"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const defaults = require("@aws-solutions-konstruk/core");
const core_1 = require("@aws-cdk/core");
const aws_logs_1 = require("@aws-cdk/aws-logs");
/**
 * @summary The ApiGatewayToLambda class.
 */
class ApiGatewayToLambda extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the ApiGatewayToLambda class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {CloudFrontToApiGatewayToLambdaProps} props - user provided props for the construct
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        // Setup the Lambda function
        this.fn = defaults.buildLambdaFunction(scope, {
            deployLambda: props.deployLambda,
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        // Setup the API Gateway
        this.api = defaults.GlobalLambdaRestApi(scope, this.fn, props.apiGatewayProps);
        // Setup the log group
        const logGroup = new aws_logs_1.LogGroup(this, 'ApiAccessLogGroup');
        // Configure API Gateway Access logging
        const stage = this.api.deploymentStage.node.findChild('Resource');
        stage.accessLogSetting = {
            destinationArn: logGroup.logGroupArn,
            format: "$context.identity.sourceIp $context.identity.caller $context.identity.user [$context.requestTime] \"$context.httpMethod $context.resourcePath $context.protocol\" $context.status $context.responseLength $context.requestId"
        };
        const deployment = (_a = this.api.latestDeployment) === null || _a === void 0 ? void 0 : _a.node.findChild('Resource');
        deployment.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W45',
                        reason: `ApiGateway does have access logging configured as part of AWS::ApiGateway::Stage.`
                    }]
            }
        };
    }
    /**
     * @summary Returns an instance of lambda.Function created by the construct.
     * @returns { lambda.Function } Instance of Function created by the construct
     * @since 0.8.0
     * @access public
     */
    lambdaFunction() {
        return this.fn;
    }
    /**
     * @summary Returns an instance of api.LambdaRestApi created by the construct.
     * @returns { api.LambdaRestApi } Instance of LambdaRestApi created by the construct
     * @since 0.8.0
     * @access public
     */
    restApi() {
        return this.api;
    }
}
exports.ApiGatewayToLambda = ApiGatewayToLambda;
//# sourceMappingURL=data:application/json;base64,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