# encoding: utf-8
from django.contrib.auth.models import User
from django.core.files.uploadedfile import SimpleUploadedFile
from django.http import FileResponse
from django.test import RequestFactory

from private_storage.tests.models import CustomerDossier
from private_storage.tests.utils import PrivateFileTestCase
from private_storage.views import PrivateStorageDetailView, PrivateStorageView


class ViewTests(PrivateFileTestCase):

    def test_detail_view(self):
        """
        Test the detail view that returns the object
        """
        CustomerDossier.objects.create(
            customer='cust1',
            file=SimpleUploadedFile('test4.txt', b'test4')
        )

        superuser = User.objects.create_superuser('admin', 'admin@example.com', 'admin')

        # Test HEAD calls too
        for method in ('GET', 'HEAD'):
            request = RequestFactory().generic(method, '/cust1/file/')
            request.user = superuser

            # Initialize locally, no need for urls.py etc..
            # This behaves like a standard DetailView
            view = PrivateStorageDetailView.as_view(
                model=CustomerDossier,
                slug_url_kwarg='customer',
                slug_field='customer',
                model_file_field='file'
            )

            response = view(
                request,
                customer='cust1',
            )
            if method == 'HEAD':
                self.assertNotIsInstance(response, FileResponse)
                self.assertEqual(response.content, b'')
            else:
                self.assertEqual(list(response.streaming_content), [b'test4'])
            self.assertEqual(response['Content-Type'], 'text/plain')
            self.assertEqual(response['Content-Length'], '5')
            self.assertIn('Last-Modified', response)

    def test_private_file_view(self):
        """
        Test the detail view that returns the object
        """
        obj = CustomerDossier.objects.create(
            customer='cust2',
            file=SimpleUploadedFile('test5.txt', b'test5')
        )
        self.assertExists('CustomerDossier', 'cust2', 'test5.txt')

        # Initialize locally, no need for urls.py etc..
        # This behaves like a standard DetailView
        view = PrivateStorageView.as_view(
            content_disposition='attachment',
        )
        superuser = User.objects.create_superuser('admin', 'admin@example.com', 'admin')

        # Test HEAD calls too
        for method in ('GET', 'HEAD'):
            request = RequestFactory().generic(method, '/cust1/file/')
            request.user = superuser
            request.META['HTTP_USER_AGENT'] = 'Test'

            response = view(
                request,
                path='CustomerDossier/cust2/test5.txt'
            )
            if method == 'HEAD':
                self.assertNotIsInstance(response, FileResponse)
                self.assertEqual(response.content, b'')
            else:
                self.assertEqual(list(response.streaming_content), [b'test5'])
            self.assertEqual(response['Content-Type'], 'text/plain')
            self.assertEqual(response['Content-Length'], '5')
            self.assertEqual(response['Content-Disposition'], "attachment; filename*=UTF-8''test5.txt")
            self.assertIn('Last-Modified', response)

    def test_private_file_view_utf8(self):
        """
        Test the detail view that returns the object
        """
        obj = CustomerDossier.objects.create(
            customer='cust2',
            file=SimpleUploadedFile(u'Heizölrückstoßabdämpfung.txt', b'test5')
        )
        self.assertExists('CustomerDossier', 'cust2', u'Heizölrückstoßabdämpfung.txt')

        # Initialize locally, no need for urls.py etc..
        # This behaves like a standard DetailView
        view = PrivateStorageView.as_view(
            content_disposition='attachment',
        )
        superuser = User.objects.create_superuser('admin', 'admin@example.com', 'admin')

        for user_agent, expect_header in [
                ('Firefox', "attachment; filename*=UTF-8''Heiz%C3%B6lr%C3%BCcksto%C3%9Fabd%C3%A4mpfung.txt"),
                ('WebKit', 'attachment; filename=Heiz\xc3\xb6lr\xc3\xbccksto\xc3\x9fabd\xc3\xa4mpfung.txt'),
                ('MSIE', 'attachment; filename=Heiz%C3%B6lr%C3%BCcksto%C3%9Fabd%C3%A4mpfung.txt'),
                ]:

            for method in ('GET', 'HEAD'):
                request = RequestFactory().generic(method, '/cust1/file/')
                request.user = superuser
                request.META['HTTP_USER_AGENT'] = user_agent

                response = view(
                    request,
                    path=u'CustomerDossier/cust2/Heizölrückstoßabdämpfung.txt'
                )
                if method == 'HEAD':
                    self.assertNotIsInstance(response, FileResponse)
                    self.assertEqual(response.content, b'')
                else:
                    self.assertEqual(list(response.streaming_content), [b'test5'])
                self.assertEqual(response['Content-Type'], 'text/plain')
                self.assertEqual(response['Content-Length'], '5')
                self.assertEqual(response['Content-Disposition'], expect_header, user_agent)
                self.assertIn('Last-Modified', response)
