"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const assertions_1 = require("aws-cdk-lib/assertions");
const cdk = require("aws-cdk-lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const lib_1 = require("../lib");
const defaults = require("@aws-solutions-constructs/core");
const test_helper_1 = require("./test-helper");
const ec2 = require("aws-cdk-lib/aws-ec2");
test('Test that properties are set correctly', () => {
    // Stack
    const stack = new cdk.Stack();
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    const construct = new lib_1.LambdaToKinesisFirehose(stack, 'test-target', {
        existingKinesisFirehose: destination.kinesisFirehose,
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        }
    });
    expect(construct.kinesisFirehose).toBeDefined();
    expect(construct.lambdaFunction).toBeDefined();
    expect(construct.vpc).not.toBeDefined();
});
test('Test existing function is utilized correctly', () => {
    // Stack
    const stack = new cdk.Stack();
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    const testName = 'test-name';
    const existingFunction = new lambda.Function(stack, 'test-function', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: "index.handler",
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        functionName: testName
    });
    new lib_1.LambdaToKinesisFirehose(stack, 'test-target', {
        existingKinesisFirehose: destination.kinesisFirehose,
        existingLambdaObj: existingFunction
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: testName
    });
    // This is 2 because there's a lambda function in the custom resource to
    // delete all the objects when cleaning up the s3 bucket in kinesisfirehose-s3
    template.resourceCountIs('AWS::Lambda::Function', 2);
});
test('Test that lambda function props are incorporated', () => {
    // Stack
    const stack = new cdk.Stack();
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    const testName = 'test-name';
    new lib_1.LambdaToKinesisFirehose(stack, 'test-target', {
        existingKinesisFirehose: destination.kinesisFirehose,
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            functionName: testName
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: testName
    });
});
test('Test that new VPC is created', () => {
    // Stack
    const stack = new cdk.Stack();
    const cidrRange = '172.0.0.0/16';
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    const construct = new lib_1.LambdaToKinesisFirehose(stack, 'test-target', {
        existingKinesisFirehose: destination.kinesisFirehose,
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        deployVpc: true,
        vpcProps: {
            ipAddresses: ec2.IpAddresses.cidr(cidrRange)
        }
    });
    expect(construct.vpc).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: cidrRange
    });
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        ServiceName: {
            "Fn::Join": [
                "",
                [
                    "com.amazonaws.",
                    {
                        Ref: "AWS::Region"
                    },
                    ".kinesis-firehose"
                ]
            ]
        },
    });
});
test('Test that existing VPC is used', () => {
    // Stack
    const stack = new cdk.Stack();
    const cidrInGetTestVpc = '172.168.0.0/16';
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    const myVpc = defaults.getTestVpc(stack, false);
    const construct = new lib_1.LambdaToKinesisFirehose(stack, 'test-target', {
        existingKinesisFirehose: destination.kinesisFirehose,
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        existingVpc: myVpc
    });
    expect(construct.vpc).toBeDefined();
    // Make sure we didn't deploy a new one anyway
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::EC2::VPC", 1);
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: cidrInGetTestVpc
    });
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        ServiceName: {
            "Fn::Join": [
                "",
                [
                    "com.amazonaws.",
                    {
                        Ref: "AWS::Region"
                    },
                    ".kinesis-firehose"
                ]
            ]
        },
    });
});
test('Test fail if existingFirehose does not have a stream name', () => {
    // Stack
    const stack = new cdk.Stack();
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    const testName = 'test-name';
    const mutatedFirehose = defaults.overrideProps(destination.kinesisFirehose, {});
    mutatedFirehose.deliveryStreamName = undefined;
    const app = () => {
        new lib_1.LambdaToKinesisFirehose(stack, 'test-target', {
            existingKinesisFirehose: mutatedFirehose,
            lambdaFunctionProps: {
                runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                handler: "index.handler",
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
                functionName: testName
            }
        });
    };
    expect(app).toThrowError(/existingKinesisFirehose must have a defined deliveryStreamName/);
});
test('Test fail Vpc check with vpcProps yet deployVpc is false', () => {
    const stack = new cdk.Stack();
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    const app = () => {
        new lib_1.LambdaToKinesisFirehose(stack, 'test-target', {
            existingKinesisFirehose: destination.kinesisFirehose,
            lambdaFunctionProps: {
                runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                handler: "index.handler",
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            },
            deployVpc: false,
            vpcProps: {
                ipAddresses: ec2.IpAddresses.cidr("10.0.0.0/16"),
            },
        });
    };
    // Assertion
    expect(app).toThrowError('Error - If deployVpc is not true, then vpcProps is ignored');
});
test('Test fail Vpc check with vpcProps yet deployVpc is undefined', () => {
    const stack = new cdk.Stack();
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    const app = () => {
        new lib_1.LambdaToKinesisFirehose(stack, 'test-target', {
            existingKinesisFirehose: destination.kinesisFirehose,
            lambdaFunctionProps: {
                runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                handler: "index.handler",
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            },
            vpcProps: {
                ipAddresses: ec2.IpAddresses.cidr("10.0.0.0/16"),
            },
        });
    };
    // Assertion
    expect(app).toThrowError('Error - If deployVpc is not true, then vpcProps is ignored');
});
test('Confirm CheckVpcProps() is being called', () => {
    const stack = new cdk.Stack();
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    const app = () => {
        // Helper declaration
        new lib_1.LambdaToKinesisFirehose(stack, 'test-lambda-kinesisfirehose', {
            existingKinesisFirehose: destination.kinesisFirehose,
            lambdaFunctionProps: {
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
                runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                handler: 'index.handler'
            },
            deployVpc: true,
            existingVpc: vpc
        });
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test('Confirm call to CheckLambdaProps', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const lambdaFunction = new lambda.Function(stack, 'a-function', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const destination = (0, test_helper_1.GetTestFirehoseDestination)(stack, 'test-destination');
    const props = {
        existingKinesisFirehose: destination.kinesisFirehose,
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        existingLambdaObj: lambdaFunction,
    };
    const app = () => {
        new lib_1.LambdaToKinesisFirehose(stack, 'test-construct', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n');
});
//# sourceMappingURL=data:application/json;base64,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