"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SsmDocument = exports.DocumentFormat = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const os_1 = require("os");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ssm_1 = require("aws-cdk-lib/aws-ssm");
const constructs_1 = require("constructs");
const string_variable_1 = require("../interface/variables/string-variable");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const yaml = require('js-yaml');
var DocumentFormat;
(function (DocumentFormat) {
    DocumentFormat[DocumentFormat["JSON"] = 0] = "JSON";
    DocumentFormat[DocumentFormat["YAML"] = 1] = "YAML";
})(DocumentFormat = exports.DocumentFormat || (exports.DocumentFormat = {}));
class SsmDocument extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.documentName = props.documentName ?? id;
        this.description = props.description ?? this.documentName;
        this.header = props.header;
        this.assumeRole = props.assumeRole;
        this.docOutputs = props.docOutputs ?? [];
        this.docInputs = props.docInputs ?? [];
        if (this.assumeRole && this.assumeRole instanceof string_variable_1.StringVariable &&
            !this.docInputs.map(i => i.name).includes(this.assumeRole.reference)) {
            throw new Error('Assume role specified but not provided in inputs: ' + this.assumeRole);
        }
        this.props = props;
        const isYaml = this.props.documentFormat == DocumentFormat.YAML;
        new aws_ssm_1.CfnDocument(this, 'Resource', {
            ...this.props,
            ...{
                content: aws_cdk_lib_1.Lazy.any({
                    produce: () => {
                        const doc = this.buildSsmDocument();
                        return JSON.parse(JSON.stringify(doc));
                    },
                }),
                documentFormat: isYaml ? 'YAML' : 'JSON',
                documentType: this.documentType(),
                tags: [{ key: 'CdkGenerated', value: 'true' }, ...(this.props.tags || [])],
            },
        });
    }
    /**
     * The SSM Document inputs optionally specify a number of parameters including allowedValues, minItems etc.
     * This function builds an object containing the relevant (declared) input parameters.
     * The return object will be used to build the yaml/json representation of the document.
     */
    formatInputs() {
        const ssmInputs = {};
        this.docInputs.forEach(inp => {
            ssmInputs[inp.name] = inp.toSsm();
        });
        return ssmInputs;
    }
    /**
       * Synthesize before calling this function!
       * You can use this to Synthesize: cdk.SynthUtils.synthesize(stack);
       *
       * Converts the objects define in the SSM Document (including all of the steps) to an SSM document string.
       * The format is dependency on the documentFormat property provided to the class.
       * The yaml can be used as is and will behave (or at least should behave) as was simulated in the runSimulation().
       * @returns a string representation of this document as an SSM formatted yaml/json.
       */
    print() {
        const isYaml = this.props.documentFormat == DocumentFormat.YAML;
        // format as JSON first so that the JSON printing goes into effect
        const doc = JSON.parse(JSON.stringify(this.buildSsmDocument()));
        if (isYaml) {
            // Prepend the document with the header if defined
            return this.headerWithComments() + yaml.dump(doc);
        }
        else {
            // There is no way to provide the header comment in a json document
            return JSON.stringify(doc);
        }
    }
    /**
       * Prepend header line with # if it does not start with #.
       * If no header is provided, this method will return an emptry string.
       * Headers are only available in yaml format.
       */
    headerWithComments() {
        if (this.header == undefined) {
            return '';
        }
        return this.header.split(os_1.EOL).map(line => line.trim().startsWith('#') ? line : '# ' + line).join(os_1.EOL) +
            // "---" demarks the end of the header and the start of the document.
            '\n---\n';
    }
}
exports.SsmDocument = SsmDocument;
_a = JSII_RTTI_SYMBOL_1;
SsmDocument[_a] = { fqn: "@cdklabs/cdk-ssm-documents.SsmDocument", version: "0.0.19" };
//# sourceMappingURL=data:application/json;base64,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