"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const index_1 = require("../lib/index");
const core_1 = require("@aws-cdk/core");
require("@aws-cdk/assert/jest");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
test('Create the Runner', () => {
    const mockApp = new core_1.App();
    const stack = new core_1.Stack(mockApp, 'testing-stack');
    new index_1.GitlabContainerRunner(stack, 'testing', { gitlabtoken: 'GITLAB_TOKEN' });
    expect(stack).toHaveResource('AWS::EC2::Instance', {
        UserData: {
            'Fn::Base64': '#!/bin/bash\nyum update -y\nyum install docker -y\nservice docker start\nusermod -aG docker ec2-user\nchmod +x /var/run/docker.sock\nservice docker restart &&  chkconfig docker on\ndocker run -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock --name gitlab-runner-register gitlab/gitlab-runner:alpine register --non-interactive --url https://gitlab.com/ --registration-token GITLAB_TOKEN --docker-pull-policy if-not-present --docker-volumes \"/var/run/docker.sock:/var/run/docker.sock\" --executor docker --docker-image \"alpine:latest\" --description \"Docker Runner\" --tag-list \"gitlab,awscdk,runner\" --docker-privileged\nsleep 2 && docker run --restart always -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock --name gitlab-runner gitlab/gitlab-runner:alpine\nusermod -aG docker ssm-user',
        },
    });
    expect(stack).toHaveResource('AWS::IAM::Role');
    expect(stack).toHaveResource('AWS::EC2::SecurityGroup', {
        SecurityGroupEgress: [{
                CidrIp: '0.0.0.0/0',
                Description: 'Allow all outbound traffic by default',
                IpProtocol: '-1',
            }],
    });
});
test('Testing runner tag change ', () => {
    const mockApp = new core_1.App();
    const stack = new core_1.Stack(mockApp, 'testing-stack');
    new index_1.GitlabContainerRunner(stack, 'testing-have-type-tag', { gitlabtoken: 'GITLAB_TOKEN', tag1: 'aa', tag2: 'bb', tag3: 'cc' });
    expect(stack).toHaveResource('AWS::EC2::Instance', {
        UserData: {
            'Fn::Base64': '#!/bin/bash\nyum update -y\nyum install docker -y\nservice docker start\nusermod -aG docker ec2-user\nchmod +x /var/run/docker.sock\nservice docker restart &&  chkconfig docker on\ndocker run -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock --name gitlab-runner-register gitlab/gitlab-runner:alpine register --non-interactive --url https://gitlab.com/ --registration-token GITLAB_TOKEN --docker-pull-policy if-not-present --docker-volumes \"/var/run/docker.sock:/var/run/docker.sock\" --executor docker --docker-image \"alpine:latest\" --description \"Docker Runner\" --tag-list \"aa,bb,cc\" --docker-privileged\nsleep 2 && docker run --restart always -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock --name gitlab-runner gitlab/gitlab-runner:alpine\nusermod -aG docker ssm-user',
        },
    });
    expect(stack).toHaveResource('AWS::EC2::SecurityGroup', {
        SecurityGroupEgress: [{
                CidrIp: '0.0.0.0/0',
                Description: 'Allow all outbound traffic by default',
                IpProtocol: '-1',
            }],
    });
});
test('Testing Runner Instance Type Change ', () => {
    const mockApp = new core_1.App();
    const stack = new core_1.Stack(mockApp, 'testing-stack');
    new index_1.GitlabContainerRunner(stack, 'testing', { gitlabtoken: 'GITLAB_TOKEN', ec2type: 't2.micro' });
    expect(stack).toHaveResource('AWS::EC2::Instance', {
        InstanceType: 't2.micro',
    });
    expect(stack).toHaveResource('AWS::EC2::SecurityGroup', {
        SecurityGroupEgress: [{
                CidrIp: '0.0.0.0/0',
                Description: 'Allow all outbound traffic by default',
                IpProtocol: '-1',
            }],
    });
});
test('Runner Can Add Ingress ', () => {
    const mockApp = new core_1.App();
    const stack = new core_1.Stack(mockApp, 'testing-stack');
    const runner = new index_1.GitlabContainerRunner(stack, 'testing', { gitlabtoken: 'GITLAB_TOKEN', ec2type: 't2.micro', tag1: 'aa', tag2: 'bb', tag3: 'cc' });
    runner.runnerEc2.connections.allowFrom(aws_ec2_1.Peer.ipv4('1.2.3.4/8'), aws_ec2_1.Port.tcp(80));
    expect(stack).toHaveResource('AWS::EC2::Instance', {
        InstanceType: 't2.micro',
    });
    expect(stack).toHaveResource('AWS::EC2::VPC', {
        CidrBlock: '10.0.0.0/16',
    });
    expect(stack).toHaveResource('AWS::EC2::SecurityGroup', {
        SecurityGroupIngress: [{
                CidrIp: '1.2.3.4/8',
                Description: 'from 1.2.3.4/8:80',
                FromPort: 80,
                IpProtocol: 'tcp',
                ToPort: 80,
            }],
    });
});
test('Runner Can Use Self VPC ', () => {
    const mockApp = new core_1.App();
    const stack = new core_1.Stack(mockApp, 'testing-stack');
    const newvpc = new aws_ec2_1.Vpc(stack, 'NEWVPC', {
        cidr: '10.1.0.0/16',
        maxAzs: 2,
        subnetConfiguration: [{
                cidrMask: 26,
                name: 'RunnerVPC',
                subnetType: aws_ec2_1.SubnetType.PUBLIC,
            }],
        natGateways: 0,
    });
    new index_1.GitlabContainerRunner(stack, 'testing', { gitlabtoken: 'GITLAB_TOKEN', ec2type: 't2.micro', selfvpc: newvpc });
    expect(stack).toHaveResource('AWS::EC2::Instance', {
        InstanceType: 't2.micro',
    });
    expect(stack).not.toHaveResource('AWS::S3::Bucket');
    expect(stack).toHaveResource('AWS::EC2::VPC', {
        CidrBlock: '10.1.0.0/16',
    });
});
test('Runner Can Use Self Role ', () => {
    const mockApp = new core_1.App();
    const stack = new core_1.Stack(mockApp, 'testing-stack');
    const role = new aws_iam_1.Role(stack, 'runner-role', {
        assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
        description: 'For Gitlab EC2 Runner Test Role',
        roleName: 'TestRole',
    });
    new index_1.GitlabContainerRunner(stack, 'testing', { gitlabtoken: 'GITLAB_TOKEN', ec2type: 't2.micro', ec2iamrole: role });
    expect(stack).toHaveResource('AWS::EC2::Instance', {
        InstanceType: 't2.micro',
    });
    expect(stack).toHaveResource('AWS::IAM::Role', {
        RoleName: 'TestRole',
    });
});
test('Can Use Coustom Gitlab Url', () => {
    const mockApp = new core_1.App();
    const stack = new core_1.Stack(mockApp, 'testing-stack');
    const role = new aws_iam_1.Role(stack, 'runner-role', {
        assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
        description: 'For Gitlab EC2 Runner Test Role',
        roleName: 'TestRole',
    });
    new index_1.GitlabContainerRunner(stack, 'testing', { gitlabtoken: 'GITLAB_TOKEN', ec2type: 't2.micro', ec2iamrole: role, gitlaburl: 'https://gitlab.my.com/' });
    expect(stack).toHaveResource('AWS::EC2::Instance', {
        InstanceType: 't2.micro',
    });
    expect(stack).toHaveResource('AWS::EC2::Instance', {
        UserData: {
            'Fn::Base64': '#!/bin/bash\nyum update -y\nyum install docker -y\nservice docker start\nusermod -aG docker ec2-user\nchmod +x /var/run/docker.sock\nservice docker restart &&  chkconfig docker on\ndocker run -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock --name gitlab-runner-register gitlab/gitlab-runner:alpine register --non-interactive --url https://gitlab.my.com/ --registration-token GITLAB_TOKEN --docker-pull-policy if-not-present --docker-volumes \"/var/run/docker.sock:/var/run/docker.sock\" --executor docker --docker-image \"alpine:latest\" --description \"Docker Runner\" --tag-list \"gitlab,awscdk,runner\" --docker-privileged\nsleep 2 && docker run --restart always -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock --name gitlab-runner gitlab/gitlab-runner:alpine\nusermod -aG docker ssm-user',
        },
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXgudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImluZGV4LnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSx3Q0FBcUQ7QUFDckQsd0NBQTJDO0FBQzNDLGdDQUE4QjtBQUM5Qiw4Q0FBK0Q7QUFDL0QsOENBQXlEO0FBR3pELElBQUksQ0FBQyxtQkFBbUIsRUFBRSxHQUFHLEVBQUU7SUFDN0IsTUFBTSxPQUFPLEdBQUcsSUFBSSxVQUFHLEVBQUUsQ0FBQztJQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssQ0FBQyxPQUFPLEVBQUUsZUFBZSxDQUFDLENBQUM7SUFDbEQsSUFBSSw2QkFBcUIsQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFLEVBQUUsV0FBVyxFQUFFLGNBQWMsRUFBRSxDQUFDLENBQUM7SUFDN0UsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxvQkFBb0IsRUFBRTtRQUNqRCxRQUFRLEVBQUU7WUFDUixZQUFZLEVBQUUsODNCQUE4M0I7U0FDNzRCO0tBQ0YsQ0FBQyxDQUFDO0lBQ0gsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO0lBQy9DLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMseUJBQXlCLEVBQUU7UUFDdEQsbUJBQW1CLEVBQUUsQ0FBQztnQkFDcEIsTUFBTSxFQUFFLFdBQVc7Z0JBQ25CLFdBQVcsRUFBRSx1Q0FBdUM7Z0JBQ3BELFVBQVUsRUFBRSxJQUFJO2FBQ2pCLENBQUM7S0FDSCxDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw0QkFBNEIsRUFBRSxHQUFHLEVBQUU7SUFDdEMsTUFBTSxPQUFPLEdBQUcsSUFBSSxVQUFHLEVBQUUsQ0FBQztJQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssQ0FBQyxPQUFPLEVBQUUsZUFBZSxDQUFDLENBQUM7SUFDbEQsSUFBSSw2QkFBcUIsQ0FBQyxLQUFLLEVBQUUsdUJBQXVCLEVBQUUsRUFBRSxXQUFXLEVBQUUsY0FBYyxFQUFFLElBQUksRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztJQUMvSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLG9CQUFvQixFQUFFO1FBQ2pELFFBQVEsRUFBRTtZQUNSLFlBQVksRUFBRSxrM0JBQWszQjtTQUNqNEI7S0FDRixDQUFDLENBQUM7SUFDSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLHlCQUF5QixFQUFFO1FBQ3RELG1CQUFtQixFQUFFLENBQUM7Z0JBQ3BCLE1BQU0sRUFBRSxXQUFXO2dCQUNuQixXQUFXLEVBQUUsdUNBQXVDO2dCQUNwRCxVQUFVLEVBQUUsSUFBSTthQUNqQixDQUFDO0tBQ0gsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsc0NBQXNDLEVBQUUsR0FBRyxFQUFFO0lBQ2hELE1BQU0sT0FBTyxHQUFHLElBQUksVUFBRyxFQUFFLENBQUM7SUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLENBQUMsT0FBTyxFQUFFLGVBQWUsQ0FBQyxDQUFDO0lBQ2xELElBQUksNkJBQXFCLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRSxFQUFFLFdBQVcsRUFBRSxjQUFjLEVBQUUsT0FBTyxFQUFFLFVBQVUsRUFBRSxDQUFDLENBQUM7SUFDbEcsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxvQkFBb0IsRUFBRTtRQUNqRCxZQUFZLEVBQUUsVUFBVTtLQUN6QixDQUFDLENBQUM7SUFDSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLHlCQUF5QixFQUFFO1FBQ3RELG1CQUFtQixFQUFFLENBQUM7Z0JBQ3BCLE1BQU0sRUFBRSxXQUFXO2dCQUNuQixXQUFXLEVBQUUsdUNBQXVDO2dCQUNwRCxVQUFVLEVBQUUsSUFBSTthQUNqQixDQUFDO0tBQ0gsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMseUJBQXlCLEVBQUUsR0FBRyxFQUFFO0lBQ25DLE1BQU0sT0FBTyxHQUFHLElBQUksVUFBRyxFQUFFLENBQUM7SUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLENBQUMsT0FBTyxFQUFFLGVBQWUsQ0FBQyxDQUFDO0lBQ2xELE1BQU0sTUFBTSxHQUFHLElBQUksNkJBQXFCLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRSxFQUFFLFdBQVcsRUFBRSxjQUFjLEVBQUUsT0FBTyxFQUFFLFVBQVUsRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7SUFDckosTUFBTSxDQUFDLFNBQVMsQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLGNBQUksQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLEVBQUUsY0FBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQzdFLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsb0JBQW9CLEVBQUU7UUFDakQsWUFBWSxFQUFFLFVBQVU7S0FDekIsQ0FBQyxDQUFDO0lBQ0gsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxlQUFlLEVBQUU7UUFDNUMsU0FBUyxFQUFFLGFBQWE7S0FDekIsQ0FDQSxDQUFDO0lBQ0YsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyx5QkFBeUIsRUFBRTtRQUN0RCxvQkFBb0IsRUFBRSxDQUFDO2dCQUNyQixNQUFNLEVBQUUsV0FBVztnQkFDbkIsV0FBVyxFQUFFLG1CQUFtQjtnQkFDaEMsUUFBUSxFQUFFLEVBQUU7Z0JBQ1osVUFBVSxFQUFFLEtBQUs7Z0JBQ2pCLE1BQU0sRUFBRSxFQUFFO2FBQ1gsQ0FBQztLQUNILENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDBCQUEwQixFQUFFLEdBQUcsRUFBRTtJQUNwQyxNQUFNLE9BQU8sR0FBRyxJQUFJLFVBQUcsRUFBRSxDQUFDO0lBQzFCLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxDQUFDLE9BQU8sRUFBRSxlQUFlLENBQUMsQ0FBQztJQUNsRCxNQUFNLE1BQU0sR0FBRyxJQUFJLGFBQUcsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1FBQ3RDLElBQUksRUFBRSxhQUFhO1FBQ25CLE1BQU0sRUFBRSxDQUFDO1FBQ1QsbUJBQW1CLEVBQUUsQ0FBQztnQkFDcEIsUUFBUSxFQUFFLEVBQUU7Z0JBQ1osSUFBSSxFQUFFLFdBQVc7Z0JBQ2pCLFVBQVUsRUFBRSxvQkFBVSxDQUFDLE1BQU07YUFDOUIsQ0FBQztRQUNGLFdBQVcsRUFBRSxDQUFDO0tBQ2YsQ0FBQyxDQUFDO0lBQ0gsSUFBSSw2QkFBcUIsQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFLEVBQUUsV0FBVyxFQUFFLGNBQWMsRUFBRSxPQUFPLEVBQUUsVUFBVSxFQUFFLE9BQU8sRUFBRSxNQUFNLEVBQUUsQ0FBQyxDQUFDO0lBQ25ILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsb0JBQW9CLEVBQUU7UUFDakQsWUFBWSxFQUFFLFVBQVU7S0FDekIsQ0FBQyxDQUFDO0lBQ0gsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEdBQUcsQ0FBQyxjQUFjLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNwRCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLGVBQWUsRUFBRTtRQUM1QyxTQUFTLEVBQUUsYUFBYTtLQUN6QixDQUNBLENBQUM7QUFDSixDQUFDLENBQUMsQ0FBQztBQUdILElBQUksQ0FBQywyQkFBMkIsRUFBRSxHQUFHLEVBQUU7SUFDckMsTUFBTSxPQUFPLEdBQUcsSUFBSSxVQUFHLEVBQUUsQ0FBQztJQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssQ0FBQyxPQUFPLEVBQUUsZUFBZSxDQUFDLENBQUM7SUFDbEQsTUFBTSxJQUFJLEdBQUcsSUFBSSxjQUFJLENBQUMsS0FBSyxFQUFFLGFBQWEsRUFBRTtRQUMxQyxTQUFTLEVBQUUsSUFBSSwwQkFBZ0IsQ0FBQyxtQkFBbUIsQ0FBQztRQUNwRCxXQUFXLEVBQUUsaUNBQWlDO1FBQzlDLFFBQVEsRUFBRSxVQUFVO0tBQ3JCLENBQUMsQ0FBQztJQUNILElBQUksNkJBQXFCLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRSxFQUFFLFdBQVcsRUFBRSxjQUFjLEVBQUUsT0FBTyxFQUFFLFVBQVUsRUFBRSxVQUFVLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztJQUNwSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLG9CQUFvQixFQUFFO1FBQ2pELFlBQVksRUFBRSxVQUFVO0tBQ3pCLENBQUMsQ0FBQztJQUNILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsZ0JBQWdCLEVBQUU7UUFDN0MsUUFBUSxFQUFFLFVBQVU7S0FDckIsQ0FDQSxDQUFDO0FBQ0osQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsNEJBQTRCLEVBQUUsR0FBRyxFQUFFO0lBQ3RDLE1BQU0sT0FBTyxHQUFHLElBQUksVUFBRyxFQUFFLENBQUM7SUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLENBQUMsT0FBTyxFQUFFLGVBQWUsQ0FBQyxDQUFDO0lBQ2xELE1BQU0sSUFBSSxHQUFHLElBQUksY0FBSSxDQUFDLEtBQUssRUFBRSxhQUFhLEVBQUU7UUFDMUMsU0FBUyxFQUFFLElBQUksMEJBQWdCLENBQUMsbUJBQW1CLENBQUM7UUFDcEQsV0FBVyxFQUFFLGlDQUFpQztRQUM5QyxRQUFRLEVBQUUsVUFBVTtLQUNyQixDQUFDLENBQUM7SUFDSCxJQUFJLDZCQUFxQixDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUUsRUFBRSxXQUFXLEVBQUUsY0FBYyxFQUFFLE9BQU8sRUFBRSxVQUFVLEVBQUUsVUFBVSxFQUFFLElBQUksRUFBRSxTQUFTLEVBQUUsd0JBQXdCLEVBQUMsQ0FBQyxDQUFDO0lBQ3hKLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsb0JBQW9CLEVBQUU7UUFDakQsWUFBWSxFQUFFLFVBQVU7S0FDekIsQ0FBQyxDQUFDO0lBQ0gsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxvQkFBb0IsRUFBRTtRQUNqRCxRQUFRLEVBQUU7WUFDUixZQUFZLEVBQUUsaTRCQUFpNEI7U0FDaDVCO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBHaXRsYWJDb250YWluZXJSdW5uZXIgfSBmcm9tICcuLi9saWIvaW5kZXgnO1xuaW1wb3J0IHsgQXBwLCBTdGFjayB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0ICdAYXdzLWNkay9hc3NlcnQvamVzdCc7XG5pbXBvcnQgeyBQZWVyLCBQb3J0LCBWcGMsIFN1Ym5ldFR5cGUgfSBmcm9tICdAYXdzLWNkay9hd3MtZWMyJztcbmltcG9ydCB7IFJvbGUsIFNlcnZpY2VQcmluY2lwYWwgfSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJ1xuXG5cbnRlc3QoJ0NyZWF0ZSB0aGUgUnVubmVyJywgKCkgPT4ge1xuICBjb25zdCBtb2NrQXBwID0gbmV3IEFwcCgpO1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjayhtb2NrQXBwLCAndGVzdGluZy1zdGFjaycpO1xuICBuZXcgR2l0bGFiQ29udGFpbmVyUnVubmVyKHN0YWNrLCAndGVzdGluZycsIHsgZ2l0bGFidG9rZW46ICdHSVRMQUJfVE9LRU4nIH0pO1xuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OkVDMjo6SW5zdGFuY2UnLCB7XG4gICAgVXNlckRhdGE6IHtcbiAgICAgICdGbjo6QmFzZTY0JzogJyMhL2Jpbi9iYXNoXFxueXVtIHVwZGF0ZSAteVxcbnl1bSBpbnN0YWxsIGRvY2tlciAteVxcbnNlcnZpY2UgZG9ja2VyIHN0YXJ0XFxudXNlcm1vZCAtYUcgZG9ja2VyIGVjMi11c2VyXFxuY2htb2QgK3ggL3Zhci9ydW4vZG9ja2VyLnNvY2tcXG5zZXJ2aWNlIGRvY2tlciByZXN0YXJ0ICYmICBjaGtjb25maWcgZG9ja2VyIG9uXFxuZG9ja2VyIHJ1biAtZCAtdiAvaG9tZS9lYzItdXNlci8uZ2l0bGFiLXJ1bm5lcjovZXRjL2dpdGxhYi1ydW5uZXIgLXYgL3Zhci9ydW4vZG9ja2VyLnNvY2s6L3Zhci9ydW4vZG9ja2VyLnNvY2sgLS1uYW1lIGdpdGxhYi1ydW5uZXItcmVnaXN0ZXIgZ2l0bGFiL2dpdGxhYi1ydW5uZXI6YWxwaW5lIHJlZ2lzdGVyIC0tbm9uLWludGVyYWN0aXZlIC0tdXJsIGh0dHBzOi8vZ2l0bGFiLmNvbS8gLS1yZWdpc3RyYXRpb24tdG9rZW4gR0lUTEFCX1RPS0VOIC0tZG9ja2VyLXB1bGwtcG9saWN5IGlmLW5vdC1wcmVzZW50IC0tZG9ja2VyLXZvbHVtZXMgXFxcIi92YXIvcnVuL2RvY2tlci5zb2NrOi92YXIvcnVuL2RvY2tlci5zb2NrXFxcIiAtLWV4ZWN1dG9yIGRvY2tlciAtLWRvY2tlci1pbWFnZSBcXFwiYWxwaW5lOmxhdGVzdFxcXCIgLS1kZXNjcmlwdGlvbiBcXFwiRG9ja2VyIFJ1bm5lclxcXCIgLS10YWctbGlzdCBcXFwiZ2l0bGFiLGF3c2NkayxydW5uZXJcXFwiIC0tZG9ja2VyLXByaXZpbGVnZWRcXG5zbGVlcCAyICYmIGRvY2tlciBydW4gLS1yZXN0YXJ0IGFsd2F5cyAtZCAtdiAvaG9tZS9lYzItdXNlci8uZ2l0bGFiLXJ1bm5lcjovZXRjL2dpdGxhYi1ydW5uZXIgLXYgL3Zhci9ydW4vZG9ja2VyLnNvY2s6L3Zhci9ydW4vZG9ja2VyLnNvY2sgLS1uYW1lIGdpdGxhYi1ydW5uZXIgZ2l0bGFiL2dpdGxhYi1ydW5uZXI6YWxwaW5lXFxudXNlcm1vZCAtYUcgZG9ja2VyIHNzbS11c2VyJyxcbiAgICB9LFxuICB9KTtcbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlJvbGUnKTtcbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpFQzI6OlNlY3VyaXR5R3JvdXAnLCB7XG4gICAgU2VjdXJpdHlHcm91cEVncmVzczogW3tcbiAgICAgIENpZHJJcDogJzAuMC4wLjAvMCcsXG4gICAgICBEZXNjcmlwdGlvbjogJ0FsbG93IGFsbCBvdXRib3VuZCB0cmFmZmljIGJ5IGRlZmF1bHQnLFxuICAgICAgSXBQcm90b2NvbDogJy0xJyxcbiAgICB9XSxcbiAgfSk7XG59KTtcblxudGVzdCgnVGVzdGluZyBydW5uZXIgdGFnIGNoYW5nZSAnLCAoKSA9PiB7XG4gIGNvbnN0IG1vY2tBcHAgPSBuZXcgQXBwKCk7XG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKG1vY2tBcHAsICd0ZXN0aW5nLXN0YWNrJyk7XG4gIG5ldyBHaXRsYWJDb250YWluZXJSdW5uZXIoc3RhY2ssICd0ZXN0aW5nLWhhdmUtdHlwZS10YWcnLCB7IGdpdGxhYnRva2VuOiAnR0lUTEFCX1RPS0VOJywgdGFnMTogJ2FhJywgdGFnMjogJ2JiJywgdGFnMzogJ2NjJyB9KTtcbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpFQzI6Okluc3RhbmNlJywge1xuICAgIFVzZXJEYXRhOiB7XG4gICAgICAnRm46OkJhc2U2NCc6ICcjIS9iaW4vYmFzaFxcbnl1bSB1cGRhdGUgLXlcXG55dW0gaW5zdGFsbCBkb2NrZXIgLXlcXG5zZXJ2aWNlIGRvY2tlciBzdGFydFxcbnVzZXJtb2QgLWFHIGRvY2tlciBlYzItdXNlclxcbmNobW9kICt4IC92YXIvcnVuL2RvY2tlci5zb2NrXFxuc2VydmljZSBkb2NrZXIgcmVzdGFydCAmJiAgY2hrY29uZmlnIGRvY2tlciBvblxcbmRvY2tlciBydW4gLWQgLXYgL2hvbWUvZWMyLXVzZXIvLmdpdGxhYi1ydW5uZXI6L2V0Yy9naXRsYWItcnVubmVyIC12IC92YXIvcnVuL2RvY2tlci5zb2NrOi92YXIvcnVuL2RvY2tlci5zb2NrIC0tbmFtZSBnaXRsYWItcnVubmVyLXJlZ2lzdGVyIGdpdGxhYi9naXRsYWItcnVubmVyOmFscGluZSByZWdpc3RlciAtLW5vbi1pbnRlcmFjdGl2ZSAtLXVybCBodHRwczovL2dpdGxhYi5jb20vIC0tcmVnaXN0cmF0aW9uLXRva2VuIEdJVExBQl9UT0tFTiAtLWRvY2tlci1wdWxsLXBvbGljeSBpZi1ub3QtcHJlc2VudCAtLWRvY2tlci12b2x1bWVzIFxcXCIvdmFyL3J1bi9kb2NrZXIuc29jazovdmFyL3J1bi9kb2NrZXIuc29ja1xcXCIgLS1leGVjdXRvciBkb2NrZXIgLS1kb2NrZXItaW1hZ2UgXFxcImFscGluZTpsYXRlc3RcXFwiIC0tZGVzY3JpcHRpb24gXFxcIkRvY2tlciBSdW5uZXJcXFwiIC0tdGFnLWxpc3QgXFxcImFhLGJiLGNjXFxcIiAtLWRvY2tlci1wcml2aWxlZ2VkXFxuc2xlZXAgMiAmJiBkb2NrZXIgcnVuIC0tcmVzdGFydCBhbHdheXMgLWQgLXYgL2hvbWUvZWMyLXVzZXIvLmdpdGxhYi1ydW5uZXI6L2V0Yy9naXRsYWItcnVubmVyIC12IC92YXIvcnVuL2RvY2tlci5zb2NrOi92YXIvcnVuL2RvY2tlci5zb2NrIC0tbmFtZSBnaXRsYWItcnVubmVyIGdpdGxhYi9naXRsYWItcnVubmVyOmFscGluZVxcbnVzZXJtb2QgLWFHIGRvY2tlciBzc20tdXNlcicsXG4gICAgfSxcbiAgfSk7XG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6RUMyOjpTZWN1cml0eUdyb3VwJywge1xuICAgIFNlY3VyaXR5R3JvdXBFZ3Jlc3M6IFt7XG4gICAgICBDaWRySXA6ICcwLjAuMC4wLzAnLFxuICAgICAgRGVzY3JpcHRpb246ICdBbGxvdyBhbGwgb3V0Ym91bmQgdHJhZmZpYyBieSBkZWZhdWx0JyxcbiAgICAgIElwUHJvdG9jb2w6ICctMScsXG4gICAgfV0sXG4gIH0pO1xufSk7XG5cbnRlc3QoJ1Rlc3RpbmcgUnVubmVyIEluc3RhbmNlIFR5cGUgQ2hhbmdlICcsICgpID0+IHtcbiAgY29uc3QgbW9ja0FwcCA9IG5ldyBBcHAoKTtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2sobW9ja0FwcCwgJ3Rlc3Rpbmctc3RhY2snKTtcbiAgbmV3IEdpdGxhYkNvbnRhaW5lclJ1bm5lcihzdGFjaywgJ3Rlc3RpbmcnLCB7IGdpdGxhYnRva2VuOiAnR0lUTEFCX1RPS0VOJywgZWMydHlwZTogJ3QyLm1pY3JvJyB9KTtcbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpFQzI6Okluc3RhbmNlJywge1xuICAgIEluc3RhbmNlVHlwZTogJ3QyLm1pY3JvJyxcbiAgfSk7XG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6RUMyOjpTZWN1cml0eUdyb3VwJywge1xuICAgIFNlY3VyaXR5R3JvdXBFZ3Jlc3M6IFt7XG4gICAgICBDaWRySXA6ICcwLjAuMC4wLzAnLFxuICAgICAgRGVzY3JpcHRpb246ICdBbGxvdyBhbGwgb3V0Ym91bmQgdHJhZmZpYyBieSBkZWZhdWx0JyxcbiAgICAgIElwUHJvdG9jb2w6ICctMScsXG4gICAgfV0sXG4gIH0pO1xufSk7XG5cbnRlc3QoJ1J1bm5lciBDYW4gQWRkIEluZ3Jlc3MgJywgKCkgPT4ge1xuICBjb25zdCBtb2NrQXBwID0gbmV3IEFwcCgpO1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjayhtb2NrQXBwLCAndGVzdGluZy1zdGFjaycpO1xuICBjb25zdCBydW5uZXIgPSBuZXcgR2l0bGFiQ29udGFpbmVyUnVubmVyKHN0YWNrLCAndGVzdGluZycsIHsgZ2l0bGFidG9rZW46ICdHSVRMQUJfVE9LRU4nLCBlYzJ0eXBlOiAndDIubWljcm8nLCB0YWcxOiAnYWEnLCB0YWcyOiAnYmInLCB0YWczOiAnY2MnIH0pO1xuICBydW5uZXIucnVubmVyRWMyLmNvbm5lY3Rpb25zLmFsbG93RnJvbShQZWVyLmlwdjQoJzEuMi4zLjQvOCcpLCBQb3J0LnRjcCg4MCkpO1xuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OkVDMjo6SW5zdGFuY2UnLCB7XG4gICAgSW5zdGFuY2VUeXBlOiAndDIubWljcm8nLFxuICB9KTtcbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpFQzI6OlZQQycsIHtcbiAgICBDaWRyQmxvY2s6ICcxMC4wLjAuMC8xNicsXG4gIH0sXG4gICk7XG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6RUMyOjpTZWN1cml0eUdyb3VwJywge1xuICAgIFNlY3VyaXR5R3JvdXBJbmdyZXNzOiBbe1xuICAgICAgQ2lkcklwOiAnMS4yLjMuNC84JyxcbiAgICAgIERlc2NyaXB0aW9uOiAnZnJvbSAxLjIuMy40Lzg6ODAnLFxuICAgICAgRnJvbVBvcnQ6IDgwLFxuICAgICAgSXBQcm90b2NvbDogJ3RjcCcsXG4gICAgICBUb1BvcnQ6IDgwLFxuICAgIH1dLFxuICB9KTtcbn0pO1xuXG50ZXN0KCdSdW5uZXIgQ2FuIFVzZSBTZWxmIFZQQyAnLCAoKSA9PiB7XG4gIGNvbnN0IG1vY2tBcHAgPSBuZXcgQXBwKCk7XG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKG1vY2tBcHAsICd0ZXN0aW5nLXN0YWNrJyk7XG4gIGNvbnN0IG5ld3ZwYyA9IG5ldyBWcGMoc3RhY2ssICdORVdWUEMnLCB7XG4gICAgY2lkcjogJzEwLjEuMC4wLzE2JyxcbiAgICBtYXhBenM6IDIsXG4gICAgc3VibmV0Q29uZmlndXJhdGlvbjogW3tcbiAgICAgIGNpZHJNYXNrOiAyNixcbiAgICAgIG5hbWU6ICdSdW5uZXJWUEMnLFxuICAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QVUJMSUMsXG4gICAgfV0sXG4gICAgbmF0R2F0ZXdheXM6IDAsXG4gIH0pO1xuICBuZXcgR2l0bGFiQ29udGFpbmVyUnVubmVyKHN0YWNrLCAndGVzdGluZycsIHsgZ2l0bGFidG9rZW46ICdHSVRMQUJfVE9LRU4nLCBlYzJ0eXBlOiAndDIubWljcm8nLCBzZWxmdnBjOiBuZXd2cGMgfSk7XG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6RUMyOjpJbnN0YW5jZScsIHtcbiAgICBJbnN0YW5jZVR5cGU6ICd0Mi5taWNybycsXG4gIH0pO1xuICBleHBlY3Qoc3RhY2spLm5vdC50b0hhdmVSZXNvdXJjZSgnQVdTOjpTMzo6QnVja2V0Jyk7XG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6RUMyOjpWUEMnLCB7XG4gICAgQ2lkckJsb2NrOiAnMTAuMS4wLjAvMTYnLFxuICB9LFxuICApO1xufSk7XG5cblxudGVzdCgnUnVubmVyIENhbiBVc2UgU2VsZiBSb2xlICcsICgpID0+IHtcbiAgY29uc3QgbW9ja0FwcCA9IG5ldyBBcHAoKTtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2sobW9ja0FwcCwgJ3Rlc3Rpbmctc3RhY2snKTtcbiAgY29uc3Qgcm9sZSA9IG5ldyBSb2xlKHN0YWNrLCAncnVubmVyLXJvbGUnLCB7XG4gICAgYXNzdW1lZEJ5OiBuZXcgU2VydmljZVByaW5jaXBhbCgnZWMyLmFtYXpvbmF3cy5jb20nKSxcbiAgICBkZXNjcmlwdGlvbjogJ0ZvciBHaXRsYWIgRUMyIFJ1bm5lciBUZXN0IFJvbGUnLFxuICAgIHJvbGVOYW1lOiAnVGVzdFJvbGUnLFxuICB9KTtcbiAgbmV3IEdpdGxhYkNvbnRhaW5lclJ1bm5lcihzdGFjaywgJ3Rlc3RpbmcnLCB7IGdpdGxhYnRva2VuOiAnR0lUTEFCX1RPS0VOJywgZWMydHlwZTogJ3QyLm1pY3JvJywgZWMyaWFtcm9sZTogcm9sZSB9KTtcbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpFQzI6Okluc3RhbmNlJywge1xuICAgIEluc3RhbmNlVHlwZTogJ3QyLm1pY3JvJyxcbiAgfSk7XG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6SUFNOjpSb2xlJywge1xuICAgIFJvbGVOYW1lOiAnVGVzdFJvbGUnLFxuICB9LFxuICApO1xufSk7XG5cbnRlc3QoJ0NhbiBVc2UgQ291c3RvbSBHaXRsYWIgVXJsJywgKCkgPT4ge1xuICBjb25zdCBtb2NrQXBwID0gbmV3IEFwcCgpO1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjayhtb2NrQXBwLCAndGVzdGluZy1zdGFjaycpO1xuICBjb25zdCByb2xlID0gbmV3IFJvbGUoc3RhY2ssICdydW5uZXItcm9sZScsIHtcbiAgICBhc3N1bWVkQnk6IG5ldyBTZXJ2aWNlUHJpbmNpcGFsKCdlYzIuYW1hem9uYXdzLmNvbScpLFxuICAgIGRlc2NyaXB0aW9uOiAnRm9yIEdpdGxhYiBFQzIgUnVubmVyIFRlc3QgUm9sZScsXG4gICAgcm9sZU5hbWU6ICdUZXN0Um9sZScsXG4gIH0pO1xuICBuZXcgR2l0bGFiQ29udGFpbmVyUnVubmVyKHN0YWNrLCAndGVzdGluZycsIHsgZ2l0bGFidG9rZW46ICdHSVRMQUJfVE9LRU4nLCBlYzJ0eXBlOiAndDIubWljcm8nLCBlYzJpYW1yb2xlOiByb2xlICxnaXRsYWJ1cmw6ICdodHRwczovL2dpdGxhYi5teS5jb20vJ30pO1xuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OkVDMjo6SW5zdGFuY2UnLCB7XG4gICAgSW5zdGFuY2VUeXBlOiAndDIubWljcm8nLFxuICB9KTtcbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpFQzI6Okluc3RhbmNlJywge1xuICAgIFVzZXJEYXRhOiB7XG4gICAgICAnRm46OkJhc2U2NCc6ICcjIS9iaW4vYmFzaFxcbnl1bSB1cGRhdGUgLXlcXG55dW0gaW5zdGFsbCBkb2NrZXIgLXlcXG5zZXJ2aWNlIGRvY2tlciBzdGFydFxcbnVzZXJtb2QgLWFHIGRvY2tlciBlYzItdXNlclxcbmNobW9kICt4IC92YXIvcnVuL2RvY2tlci5zb2NrXFxuc2VydmljZSBkb2NrZXIgcmVzdGFydCAmJiAgY2hrY29uZmlnIGRvY2tlciBvblxcbmRvY2tlciBydW4gLWQgLXYgL2hvbWUvZWMyLXVzZXIvLmdpdGxhYi1ydW5uZXI6L2V0Yy9naXRsYWItcnVubmVyIC12IC92YXIvcnVuL2RvY2tlci5zb2NrOi92YXIvcnVuL2RvY2tlci5zb2NrIC0tbmFtZSBnaXRsYWItcnVubmVyLXJlZ2lzdGVyIGdpdGxhYi9naXRsYWItcnVubmVyOmFscGluZSByZWdpc3RlciAtLW5vbi1pbnRlcmFjdGl2ZSAtLXVybCBodHRwczovL2dpdGxhYi5teS5jb20vIC0tcmVnaXN0cmF0aW9uLXRva2VuIEdJVExBQl9UT0tFTiAtLWRvY2tlci1wdWxsLXBvbGljeSBpZi1ub3QtcHJlc2VudCAtLWRvY2tlci12b2x1bWVzIFxcXCIvdmFyL3J1bi9kb2NrZXIuc29jazovdmFyL3J1bi9kb2NrZXIuc29ja1xcXCIgLS1leGVjdXRvciBkb2NrZXIgLS1kb2NrZXItaW1hZ2UgXFxcImFscGluZTpsYXRlc3RcXFwiIC0tZGVzY3JpcHRpb24gXFxcIkRvY2tlciBSdW5uZXJcXFwiIC0tdGFnLWxpc3QgXFxcImdpdGxhYixhd3NjZGsscnVubmVyXFxcIiAtLWRvY2tlci1wcml2aWxlZ2VkXFxuc2xlZXAgMiAmJiBkb2NrZXIgcnVuIC0tcmVzdGFydCBhbHdheXMgLWQgLXYgL2hvbWUvZWMyLXVzZXIvLmdpdGxhYi1ydW5uZXI6L2V0Yy9naXRsYWItcnVubmVyIC12IC92YXIvcnVuL2RvY2tlci5zb2NrOi92YXIvcnVuL2RvY2tlci5zb2NrIC0tbmFtZSBnaXRsYWItcnVubmVyIGdpdGxhYi9naXRsYWItcnVubmVyOmFscGluZVxcbnVzZXJtb2QgLWFHIGRvY2tlciBzc20tdXNlcicsXG4gICAgfSxcbiAgfSk7XG59KTsiXX0=