"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KongEksDataPlane = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const KongDataPlane_1 = require("../../../KongDataPlane");
class KongEksDataPlane extends KongDataPlane_1.KongDataPlane {
    constructor(scope, id, props) {
        super(scope, id);
        const kong_namespace = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongNamespace', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'Namespace',
                    metadata: { name: props.namespace },
                },
            ],
        });
        const kong_certificate = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongCert', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'cert-manager.io/v1',
                    kind: 'Certificate',
                    metadata: {
                        name: props.cacertname,
                        namespace: props.namespace,
                    },
                    spec: {
                        commonName: props.endpoints.hostedZoneName,
                        dnsNames: [
                            props.endpoints.clusterDns,
                            props.endpoints.telemetryDns,
                            props.endpoints.hostedZoneName,
                        ],
                        duration: '2160h0m0s',
                        issuerRef: {
                            group: 'awspca.cert-manager.io',
                            kind: 'AWSPCAClusterIssuer',
                            name: props.clusterIssuerName,
                        },
                        renewBefore: '360h0m0s',
                        secretName: `${props.clusterIssuerName}-secret`,
                        usages: [
                            'server auth',
                            'client auth',
                        ],
                        privateKey: {
                            algorithm: 'RSA',
                            size: 2048,
                        },
                    },
                },
            ],
        });
        kong_certificate.node.addDependency(kong_namespace);
        const kong_dp_service_account = new aws_cdk_lib_1.aws_eks.ServiceAccount(this, 'KongDpSA', {
            cluster: props.cluster,
            name: 'kong-dp-service-account',
            namespace: props.namespace,
        });
        kong_dp_service_account.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'secretsmanager:GetSecretValue',
                'secretsmanager:DescribeSecret',
            ],
        }));
        kong_dp_service_account.node.addDependency(kong_namespace);
        const secrets_crd_mount = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'SecretsMount', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'secrets-store.csi.x-k8s.io/v1',
                    kind: 'SecretProviderClass',
                    metadata: {
                        name: props.license_secret_name,
                        namespace: props.namespace,
                    },
                    spec: {
                        provider: 'aws',
                        secretObjects: [
                            {
                                secretName: props.license_secret_name,
                                type: 'opaque',
                                data: [
                                    {
                                        objectName: props.license_secret_name,
                                        key: 'license',
                                    },
                                ],
                            },
                        ],
                        parameters: {
                            objects: `- objectName: "${props.license_secret_name}"\n  objectType: "secretsmanager"\n`,
                        },
                    },
                },
            ],
        });
        secrets_crd_mount.node.addDependency(kong_namespace);
        const deploy_kong_dp_helm = new aws_cdk_lib_1.aws_eks.HelmChart(this, 'KongDpHelm', {
            cluster: props.cluster,
            repository: 'https://charts.konghq.com',
            chart: 'kong',
            release: 'kong',
            namespace: props.namespace,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            wait: true,
            createNamespace: true,
            values: {
                resources: {
                    requests: {
                        cpu: '300m',
                        memory: '300Mi',
                    },
                    limits: {
                        cpu: '1200m',
                        memory: '800Mi',
                    },
                },
                autoscaling: {
                    enabled: true,
                    minReplicas: 1,
                    maxReplicas: 20,
                    metrics: [
                        {
                            type: 'Resource',
                            resource: {
                                name: 'cpu',
                                target: {
                                    type: 'Utilization',
                                    averageUtilization: 75,
                                },
                            },
                        },
                    ],
                },
                ingressController: {
                    enabled: false,
                },
                // image: {
                //   repository: 'kong/kong-gateway',
                //   tag: '2.5.0.0-alpine',
                // },
                env: {
                    database: 'off',
                    role: 'data_plane',
                    cluster_mtls: 'pki',
                    cluster_cert: '/etc/secrets/kong-dp-cluster-issuer-secret/tls.crt',
                    cluster_cert_key: '/etc/secrets/kong-dp-cluster-issuer-secret/tls.key',
                    cluster_ca_cert: '/etc/secrets/kong-dp-cluster-issuer-secret/ca.crt',
                    cluster_server_name: 'kong-cp.internal',
                    cluster_control_plane: `${props.endpoints.clusterDns}:443`,
                    cluster_telemetry_endpoint: `${props.endpoints.telemetryDns}:443`,
                },
                proxy: {
                    http: {
                        enabled: false,
                    },
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'service.beta.kubernetes.io/aws-load-balancer-additional-resource-tags': 'Type=proxy',
                        'external-dns.alpha.kubernetes.io/hostname': props.endpoints.proxyDns,
                        'service.beta.kubernetes.io/aws-load-balancer-name': `${props.cluster.clusterName}-eks-proxy`,
                        'service.beta.kubernetes.io/aws-load-balancer-cross-zone-load-balancing-enabled': true,
                    },
                },
                deployment: {
                    serviceAccount: {
                        create: false,
                        name: kong_dp_service_account.serviceAccountName,
                    },
                    userDefinedVolumes: [
                        {
                            name: props.license_secret_name,
                            csi: {
                                driver: 'secrets-store.csi.k8s.io',
                                readOnly: true,
                                volumeAttributes: {
                                    secretProviderClass: props.license_secret_name,
                                },
                            },
                        },
                    ],
                    userDefinedVolumeMounts: [
                        {
                            name: props.license_secret_name,
                            mountPath: '/mnt/secrets',
                            readOnly: true,
                        },
                    ],
                },
                enterprise: {
                    enabled: true,
                    license_secret: props.license_secret_name,
                },
                manager: {
                    enabled: false,
                },
                portal: {
                    enabled: false,
                },
                portalapi: {
                    enabled: false,
                },
                secretVolumes: [
                    'kong-dp-cluster-issuer-secret',
                ],
            },
        });
        deploy_kong_dp_helm.node.addDependency(secrets_crd_mount);
    }
}
exports.KongEksDataPlane = KongEksDataPlane;
//# sourceMappingURL=data:application/json;base64,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