#!/usr/bin/env python3
"""Replay results of "run_evaluate_policy_all_levels.py" and compute reward.

Reads the files generated by "run_evaluate_policy_all_levels.py" and replays
the action logs to verify the result and to compute the total reward over all
runs.
"""

# IMPORTANT:  DO NOT MODIFY THIS FILE!
# Submissions will be evaluate on our side with our own version of this script.
# To make sure that your code is compatible with our evaluation script, make
# sure it runs with this one without any modifications.

import argparse
import os
import pickle
import sys
import typing

import numpy as np

from . import replay_action_log


class TestSample(typing.NamedTuple):
    difficulty: int
    iteration: int
    init_pose_json: str
    goal_pose_json: str
    logfile: str


def main(input_directory: str):
    try:
        if not os.path.isdir(input_directory):
            print(
                "'{}' does not exist or is not a directory.".format(
                    input_directory
                )
            )
            sys.exit(1)

        levels = (1, 2, 3, 4)

        # load samples
        sample_file = os.path.join(input_directory, "test_data.p")
        with open(sample_file, "rb") as fb:
            test_data = pickle.load(fb)

        # run "replay_action_log.py" for each sample
        level_rewards: dict = {level: [] for level in levels}
        for sample in test_data:
            print(
                "Replay level {} sample {}".format(
                    sample.difficulty, sample.iteration
                )
            )
            reward = replay_action_log.replay_action_log(
                sample.logfile,
                sample.difficulty,
                sample.init_pose_json,
                sample.goal_pose_json,
            )
            level_rewards[sample.difficulty].append(reward)

        # report
        print("\n=======================================================\n")

        report = ""
        total_reward = 0
        for level, rewards in level_rewards.items():
            rewards = np.asarray(rewards)
            mean = rewards.mean()
            report += "Level {} mean reward:\t{:.3f},\tstd: {:.3f}\n".format(
                level, mean, rewards.std()
            )
            total_reward += level * mean

        report += "-------------------------------------------------------\n"
        report += "Total Weighted Reward: {:.3f}\n".format(total_reward)

        print(report)

        # save report to file
        report_file = os.path.join(input_directory, "reward.txt")
        with open(report_file, "w") as f:
            f.write(report)

    except Exception as e:
        print(e, file=sys.stderr)
        sys.exit(1)


def add_arguments(parser):
    parser.add_argument(
        "input_directory",
        type=str,
        help="Directory containing the generated log files.",
    )


if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description=__doc__,
        formatter_class=argparse.RawDescriptionHelpFormatter,
    )
    add_arguments(parser)
    args = parser.parse_args()

    main(args.input_directory)
