import numpy as np
import logging
from os.path import join

from pyspextool import config as setup
from pyspextool.extract import config as extract
from pyspextool.extract.profiles import combine_aperturesigns
from pyspextool.io.check import check_parameter, check_qakeywords
from pyspextool.io.files import extract_filestring
from pyspextool.plot.plot_profiles import plot_profiles
from pyspextool.pyspextoolerror import pySpextoolError


def select_orders(include:int | str | list=None,
                  exclude:int | str | list=None,
                  verbose:bool=None,
                  qa_show:bool=None,
                  qa_showscale:float | int=None,
                  qa_showblock:bool=None,
                  qa_write:bool=None):

    """
    To set which orders are to be traced and extracted

    Parameters
    ----------
    include : int, list, str, optional
        If the type is int, the single order to include.
        If the type is list, a list of integer orders to include.
        If the type is str, a str giving the orders, e.g. '1-3,4,5'.

    exclude : int, list, str, optional
        If the type is int, the single order to exclude.  
        If the type is list, a list of integer orders to include.
        If the type is str, a str giving the orders, e.g. '1-3,4,5'.

    qa_show : {None, True, False}, optional
        Set to True/False to override config.state['qa_show'] in the
        pyspextool config file.  If set to True, quality assurance
        plots will be interactively generated.

    qa_showsize : tuple, default=(6,6)
        A (2,) tuple giving the plot size that is passed to matplotlib as,
        pl.figure(figsize=(qa_showsize)) for the interactive plot.

    qa_write : {None, True, False}, optional
        Set to True/False to override config.state['qa_write'] in the
        pyspextool config file.  If set to True, quality assurance
        plots will be written to disk.

    verbose : {None, True, False}, optional
        Set to True/False to override config.state['verbose'] in the
        pyspextool config file.

    Returns
    -------
    None
    Updates the config.extract['xsdoorders'] or config.extract['psdoorders'] 
    variable, updates the QA plot generated by locate_aperture_positions, 
    and reruns the trace_apertures() function if 
    config.extract['exttype'] = 'xs'.

    Examples
    --------
    select_orders(include=3)
    select_orders(include=[3, 4])
    select_orders(include='3-4,7-9')
    select_orders(exclude=3)
    select_orders(exclude=[3, 4])
    select_orders(exclude='3-4,7-9')

    """

    #
    # Check if we can proceed
    #

    if extract.state['apertures_done'] is False:

        message = " Previous steps not complete.  Please run locate_apertures."
        raise pySpextoolError(message)        

    #
    # Check parameters and QA keywords
    #

    check_parameter('select_orders', 'include', include,
                    ['NoneType', 'int', 'list', 'str'])

    check_parameter('select_orders', 'exclude', exclude,
                    ['NoneType', 'int', 'list', 'str'])

    check_parameter('select_orders', 'verbose', verbose, ['NoneType', 'bool'])

    check_parameter('select_orders', 'qa_write', qa_write, ['NoneType', 'bool'])

    check_parameter('select_orders', 'qa_show', qa_show, ['NoneType', 'bool'])

    check_parameter('select_orders', 'qa_showscale', qa_showscale,
                    ['int', 'float','NoneType'])

    check_parameter('select_orders', 'qa_showblock', qa_showblock,
                    ['NoneType', 'bool'])
    
    qa = check_qakeywords(verbose=verbose,
                          show=qa_show,
                          showscale=qa_showscale,
                          showblock=qa_showblock,
                          write=qa_write)
    
    #
    # Ensure only one optional argument is passed
    #

    if include is not None and exclude is not None:

        message = 'Cannot use keywords `include` and `exclude` at the '+ \
            'same time.'
        raise pySpextoolError(message)

    #
    # Update command line if requested.
    #

    logging.info(' Selecting orders for extraction.')
    
    #
    # Do the checks
    #

    if include is None and exclude is None:

        doorders = np.full(extract.state['norders'], 1, dtype=int)
            
    if include is not None:

        if isinstance(include, int) is True:
            include = np.array(include, dtype=int)

        if isinstance(include, list) is True:
            include = np.array(include, dtype=int)

        if isinstance(include, str) is True:
            include = np.array(extract_filestring(include, 'index'), dtype=int)

        # Find the overlap 

        doorders = np.isin(extract.state['orders'], include)

        # Test to make sure they are orders you are allowed work with

        if np.sum(doorders) != np.size(include):

            message = ' A requested order does not exist.'
            raise pySpextoolError(message)

    if exclude is not None:

        if isinstance(exclude, int) is True:
            exclude = np.array(exclude, dtype=int)

        if isinstance(exclude, list) is True:
            exclude = np.array(exclude, dtype=int)

        if isinstance(exclude, str):
            exclude = np.array(extract_filestring(exclude, 'index'), dtype=int)

        # Find the overlap 

        doorders = ~np.isin(extract.state['orders'], exclude)

        # Test to make sure they are orders you are allowed work with

        if np.sum(~doorders) != np.size(exclude):

            message = ' A requested order does not exist.'
            raise pySpextoolError(message)

    #
    # Set the doorders variable
    #

    extract.state['doorders'] = doorders

    # 
    # Update the aperture signs
    #

    z = doorders == 1

    results = combine_aperturesigns(extract.state['aperture_signs'][z,:])
    average_aperturesigns = results[0]
    label_aperturesigns = results[1]

    extract.state['average_aperturesigns'] = average_aperturesigns

    message = ' Aperture signs are (' + label_aperturesigns + ').'
    logging.info(message)

    #
    # Do the QA plotting
    #
    
    if qa['show'] is True:
               
        plot_profiles(extract.state['profiles'],
                      extract.state['slith_arc'],
                      doorders,
                      aperture_positions=extract.state['aperture_positions'],
                      plot_number=setup.plots['profiles'],
                      profilestack_max=setup.plots['profilestack_max'],
                      profile_size=setup.plots['profile_size'],
                      font_size=setup.plots['font_size'],
                      showscale=qa['showscale'],
                      showblock=qa['showblock'])

    if qa['write'] is True:

        filename = extract.state['qafilename'] + '_extractedprofiles' + \
            setup.state['qa_extension']
        fullpath = join(setup.state['qa_path'],filename)

        plot_profiles(extract.state['profiles'],
                      extract.state['slith_arc'],
                      doorders,
                      aperture_positions=extract.state['aperture_positions'],
                      profilestack_max=setup.plots['profilestack_max'],
                      profile_size=setup.plots['profile_size'],
                      font_size=setup.plots['font_size'],
                      output_fullpath=fullpath)

        

    #
    # Set the done variable
    #

    extract.state['select_done'] = True
