`timescale 1 ns / 1 ps

module axi_stub #(
    parameter delay = 16,
    parameter max_outstanding_requests = 8,
    parameter AXI_ID_WIDTH = 1,
    parameter AXI_DATA_WIDTH = 32,
    parameter AXI_ADDR_WIDTH = 32
) (
    input wire aclk,
    input wire aresetn,
    input wire [AXI_ID_WIDTH-1:0] s_axi_awid,
    input wire [AXI_ADDR_WIDTH-1:0] s_axi_awaddr,
    input wire [7:0] s_axi_awlen,
    input wire [2:0] s_axi_awsize,
    input wire [1:0] s_axi_awburst,
    input wire s_axi_awlock,
    input wire [3:0] s_axi_awcache,
    input wire [2:0] s_axi_awprot,
    input wire [3:0] s_axi_awqos,
    input wire [3:0] s_axi_awregion,
    input wire s_axi_awvalid,
    output wire s_axi_awready,
    input wire [AXI_DATA_WIDTH-1:0] s_axi_wdata,
    input wire [(AXI_DATA_WIDTH/8)-1:0] s_axi_wstrb,
    input wire s_axi_wlast,
    input wire s_axi_wvalid,
    output wire s_axi_wready,
    output wire [AXI_ID_WIDTH-1:0] s_axi_bid,
    output wire [1:0] s_axi_bresp,
    output wire s_axi_bvalid,
    input wire s_axi_bready,
    input wire [AXI_ID_WIDTH-1:0] s_axi_arid,
    input wire [AXI_ADDR_WIDTH-1:0] s_axi_araddr,
    input wire [7:0] s_axi_arlen,
    input wire [2:0] s_axi_arsize,
    input wire [1:0] s_axi_arburst,
    input wire s_axi_arlock,
    input wire [3:0] s_axi_arcache,
    input wire [2:0] s_axi_arprot,
    input wire [3:0] s_axi_arqos,
    input wire [3:0] s_axi_arregion,
    input wire s_axi_arvalid,
    output wire s_axi_arready,
    output wire [AXI_ID_WIDTH-1:0] s_axi_rid,
    output wire [AXI_DATA_WIDTH-1:0] s_axi_rdata,
    output wire [1:0] s_axi_rresp,
    output wire s_axi_rlast,
    output wire s_axi_rvalid,
    input wire s_axi_rready
);

    assign s_axi_bresp = s_axi_bvalid ? 2'd0 : 2'dX;
    assign s_axi_rdata = s_axi_rvalid ? {AXI_DATA_WIDTH{1'b0}} : {AXI_DATA_WIDTH{1'bX}};
    assign s_axi_rresp = s_axi_rvalid ? 2'd0 : 2'dX;
    
    localparam READ_FIFO_WIDTH = AXI_ID_WIDTH+48;
    localparam WRITE_FIFO_WIDTH = AXI_ID_WIDTH+40;
    
    wire read_fifo_full;
    wire [READ_FIFO_WIDTH-1:0] read_fifo_din;
    wire read_fifo_wr_en;
    
    wire read_fifo_empty;
    reg [READ_FIFO_WIDTH-1:0] read_fifo_dout;
    wire read_fifo_rd_en;
        
    wire write_fifo_full;
    wire [WRITE_FIFO_WIDTH-1:0] write_fifo_din;
    wire write_fifo_wr_en;
    
    wire write_fifo_empty;
    reg [WRITE_FIFO_WIDTH-1:0] write_fifo_dout;
    wire write_fifo_rd_en;
        
    reg [39:0] count;
    reg [7:0] issue_read_count;
    
    localparam ISSUE_READ_IDLE = 0;
    localparam ISSUE_READ_ISSUE = 1;
    
    localparam WRITE_IDLE = 0;
    localparam WRITE_READ_DATA = 1;
    
    localparam WRITE_RESPONSE_IDLE = 0;
    localparam WRITE_RESPONSE_ISSUE = 1;
    
    reg[0:0] issue_read_state;
    reg[0:0] write_state;
    reg[0:0] write_response_state;

    assign s_axi_arready = !read_fifo_full;
    assign read_fifo_wr_en = s_axi_arvalid && s_axi_arready;
    assign read_fifo_din[39:0] = count;
    assign read_fifo_din[47:40] = s_axi_arlen;
    assign read_fifo_din[48+AXI_ID_WIDTH-1:48] = s_axi_arid;
    assign s_axi_rid = s_axi_rvalid ? read_fifo_dout[48+AXI_ID_WIDTH-1:48] : {AXI_ID_WIDTH{1'bX}};

    assign s_axi_rvalid = issue_read_state == ISSUE_READ_ISSUE;
    assign s_axi_rlast = s_axi_rvalid ? issue_read_count == read_fifo_dout[47:40] : 1'bX;
    assign read_fifo_rd_en = s_axi_rready && issue_read_state == ISSUE_READ_ISSUE && issue_read_count == read_fifo_dout[47:40];

    always @(posedge aclk) begin
        if (!aresetn) begin
            issue_read_state <= ISSUE_READ_IDLE;
            count <= 0;
        end else begin
            count <= count + 1;
            case (issue_read_state)

                ISSUE_READ_IDLE: begin
                    issue_read_count <= 0;
                    if (!read_fifo_empty && read_fifo_dout[39:0]+delay <= count) begin
                        issue_read_state <= ISSUE_READ_ISSUE;
                    end
                end

                ISSUE_READ_ISSUE: begin
                    if (s_axi_rready) begin
                        issue_read_count <= issue_read_count+1;
                        if (issue_read_count == read_fifo_dout[47:40]) begin
                            issue_read_state <= ISSUE_READ_IDLE;
                        end
                    end
                end

            endcase
        end
    end

    assign s_axi_awready = write_state == WRITE_IDLE && !write_fifo_full;
    assign s_axi_wready = write_state == WRITE_READ_DATA;
    assign write_fifo_wr_en = s_axi_wvalid && s_axi_wready && s_axi_wlast;
    assign write_fifo_din[39:0] = count;
    assign write_fifo_din[40+AXI_ID_WIDTH-1:40] = s_axi_awid;
    assign s_axi_bid = s_axi_bvalid ? write_fifo_dout[40+AXI_ID_WIDTH-1:40] : {AXI_ID_WIDTH{1'bX}};

    always @(posedge aclk) begin
        if (!aresetn) begin
            write_state <= WRITE_IDLE;
        end else begin
            case (write_state)

                WRITE_IDLE: begin
                    if (s_axi_awvalid && !write_fifo_full) begin
                        write_state <= WRITE_READ_DATA;
                    end
                end

                WRITE_READ_DATA: begin
                    if (s_axi_wvalid && s_axi_wlast) begin
                        write_state <= WRITE_IDLE;
                    end
                end

            endcase
        end
    end

    assign write_fifo_rd_en = write_response_state == WRITE_RESPONSE_ISSUE && s_axi_bready;
    assign s_axi_bvalid = write_response_state == WRITE_RESPONSE_ISSUE;

    always @(posedge aclk) begin
        if (!aresetn) begin
            write_response_state <= WRITE_RESPONSE_IDLE;
        end else begin
            case(write_response_state)

                WRITE_RESPONSE_IDLE: begin
                    if (!write_fifo_empty && write_fifo_dout[39:0]+delay <= count) begin
                        write_response_state <= WRITE_RESPONSE_ISSUE;
                    end
                end

                WRITE_RESPONSE_ISSUE: begin
                    if (s_axi_bready) begin
                        write_response_state <= WRITE_RESPONSE_IDLE;
                    end
                end

            endcase
        end
    end
    
    localparam LEN = max_outstanding_requests;
    localparam ADDR_BITS = $clog2(LEN);
    
    /// READ FIFO BEGIN
    reg [READ_FIFO_WIDTH-1:0] read_fifo_mem[0:LEN-1];

    reg [ADDR_BITS:0] read_fifo_rd_addr;
    reg [ADDR_BITS:0] read_fifo_wr_addr;

    assign read_fifo_empty = read_fifo_rd_addr == read_fifo_wr_addr;
    assign read_fifo_full = read_fifo_rd_addr[ADDR_BITS] != read_fifo_wr_addr[ADDR_BITS] && read_fifo_rd_addr[ADDR_BITS-1:0] == read_fifo_wr_addr[ADDR_BITS-1:0];

    always @(posedge aclk) begin
        read_fifo_dout <= read_fifo_mem[read_fifo_rd_addr[ADDR_BITS-1:0]];

        if (read_fifo_wr_en) begin
            read_fifo_mem[read_fifo_wr_addr[ADDR_BITS-1:0]] <= read_fifo_din;
            if (read_fifo_wr_addr[ADDR_BITS-1:0] == LEN-1) begin
                read_fifo_wr_addr[ADDR_BITS-1:0] <= 0;
                read_fifo_wr_addr[ADDR_BITS] <= !read_fifo_wr_addr[ADDR_BITS];
            end else begin
                read_fifo_wr_addr[ADDR_BITS-1:0] <= read_fifo_wr_addr[ADDR_BITS-1:0]+1;
            end
            if (read_fifo_empty) begin
                read_fifo_dout <= read_fifo_din;
            end
        end
        if (read_fifo_rd_en) begin
            if (read_fifo_rd_addr[ADDR_BITS-1:0] == LEN-1) begin
                read_fifo_rd_addr[ADDR_BITS-1:0] <= 0;
                read_fifo_rd_addr[ADDR_BITS] <= !read_fifo_rd_addr[ADDR_BITS];
                read_fifo_dout <= read_fifo_mem[0];
            end else begin
                read_fifo_rd_addr[ADDR_BITS-1:0] <= read_fifo_rd_addr[ADDR_BITS-1:0]+1;
                read_fifo_dout <= read_fifo_mem[read_fifo_rd_addr[ADDR_BITS-1:0]+1];
            end
        end

        if (!aresetn) begin
            read_fifo_rd_addr <= 0;
            read_fifo_wr_addr <= 0;
        end
    end
    /// WRITE FIFO BEGIN
    reg [WRITE_FIFO_WIDTH-1:0] write_fifo_mem[0:LEN-1];

    reg [ADDR_BITS:0] write_fifo_rd_addr;
    reg [ADDR_BITS:0] write_fifo_wr_addr;

    assign write_fifo_empty = write_fifo_rd_addr == write_fifo_wr_addr;
    assign write_fifo_full = write_fifo_rd_addr[ADDR_BITS] != write_fifo_wr_addr[ADDR_BITS] && write_fifo_rd_addr[ADDR_BITS-1:0] == write_fifo_wr_addr[ADDR_BITS-1:0];

    always @(posedge aclk) begin
        write_fifo_dout <= write_fifo_mem[write_fifo_rd_addr[ADDR_BITS-1:0]];

        if (write_fifo_wr_en) begin
            write_fifo_mem[write_fifo_wr_addr[ADDR_BITS-1:0]] <= write_fifo_din;
            if (write_fifo_wr_addr[ADDR_BITS-1:0] == LEN-1) begin
                write_fifo_wr_addr[ADDR_BITS-1:0] <= 0;
                write_fifo_wr_addr[ADDR_BITS] <= !write_fifo_wr_addr[ADDR_BITS];
            end else begin
                write_fifo_wr_addr[ADDR_BITS-1:0] <= write_fifo_wr_addr[ADDR_BITS-1:0]+1;
            end
            if (write_fifo_empty) begin
                write_fifo_dout <= write_fifo_din;
            end
        end
        if (write_fifo_rd_en) begin
            if (write_fifo_rd_addr[ADDR_BITS-1:0] == LEN-1) begin
                write_fifo_rd_addr[ADDR_BITS-1:0] <= 0;
                write_fifo_rd_addr[ADDR_BITS] <= !write_fifo_rd_addr[ADDR_BITS];
                write_fifo_dout <= write_fifo_mem[0];
            end else begin
                write_fifo_rd_addr[ADDR_BITS-1:0] <= write_fifo_rd_addr[ADDR_BITS-1:0]+1;
                write_fifo_dout <= write_fifo_mem[write_fifo_rd_addr[ADDR_BITS-1:0]+1];
            end
        end

        if (!aresetn) begin
            write_fifo_rd_addr <= 0;
            write_fifo_wr_addr <= 0;
        end
    end

endmodule