"""
# Why `$λ`?

This package provides an alternative to [`argparse`](https://docs.python.org/3/library/argparse.html)
based on functional first principles. This means that this package can handle many kinds of argument-parsing patterns
that are either very awkward, difficult, or impossible with `argparse`.

### Versatile
`$λ` provides high-level functionality equivalent to other parsers. But unlike other parsers,
it permits low-level customization to handle arbitrarily complex parsing patterns. As we'll see in the tutorial,
there are many parsing patterns that `$λ` can handle which are not possible with other parsing libraries.

### Type-safe
`$λ` uses type annotations as much as Python allows. Types are checked using [`MyPy`](
https://mypy.readthedocs.io/en/stable/index.html#) and exported with the package so that users can also benefit from
the type system. Furthermore, with rare exceptions, `$λ` avoids mutations and side-effects and preserves [referential
transparency](https://en.wikipedia.org/wiki/Referential_transparency). This makes it easier for the type-checker _and
for the user_ to reason about the code.

### Concise
As we'll demonstrate in the tutorial, `$λ` provides three main
syntactic shortcuts for cutting down boilerplate:

- operators like `>>`, `|`, and `+` (and `>=` if you want to get fancy)
- the `command` decorator and the `CommandTree` object for building tree-shaped parsers
- the `Args` syntax built on top of python `dataclasses`.


As a rule, `$λ` avoids reproducing python functionality and focuses on the main job of
an argument parser: parsing. Arguably, `$λ` is way more expressive than any reasonable
person would ever need... but even if it's not the parser that we need, it's the parser we deserve.

# Installation
You guessed it:
```
pip install dollar-lambda
```

# Highlights
`$λ` comes with syntactic sugar that came make building parsers completely boilerplate-free.
However, with more concise syntax comes less flexibility. As we go we will demonstrate the
building blocks that lie behind the syntactic sugar and which enable users to build much more
logically complex parsers.

## The `command` decorator
This syntax is best for simple parsers that take a set of unordered arguments:

>>> @command()
... def main(x: int, y: int, verbose: bool = False, quiet: bool = False):
...     return dict(x=x, y=y, verbose=verbose, quiet=quiet)

Here is the help text generated by this parser:

>>> main("-h")
usage:
    -x X
    -y Y
    --verbose
    --quiet

And here it is in action:

>>> main("-x", "1", "-y", "2", "--verbose")
{'x': 1, 'y': 2, 'verbose': True, 'quiet': False}

`command` takes arguments that allow you to supply
help strings and custom types:

>>> @command(types=dict(x=lambda x: int(x) + 1, help=dict(x="A number that gets incremented.")))
... def main(x: int, y: int, verbose: bool = False, quiet: bool = False):
...     return dict(x=x, y=y, verbose=verbose, quiet=quiet)
...

>>> main("-h")
usage:
    -x X
    -y Y
    --verbose
    --quiet

>>> main("-x", "1", "-y", "2", "--verbose")
{'x': 2, 'y': 2, 'verbose': True, 'quiet': False}

### Equivalent `$λ` syntax

Under the syntactic sugar, this is how `$λ` expresses the first parser:

>>> p = nonpositional(
...     option("x", type=lambda x: int(x) + 1, help="A number that gets incremented."),
...     option("y", type=int),
...     flag("verbose", default=False),
...     flag("quiet", default=False),
... ) >> done()

>>> p.parse_args("-h")
usage:
    -x X
    -y Y
    --verbose
    --quiet
x: A number that gets incremented.

>>> p.parse_args("-x", "1", "-y", "2", "--verbose")
{'x': 2, 'y': 2, 'verbose': True, 'quiet': False}

As you can see, `flag` denotes boolean-valued command-line arguments and `option` denotes command-line
arguments that take explicit parameters. `nonpositional` allows these parsers to be applied in any order
-- i.e. it takes command-line arguments nonpositionally. `>>` allows parsers to be sequenced and `done`
 succeeds only at the end of input. For more details on all these functions, see the
  [Parser Combinators section](#parser-combinators) of the [tutorial](#tutorial).

This logic will cover many common use cases. But let's consider a more complex example:
What if you want `--verbose` and `--quiet` to be mutually exclusive?

## `|` for alternatives
In order to force a choice between two parsers, you can use `|`:

>>> p = nonpositional(
...     option("x", type=lambda x: int(x) + 1, help="A number that gets incremented."),
...     option("y", type=int),
...     flag("verbose") | flag("quiet")
... ) >> done()

>>> p.parse_args("-h")
usage: -x X -y Y [--verbose | --quiet]
x: A number that gets incremented.

>>> p.parse_args("-x", "1", "-y", "2", "--verbose")
{'x': 2, 'y': 2, 'verbose': True}

This will fail if both `--verbose` and `--quiet` are given:
>>> p.parse_args("-x", "1", "-y", "2", "--verbose", "--quiet")
usage: -x X -y Y [--verbose | --quiet]
x: A number that gets incremented.
Unrecognized argument: --quiet

It will also fail if neither is given:
>>> p.parse_args("-x", "1", "-y", "2")
usage: -x X -y Y [--verbose | --quiet]
x: A number that gets incremented.
The following arguments are required: --verbose

## `CommandTree` for dynamic dispatch
`$λ` offers a special syntax for a common use case with alternative
command line inputs.  Often, we want a program to do one thing when a user
enters one set of command line arguments and another thing for another set.
Returning to our earlier example, what if we wanted to execute one
function if the user gave the `--verbose` flag and another if they gave `--quiet` flag?

Meet `CommandTree`:
>>> tree = CommandTree()
...
>>> @tree.command()
... def base_function(x: int, y: int):
...     raise RuntimeError("Does not execute because children are required.")
...
>>> @base_function.command()
... def verbose_function(x: int, y: int, verbose: bool):
...     print(dict(x=x, y=y, verbose=verbose))
...
>>> @base_function.command()
... def quiet_function(x: int, y: int, quiet: bool):
...     print(dict(x=x, y=y, quiet=quiet))

Let's see how this parser handles different inputs.
If we provide the `--verbose` flag, `$λ` automatically invokes
the `verbose_function` with the parsed arguments:

>>> tree("-x", "1", "-y", "2", "--verbose")
{'x': 1, 'y': 2, 'verbose': True}

If we provide the `--quiet` flag, `$λ` invokes the `quiet_function`:

>>> tree("-x", "1", "-y", "2", "--quiet")  # execute quiet_function
{'x': 1, 'y': 2, 'quiet': True}

With this configuration, the parser will fail if neither
`--verbose` or `--quiet` is given:

>>> tree("-x", "1", "-y", "2")  # fails
usage: -x X -y Y [--verbose | --quiet]
The following arguments are required: --verbose

However, there are many other ways to use `CommandTree`,
including some that make use of the `base_function`.
To learn more, we recommend the [`CommandTree` tutorial](#commandtree-tutorial).

# Tutorial

We've already seen many of the concepts that power `$λ` in the
[Highlights](#highlights) section. This tutorial will address
these concepts one at a time and expose the reader to some
nuances of usage.

## An example from `argparse`

Many of you are already familiar with `argparse`.
You may even recognize this example from the `argparse` docs:

```
import argparse
parser = argparse.ArgumentParser(description="calculate X to the power of Y")
group = parser.add_mutually_exclusive_group()
group.add_argument("-v", "--verbose", action="store_true")
group.add_argument("-q", "--quiet", action="store_true")
parser.add_argument("x", type=int, help="the base")
parser.add_argument("y", type=int, help="the exponent")
args = parser.parse_args()
```

Here is the exact equivalent in this package:

>>> p = nonpositional(
...     (flag("verbose") | flag("quiet") | empty()),
...     option("x", type=int, help="the base"),
...     option("y", type=int, help="the exponent"),
... ) >> done()
...
>>> def main(x, y, verbose=False, quiet=False):
...     return dict(x=x, y=y, verbose=verbose, quiet=quiet)

This is similar to what we saw in the
[earlier section on alternatives](##for-alternatives),
 except that now we use `| empty()` to allow both
`--verbose` and `--quiet` to be omitted:

>>> main(**p.parse_args("-x", "1", "-y", "2"))
{'x': 1, 'y': 2, 'verbose': False, 'quiet': False}

Let's walk through this step by step.

## High-Level Parsers
So far we've seen a few different parser constructors.
`flag` binds a boolean value to a variable whereas `option` binds an arbitrary value to a variable.
`empty` and `done` do not bind any values to variables, but `empty` always succeeds whereas `done` only
succeeds on the end of input.

### `flag`
>>> p = flag("verbose")
>>> p.parse_args("--verbose")
{'verbose': True}

By default `flag` fails when it does not receive expected input:
>>> p.parse_args()
usage: --verbose
The following arguments are required: --verbose

Alternately, you can set a default value:
>>> flag("verbose", default=False).parse_args()
{'verbose': False}

### `option`
`option` is similar but takes an argument:
By default, `option` expects a single `-` for single-character variable names (as in `-x`),
as opposed to `--` for longer names (as in `--xenophon`):

>>> option("x").parse_args("-x", "1")
{'x': '1'}
>>> option("xenophon").parse_args("--xenophon", "1")
{'xenophon': '1'}

Use the `type` argument to convert the input to a different type:
>>> option("x", type=int).parse_args("-x", "1")  # converts "1" to an int
{'x': 1}

### `empty`
As we've said, `empty` always succeeds:

>>> empty().parse_args("any", "arguments")
{}

For a more type-safe version, you can use `Parser.empty`:

>>> p = Parser[int].empty()

`empty` is useful when you want to make an argument optional --
the definition of `Parser.optional` is:

```
def optional(self):
    return self | self.empty()
```

### `done`
Without `done` the parser will not complain about leftover (unparsed) input:

>>> flag("verbose").parse_args("--verbose", "--quiet")
{'verbose': True}

`--quiet` is not parsed here but this does not cause the parser to fail.
If we want to prevent leftover inputs, we can use `done`:

>>> (flag("verbose") >> done()).parse_args("--verbose", "--quiet")
usage: --verbose
Unrecognized argument: --quiet

`done` is usually necessary to get `nonpositional` to behave in the way that you expect,
but more on that later.

## Parser Combinators
Parser combinators are functions that combine multiple parsers into new, more complex parsers.
Our example uses three such functions: `nonpositional`, `|` or `Parser.__or__`,
and `>>` or `Parser.__rshift__`.

### `Parser.__or__`

The `|` operator is used for alternatives. Specifically, it will try the first parser,
and if that fails, try the second:

>>> p = flag("verbose") | flag("quiet")
>>> p.parse_args("--quiet") # flag("verbose") fails
{'quiet': True}
>>> p.parse_args("--verbose") # flag("verbose") succeeds
{'verbose': True}

By default one of the two flags would be required to prevent failure:
>>> p.parse_args() # neither flag is provided so this fails
usage: [--verbose | --quiet]
The following arguments are required: --verbose

We can permit the omission of both flags
by using `empty`, as we saw earlier, or we can supply a default value:

>>> (flag("verbose") | flag("quiet") | empty()).parse_args() # flags fail, but empty() succeeds
{}
>>> (flag("verbose") | flag("quiet", default=False)).parse_args() # flag("verbose") fails but flag("quiet", default=False) succeeds
{'quiet': False}

This is just sugar for

>>> (flag("verbose") | flag("quiet") | defaults(quiet=False)).parse_args() # flag("verbose") fails but flag("quiet", default=False) succeeds
{'quiet': False}

### `Parser.__rshift__`

The `>>` operator is used for sequential composition. It applies the first parser and then
hands the output of the first parser to the second parser. If either parser fails, the composition fails:

>>> p = flag("verbose") >> done()
>>> p.parse_args("--verbose")
{'verbose': True}
>>> p.parse_args("--something-else")  # first parser will fail
usage: --verbose
Expected '--verbose'. Got '--something-else'
>>> p.parse_args("--verbose", "--something-else")  # second parser will fail
usage: --verbose
Unrecognized argument: --something-else

### `nonpositional` and `Parser.__add__`
`nonpositional` takes a sequence of parsers as arguments and attempts all permutations of them,
returning the first permutations that is successful:

>>> p = nonpositional(flag("verbose"), flag("quiet"))
>>> p.parse_args("--verbose", "--quiet")
{'verbose': True, 'quiet': True}
>>> p.parse_args("--quiet", "--verbose")  # reverse order also works
{'quiet': True, 'verbose': True}

For just two parsers you can use `+`, or `Parser.__add__`, instead of `nonpositional`:
>>> p = flag("verbose") + flag("quiet")
>>> p.parse_args("--verbose", "--quiet")
{'verbose': True, 'quiet': True}
>>> p.parse_args("--quiet", "--verbose")  # reverse order also works
{'quiet': True, 'verbose': True}

This will not cover all permutations for more than two parsers:
>>> p = flag("verbose") + flag("quiet") + option("x")
>>> p.parse_args("--verbose", "-x", "1", "--quiet")
usage: --verbose --quiet -x X
Expected '--quiet'. Got '-x'

To see why note the implicit parentheses:
>>> p = (flag("verbose") + flag("quiet")) + option("x")

In order to cover the case where `-x` comes between `--verbose` and `--quiet`,
use `nonpositional`:
>>> p = nonpositional(flag("verbose"), flag("quiet"), option("x"))
>>> p.parse_args("--verbose", "-x", "1", "--quiet")  # works
{'verbose': True, 'x': '1', 'quiet': True}

If alternatives or defaults appear among the arguments to `nonpositional`, you will probably want
to add `>>` followed by `done` (or another parser) after `nonpositional`. Otherwise,
the parser will not behave as expected:

>>> p = nonpositional(flag("verbose", default=False), flag("quiet"))
>>> p.parse_args("--quiet", "--verbose")  # you expect this to set verbose to True, but it doesn't
{'verbose': False, 'quiet': True}

Why is happening? There are two permutions:

- `flag("verbose", default=False) >> flag("quiet")` and
- `flag("quiet") >> flag("verbose", default=False)`

In our example, both permutations are actually succeeding. This first succeeds by falling
back to the default, and leaving the last word of the input, `--verbose`, unparsed.
Either interpretation is valid, and `nonpositional` returns one arbitrarily -- just not the one we expected.

Now let's add `>> done()` to the end:
>>> p = nonpositional(flag("verbose", default=False), flag("quiet")) >> done()

This ensures that the first permutation will fail because the leftover `--verbose` input will
cause the `done` parser to fail:
>>> p.parse_args("--quiet", "--verbose")
{'quiet': True, 'verbose': True}

## Putting it all together
Let's recall the original example:

>>> p = nonpositional(
...     (flag("verbose") | flag("quiet") | empty()),
...     option("x", type=int, help="the base"),
...     option("y", type=int, help="the exponent"),
... ) >> done()
...
>>> def main(x, y, verbose=False, quiet=False):
...     return dict(x=x, y=y, verbose=verbose, quiet=quiet)

As we've seen, `flag("verbose") | flag("quiet") | empty()` succeeds on either `--verbose` or `--quiet`
or neither.

`option("x", type=int)` succeeds on `-x X`, where `X` is
some integer, binding that integer to the variable `"x"`. Similarly for `option("y", type=int)`.

`nonpositional` takes the three parsers:

- `flag("verbose") | flag("quiet") | empty()`
- `option("x", type=int)`
- `option("y", type=int)`

and applies them in every order, until some order succeeds.
Finally `done()` ensures that only one of these parser permutations will succeed, preventing ambiguity.

## Variations on the example
### Variable numbers of arguments

What if there was a special argument, `verbosity`,
that only makes sense if the user chooses `--verbose`?

>>> p = (
...     nonpositional(
...         ((flag("verbose") + option("verbosity", type=int)) | flag("quiet")),
...         option("x", type=int),
...         option("y", type=int),
...     )
...     >> done()
... )

Remember that `+` or `Parser.__add__` evaluates two parsers in both orders
and stopping at the first order that succeeds. So this allows us to
supply `--verbose` and `--verbosity` in any order.

>>> p.parse_args("-x", "1", "-y", "2", "--quiet")
{'x': 1, 'y': 2, 'quiet': True}
>>> p.parse_args("-x", "1", "-y", "2", "--verbose", "--verbosity", "3")
{'x': 1, 'y': 2, 'verbose': True, 'verbosity': 3}
>>> p.parse_args("-x", "1", "-y", "2", "--verbose")
usage: [--verbose --verbosity VERBOSITY | --quiet] -x X -y Y
Expected '--verbose'. Got '-x'

This is also a case where you might want to use `CommandTree`:

>>> tree = CommandTree()
...
>>> @tree.command(help=dict(x="the base", y="the exponent"))
... def base_function(x: int, y: int):
...     raise RuntimeError("This function will not execute.")
...
>>> @base_function.command()
... def verbose_function(x: int, y: int, verbose: bool, verbosity: int):
...     print(dict(x=x, y=y, verbose=verbose, verbosity=verbosity))
...
>>> @base_function.command()
... def quiet_function(x: int, y: int, quiet: bool):
...     print(dict(x=x, y=y, quiet=quiet))
...
>>> tree("-x", "1", "-y", "2", "--verbose", "--verbosity", "3")
{'x': 1, 'y': 2, 'verbose': True, 'verbosity': 3}

### `Parser.many`

What if we want to specify verbosity by the number of times that `--verbose` appears?
For this we need `Parser.many`. Before showing how we could use `Parser.many` in this setting,
let's look at how it works.

`parser.many` takes `parser` and tries to apply it as many times as possible.
`Parser.many` is a bit like the `*` pattern, if you are familiar with regexes.
`parser.many` always succeeds:

>>> p = flag("verbose").many()
>>> p.parse_args()  # succeeds
{}
>>> p.parse_args("blah")  # still succeeds
{}
>>> p.parse_args("--verbose", "blah")  # still succeeds
{'verbose': True}
>>> p.parse_args("--verbose", "--verbose", return_dict=False)
[('verbose', True), ('verbose', True)]

As you can see, `return_dict=False` returns a list of tuples instead of a dict, so that you
can have duplicate keys.

Now returning to the original example:

>>> p = (
...     nonpositional(
...         flag("verbose").many(),
...         option("x", type=int),
...         option("y", type=int),
...     )
...     >> done()
... )
>>> args = p.parse_args("-x", "1", "-y", "2", "--verbose", "--verbose", return_dict=False)
>>> args
[('x', 1), ('y', 2), ('verbose', True), ('verbose', True)]
>>> verbosity = args.count(('verbose', True))
>>> verbosity
2

### `Parser.many1`

In the previous example, the parse will default to `verbosity=0` if no `--verbose` flags
are given.  What if we wanted users to be explicit about choosing a "quiet" setting?
In other words, what if the user actually had to provide an explicit `--quiet` flag when
no `--verbose` flags were given?

For this, we use `Parser.many1`. This method is like `Parser.many` except that it fails
when on zero successes (recall that `Parser.many` always succeeds). So if `Parser.many`
is like regex `*`, `Parser.many1` is like `+`. Take a look:

>>> p = flag("verbose").many()
>>> p.parse_args()  # succeeds
{}
>>> p = flag("verbose").many1()
>>> p.parse_args()  # fails
usage: --verbose [--verbose ...]
The following arguments are required: --verbose
>>> p.parse_args("--verbose")  # succeeds
{'verbose': True}

To compell that `--quiet` flag from our users, we can do the following:

>>> p = (
...     nonpositional(
...         ((flag("verbose").many1()) | flag("quiet")),
...         option("x", type=int),
...         option("y", type=int),
...     )
...     >> done()
... )

Now omitting both `--verbose` and `--quiet` will fail:
>>> p.parse_args("-x", "1", "-y", "2")
usage: [--verbose [--verbose ...] | --quiet] -x X -y Y
Expected '--verbose'. Got '-x'
>>> p.parse_args("--verbose", "-x", "1", "-y", "2") # this succeeds
{'verbose': True, 'x': 1, 'y': 2}
>>> p.parse_args("--quiet", "-x", "1", "-y", "2") # and this succeeds
{'quiet': True, 'x': 1, 'y': 2}

# `CommandTree` Tutorial
`CommandTree` has already shown up in the
[Highlights section](#commandtree-for-dynamic-dispatch)
and in the earlier [tutorial](#variations-on-the-example).
In this section we will give a more thorough treatment,
exposing some of the underlying logic and covering all
the variations in functionality that `CommandTree`
offers.

## `CommandTree.command`

First let's walk through the use of the `CommandTree.command` decorator, one step
at a time. First we define the object:

>>> tree = CommandTree()

Now we define at least one child function:

>>> @tree.command()
... def f1(a: int):
...     return dict(f1=dict(a=a))

At this point `tree` is just a parser that takes a single option `-a`:

>>> tree("-h")
usage: -a A

Now let's add a second child function:

>>> @tree.command()
... def f2(b: bool):
...     return dict(f2=dict(b=b))
...
>>> tree("-h")
usage: [-a A | -b]

`tree` will execute either `f1` or `f2` based on which of the parsers succeeds.
This will execute `f1`:

>>> tree("-a", "1")
{'f1': {'a': 1}}

This will execute `f2`:

>>> tree("-b")
{'f2': {'b': True}}

This fails:

>>> tree()
usage: [-a A | -b]
The following arguments are required: -a

Often in cases where there are alternative sets of argument like this,
there is also a set of shared arguments. It would be cumbersome to have to
repeat these for both child functions. Instead we can define a parent function
as follows:

>>> tree = CommandTree()
...
>>> @tree.command()
... def f1(a: int):
...     raise RuntimeError("This function will not execute.")

And a child function, `g1`:

>>> @f1.command()  # note f1, not tree
... def g1(a:int, b: bool):
...     return dict(g1=dict(b=b))

Make sure to include all the arguments of `f1` in `g1` or else
`g1` will fail when it is invoked. In its current state, `tree` sequences
 the arguments of `f1` and `g1`:

>>> tree("-h")
usage: -a A -b

As before we can define an additional child function to induce alternative
argument sets:

>>> @f1.command()  # note f1, not tree
... def g2(a: int, c: str):
...     return dict(g2=dict(c=c))

Note that our usage message shows `-a A` preceding the brackets:
>>> tree("-h")
usage: -a A [-b | -c C]

To execute `g1`, we give the `-b` flag:
>>> tree("-a", "1", "-b")
{'g1': {'b': True}}

To execute `g2`, we give the `-c` flag:
>>> tree("-a", "1", "-c", "foo")
{'g2': {'c': 'foo'}}

Also, note that `tree` can have arbitrary depth:

>>> @g1.command()  # h1 is a child of g1
... def h1(a: int, b: bool, d: float):
...    return dict(h1=dict(d=d))

>>> tree("-h")
usage: -a A [-b -d D | -c C]

## `CommandTree.subcommand`
Often we want to explicitly specify which function to execute by naming it on the command line.
This would implement functionality similar to
[`ArgumentParser.add_subparsers`](https://docs.python.org/3/library/argparse.html#argparse.ArgumentParser.add_subparsers)

For this we would use the `CommandTree.subcommand` decorator:

>>> tree = CommandTree()
...
>>> @tree.command()
... def f1(a: int):
...     raise RuntimeError("This function should not be called")
...
>>> @f1.subcommand()  # note subcommand, not command
... def g1(a:int, b: bool):
...     return dict(g1=dict(b=b))
...
>>> @f1.subcommand()  # again, subcommand, not command
... def g2(a: int, c: str):
...     return dict(g2=dict(c=c))

Now the usage message indicates that `g1` and `g2` are required arguments:
>>> tree("-h")
usage: -a A [g1 -b | g2 -c C]

Now we would select g1 as follows:
>>> tree("-a", "1", "g1", "-b")
{'g1': {'b': True}}

And g2 as follows:
>>> tree("-a", "1", "g2", "-c", "foo")
{'g2': {'c': 'foo'}}
"""


__pdoc__ = {}

from dollar_lambda.args import Args, field
from dollar_lambda.decorators import CommandTree, command
from dollar_lambda.parser import (
    Parser,
    apply,
    apply_item,
    argument,
    defaults,
    done,
    empty,
    equals,
    flag,
    item,
    nonpositional,
    option,
    sat,
    sat_item,
    type_,
    wrap_help,
)

__all__ = [
    "Parser",
    "empty",
    "apply",
    "apply_item",
    "argument",
    "done",
    "equals",
    "flag",
    "item",
    "nonpositional",
    "option",
    "sat",
    "sat_item",
    "type_",
    "Args",
    "defaults",
    "field",
    "wrap_help",
    "command",
    "CommandTree",
]


__pdoc__["Parser.__add__"] = True
__pdoc__["Parser.__or__"] = True
__pdoc__["Parser.__rshift__"] = True
__pdoc__["Parser.__ge__"] = True
