import * as cloudwatch from '@aws-cdk/aws-cloudwatch';
import * as iam from '@aws-cdk/aws-iam';
import * as logs from '@aws-cdk/aws-logs';
import { Duration, IResource, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { IChainable } from './types';
/**
 * Two types of state machines are available in AWS Step Functions: EXPRESS AND STANDARD.
 *
 * @default STANDARD
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/concepts-standard-vs-express.html
 * @stability stable
 */
export declare enum StateMachineType {
    /**
     * Express Workflows are ideal for high-volume, event processing workloads.
     *
     * @stability stable
     */
    EXPRESS = "EXPRESS",
    /**
     * Standard Workflows are ideal for long-running, durable, and auditable workflows.
     *
     * @stability stable
     */
    STANDARD = "STANDARD"
}
/**
 * Defines which category of execution history events are logged.
 *
 * @default ERROR
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/cloudwatch-log-level.html
 * @stability stable
 */
export declare enum LogLevel {
    /**
     * No Logging.
     *
     * @stability stable
     */
    OFF = "OFF",
    /**
     * Log everything.
     *
     * @stability stable
     */
    ALL = "ALL",
    /**
     * Log all errors.
     *
     * @stability stable
     */
    ERROR = "ERROR",
    /**
     * Log fatal errors.
     *
     * @stability stable
     */
    FATAL = "FATAL"
}
/**
 * Defines what execution history events are logged and where they are logged.
 *
 * @stability stable
 */
export interface LogOptions {
    /**
     * The log group where the execution history events will be logged.
     *
     * @stability stable
     */
    readonly destination: logs.ILogGroup;
    /**
     * Determines whether execution data is included in your log.
     *
     * @default true
     * @stability stable
     */
    readonly includeExecutionData?: boolean;
    /**
     * Defines which category of execution history events are logged.
     *
     * @default ERROR
     * @stability stable
     */
    readonly level?: LogLevel;
}
/**
 * Properties for defining a State Machine.
 *
 * @stability stable
 */
export interface StateMachineProps {
    /**
     * A name for the state machine.
     *
     * @default A name is automatically generated
     * @stability stable
     */
    readonly stateMachineName?: string;
    /**
     * Definition for this state machine.
     *
     * @stability stable
     */
    readonly definition: IChainable;
    /**
     * The execution role for the state machine service.
     *
     * @default A role is automatically created
     * @stability stable
     */
    readonly role?: iam.IRole;
    /**
     * Maximum run time for this state machine.
     *
     * @default No timeout
     * @stability stable
     */
    readonly timeout?: Duration;
    /**
     * Type of the state machine.
     *
     * @default StateMachineType.STANDARD
     * @stability stable
     */
    readonly stateMachineType?: StateMachineType;
    /**
     * Defines what execution history events are logged and where they are logged.
     *
     * @default No logging
     * @stability stable
     */
    readonly logs?: LogOptions;
    /**
     * Specifies whether Amazon X-Ray tracing is enabled for this state machine.
     *
     * @default false
     * @stability stable
     */
    readonly tracingEnabled?: boolean;
}
/**
 * A new or imported state machine.
 */
declare abstract class StateMachineBase extends Resource implements IStateMachine {
    /**
     * Import a state machine.
     *
     * @stability stable
     */
    static fromStateMachineArn(scope: Construct, id: string, stateMachineArn: string): IStateMachine;
    abstract readonly stateMachineArn: string;
    /**
     * The principal this state machine is running as
     */
    abstract readonly grantPrincipal: iam.IPrincipal;
    /**
     * Grant the given identity permissions to start an execution of this state machine.
     *
     * @stability stable
     */
    grantStartExecution(identity: iam.IGrantable): iam.Grant;
    /**
     * Grant the given identity permissions to read results from state machine.
     *
     * @stability stable
     */
    grantRead(identity: iam.IGrantable): iam.Grant;
    /**
     * Grant the given identity task response permissions on a state machine.
     *
     * @stability stable
     */
    grantTaskResponse(identity: iam.IGrantable): iam.Grant;
    /**
     * Grant the given identity permissions on all executions of the state machine.
     *
     * @stability stable
     */
    grantExecution(identity: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Grant the given identity custom permissions.
     *
     * @stability stable
     */
    grant(identity: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Return the given named metric for this State Machine's executions.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of executions that failed.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricFailed(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of executions that were throttled.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricThrottled(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of executions that were aborted.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricAborted(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of executions that succeeded.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricSucceeded(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of executions that timed out.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricTimedOut(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of executions that were started.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricStarted(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the interval, in milliseconds, between the time the execution starts and the time it closes.
     *
     * @default - average over 5 minutes
     * @stability stable
     */
    metricTime(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Returns the pattern for the execution ARN's of the state machine
     */
    private executionArn;
    private cannedMetric;
}
/**
 * Define a StepFunctions State Machine.
 *
 * @stability stable
 */
export declare class StateMachine extends StateMachineBase {
    /**
     * Execution role of this state machine.
     *
     * @stability stable
     */
    readonly role: iam.IRole;
    /**
     * The name of the state machine.
     *
     * @stability stable
     * @attribute true
     */
    readonly stateMachineName: string;
    /**
     * The ARN of the state machine.
     *
     * @stability stable
     */
    readonly stateMachineArn: string;
    /**
     * Type of the state machine.
     *
     * @stability stable
     * @attribute true
     */
    readonly stateMachineType: StateMachineType;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: StateMachineProps);
    /**
     * The principal this state machine is running as.
     *
     * @stability stable
     */
    get grantPrincipal(): iam.IPrincipal;
    /**
     * Add the given statement to the role's policy.
     *
     * @stability stable
     */
    addToRolePolicy(statement: iam.PolicyStatement): void;
    private validateStateMachineName;
    private buildLoggingConfiguration;
    private buildTracingConfiguration;
}
/**
 * A State Machine.
 *
 * @stability stable
 */
export interface IStateMachine extends IResource, iam.IGrantable {
    /**
     * The ARN of the state machine.
     *
     * @stability stable
     * @attribute true
     */
    readonly stateMachineArn: string;
    /**
     * Grant the given identity permissions to start an execution of this state machine.
     *
     * @param identity The principal.
     * @stability stable
     */
    grantStartExecution(identity: iam.IGrantable): iam.Grant;
    /**
     * Grant the given identity read permissions for this state machine.
     *
     * @param identity The principal.
     * @stability stable
     */
    grantRead(identity: iam.IGrantable): iam.Grant;
    /**
     * Grant the given identity read permissions for this state machine.
     *
     * @param identity The principal.
     * @stability stable
     */
    grantTaskResponse(identity: iam.IGrantable): iam.Grant;
    /**
     * Grant the given identity permissions for all executions of a state machine.
     *
     * @param identity The principal.
     * @param actions The list of desired actions.
     * @stability stable
     */
    grantExecution(identity: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Grant the given identity custom permissions.
     *
     * @param identity The principal.
     * @param actions The list of desired actions.
     * @stability stable
     */
    grant(identity: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Return the given named metric for this State Machine's executions.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of executions that failed.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricFailed(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of executions that were throttled.
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    metricThrottled(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of executions that were aborted.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricAborted(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of executions that succeeded.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricSucceeded(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of executions that timed out.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricTimedOut(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of executions that were started.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricStarted(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the interval, in milliseconds, between the time the execution starts and the time it closes.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricTime(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
}
export {};
