from . import database


class DataBase:
    """Class to read and delete data from a database generated by qupled"""

    # Read runs in the database
    @staticmethod
    def inspect_runs(database_name: str | None = None) -> dict:
        """Reads runs from the database and returns the content in the form of a dictionary.

        Args:
            database_name: Name of the database to read from. Defaults to None.

        Returns:
            A dictionary whose keys are the run ids and values are the corresponding runs information.
        """
        db_handler = database.DataBaseHandler(database_name)
        return db_handler.inspect_runs()

    # Read runs in the database
    @staticmethod
    def read_run(
        run_id: int,
        database_name: str | None = None,
        input_names: list[str] | None = None,
        result_names: list[str] | None = None,
    ) -> dict:
        """
        Reads a run from the database.

        Args:
            run_id: The ID of the run to read.
            database_name: The name of the database. Defaults to None.
            input_names: A list of input names to retrieve. Defaults to None.
            result_names: A list of result names to retrieve. Defaults to None.

        Returns:
            dict: A dictionary containing the run data.
        """
        db_handler = database.DataBaseHandler(database_name)
        return db_handler.get_run(run_id, input_names, result_names)

    # Read inputs in the database
    @staticmethod
    def read_inputs(
        run_id: int, database_name: str | None = None, names: list[str] | None = None
    ) -> dict:
        """Reads inputs from the database and returns the content in the form of a dictionary.

        Args:
            run_id: Identifier of the run to read input for.
            database_name: Name of the database to read from (default is None).
            names: A list of quantities to read (default is None, which reads all available quantities).

        Returns:
            A dictionary whose keys are the quantities listed in names and values are the corresponding inputs.
        """
        db_handler = database.DataBaseHandler(database_name)
        return db_handler.get_inputs(run_id, names if names is not None else [])

    # Read results in the database
    @staticmethod
    def read_results(
        run_id: int, database_name: str | None = None, names: list[str] | None = None
    ) -> dict:
        """Reads results from the database and returns the content in the form of a dictionary.

        Args:
            run_id: Identifier of the run to read results for.
            database_name: Name of the database to read from (default is None).
            names: A list of quantities to read (default is None, which reads all available quantities).

        Returns:
            A dictionary whose keys are the quantities listed in names and values are the corresponding results.
        """
        db_handler = database.DataBaseHandler(database_name)
        return db_handler.get_results(run_id, names)

    # Delete results from the database
    @staticmethod
    def delete_run(run_id: int, database_name: str | None = None):
        """
        Deletes a run entry from the database based on the provided run ID.

        Args:
            run_id: The unique identifier of the run to be deleted.
            database_name: The name of the database to connect to.
                If None, the default database will be used.
        """
        db_handler = database.DataBaseHandler(database_name)
        return db_handler.delete_run(run_id)
