# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/02_data_pipeline.ipynb (unless otherwise specified).

__all__ = ['embedding_cols', 'feature_cols', 'cols', 'add_targets', 'stack_into_df', 'scale', 'encode_col',
           'encode_embedding_cols', 'at_bat_aggs', 'prep_data_for_modeling']

# Internal Cell
from . import processing
from .acquisition import query_db
from .processing import last
from fastscript import *
import pandas as pd
import numpy as np
from typing import List
from sklearn.preprocessing import StandardScaler
import pickle
from scipy.stats import bernoulli

# Cell

embedding_cols = ["game_pk", "game_type", "pitcher", "pitcher_team_year"]
feature_cols = ["post_bat_score", "score_diff", "end_inning", "inning", "postouts", "cum_sb_ratio",
                "times_thru_order", "post_total_runners", "tying_run_on", "pitch_total", "post_opposite_hand",
                "walk", 'walk_cumsum', 'strikeout_cumsum', 'home_run_cumsum', 'bases_cumsum']
cols = embedding_cols + feature_cols

# Cell

# adding targets to each
def add_targets(starts: List):
    """adding target as last col to each start"""
    for i, start in enumerate(starts):
        y = np.zeros((start.shape[0], 1))
        y[-1, 0] = 1
        starts[i] = np.concatenate([start, y], axis=1)
    return starts

# Cell

def stack_into_df(starts: List):
    # concatenating into big dfs
    df = pd.DataFrame(np.concatenate(starts, axis=0), columns=cols+["pulled"])

    # correcting data types
    for col in feature_cols + ["pulled"]:
        df[col] = df[col].astype(float)
    return df

# Cell

def scale(train: pd.DataFrame, test: pd.DataFrame):
    # scaling data
    scaler = StandardScaler()
    scaler.fit(train[feature_cols])
    train[feature_cols] = scaler.transform(train[feature_cols])
    test[feature_cols] = scaler.transform(test[feature_cols])
    return train, test, scaler

# Cell

def encode_col(train, valid, col="pitcher_id"):

    # encoding movies and user ids with continous ids
    train_ids = np.sort(np.unique(train[col].values))

    # number of unique ids
    num_users = len(train_ids)
    print(f"There are {num_users} unique {col}'s in this dataset")

    # making changes in df
    id2idx = {o:i for i,o in enumerate(train_ids)}
    train[col] = train[col].apply(lambda x: id2idx[x])
    valid[col] = valid[col].apply(lambda x: id2idx.get(x, -1)) # -1 for users not in training

    # getting rid of users not in training set
    valid = valid[valid[col] >= 0].copy()
    return train, valid, id2idx


def encode_embedding_cols(train, test, cols=["game_pk", "game_type", "pitcher", "pitcher_team_year"]):
    # adding a row of zeros that act as "null" or "unknown"
    # embeddings for the zero-padded rows
    zero_row = pd.DataFrame(np.zeros((1, train.shape[1])), columns=train.columns)
    train = pd.concat([zero_row, train], axis=0)
    test = pd.concat([zero_row, test], axis=0)

    # changing dtypes in order to encode for embeddings
    for cat in ["game_type", "pitcher_team_year"]:
        train[cat] = train[cat].astype(str)
        test[cat] = test[cat].astype(str)

    mappers = dict()
    # not embedding game_pk, just switching to int for easier type casting
    for col in cols:
        train, test, mapper = encode_col(train, test, col=col)
        mappers[col] = mapper

    return train, test, mappers

# Cell
at_bat_aggs = {
    "balls": "max",
    "strikes": "max",
    "pitch_number": "max",
    "post_bat_score": last,
    "post_fld_score": last,
    "events": "max",
    "postouts": last,
    "post_on_1b": last,
    "post_on_2b": last,
    "post_on_3b": last,
    "game_type": last,
    "home_team": last,
    "away_team": last,
    "inning": last,
    "inning_topbot": last,
    "post_opposite_hand": last,
    "game_year": last,
    "pitcher_team": last
}

# Cell


@call_parse
def prep_data_for_modeling(
    db_path: Param(
        help="Path to db with statcast data", type=str
    ) = "./data/raw/statcast_pitches.db",
    years: Param(help="Year of statcast data to process", type=str, nargs="+") = [
        "2019"
    ],
    verbose: Param(
        help="Whether to print out updates on processing", type=bool_arg
    ) = True,
    train_test_split_by: Param(
        help="How to split into train/test sets. One of {'start', 'year'}.", type=str
    ) = "start",
    test_size: Param(help="Percent of data to allocate to test set", type=float) = 0.25,
    output_path: Param(
        help="Path to save processed csv files", type=str
    ) = "./data/processed/",
):
    # getting all dfs from all years into a single df
    dfs = []
    for year in years:
        df_year = query_db(db_path, year, verbose=verbose)
        dfs.append(df_year)
    df = pd.concat(dfs, axis=0)

    # identifying eligible game-pitcher-year combos
    games_pitchers_years = processing.get_games_pitchers_years(df, verbose)

    # deciding which outings to allocate to train or test set
    if train_test_split_by == "start":
        # pre-determining which starts will go into train/test sets
        test_flags = bernoulli(p=test_size).rvs(
            len(games_pitchers_years), random_state=742
        )
        train_year = test_year = years
    elif train_test_split_by == "year":
        # identifying year of test starts
        test_year = list(np.sort(df["game_date"].str[:4].unique())[-1])
        train_year = list(set(years).difference(set([test_year])))
        test_flags = [
            1 if str(y) == test_year[0] else 0 for (g, p, y) in games_pitchers_years
        ]
    else:
        # no starts go to test set
        test_flags = np.zeros(len(games_pitchers))

    # processing dfs of data from eligible pitchers
    train_starts = []
    test_starts = []
    for i, (test_flag, (g, p, y)) in enumerate(zip(test_flags, games_pitchers_years)):
        if verbose:
            if i % 100 == 0:
                print(f"Just processed {i}th start.")

        cleaned_df = processing.preliminary_clean(df, g, p)
        agged_df = processing.aggregate_at_bats(cleaned_df, at_bat_aggs)
        feature_engineered_df = processing.feature_engineering(agged_df)

        # making sure starting pitcher is in AL -> this _should_ no longer be necessary
        if feature_engineered_df.shape[0] > 0:
            if test_flag:
                test_starts.append(feature_engineered_df[cols])
            else:
                train_starts.append(feature_engineered_df[cols])
        else:
            print("empty df")

    # adding binary targets (pitcher always removed in last at-bat)
    train_starts = add_targets(train_starts)
    test_starts = add_targets(test_starts)

    # stacking starts into dfs for scaling and categorical encoding
    train = stack_into_df(train_starts)
    test = stack_into_df(test_starts)

    # standard scaling (mean of 0, sd of 1)
#     train, test, scaler = scale(train, test)

    # encoding categoricals for embeddings later
    train, test, mappers = encode_embedding_cols(train, test, cols=["pitcher"])

    # saving train, test sets, along with categorical mapper to output path
    train.to_csv(f"{output_path}/train_{'_'.join(train_year)}.csv", index=False)
    test.to_csv(f"{output_path}/test_{'_'.join(test_year)}.csv", index=False)
    with open(
        f"{output_path}/mappers_{'_'.join(train_year + test_year)}.pkl", "wb"
    ) as f:
        pickle.dump(mappers, f)

    if verbose:
        print(f"{years} data ready for modeling and saved at {output_path}.")
