# This file is part of Py6S.
#
# Copyright 2012 Robin Wilson and contributors listed in the CONTRIBUTORS file.
#
# Py6S is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Py6S is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with Py6S.  If not, see <http://www.gnu.org/licenses/>.

from .. import sixs_exceptions
import collections
import numpy as np


def Wavelength(start_wavelength, end_wavelength=None, filter=None):
    """Select one or more wavelengths for the 6S simulation.

    There are a number of ways to do this:

    1. Pass a single value of a wavelength in micrometres.
       The simulation will be performed for just this wavelength::

        Wavelength(0.43)

    2. Pass a start and end wavelength in micrometres.
       The simulation will be performed across this wavelength range with a
       constant filter function (spectral response function) of 1.0::

        Wavelength(0.43, 0.50)

    3. Pass a start and end wavelength, and a filter given at 2.5nm intervals.
       The simulation will be performed across this wavelength range using the
       given filter function::

        Wavelength(0.400, 0.410, [0.7, 0.9, 1.0, 0.3, 0.15])

       The filter function must include values for the start and end wavelengths,
       plus values every 2.5nm (0.0025um) in between. So, in the example above,
       there are five values given: one each for 0.400, 0.4025, 0.405, 0.4075, 0.410.

    4. Pass a constant (as defined in this class) for a pre-defined wavelength
       range::

        Wavelength(PredefinedWavelengths.LANDSAT_TM_B1)

    """
    try:
        wv_id = start_wavelength[0]
        if wv_id > 0:
            # It's one of the new predefined wavelengths that I've added
            return Wavelength(start_wavelength[1], start_wavelength[2], start_wavelength[3])
        else:
            wv_type = "%d (Chosen Band)\n" % (-1 * wv_id)
            data = None
            min_wv = start_wavelength[1]
            max_wv = start_wavelength[2]
    except:
        if end_wavelength is None:
            # It's simply a wavelength value
            if start_wavelength > PredefinedWavelengths.MAX_ALLOWABLE_WAVELENGTH or start_wavelength < PredefinedWavelengths.MIN_ALLOWABLE_WAVELENGTH:
                raise sixs_exceptions.ParameterError("wavelength", "Wavelength must be between %f and %f" % (PredefinedWavelengths.MIN_ALLOWABLE_WAVELENGTH, PredefinedWavelengths.MAX_ALLOWABLE_WAVELENGTH))
            wv_type = "-1"
            data = "%f" % start_wavelength
            min_wv = start_wavelength
            max_wv = start_wavelength
        else:
            if start_wavelength > PredefinedWavelengths.MAX_ALLOWABLE_WAVELENGTH or start_wavelength < PredefinedWavelengths.MIN_ALLOWABLE_WAVELENGTH or end_wavelength > PredefinedWavelengths.MAX_ALLOWABLE_WAVELENGTH or end_wavelength < PredefinedWavelengths.MIN_ALLOWABLE_WAVELENGTH:
                raise sixs_exceptions.ParameterError("wavelength", "Wavelength must be between %f and %f" % (PredefinedWavelengths.MIN_ALLOWABLE_WAVELENGTH, PredefinedWavelengths.MAX_ALLOWABLE_WAVELENGTH))
            min_wv = start_wavelength
            max_wv = end_wavelength
            if filter is None:
                # They haven't given a filter, so assume filter is constant at 1
                wv_type = "0 constant filter function"
                data = "%f %f" % (start_wavelength, end_wavelength)
            else:
                # Filter has been given, so we must use it.
                # We check filter has been given at 2.5nm intervals
                n_req_filter_values = round((end_wavelength - start_wavelength) / 0.0025) + 1
                if len(filter) != n_req_filter_values:
                    raise sixs_exceptions.ParameterError("wavelength", "You must provide a filter value at 2.5nm (0.0025um) intervals, including both the start and end wavelengths.")
                wv_type = "1 User's defined filtered function"
                data = """%f %f
    %s""" % (start_wavelength, end_wavelength, " ".join(map(str, filter)))

    if data is None:
        return_string = wv_type
    else:
        return_string =  """%s
%s\n""" % (wv_type, data)

    return (return_string, min_wv, max_wv)


class PredefinedWavelengths:
    MAX_ALLOWABLE_WAVELENGTH = 4
    MIN_ALLOWABLE_WAVELENGTH = 0.2

    # New predefined wavelengths that I've added to Py6S
    # CONSTANT_NAME = (ID, Start Wavelength, End Wavelength, Filter Function)
    # Note: IDs must be > 1 for new predefined wavelengths

    # Landsat OLI
    # Taken from spreadsheet downloadable from http://landsat.gsfc.nasa.gov/?p=5779
    # Interpolated to 2.5nm intervals, as required by 6S
    LANDSAT_OLI_B1 = (1, 0.427, 0.457,
                      np.array([7.30000000e-05, 2.52450000e-03, 2.47670000e-02,
                                3.85985000e-01, 9.08749000e-01, 9.80591500e-01,
                                9.86713000e-01, 9.96568500e-01, 9.82780000e-01,
                                8.25707000e-01, 2.26412000e-01, 2.55700000e-02,
                                2.41400000e-03]))

    LANDSAT_OLI_B2 = (2, 0.436, 0.526,
                      np.array([1.00000000e-05, 1.79000000e-04, 4.55000000e-04,
                                1.63350000e-03, 6.86900000e-03, 4.28880000e-02,
                                2.71370000e-01, 7.90740500e-01, 9.03034000e-01,
                                9.04677500e-01, 8.89667000e-01, 8.79232000e-01,
                                8.79688000e-01, 8.89796500e-01, 8.48533000e-01,
                                8.36270500e-01, 8.68497000e-01, 9.11461500e-01,
                                9.31726000e-01, 9.54896500e-01, 9.56424000e-01,
                                9.83834000e-01, 9.89469000e-01, 9.68066500e-01,
                                9.88729000e-01, 9.61057500e-01, 9.66125000e-01,
                                9.82077000e-01, 9.63135000e-01, 9.98249000e-01,
                                8.44893000e-01, 1.19533500e-01, 5.32800000e-03,
                                1.32850000e-03, 5.16000000e-04, 1.17000000e-04,
                                2.30000000e-05]))

    LANDSAT_OLI_B3 = (3, 0.512, 0.610,
                      np.array([-4.60000000e-05, 1.78500000e-04, 6.48000000e-04,
                                1.57400000e-03, 3.44600000e-03, 8.73250000e-03,
                                2.55130000e-02, 9.69975000e-02, 3.53885000e-01,
                                8.03215000e-01, 9.54627000e-01, 9.60271500e-01,
                                9.69873000e-01, 9.69833500e-01, 9.77001000e-01,
                                9.95392000e-01, 9.82642000e-01, 9.71423000e-01,
                                9.46245000e-01, 9.62786000e-01, 9.66447000e-01,
                                9.64176500e-01, 9.83397000e-01, 9.70875500e-01,
                                9.78208000e-01, 9.77182500e-01, 9.69181000e-01,
                                9.81277000e-01, 9.68886000e-01, 9.80432000e-01,
                                9.04478000e-01, 6.05139000e-01, 1.90467000e-01,
                                2.47350000e-02, 2.57400000e-03, 2.39500000e-04,
                                0, 0, 0,
                                0]))

    LANDSAT_OLI_B4 = (4, 0.625, 0.690,
                      np.array([-3.42000000e-04, 1.37250000e-03, 7.19700000e-03,
                                4.86465000e-02, 2.99778000e-01, 8.34958000e-01,
                                9.50823000e-01, 9.57268000e-01, 9.84173000e-01,
                                9.83172500e-01, 9.59441000e-01, 9.54442500e-01,
                                9.81688000e-01, 9.88501500e-01, 9.76960000e-01,
                                9.88942000e-01, 9.80678000e-01, 9.66466000e-01,
                                9.66928000e-01, 7.29107000e-01, 1.23946000e-01,
                                1.25175000e-02, 1.40200000e-03, 0,
                                0, 0, 0]))

    LANDSAT_OLI_B5 = (5, 0.829, 0.899,
                      np.array([0, 7.50000000e-05, 3.14000000e-04,
                                8.52500000e-04, 2.10700000e-03, 5.90150000e-03,
                                1.73460000e-02, 6.62770000e-02, 2.49733000e-01,
                                6.63830000e-01, 9.60215000e-01, 9.76869500e-01,
                                1.00000000e+00, 9.78334000e-01, 9.57357000e-01,
                                9.50103000e-01, 9.48450000e-01, 9.53355500e-01,
                                9.69821000e-01, 8.39899500e-01, 4.48364000e-01,
                                1.37481000e-01, 3.45320000e-02, 1.00205000e-02,
                                2.94400000e-03, 9.67500000e-04, 2.41000000e-04,
                                1.55000000e-05, 0]))

    LANDSAT_OLI_B6 = (6, 1.515, 1.695,
                      np.array([0, 2.00000000e-04, 4.66000000e-04,
                                8.45000000e-04, 1.36900000e-03, 2.01550000e-03,
                                2.88100000e-03, 4.02150000e-03, 5.52800000e-03,
                                7.88900000e-03, 1.09890000e-02, 1.52755000e-02,
                                2.18310000e-02, 3.25615000e-02, 4.78640000e-02,
                                7.09490000e-02, 1.01893000e-01, 1.50884500e-01,
                                2.20261000e-01, 3.10649000e-01, 4.21470000e-01,
                                5.52234000e-01, 6.76683000e-01, 7.71509000e-01,
                                8.54065000e-01, 8.95823500e-01, 9.13009000e-01,
                                9.25058000e-01, 9.26413000e-01, 9.23818000e-01,
                                9.22828000e-01, 9.22408500e-01, 9.26605000e-01,
                                9.43438000e-01, 9.46175000e-01, 9.47297500e-01,
                                9.52859000e-01, 9.51358500e-01, 9.59047000e-01,
                                9.59191500e-01, 9.61470000e-01, 9.60494000e-01,
                                9.64703000e-01, 9.69951500e-01, 9.76906000e-01,
                                9.81271500e-01, 9.88609000e-01, 9.99010500e-01,
                                9.99642000e-01, 9.89828000e-01, 9.67125000e-01,
                                9.26719000e-01, 8.40967000e-01, 7.23103000e-01,
                                5.73232000e-01, 4.22998000e-01, 2.91752000e-01,
                                1.95988000e-01, 1.28463000e-01, 8.28380000e-02,
                                5.27520000e-02, 3.45655000e-02, 2.25040000e-02,
                                1.47195000e-02, 9.58700000e-03, 6.39600000e-03,
                                4.25700000e-03, 2.79800000e-03, 1.78100000e-03,
                                1.14200000e-03, 6.77000000e-04, 3.55000000e-04,
                                1.12000000e-04]))

    LANDSAT_OLI_B7 = (7, 2.037, 2.354,
                      np.array([0, 1.07000000e-04, 2.40000000e-04,
                                3.99000000e-04, 5.99000000e-04, 8.80500000e-04,
                                1.22200000e-03, 1.64550000e-03, 2.18700000e-03,
                                2.85300000e-03, 3.73300000e-03, 4.88200000e-03,
                                6.33700000e-03, 8.44950000e-03, 1.10050000e-02,
                                1.42985000e-02, 1.88990000e-02, 2.45090000e-02,
                                3.20710000e-02, 4.28190000e-02, 5.64290000e-02,
                                7.48955000e-02, 1.00640000e-01, 1.36480000e-01,
                                1.79714000e-01, 2.40483000e-01, 3.11347000e-01,
                                3.94832500e-01, 4.88816000e-01, 5.73971000e-01,
                                6.63067000e-01, 7.39406500e-01, 7.92667000e-01,
                                8.41172500e-01, 8.67845000e-01, 8.86269000e-01,
                                9.06527000e-01, 9.14538000e-01, 9.29693000e-01,
                                9.38975000e-01, 9.42952000e-01, 9.44181000e-01,
                                9.48776000e-01, 9.49521500e-01, 9.56635000e-01,
                                9.48258500e-01, 9.50874000e-01, 9.47049500e-01,
                                9.57717000e-01, 9.47095000e-01, 9.51641000e-01,
                                9.46800000e-01, 9.40311000e-01, 9.46466500e-01,
                                9.38737000e-01, 9.44439000e-01, 9.44482000e-01,
                                9.50472000e-01, 9.39939000e-01, 9.37156500e-01,
                                9.38955000e-01, 9.28123500e-01, 9.30508000e-01,
                                9.30946000e-01, 9.36472000e-01, 9.34327500e-01,
                                9.46217000e-01, 9.53826000e-01, 9.63135000e-01,
                                9.63944000e-01, 9.62905000e-01, 9.61607000e-01,
                                9.57814000e-01, 9.55657500e-01, 9.51706000e-01,
                                9.60275500e-01, 9.47696000e-01, 9.59807000e-01,
                                9.55750000e-01, 9.56607500e-01, 9.66786000e-01,
                                9.62823000e-01, 9.77637000e-01, 9.83457500e-01,
                                9.85056000e-01, 9.98627000e-01, 9.92469000e-01,
                                9.97947000e-01, 9.97261000e-01, 9.89437000e-01,
                                9.86037000e-01, 9.81280000e-01, 9.72794000e-01,
                                9.76369500e-01, 9.74409000e-01, 9.63698500e-01,
                                9.55095000e-01, 9.51391500e-01, 9.22405000e-01,
                                8.89264000e-01, 8.23876000e-01, 7.21272500e-01,
                                6.02539000e-01, 4.77695500e-01, 3.55569000e-01,
                                2.61452500e-01, 1.86151000e-01, 1.31725000e-01,
                                9.20290000e-02, 6.49895000e-02, 4.63320000e-02,
                                3.34235000e-02, 2.40000000e-02, 1.76250000e-02,
                                1.29300000e-02, 9.55700000e-03, 7.08800000e-03,
                                5.33100000e-03, 3.90300000e-03, 2.83800000e-03,
                                2.04700000e-03, 1.44950000e-03, 9.74000000e-04,
                                6.20000000e-04, 3.20000000e-04, 7.35000000e-05,
                                0, 0]))

    LANDSAT_OLI_B8 = (8, 0.488, 0.691,
                       np.array([2.16000000e-04, 1.30000000e-03, 3.84100000e-03,
                                 1.22590000e-02, 4.27230000e-02, 1.60137500e-01,
                                 4.72496000e-01, 7.45412500e-01, 8.31881000e-01,
                                 8.55321500e-01, 8.59640000e-01, 8.57696000e-01,
                                 8.58455000e-01, 8.58301000e-01, 8.50183000e-01,
                                 8.58223500e-01, 8.61508000e-01, 8.57683500e-01,
                                 8.79249000e-01, 8.91710500e-01, 9.06294000e-01,
                                 9.12867000e-01, 9.02939000e-01, 9.20739500e-01,
                                 9.13020000e-01, 8.85650500e-01, 8.79443000e-01,
                                 8.74179000e-01, 8.75361000e-01, 8.91665000e-01,
                                 8.74097000e-01, 8.86888500e-01, 9.03528000e-01,
                                 9.10950500e-01, 9.13190000e-01, 9.20178000e-01,
                                 9.24431000e-01, 9.29809500e-01, 9.48863000e-01,
                                 9.40543000e-01, 9.45674000e-01, 9.39380000e-01,
                                 9.46659000e-01, 9.34044500e-01, 9.40838000e-01,
                                 9.58039500e-01, 9.68241000e-01, 9.66480500e-01,
                                 9.57232000e-01, 9.47675500e-01, 9.52465000e-01,
                                 9.57481500e-01, 9.64158000e-01, 9.67366000e-01,
                                 9.77026000e-01, 9.76029500e-01, 9.69583000e-01,
                                 9.72807000e-01, 9.65780000e-01, 9.66738000e-01,
                                 9.72067000e-01, 9.79346500e-01, 9.71123000e-01,
                                 9.53377000e-01, 9.63851000e-01, 9.67101500e-01,
                                 9.70613000e-01, 9.79974500e-01, 9.88302000e-01,
                                 9.91753000e-01, 1.00000000e+00, 9.98476000e-01,
                                 9.92555000e-01, 9.85811500e-01, 9.13945000e-01,
                                 5.24376500e-01, 1.67313000e-01, 4.61755000e-02,
                                 1.51780000e-02, 6.70950000e-03, 3.22000000e-03,
                                 1.21200000e-03]))

    LANDSAT_OLI_PAN = LANDSAT_OLI_B8

    LANDSAT_OLI_B9 = (9, 1.340, 1.407,
                      np.array([0, 2.57500000e-04, 6.47000000e-04,
                                1.27400000e-03, 2.31800000e-03, 4.70450000e-03,
                                1.11240000e-02, 3.45385000e-02, 1.15351000e-01,
                                3.86681500e-01, 7.72118000e-01, 9.00941500e-01,
                                9.31247000e-01, 9.91687000e-01, 1.00000000e+00,
                                9.77080500e-01, 8.71343000e-01, 6.54888000e-01,
                                2.97920000e-01, 8.96145000e-02, 2.60840000e-02,
                                8.30650000e-03, 2.78000000e-03, 1.11900000e-03,
                                2.20000000e-04, 0, 0,
                                0]))


    # RapidEye
    # Taken from http://blackbridge.com/rapideye/upload/Spectral_Response_Curves.pdf
    # Interpolated to 2.5nm intervals, as required by 6S
#     RAPIDEYE_B1 = (10, 0.420, 0.514,
#                     np.array([ 0.002 ,  0.    ,  0.    ,  0.0015,  0.    ,  0.    ,  0.001 ,
#                             0.0095,  0.321 ,  0.725 ,  0.74  ,  0.759 ,  0.77  ,  0.781 ,
#                             0.784 ,  0.7935,  0.796 ,  0.8005,  0.806 ,  0.804 ,  0.807 ,
#                             0.817 ,  0.82  ,  0.8275,  0.84  ,  0.847 ,  0.862 ,  0.8765,
#                             0.886 ,  0.9105,  0.928 ,  0.9415,  0.969 ,  0.9685,  1.    ,
#                             0.9875,  0.437 ,  0.019, 0.0]))

#     RAPIDEYE_B2 = (11, 0.462, 0.595,
#                     np.array([ 0.002 ,  0.0025,  0.002 ,  0.021 ,  0.    ,  0.    ,  0.    ,
#                             0.    ,  0.    ,  0.    ,  0.    ,  0.0145,  0.    ,  0.    ,
#                             0.    ,  0.    ,  0.    ,  0.    ,  0.    ,  0.001 ,  0.002 ,
#                             0.01  ,  0.054 ,  0.4055,  0.868 ,  0.866 ,  0.877 ,  0.872 ,
#                             0.874 ,  0.8815,  0.882 ,  0.8805,  0.886 ,  0.8955,  0.899 ,
#                             0.8995,  0.91  ,  0.922 ,  0.928 ,  0.9345,  0.946 ,  0.9525,
#                             0.96  ,  0.972 ,  0.976 ,  0.975 ,  0.989 ,  0.9905,  0.984 ,
#                             0.997 ,  0.97  ,  0.6535,  0.039 ,  0.004 ]))

#     RAPIDEYE_B3 = (12, 0.486, 0.691,
#                     np.array([  2.00000000e-03,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              1.00000000e-03,   5.00000000e-04,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   5.00000000e-04,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   5.00000000e-04,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              1.00000000e-03,   4.00000000e-03,   1.80000000e-02,
#                              1.43000000e-01,   7.23000000e-01,   8.57500000e-01,
#                              8.65000000e-01,   8.81500000e-01,   8.82000000e-01,
#                              8.93000000e-01,   9.07000000e-01,   9.12500000e-01,
#                              9.18000000e-01,   9.31000000e-01,   9.44000000e-01,
#                              9.52500000e-01,   9.61000000e-01,   9.67000000e-01,
#                              9.73000000e-01,   9.83000000e-01,   9.91000000e-01,
#                              9.87000000e-01,   9.89000000e-01,   9.99000000e-01,
#                              9.81000000e-01,   9.28500000e-01,   1.74000000e-01,
#                              1.70000000e-02, 0.0]))

#     RAPIDEYE_B4 = (13, 0.500, 0.740,
#                     np.array([  2.70000000e-02,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   5.00000000e-04,
#                              0.00000000e+00,   5.00000000e-04,   1.00000000e-03,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   5.00000000e-04,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   2.20000000e-02,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   5.00000000e-04,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   1.00000000e-03,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   1.00000000e-03,
#                              2.00000000e-03,   5.50000000e-03,   2.10000000e-02,
#                              1.08500000e-01,   4.91000000e-01,   9.49000000e-01,
#                              9.98000000e-01,   9.99500000e-01,   9.98000000e-01,
#                              9.91500000e-01,   9.87000000e-01,   9.85500000e-01,
#                              9.82000000e-01,   9.74000000e-01,   9.66000000e-01,
#                              9.64000000e-01,   9.61000000e-01,   9.45500000e-01,
#                              9.39000000e-01,   8.98000000e-01,   4.25000000e-01,
#                              9.40000000e-02,   2.00000000e-02,   5.50000000e-03, 0.0]))

#     RAPIDEYE_B5 = (14, 0.519, 0.866,
#                     np.array([  2.00000000e-03,   5.00000000e-04,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   1.00000000e-03,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   2.00000000e-03,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   1.00000000e-03,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   4.00000000e-03,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              1.00000000e-03,   2.00000000e-03,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   1.00000000e-03,   2.00000000e-03,
#                              5.00000000e-04,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   5.00000000e-04,   2.20000000e-02,
#                              5.00000000e-04,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
#                              0.00000000e+00,   1.00000000e-03,   1.00000000e-03,
#                              2.00000000e-03,   4.50000000e-03,   9.00000000e-03,
#                              1.95000000e-02,   4.60000000e-02,   1.27500000e-01,
#                              3.45000000e-01,   7.49000000e-01,   9.88000000e-01,
#                              9.88000000e-01,   9.69000000e-01,   9.71500000e-01,
#                              9.81000000e-01,   9.82000000e-01,   9.74000000e-01,
#                              9.65000000e-01,   9.60000000e-01,   9.57500000e-01,
#                              9.59000000e-01,   9.59000000e-01,   9.57000000e-01,
#                              9.54500000e-01,   9.50000000e-01,   9.45000000e-01,
#                              9.39000000e-01,   9.34000000e-01,   9.29000000e-01,
#                              9.27000000e-01,   9.26000000e-01,   9.26000000e-01,
#                              9.25000000e-01,   9.19500000e-01,   9.11000000e-01,
#                              9.00000000e-01,   8.91000000e-01,   8.85500000e-01,
#                              8.83000000e-01,   8.78000000e-01,   8.65000000e-01,
#                              8.45500000e-01,   8.38000000e-01,   8.37000000e-01,
#                              7.12000000e-01,   3.63000000e-01,   1.24000000e-01,
#                              4.30000000e-02,   1.60000000e-02,   7.00000000e-03,
#                              3.00000000e-03], ))


#     # Pleiades 1B
#     # Taken from ENVI Spectral Library file
#     # Interpolated to 2.5nm intervals, as required by 6S
#     #
#     # NOTE: There are out-of-band responses for a number of these
#     # bands (ie. responses in the NIR for the blue band) and these
#     # *are* included, giving strange min/max wavelengths
#     #
#     PLEIADES_B1 = (15, 0.438, 0.915, # Blue
#                     np.array([ 0.001603,  0.001338,  0.004344,  0.011072,  0.017627,  0.023367,
#                     0.057403,  0.134555,  0.223763,  0.308543,  0.461557,  0.650821,
#                     0.755369,  0.74772 ,  0.716819,  0.718538,  0.756696,  0.810109,
#                     0.842166,  0.823437,  0.775247,  0.752701,  0.78022 ,  0.819927,
#                     0.851663,  0.860275,  0.858684,  0.865826,  0.882822,  0.904041,
#                     0.919695,  0.932596,  0.950233,  0.975798,  0.994977,  1.      ,
#                     0.995063,  0.980628,  0.94175 ,  0.843622,  0.671142,  0.46334 ,
#                     0.288865,  0.167078,  0.09018 ,  0.050519,  0.031488,  0.023814,
#                     0.021311,  0.02063 ,  0.01956 ,  0.016794,  0.011358,  0.006652,
#                     0.004144,  0.00303 ,  0.002416,  0.00199 ,  0.001568,  0.001136,
#                     0.001253,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.001109,  0.001987,  0.00122 ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.001119,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.001629,
#                     0.0024  ,  0.001032,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.001028,  0.001563,  0.002609,
#                     0.004575,  0.009062,  0.017163,  0.023793,  0.021523,  0.015914,
#                     0.011956,  0.009482,  0.007869,  0.007224,  0.007242,  0.007568,
#                     0.008379,  0.009864,  0.012259,  0.016267,  0.022602,  0.032027,
#                     0.04443 ,  0.054669,  0.056424,  0.048004,  0.035799,  0.025834,
#                     0.018887,  0.014439,  0.011679,  0.009911,  0.008647,  0.00781 ,
#                     0.007227,  0.006996,  0.006923,  0.006914,  0.007021,  0.007253,
#                     0.007373,  0.007505,  0.00747 ,  0.007067]))

#     PLEIADES_B2 = (16, 0.490, 0.950, # Green
#                     np.array([ 0.001521,  0.00413 ,  0.009954,  0.019642,  0.032548,  0.056793,
#                     0.123791,  0.285909,  0.528976,  0.771625,  0.883804,  0.907957,
#                     0.913146,  0.913728,  0.922484,  0.936708,  0.94976 ,  0.954499,
#                     0.958582,  0.964206,  0.970527,  0.972265,  0.967518,  0.95891 ,
#                     0.952449,  0.952466,  0.956048,  0.955179,  0.94899 ,  0.947145,
#                     0.95445 ,  0.97106 ,  0.989818,  1.      ,  0.99536 ,  0.969822,
#                     0.925304,  0.863324,  0.794828,  0.723897,  0.645327,  0.543852,
#                     0.417028,  0.276671,  0.157527,  0.085607,  0.049226,  0.032724,
#                     0.023793,  0.018197,  0.014062,  0.009966,  0.005845,  0.003038,
#                     0.001536,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.003185,  0.014191,  0.008339,  0.002244,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.001111,  0.001938,  0.003691,
#                     0.006357,  0.010271,  0.013108,  0.013261,  0.011115,  0.008366,
#                     0.006564,  0.005685,  0.00538 ,  0.005623,  0.0062  ,  0.007239,
#                     0.00892 ,  0.01151 ,  0.014942,  0.019269,  0.023479,  0.02644 ,
#                     0.027049,  0.025545,  0.023397,  0.021395,  0.019944,  0.019253,
#                     0.019074,  0.019689,  0.020694,  0.022011,  0.02323 ,  0.023757,
#                     0.022986,  0.02066 ,  0.017225,  0.013292, 0.0]))


#     PLEIADES_B3 = (17, 0.448, 0.940, # RED
#                     np.array([ 0.001195,  0.005142,  0.008003,  0.006693,  0.010859,  0.024691,
#                             0.065359,  0.122542,  0.057009,  0.021375,  0.012797,  0.006278,
#                             0.002354,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                             0.      ,  0.001013,  0.      ,  0.      ,  0.      ,  0.      ,
#                             0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                             0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                             0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                             0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                             0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                             0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                             0.      ,  0.      ,  0.      ,  0.      ,  0.001244,  0.002142,
#                             0.003819,  0.006805,  0.012333,  0.022178,  0.041333,  0.078071,
#                             0.151934,  0.277675,  0.451038,  0.629132,  0.762549,  0.832945,
#                             0.857906,  0.865887,  0.869263,  0.875221,  0.885776,  0.900593,
#                             0.917488,  0.93488 ,  0.947811,  0.956953,  0.96233 ,  0.964767,
#                             0.962429,  0.961307,  0.962025,  0.969915,  0.981157,  0.993393,
#                             1.      ,  0.980951,  0.952263,  0.913173,  0.869401,  0.825208,
#                             0.783047,  0.736127,  0.673489,  0.587753,  0.480491,  0.363007,
#                             0.252303,  0.162603,  0.102221,  0.064127,  0.041916,  0.028464,
#                             0.020455,  0.01537 ,  0.012117,  0.009881,  0.008317,  0.007102,
#                             0.006095,  0.005172,  0.004314,  0.003495,  0.002771,  0.003589,
#                             0.003031,  0.002317,  0.001784,  0.001331,  0.001021,  0.      ,
#                             0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                             0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                             0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                             0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                             0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                             0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                             0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                             0.      ,  0.      ,  0.      ,  0.001308,  0.001761,  0.002403,
#                             0.003268,  0.004364,  0.005662,  0.00689 ,  0.007822,  0.00833 ,
#                             0.008436,  0.008185,  0.008138,  0.008001,  0.007768,  0.007784,
#                             0.007998,  0.008152,  0.008506,  0.008793,  0.009312,  0.009753,
#                             0.010136,  0.010387,  0.010406,  0.010171,  0.009522,  0.008609,
#                             0.007337]))

#     PLEIADES_B4 = (18, 0.750, 0.945, # NIR
#                     np.array([ 0.001687,  0.002665,  0.001225,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.001549,  0.00218 ,  0.001988,  0.001045,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.001224,  0.002266,  0.002879,
#                     0.002041,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
#                     0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.001154,
#                     0.001352,  0.001554,  0.001756,  0.001989,  0.002251,  0.002534,
#                     0.002903,  0.005236,  0.006401,  0.008013,  0.010147,  0.013109,
#                     0.017135,  0.022905,  0.030978,  0.042662,  0.05919 ,  0.083507,
#                     0.117888,  0.166378,  0.231114,  0.315936,  0.417216,  0.528495,
#                     0.640959,  0.747239,  0.838552,  0.908812,  0.959366,  0.988114,
#                     1.      ,  0.999206,  0.989642,  0.967696,  0.951436,  0.937494,
#                     0.925472,  0.915223,  0.908783,  0.902608,  0.896683,  0.892483,
#                     0.885491,  0.877655,  0.867639,  0.856896,  0.847441,  0.836048,
#                     0.823698,  0.813044,  0.801627,  0.792162,  0.782795,  0.776935,
#                     0.771465,  0.763145,  0.75434 ,  0.745237,  0.734921,  0.725512,
#                     0.714614,  0.703089,  0.691948,  0.681648,  0.67052 ,  0.659536,
#                     0.647596,  0.629185,  0.611093,  0.593432,  0.576059,  0.559994,
#                     0.544875,  0.530614,  0.515483,  0.497676,  0.47341 ,  0.43877 ,
#                     0.393422,  0.336769,  0.274273,  0.211434,  0.153554,  0.106341,
#                     0.070442,  0.045912,  0.029535,  0.019275,  0.01268 ,  0.008462]))

#     PLEIADES_PAN = (19, 0.460, 0.845, # Panchromatic
#                     np.array([ 0.002795,  0.005578,  0.011986,  0.022135,  0.038918,  0.066419,
#                                 0.106702,  0.161427,  0.227202,  0.298655,  0.367815,  0.425804,
#                                 0.46974 ,  0.499166,  0.519316,  0.53325 ,  0.546726,  0.560304,
#                                 0.573644,  0.585534,  0.595662,  0.604104,  0.612015,  0.618774,
#                                 0.625565,  0.631905,  0.637794,  0.642186,  0.644395,  0.644882,
#                                 0.644885,  0.645114,  0.646836,  0.651318,  0.659091,  0.669141,
#                                 0.681181,  0.693702,  0.705609,  0.715769,  0.725057,  0.732351,
#                                 0.738419,  0.743876,  0.749253,  0.753645,  0.758079,  0.761089,
#                                 0.763913,  0.765235,  0.766172,  0.766597,  0.767495,  0.768755,
#                                 0.772303,  0.777108,  0.783383,  0.791719,  0.801545,  0.810952,
#                                 0.820033,  0.829068,  0.836465,  0.842652,  0.848188,  0.851917,
#                                 0.855505,  0.858336,  0.862017,  0.866684,  0.87158 ,  0.878019,
#                                 0.884707,  0.891347,  0.898891,  0.904582,  0.910509,  0.915462,
#                                 0.920794,  0.925316,  0.928988,  0.932893,  0.936956,  0.94008 ,
#                                 0.94416 ,  0.94693 ,  0.950347,  0.952109,  0.95428 ,  0.957441,
#                                 0.960771,  0.966315,  0.9721  ,  0.978828,  0.984596,  0.991222,
#                                 0.995687,  0.999088,  1.      ,  0.996327,  0.991468,  0.986969,
#                                 0.981702,  0.977146,  0.974597,  0.972901,  0.973932,  0.974192,
#                                 0.976871,  0.979825,  0.982772,  0.984573,  0.985077,  0.983727,
#                                 0.982597,  0.979023,  0.974595,  0.969741,  0.964814,  0.959839,
#                                 0.953335,  0.948357,  0.944548,  0.939689,  0.934456,  0.929716,
#                                 0.924834,  0.918336,  0.911194,  0.9061  ,  0.900618,  0.895346,
#                                 0.890809,  0.889495,  0.888541,  0.880978,  0.870784,  0.856668,
#                                 0.836935,  0.810241,  0.772852,  0.718955,  0.65285 ,  0.571081,
#                                 0.479585,  0.384649,  0.294773,  0.216645,  0.153132,  0.105705,
#                                 0.07168 ,  0.049143,  0.033776,  0.025699,  0.018176,  0.013197,
#                                 0.012156,  0.00902 ,  0.008416,  0.00428 ,  0.003313,  0.002669,
#                                 0.002212,  0.001829,  0.001529,  0.001363,  0.001215]))

    # Sentinel-2A MSI spectral response functions
    # Taken from https://earth.esa.int/web/sentinel/user-guides/sentinel-2-msi/document-library/-/asset_publisher/Wk0TKajiISaR/content/sentinel-2a-spectral-responses
    S2A_MSI_01 = (53, 0.4120, 0.4570, 
                     np.array([0.00177574,0.00357067,0.00378029,0.00162485,0.00125044,
                     0.00109536,0.00182373,0.01670008,0.32387506,0.73834764,
                     0.80862387,0.83917843,0.88731097,0.99114075,0.93596338,
                     0.8379525 ,0.61417422,0.08388263,0.        ]))

    S2A_MSI_02 = (54, 0.4390, 0.5340, 
                     np.array([0.01031543,0.02550952,0.02269349,0.01904007,0.02130721,
                     0.01793204,0.01924643,0.0574268 ,0.32799225,0.73148173,
                     0.75668081,0.77382288,0.86179176,0.9192643 ,0.90035366,
                     0.87919289,0.90695544,0.94165397,0.8860231 ,0.79745698,
                     0.77209054,0.82911115,0.92793997,0.96755782,0.95832003,
                     0.95438995,0.9770514 ,0.96669894,0.90190134,0.83341648,
                     0.82057639,0.89259188,0.99163699,0.9474343 ,0.50395858,
                     0.14371473,0.04207746,0.00849869,0.        ]))

    S2A_MSI_03 = (55, 0.5380, 0.5830, 
                     np.array([0.01448181,0.11395589,0.55322279,0.87331261,0.92572845,
                     0.8767134 ,0.8387572 ,0.87529183,0.96017974,0.9993004 ,
                     0.94522089,0.83711029,0.76460653,0.755965  ,0.79890086,
                     0.79383741,0.40397555,0.07754079,0.00108588]))

    S2A_MSI_04 = (56, 0.6460, 0.6860, 
                     np.array([0.00141521,0.25369897,0.94485805,0.99703154,0.99052772,
                     0.90975472,0.77345952,0.7712748 ,0.83612975,0.89873936,
                     0.95042063,0.95976475,0.95527459,0.81958553,0.22960399,
                     0.01516159,0.        ]))

    S2A_MSI_05 = (57, 0.6950, 0.7150, 
                     np.array([2.83578563e-02,5.77459223e-01,9.98895232e-01,
                     9.88670233e-01,9.48473886e-01,8.94935856e-01,
                     7.90753191e-01,1.60051275e-01,2.74780199e-15]))

    S2A_MSI_06 = (58, 0.7310, 0.7510, 
                     np.array([0.00171088,0.45264387,0.90232877,0.9520792 ,0.9770217 ,
                     0.99440139,0.90126739,0.11597958,0.        ]))

    S2A_MSI_07 = (59, 0.7690, 0.7990, 
                     np.array([4.58986862e-04,1.08907239e-01,6.36854532e-01,
                     9.80612493e-01,1.00000000e+00,8.95890946e-01,
                     7.94076736e-01,8.11671368e-01,8.20118286e-01,
                     6.54111768e-01,2.30888169e-01,1.62639307e-02,
                     0.00000000e+00]))

    S2A_MSI_08 = (60, 0.7600, 0.9075, 
                     np.array([6.72586786e-04,0.00000000e+00,0.00000000e+00,
                     0.00000000e+00,0.00000000e+00,1.44779746e-04,
                     1.75239146e-02,6.81427309e-02,1.95910652e-01,
                     5.03477951e-01,8.03038519e-01,9.70893721e-01,
                     9.88406528e-01,9.39513050e-01,9.53957178e-01,
                     9.82554949e-01,9.83911811e-01,9.77666900e-01,
                     9.87645575e-01,9.83475703e-01,9.57210894e-01,
                     9.06465733e-01,8.51603307e-01,7.91559886e-01,
                     7.49627707e-01,7.26753218e-01,7.25638556e-01,
                     7.19227875e-01,7.08269542e-01,7.00091470e-01,
                     7.18240007e-01,7.58295756e-01,7.96838100e-01,
                     8.03464289e-01,7.89203043e-01,7.58190619e-01,
                     7.22623995e-01,6.74426662e-01,6.30051247e-01,
                     5.96192744e-01,5.71365058e-01,5.49016706e-01,
                     5.35102151e-01,5.30005852e-01,5.32114628e-01,
                     5.32889818e-01,5.30510552e-01,5.31251103e-01,
                     5.42353442e-01,5.57670949e-01,5.48219844e-01,
                     4.92873763e-01,4.27984659e-01,4.00632204e-01,
                     4.20865558e-01,4.07967544e-01,2.57205092e-01,
                     9.34112378e-02,2.59315420e-02,1.97162898e-03]))

    S2A_MSI_8A = (61, 0.8370, 0.8820, 
                     np.array([3.00966201e-04,0.00000000e+00,0.00000000e+00,
                     0.00000000e+00,1.57217434e-03,1.97345125e-02,
                     1.07797093e-01,5.03542566e-01,9.32244502e-01,
                     9.75417653e-01,9.76078281e-01,9.88891962e-01,
                     9.99846733e-01,9.95645780e-01,9.96407621e-01,
                     6.15518275e-01,1.12212459e-01,1.67962607e-02,
                     1.08051589e-15]))

    S2A_MSI_09 = (62, 0.9320, 0.9595, 
                     np.array([0.01662953,0.28176774,0.87454114,0.97842462,1.        ,
                     0.98646234,0.99428313,0.9642169 ,0.92240308,0.58699885,
                     0.06534121,0.        ]))

    S2A_MSI_10 = (63, 1.3370, 1.4120, 
                     np.array([2.38032000e-04,2.95805000e-05,1.09542000e-04,
                     1.50071000e-04,3.71868000e-04,8.73727500e-04,
                     3.73628100e-03,3.17506340e-02,2.00854803e-01,
                     5.84484875e-01,8.61210463e-01,9.52070852e-01,
                     9.81720971e-01,9.94644797e-01,1.00000000e+00,
                     9.93502876e-01,9.78085355e-01,9.54744576e-01,
                     9.15013686e-01,8.49689704e-01,6.55300550e-01,
                     2.85381533e-01,6.83716420e-02,9.59380350e-03,
                     1.12529200e-03,2.63485500e-04,1.96998000e-04,
                     5.54815000e-05,5.61960000e-05,7.65570000e-05,
                     2.54510000e-05]))

    S2A_MSI_11 = (64, 1.5390, 1.6840, 
                     np.array([6.93400000e-06,1.81645000e-05,9.07460000e-05,
                     3.65801000e-04,6.62261000e-04,1.42506250e-03,
                     3.56747000e-03,1.04731085e-02,2.77806180e-02,
                     6.48974280e-02,1.35475748e-01,2.89005843e-01,
                     5.37634456e-01,7.79371857e-01,8.79668577e-01,
                     8.90257398e-01,9.06630520e-01,9.30950552e-01,
                     9.47506732e-01,9.55341795e-01,9.60687866e-01,
                     9.66988645e-01,9.73379886e-01,9.79229624e-01,
                     9.80912334e-01,9.80974468e-01,9.81066319e-01,
                     9.86464474e-01,9.92015684e-01,9.91470684e-01,
                     9.83502057e-01,9.74371426e-01,9.67709002e-01,
                     9.65949699e-01,9.69869577e-01,9.81077488e-01,
                     9.91875079e-01,9.98077305e-01,9.99234469e-01,
                     9.99420043e-01,1.00000000e+00,9.94156469e-01,
                     9.78569228e-01,9.57091545e-01,9.41712152e-01,
                     9.31801621e-01,8.80597830e-01,6.92504394e-01,
                     4.26963072e-01,1.99127837e-01,7.24778960e-02,
                     2.43795985e-02,9.31685100e-03,4.13662100e-03,
                     1.95831500e-03,7.76503500e-04,1.59180000e-04,
                     2.89595000e-05,0.00000000e+00]))

    S2A_MSI_12 = (65, 2.0780, 2.3205, 
                     np.array([6.39264000e-04,3.44108750e-03,7.65982600e-03,
                     8.92000050e-03,1.09721780e-02,1.41907265e-02,
                     1.97803200e-02,3.04417690e-02,4.88826620e-02,
                     8.18549335e-02,1.36902990e-01,2.23166080e-01,
                     3.27177957e-01,4.26390454e-01,5.03106894e-01,
                     5.50302429e-01,5.79029932e-01,6.04436146e-01,
                     6.34282516e-01,6.71876803e-01,7.11964139e-01,
                     7.49353252e-01,7.85580740e-01,8.14689405e-01,
                     8.36721896e-01,8.52179769e-01,8.56174190e-01,
                     8.60495823e-01,8.65138808e-01,8.71999165e-01,
                     8.82438825e-01,8.91349430e-01,8.99896619e-01,
                     9.05177679e-01,9.05502587e-01,9.04432342e-01,
                     9.02760831e-01,9.02703264e-01,9.05315323e-01,
                     9.09107627e-01,9.15584774e-01,9.22819789e-01,
                     9.29740263e-01,9.35024139e-01,9.37641167e-01,
                     9.39046421e-01,9.43259037e-01,9.41374377e-01,
                     9.36660033e-01,9.29951119e-01,9.15482617e-01,
                     9.07788182e-01,9.11168883e-01,9.21729901e-01,
                     9.19489205e-01,9.26129478e-01,9.36769555e-01,
                     9.49002300e-01,9.57046915e-01,9.63698163e-01,
                     9.66647037e-01,9.65150671e-01,9.61698492e-01,
                     9.56233764e-01,9.52731790e-01,9.51938341e-01,
                     9.54056643e-01,9.62430392e-01,9.72691041e-01,
                     9.85021423e-01,9.93982826e-01,9.99376452e-01,
                     9.99151921e-01,9.91784832e-01,9.76212761e-01,
                     9.53391519e-01,9.27724853e-01,8.99933858e-01,
                     8.68647160e-01,8.33317814e-01,7.91866591e-01,
                     7.34726119e-01,6.51282733e-01,5.43500424e-01,
                     4.29259293e-01,3.21385902e-01,2.27703987e-01,
                     1.56384870e-01,1.08237331e-01,7.69315535e-02,
                     5.49854480e-02,4.07311725e-02,3.19988480e-02,
                     2.64975445e-02,2.16716920e-02,1.36598615e-02,
                     6.51060100e-03,1.03255350e-03]))

    # Sentinel-3A OLCI spectral response functions
    # changed to center at the max value of each  
    # band rsr from https://sentinel.esa.int/web/sentinel/technical-guides/sentinel-3-olci/olci-instrument/spectral-response-function-data
    # the mean dataset
    S3A_OLCI_01 = (66, 0.3850, 0.4125, 
                    np.array([0.00000000e+00,0.00000000e+00,1.44698925e-03,
                    3.90075197e-01,6.64325016e-01,7.79951131e-01,
                    9.02794193e-01,9.59328741e-01,9.98980457e-01,
                    2.15052943e-01,6.91049895e-05,0.00000000e+00]))

    S3A_OLCI_02 = (67, 0.4000, 0.4225, 
                    np.array([0.00000000e+00,0.00000000e+00,5.67130629e-03,
                    7.80520608e-01,9.92336528e-01,9.98163685e-01,
                    9.64577067e-01,1.62142192e-01,3.65144365e-05,
                    0.00000000e+00]))

    S3A_OLCI_03 = (68, 0.4300, 0.4525, 
                    np.array([0.00000000e+00,0.00000000e+00,8.82327515e-05,
                    2.50507297e-01,9.84845256e-01,9.98648197e-01,
                    9.95748807e-01,7.25352741e-01,3.29665330e-03,
                    0.00000000e+00]))

    S3A_OLCI_04 = (69, 0.4775, 0.5000, 
                    np.array([0.00000000e+00,0.00000000e+00,8.88119025e-05,
                    2.58959751e-01,9.67644787e-01,9.78045026e-01,
                    9.96403711e-01,7.45344017e-01,3.57228954e-03,
                    0.00000000e+00]))

    S3A_OLCI_05 = (70, 0.4975, 0.5200, 
                    np.array([0.00000000e+00,0.00000000e+00,9.03196596e-05,
                    2.61921725e-01,9.68992890e-01,9.86484682e-01,
                    9.97900957e-01,7.26184125e-01,3.04349960e-03,
                    0.00000000e+00]))

    S3A_OLCI_06 = (71, 0.5475, 0.5700, 
                    np.array([0.00000000e+00,0.00000000e+00,9.12138310e-05,
                    2.71055434e-01,9.85744910e-01,9.93709995e-01,
                    9.99188281e-01,7.36202217e-01,3.20926544e-03,
                    0.00000000e+00]))

    S3A_OLCI_07 = (72, 0.6075, 0.6300, 
                    np.array([0.00000000e+00,0.00000000e+00,9.50077324e-05,
                    2.77317100e-01,9.76962724e-01,9.87228638e-01,
                    9.97972350e-01,6.98892119e-01,2.26772837e-03,
                    0.00000000e+00]))

    S3A_OLCI_08 = (73, 0.6525, 0.6750, 
                    np.array([0.00000000e+00,0.00000000e+00,1.70652583e-04,
                    3.53389797e-01,9.80447862e-01,9.91507321e-01,
                    9.97394347e-01,6.44129999e-01,1.36300229e-03,
                    0.00000000e+00]))

    S3A_OLCI_09 = (74, 0.6625, 0.6825, 
                    np.array([0.00000000e+00,0.00000000e+00,1.66013218e-04,
                    3.54497132e-01,9.91183060e-01,9.96152868e-01,
                    6.60733265e-01,1.52801384e-03,0.00000000e+00]))

    S3A_OLCI_10 = (75, 0.6700, 0.6900, 
                    np.array([0.00000000e+00,0.00000000e+00,1.38022478e-04,
                    3.33541643e-01,9.85871946e-01,9.97184790e-01,
                    6.83203266e-01,1.78238692e-03,0.00000000e+00]))

    S3A_OLCI_11 = (76, 0.6950, 0.7175, 
                    np.array([0.00000000e+00,0.00000000e+00,1.29485319e-07,
                    1.33565896e-02,8.95928951e-01,9.96556200e-01,
                    9.97534770e-01,9.86969653e-01,1.02877809e-01,
                    6.99836795e-06]))

    S3A_OLCI_12 = (77, 0.7425, 0.7625, 
                    np.array([0.00000000e+00,0.00000000e+00,5.73078846e-05,
                    2.62775426e-01,9.96285919e-01,9.94959949e-01,
                    7.44938946e-01,2.43649259e-03,0.00000000e+00]))

    S3A_OLCI_13 = (78, 0.7550, 0.7700, 
                    np.array([0.00000000e+00,4.89582545e-05,2.62646899e-01,
                    8.24031122e-01,3.06534953e-03,0.00000000e+00,
                    0.00000000e+00]))

    S3A_OLCI_14 = (79, 0.7575, 0.7725, 
                    np.array([0.00000000e+00,4.38488689e-05,2.43677851e-01,
                    9.97014247e-01,1.17088589e-01,7.61572220e-06,
                    0.00000000e+00]))

    S3A_OLCI_15 = (80, 0.7600, 0.7750, 
                    np.array([0.00000000e+00,5.87943065e-08,1.02946446e-02,
                    9.49694946e-01,1.19850202e-01,7.39186874e-06,
                    0.00000000e+00]))

    S3A_OLCI_16 = (81, 0.7625, 0.7900, 
                    np.array([0.00000000e+00,0.00000000e+00,4.54725243e-08,
                    8.66502455e-03,8.54786953e-01,9.76876743e-01,
                    9.86449152e-01,9.97465390e-01,9.97967879e-01,
                    9.87477514e-01,1.31193176e-01,9.42330409e-06]))

    S3A_OLCI_17 = (82, 0.8475, 0.8800, 
                    np.array([0.00000000e+00,0.00000000e+00,1.98596616e-05,
                    1.94106940e-01,9.98422526e-01,9.92162445e-01,
                    9.82368240e-01,9.72067967e-01,9.59774999e-01,
                    9.47635726e-01,9.35168865e-01,7.55614274e-01,
                    3.37197717e-03,0.00000000e+00]))

    S3A_OLCI_18 = (83, 0.8725, 0.8950, 
                    np.array([0.00000000e+00,1.20008056e-08,4.44776154e-03,
                    8.44464399e-01,9.95280744e-01,9.81079588e-01,
                    9.61809761e-01,1.58916299e-01,1.48886924e-05,
                    0.00000000e+00]))

    S3A_OLCI_19 = (84, 0.8875, 0.9100, 
                    np.array([0.00000000e+00,1.21562504e-08,4.26317347e-03,
                    8.40394746e-01,9.94028495e-01,9.76434111e-01,
                    9.53033360e-01,1.61291816e-01,1.75069782e-05,
                    0.00000000e+00]))

    S3A_OLCI_20 = (85, 0.9225, 0.9550, 
                    np.array([0.00000000e+00,0.00000000e+00,4.38829991e-03,
                    8.39806175e-01,9.85099657e-01,9.48581070e-01,
                    9.12425617e-01,8.77435250e-01,8.52489956e-01,
                    8.32325676e-01,8.08894612e-01,1.56714011e-01,
                    3.28157749e-05,0.00000000e+00]))

    S3A_OLCI_21 = (86, 0.9925, 1.0450, 
                    np.array([0.00000000e+00,0.00000000e+00,5.95267671e-03,
                    8.29963638e-01,9.76125513e-01,9.17196169e-01,
                    8.59092046e-01,8.01654231e-01,7.43763056e-01,
                    6.86917212e-01,6.30841661e-01,5.78254852e-01,
                    5.46055402e-01,5.16377240e-01,4.87388894e-01,
                    4.59188029e-01,4.31652996e-01,4.04096076e-01,
                    3.73472735e-01,6.44893266e-02,8.19798435e-05,
                    0.00000000e+00]))
    # Sentinel-3A SLSTR spectral response functions
    # (PREFLIGHT)
    # Code used to create these things in
    # https://github.com/jgomezdans/sentinel_SRF

    S3A_SLSTR_01 = (87, 0.54000, 0.57000,
                    np.array(
                        [0.01463, 0.14031, 0.57941, 0.95383, 0.99927, 0.98361,
                         0.98065, 0.99224, 0.86356, 0.74371, 0.25806, 0.05162,
                         0.00906, ]))

    S3A_SLSTR_02 = (88, 0.64500, 0.67500,
                    np.array(
                        [0.00868, 0.10051, 0.54166, 0.94638, 0.97952, 0.99970,
                         0.99593, 0.98108, 0.98249, 0.81191, 0.28596, 0.03422,
                         0.00557, ]))

    S3A_SLSTR_03 = (89, 0.85250, 0.88500,
                    np.array(
                        [0.01457, 0.09982, 0.45901, 0.94725, 0.99882, 0.99531,
                         0.97779, 0.89514, 0.91546, 0.92751, 0.67702, 0.17794,
                         0.02665, 0.00552, ]))

    S3A_SLSTR_04 = (90, 1.36250, 1.39000,
                    np.array(
                        [0.08117, 0.62916, 0.97652, 0.96779, 0.98766, 0.99534,
                         0.99847, 0.99996, 0.95454, 0.51479, 0.08519,
                         0.00576, ]))

    S3A_SLSTR_05 = (91, 1.56500, 1.66250,
                    np.array(
                        [0.00951, 0.02064, 0.04194, 0.08240, 0.16215, 0.30042,
                         0.49672, 0.70549, 0.85567, 0.93327, 0.96306, 0.97290,
                         0.97634, 0.98050, 0.98407, 0.98816, 0.99064, 0.99422,
                         0.99715, 0.99887, 0.99974, 0.99874, 0.99389, 0.98591,
                         0.97221, 0.95534, 0.93706, 0.92338, 0.92452, 0.93923,
                         0.95034, 0.89146, 0.68978, 0.41600, 0.21213, 0.10306,
                         0.05085, 0.02604, 0.01343, 0.00755, ]))

    S3A_SLSTR_06 = (92, 2.21750, 2.29750,
                    np.array(
                        [0.01257, 0.03348, 0.07603, 0.15844, 0.30026, 0.50042,
                         0.71448, 0.87789, 0.95880, 0.97991, 0.97956, 0.97917,
                         0.98339, 0.98979, 0.99621, 0.99969, 0.99941, 0.99572,
                         0.99031, 0.98586, 0.98433, 0.98171, 0.96554, 0.91002,
                         0.78868, 0.60765, 0.41211, 0.25045, 0.14086, 0.07530,
                         0.03827, 0.01809, 0.00721, ]))
    
    # Sentinel-2B MSI spectral response functions
    # Taken from https://earth.esa.int/web/sentinel/user-guides/sentinel-2-msi/document-library/-/asset_publisher/Wk0TKajiISaR/content/sentinel-2a-spectral-responses
    S2B_MSI_01 = (93, 0.4110, 0.4585, 
                     np.array([6.79779800e-03,5.38819850e-03,7.13011000e-03,
                     5.32861200e-03,6.21707000e-04,4.19444050e-03,
                     2.58671000e-03,8.79308600e-03,1.88722410e-01,
                     7.88710046e-01,8.81448354e-01,9.26045092e-01,
                     9.50410310e-01,9.84288926e-01,9.76368067e-01,
                     9.38960136e-01,8.09016274e-01,2.07646870e-01,
                     8.93513100e-03,0.00000000e+00]))

    S2B_MSI_02 = (94, 0.4380, 0.5330, 
                     np.array([0.00189331,0.01708433,0.01544887,0.0138867 ,0.01450069,
                     0.01344492,0.01471763,0.0430435 ,0.25226095,0.71496001,
                     0.8320771 ,0.85144357,0.90569382,0.9338208 ,0.91662274,
                     0.90050851,0.92085944,0.91341249,0.88735796,0.85194722,
                     0.85792477,0.91177803,0.96940661,0.99966032,0.98416305,
                     0.97523217,0.95355824,0.917184  ,0.8922514 ,0.89550827,
                     0.9246363 ,0.96267905,0.99679122,0.97030435,0.7360636 ,
                     0.26382961,0.07513881,0.02364449,0.        ]))

    S2B_MSI_03 = (95, 0.5360, 0.5835, 
                     np.array([0.00184908,0.05892162,0.33349948,0.86269252,0.94917532,
                     0.92057652,0.90085865,0.93279058,0.98347041,0.99597786,
                     0.95115183,0.88000476,0.83920859,0.83301579,0.86656355,
                     0.86381757,0.67746396,0.18777902,0.02572436,0.        ]))

    S2B_MSI_04 = (96, 0.6460, 0.6860, 
                     np.array([5.09031200e-03,2.29151295e-01,8.10967523e-01,
                     9.23442650e-01,9.02005739e-01,8.99869183e-01,
                     9.06886173e-01,9.34405970e-01,9.66908428e-01,
                     9.95809639e-01,9.94046577e-01,9.63150260e-01,
                     9.25809770e-01,7.68490499e-01,2.76900547e-01,
                     1.86252340e-02,7.44413212e-16]))

    S2B_MSI_05 = (97, 0.6940, 0.7140, 
                     np.array([0.01047183,0.38095721,0.96844659,0.99754666,0.9794087 ,
                     0.95094835,0.89455748,0.34237254,0.00243694]))

    S2B_MSI_06 = (98, 0.7300, 0.7500, 
                     np.array([0.01743381,0.58418185,0.91641661,0.93978171,0.97855276,
                     0.99997792,0.81917496,0.05197745,0.        ]))

    S2B_MSI_07 = (99, 0.7660, 0.7960, 
                     np.array([0.01028203,0.1849904 ,0.73021678,0.96816857,0.96492559,
                     0.99902331,0.96569621,0.91842469,0.85951719,0.72377841,
                     0.23625194,0.00734482,0.        ]))

    S2B_MSI_08 = (100, 0.7740, 0.9090, 
                     np.array([3.86696000e-04,2.25160330e-02,8.62638800e-02,
                     3.41768486e-01,8.03109115e-01,9.42316692e-01,
                     9.63880684e-01,9.93998947e-01,9.98257939e-01,
                     9.86097031e-01,9.58222106e-01,9.45610993e-01,
                     9.46461415e-01,9.22310559e-01,8.73037871e-01,
                     8.30374120e-01,8.10955964e-01,7.87625180e-01,
                     7.54167357e-01,7.23279542e-01,7.18941021e-01,
                     7.44267400e-01,7.69900245e-01,7.83440215e-01,
                     7.81644143e-01,7.81572121e-01,7.80064535e-01,
                     7.67703381e-01,7.43324604e-01,7.17699319e-01,
                     6.88575227e-01,6.50641275e-01,6.14005803e-01,
                     5.91537372e-01,5.82889219e-01,5.77470921e-01,
                     5.72399696e-01,5.70158326e-01,5.72213154e-01,
                     5.78703807e-01,5.78302049e-01,5.65806515e-01,
                     5.49385012e-01,5.42215679e-01,5.37997281e-01,
                     5.13924174e-01,4.69610434e-01,4.44484344e-01,
                     4.47532506e-01,3.93848039e-01,2.35014942e-01,
                     1.02922870e-01,3.80060080e-02,1.04894310e-02,
                     0.00000000e+00]))

    S2B_MSI_8A = (101, 0.8480, 0.8805, 
                     np.array([1.66175100e-03,5.28325840e-02,3.45506138e-01,
                     8.66965557e-01,9.87241334e-01,9.99376033e-01,
                     9.94482469e-01,9.81491925e-01,9.72818786e-01,
                     9.50032949e-01,7.78588669e-01,2.81014567e-01,
                     3.33933370e-02,8.12020000e-04]))

    S2B_MSI_09 = (102, 0.9300, 0.9575, 
                     np.array([0.01212112,0.2550252 ,0.83734232,0.94936094,0.99744452,
                     0.980285  ,0.97377663,0.9278809 ,0.85513588,0.62111363,
                     0.09480081,0.00171671]))

    S2B_MSI_10 = (103, 1.3390, 1.4165, 
                     np.array([2.45190000e-05,1.06782500e-04,9.93550000e-05,
                     1.74974500e-04,3.22212000e-04,4.58060500e-04,
                     2.18105000e-03,1.89617750e-02,1.11412066e-01,
                     3.87289252e-01,7.70430827e-01,9.59359012e-01,
                     9.93032255e-01,9.99720196e-01,9.90856204e-01,
                     9.73640319e-01,9.57577366e-01,9.43105961e-01,
                     9.36274454e-01,9.23628129e-01,8.31894981e-01,
                     5.67223531e-01,2.13253720e-01,3.33423315e-02,
                     3.16376500e-03,6.91521500e-04,1.94334000e-04,
                     2.03841500e-04,1.54509000e-04,1.61088500e-04,
                     1.19559000e-04,0.00000000e+00]))

    S2B_MSI_11 = (104, 1.5380, 1.6805, 
                     np.array([1.15000000e-05,1.25332000e-04,3.43776000e-04,
                     7.36988500e-04,1.42877600e-03,3.17141100e-03,
                     7.57268400e-03,2.19172390e-02,6.29047140e-02,
                     1.70251784e-01,3.80879812e-01,6.60381371e-01,
                     8.69866550e-01,9.55250846e-01,9.77057349e-01,
                     9.76114190e-01,9.71480798e-01,9.69396227e-01,
                     9.72736269e-01,9.79977731e-01,9.84837133e-01,
                     9.83646395e-01,9.79391949e-01,9.75827865e-01,
                     9.76805245e-01,9.83603737e-01,9.92331977e-01,
                     9.95900604e-01,9.92101664e-01,9.83005073e-01,
                     9.73545660e-01,9.70486171e-01,9.75244492e-01,
                     9.85528272e-01,9.96435703e-01,9.99518223e-01,
                     9.93293536e-01,9.83975876e-01,9.79473331e-01,
                     9.84232683e-01,9.91723341e-01,9.85747311e-01,
                     9.53186779e-01,8.96044076e-01,8.50889881e-01,
                     8.45236530e-01,8.40540222e-01,7.13618196e-01,
                     4.57566037e-01,2.16596874e-01,7.35127680e-02,
                     2.38199245e-02,8.78944000e-03,3.97113050e-03,
                     1.99073900e-03,9.53045500e-04,2.81204000e-04,
                     0.00000000e+00]))

    S2B_MSI_12 = (105, 2.0650, 2.3050, 
                     np.array([2.27567000e-04,2.33818100e-03,5.92493500e-03,
                     9.50674450e-03,1.45934790e-02,2.32456505e-02,
                     3.72344570e-02,6.31897490e-02,1.10694417e-01,
                     1.93831191e-01,3.17227795e-01,4.86127891e-01,
                     6.72128213e-01,8.14294745e-01,8.86329697e-01,
                     9.01154288e-01,8.92352598e-01,8.83263626e-01,
                     8.82697820e-01,8.95818268e-01,9.12841006e-01,
                     9.30003885e-01,9.43377038e-01,9.47987900e-01,
                     9.48111791e-01,9.45498813e-01,9.41894052e-01,
                     9.43759006e-01,9.46212496e-01,9.48440897e-01,
                     9.50676414e-01,9.51929392e-01,9.52552047e-01,
                     9.52808031e-01,9.53779111e-01,9.55188146e-01,
                     9.57986198e-01,9.60932208e-01,9.64122014e-01,
                     9.62929248e-01,9.60730243e-01,9.57674768e-01,
                     9.53076144e-01,9.55447935e-01,9.55356546e-01,
                     9.52789244e-01,9.48594073e-01,9.48422211e-01,
                     9.46620434e-01,9.42809458e-01,9.38918576e-01,
                     9.44765225e-01,9.48481875e-01,9.51379486e-01,
                     9.55262594e-01,9.50105202e-01,9.49712169e-01,
                     9.57395090e-01,9.73886163e-01,9.68734909e-01,
                     9.67555279e-01,9.68732559e-01,9.69967732e-01,
                     9.67903993e-01,9.65459532e-01,9.62297699e-01,
                     9.59368085e-01,9.61002221e-01,9.63700810e-01,
                     9.65931017e-01,9.68868150e-01,9.73363783e-01,
                     9.77203216e-01,9.80985309e-01,9.83336508e-01,
                     9.87941136e-01,9.93184480e-01,9.97493542e-01,
                     1.00000000e+00,9.93179265e-01,9.68715091e-01,
                     9.15824428e-01,8.31947776e-01,7.09102747e-01,
                     5.64024643e-01,4.25931844e-01,3.09839746e-01,
                     2.18177603e-01,1.49356419e-01,1.04215226e-01,
                     7.40635240e-02,5.28513985e-02,3.79860860e-02,
                     2.70345635e-02,1.38379210e-02,2.64583650e-03,
                     0.00000000e+00]))
    
    
    # Sentinel-3B OLCI spectral response functions
    # changed to center at the max value of each  
    # band rsr from https://sentinel.esa.int/web/sentinel/technical-guides/sentinel-3-olci/olci-instrument/spectral-response-function-data
    # the mean dataset
    S3B_OLCI_01 = (106, 0.3850, 0.4125, 
                    np.array([0.00000000e+00,0.00000000e+00,9.34810332e-05,
                    3.01161089e-01,5.79783356e-01,7.28688965e-01,
                    8.90776697e-01,9.53314424e-01,9.97316285e-01,
                    2.06989277e-01,4.47201260e-06,0.00000000e+00]))

    S3B_OLCI_02 = (107, 0.4000, 0.4225, 
                    np.array([0.00000000e+00,0.00000000e+00,1.19249516e-03,
                    7.79853543e-01,9.87822759e-01,9.98838473e-01,
                    9.87925591e-01,1.65533011e-01,3.38914146e-06,
                    0.00000000e+00]))

    S3B_OLCI_03 = (108, 0.4300, 0.4525, 
                    np.array([0.00000000e+00,0.00000000e+00,9.30282695e-06,
                    2.06052543e-01,9.98428589e-01,9.97182974e-01,
                    9.85843591e-01,7.51246618e-01,1.85549607e-03,
                    0.00000000e+00]))

    S3B_OLCI_04 = (109, 0.4775, 0.5000, 
                    np.array([0.00000000e+00,0.00000000e+00,4.06074021e-05,
                    2.85736497e-01,9.80498411e-01,9.87358384e-01,
                    9.97301845e-01,7.10387496e-01,1.51227169e-03,
                    0.00000000e+00]))

    S3B_OLCI_05 = (110, 0.4975, 0.5200, 
                    np.array([0.00000000e+00,0.00000000e+00,4.63640882e-05,
                    2.88108060e-01,9.70176583e-01,9.82820364e-01,
                    9.96720776e-01,7.03952046e-01,1.49862288e-03,
                    0.00000000e+00]))

    S3B_OLCI_06 = (111, 0.5475, 0.5700, 
                    np.array([0.00000000e+00,0.00000000e+00,5.04591202e-05,
                    3.02121393e-01,9.87815717e-01,9.94139281e-01,
                    9.99052182e-01,6.99925598e-01,1.38906899e-03,
                    0.00000000e+00]))

    S3B_OLCI_07 = (112, 0.6075, 0.6300, 
                    np.array([0.00000000e+00,0.00000000e+00,5.24770401e-05,
                    3.28445148e-01,9.78249177e-01,9.86453480e-01,
                    9.96904960e-01,6.37481108e-01,6.57750246e-04,
                    0.00000000e+00]))

    S3B_OLCI_08 = (113, 0.6525, 0.6750, 
                    np.array([0.00000000e+00,0.00000000e+00,1.02038317e-04,
                    4.23731626e-01,9.80510445e-01,9.89802816e-01,
                    9.95992727e-01,5.64721221e-01,2.87047474e-04,
                    0.00000000e+00]))

    S3B_OLCI_09 = (114, 0.6650, 0.6850, 
                    np.array([0.00000000e+00,1.01088488e-04,4.30582345e-01,
                    9.90586594e-01,9.95627137e-01,5.70430795e-01,
                    2.99212934e-04,0.00000000e+00,0.00000000e+00]))

    S3B_OLCI_10 = (115, 0.6700, 0.6900, 
                    np.array([0.00000000e+00,0.00000000e+00,9.20443879e-05,
                    4.24395382e-01,9.93044213e-01,9.96732724e-01,
                    5.87914764e-01,3.33714303e-04,0.00000000e+00]))

    S3B_OLCI_11 = (116, 0.6950, 0.7175, 
                    np.array([0.00000000e+00,0.00000000e+00,2.27527154e-08,
                    1.43013420e-02,9.30039754e-01,9.89869809e-01,
                    9.95153682e-01,9.85467850e-01,6.04404649e-02,
                    4.07944824e-07]))

    S3B_OLCI_12 = (117, 0.7425, 0.7625, 
                    np.array([0.00000000e+00,0.00000000e+00,3.32680058e-05,
                    3.35481137e-01,9.96134240e-01,9.95335488e-01,
                    6.74603784e-01,5.81641718e-04,0.00000000e+00]))

    S3B_OLCI_13 = (118, 0.7550, 0.7700, 
                    np.array([0.00000000e+00,2.85560325e-05,3.35218844e-01,
                    7.33046075e-01,6.97655972e-04,0.00000000e+00,
                    0.00000000e+00]))

    S3B_OLCI_14 = (119, 0.7550, 0.7725, 
                    np.array([0.00000000e+00,0.00000000e+00,2.51718059e-05,
                    3.10743254e-01,9.93217466e-01,7.62547197e-02,
                    6.40188179e-07,0.00000000e+00]))

    S3B_OLCI_15 = (120, 0.7600, 0.7750, 
                    np.array([0.00000000e+00,9.44449935e-09,9.67829515e-03,
                    9.72499368e-01,8.17682936e-02,7.17736029e-07,
                    0.00000000e+00]))

    S3B_OLCI_16 = (121, 0.7650, 0.7925, 
                    np.array([0.00000000e+00,8.64377579e-09,8.84274122e-03,
                    9.17740240e-01,9.96150751e-01,9.96332905e-01,
                    9.96195066e-01,9.94333040e-01,9.83695296e-01,
                    8.05458528e-02,7.66316013e-07,0.00000000e+00]))

    S3B_OLCI_17 = (122, 0.8475, 0.8800, 
                    np.array([0.00000000e+00,0.00000000e+00,1.47858085e-05,
                    2.54914009e-01,9.99170692e-01,9.92365948e-01,
                    9.84295400e-01,9.75537469e-01,9.62032596e-01,
                    9.47783211e-01,9.33310423e-01,6.88572549e-01,
                    1.08104259e-03,0.00000000e+00]))

    S3B_OLCI_18 = (123, 0.8700, 0.8925, 
                    np.array([0.00000000e+00,0.00000000e+00,5.01495276e-09,
                    6.26933249e-03,9.02104161e-01,9.91985752e-01,
                    9.75609451e-01,9.53379357e-01,1.00137591e-01,
                    1.48780466e-06]))

    S3B_OLCI_19 = (124, 0.8850, 0.9075, 
                    np.array([0.00000000e+00,0.00000000e+00,4.91117993e-09,
                    6.14792354e-03,9.01922678e-01,9.90850645e-01,
                    9.72235276e-01,9.44141158e-01,1.00942430e-01,
                    1.52794114e-06]))

    S3B_OLCI_20 = (125, 0.9225, 0.9550, 
                    np.array([0.00000000e+00,3.42848643e-09,5.17534750e-03,
                    8.95989159e-01,9.82463232e-01,9.47765152e-01,
                    9.13370826e-01,8.80142365e-01,8.52126137e-01,
                    8.28556200e-01,8.01942644e-01,1.01521422e-01,
                    1.74331314e-06,0.00000000e+00]))

    S3B_OLCI_21 = (126, 0.9925, 1.0450, 
                    np.array([0.00000000e+00,0.00000000e+00,2.16191906e-03,
                    8.62689484e-01,9.74315053e-01,9.15567438e-01,
                    8.57471704e-01,7.98987260e-01,7.41776209e-01,
                    6.85363224e-01,6.29509263e-01,5.76679778e-01,
                    5.44270496e-01,5.14648842e-01,4.85713953e-01,
                    4.57540874e-01,4.29559156e-01,4.02004281e-01,
                    3.74596536e-01,4.46248175e-02,4.02350000e-07,
                    0.00000000e+00]))

    # Sentinel-3B SLSTR spectral response functions
    # (PREFLIGHT)
    # Code used to create these things in
    # https://github.com/jgomezdans/sentinel_SRF
    S3B_SLSTR_01 = (114, 0.54250, 0.57000,
		     np.array([0.01206, 0.27425, 0.80178, 0.93965,
		     0.95802, 0.96983, 0.98111, 0.93155, 0.98239,
		     0.51100, 0.12423, 0.01377, ]))

    S3B_SLSTR_02 = (115, 0.645000, 0.67500,
		     np.array([0.00923, 0.03648, 0.39244, 0.98581,
		     0.98523, 0.99845, 0.98163, 0.94989, 0.92945,
		     0.88710, 0.46271, 0.05538, 0.00904, ]))

    S3B_SLSTR_03 = (116, 0.84750, 0.887500,
		     np.array([0.00812, 0.00967, 0.02361, 0.10105,
		     0.58329, 0.96260, 0.96818, 0.94752, 0.89536,
		     0.96613, 0.98339, 0.99422, 0.63677, 0.12622,
		     0.02013, 0.00702, 0.00674, ]))

    S3B_SLSTR_04 = (117, 1.36250, 1.39000,
		     np.array([0.06640, 0.55261, 0.98999, 0.97825,
		     0.99820, 0.99424, 0.99060, 0.99198,
		     0.96408, 0.59000, 0.11305, 0.01122, ]))

    S3B_SLSTR_05 = (118, 1.56500, 1.65750, 
		     np.array([0.01293, 0.03096, 0.06178, 0.12220,
		     0.23370, 0.40437, 0.61348, 0.80127, 0.91617,
		     0.97052, 0.98583, 0.99257, 0.99218, 0.99289,
		     0.99469, 0.99441, 0.99995, 0.99743, 0.99660,
		     0.99717, 0.99956, 0.99335, 0.98563, 0.97131,
		     0.95391, 0.93399, 0.91583, 0.90968, 0.91672,
		     0.93473, 0.92294, 0.79910, 0.54450, 0.29666,
		     0.14590, 0.06893, 0.03427, 0.01641, ]))

    S3B_SLSTR_06 = (119, 2.21750, 2.29250, 
		     np.array([0.01990, 0.05259, 0.12066, 0.25159, 
		     0.45970, 0.70491, 0.89585, 0.98337, 1.00001,
		     0.99366, 0.98682, 0.98293, 0.98233, 0.98105,
		     0.97767, 0.97383, 0.97063, 0.96946, 0.97025,
		     0.97293, 0.97185, 0.94923, 0.87905, 0.74452,
		     0.56139, 0.37587, 0.22824, 0.12957, 0.07039,
		     0.03621, 0.01760, ]))

    # Redefined MODIS TERRA and AQUA bands to a more accurate spectral
    # sampling actually measured from each satellite
    # and also added the ocean bands
    # I'm using the spectra found on
    # * TERRA: http://oceancolor.gsfc.nasa.gov/DOCS/RSR/HMODIST_RSRs.txt
    # * AQUA: http://oceancolor.gsfc.nasa.gov/DOCS/RSR/HMODISA_RSRs.txt
    # And the code that creates the bands is available here:
    # https://gist.github.com/jgomezdans/0cd6fc1537e5a76e5d3971ad167badd6
    ACCURATE_MODIS_TERRA_1 = (20, 0.61500, 0.68000,
                                      np.array(
                                          [0.02807, 0.14457, 0.37699, 0.63749,
                                           0.76990,
                                           0.78074, 0.78695, 0.79042, 0.79757,
                                           0.81003, 0.82398,
                                           0.83735, 0.85865, 0.89204, 0.91952,
                                           0.96325, 0.99380,
                                           0.99782, 0.95684, 0.83402, 0.72271,
                                           0.57700, 0.39879,
                                           0.22151, 0.09847, 0.04423, 0.01564,
                                           ]))

    ACCURATE_MODIS_TERRA_2 = (21, 0.82000, 0.89750,
                                      np.array(
                                          [0.01046, 0.02323, 0.04527, 0.08294,
                                           0.13231,
                                           0.20333, 0.29725, 0.45232, 0.64668,
                                           0.82575, 0.96775,
                                           0.99464, 0.98512, 0.97220, 0.96665,
                                           0.96893, 0.96614,
                                           0.98157, 0.98110, 0.96206, 0.91074,
                                           0.79448, 0.63300,
                                           0.40130, 0.20118, 0.10805, 0.05918,
                                           0.04157, 0.03105,
                                           0.02474, 0.01912, 0.01369,
                                           ]))

    ACCURATE_MODIS_TERRA_3 = (22, 0.45250, 0.48000,
                                      np.array(
                                          [0.02715, 0.19656, 0.66544, 0.98628,
                                           0.93702,
                                           0.85392, 0.87385, 0.92127, 0.85865,
                                           0.59029, 0.21396,
                                           0.02998,
                                           ]))

    ACCURATE_MODIS_TERRA_4 = (23, 0.54000, 0.56750,
                                      np.array(
                                          [0.04011, 0.24383, 0.65195, 0.94575,
                                           0.99606,
                                           0.99064, 0.98572, 0.97833, 0.93210,
                                           0.70581, 0.31567,
                                           0.05180,
                                           ]))

    ACCURATE_MODIS_TERRA_5 = (24, 1.21500, 1.27000,
                                       np.array(
                                           [0.01581, 0.03020, 0.05587, 0.10725,
                                            0.19486,
                                            0.33641, 0.50475, 0.70053, 0.84740,
                                            0.94598, 0.98540,
                                            0.99393, 0.97463, 0.88957, 0.76061,
                                            0.56817, 0.34327,
                                            0.20250, 0.10511, 0.05389, 0.03478,
                                            0.01910, 0.01206,
                                            ]))

    ACCURATE_MODIS_TERRA_6 = (25, 1.59750, 1.66000,
                                       np.array(
                                           [0.01510, 0.02596, 0.04589, 0.08177,
                                            0.14737,
                                            0.25125, 0.39260, 0.55170, 0.72028,
                                            0.88082, 0.98416,
                                            0.99335, 0.97750, 0.96290, 0.92923,
                                            0.86426, 0.74879,
                                            0.60357, 0.45702, 0.33580, 0.22364,
                                            0.11741, 0.06227,
                                            0.04009, 0.02624, 0.01373,
                                            ]))

    ACCURATE_MODIS_TERRA_7 = (26, 2.06000, 2.17500,
                                       np.array(
                                           [0.01090, 0.01563, 0.02103, 0.03128,
                                            0.04292,
                                            0.06585, 0.09211, 0.14750, 0.21146,
                                            0.31098, 0.42048,
                                            0.54758, 0.67979, 0.79133, 0.89692,
                                            0.94566, 0.97822,
                                            0.99110, 0.99837, 0.96789, 0.92654,
                                            0.89933, 0.87611,
                                            0.84122, 0.80300, 0.76182, 0.71977,
                                            0.68366, 0.64921,
                                            0.61062, 0.57081, 0.53225, 0.49405,
                                            0.44690, 0.39718,
                                            0.34026, 0.28131, 0.22290, 0.16465,
                                            0.12282, 0.08566,
                                            0.06288, 0.04430, 0.03179, 0.02100,
                                            0.01462, 0.01036,
                                            ]))

    ACCURATE_MODIS_TERRA_8 = (27, 0.40000, 0.42250,
                                      np.array(
                                          [0.01396, 0.13170, 0.59932, 0.67268,
                                           0.79013,
                                           0.48200, 0.89280, 0.82045, 0.24315,
                                           0.03272,
                                           ]))

    ACCURATE_MODIS_TERRA_9 = (28, 0.43500, 0.45000,
                                      np.array(
                                          [0.11465, 0.52805, 0.94626, 0.99552,
                                           0.91925,
                                           0.37027, 0.03831,
                                           ]))

    ACCURATE_MODIS_TERRA_10 = (29, 0.47750, 0.49500,
                                       np.array(
                                           [0.01416, 0.15956, 0.66826, 0.94806,
                                            0.99412,
                                            0.95485, 0.46874, 0.02232,
                                            ]))

    ACCURATE_MODIS_TERRA_11 = (30, 0.52000, 0.54000,
                                       np.array(
                                           [0.01480, 0.24145, 0.76250, 0.97399,
                                            0.99901,
                                            0.90343, 0.63643, 0.18019, 0.01435,
                                            ]))

    ACCURATE_MODIS_TERRA_12 = (31, 0.53750, 0.55500,
                                       np.array(
                                           [0.01470, 0.17968, 0.64797, 0.98082,
                                            0.99460,
                                            0.86391, 0.42490, 0.04215,
                                            ]))

    ACCURATE_MODIS_TERRA_13 = (32, 0.65750, 0.67500,
                                       np.array(
                                           [0.06338, 0.37724, 0.81862, 0.99604,
                                            0.88368,
                                            0.62629, 0.25126, 0.01371,
                                            ]))

    ACCURATE_MODIS_TERRA_14 = (33, 0.66750, 0.68750,
                                       np.array(
                                           [0.05618, 0.27689, 0.68142, 0.97341,
                                            0.98489,
                                            0.86696, 0.53632, 0.14128, 0.01578,
                                            ]))

    ACCURATE_MODIS_TERRA_15 = (34, 0.73750, 0.75500,
                                       np.array(
                                           [0.04077, 0.23050, 0.62169, 0.94550,
                                            0.97978,
                                            0.75860, 0.35374, 0.05302,
                                            ]))

    ACCURATE_MODIS_TERRA_16 = (35, 0.85250, 0.88000,
                                       np.array(
                                           [0.02770, 0.11693, 0.34382, 0.66672,
                                            0.91719,
                                            0.99906, 0.98123, 0.91350, 0.70598,
                                            0.41804, 0.10558,
                                            0.02056,
                                            ]))

    ACCURATE_MODIS_AQUA_1 = (36, 0.61500, 0.68000,
                                     np.array(
                                         [0.02807, 0.14457, 0.37699, 0.63749,
                                          0.76990,
                                          0.78074, 0.78695, 0.79042, 0.79757,
                                          0.81003, 0.82398,
                                          0.83735, 0.85865, 0.89204, 0.91952,
                                          0.96325, 0.99380,
                                          0.99782, 0.95684, 0.83402, 0.72271,
                                          0.57700, 0.39879,
                                          0.22151, 0.09847, 0.04423, 0.01564,
                                          ]))

    ACCURATE_MODIS_AQUA_2 = (37, 0.82000, 0.89750,
                                     np.array(
                                         [0.01046, 0.02323, 0.04527, 0.08294,
                                          0.13231,
                                          0.20333, 0.29725, 0.45232, 0.64668,
                                          0.82575, 0.96775,
                                          0.99464, 0.98512, 0.97220, 0.96665,
                                          0.96893, 0.96614,
                                          0.98157, 0.98110, 0.96206, 0.91074,
                                          0.79448, 0.63300,
                                          0.40130, 0.20118, 0.10805, 0.05918,
                                          0.04157, 0.03105,
                                          0.02474, 0.01912, 0.01369,
                                          ]))

    ACCURATE_MODIS_AQUA_3 = (38, 0.45250, 0.48000,
                                     np.array(
                                         [0.02715, 0.19656, 0.66544, 0.98628,
                                          0.93702,
                                          0.85392, 0.87385, 0.92127, 0.85865,
                                          0.59029, 0.21396,
                                          0.02998,
                                          ]))

    ACCURATE_MODIS_AQUA_4 = (39, 0.54000, 0.56750,
                                     np.array(
                                         [0.04011, 0.24383, 0.65195, 0.94575,
                                          0.99606,
                                          0.99064, 0.98572, 0.97833, 0.93210,
                                          0.70581, 0.31567,
                                          0.05180,
                                          ]))

    ACCURATE_MODIS_AQUA_5 = (40, 1.21500, 1.27000,
                                      np.array(
                                          [0.01581, 0.03020, 0.05587, 0.10725,
                                           0.19486,
                                           0.33641, 0.50475, 0.70053, 0.84740,
                                           0.94598, 0.98540,
                                           0.99393, 0.97463, 0.88957, 0.76061,
                                           0.56817, 0.34327,
                                           0.20250, 0.10511, 0.05389, 0.03478,
                                           0.01910, 0.01206,
                                           ]))

    ACCURATE_MODIS_AQUA_6 = (41, 1.59750, 1.66000,
                                      np.array(
                                          [0.01510, 0.02596, 0.04589, 0.08177,
                                           0.14737,
                                           0.25125, 0.39260, 0.55170, 0.72028,
                                           0.88082, 0.98416,
                                           0.99335, 0.97750, 0.96290, 0.92923,
                                           0.86426, 0.74879,
                                           0.60357, 0.45702, 0.33580, 0.22364,
                                           0.11741, 0.06227,
                                           0.04009, 0.02624, 0.01373,
                                           ]))

    ACCURATE_MODIS_AQUA_7 = (42, 2.06000, 2.17500,
                                      np.array(
                                          [0.01090, 0.01563, 0.02103, 0.03128,
                                           0.04292,
                                           0.06585, 0.09211, 0.14750, 0.21146,
                                           0.31098, 0.42048,
                                           0.54758, 0.67979, 0.79133, 0.89692,
                                           0.94566, 0.97822,
                                           0.99110, 0.99837, 0.96789, 0.92654,
                                           0.89933, 0.87611,
                                           0.84122, 0.80300, 0.76182, 0.71977,
                                           0.68366, 0.64921,
                                           0.61062, 0.57081, 0.53225, 0.49405,
                                           0.44690, 0.39718,
                                           0.34026, 0.28131, 0.22290, 0.16465,
                                           0.12282, 0.08566,
                                           0.06288, 0.04430, 0.03179, 0.02100,
                                           0.01462, 0.01036,
                                           ]))

    ACCURATE_MODIS_AQUA_8 = (43, 0.40250, 0.4225,
                                     np.array(
                                         [0.08730698,  0.5009796 ,  0.62325783,
                                          0.73623119,  0.52280358,  0.83059659,
                                          0.91752085,  0.32830619,  0.0332021
                                          ]))

    ACCURATE_MODIS_AQUA_9 = (44, 0.43250, 0.45000,
                                     np.array(
                                         [0.01902, 0.09698, 0.51329, 0.92505,
                                          0.99995,
                                          0.94153, 0.39117, 0.03671,
                                          ]))

    ACCURATE_MODIS_AQUA_10 = (45, 0.47750, 0.49500,
                                      np.array(
                                          [0.01964, 0.09723, 0.58592, 0.96018,
                                           0.98957,
                                           0.98879, 0.59678, 0.03983,
                                           ]))

    ACCURATE_MODIS_AQUA_11 = (46, 0.52000, 0.54000,
                                      np.array(
                                          [0.01215, 0.19041, 0.68607, 0.98495,
                                           0.99800,
                                           0.93087, 0.70338, 0.25414, 0.02052,
                                           ]))

    ACCURATE_MODIS_AQUA_12 = (47, 0.53750, 0.55500,
                                      np.array(
                                          [0.02111, 0.16232, 0.57148, 0.97131,
                                           0.99150,
                                           0.87360, 0.50867, 0.07715,
                                           ]))

    ACCURATE_MODIS_AQUA_13 = (48, 0.65750, 0.67500,
                                      np.array(
                                          [0.04175, 0.32634, 0.79150, 0.99988,
                                           0.89403,
                                           0.68671, 0.28090, 0.01440,
                                           ]))

    ACCURATE_MODIS_AQUA_14 = (49, 0.66750, 0.68750,
                                      np.array(
                                          [0.03422, 0.18745, 0.58587, 0.93366,
                                           0.99127,
                                           0.93329, 0.62790, 0.22864, 0.02571,
                                           ]))

    ACCURATE_MODIS_AQUA_15 = (50, 0.73500, 0.75500,
                                      np.array(
                                          [0.01266, 0.02854, 0.19037, 0.58331,
                                           0.92908,
                                           0.99234, 0.77888, 0.37822, 0.05551,
                                           ]))

    ACCURATE_MODIS_AQUA_16 = (51, 0.85250, 0.88000,
                                      np.array(
                                          [0.01859, 0.08363, 0.28236, 0.59771,
                                           0.87955,
                                           0.99942, 0.97670, 0.94504, 0.75962,
                                           0.47919, 0.14822,
                                           0.02755,
                                           ]))


  # PROBAV_CAMERA_BAND
    PROBAV_1_01 = (127, 0.43250, 0.49000,
                   np.array(
                       [0.00701, 0.01403, 0.29759, 0.76866,
                        0.85090, 0.85261, 0.86397, 0.86315,
                        0.87876, 0.89403, 0.90585, 0.91606,
                        0.86274, 0.87280, 0.89960, 0.92657,
                        0.90944, 0.91771, 0.92900, 0.95977,
                        0.98987, 0.93182, 0.23845, 0.01315, ]))
    PROBAV_2_01 = (128, 0.43750, 0.49000,
                   np.array(
                       [0.11753, 0.68816, 0.83312, 0.86384,
                        0.85535, 0.84936, 0.91241, 0.94894,
                        0.96350, 0.97903, 0.93099, 0.93834,
                        0.95654, 0.98125, 0.94097, 0.94453,
                        0.96126, 0.97506, 0.98012, 0.97479,
                        0.35092, 0.02275, ]))
    PROBAV_3_01 = (129, 0.43250, 0.49000,
                   np.array(
                       [0.00831, 0.01630, 0.26908, 0.75646,
                        0.84782, 0.85869, 0.87579, 0.87686,
                        0.92334, 0.94334, 0.97579, 0.98704,
                        0.92646, 0.93014, 0.95685, 0.97380,
                        0.94210, 0.94585, 0.95847, 0.97944,
                        0.99017, 0.94958, 0.26846, 0.01759, ]))
    PROBAV_1_02 = (130, 0.61250, 0.70000,
                   np.array(
                       [0.09219, 0.79037, 0.93792, 0.97300,
                        0.95793, 0.94941, 0.94262, 0.94200,
                        0.94969, 0.94499, 0.94095, 0.88377,
                        0.88352, 0.88640, 0.90464, 0.88917,
                        0.86092, 0.85622, 0.84943, 0.87536,
                        0.92689, 0.94288, 0.93296, 0.90110,
                        0.91734, 0.94771, 0.97572, 0.93463,
                        0.93636, 0.94238, 0.92302, 0.95991,
                        0.99355, 0.66343, 0.16410, 0.02510, ]))
    PROBAV_2_02 = (131, 0.61250, 0.70250,
                   np.array(
                       [0.05112, 0.63546, 0.94863, 0.99405,
                        0.98435, 0.98583, 0.97631, 0.97432,
                        0.99088, 0.99118, 0.97510, 0.94569,
                        0.92452, 0.93961, 0.94746, 0.92908,
                        0.91461, 0.88604, 0.88331, 0.90463,
                        0.94046, 0.96590, 0.95591, 0.91342,
                        0.91923, 0.94040, 0.95123, 0.94757,
                        0.93538, 0.93531, 0.92470, 0.95749,
                        0.98006, 0.65677, 0.16783, 0.02719,
                        0.00596, ]))
    PROBAV_3_02 = (132, 0.61000, 0.70250,
                   np.array(
                       [0.00693, 0.11182, 0.84741, 0.95372,
                        0.99692, 0.98211, 0.98042, 0.97529,
                        0.98737, 0.98246, 0.99340, 0.99401,
                        0.94669, 0.92033, 0.93551, 0.95167,
                        0.92947, 0.90131, 0.89177, 0.87599,
                        0.91253, 0.94400, 0.96551, 0.93722,
                        0.91432, 0.90637, 0.93886, 0.95319,
                        0.93674, 0.93029, 0.92951, 0.91037,
                        0.94481, 0.97393, 0.67624, 0.17566,
                        0.02900, 0.00671, ]))
    PROBAV_1_03 = (133, 0.75750, 0.92250,
                   np.array(
                       [0.00745, 0.01299, 0.02989, 0.07570,
                        0.18032, 0.38076, 0.67196, 0.90632,
                        0.99224, 0.98506, 0.97271, 0.96036,
                        0.95169, 0.94302, 0.93462, 0.92623,
                        0.91625, 0.90627, 0.90337, 0.90048,
                        0.89904, 0.89761, 0.88804, 0.87848,
                        0.87390, 0.86933, 0.86619, 0.86305,
                        0.85216, 0.84127, 0.83236, 0.82345,
                        0.81794, 0.81244, 0.80895, 0.80547,
                        0.79105, 0.77663, 0.76090, 0.74517,
                        0.74453, 0.74390, 0.72234, 0.70079,
                        0.68596, 0.67113, 0.65512, 0.63912,
                        0.62617, 0.61322, 0.60063, 0.58805,
                        0.58343, 0.57882, 0.56859, 0.55837,
                        0.53969, 0.52101, 0.50665, 0.49770,
                        0.48863, 0.47636, 0.42195, 0.28948,
                        0.13073, 0.04062, 0.01252, ]))
    PROBAV_2_03 = (134, 0.75750, 0.92500,
                   np.array(
                       [0.00675, 0.01079, 0.02156, 0.04730,
                        0.10797, 0.23157, 0.46667, 0.74601,
                        0.94127, 1.00000, 0.98746, 0.97493,
                        0.98095, 0.98698, 0.97258, 0.95819,
                        0.95082, 0.94345, 0.93537, 0.92730,
                        0.91901, 0.91072, 0.90696, 0.90321,
                        0.89724, 0.89127, 0.88293, 0.87458,
                        0.86800, 0.86141, 0.85505, 0.84869,
                        0.83384, 0.81899, 0.81435, 0.80971,
                        0.79835, 0.78699, 0.78169, 0.77639,
                        0.76419, 0.75199, 0.73214, 0.71230,
                        0.69639, 0.68048, 0.67196, 0.66344,
                        0.64948, 0.63552, 0.62135, 0.60718,
                        0.59919, 0.59120, 0.58136, 0.57152,
                        0.55307, 0.53462, 0.52212, 0.50539,
                        0.49683, 0.48331, 0.45572, 0.35616,
                        0.19922, 0.07619, 0.02540, 0.00794, ]))
    PROBAV_3_03 = (135, 0.75750, 0.92250,
                   np.array(
                       [0.01412, 0.02358, 0.04875, 0.11377,
                        0.24868, 0.48046, 0.76516, 0.95847,
                        0.99996, 0.97437, 0.96656, 0.95875,
                        0.94469, 0.93063, 0.92043, 0.91024,
                        0.90635, 0.90246, 0.90033, 0.89819,
                        0.88361, 0.86903, 0.86105, 0.85306,
                        0.84976, 0.84645, 0.83716, 0.82787,
                        0.82268, 0.81750, 0.80487, 0.79225,
                        0.78843, 0.78460, 0.78373, 0.78285,
                        0.76894, 0.75503, 0.75029, 0.74555,
                        0.73365, 0.72175, 0.70233, 0.68292,
                        0.67326, 0.66361, 0.65092, 0.63823,
                        0.62028, 0.60233, 0.59043, 0.57853,
                        0.57334, 0.56815, 0.55428, 0.54040,
                        0.52628, 0.51217, 0.49737, 0.48644,
                        0.48074, 0.46443, 0.38751, 0.24328,
                        0.09891, 0.02944, 0.00937, ]))
    PROBAV_1_04 = (136, 1.53250, 1.67250,
                   np.array(
                       [0.00534, 0.00696, 0.01016, 0.01336,
                        0.01887, 0.02438, 0.03556, 0.04675,
                        0.06724, 0.08774, 0.12937, 0.17099,
                        0.25417, 0.33734, 0.45779, 0.57823,
                        0.69640, 0.81458, 0.88284, 0.95110,
                        0.97094, 0.99078, 0.99461, 0.99845,
                        0.99841, 0.99838, 0.99286, 0.98735,
                        0.97800, 0.96866, 0.95989, 0.95112,
                        0.95260, 0.95408, 0.96743, 0.98078,
                        0.99039, 1.00000, 0.94894, 0.89787,
                        0.76385, 0.62983, 0.48925, 0.34868,
                        0.26464, 0.18059, 0.13647, 0.09235,
                        0.07131, 0.05027, 0.03929, 0.02830,
                        0.02209, 0.01588, 0.01235, 0.00882,
                        0.00660, ]))
    PROBAV_2_04 = (137, 1.53250, 1.67250,
                   np.array(
                       [0.00509, 0.00663, 0.00939, 0.01215,
                        0.01793, 0.02370, 0.03610, 0.04850,
                        0.07042, 0.09233, 0.13681, 0.18130,
                        0.26779, 0.35429, 0.47954, 0.60479,
                        0.72485, 0.84490, 0.90797, 0.97104,
                        0.98552, 1.00000, 0.99940, 0.99879,
                        0.99606, 0.99333, 0.98735, 0.98138,
                        0.97169, 0.96201, 0.95450, 0.94698,
                        0.94917, 0.95135, 0.96560, 0.97984,
                        0.98773, 0.99562, 0.93668, 0.87774,
                        0.73997, 0.60221, 0.46463, 0.32705,
                        0.24772, 0.16839, 0.12672, 0.08505,
                        0.06463, 0.04421, 0.03413, 0.02405,
                        0.01846, 0.01287, 0.01005, 0.00724,
                        0.00559, ]))
    PROBAV_3_04 = (138, 1.53750, 1.66750,
                   np.array(
                       [0.00743, 0.01027, 0.01657, 0.02287,
                        0.03624, 0.04961, 0.07428, 0.09896,
                        0.14959, 0.20021, 0.29657, 0.39293,
                        0.52013, 0.64734, 0.75301, 0.85867,
                        0.90658, 0.95449, 0.96447, 0.97445,
                        0.97526, 0.97606, 0.97496, 0.97385,
                        0.96927, 0.96469, 0.95776, 0.95082,
                        0.94890, 0.94698, 0.95556, 0.96413,
                        0.98207, 1.00000, 0.99839, 0.99678,
                        0.91467, 0.83256, 0.68482, 0.53707,
                        0.41053, 0.28399, 0.21417, 0.14435,
                        0.10815, 0.07194, 0.05424, 0.03654,
                        0.02704, 0.01755, 0.01287, 0.00819,
                        0.00619, ]))

    # All of the original predefined wavelengths from 6S
    # CONSTANT_NAME = (ID for Constant, Start Wavelength, End Wavelength)
    METEOSAT_VISIBLE = (-2, 0.35, 1.11)
    GOES_EAST_VISIBLE = (-3, 0.49, 0.9)
    GOES_WEST_VISIBLE = (-4, 0.49, 0.9)
    AVHRR_NOAA6_B1 = (-5, 0.55, 0.75)
    AVHRR_NOAA6_B2 = (-6, 0.69, 1.12)
    AVHRR_NOAA7_B1 = (-7, 0.5, 0.8)
    AVHRR_NOAA7_B2 = (-8, 0.64, 1.17)
    AVHRR_NOAA8_B1 = (-9, 0.54, 1.01)
    AVHRR_NOAA8_B2 = (-10, 0.68, 1.12)
    AVHRR_NOAA9_B1 = (-11, 0.53, 0.81)
    AVHRR_NOAA9_B2 = (-12, 0.68, 1.17)
    AVHRR_NOAA10_B1 = (-13, 0.53, 0.78)
    AVHRR_NOAA10_B2 = (-14, 0.6, 1.19)
    AVHRR_NOAA11_B1 = (-15, 0.54, 0.82)
    AVHRR_NOAA11_B2 = (-16, 0.6, 1.12)
    SPOT_HRV1_B1 = (-17, 0.47, 0.65)
    SPOT_HRV1_B2 = (-18, 0.6, 0.72)
    SPOT_HRV1_B3 = (-19, 0.73, 0.93)
    SPOT_HRV1_PAN = (-20, 0.47, 0.79)
    SPOT_HRV2_B1 = (-21, 0.47, 0.65)
    SPOT_HRV2_B2 = (-22, 0.59, 0.73)
    SPOT_HRV2_B3 = (-23, 0.74, 0.94)
    SPOT_HRV2_PAN = (-24, 0.47, 0.79)
    LANDSAT_TM_B1 = (-25, 0.43, 0.56)
    LANDSAT_TM_B2 = (-26, 0.5, 0.65)
    LANDSAT_TM_B3 = (-27, 0.58, 0.74)
    LANDSAT_TM_B4 = (-28, 0.73, 0.95)
    LANDSAT_TM_B5 = (-29, 1.5025, 1.89)
    LANDSAT_TM_B7 = (-30, 1.95, 2.41)
    LANDSAT_MSS_B1 = (-31, 0.475, 0.64)
    LANDSAT_MSS_B2 = (-32, 0.58, 0.75)
    LANDSAT_MSS_B3 = (-33, 0.655, 0.855)
    LANDSAT_MSS_B4 = (-34, 0.785, 1.1)
    ER2_MAS_B1 = (-35, 0.5025, 0.5875)
    ER2_MAS_B2 = (-36, 0.6075, 0.7)
    ER2_MAS_B3 = (-37, 0.83, 0.9125)
    ER2_MAS_B4 = (-38, 0.9, 0.9975)
    ER2_MAS_B5 = (-39, 1.82, 1.9575)
    ER2_MAS_B6 = (-40, 2.0950, 2.1925)
    ER2_MAS_B7 = (-41, 3.58, 3.87)
    MODIS_B1 = (-42, 0.61, 0.685)
    MODIS_B2 = (-43, 0.82, 0.9025)
    MODIS_B3 = (-44, 0.45, 0.4825)
    MODIS_B4 = (-45, 0.54, 0.57)
    MODIS_B5 = (-46, 1.2150, 1.27)
    MODIS_B6 = (-47, 1.6, 1.665)
    MODIS_B7 = (-48, 2.0575, 2.1825)
    MODIS_B8 = (-49, 0.4025, 0.4225)
    AVHRR_NOAA12_B1 = (-50, 0.5, 1.0)
    AVHRR_NOAA12_B2 = (-51, 0.65, 1.12)
    AVHRR_NOAA14_B1 = (-52, 0.5, 1.11)
    AVHRR_NOAA14_B2 = (-53, 0.68, 1.1)
    POLDER_B1 = (-54, 0.4125, 0.4775)
    POLDER_B2 = (-55, 0.41, 0.5225)
    POLDER_B3 = (-56, 0.5325, 0.595)
    POLDER_B4 = (-57, 0.63, 0.7025)
    POLDER_B5 = (-58, 0.745, 0.78)
    POLDER_B6 = (-59, 0.7, 0.83)
    POLDER_B7 = (-60, 0.81, 0.92)
    POLDER_B8 = (-61, 0.8650, 0.94)
    SEAWIFS_B1 = (-62, 0.3825, 0.7)
    SEAWIFS_B2 = (-63, 0.38, 0.58)
    SEAWIFS_B3 = (-64, 0.38, 1.02)
    SEAWIFS_B4 = (-65, 0.38, 1.02)
    SEAWIFS_B5 = (-66, 0.3825, 1.15)
    SEAWIFS_B6 = (-67, 0.3825, 1.05)
    SEAWIFS_B7 = (-68, 0.38, 1.15)
    SEAWIFS_B8 = (-69, 0.38, 1.15)
    AATSR_B1 = (-70, 0.525, 0.5925)
    AATSR_B2 = (-71, 0.6275, 0.6975)
    AATSR_B3 = (-72, 0.8325, 0.9025)
    AATSR_B4 = (-73, 1.4475, 1.7775)
    MERIS_B1 = (-74, 0.412, 0.412 + 0.00998)
    MERIS_B2 = (-75, 0.442, 0.442 + 0.00997)
    MERIS_B3 = (-76, 0.489, 0.489 + 0.00997)
    MERIS_B4 = (-77, 0.509, 0.509 + 0.00997)
    MERIS_B5 = (-78, 0.559, 0.559 + 0.00997)
    MERIS_B6 = (-79, 0.619, 0.619 + 0.00997)
    MERIS_B7 = (-80, 0.664, 0.664 + 0.00998)
    MERIS_B8 = (-81, 0.681, 0.681 + 0.00749)
    MERIS_B9 = (-82, 0.708, 0.708 + 0.00999)
    MERIS_B10 = (-83, 0.753, 0.753 + 0.00749)
    MERIS_B11 = (-84, 0.760, 0.760 + 0.00374)
    MERIS_B12 = (-85, 0.778, 0.778 + 0.00150)
    MERIS_B13 = (-86, 0.865, 0.865 + 0.002)
    MERIS_B14 = (-87, 0.885, 0.885 + 0.001)
    MERIS_B15 = (-88, 0.9, 0.9 + 0.001)
    GLI_B1 = (-89, 0.37, 0.3925)
    GLI_B2 = (-90, 0.3875, 0.4125)
    GLI_B3 = (-91, 0.3975, 0.4275)
    GLI_B4 = (-92, 0.4475, 0.505)
    GLI_B5 = (-93, 0.4475, 0.46)
    GLI_B6 = (-94, 0.475, 0.505)
    GLI_B7 = (-95, 0.5075, 0.5325)
    GLI_B8 = (-96, 0.5265, 0.56)
    GLI_B9 = (-97, 0.5475, 0.5825)
    GLI_B10 = (-98, 0.61, 0.64)
    GLI_B11 = (-99, 0.6525, 0.6825)
    GLI_B12 = (-100, 0.665, 0.695)
    GLI_B13 = (-101, 0.6625, 0.6975)
    GLI_B14 = (-102, 0.6925, 0.7275)
    GLI_B15 = (-103, 0.6925, 0.7275)
    GLI_B16 = (-104, 0.7325, 0.7675)
    GLI_B17 = (-105, 0.75, 0.775)
    GLI_B18 = (-106, 0.84, 0.8925)
    GLI_B19 = (-107, 0.85, 0.88)
    GLI_B20 = (-108, 0.415, 0.5075)
    GLI_B21 = (-109, 0.505, 0.58)
    GLI_B22 = (-110, 0.6075, 0.715)
    GLI_B23 = (-111, 0.745, 0.9075)
    GLI_B24 = (-112, 1.03, 1.07)
    GLI_B25 = (-113, 1.085, 1.19)
    GLI_B26 = (-114, 1.22, 1.2625)
    GLI_B27 = (-115, 1.3475, 1.415)
    GLI_B28 = (-116, 1.515, 1.77)
    GLI_B29 = (-117, 2.055, 2.345)
    GLI_B30 = (-118, 3.22, 4.0)
    ALI_B1P = (-119, 0.4225, 0.4625)
    ALI_B1 = (-120, 0.4325, 0.550)
    ALI_B2 = (-121, 0.5, 0.63)
    ALI_B3 = (-122, 0.5775, 0.750)
    ALI_B4 = (-123, 0.7525, 0.8375)
    ALI_B4P = (-124, 0.8025, 0.935)
    ALI_B5P = (-125, 1.130, 1.345)
    ALI_B5 = (-126, 1.47, 1.820)
    ALI_B7 = (-127, 1.98, 2.53)
    ASTER_B1 = (-128, 0.485, 0.6425)
    ASTER_B2 = (-129, 0.590, 0.730)
    ASTER_B3N = (-130, 0.720, 0.9075)
    ASTER_B3B = (-131, 0.720, 0.9225)
    ASTER_B4 = (-132, 1.57, 1.7675)
    ASTER_B5 = (-133, 2.120, 2.2825)
    ASTER_B6 = (-134, 2.150, 2.295)
    ASTER_B7 = (-135, 2.210, 2.39)
    ASTER_B8 = (-136, 2.250, 2.244)
    ASTER_B9 = (-137, 2.2975, 2.4875)
    LANDSAT_ETM_B1 = (-138, 0.435, 0.52)
    LANDSAT_ETM_B2 = (-139, 0.5, 0.6225)
    LANDSAT_ETM_B3 = (-140, 0.615, 0.7025)
    LANDSAT_ETM_B4 = (-141, 0.74, 0.9125)
    LANDSAT_ETM_B5 = (-142, 1.51, 1.7875)
    LANDSAT_ETM_B7 = (-143, 2.015, 2.3775)
    HYPBLUE_B1 = (-144, 0.4375, 0.5)
    HYPBLUE_B2 = (-145, 0.435, 0.52)
    SPOT_VGT_B1 = (-146, 0.4175, 0.5)
    SPOT_VGT_B2 = (-147, 0.5975, 0.7675)
    SPOT_VGT_B3 = (-148, 0.7325, 0.9575)
    SPOT_VGT_B4 = (-149, 1.5225, 1.8)
    VIIRS_BM1 = (-150, 0.4025, 0.4225)
    VIIRS_BM2 = (-151, 0.4350, 0.4550)
    VIIRS_BM3 = (-152, 0.4775, 0.4975)
    VIIRS_BM4 = (-153, 0.5450, 0.565)
    VIIRS_BM5 = (-154, 0.6625, 0.6825)
    VIIRS_BM6 = (-155, 0.7375, 0.7525)
    VIIRS_BM7 = (-156, 0.8450, 0.8850)
    VIIRS_BM8 = (-157, 1.23, 1.25)
    VIIRS_BM9 = (-158, 1.37, 1.385)
    VIIRS_BM10 = (-159, 1.58, 1.64)
    VIIRS_BM11 = (-160, 2.225, 2.275)
    VIIRS_BM12 = (-161, 3.61, 3.79)
    VIIRS_BI1 = (-162, 0.6, 0.68)
    VIIRS_BI2 = (-163, 0.845, 0.885)
    VIIRS_BI3 = (-164, 1.58, 1.64)
    VIIRS_BI4 = (-165, 3.55, 3.93)
