"""Core migrator code"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/00_core.ipynb.

# %% auto 0
__all__ = ['Migrator', 'database']

# %% ../nbs/00_core.ipynb 3
from fastlite import *
from apswutils.db import Database
from fastcore.all import *
import fastlite.kw

# %% ../nbs/00_core.ipynb 6
class Migrator():
    def __init__(self, db):
        self.migrations, self.rollbacks, self.db = dict(), dict(), db
        db.execute("""
        CREATE TABLE IF NOT EXISTS migrations (
            id INTEGER PRIMARY KEY, name TEXT, inserted_at TEXT DEFAULT CURRENT_TIMESTAMP NOT NULL
        ) STRICT;
        """)
        self.Migration = db.t.migrations.dataclass()
        self.db_migrations = db.t.migrations

# %% ../nbs/00_core.ipynb 9
@patch
def add_migration(self: Migrator, migration_id:int):
    assert type(migration_id) is int, "migration_id must be an integer"
    def decorator(migration:callable):
        assert callable(migration), "migration must be a callable"
        self.migrations[migration_id] = migration
    return decorator

@patch
def add_rollback(self: Migrator, rollback_id:int):
    assert type(rollback_id) is int, "rollback_id must be an integer"
    def decorator(rollback:callable):
        assert callable(rollback), "migration must be a callable"
        self.rollbacks[rollback_id] = rollback
    return decorator

# %% ../nbs/00_core.ipynb 10
@patch
def migrate(self: Migrator):
    keys = sorted(self.migrations.keys())
    assert keys[0] == 0 and keys[-1] == len(keys) - 1, "migration_ids must go like 0,1,2,..."

    for id, migration in sorted(self.migrations.items()):
        if id in self.db.t.migrations:
            continue
        print(id, migration.__name__)
        with self.db.conn:
            self.db_migrations.insert(self.Migration(id=id, name=migration.__name__))
            migration(self.db)

# %% ../nbs/00_core.ipynb 17
@patch
def last_applied_migration(self: Migrator):
    return self.db_migrations('id = (SELECT MAX(id) FROM migrations)')[0]

# %% ../nbs/00_core.ipynb 18
@patch
def applied_migrations(self: Migrator):
    return self.db_migrations()

# %% ../nbs/00_core.ipynb 19
@patch
def rollback(self: Migrator):
    latest_migration = self.last_applied_migration()
    last_id = latest_migration.id
    if last_id not in self.rollbacks:
        print(f"No rollback for the latest applied migration found: {latest_migration}")
        return

    with self.db.conn:
        rollback = self.rollbacks[last_id]
        print(last_id, rollback.__name__)

        rollback(self.db)
        self.db_migrations.delete(last_id)

# %% ../nbs/00_core.ipynb 40
_orig_database = database

# %% ../nbs/00_core.ipynb 41
def database(path, wal=True):
    db = _orig_database(path, wal)
    db.migrator = Migrator(db)
    # Avoid name collision with transaction rollback
    db.rollback_migration = db.migrator.rollback

    # Everything else is ok to copy automatically
    #names = [x for x in dir(Migrator) if "__" not in x and "rollback" not in x] + ["add_rollback"]    
    db.add_migration = db.migrator.add_migration
    db.add_rollback = db.migrator.add_rollback
    db.applied_migrations = db.migrator.applied_migrations
    db.last_applied_migration = db.migrator.last_applied_migration
    db.migrate = db.migrator.migrate

    return db
