/*
 *  brplot - Author: Branimir Ričko
 *    The MIT License (MIT)
 *    Copyright © Branimir Ričko [branc116]
 *
 *    Permission is hereby granted, free of charge, to any person  obtaining a copy of
 *    this software and associated  documentation files (the   “Software”), to deal in
 *    the Software without  restriction,  including without  limitation  the rights to
 *    use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
 *    the Software,  and to permit persons to whom the Software is furnished to do so,
 *    subject to the following conditions:
 *
 *    The above  copyright notice and this  permission notice shall be included in all
 *    copies or substantial portions of the Software.
 *
 *    THE SOFTWARE IS PROVIDED   “AS IS”,   WITHOUT WARRANTY OF ANY KIND,   EXPRESS OR
 *    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 *    FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.   IN NO EVENT SHALL THE AUTHORS OR
 *    COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,  DAMAGES OR OTHER LIABILITY,  WHETHER
 *    IN  AN  ACTION  OF CONTRACT,   TORT OR OTHERWISE,   ARISING FROM,   OUT OF OR IN
 *    CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Brplot - Brplot is a plotting library and application.
 * There is a simple and a more complicated api that you can use.
 *
 * Example of simple API:
 * ```main.c
      #define BRPLOT_IMPLEMENTATION
      #include "brplot.c"

      int main(void) {
         for (int i = -10; i < 10; ++i)           brp_1(y_valuei*i,               0); //          y-value, group_id
         for (float i = -10.f; i < 10.f; i+=0.1f) brp_2(0.1f*sin(i), 0.1f*cos(i), 2); // x-value, y-value, group_id
         brp_wait(); // Wait until plot window is closed
      }
 * ```
 * Compile as: cc main.c
 * Functions that are part of simple api can be found by following a tag *SIMPLEAPI*
 *
 * Example of a bit more compilcate API
 * ``` main.c
       #define BRPLOT_IMPLEMENTATION
       #include "brplot.h"

       int main(void) {
         br_plotter_t* plotter = br_plotter_new(br_plotter_default_ctor());
         br_plot_id plot = br_plot_new(plotter, br_plot_default_ctor());
         br_data_id data = br_data_new(plotter, br_data_default_ctor());
         for (int i = -10; i < 10; ++i) br_data_add_v1(plotter, data, 10.f);
         br_plotter_wait(plotter);
       }
 * ```
 * Compile as: cc main.c
 * Functions that are part of a bit more complicated API can be found by following a tag *ABITMORECOMPILCATED*
 *
 * */
#define BR_VERSION {0, 0, 1}

#if defined(__EMSCRIPTEN__)
#  include <emscripten.h>
#  define BR_EXPORT EMSCRIPTEN_KEEPALIVE
#elif defined(__GNUC__)
#  define BR_EXPORT __attribute__((visibility ("default")))
#elif defined(_WIN32)
#  define BR_EXPORT __declspec(dllexport)
#else
#  define BR_EXPORT
#endif

#if defined(__cplusplus)
extern "C" {
#endif

// --------------------------------------------- SIMPLEAPI ----------------------------------------------
/*
 * This functions behaves more or less like the more complicated api, with the difference that you don't have to pass br_plotter_t argument.
 * These functions use global instance of br_plotter_t
 * For more  details what each parameter means jump to *PARAMETER_DEFINITIONS*
 * */
typedef int br_data_id;
BR_EXPORT br_data_id brp_1   (float x,                                            br_data_id data_id);
BR_EXPORT br_data_id brp_1n  (float const* points, int n,                         br_data_id data_id);
BR_EXPORT br_data_id brp_1ns (float const* points, int n, int stride, int offset, br_data_id data_id);

BR_EXPORT br_data_id brp_2   (float x, float y,                                                                br_data_id data_id);
BR_EXPORT br_data_id brp_2n  (float const* v,                   int n,                                         br_data_id data_id);
BR_EXPORT br_data_id brp_2ns (float const* v,                   int n, int stride, int offset_x, int offset_y, br_data_id data_id);
BR_EXPORT br_data_id brp_2nd (float const* xs, float const* ys, int n,                                         br_data_id data_id);
BR_EXPORT br_data_id brp_2nds(float const* xs, float const* ys, int n, int stride, int offset_x, int offset_y, br_data_id data_id);

BR_EXPORT br_data_id brp_3   (float x, float y, float z,                                                                                      br_data_id data_id);
BR_EXPORT br_data_id brp_3n  (float const* v,                                    int n,                                                       br_data_id data_id);
BR_EXPORT br_data_id brp_3ns (float const* v,                                    int n, int stride, int offset_x, int offset_y, int offset_z, br_data_id data_id);
BR_EXPORT br_data_id brp_3nd (float const* xs, float const* ys, float const* zs, int n,                                                       br_data_id data_id);
BR_EXPORT br_data_id brp_3nds(float const* xs, float const* ys, float const* zs, int n, int stride, int offset_x, int offset_y, int offset_z, br_data_id data_id);

// Remove all data in data.id == data_id
BR_EXPORT void brp_empty(br_data_id data_id);

// Resize the plots such that data is visible
BR_EXPORT void brp_focus_all(void);

// Wait until all previous datapoints are drawn
BR_EXPORT void brp_flush(void);

// Wait until the plot window is closed
BR_EXPORT void brp_wait(void);

/*
 * This macro is used to extend structures so that backward compatibility is
 * preserved. So Each _ctor structure is exactly 128 bytes. That should be
 * more than enough for constructing every object that can be constructed from
 * now until eternity.
 **/
#define PADDED_STRUCT(name, internal_name) typedef struct name { \
  int version; \
  union { \
    struct internal_name ctor; \
    unsigned long long internal__padding[128 / 8]; \
  }; \
} name



typedef struct br_plotter_t br_plotter_t;
typedef int br_plot_id;

typedef enum {
  br_plotter_ui_kind_minimal = 0,
} br_plotter_ui_kind_t;
#define BR_PLOTTER_UI_KIND_T

#if !defined(BR_PLOT_KIND_T)
typedef enum {
  br_plot_kind_2d,
  br_plot_kind_3d
} br_plot_kind_t;
#define BR_PLOT_KIND_T
#endif

#if !defined(BR_DATA_KIND_T)
typedef enum {
  br_data_kind_2d,
  br_data_kind_3d
} br_data_kind_t;
#define BR_DATA_KIND_T
#endif

struct br_plotter_ctor_int_t {
  int width, height;
  br_plotter_ui_kind_t kind;
  int use_permastate;
  int use_stdin;
};

struct br_plot_ctor_int_t {
  br_plot_kind_t kind;
  float width; // From 0 to 1
  float height; // From 0 to 1
  int rearange_others;
};

struct br_data_ctor_int_t {
  br_data_kind_t kind;
};

PADDED_STRUCT(br_plotter_ctor_t, br_plotter_ctor_int_t);
PADDED_STRUCT(br_plot_ctor_t, br_plot_ctor_int_t);
PADDED_STRUCT(br_data_ctor_t, br_data_ctor_int_t);


// --------------------------------------------- ABITMORECOMPILCATED ---------------------------------------------

BR_EXPORT br_plotter_ctor_t* br_plotter_default_ctor(void);
BR_EXPORT br_plotter_t* br_plotter_new(br_plotter_ctor_t const* ctor);
BR_EXPORT void br_plotter_wait(br_plotter_t const* plotter);
BR_EXPORT void br_plotter_free(br_plotter_t* plotter);

BR_EXPORT br_plot_ctor_t* br_plot_default_ctor(void);
BR_EXPORT br_plot_id br_plot_new(br_plotter_t* plotter, br_plot_ctor_t const* ctor);
BR_EXPORT void br_plot_free(br_plotter_t* plotter, br_plot_id plot);

BR_EXPORT br_data_ctor_t* br_data_default_ctor(void);
BR_EXPORT br_data_id br_data_new(br_plotter_t* plotter, br_data_ctor_t const* ctor);

// --------------------------------------------- PARAMETER_DEFINITIONS ---------------------------------------------
/*
 * br_data_add_ is a set of functions that serve the same purpose.
 *              add points to data object. Form of all functions in
 *              the is is: br_data_add_v[dimensions][n[d][s]]
 *              dimensions can be: 1, 2, 3 - what is the dimension of the data
 *              n -> Flag - There is N data points.
 *              d -> Flag ( Only if n flag is on ) - Data points are in [D]ifferent addresses.
 *              s -> Flag ( Only if n flag is on ) - Data points have non standard [S]tride and offset.
 * All functions in this set are:
 *   br_data_add_v1, br_data_add_v1n, br_data_add_v1ns
 *   br_data_add_v2, br_data_add_v2n, br_data_add_v2ns, br_data_add_v2nd, br_data_add_v2nds
 *   br_data_add_v3, br_data_add_v3n, br_data_add_v3ns, br_data_add_v3nd, br_data_add_v3nds
 * Each br_data_add_ function returns number of element inserted into data object.
 *
 * Examples:
 * This is array x
 * |------------------------------------------------------------------------------------|
 * |    x1    |    x2    |    x3    |   x4    |   x5    |   x6    |    x7    |    x8    |
 * |------------------------------------------------------------------------------------|
 * n=8
 *
 * br_data_add_v1n: Add each element to the data object.
 *   E.G.:
 *     br_data_add_v1n(data, x, 8) =>  [x1, x2, x3, x4, x5, x6, x7, x8]
 *
 * br_data_add_v1ns: stride and offset is taken into account. E.G.:
 *   If stride=1, offset=0 => same as br_data_add_v1n(data, x, 8)
 *   If stride=2, offset=0 => Data inserted=[x1, x3, x5, x7]
 *   If stride=2, offset=1 => Data inserted=[x2, x4, x6, x8]
 *   If stride=3, offset=0 => Data inserted=[x1, x4] NOTE: Not inserting x7 because it's
 *                                                         not filling the entire stride
 *   If stride=3, offset=2 => Data inserted=[x3, x6]
 * NOTE: offset < stride !!!!
 * NOTE: stride != 0 !!!!
 *
 * br_data_add_v2n: added points will be: [ [x1, x2], [x3, x4], [x5, x6], [x7, x8] ]
 *
 * br_data_add_v2ns: stide and offset are taken into account. E.G.:
 *   stride=2, offset_x=0, offset_y=1 => same as br_data_add_v2n
 *   stride=1, offset_x=0, offset_y=0 => [ [x1, x1], [x2, x2], [x3, x3], [x4, x4], [x5, x5], [x6, x6], [x7, x7], [x8, x8] ]
 *
 *
 * br_data_add_v2nd: For this variand y array needs to be defined. It may be same as array x.
 *     If This is array y:
 *     |------------------------------------------------------------------------------------|
 *     |    y1    |    y2    |    y3    |   y4    |   y5    |   y6    |    y7    |    y8    |
 *     |------------------------------------------------------------------------------------|
 *     E.G.:
 *       br_data_add_v2nd(data, xs=x, ys=y, n=8) =>
 *         [ [x1, y1], [x2, y2], [x3, y3], [x4, y4], [x5, y5], [x6, y6], [x7, y7], [x8, y8] ]
 *
 * br_data_add_v2nds: For this variand y array needs to be defined. It may be same as array x.
 *     br_data_add_v2nds(data, xs=x, ys=x, n=8, stride=2, offset_x=0, offset_y=1) => same as br_data_add_v2n(data, x, 8)
 *     br_data_add_v2nds(data, xs=x, ys=y, n=8, stride=1, offset_x=0, offset_y=0) => same as br_data_add_v2nd(data, xs=x, ys=y, n=8)
 *
 * */
BR_EXPORT int br_data_add_v1   (br_plotter_t* plotter, float x, br_data_id data);
BR_EXPORT int br_data_add_v1n  (br_plotter_t* plotter, float const* x, int n, br_data_id data);
BR_EXPORT int br_data_add_v1ns (br_plotter_t* plotter, float const* x, int n, int stride, int offset, br_data_id data);
BR_EXPORT int br_data_add_v2   (br_plotter_t* plotter, float x, float y, br_data_id data);
BR_EXPORT int br_data_add_v2n  (br_plotter_t* plotter, float const* v, int n, br_data_id data);
BR_EXPORT int br_data_add_v2ns (br_plotter_t* plotter, float const* v, int n, int stride, int offset_x, int offset_y, br_data_id data);
BR_EXPORT int br_data_add_v2nd (br_plotter_t* plotter, float const* xs, float const* ys, int n, br_data_id data);
BR_EXPORT int br_data_add_v2nds(br_plotter_t* plotter, float const* xs, float const* ys, int n, int stride, int offset_x, int offset_y, br_data_id data);
BR_EXPORT int br_data_add_v3   (br_plotter_t* plotter, float x, float y, float z, br_data_id data);
BR_EXPORT int br_data_add_v3n  (br_plotter_t* plotter, float const* v, int n, br_data_id data);
BR_EXPORT int br_data_add_v3ns (br_plotter_t* plotter, float const* v, int n, int stride, int offset_x, int offset_y, int offset_z, br_data_id data);
BR_EXPORT int br_data_add_v3nd (br_plotter_t* plotter, float const* xs, float const* ys, float const* zs, int n, br_data_id data);
BR_EXPORT int br_data_add_v3nds(br_plotter_t* plotter, float const* xs, float const* ys, float const* zs, int n, int stride, int offset_x, int offset_y, int offset_z, br_data_id data);

BR_EXPORT void br_data_free(br_data_id data);

BR_EXPORT void br_plot_show_data(br_plotter_t* plotter, br_plot_id plot, br_data_id data);
BR_EXPORT void br_plot_hide_data(br_plotter_t* plotter, br_plot_id plot, br_data_id data);

#if defined(__cplusplus)
}
#endif

// If you just wanna build brplot as an app you have to define this
#if defined(BRPLOT_IMPLEMENTATION) || defined(BRPLOT_APP)
/* #  include "src/br_pp.h" */


// --------------------------------------------- DEFINES ---------------------------------------------

#if !defined(BR_DISABLE_LOG)
#  define LOG(...)
#  define LOGI(format, ...) fprintf(stderr, "[INFO][" __FILE__ ":%d] " format "\n", __LINE__, ##__VA_ARGS__)
#  define LOGW(format, ...) fprintf(stderr, "[WARNING][" __FILE__ ":%d] " format "\n", __LINE__, ##__VA_ARGS__)
#  define LOGE(format, ...) fprintf(stderr, "[ERROR][" __FILE__ ":%d] " format "\n", __LINE__, ##__VA_ARGS__)
#else
#  define LOG(...)
#  define LOGI(...)
#  define LOGW(...)
#  define LOGE(...)
#endif

#define LOGF(format, ...) do { \
  fprintf(stderr, "[FATAL][" __FILE__ ":%d] " format "\n", __LINE__, ##__VA_ARGS__); \
  abort(); \
} while(0)

#define BR_LOG_GL_ERROR(ERROR) do { \
  if (0 != (ERROR)) { \
    LOGF("GL Error: %d", (ERROR)); \
  } \
} while(0)

#define BR_MALLOC(size) malloc(size)
#define BR_CALLOC calloc
#define BR_REALLOC realloc
#define BR_FREE free
#define BR_ASSERT(x) do { \
  if (!(x)) { \
     LOGE("ASSERT FAILED: `" #x "`"); \
     BR_BREAKPOINT(); \
     LOGF("Exiting"); \
  } \
} while (0)
#define BR_ASSERTF(x, fmt, ...) do { \
   if (!(x)) { \
     LOGE("ASSERT FAILED: `" #x "`" fmt, ##__VA_ARGS__); \
     BR_BREAKPOINT(); \
     LOGF("Exiting"); \
   } \
} while (0)

#if defined(BRPLOT_IMPLEMENTATION)
#  define BR_LIB
#endif

#if defined(DEBUG) || defined(RELEASE) || defined(BR_RELEASE)
#  error "Please don't define DEBUG or RELEASE or BR_RELEASE macros"
#endif
#if !defined(BR_DEBUG)
#  define BR_RELEASE
#endif

#if defined(BR_DEBUG) && !defined(__linux__) && defined(UNIT_TEST)
   // IT don't work on windows....
#  undef UNIT_TEST
#endif

#if defined(BR_MUSL)
#elif defined(__linux__)
#  define LOCK_T pthread_mutex_t
#endif

#if defined(LOCK_T)
#  define LOCK(x) LOCK_T x;
#else
#  define LOCK(x)
#endif


#ifdef __cplusplus
extern "C" {
#endif

#ifdef __cplusplus
}
#endif

#if defined(BR_DEBUG)
#  if defined(_MSC_VER)
#    define BR_BREAKPOINT() __debugbreak()
#  else
#    define BR_BREAKPOINT() __builtin_trap()
#  endif
#else
# define BR_BREAKPOINT()
#endif

#if !defined(BR_HAS_HOTRELOAD)
#  if defined(BR_DEBUG) && defined(IMGUI) && defined(__linux__) && !defined(LIB)
#    define BR_HAS_HOTRELOAD 1
#  else
#    define BR_HAS_HOTRELOAD 0
#  endif
#endif

#if defined(BR_DEBUG) && defined(__linux__)
#  define BR_HAS_SHADER_RELOAD 1
#else
#  define BR_HAS_SHADER_RELOAD 0
#endif

#if defined(_MSC_VER)
#  define ssize_t long long int
#  define TEST_ONLY
#else
#  define TEST_ONLY __attribute__((__unused__))
#endif

#if defined(__cplusplus) &&  __cplusplus >= 201103L
#  define BR_THREAD_LOCAL       thread_local
#elif defined(_MSC_VER)
#  define BR_THREAD_LOCAL       __declspec(thread)
#elif defined (__STDC_VERSION__) && __STDC_VERSION__ >= 201112L && !defined(__STDC_NO_THREADS__)
#  define BR_THREAD_LOCAL       _Thread_local
#elif defined(__GNUC__) && __GNUC__ < 5
#  define BR_THREAD_LOCAL       __thread
#elif defined(__GNUC__)
#  define BR_THREAD_LOCAL       __thread
#endif

#if !defined(_CRT_SECURE_NO_WARNINGS)
#  define _CRT_SECURE_NO_WARNINGS // Windows bullshit
#endif
#if !defined(_GNU_SOURCE)
#  define _GNU_SOURCE // Linux bullshit
#endif

/* #include "external/Tracy/tracy/TracyC.h" */
#ifndef __TRACYC_HPP__
#define __TRACYC_HPP__

#include <stddef.h>
#include <stdint.h>

/* #include "external/Tracy/client/TracyCallstack.h" */
#ifndef __TRACYCALLSTACK_H__
#define __TRACYCALLSTACK_H__

#ifndef TRACY_NO_CALLSTACK

#  if !defined _WIN32
#    include <sys/param.h>
#  endif

#  if defined _WIN32
/* #    include "external/Tracy/common/TracyUwp.hpp" */
#ifndef __TRACYUWP_HPP__
#define __TRACYUWP_HPP__

#ifdef _WIN32
#  include <winapifamily.h>
#  if WINAPI_FAMILY_PARTITION(WINAPI_PARTITION_APP) && !WINAPI_FAMILY_PARTITION(WINAPI_PARTITION_DESKTOP)
#    define TRACY_UWP
#  endif
#endif

#endif

#    ifndef TRACY_UWP
#      define TRACY_HAS_CALLSTACK 1
#    endif
#  elif defined __ANDROID__
#    if !defined __arm__ || __ANDROID_API__ >= 21
#      define TRACY_HAS_CALLSTACK 2
#    else
#      define TRACY_HAS_CALLSTACK 5
#    endif
#  elif defined __linux
#    if defined _GNU_SOURCE && defined __GLIBC__
#      define TRACY_HAS_CALLSTACK 3
#    else
#      define TRACY_HAS_CALLSTACK 2
#    endif
#  elif defined __APPLE__
#    define TRACY_HAS_CALLSTACK 4
#  elif defined BSD
#    define TRACY_HAS_CALLSTACK 6
#  endif

#endif

#endif

/* #include "external/Tracy/common/TracyApi.h" */
#ifndef __TRACYAPI_H__
#define __TRACYAPI_H__

#if defined _WIN32
#  if defined TRACY_EXPORTS
#    define TRACY_API __declspec(dllexport)
#  elif defined TRACY_IMPORTS
#    define TRACY_API __declspec(dllimport)
#  else
#    define TRACY_API
#  endif
#else
#  define TRACY_API __attribute__((visibility("default")))
#endif

#endif    // __TRACYAPI_H__


#ifdef __cplusplus
extern "C" {
#endif

enum TracyPlotFormatEnum
{
    TracyPlotFormatNumber,
    TracyPlotFormatMemory,
    TracyPlotFormatPercentage,
    TracyPlotFormatWatt
};

TRACY_API void ___tracy_set_thread_name( const char* name );

#define TracyCSetThreadName( name ) ___tracy_set_thread_name( name );

#ifndef TracyFunction
#  define TracyFunction __FUNCTION__
#endif

#ifndef TracyFile
#  define TracyFile __FILE__
#endif

#ifndef TracyLine
#  define TracyLine __LINE__
#endif

#ifndef TRACY_ENABLE

typedef const void* TracyCZoneCtx;

#define TracyCZone(c,x)
#define TracyCZoneN(c,x,y)
#define TracyCZoneC(c,x,y)
#define TracyCZoneNC(c,x,y,z)
#define TracyCZoneEnd(c)
#define TracyCZoneText(c,x,y)
#define TracyCZoneName(c,x,y)
#define TracyCZoneColor(c,x)
#define TracyCZoneValue(c,x)

#define TracyCAlloc(x,y)
#define TracyCFree(x)
#define TracyCSecureAlloc(x,y)
#define TracyCSecureFree(x)

#define TracyCAllocN(x,y,z)
#define TracyCFreeN(x,y)
#define TracyCSecureAllocN(x,y,z)
#define TracyCSecureFreeN(x,y)

#define TracyCFrameMark
#define TracyCFrameMarkNamed(x)
#define TracyCFrameMarkStart(x)
#define TracyCFrameMarkEnd(x)
#define TracyCFrameImage(x,y,z,w,a)

#define TracyCPlot(x,y)
#define TracyCPlotF(x,y)
#define TracyCPlotI(x,y)
#define TracyCPlotConfig(x,y,z,w,a)

#define TracyCMessage(x,y)
#define TracyCMessageL(x)
#define TracyCMessageC(x,y,z)
#define TracyCMessageLC(x,y)
#define TracyCAppInfo(x,y)

#define TracyCZoneS(x,y,z)
#define TracyCZoneNS(x,y,z,w)
#define TracyCZoneCS(x,y,z,w)
#define TracyCZoneNCS(x,y,z,w,a)

#define TracyCAllocS(x,y,z)
#define TracyCFreeS(x,y)
#define TracyCSecureAllocS(x,y,z)
#define TracyCSecureFreeS(x,y)

#define TracyCAllocNS(x,y,z,w)
#define TracyCFreeNS(x,y,z)
#define TracyCSecureAllocNS(x,y,z,w)
#define TracyCSecureFreeNS(x,y,z)

#define TracyCMessageS(x,y,z)
#define TracyCMessageLS(x,y)
#define TracyCMessageCS(x,y,z,w)
#define TracyCMessageLCS(x,y,z)

#define TracyCIsConnected 0

#ifdef TRACY_FIBERS
#  define TracyCFiberEnter(fiber)
#  define TracyCFiberLeave
#endif

#else

#ifndef TracyConcat
#  define TracyConcat(x,y) TracyConcatIndirect(x,y)
#endif
#ifndef TracyConcatIndirect
#  define TracyConcatIndirect(x,y) x##y
#endif

struct ___tracy_source_location_data
{
    const char* name;
    const char* function;
    const char* file;
    uint32_t line;
    uint32_t color;
};

struct ___tracy_c_zone_context
{
    uint32_t id;
    int active;
};

struct ___tracy_gpu_time_data
{
    int64_t gpuTime;
    uint16_t queryId;
    uint8_t context;
};

struct ___tracy_gpu_zone_begin_data {
    uint64_t srcloc;
    uint16_t queryId;
    uint8_t context;
};

struct ___tracy_gpu_zone_begin_callstack_data {
    uint64_t srcloc;
    int depth;
    uint16_t queryId;
    uint8_t context;
};

struct ___tracy_gpu_zone_end_data {
    uint16_t queryId;
    uint8_t context;
};

struct ___tracy_gpu_new_context_data {
    int64_t gpuTime;
    float period;
    uint8_t context;
    uint8_t flags;
    uint8_t type;
};

struct ___tracy_gpu_context_name_data {
    uint8_t context;
    const char* name;
    uint16_t len;
};

struct ___tracy_gpu_calibration_data {
    int64_t gpuTime;
    int64_t cpuDelta;
    uint8_t context;
};

// Some containers don't support storing const types.
// This struct, as visible to user, is immutable, so treat it as if const was declared here.
typedef /*const*/ struct ___tracy_c_zone_context TracyCZoneCtx;


#ifdef TRACY_MANUAL_LIFETIME
TRACY_API void ___tracy_startup_profiler(void);
TRACY_API void ___tracy_shutdown_profiler(void);
#endif

TRACY_API uint64_t ___tracy_alloc_srcloc( uint32_t line, const char* source, size_t sourceSz, const char* function, size_t functionSz );
TRACY_API uint64_t ___tracy_alloc_srcloc_name( uint32_t line, const char* source, size_t sourceSz, const char* function, size_t functionSz, const char* name, size_t nameSz );

TRACY_API TracyCZoneCtx ___tracy_emit_zone_begin( const struct ___tracy_source_location_data* srcloc, int active );
TRACY_API TracyCZoneCtx ___tracy_emit_zone_begin_callstack( const struct ___tracy_source_location_data* srcloc, int depth, int active );
TRACY_API TracyCZoneCtx ___tracy_emit_zone_begin_alloc( uint64_t srcloc, int active );
TRACY_API TracyCZoneCtx ___tracy_emit_zone_begin_alloc_callstack( uint64_t srcloc, int depth, int active );
TRACY_API void ___tracy_emit_zone_end( TracyCZoneCtx ctx );
TRACY_API void ___tracy_emit_zone_text( TracyCZoneCtx ctx, const char* txt, size_t size );
TRACY_API void ___tracy_emit_zone_name( TracyCZoneCtx ctx, const char* txt, size_t size );
TRACY_API void ___tracy_emit_zone_color( TracyCZoneCtx ctx, uint32_t color );
TRACY_API void ___tracy_emit_zone_value( TracyCZoneCtx ctx, uint64_t value );

TRACY_API void ___tracy_emit_gpu_zone_begin( const struct ___tracy_gpu_zone_begin_data );
TRACY_API void ___tracy_emit_gpu_zone_begin_callstack( const struct ___tracy_gpu_zone_begin_callstack_data );
TRACY_API void ___tracy_emit_gpu_zone_begin_alloc( const struct ___tracy_gpu_zone_begin_data );
TRACY_API void ___tracy_emit_gpu_zone_begin_alloc_callstack( const struct ___tracy_gpu_zone_begin_callstack_data );
TRACY_API void ___tracy_emit_gpu_zone_end( const struct ___tracy_gpu_zone_end_data data );
TRACY_API void ___tracy_emit_gpu_time( const struct ___tracy_gpu_time_data );
TRACY_API void ___tracy_emit_gpu_new_context( const struct ___tracy_gpu_new_context_data );
TRACY_API void ___tracy_emit_gpu_context_name( const struct ___tracy_gpu_context_name_data );
TRACY_API void ___tracy_emit_gpu_calibration( const struct ___tracy_gpu_calibration_data );

TRACY_API void ___tracy_emit_gpu_zone_begin_serial( const struct ___tracy_gpu_zone_begin_data );
TRACY_API void ___tracy_emit_gpu_zone_begin_callstack_serial( const struct ___tracy_gpu_zone_begin_callstack_data );
TRACY_API void ___tracy_emit_gpu_zone_begin_alloc_serial( const struct ___tracy_gpu_zone_begin_data );
TRACY_API void ___tracy_emit_gpu_zone_begin_alloc_callstack_serial( const struct ___tracy_gpu_zone_begin_callstack_data );
TRACY_API void ___tracy_emit_gpu_zone_end_serial( const struct ___tracy_gpu_zone_end_data data );
TRACY_API void ___tracy_emit_gpu_time_serial( const struct ___tracy_gpu_time_data );
TRACY_API void ___tracy_emit_gpu_new_context_serial( const struct ___tracy_gpu_new_context_data );
TRACY_API void ___tracy_emit_gpu_context_name_serial( const struct ___tracy_gpu_context_name_data );
TRACY_API void ___tracy_emit_gpu_calibration_serial( const struct ___tracy_gpu_calibration_data );

TRACY_API int ___tracy_connected(void);

#if defined TRACY_HAS_CALLSTACK && defined TRACY_CALLSTACK
#  define TracyCZone( ctx, active ) static const struct ___tracy_source_location_data TracyConcat(__tracy_source_location,TracyLine) = { NULL, __func__,  TracyFile, (uint32_t)TracyLine, 0 }; TracyCZoneCtx ctx = ___tracy_emit_zone_begin_callstack( &TracyConcat(__tracy_source_location,TracyLine), TRACY_CALLSTACK, active );
#  define TracyCZoneN( ctx, name, active ) static const struct ___tracy_source_location_data TracyConcat(__tracy_source_location,TracyLine) = { name, __func__,  TracyFile, (uint32_t)TracyLine, 0 }; TracyCZoneCtx ctx = ___tracy_emit_zone_begin_callstack( &TracyConcat(__tracy_source_location,TracyLine), TRACY_CALLSTACK, active );
#  define TracyCZoneC( ctx, color, active ) static const struct ___tracy_source_location_data TracyConcat(__tracy_source_location,TracyLine) = { NULL, __func__,  TracyFile, (uint32_t)TracyLine, color }; TracyCZoneCtx ctx = ___tracy_emit_zone_begin_callstack( &TracyConcat(__tracy_source_location,TracyLine), TRACY_CALLSTACK, active );
#  define TracyCZoneNC( ctx, name, color, active ) static const struct ___tracy_source_location_data TracyConcat(__tracy_source_location,TracyLine) = { name, __func__,  TracyFile, (uint32_t)TracyLine, color }; TracyCZoneCtx ctx = ___tracy_emit_zone_begin_callstack( &TracyConcat(__tracy_source_location,TracyLine), TRACY_CALLSTACK, active );
#else
#  define TracyCZone( ctx, active ) static const struct ___tracy_source_location_data TracyConcat(__tracy_source_location,TracyLine) = { NULL, __func__,  TracyFile, (uint32_t)TracyLine, 0 }; TracyCZoneCtx ctx = ___tracy_emit_zone_begin( &TracyConcat(__tracy_source_location,TracyLine), active );
#  define TracyCZoneN( ctx, name, active ) static const struct ___tracy_source_location_data TracyConcat(__tracy_source_location,TracyLine) = { name, __func__,  TracyFile, (uint32_t)TracyLine, 0 }; TracyCZoneCtx ctx = ___tracy_emit_zone_begin( &TracyConcat(__tracy_source_location,TracyLine), active );
#  define TracyCZoneC( ctx, color, active ) static const struct ___tracy_source_location_data TracyConcat(__tracy_source_location,TracyLine) = { NULL, __func__,  TracyFile, (uint32_t)TracyLine, color }; TracyCZoneCtx ctx = ___tracy_emit_zone_begin( &TracyConcat(__tracy_source_location,TracyLine), active );
#  define TracyCZoneNC( ctx, name, color, active ) static const struct ___tracy_source_location_data TracyConcat(__tracy_source_location,TracyLine) = { name, __func__,  TracyFile, (uint32_t)TracyLine, color }; TracyCZoneCtx ctx = ___tracy_emit_zone_begin( &TracyConcat(__tracy_source_location,TracyLine), active );
#endif

#define TracyCZoneEnd( ctx ) ___tracy_emit_zone_end( ctx );

#define TracyCZoneText( ctx, txt, size ) ___tracy_emit_zone_text( ctx, txt, size );
#define TracyCZoneName( ctx, txt, size ) ___tracy_emit_zone_name( ctx, txt, size );
#define TracyCZoneColor( ctx, color ) ___tracy_emit_zone_color( ctx, color );
#define TracyCZoneValue( ctx, value ) ___tracy_emit_zone_value( ctx, value );


TRACY_API void ___tracy_emit_memory_alloc( const void* ptr, size_t size, int secure );
TRACY_API void ___tracy_emit_memory_alloc_callstack( const void* ptr, size_t size, int depth, int secure );
TRACY_API void ___tracy_emit_memory_free( const void* ptr, int secure );
TRACY_API void ___tracy_emit_memory_free_callstack( const void* ptr, int depth, int secure );
TRACY_API void ___tracy_emit_memory_alloc_named( const void* ptr, size_t size, int secure, const char* name );
TRACY_API void ___tracy_emit_memory_alloc_callstack_named( const void* ptr, size_t size, int depth, int secure, const char* name );
TRACY_API void ___tracy_emit_memory_free_named( const void* ptr, int secure, const char* name );
TRACY_API void ___tracy_emit_memory_free_callstack_named( const void* ptr, int depth, int secure, const char* name );

TRACY_API void ___tracy_emit_message( const char* txt, size_t size, int callstack );
TRACY_API void ___tracy_emit_messageL( const char* txt, int callstack );
TRACY_API void ___tracy_emit_messageC( const char* txt, size_t size, uint32_t color, int callstack );
TRACY_API void ___tracy_emit_messageLC( const char* txt, uint32_t color, int callstack );

#if defined TRACY_HAS_CALLSTACK && defined TRACY_CALLSTACK
#  define TracyCAlloc( ptr, size ) ___tracy_emit_memory_alloc_callstack( ptr, size, TRACY_CALLSTACK, 0 )
#  define TracyCFree( ptr ) ___tracy_emit_memory_free_callstack( ptr, TRACY_CALLSTACK, 0 )
#  define TracyCSecureAlloc( ptr, size ) ___tracy_emit_memory_alloc_callstack( ptr, size, TRACY_CALLSTACK, 1 )
#  define TracyCSecureFree( ptr ) ___tracy_emit_memory_free_callstack( ptr, TRACY_CALLSTACK, 1 )

#  define TracyCAllocN( ptr, size, name ) ___tracy_emit_memory_alloc_callstack_named( ptr, size, TRACY_CALLSTACK, 0, name )
#  define TracyCFreeN( ptr, name ) ___tracy_emit_memory_free_callstack_named( ptr, TRACY_CALLSTACK, 0, name )
#  define TracyCSecureAllocN( ptr, size, name ) ___tracy_emit_memory_alloc_callstack_named( ptr, size, TRACY_CALLSTACK, 1, name )
#  define TracyCSecureFreeN( ptr, name ) ___tracy_emit_memory_free_callstack_named( ptr, TRACY_CALLSTACK, 1, name )

#  define TracyCMessage( txt, size ) ___tracy_emit_message( txt, size, TRACY_CALLSTACK );
#  define TracyCMessageL( txt ) ___tracy_emit_messageL( txt, TRACY_CALLSTACK );
#  define TracyCMessageC( txt, size, color ) ___tracy_emit_messageC( txt, size, color, TRACY_CALLSTACK );
#  define TracyCMessageLC( txt, color ) ___tracy_emit_messageLC( txt, color, TRACY_CALLSTACK );
#else
#  define TracyCAlloc( ptr, size ) ___tracy_emit_memory_alloc( ptr, size, 0 );
#  define TracyCFree( ptr ) ___tracy_emit_memory_free( ptr, 0 );
#  define TracyCSecureAlloc( ptr, size ) ___tracy_emit_memory_alloc( ptr, size, 1 );
#  define TracyCSecureFree( ptr ) ___tracy_emit_memory_free( ptr, 1 );

#  define TracyCAllocN( ptr, size, name ) ___tracy_emit_memory_alloc_named( ptr, size, 0, name );
#  define TracyCFreeN( ptr, name ) ___tracy_emit_memory_free_named( ptr, 0, name );
#  define TracyCSecureAllocN( ptr, size, name ) ___tracy_emit_memory_alloc_named( ptr, size, 1, name );
#  define TracyCSecureFreeN( ptr, name ) ___tracy_emit_memory_free_named( ptr, 1, name );

#  define TracyCMessage( txt, size ) ___tracy_emit_message( txt, size, 0 );
#  define TracyCMessageL( txt ) ___tracy_emit_messageL( txt, 0 );
#  define TracyCMessageC( txt, size, color ) ___tracy_emit_messageC( txt, size, color, 0 );
#  define TracyCMessageLC( txt, color ) ___tracy_emit_messageLC( txt, color, 0 );
#endif


TRACY_API void ___tracy_emit_frame_mark( const char* name );
TRACY_API void ___tracy_emit_frame_mark_start( const char* name );
TRACY_API void ___tracy_emit_frame_mark_end( const char* name );
TRACY_API void ___tracy_emit_frame_image( const void* image, uint16_t w, uint16_t h, uint8_t offset, int flip );

#define TracyCFrameMark ___tracy_emit_frame_mark( 0 );
#define TracyCFrameMarkNamed( name ) ___tracy_emit_frame_mark( name );
#define TracyCFrameMarkStart( name ) ___tracy_emit_frame_mark_start( name );
#define TracyCFrameMarkEnd( name ) ___tracy_emit_frame_mark_end( name );
#define TracyCFrameImage( image, width, height, offset, flip ) ___tracy_emit_frame_image( image, width, height, offset, flip );


TRACY_API void ___tracy_emit_plot( const char* name, double val );
TRACY_API void ___tracy_emit_plot_float( const char* name, float val );
TRACY_API void ___tracy_emit_plot_int( const char* name, int64_t val );
TRACY_API void ___tracy_emit_plot_config( const char* name, int type, int step, int fill, uint32_t color );
TRACY_API void ___tracy_emit_message_appinfo( const char* txt, size_t size );

#define TracyCPlot( name, val ) ___tracy_emit_plot( name, val );
#define TracyCPlotF( name, val ) ___tracy_emit_plot_float( name, val );
#define TracyCPlotI( name, val ) ___tracy_emit_plot_int( name, val );
#define TracyCPlotConfig( name, type, step, fill, color ) ___tracy_emit_plot_config( name, type, step, fill, color );
#define TracyCAppInfo( txt, size ) ___tracy_emit_message_appinfo( txt, size );


#ifdef TRACY_HAS_CALLSTACK
#  define TracyCZoneS( ctx, depth, active ) static const struct ___tracy_source_location_data TracyConcat(__tracy_source_location,TracyLine) = { NULL, __func__,  TracyFile, (uint32_t)TracyLine, 0 }; TracyCZoneCtx ctx = ___tracy_emit_zone_begin_callstack( &TracyConcat(__tracy_source_location,TracyLine), depth, active );
#  define TracyCZoneNS( ctx, name, depth, active ) static const struct ___tracy_source_location_data TracyConcat(__tracy_source_location,TracyLine) = { name, __func__,  TracyFile, (uint32_t)TracyLine, 0 }; TracyCZoneCtx ctx = ___tracy_emit_zone_begin_callstack( &TracyConcat(__tracy_source_location,TracyLine), depth, active );
#  define TracyCZoneCS( ctx, color, depth, active ) static const struct ___tracy_source_location_data TracyConcat(__tracy_source_location,TracyLine) = { NULL, __func__,  TracyFile, (uint32_t)TracyLine, color }; TracyCZoneCtx ctx = ___tracy_emit_zone_begin_callstack( &TracyConcat(__tracy_source_location,TracyLine), depth, active );
#  define TracyCZoneNCS( ctx, name, color, depth, active ) static const struct ___tracy_source_location_data TracyConcat(__tracy_source_location,TracyLine) = { name, __func__,  TracyFile, (uint32_t)TracyLine, color }; TracyCZoneCtx ctx = ___tracy_emit_zone_begin_callstack( &TracyConcat(__tracy_source_location,TracyLine), depth, active );

#  define TracyCAllocS( ptr, size, depth ) ___tracy_emit_memory_alloc_callstack( ptr, size, depth, 0 )
#  define TracyCFreeS( ptr, depth ) ___tracy_emit_memory_free_callstack( ptr, depth, 0 )
#  define TracyCSecureAllocS( ptr, size, depth ) ___tracy_emit_memory_alloc_callstack( ptr, size, depth, 1 )
#  define TracyCSecureFreeS( ptr, depth ) ___tracy_emit_memory_free_callstack( ptr, depth, 1 )

#  define TracyCAllocNS( ptr, size, depth, name ) ___tracy_emit_memory_alloc_callstack_named( ptr, size, depth, 0, name )
#  define TracyCFreeNS( ptr, depth, name ) ___tracy_emit_memory_free_callstack_named( ptr, depth, 0, name )
#  define TracyCSecureAllocNS( ptr, size, depth, name ) ___tracy_emit_memory_alloc_callstack_named( ptr, size, depth, 1, name )
#  define TracyCSecureFreeNS( ptr, depth, name ) ___tracy_emit_memory_free_callstack_named( ptr, depth, 1, name )

#  define TracyCMessageS( txt, size, depth ) ___tracy_emit_message( txt, size, depth );
#  define TracyCMessageLS( txt, depth ) ___tracy_emit_messageL( txt, depth );
#  define TracyCMessageCS( txt, size, color, depth ) ___tracy_emit_messageC( txt, size, color, depth );
#  define TracyCMessageLCS( txt, color, depth ) ___tracy_emit_messageLC( txt, color, depth );
#else
#  define TracyCZoneS( ctx, depth, active ) TracyCZone( ctx, active )
#  define TracyCZoneNS( ctx, name, depth, active ) TracyCZoneN( ctx, name, active )
#  define TracyCZoneCS( ctx, color, depth, active ) TracyCZoneC( ctx, color, active )
#  define TracyCZoneNCS( ctx, name, color, depth, active ) TracyCZoneNC( ctx, name, color, active )

#  define TracyCAllocS( ptr, size, depth ) TracyCAlloc( ptr, size )
#  define TracyCFreeS( ptr, depth ) TracyCFree( ptr )
#  define TracyCSecureAllocS( ptr, size, depth ) TracyCSecureAlloc( ptr, size )
#  define TracyCSecureFreeS( ptr, depth ) TracyCSecureFree( ptr )

#  define TracyCAllocNS( ptr, size, depth, name ) TracyCAllocN( ptr, size, name )
#  define TracyCFreeNS( ptr, depth, name ) TracyCFreeN( ptr, name )
#  define TracyCSecureAllocNS( ptr, size, depth, name ) TracyCSecureAllocN( ptr, size, name )
#  define TracyCSecureFreeNS( ptr, depth, name ) TracyCSecureFreeN( ptr, name )

#  define TracyCMessageS( txt, size, depth ) TracyCMessage( txt, size )
#  define TracyCMessageLS( txt, depth ) TracyCMessageL( txt )
#  define TracyCMessageCS( txt, size, color, depth ) TracyCMessageC( txt, size, color )
#  define TracyCMessageLCS( txt, color, depth ) TracyCMessageLC( txt, color )
#endif

#define TracyCIsConnected ___tracy_connected()

#ifdef TRACY_FIBERS
TRACY_API void ___tracy_fiber_enter( const char* fiber );
TRACY_API void ___tracy_fiber_leave( void );

#  define TracyCFiberEnter( fiber ) ___tracy_fiber_enter( fiber );
#  define TracyCFiberLeave ___tracy_fiber_leave();
#endif

#endif

#ifdef __cplusplus
}
#endif

#endif



/* #  include "tools/unity/brplot.c" */
#define _GNU_SOURCE
/* #include "include/brplot.h" */

/* #include "src/platform.c" */
/* #include "src/br_plot.h" */

/* #include "src/br_shaders.h" */

#ifndef DEBUG_MACROS
/* #  include "src/br_math.h" */

#include <math.h>
#include <stddef.h>
#include <float.h>
#include <stdbool.h>

#define BR_MAX(X, Y) ((X) > (Y) ? (X) : (Y))
#define BR_MIN(X, Y) ((X) < (Y) ? (X) : (Y))

#define BR_VEC2(X, Y) ((br_vec2_t) { .x = (X), .y = (Y) })
#define BR_VEC2I(X, Y) ((br_vec2i_t) { .x = (X), .y = (Y) })
#define BR_VEC2_TOI(V) ((br_vec2i_t) { .x = (int)(V).x, .y = (int)(V).y })
#define BR_VEC2I_TOF(V) ((br_vec2_t) { .x = (float)(V).x, .y = (float)(V).y })
#define BR_VEC2I_SUB(A, B) ((br_vec2i_t) { .x = (A).x - (B).x, .y = (A).y - (B).y })
#define BR_VEC2I_SCALE(V, B) ((br_vec2i_t) { .x = (V).x * (B), .y = (V).y * (B) })
#define BR_VEC3_TOC(V, A) ((br_color_t) { .r = (unsigned char)((V).x * 255.f), .g = (unsigned char)((V).y * 255.f), .b = (unsigned char)((V).z * 255.f), .a = (A) })
#define BR_SIZE(WIDTH, HEIGHT) ((br_size_t) { .width = (WIDTH), .height = (HEIGHT) })
#define BR_SIZE_TOI(SZ) ((br_sizei_t) { .width = (int)((SZ).width), .height = (int)((SZ).height) })
#define BR_SIZEI(WIDTH, HEIGHT) ((br_sizei_t) { .width = (WIDTH), .height = (HEIGHT) })
#define BR_SIZEI_TOF(SZ) ((br_size_t) { .width = (float)((SZ).width), .height = (float)((SZ).height) })

#define BR_EXTENT_ASPECT(E) ((E).height / (E).width)
#define BR_EXTENT(X, Y, WIDTH, HEIGHT) (br_extent_t) { .arr = { (X), (Y), (WIDTH), (HEIGHT) } }
#define BR_EXTENT2(POS, SIZE) (br_extent_t) { .pos = (POS), .size = (SIZE) }
#define BR_EXTENT_TOBB(E) ((br_bb_t) { .min_x = (E).x, .min_y = (E).y, .max_x = (E).x + (E).width, .max_y = (E).y + (E).height })
#define BR_EXTENTI_TOF(E) ((br_extent_t) { .arr = { (float)(E).arr[0], (float)(E).arr[1], (float)(E).arr[2], (float)(E).arr[3] } })
#define BR_EXTENTI_TOBB(E) ((br_bb_t) { .min_x = (float)(E).x, .min_y = (float)(E).y, .max_x = (float)(E).x + (float)(E).width, .max_y = (float)(E).y + (float)(E).height })
#define BR_EXTENTI_ASPECT(E) ((float)(E).height / (float)(E).width)
#define BR_EXTENTI(X, Y, WIDTH, HEIGHT) (br_extenti_t) { .arr = { (X), (Y), (WIDTH), (HEIGHT) } }
#define BR_EXTENTI2(POS, SIZE) (br_extenti_t) { { .pos = POS, .size = SIZE } }
#define BR_EXTENT_(EX) (EX).x, (EX).y, (EX).width, (EX).height

#define BR_BB(Xm, Ym, XM, YM) (br_bb_t) { .arr = { (Xm), (Ym), (XM), (YM) } }
#define BR_BB_TOEX(BB) (br_extent_t) { .arr = { (BB).min_x, (BB).min_y, (BB).max_x - (BB).min_x, (BB).max_y - (BB).min_y } }
#define BR_BBW(BB) ((BB).max_x - (BB).min_x)
#define BR_BBH(BB) ((BB).max_y - (BB).min_y)
#define BR_BB_(BB) (BB).min_x, (BB).min_y, (BB).max_x, (BB).max_y

#define BR_VEC4(X, Y, Z, W) ((br_vec4_t) { .x = (X), .y = (Y), .z = (Z), .w = (W) }) 
#define BR_VEC42(XY, ZW) ((br_vec4_t) { .xy = (XY), .zw = (ZW)  }) 
#define BR_VEC3(X, Y, Z) ((br_vec3_t) { .x = (X), .y = (Y), .z = (Z) }) 
#define BR_VEC_ELS(X) (sizeof((X).arr) / sizeof((X).arr[0]))
#define BR_COLOR_TO4(X)  BR_VEC4(((X).r) / 255.f, ((X).g) / 255.f, ((X).b) / 255.f, ((X).a) / 255.f)
#define BR_COLOR_COMP(X)  ((X).r), ((X).g), ((X).b), ((X).a)
#define BR_COLOR_COMPF(X)  ((float)(X).r / 255.f), ((float)(X).g / 255.f), ((float)(X).b / 255.f), ((float)(X).a / 255.f)
#define BR_COLOR(R, G, B, A) ((br_color_t) { .r = (R), .g = (G), .b = (B), .a = (A) })

#define BR_RED BR_COLOR(200, 10, 10, 255)
#define BR_LIME BR_COLOR(10, 200, 10, 255)
#define BR_ORANGE BR_COLOR(180, 180, 10, 255)
#define BR_WHITE BR_COLOR(180, 180, 180, 255)
#define BR_GREEN BR_COLOR(10, 200, 10, 255)
#define BR_BLUE BR_COLOR(10, 10, 200, 255)
#define BR_LIGHTGRAY BR_COLOR(100, 100, 100, 255)
#define BR_PINK BR_COLOR(150, 80, 80, 255)
#define BR_GOLD BR_COLOR(50, 180, 50, 255)
#define BR_VIOLET BR_COLOR(180, 10, 180, 255)
#define BR_DARKPURPLE BR_COLOR(80, 10, 80, 255)

#define BR_PI 3.14159265358979323846f
#define BR_Z_MAX 100000
#define BR_Z_TO_GL(__Z) (1 - 2.f*((float)(__Z)/(float)BR_Z_MAX))
#define BR_SCREEN_TO_GL(point, screen) BR_VEC2((float)(point).x / (float)(screen).width * 2.f - 1.f, (float)(point).y / (float)(screen).height * 2.f - 1.f)

typedef struct {
  union {
    struct {
      float x, y;
    };
    float arr[2];
  };
} br_vec2_t;

typedef struct {
  union {
    struct {
      double x, y;
    };
    double arr[2];
  };
} br_vec2d_t;

typedef struct {
  union {
    struct {
      int x, y;
    };
    int arr[2];
  };
} br_vec2i_t;

typedef struct {
  union {
    struct {
      int width, height;
    };
    br_vec2i_t vec;
    int arr[2];
  };
} br_sizei_t;

typedef struct {
  union {
    struct {
      float width, height;
    };
    br_vec2_t vec;
    float arr[2];
  };
} br_size_t;

typedef struct {
  union {
    struct {
      int x, y, width, height;
    };
    struct {
      br_vec2i_t pos;
      br_sizei_t size;
    };
    int arr[4];
  };
} br_extenti_t;

typedef struct {
  union {
    struct {
      float x, y, width, height;
    };
    struct {
      br_vec2_t pos;
      br_size_t size;
    };
    float arr[4];
  };
} br_extent_t;

typedef struct {
  union {
    struct {
      float min_x, min_y, max_x, max_y;
    };
    struct {
      br_vec2_t min;
      br_vec2_t max;
    };
    float arr[4];
  };
} br_bb_t;

typedef struct {
  union {
    struct {
      float x, y, z;
    };
    br_vec2_t xy;
    float arr[3];
  };
}  br_vec3_t;

typedef struct {
  union {
    struct {
      float x, y, z, w;
    };
    struct {
      float r, g, b, a;
    };
    struct {
      br_vec2_t xy;
      br_vec2_t zw;
    };
    struct {
      br_vec3_t xyz;
    };
    float arr[4];
  };
}  br_vec4_t;

typedef struct {
  unsigned char r, g, b, a;
} br_color_t;

typedef struct {
  union {
    struct {
      float m0, m4, m8, m12;
      float m1, m5, m9, m13;
      float m2, m6, m10, m14;
      float m3, m7, m11, m15;
    };
    float arr[16];
    br_vec4_t rows[4];
  };
} br_mat_t;

//------------------------float----------------------------------

static inline float br_float_clamp(float x, float m, float M) {
  if (x < m) return m;
  if (x > M) return M;
  return x;
}

static inline float br_float_lerp(float from, float to, float factor) {
  return from * (1 - factor) + to * (factor);
}

static inline bool br_float_near_zero(float value) {
  return fabsf(value) < 1e-6;
}

//------------------------br_vec2_t------------------------------

static inline br_vec2_t br_vec2i_tof(br_vec2i_t v) {
  return (br_vec2_t) { .x = (float)v.x, .y = (float)v.y };
}

static inline br_vec2i_t br_vec2_toi(br_vec2_t v) {
  return (br_vec2i_t) { .x = (int)v.x, .y = (int)v.y };
}

static inline br_vec2_t br_vec2_max(br_vec2_t a, br_vec2_t b) {
  return BR_VEC2(fmaxf(a.x, b.x), fmaxf(a.y, b.y));
}

static inline br_vec2_t br_vec2_scale(br_vec2_t v, float s) {
  return ((br_vec2_t) { .x = v.x * s, .y = v.y * s });
}

static inline br_vec2_t br_vec2_add(br_vec2_t a, br_vec2_t b) {
  return BR_VEC2(a.x + b.x, a.y + b.y);
}

static inline br_vec2_t br_vec2_adds(br_vec2_t a, br_size_t b) {
  return BR_VEC2(a.x + b.width, a.y + b.height);
}

static inline br_vec2_t br_vec2_sub(br_vec2_t a, br_vec2_t b) {
  return BR_VEC2(a.x - b.x, a.y - b.y);
}

static inline bool br_vec2_eq(br_vec2_t a, br_vec2_t b) {
  return a.x == b.x && a.y == b.y;
}

static inline br_vec2i_t br_vec2i_add(br_vec2i_t a, br_vec2i_t b) {
  return BR_VEC2I(a.x + b.x, a.y + b.y);
}

static inline br_vec2i_t br_vec2i_sub(br_vec2i_t a, br_vec2i_t b) {
  return BR_VEC2I(a.x - b.x, a.y - b.y);
}

static inline float br_vec2_len2(br_vec2_t a) {
  float sum = 0.f;
  for (size_t i = 0; i < BR_VEC_ELS(a); ++i) sum += a.arr[i] * a.arr[i];
  return sum;
}

static inline float br_vec2_len(br_vec2_t a) {
  return sqrtf(br_vec2_len2(a));
}

static inline br_vec2_t br_vec2_mul(br_vec2_t a, br_vec2_t b) {
  return BR_VEC2(a.x * b.x, a.y * b.y);
}

static inline br_vec2_t br_vec2_div(br_vec2_t a, br_vec2_t b) {
  return BR_VEC2(a.x / b.x, a.y / b.y);
}

static inline float br_vec2_dist2(br_vec2_t a, br_vec2_t b) {
  float len2 = br_vec2_len2(br_vec2_sub(a, b));
  return len2;
}

static inline br_vec3_t br_vec2_transform_scale(br_vec2_t v, br_mat_t mat) {
  br_vec3_t result = { 0 };
  float x = v.x;
  float y = v.y;
  result.x = (mat.m0*x + mat.m4*y + mat.m12);
  result.y = (mat.m1*x + mat.m5*y + mat.m13);
  result.z = (mat.m2*x + mat.m6*y + mat.m14);
  if (fabsf(result.z) > 0.00001f) {
    result.x /= fabsf(result.z);
    result.y /= fabsf(result.z);
  }
  return result;
}

static inline br_vec2_t br_vec2_stog(br_vec2_t v, br_sizei_t screen) {
  return BR_VEC2(v.x / (float)screen.width * 2.f - 1.f, (1.f - v.y / (float)screen.height) * 2.f - 1.f);
}

//------------------------size------------------------------

static inline br_sizei_t br_sizei_sub(br_sizei_t a, br_sizei_t b) {
  return BR_SIZEI(a.width - b.width, a.height - b.height);
}

static inline br_size_t br_size_sub(br_size_t a, br_size_t b) {
  return BR_SIZE(a.width - b.width, a.height - b.height);
}

static inline br_size_t br_size_subv(br_size_t a, br_vec2_t b) {
  return BR_SIZE(a.width - b.x, a.height - b.y);
}

static inline br_sizei_t br_sizei_subvi(br_sizei_t a, br_vec2i_t b) {
  return BR_SIZEI(a.width - b.x, a.height - b.y);
}

static inline br_size_t br_size_addv(br_size_t a, br_vec2_t b) {
  return BR_SIZE(a.width + b.x, a.height + b.y);
}

//------------------------vec3------------------------------

static inline br_vec3_t br_vec3_add(br_vec3_t a, br_vec3_t b) {
  return BR_VEC3(a.x + b.x, a.y + b.y, a.z + b.z);
}

static inline br_vec3_t br_vec3_sub(br_vec3_t a, br_vec3_t b) {
  return BR_VEC3(a.x - b.x, a.y - b.y, a.z - b.z);
}

static inline br_vec3_t br_vec3_scale(br_vec3_t a, float s) {
  return BR_VEC3(a.x * s, a.y * s, a.z * s);
}

static inline br_vec3_t br_vec3_pow(br_vec3_t a, float exponent) {
  return BR_VEC3(powf(a.x, exponent), powf(a.y, exponent), powf(a.z, exponent));
}

static inline float br_vec3_len2(br_vec3_t a) {
  float sum = 0.f;
  for (size_t i = 0; i < BR_VEC_ELS(a); ++i) sum += a.arr[i] * a.arr[i];
  return sum;
}

static inline float br_vec3_len(br_vec3_t a) {
  return sqrtf(br_vec3_len2(a));
}

static inline float br_vec3_dist2(br_vec3_t a, br_vec3_t b) {
  float len2 = br_vec3_len2(br_vec3_sub(a, b));
  return len2;
}

static inline float br_vec3_dist(br_vec3_t a, br_vec3_t b) {
  return sqrtf(br_vec3_dist2(a, b));
}

static inline br_vec3_t br_vec3_mul(br_vec3_t a, br_vec3_t b) {
  return BR_VEC3(a.x*b.x, a.y*b.y, a.z*b.z);
}

static inline float br_vec3_dot(br_vec3_t a, br_vec3_t b) {
  return a.x*b.x + a.y*b.y + a.z*b.z;
}

static inline br_vec3_t br_vec3_cross(br_vec3_t a, br_vec3_t b) {
  return BR_VEC3(
    a.y*b.z - a.z*b.y,
    a.z*b.x - a.x*b.z,
    a.x*b.y - a.y*b.x);
}

static inline br_vec3_t br_vec3_normalize(br_vec3_t a) {
  float len2 = br_vec3_len2(a);
  if (fabsf(len2) > FLT_EPSILON) {
    float len = sqrtf(len2);
    for (size_t i = 0; i < BR_VEC_ELS(a); ++i) a.arr[i] /= len;
    return a;
  }
  return BR_VEC3(0,0,0);
}

// TODO: This look sus...
static inline br_vec3_t br_vec3_transform_scale(br_vec3_t v, br_mat_t mat) {
  br_vec3_t result = { 0 };

  float x = v.x, y = v.y, z = v.z;

  result.x = mat.m0*x + mat.m4*y + mat.m8*z + mat.m12;
  result.y = mat.m1*x + mat.m5*y + mat.m9*z + mat.m13;
  result.z = mat.m2*x + mat.m6*y + mat.m10*z + mat.m14;
  //float w =  mat.m3*x + mat.m7*y + mat.m11*z + mat.m15;

  if (fabsf(result.z) > 0.00001f) {
    result.x /= fabsf(result.z);
    result.y /= fabsf(result.z);
  }

  return result;
}

static inline float br_vec3_angle(br_vec3_t v1, br_vec3_t v2) {
  float len = br_vec3_len(br_vec3_cross(v1, v2));
  float dot = br_vec3_dot(v1, v2);
  return atan2f(len, dot);
}

static inline br_vec3_t br_vec3_rot(br_vec3_t v, br_vec3_t axis, float angle) { 
  angle *= .5f;
  br_vec3_t w = br_vec3_scale(axis, sinf(angle));
  br_vec3_t wv = br_vec3_cross(w, v);
  br_vec3_t wwv = br_vec3_cross(w, wv);
  wv = br_vec3_scale(wv, cosf(angle) * 2.f);
  v = br_vec3_add(v, wv);
  v = br_vec3_add(v, br_vec3_scale(wwv, 2.f));
  return v;
}

static inline br_vec3_t br_vec3_perpendicular(br_vec3_t v) {
  size_t min = 0;
  float min_val = v.arr[0];
  br_vec3_t ord = { 0 };

  for (size_t i = 1; i < 3; ++i) {
    float c = fabsf(v.arr[i]);
    if (c < min_val) min = i, min_val = c;
  }
  ord.arr[min] = 1;
  return br_vec3_cross(v, ord);
}

static inline br_vec3_t br_vec3_clamp(br_vec3_t v, float m, float M) {
  v.x = br_float_clamp(v.x, m, M);
  v.y = br_float_clamp(v.y, m, M);
  v.z = br_float_clamp(v.z, m, M);
  return v;
}

// ------------------br_color_t-----------------

static inline br_color_t br_color_lighter(br_color_t c, float factor) {
  br_vec3_t cv = BR_COLOR_TO4(c).xyz;
  br_vec3_t newc = br_vec3_clamp(br_vec3_scale(cv, factor + 1.f), 0.f, 1.f);
  return BR_VEC3_TOC(newc, c.a);
}

static inline br_color_t br_color_darker(br_color_t c, float factor) {
  return br_color_lighter(c, -factor);
}

static inline br_color_t br_color_greener(br_color_t c, float factor) {
  br_vec3_t cv = BR_COLOR_TO4(c).xyz;
  float g = (1+factor);
  float g2 = g*g;
  float k2 = 1 + (cv.y*cv.y)*(1.f - g2)/(cv.x*cv.x+cv.z*cv.z);
  float k = sqrtf(k2);
  cv.x *= k;
  cv.y *= g;
  cv.z *= k;
  return BR_VEC3_TOC(cv, c.a);
}

// ------------------br_extent_t-----------------

static inline br_vec2_t br_extent_tr(br_extent_t extent) {
  return BR_VEC2(extent.x + extent.width, extent.y);
}

static inline br_vec2_t br_extent_bl(br_extent_t extent) {
  return BR_VEC2(extent.x, extent.y + extent.height);
}

static inline br_vec2_t br_extent_br(br_extent_t extent) {
  return BR_VEC2(extent.x + extent.width, extent.y + extent.height);
}

static inline br_vec2_t br_extent_tl2(br_extent_t extent, float x, float y) {
  return BR_VEC2(extent.x + x, extent.y + y);
}

static inline br_vec2_t br_extent_tr2(br_extent_t extent, float x, float y) {
  return BR_VEC2(extent.x + extent.width - x, extent.y + y);
}

static inline bool br_extenti_eq(br_extenti_t a, br_extenti_t b) {
  if (a.x != b.x) return false;
  if (a.y != b.y) return false;
  if (a.width != b.width) return false;
  if (a.height != b.height) return false;
  return true;
}

// ------------------br_mat_t--------------------
static inline br_mat_t br_mat_perspective(float fovY, float aspect, float nearPlane, float farPlane) {
  br_mat_t result = { 0 };

  float top = nearPlane*tanf(fovY*0.5f);
  float bottom = -top;
  float right = top*aspect;
  float left = -right;

  // TODO: Check this out..
  // MatrixFrustum(-right, right, -top, top, near, far);
  float rl = (right - left);
  float tb = (top - bottom);
  float fn = (farPlane - nearPlane);

  result.m0 = (nearPlane*2.0f)/rl;
  result.m5 = (nearPlane*2.0f)/tb;
  result.m8 = (right + left)/rl;
  result.m9 = (top + bottom)/tb;
  result.m10 = -(farPlane + nearPlane)/fn;
  result.m11 = -1.0f;
  result.m14 = -(farPlane*nearPlane*2.0f)/fn;

  return result;
}

static inline br_mat_t br_mat_look_at(br_vec3_t eye, br_vec3_t target, br_vec3_t up) {
  br_vec3_t vz = br_vec3_normalize(br_vec3_sub(eye, target));
  br_vec3_t vx = br_vec3_normalize(br_vec3_cross(up, vz));
  br_vec3_t vy = br_vec3_cross(vz, vx);

  return (br_mat_t) { .rows = {
    BR_VEC4(vx.x, vx.y, vx.z, -br_vec3_dot(vx, eye)),
    BR_VEC4(vy.x, vy.y, vy.z, -br_vec3_dot(vy, eye)),
    BR_VEC4(vz.x, vz.y, vz.z, -br_vec3_dot(vz, eye)),
    BR_VEC4(   0,    0,    0,                   1.f)
  }};
}

static inline br_mat_t br_mat_mul(br_mat_t left, br_mat_t right) {
  br_mat_t result = { 0 };

  result.m0 = left.m0*right.m0 + left.m1*right.m4 + left.m2*right.m8 + left.m3*right.m12;
  result.m1 = left.m0*right.m1 + left.m1*right.m5 + left.m2*right.m9 + left.m3*right.m13;
  result.m2 = left.m0*right.m2 + left.m1*right.m6 + left.m2*right.m10 + left.m3*right.m14;
  result.m3 = left.m0*right.m3 + left.m1*right.m7 + left.m2*right.m11 + left.m3*right.m15;
  result.m4 = left.m4*right.m0 + left.m5*right.m4 + left.m6*right.m8 + left.m7*right.m12;
  result.m5 = left.m4*right.m1 + left.m5*right.m5 + left.m6*right.m9 + left.m7*right.m13;
  result.m6 = left.m4*right.m2 + left.m5*right.m6 + left.m6*right.m10 + left.m7*right.m14;
  result.m7 = left.m4*right.m3 + left.m5*right.m7 + left.m6*right.m11 + left.m7*right.m15;
  result.m8 = left.m8*right.m0 + left.m9*right.m4 + left.m10*right.m8 + left.m11*right.m12;
  result.m9 = left.m8*right.m1 + left.m9*right.m5 + left.m10*right.m9 + left.m11*right.m13;
  result.m10 = left.m8*right.m2 + left.m9*right.m6 + left.m10*right.m10 + left.m11*right.m14;
  result.m11 = left.m8*right.m3 + left.m9*right.m7 + left.m10*right.m11 + left.m11*right.m15;
  result.m12 = left.m12*right.m0 + left.m13*right.m4 + left.m14*right.m8 + left.m15*right.m12;
  result.m13 = left.m12*right.m1 + left.m13*right.m5 + left.m14*right.m9 + left.m15*right.m13;
  result.m14 = left.m12*right.m2 + left.m13*right.m6 + left.m14*right.m10 + left.m15*right.m14;
  result.m15 = left.m12*right.m3 + left.m13*right.m7 + left.m14*right.m11 + left.m15*right.m15;

  return result;
}

// ------------------collisions--------------------
static inline bool br_col_vec2_extent(br_extent_t ex, br_vec2_t point) {
  return (point.x >= ex.x) && (point.x < (ex.x + ex.width)) && (point.y >= ex.y) && (point.y < (ex.y + ex.height));
}
static inline bool br_col_vec2_bb(br_bb_t ex, br_vec2_t point) {
  return (point.x >= ex.min_x) && (point.x < (ex.max_x)) && (point.y >= ex.min_y) && (point.y < ex.max_y);
}

static inline bool br_col_extents(br_extent_t a, br_extent_t b) {
  return ((a.x < (b.x + b.width) && (a.x + a.width) > b.x) &&
          (a.y < (b.y + b.height) && (a.y + a.height) > b.y));
}

// ----------------br_bb_t-------------------------

static inline br_vec2_t br_bb_tr(br_bb_t bb) {
  return BR_VEC2(bb.max.x, bb.min.y);
}

static inline br_vec2_t br_bb_bl(br_bb_t bb) {
  return BR_VEC2(bb.min.x, bb.max.y);
}

static inline br_bb_t br_bb_sub(br_bb_t bb, br_vec2_t vec) {
  return (br_bb_t) { .min = br_vec2_sub(bb.min, vec), .max = br_vec2_sub(bb.max, vec) };
}

static inline br_vec4_t br_bb_clip_dists(br_bb_t limit, br_vec2_t pos) {
  br_vec4_t p = BR_VEC4(
      (pos.x - limit.min_x),
      (pos.y - limit.min_y),
      (limit.max_x - pos.x),
      (limit.max_y - pos.y)
  );
  return p;
}



#  include <stdlib.h>
#endif

#if defined(BR_SHADER_TOOL) || defined(BR_DEBUG)
#  define grid_fs           "src/shaders/grid.fs"
#  define grid_vs           "src/shaders/grid.vs"
#  define line_fs           "src/shaders/line.fs"
#  define line_vs           "src/shaders/line.vs"
#  define quad_fs           "src/shaders/quad.fs"
#  define quad_vs           "src/shaders/quad.vs"
#  define grid_3d_fs        "src/shaders/grid_3d.fs"
#  define grid_3d_vs        "src/shaders/grid_3d.vs"
#  define line_3d_fs        "src/shaders/line_3d.fs"
#  define line_3d_vs        "src/shaders/line_3d.vs"
#  define line_3d_simple_fs "src/shaders/line_3d_simple.fs"
#  define line_3d_simple_vs "src/shaders/line_3d_simple.vs"
#  define font_fs           "src/shaders/font.fs"
#  define font_vs           "src/shaders/font.vs"
#  define img_fs            "src/shaders/img.fs"
#  define img_vs            "src/shaders/img.vs"
#  define icon_fs           "src/shaders/icon.fs"
#  define icon_vs           "src/shaders/icon.vs"
#endif

#define NOP2(N, M)

#define BR_ALL_SHADERS(X, X_VEC, X_BUF) \
  X(grid, 2,                            \
      X_VEC(zoom, 2)                    \
      X_VEC(offset, 2)                  \
      X_VEC(bg_color, 4)                \
      X_VEC(lines_color, 4)             \
      X_VEC(screen, 2),                 \
                                        \
      X_BUF(vertexPosition, 2)          \
    )                                   \
  X(grid_3d, 8,                         \
      X_VEC(eye, 3)                     \
      X_VEC(look_dir, 3)                \
      X_VEC(m_mvp, 16),                 \
                                        \
      X_BUF(vertexPosition, 3)          \
      X_BUF(vertexColor, 3)             \
      X_BUF(z, 1)                       \
    )                                   \
  X(line_3d_simple, 1024,               \
      X_VEC(m_mvp, 16),                 \
                                        \
      X_BUF(vertexPosition, 3)          \
      X_BUF(vertexNormal, 3)            \
      X_BUF(vertexColor, 3)             \
    )                                   \
  X(line_3d, 16*1024,                   \
      X_VEC(eye, 3)                     \
      X_VEC(m_mvp, 16),                 \
                                        \
      X_BUF(vertexPosition, 3)          \
      X_BUF(vertexNormal, 3)            \
      X_BUF(vertexColor, 3)             \
    )                                   \
  X(line, 2048,                         \
      X_VEC(offset, 2)                  \
      X_VEC(zoom, 2)                    \
      X_VEC(screen, 2),                 \
                                        \
      X_BUF(vertexX, 1)                 \
      X_BUF(vertexY, 1)                 \
      X_BUF(delta, 2)                   \
      X_BUF(vertexColor, 3)             \
    )                                   \
  X(icon, 1024,                         \
      X_VEC(atlas, _TEX),               \
                                        \
      X_BUF(pos, 4)                     \
      X_BUF(fg, 4)                      \
      X_BUF(bg, 4)                      \
      X_BUF(clip_dists, 4)              \
      X_BUF(z, 1)                       \
    )                                   \
  X(font, 1024,                         \
      X_VEC(sub_pix_aa_map, 3)          \
      X_VEC(sub_pix_aa_scale, 1)        \
      X_VEC(atlas, _TEX),               \
                                        \
      X_BUF(pos, 4)                     \
      X_BUF(color, 4)                   \
      X_BUF(clip_dists, 4)              \
      X_BUF(z, 1)                       \
    )                                   \
  X(img, 2,                             \
      X_VEC(image, _TEX),               \
                                        \
      X_BUF(pos, 4)                     \
      X_BUF(z, 1)                       \
    )                                   \

#define VEC_TYPE1 float
#define VEC_TYPE2 br_vec2_t
#define VEC_TYPE3 br_vec3_t
#define VEC_TYPE4 br_vec4_t
#define VEC_TYPE16 br_mat_t
#define VEC_TYPE_TEX unsigned int
#define X_U(NAME, DIM) VEC_TYPE ## DIM NAME ## _uv;
#define X(NAME, CAP, U_VEC, BUFFS) \
  typedef struct {                 \
    U_VEC                          \
  } br_shader_ ## NAME ## _uvs_t;
BR_ALL_SHADERS(X, X_U, NOP2)
#undef X
#undef X_U

#define X_U(NAME, DIM) VEC_TYPE ## DIM NAME;
#define X(NAME, CAP, U_VEC, BUFFS) \
  typedef struct {                 \
    BUFFS                          \
  } br_shader_ ## NAME ## _el_t;
BR_ALL_SHADERS(X, NOP2, X_U)
#undef X
#undef X_U

#undef VEC_TYPE_TEX
#undef VEC_TYPE16
#undef VEC_TYPE4
#undef VEC_TYPE3
#undef VEC_TYPE2
#undef VEC_TYPE1

#define X_U(NAME, DIM) int NAME ## _u;
#define X_B(NAME, DIM)                \
  unsigned int NAME ## _vbo_id;       \
  int NAME ## _loc;                   \
  float* NAME ## _vbo;
#define X(NAME, CAP, U_VEC, BUFFS)    \
  typedef struct br_shader_ ## NAME ## _t { \
    br_shader_ ## NAME ## _uvs_t uvs; \
    U_VEC                             \
    BUFFS                             \
    int len, cap;                     \
    unsigned int id;                  \
    unsigned int vao_id;              \
  } br_shader_ ## NAME ## _t;
BR_ALL_SHADERS(X, X_U, X_B)
#undef X
#undef X_B
#undef X_U

#ifdef __cplusplus
extern "C" {
#endif

typedef struct br_shaders_t {
#define X(NAME, CAP, UNIF, BUFF) br_shader_ ## NAME ## _t* NAME;
BR_ALL_SHADERS(X, NOP2, NOP2)
#undef X
} br_shaders_t;

#define X(NAME, CAP, U_VEC, BUFF) void br_shader_ ## NAME ## _draw(br_shader_ ## NAME ## _t* shader);
BR_ALL_SHADERS(X, NOP2, NOP2)
#undef X

br_shaders_t br_shaders_malloc(void);
void br_shaders_draw_all(br_shaders_t shaders);
void br_shaders_free(br_shaders_t shaders);
void br_shaders_refresh(br_shaders_t shaders);

#define X(NAME, CAP, U_VEC, BUFF) void br_shader_ ## NAME ## _push_tri(br_shader_ ## NAME ## _t* shader, br_shader_ ## NAME ## _el_t el[3]);
BR_ALL_SHADERS(X, NOP2, NOP2)
#undef X

#define X(NAME, CAP, U_VEC, BUFF) void br_shader_ ## NAME ## _push_quad(br_shader_ ## NAME ## _t* shader, br_shader_ ## NAME ## _el_t el[4]);
BR_ALL_SHADERS(X, NOP2, NOP2)
#undef X

#ifdef __cplusplus
}
#endif

/* #include "src/br_data.h" */

/* #include "src/br_pp.h" */

/* #include "src/br_str.h" */

/* #include "src/br_pp.h" */


#include <stdint.h>
#include <stdlib.h>
#include <stdbool.h>
#include <stdio.h>

#ifdef __cplusplus
extern "C" {
#define BR_LITERAL(TYPE) TYPE
#else
#define BR_LITERAL(TYPE) (TYPE)
#endif



#define br_str_invalidata(s) s = BR_LITERAL(br_str_t){0}
#define br_str_push_literal(s, literal) br_str_push_br_strv(s, br_strv_from_literal(literal))
#define br_strv_from_literal(literal) (BR_LITERAL(br_strv_t) { literal, sizeof(literal) - 1 })

#ifdef __cplusplus
#else
#define br_str_sub(s, start, new_length) ((br_strv_t) { .str = s.str + (start), .len = (new_length) })
#define br_str_sub1(s, start) ((br_strv_t) { .str = s.str + (start), .len = s.len - (start) })
#define br_str_as_view(s) ((br_strv_t) { .str = s.str, .len = s.len })

#define br_strv_sub(s, start, new_length) ((br_strv_t) { .str = s.str + (start), .len = (new_length) })
#define br_strv_sub1(s, start) ((br_strv_t) { .str = s.str + (start), .len = s.len - (start) })
#endif

#define BR_STRV(STR, LEN) ((br_strv_t) { .str = (STR), .len = (LEN) })
#define BR_STRL(STR) ((br_strv_t) { .str = (STR), .len = (uint32_t)(sizeof((STR)) - 1) })


typedef struct br_str_t {
  char* str;
  unsigned int len;
  unsigned int cap;
} br_str_t;

typedef struct br_strv_t {
  const char* str;
  uint32_t len;
} br_strv_t;

br_str_t   br_str_malloc(size_t size);
void       br_str_free(br_str_t str);
bool       br_str_realloc(br_str_t* s, size_t new_cap);
bool       br_str_push_char(br_str_t* s, char c);
bool       br_str_push_int(br_str_t* s, int c);
bool       br_str_push_float1(br_str_t* s, float c, int decimals);
bool       br_str_push_float(br_str_t* s, float c);
bool       br_str_push_br_str(br_str_t* s, br_str_t const c);
bool       br_str_push_br_strv(br_str_t* s, br_strv_t const c);
bool       br_str_push_c_str(br_str_t* s, char const* c);
bool       br_str_push_uninitialized(br_str_t* s, unsigned int n);
char*      br_str_to_c_str(br_str_t s);
char*      br_str_move_to_c_str(br_str_t* s);
br_str_t   br_str_copy(br_str_t s);
br_str_t   br_str_from_c_str(const char* str);
void       br_str_to_c_str1(br_str_t s, char* out_s);
char*      br_str_to_scrach(br_str_t s);
char*      br_str_move_to_scrach(br_str_t s);

bool       br_strv_eq(br_strv_t s1, br_strv_t s2);
char*      br_strv_to_scrach(br_strv_t s);
char*      br_strv_to_c_str(br_strv_t s);
void       br_strv_to_c_str1(br_strv_t s, char* out_s);
br_strv_t  br_strv_from_c_str(const char* s);
int        br_strv_to_int(br_strv_t str);
br_strv_t  br_strv_trim_zeros(br_strv_t buff);
br_strv_t  br_strv_splitrs(br_strv_t buff, br_strv_t split_strv);
br_strv_t  br_strv_splitr(br_strv_t buff, char splitc);
br_strv_t  br_strv_splitl(br_strv_t buff, char splitc);
br_strv_t  br_strv_skip(br_strv_t buff, char to_skip);
bool       br_strv_starts_with(br_strv_t buff, br_strv_t starts_with);
int        br_strv_count(br_strv_t buff, char ch);

#if defined(BR_RELEASE)
char*      br_scrach_get(size_t len);
void       br_scrach_free(void);
#else
char*      _br_scrach_get(size_t len);
void       _br_scrach_free(void);
extern BR_THREAD_LOCAL const char* br_scrach_alloc_at;
extern BR_THREAD_LOCAL int br_scrach_alloc_at_line;
extern BR_THREAD_LOCAL bool is_taken;
bool br_scrach_check_is_taken(void);
#define br_scrach_get(len) ( \
    br_scrach_check_is_taken(), \
    br_scrach_alloc_at = __FILE__, \
    br_scrach_alloc_at_line = __LINE__, \
    is_taken = true, \
    _br_scrach_get(len))
#define br_scrach_free() do { \
  is_taken = false; \
  _br_scrach_free(); \
} while(0)

#endif

#ifdef __cplusplus
}
#endif

/* #include "src/br_shaders.h" */

/* #include "src/br_math.h" */


#include <stdio.h>

#ifdef __cplusplus
extern "C" {
#endif

typedef struct resampling2_t resampling2_t;
typedef struct br_plot_t br_plot_t;
typedef struct br_plots_t br_plots_t;
typedef struct br_dagens_t br_dagens_t;

typedef struct min_distances_t {
  br_vec2_t graph_point;
  br_vec2_t graph_point_x;
  br_vec2_t graph_point_y;
} min_distances_t;

typedef struct bb_3d_t {
  float xmin, ymin, zmin, xmax, ymax, zmax;
} bb_3d_t;

typedef struct bb_t {
  float xmin, ymin, xmax, ymax;
} bb_t;

#if !defined(BR_DATA_KIND_T)
typedef enum {
  br_data_kind_2d,
  br_data_kind_3d
} br_data_kind_t;
#define BR_DATA_KIND_T
#endif

typedef struct br_data_2d_t {
  float* xs;
  float* ys;
  bb_t bounding_box;
} br_data_2d_t;

typedef struct br_data_3d_t {
  float* xs;
  float* ys;
  float* zs;
  bb_3d_t bounding_box;
} br_data_3d_t;

typedef struct br_data_t {
  resampling2_t* resampling;
  size_t cap, len;
  br_data_kind_t kind;
  int group_id;
  br_color_t color;
  br_str_t name;
  bool is_new;
  union {
    br_data_2d_t dd;
    br_data_3d_t ddd;
  };
} br_data_t;

typedef struct br_datas_t {
  size_t cap, len;
  br_data_t* arr;
} br_datas_t;

typedef struct {
  int group_id;
  br_str_t name;
} br_data_desc_t;

typedef struct {
  br_data_desc_t* arr;
  size_t len, cap;
} br_data_descs_t;

void br_data_construct(void);
int br_datas_get_new_id(br_datas_t* datas);
br_data_t* br_datas_create(br_datas_t* datas, int group_id, br_data_kind_t kind);
br_data_t* br_datas_create(br_datas_t* datas, int group_id, br_data_kind_t kind);
br_data_t* br_datas_create2(br_datas_t* datas, int group_id, br_data_kind_t kind, br_color_t color, size_t cap, br_str_t name);
br_data_t* br_data_get(br_datas_t* pg_array, int group);
br_data_t* br_data_get1(br_datas_t pg, int group);
br_data_t* br_data_get2(br_datas_t* pg_array, int group, br_data_kind_t kind);
void br_data_set_name(br_datas_t* pg_array, int group, br_str_t name);
void br_data_push_y(br_datas_t* pg, float y, int group);
void br_data_push_x(br_datas_t* pg, float x, int group);
void br_data_push_xy(br_datas_t* pg, float x, float y, int group);
void br_data_push_xyz(br_datas_t* pg, float x, float y, float z, int group);
// TODO: this should be br_plotter_clear_data()
void br_data_clear(br_datas_t* pg, br_plots_t* plots, int group_id);
// Only remove all points from a group, don't remove the group itself.
void br_data_empty(br_data_t* pg);
void br_data_export(br_data_t const* pg, FILE* file);
void br_data_export_csv(br_data_t const* pg, FILE* file);
void br_datas_draw(br_datas_t pg_array, br_plot_t* shader, br_shaders_t* shaders);
void br_datas_add_test_points(br_datas_t* pg_array);
void br_datas_deinit(br_datas_t* pg_array);
// Only remove all points from all groups, don't remove groups themselfs.
void br_datas_empty(br_datas_t* pg_array);
void br_datas_export(br_datas_t const* pg_array, FILE* file);
void br_datas_export_csv(br_datas_t const* pg_array, FILE* file);
br_color_t br_data_get_default_color(int group_id);
bool br_data_is_generated(br_dagens_t const* dagens, int groups_id);
bool br_data_realloc(br_data_t* data, size_t new_cap);

size_t br_data_element_size(br_data_kind_t kind);

#ifdef __cplusplus
}
#endif

/* #include "src/br_text_renderer.h" */

/* #include "src/br_str.h" */

/* #include "src/br_math.h" */


typedef struct br_text_renderer_t br_text_renderer_t;
#if !defined(br_shader_font_t)
typedef struct br_shader_font_t br_shader_font_t;
typedef struct br_shader_fontbg_t br_shader_fontbg_t;
#endif

typedef enum {
  br_text_renderer_ancor_x_left  = 1 << 1,
  br_text_renderer_ancor_x_mid   = 1 << 2,
  br_text_renderer_ancor_x_right = 1 << 3,
  br_text_renderer_ancor_y_up    = 1 << 4,
  br_text_renderer_ancor_y_mid   = 1 << 5,
  br_text_renderer_ancor_y_down  = 1 << 6,

  br_text_renderer_ancor_left_up    = br_text_renderer_ancor_x_left  | br_text_renderer_ancor_y_up,
  br_text_renderer_ancor_left_mid   = br_text_renderer_ancor_x_left  | br_text_renderer_ancor_y_mid,
  br_text_renderer_ancor_left_down  = br_text_renderer_ancor_x_left  | br_text_renderer_ancor_y_down,
  br_text_renderer_ancor_mid_up     = br_text_renderer_ancor_x_mid   | br_text_renderer_ancor_y_up,
  br_text_renderer_ancor_mid_mid    = br_text_renderer_ancor_x_mid   | br_text_renderer_ancor_y_mid,
  br_text_renderer_ancor_mid_down   = br_text_renderer_ancor_x_mid   | br_text_renderer_ancor_y_down,
  br_text_renderer_ancor_right_up   = br_text_renderer_ancor_x_right | br_text_renderer_ancor_y_up,
  br_text_renderer_ancor_right_mid  = br_text_renderer_ancor_x_right | br_text_renderer_ancor_y_mid,
  br_text_renderer_ancor_right_down = br_text_renderer_ancor_x_right | br_text_renderer_ancor_y_down,
} br_text_renderer_ancor_t;


#if defined(__cplusplus)
extern "C" {
#endif

br_text_renderer_t* br_text_renderer_malloc(int bitmap_width, int bitmap_height, unsigned char const* font_data, br_shader_font_t** shader);
void br_text_renderer_free(br_text_renderer_t* r);
void br_text_renderer_dump(br_text_renderer_t* r);
br_strv_t br_text_renderer_fit(br_text_renderer_t* r, br_size_t size, int font_size, br_strv_t text);
br_size_t br_text_renderer_measure(br_text_renderer_t* r, int font_size, br_strv_t str);

br_extent_t br_text_renderer_push0(br_text_renderer_t* r, br_vec3_t pos, int font_size, br_color_t color, const char* text);
br_extent_t br_text_renderer_push_strv0(br_text_renderer_t* r, br_vec3_t pos, int font_size, br_color_t color, br_strv_t text);
br_extent_t br_text_renderer_push(br_text_renderer_t* r, br_vec3_t pos, int font_size, br_color_t color, const char* text, br_bb_t limit);
br_extent_t br_text_renderer_push_strv(br_text_renderer_t* r, br_vec3_t pos, int font_size, br_color_t color, br_strv_t text, br_bb_t limit);
br_extent_t br_text_renderer_push2(br_text_renderer_t* r, br_vec3_t pos, int font_size, br_color_t color, br_strv_t str, br_bb_t limit, br_text_renderer_ancor_t ancor);

#if defined(__cplusplus)
}
#endif


/* #include "src/br_math.h" */

/* #include "src/br_data.h" */


#include <stddef.h>

#ifdef __cplusplus
extern "C" {
#endif

typedef struct br_hotreload_state_t br_hotreload_state_t;

#if !defined(BR_PLOT_KIND_T)
typedef enum {
  br_plot_kind_2d,
  br_plot_kind_3d
} br_plot_kind_t;
#define BR_PLOT_KIND_T
#endif

typedef struct br_plot_2d_t {
  // graph_rect is in the graph coordinates.
  //            That is if you zoom in and out, graph_rect will change.
  br_extent_t graph_rect;

  float recoil;
  br_vec2_t mouse_pos;
  br_vec2_t zoom;
  br_vec2_t offset;
  br_vec2_t delta;
  bool show_closest, show_x_closest, show_y_closest;
} br_plot_2d_t;

typedef struct br_plot_3d_t {
  br_vec3_t eye, target, up;
  float fov_y, near_plane, far_plane;
} br_plot_3d_t;

typedef struct br_plot_t {
  struct {
    int* arr;
    int len, cap; 
  } groups_to_show;
  br_extenti_t cur_extent;
  int extent_handle;
  int menu_extent_handle;

  unsigned int texture_id;

  bool follow;
  bool jump_around;
  bool mouse_inside_graph;
  bool is_deleted;

  br_plot_kind_t kind;
  union {
    br_plot_2d_t dd;
    br_plot_3d_t ddd;
  };
} br_plot_t;

typedef struct br_plots_t {
  br_plot_t* arr;
  int len, cap;
} br_plots_t;

typedef struct context_t {
  float min_sampling;
  float cull_min;
  bool debug_bounds;
  size_t alloc_size, alloc_count, alloc_total_size, alloc_total_count, alloc_max_size, alloc_max_count, free_of_unknown_memory;
} context_t;
typedef struct br_plotter_t br_plotter_t;

extern context_t context;

br_vec2_t br_graph_to_screen(br_extent_t graph_rect, br_extenti_t screen_rect, br_vec2_t point);

void br_plot_deinit(br_plot_t* plot);
void br_plot_create_texture(br_plot_t* br);
void br_plot_draw(br_plot_t* plot, br_datas_t datas, br_shaders_t* shaders);
void br_plot_screenshot(br_text_renderer_t* tr, br_plot_t* br, br_shaders_t* shaders, br_datas_t groups, char const* path);
void br_keybinding_handle_keys(br_plotter_t* br, br_plot_t* plot);

void read_input_start(br_plotter_t* br);
void read_input_main_worker(br_plotter_t* br);
int  read_input_read_next(void);
void read_input_stop(void);

min_distances_t min_distances_get(br_vec2_t const* points, size_t points_len, br_vec2_t to);
void            min_distances_get1(min_distances_t* m, br_vec2_t const* points, size_t points_len, br_vec2_t to);


#if BR_HAS_SHADER_RELOAD
// Start watching shaders folder for changes and
// mark gv->shader_dirty flag to true if there were any change to shaders.
void start_refreshing_shaders(br_plotter_t* br);
#endif
#if BR_HAS_HOTRELOAD
void br_hotreload_start(br_hotreload_state_t* s);
#endif

#ifdef __cplusplus
}
#endif


/* #include "src/br_plotter.h" */

/* #include "src/br_data.h" */

/* #include "src/br_plot.h" */

/* #include "src/br_pp.h" */

/* #include "src/br_data_generator.h" */

/* #include "src/br_data.h" */


#include <stdio.h>

typedef enum {
  br_dagen_state_inprogress,
  br_dagen_state_paused,
  br_dagen_state_finished,
  br_dagen_state_failed,
} br_dagen_state_t;

typedef enum {
  br_dagen_kind_file,
  br_dagen_kind_expr,
} br_dagen_kind_t;

typedef struct {
  FILE* file;
  size_t x_left, y_left, z_left;
  size_t num_points;
} br_dagen_file_t;

typedef enum {
  br_dagen_expr_kind_reference_x,
  br_dagen_expr_kind_reference_y,
  br_dagen_expr_kind_reference_z,
  br_dagen_expr_kind_add,
  br_dagen_expr_kind_mul,
} br_dagen_expr_kind_t;

typedef struct br_dagen_expr_t {
  br_dagen_expr_kind_t kind;
  union {
    int group_id;
    struct {
      uint32_t op1;
      uint32_t op2;
    } operands;
  };
} br_dagen_expr_t;

typedef struct {
  br_dagen_expr_t* arr;
  size_t len, cap;
} br_dagen_exprs_t;

typedef struct br_dagen_2d_expr_t {
  uint32_t x_expr_index, y_expr_index;
  br_dagen_exprs_t arena;
} br_dagen_expr_2d_t;

typedef struct {
  br_dagen_kind_t kind;
  br_data_kind_t data_kind;
  br_dagen_state_t state;
  int group_id;
  union {
    br_dagen_file_t file;
    br_dagen_expr_2d_t expr_2d;
  };
} br_dagen_t;

typedef struct br_dagens_t {
  br_dagen_t* arr;
  size_t len, cap;
} br_dagens_t;

typedef struct br_plotter_t br_plotter_t;

#if defined(__cplusplus)
extern "C" {
#endif
bool br_dagen_push_expr_xy(br_dagens_t* dagens, br_datas_t* datas, br_dagen_exprs_t arena, uint32_t x, uint32_t y, int group_id);
bool br_dagens_add_expr_str(br_dagens_t* dagens, br_datas_t* datas, br_strv_t str, int group_id);
bool br_dagen_push_file(br_dagens_t* dagens, br_datas_t* datas, br_data_desc_t* temp_data, FILE* file);
void br_dagens_remove_dependent(br_datas_t* datas, int group_id);
void br_dagens_handle(br_datas_t* datas, br_dagens_t* dagens, br_plots_t* plots, double until);
void br_dagens_free(br_dagens_t* dagens);
#if defined(__cplusplus)
}
#endif

/* #include "src/br_math.h" */

/* #include "src/br_theme.h" */

/* #include "src/br_math.h" */


typedef struct br_theme_t {
  struct {
    br_color_t btn_inactive;
    br_color_t btn_hovered;
    br_color_t btn_active;
    br_color_t btn_txt_inactive;
    br_color_t btn_txt_hovered;
    br_color_t btn_txt_active;

    br_color_t grid_nums;
    br_color_t grid_nums_bg;
    br_color_t grid_lines;

    br_color_t plot_bg;
    br_color_t plot_menu_color;

    br_color_t ui_edge_inactive;
    br_color_t ui_edge_bg_inactive;
    br_color_t ui_edge_active;
    br_color_t ui_edge_bg_active;

    br_color_t bg;
    float highlite_factor;
  } colors;

  struct {
    br_vec2_t padding;
    int font_size;
    float border_thick;
  } ui;
} br_theme_t;

void br_theme_dark(void);
void br_theme_light(void);

void br_theme_reset_ui(void);


/* #include "src/br_permastate.h" */

#include <stdbool.h>

#ifdef __cplusplus
extern "C" {
#endif

typedef struct br_plotter_t br_plotter_t;

typedef enum {
  br_save_state_command_none = 0,
  // command       - br_save_state_command_t
  // N             - int
  // |for i in 0:N 
  // |   plot data - br_plot_t[N]
  // |_ CRC        - uint32_t
  br_save_state_command_save_plots,
  // command                   - br_save_state_command_t
  // number of points (len2)   - size_t
  //|points                    - Vector2[len2]
  //|_CRC                      - uint32_t
  br_save_state_command_save_data_2d,
  // command                   - br_save_state_command_t
  // number of points (len2)   - size_t
  //|points                    - Vector2[len2]
  //|_CRC                      - uint32_t
  br_save_state_command_save_data_3d,
  // command                   - br_save_state_command_t
  // number of datasets        - size_t
  // for dataset in datasets:
  //   dataset id              - size_t
  //   length of dataset name  - size_t
  //   dataset name            - char[len]
  // active plot index         - int
  br_save_state_command_plotter
  
} br_save_state_command_t;

typedef enum br_permastate_status_t {
  br_permastate_status_failed,
  br_permastate_status_ui_loaded,
  br_permastate_status_ok,
} br_permastate_status_t;


void                   br_permastate_save(br_plotter_t* br);
br_permastate_status_t br_permastate_load(br_plotter_t* br);

#ifdef __cplusplus
}
#endif


#ifdef __cplusplus
extern "C" {
#endif

typedef struct br_plotter_t br_plotter_t;
typedef struct q_commands q_commands;
typedef struct br_text_renderer_t br_text_renderer_t;
typedef struct GLFWwindow GLFWwindow;

#if BR_HAS_HOTRELOAD
typedef struct br_hotreload_state_t {
  LOCK(lock)
  void (*func_loop)(struct br_plotter_t* gv);
  void (*func_init)(struct br_plotter_t* gv);
  bool is_init_called;
  void* handl;
} br_hotreload_state_t;
#endif

typedef struct br_plotter_t {
  br_datas_t groups;
  br_plots_t plots;
  br_shaders_t shaders;
  br_dagens_t dagens;

  br_text_renderer_t* text;
  // Any thread can write to this q, only render thread can pop
  q_commands* commands;

  struct {
    GLFWwindow* glfw;
    br_sizei_t size;
    br_extenti_t viewport;
  } win;

  struct {
    br_vec2_t old_pos;
    br_vec2_t pos;
    br_vec2_t delta;
    br_vec2_t scroll;
    struct {
      bool left:1;
      bool right:1;
    } down;
    struct {
      bool left:1;
      bool right:1;
    } pressed;
  } mouse;

  struct {
    bool down[64];
    bool pressed[64];
    int mod;
  } key;

  struct {
    double old; 
    double now; 
    double frame; 
  } time;

#if BR_HAS_HOTRELOAD
  br_hotreload_state_t hot_state;
#endif
  int active_plot_index;
  int menu_extent_handle;

  br_permastate_status_t loaded_status;
#if BR_HAS_SHADER_RELOAD
  bool shaders_dirty;
#endif
  bool should_close;
  bool switch_to_active;
  bool any_2d, any_3d;
  struct {
    br_theme_t theme;
    bool file_saver_inited;
    bool dark_theme;
    bool expand_plots;
    bool expand_optimizations;
    bool expand_ui_styles;
    bool expand_export;
    bool expand_data;
  } ui;
} br_plotter_t;

br_plotter_t* br_plotter_malloc(void);
void        br_plotter_init(br_plotter_t* br);
void        br_plotter_free(br_plotter_t* br);

void        br_plotter_resize(br_plotter_t* br, float width, float height);
br_datas_t* br_plotter_get_br_datas(br_plotter_t* br);
void        br_plotter_switch_2d(br_plotter_t* br);
void        br_plotter_switch_3d(br_plotter_t* br);
void        br_plotter_set_bottom_left(br_plot_t* plot, float left, float bottom);
void        br_plotter_set_top_right(br_plot_t* plot, float right, float top);
void        br_plots_focus_visible(br_plots_t plot, br_datas_t groups);
void        br_plot_focus_visible(br_plot_t* plot, br_datas_t groups);
int                br_plotter_add_plot_2d(br_plotter_t* br);
int                br_plotter_add_plot_3d(br_plotter_t* br);
void               br_plotter_remove_plot(br_plotter_t* br, int plot_index);
void               br_plotter_export(br_plotter_t const* br, char const* path);
void               br_plotter_export_csv(br_plotter_t const* br, char const* path);
void        br_plotter_draw(br_plotter_t* br);
void        br_plotter_minimal(br_plotter_t* br);
void        br_plotter_frame_end(br_plotter_t* br);
void        br_plotter_datas_deinit(br_plotter_t* br);

// Platform specific
void br_plotter_wait(br_plotter_t const* br);
void br_plotter_init_specifics_platform(br_plotter_t* br, int width, int height);
void br_plotter_deinit_specifics_platform(br_plotter_t* br);
void br_plotter_begin_drawing(br_plotter_t* br);
void br_plotter_end_drawing(br_plotter_t* br);

#ifdef __cplusplus
}
#endif


/* #include "src/br_pp.h" */

/* #include "src/br_gl.h" */


#define BRGL_TEX_GRAY 1


/* Stencil */
#define GL_STENCIL_BITS				0x0D57
#define GL_STENCIL_TEST				0x0B90
#define GL_STENCIL_CLEAR_VALUE			0x0B91
#define GL_STENCIL_FUNC				0x0B92
#define GL_STENCIL_VALUE_MASK			0x0B93
#define GL_STENCIL_FAIL				0x0B94
#define GL_STENCIL_PASS_DEPTH_FAIL		0x0B95
#define GL_STENCIL_PASS_DEPTH_PASS		0x0B96
#define GL_STENCIL_REF				0x0B97
#define GL_STENCIL_WRITEMASK			0x0B98
#define GL_STENCIL_INDEX			0x1901
#define GL_KEEP					0x1E00
#define GL_REPLACE				0x1E01
#define GL_INCR					0x1E02
#define GL_DECR					0x1E03

/* Blending */
#define GL_BLEND				0x0BE2
#define GL_BLEND_SRC				0x0BE1
#define GL_BLEND_DST				0x0BE0
#define GL_SRC_COLOR				0x0300
#define GL_ONE_MINUS_SRC_COLOR			0x0301
#define GL_SRC_ALPHA				0x0302
#define GL_ONE_MINUS_SRC_ALPHA			0x0303
#define GL_DST_ALPHA				0x0304
#define GL_ONE_MINUS_DST_ALPHA			0x0305
#define GL_DST_COLOR				0x0306
#define GL_ONE_MINUS_DST_COLOR			0x0307
#define GL_SRC_ALPHA_SATURATE			0x0308

#define GL_FALSE 0
#define GL_ONE 1
#define GL_TRIANGLES 0x0004
#define GL_SRC_COLOR 0x0300
#define GL_ONE_MINUS_SRC_COLOR 0x0301
#define GL_SRC_ALPHA 0x0302
#define GL_DST_ALPHA 0x0304
#define GL_DST_COLOR 0x0306
#define GL_CULL_FACE 0x0B44
#define GL_DEPTH_TEST 0x0B71
#define GL_BLEND 0x0BE2
#define GL_UNPACK_ALIGNMENT 0x0CF5
#define GL_TEXTURE_2D 0x0DE1
#define GL_UNSIGNED_BYTE 0x1401
#define GL_TEXTURE0 0x84C0
#define GL_FLOAT 0x1406
#define GL_RED 0x1903
#define GL_RGB 0x1907
#define GL_RGBA	0x1908
#define GL_CLIP_DISTANCE0 0x3000
#define GL_FUNC_ADD	0x8006
#define GL_MIN					0x8007
#define GL_MAX 0x8008
#define GL_R8 0x8229
#define GL_ARRAY_BUFFER 0x8892
#define GL_STATIC_DRAW 0x88E4
#define GL_DYNAMIC_DRAW 0x88E8
#define GL_DEPTH24_STENCIL8 0x88F0
#define GL_FRAGMENT_SHADER 0x8B30
#define GL_VERTEX_SHADER 0x8B31
#define GL_COMPILE_STATUS 0x8B81
#define GL_LINK_STATUS 0x8B82
#define GL_INFO_LOG_LENGTH 0x8B84
#define GL_DEPTH_STENCIL_ATTACHMENT 0x821A
#define GL_COLOR_ATTACHMENT0 0x8CE0
#define GL_FRAMEBUFFER 0x8D40
#define GL_RENDERBUFFER 0x8D41
#define GL_DRAW_FRAMEBUFFER 0x8CA9
#define GL_TEXTURE_SWIZZLE_RGBA 0x8E46

#define BR_KEY_TWO 50
#define BR_KEY_THREE 51
#define BR_KEY_C 67
#define BR_KEY_D 68
#define BR_KEY_F 70
#define BR_KEY_H 72
#define BR_KEY_J 74
#define BR_KEY_K 75
#define BR_KEY_R 82
#define BR_KEY_S 83
#define BR_KEY_T 84
#define BR_KEY_U 85
#define BR_KEY_X 88
#define BR_KEY_Y 89

typedef int GLint;
typedef unsigned int GLenum;
typedef unsigned int GLuint;
typedef int GLsizei;
typedef char GLchar;
typedef unsigned char GLboolean;
typedef float GLfloat;
typedef unsigned int GLbitfield;

#if defined(_WIN64)
typedef signed   long long int GLsizeiptr;
typedef unsigned long long int khronos_usize_t;
typedef signed   long long int GLintptr;
typedef unsigned long long int khronos_uintptr_t;
#else
typedef signed   long  int     GLsizeiptr;
typedef unsigned long  int     khronos_usize_t;
typedef signed   long  int     GLintptr;
typedef unsigned long  int     khronos_uintptr_t;
#endif

void brgl_load(void);
unsigned int brgl_load_shader(const char* vs, const char* fs, int* out_ok);
unsigned int brgl_compile_shader(const char* code, GLenum type);
void brgl_unload_shader(GLuint id);
void brgl_enable_shader(GLuint id);
void brgl_blend_equation(GLenum mode);
void brgl_blend_func(GLenum sfactor, GLenum dfactor);
void brgl_blend_func_sep(GLenum srcRGB, GLenum dstRGB, GLenum srcAlpha, GLenum dstAlpha);
void brgl_enable(GLenum mode);
void brgl_disable(GLenum mode);
void brgl_enable_back_face_cull(void);
void brgl_disable_back_face_cull(void);
void brgl_enable_depth_test(void);
void brgl_disable_depth_test(void);
void brgl_enable_clip_distance(void);
void brgl_disable_clip_distance(void);
void brgl_viewport(GLint x, GLint y, GLsizei width, GLsizei height);
GLuint brgl_load_texture(const void* data, int width, int height, int format);
void brgl_unload_texture(GLuint tex_id);

GLuint brgl_create_framebuffer(int width, int height);
GLuint brgl_framebuffer_to_texture(GLuint br_id);
void brgl_enable_framebuffer(GLuint fb_id, int new_width, int new_height);
void brgl_destroy_framebuffer(GLuint fb_id);

GLuint brgl_load_vao(void);
void brgl_enable_vao(GLuint vao);
void brgl_draw_vao(GLint first, GLsizei count);
void brgl_disable_vao(void);
void brgl_unload_vao(GLuint id);

GLuint brgl_load_vbo(void const* data, GLsizeiptr len, GLboolean dynamic);
void brgl_update_vbo(GLuint id, void const* data, GLsizeiptr size, GLintptr offset);
void brgl_enable_vbo(GLuint id);
void brgl_unload_vbo(GLuint id);

void brgl_enable_vattr(GLuint id);
void brgl_set_vattr(GLuint index, GLint compSize, GLenum type, GLboolean normalized, GLint stride, void const* pointer);
void brgl_set_usamp(GLint uni, GLuint tex);
void brgl_set_umatrix(GLint uni, float* tex);
void brgl_set_u(GLint uni, float* tex, int els, int n);
GLint brgl_get_loca(GLuint shader_id, char const* name);
GLint brgl_get_locu(GLuint shader_id, char const* name);

void brgl_clear(float r, float g, float b, float a);

/* #include "src/br_tl.h" */

// Thread local stuff - this is implemented somewhere in the desktop/platfrom.c
/* #include "src/br_math.h" */


#include <stdbool.h>
#define BR_THEME (*brtl_theme())

typedef struct br_plotter_t br_plotter_t;
typedef struct br_shaders_t br_shaders_t;
typedef struct br_text_renderer_t br_text_renderer_t;
typedef struct br_theme_t br_theme_t;

#ifdef __cplusplus
extern "C" {
#endif

br_vec2_t brtl_mouse_scroll(void);
br_vec2_t brtl_mouse_pos(void);
br_vec2_t brtl_mouse_delta(void);
bool brtl_mousel_down(void);
bool brtl_mouser_down(void);
bool brtl_mousel_pressed(void);
bool brtl_mouser_pressed(void);
double brtl_time(void);
float brtl_frame_time(void);
int brtl_fps(void);
bool brtl_key_down(int key);
bool brtl_key_pressed(int key);
bool brtl_key_ctrl(void);
bool brtl_key_alt(void);
bool brtl_key_shift(void);
br_theme_t* brtl_theme(void);

br_sizei_t brtl_window_size(void);
void brtl_window_size_set(int width, int height);

br_extenti_t brtl_viewport(void);
void brtl_viewport_set(br_extenti_t ex);

br_plotter_t* brtl_plotter(void);
br_shaders_t* brtl_shaders(void);
br_text_renderer_t* brtl_text_renderer(void);

#ifdef __cplusplus
}
#endif

/* #include "src/br_q.h" */

/* #include "src/br_data.h" */

/* #include "src/br_pp.h" */

/* #include "src/br_str.h" */


#ifdef __cplusplus
extern "C" {
#endif

typedef struct br_plotter_t br_plotter_t;

typedef enum {
  q_command_none,
  q_command_set_zoom_x,
  q_command_set_zoom_y,
  q_command_set_offset_x,
  q_command_set_offset_y,
  q_command_push_point_x,
  q_command_push_point_y,
  q_command_push_point_xy,
  q_command_push_point_xyz,
  q_command_pop,
  q_command_empty,
  q_command_clear,
  q_command_clear_all,
  q_command_screenshot,
  q_command_export,
  q_command_exportcsv,
  q_command_hide,
  q_command_show,
  q_command_set_name,
  q_command_focus,
  q_command_flush,
  q_command_new_data
} q_command_type;

typedef struct q_command {
  q_command_type type;
  union {
    float value;
    struct {
      int group;
      float x;
    } push_point_x;
    struct {
      int group;
      float y;
    } push_point_y;
    struct {
      int group;
      float x, y;
    } push_point_xy;
    struct {
      int group;
      float x, y, z;
    } push_point_xyz;
    struct {
      int group;
    } pop;
    struct {
      int group;
    } clear;
    struct {
      // Should be freed by UI thread
      char* path;
    } path_arg;
    struct {
      int group;
    } hide_show;
    struct {
      int group;
      // Should be freed by UI thread
      br_str_t str;
    } set_quoted_str;
    struct {
      int data_id;
      br_data_kind_t kind;
    } new_data;
  };
} q_command;

typedef struct q_commands {
  size_t read_index, write_index;
  size_t capacity;
  q_command* commands;
  LOCK(push_mutex)
} q_commands;

q_commands* q_malloc(void);
void q_free(q_commands* q);

#ifdef LOCK_T
// If you know that only one thread writes to q use q_push, else use this.
bool q_push_safe(q_commands* q, q_command command);
#endif
// If you know that only one thread writes to q us this, else use q_push_safe.
bool q_push(q_commands* q, q_command command);
q_command q_pop(q_commands* q);
q_command q_peek(q_commands* q);

void handle_all_commands(br_plotter_t* br, q_commands* commands);

#ifdef __cplusplus
}
#endif

/* #include "src/br_icons.h" */

/* #include ".generated/icons.h" */

// Generated by tools/pack_icons.c
/* #include "src/br_math.h" */


typedef struct {
  struct {
    br_extent_t size_16;
    br_extent_t size_32;
  } menu;
  struct {
    br_extent_t size_32;
  } back;
  struct {
    br_extent_t size_32;
  } cb_0;
  struct {
    br_extent_t size_32;
  } cb_1;
  struct {
    br_extent_t size_8;
  } edge;
} br_icons_t;
extern const br_icons_t br_icons;
extern const unsigned char br_icons_atlas[128*128];
extern const int br_icons_atlas_width;
extern const int br_icons_atlas_height;


#if defined(__cplusplus)
extern "C" {
#endif

typedef struct br_shader_icon_t br_shader_icon_t;


extern struct br_extra_icons_t {
  struct {
    br_extent_t size_8;
  } edge_tl;
  struct {
    br_extent_t size_8;
  } edge_tr;
  struct {
    br_extent_t size_8;
  } edge_br;
  struct {
    br_extent_t size_8;
  } edge_b;
  struct {
    br_extent_t size_8;
  } edge_t;
  struct {
    br_extent_t size_8;
  } edge_l;
  struct {
    br_extent_t size_8;
  } edge_r;
} br_extra_icons;

extern unsigned char const br_font_data[];
extern long long const br_font_data_size;

void br_icons_init(br_shader_icon_t* shader);
void br_icons_draw(br_bb_t screen, br_bb_t atlas, br_color_t bg, br_color_t fg, br_bb_t limit, int z);
void br_icons_deinit(void);
br_extent_t br_icons_y_mirror(br_extent_t icon);
br_extent_t br_icons_top(br_extent_t icon, float percent);

br_extent_t br_icons_bot(br_extent_t icon, float percent);
br_extent_t br_icons_left(br_extent_t icon, float percent);
br_extent_t br_icons_right(br_extent_t icon, float percent);

br_extent_t br_icons_tc(br_extent_t icon, float top, float center);
br_extent_t br_icons_lc(br_extent_t icon, float left, float center);
br_extent_t br_icons_bc(br_extent_t icon, float top, float center);
br_extent_t br_icons_rc(br_extent_t icon, float left, float center);

#if defined(__cplusplus)
}
#endif


#include <string.h>

#define GLAD_MALLOC BR_MALLOC
#define GLAD_FREE BR_FREE

#if !defined(_MSC_VER)
#  pragma GCC diagnostic push
#  pragma GCC diagnostic ignored "-Wreturn-type"
#  pragma GCC diagnostic ignored "-Wunused-parameter"
#  pragma GCC diagnostic ignored "-Wsign-conversion"
#  pragma GCC diagnostic ignored "-Wsign-compare"
#  pragma GCC diagnostic ignored "-Wfloat-conversion"
#  pragma GCC diagnostic ignored "-Wmissing-field-initializers"
#  pragma GCC diagnostic ignored "-Wpedantic"
#  pragma GCC diagnostic ignored "-Wconversion"
#  pragma GCC diagnostic ignored "-Wshadow"
#  pragma GCC diagnostic ignored "-Wcast-function-type"
#endif

/* #include "external/glfw/include/GLFW/glfw3.h" */
/*************************************************************************
 * GLFW 3.4 - www.glfw.org
 * A library for OpenGL, window and input
 *------------------------------------------------------------------------
 * Copyright (c) 2002-2006 Marcus Geelnard
 * Copyright (c) 2006-2019 Camilla Löwy <elmindreda@glfw.org>
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would
 *    be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must not
 *    be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 *************************************************************************/

#ifndef _glfw3_h_
#define _glfw3_h_

#ifdef __cplusplus
extern "C" {
#endif


/*************************************************************************
 * Doxygen documentation
 *************************************************************************/

/*! @file glfw3.h
 *  @brief The header of the GLFW 3 API.
 *
 *  This is the header file of the GLFW 3 API.  It defines all its types and
 *  declares all its functions.
 *
 *  For more information about how to use this file, see @ref build_include.
 */
/*! @defgroup context Context reference
 *  @brief Functions and types related to OpenGL and OpenGL ES contexts.
 *
 *  This is the reference documentation for OpenGL and OpenGL ES context related
 *  functions.  For more task-oriented information, see the @ref context_guide.
 */
/*! @defgroup vulkan Vulkan support reference
 *  @brief Functions and types related to Vulkan.
 *
 *  This is the reference documentation for Vulkan related functions and types.
 *  For more task-oriented information, see the @ref vulkan_guide.
 */
/*! @defgroup init Initialization, version and error reference
 *  @brief Functions and types related to initialization and error handling.
 *
 *  This is the reference documentation for initialization and termination of
 *  the library, version management and error handling.  For more task-oriented
 *  information, see the @ref intro_guide.
 */
/*! @defgroup input Input reference
 *  @brief Functions and types related to input handling.
 *
 *  This is the reference documentation for input related functions and types.
 *  For more task-oriented information, see the @ref input_guide.
 */
/*! @defgroup monitor Monitor reference
 *  @brief Functions and types related to monitors.
 *
 *  This is the reference documentation for monitor related functions and types.
 *  For more task-oriented information, see the @ref monitor_guide.
 */
/*! @defgroup window Window reference
 *  @brief Functions and types related to windows.
 *
 *  This is the reference documentation for window related functions and types,
 *  including creation, deletion and event polling.  For more task-oriented
 *  information, see the @ref window_guide.
 */


/*************************************************************************
 * Compiler- and platform-specific preprocessor work
 *************************************************************************/

/* If we are we on Windows, we want a single define for it.
 */
#if !defined(_WIN32) && (defined(__WIN32__) || defined(WIN32) || defined(__MINGW32__))
 #define _WIN32
#endif /* _WIN32 */

/* Include because most Windows GLU headers need wchar_t and
 * the macOS OpenGL header blocks the definition of ptrdiff_t by glext.h.
 * Include it unconditionally to avoid surprising side-effects.
 */
#include <stddef.h>

/* Include because it is needed by Vulkan and related functions.
 * Include it unconditionally to avoid surprising side-effects.
 */
#include <stdint.h>

/* It is customary to use APIENTRY for OpenGL function pointer declarations on
 * all platforms.  Additionally, the Windows OpenGL header needs APIENTRY.
 */
#if !defined(APIENTRY)
 #if defined(_WIN32)
  #define APIENTRY __stdcall
 #else
  #define APIENTRY
 #endif
 #define GLFW_APIENTRY_DEFINED
#endif /* APIENTRY */

/* Some Windows OpenGL headers need this.
 */
#if !defined(WINGDIAPI) && defined(_WIN32)
 #define WINGDIAPI __declspec(dllimport)
 #define GLFW_WINGDIAPI_DEFINED
#endif /* WINGDIAPI */

/* Some Windows GLU headers need this.
 */
#if !defined(CALLBACK) && defined(_WIN32)
 #define CALLBACK __stdcall
 #define GLFW_CALLBACK_DEFINED
#endif /* CALLBACK */

/* Include the chosen OpenGL or OpenGL ES headers.
 */
#if defined(GLFW_INCLUDE_ES1)

 #include <GLES/gl.h>
 #if defined(GLFW_INCLUDE_GLEXT)
  #include <GLES/glext.h>
 #endif

#elif defined(GLFW_INCLUDE_ES2)

 #include <GLES2/gl2.h>
 #if defined(GLFW_INCLUDE_GLEXT)
  #include <GLES2/gl2ext.h>
 #endif

#elif defined(GLFW_INCLUDE_ES3)

 #include <GLES3/gl3.h>
 #if defined(GLFW_INCLUDE_GLEXT)
  #include <GLES2/gl2ext.h>
 #endif

#elif defined(GLFW_INCLUDE_ES31)

 #include <GLES3/gl31.h>
 #if defined(GLFW_INCLUDE_GLEXT)
  #include <GLES2/gl2ext.h>
 #endif

#elif defined(GLFW_INCLUDE_ES32)

 #include <GLES3/gl32.h>
 #if defined(GLFW_INCLUDE_GLEXT)
  #include <GLES2/gl2ext.h>
 #endif

#elif defined(GLFW_INCLUDE_GLCOREARB)

 #if defined(__APPLE__)

  #include <OpenGL/gl3.h>
  #if defined(GLFW_INCLUDE_GLEXT)
   #include <OpenGL/gl3ext.h>
  #endif /*GLFW_INCLUDE_GLEXT*/

 #else /*__APPLE__*/

  #include <GL/glcorearb.h>
  #if defined(GLFW_INCLUDE_GLEXT)
   #include <GL/glext.h>
  #endif

 #endif /*__APPLE__*/

#elif defined(GLFW_INCLUDE_GLU)

 #if defined(__APPLE__)

  #if defined(GLFW_INCLUDE_GLU)
   #include <OpenGL/glu.h>
  #endif

 #else /*__APPLE__*/

  #if defined(GLFW_INCLUDE_GLU)
   #include <GL/glu.h>
  #endif

 #endif /*__APPLE__*/

#elif !defined(GLFW_INCLUDE_NONE) && \
      !defined(__gl_h_) && \
      !defined(__gles1_gl_h_) && \
      !defined(__gles2_gl2_h_) && \
      !defined(__gles2_gl3_h_) && \
      !defined(__gles2_gl31_h_) && \
      !defined(__gles2_gl32_h_) && \
      !defined(__gl_glcorearb_h_) && \
      !defined(__gl2_h_) /*legacy*/ && \
      !defined(__gl3_h_) /*legacy*/ && \
      !defined(__gl31_h_) /*legacy*/ && \
      !defined(__gl32_h_) /*legacy*/ && \
      !defined(__glcorearb_h_) /*legacy*/ && \
      !defined(__GL_H__) /*non-standard*/ && \
      !defined(__gltypes_h_) /*non-standard*/ && \
      !defined(__glee_h_) /*non-standard*/

 #if defined(__APPLE__)

//  #if !defined(GLFW_INCLUDE_GLEXT)
//   #define GL_GLEXT_LEGACY
//  #endif
//  #include <OpenGL/gl.h>

 #else /*__APPLE__*/

//  #include <GL/gl.h>
//  #if defined(GLFW_INCLUDE_GLEXT)
//   #include <GL/glext.h>
//  #endif

 #endif /*__APPLE__*/

#endif /* OpenGL and OpenGL ES headers */

#if defined(GLFW_DLL) && defined(_GLFW_BUILD_DLL)
 /* GLFW_DLL must be defined by applications that are linking against the DLL
  * version of the GLFW library.  _GLFW_BUILD_DLL is defined by the GLFW
  * configuration header when compiling the DLL version of the library.
  */
 #error "You must not have both GLFW_DLL and _GLFW_BUILD_DLL defined"
#endif

/* GLFWAPI is used to declare public API functions for export
 * from the DLL / shared library / dynamic library.
 */
#if defined(_WIN32) && defined(_GLFW_BUILD_DLL)
 /* We are building GLFW as a Win32 DLL */
 #define GLFWAPI __declspec(dllexport)
#elif defined(_WIN32) && defined(GLFW_DLL)
 /* We are calling a GLFW Win32 DLL */
 #define GLFWAPI __declspec(dllimport)
#elif defined(__GNUC__) && defined(_GLFW_BUILD_DLL)
 /* We are building GLFW as a Unix shared library */
 #define GLFWAPI __attribute__((visibility("default")))
#else
 #define GLFWAPI
#endif


/*************************************************************************
 * GLFW API tokens
 *************************************************************************/

/*! @name GLFW version macros
 *  @{ */
/*! @brief The major version number of the GLFW header.
 *
 *  The major version number of the GLFW header.  This is incremented when the
 *  API is changed in non-compatible ways.
 *  @ingroup init
 */
#define GLFW_VERSION_MAJOR          3
/*! @brief The minor version number of the GLFW header.
 *
 *  The minor version number of the GLFW header.  This is incremented when
 *  features are added to the API but it remains backward-compatible.
 *  @ingroup init
 */
#define GLFW_VERSION_MINOR          4
/*! @brief The revision number of the GLFW header.
 *
 *  The revision number of the GLFW header.  This is incremented when a bug fix
 *  release is made that does not contain any API changes.
 *  @ingroup init
 */
#define GLFW_VERSION_REVISION       0
/*! @} */

/*! @brief One.
 *
 *  This is only semantic sugar for the number 1.  You can instead use `1` or
 *  `true` or `_True` or `GL_TRUE` or `VK_TRUE` or anything else that is equal
 *  to one.
 *
 *  @ingroup init
 */
#define GLFW_TRUE                   1
/*! @brief Zero.
 *
 *  This is only semantic sugar for the number 0.  You can instead use `0` or
 *  `false` or `_False` or `GL_FALSE` or `VK_FALSE` or anything else that is
 *  equal to zero.
 *
 *  @ingroup init
 */
#define GLFW_FALSE                  0

/*! @name Key and button actions
 *  @{ */
/*! @brief The key or mouse button was released.
 *
 *  The key or mouse button was released.
 *
 *  @ingroup input
 */
#define GLFW_RELEASE                0
/*! @brief The key or mouse button was pressed.
 *
 *  The key or mouse button was pressed.
 *
 *  @ingroup input
 */
#define GLFW_PRESS                  1
/*! @brief The key was held down until it repeated.
 *
 *  The key was held down until it repeated.
 *
 *  @ingroup input
 */
#define GLFW_REPEAT                 2
/*! @} */

/*! @ingroup input
 */
#define GLFW_KEY_UNKNOWN            -1

/*! @} */

/*! @defgroup keys Keyboard key tokens
 *  @brief Keyboard key tokens.
 *
 *  See [key input](@ref input_key) for how these are used.
 *
 *  These key codes are inspired by the _USB HID Usage Tables v1.12_ (p. 53-60),
 *  but re-arranged to map to 7-bit ASCII for printable keys (function keys are
 *  put in the 256+ range).
 *
 *  The naming of the key codes follow these rules:
 *   - The US keyboard layout is used
 *   - Names of printable alphanumeric characters are used (e.g. "A", "R",
 *     "3", etc.)
 *   - For non-alphanumeric characters, Unicode:ish names are used (e.g.
 *     "COMMA", "LEFT_SQUARE_BRACKET", etc.). Note that some names do not
 *     correspond to the Unicode standard (usually for brevity)
 *   - Keys that lack a clear US mapping are named "WORLD_x"
 *   - For non-printable keys, custom names are used (e.g. "F4",
 *     "BACKSPACE", etc.)
 *
 *  @ingroup input
 *  @{
 */

/* Printable keys */
#define GLFW_KEY_SPACE              32
#define GLFW_KEY_APOSTROPHE         39  /* ' */
#define GLFW_KEY_COMMA              44  /* , */
#define GLFW_KEY_MINUS              45  /* - */
#define GLFW_KEY_PERIOD             46  /* . */
#define GLFW_KEY_SLASH              47  /* / */
#define GLFW_KEY_0                  48
#define GLFW_KEY_1                  49
#define GLFW_KEY_2                  50
#define GLFW_KEY_3                  51
#define GLFW_KEY_4                  52
#define GLFW_KEY_5                  53
#define GLFW_KEY_6                  54
#define GLFW_KEY_7                  55
#define GLFW_KEY_8                  56
#define GLFW_KEY_9                  57
#define GLFW_KEY_SEMICOLON          59  /* ; */
#define GLFW_KEY_EQUAL              61  /* = */
#define GLFW_KEY_A                  65
#define GLFW_KEY_B                  66
#define GLFW_KEY_C                  67
#define GLFW_KEY_D                  68
#define GLFW_KEY_E                  69
#define GLFW_KEY_F                  70
#define GLFW_KEY_G                  71
#define GLFW_KEY_H                  72
#define GLFW_KEY_I                  73
#define GLFW_KEY_J                  74
#define GLFW_KEY_K                  75
#define GLFW_KEY_L                  76
#define GLFW_KEY_M                  77
#define GLFW_KEY_N                  78
#define GLFW_KEY_O                  79
#define GLFW_KEY_P                  80
#define GLFW_KEY_Q                  81
#define GLFW_KEY_R                  82
#define GLFW_KEY_S                  83
#define GLFW_KEY_T                  84
#define GLFW_KEY_U                  85
#define GLFW_KEY_V                  86
#define GLFW_KEY_W                  87
#define GLFW_KEY_X                  88
#define GLFW_KEY_Y                  89
#define GLFW_KEY_Z                  90
#define GLFW_KEY_LEFT_BRACKET       91  /* [ */
#define GLFW_KEY_BACKSLASH          92  /* \ */
#define GLFW_KEY_RIGHT_BRACKET      93  /* ] */
#define GLFW_KEY_GRAVE_ACCENT       96  /* ` */
#define GLFW_KEY_WORLD_1            161 /* non-US #1 */
#define GLFW_KEY_WORLD_2            162 /* non-US #2 */

/* Function keys */
#define GLFW_KEY_ESCAPE             256
#define GLFW_KEY_ENTER              257
#define GLFW_KEY_TAB                258
#define GLFW_KEY_BACKSPACE          259
#define GLFW_KEY_INSERT             260
#define GLFW_KEY_DELETE             261
#define GLFW_KEY_RIGHT              262
#define GLFW_KEY_LEFT               263
#define GLFW_KEY_DOWN               264
#define GLFW_KEY_UP                 265
#define GLFW_KEY_PAGE_UP            266
#define GLFW_KEY_PAGE_DOWN          267
#define GLFW_KEY_HOME               268
#define GLFW_KEY_END                269
#define GLFW_KEY_CAPS_LOCK          280
#define GLFW_KEY_SCROLL_LOCK        281
#define GLFW_KEY_NUM_LOCK           282
#define GLFW_KEY_PRINT_SCREEN       283
#define GLFW_KEY_PAUSE              284
#define GLFW_KEY_F1                 290
#define GLFW_KEY_F2                 291
#define GLFW_KEY_F3                 292
#define GLFW_KEY_F4                 293
#define GLFW_KEY_F5                 294
#define GLFW_KEY_F6                 295
#define GLFW_KEY_F7                 296
#define GLFW_KEY_F8                 297
#define GLFW_KEY_F9                 298
#define GLFW_KEY_F10                299
#define GLFW_KEY_F11                300
#define GLFW_KEY_F12                301
#define GLFW_KEY_F13                302
#define GLFW_KEY_F14                303
#define GLFW_KEY_F15                304
#define GLFW_KEY_F16                305
#define GLFW_KEY_F17                306
#define GLFW_KEY_F18                307
#define GLFW_KEY_F19                308
#define GLFW_KEY_F20                309
#define GLFW_KEY_F21                310
#define GLFW_KEY_F22                311
#define GLFW_KEY_F23                312
#define GLFW_KEY_F24                313
#define GLFW_KEY_F25                314
#define GLFW_KEY_KP_0               320
#define GLFW_KEY_KP_1               321
#define GLFW_KEY_KP_2               322
#define GLFW_KEY_KP_3               323
#define GLFW_KEY_KP_4               324
#define GLFW_KEY_KP_5               325
#define GLFW_KEY_KP_6               326
#define GLFW_KEY_KP_7               327
#define GLFW_KEY_KP_8               328
#define GLFW_KEY_KP_9               329
#define GLFW_KEY_KP_DECIMAL         330
#define GLFW_KEY_KP_DIVIDE          331
#define GLFW_KEY_KP_MULTIPLY        332
#define GLFW_KEY_KP_SUBTRACT        333
#define GLFW_KEY_KP_ADD             334
#define GLFW_KEY_KP_ENTER           335
#define GLFW_KEY_KP_EQUAL           336
#define GLFW_KEY_LEFT_SHIFT         340
#define GLFW_KEY_LEFT_CONTROL       341
#define GLFW_KEY_LEFT_ALT           342
#define GLFW_KEY_LEFT_SUPER         343
#define GLFW_KEY_RIGHT_SHIFT        344
#define GLFW_KEY_RIGHT_CONTROL      345
#define GLFW_KEY_RIGHT_ALT          346
#define GLFW_KEY_RIGHT_SUPER        347
#define GLFW_KEY_MENU               348

#define GLFW_KEY_LAST               GLFW_KEY_MENU

/*! @} */

/*! @defgroup mods Modifier key flags
 *  @brief Modifier key flags.
 *
 *  See [key input](@ref input_key) for how these are used.
 *
 *  @ingroup input
 *  @{ */

/*! @brief If this bit is set one or more Shift keys were held down.
 *
 *  If this bit is set one or more Shift keys were held down.
 */
#define GLFW_MOD_SHIFT           0x0001
/*! @brief If this bit is set one or more Control keys were held down.
 *
 *  If this bit is set one or more Control keys were held down.
 */
#define GLFW_MOD_CONTROL         0x0002
/*! @brief If this bit is set one or more Alt keys were held down.
 *
 *  If this bit is set one or more Alt keys were held down.
 */
#define GLFW_MOD_ALT             0x0004
/*! @brief If this bit is set one or more Super keys were held down.
 *
 *  If this bit is set one or more Super keys were held down.
 */
#define GLFW_MOD_SUPER           0x0008
/*! @brief If this bit is set the Caps Lock key is enabled.
 *
 *  If this bit is set the Caps Lock key is enabled and the @ref
 *  GLFW_LOCK_KEY_MODS input mode is set.
 */
#define GLFW_MOD_CAPS_LOCK       0x0010
/*! @brief If this bit is set the Num Lock key is enabled.
 *
 *  If this bit is set the Num Lock key is enabled and the @ref
 *  GLFW_LOCK_KEY_MODS input mode is set.
 */
#define GLFW_MOD_NUM_LOCK        0x0020

/*! @} */

/*! @defgroup buttons Mouse buttons
 *  @brief Mouse button IDs.
 *
 *  See [mouse button input](@ref input_mouse_button) for how these are used.
 *
 *  @ingroup input
 *  @{ */
#define GLFW_MOUSE_BUTTON_1         0
#define GLFW_MOUSE_BUTTON_2         1
#define GLFW_MOUSE_BUTTON_3         2
#define GLFW_MOUSE_BUTTON_4         3
#define GLFW_MOUSE_BUTTON_5         4
#define GLFW_MOUSE_BUTTON_6         5
#define GLFW_MOUSE_BUTTON_7         6
#define GLFW_MOUSE_BUTTON_8         7
#define GLFW_MOUSE_BUTTON_LAST      GLFW_MOUSE_BUTTON_8
#define GLFW_MOUSE_BUTTON_LEFT      GLFW_MOUSE_BUTTON_1
#define GLFW_MOUSE_BUTTON_RIGHT     GLFW_MOUSE_BUTTON_2
#define GLFW_MOUSE_BUTTON_MIDDLE    GLFW_MOUSE_BUTTON_3
/*! @} */

/*! @defgroup errors Error codes
 *  @brief Error codes.
 *
 *  See [error handling](@ref error_handling) for how these are used.
 *
 *  @ingroup init
 *  @{ */
/*! @brief No error has occurred.
 *
 *  No error has occurred.
 *
 *  @analysis Yay.
 */
#define GLFW_NO_ERROR               0
/*! @brief GLFW has not been initialized.
 *
 *  This occurs if a GLFW function was called that must not be called unless the
 *  library is [initialized](@ref intro_init).
 *
 *  @analysis Application programmer error.  Initialize GLFW before calling any
 *  function that requires initialization.
 */
#define GLFW_NOT_INITIALIZED        0x00010001
/*! @brief No context is current for this thread.
 *
 *  This occurs if a GLFW function was called that needs and operates on the
 *  current OpenGL or OpenGL ES context but no context is current on the calling
 *  thread.  One such function is @ref glfwSwapInterval.
 *
 *  @analysis Application programmer error.  Ensure a context is current before
 *  calling functions that require a current context.
 */
#define GLFW_NO_CURRENT_CONTEXT     0x00010002
/*! @brief One of the arguments to the function was an invalid enum value.
 *
 *  One of the arguments to the function was an invalid enum value, for example
 *  requesting @ref GLFW_RED_BITS with @ref glfwGetWindowAttrib.
 *
 *  @analysis Application programmer error.  Fix the offending call.
 */
#define GLFW_INVALID_ENUM           0x00010003
/*! @brief One of the arguments to the function was an invalid value.
 *
 *  One of the arguments to the function was an invalid value, for example
 *  requesting a non-existent OpenGL or OpenGL ES version like 2.7.
 *
 *  Requesting a valid but unavailable OpenGL or OpenGL ES version will instead
 *  result in a @ref GLFW_VERSION_UNAVAILABLE error.
 *
 *  @analysis Application programmer error.  Fix the offending call.
 */
#define GLFW_INVALID_VALUE          0x00010004
/*! @brief A memory allocation failed.
 *
 *  A memory allocation failed.
 *
 *  @analysis A bug in GLFW or the underlying operating system.  Report the bug
 *  to our [issue tracker](https://github.com/glfw/glfw/issues).
 */
#define GLFW_OUT_OF_MEMORY          0x00010005
/*! @brief GLFW could not find support for the requested API on the system.
 *
 *  GLFW could not find support for the requested API on the system.
 *
 *  @analysis The installed graphics driver does not support the requested
 *  API, or does not support it via the chosen context creation API.
 *  Below are a few examples.
 *
 *  @par
 *  Some pre-installed Windows graphics drivers do not support OpenGL.  AMD only
 *  supports OpenGL ES via EGL, while Nvidia and Intel only support it via
 *  a WGL or GLX extension.  macOS does not provide OpenGL ES at all.  The Mesa
 *  EGL, OpenGL and OpenGL ES libraries do not interface with the Nvidia binary
 *  driver.  Older graphics drivers do not support Vulkan.
 */
#define GLFW_API_UNAVAILABLE        0x00010006
/*! @brief The requested OpenGL or OpenGL ES version is not available.
 *
 *  The requested OpenGL or OpenGL ES version (including any requested context
 *  or framebuffer hints) is not available on this machine.
 *
 *  @analysis The machine does not support your requirements.  If your
 *  application is sufficiently flexible, downgrade your requirements and try
 *  again.  Otherwise, inform the user that their machine does not match your
 *  requirements.
 *
 *  @par
 *  Future invalid OpenGL and OpenGL ES versions, for example OpenGL 4.8 if 5.0
 *  comes out before the 4.x series gets that far, also fail with this error and
 *  not @ref GLFW_INVALID_VALUE, because GLFW cannot know what future versions
 *  will exist.
 */
#define GLFW_VERSION_UNAVAILABLE    0x00010007
/*! @brief A platform-specific error occurred that does not match any of the
 *  more specific categories.
 *
 *  A platform-specific error occurred that does not match any of the more
 *  specific categories.
 *
 *  @analysis A bug or configuration error in GLFW, the underlying operating
 *  system or its drivers, or a lack of required resources.  Report the issue to
 *  our [issue tracker](https://github.com/glfw/glfw/issues).
 */
#define GLFW_PLATFORM_ERROR         0x00010008
/*! @brief The requested format is not supported or available.
 *
 *  If emitted during window creation, the requested pixel format is not
 *  supported.
 *
 *  If emitted when querying the clipboard, the contents of the clipboard could
 *  not be converted to the requested format.
 *
 *  @analysis If emitted during window creation, one or more
 *  [hard constraints](@ref window_hints_hard) did not match any of the
 *  available pixel formats.  If your application is sufficiently flexible,
 *  downgrade your requirements and try again.  Otherwise, inform the user that
 *  their machine does not match your requirements.
 *
 *  @par
 *  If emitted when querying the clipboard, ignore the error or report it to
 *  the user, as appropriate.
 */
#define GLFW_FORMAT_UNAVAILABLE     0x00010009
/*! @brief The specified window does not have an OpenGL or OpenGL ES context.
 *
 *  A window that does not have an OpenGL or OpenGL ES context was passed to
 *  a function that requires it to have one.
 *
 *  @analysis Application programmer error.  Fix the offending call.
 */
#define GLFW_NO_WINDOW_CONTEXT      0x0001000A
/*! @brief The specified cursor shape is not available.
 *
 *  The specified standard cursor shape is not available, either because the
 *  current platform cursor theme does not provide it or because it is not
 *  available on the platform.
 *
 *  @analysis Platform or system settings limitation.  Pick another
 *  [standard cursor shape](@ref shapes) or create a
 *  [custom cursor](@ref cursor_custom).
 */
#define GLFW_CURSOR_UNAVAILABLE     0x0001000B
/*! @brief The requested feature is not provided by the platform.
 *
 *  The requested feature is not provided by the platform, so GLFW is unable to
 *  implement it.  The documentation for each function notes if it could emit
 *  this error.
 *
 *  @analysis Platform or platform version limitation.  The error can be ignored
 *  unless the feature is critical to the application.
 *
 *  @par
 *  A function call that emits this error has no effect other than the error and
 *  updating any existing out parameters.
 */
#define GLFW_FEATURE_UNAVAILABLE    0x0001000C
/*! @brief The requested feature is not implemented for the platform.
 *
 *  The requested feature has not yet been implemented in GLFW for this platform.
 *
 *  @analysis An incomplete implementation of GLFW for this platform, hopefully
 *  fixed in a future release.  The error can be ignored unless the feature is
 *  critical to the application.
 *
 *  @par
 *  A function call that emits this error has no effect other than the error and
 *  updating any existing out parameters.
 */
#define GLFW_FEATURE_UNIMPLEMENTED  0x0001000D
/*! @brief Platform unavailable or no matching platform was found.
 *
 *  If emitted during initialization, no matching platform was found.  If the @ref
 *  GLFW_PLATFORM init hint was set to `GLFW_ANY_PLATFORM`, GLFW could not detect any of
 *  the platforms supported by this library binary, except for the Null platform.  If the
 *  init hint was set to a specific platform, it is either not supported by this library
 *  binary or GLFW was not able to detect it.
 *
 *  If emitted by a native access function, GLFW was initialized for a different platform
 *  than the function is for.
 *
 *  @analysis Failure to detect any platform usually only happens on non-macOS Unix
 *  systems, either when no window system is running or the program was run from
 *  a terminal that does not have the necessary environment variables.  Fall back to
 *  a different platform if possible or notify the user that no usable platform was
 *  detected.
 *
 *  Failure to detect a specific platform may have the same cause as above or be because
 *  support for that platform was not compiled in.  Call @ref glfwPlatformSupported to
 *  check whether a specific platform is supported by a library binary.
 */
#define GLFW_PLATFORM_UNAVAILABLE   0x0001000E
/*! @} */

/*! @addtogroup window
 *  @{ */
/*! @brief Input focus window hint and attribute
 *
 *  Input focus [window hint](@ref GLFW_FOCUSED_hint) or
 *  [window attribute](@ref GLFW_FOCUSED_attrib).
 */
#define GLFW_FOCUSED                0x00020001
/*! @brief Window iconification window attribute
 *
 *  Window iconification [window attribute](@ref GLFW_ICONIFIED_attrib).
 */
#define GLFW_ICONIFIED              0x00020002
/*! @brief Window resize-ability window hint and attribute
 *
 *  Window resize-ability [window hint](@ref GLFW_RESIZABLE_hint) and
 *  [window attribute](@ref GLFW_RESIZABLE_attrib).
 */
#define GLFW_RESIZABLE              0x00020003
/*! @brief Window visibility window hint and attribute
 *
 *  Window visibility [window hint](@ref GLFW_VISIBLE_hint) and
 *  [window attribute](@ref GLFW_VISIBLE_attrib).
 */
#define GLFW_VISIBLE                0x00020004
/*! @brief Window decoration window hint and attribute
 *
 *  Window decoration [window hint](@ref GLFW_DECORATED_hint) and
 *  [window attribute](@ref GLFW_DECORATED_attrib).
 */
#define GLFW_DECORATED              0x00020005
/*! @brief Window auto-iconification window hint and attribute
 *
 *  Window auto-iconification [window hint](@ref GLFW_AUTO_ICONIFY_hint) and
 *  [window attribute](@ref GLFW_AUTO_ICONIFY_attrib).
 */
#define GLFW_AUTO_ICONIFY           0x00020006
/*! @brief Window decoration window hint and attribute
 *
 *  Window decoration [window hint](@ref GLFW_FLOATING_hint) and
 *  [window attribute](@ref GLFW_FLOATING_attrib).
 */
#define GLFW_FLOATING               0x00020007
/*! @brief Window maximization window hint and attribute
 *
 *  Window maximization [window hint](@ref GLFW_MAXIMIZED_hint) and
 *  [window attribute](@ref GLFW_MAXIMIZED_attrib).
 */
#define GLFW_MAXIMIZED              0x00020008
/*! @brief Cursor centering window hint
 *
 *  Cursor centering [window hint](@ref GLFW_CENTER_CURSOR_hint).
 */
#define GLFW_CENTER_CURSOR          0x00020009
/*! @brief Window framebuffer transparency hint and attribute
 *
 *  Window framebuffer transparency
 *  [window hint](@ref GLFW_TRANSPARENT_FRAMEBUFFER_hint) and
 *  [window attribute](@ref GLFW_TRANSPARENT_FRAMEBUFFER_attrib).
 */
#define GLFW_TRANSPARENT_FRAMEBUFFER 0x0002000A
/*! @brief Mouse cursor hover window attribute.
 *
 *  Mouse cursor hover [window attribute](@ref GLFW_HOVERED_attrib).
 */
#define GLFW_HOVERED                0x0002000B
/*! @brief Input focus on calling show window hint and attribute
 *
 *  Input focus [window hint](@ref GLFW_FOCUS_ON_SHOW_hint) or
 *  [window attribute](@ref GLFW_FOCUS_ON_SHOW_attrib).
 */
#define GLFW_FOCUS_ON_SHOW          0x0002000C

/*! @brief Mouse input transparency window hint and attribute
 *
 *  Mouse input transparency [window hint](@ref GLFW_MOUSE_PASSTHROUGH_hint) or
 *  [window attribute](@ref GLFW_MOUSE_PASSTHROUGH_attrib).
 */
#define GLFW_MOUSE_PASSTHROUGH      0x0002000D

/*! @brief Initial position x-coordinate window hint.
 *
 *  Initial position x-coordinate [window hint](@ref GLFW_POSITION_X).
 */
#define GLFW_POSITION_X             0x0002000E

/*! @brief Initial position y-coordinate window hint.
 *
 *  Initial position y-coordinate [window hint](@ref GLFW_POSITION_Y).
 */
#define GLFW_POSITION_Y             0x0002000F

/*! @brief Framebuffer bit depth hint.
 *
 *  Framebuffer bit depth [hint](@ref GLFW_RED_BITS).
 */
#define GLFW_RED_BITS               0x00021001
/*! @brief Framebuffer bit depth hint.
 *
 *  Framebuffer bit depth [hint](@ref GLFW_GREEN_BITS).
 */
#define GLFW_GREEN_BITS             0x00021002
/*! @brief Framebuffer bit depth hint.
 *
 *  Framebuffer bit depth [hint](@ref GLFW_BLUE_BITS).
 */
#define GLFW_BLUE_BITS              0x00021003
/*! @brief Framebuffer bit depth hint.
 *
 *  Framebuffer bit depth [hint](@ref GLFW_ALPHA_BITS).
 */
#define GLFW_ALPHA_BITS             0x00021004
/*! @brief Framebuffer bit depth hint.
 *
 *  Framebuffer bit depth [hint](@ref GLFW_DEPTH_BITS).
 */
#define GLFW_DEPTH_BITS             0x00021005
/*! @brief Framebuffer bit depth hint.
 *
 *  Framebuffer bit depth [hint](@ref GLFW_STENCIL_BITS).
 */
#define GLFW_STENCIL_BITS           0x00021006
/*! @brief Framebuffer bit depth hint.
 *
 *  Framebuffer bit depth [hint](@ref GLFW_ACCUM_RED_BITS).
 */
#define GLFW_ACCUM_RED_BITS         0x00021007
/*! @brief Framebuffer bit depth hint.
 *
 *  Framebuffer bit depth [hint](@ref GLFW_ACCUM_GREEN_BITS).
 */
#define GLFW_ACCUM_GREEN_BITS       0x00021008
/*! @brief Framebuffer bit depth hint.
 *
 *  Framebuffer bit depth [hint](@ref GLFW_ACCUM_BLUE_BITS).
 */
#define GLFW_ACCUM_BLUE_BITS        0x00021009
/*! @brief Framebuffer bit depth hint.
 *
 *  Framebuffer bit depth [hint](@ref GLFW_ACCUM_ALPHA_BITS).
 */
#define GLFW_ACCUM_ALPHA_BITS       0x0002100A
/*! @brief Framebuffer auxiliary buffer hint.
 *
 *  Framebuffer auxiliary buffer [hint](@ref GLFW_AUX_BUFFERS).
 */
#define GLFW_AUX_BUFFERS            0x0002100B
/*! @brief OpenGL stereoscopic rendering hint.
 *
 *  OpenGL stereoscopic rendering [hint](@ref GLFW_STEREO).
 */
#define GLFW_STEREO                 0x0002100C
/*! @brief Framebuffer MSAA samples hint.
 *
 *  Framebuffer MSAA samples [hint](@ref GLFW_SAMPLES).
 */
#define GLFW_SAMPLES                0x0002100D
/*! @brief Framebuffer sRGB hint.
 *
 *  Framebuffer sRGB [hint](@ref GLFW_SRGB_CAPABLE).
 */
#define GLFW_SRGB_CAPABLE           0x0002100E
/*! @brief Monitor refresh rate hint.
 *
 *  Monitor refresh rate [hint](@ref GLFW_REFRESH_RATE).
 */
#define GLFW_REFRESH_RATE           0x0002100F
/*! @brief Framebuffer double buffering hint and attribute.
 *
 *  Framebuffer double buffering [hint](@ref GLFW_DOUBLEBUFFER_hint) and
 *  [attribute](@ref GLFW_DOUBLEBUFFER_attrib).
 */
#define GLFW_DOUBLEBUFFER           0x00021010

/*! @brief Context client API hint and attribute.
 *
 *  Context client API [hint](@ref GLFW_CLIENT_API_hint) and
 *  [attribute](@ref GLFW_CLIENT_API_attrib).
 */
#define GLFW_CLIENT_API             0x00022001
/*! @brief Context client API major version hint and attribute.
 *
 *  Context client API major version [hint](@ref GLFW_CONTEXT_VERSION_MAJOR_hint)
 *  and [attribute](@ref GLFW_CONTEXT_VERSION_MAJOR_attrib).
 */
#define GLFW_CONTEXT_VERSION_MAJOR  0x00022002
/*! @brief Context client API minor version hint and attribute.
 *
 *  Context client API minor version [hint](@ref GLFW_CONTEXT_VERSION_MINOR_hint)
 *  and [attribute](@ref GLFW_CONTEXT_VERSION_MINOR_attrib).
 */
#define GLFW_CONTEXT_VERSION_MINOR  0x00022003
/*! @brief Context client API revision number attribute.
 *
 *  Context client API revision number
 *  [attribute](@ref GLFW_CONTEXT_REVISION_attrib).
 */
#define GLFW_CONTEXT_REVISION       0x00022004
/*! @brief Context robustness hint and attribute.
 *
 *  Context client API revision number [hint](@ref GLFW_CONTEXT_ROBUSTNESS_hint)
 *  and [attribute](@ref GLFW_CONTEXT_ROBUSTNESS_attrib).
 */
#define GLFW_CONTEXT_ROBUSTNESS     0x00022005
/*! @brief OpenGL forward-compatibility hint and attribute.
 *
 *  OpenGL forward-compatibility [hint](@ref GLFW_OPENGL_FORWARD_COMPAT_hint)
 *  and [attribute](@ref GLFW_OPENGL_FORWARD_COMPAT_attrib).
 */
#define GLFW_OPENGL_FORWARD_COMPAT  0x00022006
/*! @brief Debug mode context hint and attribute.
 *
 *  Debug mode context [hint](@ref GLFW_CONTEXT_DEBUG_hint) and
 *  [attribute](@ref GLFW_CONTEXT_DEBUG_attrib).
 */
#define GLFW_CONTEXT_DEBUG          0x00022007
/*! @brief Legacy name for compatibility.
 *
 *  This is an alias for compatibility with earlier versions.
 */
#define GLFW_OPENGL_DEBUG_CONTEXT   GLFW_CONTEXT_DEBUG
/*! @brief OpenGL profile hint and attribute.
 *
 *  OpenGL profile [hint](@ref GLFW_OPENGL_PROFILE_hint) and
 *  [attribute](@ref GLFW_OPENGL_PROFILE_attrib).
 */
#define GLFW_OPENGL_PROFILE         0x00022008
/*! @brief Context flush-on-release hint and attribute.
 *
 *  Context flush-on-release [hint](@ref GLFW_CONTEXT_RELEASE_BEHAVIOR_hint) and
 *  [attribute](@ref GLFW_CONTEXT_RELEASE_BEHAVIOR_attrib).
 */
#define GLFW_CONTEXT_RELEASE_BEHAVIOR 0x00022009
/*! @brief Context error suppression hint and attribute.
 *
 *  Context error suppression [hint](@ref GLFW_CONTEXT_NO_ERROR_hint) and
 *  [attribute](@ref GLFW_CONTEXT_NO_ERROR_attrib).
 */
#define GLFW_CONTEXT_NO_ERROR       0x0002200A
/*! @brief Context creation API hint and attribute.
 *
 *  Context creation API [hint](@ref GLFW_CONTEXT_CREATION_API_hint) and
 *  [attribute](@ref GLFW_CONTEXT_CREATION_API_attrib).
 */
#define GLFW_CONTEXT_CREATION_API   0x0002200B
/*! @brief Window content area scaling window
 *  [window hint](@ref GLFW_SCALE_TO_MONITOR).
 */
#define GLFW_SCALE_TO_MONITOR       0x0002200C
/*! @brief Window framebuffer scaling
 *  [window hint](@ref GLFW_SCALE_FRAMEBUFFER_hint).
 */
#define GLFW_SCALE_FRAMEBUFFER      0x0002200D
/*! @brief Legacy name for compatibility.
 *
 *  This is an alias for the
 *  [GLFW_SCALE_FRAMEBUFFER](@ref GLFW_SCALE_FRAMEBUFFER_hint) window hint for
 *  compatibility with earlier versions.
 */
#define GLFW_COCOA_RETINA_FRAMEBUFFER 0x00023001
/*! @brief macOS specific
 *  [window hint](@ref GLFW_COCOA_FRAME_NAME_hint).
 */
#define GLFW_COCOA_FRAME_NAME         0x00023002
/*! @brief macOS specific
 *  [window hint](@ref GLFW_COCOA_GRAPHICS_SWITCHING_hint).
 */
#define GLFW_COCOA_GRAPHICS_SWITCHING 0x00023003
/*! @brief X11 specific
 *  [window hint](@ref GLFW_X11_CLASS_NAME_hint).
 */
#define GLFW_X11_CLASS_NAME         0x00024001
/*! @brief X11 specific
 *  [window hint](@ref GLFW_X11_CLASS_NAME_hint).
 */
#define GLFW_X11_INSTANCE_NAME      0x00024002
#define GLFW_WIN32_KEYBOARD_MENU    0x00025001
/*! @brief Win32 specific [window hint](@ref GLFW_WIN32_SHOWDEFAULT_hint).
 */
#define GLFW_WIN32_SHOWDEFAULT      0x00025002
/*! @brief Wayland specific
 *  [window hint](@ref GLFW_WAYLAND_APP_ID_hint).
 *  
 *  Allows specification of the Wayland app_id.
 */
#define GLFW_WAYLAND_APP_ID         0x00026001
/*! @} */

#define GLFW_NO_API                          0
#define GLFW_OPENGL_API             0x00030001
#define GLFW_OPENGL_ES_API          0x00030002

#define GLFW_NO_ROBUSTNESS                   0
#define GLFW_NO_RESET_NOTIFICATION  0x00031001
#define GLFW_LOSE_CONTEXT_ON_RESET  0x00031002

#define GLFW_OPENGL_ANY_PROFILE              0
#define GLFW_OPENGL_CORE_PROFILE    0x00032001
#define GLFW_OPENGL_COMPAT_PROFILE  0x00032002

#define GLFW_CURSOR                 0x00033001
#define GLFW_STICKY_KEYS            0x00033002
#define GLFW_STICKY_MOUSE_BUTTONS   0x00033003
#define GLFW_LOCK_KEY_MODS          0x00033004
#define GLFW_RAW_MOUSE_MOTION       0x00033005

#define GLFW_CURSOR_NORMAL          0x00034001
#define GLFW_CURSOR_HIDDEN          0x00034002
#define GLFW_CURSOR_DISABLED        0x00034003
#define GLFW_CURSOR_CAPTURED        0x00034004

#define GLFW_ANY_RELEASE_BEHAVIOR            0
#define GLFW_RELEASE_BEHAVIOR_FLUSH 0x00035001
#define GLFW_RELEASE_BEHAVIOR_NONE  0x00035002

#define GLFW_NATIVE_CONTEXT_API     0x00036001
#define GLFW_EGL_CONTEXT_API        0x00036002
#define GLFW_OSMESA_CONTEXT_API     0x00036003

#define GLFW_ANGLE_PLATFORM_TYPE_NONE    0x00037001
#define GLFW_ANGLE_PLATFORM_TYPE_OPENGL  0x00037002
#define GLFW_ANGLE_PLATFORM_TYPE_OPENGLES 0x00037003
#define GLFW_ANGLE_PLATFORM_TYPE_D3D9    0x00037004
#define GLFW_ANGLE_PLATFORM_TYPE_D3D11   0x00037005
#define GLFW_ANGLE_PLATFORM_TYPE_VULKAN  0x00037007
#define GLFW_ANGLE_PLATFORM_TYPE_METAL   0x00037008

#define GLFW_WAYLAND_PREFER_LIBDECOR    0x00038001
#define GLFW_WAYLAND_DISABLE_LIBDECOR   0x00038002

#define GLFW_ANY_POSITION           0x80000000

/*! @defgroup shapes Standard cursor shapes
 *  @brief Standard system cursor shapes.
 *
 *  These are the [standard cursor shapes](@ref cursor_standard) that can be
 *  requested from the platform (window system).
 *
 *  @ingroup input
 *  @{ */

/*! @brief The regular arrow cursor shape.
 *
 *  The regular arrow cursor shape.
 */
#define GLFW_ARROW_CURSOR           0x00036001
/*! @brief The text input I-beam cursor shape.
 *
 *  The text input I-beam cursor shape.
 */
#define GLFW_IBEAM_CURSOR           0x00036002
/*! @brief The crosshair cursor shape.
 *
 *  The crosshair cursor shape.
 */
#define GLFW_CROSSHAIR_CURSOR       0x00036003
/*! @brief The pointing hand cursor shape.
 *
 *  The pointing hand cursor shape.
 */
#define GLFW_POINTING_HAND_CURSOR   0x00036004
/*! @brief The horizontal resize/move arrow shape.
 *
 *  The horizontal resize/move arrow shape.  This is usually a horizontal
 *  double-headed arrow.
 */
#define GLFW_RESIZE_EW_CURSOR       0x00036005
/*! @brief The vertical resize/move arrow shape.
 *
 *  The vertical resize/move shape.  This is usually a vertical double-headed
 *  arrow.
 */
#define GLFW_RESIZE_NS_CURSOR       0x00036006
/*! @brief The top-left to bottom-right diagonal resize/move arrow shape.
 *
 *  The top-left to bottom-right diagonal resize/move shape.  This is usually
 *  a diagonal double-headed arrow.
 *
 *  @note @macos This shape is provided by a private system API and may fail
 *  with @ref GLFW_CURSOR_UNAVAILABLE in the future.
 *
 *  @note @wayland This shape is provided by a newer standard not supported by
 *  all cursor themes.
 *
 *  @note @x11 This shape is provided by a newer standard not supported by all
 *  cursor themes.
 */
#define GLFW_RESIZE_NWSE_CURSOR     0x00036007
/*! @brief The top-right to bottom-left diagonal resize/move arrow shape.
 *
 *  The top-right to bottom-left diagonal resize/move shape.  This is usually
 *  a diagonal double-headed arrow.
 *
 *  @note @macos This shape is provided by a private system API and may fail
 *  with @ref GLFW_CURSOR_UNAVAILABLE in the future.
 *
 *  @note @wayland This shape is provided by a newer standard not supported by
 *  all cursor themes.
 *
 *  @note @x11 This shape is provided by a newer standard not supported by all
 *  cursor themes.
 */
#define GLFW_RESIZE_NESW_CURSOR     0x00036008
/*! @brief The omni-directional resize/move cursor shape.
 *
 *  The omni-directional resize cursor/move shape.  This is usually either
 *  a combined horizontal and vertical double-headed arrow or a grabbing hand.
 */
#define GLFW_RESIZE_ALL_CURSOR      0x00036009
/*! @brief The operation-not-allowed shape.
 *
 *  The operation-not-allowed shape.  This is usually a circle with a diagonal
 *  line through it.
 *
 *  @note @wayland This shape is provided by a newer standard not supported by
 *  all cursor themes.
 *
 *  @note @x11 This shape is provided by a newer standard not supported by all
 *  cursor themes.
 */
#define GLFW_NOT_ALLOWED_CURSOR     0x0003600A
/*! @brief Legacy name for compatibility.
 *
 *  This is an alias for compatibility with earlier versions.
 */
#define GLFW_HRESIZE_CURSOR         GLFW_RESIZE_EW_CURSOR
/*! @brief Legacy name for compatibility.
 *
 *  This is an alias for compatibility with earlier versions.
 */
#define GLFW_VRESIZE_CURSOR         GLFW_RESIZE_NS_CURSOR
/*! @brief Legacy name for compatibility.
 *
 *  This is an alias for compatibility with earlier versions.
 */
#define GLFW_HAND_CURSOR            GLFW_POINTING_HAND_CURSOR
/*! @} */

#define GLFW_CONNECTED              0x00040001
#define GLFW_DISCONNECTED           0x00040002

/*! @brief ANGLE rendering backend init hint.
 *
 *  ANGLE rendering backend [init hint](@ref GLFW_ANGLE_PLATFORM_TYPE_hint).
 */
#define GLFW_ANGLE_PLATFORM_TYPE    0x00050002
/*! @brief Platform selection init hint.
 *
 *  Platform selection [init hint](@ref GLFW_PLATFORM).
 */
#define GLFW_PLATFORM               0x00050003
/*! @brief macOS specific init hint.
 *
 *  macOS specific [init hint](@ref GLFW_COCOA_CHDIR_RESOURCES_hint).
 */
#define GLFW_COCOA_CHDIR_RESOURCES  0x00051001
/*! @brief macOS specific init hint.
 *
 *  macOS specific [init hint](@ref GLFW_COCOA_MENUBAR_hint).
 */
#define GLFW_COCOA_MENUBAR          0x00051002
/*! @brief X11 specific init hint.
 *
 *  X11 specific [init hint](@ref GLFW_X11_XCB_VULKAN_SURFACE_hint).
 */
#define GLFW_X11_XCB_VULKAN_SURFACE 0x00052001
/*! @brief Wayland specific init hint.
 *
 *  Wayland specific [init hint](@ref GLFW_WAYLAND_LIBDECOR_hint).
 */
#define GLFW_WAYLAND_LIBDECOR       0x00053001
/*! @} */

/*! @addtogroup init
 *  @{ */
/*! @brief Hint value that enables automatic platform selection.
 *
 *  Hint value for @ref GLFW_PLATFORM that enables automatic platform selection.
 */
#define GLFW_ANY_PLATFORM           0x00060000
#define GLFW_PLATFORM_WIN32         0x00060001
#define GLFW_PLATFORM_COCOA         0x00060002
#define GLFW_PLATFORM_WAYLAND       0x00060003
#define GLFW_PLATFORM_X11           0x00060004
#define GLFW_PLATFORM_NULL          0x00060005
/*! @} */

#define GLFW_DONT_CARE              -1


/*************************************************************************
 * GLFW API types
 *************************************************************************/

/*! @brief Client API function pointer type.
 *
 *  Generic function pointer used for returning client API function pointers
 *  without forcing a cast from a regular pointer.
 *
 *  @sa @ref context_glext
 *  @sa @ref glfwGetProcAddress
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup context
 */
typedef void (*GLFWglproc)(void);

/*! @brief Vulkan API function pointer type.
 *
 *  Generic function pointer used for returning Vulkan API function pointers
 *  without forcing a cast from a regular pointer.
 *
 *  @sa @ref vulkan_proc
 *  @sa @ref glfwGetInstanceProcAddress
 *
 *  @since Added in version 3.2.
 *
 *  @ingroup vulkan
 */
typedef void (*GLFWvkproc)(void);

/*! @brief Opaque monitor object.
 *
 *  Opaque monitor object.
 *
 *  @see @ref monitor_object
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup monitor
 */
typedef struct GLFWmonitor GLFWmonitor;

/*! @brief Opaque window object.
 *
 *  Opaque window object.
 *
 *  @see @ref window_object
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup window
 */
typedef struct GLFWwindow GLFWwindow;

/*! @brief Opaque cursor object.
 *
 *  Opaque cursor object.
 *
 *  @see @ref cursor_object
 *
 *  @since Added in version 3.1.
 *
 *  @ingroup input
 */
typedef struct GLFWcursor GLFWcursor;

/*! @brief The function pointer type for memory allocation callbacks.
 *
 *  This is the function pointer type for memory allocation callbacks.  A memory
 *  allocation callback function has the following signature:
 *  @code
 *  void* function_name(size_t size, void* user)
 *  @endcode
 *
 *  This function must return either a memory block at least `size` bytes long,
 *  or `NULL` if allocation failed.  Note that not all parts of GLFW handle allocation
 *  failures gracefully yet.
 *
 *  This function must support being called during @ref glfwInit but before the library is
 *  flagged as initialized, as well as during @ref glfwTerminate after the library is no
 *  longer flagged as initialized.
 *
 *  Any memory allocated via this function will be deallocated via the same allocator
 *  during library termination or earlier.
 *
 *  Any memory allocated via this function must be suitably aligned for any object type.
 *  If you are using C99 or earlier, this alignment is platform-dependent but will be the
 *  same as what `malloc` provides.  If you are using C11 or later, this is the value of
 *  `alignof(max_align_t)`.
 *
 *  The size will always be greater than zero.  Allocations of size zero are filtered out
 *  before reaching the custom allocator.
 *
 *  If this function returns `NULL`, GLFW will emit @ref GLFW_OUT_OF_MEMORY.
 *
 *  This function must not call any GLFW function.
 *
 *  @param[in] size The minimum size, in bytes, of the memory block.
 *  @param[in] user The user-defined pointer from the allocator.
 *  @return The address of the newly allocated memory block, or `NULL` if an
 *  error occurred.
 *
 *  @pointer_lifetime The returned memory block must be valid at least until it
 *  is deallocated.
 *
 *  @reentrancy This function should not call any GLFW function.
 *
 *  @thread_safety This function must support being called from any thread that calls GLFW
 *  functions.
 *
 *  @sa @ref init_allocator
 *  @sa @ref GLFWallocator
 *
 *  @since Added in version 3.4.
 *
 *  @ingroup init
 */
typedef void* (* GLFWallocatefun)(size_t size, void* user);

/*! @brief The function pointer type for memory reallocation callbacks.
 *
 *  This is the function pointer type for memory reallocation callbacks.
 *  A memory reallocation callback function has the following signature:
 *  @code
 *  void* function_name(void* block, size_t size, void* user)
 *  @endcode
 *
 *  This function must return a memory block at least `size` bytes long, or
 *  `NULL` if allocation failed.  Note that not all parts of GLFW handle allocation
 *  failures gracefully yet.
 *
 *  This function must support being called during @ref glfwInit but before the library is
 *  flagged as initialized, as well as during @ref glfwTerminate after the library is no
 *  longer flagged as initialized.
 *
 *  Any memory allocated via this function will be deallocated via the same allocator
 *  during library termination or earlier.
 *
 *  Any memory allocated via this function must be suitably aligned for any object type.
 *  If you are using C99 or earlier, this alignment is platform-dependent but will be the
 *  same as what `realloc` provides.  If you are using C11 or later, this is the value of
 *  `alignof(max_align_t)`.
 *
 *  The block address will never be `NULL` and the size will always be greater than zero.
 *  Reallocations of a block to size zero are converted into deallocations before reaching
 *  the custom allocator.  Reallocations of `NULL` to a non-zero size are converted into
 *  regular allocations before reaching the custom allocator.
 *
 *  If this function returns `NULL`, GLFW will emit @ref GLFW_OUT_OF_MEMORY.
 *
 *  This function must not call any GLFW function.
 *
 *  @param[in] block The address of the memory block to reallocate.
 *  @param[in] size The new minimum size, in bytes, of the memory block.
 *  @param[in] user The user-defined pointer from the allocator.
 *  @return The address of the newly allocated or resized memory block, or
 *  `NULL` if an error occurred.
 *
 *  @pointer_lifetime The returned memory block must be valid at least until it
 *  is deallocated.
 *
 *  @reentrancy This function should not call any GLFW function.
 *
 *  @thread_safety This function must support being called from any thread that calls GLFW
 *  functions.
 *
 *  @sa @ref init_allocator
 *  @sa @ref GLFWallocator
 *
 *  @since Added in version 3.4.
 *
 *  @ingroup init
 */
typedef void* (* GLFWreallocatefun)(void* block, size_t size, void* user);

/*! @brief The function pointer type for memory deallocation callbacks.
 *
 *  This is the function pointer type for memory deallocation callbacks.
 *  A memory deallocation callback function has the following signature:
 *  @code
 *  void function_name(void* block, void* user)
 *  @endcode
 *
 *  This function may deallocate the specified memory block.  This memory block
 *  will have been allocated with the same allocator.
 *
 *  This function must support being called during @ref glfwInit but before the library is
 *  flagged as initialized, as well as during @ref glfwTerminate after the library is no
 *  longer flagged as initialized.
 *
 *  The block address will never be `NULL`.  Deallocations of `NULL` are filtered out
 *  before reaching the custom allocator.
 *
 *  If this function returns `NULL`, GLFW will emit @ref GLFW_OUT_OF_MEMORY.
 *
 *  This function must not call any GLFW function.
 *
 *  @param[in] block The address of the memory block to deallocate.
 *  @param[in] user The user-defined pointer from the allocator.
 *
 *  @pointer_lifetime The specified memory block will not be accessed by GLFW
 *  after this function is called.
 *
 *  @reentrancy This function should not call any GLFW function.
 *
 *  @thread_safety This function must support being called from any thread that calls GLFW
 *  functions.
 *
 *  @sa @ref init_allocator
 *  @sa @ref GLFWallocator
 *
 *  @since Added in version 3.4.
 *
 *  @ingroup init
 */
typedef void (* GLFWdeallocatefun)(void* block, void* user);

/*! @brief The function pointer type for error callbacks.
 *
 *  This is the function pointer type for error callbacks.  An error callback
 *  function has the following signature:
 *  @code
 *  void callback_name(int error_code, const char* description)
 *  @endcode
 *
 *  @param[in] error_code An [error code](@ref errors).  Future releases may add
 *  more error codes.
 *  @param[in] description A UTF-8 encoded string describing the error.
 *
 *  @pointer_lifetime The error description string is valid until the callback
 *  function returns.
 *
 *  @sa @ref error_handling
 *  @sa @ref glfwSetErrorCallback
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup init
 */
typedef void (* GLFWerrorfun)(int error_code, const char* description);

/*! @brief The function pointer type for window position callbacks.
 *
 *  This is the function pointer type for window position callbacks.  A window
 *  position callback function has the following signature:
 *  @code
 *  void callback_name(GLFWwindow* window, int xpos, int ypos)
 *  @endcode
 *
 *  @param[in] window The window that was moved.
 *  @param[in] xpos The new x-coordinate, in screen coordinates, of the
 *  upper-left corner of the content area of the window.
 *  @param[in] ypos The new y-coordinate, in screen coordinates, of the
 *  upper-left corner of the content area of the window.
 *
 *  @sa @ref window_pos
 *  @sa @ref glfwSetWindowPosCallback
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup window
 */
typedef void (* GLFWwindowposfun)(GLFWwindow* window, int xpos, int ypos);

/*! @brief The function pointer type for window size callbacks.
 *
 *  This is the function pointer type for window size callbacks.  A window size
 *  callback function has the following signature:
 *  @code
 *  void callback_name(GLFWwindow* window, int width, int height)
 *  @endcode
 *
 *  @param[in] window The window that was resized.
 *  @param[in] width The new width, in screen coordinates, of the window.
 *  @param[in] height The new height, in screen coordinates, of the window.
 *
 *  @sa @ref window_size
 *  @sa @ref glfwSetWindowSizeCallback
 *
 *  @since Added in version 1.0.
 *  @glfw3 Added window handle parameter.
 *
 *  @ingroup window
 */
typedef void (* GLFWwindowsizefun)(GLFWwindow* window, int width, int height);

/*! @brief The function pointer type for window close callbacks.
 *
 *  This is the function pointer type for window close callbacks.  A window
 *  close callback function has the following signature:
 *  @code
 *  void function_name(GLFWwindow* window)
 *  @endcode
 *
 *  @param[in] window The window that the user attempted to close.
 *
 *  @sa @ref window_close
 *  @sa @ref glfwSetWindowCloseCallback
 *
 *  @since Added in version 2.5.
 *  @glfw3 Added window handle parameter.
 *
 *  @ingroup window
 */
typedef void (* GLFWwindowclosefun)(GLFWwindow* window);

/*! @brief The function pointer type for window content refresh callbacks.
 *
 *  This is the function pointer type for window content refresh callbacks.
 *  A window content refresh callback function has the following signature:
 *  @code
 *  void function_name(GLFWwindow* window);
 *  @endcode
 *
 *  @param[in] window The window whose content needs to be refreshed.
 *
 *  @sa @ref window_refresh
 *  @sa @ref glfwSetWindowRefreshCallback
 *
 *  @since Added in version 2.5.
 *  @glfw3 Added window handle parameter.
 *
 *  @ingroup window
 */
typedef void (* GLFWwindowrefreshfun)(GLFWwindow* window);

/*! @brief The function pointer type for window focus callbacks.
 *
 *  This is the function pointer type for window focus callbacks.  A window
 *  focus callback function has the following signature:
 *  @code
 *  void function_name(GLFWwindow* window, int focused)
 *  @endcode
 *
 *  @param[in] window The window that gained or lost input focus.
 *  @param[in] focused `GLFW_TRUE` if the window was given input focus, or
 *  `GLFW_FALSE` if it lost it.
 *
 *  @sa @ref window_focus
 *  @sa @ref glfwSetWindowFocusCallback
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup window
 */
typedef void (* GLFWwindowfocusfun)(GLFWwindow* window, int focused);

/*! @brief The function pointer type for window iconify callbacks.
 *
 *  This is the function pointer type for window iconify callbacks.  A window
 *  iconify callback function has the following signature:
 *  @code
 *  void function_name(GLFWwindow* window, int iconified)
 *  @endcode
 *
 *  @param[in] window The window that was iconified or restored.
 *  @param[in] iconified `GLFW_TRUE` if the window was iconified, or
 *  `GLFW_FALSE` if it was restored.
 *
 *  @sa @ref window_iconify
 *  @sa @ref glfwSetWindowIconifyCallback
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup window
 */
typedef void (* GLFWwindowiconifyfun)(GLFWwindow* window, int iconified);

/*! @brief The function pointer type for window maximize callbacks.
 *
 *  This is the function pointer type for window maximize callbacks.  A window
 *  maximize callback function has the following signature:
 *  @code
 *  void function_name(GLFWwindow* window, int maximized)
 *  @endcode
 *
 *  @param[in] window The window that was maximized or restored.
 *  @param[in] maximized `GLFW_TRUE` if the window was maximized, or
 *  `GLFW_FALSE` if it was restored.
 *
 *  @sa @ref window_maximize
 *  @sa glfwSetWindowMaximizeCallback
 *
 *  @since Added in version 3.3.
 *
 *  @ingroup window
 */
typedef void (* GLFWwindowmaximizefun)(GLFWwindow* window, int maximized);

/*! @brief The function pointer type for framebuffer size callbacks.
 *
 *  This is the function pointer type for framebuffer size callbacks.
 *  A framebuffer size callback function has the following signature:
 *  @code
 *  void function_name(GLFWwindow* window, int width, int height)
 *  @endcode
 *
 *  @param[in] window The window whose framebuffer was resized.
 *  @param[in] width The new width, in pixels, of the framebuffer.
 *  @param[in] height The new height, in pixels, of the framebuffer.
 *
 *  @sa @ref window_fbsize
 *  @sa @ref glfwSetFramebufferSizeCallback
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup window
 */
typedef void (* GLFWframebuffersizefun)(GLFWwindow* window, int width, int height);

/*! @brief The function pointer type for window content scale callbacks.
 *
 *  This is the function pointer type for window content scale callbacks.
 *  A window content scale callback function has the following signature:
 *  @code
 *  void function_name(GLFWwindow* window, float xscale, float yscale)
 *  @endcode
 *
 *  @param[in] window The window whose content scale changed.
 *  @param[in] xscale The new x-axis content scale of the window.
 *  @param[in] yscale The new y-axis content scale of the window.
 *
 *  @sa @ref window_scale
 *  @sa @ref glfwSetWindowContentScaleCallback
 *
 *  @since Added in version 3.3.
 *
 *  @ingroup window
 */
typedef void (* GLFWwindowcontentscalefun)(GLFWwindow* window, float xscale, float yscale);

/*! @brief The function pointer type for mouse button callbacks.
 *
 *  This is the function pointer type for mouse button callback functions.
 *  A mouse button callback function has the following signature:
 *  @code
 *  void function_name(GLFWwindow* window, int button, int action, int mods)
 *  @endcode
 *
 *  @param[in] window The window that received the event.
 *  @param[in] button The [mouse button](@ref buttons) that was pressed or
 *  released.
 *  @param[in] action One of `GLFW_PRESS` or `GLFW_RELEASE`.  Future releases
 *  may add more actions.
 *  @param[in] mods Bit field describing which [modifier keys](@ref mods) were
 *  held down.
 *
 *  @sa @ref input_mouse_button
 *  @sa @ref glfwSetMouseButtonCallback
 *
 *  @since Added in version 1.0.
 *  @glfw3 Added window handle and modifier mask parameters.
 *
 *  @ingroup input
 */
typedef void (* GLFWmousebuttonfun)(GLFWwindow* window, int button, int action, int mods);

/*! @brief The function pointer type for cursor position callbacks.
 *
 *  This is the function pointer type for cursor position callbacks.  A cursor
 *  position callback function has the following signature:
 *  @code
 *  void function_name(GLFWwindow* window, double xpos, double ypos);
 *  @endcode
 *
 *  @param[in] window The window that received the event.
 *  @param[in] xpos The new cursor x-coordinate, relative to the left edge of
 *  the content area.
 *  @param[in] ypos The new cursor y-coordinate, relative to the top edge of the
 *  content area.
 *
 *  @sa @ref cursor_pos
 *  @sa @ref glfwSetCursorPosCallback
 *
 *  @since Added in version 3.0.  Replaces `GLFWmouseposfun`.
 *
 *  @ingroup input
 */
typedef void (* GLFWcursorposfun)(GLFWwindow* window, double xpos, double ypos);

/*! @brief The function pointer type for cursor enter/leave callbacks.
 *
 *  This is the function pointer type for cursor enter/leave callbacks.
 *  A cursor enter/leave callback function has the following signature:
 *  @code
 *  void function_name(GLFWwindow* window, int entered)
 *  @endcode
 *
 *  @param[in] window The window that received the event.
 *  @param[in] entered `GLFW_TRUE` if the cursor entered the window's content
 *  area, or `GLFW_FALSE` if it left it.
 *
 *  @sa @ref cursor_enter
 *  @sa @ref glfwSetCursorEnterCallback
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup input
 */
typedef void (* GLFWcursorenterfun)(GLFWwindow* window, int entered);

/*! @brief The function pointer type for scroll callbacks.
 *
 *  This is the function pointer type for scroll callbacks.  A scroll callback
 *  function has the following signature:
 *  @code
 *  void function_name(GLFWwindow* window, double xoffset, double yoffset)
 *  @endcode
 *
 *  @param[in] window The window that received the event.
 *  @param[in] xoffset The scroll offset along the x-axis.
 *  @param[in] yoffset The scroll offset along the y-axis.
 *
 *  @sa @ref scrolling
 *  @sa @ref glfwSetScrollCallback
 *
 *  @since Added in version 3.0.  Replaces `GLFWmousewheelfun`.
 *
 *  @ingroup input
 */
typedef void (* GLFWscrollfun)(GLFWwindow* window, double xoffset, double yoffset);

/*! @brief The function pointer type for keyboard key callbacks.
 *
 *  This is the function pointer type for keyboard key callbacks.  A keyboard
 *  key callback function has the following signature:
 *  @code
 *  void function_name(GLFWwindow* window, int key, int scancode, int action, int mods)
 *  @endcode
 *
 *  @param[in] window The window that received the event.
 *  @param[in] key The [keyboard key](@ref keys) that was pressed or released.
 *  @param[in] scancode The platform-specific scancode of the key.
 *  @param[in] action `GLFW_PRESS`, `GLFW_RELEASE` or `GLFW_REPEAT`.  Future
 *  releases may add more actions.
 *  @param[in] mods Bit field describing which [modifier keys](@ref mods) were
 *  held down.
 *
 *  @sa @ref input_key
 *  @sa @ref glfwSetKeyCallback
 *
 *  @since Added in version 1.0.
 *  @glfw3 Added window handle, scancode and modifier mask parameters.
 *
 *  @ingroup input
 */
typedef void (* GLFWkeyfun)(GLFWwindow* window, int key, int scancode, int action, int mods);

/*! @brief The function pointer type for Unicode character callbacks.
 *
 *  This is the function pointer type for Unicode character callbacks.
 *  A Unicode character callback function has the following signature:
 *  @code
 *  void function_name(GLFWwindow* window, unsigned int codepoint)
 *  @endcode
 *
 *  @param[in] window The window that received the event.
 *  @param[in] codepoint The Unicode code point of the character.
 *
 *  @sa @ref input_char
 *  @sa @ref glfwSetCharCallback
 *
 *  @since Added in version 2.4.
 *  @glfw3 Added window handle parameter.
 *
 *  @ingroup input
 */
typedef void (* GLFWcharfun)(GLFWwindow* window, unsigned int codepoint);

/*! @brief The function pointer type for Unicode character with modifiers
 *  callbacks.
 *
 *  This is the function pointer type for Unicode character with modifiers
 *  callbacks.  It is called for each input character, regardless of what
 *  modifier keys are held down.  A Unicode character with modifiers callback
 *  function has the following signature:
 *  @code
 *  void function_name(GLFWwindow* window, unsigned int codepoint, int mods)
 *  @endcode
 *
 *  @param[in] window The window that received the event.
 *  @param[in] codepoint The Unicode code point of the character.
 *  @param[in] mods Bit field describing which [modifier keys](@ref mods) were
 *  held down.
 *
 *  @sa @ref input_char
 *  @sa @ref glfwSetCharModsCallback
 *
 *  @deprecated Scheduled for removal in version 4.0.
 *
 *  @since Added in version 3.1.
 *
 *  @ingroup input
 */
typedef void (* GLFWcharmodsfun)(GLFWwindow* window, unsigned int codepoint, int mods);

/*! @brief The function pointer type for path drop callbacks.
 *
 *  This is the function pointer type for path drop callbacks.  A path drop
 *  callback function has the following signature:
 *  @code
 *  void function_name(GLFWwindow* window, int path_count, const char* paths[])
 *  @endcode
 *
 *  @param[in] window The window that received the event.
 *  @param[in] path_count The number of dropped paths.
 *  @param[in] paths The UTF-8 encoded file and/or directory path names.
 *
 *  @pointer_lifetime The path array and its strings are valid until the
 *  callback function returns.
 *
 *  @sa @ref path_drop
 *  @sa @ref glfwSetDropCallback
 *
 *  @since Added in version 3.1.
 *
 *  @ingroup input
 */
typedef void (* GLFWdropfun)(GLFWwindow* window, int path_count, const char* paths[]);

/*! @brief The function pointer type for monitor configuration callbacks.
 *
 *  This is the function pointer type for monitor configuration callbacks.
 *  A monitor callback function has the following signature:
 *  @code
 *  void function_name(GLFWmonitor* monitor, int event)
 *  @endcode
 *
 *  @param[in] monitor The monitor that was connected or disconnected.
 *  @param[in] event One of `GLFW_CONNECTED` or `GLFW_DISCONNECTED`.  Future
 *  releases may add more events.
 *
 *  @sa @ref monitor_event
 *  @sa @ref glfwSetMonitorCallback
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup monitor
 */
typedef void (* GLFWmonitorfun)(GLFWmonitor* monitor, int event);

/*! @brief Video mode type.
 *
 *  This describes a single video mode.
 *
 *  @sa @ref monitor_modes
 *  @sa @ref glfwGetVideoMode
 *  @sa @ref glfwGetVideoModes
 *
 *  @since Added in version 1.0.
 *  @glfw3 Added refresh rate member.
 *
 *  @ingroup monitor
 */
typedef struct GLFWvidmode
{
    /*! The width, in screen coordinates, of the video mode.
     */
    int width;
    /*! The height, in screen coordinates, of the video mode.
     */
    int height;
    /*! The bit depth of the red channel of the video mode.
     */
    int redBits;
    /*! The bit depth of the green channel of the video mode.
     */
    int greenBits;
    /*! The bit depth of the blue channel of the video mode.
     */
    int blueBits;
    /*! The refresh rate, in Hz, of the video mode.
     */
    int refreshRate;
} GLFWvidmode;

/*! @brief Gamma ramp.
 *
 *  This describes the gamma ramp for a monitor.
 *
 *  @sa @ref monitor_gamma
 *  @sa @ref glfwGetGammaRamp
 *  @sa @ref glfwSetGammaRamp
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup monitor
 */
typedef struct GLFWgammaramp
{
    /*! An array of value describing the response of the red channel.
     */
    unsigned short* red;
    /*! An array of value describing the response of the green channel.
     */
    unsigned short* green;
    /*! An array of value describing the response of the blue channel.
     */
    unsigned short* blue;
    /*! The number of elements in each array.
     */
    unsigned int size;
} GLFWgammaramp;

/*! @brief Image data.
 *
 *  This describes a single 2D image.  See the documentation for each related
 *  function what the expected pixel format is.
 *
 *  @sa @ref cursor_custom
 *  @sa @ref window_icon
 *
 *  @since Added in version 2.1.
 *  @glfw3 Removed format and bytes-per-pixel members.
 *
 *  @ingroup window
 */
typedef struct GLFWimage
{
    /*! The width, in pixels, of this image.
     */
    int width;
    /*! The height, in pixels, of this image.
     */
    int height;
    /*! The pixel data of this image, arranged left-to-right, top-to-bottom.
     */
    unsigned char* pixels;
} GLFWimage;

/*! @brief Custom heap memory allocator.
 *
 *  This describes a custom heap memory allocator for GLFW.  To set an allocator, pass it
 *  to @ref glfwInitAllocator before initializing the library.
 *
 *  @sa @ref init_allocator
 *  @sa @ref glfwInitAllocator
 *
 *  @since Added in version 3.4.
 *
 *  @ingroup init
 */
typedef struct GLFWallocator
{
    /*! The memory allocation function.  See @ref GLFWallocatefun for details about
     *  allocation function.
     */
    GLFWallocatefun allocate;
    /*! The memory reallocation function.  See @ref GLFWreallocatefun for details about
     *  reallocation function.
     */
    GLFWreallocatefun reallocate;
    /*! The memory deallocation function.  See @ref GLFWdeallocatefun for details about
     *  deallocation function.
     */
    GLFWdeallocatefun deallocate;
    /*! The user pointer for this custom allocator.  This value will be passed to the
     *  allocator functions.
     */
    void* user;
} GLFWallocator;


/*************************************************************************
 * GLFW API functions
 *************************************************************************/

/*! @brief Initializes the GLFW library.
 *
 *  This function initializes the GLFW library.  Before most GLFW functions can
 *  be used, GLFW must be initialized, and before an application terminates GLFW
 *  should be terminated in order to free any resources allocated during or
 *  after initialization.
 *
 *  If this function fails, it calls @ref glfwTerminate before returning.  If it
 *  succeeds, you should call @ref glfwTerminate before the application exits.
 *
 *  Additional calls to this function after successful initialization but before
 *  termination will return `GLFW_TRUE` immediately.
 *
 *  The @ref GLFW_PLATFORM init hint controls which platforms are considered during
 *  initialization.  This also depends on which platforms the library was compiled to
 *  support.
 *
 *  @return `GLFW_TRUE` if successful, or `GLFW_FALSE` if an
 *  [error](@ref error_handling) occurred.
 *
 *  @errors Possible errors include @ref GLFW_PLATFORM_UNAVAILABLE and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @remark @macos This function will change the current directory of the
 *  application to the `Contents/Resources` subdirectory of the application's
 *  bundle, if present.  This can be disabled with the @ref
 *  GLFW_COCOA_CHDIR_RESOURCES init hint.
 *
 *  @remark @macos This function will create the main menu and dock icon for the
 *  application.  If GLFW finds a `MainMenu.nib` it is loaded and assumed to
 *  contain a menu bar.  Otherwise a minimal menu bar is created manually with
 *  common commands like Hide, Quit and About.  The About entry opens a minimal
 *  about dialog with information from the application's bundle.  The menu bar
 *  and dock icon can be disabled entirely with the @ref GLFW_COCOA_MENUBAR init
 *  hint.
 *
 *  @remark __Wayland, X11:__ If the library was compiled with support for both
 *  Wayland and X11, and the @ref GLFW_PLATFORM init hint is set to
 *  `GLFW_ANY_PLATFORM`, the `XDG_SESSION_TYPE` environment variable affects
 *  which platform is picked.  If the environment variable is not set, or is set
 *  to something other than `wayland` or `x11`, the regular detection mechanism
 *  will be used instead.
 *
 *  @remark @x11 This function will set the `LC_CTYPE` category of the
 *  application locale according to the current environment if that category is
 *  still "C".  This is because the "C" locale breaks Unicode text input.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref intro_init
 *  @sa @ref glfwInitHint
 *  @sa @ref glfwInitAllocator
 *  @sa @ref glfwTerminate
 *
 *  @since Added in version 1.0.
 *
 *  @ingroup init
 */
GLFWAPI int glfwInit(void);

/*! @brief Terminates the GLFW library.
 *
 *  This function destroys all remaining windows and cursors, restores any
 *  modified gamma ramps and frees any other allocated resources.  Once this
 *  function is called, you must again call @ref glfwInit successfully before
 *  you will be able to use most GLFW functions.
 *
 *  If GLFW has been successfully initialized, this function should be called
 *  before the application exits.  If initialization fails, there is no need to
 *  call this function, as it is called by @ref glfwInit before it returns
 *  failure.
 *
 *  This function has no effect if GLFW is not initialized.
 *
 *  @errors Possible errors include @ref GLFW_PLATFORM_ERROR.
 *
 *  @remark This function may be called before @ref glfwInit.
 *
 *  @warning The contexts of any remaining windows must not be current on any
 *  other thread when this function is called.
 *
 *  @reentrancy This function must not be called from a callback.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref intro_init
 *  @sa @ref glfwInit
 *
 *  @since Added in version 1.0.
 *
 *  @ingroup init
 */
GLFWAPI void glfwTerminate(void);

/*! @brief Sets the specified init hint to the desired value.
 *
 *  This function sets hints for the next initialization of GLFW.
 *
 *  The values you set hints to are never reset by GLFW, but they only take
 *  effect during initialization.  Once GLFW has been initialized, any values
 *  you set will be ignored until the library is terminated and initialized
 *  again.
 *
 *  Some hints are platform specific.  These may be set on any platform but they
 *  will only affect their specific platform.  Other platforms will ignore them.
 *  Setting these hints requires no platform specific headers or functions.
 *
 *  @param[in] hint The [init hint](@ref init_hints) to set.
 *  @param[in] value The new value of the init hint.
 *
 *  @errors Possible errors include @ref GLFW_INVALID_ENUM and @ref
 *  GLFW_INVALID_VALUE.
 *
 *  @remarks This function may be called before @ref glfwInit.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa init_hints
 *  @sa glfwInit
 *
 *  @since Added in version 3.3.
 *
 *  @ingroup init
 */
GLFWAPI void glfwInitHint(int hint, int value);

/*! @brief Sets the init allocator to the desired value.
 *
 *  To use the default allocator, call this function with a `NULL` argument.
 *
 *  If you specify an allocator struct, every member must be a valid function
 *  pointer.  If any member is `NULL`, this function will emit @ref
 *  GLFW_INVALID_VALUE and the init allocator will be unchanged.
 *
 *  The functions in the allocator must fulfil a number of requirements.  See the
 *  documentation for @ref GLFWallocatefun, @ref GLFWreallocatefun and @ref
 *  GLFWdeallocatefun for details.
 *
 *  @param[in] allocator The allocator to use at the next initialization, or
 *  `NULL` to use the default one.
 *
 *  @errors Possible errors include @ref GLFW_INVALID_VALUE.
 *
 *  @pointer_lifetime The specified allocator is copied before this function
 *  returns.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref init_allocator
 *  @sa @ref glfwInit
 *
 *  @since Added in version 3.4.
 *
 *  @ingroup init
 */
GLFWAPI void glfwInitAllocator(const GLFWallocator* allocator);

#if defined(VK_VERSION_1_0)

/*! @brief Sets the desired Vulkan `vkGetInstanceProcAddr` function.
 *
 *  This function sets the `vkGetInstanceProcAddr` function that GLFW will use for all
 *  Vulkan related entry point queries.
 *
 *  This feature is mostly useful on macOS, if your copy of the Vulkan loader is in
 *  a location where GLFW cannot find it through dynamic loading, or if you are still
 *  using the static library version of the loader.
 *
 *  If set to `NULL`, GLFW will try to load the Vulkan loader dynamically by its standard
 *  name and get this function from there.  This is the default behavior.
 *
 *  The standard name of the loader is `vulkan-1.dll` on Windows, `libvulkan.so.1` on
 *  Linux and other Unix-like systems and `libvulkan.1.dylib` on macOS.  If your code is
 *  also loading it via these names then you probably don't need to use this function.
 *
 *  The function address you set is never reset by GLFW, but it only takes effect during
 *  initialization.  Once GLFW has been initialized, any updates will be ignored until the
 *  library is terminated and initialized again.
 *
 *  @param[in] loader The address of the function to use, or `NULL`.
 *
 *  @par Loader function signature
 *  @code
 *  PFN_vkVoidFunction vkGetInstanceProcAddr(VkInstance instance, const char* name)
 *  @endcode
 *  For more information about this function, see the
 *  [Vulkan Registry](https://www.khronos.org/registry/vulkan/).
 *
 *  @errors None.
 *
 *  @remark This function may be called before @ref glfwInit.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref vulkan_loader
 *  @sa @ref glfwInit
 *
 *  @since Added in version 3.4.
 *
 *  @ingroup init
 */
GLFWAPI void glfwInitVulkanLoader(PFN_vkGetInstanceProcAddr loader);

#endif /*VK_VERSION_1_0*/

/*! @brief Retrieves the version of the GLFW library.
 *
 *  This function retrieves the major, minor and revision numbers of the GLFW
 *  library.  It is intended for when you are using GLFW as a shared library and
 *  want to ensure that you are using the minimum required version.
 *
 *  Any or all of the version arguments may be `NULL`.
 *
 *  @param[out] major Where to store the major version number, or `NULL`.
 *  @param[out] minor Where to store the minor version number, or `NULL`.
 *  @param[out] rev Where to store the revision number, or `NULL`.
 *
 *  @errors None.
 *
 *  @remark This function may be called before @ref glfwInit.
 *
 *  @thread_safety This function may be called from any thread.
 *
 *  @sa @ref intro_version
 *  @sa @ref glfwGetVersionString
 *
 *  @since Added in version 1.0.
 *
 *  @ingroup init
 */
GLFWAPI void glfwGetVersion(int* major, int* minor, int* rev);

/*! @brief Returns a string describing the compile-time configuration.
 *
 *  This function returns the compile-time generated
 *  [version string](@ref intro_version_string) of the GLFW library binary.  It describes
 *  the version, platforms, compiler and any platform or operating system specific
 *  compile-time options.  It should not be confused with the OpenGL or OpenGL ES version
 *  string, queried with `glGetString`.
 *
 *  __Do not use the version string__ to parse the GLFW library version.  The
 *  @ref glfwGetVersion function provides the version of the running library
 *  binary in numerical format.
 *
 *  __Do not use the version string__ to parse what platforms are supported.  The @ref
 *  glfwPlatformSupported function lets you query platform support.
 *
 *  @return The ASCII encoded GLFW version string.
 *
 *  @errors None.
 *
 *  @remark This function may be called before @ref glfwInit.
 *
 *  @pointer_lifetime The returned string is static and compile-time generated.
 *
 *  @thread_safety This function may be called from any thread.
 *
 *  @sa @ref intro_version
 *  @sa @ref glfwGetVersion
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup init
 */
GLFWAPI const char* glfwGetVersionString(void);

/*! @brief Returns and clears the last error for the calling thread.
 *
 *  This function returns and clears the [error code](@ref errors) of the last
 *  error that occurred on the calling thread, and optionally a UTF-8 encoded
 *  human-readable description of it.  If no error has occurred since the last
 *  call, it returns @ref GLFW_NO_ERROR (zero) and the description pointer is
 *  set to `NULL`.
 *
 *  @param[in] description Where to store the error description pointer, or `NULL`.
 *  @return The last error code for the calling thread, or @ref GLFW_NO_ERROR
 *  (zero).
 *
 *  @errors None.
 *
 *  @pointer_lifetime The returned string is allocated and freed by GLFW.  You
 *  should not free it yourself.  It is guaranteed to be valid only until the
 *  next error occurs or the library is terminated.
 *
 *  @remark This function may be called before @ref glfwInit.
 *
 *  @thread_safety This function may be called from any thread.
 *
 *  @sa @ref error_handling
 *  @sa @ref glfwSetErrorCallback
 *
 *  @since Added in version 3.3.
 *
 *  @ingroup init
 */
GLFWAPI int glfwGetError(const char** description);

/*! @brief Sets the error callback.
 *
 *  This function sets the error callback, which is called with an error code
 *  and a human-readable description each time a GLFW error occurs.
 *
 *  The error code is set before the callback is called.  Calling @ref
 *  glfwGetError from the error callback will return the same value as the error
 *  code argument.
 *
 *  The error callback is called on the thread where the error occurred.  If you
 *  are using GLFW from multiple threads, your error callback needs to be
 *  written accordingly.
 *
 *  Because the description string may have been generated specifically for that
 *  error, it is not guaranteed to be valid after the callback has returned.  If
 *  you wish to use it after the callback returns, you need to make a copy.
 *
 *  Once set, the error callback remains set even after the library has been
 *  terminated.
 *
 *  @param[in] callback The new callback, or `NULL` to remove the currently set
 *  callback.
 *  @return The previously set callback, or `NULL` if no callback was set.
 *
 *  @callback_signature
 *  @code
 *  void callback_name(int error_code, const char* description)
 *  @endcode
 *  For more information about the callback parameters, see the
 *  [callback pointer type](@ref GLFWerrorfun).
 *
 *  @errors None.
 *
 *  @remark This function may be called before @ref glfwInit.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref error_handling
 *  @sa @ref glfwGetError
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup init
 */
GLFWAPI GLFWerrorfun glfwSetErrorCallback(GLFWerrorfun callback);

/*! @brief Returns the currently selected platform.
 *
 *  This function returns the platform that was selected during initialization.  The
 *  returned value will be one of `GLFW_PLATFORM_WIN32`, `GLFW_PLATFORM_COCOA`,
 *  `GLFW_PLATFORM_WAYLAND`, `GLFW_PLATFORM_X11` or `GLFW_PLATFORM_NULL`.
 *
 *  @return The currently selected platform, or zero if an error occurred.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function may be called from any thread.
 *
 *  @sa @ref platform
 *  @sa @ref glfwPlatformSupported
 *
 *  @since Added in version 3.4.
 *
 *  @ingroup init
 */
GLFWAPI int glfwGetPlatform(void);

/*! @brief Returns whether the library includes support for the specified platform.
 *
 *  This function returns whether the library was compiled with support for the specified
 *  platform.  The platform must be one of `GLFW_PLATFORM_WIN32`, `GLFW_PLATFORM_COCOA`,
 *  `GLFW_PLATFORM_WAYLAND`, `GLFW_PLATFORM_X11` or `GLFW_PLATFORM_NULL`.
 *
 *  @param[in] platform The platform to query.
 *  @return `GLFW_TRUE` if the platform is supported, or `GLFW_FALSE` otherwise.
 *
 *  @errors Possible errors include @ref GLFW_INVALID_ENUM.
 *
 *  @remark This function may be called before @ref glfwInit.
 *
 *  @thread_safety This function may be called from any thread.
 *
 *  @sa @ref platform
 *  @sa @ref glfwGetPlatform
 *
 *  @since Added in version 3.4.
 *
 *  @ingroup init
 */
GLFWAPI int glfwPlatformSupported(int platform);

/*! @brief Returns the currently connected monitors.
 *
 *  This function returns an array of handles for all currently connected
 *  monitors.  The primary monitor is always first in the returned array.  If no
 *  monitors were found, this function returns `NULL`.
 *
 *  @param[out] count Where to store the number of monitors in the returned
 *  array.  This is set to zero if an error occurred.
 *  @return An array of monitor handles, or `NULL` if no monitors were found or
 *  if an [error](@ref error_handling) occurred.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @pointer_lifetime The returned array is allocated and freed by GLFW.  You
 *  should not free it yourself.  It is guaranteed to be valid only until the
 *  monitor configuration changes or the library is terminated.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref monitor_monitors
 *  @sa @ref monitor_event
 *  @sa @ref glfwGetPrimaryMonitor
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup monitor
 */
GLFWAPI GLFWmonitor** glfwGetMonitors(int* count);

/*! @brief Returns the primary monitor.
 *
 *  This function returns the primary monitor.  This is usually the monitor
 *  where elements like the task bar or global menu bar are located.
 *
 *  @return The primary monitor, or `NULL` if no monitors were found or if an
 *  [error](@ref error_handling) occurred.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @remark The primary monitor is always first in the array returned by @ref
 *  glfwGetMonitors.
 *
 *  @sa @ref monitor_monitors
 *  @sa @ref glfwGetMonitors
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup monitor
 */
GLFWAPI GLFWmonitor* glfwGetPrimaryMonitor(void);

/*! @brief Returns the position of the monitor's viewport on the virtual screen.
 *
 *  This function returns the position, in screen coordinates, of the upper-left
 *  corner of the specified monitor.
 *
 *  Any or all of the position arguments may be `NULL`.  If an error occurs, all
 *  non-`NULL` position arguments will be set to zero.
 *
 *  @param[in] monitor The monitor to query.
 *  @param[out] xpos Where to store the monitor x-coordinate, or `NULL`.
 *  @param[out] ypos Where to store the monitor y-coordinate, or `NULL`.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref monitor_properties
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup monitor
 */
GLFWAPI void glfwGetMonitorPos(GLFWmonitor* monitor, int* xpos, int* ypos);

/*! @brief Retrieves the work area of the monitor.
 *
 *  This function returns the position, in screen coordinates, of the upper-left
 *  corner of the work area of the specified monitor along with the work area
 *  size in screen coordinates. The work area is defined as the area of the
 *  monitor not occluded by the window system task bar where present. If no
 *  task bar exists then the work area is the monitor resolution in screen
 *  coordinates.
 *
 *  Any or all of the position and size arguments may be `NULL`.  If an error
 *  occurs, all non-`NULL` position and size arguments will be set to zero.
 *
 *  @param[in] monitor The monitor to query.
 *  @param[out] xpos Where to store the monitor x-coordinate, or `NULL`.
 *  @param[out] ypos Where to store the monitor y-coordinate, or `NULL`.
 *  @param[out] width Where to store the monitor width, or `NULL`.
 *  @param[out] height Where to store the monitor height, or `NULL`.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref monitor_workarea
 *
 *  @since Added in version 3.3.
 *
 *  @ingroup monitor
 */
GLFWAPI void glfwGetMonitorWorkarea(GLFWmonitor* monitor, int* xpos, int* ypos, int* width, int* height);

/*! @brief Returns the physical size of the monitor.
 *
 *  This function returns the size, in millimetres, of the display area of the
 *  specified monitor.
 *
 *  Some platforms do not provide accurate monitor size information, either
 *  because the monitor [EDID][] data is incorrect or because the driver does
 *  not report it accurately.
 *
 *  [EDID]: https://en.wikipedia.org/wiki/Extended_display_identification_data
 *
 *  Any or all of the size arguments may be `NULL`.  If an error occurs, all
 *  non-`NULL` size arguments will be set to zero.
 *
 *  @param[in] monitor The monitor to query.
 *  @param[out] widthMM Where to store the width, in millimetres, of the
 *  monitor's display area, or `NULL`.
 *  @param[out] heightMM Where to store the height, in millimetres, of the
 *  monitor's display area, or `NULL`.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @remark @win32 On Windows 8 and earlier the physical size is calculated from
 *  the current resolution and system DPI instead of querying the monitor EDID data.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref monitor_properties
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup monitor
 */
GLFWAPI void glfwGetMonitorPhysicalSize(GLFWmonitor* monitor, int* widthMM, int* heightMM);

/*! @brief Retrieves the content scale for the specified monitor.
 *
 *  This function retrieves the content scale for the specified monitor.  The
 *  content scale is the ratio between the current DPI and the platform's
 *  default DPI.  This is especially important for text and any UI elements.  If
 *  the pixel dimensions of your UI scaled by this look appropriate on your
 *  machine then it should appear at a reasonable size on other machines
 *  regardless of their DPI and scaling settings.  This relies on the system DPI
 *  and scaling settings being somewhat correct.
 *
 *  The content scale may depend on both the monitor resolution and pixel
 *  density and on user settings.  It may be very different from the raw DPI
 *  calculated from the physical size and current resolution.
 *
 *  @param[in] monitor The monitor to query.
 *  @param[out] xscale Where to store the x-axis content scale, or `NULL`.
 *  @param[out] yscale Where to store the y-axis content scale, or `NULL`.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @remark @wayland Fractional scaling information is not yet available for
 *  monitors, so this function only returns integer content scales.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref monitor_scale
 *  @sa @ref glfwGetWindowContentScale
 *
 *  @since Added in version 3.3.
 *
 *  @ingroup monitor
 */
GLFWAPI void glfwGetMonitorContentScale(GLFWmonitor* monitor, float* xscale, float* yscale);

/*! @brief Returns the name of the specified monitor.
 *
 *  This function returns a human-readable name, encoded as UTF-8, of the
 *  specified monitor.  The name typically reflects the make and model of the
 *  monitor and is not guaranteed to be unique among the connected monitors.
 *
 *  @param[in] monitor The monitor to query.
 *  @return The UTF-8 encoded name of the monitor, or `NULL` if an
 *  [error](@ref error_handling) occurred.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @pointer_lifetime The returned string is allocated and freed by GLFW.  You
 *  should not free it yourself.  It is valid until the specified monitor is
 *  disconnected or the library is terminated.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref monitor_properties
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup monitor
 */
GLFWAPI const char* glfwGetMonitorName(GLFWmonitor* monitor);

/*! @brief Sets the user pointer of the specified monitor.
 *
 *  This function sets the user-defined pointer of the specified monitor.  The
 *  current value is retained until the monitor is disconnected.  The initial
 *  value is `NULL`.
 *
 *  This function may be called from the monitor callback, even for a monitor
 *  that is being disconnected.
 *
 *  @param[in] monitor The monitor whose pointer to set.
 *  @param[in] pointer The new value.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function may be called from any thread.  Access is not
 *  synchronized.
 *
 *  @sa @ref monitor_userptr
 *  @sa @ref glfwGetMonitorUserPointer
 *
 *  @since Added in version 3.3.
 *
 *  @ingroup monitor
 */
GLFWAPI void glfwSetMonitorUserPointer(GLFWmonitor* monitor, void* pointer);

/*! @brief Returns the user pointer of the specified monitor.
 *
 *  This function returns the current value of the user-defined pointer of the
 *  specified monitor.  The initial value is `NULL`.
 *
 *  This function may be called from the monitor callback, even for a monitor
 *  that is being disconnected.
 *
 *  @param[in] monitor The monitor whose pointer to return.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function may be called from any thread.  Access is not
 *  synchronized.
 *
 *  @sa @ref monitor_userptr
 *  @sa @ref glfwSetMonitorUserPointer
 *
 *  @since Added in version 3.3.
 *
 *  @ingroup monitor
 */
GLFWAPI void* glfwGetMonitorUserPointer(GLFWmonitor* monitor);

/*! @brief Sets the monitor configuration callback.
 *
 *  This function sets the monitor configuration callback, or removes the
 *  currently set callback.  This is called when a monitor is connected to or
 *  disconnected from the system.
 *
 *  @param[in] callback The new callback, or `NULL` to remove the currently set
 *  callback.
 *  @return The previously set callback, or `NULL` if no callback was set or the
 *  library had not been [initialized](@ref intro_init).
 *
 *  @callback_signature
 *  @code
 *  void function_name(GLFWmonitor* monitor, int event)
 *  @endcode
 *  For more information about the callback parameters, see the
 *  [function pointer type](@ref GLFWmonitorfun).
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref monitor_event
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup monitor
 */
GLFWAPI GLFWmonitorfun glfwSetMonitorCallback(GLFWmonitorfun callback);

/*! @brief Returns the available video modes for the specified monitor.
 *
 *  This function returns an array of all video modes supported by the specified
 *  monitor.  The returned array is sorted in ascending order, first by color
 *  bit depth (the sum of all channel depths), then by resolution area (the
 *  product of width and height), then resolution width and finally by refresh
 *  rate.
 *
 *  @param[in] monitor The monitor to query.
 *  @param[out] count Where to store the number of video modes in the returned
 *  array.  This is set to zero if an error occurred.
 *  @return An array of video modes, or `NULL` if an
 *  [error](@ref error_handling) occurred.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @pointer_lifetime The returned array is allocated and freed by GLFW.  You
 *  should not free it yourself.  It is valid until the specified monitor is
 *  disconnected, this function is called again for that monitor or the library
 *  is terminated.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref monitor_modes
 *  @sa @ref glfwGetVideoMode
 *
 *  @since Added in version 1.0.
 *  @glfw3 Changed to return an array of modes for a specific monitor.
 *
 *  @ingroup monitor
 */
GLFWAPI const GLFWvidmode* glfwGetVideoModes(GLFWmonitor* monitor, int* count);

/*! @brief Returns the current mode of the specified monitor.
 *
 *  This function returns the current video mode of the specified monitor.  If
 *  you have created a full screen window for that monitor, the return value
 *  will depend on whether that window is iconified.
 *
 *  @param[in] monitor The monitor to query.
 *  @return The current mode of the monitor, or `NULL` if an
 *  [error](@ref error_handling) occurred.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @pointer_lifetime The returned array is allocated and freed by GLFW.  You
 *  should not free it yourself.  It is valid until the specified monitor is
 *  disconnected or the library is terminated.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref monitor_modes
 *  @sa @ref glfwGetVideoModes
 *
 *  @since Added in version 3.0.  Replaces `glfwGetDesktopMode`.
 *
 *  @ingroup monitor
 */
GLFWAPI const GLFWvidmode* glfwGetVideoMode(GLFWmonitor* monitor);

/*! @brief Generates a gamma ramp and sets it for the specified monitor.
 *
 *  This function generates an appropriately sized gamma ramp from the specified
 *  exponent and then calls @ref glfwSetGammaRamp with it.  The value must be
 *  a finite number greater than zero.
 *
 *  The software controlled gamma ramp is applied _in addition_ to the hardware
 *  gamma correction, which today is usually an approximation of sRGB gamma.
 *  This means that setting a perfectly linear ramp, or gamma 1.0, will produce
 *  the default (usually sRGB-like) behavior.
 *
 *  For gamma correct rendering with OpenGL or OpenGL ES, see the @ref
 *  GLFW_SRGB_CAPABLE hint.
 *
 *  @param[in] monitor The monitor whose gamma ramp to set.
 *  @param[in] gamma The desired exponent.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref GLFW_INVALID_VALUE,
 *  @ref GLFW_PLATFORM_ERROR and @ref GLFW_FEATURE_UNAVAILABLE (see remarks).
 *
 *  @remark @wayland Gamma handling is a privileged protocol, this function
 *  will thus never be implemented and emits @ref GLFW_FEATURE_UNAVAILABLE.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref monitor_gamma
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup monitor
 */
GLFWAPI void glfwSetGamma(GLFWmonitor* monitor, float gamma);

/*! @brief Returns the current gamma ramp for the specified monitor.
 *
 *  This function returns the current gamma ramp of the specified monitor.
 *
 *  @param[in] monitor The monitor to query.
 *  @return The current gamma ramp, or `NULL` if an
 *  [error](@ref error_handling) occurred.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref GLFW_PLATFORM_ERROR
 *  and @ref GLFW_FEATURE_UNAVAILABLE (see remarks).
 *
 *  @remark @wayland Gamma handling is a privileged protocol, this function
 *  will thus never be implemented and emits @ref GLFW_FEATURE_UNAVAILABLE while
 *  returning `NULL`.
 *
 *  @pointer_lifetime The returned structure and its arrays are allocated and
 *  freed by GLFW.  You should not free them yourself.  They are valid until the
 *  specified monitor is disconnected, this function is called again for that
 *  monitor or the library is terminated.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref monitor_gamma
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup monitor
 */
GLFWAPI const GLFWgammaramp* glfwGetGammaRamp(GLFWmonitor* monitor);

/*! @brief Sets the current gamma ramp for the specified monitor.
 *
 *  This function sets the current gamma ramp for the specified monitor.  The
 *  original gamma ramp for that monitor is saved by GLFW the first time this
 *  function is called and is restored by @ref glfwTerminate.
 *
 *  The software controlled gamma ramp is applied _in addition_ to the hardware
 *  gamma correction, which today is usually an approximation of sRGB gamma.
 *  This means that setting a perfectly linear ramp, or gamma 1.0, will produce
 *  the default (usually sRGB-like) behavior.
 *
 *  For gamma correct rendering with OpenGL or OpenGL ES, see the @ref
 *  GLFW_SRGB_CAPABLE hint.
 *
 *  @param[in] monitor The monitor whose gamma ramp to set.
 *  @param[in] ramp The gamma ramp to use.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref GLFW_PLATFORM_ERROR
 *  and @ref GLFW_FEATURE_UNAVAILABLE (see remarks).
 *
 *  @remark The size of the specified gamma ramp should match the size of the
 *  current ramp for that monitor.
 *
 *  @remark @win32 The gamma ramp size must be 256.
 *
 *  @remark @wayland Gamma handling is a privileged protocol, this function
 *  will thus never be implemented and emits @ref GLFW_FEATURE_UNAVAILABLE.
 *
 *  @pointer_lifetime The specified gamma ramp is copied before this function
 *  returns.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref monitor_gamma
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup monitor
 */
GLFWAPI void glfwSetGammaRamp(GLFWmonitor* monitor, const GLFWgammaramp* ramp);

/*! @brief Resets all window hints to their default values.
 *
 *  This function resets all window hints to their
 *  [default values](@ref window_hints_values).
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_hints
 *  @sa @ref glfwWindowHint
 *  @sa @ref glfwWindowHintString
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup window
 */
GLFWAPI void glfwDefaultWindowHints(void);

/*! @brief Sets the specified window hint to the desired value.
 *
 *  This function sets hints for the next call to @ref glfwCreateWindow.  The
 *  hints, once set, retain their values until changed by a call to this
 *  function or @ref glfwDefaultWindowHints, or until the library is terminated.
 *
 *  Only integer value hints can be set with this function.  String value hints
 *  are set with @ref glfwWindowHintString.
 *
 *  This function does not check whether the specified hint values are valid.
 *  If you set hints to invalid values this will instead be reported by the next
 *  call to @ref glfwCreateWindow.
 *
 *  Some hints are platform specific.  These may be set on any platform but they
 *  will only affect their specific platform.  Other platforms will ignore them.
 *  Setting these hints requires no platform specific headers or functions.
 *
 *  @param[in] hint The [window hint](@ref window_hints) to set.
 *  @param[in] value The new value of the window hint.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_INVALID_ENUM.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_hints
 *  @sa @ref glfwWindowHintString
 *  @sa @ref glfwDefaultWindowHints
 *
 *  @since Added in version 3.0.  Replaces `glfwOpenWindowHint`.
 *
 *  @ingroup window
 */
GLFWAPI void glfwWindowHint(int hint, int value);

/*! @brief Sets the specified window hint to the desired value.
 *
 *  This function sets hints for the next call to @ref glfwCreateWindow.  The
 *  hints, once set, retain their values until changed by a call to this
 *  function or @ref glfwDefaultWindowHints, or until the library is terminated.
 *
 *  Only string type hints can be set with this function.  Integer value hints
 *  are set with @ref glfwWindowHint.
 *
 *  This function does not check whether the specified hint values are valid.
 *  If you set hints to invalid values this will instead be reported by the next
 *  call to @ref glfwCreateWindow.
 *
 *  Some hints are platform specific.  These may be set on any platform but they
 *  will only affect their specific platform.  Other platforms will ignore them.
 *  Setting these hints requires no platform specific headers or functions.
 *
 *  @param[in] hint The [window hint](@ref window_hints) to set.
 *  @param[in] value The new value of the window hint.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_INVALID_ENUM.
 *
 *  @pointer_lifetime The specified string is copied before this function
 *  returns.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_hints
 *  @sa @ref glfwWindowHint
 *  @sa @ref glfwDefaultWindowHints
 *
 *  @since Added in version 3.3.
 *
 *  @ingroup window
 */
GLFWAPI void glfwWindowHintString(int hint, const char* value);

/*! @brief Creates a window and its associated context.
 *
 *  This function creates a window and its associated OpenGL or OpenGL ES
 *  context.  Most of the options controlling how the window and its context
 *  should be created are specified with [window hints](@ref window_hints).
 *
 *  Successful creation does not change which context is current.  Before you
 *  can use the newly created context, you need to
 *  [make it current](@ref context_current).  For information about the `share`
 *  parameter, see @ref context_sharing.
 *
 *  The created window, framebuffer and context may differ from what you
 *  requested, as not all parameters and hints are
 *  [hard constraints](@ref window_hints_hard).  This includes the size of the
 *  window, especially for full screen windows.  To query the actual attributes
 *  of the created window, framebuffer and context, see @ref
 *  glfwGetWindowAttrib, @ref glfwGetWindowSize and @ref glfwGetFramebufferSize.
 *
 *  To create a full screen window, you need to specify the monitor the window
 *  will cover.  If no monitor is specified, the window will be windowed mode.
 *  Unless you have a way for the user to choose a specific monitor, it is
 *  recommended that you pick the primary monitor.  For more information on how
 *  to query connected monitors, see @ref monitor_monitors.
 *
 *  For full screen windows, the specified size becomes the resolution of the
 *  window's _desired video mode_.  As long as a full screen window is not
 *  iconified, the supported video mode most closely matching the desired video
 *  mode is set for the specified monitor.  For more information about full
 *  screen windows, including the creation of so called _windowed full screen_
 *  or _borderless full screen_ windows, see @ref window_windowed_full_screen.
 *
 *  Once you have created the window, you can switch it between windowed and
 *  full screen mode with @ref glfwSetWindowMonitor.  This will not affect its
 *  OpenGL or OpenGL ES context.
 *
 *  By default, newly created windows use the placement recommended by the
 *  window system.  To create the window at a specific position, set the @ref
 *  GLFW_POSITION_X and @ref GLFW_POSITION_Y window hints before creation.  To
 *  restore the default behavior, set either or both hints back to
 *  `GLFW_ANY_POSITION`.
 *
 *  As long as at least one full screen window is not iconified, the screensaver
 *  is prohibited from starting.
 *
 *  Window systems put limits on window sizes.  Very large or very small window
 *  dimensions may be overridden by the window system on creation.  Check the
 *  actual [size](@ref window_size) after creation.
 *
 *  The [swap interval](@ref buffer_swap) is not set during window creation and
 *  the initial value may vary depending on driver settings and defaults.
 *
 *  @param[in] width The desired width, in screen coordinates, of the window.
 *  This must be greater than zero.
 *  @param[in] height The desired height, in screen coordinates, of the window.
 *  This must be greater than zero.
 *  @param[in] title The initial, UTF-8 encoded window title.
 *  @param[in] monitor The monitor to use for full screen mode, or `NULL` for
 *  windowed mode.
 *  @param[in] share The window whose context to share resources with, or `NULL`
 *  to not share resources.
 *  @return The handle of the created window, or `NULL` if an
 *  [error](@ref error_handling) occurred.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref
 *  GLFW_INVALID_ENUM, @ref GLFW_INVALID_VALUE, @ref GLFW_API_UNAVAILABLE, @ref
 *  GLFW_VERSION_UNAVAILABLE, @ref GLFW_FORMAT_UNAVAILABLE, @ref
 *  GLFW_NO_WINDOW_CONTEXT and @ref GLFW_PLATFORM_ERROR.
 *
 *  @remark @win32 Window creation will fail if the Microsoft GDI software
 *  OpenGL implementation is the only one available.
 *
 *  @remark @win32 If the executable has an icon resource named `GLFW_ICON,` it
 *  will be set as the initial icon for the window.  If no such icon is present,
 *  the `IDI_APPLICATION` icon will be used instead.  To set a different icon,
 *  see @ref glfwSetWindowIcon.
 *
 *  @remark @win32 The context to share resources with must not be current on
 *  any other thread.
 *
 *  @remark @macos The OS only supports core profile contexts for OpenGL
 *  versions 3.2 and later.  Before creating an OpenGL context of version 3.2 or
 *  later you must set the [GLFW_OPENGL_PROFILE](@ref GLFW_OPENGL_PROFILE_hint)
 *  hint accordingly.  OpenGL 3.0 and 3.1 contexts are not supported at all
 *  on macOS.
 *
 *  @remark @macos The GLFW window has no icon, as it is not a document
 *  window, but the dock icon will be the same as the application bundle's icon.
 *  For more information on bundles, see the
 *  [Bundle Programming Guide][bundle-guide] in the Mac Developer Library.
 *
 *  [bundle-guide]: https://developer.apple.com/library/mac/documentation/CoreFoundation/Conceptual/CFBundles/
 *
 *  @remark @macos On OS X 10.10 and later the window frame will not be rendered
 *  at full resolution on Retina displays unless the
 *  [GLFW_SCALE_FRAMEBUFFER](@ref GLFW_SCALE_FRAMEBUFFER_hint)
 *  hint is `GLFW_TRUE` and the `NSHighResolutionCapable` key is enabled in the
 *  application bundle's `Info.plist`.  For more information, see
 *  [High Resolution Guidelines for OS X][hidpi-guide] in the Mac Developer
 *  Library.  The GLFW test and example programs use a custom `Info.plist`
 *  template for this, which can be found as `CMake/Info.plist.in` in the source
 *  tree.
 *
 *  [hidpi-guide]: https://developer.apple.com/library/mac/documentation/GraphicsAnimation/Conceptual/HighResolutionOSX/Explained/Explained.html
 *
 *  @remark @macos When activating frame autosaving with
 *  [GLFW_COCOA_FRAME_NAME](@ref GLFW_COCOA_FRAME_NAME_hint), the specified
 *  window size and position may be overridden by previously saved values.
 *
 *  @remark @wayland GLFW uses [libdecor][] where available to create its window
 *  decorations.  This in turn uses server-side XDG decorations where available
 *  and provides high quality client-side decorations on compositors like GNOME.
 *  If both XDG decorations and libdecor are unavailable, GLFW falls back to
 *  a very simple set of window decorations that only support moving, resizing
 *  and the window manager's right-click menu.
 *
 *  [libdecor]: https://gitlab.freedesktop.org/libdecor/libdecor
 *
 *  @remark @x11 Some window managers will not respect the placement of
 *  initially hidden windows.
 *
 *  @remark @x11 Due to the asynchronous nature of X11, it may take a moment for
 *  a window to reach its requested state.  This means you may not be able to
 *  query the final size, position or other attributes directly after window
 *  creation.
 *
 *  @remark @x11 The class part of the `WM_CLASS` window property will by
 *  default be set to the window title passed to this function.  The instance
 *  part will use the contents of the `RESOURCE_NAME` environment variable, if
 *  present and not empty, or fall back to the window title.  Set the
 *  [GLFW_X11_CLASS_NAME](@ref GLFW_X11_CLASS_NAME_hint) and
 *  [GLFW_X11_INSTANCE_NAME](@ref GLFW_X11_INSTANCE_NAME_hint) window hints to
 *  override this.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_creation
 *  @sa @ref glfwDestroyWindow
 *
 *  @since Added in version 3.0.  Replaces `glfwOpenWindow`.
 *
 *  @ingroup window
 */
GLFWAPI GLFWwindow* glfwCreateWindow(int width, int height, const char* title, GLFWmonitor* monitor, GLFWwindow* share);

/*! @brief Destroys the specified window and its context.
 *
 *  This function destroys the specified window and its context.  On calling
 *  this function, no further callbacks will be called for that window.
 *
 *  If the context of the specified window is current on the main thread, it is
 *  detached before being destroyed.
 *
 *  @param[in] window The window to destroy.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @note The context of the specified window must not be current on any other
 *  thread when this function is called.
 *
 *  @reentrancy This function must not be called from a callback.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_creation
 *  @sa @ref glfwCreateWindow
 *
 *  @since Added in version 3.0.  Replaces `glfwCloseWindow`.
 *
 *  @ingroup window
 */
GLFWAPI void glfwDestroyWindow(GLFWwindow* window);

/*! @brief Checks the close flag of the specified window.
 *
 *  This function returns the value of the close flag of the specified window.
 *
 *  @param[in] window The window to query.
 *  @return The value of the close flag.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function may be called from any thread.  Access is not
 *  synchronized.
 *
 *  @sa @ref window_close
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup window
 */
GLFWAPI int glfwWindowShouldClose(GLFWwindow* window);

/*! @brief Sets the close flag of the specified window.
 *
 *  This function sets the value of the close flag of the specified window.
 *  This can be used to override the user's attempt to close the window, or
 *  to signal that it should be closed.
 *
 *  @param[in] window The window whose flag to change.
 *  @param[in] value The new value.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function may be called from any thread.  Access is not
 *  synchronized.
 *
 *  @sa @ref window_close
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup window
 */
GLFWAPI void glfwSetWindowShouldClose(GLFWwindow* window, int value);

/*! @brief Returns the title of the specified window.
 *
 *  This function returns the window title, encoded as UTF-8, of the specified
 *  window.  This is the title set previously by @ref glfwCreateWindow
 *  or @ref glfwSetWindowTitle.
 *
 *  @param[in] window The window to query.
 *  @return The UTF-8 encoded window title, or `NULL` if an
 *  [error](@ref error_handling) occurred.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @remark The returned title is currently a copy of the title last set by @ref
 *  glfwCreateWindow or @ref glfwSetWindowTitle.  It does not include any
 *  additional text which may be appended by the platform or another program.
 *
 *  @pointer_lifetime The returned string is allocated and freed by GLFW.  You
 *  should not free it yourself.  It is valid until the next call to @ref
 *  glfwGetWindowTitle or @ref glfwSetWindowTitle, or until the library is
 *  terminated.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_title
 *  @sa @ref glfwSetWindowTitle
 *
 *  @since Added in version 3.4.
 *
 *  @ingroup window
 */
GLFWAPI const char* glfwGetWindowTitle(GLFWwindow* window);

/*! @brief Sets the title of the specified window.
 *
 *  This function sets the window title, encoded as UTF-8, of the specified
 *  window.
 *
 *  @param[in] window The window whose title to change.
 *  @param[in] title The UTF-8 encoded window title.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @remark @macos The window title will not be updated until the next time you
 *  process events.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_title
 *  @sa @ref glfwGetWindowTitle
 *
 *  @since Added in version 1.0.
 *  @glfw3 Added window handle parameter.
 *
 *  @ingroup window
 */
GLFWAPI void glfwSetWindowTitle(GLFWwindow* window, const char* title);

/*! @brief Sets the icon for the specified window.
 *
 *  This function sets the icon of the specified window.  If passed an array of
 *  candidate images, those of or closest to the sizes desired by the system are
 *  selected.  If no images are specified, the window reverts to its default
 *  icon.
 *
 *  The pixels are 32-bit, little-endian, non-premultiplied RGBA, i.e. eight
 *  bits per channel with the red channel first.  They are arranged canonically
 *  as packed sequential rows, starting from the top-left corner.
 *
 *  The desired image sizes varies depending on platform and system settings.
 *  The selected images will be rescaled as needed.  Good sizes include 16x16,
 *  32x32 and 48x48.
 *
 *  @param[in] window The window whose icon to set.
 *  @param[in] count The number of images in the specified array, or zero to
 *  revert to the default window icon.
 *  @param[in] images The images to create the icon from.  This is ignored if
 *  count is zero.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref
 *  GLFW_INVALID_VALUE, @ref GLFW_PLATFORM_ERROR and @ref
 *  GLFW_FEATURE_UNAVAILABLE (see remarks).
 *
 *  @pointer_lifetime The specified image data is copied before this function
 *  returns.
 *
 *  @remark @macos Regular windows do not have icons on macOS.  This function
 *  will emit @ref GLFW_FEATURE_UNAVAILABLE.  The dock icon will be the same as
 *  the application bundle's icon.  For more information on bundles, see the
 *  [Bundle Programming Guide][bundle-guide] in the Mac Developer Library.
 *
 *  [bundle-guide]: https://developer.apple.com/library/mac/documentation/CoreFoundation/Conceptual/CFBundles/
 *
 *  @remark @wayland There is no existing protocol to change an icon, the
 *  window will thus inherit the one defined in the application's desktop file.
 *  This function will emit @ref GLFW_FEATURE_UNAVAILABLE.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_icon
 *
 *  @since Added in version 3.2.
 *
 *  @ingroup window
 */
GLFWAPI void glfwSetWindowIcon(GLFWwindow* window, int count, const GLFWimage* images);

/*! @brief Retrieves the position of the content area of the specified window.
 *
 *  This function retrieves the position, in screen coordinates, of the
 *  upper-left corner of the content area of the specified window.
 *
 *  Any or all of the position arguments may be `NULL`.  If an error occurs, all
 *  non-`NULL` position arguments will be set to zero.
 *
 *  @param[in] window The window to query.
 *  @param[out] xpos Where to store the x-coordinate of the upper-left corner of
 *  the content area, or `NULL`.
 *  @param[out] ypos Where to store the y-coordinate of the upper-left corner of
 *  the content area, or `NULL`.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref
 *  GLFW_PLATFORM_ERROR and @ref GLFW_FEATURE_UNAVAILABLE (see remarks).
 *
 *  @remark @wayland There is no way for an application to retrieve the global
 *  position of its windows.  This function will emit @ref
 *  GLFW_FEATURE_UNAVAILABLE.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_pos
 *  @sa @ref glfwSetWindowPos
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup window
 */
GLFWAPI void glfwGetWindowPos(GLFWwindow* window, int* xpos, int* ypos);

/*! @brief Sets the position of the content area of the specified window.
 *
 *  This function sets the position, in screen coordinates, of the upper-left
 *  corner of the content area of the specified windowed mode window.  If the
 *  window is a full screen window, this function does nothing.
 *
 *  __Do not use this function__ to move an already visible window unless you
 *  have very good reasons for doing so, as it will confuse and annoy the user.
 *
 *  The window manager may put limits on what positions are allowed.  GLFW
 *  cannot and should not override these limits.
 *
 *  @param[in] window The window to query.
 *  @param[in] xpos The x-coordinate of the upper-left corner of the content area.
 *  @param[in] ypos The y-coordinate of the upper-left corner of the content area.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref
 *  GLFW_PLATFORM_ERROR and @ref GLFW_FEATURE_UNAVAILABLE (see remarks).
 *
 *  @remark @wayland There is no way for an application to set the global
 *  position of its windows.  This function will emit @ref
 *  GLFW_FEATURE_UNAVAILABLE.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_pos
 *  @sa @ref glfwGetWindowPos
 *
 *  @since Added in version 1.0.
 *  @glfw3 Added window handle parameter.
 *
 *  @ingroup window
 */
GLFWAPI void glfwSetWindowPos(GLFWwindow* window, int xpos, int ypos);

/*! @brief Retrieves the size of the content area of the specified window.
 *
 *  This function retrieves the size, in screen coordinates, of the content area
 *  of the specified window.  If you wish to retrieve the size of the
 *  framebuffer of the window in pixels, see @ref glfwGetFramebufferSize.
 *
 *  Any or all of the size arguments may be `NULL`.  If an error occurs, all
 *  non-`NULL` size arguments will be set to zero.
 *
 *  @param[in] window The window whose size to retrieve.
 *  @param[out] width Where to store the width, in screen coordinates, of the
 *  content area, or `NULL`.
 *  @param[out] height Where to store the height, in screen coordinates, of the
 *  content area, or `NULL`.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_size
 *  @sa @ref glfwSetWindowSize
 *
 *  @since Added in version 1.0.
 *  @glfw3 Added window handle parameter.
 *
 *  @ingroup window
 */
GLFWAPI void glfwGetWindowSize(GLFWwindow* window, int* width, int* height);

/*! @brief Sets the size limits of the specified window.
 *
 *  This function sets the size limits of the content area of the specified
 *  window.  If the window is full screen, the size limits only take effect
 *  once it is made windowed.  If the window is not resizable, this function
 *  does nothing.
 *
 *  The size limits are applied immediately to a windowed mode window and may
 *  cause it to be resized.
 *
 *  The maximum dimensions must be greater than or equal to the minimum
 *  dimensions and all must be greater than or equal to zero.
 *
 *  @param[in] window The window to set limits for.
 *  @param[in] minwidth The minimum width, in screen coordinates, of the content
 *  area, or `GLFW_DONT_CARE`.
 *  @param[in] minheight The minimum height, in screen coordinates, of the
 *  content area, or `GLFW_DONT_CARE`.
 *  @param[in] maxwidth The maximum width, in screen coordinates, of the content
 *  area, or `GLFW_DONT_CARE`.
 *  @param[in] maxheight The maximum height, in screen coordinates, of the
 *  content area, or `GLFW_DONT_CARE`.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref
 *  GLFW_INVALID_VALUE and @ref GLFW_PLATFORM_ERROR.
 *
 *  @remark If you set size limits and an aspect ratio that conflict, the
 *  results are undefined.
 *
 *  @remark @wayland The size limits will not be applied until the window is
 *  actually resized, either by the user or by the compositor.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_sizelimits
 *  @sa @ref glfwSetWindowAspectRatio
 *
 *  @since Added in version 3.2.
 *
 *  @ingroup window
 */
GLFWAPI void glfwSetWindowSizeLimits(GLFWwindow* window, int minwidth, int minheight, int maxwidth, int maxheight);

/*! @brief Sets the aspect ratio of the specified window.
 *
 *  This function sets the required aspect ratio of the content area of the
 *  specified window.  If the window is full screen, the aspect ratio only takes
 *  effect once it is made windowed.  If the window is not resizable, this
 *  function does nothing.
 *
 *  The aspect ratio is specified as a numerator and a denominator and both
 *  values must be greater than zero.  For example, the common 16:9 aspect ratio
 *  is specified as 16 and 9, respectively.
 *
 *  If the numerator and denominator is set to `GLFW_DONT_CARE` then the aspect
 *  ratio limit is disabled.
 *
 *  The aspect ratio is applied immediately to a windowed mode window and may
 *  cause it to be resized.
 *
 *  @param[in] window The window to set limits for.
 *  @param[in] numer The numerator of the desired aspect ratio, or
 *  `GLFW_DONT_CARE`.
 *  @param[in] denom The denominator of the desired aspect ratio, or
 *  `GLFW_DONT_CARE`.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref
 *  GLFW_INVALID_VALUE and @ref GLFW_PLATFORM_ERROR.
 *
 *  @remark If you set size limits and an aspect ratio that conflict, the
 *  results are undefined.
 *
 *  @remark @wayland The aspect ratio will not be applied until the window is
 *  actually resized, either by the user or by the compositor.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_sizelimits
 *  @sa @ref glfwSetWindowSizeLimits
 *
 *  @since Added in version 3.2.
 *
 *  @ingroup window
 */
GLFWAPI void glfwSetWindowAspectRatio(GLFWwindow* window, int numer, int denom);

/*! @brief Sets the size of the content area of the specified window.
 *
 *  This function sets the size, in screen coordinates, of the content area of
 *  the specified window.
 *
 *  For full screen windows, this function updates the resolution of its desired
 *  video mode and switches to the video mode closest to it, without affecting
 *  the window's context.  As the context is unaffected, the bit depths of the
 *  framebuffer remain unchanged.
 *
 *  If you wish to update the refresh rate of the desired video mode in addition
 *  to its resolution, see @ref glfwSetWindowMonitor.
 *
 *  The window manager may put limits on what sizes are allowed.  GLFW cannot
 *  and should not override these limits.
 *
 *  @param[in] window The window to resize.
 *  @param[in] width The desired width, in screen coordinates, of the window
 *  content area.
 *  @param[in] height The desired height, in screen coordinates, of the window
 *  content area.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_size
 *  @sa @ref glfwGetWindowSize
 *  @sa @ref glfwSetWindowMonitor
 *
 *  @since Added in version 1.0.
 *  @glfw3 Added window handle parameter.
 *
 *  @ingroup window
 */
GLFWAPI void glfwSetWindowSize(GLFWwindow* window, int width, int height);

/*! @brief Retrieves the size of the framebuffer of the specified window.
 *
 *  This function retrieves the size, in pixels, of the framebuffer of the
 *  specified window.  If you wish to retrieve the size of the window in screen
 *  coordinates, see @ref glfwGetWindowSize.
 *
 *  Any or all of the size arguments may be `NULL`.  If an error occurs, all
 *  non-`NULL` size arguments will be set to zero.
 *
 *  @param[in] window The window whose framebuffer to query.
 *  @param[out] width Where to store the width, in pixels, of the framebuffer,
 *  or `NULL`.
 *  @param[out] height Where to store the height, in pixels, of the framebuffer,
 *  or `NULL`.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_fbsize
 *  @sa @ref glfwSetFramebufferSizeCallback
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup window
 */
GLFWAPI void glfwGetFramebufferSize(GLFWwindow* window, int* width, int* height);

/*! @brief Retrieves the size of the frame of the window.
 *
 *  This function retrieves the size, in screen coordinates, of each edge of the
 *  frame of the specified window.  This size includes the title bar, if the
 *  window has one.  The size of the frame may vary depending on the
 *  [window-related hints](@ref window_hints_wnd) used to create it.
 *
 *  Because this function retrieves the size of each window frame edge and not
 *  the offset along a particular coordinate axis, the retrieved values will
 *  always be zero or positive.
 *
 *  Any or all of the size arguments may be `NULL`.  If an error occurs, all
 *  non-`NULL` size arguments will be set to zero.
 *
 *  @param[in] window The window whose frame size to query.
 *  @param[out] left Where to store the size, in screen coordinates, of the left
 *  edge of the window frame, or `NULL`.
 *  @param[out] top Where to store the size, in screen coordinates, of the top
 *  edge of the window frame, or `NULL`.
 *  @param[out] right Where to store the size, in screen coordinates, of the
 *  right edge of the window frame, or `NULL`.
 *  @param[out] bottom Where to store the size, in screen coordinates, of the
 *  bottom edge of the window frame, or `NULL`.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_size
 *
 *  @since Added in version 3.1.
 *
 *  @ingroup window
 */
GLFWAPI void glfwGetWindowFrameSize(GLFWwindow* window, int* left, int* top, int* right, int* bottom);

/*! @brief Retrieves the content scale for the specified window.
 *
 *  This function retrieves the content scale for the specified window.  The
 *  content scale is the ratio between the current DPI and the platform's
 *  default DPI.  This is especially important for text and any UI elements.  If
 *  the pixel dimensions of your UI scaled by this look appropriate on your
 *  machine then it should appear at a reasonable size on other machines
 *  regardless of their DPI and scaling settings.  This relies on the system DPI
 *  and scaling settings being somewhat correct.
 *
 *  On platforms where each monitors can have its own content scale, the window
 *  content scale will depend on which monitor the system considers the window
 *  to be on.
 *
 *  @param[in] window The window to query.
 *  @param[out] xscale Where to store the x-axis content scale, or `NULL`.
 *  @param[out] yscale Where to store the y-axis content scale, or `NULL`.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_scale
 *  @sa @ref glfwSetWindowContentScaleCallback
 *  @sa @ref glfwGetMonitorContentScale
 *
 *  @since Added in version 3.3.
 *
 *  @ingroup window
 */
GLFWAPI void glfwGetWindowContentScale(GLFWwindow* window, float* xscale, float* yscale);

/*! @brief Returns the opacity of the whole window.
 *
 *  This function returns the opacity of the window, including any decorations.
 *
 *  The opacity (or alpha) value is a positive finite number between zero and
 *  one, where zero is fully transparent and one is fully opaque.  If the system
 *  does not support whole window transparency, this function always returns one.
 *
 *  The initial opacity value for newly created windows is one.
 *
 *  @param[in] window The window to query.
 *  @return The opacity value of the specified window.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_transparency
 *  @sa @ref glfwSetWindowOpacity
 *
 *  @since Added in version 3.3.
 *
 *  @ingroup window
 */
GLFWAPI float glfwGetWindowOpacity(GLFWwindow* window);

/*! @brief Sets the opacity of the whole window.
 *
 *  This function sets the opacity of the window, including any decorations.
 *
 *  The opacity (or alpha) value is a positive finite number between zero and
 *  one, where zero is fully transparent and one is fully opaque.
 *
 *  The initial opacity value for newly created windows is one.
 *
 *  A window created with framebuffer transparency may not use whole window
 *  transparency.  The results of doing this are undefined.
 *
 *  @param[in] window The window to set the opacity for.
 *  @param[in] opacity The desired opacity of the specified window.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref
 *  GLFW_PLATFORM_ERROR and @ref GLFW_FEATURE_UNAVAILABLE (see remarks).
 *
 *  @remark @wayland There is no way to set an opacity factor for a window.
 *  This function will emit @ref GLFW_FEATURE_UNAVAILABLE.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_transparency
 *  @sa @ref glfwGetWindowOpacity
 *
 *  @since Added in version 3.3.
 *
 *  @ingroup window
 */
GLFWAPI void glfwSetWindowOpacity(GLFWwindow* window, float opacity);

/*! @brief Iconifies the specified window.
 *
 *  This function iconifies (minimizes) the specified window if it was
 *  previously restored.  If the window is already iconified, this function does
 *  nothing.
 *
 *  If the specified window is a full screen window, GLFW restores the original
 *  video mode of the monitor.  The window's desired video mode is set again
 *  when the window is restored.
 *
 *  @param[in] window The window to iconify.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @remark @wayland Once a window is iconified, @ref glfwRestoreWindow won’t
 *  be able to restore it.  This is a design decision of the xdg-shell
 *  protocol.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_iconify
 *  @sa @ref glfwRestoreWindow
 *  @sa @ref glfwMaximizeWindow
 *
 *  @since Added in version 2.1.
 *  @glfw3 Added window handle parameter.
 *
 *  @ingroup window
 */
GLFWAPI void glfwIconifyWindow(GLFWwindow* window);

/*! @brief Restores the specified window.
 *
 *  This function restores the specified window if it was previously iconified
 *  (minimized) or maximized.  If the window is already restored, this function
 *  does nothing.
 *
 *  If the specified window is an iconified full screen window, its desired
 *  video mode is set again for its monitor when the window is restored.
 *
 *  @param[in] window The window to restore.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_iconify
 *  @sa @ref glfwIconifyWindow
 *  @sa @ref glfwMaximizeWindow
 *
 *  @since Added in version 2.1.
 *  @glfw3 Added window handle parameter.
 *
 *  @ingroup window
 */
GLFWAPI void glfwRestoreWindow(GLFWwindow* window);

/*! @brief Maximizes the specified window.
 *
 *  This function maximizes the specified window if it was previously not
 *  maximized.  If the window is already maximized, this function does nothing.
 *
 *  If the specified window is a full screen window, this function does nothing.
 *
 *  @param[in] window The window to maximize.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @par Thread Safety
 *  This function may only be called from the main thread.
 *
 *  @sa @ref window_iconify
 *  @sa @ref glfwIconifyWindow
 *  @sa @ref glfwRestoreWindow
 *
 *  @since Added in GLFW 3.2.
 *
 *  @ingroup window
 */
GLFWAPI void glfwMaximizeWindow(GLFWwindow* window);

/*! @brief Makes the specified window visible.
 *
 *  This function makes the specified window visible if it was previously
 *  hidden.  If the window is already visible or is in full screen mode, this
 *  function does nothing.
 *
 *  By default, windowed mode windows are focused when shown
 *  Set the [GLFW_FOCUS_ON_SHOW](@ref GLFW_FOCUS_ON_SHOW_hint) window hint
 *  to change this behavior for all newly created windows, or change the
 *  behavior for an existing window with @ref glfwSetWindowAttrib.
 *
 *  @param[in] window The window to make visible.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @remark @wayland Because Wayland wants every frame of the desktop to be
 *  complete, this function does not immediately make the window visible.
 *  Instead it will become visible the next time the window framebuffer is
 *  updated after this call.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_hide
 *  @sa @ref glfwHideWindow
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup window
 */
GLFWAPI void glfwShowWindow(GLFWwindow* window);

/*! @brief Hides the specified window.
 *
 *  This function hides the specified window if it was previously visible.  If
 *  the window is already hidden or is in full screen mode, this function does
 *  nothing.
 *
 *  @param[in] window The window to hide.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_hide
 *  @sa @ref glfwShowWindow
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup window
 */
GLFWAPI void glfwHideWindow(GLFWwindow* window);

/*! @brief Brings the specified window to front and sets input focus.
 *
 *  This function brings the specified window to front and sets input focus.
 *  The window should already be visible and not iconified.
 *
 *  By default, both windowed and full screen mode windows are focused when
 *  initially created.  Set the [GLFW_FOCUSED](@ref GLFW_FOCUSED_hint) to
 *  disable this behavior.
 *
 *  Also by default, windowed mode windows are focused when shown
 *  with @ref glfwShowWindow. Set the
 *  [GLFW_FOCUS_ON_SHOW](@ref GLFW_FOCUS_ON_SHOW_hint) to disable this behavior.
 *
 *  __Do not use this function__ to steal focus from other applications unless
 *  you are certain that is what the user wants.  Focus stealing can be
 *  extremely disruptive.
 *
 *  For a less disruptive way of getting the user's attention, see
 *  [attention requests](@ref window_attention).
 *
 *  @param[in] window The window to give input focus.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @remark @wayland The compositor will likely ignore focus requests unless
 *  another window created by the same application already has input focus.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_focus
 *  @sa @ref window_attention
 *
 *  @since Added in version 3.2.
 *
 *  @ingroup window
 */
GLFWAPI void glfwFocusWindow(GLFWwindow* window);

/*! @brief Requests user attention to the specified window.
 *
 *  This function requests user attention to the specified window.  On
 *  platforms where this is not supported, attention is requested to the
 *  application as a whole.
 *
 *  Once the user has given attention, usually by focusing the window or
 *  application, the system will end the request automatically.
 *
 *  @param[in] window The window to request attention to.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @remark @macos Attention is requested to the application as a whole, not the
 *  specific window.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_attention
 *
 *  @since Added in version 3.3.
 *
 *  @ingroup window
 */
GLFWAPI void glfwRequestWindowAttention(GLFWwindow* window);

/*! @brief Returns the monitor that the window uses for full screen mode.
 *
 *  This function returns the handle of the monitor that the specified window is
 *  in full screen on.
 *
 *  @param[in] window The window to query.
 *  @return The monitor, or `NULL` if the window is in windowed mode or an
 *  [error](@ref error_handling) occurred.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_monitor
 *  @sa @ref glfwSetWindowMonitor
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup window
 */
GLFWAPI GLFWmonitor* glfwGetWindowMonitor(GLFWwindow* window);

/*! @brief Sets the mode, monitor, video mode and placement of a window.
 *
 *  This function sets the monitor that the window uses for full screen mode or,
 *  if the monitor is `NULL`, makes it windowed mode.
 *
 *  When setting a monitor, this function updates the width, height and refresh
 *  rate of the desired video mode and switches to the video mode closest to it.
 *  The window position is ignored when setting a monitor.
 *
 *  When the monitor is `NULL`, the position, width and height are used to
 *  place the window content area.  The refresh rate is ignored when no monitor
 *  is specified.
 *
 *  If you only wish to update the resolution of a full screen window or the
 *  size of a windowed mode window, see @ref glfwSetWindowSize.
 *
 *  When a window transitions from full screen to windowed mode, this function
 *  restores any previous window settings such as whether it is decorated,
 *  floating, resizable, has size or aspect ratio limits, etc.
 *
 *  @param[in] window The window whose monitor, size or video mode to set.
 *  @param[in] monitor The desired monitor, or `NULL` to set windowed mode.
 *  @param[in] xpos The desired x-coordinate of the upper-left corner of the
 *  content area.
 *  @param[in] ypos The desired y-coordinate of the upper-left corner of the
 *  content area.
 *  @param[in] width The desired with, in screen coordinates, of the content
 *  area or video mode.
 *  @param[in] height The desired height, in screen coordinates, of the content
 *  area or video mode.
 *  @param[in] refreshRate The desired refresh rate, in Hz, of the video mode,
 *  or `GLFW_DONT_CARE`.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @remark The OpenGL or OpenGL ES context will not be destroyed or otherwise
 *  affected by any resizing or mode switching, although you may need to update
 *  your viewport if the framebuffer size has changed.
 *
 *  @remark @wayland The desired window position is ignored, as there is no way
 *  for an application to set this property.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_monitor
 *  @sa @ref window_full_screen
 *  @sa @ref glfwGetWindowMonitor
 *  @sa @ref glfwSetWindowSize
 *
 *  @since Added in version 3.2.
 *
 *  @ingroup window
 */
GLFWAPI void glfwSetWindowMonitor(GLFWwindow* window, GLFWmonitor* monitor, int xpos, int ypos, int width, int height, int refreshRate);

/*! @brief Returns an attribute of the specified window.
 *
 *  This function returns the value of an attribute of the specified window or
 *  its OpenGL or OpenGL ES context.
 *
 *  @param[in] window The window to query.
 *  @param[in] attrib The [window attribute](@ref window_attribs) whose value to
 *  return.
 *  @return The value of the attribute, or zero if an
 *  [error](@ref error_handling) occurred.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref
 *  GLFW_INVALID_ENUM and @ref GLFW_PLATFORM_ERROR.
 *
 *  @remark Framebuffer related hints are not window attributes.  See @ref
 *  window_attribs_fb for more information.
 *
 *  @remark Zero is a valid value for many window and context related
 *  attributes so you cannot use a return value of zero as an indication of
 *  errors.  However, this function should not fail as long as it is passed
 *  valid arguments and the library has been [initialized](@ref intro_init).
 *
 *  @remark @wayland The Wayland protocol provides no way to check whether a
 *  window is iconfied, so @ref GLFW_ICONIFIED always returns `GLFW_FALSE`.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_attribs
 *  @sa @ref glfwSetWindowAttrib
 *
 *  @since Added in version 3.0.  Replaces `glfwGetWindowParam` and
 *  `glfwGetGLVersion`.
 *
 *  @ingroup window
 */
GLFWAPI int glfwGetWindowAttrib(GLFWwindow* window, int attrib);

/*! @brief Sets an attribute of the specified window.
 *
 *  This function sets the value of an attribute of the specified window.
 *
 *  The supported attributes are [GLFW_DECORATED](@ref GLFW_DECORATED_attrib),
 *  [GLFW_RESIZABLE](@ref GLFW_RESIZABLE_attrib),
 *  [GLFW_FLOATING](@ref GLFW_FLOATING_attrib),
 *  [GLFW_AUTO_ICONIFY](@ref GLFW_AUTO_ICONIFY_attrib) and
 *  [GLFW_FOCUS_ON_SHOW](@ref GLFW_FOCUS_ON_SHOW_attrib).
 *  [GLFW_MOUSE_PASSTHROUGH](@ref GLFW_MOUSE_PASSTHROUGH_attrib)
 *
 *  Some of these attributes are ignored for full screen windows.  The new
 *  value will take effect if the window is later made windowed.
 *
 *  Some of these attributes are ignored for windowed mode windows.  The new
 *  value will take effect if the window is later made full screen.
 *
 *  @param[in] window The window to set the attribute for.
 *  @param[in] attrib A supported window attribute.
 *  @param[in] value `GLFW_TRUE` or `GLFW_FALSE`.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref
 *  GLFW_INVALID_ENUM, @ref GLFW_INVALID_VALUE, @ref GLFW_PLATFORM_ERROR and @ref
 *  GLFW_FEATURE_UNAVAILABLE (see remarks).
 *
 *  @remark Calling @ref glfwGetWindowAttrib will always return the latest
 *  value, even if that value is ignored by the current mode of the window.
 *
 *  @remark @wayland The [GLFW_FLOATING](@ref GLFW_FLOATING_attrib) window attribute is
 *  not supported.  Setting this will emit @ref GLFW_FEATURE_UNAVAILABLE.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_attribs
 *  @sa @ref glfwGetWindowAttrib
 *
 *  @since Added in version 3.3.
 *
 *  @ingroup window
 */
GLFWAPI void glfwSetWindowAttrib(GLFWwindow* window, int attrib, int value);

/*! @brief Sets the user pointer of the specified window.
 *
 *  This function sets the user-defined pointer of the specified window.  The
 *  current value is retained until the window is destroyed.  The initial value
 *  is `NULL`.
 *
 *  @param[in] window The window whose pointer to set.
 *  @param[in] pointer The new value.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function may be called from any thread.  Access is not
 *  synchronized.
 *
 *  @sa @ref window_userptr
 *  @sa @ref glfwGetWindowUserPointer
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup window
 */
GLFWAPI void glfwSetWindowUserPointer(GLFWwindow* window, void* pointer);

/*! @brief Returns the user pointer of the specified window.
 *
 *  This function returns the current value of the user-defined pointer of the
 *  specified window.  The initial value is `NULL`.
 *
 *  @param[in] window The window whose pointer to return.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function may be called from any thread.  Access is not
 *  synchronized.
 *
 *  @sa @ref window_userptr
 *  @sa @ref glfwSetWindowUserPointer
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup window
 */
GLFWAPI void* glfwGetWindowUserPointer(GLFWwindow* window);

/*! @brief Sets the position callback for the specified window.
 *
 *  This function sets the position callback of the specified window, which is
 *  called when the window is moved.  The callback is provided with the
 *  position, in screen coordinates, of the upper-left corner of the content
 *  area of the window.
 *
 *  @param[in] window The window whose callback to set.
 *  @param[in] callback The new callback, or `NULL` to remove the currently set
 *  callback.
 *  @return The previously set callback, or `NULL` if no callback was set or the
 *  library had not been [initialized](@ref intro_init).
 *
 *  @callback_signature
 *  @code
 *  void function_name(GLFWwindow* window, int xpos, int ypos)
 *  @endcode
 *  For more information about the callback parameters, see the
 *  [function pointer type](@ref GLFWwindowposfun).
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @remark @wayland This callback will never be called, as there is no way for
 *  an application to know its global position.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_pos
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup window
 */
GLFWAPI GLFWwindowposfun glfwSetWindowPosCallback(GLFWwindow* window, GLFWwindowposfun callback);

/*! @brief Sets the size callback for the specified window.
 *
 *  This function sets the size callback of the specified window, which is
 *  called when the window is resized.  The callback is provided with the size,
 *  in screen coordinates, of the content area of the window.
 *
 *  @param[in] window The window whose callback to set.
 *  @param[in] callback The new callback, or `NULL` to remove the currently set
 *  callback.
 *  @return The previously set callback, or `NULL` if no callback was set or the
 *  library had not been [initialized](@ref intro_init).
 *
 *  @callback_signature
 *  @code
 *  void function_name(GLFWwindow* window, int width, int height)
 *  @endcode
 *  For more information about the callback parameters, see the
 *  [function pointer type](@ref GLFWwindowsizefun).
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_size
 *
 *  @since Added in version 1.0.
 *  @glfw3 Added window handle parameter and return value.
 *
 *  @ingroup window
 */
GLFWAPI GLFWwindowsizefun glfwSetWindowSizeCallback(GLFWwindow* window, GLFWwindowsizefun callback);

/*! @brief Sets the close callback for the specified window.
 *
 *  This function sets the close callback of the specified window, which is
 *  called when the user attempts to close the window, for example by clicking
 *  the close widget in the title bar.
 *
 *  The close flag is set before this callback is called, but you can modify it
 *  at any time with @ref glfwSetWindowShouldClose.
 *
 *  The close callback is not triggered by @ref glfwDestroyWindow.
 *
 *  @param[in] window The window whose callback to set.
 *  @param[in] callback The new callback, or `NULL` to remove the currently set
 *  callback.
 *  @return The previously set callback, or `NULL` if no callback was set or the
 *  library had not been [initialized](@ref intro_init).
 *
 *  @callback_signature
 *  @code
 *  void function_name(GLFWwindow* window)
 *  @endcode
 *  For more information about the callback parameters, see the
 *  [function pointer type](@ref GLFWwindowclosefun).
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @remark @macos Selecting Quit from the application menu will trigger the
 *  close callback for all windows.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_close
 *
 *  @since Added in version 2.5.
 *  @glfw3 Added window handle parameter and return value.
 *
 *  @ingroup window
 */
GLFWAPI GLFWwindowclosefun glfwSetWindowCloseCallback(GLFWwindow* window, GLFWwindowclosefun callback);

/*! @brief Sets the refresh callback for the specified window.
 *
 *  This function sets the refresh callback of the specified window, which is
 *  called when the content area of the window needs to be redrawn, for example
 *  if the window has been exposed after having been covered by another window.
 *
 *  On compositing window systems such as Aero, Compiz, Aqua or Wayland, where
 *  the window contents are saved off-screen, this callback may be called only
 *  very infrequently or never at all.
 *
 *  @param[in] window The window whose callback to set.
 *  @param[in] callback The new callback, or `NULL` to remove the currently set
 *  callback.
 *  @return The previously set callback, or `NULL` if no callback was set or the
 *  library had not been [initialized](@ref intro_init).
 *
 *  @callback_signature
 *  @code
 *  void function_name(GLFWwindow* window);
 *  @endcode
 *  For more information about the callback parameters, see the
 *  [function pointer type](@ref GLFWwindowrefreshfun).
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_refresh
 *
 *  @since Added in version 2.5.
 *  @glfw3 Added window handle parameter and return value.
 *
 *  @ingroup window
 */
GLFWAPI GLFWwindowrefreshfun glfwSetWindowRefreshCallback(GLFWwindow* window, GLFWwindowrefreshfun callback);

/*! @brief Sets the focus callback for the specified window.
 *
 *  This function sets the focus callback of the specified window, which is
 *  called when the window gains or loses input focus.
 *
 *  After the focus callback is called for a window that lost input focus,
 *  synthetic key and mouse button release events will be generated for all such
 *  that had been pressed.  For more information, see @ref glfwSetKeyCallback
 *  and @ref glfwSetMouseButtonCallback.
 *
 *  @param[in] window The window whose callback to set.
 *  @param[in] callback The new callback, or `NULL` to remove the currently set
 *  callback.
 *  @return The previously set callback, or `NULL` if no callback was set or the
 *  library had not been [initialized](@ref intro_init).
 *
 *  @callback_signature
 *  @code
 *  void function_name(GLFWwindow* window, int focused)
 *  @endcode
 *  For more information about the callback parameters, see the
 *  [function pointer type](@ref GLFWwindowfocusfun).
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_focus
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup window
 */
GLFWAPI GLFWwindowfocusfun glfwSetWindowFocusCallback(GLFWwindow* window, GLFWwindowfocusfun callback);

/*! @brief Sets the iconify callback for the specified window.
 *
 *  This function sets the iconification callback of the specified window, which
 *  is called when the window is iconified or restored.
 *
 *  @param[in] window The window whose callback to set.
 *  @param[in] callback The new callback, or `NULL` to remove the currently set
 *  callback.
 *  @return The previously set callback, or `NULL` if no callback was set or the
 *  library had not been [initialized](@ref intro_init).
 *
 *  @callback_signature
 *  @code
 *  void function_name(GLFWwindow* window, int iconified)
 *  @endcode
 *  For more information about the callback parameters, see the
 *  [function pointer type](@ref GLFWwindowiconifyfun).
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_iconify
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup window
 */
GLFWAPI GLFWwindowiconifyfun glfwSetWindowIconifyCallback(GLFWwindow* window, GLFWwindowiconifyfun callback);

/*! @brief Sets the maximize callback for the specified window.
 *
 *  This function sets the maximization callback of the specified window, which
 *  is called when the window is maximized or restored.
 *
 *  @param[in] window The window whose callback to set.
 *  @param[in] callback The new callback, or `NULL` to remove the currently set
 *  callback.
 *  @return The previously set callback, or `NULL` if no callback was set or the
 *  library had not been [initialized](@ref intro_init).
 *
 *  @callback_signature
 *  @code
 *  void function_name(GLFWwindow* window, int maximized)
 *  @endcode
 *  For more information about the callback parameters, see the
 *  [function pointer type](@ref GLFWwindowmaximizefun).
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_maximize
 *
 *  @since Added in version 3.3.
 *
 *  @ingroup window
 */
GLFWAPI GLFWwindowmaximizefun glfwSetWindowMaximizeCallback(GLFWwindow* window, GLFWwindowmaximizefun callback);

/*! @brief Sets the framebuffer resize callback for the specified window.
 *
 *  This function sets the framebuffer resize callback of the specified window,
 *  which is called when the framebuffer of the specified window is resized.
 *
 *  @param[in] window The window whose callback to set.
 *  @param[in] callback The new callback, or `NULL` to remove the currently set
 *  callback.
 *  @return The previously set callback, or `NULL` if no callback was set or the
 *  library had not been [initialized](@ref intro_init).
 *
 *  @callback_signature
 *  @code
 *  void function_name(GLFWwindow* window, int width, int height)
 *  @endcode
 *  For more information about the callback parameters, see the
 *  [function pointer type](@ref GLFWframebuffersizefun).
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_fbsize
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup window
 */
GLFWAPI GLFWframebuffersizefun glfwSetFramebufferSizeCallback(GLFWwindow* window, GLFWframebuffersizefun callback);

/*! @brief Sets the window content scale callback for the specified window.
 *
 *  This function sets the window content scale callback of the specified window,
 *  which is called when the content scale of the specified window changes.
 *
 *  @param[in] window The window whose callback to set.
 *  @param[in] callback The new callback, or `NULL` to remove the currently set
 *  callback.
 *  @return The previously set callback, or `NULL` if no callback was set or the
 *  library had not been [initialized](@ref intro_init).
 *
 *  @callback_signature
 *  @code
 *  void function_name(GLFWwindow* window, float xscale, float yscale)
 *  @endcode
 *  For more information about the callback parameters, see the
 *  [function pointer type](@ref GLFWwindowcontentscalefun).
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref window_scale
 *  @sa @ref glfwGetWindowContentScale
 *
 *  @since Added in version 3.3.
 *
 *  @ingroup window
 */
GLFWAPI GLFWwindowcontentscalefun glfwSetWindowContentScaleCallback(GLFWwindow* window, GLFWwindowcontentscalefun callback);

/*! @brief Processes all pending events.
 *
 *  This function processes only those events that are already in the event
 *  queue and then returns immediately.  Processing events will cause the window
 *  and input callbacks associated with those events to be called.
 *
 *  On some platforms, a window move, resize or menu operation will cause event
 *  processing to block.  This is due to how event processing is designed on
 *  those platforms.  You can use the
 *  [window refresh callback](@ref window_refresh) to redraw the contents of
 *  your window when necessary during such operations.
 *
 *  Do not assume that callbacks you set will _only_ be called in response to
 *  event processing functions like this one.  While it is necessary to poll for
 *  events, window systems that require GLFW to register callbacks of its own
 *  can pass events to GLFW in response to many window system function calls.
 *  GLFW will pass those events on to the application callbacks before
 *  returning.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @reentrancy This function must not be called from a callback.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref events
 *  @sa @ref glfwWaitEvents
 *  @sa @ref glfwWaitEventsTimeout
 *
 *  @since Added in version 1.0.
 *
 *  @ingroup window
 */
GLFWAPI void glfwPollEvents(void);

/*! @brief Waits until events are queued and processes them.
 *
 *  This function puts the calling thread to sleep until at least one event is
 *  available in the event queue.  Once one or more events are available,
 *  it behaves exactly like @ref glfwPollEvents, i.e. the events in the queue
 *  are processed and the function then returns immediately.  Processing events
 *  will cause the window and input callbacks associated with those events to be
 *  called.
 *
 *  Since not all events are associated with callbacks, this function may return
 *  without a callback having been called even if you are monitoring all
 *  callbacks.
 *
 *  On some platforms, a window move, resize or menu operation will cause event
 *  processing to block.  This is due to how event processing is designed on
 *  those platforms.  You can use the
 *  [window refresh callback](@ref window_refresh) to redraw the contents of
 *  your window when necessary during such operations.
 *
 *  Do not assume that callbacks you set will _only_ be called in response to
 *  event processing functions like this one.  While it is necessary to poll for
 *  events, window systems that require GLFW to register callbacks of its own
 *  can pass events to GLFW in response to many window system function calls.
 *  GLFW will pass those events on to the application callbacks before
 *  returning.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @reentrancy This function must not be called from a callback.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref events
 *  @sa @ref glfwPollEvents
 *  @sa @ref glfwWaitEventsTimeout
 *
 *  @since Added in version 2.5.
 *
 *  @ingroup window
 */
GLFWAPI void glfwWaitEvents(void);

/*! @brief Waits with timeout until events are queued and processes them.
 *
 *  This function puts the calling thread to sleep until at least one event is
 *  available in the event queue, or until the specified timeout is reached.  If
 *  one or more events are available, it behaves exactly like @ref
 *  glfwPollEvents, i.e. the events in the queue are processed and the function
 *  then returns immediately.  Processing events will cause the window and input
 *  callbacks associated with those events to be called.
 *
 *  The timeout value must be a positive finite number.
 *
 *  Since not all events are associated with callbacks, this function may return
 *  without a callback having been called even if you are monitoring all
 *  callbacks.
 *
 *  On some platforms, a window move, resize or menu operation will cause event
 *  processing to block.  This is due to how event processing is designed on
 *  those platforms.  You can use the
 *  [window refresh callback](@ref window_refresh) to redraw the contents of
 *  your window when necessary during such operations.
 *
 *  Do not assume that callbacks you set will _only_ be called in response to
 *  event processing functions like this one.  While it is necessary to poll for
 *  events, window systems that require GLFW to register callbacks of its own
 *  can pass events to GLFW in response to many window system function calls.
 *  GLFW will pass those events on to the application callbacks before
 *  returning.
 *
 *  @param[in] timeout The maximum amount of time, in seconds, to wait.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref
 *  GLFW_INVALID_VALUE and @ref GLFW_PLATFORM_ERROR.
 *
 *  @reentrancy This function must not be called from a callback.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref events
 *  @sa @ref glfwPollEvents
 *  @sa @ref glfwWaitEvents
 *
 *  @since Added in version 3.2.
 *
 *  @ingroup window
 */
GLFWAPI void glfwWaitEventsTimeout(double timeout);

/*! @brief Posts an empty event to the event queue.
 *
 *  This function posts an empty event from the current thread to the event
 *  queue, causing @ref glfwWaitEvents or @ref glfwWaitEventsTimeout to return.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @thread_safety This function may be called from any thread.
 *
 *  @sa @ref events
 *  @sa @ref glfwWaitEvents
 *  @sa @ref glfwWaitEventsTimeout
 *
 *  @since Added in version 3.1.
 *
 *  @ingroup window
 */
GLFWAPI void glfwPostEmptyEvent(void);

/*! @brief Returns the value of an input option for the specified window.
 *
 *  This function returns the value of an input option for the specified window.
 *  The mode must be one of @ref GLFW_CURSOR, @ref GLFW_STICKY_KEYS,
 *  @ref GLFW_STICKY_MOUSE_BUTTONS, @ref GLFW_LOCK_KEY_MODS or
 *  @ref GLFW_RAW_MOUSE_MOTION.
 *
 *  @param[in] window The window to query.
 *  @param[in] mode One of `GLFW_CURSOR`, `GLFW_STICKY_KEYS`,
 *  `GLFW_STICKY_MOUSE_BUTTONS`, `GLFW_LOCK_KEY_MODS` or
 *  `GLFW_RAW_MOUSE_MOTION`.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_INVALID_ENUM.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref glfwSetInputMode
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup input
 */
GLFWAPI int glfwGetInputMode(GLFWwindow* window, int mode);

/*! @brief Sets an input option for the specified window.
 *
 *  This function sets an input mode option for the specified window.  The mode
 *  must be one of @ref GLFW_CURSOR, @ref GLFW_STICKY_KEYS,
 *  @ref GLFW_STICKY_MOUSE_BUTTONS, @ref GLFW_LOCK_KEY_MODS or
 *  @ref GLFW_RAW_MOUSE_MOTION.
 *
 *  If the mode is `GLFW_CURSOR`, the value must be one of the following cursor
 *  modes:
 *  - `GLFW_CURSOR_NORMAL` makes the cursor visible and behaving normally.
 *  - `GLFW_CURSOR_HIDDEN` makes the cursor invisible when it is over the
 *    content area of the window but does not restrict the cursor from leaving.
 *  - `GLFW_CURSOR_DISABLED` hides and grabs the cursor, providing virtual
 *    and unlimited cursor movement.  This is useful for implementing for
 *    example 3D camera controls.
 *  - `GLFW_CURSOR_CAPTURED` makes the cursor visible and confines it to the
 *    content area of the window.
 *
 *  If the mode is `GLFW_STICKY_KEYS`, the value must be either `GLFW_TRUE` to
 *  enable sticky keys, or `GLFW_FALSE` to disable it.  If sticky keys are
 *  enabled, a key press will ensure that @ref glfwGetKey returns `GLFW_PRESS`
 *  the next time it is called even if the key had been released before the
 *  call.  This is useful when you are only interested in whether keys have been
 *  pressed but not when or in which order.
 *
 *  If the mode is `GLFW_STICKY_MOUSE_BUTTONS`, the value must be either
 *  `GLFW_TRUE` to enable sticky mouse buttons, or `GLFW_FALSE` to disable it.
 *  If sticky mouse buttons are enabled, a mouse button press will ensure that
 *  @ref glfwGetMouseButton returns `GLFW_PRESS` the next time it is called even
 *  if the mouse button had been released before the call.  This is useful when
 *  you are only interested in whether mouse buttons have been pressed but not
 *  when or in which order.
 *
 *  If the mode is `GLFW_LOCK_KEY_MODS`, the value must be either `GLFW_TRUE` to
 *  enable lock key modifier bits, or `GLFW_FALSE` to disable them.  If enabled,
 *  callbacks that receive modifier bits will also have the @ref
 *  GLFW_MOD_CAPS_LOCK bit set when the event was generated with Caps Lock on,
 *  and the @ref GLFW_MOD_NUM_LOCK bit when Num Lock was on.
 *
 *  If the mode is `GLFW_RAW_MOUSE_MOTION`, the value must be either `GLFW_TRUE`
 *  to enable raw (unscaled and unaccelerated) mouse motion when the cursor is
 *  disabled, or `GLFW_FALSE` to disable it.  If raw motion is not supported,
 *  attempting to set this will emit @ref GLFW_FEATURE_UNAVAILABLE.  Call @ref
 *  glfwRawMouseMotionSupported to check for support.
 *
 *  @param[in] window The window whose input mode to set.
 *  @param[in] mode One of `GLFW_CURSOR`, `GLFW_STICKY_KEYS`,
 *  `GLFW_STICKY_MOUSE_BUTTONS`, `GLFW_LOCK_KEY_MODS` or
 *  `GLFW_RAW_MOUSE_MOTION`.
 *  @param[in] value The new value of the specified input mode.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref
 *  GLFW_INVALID_ENUM, @ref GLFW_PLATFORM_ERROR and @ref
 *  GLFW_FEATURE_UNAVAILABLE (see above).
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref glfwGetInputMode
 *
 *  @since Added in version 3.0.  Replaces `glfwEnable` and `glfwDisable`.
 *
 *  @ingroup input
 */
GLFWAPI void glfwSetInputMode(GLFWwindow* window, int mode, int value);

/*! @brief Returns whether raw mouse motion is supported.
 *
 *  This function returns whether raw mouse motion is supported on the current
 *  system.  This status does not change after GLFW has been initialized so you
 *  only need to check this once.  If you attempt to enable raw motion on
 *  a system that does not support it, @ref GLFW_PLATFORM_ERROR will be emitted.
 *
 *  Raw mouse motion is closer to the actual motion of the mouse across
 *  a surface.  It is not affected by the scaling and acceleration applied to
 *  the motion of the desktop cursor.  That processing is suitable for a cursor
 *  while raw motion is better for controlling for example a 3D camera.  Because
 *  of this, raw mouse motion is only provided when the cursor is disabled.
 *
 *  @return `GLFW_TRUE` if raw mouse motion is supported on the current machine,
 *  or `GLFW_FALSE` otherwise.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref raw_mouse_motion
 *  @sa @ref glfwSetInputMode
 *
 *  @since Added in version 3.3.
 *
 *  @ingroup input
 */
GLFWAPI int glfwRawMouseMotionSupported(void);

/*! @brief Returns the platform-specific scancode of the specified key.
 *
 *  This function returns the platform-specific scancode of the specified key.
 *
 *  If the specified [key token](@ref keys) corresponds to a physical key not
 *  supported on the current platform then this method will return `-1`.
 *  Calling this function with anything other than a key token will return `-1`
 *  and generate a @ref GLFW_INVALID_ENUM error.
 *
 *  @param[in] key Any [key token](@ref keys).
 *  @return The platform-specific scancode for the key, or `-1` if the key is
 *  not supported on the current platform or an [error](@ref error_handling)
 *  occurred.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_INVALID_ENUM.
 *
 *  @thread_safety This function may be called from any thread.
 *
 *  @sa @ref input_key
 *
 *  @since Added in version 3.3.
 *
 *  @ingroup input
 */
GLFWAPI int glfwGetKeyScancode(int key);

/*! @brief Returns the last reported state of a keyboard key for the specified
 *  window.
 *
 *  This function returns the last state reported for the specified key to the
 *  specified window.  The returned state is one of `GLFW_PRESS` or
 *  `GLFW_RELEASE`.  The action `GLFW_REPEAT` is only reported to the key callback.
 *
 *  If the @ref GLFW_STICKY_KEYS input mode is enabled, this function returns
 *  `GLFW_PRESS` the first time you call it for a key that was pressed, even if
 *  that key has already been released.
 *
 *  The key functions deal with physical keys, with [key tokens](@ref keys)
 *  named after their use on the standard US keyboard layout.  If you want to
 *  input text, use the Unicode character callback instead.
 *
 *  The [modifier key bit masks](@ref mods) are not key tokens and cannot be
 *  used with this function.
 *
 *  __Do not use this function__ to implement [text input](@ref input_char).
 *
 *  @param[in] window The desired window.
 *  @param[in] key The desired [keyboard key](@ref keys).  `GLFW_KEY_UNKNOWN` is
 *  not a valid key for this function.
 *  @return One of `GLFW_PRESS` or `GLFW_RELEASE`.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_INVALID_ENUM.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref input_key
 *
 *  @since Added in version 1.0.
 *  @glfw3 Added window handle parameter.
 *
 *  @ingroup input
 */
GLFWAPI int glfwGetKey(GLFWwindow* window, int key);

/*! @brief Returns the last reported state of a mouse button for the specified
 *  window.
 *
 *  This function returns the last state reported for the specified mouse button
 *  to the specified window.  The returned state is one of `GLFW_PRESS` or
 *  `GLFW_RELEASE`.
 *
 *  If the @ref GLFW_STICKY_MOUSE_BUTTONS input mode is enabled, this function
 *  returns `GLFW_PRESS` the first time you call it for a mouse button that was
 *  pressed, even if that mouse button has already been released.
 *
 *  @param[in] window The desired window.
 *  @param[in] button The desired [mouse button](@ref buttons).
 *  @return One of `GLFW_PRESS` or `GLFW_RELEASE`.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_INVALID_ENUM.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref input_mouse_button
 *
 *  @since Added in version 1.0.
 *  @glfw3 Added window handle parameter.
 *
 *  @ingroup input
 */
GLFWAPI int glfwGetMouseButton(GLFWwindow* window, int button);

/*! @brief Retrieves the position of the cursor relative to the content area of
 *  the window.
 *
 *  This function returns the position of the cursor, in screen coordinates,
 *  relative to the upper-left corner of the content area of the specified
 *  window.
 *
 *  If the cursor is disabled (with `GLFW_CURSOR_DISABLED`) then the cursor
 *  position is unbounded and limited only by the minimum and maximum values of
 *  a `double`.
 *
 *  The coordinate can be converted to their integer equivalents with the
 *  `floor` function.  Casting directly to an integer type works for positive
 *  coordinates, but fails for negative ones.
 *
 *  Any or all of the position arguments may be `NULL`.  If an error occurs, all
 *  non-`NULL` position arguments will be set to zero.
 *
 *  @param[in] window The desired window.
 *  @param[out] xpos Where to store the cursor x-coordinate, relative to the
 *  left edge of the content area, or `NULL`.
 *  @param[out] ypos Where to store the cursor y-coordinate, relative to the to
 *  top edge of the content area, or `NULL`.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref cursor_pos
 *  @sa @ref glfwSetCursorPos
 *
 *  @since Added in version 3.0.  Replaces `glfwGetMousePos`.
 *
 *  @ingroup input
 */
GLFWAPI void glfwGetCursorPos(GLFWwindow* window, double* xpos, double* ypos);

/*! @brief Sets the position of the cursor, relative to the content area of the
 *  window.
 *
 *  This function sets the position, in screen coordinates, of the cursor
 *  relative to the upper-left corner of the content area of the specified
 *  window.  The window must have input focus.  If the window does not have
 *  input focus when this function is called, it fails silently.
 *
 *  __Do not use this function__ to implement things like camera controls.  GLFW
 *  already provides the `GLFW_CURSOR_DISABLED` cursor mode that hides the
 *  cursor, transparently re-centers it and provides unconstrained cursor
 *  motion.  See @ref glfwSetInputMode for more information.
 *
 *  If the cursor mode is `GLFW_CURSOR_DISABLED` then the cursor position is
 *  unconstrained and limited only by the minimum and maximum values of
 *  a `double`.
 *
 *  @param[in] window The desired window.
 *  @param[in] xpos The desired x-coordinate, relative to the left edge of the
 *  content area.
 *  @param[in] ypos The desired y-coordinate, relative to the top edge of the
 *  content area.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref
 *  GLFW_PLATFORM_ERROR and @ref GLFW_FEATURE_UNAVAILABLE (see remarks).
 *
 *  @remark @wayland This function will only work when the cursor mode is
 *  `GLFW_CURSOR_DISABLED`, otherwise it will emit @ref GLFW_FEATURE_UNAVAILABLE.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref cursor_pos
 *  @sa @ref glfwGetCursorPos
 *
 *  @since Added in version 3.0.  Replaces `glfwSetMousePos`.
 *
 *  @ingroup input
 */
GLFWAPI void glfwSetCursorPos(GLFWwindow* window, double xpos, double ypos);

/*! @brief Creates a custom cursor.
 *
 *  Creates a new custom cursor image that can be set for a window with @ref
 *  glfwSetCursor.  The cursor can be destroyed with @ref glfwDestroyCursor.
 *  Any remaining cursors are destroyed by @ref glfwTerminate.
 *
 *  The pixels are 32-bit, little-endian, non-premultiplied RGBA, i.e. eight
 *  bits per channel with the red channel first.  They are arranged canonically
 *  as packed sequential rows, starting from the top-left corner.
 *
 *  The cursor hotspot is specified in pixels, relative to the upper-left corner
 *  of the cursor image.  Like all other coordinate systems in GLFW, the X-axis
 *  points to the right and the Y-axis points down.
 *
 *  @param[in] image The desired cursor image.
 *  @param[in] xhot The desired x-coordinate, in pixels, of the cursor hotspot.
 *  @param[in] yhot The desired y-coordinate, in pixels, of the cursor hotspot.
 *  @return The handle of the created cursor, or `NULL` if an
 *  [error](@ref error_handling) occurred.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref
 *  GLFW_INVALID_VALUE and @ref GLFW_PLATFORM_ERROR.
 *
 *  @pointer_lifetime The specified image data is copied before this function
 *  returns.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref cursor_object
 *  @sa @ref glfwDestroyCursor
 *  @sa @ref glfwCreateStandardCursor
 *
 *  @since Added in version 3.1.
 *
 *  @ingroup input
 */
GLFWAPI GLFWcursor* glfwCreateCursor(const GLFWimage* image, int xhot, int yhot);

/*! @brief Creates a cursor with a standard shape.
 *
 *  Returns a cursor with a standard shape, that can be set for a window with
 *  @ref glfwSetCursor.  The images for these cursors come from the system
 *  cursor theme and their exact appearance will vary between platforms.
 *
 *  Most of these shapes are guaranteed to exist on every supported platform but
 *  a few may not be present.  See the table below for details.
 *
 *  Cursor shape                   | Windows | macOS | X11    | Wayland
 *  ------------------------------ | ------- | ----- | ------ | -------
 *  @ref GLFW_ARROW_CURSOR         | Yes     | Yes   | Yes    | Yes
 *  @ref GLFW_IBEAM_CURSOR         | Yes     | Yes   | Yes    | Yes
 *  @ref GLFW_CROSSHAIR_CURSOR     | Yes     | Yes   | Yes    | Yes
 *  @ref GLFW_POINTING_HAND_CURSOR | Yes     | Yes   | Yes    | Yes
 *  @ref GLFW_RESIZE_EW_CURSOR     | Yes     | Yes   | Yes    | Yes
 *  @ref GLFW_RESIZE_NS_CURSOR     | Yes     | Yes   | Yes    | Yes
 *  @ref GLFW_RESIZE_NWSE_CURSOR   | Yes     | Yes<sup>1</sup> | Maybe<sup>2</sup> | Maybe<sup>2</sup>
 *  @ref GLFW_RESIZE_NESW_CURSOR   | Yes     | Yes<sup>1</sup> | Maybe<sup>2</sup> | Maybe<sup>2</sup>
 *  @ref GLFW_RESIZE_ALL_CURSOR    | Yes     | Yes   | Yes    | Yes
 *  @ref GLFW_NOT_ALLOWED_CURSOR   | Yes     | Yes   | Maybe<sup>2</sup> | Maybe<sup>2</sup>
 *
 *  1) This uses a private system API and may fail in the future.
 *
 *  2) This uses a newer standard that not all cursor themes support.
 *
 *  If the requested shape is not available, this function emits a @ref
 *  GLFW_CURSOR_UNAVAILABLE error and returns `NULL`.
 *
 *  @param[in] shape One of the [standard shapes](@ref shapes).
 *  @return A new cursor ready to use or `NULL` if an
 *  [error](@ref error_handling) occurred.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref
 *  GLFW_INVALID_ENUM, @ref GLFW_CURSOR_UNAVAILABLE and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref cursor_standard
 *  @sa @ref glfwCreateCursor
 *
 *  @since Added in version 3.1.
 *
 *  @ingroup input
 */
GLFWAPI GLFWcursor* glfwCreateStandardCursor(int shape);

/*! @brief Destroys a cursor.
 *
 *  This function destroys a cursor previously created with @ref
 *  glfwCreateCursor.  Any remaining cursors will be destroyed by @ref
 *  glfwTerminate.
 *
 *  If the specified cursor is current for any window, that window will be
 *  reverted to the default cursor.  This does not affect the cursor mode.
 *
 *  @param[in] cursor The cursor object to destroy.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @reentrancy This function must not be called from a callback.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref cursor_object
 *  @sa @ref glfwCreateCursor
 *
 *  @since Added in version 3.1.
 *
 *  @ingroup input
 */
GLFWAPI void glfwDestroyCursor(GLFWcursor* cursor);

/*! @brief Sets the cursor for the window.
 *
 *  This function sets the cursor image to be used when the cursor is over the
 *  content area of the specified window.  The set cursor will only be visible
 *  when the [cursor mode](@ref cursor_mode) of the window is
 *  `GLFW_CURSOR_NORMAL`.
 *
 *  On some platforms, the set cursor may not be visible unless the window also
 *  has input focus.
 *
 *  @param[in] window The window to set the cursor for.
 *  @param[in] cursor The cursor to set, or `NULL` to switch back to the default
 *  arrow cursor.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref cursor_object
 *
 *  @since Added in version 3.1.
 *
 *  @ingroup input
 */
GLFWAPI void glfwSetCursor(GLFWwindow* window, GLFWcursor* cursor);

/*! @brief Sets the key callback.
 *
 *  This function sets the key callback of the specified window, which is called
 *  when a key is pressed, repeated or released.
 *
 *  The key functions deal with physical keys, with layout independent
 *  [key tokens](@ref keys) named after their values in the standard US keyboard
 *  layout.  If you want to input text, use the
 *  [character callback](@ref glfwSetCharCallback) instead.
 *
 *  When a window loses input focus, it will generate synthetic key release
 *  events for all pressed keys with associated key tokens.  You can tell these
 *  events from user-generated events by the fact that the synthetic ones are
 *  generated after the focus loss event has been processed, i.e. after the
 *  [window focus callback](@ref glfwSetWindowFocusCallback) has been called.
 *
 *  The scancode of a key is specific to that platform or sometimes even to that
 *  machine.  Scancodes are intended to allow users to bind keys that don't have
 *  a GLFW key token.  Such keys have `key` set to `GLFW_KEY_UNKNOWN`, their
 *  state is not saved and so it cannot be queried with @ref glfwGetKey.
 *
 *  Sometimes GLFW needs to generate synthetic key events, in which case the
 *  scancode may be zero.
 *
 *  @param[in] window The window whose callback to set.
 *  @param[in] callback The new key callback, or `NULL` to remove the currently
 *  set callback.
 *  @return The previously set callback, or `NULL` if no callback was set or the
 *  library had not been [initialized](@ref intro_init).
 *
 *  @callback_signature
 *  @code
 *  void function_name(GLFWwindow* window, int key, int scancode, int action, int mods)
 *  @endcode
 *  For more information about the callback parameters, see the
 *  [function pointer type](@ref GLFWkeyfun).
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref input_key
 *
 *  @since Added in version 1.0.
 *  @glfw3 Added window handle parameter and return value.
 *
 *  @ingroup input
 */
GLFWAPI GLFWkeyfun glfwSetKeyCallback(GLFWwindow* window, GLFWkeyfun callback);

/*! @brief Sets the Unicode character callback.
 *
 *  This function sets the character callback of the specified window, which is
 *  called when a Unicode character is input.
 *
 *  The character callback is intended for Unicode text input.  As it deals with
 *  characters, it is keyboard layout dependent, whereas the
 *  [key callback](@ref glfwSetKeyCallback) is not.  Characters do not map 1:1
 *  to physical keys, as a key may produce zero, one or more characters.  If you
 *  want to know whether a specific physical key was pressed or released, see
 *  the key callback instead.
 *
 *  The character callback behaves as system text input normally does and will
 *  not be called if modifier keys are held down that would prevent normal text
 *  input on that platform, for example a Super (Command) key on macOS or Alt key
 *  on Windows.
 *
 *  @param[in] window The window whose callback to set.
 *  @param[in] callback The new callback, or `NULL` to remove the currently set
 *  callback.
 *  @return The previously set callback, or `NULL` if no callback was set or the
 *  library had not been [initialized](@ref intro_init).
 *
 *  @callback_signature
 *  @code
 *  void function_name(GLFWwindow* window, unsigned int codepoint)
 *  @endcode
 *  For more information about the callback parameters, see the
 *  [function pointer type](@ref GLFWcharfun).
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref input_char
 *
 *  @since Added in version 2.4.
 *  @glfw3 Added window handle parameter and return value.
 *
 *  @ingroup input
 */
GLFWAPI GLFWcharfun glfwSetCharCallback(GLFWwindow* window, GLFWcharfun callback);

/*! @brief Sets the Unicode character with modifiers callback.
 *
 *  This function sets the character with modifiers callback of the specified
 *  window, which is called when a Unicode character is input regardless of what
 *  modifier keys are used.
 *
 *  The character with modifiers callback is intended for implementing custom
 *  Unicode character input.  For regular Unicode text input, see the
 *  [character callback](@ref glfwSetCharCallback).  Like the character
 *  callback, the character with modifiers callback deals with characters and is
 *  keyboard layout dependent.  Characters do not map 1:1 to physical keys, as
 *  a key may produce zero, one or more characters.  If you want to know whether
 *  a specific physical key was pressed or released, see the
 *  [key callback](@ref glfwSetKeyCallback) instead.
 *
 *  @param[in] window The window whose callback to set.
 *  @param[in] callback The new callback, or `NULL` to remove the currently set
 *  callback.
 *  @return The previously set callback, or `NULL` if no callback was set or an
 *  [error](@ref error_handling) occurred.
 *
 *  @callback_signature
 *  @code
 *  void function_name(GLFWwindow* window, unsigned int codepoint, int mods)
 *  @endcode
 *  For more information about the callback parameters, see the
 *  [function pointer type](@ref GLFWcharmodsfun).
 *
 *  @deprecated Scheduled for removal in version 4.0.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref input_char
 *
 *  @since Added in version 3.1.
 *
 *  @ingroup input
 */
GLFWAPI GLFWcharmodsfun glfwSetCharModsCallback(GLFWwindow* window, GLFWcharmodsfun callback);

/*! @brief Sets the mouse button callback.
 *
 *  This function sets the mouse button callback of the specified window, which
 *  is called when a mouse button is pressed or released.
 *
 *  When a window loses input focus, it will generate synthetic mouse button
 *  release events for all pressed mouse buttons.  You can tell these events
 *  from user-generated events by the fact that the synthetic ones are generated
 *  after the focus loss event has been processed, i.e. after the
 *  [window focus callback](@ref glfwSetWindowFocusCallback) has been called.
 *
 *  @param[in] window The window whose callback to set.
 *  @param[in] callback The new callback, or `NULL` to remove the currently set
 *  callback.
 *  @return The previously set callback, or `NULL` if no callback was set or the
 *  library had not been [initialized](@ref intro_init).
 *
 *  @callback_signature
 *  @code
 *  void function_name(GLFWwindow* window, int button, int action, int mods)
 *  @endcode
 *  For more information about the callback parameters, see the
 *  [function pointer type](@ref GLFWmousebuttonfun).
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref input_mouse_button
 *
 *  @since Added in version 1.0.
 *  @glfw3 Added window handle parameter and return value.
 *
 *  @ingroup input
 */
GLFWAPI GLFWmousebuttonfun glfwSetMouseButtonCallback(GLFWwindow* window, GLFWmousebuttonfun callback);

/*! @brief Sets the cursor position callback.
 *
 *  This function sets the cursor position callback of the specified window,
 *  which is called when the cursor is moved.  The callback is provided with the
 *  position, in screen coordinates, relative to the upper-left corner of the
 *  content area of the window.
 *
 *  @param[in] window The window whose callback to set.
 *  @param[in] callback The new callback, or `NULL` to remove the currently set
 *  callback.
 *  @return The previously set callback, or `NULL` if no callback was set or the
 *  library had not been [initialized](@ref intro_init).
 *
 *  @callback_signature
 *  @code
 *  void function_name(GLFWwindow* window, double xpos, double ypos);
 *  @endcode
 *  For more information about the callback parameters, see the
 *  [function pointer type](@ref GLFWcursorposfun).
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref cursor_pos
 *
 *  @since Added in version 3.0.  Replaces `glfwSetMousePosCallback`.
 *
 *  @ingroup input
 */
GLFWAPI GLFWcursorposfun glfwSetCursorPosCallback(GLFWwindow* window, GLFWcursorposfun callback);

/*! @brief Sets the cursor enter/leave callback.
 *
 *  This function sets the cursor boundary crossing callback of the specified
 *  window, which is called when the cursor enters or leaves the content area of
 *  the window.
 *
 *  @param[in] window The window whose callback to set.
 *  @param[in] callback The new callback, or `NULL` to remove the currently set
 *  callback.
 *  @return The previously set callback, or `NULL` if no callback was set or the
 *  library had not been [initialized](@ref intro_init).
 *
 *  @callback_signature
 *  @code
 *  void function_name(GLFWwindow* window, int entered)
 *  @endcode
 *  For more information about the callback parameters, see the
 *  [function pointer type](@ref GLFWcursorenterfun).
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref cursor_enter
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup input
 */
GLFWAPI GLFWcursorenterfun glfwSetCursorEnterCallback(GLFWwindow* window, GLFWcursorenterfun callback);

/*! @brief Sets the scroll callback.
 *
 *  This function sets the scroll callback of the specified window, which is
 *  called when a scrolling device is used, such as a mouse wheel or scrolling
 *  area of a touchpad.
 *
 *  The scroll callback receives all scrolling input, like that from a mouse
 *  wheel or a touchpad scrolling area.
 *
 *  @param[in] window The window whose callback to set.
 *  @param[in] callback The new scroll callback, or `NULL` to remove the
 *  currently set callback.
 *  @return The previously set callback, or `NULL` if no callback was set or the
 *  library had not been [initialized](@ref intro_init).
 *
 *  @callback_signature
 *  @code
 *  void function_name(GLFWwindow* window, double xoffset, double yoffset)
 *  @endcode
 *  For more information about the callback parameters, see the
 *  [function pointer type](@ref GLFWscrollfun).
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref scrolling
 *
 *  @since Added in version 3.0.  Replaces `glfwSetMouseWheelCallback`.
 *
 *  @ingroup input
 */
GLFWAPI GLFWscrollfun glfwSetScrollCallback(GLFWwindow* window, GLFWscrollfun callback);

/*! @brief Sets the path drop callback.
 *
 *  This function sets the path drop callback of the specified window, which is
 *  called when one or more dragged paths are dropped on the window.
 *
 *  Because the path array and its strings may have been generated specifically
 *  for that event, they are not guaranteed to be valid after the callback has
 *  returned.  If you wish to use them after the callback returns, you need to
 *  make a deep copy.
 *
 *  @param[in] window The window whose callback to set.
 *  @param[in] callback The new file drop callback, or `NULL` to remove the
 *  currently set callback.
 *  @return The previously set callback, or `NULL` if no callback was set or the
 *  library had not been [initialized](@ref intro_init).
 *
 *  @callback_signature
 *  @code
 *  void function_name(GLFWwindow* window, int path_count, const char* paths[])
 *  @endcode
 *  For more information about the callback parameters, see the
 *  [function pointer type](@ref GLFWdropfun).
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref path_drop
 *
 *  @since Added in version 3.1.
 *
 *  @ingroup input
 */
GLFWAPI GLFWdropfun glfwSetDropCallback(GLFWwindow* window, GLFWdropfun callback);

/*! @brief Sets the clipboard to the specified string.
 *
 *  This function sets the system clipboard to the specified, UTF-8 encoded
 *  string.
 *
 *  @param[in] window Deprecated.  Any valid window or `NULL`.
 *  @param[in] string A UTF-8 encoded string.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @remark @win32 The clipboard on Windows has a single global lock for reading and
 *  writing.  GLFW tries to acquire it a few times, which is almost always enough.  If it
 *  cannot acquire the lock then this function emits @ref GLFW_PLATFORM_ERROR and returns.
 *  It is safe to try this multiple times.
 *
 *  @pointer_lifetime The specified string is copied before this function
 *  returns.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref clipboard
 *  @sa @ref glfwGetClipboardString
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup input
 */
GLFWAPI void glfwSetClipboardString(GLFWwindow* window, const char* string);

/*! @brief Returns the contents of the clipboard as a string.
 *
 *  This function returns the contents of the system clipboard, if it contains
 *  or is convertible to a UTF-8 encoded string.  If the clipboard is empty or
 *  if its contents cannot be converted, `NULL` is returned and a @ref
 *  GLFW_FORMAT_UNAVAILABLE error is generated.
 *
 *  @param[in] window Deprecated.  Any valid window or `NULL`.
 *  @return The contents of the clipboard as a UTF-8 encoded string, or `NULL`
 *  if an [error](@ref error_handling) occurred.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref
 *  GLFW_FORMAT_UNAVAILABLE and @ref GLFW_PLATFORM_ERROR.
 *
 *  @remark @win32 The clipboard on Windows has a single global lock for reading and
 *  writing.  GLFW tries to acquire it a few times, which is almost always enough.  If it
 *  cannot acquire the lock then this function emits @ref GLFW_PLATFORM_ERROR and returns.
 *  It is safe to try this multiple times.
 *
 *  @pointer_lifetime The returned string is allocated and freed by GLFW.  You
 *  should not free it yourself.  It is valid until the next call to @ref
 *  glfwGetClipboardString or @ref glfwSetClipboardString, or until the library
 *  is terminated.
 *
 *  @thread_safety This function must only be called from the main thread.
 *
 *  @sa @ref clipboard
 *  @sa @ref glfwSetClipboardString
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup input
 */
GLFWAPI const char* glfwGetClipboardString(GLFWwindow* window);

/*! @brief Returns the GLFW time.
 *
 *  This function returns the current GLFW time, in seconds.  Unless the time
 *  has been set using @ref glfwSetTime it measures time elapsed since GLFW was
 *  initialized.
 *
 *  This function and @ref glfwSetTime are helper functions on top of @ref
 *  glfwGetTimerFrequency and @ref glfwGetTimerValue.
 *
 *  The resolution of the timer is system dependent, but is usually on the order
 *  of a few micro- or nanoseconds.  It uses the highest-resolution monotonic
 *  time source on each operating system.
 *
 *  @return The current time, in seconds, or zero if an
 *  [error](@ref error_handling) occurred.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function may be called from any thread.  Reading and
 *  writing of the internal base time is not atomic, so it needs to be
 *  externally synchronized with calls to @ref glfwSetTime.
 *
 *  @sa @ref time
 *
 *  @since Added in version 1.0.
 *
 *  @ingroup input
 */
GLFWAPI double glfwGetTime(void);

/*! @brief Sets the GLFW time.
 *
 *  This function sets the current GLFW time, in seconds.  The value must be
 *  a positive finite number less than or equal to 18446744073.0, which is
 *  approximately 584.5 years.
 *
 *  This function and @ref glfwGetTime are helper functions on top of @ref
 *  glfwGetTimerFrequency and @ref glfwGetTimerValue.
 *
 *  @param[in] time The new value, in seconds.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_INVALID_VALUE.
 *
 *  @remark The upper limit of GLFW time is calculated as
 *  floor((2<sup>64</sup> - 1) / 10<sup>9</sup>) and is due to implementations
 *  storing nanoseconds in 64 bits.  The limit may be increased in the future.
 *
 *  @thread_safety This function may be called from any thread.  Reading and
 *  writing of the internal base time is not atomic, so it needs to be
 *  externally synchronized with calls to @ref glfwGetTime.
 *
 *  @sa @ref time
 *
 *  @since Added in version 2.2.
 *
 *  @ingroup input
 */
GLFWAPI void glfwSetTime(double time);

/*! @brief Returns the current value of the raw timer.
 *
 *  This function returns the current value of the raw timer, measured in
 *  1&nbsp;/&nbsp;frequency seconds.  To get the frequency, call @ref
 *  glfwGetTimerFrequency.
 *
 *  @return The value of the timer, or zero if an
 *  [error](@ref error_handling) occurred.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function may be called from any thread.
 *
 *  @sa @ref time
 *  @sa @ref glfwGetTimerFrequency
 *
 *  @since Added in version 3.2.
 *
 *  @ingroup input
 */
GLFWAPI uint64_t glfwGetTimerValue(void);

/*! @brief Returns the frequency, in Hz, of the raw timer.
 *
 *  This function returns the frequency, in Hz, of the raw timer.
 *
 *  @return The frequency of the timer, in Hz, or zero if an
 *  [error](@ref error_handling) occurred.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function may be called from any thread.
 *
 *  @sa @ref time
 *  @sa @ref glfwGetTimerValue
 *
 *  @since Added in version 3.2.
 *
 *  @ingroup input
 */
GLFWAPI uint64_t glfwGetTimerFrequency(void);

/*! @brief Makes the context of the specified window current for the calling
 *  thread.
 *
 *  This function makes the OpenGL or OpenGL ES context of the specified window
 *  current on the calling thread.  It can also detach the current context from
 *  the calling thread without making a new one current by passing in `NULL`.
 *
 *  A context must only be made current on a single thread at a time and each
 *  thread can have only a single current context at a time.  Making a context
 *  current detaches any previously current context on the calling thread.
 *
 *  When moving a context between threads, you must detach it (make it
 *  non-current) on the old thread before making it current on the new one.
 *
 *  By default, making a context non-current implicitly forces a pipeline flush.
 *  On machines that support `GL_KHR_context_flush_control`, you can control
 *  whether a context performs this flush by setting the
 *  [GLFW_CONTEXT_RELEASE_BEHAVIOR](@ref GLFW_CONTEXT_RELEASE_BEHAVIOR_hint)
 *  hint.
 *
 *  The specified window must have an OpenGL or OpenGL ES context.  Specifying
 *  a window without a context will generate a @ref GLFW_NO_WINDOW_CONTEXT
 *  error.
 *
 *  @param[in] window The window whose context to make current, or `NULL` to
 *  detach the current context.
 *
 *  @remarks If the previously current context was created via a different
 *  context creation API than the one passed to this function, GLFW will still
 *  detach the previous one from its API before making the new one current.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref
 *  GLFW_NO_WINDOW_CONTEXT and @ref GLFW_PLATFORM_ERROR.
 *
 *  @thread_safety This function may be called from any thread.
 *
 *  @sa @ref context_current
 *  @sa @ref glfwGetCurrentContext
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup context
 */
GLFWAPI void glfwMakeContextCurrent(GLFWwindow* window);

/*! @brief Returns the window whose context is current on the calling thread.
 *
 *  This function returns the window whose OpenGL or OpenGL ES context is
 *  current on the calling thread.
 *
 *  @return The window whose context is current, or `NULL` if no window's
 *  context is current.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function may be called from any thread.
 *
 *  @sa @ref context_current
 *  @sa @ref glfwMakeContextCurrent
 *
 *  @since Added in version 3.0.
 *
 *  @ingroup context
 */
GLFWAPI GLFWwindow* glfwGetCurrentContext(void);

/*! @brief Swaps the front and back buffers of the specified window.
 *
 *  This function swaps the front and back buffers of the specified window when
 *  rendering with OpenGL or OpenGL ES.  If the swap interval is greater than
 *  zero, the GPU driver waits the specified number of screen updates before
 *  swapping the buffers.
 *
 *  The specified window must have an OpenGL or OpenGL ES context.  Specifying
 *  a window without a context will generate a @ref GLFW_NO_WINDOW_CONTEXT
 *  error.
 *
 *  This function does not apply to Vulkan.  If you are rendering with Vulkan,
 *  see `vkQueuePresentKHR` instead.
 *
 *  @param[in] window The window whose buffers to swap.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref
 *  GLFW_NO_WINDOW_CONTEXT and @ref GLFW_PLATFORM_ERROR.
 *
 *  @remark __EGL:__ The context of the specified window must be current on the
 *  calling thread.
 *
 *  @thread_safety This function may be called from any thread.
 *
 *  @sa @ref buffer_swap
 *  @sa @ref glfwSwapInterval
 *
 *  @since Added in version 1.0.
 *  @glfw3 Added window handle parameter.
 *
 *  @ingroup window
 */
GLFWAPI void glfwSwapBuffers(GLFWwindow* window);

/*! @brief Sets the swap interval for the current context.
 *
 *  This function sets the swap interval for the current OpenGL or OpenGL ES
 *  context, i.e. the number of screen updates to wait from the time @ref
 *  glfwSwapBuffers was called before swapping the buffers and returning.  This
 *  is sometimes called _vertical synchronization_, _vertical retrace
 *  synchronization_ or just _vsync_.
 *
 *  A context that supports either of the `WGL_EXT_swap_control_tear` and
 *  `GLX_EXT_swap_control_tear` extensions also accepts _negative_ swap
 *  intervals, which allows the driver to swap immediately even if a frame
 *  arrives a little bit late.  You can check for these extensions with @ref
 *  glfwExtensionSupported.
 *
 *  A context must be current on the calling thread.  Calling this function
 *  without a current context will cause a @ref GLFW_NO_CURRENT_CONTEXT error.
 *
 *  This function does not apply to Vulkan.  If you are rendering with Vulkan,
 *  see the present mode of your swapchain instead.
 *
 *  @param[in] interval The minimum number of screen updates to wait for
 *  until the buffers are swapped by @ref glfwSwapBuffers.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref
 *  GLFW_NO_CURRENT_CONTEXT and @ref GLFW_PLATFORM_ERROR.
 *
 *  @remark This function is not called during context creation, leaving the
 *  swap interval set to whatever is the default for that API.  This is done
 *  because some swap interval extensions used by GLFW do not allow the swap
 *  interval to be reset to zero once it has been set to a non-zero value.
 *
 *  @remark Some GPU drivers do not honor the requested swap interval, either
 *  because of a user setting that overrides the application's request or due to
 *  bugs in the driver.
 *
 *  @thread_safety This function may be called from any thread.
 *
 *  @sa @ref buffer_swap
 *  @sa @ref glfwSwapBuffers
 *
 *  @since Added in version 1.0.
 *
 *  @ingroup context
 */
GLFWAPI void glfwSwapInterval(int interval);

/*! @brief Returns whether the specified extension is available.
 *
 *  This function returns whether the specified
 *  [API extension](@ref context_glext) is supported by the current OpenGL or
 *  OpenGL ES context.  It searches both for client API extension and context
 *  creation API extensions.
 *
 *  A context must be current on the calling thread.  Calling this function
 *  without a current context will cause a @ref GLFW_NO_CURRENT_CONTEXT error.
 *
 *  As this functions retrieves and searches one or more extension strings each
 *  call, it is recommended that you cache its results if it is going to be used
 *  frequently.  The extension strings will not change during the lifetime of
 *  a context, so there is no danger in doing this.
 *
 *  This function does not apply to Vulkan.  If you are using Vulkan, see @ref
 *  glfwGetRequiredInstanceExtensions, `vkEnumerateInstanceExtensionProperties`
 *  and `vkEnumerateDeviceExtensionProperties` instead.
 *
 *  @param[in] extension The ASCII encoded name of the extension.
 *  @return `GLFW_TRUE` if the extension is available, or `GLFW_FALSE`
 *  otherwise.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref
 *  GLFW_NO_CURRENT_CONTEXT, @ref GLFW_INVALID_VALUE and @ref
 *  GLFW_PLATFORM_ERROR.
 *
 *  @thread_safety This function may be called from any thread.
 *
 *  @sa @ref context_glext
 *  @sa @ref glfwGetProcAddress
 *
 *  @since Added in version 1.0.
 *
 *  @ingroup context
 */
GLFWAPI int glfwExtensionSupported(const char* extension);

/*! @brief Returns the address of the specified function for the current
 *  context.
 *
 *  This function returns the address of the specified OpenGL or OpenGL ES
 *  [core or extension function](@ref context_glext), if it is supported
 *  by the current context.
 *
 *  A context must be current on the calling thread.  Calling this function
 *  without a current context will cause a @ref GLFW_NO_CURRENT_CONTEXT error.
 *
 *  This function does not apply to Vulkan.  If you are rendering with Vulkan,
 *  see @ref glfwGetInstanceProcAddress, `vkGetInstanceProcAddr` and
 *  `vkGetDeviceProcAddr` instead.
 *
 *  @param[in] procname The ASCII encoded name of the function.
 *  @return The address of the function, or `NULL` if an
 *  [error](@ref error_handling) occurred.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref
 *  GLFW_NO_CURRENT_CONTEXT and @ref GLFW_PLATFORM_ERROR.
 *
 *  @remark The address of a given function is not guaranteed to be the same
 *  between contexts.
 *
 *  @remark This function may return a non-`NULL` address despite the
 *  associated version or extension not being available.  Always check the
 *  context version or extension string first.
 *
 *  @pointer_lifetime The returned function pointer is valid until the context
 *  is destroyed or the library is terminated.
 *
 *  @thread_safety This function may be called from any thread.
 *
 *  @sa @ref context_glext
 *  @sa @ref glfwExtensionSupported
 *
 *  @since Added in version 1.0.
 *
 *  @ingroup context
 */
GLFWAPI GLFWglproc glfwGetProcAddress(const char* procname);

/*! @brief Returns whether the Vulkan loader and an ICD have been found.
 *
 *  This function returns whether the Vulkan loader and any minimally functional
 *  ICD have been found.
 *
 *  The availability of a Vulkan loader and even an ICD does not by itself guarantee that
 *  surface creation or even instance creation is possible.  Call @ref
 *  glfwGetRequiredInstanceExtensions to check whether the extensions necessary for Vulkan
 *  surface creation are available and @ref glfwGetPhysicalDevicePresentationSupport to
 *  check whether a queue family of a physical device supports image presentation.
 *
 *  @return `GLFW_TRUE` if Vulkan is minimally available, or `GLFW_FALSE`
 *  otherwise.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED.
 *
 *  @thread_safety This function may be called from any thread.
 *
 *  @sa @ref vulkan_support
 *
 *  @since Added in version 3.2.
 *
 *  @ingroup vulkan
 */
GLFWAPI int glfwVulkanSupported(void);

/*! @brief Returns the Vulkan instance extensions required by GLFW.
 *
 *  This function returns an array of names of Vulkan instance extensions required
 *  by GLFW for creating Vulkan surfaces for GLFW windows.  If successful, the
 *  list will always contain `VK_KHR_surface`, so if you don't require any
 *  additional extensions you can pass this list directly to the
 *  `VkInstanceCreateInfo` struct.
 *
 *  If Vulkan is not available on the machine, this function returns `NULL` and
 *  generates a @ref GLFW_API_UNAVAILABLE error.  Call @ref glfwVulkanSupported
 *  to check whether Vulkan is at least minimally available.
 *
 *  If Vulkan is available but no set of extensions allowing window surface
 *  creation was found, this function returns `NULL`.  You may still use Vulkan
 *  for off-screen rendering and compute work.
 *
 *  @param[out] count Where to store the number of extensions in the returned
 *  array.  This is set to zero if an error occurred.
 *  @return An array of ASCII encoded extension names, or `NULL` if an
 *  [error](@ref error_handling) occurred.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_API_UNAVAILABLE.
 *
 *  @remark Additional extensions may be required by future versions of GLFW.
 *  You should check if any extensions you wish to enable are already in the
 *  returned array, as it is an error to specify an extension more than once in
 *  the `VkInstanceCreateInfo` struct.
 *
 *  @pointer_lifetime The returned array is allocated and freed by GLFW.  You
 *  should not free it yourself.  It is guaranteed to be valid only until the
 *  library is terminated.
 *
 *  @thread_safety This function may be called from any thread.
 *
 *  @sa @ref vulkan_ext
 *  @sa @ref glfwCreateWindowSurface
 *
 *  @since Added in version 3.2.
 *
 *  @ingroup vulkan
 */
GLFWAPI const char** glfwGetRequiredInstanceExtensions(uint32_t* count);

#if defined(VK_VERSION_1_0)

/*! @brief Returns the address of the specified Vulkan instance function.
 *
 *  This function returns the address of the specified Vulkan core or extension
 *  function for the specified instance.  If instance is set to `NULL` it can
 *  return any function exported from the Vulkan loader, including at least the
 *  following functions:
 *
 *  - `vkEnumerateInstanceExtensionProperties`
 *  - `vkEnumerateInstanceLayerProperties`
 *  - `vkCreateInstance`
 *  - `vkGetInstanceProcAddr`
 *
 *  If Vulkan is not available on the machine, this function returns `NULL` and
 *  generates a @ref GLFW_API_UNAVAILABLE error.  Call @ref glfwVulkanSupported
 *  to check whether Vulkan is at least minimally available.
 *
 *  This function is equivalent to calling `vkGetInstanceProcAddr` with
 *  a platform-specific query of the Vulkan loader as a fallback.
 *
 *  @param[in] instance The Vulkan instance to query, or `NULL` to retrieve
 *  functions related to instance creation.
 *  @param[in] procname The ASCII encoded name of the function.
 *  @return The address of the function, or `NULL` if an
 *  [error](@ref error_handling) occurred.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED and @ref
 *  GLFW_API_UNAVAILABLE.
 *
 *  @pointer_lifetime The returned function pointer is valid until the library
 *  is terminated.
 *
 *  @thread_safety This function may be called from any thread.
 *
 *  @sa @ref vulkan_proc
 *
 *  @since Added in version 3.2.
 *
 *  @ingroup vulkan
 */
GLFWAPI GLFWvkproc glfwGetInstanceProcAddress(VkInstance instance, const char* procname);

/*! @brief Returns whether the specified queue family can present images.
 *
 *  This function returns whether the specified queue family of the specified
 *  physical device supports presentation to the platform GLFW was built for.
 *
 *  If Vulkan or the required window surface creation instance extensions are
 *  not available on the machine, or if the specified instance was not created
 *  with the required extensions, this function returns `GLFW_FALSE` and
 *  generates a @ref GLFW_API_UNAVAILABLE error.  Call @ref glfwVulkanSupported
 *  to check whether Vulkan is at least minimally available and @ref
 *  glfwGetRequiredInstanceExtensions to check what instance extensions are
 *  required.
 *
 *  @param[in] instance The instance that the physical device belongs to.
 *  @param[in] device The physical device that the queue family belongs to.
 *  @param[in] queuefamily The index of the queue family to query.
 *  @return `GLFW_TRUE` if the queue family supports presentation, or
 *  `GLFW_FALSE` otherwise.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref
 *  GLFW_API_UNAVAILABLE and @ref GLFW_PLATFORM_ERROR.
 *
 *  @remark @macos This function currently always returns `GLFW_TRUE`, as the
 *  `VK_MVK_macos_surface` and `VK_EXT_metal_surface` extensions do not provide
 *  a `vkGetPhysicalDevice*PresentationSupport` type function.
 *
 *  @thread_safety This function may be called from any thread.  For
 *  synchronization details of Vulkan objects, see the Vulkan specification.
 *
 *  @sa @ref vulkan_present
 *
 *  @since Added in version 3.2.
 *
 *  @ingroup vulkan
 */
GLFWAPI int glfwGetPhysicalDevicePresentationSupport(VkInstance instance, VkPhysicalDevice device, uint32_t queuefamily);

/*! @brief Creates a Vulkan surface for the specified window.
 *
 *  This function creates a Vulkan surface for the specified window.
 *
 *  If the Vulkan loader or at least one minimally functional ICD were not found,
 *  this function returns `VK_ERROR_INITIALIZATION_FAILED` and generates a @ref
 *  GLFW_API_UNAVAILABLE error.  Call @ref glfwVulkanSupported to check whether
 *  Vulkan is at least minimally available.
 *
 *  If the required window surface creation instance extensions are not
 *  available or if the specified instance was not created with these extensions
 *  enabled, this function returns `VK_ERROR_EXTENSION_NOT_PRESENT` and
 *  generates a @ref GLFW_API_UNAVAILABLE error.  Call @ref
 *  glfwGetRequiredInstanceExtensions to check what instance extensions are
 *  required.
 *
 *  The window surface cannot be shared with another API so the window must
 *  have been created with the [client api hint](@ref GLFW_CLIENT_API_attrib)
 *  set to `GLFW_NO_API` otherwise it generates a @ref GLFW_INVALID_VALUE error
 *  and returns `VK_ERROR_NATIVE_WINDOW_IN_USE_KHR`.
 *
 *  The window surface must be destroyed before the specified Vulkan instance.
 *  It is the responsibility of the caller to destroy the window surface.  GLFW
 *  does not destroy it for you.  Call `vkDestroySurfaceKHR` to destroy the
 *  surface.
 *
 *  @param[in] instance The Vulkan instance to create the surface in.
 *  @param[in] window The window to create the surface for.
 *  @param[in] allocator The allocator to use, or `NULL` to use the default
 *  allocator.
 *  @param[out] surface Where to store the handle of the surface.  This is set
 *  to `VK_NULL_HANDLE` if an error occurred.
 *  @return `VK_SUCCESS` if successful, or a Vulkan error code if an
 *  [error](@ref error_handling) occurred.
 *
 *  @errors Possible errors include @ref GLFW_NOT_INITIALIZED, @ref
 *  GLFW_API_UNAVAILABLE, @ref GLFW_PLATFORM_ERROR and @ref GLFW_INVALID_VALUE
 *
 *  @remark If an error occurs before the creation call is made, GLFW returns
 *  the Vulkan error code most appropriate for the error.  Appropriate use of
 *  @ref glfwVulkanSupported and @ref glfwGetRequiredInstanceExtensions should
 *  eliminate almost all occurrences of these errors.
 *
 *  @remark @macos GLFW prefers the `VK_EXT_metal_surface` extension, with the
 *  `VK_MVK_macos_surface` extension as a fallback.  The name of the selected
 *  extension, if any, is included in the array returned by @ref
 *  glfwGetRequiredInstanceExtensions.
 *
 *  @remark @macos This function creates and sets a `CAMetalLayer` instance for
 *  the window content view, which is required for MoltenVK to function.
 *
 *  @remark @x11 By default GLFW prefers the `VK_KHR_xcb_surface` extension,
 *  with the `VK_KHR_xlib_surface` extension as a fallback.  You can make
 *  `VK_KHR_xlib_surface` the preferred extension by setting the
 *  [GLFW_X11_XCB_VULKAN_SURFACE](@ref GLFW_X11_XCB_VULKAN_SURFACE_hint) init
 *  hint.  The name of the selected extension, if any, is included in the array
 *  returned by @ref glfwGetRequiredInstanceExtensions.
 *
 *  @thread_safety This function may be called from any thread.  For
 *  synchronization details of Vulkan objects, see the Vulkan specification.
 *
 *  @sa @ref vulkan_surface
 *  @sa @ref glfwGetRequiredInstanceExtensions
 *
 *  @since Added in version 3.2.
 *
 *  @ingroup vulkan
 */
GLFWAPI VkResult glfwCreateWindowSurface(VkInstance instance, GLFWwindow* window, const VkAllocationCallbacks* allocator, VkSurfaceKHR* surface);

#endif /*VK_VERSION_1_0*/


/*************************************************************************
 * Global definition cleanup
 *************************************************************************/

/* ------------------- BEGIN SYSTEM/COMPILER SPECIFIC -------------------- */

#ifdef GLFW_WINGDIAPI_DEFINED
 #undef WINGDIAPI
 #undef GLFW_WINGDIAPI_DEFINED
#endif

#ifdef GLFW_CALLBACK_DEFINED
 #undef CALLBACK
 #undef GLFW_CALLBACK_DEFINED
#endif

/* Some OpenGL related headers need GLAPIENTRY, but it is unconditionally
 * defined by some gl.h variants (OpenBSD) so define it after if needed.
 */
#ifndef GLAPIENTRY
 #define GLAPIENTRY APIENTRY
 #define GLFW_GLAPIENTRY_DEFINED
#endif

/* -------------------- END SYSTEM/COMPILER SPECIFIC --------------------- */


#ifdef __cplusplus
}
#endif

#endif /* _glfw3_h_ */



#if defined(_WIN32) || defined(__CYGWIN__)
#  define _GLFW_WIN32
#  pragma comment(lib, "user32.lib")
#  pragma comment(lib, "Gdi32.lib")
#  pragma comment(lib, "Shell32.lib")
#endif

#if defined(__linux__)
#  if !defined(BR_NO_X11)
#    define _GLFW_X11
#  endif
#  if !defined(BR_NO_WAYLAND)
#    define _GLFW_WAYLAND
#  endif
#  if defined(HEADLESS)
#    define _GLFW_NULL
#  endif
#endif
#if defined(__FreeBSD__) || defined(__OpenBSD__) || defined(__NetBSD__) || defined(__DragonFly__)
#  if !defined(BR_NO_X11)
#    define _GLFW_X11
#  endif
#  if defined(HEADLESS)
#    define _GLFW_NULL
#  endif
#  define BR_NO_WAYLAND
#endif
#if defined(__APPLE__)
#  define _GLFW_COCOA
#endif
#if defined(__TINYC__)
#  define _WIN32_WINNT_WINXP      0x0501
#endif


#if defined(_GLFW_X11) || defined(_GLFW_WAYLAND)
/* #  include "external/glfw/src/xkb_unicode.h" */
//========================================================================
// GLFW 3.4 Linux - www.glfw.org
//------------------------------------------------------------------------
// Copyright (c) 2014 Jonas Ådahl <jadahl@gmail.com>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any damages
// arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it
// freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you must not
//    claim that you wrote the original software. If you use this software
//    in a product, an acknowledgment in the product documentation would
//    be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such, and must not
//    be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source
//    distribution.
//
//========================================================================

#define GLFW_INVALID_CODEPOINT 0xffffffffu

uint32_t _glfwKeySym2Unicode(unsigned int keysym);


/* #  include "external/glfw/src/posix_poll.h" */
//========================================================================
// GLFW 3.4 POSIX - www.glfw.org
//------------------------------------------------------------------------
// Copyright (c) 2022 Camilla Löwy <elmindreda@glfw.org>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any damages
// arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it
// freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you must not
//    claim that you wrote the original software. If you use this software
//    in a product, an acknowledgment in the product documentation would
//    be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such, and must not
//    be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source
//    distribution.
//
//========================================================================

#include <poll.h>

int _glfwPollPOSIX(struct pollfd* fds, nfds_t count, double* timeout);


#endif

#if defined(_GLFW_X11)
/* #  include "external/X11/cursorfont.h" */
/*

Copyright 1987, 1998  The Open Group

Permission to use, copy, modify, distribute, and sell this software and its
documentation for any purpose is hereby granted without fee, provided that
the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation.

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE OPEN GROUP BE LIABLE FOR ANY CLAIM, DAMAGES OR
OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of The Open Group shall
not be used in advertising or otherwise to promote the sale, use or
other dealings in this Software without prior written authorization
from The Open Group.

*/

#ifndef _X11_CURSORFONT_H_
#define _X11_CURSORFONT_H_

#define XC_num_glyphs 154
#define XC_X_cursor 0
#define XC_arrow 2
#define XC_based_arrow_down 4
#define XC_based_arrow_up 6
#define XC_boat 8
#define XC_bogosity 10
#define XC_bottom_left_corner 12
#define XC_bottom_right_corner 14
#define XC_bottom_side 16
#define XC_bottom_tee 18
#define XC_box_spiral 20
#define XC_center_ptr 22
#define XC_circle 24
#define XC_clock 26
#define XC_coffee_mug 28
#define XC_cross 30
#define XC_cross_reverse 32
#define XC_crosshair 34
#define XC_diamond_cross 36
#define XC_dot 38
#define XC_dotbox 40
#define XC_double_arrow 42
#define XC_draft_large 44
#define XC_draft_small 46
#define XC_draped_box 48
#define XC_exchange 50
#define XC_fleur 52
#define XC_gobbler 54
#define XC_gumby 56
#define XC_hand1 58
#define XC_hand2 60
#define XC_heart 62
#define XC_icon 64
#define XC_iron_cross 66
#define XC_left_ptr 68
#define XC_left_side 70
#define XC_left_tee 72
#define XC_leftbutton 74
#define XC_ll_angle 76
#define XC_lr_angle 78
#define XC_man 80
#define XC_middlebutton 82
#define XC_mouse 84
#define XC_pencil 86
#define XC_pirate 88
#define XC_plus 90
#define XC_question_arrow 92
#define XC_right_ptr 94
#define XC_right_side 96
#define XC_right_tee 98
#define XC_rightbutton 100
#define XC_rtl_logo 102
#define XC_sailboat 104
#define XC_sb_down_arrow 106
#define XC_sb_h_double_arrow 108
#define XC_sb_left_arrow 110
#define XC_sb_right_arrow 112
#define XC_sb_up_arrow 114
#define XC_sb_v_double_arrow 116
#define XC_shuttle 118
#define XC_sizing 120
#define XC_spider 122
#define XC_spraycan 124
#define XC_star 126
#define XC_target 128
#define XC_tcross 130
#define XC_top_left_arrow 132
#define XC_top_left_corner 134
#define XC_top_right_corner 136
#define XC_top_side 138
#define XC_top_tee 140
#define XC_trek 142
#define XC_ul_angle 144
#define XC_umbrella 146
#define XC_ur_angle 148
#define XC_watch 150
#define XC_xterm 152

#endif /* _X11_CURSORFONT_H_ */
/* #  include "external/X11/Xmd.h" */
/***********************************************************

Copyright 1987, 1998  The Open Group

Permission to use, copy, modify, distribute, and sell this software and its
documentation for any purpose is hereby granted without fee, provided that
the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation.

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
OPEN GROUP BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of The Open Group shall not be
used in advertising or otherwise to promote the sale, use or other dealings
in this Software without prior written authorization from The Open Group.


Copyright 1987 by Digital Equipment Corporation, Maynard, Massachusetts.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of Digital not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

DIGITAL DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
DIGITAL BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

******************************************************************/
#ifndef XMD_H
# define XMD_H 1
/*
 *  Xmd.h: MACHINE DEPENDENT DECLARATIONS.
 */

/*
 * Special per-machine configuration flags.
 */
# if defined(__sun) && defined(__SVR4)
#  include <sys/isa_defs.h> /* Solaris: defines _LP64 if necessary */
# endif

#if defined(__SIZEOF_LONG__)
# if __SIZEOF_LONG__ == 8
#  define LONG64				/* 32/64-bit architecture */
# endif
# elif defined (_LP64) || defined(__LP64__) || \
     defined(__alpha) || defined(__alpha__) || \
     defined(__ia64__) || defined(ia64) || \
     defined(__sparc64__) || \
     defined(__s390x__) || \
     defined(__amd64__) || defined(amd64) || \
     defined(__powerpc64__)
#  if !defined(__ILP32__) /* amd64-x32 is 32bit */
#   define LONG64				/* 32/64-bit architecture */
#  endif /* !__ILP32__ */
# endif

/*
 * Definition of macro used to set constants for size of network structures;
 * machines with preprocessors that can't handle all of the sz_ symbols
 * can define this macro to be sizeof(x) if and only if their compiler doesn't
 * pad out structures (esp. the xTextElt structure which contains only two
 * one-byte fields).  Network structures should always define sz_symbols.
 *
 * The sz_ prefix is used instead of something more descriptive so that the
 * symbols are no more than 32 characters long (which causes problems for some
 * compilers and preprocessors).
 *
 * The extra indirection is to get macro arguments to expand correctly before
 * the concatenation, rather than afterward.
 */
# define _SIZEOF(x) sz_##x
# define SIZEOF(x) _SIZEOF(x)

/*
 * Bitfield suffixes for the protocol structure elements, if you
 * need them.  Note that bitfields are not guaranteed to be signed
 * (or even unsigned) according to ANSI C.
 */
# define B32 /* bitfield not needed on architectures with native 32-bit type */
# define B16 /* bitfield not needed on architectures with native 16-bit type */
# ifdef LONG64
typedef long INT64;
typedef int INT32;
# else
typedef long INT32;
# endif
typedef short INT16;

typedef signed char    INT8;

# ifdef LONG64
typedef unsigned long CARD64;
typedef unsigned int CARD32;
# else
typedef unsigned long long CARD64;
typedef unsigned long CARD32;
# endif
typedef unsigned short CARD16;
typedef unsigned char  CARD8;

typedef CARD32		BITS32;
typedef CARD16		BITS16;

typedef CARD8		BYTE;
typedef CARD8		BOOL;

/*
 * was definitions for sign-extending bitfields on architectures without
 * native types smaller than 64-bit, now just backwards compatibility
 */
# define cvtINT8toInt(val) (val)
# define cvtINT16toInt(val) (val)
# define cvtINT32toInt(val) (val)
# define cvtINT8toShort(val) (val)
# define cvtINT16toShort(val) (val)
# define cvtINT32toShort(val) (val)
# define cvtINT8toLong(val) (val)
# define cvtINT16toLong(val) (val)
# define cvtINT32toLong(val) (val)

/*
 * this version should leave result of type (t *), but that should only be
 * used when not in MUSTCOPY
 */
# define NEXTPTR(p,t) (((t *)(p)) + 1)

#endif /* XMD_H */
/* #  include "external/xkbcommon/xkbcommon.h" */
/*
 * Copyright 1985, 1987, 1990, 1998  The Open Group
 * Copyright 2008  Dan Nicholson
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the names of the authors or their
 * institutions shall not be used in advertising or otherwise to promote the
 * sale, use or other dealings in this Software without prior written
 * authorization from the authors.
 */

/************************************************************
 * Copyright (c) 1993 by Silicon Graphics Computer Systems, Inc.
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting
 * documentation, and that the name of Silicon Graphics not be
 * used in advertising or publicity pertaining to distribution
 * of the software without specific prior written permission.
 * Silicon Graphics makes no representation about the suitability
 * of this software for any purpose. It is provided "as is"
 * without any express or implied warranty.
 *
 * SILICON GRAPHICS DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL SILICON
 * GRAPHICS BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION  WITH
 * THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 ********************************************************/

/*
 * Copyright © 2009-2012 Daniel Stone
 * Copyright © 2012 Intel Corporation
 * Copyright © 2012 Ran Benita
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * Author: Daniel Stone <daniel@fooishbar.org>
 */

#ifndef _XKBCOMMON_H_
#define _XKBCOMMON_H_

#include <stdint.h>
#include <stdio.h>
#include <stdarg.h>

/* #include "external/xkbcommon/xkbcommon-names.h" */
/*
 * Copyright © 2012 Intel Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * Author: Daniel Stone <daniel@fooishbar.org>
 */

#ifndef _XKBCOMMON_NAMES_H
#define _XKBCOMMON_NAMES_H

/**
 * @file
 * @brief Predefined names for common modifiers and LEDs.
 */

#define XKB_MOD_NAME_SHIFT      "Shift"
#define XKB_MOD_NAME_CAPS       "Lock"
#define XKB_MOD_NAME_CTRL       "Control"
#define XKB_MOD_NAME_ALT        "Mod1"
#define XKB_MOD_NAME_NUM        "Mod2"
#define XKB_MOD_NAME_LOGO       "Mod4"

#define XKB_LED_NAME_CAPS       "Caps Lock"
#define XKB_LED_NAME_NUM        "Num Lock"
#define XKB_LED_NAME_SCROLL     "Scroll Lock"

#endif

/* #include "external/xkbcommon/xkbcommon-keysyms.h" */
#ifndef _XKBCOMMON_KEYSYMS_H
#define _XKBCOMMON_KEYSYMS_H

/* This file is autogenerated; please do not commit directly. */

/**
 * @file
 * Key symbols (keysyms) definitions.
 */

#define XKB_KEY_NoSymbol                    0x000000  /* Special KeySym */

/***********************************************************
Copyright 1987, 1994, 1998  The Open Group

Permission to use, copy, modify, distribute, and sell this software and its
documentation for any purpose is hereby granted without fee, provided that
the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation.

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE OPEN GROUP BE LIABLE FOR ANY CLAIM, DAMAGES OR
OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of The Open Group shall
not be used in advertising or otherwise to promote the sale, use or
other dealings in this Software without prior written authorization
from The Open Group.


Copyright 1987 by Digital Equipment Corporation, Maynard, Massachusetts

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of Digital not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

DIGITAL DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
DIGITAL BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

******************************************************************/

/*
 * The "X11 Window System Protocol" standard defines in Appendix A the
 * keysym codes. These 29-bit integer values identify characters or
 * functions associated with each key (e.g., via the visible
 * engraving) of a keyboard layout. This file assigns mnemonic macro
 * names for these keysyms.
 *
 * This file is also compiled (by src/util/makekeys.c in libX11) into
 * hash tables that can be accessed with X11 library functions such as
 * XStringToKeysym() and XKeysymToString().
 *
 * Where a keysym corresponds one-to-one to an ISO 10646 / Unicode
 * character, this is noted in a comment that provides both the U+xxxx
 * Unicode position, as well as the official Unicode name of the
 * character.
 *
 * Some keysyms map to a character already mapped by another keysym,
 * with compatible but more precise semantics, such as the keypad-
 * related keysyms. In this case, none of the keysym are deprecated.
 * The most generic keysym is annotated as previously and more specific
 * keysyms have the same annotation between angle brackets:
 *
 *     #define XKB_KEY_space                 0x0020  // U+0020 SPACE
 *     #define XKB_KEY_KP_Space              0xff80  //<U+0020 SPACE>
 *
 * Where the correspondence is either not one-to-one or semantically
 * unclear, the Unicode position and name are enclosed in
 * parentheses. Such legacy keysyms should be considered deprecated
 * and are not recommended for use in future keyboard mappings.
 *
 * For any future extension of the keysyms with characters already
 * found in ISO 10646 / Unicode, the following algorithm shall be
 * used. The new keysym code position will simply be the character's
 * Unicode number plus 0x01000000. The keysym values in the range
 * 0x01000100 to 0x0110ffff are reserved to represent Unicode
 * characters in the range U+0100 to U+10FFFF.
 *
 * While most newer Unicode-based X11 clients do already accept
 * Unicode-mapped keysyms in the range 0x01000100 to 0x0110ffff, it
 * will remain necessary for clients -- in the interest of
 * compatibility with existing servers -- to also understand the
 * existing legacy keysym values in the range 0x0100 to 0x20ff.
 *
 * Where several mnemonic names are defined for the same keysym in this
 * file, the first one listed is considered the "canonical" name. This
 * is the name that should be used when retrieving a keysym name from
 * its code. The next names are considered "aliases" to the canonical
 * name.
 *
 * Aliases are made explicit by writing in their comment "alias for",
 * followed by the corresponding canonical name. Example:
 *
 *     #define XKB_KEY_dead_tilde            0xfe53
 *     #define XKB_KEY_dead_perispomeni      0xfe53 // alias for dead_tilde
 *
 * The rules to consider a keysym mnemonic name deprecated are:
 *
 *   1. A legacy keysym with its Unicode mapping in parentheses is
 *      deprecated (see above).
 *
 *   2. A keysym name is *explicitly* deprecated by starting its comment
 *      with "deprecated". Examples:
 *
 *        #define XKB_KEY_L1           0xffc8  // deprecated alias for F11
 *        #define XKB_KEY_quoteleft    0x0060  // deprecated
 *
 *   3. A keysym name is *explicitly* *not* deprecated by starting its
 *      comment with "non-deprecated alias". Examples:
 *
 *       #define XKB_KEY_dead_tilde       0xfe53
 *       #define XKB_KEY_dead_perispomeni 0xfe53 // non-deprecated alias for dead_tilde
 *
 *   4. If none of the previous rules apply, an alias is *implicitly*
 *      deprecated if there is at least one previous name for the
 *      corresponding keysym that is *not* explicitly deprecated.
 *
 *      Examples:
 *
 *        // SingleCandidate is the canonical name
 *        #define XKB_KEY_SingleCandidate        0xff3c
 *        // Hangul_SingleCandidate is deprecated because it is an alias
 *        // and it does not start with "non-deprecated alias"
 *        #define XKB_KEY_Hangul_SingleCandidate 0xff3c // Single candidate
 *
 *        // guillemotleft is the canonical name, but it is deprecated
 *        #define XKB_KEY_guillemotleft  0x00ab // deprecated alias for guillemetleft (misspelling)
 *        // guillemetleft is not deprecated, because the keysym has no endorsed name before it.
 *        #define XKB_KEY_guillemetleft  0x00ab // U+00AB LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
 *        // The following hypothetical name is deprecated because guillemetleft come before.
 *        #define XKB_KEY_guillemetleft2 0x00ab
 *
 * Mnemonic names for keysyms are defined in this file with lines
 * that match one of these Perl regular expressions:
 *
 *    /^\#define XKB_KEY_([a-zA-Z_0-9]+)\s+0x([0-9a-f]+)\s*\/\* U\+([0-9A-F]{4,6}) (.*) \*\/\s*$/
 *    /^\#define XKB_KEY_([a-zA-Z_0-9]+)\s+0x([0-9a-f]+)\s*\/\*<U\+([0-9A-F]{4,6}) (.*)>\*\/\s*$/
 *    /^\#define XKB_KEY_([a-zA-Z_0-9]+)\s+0x([0-9a-f]+)\s*\/\*\(U\+([0-9A-F]{4,6}) (.*)\)\*\/\s*$/
 *    /^\#define XKB_KEY_([a-zA-Z_0-9]+)\s+0x([0-9a-f]+)\s*(\/\*\s*(.*)\s*\*\/)?\s*$/
 *
 * Before adding new keysyms, please do consider the following: In
 * addition to the keysym names defined in this file, the
 * XStringToKeysym() and XKeysymToString() functions will also handle
 * any keysym string of the form "U0020" to "U007E" and "U00A0" to
 * "U10FFFF" for all possible Unicode characters. In other words,
 * every possible Unicode character has already a keysym string
 * defined algorithmically, even if it is not listed here. Therefore,
 * defining an additional keysym macro is only necessary where a
 * non-hexadecimal mnemonic name is needed, or where the new keysym
 * does not represent any existing Unicode character.
 *
 * When adding new keysyms to this file, do not forget to also update the
 * following as needed:
 *
 *   - the mappings in src/KeyBind.c in the libX11 repo
 *     https://gitlab.freedesktop.org/xorg/lib/libx11
 *
 *   - the protocol specification in specs/keysyms.xml in this repo
 *     https://gitlab.freedesktop.org/xorg/proto/xorgproto
 *
 * Before removing or changing the order of the keysyms, please consider
 * the following: it is very difficult to know what keysyms are used and
 * how.
 *
 *   - A sandboxed application may have incompatibilities with the host
 *     system. For example, if new keysym name is introduced and is made
 *     the canonical name, then an application with an older keysym parser
 *     will not be able to parse the new name.
 *   - Customization of keyboard layout and Compose files are two popular
 *     use cases. Checking the standard keyboard layout database xkeyboard-config
 *     https://gitlab.freedesktop.org/xkeyboard-config/xkeyboard-config
 *     and the standard Compose files in libx11
 *     https://gitlab.freedesktop.org/xorg/lib/libx11 is a mandatory
 *     step, but may *not* be enough for a proper impact assessment for
 *     e.g. keysyms removals.
 *
 * Therefore, it is advised to proceed to no removal and to make a new
 * name canonical only 10 years after its introduction. This means that
 * some keysyms may have their first listed name deprecated during the
 * period of transition. Once this period is over, the deprecated name
 * should be moved after the new canonical name.
 */

#define XKB_KEY_VoidSymbol                  0xffffff  /* Void symbol */

/*
 * TTY function keys, cleverly chosen to map to ASCII, for convenience of
 * programming, but could have been arbitrary (at the cost of lookup
 * tables in client code).
 */

#define XKB_KEY_BackSpace                     0xff08  /* U+0008 BACKSPACE */
#define XKB_KEY_Tab                           0xff09  /* U+0009 CHARACTER TABULATION */
#define XKB_KEY_Linefeed                      0xff0a  /* U+000A LINE FEED */
#define XKB_KEY_Clear                         0xff0b  /* U+000B LINE TABULATION */
#define XKB_KEY_Return                        0xff0d  /* U+000D CARRIAGE RETURN */
#define XKB_KEY_Pause                         0xff13  /* Pause, hold */
#define XKB_KEY_Scroll_Lock                   0xff14
#define XKB_KEY_Sys_Req                       0xff15
#define XKB_KEY_Escape                        0xff1b  /* U+001B ESCAPE */
#define XKB_KEY_Delete                        0xffff  /* U+007F DELETE */



/* International & multi-key character composition */

#define XKB_KEY_Multi_key                     0xff20  /* Multi-key character compose */
#define XKB_KEY_Codeinput                     0xff37
#define XKB_KEY_SingleCandidate               0xff3c
#define XKB_KEY_MultipleCandidate             0xff3d
#define XKB_KEY_PreviousCandidate             0xff3e

/* Japanese keyboard support */

#define XKB_KEY_Kanji                         0xff21  /* Kanji, Kanji convert */
#define XKB_KEY_Muhenkan                      0xff22  /* Cancel Conversion */
#define XKB_KEY_Henkan_Mode                   0xff23  /* Start/Stop Conversion */
#define XKB_KEY_Henkan                        0xff23  /* non-deprecated alias for Henkan_Mode */
#define XKB_KEY_Romaji                        0xff24  /* to Romaji */
#define XKB_KEY_Hiragana                      0xff25  /* to Hiragana */
#define XKB_KEY_Katakana                      0xff26  /* to Katakana */
#define XKB_KEY_Hiragana_Katakana             0xff27  /* Hiragana/Katakana toggle */
#define XKB_KEY_Zenkaku                       0xff28  /* to Zenkaku */
#define XKB_KEY_Hankaku                       0xff29  /* to Hankaku */
#define XKB_KEY_Zenkaku_Hankaku               0xff2a  /* Zenkaku/Hankaku toggle */
#define XKB_KEY_Touroku                       0xff2b  /* Add to Dictionary */
#define XKB_KEY_Massyo                        0xff2c  /* Delete from Dictionary */
#define XKB_KEY_Kana_Lock                     0xff2d  /* Kana Lock */
#define XKB_KEY_Kana_Shift                    0xff2e  /* Kana Shift */
#define XKB_KEY_Eisu_Shift                    0xff2f  /* Alphanumeric Shift */
#define XKB_KEY_Eisu_toggle                   0xff30  /* Alphanumeric toggle */
#define XKB_KEY_Kanji_Bangou                  0xff37  /* Codeinput */
#define XKB_KEY_Zen_Koho                      0xff3d  /* Multiple/All Candidate(s) */
#define XKB_KEY_Mae_Koho                      0xff3e  /* Previous Candidate */

/* 0xff31 thru 0xff3f are under XK_KOREAN */

/* Cursor control & motion */

#define XKB_KEY_Home                          0xff50
#define XKB_KEY_Left                          0xff51  /* Move left, left arrow */
#define XKB_KEY_Up                            0xff52  /* Move up, up arrow */
#define XKB_KEY_Right                         0xff53  /* Move right, right arrow */
#define XKB_KEY_Down                          0xff54  /* Move down, down arrow */
#define XKB_KEY_Prior                         0xff55  /* Prior, previous */
#define XKB_KEY_Page_Up                       0xff55  /* deprecated alias for Prior */
#define XKB_KEY_Next                          0xff56  /* Next */
#define XKB_KEY_Page_Down                     0xff56  /* deprecated alias for Next */
#define XKB_KEY_End                           0xff57  /* EOL */
#define XKB_KEY_Begin                         0xff58  /* BOL */


/* Misc functions */

#define XKB_KEY_Select                        0xff60  /* Select, mark */
#define XKB_KEY_Print                         0xff61
#define XKB_KEY_Execute                       0xff62  /* Execute, run, do */
#define XKB_KEY_Insert                        0xff63  /* Insert, insert here */
#define XKB_KEY_Undo                          0xff65
#define XKB_KEY_Redo                          0xff66  /* Redo, again */
#define XKB_KEY_Menu                          0xff67
#define XKB_KEY_Find                          0xff68  /* Find, search */
#define XKB_KEY_Cancel                        0xff69  /* Cancel, stop, abort, exit */
#define XKB_KEY_Help                          0xff6a  /* Help */
#define XKB_KEY_Break                         0xff6b
#define XKB_KEY_Mode_switch                   0xff7e  /* Character set switch */
#define XKB_KEY_script_switch                 0xff7e  /* non-deprecated alias for Mode_switch */
#define XKB_KEY_Num_Lock                      0xff7f

/* Keypad functions, keypad numbers cleverly chosen to map to ASCII */

#define XKB_KEY_KP_Space                      0xff80  /*<U+0020 SPACE>*/
#define XKB_KEY_KP_Tab                        0xff89  /*<U+0009 CHARACTER TABULATION>*/
#define XKB_KEY_KP_Enter                      0xff8d  /*<U+000D CARRIAGE RETURN>*/
#define XKB_KEY_KP_F1                         0xff91  /* PF1, KP_A, ... */
#define XKB_KEY_KP_F2                         0xff92
#define XKB_KEY_KP_F3                         0xff93
#define XKB_KEY_KP_F4                         0xff94
#define XKB_KEY_KP_Home                       0xff95
#define XKB_KEY_KP_Left                       0xff96
#define XKB_KEY_KP_Up                         0xff97
#define XKB_KEY_KP_Right                      0xff98
#define XKB_KEY_KP_Down                       0xff99
#define XKB_KEY_KP_Prior                      0xff9a
#define XKB_KEY_KP_Page_Up                    0xff9a  /* deprecated alias for KP_Prior */
#define XKB_KEY_KP_Next                       0xff9b
#define XKB_KEY_KP_Page_Down                  0xff9b  /* deprecated alias for KP_Next */
#define XKB_KEY_KP_End                        0xff9c
#define XKB_KEY_KP_Begin                      0xff9d
#define XKB_KEY_KP_Insert                     0xff9e
#define XKB_KEY_KP_Delete                     0xff9f
#define XKB_KEY_KP_Equal                      0xffbd  /*<U+003D EQUALS SIGN>*/
#define XKB_KEY_KP_Multiply                   0xffaa  /*<U+002A ASTERISK>*/
#define XKB_KEY_KP_Add                        0xffab  /*<U+002B PLUS SIGN>*/
#define XKB_KEY_KP_Separator                  0xffac  /*<U+002C COMMA>*/
#define XKB_KEY_KP_Subtract                   0xffad  /*<U+002D HYPHEN-MINUS>*/
#define XKB_KEY_KP_Decimal                    0xffae  /*<U+002E FULL STOP>*/
#define XKB_KEY_KP_Divide                     0xffaf  /*<U+002F SOLIDUS>*/

#define XKB_KEY_KP_0                          0xffb0  /*<U+0030 DIGIT ZERO>*/
#define XKB_KEY_KP_1                          0xffb1  /*<U+0031 DIGIT ONE>*/
#define XKB_KEY_KP_2                          0xffb2  /*<U+0032 DIGIT TWO>*/
#define XKB_KEY_KP_3                          0xffb3  /*<U+0033 DIGIT THREE>*/
#define XKB_KEY_KP_4                          0xffb4  /*<U+0034 DIGIT FOUR>*/
#define XKB_KEY_KP_5                          0xffb5  /*<U+0035 DIGIT FIVE>*/
#define XKB_KEY_KP_6                          0xffb6  /*<U+0036 DIGIT SIX>*/
#define XKB_KEY_KP_7                          0xffb7  /*<U+0037 DIGIT SEVEN>*/
#define XKB_KEY_KP_8                          0xffb8  /*<U+0038 DIGIT EIGHT>*/
#define XKB_KEY_KP_9                          0xffb9  /*<U+0039 DIGIT NINE>*/



/*
 * Auxiliary functions; note the duplicate definitions for left and right
 * function keys;  Sun keyboards and a few other manufacturers have such
 * function key groups on the left and/or right sides of the keyboard.
 * We've not found a keyboard with more than 35 function keys total.
 */

#define XKB_KEY_F1                            0xffbe
#define XKB_KEY_F2                            0xffbf
#define XKB_KEY_F3                            0xffc0
#define XKB_KEY_F4                            0xffc1
#define XKB_KEY_F5                            0xffc2
#define XKB_KEY_F6                            0xffc3
#define XKB_KEY_F7                            0xffc4
#define XKB_KEY_F8                            0xffc5
#define XKB_KEY_F9                            0xffc6
#define XKB_KEY_F10                           0xffc7
#define XKB_KEY_F11                           0xffc8
#define XKB_KEY_L1                            0xffc8  /* deprecated alias for F11 */
#define XKB_KEY_F12                           0xffc9
#define XKB_KEY_L2                            0xffc9  /* deprecated alias for F12 */
#define XKB_KEY_F13                           0xffca
#define XKB_KEY_L3                            0xffca  /* deprecated alias for F13 */
#define XKB_KEY_F14                           0xffcb
#define XKB_KEY_L4                            0xffcb  /* deprecated alias for F14 */
#define XKB_KEY_F15                           0xffcc
#define XKB_KEY_L5                            0xffcc  /* deprecated alias for F15 */
#define XKB_KEY_F16                           0xffcd
#define XKB_KEY_L6                            0xffcd  /* deprecated alias for F16 */
#define XKB_KEY_F17                           0xffce
#define XKB_KEY_L7                            0xffce  /* deprecated alias for F17 */
#define XKB_KEY_F18                           0xffcf
#define XKB_KEY_L8                            0xffcf  /* deprecated alias for F18 */
#define XKB_KEY_F19                           0xffd0
#define XKB_KEY_L9                            0xffd0  /* deprecated alias for F19 */
#define XKB_KEY_F20                           0xffd1
#define XKB_KEY_L10                           0xffd1  /* deprecated alias for F20 */
#define XKB_KEY_F21                           0xffd2
#define XKB_KEY_R1                            0xffd2  /* deprecated alias for F21 */
#define XKB_KEY_F22                           0xffd3
#define XKB_KEY_R2                            0xffd3  /* deprecated alias for F22 */
#define XKB_KEY_F23                           0xffd4
#define XKB_KEY_R3                            0xffd4  /* deprecated alias for F23 */
#define XKB_KEY_F24                           0xffd5
#define XKB_KEY_R4                            0xffd5  /* deprecated alias for F24 */
#define XKB_KEY_F25                           0xffd6
#define XKB_KEY_R5                            0xffd6  /* deprecated alias for F25 */
#define XKB_KEY_F26                           0xffd7
#define XKB_KEY_R6                            0xffd7  /* deprecated alias for F26 */
#define XKB_KEY_F27                           0xffd8
#define XKB_KEY_R7                            0xffd8  /* deprecated alias for F27 */
#define XKB_KEY_F28                           0xffd9
#define XKB_KEY_R8                            0xffd9  /* deprecated alias for F28 */
#define XKB_KEY_F29                           0xffda
#define XKB_KEY_R9                            0xffda  /* deprecated alias for F29 */
#define XKB_KEY_F30                           0xffdb
#define XKB_KEY_R10                           0xffdb  /* deprecated alias for F30 */
#define XKB_KEY_F31                           0xffdc
#define XKB_KEY_R11                           0xffdc  /* deprecated alias for F31 */
#define XKB_KEY_F32                           0xffdd
#define XKB_KEY_R12                           0xffdd  /* deprecated alias for F32 */
#define XKB_KEY_F33                           0xffde
#define XKB_KEY_R13                           0xffde  /* deprecated alias for F33 */
#define XKB_KEY_F34                           0xffdf
#define XKB_KEY_R14                           0xffdf  /* deprecated alias for F34 */
#define XKB_KEY_F35                           0xffe0
#define XKB_KEY_R15                           0xffe0  /* deprecated alias for F35 */

/* Modifiers */

#define XKB_KEY_Shift_L                       0xffe1  /* Left shift */
#define XKB_KEY_Shift_R                       0xffe2  /* Right shift */
#define XKB_KEY_Control_L                     0xffe3  /* Left control */
#define XKB_KEY_Control_R                     0xffe4  /* Right control */
#define XKB_KEY_Caps_Lock                     0xffe5  /* Caps lock */
#define XKB_KEY_Shift_Lock                    0xffe6  /* Shift lock */

#define XKB_KEY_Meta_L                        0xffe7  /* Left meta */
#define XKB_KEY_Meta_R                        0xffe8  /* Right meta */
#define XKB_KEY_Alt_L                         0xffe9  /* Left alt */
#define XKB_KEY_Alt_R                         0xffea  /* Right alt */
#define XKB_KEY_Super_L                       0xffeb  /* Left super */
#define XKB_KEY_Super_R                       0xffec  /* Right super */
#define XKB_KEY_Hyper_L                       0xffed  /* Left hyper */
#define XKB_KEY_Hyper_R                       0xffee  /* Right hyper */

/*
 * Keyboard (XKB) Extension function and modifier keys
 * (from Appendix C of "The X Keyboard Extension: Protocol Specification")
 * Byte 3 = 0xfe
 */

#define XKB_KEY_ISO_Lock                      0xfe01
#define XKB_KEY_ISO_Level2_Latch              0xfe02
#define XKB_KEY_ISO_Level3_Shift              0xfe03
#define XKB_KEY_ISO_Level3_Latch              0xfe04
#define XKB_KEY_ISO_Level3_Lock               0xfe05
#define XKB_KEY_ISO_Level5_Shift              0xfe11
#define XKB_KEY_ISO_Level5_Latch              0xfe12
#define XKB_KEY_ISO_Level5_Lock               0xfe13
#define XKB_KEY_ISO_Group_Shift               0xff7e  /* non-deprecated alias for Mode_switch */
#define XKB_KEY_ISO_Group_Latch               0xfe06
#define XKB_KEY_ISO_Group_Lock                0xfe07
#define XKB_KEY_ISO_Next_Group                0xfe08
#define XKB_KEY_ISO_Next_Group_Lock           0xfe09
#define XKB_KEY_ISO_Prev_Group                0xfe0a
#define XKB_KEY_ISO_Prev_Group_Lock           0xfe0b
#define XKB_KEY_ISO_First_Group               0xfe0c
#define XKB_KEY_ISO_First_Group_Lock          0xfe0d
#define XKB_KEY_ISO_Last_Group                0xfe0e
#define XKB_KEY_ISO_Last_Group_Lock           0xfe0f

#define XKB_KEY_ISO_Left_Tab                  0xfe20
#define XKB_KEY_ISO_Move_Line_Up              0xfe21
#define XKB_KEY_ISO_Move_Line_Down            0xfe22
#define XKB_KEY_ISO_Partial_Line_Up           0xfe23
#define XKB_KEY_ISO_Partial_Line_Down         0xfe24
#define XKB_KEY_ISO_Partial_Space_Left        0xfe25
#define XKB_KEY_ISO_Partial_Space_Right       0xfe26
#define XKB_KEY_ISO_Set_Margin_Left           0xfe27
#define XKB_KEY_ISO_Set_Margin_Right          0xfe28
#define XKB_KEY_ISO_Release_Margin_Left       0xfe29
#define XKB_KEY_ISO_Release_Margin_Right      0xfe2a
#define XKB_KEY_ISO_Release_Both_Margins      0xfe2b
#define XKB_KEY_ISO_Fast_Cursor_Left          0xfe2c
#define XKB_KEY_ISO_Fast_Cursor_Right         0xfe2d
#define XKB_KEY_ISO_Fast_Cursor_Up            0xfe2e
#define XKB_KEY_ISO_Fast_Cursor_Down          0xfe2f
#define XKB_KEY_ISO_Continuous_Underline      0xfe30
#define XKB_KEY_ISO_Discontinuous_Underline   0xfe31
#define XKB_KEY_ISO_Emphasize                 0xfe32
#define XKB_KEY_ISO_Center_Object             0xfe33
#define XKB_KEY_ISO_Enter                     0xfe34

#define XKB_KEY_dead_grave                    0xfe50
#define XKB_KEY_dead_acute                    0xfe51
#define XKB_KEY_dead_circumflex               0xfe52
#define XKB_KEY_dead_tilde                    0xfe53
#define XKB_KEY_dead_perispomeni              0xfe53  /* non-deprecated alias for dead_tilde */
#define XKB_KEY_dead_macron                   0xfe54
#define XKB_KEY_dead_breve                    0xfe55
#define XKB_KEY_dead_abovedot                 0xfe56
#define XKB_KEY_dead_diaeresis                0xfe57
#define XKB_KEY_dead_abovering                0xfe58
#define XKB_KEY_dead_doubleacute              0xfe59
#define XKB_KEY_dead_caron                    0xfe5a
#define XKB_KEY_dead_cedilla                  0xfe5b
#define XKB_KEY_dead_ogonek                   0xfe5c
#define XKB_KEY_dead_iota                     0xfe5d
#define XKB_KEY_dead_voiced_sound             0xfe5e
#define XKB_KEY_dead_semivoiced_sound         0xfe5f
#define XKB_KEY_dead_belowdot                 0xfe60
#define XKB_KEY_dead_hook                     0xfe61
#define XKB_KEY_dead_horn                     0xfe62
#define XKB_KEY_dead_stroke                   0xfe63
#define XKB_KEY_dead_abovecomma               0xfe64
#define XKB_KEY_dead_psili                    0xfe64  /* non-deprecated alias for dead_abovecomma */
#define XKB_KEY_dead_abovereversedcomma       0xfe65
#define XKB_KEY_dead_dasia                    0xfe65  /* non-deprecated alias for dead_abovereversedcomma */
#define XKB_KEY_dead_doublegrave              0xfe66
#define XKB_KEY_dead_belowring                0xfe67
#define XKB_KEY_dead_belowmacron              0xfe68
#define XKB_KEY_dead_belowcircumflex          0xfe69
#define XKB_KEY_dead_belowtilde               0xfe6a
#define XKB_KEY_dead_belowbreve               0xfe6b
#define XKB_KEY_dead_belowdiaeresis           0xfe6c
#define XKB_KEY_dead_invertedbreve            0xfe6d
#define XKB_KEY_dead_belowcomma               0xfe6e
#define XKB_KEY_dead_currency                 0xfe6f

/* extra dead elements for German T3 layout */
#define XKB_KEY_dead_lowline                  0xfe90
#define XKB_KEY_dead_aboveverticalline        0xfe91
#define XKB_KEY_dead_belowverticalline        0xfe92
#define XKB_KEY_dead_longsolidusoverlay       0xfe93

/* dead vowels for universal syllable entry */
#define XKB_KEY_dead_a                        0xfe80
#define XKB_KEY_dead_A                        0xfe81
#define XKB_KEY_dead_e                        0xfe82
#define XKB_KEY_dead_E                        0xfe83
#define XKB_KEY_dead_i                        0xfe84
#define XKB_KEY_dead_I                        0xfe85
#define XKB_KEY_dead_o                        0xfe86
#define XKB_KEY_dead_O                        0xfe87
#define XKB_KEY_dead_u                        0xfe88
#define XKB_KEY_dead_U                        0xfe89
#define XKB_KEY_dead_small_schwa              0xfe8a  /* deprecated alias for dead_schwa */
#define XKB_KEY_dead_schwa                    0xfe8a
#define XKB_KEY_dead_capital_schwa            0xfe8b  /* deprecated alias for dead_SCHWA */
#define XKB_KEY_dead_SCHWA                    0xfe8b

#define XKB_KEY_dead_greek                    0xfe8c
#define XKB_KEY_dead_hamza                    0xfe8d

#define XKB_KEY_First_Virtual_Screen          0xfed0
#define XKB_KEY_Prev_Virtual_Screen           0xfed1
#define XKB_KEY_Next_Virtual_Screen           0xfed2
#define XKB_KEY_Last_Virtual_Screen           0xfed4
#define XKB_KEY_Terminate_Server              0xfed5

#define XKB_KEY_AccessX_Enable                0xfe70
#define XKB_KEY_AccessX_Feedback_Enable       0xfe71
#define XKB_KEY_RepeatKeys_Enable             0xfe72
#define XKB_KEY_SlowKeys_Enable               0xfe73
#define XKB_KEY_BounceKeys_Enable             0xfe74
#define XKB_KEY_StickyKeys_Enable             0xfe75
#define XKB_KEY_MouseKeys_Enable              0xfe76
#define XKB_KEY_MouseKeys_Accel_Enable        0xfe77
#define XKB_KEY_Overlay1_Enable               0xfe78
#define XKB_KEY_Overlay2_Enable               0xfe79
#define XKB_KEY_AudibleBell_Enable            0xfe7a

#define XKB_KEY_Pointer_Left                  0xfee0
#define XKB_KEY_Pointer_Right                 0xfee1
#define XKB_KEY_Pointer_Up                    0xfee2
#define XKB_KEY_Pointer_Down                  0xfee3
#define XKB_KEY_Pointer_UpLeft                0xfee4
#define XKB_KEY_Pointer_UpRight               0xfee5
#define XKB_KEY_Pointer_DownLeft              0xfee6
#define XKB_KEY_Pointer_DownRight             0xfee7
#define XKB_KEY_Pointer_Button_Dflt           0xfee8
#define XKB_KEY_Pointer_Button1               0xfee9
#define XKB_KEY_Pointer_Button2               0xfeea
#define XKB_KEY_Pointer_Button3               0xfeeb
#define XKB_KEY_Pointer_Button4               0xfeec
#define XKB_KEY_Pointer_Button5               0xfeed
#define XKB_KEY_Pointer_DblClick_Dflt         0xfeee
#define XKB_KEY_Pointer_DblClick1             0xfeef
#define XKB_KEY_Pointer_DblClick2             0xfef0
#define XKB_KEY_Pointer_DblClick3             0xfef1
#define XKB_KEY_Pointer_DblClick4             0xfef2
#define XKB_KEY_Pointer_DblClick5             0xfef3
#define XKB_KEY_Pointer_Drag_Dflt             0xfef4
#define XKB_KEY_Pointer_Drag1                 0xfef5
#define XKB_KEY_Pointer_Drag2                 0xfef6
#define XKB_KEY_Pointer_Drag3                 0xfef7
#define XKB_KEY_Pointer_Drag4                 0xfef8
#define XKB_KEY_Pointer_Drag5                 0xfefd

#define XKB_KEY_Pointer_EnableKeys            0xfef9
#define XKB_KEY_Pointer_Accelerate            0xfefa
#define XKB_KEY_Pointer_DfltBtnNext           0xfefb
#define XKB_KEY_Pointer_DfltBtnPrev           0xfefc

/* Single-Stroke Multiple-Character N-Graph Keysyms For The X Input Method */

#define XKB_KEY_ch                            0xfea0
#define XKB_KEY_Ch                            0xfea1
#define XKB_KEY_CH                            0xfea2
#define XKB_KEY_c_h                           0xfea3
#define XKB_KEY_C_h                           0xfea4
#define XKB_KEY_C_H                           0xfea5


/*
 * 3270 Terminal Keys
 * Byte 3 = 0xfd
 */

#define XKB_KEY_3270_Duplicate                0xfd01
#define XKB_KEY_3270_FieldMark                0xfd02
#define XKB_KEY_3270_Right2                   0xfd03
#define XKB_KEY_3270_Left2                    0xfd04
#define XKB_KEY_3270_BackTab                  0xfd05
#define XKB_KEY_3270_EraseEOF                 0xfd06
#define XKB_KEY_3270_EraseInput               0xfd07
#define XKB_KEY_3270_Reset                    0xfd08
#define XKB_KEY_3270_Quit                     0xfd09
#define XKB_KEY_3270_PA1                      0xfd0a
#define XKB_KEY_3270_PA2                      0xfd0b
#define XKB_KEY_3270_PA3                      0xfd0c
#define XKB_KEY_3270_Test                     0xfd0d
#define XKB_KEY_3270_Attn                     0xfd0e
#define XKB_KEY_3270_CursorBlink              0xfd0f
#define XKB_KEY_3270_AltCursor                0xfd10
#define XKB_KEY_3270_KeyClick                 0xfd11
#define XKB_KEY_3270_Jump                     0xfd12
#define XKB_KEY_3270_Ident                    0xfd13
#define XKB_KEY_3270_Rule                     0xfd14
#define XKB_KEY_3270_Copy                     0xfd15
#define XKB_KEY_3270_Play                     0xfd16
#define XKB_KEY_3270_Setup                    0xfd17
#define XKB_KEY_3270_Record                   0xfd18
#define XKB_KEY_3270_ChangeScreen             0xfd19
#define XKB_KEY_3270_DeleteWord               0xfd1a
#define XKB_KEY_3270_ExSelect                 0xfd1b
#define XKB_KEY_3270_CursorSelect             0xfd1c
#define XKB_KEY_3270_PrintScreen              0xfd1d
#define XKB_KEY_3270_Enter                    0xfd1e

/*
 * Latin 1
 * (ISO/IEC 8859-1 = Unicode U+0020..U+00FF)
 * Byte 3 = 0
 */
#define XKB_KEY_space                         0x0020  /* U+0020 SPACE */
#define XKB_KEY_exclam                        0x0021  /* U+0021 EXCLAMATION MARK */
#define XKB_KEY_quotedbl                      0x0022  /* U+0022 QUOTATION MARK */
#define XKB_KEY_numbersign                    0x0023  /* U+0023 NUMBER SIGN */
#define XKB_KEY_dollar                        0x0024  /* U+0024 DOLLAR SIGN */
#define XKB_KEY_percent                       0x0025  /* U+0025 PERCENT SIGN */
#define XKB_KEY_ampersand                     0x0026  /* U+0026 AMPERSAND */
#define XKB_KEY_apostrophe                    0x0027  /* U+0027 APOSTROPHE */
#define XKB_KEY_quoteright                    0x0027  /* deprecated */
#define XKB_KEY_parenleft                     0x0028  /* U+0028 LEFT PARENTHESIS */
#define XKB_KEY_parenright                    0x0029  /* U+0029 RIGHT PARENTHESIS */
#define XKB_KEY_asterisk                      0x002a  /* U+002A ASTERISK */
#define XKB_KEY_plus                          0x002b  /* U+002B PLUS SIGN */
#define XKB_KEY_comma                         0x002c  /* U+002C COMMA */
#define XKB_KEY_minus                         0x002d  /* U+002D HYPHEN-MINUS */
#define XKB_KEY_period                        0x002e  /* U+002E FULL STOP */
#define XKB_KEY_slash                         0x002f  /* U+002F SOLIDUS */
#define XKB_KEY_0                             0x0030  /* U+0030 DIGIT ZERO */
#define XKB_KEY_1                             0x0031  /* U+0031 DIGIT ONE */
#define XKB_KEY_2                             0x0032  /* U+0032 DIGIT TWO */
#define XKB_KEY_3                             0x0033  /* U+0033 DIGIT THREE */
#define XKB_KEY_4                             0x0034  /* U+0034 DIGIT FOUR */
#define XKB_KEY_5                             0x0035  /* U+0035 DIGIT FIVE */
#define XKB_KEY_6                             0x0036  /* U+0036 DIGIT SIX */
#define XKB_KEY_7                             0x0037  /* U+0037 DIGIT SEVEN */
#define XKB_KEY_8                             0x0038  /* U+0038 DIGIT EIGHT */
#define XKB_KEY_9                             0x0039  /* U+0039 DIGIT NINE */
#define XKB_KEY_colon                         0x003a  /* U+003A COLON */
#define XKB_KEY_semicolon                     0x003b  /* U+003B SEMICOLON */
#define XKB_KEY_less                          0x003c  /* U+003C LESS-THAN SIGN */
#define XKB_KEY_equal                         0x003d  /* U+003D EQUALS SIGN */
#define XKB_KEY_greater                       0x003e  /* U+003E GREATER-THAN SIGN */
#define XKB_KEY_question                      0x003f  /* U+003F QUESTION MARK */
#define XKB_KEY_at                            0x0040  /* U+0040 COMMERCIAL AT */
#define XKB_KEY_A                             0x0041  /* U+0041 LATIN CAPITAL LETTER A */
#define XKB_KEY_B                             0x0042  /* U+0042 LATIN CAPITAL LETTER B */
#define XKB_KEY_C                             0x0043  /* U+0043 LATIN CAPITAL LETTER C */
#define XKB_KEY_D                             0x0044  /* U+0044 LATIN CAPITAL LETTER D */
#define XKB_KEY_E                             0x0045  /* U+0045 LATIN CAPITAL LETTER E */
#define XKB_KEY_F                             0x0046  /* U+0046 LATIN CAPITAL LETTER F */
#define XKB_KEY_G                             0x0047  /* U+0047 LATIN CAPITAL LETTER G */
#define XKB_KEY_H                             0x0048  /* U+0048 LATIN CAPITAL LETTER H */
#define XKB_KEY_I                             0x0049  /* U+0049 LATIN CAPITAL LETTER I */
#define XKB_KEY_J                             0x004a  /* U+004A LATIN CAPITAL LETTER J */
#define XKB_KEY_K                             0x004b  /* U+004B LATIN CAPITAL LETTER K */
#define XKB_KEY_L                             0x004c  /* U+004C LATIN CAPITAL LETTER L */
#define XKB_KEY_M                             0x004d  /* U+004D LATIN CAPITAL LETTER M */
#define XKB_KEY_N                             0x004e  /* U+004E LATIN CAPITAL LETTER N */
#define XKB_KEY_O                             0x004f  /* U+004F LATIN CAPITAL LETTER O */
#define XKB_KEY_P                             0x0050  /* U+0050 LATIN CAPITAL LETTER P */
#define XKB_KEY_Q                             0x0051  /* U+0051 LATIN CAPITAL LETTER Q */
#define XKB_KEY_R                             0x0052  /* U+0052 LATIN CAPITAL LETTER R */
#define XKB_KEY_S                             0x0053  /* U+0053 LATIN CAPITAL LETTER S */
#define XKB_KEY_T                             0x0054  /* U+0054 LATIN CAPITAL LETTER T */
#define XKB_KEY_U                             0x0055  /* U+0055 LATIN CAPITAL LETTER U */
#define XKB_KEY_V                             0x0056  /* U+0056 LATIN CAPITAL LETTER V */
#define XKB_KEY_W                             0x0057  /* U+0057 LATIN CAPITAL LETTER W */
#define XKB_KEY_X                             0x0058  /* U+0058 LATIN CAPITAL LETTER X */
#define XKB_KEY_Y                             0x0059  /* U+0059 LATIN CAPITAL LETTER Y */
#define XKB_KEY_Z                             0x005a  /* U+005A LATIN CAPITAL LETTER Z */
#define XKB_KEY_bracketleft                   0x005b  /* U+005B LEFT SQUARE BRACKET */
#define XKB_KEY_backslash                     0x005c  /* U+005C REVERSE SOLIDUS */
#define XKB_KEY_bracketright                  0x005d  /* U+005D RIGHT SQUARE BRACKET */
#define XKB_KEY_asciicircum                   0x005e  /* U+005E CIRCUMFLEX ACCENT */
#define XKB_KEY_underscore                    0x005f  /* U+005F LOW LINE */
#define XKB_KEY_grave                         0x0060  /* U+0060 GRAVE ACCENT */
#define XKB_KEY_quoteleft                     0x0060  /* deprecated */
#define XKB_KEY_a                             0x0061  /* U+0061 LATIN SMALL LETTER A */
#define XKB_KEY_b                             0x0062  /* U+0062 LATIN SMALL LETTER B */
#define XKB_KEY_c                             0x0063  /* U+0063 LATIN SMALL LETTER C */
#define XKB_KEY_d                             0x0064  /* U+0064 LATIN SMALL LETTER D */
#define XKB_KEY_e                             0x0065  /* U+0065 LATIN SMALL LETTER E */
#define XKB_KEY_f                             0x0066  /* U+0066 LATIN SMALL LETTER F */
#define XKB_KEY_g                             0x0067  /* U+0067 LATIN SMALL LETTER G */
#define XKB_KEY_h                             0x0068  /* U+0068 LATIN SMALL LETTER H */
#define XKB_KEY_i                             0x0069  /* U+0069 LATIN SMALL LETTER I */
#define XKB_KEY_j                             0x006a  /* U+006A LATIN SMALL LETTER J */
#define XKB_KEY_k                             0x006b  /* U+006B LATIN SMALL LETTER K */
#define XKB_KEY_l                             0x006c  /* U+006C LATIN SMALL LETTER L */
#define XKB_KEY_m                             0x006d  /* U+006D LATIN SMALL LETTER M */
#define XKB_KEY_n                             0x006e  /* U+006E LATIN SMALL LETTER N */
#define XKB_KEY_o                             0x006f  /* U+006F LATIN SMALL LETTER O */
#define XKB_KEY_p                             0x0070  /* U+0070 LATIN SMALL LETTER P */
#define XKB_KEY_q                             0x0071  /* U+0071 LATIN SMALL LETTER Q */
#define XKB_KEY_r                             0x0072  /* U+0072 LATIN SMALL LETTER R */
#define XKB_KEY_s                             0x0073  /* U+0073 LATIN SMALL LETTER S */
#define XKB_KEY_t                             0x0074  /* U+0074 LATIN SMALL LETTER T */
#define XKB_KEY_u                             0x0075  /* U+0075 LATIN SMALL LETTER U */
#define XKB_KEY_v                             0x0076  /* U+0076 LATIN SMALL LETTER V */
#define XKB_KEY_w                             0x0077  /* U+0077 LATIN SMALL LETTER W */
#define XKB_KEY_x                             0x0078  /* U+0078 LATIN SMALL LETTER X */
#define XKB_KEY_y                             0x0079  /* U+0079 LATIN SMALL LETTER Y */
#define XKB_KEY_z                             0x007a  /* U+007A LATIN SMALL LETTER Z */
#define XKB_KEY_braceleft                     0x007b  /* U+007B LEFT CURLY BRACKET */
#define XKB_KEY_bar                           0x007c  /* U+007C VERTICAL LINE */
#define XKB_KEY_braceright                    0x007d  /* U+007D RIGHT CURLY BRACKET */
#define XKB_KEY_asciitilde                    0x007e  /* U+007E TILDE */

#define XKB_KEY_nobreakspace                  0x00a0  /* U+00A0 NO-BREAK SPACE */
#define XKB_KEY_exclamdown                    0x00a1  /* U+00A1 INVERTED EXCLAMATION MARK */
#define XKB_KEY_cent                          0x00a2  /* U+00A2 CENT SIGN */
#define XKB_KEY_sterling                      0x00a3  /* U+00A3 POUND SIGN */
#define XKB_KEY_currency                      0x00a4  /* U+00A4 CURRENCY SIGN */
#define XKB_KEY_yen                           0x00a5  /* U+00A5 YEN SIGN */
#define XKB_KEY_brokenbar                     0x00a6  /* U+00A6 BROKEN BAR */
#define XKB_KEY_section                       0x00a7  /* U+00A7 SECTION SIGN */
#define XKB_KEY_diaeresis                     0x00a8  /* U+00A8 DIAERESIS */
#define XKB_KEY_copyright                     0x00a9  /* U+00A9 COPYRIGHT SIGN */
#define XKB_KEY_ordfeminine                   0x00aa  /* U+00AA FEMININE ORDINAL INDICATOR */
#define XKB_KEY_guillemotleft                 0x00ab  /* deprecated alias for guillemetleft (misspelling) */
#define XKB_KEY_guillemetleft                 0x00ab  /* U+00AB LEFT-POINTING DOUBLE ANGLE QUOTATION MARK */
#define XKB_KEY_notsign                       0x00ac  /* U+00AC NOT SIGN */
#define XKB_KEY_hyphen                        0x00ad  /* U+00AD SOFT HYPHEN */
#define XKB_KEY_registered                    0x00ae  /* U+00AE REGISTERED SIGN */
#define XKB_KEY_macron                        0x00af  /* U+00AF MACRON */
#define XKB_KEY_degree                        0x00b0  /* U+00B0 DEGREE SIGN */
#define XKB_KEY_plusminus                     0x00b1  /* U+00B1 PLUS-MINUS SIGN */
#define XKB_KEY_twosuperior                   0x00b2  /* U+00B2 SUPERSCRIPT TWO */
#define XKB_KEY_threesuperior                 0x00b3  /* U+00B3 SUPERSCRIPT THREE */
#define XKB_KEY_acute                         0x00b4  /* U+00B4 ACUTE ACCENT */
#define XKB_KEY_mu                            0x00b5  /* U+00B5 MICRO SIGN */
#define XKB_KEY_paragraph                     0x00b6  /* U+00B6 PILCROW SIGN */
#define XKB_KEY_periodcentered                0x00b7  /* U+00B7 MIDDLE DOT */
#define XKB_KEY_cedilla                       0x00b8  /* U+00B8 CEDILLA */
#define XKB_KEY_onesuperior                   0x00b9  /* U+00B9 SUPERSCRIPT ONE */
#define XKB_KEY_masculine                     0x00ba  /* deprecated alias for ordmasculine (inconsistent name) */
#define XKB_KEY_ordmasculine                  0x00ba  /* U+00BA MASCULINE ORDINAL INDICATOR */
#define XKB_KEY_guillemotright                0x00bb  /* deprecated alias for guillemetright (misspelling) */
#define XKB_KEY_guillemetright                0x00bb  /* U+00BB RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK */
#define XKB_KEY_onequarter                    0x00bc  /* U+00BC VULGAR FRACTION ONE QUARTER */
#define XKB_KEY_onehalf                       0x00bd  /* U+00BD VULGAR FRACTION ONE HALF */
#define XKB_KEY_threequarters                 0x00be  /* U+00BE VULGAR FRACTION THREE QUARTERS */
#define XKB_KEY_questiondown                  0x00bf  /* U+00BF INVERTED QUESTION MARK */
#define XKB_KEY_Agrave                        0x00c0  /* U+00C0 LATIN CAPITAL LETTER A WITH GRAVE */
#define XKB_KEY_Aacute                        0x00c1  /* U+00C1 LATIN CAPITAL LETTER A WITH ACUTE */
#define XKB_KEY_Acircumflex                   0x00c2  /* U+00C2 LATIN CAPITAL LETTER A WITH CIRCUMFLEX */
#define XKB_KEY_Atilde                        0x00c3  /* U+00C3 LATIN CAPITAL LETTER A WITH TILDE */
#define XKB_KEY_Adiaeresis                    0x00c4  /* U+00C4 LATIN CAPITAL LETTER A WITH DIAERESIS */
#define XKB_KEY_Aring                         0x00c5  /* U+00C5 LATIN CAPITAL LETTER A WITH RING ABOVE */
#define XKB_KEY_AE                            0x00c6  /* U+00C6 LATIN CAPITAL LETTER AE */
#define XKB_KEY_Ccedilla                      0x00c7  /* U+00C7 LATIN CAPITAL LETTER C WITH CEDILLA */
#define XKB_KEY_Egrave                        0x00c8  /* U+00C8 LATIN CAPITAL LETTER E WITH GRAVE */
#define XKB_KEY_Eacute                        0x00c9  /* U+00C9 LATIN CAPITAL LETTER E WITH ACUTE */
#define XKB_KEY_Ecircumflex                   0x00ca  /* U+00CA LATIN CAPITAL LETTER E WITH CIRCUMFLEX */
#define XKB_KEY_Ediaeresis                    0x00cb  /* U+00CB LATIN CAPITAL LETTER E WITH DIAERESIS */
#define XKB_KEY_Igrave                        0x00cc  /* U+00CC LATIN CAPITAL LETTER I WITH GRAVE */
#define XKB_KEY_Iacute                        0x00cd  /* U+00CD LATIN CAPITAL LETTER I WITH ACUTE */
#define XKB_KEY_Icircumflex                   0x00ce  /* U+00CE LATIN CAPITAL LETTER I WITH CIRCUMFLEX */
#define XKB_KEY_Idiaeresis                    0x00cf  /* U+00CF LATIN CAPITAL LETTER I WITH DIAERESIS */
#define XKB_KEY_ETH                           0x00d0  /* U+00D0 LATIN CAPITAL LETTER ETH */
#define XKB_KEY_Eth                           0x00d0  /* deprecated */
#define XKB_KEY_Ntilde                        0x00d1  /* U+00D1 LATIN CAPITAL LETTER N WITH TILDE */
#define XKB_KEY_Ograve                        0x00d2  /* U+00D2 LATIN CAPITAL LETTER O WITH GRAVE */
#define XKB_KEY_Oacute                        0x00d3  /* U+00D3 LATIN CAPITAL LETTER O WITH ACUTE */
#define XKB_KEY_Ocircumflex                   0x00d4  /* U+00D4 LATIN CAPITAL LETTER O WITH CIRCUMFLEX */
#define XKB_KEY_Otilde                        0x00d5  /* U+00D5 LATIN CAPITAL LETTER O WITH TILDE */
#define XKB_KEY_Odiaeresis                    0x00d6  /* U+00D6 LATIN CAPITAL LETTER O WITH DIAERESIS */
#define XKB_KEY_multiply                      0x00d7  /* U+00D7 MULTIPLICATION SIGN */
#define XKB_KEY_Oslash                        0x00d8  /* U+00D8 LATIN CAPITAL LETTER O WITH STROKE */
#define XKB_KEY_Ooblique                      0x00d8  /* deprecated alias for Oslash */
#define XKB_KEY_Ugrave                        0x00d9  /* U+00D9 LATIN CAPITAL LETTER U WITH GRAVE */
#define XKB_KEY_Uacute                        0x00da  /* U+00DA LATIN CAPITAL LETTER U WITH ACUTE */
#define XKB_KEY_Ucircumflex                   0x00db  /* U+00DB LATIN CAPITAL LETTER U WITH CIRCUMFLEX */
#define XKB_KEY_Udiaeresis                    0x00dc  /* U+00DC LATIN CAPITAL LETTER U WITH DIAERESIS */
#define XKB_KEY_Yacute                        0x00dd  /* U+00DD LATIN CAPITAL LETTER Y WITH ACUTE */
#define XKB_KEY_THORN                         0x00de  /* U+00DE LATIN CAPITAL LETTER THORN */
#define XKB_KEY_Thorn                         0x00de  /* deprecated */
#define XKB_KEY_ssharp                        0x00df  /* U+00DF LATIN SMALL LETTER SHARP S */
#define XKB_KEY_agrave                        0x00e0  /* U+00E0 LATIN SMALL LETTER A WITH GRAVE */
#define XKB_KEY_aacute                        0x00e1  /* U+00E1 LATIN SMALL LETTER A WITH ACUTE */
#define XKB_KEY_acircumflex                   0x00e2  /* U+00E2 LATIN SMALL LETTER A WITH CIRCUMFLEX */
#define XKB_KEY_atilde                        0x00e3  /* U+00E3 LATIN SMALL LETTER A WITH TILDE */
#define XKB_KEY_adiaeresis                    0x00e4  /* U+00E4 LATIN SMALL LETTER A WITH DIAERESIS */
#define XKB_KEY_aring                         0x00e5  /* U+00E5 LATIN SMALL LETTER A WITH RING ABOVE */
#define XKB_KEY_ae                            0x00e6  /* U+00E6 LATIN SMALL LETTER AE */
#define XKB_KEY_ccedilla                      0x00e7  /* U+00E7 LATIN SMALL LETTER C WITH CEDILLA */
#define XKB_KEY_egrave                        0x00e8  /* U+00E8 LATIN SMALL LETTER E WITH GRAVE */
#define XKB_KEY_eacute                        0x00e9  /* U+00E9 LATIN SMALL LETTER E WITH ACUTE */
#define XKB_KEY_ecircumflex                   0x00ea  /* U+00EA LATIN SMALL LETTER E WITH CIRCUMFLEX */
#define XKB_KEY_ediaeresis                    0x00eb  /* U+00EB LATIN SMALL LETTER E WITH DIAERESIS */
#define XKB_KEY_igrave                        0x00ec  /* U+00EC LATIN SMALL LETTER I WITH GRAVE */
#define XKB_KEY_iacute                        0x00ed  /* U+00ED LATIN SMALL LETTER I WITH ACUTE */
#define XKB_KEY_icircumflex                   0x00ee  /* U+00EE LATIN SMALL LETTER I WITH CIRCUMFLEX */
#define XKB_KEY_idiaeresis                    0x00ef  /* U+00EF LATIN SMALL LETTER I WITH DIAERESIS */
#define XKB_KEY_eth                           0x00f0  /* U+00F0 LATIN SMALL LETTER ETH */
#define XKB_KEY_ntilde                        0x00f1  /* U+00F1 LATIN SMALL LETTER N WITH TILDE */
#define XKB_KEY_ograve                        0x00f2  /* U+00F2 LATIN SMALL LETTER O WITH GRAVE */
#define XKB_KEY_oacute                        0x00f3  /* U+00F3 LATIN SMALL LETTER O WITH ACUTE */
#define XKB_KEY_ocircumflex                   0x00f4  /* U+00F4 LATIN SMALL LETTER O WITH CIRCUMFLEX */
#define XKB_KEY_otilde                        0x00f5  /* U+00F5 LATIN SMALL LETTER O WITH TILDE */
#define XKB_KEY_odiaeresis                    0x00f6  /* U+00F6 LATIN SMALL LETTER O WITH DIAERESIS */
#define XKB_KEY_division                      0x00f7  /* U+00F7 DIVISION SIGN */
#define XKB_KEY_oslash                        0x00f8  /* U+00F8 LATIN SMALL LETTER O WITH STROKE */
#define XKB_KEY_ooblique                      0x00f8  /* deprecated alias for oslash */
#define XKB_KEY_ugrave                        0x00f9  /* U+00F9 LATIN SMALL LETTER U WITH GRAVE */
#define XKB_KEY_uacute                        0x00fa  /* U+00FA LATIN SMALL LETTER U WITH ACUTE */
#define XKB_KEY_ucircumflex                   0x00fb  /* U+00FB LATIN SMALL LETTER U WITH CIRCUMFLEX */
#define XKB_KEY_udiaeresis                    0x00fc  /* U+00FC LATIN SMALL LETTER U WITH DIAERESIS */
#define XKB_KEY_yacute                        0x00fd  /* U+00FD LATIN SMALL LETTER Y WITH ACUTE */
#define XKB_KEY_thorn                         0x00fe  /* U+00FE LATIN SMALL LETTER THORN */
#define XKB_KEY_ydiaeresis                    0x00ff  /* U+00FF LATIN SMALL LETTER Y WITH DIAERESIS */

/*
 * Latin 2
 * Byte 3 = 1
 */

#define XKB_KEY_Aogonek                       0x01a1  /* U+0104 LATIN CAPITAL LETTER A WITH OGONEK */
#define XKB_KEY_breve                         0x01a2  /* U+02D8 BREVE */
#define XKB_KEY_Lstroke                       0x01a3  /* U+0141 LATIN CAPITAL LETTER L WITH STROKE */
#define XKB_KEY_Lcaron                        0x01a5  /* U+013D LATIN CAPITAL LETTER L WITH CARON */
#define XKB_KEY_Sacute                        0x01a6  /* U+015A LATIN CAPITAL LETTER S WITH ACUTE */
#define XKB_KEY_Scaron                        0x01a9  /* U+0160 LATIN CAPITAL LETTER S WITH CARON */
#define XKB_KEY_Scedilla                      0x01aa  /* U+015E LATIN CAPITAL LETTER S WITH CEDILLA */
#define XKB_KEY_Tcaron                        0x01ab  /* U+0164 LATIN CAPITAL LETTER T WITH CARON */
#define XKB_KEY_Zacute                        0x01ac  /* U+0179 LATIN CAPITAL LETTER Z WITH ACUTE */
#define XKB_KEY_Zcaron                        0x01ae  /* U+017D LATIN CAPITAL LETTER Z WITH CARON */
#define XKB_KEY_Zabovedot                     0x01af  /* U+017B LATIN CAPITAL LETTER Z WITH DOT ABOVE */
#define XKB_KEY_aogonek                       0x01b1  /* U+0105 LATIN SMALL LETTER A WITH OGONEK */
#define XKB_KEY_ogonek                        0x01b2  /* U+02DB OGONEK */
#define XKB_KEY_lstroke                       0x01b3  /* U+0142 LATIN SMALL LETTER L WITH STROKE */
#define XKB_KEY_lcaron                        0x01b5  /* U+013E LATIN SMALL LETTER L WITH CARON */
#define XKB_KEY_sacute                        0x01b6  /* U+015B LATIN SMALL LETTER S WITH ACUTE */
#define XKB_KEY_caron                         0x01b7  /* U+02C7 CARON */
#define XKB_KEY_scaron                        0x01b9  /* U+0161 LATIN SMALL LETTER S WITH CARON */
#define XKB_KEY_scedilla                      0x01ba  /* U+015F LATIN SMALL LETTER S WITH CEDILLA */
#define XKB_KEY_tcaron                        0x01bb  /* U+0165 LATIN SMALL LETTER T WITH CARON */
#define XKB_KEY_zacute                        0x01bc  /* U+017A LATIN SMALL LETTER Z WITH ACUTE */
#define XKB_KEY_doubleacute                   0x01bd  /* U+02DD DOUBLE ACUTE ACCENT */
#define XKB_KEY_zcaron                        0x01be  /* U+017E LATIN SMALL LETTER Z WITH CARON */
#define XKB_KEY_zabovedot                     0x01bf  /* U+017C LATIN SMALL LETTER Z WITH DOT ABOVE */
#define XKB_KEY_Racute                        0x01c0  /* U+0154 LATIN CAPITAL LETTER R WITH ACUTE */
#define XKB_KEY_Abreve                        0x01c3  /* U+0102 LATIN CAPITAL LETTER A WITH BREVE */
#define XKB_KEY_Lacute                        0x01c5  /* U+0139 LATIN CAPITAL LETTER L WITH ACUTE */
#define XKB_KEY_Cacute                        0x01c6  /* U+0106 LATIN CAPITAL LETTER C WITH ACUTE */
#define XKB_KEY_Ccaron                        0x01c8  /* U+010C LATIN CAPITAL LETTER C WITH CARON */
#define XKB_KEY_Eogonek                       0x01ca  /* U+0118 LATIN CAPITAL LETTER E WITH OGONEK */
#define XKB_KEY_Ecaron                        0x01cc  /* U+011A LATIN CAPITAL LETTER E WITH CARON */
#define XKB_KEY_Dcaron                        0x01cf  /* U+010E LATIN CAPITAL LETTER D WITH CARON */
#define XKB_KEY_Dstroke                       0x01d0  /* U+0110 LATIN CAPITAL LETTER D WITH STROKE */
#define XKB_KEY_Nacute                        0x01d1  /* U+0143 LATIN CAPITAL LETTER N WITH ACUTE */
#define XKB_KEY_Ncaron                        0x01d2  /* U+0147 LATIN CAPITAL LETTER N WITH CARON */
#define XKB_KEY_Odoubleacute                  0x01d5  /* U+0150 LATIN CAPITAL LETTER O WITH DOUBLE ACUTE */
#define XKB_KEY_Rcaron                        0x01d8  /* U+0158 LATIN CAPITAL LETTER R WITH CARON */
#define XKB_KEY_Uring                         0x01d9  /* U+016E LATIN CAPITAL LETTER U WITH RING ABOVE */
#define XKB_KEY_Udoubleacute                  0x01db  /* U+0170 LATIN CAPITAL LETTER U WITH DOUBLE ACUTE */
#define XKB_KEY_Tcedilla                      0x01de  /* U+0162 LATIN CAPITAL LETTER T WITH CEDILLA */
#define XKB_KEY_racute                        0x01e0  /* U+0155 LATIN SMALL LETTER R WITH ACUTE */
#define XKB_KEY_abreve                        0x01e3  /* U+0103 LATIN SMALL LETTER A WITH BREVE */
#define XKB_KEY_lacute                        0x01e5  /* U+013A LATIN SMALL LETTER L WITH ACUTE */
#define XKB_KEY_cacute                        0x01e6  /* U+0107 LATIN SMALL LETTER C WITH ACUTE */
#define XKB_KEY_ccaron                        0x01e8  /* U+010D LATIN SMALL LETTER C WITH CARON */
#define XKB_KEY_eogonek                       0x01ea  /* U+0119 LATIN SMALL LETTER E WITH OGONEK */
#define XKB_KEY_ecaron                        0x01ec  /* U+011B LATIN SMALL LETTER E WITH CARON */
#define XKB_KEY_dcaron                        0x01ef  /* U+010F LATIN SMALL LETTER D WITH CARON */
#define XKB_KEY_dstroke                       0x01f0  /* U+0111 LATIN SMALL LETTER D WITH STROKE */
#define XKB_KEY_nacute                        0x01f1  /* U+0144 LATIN SMALL LETTER N WITH ACUTE */
#define XKB_KEY_ncaron                        0x01f2  /* U+0148 LATIN SMALL LETTER N WITH CARON */
#define XKB_KEY_odoubleacute                  0x01f5  /* U+0151 LATIN SMALL LETTER O WITH DOUBLE ACUTE */
#define XKB_KEY_rcaron                        0x01f8  /* U+0159 LATIN SMALL LETTER R WITH CARON */
#define XKB_KEY_uring                         0x01f9  /* U+016F LATIN SMALL LETTER U WITH RING ABOVE */
#define XKB_KEY_udoubleacute                  0x01fb  /* U+0171 LATIN SMALL LETTER U WITH DOUBLE ACUTE */
#define XKB_KEY_tcedilla                      0x01fe  /* U+0163 LATIN SMALL LETTER T WITH CEDILLA */
#define XKB_KEY_abovedot                      0x01ff  /* U+02D9 DOT ABOVE */

/*
 * Latin 3
 * Byte 3 = 2
 */

#define XKB_KEY_Hstroke                       0x02a1  /* U+0126 LATIN CAPITAL LETTER H WITH STROKE */
#define XKB_KEY_Hcircumflex                   0x02a6  /* U+0124 LATIN CAPITAL LETTER H WITH CIRCUMFLEX */
#define XKB_KEY_Iabovedot                     0x02a9  /* U+0130 LATIN CAPITAL LETTER I WITH DOT ABOVE */
#define XKB_KEY_Gbreve                        0x02ab  /* U+011E LATIN CAPITAL LETTER G WITH BREVE */
#define XKB_KEY_Jcircumflex                   0x02ac  /* U+0134 LATIN CAPITAL LETTER J WITH CIRCUMFLEX */
#define XKB_KEY_hstroke                       0x02b1  /* U+0127 LATIN SMALL LETTER H WITH STROKE */
#define XKB_KEY_hcircumflex                   0x02b6  /* U+0125 LATIN SMALL LETTER H WITH CIRCUMFLEX */
#define XKB_KEY_idotless                      0x02b9  /* U+0131 LATIN SMALL LETTER DOTLESS I */
#define XKB_KEY_gbreve                        0x02bb  /* U+011F LATIN SMALL LETTER G WITH BREVE */
#define XKB_KEY_jcircumflex                   0x02bc  /* U+0135 LATIN SMALL LETTER J WITH CIRCUMFLEX */
#define XKB_KEY_Cabovedot                     0x02c5  /* U+010A LATIN CAPITAL LETTER C WITH DOT ABOVE */
#define XKB_KEY_Ccircumflex                   0x02c6  /* U+0108 LATIN CAPITAL LETTER C WITH CIRCUMFLEX */
#define XKB_KEY_Gabovedot                     0x02d5  /* U+0120 LATIN CAPITAL LETTER G WITH DOT ABOVE */
#define XKB_KEY_Gcircumflex                   0x02d8  /* U+011C LATIN CAPITAL LETTER G WITH CIRCUMFLEX */
#define XKB_KEY_Ubreve                        0x02dd  /* U+016C LATIN CAPITAL LETTER U WITH BREVE */
#define XKB_KEY_Scircumflex                   0x02de  /* U+015C LATIN CAPITAL LETTER S WITH CIRCUMFLEX */
#define XKB_KEY_cabovedot                     0x02e5  /* U+010B LATIN SMALL LETTER C WITH DOT ABOVE */
#define XKB_KEY_ccircumflex                   0x02e6  /* U+0109 LATIN SMALL LETTER C WITH CIRCUMFLEX */
#define XKB_KEY_gabovedot                     0x02f5  /* U+0121 LATIN SMALL LETTER G WITH DOT ABOVE */
#define XKB_KEY_gcircumflex                   0x02f8  /* U+011D LATIN SMALL LETTER G WITH CIRCUMFLEX */
#define XKB_KEY_ubreve                        0x02fd  /* U+016D LATIN SMALL LETTER U WITH BREVE */
#define XKB_KEY_scircumflex                   0x02fe  /* U+015D LATIN SMALL LETTER S WITH CIRCUMFLEX */


/*
 * Latin 4
 * Byte 3 = 3
 */

#define XKB_KEY_kra                           0x03a2  /* U+0138 LATIN SMALL LETTER KRA */
#define XKB_KEY_kappa                         0x03a2  /* deprecated */
#define XKB_KEY_Rcedilla                      0x03a3  /* U+0156 LATIN CAPITAL LETTER R WITH CEDILLA */
#define XKB_KEY_Itilde                        0x03a5  /* U+0128 LATIN CAPITAL LETTER I WITH TILDE */
#define XKB_KEY_Lcedilla                      0x03a6  /* U+013B LATIN CAPITAL LETTER L WITH CEDILLA */
#define XKB_KEY_Emacron                       0x03aa  /* U+0112 LATIN CAPITAL LETTER E WITH MACRON */
#define XKB_KEY_Gcedilla                      0x03ab  /* U+0122 LATIN CAPITAL LETTER G WITH CEDILLA */
#define XKB_KEY_Tslash                        0x03ac  /* U+0166 LATIN CAPITAL LETTER T WITH STROKE */
#define XKB_KEY_rcedilla                      0x03b3  /* U+0157 LATIN SMALL LETTER R WITH CEDILLA */
#define XKB_KEY_itilde                        0x03b5  /* U+0129 LATIN SMALL LETTER I WITH TILDE */
#define XKB_KEY_lcedilla                      0x03b6  /* U+013C LATIN SMALL LETTER L WITH CEDILLA */
#define XKB_KEY_emacron                       0x03ba  /* U+0113 LATIN SMALL LETTER E WITH MACRON */
#define XKB_KEY_gcedilla                      0x03bb  /* U+0123 LATIN SMALL LETTER G WITH CEDILLA */
#define XKB_KEY_tslash                        0x03bc  /* U+0167 LATIN SMALL LETTER T WITH STROKE */
#define XKB_KEY_ENG                           0x03bd  /* U+014A LATIN CAPITAL LETTER ENG */
#define XKB_KEY_eng                           0x03bf  /* U+014B LATIN SMALL LETTER ENG */
#define XKB_KEY_Amacron                       0x03c0  /* U+0100 LATIN CAPITAL LETTER A WITH MACRON */
#define XKB_KEY_Iogonek                       0x03c7  /* U+012E LATIN CAPITAL LETTER I WITH OGONEK */
#define XKB_KEY_Eabovedot                     0x03cc  /* U+0116 LATIN CAPITAL LETTER E WITH DOT ABOVE */
#define XKB_KEY_Imacron                       0x03cf  /* U+012A LATIN CAPITAL LETTER I WITH MACRON */
#define XKB_KEY_Ncedilla                      0x03d1  /* U+0145 LATIN CAPITAL LETTER N WITH CEDILLA */
#define XKB_KEY_Omacron                       0x03d2  /* U+014C LATIN CAPITAL LETTER O WITH MACRON */
#define XKB_KEY_Kcedilla                      0x03d3  /* U+0136 LATIN CAPITAL LETTER K WITH CEDILLA */
#define XKB_KEY_Uogonek                       0x03d9  /* U+0172 LATIN CAPITAL LETTER U WITH OGONEK */
#define XKB_KEY_Utilde                        0x03dd  /* U+0168 LATIN CAPITAL LETTER U WITH TILDE */
#define XKB_KEY_Umacron                       0x03de  /* U+016A LATIN CAPITAL LETTER U WITH MACRON */
#define XKB_KEY_amacron                       0x03e0  /* U+0101 LATIN SMALL LETTER A WITH MACRON */
#define XKB_KEY_iogonek                       0x03e7  /* U+012F LATIN SMALL LETTER I WITH OGONEK */
#define XKB_KEY_eabovedot                     0x03ec  /* U+0117 LATIN SMALL LETTER E WITH DOT ABOVE */
#define XKB_KEY_imacron                       0x03ef  /* U+012B LATIN SMALL LETTER I WITH MACRON */
#define XKB_KEY_ncedilla                      0x03f1  /* U+0146 LATIN SMALL LETTER N WITH CEDILLA */
#define XKB_KEY_omacron                       0x03f2  /* U+014D LATIN SMALL LETTER O WITH MACRON */
#define XKB_KEY_kcedilla                      0x03f3  /* U+0137 LATIN SMALL LETTER K WITH CEDILLA */
#define XKB_KEY_uogonek                       0x03f9  /* U+0173 LATIN SMALL LETTER U WITH OGONEK */
#define XKB_KEY_utilde                        0x03fd  /* U+0169 LATIN SMALL LETTER U WITH TILDE */
#define XKB_KEY_umacron                       0x03fe  /* U+016B LATIN SMALL LETTER U WITH MACRON */

/*
 * Latin 8
 */
#define XKB_KEY_Wcircumflex                0x1000174  /* U+0174 LATIN CAPITAL LETTER W WITH CIRCUMFLEX */
#define XKB_KEY_wcircumflex                0x1000175  /* U+0175 LATIN SMALL LETTER W WITH CIRCUMFLEX */
#define XKB_KEY_Ycircumflex                0x1000176  /* U+0176 LATIN CAPITAL LETTER Y WITH CIRCUMFLEX */
#define XKB_KEY_ycircumflex                0x1000177  /* U+0177 LATIN SMALL LETTER Y WITH CIRCUMFLEX */
#define XKB_KEY_Babovedot                  0x1001e02  /* U+1E02 LATIN CAPITAL LETTER B WITH DOT ABOVE */
#define XKB_KEY_babovedot                  0x1001e03  /* U+1E03 LATIN SMALL LETTER B WITH DOT ABOVE */
#define XKB_KEY_Dabovedot                  0x1001e0a  /* U+1E0A LATIN CAPITAL LETTER D WITH DOT ABOVE */
#define XKB_KEY_dabovedot                  0x1001e0b  /* U+1E0B LATIN SMALL LETTER D WITH DOT ABOVE */
#define XKB_KEY_Fabovedot                  0x1001e1e  /* U+1E1E LATIN CAPITAL LETTER F WITH DOT ABOVE */
#define XKB_KEY_fabovedot                  0x1001e1f  /* U+1E1F LATIN SMALL LETTER F WITH DOT ABOVE */
#define XKB_KEY_Mabovedot                  0x1001e40  /* U+1E40 LATIN CAPITAL LETTER M WITH DOT ABOVE */
#define XKB_KEY_mabovedot                  0x1001e41  /* U+1E41 LATIN SMALL LETTER M WITH DOT ABOVE */
#define XKB_KEY_Pabovedot                  0x1001e56  /* U+1E56 LATIN CAPITAL LETTER P WITH DOT ABOVE */
#define XKB_KEY_pabovedot                  0x1001e57  /* U+1E57 LATIN SMALL LETTER P WITH DOT ABOVE */
#define XKB_KEY_Sabovedot                  0x1001e60  /* U+1E60 LATIN CAPITAL LETTER S WITH DOT ABOVE */
#define XKB_KEY_sabovedot                  0x1001e61  /* U+1E61 LATIN SMALL LETTER S WITH DOT ABOVE */
#define XKB_KEY_Tabovedot                  0x1001e6a  /* U+1E6A LATIN CAPITAL LETTER T WITH DOT ABOVE */
#define XKB_KEY_tabovedot                  0x1001e6b  /* U+1E6B LATIN SMALL LETTER T WITH DOT ABOVE */
#define XKB_KEY_Wgrave                     0x1001e80  /* U+1E80 LATIN CAPITAL LETTER W WITH GRAVE */
#define XKB_KEY_wgrave                     0x1001e81  /* U+1E81 LATIN SMALL LETTER W WITH GRAVE */
#define XKB_KEY_Wacute                     0x1001e82  /* U+1E82 LATIN CAPITAL LETTER W WITH ACUTE */
#define XKB_KEY_wacute                     0x1001e83  /* U+1E83 LATIN SMALL LETTER W WITH ACUTE */
#define XKB_KEY_Wdiaeresis                 0x1001e84  /* U+1E84 LATIN CAPITAL LETTER W WITH DIAERESIS */
#define XKB_KEY_wdiaeresis                 0x1001e85  /* U+1E85 LATIN SMALL LETTER W WITH DIAERESIS */
#define XKB_KEY_Ygrave                     0x1001ef2  /* U+1EF2 LATIN CAPITAL LETTER Y WITH GRAVE */
#define XKB_KEY_ygrave                     0x1001ef3  /* U+1EF3 LATIN SMALL LETTER Y WITH GRAVE */

/*
 * Latin 9
 * Byte 3 = 0x13
 */

#define XKB_KEY_OE                            0x13bc  /* U+0152 LATIN CAPITAL LIGATURE OE */
#define XKB_KEY_oe                            0x13bd  /* U+0153 LATIN SMALL LIGATURE OE */
#define XKB_KEY_Ydiaeresis                    0x13be  /* U+0178 LATIN CAPITAL LETTER Y WITH DIAERESIS */

/*
 * Katakana
 * Byte 3 = 4
 */

#define XKB_KEY_overline                      0x047e  /* U+203E OVERLINE */
#define XKB_KEY_kana_fullstop                 0x04a1  /* U+3002 IDEOGRAPHIC FULL STOP */
#define XKB_KEY_kana_openingbracket           0x04a2  /* U+300C LEFT CORNER BRACKET */
#define XKB_KEY_kana_closingbracket           0x04a3  /* U+300D RIGHT CORNER BRACKET */
#define XKB_KEY_kana_comma                    0x04a4  /* U+3001 IDEOGRAPHIC COMMA */
#define XKB_KEY_kana_conjunctive              0x04a5  /* U+30FB KATAKANA MIDDLE DOT */
#define XKB_KEY_kana_middledot                0x04a5  /* deprecated */
#define XKB_KEY_kana_WO                       0x04a6  /* U+30F2 KATAKANA LETTER WO */
#define XKB_KEY_kana_a                        0x04a7  /* U+30A1 KATAKANA LETTER SMALL A */
#define XKB_KEY_kana_i                        0x04a8  /* U+30A3 KATAKANA LETTER SMALL I */
#define XKB_KEY_kana_u                        0x04a9  /* U+30A5 KATAKANA LETTER SMALL U */
#define XKB_KEY_kana_e                        0x04aa  /* U+30A7 KATAKANA LETTER SMALL E */
#define XKB_KEY_kana_o                        0x04ab  /* U+30A9 KATAKANA LETTER SMALL O */
#define XKB_KEY_kana_ya                       0x04ac  /* U+30E3 KATAKANA LETTER SMALL YA */
#define XKB_KEY_kana_yu                       0x04ad  /* U+30E5 KATAKANA LETTER SMALL YU */
#define XKB_KEY_kana_yo                       0x04ae  /* U+30E7 KATAKANA LETTER SMALL YO */
#define XKB_KEY_kana_tsu                      0x04af  /* U+30C3 KATAKANA LETTER SMALL TU */
#define XKB_KEY_kana_tu                       0x04af  /* deprecated */
#define XKB_KEY_prolongedsound                0x04b0  /* U+30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK */
#define XKB_KEY_kana_A                        0x04b1  /* U+30A2 KATAKANA LETTER A */
#define XKB_KEY_kana_I                        0x04b2  /* U+30A4 KATAKANA LETTER I */
#define XKB_KEY_kana_U                        0x04b3  /* U+30A6 KATAKANA LETTER U */
#define XKB_KEY_kana_E                        0x04b4  /* U+30A8 KATAKANA LETTER E */
#define XKB_KEY_kana_O                        0x04b5  /* U+30AA KATAKANA LETTER O */
#define XKB_KEY_kana_KA                       0x04b6  /* U+30AB KATAKANA LETTER KA */
#define XKB_KEY_kana_KI                       0x04b7  /* U+30AD KATAKANA LETTER KI */
#define XKB_KEY_kana_KU                       0x04b8  /* U+30AF KATAKANA LETTER KU */
#define XKB_KEY_kana_KE                       0x04b9  /* U+30B1 KATAKANA LETTER KE */
#define XKB_KEY_kana_KO                       0x04ba  /* U+30B3 KATAKANA LETTER KO */
#define XKB_KEY_kana_SA                       0x04bb  /* U+30B5 KATAKANA LETTER SA */
#define XKB_KEY_kana_SHI                      0x04bc  /* U+30B7 KATAKANA LETTER SI */
#define XKB_KEY_kana_SU                       0x04bd  /* U+30B9 KATAKANA LETTER SU */
#define XKB_KEY_kana_SE                       0x04be  /* U+30BB KATAKANA LETTER SE */
#define XKB_KEY_kana_SO                       0x04bf  /* U+30BD KATAKANA LETTER SO */
#define XKB_KEY_kana_TA                       0x04c0  /* U+30BF KATAKANA LETTER TA */
#define XKB_KEY_kana_CHI                      0x04c1  /* U+30C1 KATAKANA LETTER TI */
#define XKB_KEY_kana_TI                       0x04c1  /* deprecated */
#define XKB_KEY_kana_TSU                      0x04c2  /* U+30C4 KATAKANA LETTER TU */
#define XKB_KEY_kana_TU                       0x04c2  /* deprecated */
#define XKB_KEY_kana_TE                       0x04c3  /* U+30C6 KATAKANA LETTER TE */
#define XKB_KEY_kana_TO                       0x04c4  /* U+30C8 KATAKANA LETTER TO */
#define XKB_KEY_kana_NA                       0x04c5  /* U+30CA KATAKANA LETTER NA */
#define XKB_KEY_kana_NI                       0x04c6  /* U+30CB KATAKANA LETTER NI */
#define XKB_KEY_kana_NU                       0x04c7  /* U+30CC KATAKANA LETTER NU */
#define XKB_KEY_kana_NE                       0x04c8  /* U+30CD KATAKANA LETTER NE */
#define XKB_KEY_kana_NO                       0x04c9  /* U+30CE KATAKANA LETTER NO */
#define XKB_KEY_kana_HA                       0x04ca  /* U+30CF KATAKANA LETTER HA */
#define XKB_KEY_kana_HI                       0x04cb  /* U+30D2 KATAKANA LETTER HI */
#define XKB_KEY_kana_FU                       0x04cc  /* U+30D5 KATAKANA LETTER HU */
#define XKB_KEY_kana_HU                       0x04cc  /* deprecated */
#define XKB_KEY_kana_HE                       0x04cd  /* U+30D8 KATAKANA LETTER HE */
#define XKB_KEY_kana_HO                       0x04ce  /* U+30DB KATAKANA LETTER HO */
#define XKB_KEY_kana_MA                       0x04cf  /* U+30DE KATAKANA LETTER MA */
#define XKB_KEY_kana_MI                       0x04d0  /* U+30DF KATAKANA LETTER MI */
#define XKB_KEY_kana_MU                       0x04d1  /* U+30E0 KATAKANA LETTER MU */
#define XKB_KEY_kana_ME                       0x04d2  /* U+30E1 KATAKANA LETTER ME */
#define XKB_KEY_kana_MO                       0x04d3  /* U+30E2 KATAKANA LETTER MO */
#define XKB_KEY_kana_YA                       0x04d4  /* U+30E4 KATAKANA LETTER YA */
#define XKB_KEY_kana_YU                       0x04d5  /* U+30E6 KATAKANA LETTER YU */
#define XKB_KEY_kana_YO                       0x04d6  /* U+30E8 KATAKANA LETTER YO */
#define XKB_KEY_kana_RA                       0x04d7  /* U+30E9 KATAKANA LETTER RA */
#define XKB_KEY_kana_RI                       0x04d8  /* U+30EA KATAKANA LETTER RI */
#define XKB_KEY_kana_RU                       0x04d9  /* U+30EB KATAKANA LETTER RU */
#define XKB_KEY_kana_RE                       0x04da  /* U+30EC KATAKANA LETTER RE */
#define XKB_KEY_kana_RO                       0x04db  /* U+30ED KATAKANA LETTER RO */
#define XKB_KEY_kana_WA                       0x04dc  /* U+30EF KATAKANA LETTER WA */
#define XKB_KEY_kana_N                        0x04dd  /* U+30F3 KATAKANA LETTER N */
#define XKB_KEY_voicedsound                   0x04de  /* U+309B KATAKANA-HIRAGANA VOICED SOUND MARK */
#define XKB_KEY_semivoicedsound               0x04df  /* U+309C KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK */
#define XKB_KEY_kana_switch                   0xff7e  /* non-deprecated alias for Mode_switch */

/*
 * Arabic
 * Byte 3 = 5
 */

#define XKB_KEY_Farsi_0                    0x10006f0  /* U+06F0 EXTENDED ARABIC-INDIC DIGIT ZERO */
#define XKB_KEY_Farsi_1                    0x10006f1  /* U+06F1 EXTENDED ARABIC-INDIC DIGIT ONE */
#define XKB_KEY_Farsi_2                    0x10006f2  /* U+06F2 EXTENDED ARABIC-INDIC DIGIT TWO */
#define XKB_KEY_Farsi_3                    0x10006f3  /* U+06F3 EXTENDED ARABIC-INDIC DIGIT THREE */
#define XKB_KEY_Farsi_4                    0x10006f4  /* U+06F4 EXTENDED ARABIC-INDIC DIGIT FOUR */
#define XKB_KEY_Farsi_5                    0x10006f5  /* U+06F5 EXTENDED ARABIC-INDIC DIGIT FIVE */
#define XKB_KEY_Farsi_6                    0x10006f6  /* U+06F6 EXTENDED ARABIC-INDIC DIGIT SIX */
#define XKB_KEY_Farsi_7                    0x10006f7  /* U+06F7 EXTENDED ARABIC-INDIC DIGIT SEVEN */
#define XKB_KEY_Farsi_8                    0x10006f8  /* U+06F8 EXTENDED ARABIC-INDIC DIGIT EIGHT */
#define XKB_KEY_Farsi_9                    0x10006f9  /* U+06F9 EXTENDED ARABIC-INDIC DIGIT NINE */
#define XKB_KEY_Arabic_percent             0x100066a  /* U+066A ARABIC PERCENT SIGN */
#define XKB_KEY_Arabic_superscript_alef    0x1000670  /* U+0670 ARABIC LETTER SUPERSCRIPT ALEF */
#define XKB_KEY_Arabic_tteh                0x1000679  /* U+0679 ARABIC LETTER TTEH */
#define XKB_KEY_Arabic_peh                 0x100067e  /* U+067E ARABIC LETTER PEH */
#define XKB_KEY_Arabic_tcheh               0x1000686  /* U+0686 ARABIC LETTER TCHEH */
#define XKB_KEY_Arabic_ddal                0x1000688  /* U+0688 ARABIC LETTER DDAL */
#define XKB_KEY_Arabic_rreh                0x1000691  /* U+0691 ARABIC LETTER RREH */
#define XKB_KEY_Arabic_comma                  0x05ac  /* U+060C ARABIC COMMA */
#define XKB_KEY_Arabic_fullstop            0x10006d4  /* U+06D4 ARABIC FULL STOP */
#define XKB_KEY_Arabic_0                   0x1000660  /* U+0660 ARABIC-INDIC DIGIT ZERO */
#define XKB_KEY_Arabic_1                   0x1000661  /* U+0661 ARABIC-INDIC DIGIT ONE */
#define XKB_KEY_Arabic_2                   0x1000662  /* U+0662 ARABIC-INDIC DIGIT TWO */
#define XKB_KEY_Arabic_3                   0x1000663  /* U+0663 ARABIC-INDIC DIGIT THREE */
#define XKB_KEY_Arabic_4                   0x1000664  /* U+0664 ARABIC-INDIC DIGIT FOUR */
#define XKB_KEY_Arabic_5                   0x1000665  /* U+0665 ARABIC-INDIC DIGIT FIVE */
#define XKB_KEY_Arabic_6                   0x1000666  /* U+0666 ARABIC-INDIC DIGIT SIX */
#define XKB_KEY_Arabic_7                   0x1000667  /* U+0667 ARABIC-INDIC DIGIT SEVEN */
#define XKB_KEY_Arabic_8                   0x1000668  /* U+0668 ARABIC-INDIC DIGIT EIGHT */
#define XKB_KEY_Arabic_9                   0x1000669  /* U+0669 ARABIC-INDIC DIGIT NINE */
#define XKB_KEY_Arabic_semicolon              0x05bb  /* U+061B ARABIC SEMICOLON */
#define XKB_KEY_Arabic_question_mark          0x05bf  /* U+061F ARABIC QUESTION MARK */
#define XKB_KEY_Arabic_hamza                  0x05c1  /* U+0621 ARABIC LETTER HAMZA */
#define XKB_KEY_Arabic_maddaonalef            0x05c2  /* U+0622 ARABIC LETTER ALEF WITH MADDA ABOVE */
#define XKB_KEY_Arabic_hamzaonalef            0x05c3  /* U+0623 ARABIC LETTER ALEF WITH HAMZA ABOVE */
#define XKB_KEY_Arabic_hamzaonwaw             0x05c4  /* U+0624 ARABIC LETTER WAW WITH HAMZA ABOVE */
#define XKB_KEY_Arabic_hamzaunderalef         0x05c5  /* U+0625 ARABIC LETTER ALEF WITH HAMZA BELOW */
#define XKB_KEY_Arabic_hamzaonyeh             0x05c6  /* U+0626 ARABIC LETTER YEH WITH HAMZA ABOVE */
#define XKB_KEY_Arabic_alef                   0x05c7  /* U+0627 ARABIC LETTER ALEF */
#define XKB_KEY_Arabic_beh                    0x05c8  /* U+0628 ARABIC LETTER BEH */
#define XKB_KEY_Arabic_tehmarbuta             0x05c9  /* U+0629 ARABIC LETTER TEH MARBUTA */
#define XKB_KEY_Arabic_teh                    0x05ca  /* U+062A ARABIC LETTER TEH */
#define XKB_KEY_Arabic_theh                   0x05cb  /* U+062B ARABIC LETTER THEH */
#define XKB_KEY_Arabic_jeem                   0x05cc  /* U+062C ARABIC LETTER JEEM */
#define XKB_KEY_Arabic_hah                    0x05cd  /* U+062D ARABIC LETTER HAH */
#define XKB_KEY_Arabic_khah                   0x05ce  /* U+062E ARABIC LETTER KHAH */
#define XKB_KEY_Arabic_dal                    0x05cf  /* U+062F ARABIC LETTER DAL */
#define XKB_KEY_Arabic_thal                   0x05d0  /* U+0630 ARABIC LETTER THAL */
#define XKB_KEY_Arabic_ra                     0x05d1  /* U+0631 ARABIC LETTER REH */
#define XKB_KEY_Arabic_zain                   0x05d2  /* U+0632 ARABIC LETTER ZAIN */
#define XKB_KEY_Arabic_seen                   0x05d3  /* U+0633 ARABIC LETTER SEEN */
#define XKB_KEY_Arabic_sheen                  0x05d4  /* U+0634 ARABIC LETTER SHEEN */
#define XKB_KEY_Arabic_sad                    0x05d5  /* U+0635 ARABIC LETTER SAD */
#define XKB_KEY_Arabic_dad                    0x05d6  /* U+0636 ARABIC LETTER DAD */
#define XKB_KEY_Arabic_tah                    0x05d7  /* U+0637 ARABIC LETTER TAH */
#define XKB_KEY_Arabic_zah                    0x05d8  /* U+0638 ARABIC LETTER ZAH */
#define XKB_KEY_Arabic_ain                    0x05d9  /* U+0639 ARABIC LETTER AIN */
#define XKB_KEY_Arabic_ghain                  0x05da  /* U+063A ARABIC LETTER GHAIN */
#define XKB_KEY_Arabic_tatweel                0x05e0  /* U+0640 ARABIC TATWEEL */
#define XKB_KEY_Arabic_feh                    0x05e1  /* U+0641 ARABIC LETTER FEH */
#define XKB_KEY_Arabic_qaf                    0x05e2  /* U+0642 ARABIC LETTER QAF */
#define XKB_KEY_Arabic_kaf                    0x05e3  /* U+0643 ARABIC LETTER KAF */
#define XKB_KEY_Arabic_lam                    0x05e4  /* U+0644 ARABIC LETTER LAM */
#define XKB_KEY_Arabic_meem                   0x05e5  /* U+0645 ARABIC LETTER MEEM */
#define XKB_KEY_Arabic_noon                   0x05e6  /* U+0646 ARABIC LETTER NOON */
#define XKB_KEY_Arabic_ha                     0x05e7  /* U+0647 ARABIC LETTER HEH */
#define XKB_KEY_Arabic_heh                    0x05e7  /* deprecated */
#define XKB_KEY_Arabic_waw                    0x05e8  /* U+0648 ARABIC LETTER WAW */
#define XKB_KEY_Arabic_alefmaksura            0x05e9  /* U+0649 ARABIC LETTER ALEF MAKSURA */
#define XKB_KEY_Arabic_yeh                    0x05ea  /* U+064A ARABIC LETTER YEH */
#define XKB_KEY_Arabic_fathatan               0x05eb  /* U+064B ARABIC FATHATAN */
#define XKB_KEY_Arabic_dammatan               0x05ec  /* U+064C ARABIC DAMMATAN */
#define XKB_KEY_Arabic_kasratan               0x05ed  /* U+064D ARABIC KASRATAN */
#define XKB_KEY_Arabic_fatha                  0x05ee  /* U+064E ARABIC FATHA */
#define XKB_KEY_Arabic_damma                  0x05ef  /* U+064F ARABIC DAMMA */
#define XKB_KEY_Arabic_kasra                  0x05f0  /* U+0650 ARABIC KASRA */
#define XKB_KEY_Arabic_shadda                 0x05f1  /* U+0651 ARABIC SHADDA */
#define XKB_KEY_Arabic_sukun                  0x05f2  /* U+0652 ARABIC SUKUN */
#define XKB_KEY_Arabic_madda_above         0x1000653  /* U+0653 ARABIC MADDAH ABOVE */
#define XKB_KEY_Arabic_hamza_above         0x1000654  /* U+0654 ARABIC HAMZA ABOVE */
#define XKB_KEY_Arabic_hamza_below         0x1000655  /* U+0655 ARABIC HAMZA BELOW */
#define XKB_KEY_Arabic_jeh                 0x1000698  /* U+0698 ARABIC LETTER JEH */
#define XKB_KEY_Arabic_veh                 0x10006a4  /* U+06A4 ARABIC LETTER VEH */
#define XKB_KEY_Arabic_keheh               0x10006a9  /* U+06A9 ARABIC LETTER KEHEH */
#define XKB_KEY_Arabic_gaf                 0x10006af  /* U+06AF ARABIC LETTER GAF */
#define XKB_KEY_Arabic_noon_ghunna         0x10006ba  /* U+06BA ARABIC LETTER NOON GHUNNA */
#define XKB_KEY_Arabic_heh_doachashmee     0x10006be  /* U+06BE ARABIC LETTER HEH DOACHASHMEE */
#define XKB_KEY_Farsi_yeh                  0x10006cc  /* U+06CC ARABIC LETTER FARSI YEH */
#define XKB_KEY_Arabic_farsi_yeh           0x10006cc  /* deprecated alias for Farsi_yeh */
#define XKB_KEY_Arabic_yeh_baree           0x10006d2  /* U+06D2 ARABIC LETTER YEH BARREE */
#define XKB_KEY_Arabic_heh_goal            0x10006c1  /* U+06C1 ARABIC LETTER HEH GOAL */
#define XKB_KEY_Arabic_switch                 0xff7e  /* non-deprecated alias for Mode_switch */

/*
 * Cyrillic
 * Byte 3 = 6
 */
#define XKB_KEY_Cyrillic_GHE_bar           0x1000492  /* U+0492 CYRILLIC CAPITAL LETTER GHE WITH STROKE */
#define XKB_KEY_Cyrillic_ghe_bar           0x1000493  /* U+0493 CYRILLIC SMALL LETTER GHE WITH STROKE */
#define XKB_KEY_Cyrillic_ZHE_descender     0x1000496  /* U+0496 CYRILLIC CAPITAL LETTER ZHE WITH DESCENDER */
#define XKB_KEY_Cyrillic_zhe_descender     0x1000497  /* U+0497 CYRILLIC SMALL LETTER ZHE WITH DESCENDER */
#define XKB_KEY_Cyrillic_KA_descender      0x100049a  /* U+049A CYRILLIC CAPITAL LETTER KA WITH DESCENDER */
#define XKB_KEY_Cyrillic_ka_descender      0x100049b  /* U+049B CYRILLIC SMALL LETTER KA WITH DESCENDER */
#define XKB_KEY_Cyrillic_KA_vertstroke     0x100049c  /* U+049C CYRILLIC CAPITAL LETTER KA WITH VERTICAL STROKE */
#define XKB_KEY_Cyrillic_ka_vertstroke     0x100049d  /* U+049D CYRILLIC SMALL LETTER KA WITH VERTICAL STROKE */
#define XKB_KEY_Cyrillic_EN_descender      0x10004a2  /* U+04A2 CYRILLIC CAPITAL LETTER EN WITH DESCENDER */
#define XKB_KEY_Cyrillic_en_descender      0x10004a3  /* U+04A3 CYRILLIC SMALL LETTER EN WITH DESCENDER */
#define XKB_KEY_Cyrillic_U_straight        0x10004ae  /* U+04AE CYRILLIC CAPITAL LETTER STRAIGHT U */
#define XKB_KEY_Cyrillic_u_straight        0x10004af  /* U+04AF CYRILLIC SMALL LETTER STRAIGHT U */
#define XKB_KEY_Cyrillic_U_straight_bar    0x10004b0  /* U+04B0 CYRILLIC CAPITAL LETTER STRAIGHT U WITH STROKE */
#define XKB_KEY_Cyrillic_u_straight_bar    0x10004b1  /* U+04B1 CYRILLIC SMALL LETTER STRAIGHT U WITH STROKE */
#define XKB_KEY_Cyrillic_HA_descender      0x10004b2  /* U+04B2 CYRILLIC CAPITAL LETTER HA WITH DESCENDER */
#define XKB_KEY_Cyrillic_ha_descender      0x10004b3  /* U+04B3 CYRILLIC SMALL LETTER HA WITH DESCENDER */
#define XKB_KEY_Cyrillic_CHE_descender     0x10004b6  /* U+04B6 CYRILLIC CAPITAL LETTER CHE WITH DESCENDER */
#define XKB_KEY_Cyrillic_che_descender     0x10004b7  /* U+04B7 CYRILLIC SMALL LETTER CHE WITH DESCENDER */
#define XKB_KEY_Cyrillic_CHE_vertstroke    0x10004b8  /* U+04B8 CYRILLIC CAPITAL LETTER CHE WITH VERTICAL STROKE */
#define XKB_KEY_Cyrillic_che_vertstroke    0x10004b9  /* U+04B9 CYRILLIC SMALL LETTER CHE WITH VERTICAL STROKE */
#define XKB_KEY_Cyrillic_SHHA              0x10004ba  /* U+04BA CYRILLIC CAPITAL LETTER SHHA */
#define XKB_KEY_Cyrillic_shha              0x10004bb  /* U+04BB CYRILLIC SMALL LETTER SHHA */

#define XKB_KEY_Cyrillic_SCHWA             0x10004d8  /* U+04D8 CYRILLIC CAPITAL LETTER SCHWA */
#define XKB_KEY_Cyrillic_schwa             0x10004d9  /* U+04D9 CYRILLIC SMALL LETTER SCHWA */
#define XKB_KEY_Cyrillic_I_macron          0x10004e2  /* U+04E2 CYRILLIC CAPITAL LETTER I WITH MACRON */
#define XKB_KEY_Cyrillic_i_macron          0x10004e3  /* U+04E3 CYRILLIC SMALL LETTER I WITH MACRON */
#define XKB_KEY_Cyrillic_O_bar             0x10004e8  /* U+04E8 CYRILLIC CAPITAL LETTER BARRED O */
#define XKB_KEY_Cyrillic_o_bar             0x10004e9  /* U+04E9 CYRILLIC SMALL LETTER BARRED O */
#define XKB_KEY_Cyrillic_U_macron          0x10004ee  /* U+04EE CYRILLIC CAPITAL LETTER U WITH MACRON */
#define XKB_KEY_Cyrillic_u_macron          0x10004ef  /* U+04EF CYRILLIC SMALL LETTER U WITH MACRON */

#define XKB_KEY_Serbian_dje                   0x06a1  /* U+0452 CYRILLIC SMALL LETTER DJE */
#define XKB_KEY_Macedonia_gje                 0x06a2  /* U+0453 CYRILLIC SMALL LETTER GJE */
#define XKB_KEY_Cyrillic_io                   0x06a3  /* U+0451 CYRILLIC SMALL LETTER IO */
#define XKB_KEY_Ukrainian_ie                  0x06a4  /* U+0454 CYRILLIC SMALL LETTER UKRAINIAN IE */
#define XKB_KEY_Ukranian_je                   0x06a4  /* deprecated */
#define XKB_KEY_Macedonia_dse                 0x06a5  /* U+0455 CYRILLIC SMALL LETTER DZE */
#define XKB_KEY_Ukrainian_i                   0x06a6  /* U+0456 CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I */
#define XKB_KEY_Ukranian_i                    0x06a6  /* deprecated */
#define XKB_KEY_Ukrainian_yi                  0x06a7  /* U+0457 CYRILLIC SMALL LETTER YI */
#define XKB_KEY_Ukranian_yi                   0x06a7  /* deprecated */
#define XKB_KEY_Cyrillic_je                   0x06a8  /* U+0458 CYRILLIC SMALL LETTER JE */
#define XKB_KEY_Serbian_je                    0x06a8  /* deprecated */
#define XKB_KEY_Cyrillic_lje                  0x06a9  /* U+0459 CYRILLIC SMALL LETTER LJE */
#define XKB_KEY_Serbian_lje                   0x06a9  /* deprecated */
#define XKB_KEY_Cyrillic_nje                  0x06aa  /* U+045A CYRILLIC SMALL LETTER NJE */
#define XKB_KEY_Serbian_nje                   0x06aa  /* deprecated */
#define XKB_KEY_Serbian_tshe                  0x06ab  /* U+045B CYRILLIC SMALL LETTER TSHE */
#define XKB_KEY_Macedonia_kje                 0x06ac  /* U+045C CYRILLIC SMALL LETTER KJE */
#define XKB_KEY_Ukrainian_ghe_with_upturn     0x06ad  /* U+0491 CYRILLIC SMALL LETTER GHE WITH UPTURN */
#define XKB_KEY_Byelorussian_shortu           0x06ae  /* U+045E CYRILLIC SMALL LETTER SHORT U */
#define XKB_KEY_Cyrillic_dzhe                 0x06af  /* U+045F CYRILLIC SMALL LETTER DZHE */
#define XKB_KEY_Serbian_dze                   0x06af  /* deprecated */
#define XKB_KEY_numerosign                    0x06b0  /* U+2116 NUMERO SIGN */
#define XKB_KEY_Serbian_DJE                   0x06b1  /* U+0402 CYRILLIC CAPITAL LETTER DJE */
#define XKB_KEY_Macedonia_GJE                 0x06b2  /* U+0403 CYRILLIC CAPITAL LETTER GJE */
#define XKB_KEY_Cyrillic_IO                   0x06b3  /* U+0401 CYRILLIC CAPITAL LETTER IO */
#define XKB_KEY_Ukrainian_IE                  0x06b4  /* U+0404 CYRILLIC CAPITAL LETTER UKRAINIAN IE */
#define XKB_KEY_Ukranian_JE                   0x06b4  /* deprecated */
#define XKB_KEY_Macedonia_DSE                 0x06b5  /* U+0405 CYRILLIC CAPITAL LETTER DZE */
#define XKB_KEY_Ukrainian_I                   0x06b6  /* U+0406 CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I */
#define XKB_KEY_Ukranian_I                    0x06b6  /* deprecated */
#define XKB_KEY_Ukrainian_YI                  0x06b7  /* U+0407 CYRILLIC CAPITAL LETTER YI */
#define XKB_KEY_Ukranian_YI                   0x06b7  /* deprecated */
#define XKB_KEY_Cyrillic_JE                   0x06b8  /* U+0408 CYRILLIC CAPITAL LETTER JE */
#define XKB_KEY_Serbian_JE                    0x06b8  /* deprecated */
#define XKB_KEY_Cyrillic_LJE                  0x06b9  /* U+0409 CYRILLIC CAPITAL LETTER LJE */
#define XKB_KEY_Serbian_LJE                   0x06b9  /* deprecated */
#define XKB_KEY_Cyrillic_NJE                  0x06ba  /* U+040A CYRILLIC CAPITAL LETTER NJE */
#define XKB_KEY_Serbian_NJE                   0x06ba  /* deprecated */
#define XKB_KEY_Serbian_TSHE                  0x06bb  /* U+040B CYRILLIC CAPITAL LETTER TSHE */
#define XKB_KEY_Macedonia_KJE                 0x06bc  /* U+040C CYRILLIC CAPITAL LETTER KJE */
#define XKB_KEY_Ukrainian_GHE_WITH_UPTURN     0x06bd  /* U+0490 CYRILLIC CAPITAL LETTER GHE WITH UPTURN */
#define XKB_KEY_Byelorussian_SHORTU           0x06be  /* U+040E CYRILLIC CAPITAL LETTER SHORT U */
#define XKB_KEY_Cyrillic_DZHE                 0x06bf  /* U+040F CYRILLIC CAPITAL LETTER DZHE */
#define XKB_KEY_Serbian_DZE                   0x06bf  /* deprecated */
#define XKB_KEY_Cyrillic_yu                   0x06c0  /* U+044E CYRILLIC SMALL LETTER YU */
#define XKB_KEY_Cyrillic_a                    0x06c1  /* U+0430 CYRILLIC SMALL LETTER A */
#define XKB_KEY_Cyrillic_be                   0x06c2  /* U+0431 CYRILLIC SMALL LETTER BE */
#define XKB_KEY_Cyrillic_tse                  0x06c3  /* U+0446 CYRILLIC SMALL LETTER TSE */
#define XKB_KEY_Cyrillic_de                   0x06c4  /* U+0434 CYRILLIC SMALL LETTER DE */
#define XKB_KEY_Cyrillic_ie                   0x06c5  /* U+0435 CYRILLIC SMALL LETTER IE */
#define XKB_KEY_Cyrillic_ef                   0x06c6  /* U+0444 CYRILLIC SMALL LETTER EF */
#define XKB_KEY_Cyrillic_ghe                  0x06c7  /* U+0433 CYRILLIC SMALL LETTER GHE */
#define XKB_KEY_Cyrillic_ha                   0x06c8  /* U+0445 CYRILLIC SMALL LETTER HA */
#define XKB_KEY_Cyrillic_i                    0x06c9  /* U+0438 CYRILLIC SMALL LETTER I */
#define XKB_KEY_Cyrillic_shorti               0x06ca  /* U+0439 CYRILLIC SMALL LETTER SHORT I */
#define XKB_KEY_Cyrillic_ka                   0x06cb  /* U+043A CYRILLIC SMALL LETTER KA */
#define XKB_KEY_Cyrillic_el                   0x06cc  /* U+043B CYRILLIC SMALL LETTER EL */
#define XKB_KEY_Cyrillic_em                   0x06cd  /* U+043C CYRILLIC SMALL LETTER EM */
#define XKB_KEY_Cyrillic_en                   0x06ce  /* U+043D CYRILLIC SMALL LETTER EN */
#define XKB_KEY_Cyrillic_o                    0x06cf  /* U+043E CYRILLIC SMALL LETTER O */
#define XKB_KEY_Cyrillic_pe                   0x06d0  /* U+043F CYRILLIC SMALL LETTER PE */
#define XKB_KEY_Cyrillic_ya                   0x06d1  /* U+044F CYRILLIC SMALL LETTER YA */
#define XKB_KEY_Cyrillic_er                   0x06d2  /* U+0440 CYRILLIC SMALL LETTER ER */
#define XKB_KEY_Cyrillic_es                   0x06d3  /* U+0441 CYRILLIC SMALL LETTER ES */
#define XKB_KEY_Cyrillic_te                   0x06d4  /* U+0442 CYRILLIC SMALL LETTER TE */
#define XKB_KEY_Cyrillic_u                    0x06d5  /* U+0443 CYRILLIC SMALL LETTER U */
#define XKB_KEY_Cyrillic_zhe                  0x06d6  /* U+0436 CYRILLIC SMALL LETTER ZHE */
#define XKB_KEY_Cyrillic_ve                   0x06d7  /* U+0432 CYRILLIC SMALL LETTER VE */
#define XKB_KEY_Cyrillic_softsign             0x06d8  /* U+044C CYRILLIC SMALL LETTER SOFT SIGN */
#define XKB_KEY_Cyrillic_yeru                 0x06d9  /* U+044B CYRILLIC SMALL LETTER YERU */
#define XKB_KEY_Cyrillic_ze                   0x06da  /* U+0437 CYRILLIC SMALL LETTER ZE */
#define XKB_KEY_Cyrillic_sha                  0x06db  /* U+0448 CYRILLIC SMALL LETTER SHA */
#define XKB_KEY_Cyrillic_e                    0x06dc  /* U+044D CYRILLIC SMALL LETTER E */
#define XKB_KEY_Cyrillic_shcha                0x06dd  /* U+0449 CYRILLIC SMALL LETTER SHCHA */
#define XKB_KEY_Cyrillic_che                  0x06de  /* U+0447 CYRILLIC SMALL LETTER CHE */
#define XKB_KEY_Cyrillic_hardsign             0x06df  /* U+044A CYRILLIC SMALL LETTER HARD SIGN */
#define XKB_KEY_Cyrillic_YU                   0x06e0  /* U+042E CYRILLIC CAPITAL LETTER YU */
#define XKB_KEY_Cyrillic_A                    0x06e1  /* U+0410 CYRILLIC CAPITAL LETTER A */
#define XKB_KEY_Cyrillic_BE                   0x06e2  /* U+0411 CYRILLIC CAPITAL LETTER BE */
#define XKB_KEY_Cyrillic_TSE                  0x06e3  /* U+0426 CYRILLIC CAPITAL LETTER TSE */
#define XKB_KEY_Cyrillic_DE                   0x06e4  /* U+0414 CYRILLIC CAPITAL LETTER DE */
#define XKB_KEY_Cyrillic_IE                   0x06e5  /* U+0415 CYRILLIC CAPITAL LETTER IE */
#define XKB_KEY_Cyrillic_EF                   0x06e6  /* U+0424 CYRILLIC CAPITAL LETTER EF */
#define XKB_KEY_Cyrillic_GHE                  0x06e7  /* U+0413 CYRILLIC CAPITAL LETTER GHE */
#define XKB_KEY_Cyrillic_HA                   0x06e8  /* U+0425 CYRILLIC CAPITAL LETTER HA */
#define XKB_KEY_Cyrillic_I                    0x06e9  /* U+0418 CYRILLIC CAPITAL LETTER I */
#define XKB_KEY_Cyrillic_SHORTI               0x06ea  /* U+0419 CYRILLIC CAPITAL LETTER SHORT I */
#define XKB_KEY_Cyrillic_KA                   0x06eb  /* U+041A CYRILLIC CAPITAL LETTER KA */
#define XKB_KEY_Cyrillic_EL                   0x06ec  /* U+041B CYRILLIC CAPITAL LETTER EL */
#define XKB_KEY_Cyrillic_EM                   0x06ed  /* U+041C CYRILLIC CAPITAL LETTER EM */
#define XKB_KEY_Cyrillic_EN                   0x06ee  /* U+041D CYRILLIC CAPITAL LETTER EN */
#define XKB_KEY_Cyrillic_O                    0x06ef  /* U+041E CYRILLIC CAPITAL LETTER O */
#define XKB_KEY_Cyrillic_PE                   0x06f0  /* U+041F CYRILLIC CAPITAL LETTER PE */
#define XKB_KEY_Cyrillic_YA                   0x06f1  /* U+042F CYRILLIC CAPITAL LETTER YA */
#define XKB_KEY_Cyrillic_ER                   0x06f2  /* U+0420 CYRILLIC CAPITAL LETTER ER */
#define XKB_KEY_Cyrillic_ES                   0x06f3  /* U+0421 CYRILLIC CAPITAL LETTER ES */
#define XKB_KEY_Cyrillic_TE                   0x06f4  /* U+0422 CYRILLIC CAPITAL LETTER TE */
#define XKB_KEY_Cyrillic_U                    0x06f5  /* U+0423 CYRILLIC CAPITAL LETTER U */
#define XKB_KEY_Cyrillic_ZHE                  0x06f6  /* U+0416 CYRILLIC CAPITAL LETTER ZHE */
#define XKB_KEY_Cyrillic_VE                   0x06f7  /* U+0412 CYRILLIC CAPITAL LETTER VE */
#define XKB_KEY_Cyrillic_SOFTSIGN             0x06f8  /* U+042C CYRILLIC CAPITAL LETTER SOFT SIGN */
#define XKB_KEY_Cyrillic_YERU                 0x06f9  /* U+042B CYRILLIC CAPITAL LETTER YERU */
#define XKB_KEY_Cyrillic_ZE                   0x06fa  /* U+0417 CYRILLIC CAPITAL LETTER ZE */
#define XKB_KEY_Cyrillic_SHA                  0x06fb  /* U+0428 CYRILLIC CAPITAL LETTER SHA */
#define XKB_KEY_Cyrillic_E                    0x06fc  /* U+042D CYRILLIC CAPITAL LETTER E */
#define XKB_KEY_Cyrillic_SHCHA                0x06fd  /* U+0429 CYRILLIC CAPITAL LETTER SHCHA */
#define XKB_KEY_Cyrillic_CHE                  0x06fe  /* U+0427 CYRILLIC CAPITAL LETTER CHE */
#define XKB_KEY_Cyrillic_HARDSIGN             0x06ff  /* U+042A CYRILLIC CAPITAL LETTER HARD SIGN */

/*
 * Greek
 * (based on an early draft of, and not quite identical to, ISO/IEC 8859-7)
 * Byte 3 = 7
 */

#define XKB_KEY_Greek_ALPHAaccent             0x07a1  /* U+0386 GREEK CAPITAL LETTER ALPHA WITH TONOS */
#define XKB_KEY_Greek_EPSILONaccent           0x07a2  /* U+0388 GREEK CAPITAL LETTER EPSILON WITH TONOS */
#define XKB_KEY_Greek_ETAaccent               0x07a3  /* U+0389 GREEK CAPITAL LETTER ETA WITH TONOS */
#define XKB_KEY_Greek_IOTAaccent              0x07a4  /* U+038A GREEK CAPITAL LETTER IOTA WITH TONOS */
#define XKB_KEY_Greek_IOTAdieresis            0x07a5  /* U+03AA GREEK CAPITAL LETTER IOTA WITH DIALYTIKA */
#define XKB_KEY_Greek_IOTAdiaeresis           0x07a5  /* deprecated (old typo) */
#define XKB_KEY_Greek_OMICRONaccent           0x07a7  /* U+038C GREEK CAPITAL LETTER OMICRON WITH TONOS */
#define XKB_KEY_Greek_UPSILONaccent           0x07a8  /* U+038E GREEK CAPITAL LETTER UPSILON WITH TONOS */
#define XKB_KEY_Greek_UPSILONdieresis         0x07a9  /* U+03AB GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA */
#define XKB_KEY_Greek_OMEGAaccent             0x07ab  /* U+038F GREEK CAPITAL LETTER OMEGA WITH TONOS */
#define XKB_KEY_Greek_accentdieresis          0x07ae  /* U+0385 GREEK DIALYTIKA TONOS */
#define XKB_KEY_Greek_horizbar                0x07af  /* U+2015 HORIZONTAL BAR */
#define XKB_KEY_Greek_alphaaccent             0x07b1  /* U+03AC GREEK SMALL LETTER ALPHA WITH TONOS */
#define XKB_KEY_Greek_epsilonaccent           0x07b2  /* U+03AD GREEK SMALL LETTER EPSILON WITH TONOS */
#define XKB_KEY_Greek_etaaccent               0x07b3  /* U+03AE GREEK SMALL LETTER ETA WITH TONOS */
#define XKB_KEY_Greek_iotaaccent              0x07b4  /* U+03AF GREEK SMALL LETTER IOTA WITH TONOS */
#define XKB_KEY_Greek_iotadieresis            0x07b5  /* U+03CA GREEK SMALL LETTER IOTA WITH DIALYTIKA */
#define XKB_KEY_Greek_iotaaccentdieresis      0x07b6  /* U+0390 GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS */
#define XKB_KEY_Greek_omicronaccent           0x07b7  /* U+03CC GREEK SMALL LETTER OMICRON WITH TONOS */
#define XKB_KEY_Greek_upsilonaccent           0x07b8  /* U+03CD GREEK SMALL LETTER UPSILON WITH TONOS */
#define XKB_KEY_Greek_upsilondieresis         0x07b9  /* U+03CB GREEK SMALL LETTER UPSILON WITH DIALYTIKA */
#define XKB_KEY_Greek_upsilonaccentdieresis   0x07ba  /* U+03B0 GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOS */
#define XKB_KEY_Greek_omegaaccent             0x07bb  /* U+03CE GREEK SMALL LETTER OMEGA WITH TONOS */
#define XKB_KEY_Greek_ALPHA                   0x07c1  /* U+0391 GREEK CAPITAL LETTER ALPHA */
#define XKB_KEY_Greek_BETA                    0x07c2  /* U+0392 GREEK CAPITAL LETTER BETA */
#define XKB_KEY_Greek_GAMMA                   0x07c3  /* U+0393 GREEK CAPITAL LETTER GAMMA */
#define XKB_KEY_Greek_DELTA                   0x07c4  /* U+0394 GREEK CAPITAL LETTER DELTA */
#define XKB_KEY_Greek_EPSILON                 0x07c5  /* U+0395 GREEK CAPITAL LETTER EPSILON */
#define XKB_KEY_Greek_ZETA                    0x07c6  /* U+0396 GREEK CAPITAL LETTER ZETA */
#define XKB_KEY_Greek_ETA                     0x07c7  /* U+0397 GREEK CAPITAL LETTER ETA */
#define XKB_KEY_Greek_THETA                   0x07c8  /* U+0398 GREEK CAPITAL LETTER THETA */
#define XKB_KEY_Greek_IOTA                    0x07c9  /* U+0399 GREEK CAPITAL LETTER IOTA */
#define XKB_KEY_Greek_KAPPA                   0x07ca  /* U+039A GREEK CAPITAL LETTER KAPPA */
#define XKB_KEY_Greek_LAMDA                   0x07cb  /* U+039B GREEK CAPITAL LETTER LAMDA */
#define XKB_KEY_Greek_LAMBDA                  0x07cb  /* non-deprecated alias for Greek_LAMDA */
#define XKB_KEY_Greek_MU                      0x07cc  /* U+039C GREEK CAPITAL LETTER MU */
#define XKB_KEY_Greek_NU                      0x07cd  /* U+039D GREEK CAPITAL LETTER NU */
#define XKB_KEY_Greek_XI                      0x07ce  /* U+039E GREEK CAPITAL LETTER XI */
#define XKB_KEY_Greek_OMICRON                 0x07cf  /* U+039F GREEK CAPITAL LETTER OMICRON */
#define XKB_KEY_Greek_PI                      0x07d0  /* U+03A0 GREEK CAPITAL LETTER PI */
#define XKB_KEY_Greek_RHO                     0x07d1  /* U+03A1 GREEK CAPITAL LETTER RHO */
#define XKB_KEY_Greek_SIGMA                   0x07d2  /* U+03A3 GREEK CAPITAL LETTER SIGMA */
#define XKB_KEY_Greek_TAU                     0x07d4  /* U+03A4 GREEK CAPITAL LETTER TAU */
#define XKB_KEY_Greek_UPSILON                 0x07d5  /* U+03A5 GREEK CAPITAL LETTER UPSILON */
#define XKB_KEY_Greek_PHI                     0x07d6  /* U+03A6 GREEK CAPITAL LETTER PHI */
#define XKB_KEY_Greek_CHI                     0x07d7  /* U+03A7 GREEK CAPITAL LETTER CHI */
#define XKB_KEY_Greek_PSI                     0x07d8  /* U+03A8 GREEK CAPITAL LETTER PSI */
#define XKB_KEY_Greek_OMEGA                   0x07d9  /* U+03A9 GREEK CAPITAL LETTER OMEGA */
#define XKB_KEY_Greek_alpha                   0x07e1  /* U+03B1 GREEK SMALL LETTER ALPHA */
#define XKB_KEY_Greek_beta                    0x07e2  /* U+03B2 GREEK SMALL LETTER BETA */
#define XKB_KEY_Greek_gamma                   0x07e3  /* U+03B3 GREEK SMALL LETTER GAMMA */
#define XKB_KEY_Greek_delta                   0x07e4  /* U+03B4 GREEK SMALL LETTER DELTA */
#define XKB_KEY_Greek_epsilon                 0x07e5  /* U+03B5 GREEK SMALL LETTER EPSILON */
#define XKB_KEY_Greek_zeta                    0x07e6  /* U+03B6 GREEK SMALL LETTER ZETA */
#define XKB_KEY_Greek_eta                     0x07e7  /* U+03B7 GREEK SMALL LETTER ETA */
#define XKB_KEY_Greek_theta                   0x07e8  /* U+03B8 GREEK SMALL LETTER THETA */
#define XKB_KEY_Greek_iota                    0x07e9  /* U+03B9 GREEK SMALL LETTER IOTA */
#define XKB_KEY_Greek_kappa                   0x07ea  /* U+03BA GREEK SMALL LETTER KAPPA */
#define XKB_KEY_Greek_lamda                   0x07eb  /* U+03BB GREEK SMALL LETTER LAMDA */
#define XKB_KEY_Greek_lambda                  0x07eb  /* non-deprecated alias for Greek_lamda */
#define XKB_KEY_Greek_mu                      0x07ec  /* U+03BC GREEK SMALL LETTER MU */
#define XKB_KEY_Greek_nu                      0x07ed  /* U+03BD GREEK SMALL LETTER NU */
#define XKB_KEY_Greek_xi                      0x07ee  /* U+03BE GREEK SMALL LETTER XI */
#define XKB_KEY_Greek_omicron                 0x07ef  /* U+03BF GREEK SMALL LETTER OMICRON */
#define XKB_KEY_Greek_pi                      0x07f0  /* U+03C0 GREEK SMALL LETTER PI */
#define XKB_KEY_Greek_rho                     0x07f1  /* U+03C1 GREEK SMALL LETTER RHO */
#define XKB_KEY_Greek_sigma                   0x07f2  /* U+03C3 GREEK SMALL LETTER SIGMA */
#define XKB_KEY_Greek_finalsmallsigma         0x07f3  /* U+03C2 GREEK SMALL LETTER FINAL SIGMA */
#define XKB_KEY_Greek_tau                     0x07f4  /* U+03C4 GREEK SMALL LETTER TAU */
#define XKB_KEY_Greek_upsilon                 0x07f5  /* U+03C5 GREEK SMALL LETTER UPSILON */
#define XKB_KEY_Greek_phi                     0x07f6  /* U+03C6 GREEK SMALL LETTER PHI */
#define XKB_KEY_Greek_chi                     0x07f7  /* U+03C7 GREEK SMALL LETTER CHI */
#define XKB_KEY_Greek_psi                     0x07f8  /* U+03C8 GREEK SMALL LETTER PSI */
#define XKB_KEY_Greek_omega                   0x07f9  /* U+03C9 GREEK SMALL LETTER OMEGA */
#define XKB_KEY_Greek_switch                  0xff7e  /* non-deprecated alias for Mode_switch */

/*
 * Technical
 * (from the DEC VT330/VT420 Technical Character Set, http://vt100.net/charsets/technical.html)
 * Byte 3 = 8
 */

#define XKB_KEY_leftradical                   0x08a1  /* U+23B7 RADICAL SYMBOL BOTTOM */
#define XKB_KEY_topleftradical                0x08a2  /*(U+250C BOX DRAWINGS LIGHT DOWN AND RIGHT)*/
#define XKB_KEY_horizconnector                0x08a3  /*(U+2500 BOX DRAWINGS LIGHT HORIZONTAL)*/
#define XKB_KEY_topintegral                   0x08a4  /* U+2320 TOP HALF INTEGRAL */
#define XKB_KEY_botintegral                   0x08a5  /* U+2321 BOTTOM HALF INTEGRAL */
#define XKB_KEY_vertconnector                 0x08a6  /*(U+2502 BOX DRAWINGS LIGHT VERTICAL)*/
#define XKB_KEY_topleftsqbracket              0x08a7  /* U+23A1 LEFT SQUARE BRACKET UPPER CORNER */
#define XKB_KEY_botleftsqbracket              0x08a8  /* U+23A3 LEFT SQUARE BRACKET LOWER CORNER */
#define XKB_KEY_toprightsqbracket             0x08a9  /* U+23A4 RIGHT SQUARE BRACKET UPPER CORNER */
#define XKB_KEY_botrightsqbracket             0x08aa  /* U+23A6 RIGHT SQUARE BRACKET LOWER CORNER */
#define XKB_KEY_topleftparens                 0x08ab  /* U+239B LEFT PARENTHESIS UPPER HOOK */
#define XKB_KEY_botleftparens                 0x08ac  /* U+239D LEFT PARENTHESIS LOWER HOOK */
#define XKB_KEY_toprightparens                0x08ad  /* U+239E RIGHT PARENTHESIS UPPER HOOK */
#define XKB_KEY_botrightparens                0x08ae  /* U+23A0 RIGHT PARENTHESIS LOWER HOOK */
#define XKB_KEY_leftmiddlecurlybrace          0x08af  /* U+23A8 LEFT CURLY BRACKET MIDDLE PIECE */
#define XKB_KEY_rightmiddlecurlybrace         0x08b0  /* U+23AC RIGHT CURLY BRACKET MIDDLE PIECE */
#define XKB_KEY_topleftsummation              0x08b1
#define XKB_KEY_botleftsummation              0x08b2
#define XKB_KEY_topvertsummationconnector     0x08b3
#define XKB_KEY_botvertsummationconnector     0x08b4
#define XKB_KEY_toprightsummation             0x08b5
#define XKB_KEY_botrightsummation             0x08b6
#define XKB_KEY_rightmiddlesummation          0x08b7
#define XKB_KEY_lessthanequal                 0x08bc  /* U+2264 LESS-THAN OR EQUAL TO */
#define XKB_KEY_notequal                      0x08bd  /* U+2260 NOT EQUAL TO */
#define XKB_KEY_greaterthanequal              0x08be  /* U+2265 GREATER-THAN OR EQUAL TO */
#define XKB_KEY_integral                      0x08bf  /* U+222B INTEGRAL */
#define XKB_KEY_therefore                     0x08c0  /* U+2234 THEREFORE */
#define XKB_KEY_variation                     0x08c1  /* U+221D PROPORTIONAL TO */
#define XKB_KEY_infinity                      0x08c2  /* U+221E INFINITY */
#define XKB_KEY_nabla                         0x08c5  /* U+2207 NABLA */
#define XKB_KEY_approximate                   0x08c8  /* U+223C TILDE OPERATOR */
#define XKB_KEY_similarequal                  0x08c9  /* U+2243 ASYMPTOTICALLY EQUAL TO */
#define XKB_KEY_ifonlyif                      0x08cd  /* U+21D4 LEFT RIGHT DOUBLE ARROW */
#define XKB_KEY_implies                       0x08ce  /* U+21D2 RIGHTWARDS DOUBLE ARROW */
#define XKB_KEY_identical                     0x08cf  /* U+2261 IDENTICAL TO */
#define XKB_KEY_radical                       0x08d6  /* U+221A SQUARE ROOT */
#define XKB_KEY_includedin                    0x08da  /* U+2282 SUBSET OF */
#define XKB_KEY_includes                      0x08db  /* U+2283 SUPERSET OF */
#define XKB_KEY_intersection                  0x08dc  /* U+2229 INTERSECTION */
#define XKB_KEY_union                         0x08dd  /* U+222A UNION */
#define XKB_KEY_logicaland                    0x08de  /* U+2227 LOGICAL AND */
#define XKB_KEY_logicalor                     0x08df  /* U+2228 LOGICAL OR */
#define XKB_KEY_partialderivative             0x08ef  /* U+2202 PARTIAL DIFFERENTIAL */
#define XKB_KEY_function                      0x08f6  /* U+0192 LATIN SMALL LETTER F WITH HOOK */
#define XKB_KEY_leftarrow                     0x08fb  /* U+2190 LEFTWARDS ARROW */
#define XKB_KEY_uparrow                       0x08fc  /* U+2191 UPWARDS ARROW */
#define XKB_KEY_rightarrow                    0x08fd  /* U+2192 RIGHTWARDS ARROW */
#define XKB_KEY_downarrow                     0x08fe  /* U+2193 DOWNWARDS ARROW */

/*
 * Special
 * (from the DEC VT100 Special Graphics Character Set)
 * Byte 3 = 9
 */

#define XKB_KEY_blank                         0x09df
#define XKB_KEY_soliddiamond                  0x09e0  /* U+25C6 BLACK DIAMOND */
#define XKB_KEY_checkerboard                  0x09e1  /* U+2592 MEDIUM SHADE */
#define XKB_KEY_ht                            0x09e2  /* U+2409 SYMBOL FOR HORIZONTAL TABULATION */
#define XKB_KEY_ff                            0x09e3  /* U+240C SYMBOL FOR FORM FEED */
#define XKB_KEY_cr                            0x09e4  /* U+240D SYMBOL FOR CARRIAGE RETURN */
#define XKB_KEY_lf                            0x09e5  /* U+240A SYMBOL FOR LINE FEED */
#define XKB_KEY_nl                            0x09e8  /* U+2424 SYMBOL FOR NEWLINE */
#define XKB_KEY_vt                            0x09e9  /* U+240B SYMBOL FOR VERTICAL TABULATION */
#define XKB_KEY_lowrightcorner                0x09ea  /* U+2518 BOX DRAWINGS LIGHT UP AND LEFT */
#define XKB_KEY_uprightcorner                 0x09eb  /* U+2510 BOX DRAWINGS LIGHT DOWN AND LEFT */
#define XKB_KEY_upleftcorner                  0x09ec  /* U+250C BOX DRAWINGS LIGHT DOWN AND RIGHT */
#define XKB_KEY_lowleftcorner                 0x09ed  /* U+2514 BOX DRAWINGS LIGHT UP AND RIGHT */
#define XKB_KEY_crossinglines                 0x09ee  /* U+253C BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL */
#define XKB_KEY_horizlinescan1                0x09ef  /* U+23BA HORIZONTAL SCAN LINE-1 */
#define XKB_KEY_horizlinescan3                0x09f0  /* U+23BB HORIZONTAL SCAN LINE-3 */
#define XKB_KEY_horizlinescan5                0x09f1  /* U+2500 BOX DRAWINGS LIGHT HORIZONTAL */
#define XKB_KEY_horizlinescan7                0x09f2  /* U+23BC HORIZONTAL SCAN LINE-7 */
#define XKB_KEY_horizlinescan9                0x09f3  /* U+23BD HORIZONTAL SCAN LINE-9 */
#define XKB_KEY_leftt                         0x09f4  /* U+251C BOX DRAWINGS LIGHT VERTICAL AND RIGHT */
#define XKB_KEY_rightt                        0x09f5  /* U+2524 BOX DRAWINGS LIGHT VERTICAL AND LEFT */
#define XKB_KEY_bott                          0x09f6  /* U+2534 BOX DRAWINGS LIGHT UP AND HORIZONTAL */
#define XKB_KEY_topt                          0x09f7  /* U+252C BOX DRAWINGS LIGHT DOWN AND HORIZONTAL */
#define XKB_KEY_vertbar                       0x09f8  /* U+2502 BOX DRAWINGS LIGHT VERTICAL */

/*
 * Publishing
 * (these are probably from a long forgotten DEC Publishing
 * font that once shipped with DECwrite)
 * Byte 3 = 0x0a
 */

#define XKB_KEY_emspace                       0x0aa1  /* U+2003 EM SPACE */
#define XKB_KEY_enspace                       0x0aa2  /* U+2002 EN SPACE */
#define XKB_KEY_em3space                      0x0aa3  /* U+2004 THREE-PER-EM SPACE */
#define XKB_KEY_em4space                      0x0aa4  /* U+2005 FOUR-PER-EM SPACE */
#define XKB_KEY_digitspace                    0x0aa5  /* U+2007 FIGURE SPACE */
#define XKB_KEY_punctspace                    0x0aa6  /* U+2008 PUNCTUATION SPACE */
#define XKB_KEY_thinspace                     0x0aa7  /* U+2009 THIN SPACE */
#define XKB_KEY_hairspace                     0x0aa8  /* U+200A HAIR SPACE */
#define XKB_KEY_emdash                        0x0aa9  /* U+2014 EM DASH */
#define XKB_KEY_endash                        0x0aaa  /* U+2013 EN DASH */
#define XKB_KEY_signifblank                   0x0aac  /*(U+2423 OPEN BOX)*/
#define XKB_KEY_ellipsis                      0x0aae  /* U+2026 HORIZONTAL ELLIPSIS */
#define XKB_KEY_doubbaselinedot               0x0aaf  /* U+2025 TWO DOT LEADER */
#define XKB_KEY_onethird                      0x0ab0  /* U+2153 VULGAR FRACTION ONE THIRD */
#define XKB_KEY_twothirds                     0x0ab1  /* U+2154 VULGAR FRACTION TWO THIRDS */
#define XKB_KEY_onefifth                      0x0ab2  /* U+2155 VULGAR FRACTION ONE FIFTH */
#define XKB_KEY_twofifths                     0x0ab3  /* U+2156 VULGAR FRACTION TWO FIFTHS */
#define XKB_KEY_threefifths                   0x0ab4  /* U+2157 VULGAR FRACTION THREE FIFTHS */
#define XKB_KEY_fourfifths                    0x0ab5  /* U+2158 VULGAR FRACTION FOUR FIFTHS */
#define XKB_KEY_onesixth                      0x0ab6  /* U+2159 VULGAR FRACTION ONE SIXTH */
#define XKB_KEY_fivesixths                    0x0ab7  /* U+215A VULGAR FRACTION FIVE SIXTHS */
#define XKB_KEY_careof                        0x0ab8  /* U+2105 CARE OF */
#define XKB_KEY_figdash                       0x0abb  /* U+2012 FIGURE DASH */
#define XKB_KEY_leftanglebracket              0x0abc  /*(U+2329 LEFT-POINTING ANGLE BRACKET)*/
#define XKB_KEY_decimalpoint                  0x0abd  /*(U+002E FULL STOP)*/
#define XKB_KEY_rightanglebracket             0x0abe  /*(U+232A RIGHT-POINTING ANGLE BRACKET)*/
#define XKB_KEY_marker                        0x0abf
#define XKB_KEY_oneeighth                     0x0ac3  /* U+215B VULGAR FRACTION ONE EIGHTH */
#define XKB_KEY_threeeighths                  0x0ac4  /* U+215C VULGAR FRACTION THREE EIGHTHS */
#define XKB_KEY_fiveeighths                   0x0ac5  /* U+215D VULGAR FRACTION FIVE EIGHTHS */
#define XKB_KEY_seveneighths                  0x0ac6  /* U+215E VULGAR FRACTION SEVEN EIGHTHS */
#define XKB_KEY_trademark                     0x0ac9  /* U+2122 TRADE MARK SIGN */
#define XKB_KEY_signaturemark                 0x0aca  /*(U+2613 SALTIRE)*/
#define XKB_KEY_trademarkincircle             0x0acb
#define XKB_KEY_leftopentriangle              0x0acc  /*(U+25C1 WHITE LEFT-POINTING TRIANGLE)*/
#define XKB_KEY_rightopentriangle             0x0acd  /*(U+25B7 WHITE RIGHT-POINTING TRIANGLE)*/
#define XKB_KEY_emopencircle                  0x0ace  /*(U+25CB WHITE CIRCLE)*/
#define XKB_KEY_emopenrectangle               0x0acf  /*(U+25AF WHITE VERTICAL RECTANGLE)*/
#define XKB_KEY_leftsinglequotemark           0x0ad0  /* U+2018 LEFT SINGLE QUOTATION MARK */
#define XKB_KEY_rightsinglequotemark          0x0ad1  /* U+2019 RIGHT SINGLE QUOTATION MARK */
#define XKB_KEY_leftdoublequotemark           0x0ad2  /* U+201C LEFT DOUBLE QUOTATION MARK */
#define XKB_KEY_rightdoublequotemark          0x0ad3  /* U+201D RIGHT DOUBLE QUOTATION MARK */
#define XKB_KEY_prescription                  0x0ad4  /* U+211E PRESCRIPTION TAKE */
#define XKB_KEY_permille                      0x0ad5  /* U+2030 PER MILLE SIGN */
#define XKB_KEY_minutes                       0x0ad6  /* U+2032 PRIME */
#define XKB_KEY_seconds                       0x0ad7  /* U+2033 DOUBLE PRIME */
#define XKB_KEY_latincross                    0x0ad9  /* U+271D LATIN CROSS */
#define XKB_KEY_hexagram                      0x0ada
#define XKB_KEY_filledrectbullet              0x0adb  /*(U+25AC BLACK RECTANGLE)*/
#define XKB_KEY_filledlefttribullet           0x0adc  /*(U+25C0 BLACK LEFT-POINTING TRIANGLE)*/
#define XKB_KEY_filledrighttribullet          0x0add  /*(U+25B6 BLACK RIGHT-POINTING TRIANGLE)*/
#define XKB_KEY_emfilledcircle                0x0ade  /*(U+25CF BLACK CIRCLE)*/
#define XKB_KEY_emfilledrect                  0x0adf  /*(U+25AE BLACK VERTICAL RECTANGLE)*/
#define XKB_KEY_enopencircbullet              0x0ae0  /*(U+25E6 WHITE BULLET)*/
#define XKB_KEY_enopensquarebullet            0x0ae1  /*(U+25AB WHITE SMALL SQUARE)*/
#define XKB_KEY_openrectbullet                0x0ae2  /*(U+25AD WHITE RECTANGLE)*/
#define XKB_KEY_opentribulletup               0x0ae3  /*(U+25B3 WHITE UP-POINTING TRIANGLE)*/
#define XKB_KEY_opentribulletdown             0x0ae4  /*(U+25BD WHITE DOWN-POINTING TRIANGLE)*/
#define XKB_KEY_openstar                      0x0ae5  /*(U+2606 WHITE STAR)*/
#define XKB_KEY_enfilledcircbullet            0x0ae6  /*(U+2022 BULLET)*/
#define XKB_KEY_enfilledsqbullet              0x0ae7  /*(U+25AA BLACK SMALL SQUARE)*/
#define XKB_KEY_filledtribulletup             0x0ae8  /*(U+25B2 BLACK UP-POINTING TRIANGLE)*/
#define XKB_KEY_filledtribulletdown           0x0ae9  /*(U+25BC BLACK DOWN-POINTING TRIANGLE)*/
#define XKB_KEY_leftpointer                   0x0aea  /*(U+261C WHITE LEFT POINTING INDEX)*/
#define XKB_KEY_rightpointer                  0x0aeb  /*(U+261E WHITE RIGHT POINTING INDEX)*/
#define XKB_KEY_club                          0x0aec  /* U+2663 BLACK CLUB SUIT */
#define XKB_KEY_diamond                       0x0aed  /* U+2666 BLACK DIAMOND SUIT */
#define XKB_KEY_heart                         0x0aee  /* U+2665 BLACK HEART SUIT */
#define XKB_KEY_maltesecross                  0x0af0  /* U+2720 MALTESE CROSS */
#define XKB_KEY_dagger                        0x0af1  /* U+2020 DAGGER */
#define XKB_KEY_doubledagger                  0x0af2  /* U+2021 DOUBLE DAGGER */
#define XKB_KEY_checkmark                     0x0af3  /* U+2713 CHECK MARK */
#define XKB_KEY_ballotcross                   0x0af4  /* U+2717 BALLOT X */
#define XKB_KEY_musicalsharp                  0x0af5  /* U+266F MUSIC SHARP SIGN */
#define XKB_KEY_musicalflat                   0x0af6  /* U+266D MUSIC FLAT SIGN */
#define XKB_KEY_malesymbol                    0x0af7  /* U+2642 MALE SIGN */
#define XKB_KEY_femalesymbol                  0x0af8  /* U+2640 FEMALE SIGN */
#define XKB_KEY_telephone                     0x0af9  /* U+260E BLACK TELEPHONE */
#define XKB_KEY_telephonerecorder             0x0afa  /* U+2315 TELEPHONE RECORDER */
#define XKB_KEY_phonographcopyright           0x0afb  /* U+2117 SOUND RECORDING COPYRIGHT */
#define XKB_KEY_caret                         0x0afc  /* U+2038 CARET */
#define XKB_KEY_singlelowquotemark            0x0afd  /* U+201A SINGLE LOW-9 QUOTATION MARK */
#define XKB_KEY_doublelowquotemark            0x0afe  /* U+201E DOUBLE LOW-9 QUOTATION MARK */
#define XKB_KEY_cursor                        0x0aff

/*
 * APL
 * Byte 3 = 0x0b
 */

#define XKB_KEY_leftcaret                     0x0ba3  /*(U+003C LESS-THAN SIGN)*/
#define XKB_KEY_rightcaret                    0x0ba6  /*(U+003E GREATER-THAN SIGN)*/
#define XKB_KEY_downcaret                     0x0ba8  /*(U+2228 LOGICAL OR)*/
#define XKB_KEY_upcaret                       0x0ba9  /*(U+2227 LOGICAL AND)*/
#define XKB_KEY_overbar                       0x0bc0  /*(U+00AF MACRON)*/
#define XKB_KEY_downtack                      0x0bc2  /* U+22A4 DOWN TACK */
#define XKB_KEY_upshoe                        0x0bc3  /*(U+2229 INTERSECTION)*/
#define XKB_KEY_downstile                     0x0bc4  /* U+230A LEFT FLOOR */
#define XKB_KEY_underbar                      0x0bc6  /*(U+005F LOW LINE)*/
#define XKB_KEY_jot                           0x0bca  /* U+2218 RING OPERATOR */
#define XKB_KEY_quad                          0x0bcc  /* U+2395 APL FUNCTIONAL SYMBOL QUAD */
#define XKB_KEY_uptack                        0x0bce  /* U+22A5 UP TACK */
#define XKB_KEY_circle                        0x0bcf  /* U+25CB WHITE CIRCLE */
#define XKB_KEY_upstile                       0x0bd3  /* U+2308 LEFT CEILING */
#define XKB_KEY_downshoe                      0x0bd6  /*(U+222A UNION)*/
#define XKB_KEY_rightshoe                     0x0bd8  /*(U+2283 SUPERSET OF)*/
#define XKB_KEY_leftshoe                      0x0bda  /*(U+2282 SUBSET OF)*/
#define XKB_KEY_lefttack                      0x0bdc  /* U+22A3 LEFT TACK */
#define XKB_KEY_righttack                     0x0bfc  /* U+22A2 RIGHT TACK */

/*
 * Hebrew
 * Byte 3 = 0x0c
 */

#define XKB_KEY_hebrew_doublelowline          0x0cdf  /* U+2017 DOUBLE LOW LINE */
#define XKB_KEY_hebrew_aleph                  0x0ce0  /* U+05D0 HEBREW LETTER ALEF */
#define XKB_KEY_hebrew_bet                    0x0ce1  /* U+05D1 HEBREW LETTER BET */
#define XKB_KEY_hebrew_beth                   0x0ce1  /* deprecated */
#define XKB_KEY_hebrew_gimel                  0x0ce2  /* U+05D2 HEBREW LETTER GIMEL */
#define XKB_KEY_hebrew_gimmel                 0x0ce2  /* deprecated */
#define XKB_KEY_hebrew_dalet                  0x0ce3  /* U+05D3 HEBREW LETTER DALET */
#define XKB_KEY_hebrew_daleth                 0x0ce3  /* deprecated */
#define XKB_KEY_hebrew_he                     0x0ce4  /* U+05D4 HEBREW LETTER HE */
#define XKB_KEY_hebrew_waw                    0x0ce5  /* U+05D5 HEBREW LETTER VAV */
#define XKB_KEY_hebrew_zain                   0x0ce6  /* U+05D6 HEBREW LETTER ZAYIN */
#define XKB_KEY_hebrew_zayin                  0x0ce6  /* deprecated */
#define XKB_KEY_hebrew_chet                   0x0ce7  /* U+05D7 HEBREW LETTER HET */
#define XKB_KEY_hebrew_het                    0x0ce7  /* deprecated */
#define XKB_KEY_hebrew_tet                    0x0ce8  /* U+05D8 HEBREW LETTER TET */
#define XKB_KEY_hebrew_teth                   0x0ce8  /* deprecated */
#define XKB_KEY_hebrew_yod                    0x0ce9  /* U+05D9 HEBREW LETTER YOD */
#define XKB_KEY_hebrew_finalkaph              0x0cea  /* U+05DA HEBREW LETTER FINAL KAF */
#define XKB_KEY_hebrew_kaph                   0x0ceb  /* U+05DB HEBREW LETTER KAF */
#define XKB_KEY_hebrew_lamed                  0x0cec  /* U+05DC HEBREW LETTER LAMED */
#define XKB_KEY_hebrew_finalmem               0x0ced  /* U+05DD HEBREW LETTER FINAL MEM */
#define XKB_KEY_hebrew_mem                    0x0cee  /* U+05DE HEBREW LETTER MEM */
#define XKB_KEY_hebrew_finalnun               0x0cef  /* U+05DF HEBREW LETTER FINAL NUN */
#define XKB_KEY_hebrew_nun                    0x0cf0  /* U+05E0 HEBREW LETTER NUN */
#define XKB_KEY_hebrew_samech                 0x0cf1  /* U+05E1 HEBREW LETTER SAMEKH */
#define XKB_KEY_hebrew_samekh                 0x0cf1  /* deprecated */
#define XKB_KEY_hebrew_ayin                   0x0cf2  /* U+05E2 HEBREW LETTER AYIN */
#define XKB_KEY_hebrew_finalpe                0x0cf3  /* U+05E3 HEBREW LETTER FINAL PE */
#define XKB_KEY_hebrew_pe                     0x0cf4  /* U+05E4 HEBREW LETTER PE */
#define XKB_KEY_hebrew_finalzade              0x0cf5  /* U+05E5 HEBREW LETTER FINAL TSADI */
#define XKB_KEY_hebrew_finalzadi              0x0cf5  /* deprecated */
#define XKB_KEY_hebrew_zade                   0x0cf6  /* U+05E6 HEBREW LETTER TSADI */
#define XKB_KEY_hebrew_zadi                   0x0cf6  /* deprecated */
#define XKB_KEY_hebrew_qoph                   0x0cf7  /* U+05E7 HEBREW LETTER QOF */
#define XKB_KEY_hebrew_kuf                    0x0cf7  /* deprecated */
#define XKB_KEY_hebrew_resh                   0x0cf8  /* U+05E8 HEBREW LETTER RESH */
#define XKB_KEY_hebrew_shin                   0x0cf9  /* U+05E9 HEBREW LETTER SHIN */
#define XKB_KEY_hebrew_taw                    0x0cfa  /* U+05EA HEBREW LETTER TAV */
#define XKB_KEY_hebrew_taf                    0x0cfa  /* deprecated */
#define XKB_KEY_Hebrew_switch                 0xff7e  /* non-deprecated alias for Mode_switch */

/*
 * Thai
 * Byte 3 = 0x0d
 */

#define XKB_KEY_Thai_kokai                    0x0da1  /* U+0E01 THAI CHARACTER KO KAI */
#define XKB_KEY_Thai_khokhai                  0x0da2  /* U+0E02 THAI CHARACTER KHO KHAI */
#define XKB_KEY_Thai_khokhuat                 0x0da3  /* U+0E03 THAI CHARACTER KHO KHUAT */
#define XKB_KEY_Thai_khokhwai                 0x0da4  /* U+0E04 THAI CHARACTER KHO KHWAI */
#define XKB_KEY_Thai_khokhon                  0x0da5  /* U+0E05 THAI CHARACTER KHO KHON */
#define XKB_KEY_Thai_khorakhang               0x0da6  /* U+0E06 THAI CHARACTER KHO RAKHANG */
#define XKB_KEY_Thai_ngongu                   0x0da7  /* U+0E07 THAI CHARACTER NGO NGU */
#define XKB_KEY_Thai_chochan                  0x0da8  /* U+0E08 THAI CHARACTER CHO CHAN */
#define XKB_KEY_Thai_choching                 0x0da9  /* U+0E09 THAI CHARACTER CHO CHING */
#define XKB_KEY_Thai_chochang                 0x0daa  /* U+0E0A THAI CHARACTER CHO CHANG */
#define XKB_KEY_Thai_soso                     0x0dab  /* U+0E0B THAI CHARACTER SO SO */
#define XKB_KEY_Thai_chochoe                  0x0dac  /* U+0E0C THAI CHARACTER CHO CHOE */
#define XKB_KEY_Thai_yoying                   0x0dad  /* U+0E0D THAI CHARACTER YO YING */
#define XKB_KEY_Thai_dochada                  0x0dae  /* U+0E0E THAI CHARACTER DO CHADA */
#define XKB_KEY_Thai_topatak                  0x0daf  /* U+0E0F THAI CHARACTER TO PATAK */
#define XKB_KEY_Thai_thothan                  0x0db0  /* U+0E10 THAI CHARACTER THO THAN */
#define XKB_KEY_Thai_thonangmontho            0x0db1  /* U+0E11 THAI CHARACTER THO NANGMONTHO */
#define XKB_KEY_Thai_thophuthao               0x0db2  /* U+0E12 THAI CHARACTER THO PHUTHAO */
#define XKB_KEY_Thai_nonen                    0x0db3  /* U+0E13 THAI CHARACTER NO NEN */
#define XKB_KEY_Thai_dodek                    0x0db4  /* U+0E14 THAI CHARACTER DO DEK */
#define XKB_KEY_Thai_totao                    0x0db5  /* U+0E15 THAI CHARACTER TO TAO */
#define XKB_KEY_Thai_thothung                 0x0db6  /* U+0E16 THAI CHARACTER THO THUNG */
#define XKB_KEY_Thai_thothahan                0x0db7  /* U+0E17 THAI CHARACTER THO THAHAN */
#define XKB_KEY_Thai_thothong                 0x0db8  /* U+0E18 THAI CHARACTER THO THONG */
#define XKB_KEY_Thai_nonu                     0x0db9  /* U+0E19 THAI CHARACTER NO NU */
#define XKB_KEY_Thai_bobaimai                 0x0dba  /* U+0E1A THAI CHARACTER BO BAIMAI */
#define XKB_KEY_Thai_popla                    0x0dbb  /* U+0E1B THAI CHARACTER PO PLA */
#define XKB_KEY_Thai_phophung                 0x0dbc  /* U+0E1C THAI CHARACTER PHO PHUNG */
#define XKB_KEY_Thai_fofa                     0x0dbd  /* U+0E1D THAI CHARACTER FO FA */
#define XKB_KEY_Thai_phophan                  0x0dbe  /* U+0E1E THAI CHARACTER PHO PHAN */
#define XKB_KEY_Thai_fofan                    0x0dbf  /* U+0E1F THAI CHARACTER FO FAN */
#define XKB_KEY_Thai_phosamphao               0x0dc0  /* U+0E20 THAI CHARACTER PHO SAMPHAO */
#define XKB_KEY_Thai_moma                     0x0dc1  /* U+0E21 THAI CHARACTER MO MA */
#define XKB_KEY_Thai_yoyak                    0x0dc2  /* U+0E22 THAI CHARACTER YO YAK */
#define XKB_KEY_Thai_rorua                    0x0dc3  /* U+0E23 THAI CHARACTER RO RUA */
#define XKB_KEY_Thai_ru                       0x0dc4  /* U+0E24 THAI CHARACTER RU */
#define XKB_KEY_Thai_loling                   0x0dc5  /* U+0E25 THAI CHARACTER LO LING */
#define XKB_KEY_Thai_lu                       0x0dc6  /* U+0E26 THAI CHARACTER LU */
#define XKB_KEY_Thai_wowaen                   0x0dc7  /* U+0E27 THAI CHARACTER WO WAEN */
#define XKB_KEY_Thai_sosala                   0x0dc8  /* U+0E28 THAI CHARACTER SO SALA */
#define XKB_KEY_Thai_sorusi                   0x0dc9  /* U+0E29 THAI CHARACTER SO RUSI */
#define XKB_KEY_Thai_sosua                    0x0dca  /* U+0E2A THAI CHARACTER SO SUA */
#define XKB_KEY_Thai_hohip                    0x0dcb  /* U+0E2B THAI CHARACTER HO HIP */
#define XKB_KEY_Thai_lochula                  0x0dcc  /* U+0E2C THAI CHARACTER LO CHULA */
#define XKB_KEY_Thai_oang                     0x0dcd  /* U+0E2D THAI CHARACTER O ANG */
#define XKB_KEY_Thai_honokhuk                 0x0dce  /* U+0E2E THAI CHARACTER HO NOKHUK */
#define XKB_KEY_Thai_paiyannoi                0x0dcf  /* U+0E2F THAI CHARACTER PAIYANNOI */
#define XKB_KEY_Thai_saraa                    0x0dd0  /* U+0E30 THAI CHARACTER SARA A */
#define XKB_KEY_Thai_maihanakat               0x0dd1  /* U+0E31 THAI CHARACTER MAI HAN-AKAT */
#define XKB_KEY_Thai_saraaa                   0x0dd2  /* U+0E32 THAI CHARACTER SARA AA */
#define XKB_KEY_Thai_saraam                   0x0dd3  /* U+0E33 THAI CHARACTER SARA AM */
#define XKB_KEY_Thai_sarai                    0x0dd4  /* U+0E34 THAI CHARACTER SARA I */
#define XKB_KEY_Thai_saraii                   0x0dd5  /* U+0E35 THAI CHARACTER SARA II */
#define XKB_KEY_Thai_saraue                   0x0dd6  /* U+0E36 THAI CHARACTER SARA UE */
#define XKB_KEY_Thai_sarauee                  0x0dd7  /* U+0E37 THAI CHARACTER SARA UEE */
#define XKB_KEY_Thai_sarau                    0x0dd8  /* U+0E38 THAI CHARACTER SARA U */
#define XKB_KEY_Thai_sarauu                   0x0dd9  /* U+0E39 THAI CHARACTER SARA UU */
#define XKB_KEY_Thai_phinthu                  0x0dda  /* U+0E3A THAI CHARACTER PHINTHU */
#define XKB_KEY_Thai_maihanakat_maitho        0x0dde  /*(U+0E3E Unassigned code point)*/
#define XKB_KEY_Thai_baht                     0x0ddf  /* U+0E3F THAI CURRENCY SYMBOL BAHT */
#define XKB_KEY_Thai_sarae                    0x0de0  /* U+0E40 THAI CHARACTER SARA E */
#define XKB_KEY_Thai_saraae                   0x0de1  /* U+0E41 THAI CHARACTER SARA AE */
#define XKB_KEY_Thai_sarao                    0x0de2  /* U+0E42 THAI CHARACTER SARA O */
#define XKB_KEY_Thai_saraaimaimuan            0x0de3  /* U+0E43 THAI CHARACTER SARA AI MAIMUAN */
#define XKB_KEY_Thai_saraaimaimalai           0x0de4  /* U+0E44 THAI CHARACTER SARA AI MAIMALAI */
#define XKB_KEY_Thai_lakkhangyao              0x0de5  /* U+0E45 THAI CHARACTER LAKKHANGYAO */
#define XKB_KEY_Thai_maiyamok                 0x0de6  /* U+0E46 THAI CHARACTER MAIYAMOK */
#define XKB_KEY_Thai_maitaikhu                0x0de7  /* U+0E47 THAI CHARACTER MAITAIKHU */
#define XKB_KEY_Thai_maiek                    0x0de8  /* U+0E48 THAI CHARACTER MAI EK */
#define XKB_KEY_Thai_maitho                   0x0de9  /* U+0E49 THAI CHARACTER MAI THO */
#define XKB_KEY_Thai_maitri                   0x0dea  /* U+0E4A THAI CHARACTER MAI TRI */
#define XKB_KEY_Thai_maichattawa              0x0deb  /* U+0E4B THAI CHARACTER MAI CHATTAWA */
#define XKB_KEY_Thai_thanthakhat              0x0dec  /* U+0E4C THAI CHARACTER THANTHAKHAT */
#define XKB_KEY_Thai_nikhahit                 0x0ded  /* U+0E4D THAI CHARACTER NIKHAHIT */
#define XKB_KEY_Thai_leksun                   0x0df0  /* U+0E50 THAI DIGIT ZERO */
#define XKB_KEY_Thai_leknung                  0x0df1  /* U+0E51 THAI DIGIT ONE */
#define XKB_KEY_Thai_leksong                  0x0df2  /* U+0E52 THAI DIGIT TWO */
#define XKB_KEY_Thai_leksam                   0x0df3  /* U+0E53 THAI DIGIT THREE */
#define XKB_KEY_Thai_leksi                    0x0df4  /* U+0E54 THAI DIGIT FOUR */
#define XKB_KEY_Thai_lekha                    0x0df5  /* U+0E55 THAI DIGIT FIVE */
#define XKB_KEY_Thai_lekhok                   0x0df6  /* U+0E56 THAI DIGIT SIX */
#define XKB_KEY_Thai_lekchet                  0x0df7  /* U+0E57 THAI DIGIT SEVEN */
#define XKB_KEY_Thai_lekpaet                  0x0df8  /* U+0E58 THAI DIGIT EIGHT */
#define XKB_KEY_Thai_lekkao                   0x0df9  /* U+0E59 THAI DIGIT NINE */

/*
 * Korean
 * Byte 3 = 0x0e
 */


#define XKB_KEY_Hangul                        0xff31  /* Hangul start/stop(toggle) */
#define XKB_KEY_Hangul_Start                  0xff32  /* Hangul start */
#define XKB_KEY_Hangul_End                    0xff33  /* Hangul end, English start */
#define XKB_KEY_Hangul_Hanja                  0xff34  /* Start Hangul->Hanja Conversion */
#define XKB_KEY_Hangul_Jamo                   0xff35  /* Hangul Jamo mode */
#define XKB_KEY_Hangul_Romaja                 0xff36  /* Hangul Romaja mode */
#define XKB_KEY_Hangul_Codeinput              0xff37  /* Hangul code input mode */
#define XKB_KEY_Hangul_Jeonja                 0xff38  /* Jeonja mode */
#define XKB_KEY_Hangul_Banja                  0xff39  /* Banja mode */
#define XKB_KEY_Hangul_PreHanja               0xff3a  /* Pre Hanja conversion */
#define XKB_KEY_Hangul_PostHanja              0xff3b  /* Post Hanja conversion */
#define XKB_KEY_Hangul_SingleCandidate        0xff3c  /* Single candidate */
#define XKB_KEY_Hangul_MultipleCandidate      0xff3d  /* Multiple candidate */
#define XKB_KEY_Hangul_PreviousCandidate      0xff3e  /* Previous candidate */
#define XKB_KEY_Hangul_Special                0xff3f  /* Special symbols */
#define XKB_KEY_Hangul_switch                 0xff7e  /* non-deprecated alias for Mode_switch */

/* Hangul Consonant Characters */
#define XKB_KEY_Hangul_Kiyeog                 0x0ea1  /* U+3131 HANGUL LETTER KIYEOK */
#define XKB_KEY_Hangul_SsangKiyeog            0x0ea2  /* U+3132 HANGUL LETTER SSANGKIYEOK */
#define XKB_KEY_Hangul_KiyeogSios             0x0ea3  /* U+3133 HANGUL LETTER KIYEOK-SIOS */
#define XKB_KEY_Hangul_Nieun                  0x0ea4  /* U+3134 HANGUL LETTER NIEUN */
#define XKB_KEY_Hangul_NieunJieuj             0x0ea5  /* U+3135 HANGUL LETTER NIEUN-CIEUC */
#define XKB_KEY_Hangul_NieunHieuh             0x0ea6  /* U+3136 HANGUL LETTER NIEUN-HIEUH */
#define XKB_KEY_Hangul_Dikeud                 0x0ea7  /* U+3137 HANGUL LETTER TIKEUT */
#define XKB_KEY_Hangul_SsangDikeud            0x0ea8  /* U+3138 HANGUL LETTER SSANGTIKEUT */
#define XKB_KEY_Hangul_Rieul                  0x0ea9  /* U+3139 HANGUL LETTER RIEUL */
#define XKB_KEY_Hangul_RieulKiyeog            0x0eaa  /* U+313A HANGUL LETTER RIEUL-KIYEOK */
#define XKB_KEY_Hangul_RieulMieum             0x0eab  /* U+313B HANGUL LETTER RIEUL-MIEUM */
#define XKB_KEY_Hangul_RieulPieub             0x0eac  /* U+313C HANGUL LETTER RIEUL-PIEUP */
#define XKB_KEY_Hangul_RieulSios              0x0ead  /* U+313D HANGUL LETTER RIEUL-SIOS */
#define XKB_KEY_Hangul_RieulTieut             0x0eae  /* U+313E HANGUL LETTER RIEUL-THIEUTH */
#define XKB_KEY_Hangul_RieulPhieuf            0x0eaf  /* U+313F HANGUL LETTER RIEUL-PHIEUPH */
#define XKB_KEY_Hangul_RieulHieuh             0x0eb0  /* U+3140 HANGUL LETTER RIEUL-HIEUH */
#define XKB_KEY_Hangul_Mieum                  0x0eb1  /* U+3141 HANGUL LETTER MIEUM */
#define XKB_KEY_Hangul_Pieub                  0x0eb2  /* U+3142 HANGUL LETTER PIEUP */
#define XKB_KEY_Hangul_SsangPieub             0x0eb3  /* U+3143 HANGUL LETTER SSANGPIEUP */
#define XKB_KEY_Hangul_PieubSios              0x0eb4  /* U+3144 HANGUL LETTER PIEUP-SIOS */
#define XKB_KEY_Hangul_Sios                   0x0eb5  /* U+3145 HANGUL LETTER SIOS */
#define XKB_KEY_Hangul_SsangSios              0x0eb6  /* U+3146 HANGUL LETTER SSANGSIOS */
#define XKB_KEY_Hangul_Ieung                  0x0eb7  /* U+3147 HANGUL LETTER IEUNG */
#define XKB_KEY_Hangul_Jieuj                  0x0eb8  /* U+3148 HANGUL LETTER CIEUC */
#define XKB_KEY_Hangul_SsangJieuj             0x0eb9  /* U+3149 HANGUL LETTER SSANGCIEUC */
#define XKB_KEY_Hangul_Cieuc                  0x0eba  /* U+314A HANGUL LETTER CHIEUCH */
#define XKB_KEY_Hangul_Khieuq                 0x0ebb  /* U+314B HANGUL LETTER KHIEUKH */
#define XKB_KEY_Hangul_Tieut                  0x0ebc  /* U+314C HANGUL LETTER THIEUTH */
#define XKB_KEY_Hangul_Phieuf                 0x0ebd  /* U+314D HANGUL LETTER PHIEUPH */
#define XKB_KEY_Hangul_Hieuh                  0x0ebe  /* U+314E HANGUL LETTER HIEUH */

/* Hangul Vowel Characters */
#define XKB_KEY_Hangul_A                      0x0ebf  /* U+314F HANGUL LETTER A */
#define XKB_KEY_Hangul_AE                     0x0ec0  /* U+3150 HANGUL LETTER AE */
#define XKB_KEY_Hangul_YA                     0x0ec1  /* U+3151 HANGUL LETTER YA */
#define XKB_KEY_Hangul_YAE                    0x0ec2  /* U+3152 HANGUL LETTER YAE */
#define XKB_KEY_Hangul_EO                     0x0ec3  /* U+3153 HANGUL LETTER EO */
#define XKB_KEY_Hangul_E                      0x0ec4  /* U+3154 HANGUL LETTER E */
#define XKB_KEY_Hangul_YEO                    0x0ec5  /* U+3155 HANGUL LETTER YEO */
#define XKB_KEY_Hangul_YE                     0x0ec6  /* U+3156 HANGUL LETTER YE */
#define XKB_KEY_Hangul_O                      0x0ec7  /* U+3157 HANGUL LETTER O */
#define XKB_KEY_Hangul_WA                     0x0ec8  /* U+3158 HANGUL LETTER WA */
#define XKB_KEY_Hangul_WAE                    0x0ec9  /* U+3159 HANGUL LETTER WAE */
#define XKB_KEY_Hangul_OE                     0x0eca  /* U+315A HANGUL LETTER OE */
#define XKB_KEY_Hangul_YO                     0x0ecb  /* U+315B HANGUL LETTER YO */
#define XKB_KEY_Hangul_U                      0x0ecc  /* U+315C HANGUL LETTER U */
#define XKB_KEY_Hangul_WEO                    0x0ecd  /* U+315D HANGUL LETTER WEO */
#define XKB_KEY_Hangul_WE                     0x0ece  /* U+315E HANGUL LETTER WE */
#define XKB_KEY_Hangul_WI                     0x0ecf  /* U+315F HANGUL LETTER WI */
#define XKB_KEY_Hangul_YU                     0x0ed0  /* U+3160 HANGUL LETTER YU */
#define XKB_KEY_Hangul_EU                     0x0ed1  /* U+3161 HANGUL LETTER EU */
#define XKB_KEY_Hangul_YI                     0x0ed2  /* U+3162 HANGUL LETTER YI */
#define XKB_KEY_Hangul_I                      0x0ed3  /* U+3163 HANGUL LETTER I */

/* Hangul syllable-final (JongSeong) Characters */
#define XKB_KEY_Hangul_J_Kiyeog               0x0ed4  /* U+11A8 HANGUL JONGSEONG KIYEOK */
#define XKB_KEY_Hangul_J_SsangKiyeog          0x0ed5  /* U+11A9 HANGUL JONGSEONG SSANGKIYEOK */
#define XKB_KEY_Hangul_J_KiyeogSios           0x0ed6  /* U+11AA HANGUL JONGSEONG KIYEOK-SIOS */
#define XKB_KEY_Hangul_J_Nieun                0x0ed7  /* U+11AB HANGUL JONGSEONG NIEUN */
#define XKB_KEY_Hangul_J_NieunJieuj           0x0ed8  /* U+11AC HANGUL JONGSEONG NIEUN-CIEUC */
#define XKB_KEY_Hangul_J_NieunHieuh           0x0ed9  /* U+11AD HANGUL JONGSEONG NIEUN-HIEUH */
#define XKB_KEY_Hangul_J_Dikeud               0x0eda  /* U+11AE HANGUL JONGSEONG TIKEUT */
#define XKB_KEY_Hangul_J_Rieul                0x0edb  /* U+11AF HANGUL JONGSEONG RIEUL */
#define XKB_KEY_Hangul_J_RieulKiyeog          0x0edc  /* U+11B0 HANGUL JONGSEONG RIEUL-KIYEOK */
#define XKB_KEY_Hangul_J_RieulMieum           0x0edd  /* U+11B1 HANGUL JONGSEONG RIEUL-MIEUM */
#define XKB_KEY_Hangul_J_RieulPieub           0x0ede  /* U+11B2 HANGUL JONGSEONG RIEUL-PIEUP */
#define XKB_KEY_Hangul_J_RieulSios            0x0edf  /* U+11B3 HANGUL JONGSEONG RIEUL-SIOS */
#define XKB_KEY_Hangul_J_RieulTieut           0x0ee0  /* U+11B4 HANGUL JONGSEONG RIEUL-THIEUTH */
#define XKB_KEY_Hangul_J_RieulPhieuf          0x0ee1  /* U+11B5 HANGUL JONGSEONG RIEUL-PHIEUPH */
#define XKB_KEY_Hangul_J_RieulHieuh           0x0ee2  /* U+11B6 HANGUL JONGSEONG RIEUL-HIEUH */
#define XKB_KEY_Hangul_J_Mieum                0x0ee3  /* U+11B7 HANGUL JONGSEONG MIEUM */
#define XKB_KEY_Hangul_J_Pieub                0x0ee4  /* U+11B8 HANGUL JONGSEONG PIEUP */
#define XKB_KEY_Hangul_J_PieubSios            0x0ee5  /* U+11B9 HANGUL JONGSEONG PIEUP-SIOS */
#define XKB_KEY_Hangul_J_Sios                 0x0ee6  /* U+11BA HANGUL JONGSEONG SIOS */
#define XKB_KEY_Hangul_J_SsangSios            0x0ee7  /* U+11BB HANGUL JONGSEONG SSANGSIOS */
#define XKB_KEY_Hangul_J_Ieung                0x0ee8  /* U+11BC HANGUL JONGSEONG IEUNG */
#define XKB_KEY_Hangul_J_Jieuj                0x0ee9  /* U+11BD HANGUL JONGSEONG CIEUC */
#define XKB_KEY_Hangul_J_Cieuc                0x0eea  /* U+11BE HANGUL JONGSEONG CHIEUCH */
#define XKB_KEY_Hangul_J_Khieuq               0x0eeb  /* U+11BF HANGUL JONGSEONG KHIEUKH */
#define XKB_KEY_Hangul_J_Tieut                0x0eec  /* U+11C0 HANGUL JONGSEONG THIEUTH */
#define XKB_KEY_Hangul_J_Phieuf               0x0eed  /* U+11C1 HANGUL JONGSEONG PHIEUPH */
#define XKB_KEY_Hangul_J_Hieuh                0x0eee  /* U+11C2 HANGUL JONGSEONG HIEUH */

/* Ancient Hangul Consonant Characters */
#define XKB_KEY_Hangul_RieulYeorinHieuh       0x0eef  /* U+316D HANGUL LETTER RIEUL-YEORINHIEUH */
#define XKB_KEY_Hangul_SunkyeongeumMieum      0x0ef0  /* U+3171 HANGUL LETTER KAPYEOUNMIEUM */
#define XKB_KEY_Hangul_SunkyeongeumPieub      0x0ef1  /* U+3178 HANGUL LETTER KAPYEOUNPIEUP */
#define XKB_KEY_Hangul_PanSios                0x0ef2  /* U+317F HANGUL LETTER PANSIOS */
#define XKB_KEY_Hangul_KkogjiDalrinIeung      0x0ef3  /* U+3181 HANGUL LETTER YESIEUNG */
#define XKB_KEY_Hangul_SunkyeongeumPhieuf     0x0ef4  /* U+3184 HANGUL LETTER KAPYEOUNPHIEUPH */
#define XKB_KEY_Hangul_YeorinHieuh            0x0ef5  /* U+3186 HANGUL LETTER YEORINHIEUH */

/* Ancient Hangul Vowel Characters */
#define XKB_KEY_Hangul_AraeA                  0x0ef6  /* U+318D HANGUL LETTER ARAEA */
#define XKB_KEY_Hangul_AraeAE                 0x0ef7  /* U+318E HANGUL LETTER ARAEAE */

/* Ancient Hangul syllable-final (JongSeong) Characters */
#define XKB_KEY_Hangul_J_PanSios              0x0ef8  /* U+11EB HANGUL JONGSEONG PANSIOS */
#define XKB_KEY_Hangul_J_KkogjiDalrinIeung    0x0ef9  /* U+11F0 HANGUL JONGSEONG YESIEUNG */
#define XKB_KEY_Hangul_J_YeorinHieuh          0x0efa  /* U+11F9 HANGUL JONGSEONG YEORINHIEUH */

/* Korean currency symbol */
#define XKB_KEY_Korean_Won                    0x0eff  /*(U+20A9 WON SIGN)*/


/*
 * Armenian
 */

#define XKB_KEY_Armenian_ligature_ew       0x1000587  /* U+0587 ARMENIAN SMALL LIGATURE ECH YIWN */
#define XKB_KEY_Armenian_full_stop         0x1000589  /* U+0589 ARMENIAN FULL STOP */
#define XKB_KEY_Armenian_verjaket          0x1000589  /* deprecated alias for Armenian_full_stop */
#define XKB_KEY_Armenian_separation_mark   0x100055d  /* U+055D ARMENIAN COMMA */
#define XKB_KEY_Armenian_but               0x100055d  /* deprecated alias for Armenian_separation_mark */
#define XKB_KEY_Armenian_hyphen            0x100058a  /* U+058A ARMENIAN HYPHEN */
#define XKB_KEY_Armenian_yentamna          0x100058a  /* deprecated alias for Armenian_hyphen */
#define XKB_KEY_Armenian_exclam            0x100055c  /* U+055C ARMENIAN EXCLAMATION MARK */
#define XKB_KEY_Armenian_amanak            0x100055c  /* deprecated alias for Armenian_exclam */
#define XKB_KEY_Armenian_accent            0x100055b  /* U+055B ARMENIAN EMPHASIS MARK */
#define XKB_KEY_Armenian_shesht            0x100055b  /* deprecated alias for Armenian_accent */
#define XKB_KEY_Armenian_question          0x100055e  /* U+055E ARMENIAN QUESTION MARK */
#define XKB_KEY_Armenian_paruyk            0x100055e  /* deprecated alias for Armenian_question */
#define XKB_KEY_Armenian_AYB               0x1000531  /* U+0531 ARMENIAN CAPITAL LETTER AYB */
#define XKB_KEY_Armenian_ayb               0x1000561  /* U+0561 ARMENIAN SMALL LETTER AYB */
#define XKB_KEY_Armenian_BEN               0x1000532  /* U+0532 ARMENIAN CAPITAL LETTER BEN */
#define XKB_KEY_Armenian_ben               0x1000562  /* U+0562 ARMENIAN SMALL LETTER BEN */
#define XKB_KEY_Armenian_GIM               0x1000533  /* U+0533 ARMENIAN CAPITAL LETTER GIM */
#define XKB_KEY_Armenian_gim               0x1000563  /* U+0563 ARMENIAN SMALL LETTER GIM */
#define XKB_KEY_Armenian_DA                0x1000534  /* U+0534 ARMENIAN CAPITAL LETTER DA */
#define XKB_KEY_Armenian_da                0x1000564  /* U+0564 ARMENIAN SMALL LETTER DA */
#define XKB_KEY_Armenian_YECH              0x1000535  /* U+0535 ARMENIAN CAPITAL LETTER ECH */
#define XKB_KEY_Armenian_yech              0x1000565  /* U+0565 ARMENIAN SMALL LETTER ECH */
#define XKB_KEY_Armenian_ZA                0x1000536  /* U+0536 ARMENIAN CAPITAL LETTER ZA */
#define XKB_KEY_Armenian_za                0x1000566  /* U+0566 ARMENIAN SMALL LETTER ZA */
#define XKB_KEY_Armenian_E                 0x1000537  /* U+0537 ARMENIAN CAPITAL LETTER EH */
#define XKB_KEY_Armenian_e                 0x1000567  /* U+0567 ARMENIAN SMALL LETTER EH */
#define XKB_KEY_Armenian_AT                0x1000538  /* U+0538 ARMENIAN CAPITAL LETTER ET */
#define XKB_KEY_Armenian_at                0x1000568  /* U+0568 ARMENIAN SMALL LETTER ET */
#define XKB_KEY_Armenian_TO                0x1000539  /* U+0539 ARMENIAN CAPITAL LETTER TO */
#define XKB_KEY_Armenian_to                0x1000569  /* U+0569 ARMENIAN SMALL LETTER TO */
#define XKB_KEY_Armenian_ZHE               0x100053a  /* U+053A ARMENIAN CAPITAL LETTER ZHE */
#define XKB_KEY_Armenian_zhe               0x100056a  /* U+056A ARMENIAN SMALL LETTER ZHE */
#define XKB_KEY_Armenian_INI               0x100053b  /* U+053B ARMENIAN CAPITAL LETTER INI */
#define XKB_KEY_Armenian_ini               0x100056b  /* U+056B ARMENIAN SMALL LETTER INI */
#define XKB_KEY_Armenian_LYUN              0x100053c  /* U+053C ARMENIAN CAPITAL LETTER LIWN */
#define XKB_KEY_Armenian_lyun              0x100056c  /* U+056C ARMENIAN SMALL LETTER LIWN */
#define XKB_KEY_Armenian_KHE               0x100053d  /* U+053D ARMENIAN CAPITAL LETTER XEH */
#define XKB_KEY_Armenian_khe               0x100056d  /* U+056D ARMENIAN SMALL LETTER XEH */
#define XKB_KEY_Armenian_TSA               0x100053e  /* U+053E ARMENIAN CAPITAL LETTER CA */
#define XKB_KEY_Armenian_tsa               0x100056e  /* U+056E ARMENIAN SMALL LETTER CA */
#define XKB_KEY_Armenian_KEN               0x100053f  /* U+053F ARMENIAN CAPITAL LETTER KEN */
#define XKB_KEY_Armenian_ken               0x100056f  /* U+056F ARMENIAN SMALL LETTER KEN */
#define XKB_KEY_Armenian_HO                0x1000540  /* U+0540 ARMENIAN CAPITAL LETTER HO */
#define XKB_KEY_Armenian_ho                0x1000570  /* U+0570 ARMENIAN SMALL LETTER HO */
#define XKB_KEY_Armenian_DZA               0x1000541  /* U+0541 ARMENIAN CAPITAL LETTER JA */
#define XKB_KEY_Armenian_dza               0x1000571  /* U+0571 ARMENIAN SMALL LETTER JA */
#define XKB_KEY_Armenian_GHAT              0x1000542  /* U+0542 ARMENIAN CAPITAL LETTER GHAD */
#define XKB_KEY_Armenian_ghat              0x1000572  /* U+0572 ARMENIAN SMALL LETTER GHAD */
#define XKB_KEY_Armenian_TCHE              0x1000543  /* U+0543 ARMENIAN CAPITAL LETTER CHEH */
#define XKB_KEY_Armenian_tche              0x1000573  /* U+0573 ARMENIAN SMALL LETTER CHEH */
#define XKB_KEY_Armenian_MEN               0x1000544  /* U+0544 ARMENIAN CAPITAL LETTER MEN */
#define XKB_KEY_Armenian_men               0x1000574  /* U+0574 ARMENIAN SMALL LETTER MEN */
#define XKB_KEY_Armenian_HI                0x1000545  /* U+0545 ARMENIAN CAPITAL LETTER YI */
#define XKB_KEY_Armenian_hi                0x1000575  /* U+0575 ARMENIAN SMALL LETTER YI */
#define XKB_KEY_Armenian_NU                0x1000546  /* U+0546 ARMENIAN CAPITAL LETTER NOW */
#define XKB_KEY_Armenian_nu                0x1000576  /* U+0576 ARMENIAN SMALL LETTER NOW */
#define XKB_KEY_Armenian_SHA               0x1000547  /* U+0547 ARMENIAN CAPITAL LETTER SHA */
#define XKB_KEY_Armenian_sha               0x1000577  /* U+0577 ARMENIAN SMALL LETTER SHA */
#define XKB_KEY_Armenian_VO                0x1000548  /* U+0548 ARMENIAN CAPITAL LETTER VO */
#define XKB_KEY_Armenian_vo                0x1000578  /* U+0578 ARMENIAN SMALL LETTER VO */
#define XKB_KEY_Armenian_CHA               0x1000549  /* U+0549 ARMENIAN CAPITAL LETTER CHA */
#define XKB_KEY_Armenian_cha               0x1000579  /* U+0579 ARMENIAN SMALL LETTER CHA */
#define XKB_KEY_Armenian_PE                0x100054a  /* U+054A ARMENIAN CAPITAL LETTER PEH */
#define XKB_KEY_Armenian_pe                0x100057a  /* U+057A ARMENIAN SMALL LETTER PEH */
#define XKB_KEY_Armenian_JE                0x100054b  /* U+054B ARMENIAN CAPITAL LETTER JHEH */
#define XKB_KEY_Armenian_je                0x100057b  /* U+057B ARMENIAN SMALL LETTER JHEH */
#define XKB_KEY_Armenian_RA                0x100054c  /* U+054C ARMENIAN CAPITAL LETTER RA */
#define XKB_KEY_Armenian_ra                0x100057c  /* U+057C ARMENIAN SMALL LETTER RA */
#define XKB_KEY_Armenian_SE                0x100054d  /* U+054D ARMENIAN CAPITAL LETTER SEH */
#define XKB_KEY_Armenian_se                0x100057d  /* U+057D ARMENIAN SMALL LETTER SEH */
#define XKB_KEY_Armenian_VEV               0x100054e  /* U+054E ARMENIAN CAPITAL LETTER VEW */
#define XKB_KEY_Armenian_vev               0x100057e  /* U+057E ARMENIAN SMALL LETTER VEW */
#define XKB_KEY_Armenian_TYUN              0x100054f  /* U+054F ARMENIAN CAPITAL LETTER TIWN */
#define XKB_KEY_Armenian_tyun              0x100057f  /* U+057F ARMENIAN SMALL LETTER TIWN */
#define XKB_KEY_Armenian_RE                0x1000550  /* U+0550 ARMENIAN CAPITAL LETTER REH */
#define XKB_KEY_Armenian_re                0x1000580  /* U+0580 ARMENIAN SMALL LETTER REH */
#define XKB_KEY_Armenian_TSO               0x1000551  /* U+0551 ARMENIAN CAPITAL LETTER CO */
#define XKB_KEY_Armenian_tso               0x1000581  /* U+0581 ARMENIAN SMALL LETTER CO */
#define XKB_KEY_Armenian_VYUN              0x1000552  /* U+0552 ARMENIAN CAPITAL LETTER YIWN */
#define XKB_KEY_Armenian_vyun              0x1000582  /* U+0582 ARMENIAN SMALL LETTER YIWN */
#define XKB_KEY_Armenian_PYUR              0x1000553  /* U+0553 ARMENIAN CAPITAL LETTER PIWR */
#define XKB_KEY_Armenian_pyur              0x1000583  /* U+0583 ARMENIAN SMALL LETTER PIWR */
#define XKB_KEY_Armenian_KE                0x1000554  /* U+0554 ARMENIAN CAPITAL LETTER KEH */
#define XKB_KEY_Armenian_ke                0x1000584  /* U+0584 ARMENIAN SMALL LETTER KEH */
#define XKB_KEY_Armenian_O                 0x1000555  /* U+0555 ARMENIAN CAPITAL LETTER OH */
#define XKB_KEY_Armenian_o                 0x1000585  /* U+0585 ARMENIAN SMALL LETTER OH */
#define XKB_KEY_Armenian_FE                0x1000556  /* U+0556 ARMENIAN CAPITAL LETTER FEH */
#define XKB_KEY_Armenian_fe                0x1000586  /* U+0586 ARMENIAN SMALL LETTER FEH */
#define XKB_KEY_Armenian_apostrophe        0x100055a  /* U+055A ARMENIAN APOSTROPHE */

/*
 * Georgian
 */

#define XKB_KEY_Georgian_an                0x10010d0  /* U+10D0 GEORGIAN LETTER AN */
#define XKB_KEY_Georgian_ban               0x10010d1  /* U+10D1 GEORGIAN LETTER BAN */
#define XKB_KEY_Georgian_gan               0x10010d2  /* U+10D2 GEORGIAN LETTER GAN */
#define XKB_KEY_Georgian_don               0x10010d3  /* U+10D3 GEORGIAN LETTER DON */
#define XKB_KEY_Georgian_en                0x10010d4  /* U+10D4 GEORGIAN LETTER EN */
#define XKB_KEY_Georgian_vin               0x10010d5  /* U+10D5 GEORGIAN LETTER VIN */
#define XKB_KEY_Georgian_zen               0x10010d6  /* U+10D6 GEORGIAN LETTER ZEN */
#define XKB_KEY_Georgian_tan               0x10010d7  /* U+10D7 GEORGIAN LETTER TAN */
#define XKB_KEY_Georgian_in                0x10010d8  /* U+10D8 GEORGIAN LETTER IN */
#define XKB_KEY_Georgian_kan               0x10010d9  /* U+10D9 GEORGIAN LETTER KAN */
#define XKB_KEY_Georgian_las               0x10010da  /* U+10DA GEORGIAN LETTER LAS */
#define XKB_KEY_Georgian_man               0x10010db  /* U+10DB GEORGIAN LETTER MAN */
#define XKB_KEY_Georgian_nar               0x10010dc  /* U+10DC GEORGIAN LETTER NAR */
#define XKB_KEY_Georgian_on                0x10010dd  /* U+10DD GEORGIAN LETTER ON */
#define XKB_KEY_Georgian_par               0x10010de  /* U+10DE GEORGIAN LETTER PAR */
#define XKB_KEY_Georgian_zhar              0x10010df  /* U+10DF GEORGIAN LETTER ZHAR */
#define XKB_KEY_Georgian_rae               0x10010e0  /* U+10E0 GEORGIAN LETTER RAE */
#define XKB_KEY_Georgian_san               0x10010e1  /* U+10E1 GEORGIAN LETTER SAN */
#define XKB_KEY_Georgian_tar               0x10010e2  /* U+10E2 GEORGIAN LETTER TAR */
#define XKB_KEY_Georgian_un                0x10010e3  /* U+10E3 GEORGIAN LETTER UN */
#define XKB_KEY_Georgian_phar              0x10010e4  /* U+10E4 GEORGIAN LETTER PHAR */
#define XKB_KEY_Georgian_khar              0x10010e5  /* U+10E5 GEORGIAN LETTER KHAR */
#define XKB_KEY_Georgian_ghan              0x10010e6  /* U+10E6 GEORGIAN LETTER GHAN */
#define XKB_KEY_Georgian_qar               0x10010e7  /* U+10E7 GEORGIAN LETTER QAR */
#define XKB_KEY_Georgian_shin              0x10010e8  /* U+10E8 GEORGIAN LETTER SHIN */
#define XKB_KEY_Georgian_chin              0x10010e9  /* U+10E9 GEORGIAN LETTER CHIN */
#define XKB_KEY_Georgian_can               0x10010ea  /* U+10EA GEORGIAN LETTER CAN */
#define XKB_KEY_Georgian_jil               0x10010eb  /* U+10EB GEORGIAN LETTER JIL */
#define XKB_KEY_Georgian_cil               0x10010ec  /* U+10EC GEORGIAN LETTER CIL */
#define XKB_KEY_Georgian_char              0x10010ed  /* U+10ED GEORGIAN LETTER CHAR */
#define XKB_KEY_Georgian_xan               0x10010ee  /* U+10EE GEORGIAN LETTER XAN */
#define XKB_KEY_Georgian_jhan              0x10010ef  /* U+10EF GEORGIAN LETTER JHAN */
#define XKB_KEY_Georgian_hae               0x10010f0  /* U+10F0 GEORGIAN LETTER HAE */
#define XKB_KEY_Georgian_he                0x10010f1  /* U+10F1 GEORGIAN LETTER HE */
#define XKB_KEY_Georgian_hie               0x10010f2  /* U+10F2 GEORGIAN LETTER HIE */
#define XKB_KEY_Georgian_we                0x10010f3  /* U+10F3 GEORGIAN LETTER WE */
#define XKB_KEY_Georgian_har               0x10010f4  /* U+10F4 GEORGIAN LETTER HAR */
#define XKB_KEY_Georgian_hoe               0x10010f5  /* U+10F5 GEORGIAN LETTER HOE */
#define XKB_KEY_Georgian_fi                0x10010f6  /* U+10F6 GEORGIAN LETTER FI */

/*
 * Azeri (and other Turkic or Caucasian languages)
 */

/* latin */
#define XKB_KEY_Xabovedot                  0x1001e8a  /* U+1E8A LATIN CAPITAL LETTER X WITH DOT ABOVE */
#define XKB_KEY_Ibreve                     0x100012c  /* U+012C LATIN CAPITAL LETTER I WITH BREVE */
#define XKB_KEY_Zstroke                    0x10001b5  /* U+01B5 LATIN CAPITAL LETTER Z WITH STROKE */
#define XKB_KEY_Gcaron                     0x10001e6  /* U+01E6 LATIN CAPITAL LETTER G WITH CARON */
#define XKB_KEY_Ocaron                     0x10001d1  /* U+01D1 LATIN CAPITAL LETTER O WITH CARON */
#define XKB_KEY_Obarred                    0x100019f  /* U+019F LATIN CAPITAL LETTER O WITH MIDDLE TILDE */
#define XKB_KEY_xabovedot                  0x1001e8b  /* U+1E8B LATIN SMALL LETTER X WITH DOT ABOVE */
#define XKB_KEY_ibreve                     0x100012d  /* U+012D LATIN SMALL LETTER I WITH BREVE */
#define XKB_KEY_zstroke                    0x10001b6  /* U+01B6 LATIN SMALL LETTER Z WITH STROKE */
#define XKB_KEY_gcaron                     0x10001e7  /* U+01E7 LATIN SMALL LETTER G WITH CARON */
#define XKB_KEY_ocaron                     0x10001d2  /* U+01D2 LATIN SMALL LETTER O WITH CARON */
#define XKB_KEY_obarred                    0x1000275  /* U+0275 LATIN SMALL LETTER BARRED O */
#define XKB_KEY_SCHWA                      0x100018f  /* U+018F LATIN CAPITAL LETTER SCHWA */
#define XKB_KEY_schwa                      0x1000259  /* U+0259 LATIN SMALL LETTER SCHWA */
#define XKB_KEY_EZH                        0x10001b7  /* U+01B7 LATIN CAPITAL LETTER EZH */
#define XKB_KEY_ezh                        0x1000292  /* U+0292 LATIN SMALL LETTER EZH */
/* those are not really Caucasus */
/* For Inupiak */
#define XKB_KEY_Lbelowdot                  0x1001e36  /* U+1E36 LATIN CAPITAL LETTER L WITH DOT BELOW */
#define XKB_KEY_lbelowdot                  0x1001e37  /* U+1E37 LATIN SMALL LETTER L WITH DOT BELOW */

/*
 * Vietnamese
 */

#define XKB_KEY_Abelowdot                  0x1001ea0  /* U+1EA0 LATIN CAPITAL LETTER A WITH DOT BELOW */
#define XKB_KEY_abelowdot                  0x1001ea1  /* U+1EA1 LATIN SMALL LETTER A WITH DOT BELOW */
#define XKB_KEY_Ahook                      0x1001ea2  /* U+1EA2 LATIN CAPITAL LETTER A WITH HOOK ABOVE */
#define XKB_KEY_ahook                      0x1001ea3  /* U+1EA3 LATIN SMALL LETTER A WITH HOOK ABOVE */
#define XKB_KEY_Acircumflexacute           0x1001ea4  /* U+1EA4 LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND ACUTE */
#define XKB_KEY_acircumflexacute           0x1001ea5  /* U+1EA5 LATIN SMALL LETTER A WITH CIRCUMFLEX AND ACUTE */
#define XKB_KEY_Acircumflexgrave           0x1001ea6  /* U+1EA6 LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND GRAVE */
#define XKB_KEY_acircumflexgrave           0x1001ea7  /* U+1EA7 LATIN SMALL LETTER A WITH CIRCUMFLEX AND GRAVE */
#define XKB_KEY_Acircumflexhook            0x1001ea8  /* U+1EA8 LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE */
#define XKB_KEY_acircumflexhook            0x1001ea9  /* U+1EA9 LATIN SMALL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE */
#define XKB_KEY_Acircumflextilde           0x1001eaa  /* U+1EAA LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND TILDE */
#define XKB_KEY_acircumflextilde           0x1001eab  /* U+1EAB LATIN SMALL LETTER A WITH CIRCUMFLEX AND TILDE */
#define XKB_KEY_Acircumflexbelowdot        0x1001eac  /* U+1EAC LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND DOT BELOW */
#define XKB_KEY_acircumflexbelowdot        0x1001ead  /* U+1EAD LATIN SMALL LETTER A WITH CIRCUMFLEX AND DOT BELOW */
#define XKB_KEY_Abreveacute                0x1001eae  /* U+1EAE LATIN CAPITAL LETTER A WITH BREVE AND ACUTE */
#define XKB_KEY_abreveacute                0x1001eaf  /* U+1EAF LATIN SMALL LETTER A WITH BREVE AND ACUTE */
#define XKB_KEY_Abrevegrave                0x1001eb0  /* U+1EB0 LATIN CAPITAL LETTER A WITH BREVE AND GRAVE */
#define XKB_KEY_abrevegrave                0x1001eb1  /* U+1EB1 LATIN SMALL LETTER A WITH BREVE AND GRAVE */
#define XKB_KEY_Abrevehook                 0x1001eb2  /* U+1EB2 LATIN CAPITAL LETTER A WITH BREVE AND HOOK ABOVE */
#define XKB_KEY_abrevehook                 0x1001eb3  /* U+1EB3 LATIN SMALL LETTER A WITH BREVE AND HOOK ABOVE */
#define XKB_KEY_Abrevetilde                0x1001eb4  /* U+1EB4 LATIN CAPITAL LETTER A WITH BREVE AND TILDE */
#define XKB_KEY_abrevetilde                0x1001eb5  /* U+1EB5 LATIN SMALL LETTER A WITH BREVE AND TILDE */
#define XKB_KEY_Abrevebelowdot             0x1001eb6  /* U+1EB6 LATIN CAPITAL LETTER A WITH BREVE AND DOT BELOW */
#define XKB_KEY_abrevebelowdot             0x1001eb7  /* U+1EB7 LATIN SMALL LETTER A WITH BREVE AND DOT BELOW */
#define XKB_KEY_Ebelowdot                  0x1001eb8  /* U+1EB8 LATIN CAPITAL LETTER E WITH DOT BELOW */
#define XKB_KEY_ebelowdot                  0x1001eb9  /* U+1EB9 LATIN SMALL LETTER E WITH DOT BELOW */
#define XKB_KEY_Ehook                      0x1001eba  /* U+1EBA LATIN CAPITAL LETTER E WITH HOOK ABOVE */
#define XKB_KEY_ehook                      0x1001ebb  /* U+1EBB LATIN SMALL LETTER E WITH HOOK ABOVE */
#define XKB_KEY_Etilde                     0x1001ebc  /* U+1EBC LATIN CAPITAL LETTER E WITH TILDE */
#define XKB_KEY_etilde                     0x1001ebd  /* U+1EBD LATIN SMALL LETTER E WITH TILDE */
#define XKB_KEY_Ecircumflexacute           0x1001ebe  /* U+1EBE LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND ACUTE */
#define XKB_KEY_ecircumflexacute           0x1001ebf  /* U+1EBF LATIN SMALL LETTER E WITH CIRCUMFLEX AND ACUTE */
#define XKB_KEY_Ecircumflexgrave           0x1001ec0  /* U+1EC0 LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND GRAVE */
#define XKB_KEY_ecircumflexgrave           0x1001ec1  /* U+1EC1 LATIN SMALL LETTER E WITH CIRCUMFLEX AND GRAVE */
#define XKB_KEY_Ecircumflexhook            0x1001ec2  /* U+1EC2 LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE */
#define XKB_KEY_ecircumflexhook            0x1001ec3  /* U+1EC3 LATIN SMALL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE */
#define XKB_KEY_Ecircumflextilde           0x1001ec4  /* U+1EC4 LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND TILDE */
#define XKB_KEY_ecircumflextilde           0x1001ec5  /* U+1EC5 LATIN SMALL LETTER E WITH CIRCUMFLEX AND TILDE */
#define XKB_KEY_Ecircumflexbelowdot        0x1001ec6  /* U+1EC6 LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND DOT BELOW */
#define XKB_KEY_ecircumflexbelowdot        0x1001ec7  /* U+1EC7 LATIN SMALL LETTER E WITH CIRCUMFLEX AND DOT BELOW */
#define XKB_KEY_Ihook                      0x1001ec8  /* U+1EC8 LATIN CAPITAL LETTER I WITH HOOK ABOVE */
#define XKB_KEY_ihook                      0x1001ec9  /* U+1EC9 LATIN SMALL LETTER I WITH HOOK ABOVE */
#define XKB_KEY_Ibelowdot                  0x1001eca  /* U+1ECA LATIN CAPITAL LETTER I WITH DOT BELOW */
#define XKB_KEY_ibelowdot                  0x1001ecb  /* U+1ECB LATIN SMALL LETTER I WITH DOT BELOW */
#define XKB_KEY_Obelowdot                  0x1001ecc  /* U+1ECC LATIN CAPITAL LETTER O WITH DOT BELOW */
#define XKB_KEY_obelowdot                  0x1001ecd  /* U+1ECD LATIN SMALL LETTER O WITH DOT BELOW */
#define XKB_KEY_Ohook                      0x1001ece  /* U+1ECE LATIN CAPITAL LETTER O WITH HOOK ABOVE */
#define XKB_KEY_ohook                      0x1001ecf  /* U+1ECF LATIN SMALL LETTER O WITH HOOK ABOVE */
#define XKB_KEY_Ocircumflexacute           0x1001ed0  /* U+1ED0 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND ACUTE */
#define XKB_KEY_ocircumflexacute           0x1001ed1  /* U+1ED1 LATIN SMALL LETTER O WITH CIRCUMFLEX AND ACUTE */
#define XKB_KEY_Ocircumflexgrave           0x1001ed2  /* U+1ED2 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND GRAVE */
#define XKB_KEY_ocircumflexgrave           0x1001ed3  /* U+1ED3 LATIN SMALL LETTER O WITH CIRCUMFLEX AND GRAVE */
#define XKB_KEY_Ocircumflexhook            0x1001ed4  /* U+1ED4 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE */
#define XKB_KEY_ocircumflexhook            0x1001ed5  /* U+1ED5 LATIN SMALL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE */
#define XKB_KEY_Ocircumflextilde           0x1001ed6  /* U+1ED6 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND TILDE */
#define XKB_KEY_ocircumflextilde           0x1001ed7  /* U+1ED7 LATIN SMALL LETTER O WITH CIRCUMFLEX AND TILDE */
#define XKB_KEY_Ocircumflexbelowdot        0x1001ed8  /* U+1ED8 LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND DOT BELOW */
#define XKB_KEY_ocircumflexbelowdot        0x1001ed9  /* U+1ED9 LATIN SMALL LETTER O WITH CIRCUMFLEX AND DOT BELOW */
#define XKB_KEY_Ohornacute                 0x1001eda  /* U+1EDA LATIN CAPITAL LETTER O WITH HORN AND ACUTE */
#define XKB_KEY_ohornacute                 0x1001edb  /* U+1EDB LATIN SMALL LETTER O WITH HORN AND ACUTE */
#define XKB_KEY_Ohorngrave                 0x1001edc  /* U+1EDC LATIN CAPITAL LETTER O WITH HORN AND GRAVE */
#define XKB_KEY_ohorngrave                 0x1001edd  /* U+1EDD LATIN SMALL LETTER O WITH HORN AND GRAVE */
#define XKB_KEY_Ohornhook                  0x1001ede  /* U+1EDE LATIN CAPITAL LETTER O WITH HORN AND HOOK ABOVE */
#define XKB_KEY_ohornhook                  0x1001edf  /* U+1EDF LATIN SMALL LETTER O WITH HORN AND HOOK ABOVE */
#define XKB_KEY_Ohorntilde                 0x1001ee0  /* U+1EE0 LATIN CAPITAL LETTER O WITH HORN AND TILDE */
#define XKB_KEY_ohorntilde                 0x1001ee1  /* U+1EE1 LATIN SMALL LETTER O WITH HORN AND TILDE */
#define XKB_KEY_Ohornbelowdot              0x1001ee2  /* U+1EE2 LATIN CAPITAL LETTER O WITH HORN AND DOT BELOW */
#define XKB_KEY_ohornbelowdot              0x1001ee3  /* U+1EE3 LATIN SMALL LETTER O WITH HORN AND DOT BELOW */
#define XKB_KEY_Ubelowdot                  0x1001ee4  /* U+1EE4 LATIN CAPITAL LETTER U WITH DOT BELOW */
#define XKB_KEY_ubelowdot                  0x1001ee5  /* U+1EE5 LATIN SMALL LETTER U WITH DOT BELOW */
#define XKB_KEY_Uhook                      0x1001ee6  /* U+1EE6 LATIN CAPITAL LETTER U WITH HOOK ABOVE */
#define XKB_KEY_uhook                      0x1001ee7  /* U+1EE7 LATIN SMALL LETTER U WITH HOOK ABOVE */
#define XKB_KEY_Uhornacute                 0x1001ee8  /* U+1EE8 LATIN CAPITAL LETTER U WITH HORN AND ACUTE */
#define XKB_KEY_uhornacute                 0x1001ee9  /* U+1EE9 LATIN SMALL LETTER U WITH HORN AND ACUTE */
#define XKB_KEY_Uhorngrave                 0x1001eea  /* U+1EEA LATIN CAPITAL LETTER U WITH HORN AND GRAVE */
#define XKB_KEY_uhorngrave                 0x1001eeb  /* U+1EEB LATIN SMALL LETTER U WITH HORN AND GRAVE */
#define XKB_KEY_Uhornhook                  0x1001eec  /* U+1EEC LATIN CAPITAL LETTER U WITH HORN AND HOOK ABOVE */
#define XKB_KEY_uhornhook                  0x1001eed  /* U+1EED LATIN SMALL LETTER U WITH HORN AND HOOK ABOVE */
#define XKB_KEY_Uhorntilde                 0x1001eee  /* U+1EEE LATIN CAPITAL LETTER U WITH HORN AND TILDE */
#define XKB_KEY_uhorntilde                 0x1001eef  /* U+1EEF LATIN SMALL LETTER U WITH HORN AND TILDE */
#define XKB_KEY_Uhornbelowdot              0x1001ef0  /* U+1EF0 LATIN CAPITAL LETTER U WITH HORN AND DOT BELOW */
#define XKB_KEY_uhornbelowdot              0x1001ef1  /* U+1EF1 LATIN SMALL LETTER U WITH HORN AND DOT BELOW */
#define XKB_KEY_Ybelowdot                  0x1001ef4  /* U+1EF4 LATIN CAPITAL LETTER Y WITH DOT BELOW */
#define XKB_KEY_ybelowdot                  0x1001ef5  /* U+1EF5 LATIN SMALL LETTER Y WITH DOT BELOW */
#define XKB_KEY_Yhook                      0x1001ef6  /* U+1EF6 LATIN CAPITAL LETTER Y WITH HOOK ABOVE */
#define XKB_KEY_yhook                      0x1001ef7  /* U+1EF7 LATIN SMALL LETTER Y WITH HOOK ABOVE */
#define XKB_KEY_Ytilde                     0x1001ef8  /* U+1EF8 LATIN CAPITAL LETTER Y WITH TILDE */
#define XKB_KEY_ytilde                     0x1001ef9  /* U+1EF9 LATIN SMALL LETTER Y WITH TILDE */
#define XKB_KEY_Ohorn                      0x10001a0  /* U+01A0 LATIN CAPITAL LETTER O WITH HORN */
#define XKB_KEY_ohorn                      0x10001a1  /* U+01A1 LATIN SMALL LETTER O WITH HORN */
#define XKB_KEY_Uhorn                      0x10001af  /* U+01AF LATIN CAPITAL LETTER U WITH HORN */
#define XKB_KEY_uhorn                      0x10001b0  /* U+01B0 LATIN SMALL LETTER U WITH HORN */
#define XKB_KEY_combining_tilde            0x1000303  /* U+0303 COMBINING TILDE */
#define XKB_KEY_combining_grave            0x1000300  /* U+0300 COMBINING GRAVE ACCENT */
#define XKB_KEY_combining_acute            0x1000301  /* U+0301 COMBINING ACUTE ACCENT */
#define XKB_KEY_combining_hook             0x1000309  /* U+0309 COMBINING HOOK ABOVE */
#define XKB_KEY_combining_belowdot         0x1000323  /* U+0323 COMBINING DOT BELOW */


#define XKB_KEY_EcuSign                    0x10020a0  /* U+20A0 EURO-CURRENCY SIGN */
#define XKB_KEY_ColonSign                  0x10020a1  /* U+20A1 COLON SIGN */
#define XKB_KEY_CruzeiroSign               0x10020a2  /* U+20A2 CRUZEIRO SIGN */
#define XKB_KEY_FFrancSign                 0x10020a3  /* U+20A3 FRENCH FRANC SIGN */
#define XKB_KEY_LiraSign                   0x10020a4  /* U+20A4 LIRA SIGN */
#define XKB_KEY_MillSign                   0x10020a5  /* U+20A5 MILL SIGN */
#define XKB_KEY_NairaSign                  0x10020a6  /* U+20A6 NAIRA SIGN */
#define XKB_KEY_PesetaSign                 0x10020a7  /* U+20A7 PESETA SIGN */
#define XKB_KEY_RupeeSign                  0x10020a8  /* U+20A8 RUPEE SIGN */
#define XKB_KEY_WonSign                    0x10020a9  /* U+20A9 WON SIGN */
#define XKB_KEY_NewSheqelSign              0x10020aa  /* U+20AA NEW SHEQEL SIGN */
#define XKB_KEY_DongSign                   0x10020ab  /* U+20AB DONG SIGN */
#define XKB_KEY_EuroSign                      0x20ac  /* U+20AC EURO SIGN */

/* one, two and three are defined above. */
#define XKB_KEY_zerosuperior               0x1002070  /* U+2070 SUPERSCRIPT ZERO */
#define XKB_KEY_foursuperior               0x1002074  /* U+2074 SUPERSCRIPT FOUR */
#define XKB_KEY_fivesuperior               0x1002075  /* U+2075 SUPERSCRIPT FIVE */
#define XKB_KEY_sixsuperior                0x1002076  /* U+2076 SUPERSCRIPT SIX */
#define XKB_KEY_sevensuperior              0x1002077  /* U+2077 SUPERSCRIPT SEVEN */
#define XKB_KEY_eightsuperior              0x1002078  /* U+2078 SUPERSCRIPT EIGHT */
#define XKB_KEY_ninesuperior               0x1002079  /* U+2079 SUPERSCRIPT NINE */
#define XKB_KEY_zerosubscript              0x1002080  /* U+2080 SUBSCRIPT ZERO */
#define XKB_KEY_onesubscript               0x1002081  /* U+2081 SUBSCRIPT ONE */
#define XKB_KEY_twosubscript               0x1002082  /* U+2082 SUBSCRIPT TWO */
#define XKB_KEY_threesubscript             0x1002083  /* U+2083 SUBSCRIPT THREE */
#define XKB_KEY_foursubscript              0x1002084  /* U+2084 SUBSCRIPT FOUR */
#define XKB_KEY_fivesubscript              0x1002085  /* U+2085 SUBSCRIPT FIVE */
#define XKB_KEY_sixsubscript               0x1002086  /* U+2086 SUBSCRIPT SIX */
#define XKB_KEY_sevensubscript             0x1002087  /* U+2087 SUBSCRIPT SEVEN */
#define XKB_KEY_eightsubscript             0x1002088  /* U+2088 SUBSCRIPT EIGHT */
#define XKB_KEY_ninesubscript              0x1002089  /* U+2089 SUBSCRIPT NINE */
#define XKB_KEY_partdifferential           0x1002202  /* U+2202 PARTIAL DIFFERENTIAL */
#define XKB_KEY_emptyset                   0x1002205  /* U+2205 EMPTY SET */
#define XKB_KEY_elementof                  0x1002208  /* U+2208 ELEMENT OF */
#define XKB_KEY_notelementof               0x1002209  /* U+2209 NOT AN ELEMENT OF */
#define XKB_KEY_containsas                 0x100220b  /* U+220B CONTAINS AS MEMBER */
#define XKB_KEY_squareroot                 0x100221a  /* U+221A SQUARE ROOT */
#define XKB_KEY_cuberoot                   0x100221b  /* U+221B CUBE ROOT */
#define XKB_KEY_fourthroot                 0x100221c  /* U+221C FOURTH ROOT */
#define XKB_KEY_dintegral                  0x100222c  /* U+222C DOUBLE INTEGRAL */
#define XKB_KEY_tintegral                  0x100222d  /* U+222D TRIPLE INTEGRAL */
#define XKB_KEY_because                    0x1002235  /* U+2235 BECAUSE */
#define XKB_KEY_approxeq                   0x1002248  /*(U+2248 ALMOST EQUAL TO)*/
#define XKB_KEY_notapproxeq                0x1002247  /*(U+2247 NEITHER APPROXIMATELY NOR ACTUALLY EQUAL TO)*/
#define XKB_KEY_notidentical               0x1002262  /* U+2262 NOT IDENTICAL TO */
#define XKB_KEY_stricteq                   0x1002263  /* U+2263 STRICTLY EQUIVALENT TO */

#define XKB_KEY_braille_dot_1                 0xfff1
#define XKB_KEY_braille_dot_2                 0xfff2
#define XKB_KEY_braille_dot_3                 0xfff3
#define XKB_KEY_braille_dot_4                 0xfff4
#define XKB_KEY_braille_dot_5                 0xfff5
#define XKB_KEY_braille_dot_6                 0xfff6
#define XKB_KEY_braille_dot_7                 0xfff7
#define XKB_KEY_braille_dot_8                 0xfff8
#define XKB_KEY_braille_dot_9                 0xfff9
#define XKB_KEY_braille_dot_10                0xfffa
#define XKB_KEY_braille_blank              0x1002800  /* U+2800 BRAILLE PATTERN BLANK */
#define XKB_KEY_braille_dots_1             0x1002801  /* U+2801 BRAILLE PATTERN DOTS-1 */
#define XKB_KEY_braille_dots_2             0x1002802  /* U+2802 BRAILLE PATTERN DOTS-2 */
#define XKB_KEY_braille_dots_12            0x1002803  /* U+2803 BRAILLE PATTERN DOTS-12 */
#define XKB_KEY_braille_dots_3             0x1002804  /* U+2804 BRAILLE PATTERN DOTS-3 */
#define XKB_KEY_braille_dots_13            0x1002805  /* U+2805 BRAILLE PATTERN DOTS-13 */
#define XKB_KEY_braille_dots_23            0x1002806  /* U+2806 BRAILLE PATTERN DOTS-23 */
#define XKB_KEY_braille_dots_123           0x1002807  /* U+2807 BRAILLE PATTERN DOTS-123 */
#define XKB_KEY_braille_dots_4             0x1002808  /* U+2808 BRAILLE PATTERN DOTS-4 */
#define XKB_KEY_braille_dots_14            0x1002809  /* U+2809 BRAILLE PATTERN DOTS-14 */
#define XKB_KEY_braille_dots_24            0x100280a  /* U+280A BRAILLE PATTERN DOTS-24 */
#define XKB_KEY_braille_dots_124           0x100280b  /* U+280B BRAILLE PATTERN DOTS-124 */
#define XKB_KEY_braille_dots_34            0x100280c  /* U+280C BRAILLE PATTERN DOTS-34 */
#define XKB_KEY_braille_dots_134           0x100280d  /* U+280D BRAILLE PATTERN DOTS-134 */
#define XKB_KEY_braille_dots_234           0x100280e  /* U+280E BRAILLE PATTERN DOTS-234 */
#define XKB_KEY_braille_dots_1234          0x100280f  /* U+280F BRAILLE PATTERN DOTS-1234 */
#define XKB_KEY_braille_dots_5             0x1002810  /* U+2810 BRAILLE PATTERN DOTS-5 */
#define XKB_KEY_braille_dots_15            0x1002811  /* U+2811 BRAILLE PATTERN DOTS-15 */
#define XKB_KEY_braille_dots_25            0x1002812  /* U+2812 BRAILLE PATTERN DOTS-25 */
#define XKB_KEY_braille_dots_125           0x1002813  /* U+2813 BRAILLE PATTERN DOTS-125 */
#define XKB_KEY_braille_dots_35            0x1002814  /* U+2814 BRAILLE PATTERN DOTS-35 */
#define XKB_KEY_braille_dots_135           0x1002815  /* U+2815 BRAILLE PATTERN DOTS-135 */
#define XKB_KEY_braille_dots_235           0x1002816  /* U+2816 BRAILLE PATTERN DOTS-235 */
#define XKB_KEY_braille_dots_1235          0x1002817  /* U+2817 BRAILLE PATTERN DOTS-1235 */
#define XKB_KEY_braille_dots_45            0x1002818  /* U+2818 BRAILLE PATTERN DOTS-45 */
#define XKB_KEY_braille_dots_145           0x1002819  /* U+2819 BRAILLE PATTERN DOTS-145 */
#define XKB_KEY_braille_dots_245           0x100281a  /* U+281A BRAILLE PATTERN DOTS-245 */
#define XKB_KEY_braille_dots_1245          0x100281b  /* U+281B BRAILLE PATTERN DOTS-1245 */
#define XKB_KEY_braille_dots_345           0x100281c  /* U+281C BRAILLE PATTERN DOTS-345 */
#define XKB_KEY_braille_dots_1345          0x100281d  /* U+281D BRAILLE PATTERN DOTS-1345 */
#define XKB_KEY_braille_dots_2345          0x100281e  /* U+281E BRAILLE PATTERN DOTS-2345 */
#define XKB_KEY_braille_dots_12345         0x100281f  /* U+281F BRAILLE PATTERN DOTS-12345 */
#define XKB_KEY_braille_dots_6             0x1002820  /* U+2820 BRAILLE PATTERN DOTS-6 */
#define XKB_KEY_braille_dots_16            0x1002821  /* U+2821 BRAILLE PATTERN DOTS-16 */
#define XKB_KEY_braille_dots_26            0x1002822  /* U+2822 BRAILLE PATTERN DOTS-26 */
#define XKB_KEY_braille_dots_126           0x1002823  /* U+2823 BRAILLE PATTERN DOTS-126 */
#define XKB_KEY_braille_dots_36            0x1002824  /* U+2824 BRAILLE PATTERN DOTS-36 */
#define XKB_KEY_braille_dots_136           0x1002825  /* U+2825 BRAILLE PATTERN DOTS-136 */
#define XKB_KEY_braille_dots_236           0x1002826  /* U+2826 BRAILLE PATTERN DOTS-236 */
#define XKB_KEY_braille_dots_1236          0x1002827  /* U+2827 BRAILLE PATTERN DOTS-1236 */
#define XKB_KEY_braille_dots_46            0x1002828  /* U+2828 BRAILLE PATTERN DOTS-46 */
#define XKB_KEY_braille_dots_146           0x1002829  /* U+2829 BRAILLE PATTERN DOTS-146 */
#define XKB_KEY_braille_dots_246           0x100282a  /* U+282A BRAILLE PATTERN DOTS-246 */
#define XKB_KEY_braille_dots_1246          0x100282b  /* U+282B BRAILLE PATTERN DOTS-1246 */
#define XKB_KEY_braille_dots_346           0x100282c  /* U+282C BRAILLE PATTERN DOTS-346 */
#define XKB_KEY_braille_dots_1346          0x100282d  /* U+282D BRAILLE PATTERN DOTS-1346 */
#define XKB_KEY_braille_dots_2346          0x100282e  /* U+282E BRAILLE PATTERN DOTS-2346 */
#define XKB_KEY_braille_dots_12346         0x100282f  /* U+282F BRAILLE PATTERN DOTS-12346 */
#define XKB_KEY_braille_dots_56            0x1002830  /* U+2830 BRAILLE PATTERN DOTS-56 */
#define XKB_KEY_braille_dots_156           0x1002831  /* U+2831 BRAILLE PATTERN DOTS-156 */
#define XKB_KEY_braille_dots_256           0x1002832  /* U+2832 BRAILLE PATTERN DOTS-256 */
#define XKB_KEY_braille_dots_1256          0x1002833  /* U+2833 BRAILLE PATTERN DOTS-1256 */
#define XKB_KEY_braille_dots_356           0x1002834  /* U+2834 BRAILLE PATTERN DOTS-356 */
#define XKB_KEY_braille_dots_1356          0x1002835  /* U+2835 BRAILLE PATTERN DOTS-1356 */
#define XKB_KEY_braille_dots_2356          0x1002836  /* U+2836 BRAILLE PATTERN DOTS-2356 */
#define XKB_KEY_braille_dots_12356         0x1002837  /* U+2837 BRAILLE PATTERN DOTS-12356 */
#define XKB_KEY_braille_dots_456           0x1002838  /* U+2838 BRAILLE PATTERN DOTS-456 */
#define XKB_KEY_braille_dots_1456          0x1002839  /* U+2839 BRAILLE PATTERN DOTS-1456 */
#define XKB_KEY_braille_dots_2456          0x100283a  /* U+283A BRAILLE PATTERN DOTS-2456 */
#define XKB_KEY_braille_dots_12456         0x100283b  /* U+283B BRAILLE PATTERN DOTS-12456 */
#define XKB_KEY_braille_dots_3456          0x100283c  /* U+283C BRAILLE PATTERN DOTS-3456 */
#define XKB_KEY_braille_dots_13456         0x100283d  /* U+283D BRAILLE PATTERN DOTS-13456 */
#define XKB_KEY_braille_dots_23456         0x100283e  /* U+283E BRAILLE PATTERN DOTS-23456 */
#define XKB_KEY_braille_dots_123456        0x100283f  /* U+283F BRAILLE PATTERN DOTS-123456 */
#define XKB_KEY_braille_dots_7             0x1002840  /* U+2840 BRAILLE PATTERN DOTS-7 */
#define XKB_KEY_braille_dots_17            0x1002841  /* U+2841 BRAILLE PATTERN DOTS-17 */
#define XKB_KEY_braille_dots_27            0x1002842  /* U+2842 BRAILLE PATTERN DOTS-27 */
#define XKB_KEY_braille_dots_127           0x1002843  /* U+2843 BRAILLE PATTERN DOTS-127 */
#define XKB_KEY_braille_dots_37            0x1002844  /* U+2844 BRAILLE PATTERN DOTS-37 */
#define XKB_KEY_braille_dots_137           0x1002845  /* U+2845 BRAILLE PATTERN DOTS-137 */
#define XKB_KEY_braille_dots_237           0x1002846  /* U+2846 BRAILLE PATTERN DOTS-237 */
#define XKB_KEY_braille_dots_1237          0x1002847  /* U+2847 BRAILLE PATTERN DOTS-1237 */
#define XKB_KEY_braille_dots_47            0x1002848  /* U+2848 BRAILLE PATTERN DOTS-47 */
#define XKB_KEY_braille_dots_147           0x1002849  /* U+2849 BRAILLE PATTERN DOTS-147 */
#define XKB_KEY_braille_dots_247           0x100284a  /* U+284A BRAILLE PATTERN DOTS-247 */
#define XKB_KEY_braille_dots_1247          0x100284b  /* U+284B BRAILLE PATTERN DOTS-1247 */
#define XKB_KEY_braille_dots_347           0x100284c  /* U+284C BRAILLE PATTERN DOTS-347 */
#define XKB_KEY_braille_dots_1347          0x100284d  /* U+284D BRAILLE PATTERN DOTS-1347 */
#define XKB_KEY_braille_dots_2347          0x100284e  /* U+284E BRAILLE PATTERN DOTS-2347 */
#define XKB_KEY_braille_dots_12347         0x100284f  /* U+284F BRAILLE PATTERN DOTS-12347 */
#define XKB_KEY_braille_dots_57            0x1002850  /* U+2850 BRAILLE PATTERN DOTS-57 */
#define XKB_KEY_braille_dots_157           0x1002851  /* U+2851 BRAILLE PATTERN DOTS-157 */
#define XKB_KEY_braille_dots_257           0x1002852  /* U+2852 BRAILLE PATTERN DOTS-257 */
#define XKB_KEY_braille_dots_1257          0x1002853  /* U+2853 BRAILLE PATTERN DOTS-1257 */
#define XKB_KEY_braille_dots_357           0x1002854  /* U+2854 BRAILLE PATTERN DOTS-357 */
#define XKB_KEY_braille_dots_1357          0x1002855  /* U+2855 BRAILLE PATTERN DOTS-1357 */
#define XKB_KEY_braille_dots_2357          0x1002856  /* U+2856 BRAILLE PATTERN DOTS-2357 */
#define XKB_KEY_braille_dots_12357         0x1002857  /* U+2857 BRAILLE PATTERN DOTS-12357 */
#define XKB_KEY_braille_dots_457           0x1002858  /* U+2858 BRAILLE PATTERN DOTS-457 */
#define XKB_KEY_braille_dots_1457          0x1002859  /* U+2859 BRAILLE PATTERN DOTS-1457 */
#define XKB_KEY_braille_dots_2457          0x100285a  /* U+285A BRAILLE PATTERN DOTS-2457 */
#define XKB_KEY_braille_dots_12457         0x100285b  /* U+285B BRAILLE PATTERN DOTS-12457 */
#define XKB_KEY_braille_dots_3457          0x100285c  /* U+285C BRAILLE PATTERN DOTS-3457 */
#define XKB_KEY_braille_dots_13457         0x100285d  /* U+285D BRAILLE PATTERN DOTS-13457 */
#define XKB_KEY_braille_dots_23457         0x100285e  /* U+285E BRAILLE PATTERN DOTS-23457 */
#define XKB_KEY_braille_dots_123457        0x100285f  /* U+285F BRAILLE PATTERN DOTS-123457 */
#define XKB_KEY_braille_dots_67            0x1002860  /* U+2860 BRAILLE PATTERN DOTS-67 */
#define XKB_KEY_braille_dots_167           0x1002861  /* U+2861 BRAILLE PATTERN DOTS-167 */
#define XKB_KEY_braille_dots_267           0x1002862  /* U+2862 BRAILLE PATTERN DOTS-267 */
#define XKB_KEY_braille_dots_1267          0x1002863  /* U+2863 BRAILLE PATTERN DOTS-1267 */
#define XKB_KEY_braille_dots_367           0x1002864  /* U+2864 BRAILLE PATTERN DOTS-367 */
#define XKB_KEY_braille_dots_1367          0x1002865  /* U+2865 BRAILLE PATTERN DOTS-1367 */
#define XKB_KEY_braille_dots_2367          0x1002866  /* U+2866 BRAILLE PATTERN DOTS-2367 */
#define XKB_KEY_braille_dots_12367         0x1002867  /* U+2867 BRAILLE PATTERN DOTS-12367 */
#define XKB_KEY_braille_dots_467           0x1002868  /* U+2868 BRAILLE PATTERN DOTS-467 */
#define XKB_KEY_braille_dots_1467          0x1002869  /* U+2869 BRAILLE PATTERN DOTS-1467 */
#define XKB_KEY_braille_dots_2467          0x100286a  /* U+286A BRAILLE PATTERN DOTS-2467 */
#define XKB_KEY_braille_dots_12467         0x100286b  /* U+286B BRAILLE PATTERN DOTS-12467 */
#define XKB_KEY_braille_dots_3467          0x100286c  /* U+286C BRAILLE PATTERN DOTS-3467 */
#define XKB_KEY_braille_dots_13467         0x100286d  /* U+286D BRAILLE PATTERN DOTS-13467 */
#define XKB_KEY_braille_dots_23467         0x100286e  /* U+286E BRAILLE PATTERN DOTS-23467 */
#define XKB_KEY_braille_dots_123467        0x100286f  /* U+286F BRAILLE PATTERN DOTS-123467 */
#define XKB_KEY_braille_dots_567           0x1002870  /* U+2870 BRAILLE PATTERN DOTS-567 */
#define XKB_KEY_braille_dots_1567          0x1002871  /* U+2871 BRAILLE PATTERN DOTS-1567 */
#define XKB_KEY_braille_dots_2567          0x1002872  /* U+2872 BRAILLE PATTERN DOTS-2567 */
#define XKB_KEY_braille_dots_12567         0x1002873  /* U+2873 BRAILLE PATTERN DOTS-12567 */
#define XKB_KEY_braille_dots_3567          0x1002874  /* U+2874 BRAILLE PATTERN DOTS-3567 */
#define XKB_KEY_braille_dots_13567         0x1002875  /* U+2875 BRAILLE PATTERN DOTS-13567 */
#define XKB_KEY_braille_dots_23567         0x1002876  /* U+2876 BRAILLE PATTERN DOTS-23567 */
#define XKB_KEY_braille_dots_123567        0x1002877  /* U+2877 BRAILLE PATTERN DOTS-123567 */
#define XKB_KEY_braille_dots_4567          0x1002878  /* U+2878 BRAILLE PATTERN DOTS-4567 */
#define XKB_KEY_braille_dots_14567         0x1002879  /* U+2879 BRAILLE PATTERN DOTS-14567 */
#define XKB_KEY_braille_dots_24567         0x100287a  /* U+287A BRAILLE PATTERN DOTS-24567 */
#define XKB_KEY_braille_dots_124567        0x100287b  /* U+287B BRAILLE PATTERN DOTS-124567 */
#define XKB_KEY_braille_dots_34567         0x100287c  /* U+287C BRAILLE PATTERN DOTS-34567 */
#define XKB_KEY_braille_dots_134567        0x100287d  /* U+287D BRAILLE PATTERN DOTS-134567 */
#define XKB_KEY_braille_dots_234567        0x100287e  /* U+287E BRAILLE PATTERN DOTS-234567 */
#define XKB_KEY_braille_dots_1234567       0x100287f  /* U+287F BRAILLE PATTERN DOTS-1234567 */
#define XKB_KEY_braille_dots_8             0x1002880  /* U+2880 BRAILLE PATTERN DOTS-8 */
#define XKB_KEY_braille_dots_18            0x1002881  /* U+2881 BRAILLE PATTERN DOTS-18 */
#define XKB_KEY_braille_dots_28            0x1002882  /* U+2882 BRAILLE PATTERN DOTS-28 */
#define XKB_KEY_braille_dots_128           0x1002883  /* U+2883 BRAILLE PATTERN DOTS-128 */
#define XKB_KEY_braille_dots_38            0x1002884  /* U+2884 BRAILLE PATTERN DOTS-38 */
#define XKB_KEY_braille_dots_138           0x1002885  /* U+2885 BRAILLE PATTERN DOTS-138 */
#define XKB_KEY_braille_dots_238           0x1002886  /* U+2886 BRAILLE PATTERN DOTS-238 */
#define XKB_KEY_braille_dots_1238          0x1002887  /* U+2887 BRAILLE PATTERN DOTS-1238 */
#define XKB_KEY_braille_dots_48            0x1002888  /* U+2888 BRAILLE PATTERN DOTS-48 */
#define XKB_KEY_braille_dots_148           0x1002889  /* U+2889 BRAILLE PATTERN DOTS-148 */
#define XKB_KEY_braille_dots_248           0x100288a  /* U+288A BRAILLE PATTERN DOTS-248 */
#define XKB_KEY_braille_dots_1248          0x100288b  /* U+288B BRAILLE PATTERN DOTS-1248 */
#define XKB_KEY_braille_dots_348           0x100288c  /* U+288C BRAILLE PATTERN DOTS-348 */
#define XKB_KEY_braille_dots_1348          0x100288d  /* U+288D BRAILLE PATTERN DOTS-1348 */
#define XKB_KEY_braille_dots_2348          0x100288e  /* U+288E BRAILLE PATTERN DOTS-2348 */
#define XKB_KEY_braille_dots_12348         0x100288f  /* U+288F BRAILLE PATTERN DOTS-12348 */
#define XKB_KEY_braille_dots_58            0x1002890  /* U+2890 BRAILLE PATTERN DOTS-58 */
#define XKB_KEY_braille_dots_158           0x1002891  /* U+2891 BRAILLE PATTERN DOTS-158 */
#define XKB_KEY_braille_dots_258           0x1002892  /* U+2892 BRAILLE PATTERN DOTS-258 */
#define XKB_KEY_braille_dots_1258          0x1002893  /* U+2893 BRAILLE PATTERN DOTS-1258 */
#define XKB_KEY_braille_dots_358           0x1002894  /* U+2894 BRAILLE PATTERN DOTS-358 */
#define XKB_KEY_braille_dots_1358          0x1002895  /* U+2895 BRAILLE PATTERN DOTS-1358 */
#define XKB_KEY_braille_dots_2358          0x1002896  /* U+2896 BRAILLE PATTERN DOTS-2358 */
#define XKB_KEY_braille_dots_12358         0x1002897  /* U+2897 BRAILLE PATTERN DOTS-12358 */
#define XKB_KEY_braille_dots_458           0x1002898  /* U+2898 BRAILLE PATTERN DOTS-458 */
#define XKB_KEY_braille_dots_1458          0x1002899  /* U+2899 BRAILLE PATTERN DOTS-1458 */
#define XKB_KEY_braille_dots_2458          0x100289a  /* U+289A BRAILLE PATTERN DOTS-2458 */
#define XKB_KEY_braille_dots_12458         0x100289b  /* U+289B BRAILLE PATTERN DOTS-12458 */
#define XKB_KEY_braille_dots_3458          0x100289c  /* U+289C BRAILLE PATTERN DOTS-3458 */
#define XKB_KEY_braille_dots_13458         0x100289d  /* U+289D BRAILLE PATTERN DOTS-13458 */
#define XKB_KEY_braille_dots_23458         0x100289e  /* U+289E BRAILLE PATTERN DOTS-23458 */
#define XKB_KEY_braille_dots_123458        0x100289f  /* U+289F BRAILLE PATTERN DOTS-123458 */
#define XKB_KEY_braille_dots_68            0x10028a0  /* U+28A0 BRAILLE PATTERN DOTS-68 */
#define XKB_KEY_braille_dots_168           0x10028a1  /* U+28A1 BRAILLE PATTERN DOTS-168 */
#define XKB_KEY_braille_dots_268           0x10028a2  /* U+28A2 BRAILLE PATTERN DOTS-268 */
#define XKB_KEY_braille_dots_1268          0x10028a3  /* U+28A3 BRAILLE PATTERN DOTS-1268 */
#define XKB_KEY_braille_dots_368           0x10028a4  /* U+28A4 BRAILLE PATTERN DOTS-368 */
#define XKB_KEY_braille_dots_1368          0x10028a5  /* U+28A5 BRAILLE PATTERN DOTS-1368 */
#define XKB_KEY_braille_dots_2368          0x10028a6  /* U+28A6 BRAILLE PATTERN DOTS-2368 */
#define XKB_KEY_braille_dots_12368         0x10028a7  /* U+28A7 BRAILLE PATTERN DOTS-12368 */
#define XKB_KEY_braille_dots_468           0x10028a8  /* U+28A8 BRAILLE PATTERN DOTS-468 */
#define XKB_KEY_braille_dots_1468          0x10028a9  /* U+28A9 BRAILLE PATTERN DOTS-1468 */
#define XKB_KEY_braille_dots_2468          0x10028aa  /* U+28AA BRAILLE PATTERN DOTS-2468 */
#define XKB_KEY_braille_dots_12468         0x10028ab  /* U+28AB BRAILLE PATTERN DOTS-12468 */
#define XKB_KEY_braille_dots_3468          0x10028ac  /* U+28AC BRAILLE PATTERN DOTS-3468 */
#define XKB_KEY_braille_dots_13468         0x10028ad  /* U+28AD BRAILLE PATTERN DOTS-13468 */
#define XKB_KEY_braille_dots_23468         0x10028ae  /* U+28AE BRAILLE PATTERN DOTS-23468 */
#define XKB_KEY_braille_dots_123468        0x10028af  /* U+28AF BRAILLE PATTERN DOTS-123468 */
#define XKB_KEY_braille_dots_568           0x10028b0  /* U+28B0 BRAILLE PATTERN DOTS-568 */
#define XKB_KEY_braille_dots_1568          0x10028b1  /* U+28B1 BRAILLE PATTERN DOTS-1568 */
#define XKB_KEY_braille_dots_2568          0x10028b2  /* U+28B2 BRAILLE PATTERN DOTS-2568 */
#define XKB_KEY_braille_dots_12568         0x10028b3  /* U+28B3 BRAILLE PATTERN DOTS-12568 */
#define XKB_KEY_braille_dots_3568          0x10028b4  /* U+28B4 BRAILLE PATTERN DOTS-3568 */
#define XKB_KEY_braille_dots_13568         0x10028b5  /* U+28B5 BRAILLE PATTERN DOTS-13568 */
#define XKB_KEY_braille_dots_23568         0x10028b6  /* U+28B6 BRAILLE PATTERN DOTS-23568 */
#define XKB_KEY_braille_dots_123568        0x10028b7  /* U+28B7 BRAILLE PATTERN DOTS-123568 */
#define XKB_KEY_braille_dots_4568          0x10028b8  /* U+28B8 BRAILLE PATTERN DOTS-4568 */
#define XKB_KEY_braille_dots_14568         0x10028b9  /* U+28B9 BRAILLE PATTERN DOTS-14568 */
#define XKB_KEY_braille_dots_24568         0x10028ba  /* U+28BA BRAILLE PATTERN DOTS-24568 */
#define XKB_KEY_braille_dots_124568        0x10028bb  /* U+28BB BRAILLE PATTERN DOTS-124568 */
#define XKB_KEY_braille_dots_34568         0x10028bc  /* U+28BC BRAILLE PATTERN DOTS-34568 */
#define XKB_KEY_braille_dots_134568        0x10028bd  /* U+28BD BRAILLE PATTERN DOTS-134568 */
#define XKB_KEY_braille_dots_234568        0x10028be  /* U+28BE BRAILLE PATTERN DOTS-234568 */
#define XKB_KEY_braille_dots_1234568       0x10028bf  /* U+28BF BRAILLE PATTERN DOTS-1234568 */
#define XKB_KEY_braille_dots_78            0x10028c0  /* U+28C0 BRAILLE PATTERN DOTS-78 */
#define XKB_KEY_braille_dots_178           0x10028c1  /* U+28C1 BRAILLE PATTERN DOTS-178 */
#define XKB_KEY_braille_dots_278           0x10028c2  /* U+28C2 BRAILLE PATTERN DOTS-278 */
#define XKB_KEY_braille_dots_1278          0x10028c3  /* U+28C3 BRAILLE PATTERN DOTS-1278 */
#define XKB_KEY_braille_dots_378           0x10028c4  /* U+28C4 BRAILLE PATTERN DOTS-378 */
#define XKB_KEY_braille_dots_1378          0x10028c5  /* U+28C5 BRAILLE PATTERN DOTS-1378 */
#define XKB_KEY_braille_dots_2378          0x10028c6  /* U+28C6 BRAILLE PATTERN DOTS-2378 */
#define XKB_KEY_braille_dots_12378         0x10028c7  /* U+28C7 BRAILLE PATTERN DOTS-12378 */
#define XKB_KEY_braille_dots_478           0x10028c8  /* U+28C8 BRAILLE PATTERN DOTS-478 */
#define XKB_KEY_braille_dots_1478          0x10028c9  /* U+28C9 BRAILLE PATTERN DOTS-1478 */
#define XKB_KEY_braille_dots_2478          0x10028ca  /* U+28CA BRAILLE PATTERN DOTS-2478 */
#define XKB_KEY_braille_dots_12478         0x10028cb  /* U+28CB BRAILLE PATTERN DOTS-12478 */
#define XKB_KEY_braille_dots_3478          0x10028cc  /* U+28CC BRAILLE PATTERN DOTS-3478 */
#define XKB_KEY_braille_dots_13478         0x10028cd  /* U+28CD BRAILLE PATTERN DOTS-13478 */
#define XKB_KEY_braille_dots_23478         0x10028ce  /* U+28CE BRAILLE PATTERN DOTS-23478 */
#define XKB_KEY_braille_dots_123478        0x10028cf  /* U+28CF BRAILLE PATTERN DOTS-123478 */
#define XKB_KEY_braille_dots_578           0x10028d0  /* U+28D0 BRAILLE PATTERN DOTS-578 */
#define XKB_KEY_braille_dots_1578          0x10028d1  /* U+28D1 BRAILLE PATTERN DOTS-1578 */
#define XKB_KEY_braille_dots_2578          0x10028d2  /* U+28D2 BRAILLE PATTERN DOTS-2578 */
#define XKB_KEY_braille_dots_12578         0x10028d3  /* U+28D3 BRAILLE PATTERN DOTS-12578 */
#define XKB_KEY_braille_dots_3578          0x10028d4  /* U+28D4 BRAILLE PATTERN DOTS-3578 */
#define XKB_KEY_braille_dots_13578         0x10028d5  /* U+28D5 BRAILLE PATTERN DOTS-13578 */
#define XKB_KEY_braille_dots_23578         0x10028d6  /* U+28D6 BRAILLE PATTERN DOTS-23578 */
#define XKB_KEY_braille_dots_123578        0x10028d7  /* U+28D7 BRAILLE PATTERN DOTS-123578 */
#define XKB_KEY_braille_dots_4578          0x10028d8  /* U+28D8 BRAILLE PATTERN DOTS-4578 */
#define XKB_KEY_braille_dots_14578         0x10028d9  /* U+28D9 BRAILLE PATTERN DOTS-14578 */
#define XKB_KEY_braille_dots_24578         0x10028da  /* U+28DA BRAILLE PATTERN DOTS-24578 */
#define XKB_KEY_braille_dots_124578        0x10028db  /* U+28DB BRAILLE PATTERN DOTS-124578 */
#define XKB_KEY_braille_dots_34578         0x10028dc  /* U+28DC BRAILLE PATTERN DOTS-34578 */
#define XKB_KEY_braille_dots_134578        0x10028dd  /* U+28DD BRAILLE PATTERN DOTS-134578 */
#define XKB_KEY_braille_dots_234578        0x10028de  /* U+28DE BRAILLE PATTERN DOTS-234578 */
#define XKB_KEY_braille_dots_1234578       0x10028df  /* U+28DF BRAILLE PATTERN DOTS-1234578 */
#define XKB_KEY_braille_dots_678           0x10028e0  /* U+28E0 BRAILLE PATTERN DOTS-678 */
#define XKB_KEY_braille_dots_1678          0x10028e1  /* U+28E1 BRAILLE PATTERN DOTS-1678 */
#define XKB_KEY_braille_dots_2678          0x10028e2  /* U+28E2 BRAILLE PATTERN DOTS-2678 */
#define XKB_KEY_braille_dots_12678         0x10028e3  /* U+28E3 BRAILLE PATTERN DOTS-12678 */
#define XKB_KEY_braille_dots_3678          0x10028e4  /* U+28E4 BRAILLE PATTERN DOTS-3678 */
#define XKB_KEY_braille_dots_13678         0x10028e5  /* U+28E5 BRAILLE PATTERN DOTS-13678 */
#define XKB_KEY_braille_dots_23678         0x10028e6  /* U+28E6 BRAILLE PATTERN DOTS-23678 */
#define XKB_KEY_braille_dots_123678        0x10028e7  /* U+28E7 BRAILLE PATTERN DOTS-123678 */
#define XKB_KEY_braille_dots_4678          0x10028e8  /* U+28E8 BRAILLE PATTERN DOTS-4678 */
#define XKB_KEY_braille_dots_14678         0x10028e9  /* U+28E9 BRAILLE PATTERN DOTS-14678 */
#define XKB_KEY_braille_dots_24678         0x10028ea  /* U+28EA BRAILLE PATTERN DOTS-24678 */
#define XKB_KEY_braille_dots_124678        0x10028eb  /* U+28EB BRAILLE PATTERN DOTS-124678 */
#define XKB_KEY_braille_dots_34678         0x10028ec  /* U+28EC BRAILLE PATTERN DOTS-34678 */
#define XKB_KEY_braille_dots_134678        0x10028ed  /* U+28ED BRAILLE PATTERN DOTS-134678 */
#define XKB_KEY_braille_dots_234678        0x10028ee  /* U+28EE BRAILLE PATTERN DOTS-234678 */
#define XKB_KEY_braille_dots_1234678       0x10028ef  /* U+28EF BRAILLE PATTERN DOTS-1234678 */
#define XKB_KEY_braille_dots_5678          0x10028f0  /* U+28F0 BRAILLE PATTERN DOTS-5678 */
#define XKB_KEY_braille_dots_15678         0x10028f1  /* U+28F1 BRAILLE PATTERN DOTS-15678 */
#define XKB_KEY_braille_dots_25678         0x10028f2  /* U+28F2 BRAILLE PATTERN DOTS-25678 */
#define XKB_KEY_braille_dots_125678        0x10028f3  /* U+28F3 BRAILLE PATTERN DOTS-125678 */
#define XKB_KEY_braille_dots_35678         0x10028f4  /* U+28F4 BRAILLE PATTERN DOTS-35678 */
#define XKB_KEY_braille_dots_135678        0x10028f5  /* U+28F5 BRAILLE PATTERN DOTS-135678 */
#define XKB_KEY_braille_dots_235678        0x10028f6  /* U+28F6 BRAILLE PATTERN DOTS-235678 */
#define XKB_KEY_braille_dots_1235678       0x10028f7  /* U+28F7 BRAILLE PATTERN DOTS-1235678 */
#define XKB_KEY_braille_dots_45678         0x10028f8  /* U+28F8 BRAILLE PATTERN DOTS-45678 */
#define XKB_KEY_braille_dots_145678        0x10028f9  /* U+28F9 BRAILLE PATTERN DOTS-145678 */
#define XKB_KEY_braille_dots_245678        0x10028fa  /* U+28FA BRAILLE PATTERN DOTS-245678 */
#define XKB_KEY_braille_dots_1245678       0x10028fb  /* U+28FB BRAILLE PATTERN DOTS-1245678 */
#define XKB_KEY_braille_dots_345678        0x10028fc  /* U+28FC BRAILLE PATTERN DOTS-345678 */
#define XKB_KEY_braille_dots_1345678       0x10028fd  /* U+28FD BRAILLE PATTERN DOTS-1345678 */
#define XKB_KEY_braille_dots_2345678       0x10028fe  /* U+28FE BRAILLE PATTERN DOTS-2345678 */
#define XKB_KEY_braille_dots_12345678      0x10028ff  /* U+28FF BRAILLE PATTERN DOTS-12345678 */

/*
 * Sinhala (http://unicode.org/charts/PDF/U0D80.pdf)
 * http://www.nongnu.org/sinhala/doc/transliteration/sinhala-transliteration_6.html
 */

#define XKB_KEY_Sinh_ng                    0x1000d82  /* U+0D82 SINHALA SIGN ANUSVARAYA */
#define XKB_KEY_Sinh_h2                    0x1000d83  /* U+0D83 SINHALA SIGN VISARGAYA */
#define XKB_KEY_Sinh_a                     0x1000d85  /* U+0D85 SINHALA LETTER AYANNA */
#define XKB_KEY_Sinh_aa                    0x1000d86  /* U+0D86 SINHALA LETTER AAYANNA */
#define XKB_KEY_Sinh_ae                    0x1000d87  /* U+0D87 SINHALA LETTER AEYANNA */
#define XKB_KEY_Sinh_aee                   0x1000d88  /* U+0D88 SINHALA LETTER AEEYANNA */
#define XKB_KEY_Sinh_i                     0x1000d89  /* U+0D89 SINHALA LETTER IYANNA */
#define XKB_KEY_Sinh_ii                    0x1000d8a  /* U+0D8A SINHALA LETTER IIYANNA */
#define XKB_KEY_Sinh_u                     0x1000d8b  /* U+0D8B SINHALA LETTER UYANNA */
#define XKB_KEY_Sinh_uu                    0x1000d8c  /* U+0D8C SINHALA LETTER UUYANNA */
#define XKB_KEY_Sinh_ri                    0x1000d8d  /* U+0D8D SINHALA LETTER IRUYANNA */
#define XKB_KEY_Sinh_rii                   0x1000d8e  /* U+0D8E SINHALA LETTER IRUUYANNA */
#define XKB_KEY_Sinh_lu                    0x1000d8f  /* U+0D8F SINHALA LETTER ILUYANNA */
#define XKB_KEY_Sinh_luu                   0x1000d90  /* U+0D90 SINHALA LETTER ILUUYANNA */
#define XKB_KEY_Sinh_e                     0x1000d91  /* U+0D91 SINHALA LETTER EYANNA */
#define XKB_KEY_Sinh_ee                    0x1000d92  /* U+0D92 SINHALA LETTER EEYANNA */
#define XKB_KEY_Sinh_ai                    0x1000d93  /* U+0D93 SINHALA LETTER AIYANNA */
#define XKB_KEY_Sinh_o                     0x1000d94  /* U+0D94 SINHALA LETTER OYANNA */
#define XKB_KEY_Sinh_oo                    0x1000d95  /* U+0D95 SINHALA LETTER OOYANNA */
#define XKB_KEY_Sinh_au                    0x1000d96  /* U+0D96 SINHALA LETTER AUYANNA */
#define XKB_KEY_Sinh_ka                    0x1000d9a  /* U+0D9A SINHALA LETTER ALPAPRAANA KAYANNA */
#define XKB_KEY_Sinh_kha                   0x1000d9b  /* U+0D9B SINHALA LETTER MAHAAPRAANA KAYANNA */
#define XKB_KEY_Sinh_ga                    0x1000d9c  /* U+0D9C SINHALA LETTER ALPAPRAANA GAYANNA */
#define XKB_KEY_Sinh_gha                   0x1000d9d  /* U+0D9D SINHALA LETTER MAHAAPRAANA GAYANNA */
#define XKB_KEY_Sinh_ng2                   0x1000d9e  /* U+0D9E SINHALA LETTER KANTAJA NAASIKYAYA */
#define XKB_KEY_Sinh_nga                   0x1000d9f  /* U+0D9F SINHALA LETTER SANYAKA GAYANNA */
#define XKB_KEY_Sinh_ca                    0x1000da0  /* U+0DA0 SINHALA LETTER ALPAPRAANA CAYANNA */
#define XKB_KEY_Sinh_cha                   0x1000da1  /* U+0DA1 SINHALA LETTER MAHAAPRAANA CAYANNA */
#define XKB_KEY_Sinh_ja                    0x1000da2  /* U+0DA2 SINHALA LETTER ALPAPRAANA JAYANNA */
#define XKB_KEY_Sinh_jha                   0x1000da3  /* U+0DA3 SINHALA LETTER MAHAAPRAANA JAYANNA */
#define XKB_KEY_Sinh_nya                   0x1000da4  /* U+0DA4 SINHALA LETTER TAALUJA NAASIKYAYA */
#define XKB_KEY_Sinh_jnya                  0x1000da5  /* U+0DA5 SINHALA LETTER TAALUJA SANYOOGA NAAKSIKYAYA */
#define XKB_KEY_Sinh_nja                   0x1000da6  /* U+0DA6 SINHALA LETTER SANYAKA JAYANNA */
#define XKB_KEY_Sinh_tta                   0x1000da7  /* U+0DA7 SINHALA LETTER ALPAPRAANA TTAYANNA */
#define XKB_KEY_Sinh_ttha                  0x1000da8  /* U+0DA8 SINHALA LETTER MAHAAPRAANA TTAYANNA */
#define XKB_KEY_Sinh_dda                   0x1000da9  /* U+0DA9 SINHALA LETTER ALPAPRAANA DDAYANNA */
#define XKB_KEY_Sinh_ddha                  0x1000daa  /* U+0DAA SINHALA LETTER MAHAAPRAANA DDAYANNA */
#define XKB_KEY_Sinh_nna                   0x1000dab  /* U+0DAB SINHALA LETTER MUURDHAJA NAYANNA */
#define XKB_KEY_Sinh_ndda                  0x1000dac  /* U+0DAC SINHALA LETTER SANYAKA DDAYANNA */
#define XKB_KEY_Sinh_tha                   0x1000dad  /* U+0DAD SINHALA LETTER ALPAPRAANA TAYANNA */
#define XKB_KEY_Sinh_thha                  0x1000dae  /* U+0DAE SINHALA LETTER MAHAAPRAANA TAYANNA */
#define XKB_KEY_Sinh_dha                   0x1000daf  /* U+0DAF SINHALA LETTER ALPAPRAANA DAYANNA */
#define XKB_KEY_Sinh_dhha                  0x1000db0  /* U+0DB0 SINHALA LETTER MAHAAPRAANA DAYANNA */
#define XKB_KEY_Sinh_na                    0x1000db1  /* U+0DB1 SINHALA LETTER DANTAJA NAYANNA */
#define XKB_KEY_Sinh_ndha                  0x1000db3  /* U+0DB3 SINHALA LETTER SANYAKA DAYANNA */
#define XKB_KEY_Sinh_pa                    0x1000db4  /* U+0DB4 SINHALA LETTER ALPAPRAANA PAYANNA */
#define XKB_KEY_Sinh_pha                   0x1000db5  /* U+0DB5 SINHALA LETTER MAHAAPRAANA PAYANNA */
#define XKB_KEY_Sinh_ba                    0x1000db6  /* U+0DB6 SINHALA LETTER ALPAPRAANA BAYANNA */
#define XKB_KEY_Sinh_bha                   0x1000db7  /* U+0DB7 SINHALA LETTER MAHAAPRAANA BAYANNA */
#define XKB_KEY_Sinh_ma                    0x1000db8  /* U+0DB8 SINHALA LETTER MAYANNA */
#define XKB_KEY_Sinh_mba                   0x1000db9  /* U+0DB9 SINHALA LETTER AMBA BAYANNA */
#define XKB_KEY_Sinh_ya                    0x1000dba  /* U+0DBA SINHALA LETTER YAYANNA */
#define XKB_KEY_Sinh_ra                    0x1000dbb  /* U+0DBB SINHALA LETTER RAYANNA */
#define XKB_KEY_Sinh_la                    0x1000dbd  /* U+0DBD SINHALA LETTER DANTAJA LAYANNA */
#define XKB_KEY_Sinh_va                    0x1000dc0  /* U+0DC0 SINHALA LETTER VAYANNA */
#define XKB_KEY_Sinh_sha                   0x1000dc1  /* U+0DC1 SINHALA LETTER TAALUJA SAYANNA */
#define XKB_KEY_Sinh_ssha                  0x1000dc2  /* U+0DC2 SINHALA LETTER MUURDHAJA SAYANNA */
#define XKB_KEY_Sinh_sa                    0x1000dc3  /* U+0DC3 SINHALA LETTER DANTAJA SAYANNA */
#define XKB_KEY_Sinh_ha                    0x1000dc4  /* U+0DC4 SINHALA LETTER HAYANNA */
#define XKB_KEY_Sinh_lla                   0x1000dc5  /* U+0DC5 SINHALA LETTER MUURDHAJA LAYANNA */
#define XKB_KEY_Sinh_fa                    0x1000dc6  /* U+0DC6 SINHALA LETTER FAYANNA */
#define XKB_KEY_Sinh_al                    0x1000dca  /* U+0DCA SINHALA SIGN AL-LAKUNA */
#define XKB_KEY_Sinh_aa2                   0x1000dcf  /* U+0DCF SINHALA VOWEL SIGN AELA-PILLA */
#define XKB_KEY_Sinh_ae2                   0x1000dd0  /* U+0DD0 SINHALA VOWEL SIGN KETTI AEDA-PILLA */
#define XKB_KEY_Sinh_aee2                  0x1000dd1  /* U+0DD1 SINHALA VOWEL SIGN DIGA AEDA-PILLA */
#define XKB_KEY_Sinh_i2                    0x1000dd2  /* U+0DD2 SINHALA VOWEL SIGN KETTI IS-PILLA */
#define XKB_KEY_Sinh_ii2                   0x1000dd3  /* U+0DD3 SINHALA VOWEL SIGN DIGA IS-PILLA */
#define XKB_KEY_Sinh_u2                    0x1000dd4  /* U+0DD4 SINHALA VOWEL SIGN KETTI PAA-PILLA */
#define XKB_KEY_Sinh_uu2                   0x1000dd6  /* U+0DD6 SINHALA VOWEL SIGN DIGA PAA-PILLA */
#define XKB_KEY_Sinh_ru2                   0x1000dd8  /* U+0DD8 SINHALA VOWEL SIGN GAETTA-PILLA */
#define XKB_KEY_Sinh_e2                    0x1000dd9  /* U+0DD9 SINHALA VOWEL SIGN KOMBUVA */
#define XKB_KEY_Sinh_ee2                   0x1000dda  /* U+0DDA SINHALA VOWEL SIGN DIGA KOMBUVA */
#define XKB_KEY_Sinh_ai2                   0x1000ddb  /* U+0DDB SINHALA VOWEL SIGN KOMBU DEKA */
#define XKB_KEY_Sinh_o2                    0x1000ddc  /* U+0DDC SINHALA VOWEL SIGN KOMBUVA HAA AELA-PILLA */
#define XKB_KEY_Sinh_oo2                   0x1000ddd  /* U+0DDD SINHALA VOWEL SIGN KOMBUVA HAA DIGA AELA-PILLA */
#define XKB_KEY_Sinh_au2                   0x1000dde  /* U+0DDE SINHALA VOWEL SIGN KOMBUVA HAA GAYANUKITTA */
#define XKB_KEY_Sinh_lu2                   0x1000ddf  /* U+0DDF SINHALA VOWEL SIGN GAYANUKITTA */
#define XKB_KEY_Sinh_ruu2                  0x1000df2  /* U+0DF2 SINHALA VOWEL SIGN DIGA GAETTA-PILLA */
#define XKB_KEY_Sinh_luu2                  0x1000df3  /* U+0DF3 SINHALA VOWEL SIGN DIGA GAYANUKITTA */
#define XKB_KEY_Sinh_kunddaliya            0x1000df4  /* U+0DF4 SINHALA PUNCTUATION KUNDDALIYA */
/*
 * XFree86 vendor specific keysyms.
 *
 * The XFree86 keysym range is 0x10080001 - 0x1008ffff.
 *
 * The XF86 set of keysyms is a catch-all set of defines for keysyms found
 * on various multimedia keyboards. Originally specific to XFree86 they have
 * been been adopted over time and are considered a "standard" part of X
 * keysym definitions.
 * XFree86 never properly commented these keysyms, so we have done our
 * best to explain the semantic meaning of these keys.
 *
 * XFree86 has removed their mail archives of the period, that might have
 * shed more light on some of these definitions. Until/unless we resurrect
 * these archives, these are from memory and usage.
 */

/*
 * ModeLock
 *
 * This one is old, and not really used any more since XKB offers this
 * functionality.
 */

#define XKB_KEY_XF86ModeLock              0x1008ff01  /* Mode Switch Lock */

/* Backlight controls. */
#define XKB_KEY_XF86MonBrightnessUp       0x1008ff02  /* Monitor/panel brightness */
#define XKB_KEY_XF86MonBrightnessDown     0x1008ff03  /* Monitor/panel brightness */
#define XKB_KEY_XF86KbdLightOnOff         0x1008ff04  /* Keyboards may be lit     */
#define XKB_KEY_XF86KbdBrightnessUp       0x1008ff05  /* Keyboards may be lit     */
#define XKB_KEY_XF86KbdBrightnessDown     0x1008ff06  /* Keyboards may be lit     */
#define XKB_KEY_XF86MonBrightnessCycle    0x1008ff07  /* Monitor/panel brightness */

/*
 * Keys found on some "Internet" keyboards.
 */
#define XKB_KEY_XF86Standby               0x1008ff10  /* System into standby mode   */
#define XKB_KEY_XF86AudioLowerVolume      0x1008ff11  /* Volume control down        */
#define XKB_KEY_XF86AudioMute             0x1008ff12  /* Mute sound from the system */
#define XKB_KEY_XF86AudioRaiseVolume      0x1008ff13  /* Volume control up          */
#define XKB_KEY_XF86AudioPlay             0x1008ff14  /* Start playing of audio >   */
#define XKB_KEY_XF86AudioStop             0x1008ff15  /* Stop playing audio         */
#define XKB_KEY_XF86AudioPrev             0x1008ff16  /* Previous track             */
#define XKB_KEY_XF86AudioNext             0x1008ff17  /* Next track                 */
#define XKB_KEY_XF86HomePage              0x1008ff18  /* Display user's home page   */
#define XKB_KEY_XF86Mail                  0x1008ff19  /* Invoke user's mail program */
#define XKB_KEY_XF86Start                 0x1008ff1a  /* Start application          */
#define XKB_KEY_XF86Search                0x1008ff1b  /* Search                     */
#define XKB_KEY_XF86AudioRecord           0x1008ff1c  /* Record audio application   */

/* These are sometimes found on PDA's (e.g. Palm, PocketPC or elsewhere)   */
#define XKB_KEY_XF86Calculator            0x1008ff1d  /* Invoke calculator program  */
#define XKB_KEY_XF86Memo                  0x1008ff1e  /* Invoke Memo taking program */
#define XKB_KEY_XF86ToDoList              0x1008ff1f  /* Invoke To Do List program  */
#define XKB_KEY_XF86Calendar              0x1008ff20  /* Invoke Calendar program    */
#define XKB_KEY_XF86PowerDown             0x1008ff21  /* Deep sleep the system      */
#define XKB_KEY_XF86ContrastAdjust        0x1008ff22  /* Adjust screen contrast     */
#define XKB_KEY_XF86RockerUp              0x1008ff23  /* Rocker switches exist up   */
#define XKB_KEY_XF86RockerDown            0x1008ff24  /* and down                   */
#define XKB_KEY_XF86RockerEnter           0x1008ff25  /* and let you press them     */

/* Some more "Internet" keyboard symbols */
#define XKB_KEY_XF86Back                  0x1008ff26  /* Like back on a browser     */
#define XKB_KEY_XF86Forward               0x1008ff27  /* Like forward on a browser  */
#define XKB_KEY_XF86Stop                  0x1008ff28  /* Stop current operation     */
#define XKB_KEY_XF86Refresh               0x1008ff29  /* Refresh the page           */
#define XKB_KEY_XF86PowerOff              0x1008ff2a  /* Power off system entirely  */
#define XKB_KEY_XF86WakeUp                0x1008ff2b  /* Wake up system from sleep  */
#define XKB_KEY_XF86Eject                 0x1008ff2c  /* Eject device (e.g. DVD)    */
#define XKB_KEY_XF86ScreenSaver           0x1008ff2d  /* Invoke screensaver         */
#define XKB_KEY_XF86WWW                   0x1008ff2e  /* Invoke web browser         */
#define XKB_KEY_XF86Sleep                 0x1008ff2f  /* Put system to sleep        */
#define XKB_KEY_XF86Favorites             0x1008ff30  /* Show favorite locations    */
#define XKB_KEY_XF86AudioPause            0x1008ff31  /* Pause audio playing        */
#define XKB_KEY_XF86AudioMedia            0x1008ff32  /* Launch media collection app */
#define XKB_KEY_XF86MyComputer            0x1008ff33  /* Display "My Computer" window */
#define XKB_KEY_XF86VendorHome            0x1008ff34  /* Display vendor home web site */
#define XKB_KEY_XF86LightBulb             0x1008ff35  /* Light bulb keys exist       */
#define XKB_KEY_XF86Shop                  0x1008ff36  /* Display shopping web site   */
#define XKB_KEY_XF86History               0x1008ff37  /* Show history of web surfing */
#define XKB_KEY_XF86OpenURL               0x1008ff38  /* Open selected URL           */
#define XKB_KEY_XF86AddFavorite           0x1008ff39  /* Add URL to favorites list   */
#define XKB_KEY_XF86HotLinks              0x1008ff3a  /* Show "hot" links            */
#define XKB_KEY_XF86BrightnessAdjust      0x1008ff3b  /* Invoke brightness adj. UI   */
#define XKB_KEY_XF86Finance               0x1008ff3c  /* Display financial site      */
#define XKB_KEY_XF86Community             0x1008ff3d  /* Display user's community    */
#define XKB_KEY_XF86AudioRewind           0x1008ff3e  /* "rewind" audio track        */
#define XKB_KEY_XF86BackForward           0x1008ff3f  /* ??? */
#define XKB_KEY_XF86Launch0               0x1008ff40  /* Launch Application          */
#define XKB_KEY_XF86Launch1               0x1008ff41  /* Launch Application          */
#define XKB_KEY_XF86Launch2               0x1008ff42  /* Launch Application          */
#define XKB_KEY_XF86Launch3               0x1008ff43  /* Launch Application          */
#define XKB_KEY_XF86Launch4               0x1008ff44  /* Launch Application          */
#define XKB_KEY_XF86Launch5               0x1008ff45  /* Launch Application          */
#define XKB_KEY_XF86Launch6               0x1008ff46  /* Launch Application          */
#define XKB_KEY_XF86Launch7               0x1008ff47  /* Launch Application          */
#define XKB_KEY_XF86Launch8               0x1008ff48  /* Launch Application          */
#define XKB_KEY_XF86Launch9               0x1008ff49  /* Launch Application          */
#define XKB_KEY_XF86LaunchA               0x1008ff4a  /* Launch Application          */
#define XKB_KEY_XF86LaunchB               0x1008ff4b  /* Launch Application          */
#define XKB_KEY_XF86LaunchC               0x1008ff4c  /* Launch Application          */
#define XKB_KEY_XF86LaunchD               0x1008ff4d  /* Launch Application          */
#define XKB_KEY_XF86LaunchE               0x1008ff4e  /* Launch Application          */
#define XKB_KEY_XF86LaunchF               0x1008ff4f  /* Launch Application          */

#define XKB_KEY_XF86ApplicationLeft       0x1008ff50  /* switch to application, left */
#define XKB_KEY_XF86ApplicationRight      0x1008ff51  /* switch to application, right*/
#define XKB_KEY_XF86Book                  0x1008ff52  /* Launch bookreader           */
#define XKB_KEY_XF86CD                    0x1008ff53  /* Launch CD/DVD player        */
#define XKB_KEY_XF86Calculater            0x1008ff54  /* Launch Calculater           */
#define XKB_KEY_XF86Clear                 0x1008ff55  /* Clear window, screen        */
#define XKB_KEY_XF86Close                 0x1008ff56  /* Close window                */
#define XKB_KEY_XF86Copy                  0x1008ff57  /* Copy selection              */
#define XKB_KEY_XF86Cut                   0x1008ff58  /* Cut selection               */
#define XKB_KEY_XF86Display               0x1008ff59  /* Output switch key           */
#define XKB_KEY_XF86DOS                   0x1008ff5a  /* Launch DOS (emulation)      */
#define XKB_KEY_XF86Documents             0x1008ff5b  /* Open documents window       */
#define XKB_KEY_XF86Excel                 0x1008ff5c  /* Launch spread sheet         */
#define XKB_KEY_XF86Explorer              0x1008ff5d  /* Launch file explorer        */
#define XKB_KEY_XF86Game                  0x1008ff5e  /* Launch game                 */
#define XKB_KEY_XF86Go                    0x1008ff5f  /* Go to URL                   */
#define XKB_KEY_XF86iTouch                0x1008ff60  /* Logitech iTouch- don't use  */
#define XKB_KEY_XF86LogOff                0x1008ff61  /* Log off system              */
#define XKB_KEY_XF86Market                0x1008ff62  /* ??                          */
#define XKB_KEY_XF86Meeting               0x1008ff63  /* enter meeting in calendar   */
#define XKB_KEY_XF86MenuKB                0x1008ff65  /* distinguish keyboard from PB */
#define XKB_KEY_XF86MenuPB                0x1008ff66  /* distinguish PB from keyboard */
#define XKB_KEY_XF86MySites               0x1008ff67  /* Favourites                  */
#define XKB_KEY_XF86New                   0x1008ff68  /* New (folder, document...    */
#define XKB_KEY_XF86News                  0x1008ff69  /* News                        */
#define XKB_KEY_XF86OfficeHome            0x1008ff6a  /* Office home (old Staroffice)*/
#define XKB_KEY_XF86Open                  0x1008ff6b  /* Open                        */
#define XKB_KEY_XF86Option                0x1008ff6c  /* ?? */
#define XKB_KEY_XF86Paste                 0x1008ff6d  /* Paste                       */
#define XKB_KEY_XF86Phone                 0x1008ff6e  /* Launch phone; dial number   */
#define XKB_KEY_XF86Q                     0x1008ff70  /* Compaq's Q - don't use      */
#define XKB_KEY_XF86Reply                 0x1008ff72  /* Reply e.g., mail            */
#define XKB_KEY_XF86Reload                0x1008ff73  /* Reload web page, file, etc. */
#define XKB_KEY_XF86RotateWindows         0x1008ff74  /* Rotate windows e.g. xrandr  */
#define XKB_KEY_XF86RotationPB            0x1008ff75  /* don't use                   */
#define XKB_KEY_XF86RotationKB            0x1008ff76  /* don't use                   */
#define XKB_KEY_XF86Save                  0x1008ff77  /* Save (file, document, state */
#define XKB_KEY_XF86ScrollUp              0x1008ff78  /* Scroll window/contents up   */
#define XKB_KEY_XF86ScrollDown            0x1008ff79  /* Scrool window/contentd down */
#define XKB_KEY_XF86ScrollClick           0x1008ff7a  /* Use XKB mousekeys instead   */
#define XKB_KEY_XF86Send                  0x1008ff7b  /* Send mail, file, object     */
#define XKB_KEY_XF86Spell                 0x1008ff7c  /* Spell checker               */
#define XKB_KEY_XF86SplitScreen           0x1008ff7d  /* Split window or screen      */
#define XKB_KEY_XF86Support               0x1008ff7e  /* Get support (??)            */
#define XKB_KEY_XF86TaskPane              0x1008ff7f  /* Show tasks */
#define XKB_KEY_XF86Terminal              0x1008ff80  /* Launch terminal emulator    */
#define XKB_KEY_XF86Tools                 0x1008ff81  /* toolbox of desktop/app.     */
#define XKB_KEY_XF86Travel                0x1008ff82  /* ?? */
#define XKB_KEY_XF86UserPB                0x1008ff84  /* ?? */
#define XKB_KEY_XF86User1KB               0x1008ff85  /* ?? */
#define XKB_KEY_XF86User2KB               0x1008ff86  /* ?? */
#define XKB_KEY_XF86Video                 0x1008ff87  /* Launch video player       */
#define XKB_KEY_XF86WheelButton           0x1008ff88  /* button from a mouse wheel */
#define XKB_KEY_XF86Word                  0x1008ff89  /* Launch word processor     */
#define XKB_KEY_XF86Xfer                  0x1008ff8a
#define XKB_KEY_XF86ZoomIn                0x1008ff8b  /* zoom in view, map, etc.   */
#define XKB_KEY_XF86ZoomOut               0x1008ff8c  /* zoom out view, map, etc.  */

#define XKB_KEY_XF86Away                  0x1008ff8d  /* mark yourself as away     */
#define XKB_KEY_XF86Messenger             0x1008ff8e  /* as in instant messaging   */
#define XKB_KEY_XF86WebCam                0x1008ff8f  /* Launch web camera app.    */
#define XKB_KEY_XF86MailForward           0x1008ff90  /* Forward in mail           */
#define XKB_KEY_XF86Pictures              0x1008ff91  /* Show pictures             */
#define XKB_KEY_XF86Music                 0x1008ff92  /* Launch music application  */

#define XKB_KEY_XF86Battery               0x1008ff93  /* Display battery information */
#define XKB_KEY_XF86Bluetooth             0x1008ff94  /* Enable/disable Bluetooth    */
#define XKB_KEY_XF86WLAN                  0x1008ff95  /* Enable/disable WLAN         */
#define XKB_KEY_XF86UWB                   0x1008ff96  /* Enable/disable UWB	    */

#define XKB_KEY_XF86AudioForward          0x1008ff97  /* fast-forward audio track    */
#define XKB_KEY_XF86AudioRepeat           0x1008ff98  /* toggle repeat mode          */
#define XKB_KEY_XF86AudioRandomPlay       0x1008ff99  /* toggle shuffle mode         */
#define XKB_KEY_XF86Subtitle              0x1008ff9a  /* cycle through subtitle      */
#define XKB_KEY_XF86AudioCycleTrack       0x1008ff9b  /* cycle through audio tracks  */
#define XKB_KEY_XF86CycleAngle            0x1008ff9c  /* cycle through angles        */
#define XKB_KEY_XF86FrameBack             0x1008ff9d  /* video: go one frame back    */
#define XKB_KEY_XF86FrameForward          0x1008ff9e  /* video: go one frame forward */
#define XKB_KEY_XF86Time                  0x1008ff9f  /* display, or shows an entry for time seeking */
#define XKB_KEY_XF86Select                0x1008ffa0  /* Select button on joypads and remotes */
#define XKB_KEY_XF86View                  0x1008ffa1  /* Show a view options/properties */
#define XKB_KEY_XF86TopMenu               0x1008ffa2  /* Go to a top-level menu in a video */

#define XKB_KEY_XF86Red                   0x1008ffa3  /* Red button                  */
#define XKB_KEY_XF86Green                 0x1008ffa4  /* Green button                */
#define XKB_KEY_XF86Yellow                0x1008ffa5  /* Yellow button               */
#define XKB_KEY_XF86Blue                  0x1008ffa6  /* Blue button                 */

#define XKB_KEY_XF86Suspend               0x1008ffa7  /* Sleep to RAM                */
#define XKB_KEY_XF86Hibernate             0x1008ffa8  /* Sleep to disk               */
#define XKB_KEY_XF86TouchpadToggle        0x1008ffa9  /* Toggle between touchpad/trackstick */
#define XKB_KEY_XF86TouchpadOn            0x1008ffb0  /* The touchpad got switched on */
#define XKB_KEY_XF86TouchpadOff           0x1008ffb1  /* The touchpad got switched off */

#define XKB_KEY_XF86AudioMicMute          0x1008ffb2  /* Mute the Mic from the system */

#define XKB_KEY_XF86Keyboard              0x1008ffb3  /* User defined keyboard related action */

#define XKB_KEY_XF86WWAN                  0x1008ffb4  /* Toggle WWAN (LTE, UMTS, etc.) radio */
#define XKB_KEY_XF86RFKill                0x1008ffb5  /* Toggle radios on/off */

#define XKB_KEY_XF86AudioPreset           0x1008ffb6  /* Select equalizer preset, e.g. theatre-mode */

#define XKB_KEY_XF86RotationLockToggle    0x1008ffb7  /* Toggle screen rotation lock on/off */

#define XKB_KEY_XF86FullScreen            0x1008ffb8  /* Toggle fullscreen */

/* Keys for special action keys (hot keys) */
/* Virtual terminals on some operating systems */
#define XKB_KEY_XF86Switch_VT_1           0x1008fe01
#define XKB_KEY_XF86Switch_VT_2           0x1008fe02
#define XKB_KEY_XF86Switch_VT_3           0x1008fe03
#define XKB_KEY_XF86Switch_VT_4           0x1008fe04
#define XKB_KEY_XF86Switch_VT_5           0x1008fe05
#define XKB_KEY_XF86Switch_VT_6           0x1008fe06
#define XKB_KEY_XF86Switch_VT_7           0x1008fe07
#define XKB_KEY_XF86Switch_VT_8           0x1008fe08
#define XKB_KEY_XF86Switch_VT_9           0x1008fe09
#define XKB_KEY_XF86Switch_VT_10          0x1008fe0a
#define XKB_KEY_XF86Switch_VT_11          0x1008fe0b
#define XKB_KEY_XF86Switch_VT_12          0x1008fe0c

#define XKB_KEY_XF86Ungrab                0x1008fe20  /* force ungrab               */
#define XKB_KEY_XF86ClearGrab             0x1008fe21  /* kill application with grab */
#define XKB_KEY_XF86Next_VMode            0x1008fe22  /* next video mode available  */
#define XKB_KEY_XF86Prev_VMode            0x1008fe23  /* prev. video mode available */
#define XKB_KEY_XF86LogWindowTree         0x1008fe24  /* print window tree to log   */
#define XKB_KEY_XF86LogGrabInfo           0x1008fe25  /* print all active grabs to log */


/*
 * Reserved range for evdev symbols: 0x10081000-0x10081FFF
 *
 * Key syms within this range must match the Linux kernel
 * input-event-codes.h file in the format:
 *     XKB_KEY_XF86CamelCaseKernelName	_EVDEVK(kernel value)
 * For example, the kernel
 *   #define KEY_MACRO_RECORD_START	0x2b0
 * effectively ends up as:
 *   #define XKB_KEY_XF86MacroRecordStart	0x100812b0
 *
 * For historical reasons, some keysyms within the reserved range will be
 * missing, most notably all "normal" keys that are mapped through default
 * XKB layouts (e.g. KEY_Q).
 *
 * CamelCasing is done with a human control as last authority, e.g. see VOD
 * instead of Vod for the Video on Demand key.
 *
 * The format for #defines is strict:
 *
 * #define XKB_KEY_XF86FOO<tab...>_EVDEVK(0xABC)<tab><tab> |* kver KEY_FOO *|
 *
 * Where
 * - alignment by tabs
 * - the _EVDEVK macro must be used
 * - the hex code must be in uppercase hex
 * - the kernel version (kver) is in the form v5.10
 * - kver and key name are within a slash-star comment (a pipe is used in
 *   this example for technical reasons)
 * These #defines are parsed by scripts. Do not stray from the given format.
 *
 * Where the evdev keycode is mapped to a different symbol, please add a
 * comment line starting with Use: but otherwise the same format, e.g.
 *  Use: XKB_KEY_XF86RotationLockToggle	_EVDEVK(0x231)		   v4.16 KEY_ROTATE_LOCK_TOGGLE
 *
 */
/* Use: XKB_KEY_XF86Eject                    _EVDEVK(0x0a2)             KEY_EJECTCLOSECD */
/* Use: XKB_KEY_XF86New                      _EVDEVK(0x0b5)     v2.6.14 KEY_NEW */
/* Use: XKB_KEY_Redo                         _EVDEVK(0x0b6)     v2.6.14 KEY_REDO */
/* KEY_DASHBOARD has been mapped to LaunchB in xkeyboard-config since 2011 */
/* Use: XKB_KEY_XF86LaunchB                  _EVDEVK(0x0cc)     v2.6.28 KEY_DASHBOARD */
/* Use: XKB_KEY_XF86Display                  _EVDEVK(0x0e3)     v2.6.12 KEY_SWITCHVIDEOMODE */
/* Use: XKB_KEY_XF86KbdLightOnOff            _EVDEVK(0x0e4)     v2.6.12 KEY_KBDILLUMTOGGLE */
/* Use: XKB_KEY_XF86KbdBrightnessDown        _EVDEVK(0x0e5)     v2.6.12 KEY_KBDILLUMDOWN */
/* Use: XKB_KEY_XF86KbdBrightnessUp          _EVDEVK(0x0e6)     v2.6.12 KEY_KBDILLUMUP */
/* Use: XKB_KEY_XF86Send                     _EVDEVK(0x0e7)     v2.6.14 KEY_SEND */
/* Use: XKB_KEY_XF86Reply                    _EVDEVK(0x0e8)     v2.6.14 KEY_REPLY */
/* Use: XKB_KEY_XF86MailForward              _EVDEVK(0x0e9)     v2.6.14 KEY_FORWARDMAIL */
/* Use: XKB_KEY_XF86Save                     _EVDEVK(0x0ea)     v2.6.14 KEY_SAVE */
/* Use: XKB_KEY_XF86Documents                _EVDEVK(0x0eb)     v2.6.14 KEY_DOCUMENTS */
/* Use: XKB_KEY_XF86Battery                  _EVDEVK(0x0ec)     v2.6.17 KEY_BATTERY */
/* Use: XKB_KEY_XF86Bluetooth                _EVDEVK(0x0ed)     v2.6.19 KEY_BLUETOOTH */
/* Use: XKB_KEY_XF86WLAN                     _EVDEVK(0x0ee)     v2.6.19 KEY_WLAN */
/* Use: XKB_KEY_XF86UWB                      _EVDEVK(0x0ef)     v2.6.24 KEY_UWB */
/* Use: XKB_KEY_XF86Next_VMode               _EVDEVK(0x0f1)     v2.6.23 KEY_VIDEO_NEXT */
/* Use: XKB_KEY_XF86Prev_VMode               _EVDEVK(0x0f2)     v2.6.23 KEY_VIDEO_PREV */
/* Use: XKB_KEY_XF86MonBrightnessCycle       _EVDEVK(0x0f3)     v2.6.23 KEY_BRIGHTNESS_CYCLE */
#define XKB_KEY_XF86BrightnessAuto           0x100810f4      /* v3.16   KEY_BRIGHTNESS_AUTO */
#define XKB_KEY_XF86DisplayOff               0x100810f5      /* v2.6.23 KEY_DISPLAY_OFF */
/* Use: XKB_KEY_XF86WWAN                     _EVDEVK(0x0f6)     v3.13   KEY_WWAN */
/* Use: XKB_KEY_XF86RFKill                   _EVDEVK(0x0f7)     v2.6.33 KEY_RFKILL */
/* Use: XKB_KEY_XF86AudioMicMute             _EVDEVK(0x0f8)     v3.1    KEY_MICMUTE */
#define XKB_KEY_XF86Info                     0x10081166      /*         KEY_INFO */
/* Use: XKB_KEY_XF86CycleAngle               _EVDEVK(0x173)             KEY_ANGLE */
/* Use: XKB_KEY_XF86FullScreen               _EVDEVK(0x174)     v5.1    KEY_FULL_SCREEN */
#define XKB_KEY_XF86AspectRatio              0x10081177      /* v5.1    KEY_ASPECT_RATIO */
#define XKB_KEY_XF86DVD                      0x10081185      /*         KEY_DVD */
#define XKB_KEY_XF86Audio                    0x10081188      /*         KEY_AUDIO */
/* Use: XKB_KEY_XF86Video                    _EVDEVK(0x189)             KEY_VIDEO */
/* Use: XKB_KEY_XF86Calendar                 _EVDEVK(0x18d)             KEY_CALENDAR */
#define XKB_KEY_XF86ChannelUp                0x10081192      /*         KEY_CHANNELUP */
#define XKB_KEY_XF86ChannelDown              0x10081193      /*         KEY_CHANNELDOWN */
/* Use: XKB_KEY_XF86AudioRandomPlay          _EVDEVK(0x19a)             KEY_SHUFFLE */
#define XKB_KEY_XF86Break                    0x1008119b      /*         KEY_BREAK */
#define XKB_KEY_XF86VideoPhone               0x100811a0      /* v2.6.20 KEY_VIDEOPHONE */
/* Use: XKB_KEY_XF86Game                     _EVDEVK(0x1a1)     v2.6.20 KEY_GAMES */
/* Use: XKB_KEY_XF86ZoomIn                   _EVDEVK(0x1a2)     v2.6.20 KEY_ZOOMIN */
/* Use: XKB_KEY_XF86ZoomOut                  _EVDEVK(0x1a3)     v2.6.20 KEY_ZOOMOUT */
#define XKB_KEY_XF86ZoomReset                0x100811a4      /* v2.6.20 KEY_ZOOMRESET */
/* Use: XKB_KEY_XF86Word                     _EVDEVK(0x1a5)     v2.6.20 KEY_WORDPROCESSOR */
#define XKB_KEY_XF86Editor                   0x100811a6      /* v2.6.20 KEY_EDITOR */
/* Use: XKB_KEY_XF86Excel                    _EVDEVK(0x1a7)     v2.6.20 KEY_SPREADSHEET */
#define XKB_KEY_XF86GraphicsEditor           0x100811a8      /* v2.6.20 KEY_GRAPHICSEDITOR */
#define XKB_KEY_XF86Presentation             0x100811a9      /* v2.6.20 KEY_PRESENTATION */
#define XKB_KEY_XF86Database                 0x100811aa      /* v2.6.20 KEY_DATABASE */
/* Use: XKB_KEY_XF86News                     _EVDEVK(0x1ab)     v2.6.20 KEY_NEWS */
#define XKB_KEY_XF86Voicemail                0x100811ac      /* v2.6.20 KEY_VOICEMAIL */
#define XKB_KEY_XF86Addressbook              0x100811ad      /* v2.6.20 KEY_ADDRESSBOOK */
/* Use: XKB_KEY_XF86Messenger                _EVDEVK(0x1ae)     v2.6.20 KEY_MESSENGER */
#define XKB_KEY_XF86DisplayToggle            0x100811af      /* v2.6.20 KEY_DISPLAYTOGGLE */
#define XKB_KEY_XF86SpellCheck               0x100811b0      /* v2.6.24 KEY_SPELLCHECK */
/* Use: XKB_KEY_XF86LogOff                   _EVDEVK(0x1b1)     v2.6.24 KEY_LOGOFF */
/* Use: XKB_KEY_dollar                       _EVDEVK(0x1b2)     v2.6.24 KEY_DOLLAR */
/* Use: XKB_KEY_EuroSign                     _EVDEVK(0x1b3)     v2.6.24 KEY_EURO */
/* Use: XKB_KEY_XF86FrameBack                _EVDEVK(0x1b4)     v2.6.24 KEY_FRAMEBACK */
/* Use: XKB_KEY_XF86FrameForward             _EVDEVK(0x1b5)     v2.6.24 KEY_FRAMEFORWARD */
#define XKB_KEY_XF86ContextMenu              0x100811b6      /* v2.6.24 KEY_CONTEXT_MENU */
#define XKB_KEY_XF86MediaRepeat              0x100811b7      /* v2.6.26 KEY_MEDIA_REPEAT */
#define XKB_KEY_XF8610ChannelsUp             0x100811b8      /* v2.6.38 KEY_10CHANNELSUP */
#define XKB_KEY_XF8610ChannelsDown           0x100811b9      /* v2.6.38 KEY_10CHANNELSDOWN */
#define XKB_KEY_XF86Images                   0x100811ba      /* v2.6.39 KEY_IMAGES */
#define XKB_KEY_XF86NotificationCenter       0x100811bc      /* v5.10   KEY_NOTIFICATION_CENTER */
#define XKB_KEY_XF86PickupPhone              0x100811bd      /* v5.10   KEY_PICKUP_PHONE */
#define XKB_KEY_XF86HangupPhone              0x100811be      /* v5.10   KEY_HANGUP_PHONE */
#define XKB_KEY_XF86Fn                       0x100811d0      /*         KEY_FN */
#define XKB_KEY_XF86Fn_Esc                   0x100811d1      /*         KEY_FN_ESC */
#define XKB_KEY_XF86FnRightShift             0x100811e5      /* v5.10   KEY_FN_RIGHT_SHIFT */
/* Use: XKB_KEY_braille_dot_1                _EVDEVK(0x1f1)     v2.6.17 KEY_BRL_DOT1 */
/* Use: XKB_KEY_braille_dot_2                _EVDEVK(0x1f2)     v2.6.17 KEY_BRL_DOT2 */
/* Use: XKB_KEY_braille_dot_3                _EVDEVK(0x1f3)     v2.6.17 KEY_BRL_DOT3 */
/* Use: XKB_KEY_braille_dot_4                _EVDEVK(0x1f4)     v2.6.17 KEY_BRL_DOT4 */
/* Use: XKB_KEY_braille_dot_5                _EVDEVK(0x1f5)     v2.6.17 KEY_BRL_DOT5 */
/* Use: XKB_KEY_braille_dot_6                _EVDEVK(0x1f6)     v2.6.17 KEY_BRL_DOT6 */
/* Use: XKB_KEY_braille_dot_7                _EVDEVK(0x1f7)     v2.6.17 KEY_BRL_DOT7 */
/* Use: XKB_KEY_braille_dot_8                _EVDEVK(0x1f8)     v2.6.17 KEY_BRL_DOT8 */
/* Use: XKB_KEY_braille_dot_9                _EVDEVK(0x1f9)     v2.6.23 KEY_BRL_DOT9 */
/* Use: XKB_KEY_braille_dot_1                _EVDEVK(0x1fa)     v2.6.23 KEY_BRL_DOT10 */
#define XKB_KEY_XF86Numeric0                 0x10081200      /* v2.6.28 KEY_NUMERIC_0 */
#define XKB_KEY_XF86Numeric1                 0x10081201      /* v2.6.28 KEY_NUMERIC_1 */
#define XKB_KEY_XF86Numeric2                 0x10081202      /* v2.6.28 KEY_NUMERIC_2 */
#define XKB_KEY_XF86Numeric3                 0x10081203      /* v2.6.28 KEY_NUMERIC_3 */
#define XKB_KEY_XF86Numeric4                 0x10081204      /* v2.6.28 KEY_NUMERIC_4 */
#define XKB_KEY_XF86Numeric5                 0x10081205      /* v2.6.28 KEY_NUMERIC_5 */
#define XKB_KEY_XF86Numeric6                 0x10081206      /* v2.6.28 KEY_NUMERIC_6 */
#define XKB_KEY_XF86Numeric7                 0x10081207      /* v2.6.28 KEY_NUMERIC_7 */
#define XKB_KEY_XF86Numeric8                 0x10081208      /* v2.6.28 KEY_NUMERIC_8 */
#define XKB_KEY_XF86Numeric9                 0x10081209      /* v2.6.28 KEY_NUMERIC_9 */
#define XKB_KEY_XF86NumericStar              0x1008120a      /* v2.6.28 KEY_NUMERIC_STAR */
#define XKB_KEY_XF86NumericPound             0x1008120b      /* v2.6.28 KEY_NUMERIC_POUND */
#define XKB_KEY_XF86NumericA                 0x1008120c      /* v4.1    KEY_NUMERIC_A */
#define XKB_KEY_XF86NumericB                 0x1008120d      /* v4.1    KEY_NUMERIC_B */
#define XKB_KEY_XF86NumericC                 0x1008120e      /* v4.1    KEY_NUMERIC_C */
#define XKB_KEY_XF86NumericD                 0x1008120f      /* v4.1    KEY_NUMERIC_D */
#define XKB_KEY_XF86CameraFocus              0x10081210      /* v2.6.33 KEY_CAMERA_FOCUS */
#define XKB_KEY_XF86WPSButton                0x10081211      /* v2.6.34 KEY_WPS_BUTTON */
/* Use: XKB_KEY_XF86TouchpadToggle           _EVDEVK(0x212)     v2.6.37 KEY_TOUCHPAD_TOGGLE */
/* Use: XKB_KEY_XF86TouchpadOn               _EVDEVK(0x213)     v2.6.37 KEY_TOUCHPAD_ON */
/* Use: XKB_KEY_XF86TouchpadOff              _EVDEVK(0x214)     v2.6.37 KEY_TOUCHPAD_OFF */
#define XKB_KEY_XF86CameraZoomIn             0x10081215      /* v2.6.39 KEY_CAMERA_ZOOMIN */
#define XKB_KEY_XF86CameraZoomOut            0x10081216      /* v2.6.39 KEY_CAMERA_ZOOMOUT */
#define XKB_KEY_XF86CameraUp                 0x10081217      /* v2.6.39 KEY_CAMERA_UP */
#define XKB_KEY_XF86CameraDown               0x10081218      /* v2.6.39 KEY_CAMERA_DOWN */
#define XKB_KEY_XF86CameraLeft               0x10081219      /* v2.6.39 KEY_CAMERA_LEFT */
#define XKB_KEY_XF86CameraRight              0x1008121a      /* v2.6.39 KEY_CAMERA_RIGHT */
#define XKB_KEY_XF86AttendantOn              0x1008121b      /* v3.10   KEY_ATTENDANT_ON */
#define XKB_KEY_XF86AttendantOff             0x1008121c      /* v3.10   KEY_ATTENDANT_OFF */
#define XKB_KEY_XF86AttendantToggle          0x1008121d      /* v3.10   KEY_ATTENDANT_TOGGLE */
#define XKB_KEY_XF86LightsToggle             0x1008121e      /* v3.10   KEY_LIGHTS_TOGGLE */
#define XKB_KEY_XF86ALSToggle                0x10081230      /* v3.13   KEY_ALS_TOGGLE */
/* Use: XKB_KEY_XF86RotationLockToggle       _EVDEVK(0x231)     v4.16   KEY_ROTATE_LOCK_TOGGLE */
#define XKB_KEY_XF86Buttonconfig             0x10081240      /* v3.16   KEY_BUTTONCONFIG */
#define XKB_KEY_XF86Taskmanager              0x10081241      /* v3.16   KEY_TASKMANAGER */
#define XKB_KEY_XF86Journal                  0x10081242      /* v3.16   KEY_JOURNAL */
#define XKB_KEY_XF86ControlPanel             0x10081243      /* v3.16   KEY_CONTROLPANEL */
#define XKB_KEY_XF86AppSelect                0x10081244      /* v3.16   KEY_APPSELECT */
#define XKB_KEY_XF86Screensaver              0x10081245      /* v3.16   KEY_SCREENSAVER */
#define XKB_KEY_XF86VoiceCommand             0x10081246      /* v3.16   KEY_VOICECOMMAND */
#define XKB_KEY_XF86Assistant                0x10081247      /* v4.13   KEY_ASSISTANT */
/* Use: XKB_KEY_ISO_Next_Group               _EVDEVK(0x248)     v5.2    KEY_KBD_LAYOUT_NEXT */
#define XKB_KEY_XF86EmojiPicker              0x10081249      /* v5.13   KEY_EMOJI_PICKER */
#define XKB_KEY_XF86Dictate                  0x1008124a      /* v5.17   KEY_DICTATE */
#define XKB_KEY_XF86CameraAccessEnable       0x1008124b      /* v6.2    KEY_CAMERA_ACCESS_ENABLE */
#define XKB_KEY_XF86CameraAccessDisable      0x1008124c      /* v6.2    KEY_CAMERA_ACCESS_DISABLE */
#define XKB_KEY_XF86CameraAccessToggle       0x1008124d      /* v6.2    KEY_CAMERA_ACCESS_TOGGLE */
#define XKB_KEY_XF86BrightnessMin            0x10081250      /* v3.16   KEY_BRIGHTNESS_MIN */
#define XKB_KEY_XF86BrightnessMax            0x10081251      /* v3.16   KEY_BRIGHTNESS_MAX */
#define XKB_KEY_XF86KbdInputAssistPrev       0x10081260      /* v3.18   KEY_KBDINPUTASSIST_PREV */
#define XKB_KEY_XF86KbdInputAssistNext       0x10081261      /* v3.18   KEY_KBDINPUTASSIST_NEXT */
#define XKB_KEY_XF86KbdInputAssistPrevgroup  0x10081262      /* v3.18   KEY_KBDINPUTASSIST_PREVGROUP */
#define XKB_KEY_XF86KbdInputAssistNextgroup  0x10081263      /* v3.18   KEY_KBDINPUTASSIST_NEXTGROUP */
#define XKB_KEY_XF86KbdInputAssistAccept     0x10081264      /* v3.18   KEY_KBDINPUTASSIST_ACCEPT */
#define XKB_KEY_XF86KbdInputAssistCancel     0x10081265      /* v3.18   KEY_KBDINPUTASSIST_CANCEL */
#define XKB_KEY_XF86RightUp                  0x10081266      /* v4.7    KEY_RIGHT_UP */
#define XKB_KEY_XF86RightDown                0x10081267      /* v4.7    KEY_RIGHT_DOWN */
#define XKB_KEY_XF86LeftUp                   0x10081268      /* v4.7    KEY_LEFT_UP */
#define XKB_KEY_XF86LeftDown                 0x10081269      /* v4.7    KEY_LEFT_DOWN */
#define XKB_KEY_XF86RootMenu                 0x1008126a      /* v4.7    KEY_ROOT_MENU */
#define XKB_KEY_XF86MediaTopMenu             0x1008126b      /* v4.7    KEY_MEDIA_TOP_MENU */
#define XKB_KEY_XF86Numeric11                0x1008126c      /* v4.7    KEY_NUMERIC_11 */
#define XKB_KEY_XF86Numeric12                0x1008126d      /* v4.7    KEY_NUMERIC_12 */
#define XKB_KEY_XF86AudioDesc                0x1008126e      /* v4.7    KEY_AUDIO_DESC */
#define XKB_KEY_XF863DMode                   0x1008126f      /* v4.7    KEY_3D_MODE */
#define XKB_KEY_XF86NextFavorite             0x10081270      /* v4.7    KEY_NEXT_FAVORITE */
#define XKB_KEY_XF86StopRecord               0x10081271      /* v4.7    KEY_STOP_RECORD */
#define XKB_KEY_XF86PauseRecord              0x10081272      /* v4.7    KEY_PAUSE_RECORD */
#define XKB_KEY_XF86VOD                      0x10081273      /* v4.7    KEY_VOD */
#define XKB_KEY_XF86Unmute                   0x10081274      /* v4.7    KEY_UNMUTE */
#define XKB_KEY_XF86FastReverse              0x10081275      /* v4.7    KEY_FASTREVERSE */
#define XKB_KEY_XF86SlowReverse              0x10081276      /* v4.7    KEY_SLOWREVERSE */
#define XKB_KEY_XF86Data                     0x10081277      /* v4.7    KEY_DATA */
#define XKB_KEY_XF86OnScreenKeyboard         0x10081278      /* v4.12   KEY_ONSCREEN_KEYBOARD */
#define XKB_KEY_XF86PrivacyScreenToggle      0x10081279      /* v5.5    KEY_PRIVACY_SCREEN_TOGGLE */
#define XKB_KEY_XF86SelectiveScreenshot      0x1008127a      /* v5.6    KEY_SELECTIVE_SCREENSHOT */
#define XKB_KEY_XF86NextElement              0x1008127b      /* v5.18   KEY_NEXT_ELEMENT */
#define XKB_KEY_XF86PreviousElement          0x1008127c      /* v5.18   KEY_PREVIOUS_ELEMENT */
#define XKB_KEY_XF86AutopilotEngageToggle    0x1008127d      /* v5.18   KEY_AUTOPILOT_ENGAGE_TOGGLE */
#define XKB_KEY_XF86MarkWaypoint             0x1008127e      /* v5.18   KEY_MARK_WAYPOINT */
#define XKB_KEY_XF86Sos                      0x1008127f      /* v5.18   KEY_SOS */
#define XKB_KEY_XF86NavChart                 0x10081280      /* v5.18   KEY_NAV_CHART */
#define XKB_KEY_XF86FishingChart             0x10081281      /* v5.18   KEY_FISHING_CHART */
#define XKB_KEY_XF86SingleRangeRadar         0x10081282      /* v5.18   KEY_SINGLE_RANGE_RADAR */
#define XKB_KEY_XF86DualRangeRadar           0x10081283      /* v5.18   KEY_DUAL_RANGE_RADAR */
#define XKB_KEY_XF86RadarOverlay             0x10081284      /* v5.18   KEY_RADAR_OVERLAY */
#define XKB_KEY_XF86TraditionalSonar         0x10081285      /* v5.18   KEY_TRADITIONAL_SONAR */
#define XKB_KEY_XF86ClearvuSonar             0x10081286      /* v5.18   KEY_CLEARVU_SONAR */
#define XKB_KEY_XF86SidevuSonar              0x10081287      /* v5.18   KEY_SIDEVU_SONAR */
#define XKB_KEY_XF86NavInfo                  0x10081288      /* v5.18   KEY_NAV_INFO */
/* Use: XKB_KEY_XF86BrightnessAdjust         _EVDEVK(0x289)     v5.18   KEY_BRIGHTNESS_MENU */
#define XKB_KEY_XF86Macro1                   0x10081290      /* v5.5    KEY_MACRO1 */
#define XKB_KEY_XF86Macro2                   0x10081291      /* v5.5    KEY_MACRO2 */
#define XKB_KEY_XF86Macro3                   0x10081292      /* v5.5    KEY_MACRO3 */
#define XKB_KEY_XF86Macro4                   0x10081293      /* v5.5    KEY_MACRO4 */
#define XKB_KEY_XF86Macro5                   0x10081294      /* v5.5    KEY_MACRO5 */
#define XKB_KEY_XF86Macro6                   0x10081295      /* v5.5    KEY_MACRO6 */
#define XKB_KEY_XF86Macro7                   0x10081296      /* v5.5    KEY_MACRO7 */
#define XKB_KEY_XF86Macro8                   0x10081297      /* v5.5    KEY_MACRO8 */
#define XKB_KEY_XF86Macro9                   0x10081298      /* v5.5    KEY_MACRO9 */
#define XKB_KEY_XF86Macro10                  0x10081299      /* v5.5    KEY_MACRO10 */
#define XKB_KEY_XF86Macro11                  0x1008129a      /* v5.5    KEY_MACRO11 */
#define XKB_KEY_XF86Macro12                  0x1008129b      /* v5.5    KEY_MACRO12 */
#define XKB_KEY_XF86Macro13                  0x1008129c      /* v5.5    KEY_MACRO13 */
#define XKB_KEY_XF86Macro14                  0x1008129d      /* v5.5    KEY_MACRO14 */
#define XKB_KEY_XF86Macro15                  0x1008129e      /* v5.5    KEY_MACRO15 */
#define XKB_KEY_XF86Macro16                  0x1008129f      /* v5.5    KEY_MACRO16 */
#define XKB_KEY_XF86Macro17                  0x100812a0      /* v5.5    KEY_MACRO17 */
#define XKB_KEY_XF86Macro18                  0x100812a1      /* v5.5    KEY_MACRO18 */
#define XKB_KEY_XF86Macro19                  0x100812a2      /* v5.5    KEY_MACRO19 */
#define XKB_KEY_XF86Macro20                  0x100812a3      /* v5.5    KEY_MACRO20 */
#define XKB_KEY_XF86Macro21                  0x100812a4      /* v5.5    KEY_MACRO21 */
#define XKB_KEY_XF86Macro22                  0x100812a5      /* v5.5    KEY_MACRO22 */
#define XKB_KEY_XF86Macro23                  0x100812a6      /* v5.5    KEY_MACRO23 */
#define XKB_KEY_XF86Macro24                  0x100812a7      /* v5.5    KEY_MACRO24 */
#define XKB_KEY_XF86Macro25                  0x100812a8      /* v5.5    KEY_MACRO25 */
#define XKB_KEY_XF86Macro26                  0x100812a9      /* v5.5    KEY_MACRO26 */
#define XKB_KEY_XF86Macro27                  0x100812aa      /* v5.5    KEY_MACRO27 */
#define XKB_KEY_XF86Macro28                  0x100812ab      /* v5.5    KEY_MACRO28 */
#define XKB_KEY_XF86Macro29                  0x100812ac      /* v5.5    KEY_MACRO29 */
#define XKB_KEY_XF86Macro30                  0x100812ad      /* v5.5    KEY_MACRO30 */
#define XKB_KEY_XF86MacroRecordStart         0x100812b0      /* v5.5    KEY_MACRO_RECORD_START */
#define XKB_KEY_XF86MacroRecordStop          0x100812b1      /* v5.5    KEY_MACRO_RECORD_STOP */
#define XKB_KEY_XF86MacroPresetCycle         0x100812b2      /* v5.5    KEY_MACRO_PRESET_CYCLE */
#define XKB_KEY_XF86MacroPreset1             0x100812b3      /* v5.5    KEY_MACRO_PRESET1 */
#define XKB_KEY_XF86MacroPreset2             0x100812b4      /* v5.5    KEY_MACRO_PRESET2 */
#define XKB_KEY_XF86MacroPreset3             0x100812b5      /* v5.5    KEY_MACRO_PRESET3 */
#define XKB_KEY_XF86KbdLcdMenu1              0x100812b8      /* v5.5    KEY_KBD_LCD_MENU1 */
#define XKB_KEY_XF86KbdLcdMenu2              0x100812b9      /* v5.5    KEY_KBD_LCD_MENU2 */
#define XKB_KEY_XF86KbdLcdMenu3              0x100812ba      /* v5.5    KEY_KBD_LCD_MENU3 */
#define XKB_KEY_XF86KbdLcdMenu4              0x100812bb      /* v5.5    KEY_KBD_LCD_MENU4 */
#define XKB_KEY_XF86KbdLcdMenu5              0x100812bc      /* v5.5    KEY_KBD_LCD_MENU5 */
#undef _EVDEVK
/*
 * Copyright (c) 1991, Oracle and/or its affiliates.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */
/************************************************************

Copyright 1991, 1998  The Open Group

Permission to use, copy, modify, distribute, and sell this software and its
documentation for any purpose is hereby granted without fee, provided that
the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation.

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
OPEN GROUP BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of The Open Group shall not be
used in advertising or otherwise to promote the sale, use or other dealings
in this Software without prior written authorization from The Open Group.

***********************************************************/

/*
 * Floating Accent
 */

#define XKB_KEY_SunFA_Grave               0x1005ff00
#define XKB_KEY_SunFA_Circum              0x1005ff01
#define XKB_KEY_SunFA_Tilde               0x1005ff02
#define XKB_KEY_SunFA_Acute               0x1005ff03
#define XKB_KEY_SunFA_Diaeresis           0x1005ff04
#define XKB_KEY_SunFA_Cedilla             0x1005ff05

/*
 * Miscellaneous Functions
 */

#define XKB_KEY_SunF36                    0x1005ff10  /* Labeled F11 */
#define XKB_KEY_SunF37                    0x1005ff11  /* Labeled F12 */

#define XKB_KEY_SunSys_Req                0x1005ff60
#define XKB_KEY_SunPrint_Screen           0x0000ff61  /* Same as XKB_KEY_Print */

/*
 * International & Multi-Key Character Composition
 */

#define XKB_KEY_SunCompose                0x0000ff20  /* Same as XKB_KEY_Multi_key */
#define XKB_KEY_SunAltGraph               0x0000ff7e  /* Same as XKB_KEY_Mode_switch */

/*
 * Cursor Control
 */

#define XKB_KEY_SunPageUp                 0x0000ff55  /* Same as XKB_KEY_Prior */
#define XKB_KEY_SunPageDown               0x0000ff56  /* Same as XKB_KEY_Next */

/*
 * Open Look Functions
 */

#define XKB_KEY_SunUndo                   0x0000ff65  /* Same as XKB_KEY_Undo */
#define XKB_KEY_SunAgain                  0x0000ff66  /* Same as XKB_KEY_Redo */
#define XKB_KEY_SunFind                   0x0000ff68  /* Same as XKB_KEY_Find */
#define XKB_KEY_SunStop                   0x0000ff69  /* Same as XKB_KEY_Cancel */
#define XKB_KEY_SunProps                  0x1005ff70
#define XKB_KEY_SunFront                  0x1005ff71
#define XKB_KEY_SunCopy                   0x1005ff72
#define XKB_KEY_SunOpen                   0x1005ff73
#define XKB_KEY_SunPaste                  0x1005ff74
#define XKB_KEY_SunCut                    0x1005ff75

#define XKB_KEY_SunPowerSwitch            0x1005ff76
#define XKB_KEY_SunAudioLowerVolume       0x1005ff77
#define XKB_KEY_SunAudioMute              0x1005ff78
#define XKB_KEY_SunAudioRaiseVolume       0x1005ff79
#define XKB_KEY_SunVideoDegauss           0x1005ff7a
#define XKB_KEY_SunVideoLowerBrightness   0x1005ff7b
#define XKB_KEY_SunVideoRaiseBrightness   0x1005ff7c
#define XKB_KEY_SunPowerSwitchShift       0x1005ff7d
/***********************************************************

Copyright 1988, 1998  The Open Group

Permission to use, copy, modify, distribute, and sell this software and its
documentation for any purpose is hereby granted without fee, provided that
the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation.

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
OPEN GROUP BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of The Open Group shall not be
used in advertising or otherwise to promote the sale, use or other dealings
in this Software without prior written authorization from The Open Group.


Copyright 1988 by Digital Equipment Corporation, Maynard, Massachusetts.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of Digital not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

DIGITAL DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
DIGITAL BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

******************************************************************/

/*
 * DEC private keysyms
 * (29th bit set)
 */

/* two-key compose sequence initiators, chosen to map to Latin1 characters */

#define XKB_KEY_Dring_accent              0x1000feb0
#define XKB_KEY_Dcircumflex_accent        0x1000fe5e
#define XKB_KEY_Dcedilla_accent           0x1000fe2c
#define XKB_KEY_Dacute_accent             0x1000fe27
#define XKB_KEY_Dgrave_accent             0x1000fe60
#define XKB_KEY_Dtilde                    0x1000fe7e
#define XKB_KEY_Ddiaeresis                0x1000fe22

/* special keysym for LK2** "Remove" key on editing keypad */

#define XKB_KEY_DRemove                   0x1000ff00  /* Remove */
/*

Copyright 1987, 1998  The Open Group

Permission to use, copy, modify, distribute, and sell this software and its
documentation for any purpose is hereby granted without fee, provided that
the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation.

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE OPEN GROUP BE LIABLE FOR ANY CLAIM, DAMAGES OR
OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of The Open Group shall
not be used in advertising or otherwise to promote the sale, use or
other dealings in this Software without prior written authorization
from The Open Group.

Copyright 1987 by Digital Equipment Corporation, Maynard, Massachusetts,

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the names of Hewlett Packard
or Digital not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

DIGITAL DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
DIGITAL BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

HEWLETT-PACKARD MAKES NO WARRANTY OF ANY KIND WITH REGARD
TO THIS SOFTWARE, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE.  Hewlett-Packard shall not be liable for errors
contained herein or direct, indirect, special, incidental or
consequential damages in connection with the furnishing,
performance, or use of this material.

*/



#define XKB_KEY_hpClearLine               0x1000ff6f
#define XKB_KEY_hpInsertLine              0x1000ff70
#define XKB_KEY_hpDeleteLine              0x1000ff71
#define XKB_KEY_hpInsertChar              0x1000ff72
#define XKB_KEY_hpDeleteChar              0x1000ff73
#define XKB_KEY_hpBackTab                 0x1000ff74
#define XKB_KEY_hpKP_BackTab              0x1000ff75
#define XKB_KEY_hpModelock1               0x1000ff48
#define XKB_KEY_hpModelock2               0x1000ff49
#define XKB_KEY_hpReset                   0x1000ff6c
#define XKB_KEY_hpSystem                  0x1000ff6d
#define XKB_KEY_hpUser                    0x1000ff6e
#define XKB_KEY_hpmute_acute              0x100000a8
#define XKB_KEY_hpmute_grave              0x100000a9
#define XKB_KEY_hpmute_asciicircum        0x100000aa
#define XKB_KEY_hpmute_diaeresis          0x100000ab
#define XKB_KEY_hpmute_asciitilde         0x100000ac
#define XKB_KEY_hplira                    0x100000af
#define XKB_KEY_hpguilder                 0x100000be
#define XKB_KEY_hpYdiaeresis              0x100000ee
#define XKB_KEY_hpIO                      0x100000ee  /* deprecated alias for hpYdiaeresis */
#define XKB_KEY_hplongminus               0x100000f6
#define XKB_KEY_hpblock                   0x100000fc



#define XKB_KEY_osfCopy                   0x1004ff02
#define XKB_KEY_osfCut                    0x1004ff03
#define XKB_KEY_osfPaste                  0x1004ff04
#define XKB_KEY_osfBackTab                0x1004ff07
#define XKB_KEY_osfBackSpace              0x1004ff08
#define XKB_KEY_osfClear                  0x1004ff0b
#define XKB_KEY_osfEscape                 0x1004ff1b
#define XKB_KEY_osfAddMode                0x1004ff31
#define XKB_KEY_osfPrimaryPaste           0x1004ff32
#define XKB_KEY_osfQuickPaste             0x1004ff33
#define XKB_KEY_osfPageLeft               0x1004ff40
#define XKB_KEY_osfPageUp                 0x1004ff41
#define XKB_KEY_osfPageDown               0x1004ff42
#define XKB_KEY_osfPageRight              0x1004ff43
#define XKB_KEY_osfActivate               0x1004ff44
#define XKB_KEY_osfMenuBar                0x1004ff45
#define XKB_KEY_osfLeft                   0x1004ff51
#define XKB_KEY_osfUp                     0x1004ff52
#define XKB_KEY_osfRight                  0x1004ff53
#define XKB_KEY_osfDown                   0x1004ff54
#define XKB_KEY_osfEndLine                0x1004ff57
#define XKB_KEY_osfBeginLine              0x1004ff58
#define XKB_KEY_osfEndData                0x1004ff59
#define XKB_KEY_osfBeginData              0x1004ff5a
#define XKB_KEY_osfPrevMenu               0x1004ff5b
#define XKB_KEY_osfNextMenu               0x1004ff5c
#define XKB_KEY_osfPrevField              0x1004ff5d
#define XKB_KEY_osfNextField              0x1004ff5e
#define XKB_KEY_osfSelect                 0x1004ff60
#define XKB_KEY_osfInsert                 0x1004ff63
#define XKB_KEY_osfUndo                   0x1004ff65
#define XKB_KEY_osfMenu                   0x1004ff67
#define XKB_KEY_osfCancel                 0x1004ff69
#define XKB_KEY_osfHelp                   0x1004ff6a
#define XKB_KEY_osfSelectAll              0x1004ff71
#define XKB_KEY_osfDeselectAll            0x1004ff72
#define XKB_KEY_osfReselect               0x1004ff73
#define XKB_KEY_osfExtend                 0x1004ff74
#define XKB_KEY_osfRestore                0x1004ff78
#define XKB_KEY_osfDelete                 0x1004ffff



/**************************************************************
 * The use of the following macros is deprecated.
 * They are listed below only for backwards compatibility.
 */
#define XKB_KEY_Reset                     0x1000ff6c  /* deprecated alias for hpReset */
#define XKB_KEY_System                    0x1000ff6d  /* deprecated alias for hpSystem */
#define XKB_KEY_User                      0x1000ff6e  /* deprecated alias for hpUser */
#define XKB_KEY_ClearLine                 0x1000ff6f  /* deprecated alias for hpClearLine */
#define XKB_KEY_InsertLine                0x1000ff70  /* deprecated alias for hpInsertLine */
#define XKB_KEY_DeleteLine                0x1000ff71  /* deprecated alias for hpDeleteLine */
#define XKB_KEY_InsertChar                0x1000ff72  /* deprecated alias for hpInsertChar */
#define XKB_KEY_DeleteChar                0x1000ff73  /* deprecated alias for hpDeleteChar */
#define XKB_KEY_BackTab                   0x1000ff74  /* deprecated alias for hpBackTab */
#define XKB_KEY_KP_BackTab                0x1000ff75  /* deprecated alias for hpKP_BackTab */
#define XKB_KEY_Ext16bit_L                0x1000ff76  /* deprecated */
#define XKB_KEY_Ext16bit_R                0x1000ff77  /* deprecated */
#define XKB_KEY_mute_acute                0x100000a8  /* deprecated alias for hpmute_acute */
#define XKB_KEY_mute_grave                0x100000a9  /* deprecated alias for hpmute_grave */
#define XKB_KEY_mute_asciicircum          0x100000aa  /* deprecated alias for hpmute_asciicircum */
#define XKB_KEY_mute_diaeresis            0x100000ab  /* deprecated alias for hpmute_diaeresis */
#define XKB_KEY_mute_asciitilde           0x100000ac  /* deprecated alias for hpmute_asciitilde */
#define XKB_KEY_lira                      0x100000af  /* deprecated alias for hplira */
#define XKB_KEY_guilder                   0x100000be  /* deprecated alias for hpguilder */
#define XKB_KEY_IO                        0x100000ee  /* deprecated alias for hpYdiaeresis */
#define XKB_KEY_longminus                 0x100000f6  /* deprecated alias for hplongminus */
#define XKB_KEY_block                     0x100000fc  /* deprecated alias for hpblock */



#endif


#ifdef __cplusplus
extern "C" {
#endif

/**
 * @file
 * Main libxkbcommon API.
 */

/**
 * @struct xkb_context
 * Opaque top level library context object.
 *
 * The context contains various general library data and state, like
 * logging level and include paths.
 *
 * Objects are created in a specific context, and multiple contexts may
 * coexist simultaneously.  Objects from different contexts are completely
 * separated and do not share any memory or state.
 */
struct xkb_context;

/**
 * @struct xkb_keymap
 * Opaque compiled keymap object.
 *
 * The keymap object holds all of the static keyboard information obtained
 * from compiling XKB files.
 *
 * A keymap is immutable after it is created (besides reference counts, etc.);
 * if you need to change it, you must create a new one.
 */
struct xkb_keymap;

/**
 * @struct xkb_state
 * Opaque keyboard state object.
 *
 * State objects contain the active state of a keyboard (or keyboards), such
 * as the currently effective layout and the active modifiers.  It acts as a
 * simple state machine, wherein key presses and releases are the input, and
 * key symbols (keysyms) are the output.
 */
struct xkb_state;

/**
 * A number used to represent a physical key on a keyboard.
 *
 * A standard PC-compatible keyboard might have 102 keys.  An appropriate
 * keymap would assign each of them a keycode, by which the user should
 * refer to the key throughout the library.
 *
 * Historically, the X11 protocol, and consequentially the XKB protocol,
 * assign only 8 bits for keycodes.  This limits the number of different
 * keys that can be used simultaneously in a single keymap to 256
 * (disregarding other limitations).  This library does not share this limit;
 * keycodes beyond 255 ('extended keycodes') are not treated specially.
 * Keymaps and applications which are compatible with X11 should not use
 * these keycodes.
 *
 * The values of specific keycodes are determined by the keymap and the
 * underlying input system.  For example, with an X11-compatible keymap
 * and Linux evdev scan codes (see linux/input.h), a fixed offset is used:
 *
 * The keymap defines a canonical name for each key, plus possible aliases.
 * Historically, the XKB protocol restricts these names to at most 4 (ASCII)
 * characters, but this library does not share this limit.
 *
 * @code
 * xkb_keycode_t keycode_A = KEY_A + 8;
 * @endcode
 *
 * @sa xkb_keycode_is_legal_ext() xkb_keycode_is_legal_x11()
 */
typedef uint32_t xkb_keycode_t;

/**
 * A number used to represent the symbols generated from a key on a keyboard.
 *
 * A key, represented by a keycode, may generate different symbols according
 * to keyboard state.  For example, on a QWERTY keyboard, pressing the key
 * labled \<A\> generates the symbol ‘a’.  If the Shift key is held, it
 * generates the symbol ‘A’.  If a different layout is used, say Greek,
 * it generates the symbol ‘α’.  And so on.
 *
 * Each such symbol is represented by a *keysym* (short for “key symbol”).
 * Note that keysyms are somewhat more general, in that they can also represent
 * some “function”, such as “Left” or “Right” for the arrow keys.  For more
 * information, see: Appendix A [“KEYSYM Encoding”][encoding] of the X Window
 * System Protocol.
 *
 * Specifically named keysyms can be found in the
 * xkbcommon/xkbcommon-keysyms.h header file.  Their name does not include
 * the `XKB_KEY_` prefix.
 *
 * Besides those, any Unicode/ISO&nbsp;10646 character in the range U+0100 to
 * U+10FFFF can be represented by a keysym value in the range 0x01000100 to
 * 0x0110FFFF.  The name of Unicode keysyms is `U<codepoint>`, e.g. `UA1B2`.
 *
 * The name of other unnamed keysyms is the hexadecimal representation of
 * their value, e.g. `0xabcd1234`.
 *
 * Keysym names are case-sensitive.
 *
 * @note **Encoding:** Keysyms are 32-bit integers with the 3 most significant
 * bits always set to zero.  See: Appendix A [“KEYSYM Encoding”][encoding] of
 * the X Window System Protocol.
 *
 * [encoding]: https://www.x.org/releases/current/doc/xproto/x11protocol.html#keysym_encoding
 *
 * @ingroup keysyms
 * @sa XKB_KEYSYM_MAX
 */
typedef uint32_t xkb_keysym_t;

/**
 * Index of a keyboard layout.
 *
 * The layout index is a state component which detemines which <em>keyboard
 * layout</em> is active.  These may be different alphabets, different key
 * arrangements, etc.
 *
 * Layout indices are consecutive.  The first layout has index 0.
 *
 * Each layout is not required to have a name, and the names are not
 * guaranteed to be unique (though they are usually provided and unique).
 * Therefore, it is not safe to use the name as a unique identifier for a
 * layout.  Layout names are case-sensitive.
 *
 * Layout names are specified in the layout's definition, for example
 * "English (US)".  These are different from the (conventionally) short names
 * which are used to locate the layout, for example "us" or "us(intl)".  These
 * names are not present in a compiled keymap.
 *
 * If the user selects layouts from a list generated from the XKB registry
 * (using libxkbregistry or directly), and this metadata is needed later on, it
 * is recommended to store it along with the keymap.
 *
 * Layouts are also called "groups" by XKB.
 *
 * @sa xkb_keymap_num_layouts() xkb_keymap_num_layouts_for_key()
 */
typedef uint32_t xkb_layout_index_t;
/** A mask of layout indices. */
typedef uint32_t xkb_layout_mask_t;

/**
 * Index of a shift level.
 *
 * Any key, in any layout, can have several <em>shift levels</em>.  Each
 * shift level can assign different keysyms to the key.  The shift level
 * to use is chosen according to the current keyboard state; for example,
 * if no keys are pressed, the first level may be used; if the Left Shift
 * key is pressed, the second; if Num Lock is pressed, the third; and
 * many such combinations are possible (see xkb_mod_index_t).
 *
 * Level indices are consecutive.  The first level has index 0.
 */
typedef uint32_t xkb_level_index_t;

/**
 * Index of a modifier.
 *
 * A @e modifier is a state component which changes the way keys are
 * interpreted.  A keymap defines a set of modifiers, such as Alt, Shift,
 * Num Lock or Meta, and specifies which keys may @e activate which
 * modifiers (in a many-to-many relationship, i.e. a key can activate
 * several modifiers, and a modifier may be activated by several keys.
 * Different keymaps do this differently).
 *
 * When retrieving the keysyms for a key, the active modifier set is
 * consulted; this detemines the correct shift level to use within the
 * currently active layout (see xkb_level_index_t).
 *
 * Modifier indices are consecutive.  The first modifier has index 0.
 *
 * Each modifier must have a name, and the names are unique.  Therefore, it
 * is safe to use the name as a unique identifier for a modifier.  The names
 * of some common modifiers are provided in the xkbcommon/xkbcommon-names.h
 * header file.  Modifier names are case-sensitive.
 *
 * @sa xkb_keymap_num_mods()
 */
typedef uint32_t xkb_mod_index_t;
/** A mask of modifier indices. */
typedef uint32_t xkb_mod_mask_t;

/**
 * Index of a keyboard LED.
 *
 * LEDs are logical objects which may be @e active or @e inactive.  They
 * typically correspond to the lights on the keyboard. Their state is
 * determined by the current keyboard state.
 *
 * LED indices are non-consecutive.  The first LED has index 0.
 *
 * Each LED must have a name, and the names are unique. Therefore,
 * it is safe to use the name as a unique identifier for a LED.  The names
 * of some common LEDs are provided in the xkbcommon/xkbcommon-names.h
 * header file.  LED names are case-sensitive.
 *
 * @warning A given keymap may specify an exact index for a given LED.
 * Therefore, LED indexing is not necessarily sequential, as opposed to
 * modifiers and layouts.  This means that when iterating over the LEDs
 * in a keymap using e.g. xkb_keymap_num_leds(), some indices might be
 * invalid.  Given such an index, functions like xkb_keymap_led_get_name()
 * will return NULL, and xkb_state_led_index_is_active() will return -1.
 *
 * LEDs are also called "indicators" by XKB.
 *
 * @sa xkb_keymap_num_leds()
 */
typedef uint32_t xkb_led_index_t;
/** A mask of LED indices. */
typedef uint32_t xkb_led_mask_t;

#define XKB_KEYCODE_INVALID (0xffffffff)
#define XKB_LAYOUT_INVALID  (0xffffffff)
#define XKB_LEVEL_INVALID   (0xffffffff)
#define XKB_MOD_INVALID     (0xffffffff)
#define XKB_LED_INVALID     (0xffffffff)

#define XKB_KEYCODE_MAX     (0xffffffff - 1)

/**
 * Maximum keysym value
 *
 * @since 1.6.0
 * @sa xkb_keysym_t
 * @ingroup keysyms
 */
#define XKB_KEYSYM_MAX      0x1fffffff

/**
 * Test whether a value is a valid extended keycode.
 * @sa xkb_keycode_t
 **/
#define xkb_keycode_is_legal_ext(key) (key <= XKB_KEYCODE_MAX)

/**
 * Test whether a value is a valid X11 keycode.
 * @sa xkb_keycode_t
 */
#define xkb_keycode_is_legal_x11(key) (key >= 8 && key <= 255)

/**
 * Names to compile a keymap with, also known as RMLVO.
 *
 * The names are the common configuration values by which a user picks
 * a keymap.
 *
 * If the entire struct is NULL, then each field is taken to be NULL.
 * You should prefer passing NULL instead of choosing your own defaults.
 */
struct xkb_rule_names {
    /**
     * The rules file to use. The rules file describes how to interpret
     * the values of the model, layout, variant and options fields.
     *
     * If NULL or the empty string "", a default value is used.
     * If the XKB_DEFAULT_RULES environment variable is set, it is used
     * as the default.  Otherwise the system default is used.
     */
    const char *rules;
    /**
     * The keyboard model by which to interpret keycodes and LEDs.
     *
     * If NULL or the empty string "", a default value is used.
     * If the XKB_DEFAULT_MODEL environment variable is set, it is used
     * as the default.  Otherwise the system default is used.
     */
    const char *model;
    /**
     * A comma separated list of layouts (languages) to include in the
     * keymap.
     *
     * If NULL or the empty string "", a default value is used.
     * If the XKB_DEFAULT_LAYOUT environment variable is set, it is used
     * as the default.  Otherwise the system default is used.
     */
    const char *layout;
    /**
     * A comma separated list of variants, one per layout, which may
     * modify or augment the respective layout in various ways.
     *
     * Generally, should either be empty or have the same number of values
     * as the number of layouts. You may use empty values as in "intl,,neo".
     *
     * If NULL or the empty string "", and a default value is also used
     * for the layout, a default value is used.  Otherwise no variant is
     * used.
     * If the XKB_DEFAULT_VARIANT environment variable is set, it is used
     * as the default.  Otherwise the system default is used.
     */
    const char *variant;
    /**
     * A comma separated list of options, through which the user specifies
     * non-layout related preferences, like which key combinations are used
     * for switching layouts, or which key is the Compose key.
     *
     * If NULL, a default value is used.  If the empty string "", no
     * options are used.
     * If the XKB_DEFAULT_OPTIONS environment variable is set, it is used
     * as the default.  Otherwise the system default is used.
     */
    const char *options;
};

/**
 * @defgroup keysyms Keysyms
 * Utility functions related to *keysyms* (short for “key symbols”).
 *
 * @{
 */

/**
 * @page keysym-transformations Keysym Transformations
 *
 * Keysym translation is subject to several "keysym transformations",
 * as described in the XKB specification.  These are:
 *
 * - Capitalization transformation.  If the Caps Lock modifier is
 *   active and was not consumed by the translation process, a single
 *   keysym is transformed to its upper-case form (if applicable).
 *   Similarly, the UTF-8/UTF-32 string produced is capitalized.
 *
 *   This is described in:
 *   https://www.x.org/releases/current/doc/kbproto/xkbproto.html#Interpreting_the_Lock_Modifier
 *
 * - Control transformation.  If the Control modifier is active and
 *   was not consumed by the translation process, the string produced
 *   is transformed to its matching ASCII control character (if
 *   applicable).  Keysyms are not affected.
 *
 *   This is described in:
 *   https://www.x.org/releases/current/doc/kbproto/xkbproto.html#Interpreting_the_Control_Modifier
 *
 * Each relevant function discusses which transformations it performs.
 *
 * These transformations are not applicable when a key produces multiple
 * keysyms.
 */


/**
 * Get the name of a keysym.
 *
 * For a description of how keysyms are named, see @ref xkb_keysym_t.
 *
 * @param[in]  keysym The keysym.
 * @param[out] buffer A string buffer to write the name into.
 * @param[in]  size   Size of the buffer.
 *
 * @warning If the buffer passed is too small, the string is truncated
 * (though still NUL-terminated); a size of at least 64 bytes is recommended.
 *
 * @returns The number of bytes in the name, excluding the NUL byte. If
 * the keysym is invalid, returns -1.
 *
 * You may check if truncation has occurred by comparing the return value
 * with the length of buffer, similarly to the snprintf(3) function.
 *
 * @sa xkb_keysym_t
 */
int
xkb_keysym_get_name(xkb_keysym_t keysym, char *buffer, size_t size);

/** Flags for xkb_keysym_from_name(). */
enum xkb_keysym_flags {
    /** Do not apply any flags. */
    XKB_KEYSYM_NO_FLAGS = 0,
    /** Find keysym by case-insensitive search. */
    XKB_KEYSYM_CASE_INSENSITIVE = (1 << 0)
};

/**
 * Get a keysym from its name.
 *
 * @param name The name of a keysym. See remarks in xkb_keysym_get_name();
 * this function will accept any name returned by that function.
 * @param flags A set of flags controlling how the search is done. If
 * invalid flags are passed, this will fail with XKB_KEY_NoSymbol.
 *
 * If you use the XKB_KEYSYM_CASE_INSENSITIVE flag and two keysym names
 * differ only by case, then the lower-case keysym name is returned.  For
 * instance, for KEY_a and KEY_A, this function would return KEY_a for the
 * case-insensitive search.  If this functionality is needed, it is
 * recommended to first call this function without this flag; and if that
 * fails, only then to try with this flag, while possibly warning the user
 * he had misspelled the name, and might get wrong results.
 *
 * Case folding is done according to the C locale; the current locale is not
 * consulted.
 *
 * @returns The keysym. If the name is invalid, returns XKB_KEY_NoSymbol.
 *
 * @sa xkb_keysym_t
 */
xkb_keysym_t
xkb_keysym_from_name(const char *name, enum xkb_keysym_flags flags);

/**
 * Get the Unicode/UTF-8 representation of a keysym.
 *
 * @param[in]  keysym The keysym.
 * @param[out] buffer A buffer to write the UTF-8 string into.
 * @param[in]  size   The size of buffer.  Must be at least 7.
 *
 * @returns The number of bytes written to the buffer (including the
 * terminating byte).  If the keysym does not have a Unicode
 * representation, returns 0.  If the buffer is too small, returns -1.
 *
 * This function does not perform any @ref keysym-transformations.
 * Therefore, prefer to use xkb_state_key_get_utf8() if possible.
 *
 * @sa xkb_state_key_get_utf8()
 */
int
xkb_keysym_to_utf8(xkb_keysym_t keysym, char *buffer, size_t size);

/**
 * Get the Unicode/UTF-32 representation of a keysym.
 *
 * @returns The Unicode/UTF-32 representation of keysym, which is also
 * compatible with UCS-4.  If the keysym does not have a Unicode
 * representation, returns 0.
 *
 * This function does not perform any @ref keysym-transformations.
 * Therefore, prefer to use xkb_state_key_get_utf32() if possible.
 *
 * @sa xkb_state_key_get_utf32()
 */
uint32_t
xkb_keysym_to_utf32(xkb_keysym_t keysym);

/**
 * Get the keysym corresponding to a Unicode/UTF-32 codepoint.
 *
 * @returns The keysym corresponding to the specified Unicode
 * codepoint, or XKB_KEY_NoSymbol if there is none.
 *
 * This function is the inverse of @ref xkb_keysym_to_utf32. In cases
 * where a single codepoint corresponds to multiple keysyms, returns
 * the keysym with the lowest value.
 *
 * Unicode codepoints which do not have a special (legacy) keysym
 * encoding use a direct encoding scheme. These keysyms don't usually
 * have an associated keysym constant (XKB_KEY_*).
 *
 * For noncharacter Unicode codepoints and codepoints outside of the
 * defined Unicode planes this function returns XKB_KEY_NoSymbol.
 *
 * @sa xkb_keysym_to_utf32()
 * @since 1.0.0
 */
xkb_keysym_t
xkb_utf32_to_keysym(uint32_t ucs);

/**
 * Convert a keysym to its uppercase form.
 *
 * If there is no such form, the keysym is returned unchanged.
 *
 * The conversion rules may be incomplete; prefer to work with the Unicode
 * representation instead, when possible.
 */
xkb_keysym_t
xkb_keysym_to_upper(xkb_keysym_t ks);

/**
 * Convert a keysym to its lowercase form.
 *
 * The conversion rules may be incomplete; prefer to work with the Unicode
 * representation instead, when possible.
 */
xkb_keysym_t
xkb_keysym_to_lower(xkb_keysym_t ks);

/** @} */

/**
 * @defgroup context Library Context
 * Creating, destroying and using library contexts.
 *
 * Every keymap compilation request must have a context associated with
 * it.  The context keeps around state such as the include path.
 *
 * @{
 */

/**
 * @page envvars Environment Variables
 *
 * The user may set some environment variables which affect the library:
 *
 * - `XKB_CONFIG_ROOT`, `XKB_CONFIG_EXTRA_PATH`, `XDG_CONFIG_DIR`, `HOME` - see @ref include-path.
 * - `XKB_LOG_LEVEL` - see xkb_context_set_log_level().
 * - `XKB_LOG_VERBOSITY` - see xkb_context_set_log_verbosity().
 * - `XKB_DEFAULT_RULES`, `XKB_DEFAULT_MODEL`, `XKB_DEFAULT_LAYOUT`,
 *   `XKB_DEFAULT_VARIANT`, `XKB_DEFAULT_OPTIONS` - see xkb_rule_names.
 */

/** Flags for context creation. */
enum xkb_context_flags {
    /** Do not apply any context flags. */
    XKB_CONTEXT_NO_FLAGS = 0,
    /** Create this context with an empty include path. */
    XKB_CONTEXT_NO_DEFAULT_INCLUDES = (1 << 0),
    /**
     * Don't take RMLVO names from the environment.
     *
     * @since 0.3.0
     */
    XKB_CONTEXT_NO_ENVIRONMENT_NAMES = (1 << 1),
    /**
     * Disable the use of secure_getenv for this context, so that privileged
     * processes can use environment variables. Client uses at their own risk.
     *
     * @since 1.5.0
     */
    XKB_CONTEXT_NO_SECURE_GETENV = (1 << 2)
};

/**
 * Create a new context.
 *
 * @param flags Optional flags for the context, or 0.
 *
 * @returns A new context, or NULL on failure.
 *
 * @memberof xkb_context
 */
struct xkb_context *
xkb_context_new(enum xkb_context_flags flags);

/**
 * Take a new reference on a context.
 *
 * @returns The passed in context.
 *
 * @memberof xkb_context
 */
struct xkb_context *
xkb_context_ref(struct xkb_context *context);

/**
 * Release a reference on a context, and possibly free it.
 *
 * @param context The context.  If it is NULL, this function does nothing.
 *
 * @memberof xkb_context
 */
void
xkb_context_unref(struct xkb_context *context);

/**
 * Store custom user data in the context.
 *
 * This may be useful in conjunction with xkb_context_set_log_fn() or other
 * callbacks.
 *
 * @memberof xkb_context
 */
void
xkb_context_set_user_data(struct xkb_context *context, void *user_data);

/**
 * Retrieves stored user data from the context.
 *
 * @returns The stored user data.  If the user data wasn't set, or the
 * passed in context is NULL, returns NULL.
 *
 * This may be useful to access private user data from callbacks like a
 * custom logging function.
 *
 * @memberof xkb_context
 **/
void *
xkb_context_get_user_data(struct xkb_context *context);

/** @} */

/**
 * @defgroup include-path Include Paths
 * Manipulating the include paths in a context.
 *
 * The include paths are the file-system paths that are searched when an
 * include statement is encountered during keymap compilation.
 *
 * The default include paths are, in that lookup order:
 * - The path `$XDG_CONFIG_HOME/xkb`, with the usual `XDG_CONFIG_HOME`
 *   fallback to `$HOME/.config/` if unset.
 * - The path `$HOME/.xkb`, where $HOME is the value of the environment
 *   variable `HOME`.
 * - The `XKB_CONFIG_EXTRA_PATH` environment variable, if defined, otherwise the
 *   system configuration directory, defined at library configuration time
 *   (usually `/etc/xkb`).
 * - The `XKB_CONFIG_ROOT` environment variable, if defined, otherwise
 *   the system XKB root, defined at library configuration time.
 *
 * @{
 */

/**
 * Append a new entry to the context's include path.
 *
 * @returns 1 on success, or 0 if the include path could not be added or is
 * inaccessible.
 *
 * @memberof xkb_context
 */
int
xkb_context_include_path_append(struct xkb_context *context, const char *path);

/**
 * Append the default include paths to the context's include path.
 *
 * @returns 1 on success, or 0 if the primary include path could not be added.
 *
 * @memberof xkb_context
 */
int
xkb_context_include_path_append_default(struct xkb_context *context);

/**
 * Reset the context's include path to the default.
 *
 * Removes all entries from the context's include path, and inserts the
 * default paths.
 *
 * @returns 1 on success, or 0 if the primary include path could not be added.
 *
 * @memberof xkb_context
 */
int
xkb_context_include_path_reset_defaults(struct xkb_context *context);

/**
 * Remove all entries from the context's include path.
 *
 * @memberof xkb_context
 */
void
xkb_context_include_path_clear(struct xkb_context *context);

/**
 * Get the number of paths in the context's include path.
 *
 * @memberof xkb_context
 */
unsigned int
xkb_context_num_include_paths(struct xkb_context *context);

/**
 * Get a specific include path from the context's include path.
 *
 * @returns The include path at the specified index.  If the index is
 * invalid, returns NULL.
 *
 * @memberof xkb_context
 */
const char *
xkb_context_include_path_get(struct xkb_context *context, unsigned int index);

/** @} */

/**
 * @defgroup logging Logging Handling
 * Manipulating how logging from this library is handled.
 *
 * @{
 */

/** Specifies a logging level. */
enum xkb_log_level {
    XKB_LOG_LEVEL_CRITICAL = 10, /**< Log critical internal errors only. */
    XKB_LOG_LEVEL_ERROR = 20,    /**< Log all errors. */
    XKB_LOG_LEVEL_WARNING = 30,  /**< Log warnings and errors. */
    XKB_LOG_LEVEL_INFO = 40,     /**< Log information, warnings, and errors. */
    XKB_LOG_LEVEL_DEBUG = 50     /**< Log everything. */
};

/**
 * Set the current logging level.
 *
 * @param context The context in which to set the logging level.
 * @param level   The logging level to use.  Only messages from this level
 * and below will be logged.
 *
 * The default level is XKB_LOG_LEVEL_ERROR.  The environment variable
 * XKB_LOG_LEVEL, if set in the time the context was created, overrides the
 * default value.  It may be specified as a level number or name.
 *
 * @memberof xkb_context
 */
void
xkb_context_set_log_level(struct xkb_context *context,
                          enum xkb_log_level level);

/**
 * Get the current logging level.
 *
 * @memberof xkb_context
 */
enum xkb_log_level
xkb_context_get_log_level(struct xkb_context *context);

/**
 * Sets the current logging verbosity.
 *
 * The library can generate a number of warnings which are not helpful to
 * ordinary users of the library.  The verbosity may be increased if more
 * information is desired (e.g. when developing a new keymap).
 *
 * The default verbosity is 0.  The environment variable XKB_LOG_VERBOSITY,
 * if set in the time the context was created, overrides the default value.
 *
 * @param context   The context in which to use the set verbosity.
 * @param verbosity The verbosity to use.  Currently used values are
 * 1 to 10, higher values being more verbose.  0 would result in no verbose
 * messages being logged.
 *
 * Most verbose messages are of level XKB_LOG_LEVEL_WARNING or lower.
 *
 * @memberof xkb_context
 */
void
xkb_context_set_log_verbosity(struct xkb_context *context, int verbosity);

/**
 * Get the current logging verbosity of the context.
 *
 * @memberof xkb_context
 */
int
xkb_context_get_log_verbosity(struct xkb_context *context);

/**
 * Set a custom function to handle logging messages.
 *
 * @param context The context in which to use the set logging function.
 * @param log_fn  The function that will be called for logging messages.
 * Passing NULL restores the default function, which logs to stderr.
 *
 * By default, log messages from this library are printed to stderr.  This
 * function allows you to replace the default behavior with a custom
 * handler.  The handler is only called with messages which match the
 * current logging level and verbosity settings for the context.
 * level is the logging level of the message.  @a format and @a args are
 * the same as in the vprintf(3) function.
 *
 * You may use xkb_context_set_user_data() on the context, and then call
 * xkb_context_get_user_data() from within the logging function to provide
 * it with additional private context.
 *
 * @memberof xkb_context
 */
void
xkb_context_set_log_fn(struct xkb_context *context,
                       void (*log_fn)(struct xkb_context *context,
                                      enum xkb_log_level level,
                                      const char *format, va_list args));

/** @} */

/**
 * @defgroup keymap Keymap Creation
 * Creating and destroying keymaps.
 *
 * @{
 */

/** Flags for keymap compilation. */
enum xkb_keymap_compile_flags {
    /** Do not apply any flags. */
    XKB_KEYMAP_COMPILE_NO_FLAGS = 0
};

/**
 * Create a keymap from RMLVO names.
 *
 * The primary keymap entry point: creates a new XKB keymap from a set of
 * RMLVO (Rules + Model + Layouts + Variants + Options) names.
 *
 * @param context The context in which to create the keymap.
 * @param names   The RMLVO names to use.  See xkb_rule_names.
 * @param flags   Optional flags for the keymap, or 0.
 *
 * @returns A keymap compiled according to the RMLVO names, or NULL if
 * the compilation failed.
 *
 * @sa xkb_rule_names
 * @memberof xkb_keymap
 */
struct xkb_keymap *
xkb_keymap_new_from_names(struct xkb_context *context,
                          const struct xkb_rule_names *names,
                          enum xkb_keymap_compile_flags flags);

/** The possible keymap formats. */
enum xkb_keymap_format {
    /** The current/classic XKB text format, as generated by xkbcomp -xkb. */
    XKB_KEYMAP_FORMAT_TEXT_V1 = 1
};

/**
 * Create a keymap from a keymap file.
 *
 * @param context The context in which to create the keymap.
 * @param file    The keymap file to compile.
 * @param format  The text format of the keymap file to compile.
 * @param flags   Optional flags for the keymap, or 0.
 *
 * @returns A keymap compiled from the given XKB keymap file, or NULL if
 * the compilation failed.
 *
 * The file must contain a complete keymap.  For example, in the
 * XKB_KEYMAP_FORMAT_TEXT_V1 format, this means the file must contain one
 * top level '%xkb_keymap' section, which in turn contains other required
 * sections.
 *
 * @memberof xkb_keymap
 */
struct xkb_keymap *
xkb_keymap_new_from_file(struct xkb_context *context, FILE *file,
                         enum xkb_keymap_format format,
                         enum xkb_keymap_compile_flags flags);

/**
 * Create a keymap from a keymap string.
 *
 * This is just like xkb_keymap_new_from_file(), but instead of a file, gets
 * the keymap as one enormous string.
 *
 * @see xkb_keymap_new_from_file()
 * @memberof xkb_keymap
 */
struct xkb_keymap *
xkb_keymap_new_from_string(struct xkb_context *context, const char *string,
                           enum xkb_keymap_format format,
                           enum xkb_keymap_compile_flags flags);

/**
 * Create a keymap from a memory buffer.
 *
 * This is just like xkb_keymap_new_from_string(), but takes a length argument
 * so the input string does not have to be zero-terminated.
 *
 * @see xkb_keymap_new_from_string()
 * @memberof xkb_keymap
 * @since 0.3.0
 */
struct xkb_keymap *
xkb_keymap_new_from_buffer(struct xkb_context *context, const char *buffer,
                           size_t length, enum xkb_keymap_format format,
                           enum xkb_keymap_compile_flags flags);

/**
 * Take a new reference on a keymap.
 *
 * @returns The passed in keymap.
 *
 * @memberof xkb_keymap
 */
struct xkb_keymap *
xkb_keymap_ref(struct xkb_keymap *keymap);

/**
 * Release a reference on a keymap, and possibly free it.
 *
 * @param keymap The keymap.  If it is NULL, this function does nothing.
 *
 * @memberof xkb_keymap
 */
void
xkb_keymap_unref(struct xkb_keymap *keymap);

/**
 * Get the keymap as a string in the format from which it was created.
 * @sa xkb_keymap_get_as_string()
 **/
#define XKB_KEYMAP_USE_ORIGINAL_FORMAT ((enum xkb_keymap_format) -1)

/**
 * Get the compiled keymap as a string.
 *
 * @param keymap The keymap to get as a string.
 * @param format The keymap format to use for the string.  You can pass
 * in the special value XKB_KEYMAP_USE_ORIGINAL_FORMAT to use the format
 * from which the keymap was originally created.
 *
 * @returns The keymap as a NUL-terminated string, or NULL if unsuccessful.
 *
 * The returned string may be fed back into xkb_keymap_new_from_string() to get
 * the exact same keymap (possibly in another process, etc.).
 *
 * The returned string is dynamically allocated and should be freed by the
 * caller.
 *
 * @memberof xkb_keymap
 */
char *
xkb_keymap_get_as_string(struct xkb_keymap *keymap,
                         enum xkb_keymap_format format);

/** @} */

/**
 * @defgroup components Keymap Components
 * Enumeration of state components in a keymap.
 *
 * @{
 */

/**
 * Get the minimum keycode in the keymap.
 *
 * @sa xkb_keycode_t
 * @memberof xkb_keymap
 * @since 0.3.1
 */
xkb_keycode_t
xkb_keymap_min_keycode(struct xkb_keymap *keymap);

/**
 * Get the maximum keycode in the keymap.
 *
 * @sa xkb_keycode_t
 * @memberof xkb_keymap
 * @since 0.3.1
 */
xkb_keycode_t
xkb_keymap_max_keycode(struct xkb_keymap *keymap);

/**
 * The iterator used by xkb_keymap_key_for_each().
 *
 * @sa xkb_keymap_key_for_each
 * @memberof xkb_keymap
 * @since 0.3.1
 */
typedef void
(*xkb_keymap_key_iter_t)(struct xkb_keymap *keymap, xkb_keycode_t key,
                         void *data);

/**
 * Run a specified function for every valid keycode in the keymap.  If a
 * keymap is sparse, this function may be called fewer than
 * (max_keycode - min_keycode + 1) times.
 *
 * @sa xkb_keymap_min_keycode() xkb_keymap_max_keycode() xkb_keycode_t
 * @memberof xkb_keymap
 * @since 0.3.1
 */
void
xkb_keymap_key_for_each(struct xkb_keymap *keymap, xkb_keymap_key_iter_t iter,
                        void *data);

/**
 * Find the name of the key with the given keycode.
 *
 * This function always returns the canonical name of the key (see
 * description in xkb_keycode_t).
 *
 * @returns The key name. If no key with this keycode exists,
 * returns NULL.
 *
 * @sa xkb_keycode_t
 * @memberof xkb_keymap
 * @since 0.6.0
 */
const char *
xkb_keymap_key_get_name(struct xkb_keymap *keymap, xkb_keycode_t key);

/**
 * Find the keycode of the key with the given name.
 *
 * The name can be either a canonical name or an alias.
 *
 * @returns The keycode. If no key with this name exists,
 * returns XKB_KEYCODE_INVALID.
 *
 * @sa xkb_keycode_t
 * @memberof xkb_keymap
 * @since 0.6.0
 */
xkb_keycode_t
xkb_keymap_key_by_name(struct xkb_keymap *keymap, const char *name);

/**
 * Get the number of modifiers in the keymap.
 *
 * @sa xkb_mod_index_t
 * @memberof xkb_keymap
 */
xkb_mod_index_t
xkb_keymap_num_mods(struct xkb_keymap *keymap);

/**
 * Get the name of a modifier by index.
 *
 * @returns The name.  If the index is invalid, returns NULL.
 *
 * @sa xkb_mod_index_t
 * @memberof xkb_keymap
 */
const char *
xkb_keymap_mod_get_name(struct xkb_keymap *keymap, xkb_mod_index_t idx);

/**
 * Get the index of a modifier by name.
 *
 * @returns The index.  If no modifier with this name exists, returns
 * XKB_MOD_INVALID.
 *
 * @sa xkb_mod_index_t
 * @memberof xkb_keymap
 */
xkb_mod_index_t
xkb_keymap_mod_get_index(struct xkb_keymap *keymap, const char *name);

/**
 * Get the number of layouts in the keymap.
 *
 * @sa xkb_layout_index_t xkb_rule_names xkb_keymap_num_layouts_for_key()
 * @memberof xkb_keymap
 */
xkb_layout_index_t
xkb_keymap_num_layouts(struct xkb_keymap *keymap);

/**
 * Get the name of a layout by index.
 *
 * @returns The name.  If the index is invalid, or the layout does not have
 * a name, returns NULL.
 *
 * @sa xkb_layout_index_t
 *     For notes on layout names.
 * @memberof xkb_keymap
 */
const char *
xkb_keymap_layout_get_name(struct xkb_keymap *keymap, xkb_layout_index_t idx);

/**
 * Get the index of a layout by name.
 *
 * @returns The index.  If no layout exists with this name, returns
 * XKB_LAYOUT_INVALID.  If more than one layout in the keymap has this name,
 * returns the lowest index among them.
 *
 * @sa xkb_layout_index_t
 *     For notes on layout names.
 * @memberof xkb_keymap
 */
xkb_layout_index_t
xkb_keymap_layout_get_index(struct xkb_keymap *keymap, const char *name);

/**
 * Get the number of LEDs in the keymap.
 *
 * @warning The range [ 0...xkb_keymap_num_leds() ) includes all of the LEDs
 * in the keymap, but may also contain inactive LEDs.  When iterating over
 * this range, you need the handle this case when calling functions such as
 * xkb_keymap_led_get_name() or xkb_state_led_index_is_active().
 *
 * @sa xkb_led_index_t
 * @memberof xkb_keymap
 */
xkb_led_index_t
xkb_keymap_num_leds(struct xkb_keymap *keymap);

/**
 * Get the name of a LED by index.
 *
 * @returns The name.  If the index is invalid, returns NULL.
 *
 * @memberof xkb_keymap
 */
const char *
xkb_keymap_led_get_name(struct xkb_keymap *keymap, xkb_led_index_t idx);

/**
 * Get the index of a LED by name.
 *
 * @returns The index.  If no LED with this name exists, returns
 * XKB_LED_INVALID.
 *
 * @memberof xkb_keymap
 */
xkb_led_index_t
xkb_keymap_led_get_index(struct xkb_keymap *keymap, const char *name);

/**
 * Get the number of layouts for a specific key.
 *
 * This number can be different from xkb_keymap_num_layouts(), but is always
 * smaller.  It is the appropriate value to use when iterating over the
 * layouts of a key.
 *
 * @sa xkb_layout_index_t
 * @memberof xkb_keymap
 */
xkb_layout_index_t
xkb_keymap_num_layouts_for_key(struct xkb_keymap *keymap, xkb_keycode_t key);

/**
 * Get the number of shift levels for a specific key and layout.
 *
 * If @c layout is out of range for this key (that is, larger or equal to
 * the value returned by xkb_keymap_num_layouts_for_key()), it is brought
 * back into range in a manner consistent with xkb_state_key_get_layout().
 *
 * @sa xkb_level_index_t
 * @memberof xkb_keymap
 */
xkb_level_index_t
xkb_keymap_num_levels_for_key(struct xkb_keymap *keymap, xkb_keycode_t key,
                              xkb_layout_index_t layout);

/**
 * Retrieves every possible modifier mask that produces the specified
 * shift level for a specific key and layout.
 *
 * This API is useful for inverse key transformation; i.e. finding out
 * which modifiers need to be active in order to be able to type the
 * keysym(s) corresponding to the specific key code, layout and level.
 *
 * @warning It returns only up to masks_size modifier masks. If the
 * buffer passed is too small, some of the possible modifier combinations
 * will not be returned.
 *
 * @param[in] keymap      The keymap.
 * @param[in] key         The keycode of the key.
 * @param[in] layout      The layout for which to get modifiers.
 * @param[in] level       The shift level in the layout for which to get the
 * modifiers. This should be smaller than:
 * @code xkb_keymap_num_levels_for_key(keymap, key) @endcode
 * @param[out] masks_out  A buffer in which the requested masks should be
 * stored.
 * @param[out] masks_size The number of elements in the buffer pointed to by
 * masks_out.
 *
 * If @c layout is out of range for this key (that is, larger or equal to
 * the value returned by xkb_keymap_num_layouts_for_key()), it is brought
 * back into range in a manner consistent with xkb_state_key_get_layout().
 *
 * @returns The number of modifier masks stored in the masks_out array.
 * If the key is not in the keymap or if the specified shift level cannot
 * be reached it returns 0 and does not modify the masks_out buffer.
 *
 * @sa xkb_level_index_t
 * @sa xkb_mod_mask_t
 * @memberof xkb_keymap
 * @since 1.0.0
 */
size_t
xkb_keymap_key_get_mods_for_level(struct xkb_keymap *keymap,
                                  xkb_keycode_t key,
                                  xkb_layout_index_t layout,
                                  xkb_level_index_t level,
                                  xkb_mod_mask_t *masks_out,
                                  size_t masks_size);

/**
 * Get the keysyms obtained from pressing a key in a given layout and
 * shift level.
 *
 * This function is like xkb_state_key_get_syms(), only the layout and
 * shift level are not derived from the keyboard state but are instead
 * specified explicitly.
 *
 * @param[in] keymap    The keymap.
 * @param[in] key       The keycode of the key.
 * @param[in] layout    The layout for which to get the keysyms.
 * @param[in] level     The shift level in the layout for which to get the
 * keysyms. This should be smaller than:
 * @code xkb_keymap_num_levels_for_key(keymap, key) @endcode
 * @param[out] syms_out An immutable array of keysyms corresponding to the
 * key in the given layout and shift level.
 *
 * If @c layout is out of range for this key (that is, larger or equal to
 * the value returned by xkb_keymap_num_layouts_for_key()), it is brought
 * back into range in a manner consistent with xkb_state_key_get_layout().
 *
 * @returns The number of keysyms in the syms_out array.  If no keysyms
 * are produced by the key in the given layout and shift level, returns 0
 * and sets syms_out to NULL.
 *
 * @sa xkb_state_key_get_syms()
 * @memberof xkb_keymap
 */
int
xkb_keymap_key_get_syms_by_level(struct xkb_keymap *keymap,
                                 xkb_keycode_t key,
                                 xkb_layout_index_t layout,
                                 xkb_level_index_t level,
                                 const xkb_keysym_t **syms_out);

/**
 * Determine whether a key should repeat or not.
 *
 * A keymap may specify different repeat behaviors for different keys.
 * Most keys should generally exhibit repeat behavior; for example, holding
 * the 'a' key down in a text editor should normally insert a single 'a'
 * character every few milliseconds, until the key is released.  However,
 * there are keys which should not or do not need to be repeated.  For
 * example, repeating modifier keys such as Left/Right Shift or Caps Lock
 * is not generally useful or desired.
 *
 * @returns 1 if the key should repeat, 0 otherwise.
 *
 * @memberof xkb_keymap
 */
int
xkb_keymap_key_repeats(struct xkb_keymap *keymap, xkb_keycode_t key);

/** @} */

/**
 * @defgroup state Keyboard State
 * Creating, destroying and manipulating keyboard state objects.
 *
 * @{
 */

/**
 * Create a new keyboard state object.
 *
 * @param keymap The keymap which the state will use.
 *
 * @returns A new keyboard state object, or NULL on failure.
 *
 * @memberof xkb_state
 */
struct xkb_state *
xkb_state_new(struct xkb_keymap *keymap);

/**
 * Take a new reference on a keyboard state object.
 *
 * @returns The passed in object.
 *
 * @memberof xkb_state
 */
struct xkb_state *
xkb_state_ref(struct xkb_state *state);

/**
 * Release a reference on a keybaord state object, and possibly free it.
 *
 * @param state The state.  If it is NULL, this function does nothing.
 *
 * @memberof xkb_state
 */
void
xkb_state_unref(struct xkb_state *state);

/**
 * Get the keymap which a keyboard state object is using.
 *
 * @returns The keymap which was passed to xkb_state_new() when creating
 * this state object.
 *
 * This function does not take a new reference on the keymap; you must
 * explicitly reference it yourself if you plan to use it beyond the
 * lifetime of the state.
 *
 * @memberof xkb_state
 */
struct xkb_keymap *
xkb_state_get_keymap(struct xkb_state *state);

/**
 * @page server-client-state Server State and Client State
 * @parblock
 *
 * The xkb_state API is used by two distinct actors in most window-system
 * architectures:
 *
 * 1. A *server* - for example, a Wayland compositor, an X11 server, an evdev
 *    listener.
 *
 *    Servers maintain the XKB state for a device according to input events from
 *    the device, such as key presses and releases, and out-of-band events from
 *    the user, like UI layout switchers.
 *
 * 2. A *client* - for example, a Wayland client, an X11 client.
 *
 *    Clients do not listen to input from the device; instead, whenever the
 *    server state changes, the server serializes the state and notifies the
 *    clients that the state has changed; the clients then update the state
 *    from the serialization.
 *
 * Some entry points in the xkb_state API are only meant for servers and some
 * are only meant for clients, and the two should generally not be mixed.
 *
 * @endparblock
 */

/** Specifies the direction of the key (press / release). */
enum xkb_key_direction {
    XKB_KEY_UP,   /**< The key was released. */
    XKB_KEY_DOWN  /**< The key was pressed. */
};

/**
 * Modifier and layout types for state objects.  This enum is bitmaskable,
 * e.g. (XKB_STATE_MODS_DEPRESSED | XKB_STATE_MODS_LATCHED) is valid to
 * exclude locked modifiers.
 *
 * In XKB, the DEPRESSED components are also known as 'base'.
 */
enum xkb_state_component {
    /** Depressed modifiers, i.e. a key is physically holding them. */
    XKB_STATE_MODS_DEPRESSED = (1 << 0),
    /** Latched modifiers, i.e. will be unset after the next non-modifier
     *  key press. */
    XKB_STATE_MODS_LATCHED = (1 << 1),
    /** Locked modifiers, i.e. will be unset after the key provoking the
     *  lock has been pressed again. */
    XKB_STATE_MODS_LOCKED = (1 << 2),
    /** Effective modifiers, i.e. currently active and affect key
     *  processing (derived from the other state components).
     *  Use this unless you explicitly care how the state came about. */
    XKB_STATE_MODS_EFFECTIVE = (1 << 3),
    /** Depressed layout, i.e. a key is physically holding it. */
    XKB_STATE_LAYOUT_DEPRESSED = (1 << 4),
    /** Latched layout, i.e. will be unset after the next non-modifier
     *  key press. */
    XKB_STATE_LAYOUT_LATCHED = (1 << 5),
    /** Locked layout, i.e. will be unset after the key provoking the lock
     *  has been pressed again. */
    XKB_STATE_LAYOUT_LOCKED = (1 << 6),
    /** Effective layout, i.e. currently active and affects key processing
     *  (derived from the other state components).
     *  Use this unless you explicitly care how the state came about. */
    XKB_STATE_LAYOUT_EFFECTIVE = (1 << 7),
    /** LEDs (derived from the other state components). */
    XKB_STATE_LEDS = (1 << 8)
};

/**
 * Update the keyboard state to reflect a given key being pressed or
 * released.
 *
 * This entry point is intended for *server* applications and should not be used
 * by *client* applications; see @ref server-client-state for details.
 *
 * A series of calls to this function should be consistent; that is, a call
 * with XKB_KEY_DOWN for a key should be matched by an XKB_KEY_UP; if a key
 * is pressed twice, it should be released twice; etc. Otherwise (e.g. due
 * to missed input events), situations like "stuck modifiers" may occur.
 *
 * This function is often used in conjunction with the function
 * xkb_state_key_get_syms() (or xkb_state_key_get_one_sym()), for example,
 * when handling a key event.  In this case, you should prefer to get the
 * keysyms *before* updating the key, such that the keysyms reported for
 * the key event are not affected by the event itself.  This is the
 * conventional behavior.
 *
 * @returns A mask of state components that have changed as a result of
 * the update.  If nothing in the state has changed, returns 0.
 *
 * @memberof xkb_state
 *
 * @sa xkb_state_update_mask()
 */
enum xkb_state_component
xkb_state_update_key(struct xkb_state *state, xkb_keycode_t key,
                     enum xkb_key_direction direction);

/**
 * Update a keyboard state from a set of explicit masks.
 *
 * This entry point is intended for *client* applications; see @ref
 * server-client-state for details. *Server* applications should use
 * xkb_state_update_key() instead.
 *
 * All parameters must always be passed, or the resulting state may be
 * incoherent.
 *
 * The serialization is lossy and will not survive round trips; it must only
 * be used to feed client state objects, and must not be used to update the
 * server state.
 *
 * @returns A mask of state components that have changed as a result of
 * the update.  If nothing in the state has changed, returns 0.
 *
 * @memberof xkb_state
 *
 * @sa xkb_state_component
 * @sa xkb_state_update_key
 */
enum xkb_state_component
xkb_state_update_mask(struct xkb_state *state,
                      xkb_mod_mask_t depressed_mods,
                      xkb_mod_mask_t latched_mods,
                      xkb_mod_mask_t locked_mods,
                      xkb_layout_index_t depressed_layout,
                      xkb_layout_index_t latched_layout,
                      xkb_layout_index_t locked_layout);

/**
 * Get the keysyms obtained from pressing a particular key in a given
 * keyboard state.
 *
 * Get the keysyms for a key according to the current active layout,
 * modifiers and shift level for the key, as determined by a keyboard
 * state.
 *
 * @param[in]  state    The keyboard state object.
 * @param[in]  key      The keycode of the key.
 * @param[out] syms_out An immutable array of keysyms corresponding the
 * key in the given keyboard state.
 *
 * As an extension to XKB, this function can return more than one keysym.
 * If you do not want to handle this case, you can use
 * xkb_state_key_get_one_sym() for a simpler interface.
 *
 * This function does not perform any @ref keysym-transformations.
 * (This might change).
 *
 * @returns The number of keysyms in the syms_out array.  If no keysyms
 * are produced by the key in the given keyboard state, returns 0 and sets
 * syms_out to NULL.
 *
 * @memberof xkb_state
 */
int
xkb_state_key_get_syms(struct xkb_state *state, xkb_keycode_t key,
                       const xkb_keysym_t **syms_out);

/**
 * Get the Unicode/UTF-8 string obtained from pressing a particular key
 * in a given keyboard state.
 *
 * @param[in]  state  The keyboard state object.
 * @param[in]  key    The keycode of the key.
 * @param[out] buffer A buffer to write the string into.
 * @param[in]  size   Size of the buffer.
 *
 * @warning If the buffer passed is too small, the string is truncated
 * (though still NUL-terminated).
 *
 * @returns The number of bytes required for the string, excluding the
 * NUL byte.  If there is nothing to write, returns 0.
 *
 * You may check if truncation has occurred by comparing the return value
 * with the size of @p buffer, similarly to the snprintf(3) function.
 * You may safely pass NULL and 0 to @p buffer and @p size to find the
 * required size (without the NUL-byte).
 *
 * This function performs Capitalization and Control @ref
 * keysym-transformations.
 *
 * @memberof xkb_state
 * @since 0.4.1
 */
int
xkb_state_key_get_utf8(struct xkb_state *state, xkb_keycode_t key,
                       char *buffer, size_t size);

/**
 * Get the Unicode/UTF-32 codepoint obtained from pressing a particular
 * key in a a given keyboard state.
 *
 * @returns The UTF-32 representation for the key, if it consists of only
 * a single codepoint.  Otherwise, returns 0.
 *
 * This function performs Capitalization and Control @ref
 * keysym-transformations.
 *
 * @memberof xkb_state
 * @since 0.4.1
 */
uint32_t
xkb_state_key_get_utf32(struct xkb_state *state, xkb_keycode_t key);

/**
 * Get the single keysym obtained from pressing a particular key in a
 * given keyboard state.
 *
 * This function is similar to xkb_state_key_get_syms(), but intended
 * for users which cannot or do not want to handle the case where
 * multiple keysyms are returned (in which case this function is
 * preferred).
 *
 * @returns The keysym.  If the key does not have exactly one keysym,
 * returns XKB_KEY_NoSymbol
 *
 * This function performs Capitalization @ref keysym-transformations.
 *
 * @sa xkb_state_key_get_syms()
 * @memberof xkb_state
 */
xkb_keysym_t
xkb_state_key_get_one_sym(struct xkb_state *state, xkb_keycode_t key);

/**
 * Get the effective layout index for a key in a given keyboard state.
 *
 * @returns The layout index for the key in the given keyboard state.  If
 * the given keycode is invalid, or if the key is not included in any
 * layout at all, returns XKB_LAYOUT_INVALID.
 *
 * @invariant If the returned layout is valid, the following always holds:
 * @code
 * xkb_state_key_get_layout(state, key) < xkb_keymap_num_layouts_for_key(keymap, key)
 * @endcode
 *
 * @memberof xkb_state
 */
xkb_layout_index_t
xkb_state_key_get_layout(struct xkb_state *state, xkb_keycode_t key);

/**
 * Get the effective shift level for a key in a given keyboard state and
 * layout.
 *
 * @param state The keyboard state.
 * @param key The keycode of the key.
 * @param layout The layout for which to get the shift level.  This must be
 * smaller than:
 * @code xkb_keymap_num_layouts_for_key(keymap, key) @endcode
 * usually it would be:
 * @code xkb_state_key_get_layout(state, key) @endcode
 *
 * @return The shift level index.  If the key or layout are invalid,
 * returns XKB_LEVEL_INVALID.
 *
 * @invariant If the returned level is valid, the following always holds:
 * @code
 * xkb_state_key_get_level(state, key, layout) < xkb_keymap_num_levels_for_key(keymap, key, layout)
 * @endcode
 *
 * @memberof xkb_state
 */
xkb_level_index_t
xkb_state_key_get_level(struct xkb_state *state, xkb_keycode_t key,
                        xkb_layout_index_t layout);

/**
 * Match flags for xkb_state_mod_indices_are_active() and
 * xkb_state_mod_names_are_active(), specifying the conditions for a
 * successful match.  XKB_STATE_MATCH_NON_EXCLUSIVE is bitmaskable with
 * the other modes.
 */
enum xkb_state_match {
    /** Returns true if any of the modifiers are active. */
    XKB_STATE_MATCH_ANY = (1 << 0),
    /** Returns true if all of the modifiers are active. */
    XKB_STATE_MATCH_ALL = (1 << 1),
    /** Makes matching non-exclusive, i.e. will not return false if a
     *  modifier not specified in the arguments is active. */
    XKB_STATE_MATCH_NON_EXCLUSIVE = (1 << 16)
};

/**
 * The counterpart to xkb_state_update_mask for modifiers, to be used on
 * the server side of serialization.
 *
 * This entry point is intended for *server* applications; see @ref
 * server-client-state for details. *Client* applications should use the
 * xkb_state_mod_*_is_active API.
 *
 * @param state      The keyboard state.
 * @param components A mask of the modifier state components to serialize.
 * State components other than XKB_STATE_MODS_* are ignored.
 * If XKB_STATE_MODS_EFFECTIVE is included, all other state components are
 * ignored.
 *
 * @returns A xkb_mod_mask_t representing the given components of the
 * modifier state.
 *
 * @memberof xkb_state
 */
xkb_mod_mask_t
xkb_state_serialize_mods(struct xkb_state *state,
                         enum xkb_state_component components);

/**
 * The counterpart to xkb_state_update_mask for layouts, to be used on
 * the server side of serialization.
 *
 * This entry point is intended for *server* applications; see @ref
 * server-client-state for details. *Client* applications should use the
 * xkb_state_layout_*_is_active API.
 *
 * @param state      The keyboard state.
 * @param components A mask of the layout state components to serialize.
 * State components other than XKB_STATE_LAYOUT_* are ignored.
 * If XKB_STATE_LAYOUT_EFFECTIVE is included, all other state components are
 * ignored.
 *
 * @returns A layout index representing the given components of the
 * layout state.
 *
 * @memberof xkb_state
 */
xkb_layout_index_t
xkb_state_serialize_layout(struct xkb_state *state,
                           enum xkb_state_component components);

/**
 * Test whether a modifier is active in a given keyboard state by name.
 *
 * @returns 1 if the modifier is active, 0 if it is not.  If the modifier
 * name does not exist in the keymap, returns -1.
 *
 * @memberof xkb_state
 */
int
xkb_state_mod_name_is_active(struct xkb_state *state, const char *name,
                             enum xkb_state_component type);

/**
 * Test whether a set of modifiers are active in a given keyboard state by
 * name.
 *
 * @param state The keyboard state.
 * @param type  The component of the state against which to match the
 * given modifiers.
 * @param match The manner by which to match the state against the
 * given modifiers.
 * @param ...   The set of of modifier names to test, terminated by a NULL
 * argument (sentinel).
 *
 * @returns 1 if the modifiers are active, 0 if they are not.  If any of
 * the modifier names do not exist in the keymap, returns -1.
 *
 * @memberof xkb_state
 */
int
xkb_state_mod_names_are_active(struct xkb_state *state,
                               enum xkb_state_component type,
                               enum xkb_state_match match,
                               ...);

/**
 * Test whether a modifier is active in a given keyboard state by index.
 *
 * @returns 1 if the modifier is active, 0 if it is not.  If the modifier
 * index is invalid in the keymap, returns -1.
 *
 * @memberof xkb_state
 */
int
xkb_state_mod_index_is_active(struct xkb_state *state, xkb_mod_index_t idx,
                              enum xkb_state_component type);

/**
 * Test whether a set of modifiers are active in a given keyboard state by
 * index.
 *
 * @param state The keyboard state.
 * @param type  The component of the state against which to match the
 * given modifiers.
 * @param match The manner by which to match the state against the
 * given modifiers.
 * @param ...   The set of of modifier indices to test, terminated by a
 * XKB_MOD_INVALID argument (sentinel).
 *
 * @returns 1 if the modifiers are active, 0 if they are not.  If any of
 * the modifier indices are invalid in the keymap, returns -1.
 *
 * @memberof xkb_state
 */
int
xkb_state_mod_indices_are_active(struct xkb_state *state,
                                 enum xkb_state_component type,
                                 enum xkb_state_match match,
                                 ...);

/**
 * @page consumed-modifiers Consumed Modifiers
 * @parblock
 *
 * Some functions, like xkb_state_key_get_syms(), look at the state of
 * the modifiers in the keymap and derive from it the correct shift level
 * to use for the key.  For example, in a US layout, pressing the key
 * labeled \<A\> while the Shift modifier is active, generates the keysym
 * 'A'.  In this case, the Shift modifier is said to be "consumed".
 * However, the Num Lock modifier does not affect this translation at all,
 * even if it is active, so it is not consumed by this translation.
 *
 * It may be desirable for some application to not reuse consumed modifiers
 * for further processing, e.g. for hotkeys or keyboard shortcuts.  To
 * understand why, consider some requirements from a standard shortcut
 * mechanism, and how they are implemented:
 *
 * 1. The shortcut's modifiers must match exactly to the state.  For
 *    example, it is possible to bind separate actions to \<Alt\>\<Tab\>
 *    and to \<Alt\>\<Shift\>\<Tab\>.  Further, if only \<Alt\>\<Tab\> is
 *    bound to an action, pressing \<Alt\>\<Shift\>\<Tab\> should not
 *    trigger the shortcut.
 *    Effectively, this means that the modifiers are compared using the
 *    equality operator (==).
 *
 * 2. Only relevant modifiers are considered for the matching.  For example,
 *    Caps Lock and Num Lock should not generally affect the matching, e.g.
 *    when matching \<Alt\>\<Tab\> against the state, it does not matter
 *    whether Num Lock is active or not.  These relevant, or "significant",
 *    modifiers usually include Alt, Control, Shift, Super and similar.
 *    Effectively, this means that non-significant modifiers are masked out,
 *    before doing the comparison as described above.
 *
 * 3. The matching must be independent of the layout/keymap.  For example,
 *    the \<Plus\> (+) symbol is found on the first level on some layouts,
 *    but requires holding Shift on others.  If you simply bind the action
 *    to the \<Plus\> keysym, it would work for the unshifted kind, but
 *    not for the others, because the match against Shift would fail.  If
 *    you bind the action to \<Shift\>\<Plus\>, only the shifted kind would
 *    work.  So what is needed is to recognize that Shift is used up in the
 *    translation of the keysym itself, and therefore should not be included
 *    in the matching.
 *    Effectively, this means that consumed modifiers (Shift in this example)
 *    are masked out as well, before doing the comparison.
 *
 * In summary, this is approximately how the matching would be performed:
 * @code
 *   (keysym == shortcut_keysym) &&
 *   ((state_mods & ~consumed_mods & significant_mods) == shortcut_mods)
 * @endcode
 *
 * @c state_mods are the modifiers reported by
 * xkb_state_mod_index_is_active() and similar functions.
 * @c consumed_mods are the modifiers reported by
 * xkb_state_mod_index_is_consumed() and similar functions.
 * @c significant_mods are decided upon by the application/toolkit/user;
 * it is up to them to decide whether these are configurable or hard-coded.
 *
 * @endparblock
 */

/**
 * Consumed modifiers mode.
 *
 * There are several possible methods for deciding which modifiers are
 * consumed and which are not, each applicable for different systems or
 * situations. The mode selects the method to use.
 *
 * Keep in mind that in all methods, the keymap may decide to "preserve"
 * a modifier, meaning it is not reported as consumed even if it would
 * have otherwise.
 */
enum xkb_consumed_mode {
    /**
     * This is the mode defined in the XKB specification and used by libX11.
     *
     * A modifier is consumed if and only if it *may affect* key translation.
     *
     * For example, if `Control+Alt+<Backspace>` produces some assigned keysym,
     * then when pressing just `<Backspace>`, `Control` and `Alt` are consumed,
     * even though they are not active, since if they *were* active they would
     * have affected key translation.
     */
    XKB_CONSUMED_MODE_XKB,
    /**
     * This is the mode used by the GTK+ toolkit.
     *
     * The mode consists of the following two independent heuristics:
     *
     * - The currently active set of modifiers, excluding modifiers which do
     *   not affect the key (as described for @ref XKB_CONSUMED_MODE_XKB), are
     *   considered consumed, if the keysyms produced when all of them are
     *   active are different from the keysyms produced when no modifiers are
     *   active.
     *
     * - A single modifier is considered consumed if the keysyms produced for
     *   the key when it is the only active modifier are different from the
     *   keysyms produced when no modifiers are active.
     */
    XKB_CONSUMED_MODE_GTK
};

/**
 * Get the mask of modifiers consumed by translating a given key.
 *
 * @param state The keyboard state.
 * @param key   The keycode of the key.
 * @param mode  The consumed modifiers mode to use; see enum description.
 *
 * @returns a mask of the consumed modifiers.
 *
 * @memberof xkb_state
 * @since 0.7.0
 */
xkb_mod_mask_t
xkb_state_key_get_consumed_mods2(struct xkb_state *state, xkb_keycode_t key,
                                 enum xkb_consumed_mode mode);

/**
 * Same as xkb_state_key_get_consumed_mods2() with mode XKB_CONSUMED_MODE_XKB.
 *
 * @memberof xkb_state
 * @since 0.4.1
 */
xkb_mod_mask_t
xkb_state_key_get_consumed_mods(struct xkb_state *state, xkb_keycode_t key);

/**
 * Test whether a modifier is consumed by keyboard state translation for
 * a key.
 *
 * @param state The keyboard state.
 * @param key   The keycode of the key.
 * @param idx   The index of the modifier to check.
 * @param mode  The consumed modifiers mode to use; see enum description.
 *
 * @returns 1 if the modifier is consumed, 0 if it is not.  If the modifier
 * index is not valid in the keymap, returns -1.
 *
 * @sa xkb_state_mod_mask_remove_consumed()
 * @sa xkb_state_key_get_consumed_mods()
 * @memberof xkb_state
 * @since 0.7.0
 */
int
xkb_state_mod_index_is_consumed2(struct xkb_state *state,
                                 xkb_keycode_t key,
                                 xkb_mod_index_t idx,
                                 enum xkb_consumed_mode mode);

/**
 * Same as xkb_state_mod_index_is_consumed2() with mode XKB_CONSUMED_MOD_XKB.
 *
 * @memberof xkb_state
 * @since 0.4.1
 */
int
xkb_state_mod_index_is_consumed(struct xkb_state *state, xkb_keycode_t key,
                                xkb_mod_index_t idx);

/**
 * Remove consumed modifiers from a modifier mask for a key.
 *
 * @deprecated Use xkb_state_key_get_consumed_mods2() instead.
 *
 * Takes the given modifier mask, and removes all modifiers which are
 * consumed for that particular key (as in xkb_state_mod_index_is_consumed()).
 *
 * @sa xkb_state_mod_index_is_consumed()
 * @memberof xkb_state
 */
xkb_mod_mask_t
xkb_state_mod_mask_remove_consumed(struct xkb_state *state, xkb_keycode_t key,
                                   xkb_mod_mask_t mask);

/**
 * Test whether a layout is active in a given keyboard state by name.
 *
 * @returns 1 if the layout is active, 0 if it is not.  If no layout with
 * this name exists in the keymap, return -1.
 *
 * If multiple layouts in the keymap have this name, the one with the lowest
 * index is tested.
 *
 * @sa xkb_layout_index_t
 * @memberof xkb_state
 */
int
xkb_state_layout_name_is_active(struct xkb_state *state, const char *name,
                                enum xkb_state_component type);

/**
 * Test whether a layout is active in a given keyboard state by index.
 *
 * @returns 1 if the layout is active, 0 if it is not.  If the layout index
 * is not valid in the keymap, returns -1.
 *
 * @sa xkb_layout_index_t
 * @memberof xkb_state
 */
int
xkb_state_layout_index_is_active(struct xkb_state *state,
                                 xkb_layout_index_t idx,
                                 enum xkb_state_component type);

/**
 * Test whether a LED is active in a given keyboard state by name.
 *
 * @returns 1 if the LED is active, 0 if it not.  If no LED with this name
 * exists in the keymap, returns -1.
 *
 * @sa xkb_led_index_t
 * @memberof xkb_state
 */
int
xkb_state_led_name_is_active(struct xkb_state *state, const char *name);

/**
 * Test whether a LED is active in a given keyboard state by index.
 *
 * @returns 1 if the LED is active, 0 if it not.  If the LED index is not
 * valid in the keymap, returns -1.
 *
 * @sa xkb_led_index_t
 * @memberof xkb_state
 */
int
xkb_state_led_index_is_active(struct xkb_state *state, xkb_led_index_t idx);

/** @} */

/* Leave this include last, so it can pick up our types, etc. */
/* #include "external/xkbcommon/xkbcommon-compat.h" */
/*
 * Copyright © 2012 Daniel Stone
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * Author: Daniel Stone <daniel@fooishbar.org>
 */

#ifndef _XKBCOMMON_COMPAT_H
#define _XKBCOMMON_COMPAT_H

/**
 * Renamed keymap API.
 */
#define xkb_group_index_t xkb_layout_index_t
#define xkb_group_mask_t xkb_layout_mask_t
#define xkb_map_compile_flags xkb_keymap_compile_flags
#define XKB_GROUP_INVALID XKB_LAYOUT_INVALID

#define XKB_STATE_DEPRESSED \
    (XKB_STATE_MODS_DEPRESSED | XKB_STATE_LAYOUT_DEPRESSED)
#define XKB_STATE_LATCHED \
    (XKB_STATE_MODS_LATCHED | XKB_STATE_LAYOUT_LATCHED)
#define XKB_STATE_LOCKED \
    (XKB_STATE_MODS_LOCKED | XKB_STATE_LAYOUT_LOCKED)
#define XKB_STATE_EFFECTIVE \
    (XKB_STATE_DEPRESSED | XKB_STATE_LATCHED | XKB_STATE_LOCKED | \
     XKB_STATE_MODS_EFFECTIVE | XKB_STATE_LAYOUT_EFFECTIVE)

#define xkb_map_new_from_names(context, names, flags) \
        xkb_keymap_new_from_names(context, names, flags)
#define xkb_map_new_from_file(context, file, format, flags) \
        xkb_keymap_new_from_file(context, file, format, flags)
#define xkb_map_new_from_string(context, string, format, flags) \
        xkb_keymap_new_from_string(context, string, format, flags)
#define xkb_map_get_as_string(keymap) \
        xkb_keymap_get_as_string(keymap, XKB_KEYMAP_FORMAT_TEXT_V1)
#define xkb_map_ref(keymap) xkb_keymap_ref(keymap)
#define xkb_map_unref(keymap) xkb_keymap_unref(keymap)

#define xkb_map_num_mods(keymap) xkb_keymap_num_mods(keymap)
#define xkb_map_mod_get_name(keymap, idx) xkb_keymap_mod_get_name(keymap, idx)
#define xkb_map_mod_get_index(keymap, str) xkb_keymap_mod_get_index(keymap, str)
#define xkb_key_mod_index_is_consumed(state, key, mod) \
        xkb_state_mod_index_is_consumed(state, key, mod)
#define xkb_key_mod_mask_remove_consumed(state, key, modmask) \
        xkb_state_mod_mask_remove_consumed(state, key, modmask)

#define xkb_map_num_groups(keymap) xkb_keymap_num_layouts(keymap)
#define xkb_key_num_groups(keymap, key) \
        xkb_keymap_num_layouts_for_key(keymap, key)
#define xkb_map_group_get_name(keymap, idx) \
        xkb_keymap_layout_get_name(keymap, idx)
#define xkb_map_group_get_index(keymap, str) \
        xkb_keymap_layout_get_index(keymap, str)

#define xkb_map_num_leds(keymap) xkb_keymap_num_leds(keymap)
#define xkb_map_led_get_name(keymap, idx) xkb_keymap_led_get_name(keymap, idx)
#define xkb_map_led_get_index(keymap, str) \
        xkb_keymap_led_get_index(keymap, str)

#define xkb_key_repeats(keymap, key) xkb_keymap_key_repeats(keymap, key)

#define xkb_key_get_syms(state, key, syms_out) \
        xkb_state_key_get_syms(state, key, syms_out)

#define xkb_state_group_name_is_active(state, name, type) \
        xkb_state_layout_name_is_active(state, name, type)
#define xkb_state_group_index_is_active(state, idx, type) \
        xkb_state_layout_index_is_active(state, idx, type)

#define xkb_state_serialize_group(state, component) \
        xkb_state_serialize_layout(state, component)

#define xkb_state_get_map(state) xkb_state_get_keymap(state)

/* Not needed anymore, since there's NO_FLAGS. */
#define XKB_MAP_COMPILE_PLACEHOLDER XKB_KEYMAP_COMPILE_NO_FLAGS
#define XKB_MAP_COMPILE_NO_FLAGS XKB_KEYMAP_COMPILE_NO_FLAGS

#endif


#ifdef __cplusplus
} /* extern "C" */
#endif

#endif /* _XKBCOMMON_H_ */
#endif

#if !defined(__EMSCRIPTEN__)
// Common modules to all platforms
/* #  include "external/glfw/src/init.c" */
//========================================================================
// GLFW 3.4 - www.glfw.org
//------------------------------------------------------------------------
// Copyright (c) 2002-2006 Marcus Geelnard
// Copyright (c) 2006-2018 Camilla Löwy <elmindreda@glfw.org>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any damages
// arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it
// freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you must not
//    claim that you wrote the original software. If you use this software
//    in a product, an acknowledgment in the product documentation would
//    be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such, and must not
//    be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source
//    distribution.
//
//========================================================================

/* #include "external/glfw/src/internal.h" */
//========================================================================
// GLFW 3.4 - www.glfw.org
//------------------------------------------------------------------------
// Copyright (c) 2002-2006 Marcus Geelnard
// Copyright (c) 2006-2019 Camilla Löwy <elmindreda@glfw.org>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any damages
// arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it
// freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you must not
//    claim that you wrote the original software. If you use this software
//    in a product, an acknowledgment in the product documentation would
//    be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such, and must not
//    be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source
//    distribution.
//
//========================================================================


/* #include "src/br_pp.h" */


#if defined(GLFW_INCLUDE_GLCOREARB) || \
    defined(GLFW_INCLUDE_ES1)       || \
    defined(GLFW_INCLUDE_ES2)       || \
    defined(GLFW_INCLUDE_ES3)       || \
    defined(GLFW_INCLUDE_ES31)      || \
    defined(GLFW_INCLUDE_ES32)      || \
    defined(GLFW_INCLUDE_NONE)      || \
    defined(GLFW_INCLUDE_GLEXT)     || \
    defined(GLFW_INCLUDE_GLU)       || \
    defined(GLFW_INCLUDE_VULKAN)    || \
    defined(GLFW_DLL)
 #error "You must not define any header option macros when compiling GLFW"
#endif

#define GLFW_INCLUDE_NONE
/* #include "external/glfw/include/GLFW/glfw3.h" */


#define _GLFW_INSERT_FIRST      0
#define _GLFW_INSERT_LAST       1

#define _GLFW_POLL_PRESENCE     0
#define _GLFW_POLL_AXES         1
#define _GLFW_POLL_BUTTONS      2
#define _GLFW_POLL_ALL          (_GLFW_POLL_AXES | _GLFW_POLL_BUTTONS)

#define _GLFW_MESSAGE_SIZE      1024

typedef int GLFWbool;
typedef void (*GLFWproc)(void);

typedef struct _GLFWerror       _GLFWerror;
typedef struct _GLFWinitconfig  _GLFWinitconfig;
typedef struct _GLFWwndconfig   _GLFWwndconfig;
typedef struct _GLFWctxconfig   _GLFWctxconfig;
typedef struct _GLFWfbconfig    _GLFWfbconfig;
typedef struct _GLFWcontext     _GLFWcontext;
typedef struct _GLFWwindow      _GLFWwindow;
typedef struct _GLFWplatform    _GLFWplatform;
typedef struct _GLFWlibrary     _GLFWlibrary;
typedef struct _GLFWmonitor     _GLFWmonitor;
typedef struct _GLFWcursor      _GLFWcursor;
typedef struct _GLFWmapelement  _GLFWmapelement;
typedef struct _GLFWmapping     _GLFWmapping;
typedef struct _GLFWtls         _GLFWtls;
typedef struct _GLFWmutex       _GLFWmutex;

#define GL_VERSION 0x1f02
#define GL_NONE 0
#define GL_COLOR_BUFFER_BIT 0x00004000
#define GL_UNSIGNED_BYTE 0x1401
#define GL_EXTENSIONS 0x1f03
#define GL_NUM_EXTENSIONS 0x821d
#define GL_CONTEXT_FLAGS 0x821e
#define GL_CONTEXT_FLAG_FORWARD_COMPATIBLE_BIT 0x00000001
#define GL_CONTEXT_FLAG_DEBUG_BIT 0x00000002
#define GL_CONTEXT_PROFILE_MASK 0x9126
#define GL_CONTEXT_COMPATIBILITY_PROFILE_BIT 0x00000002
#define GL_CONTEXT_CORE_PROFILE_BIT 0x00000001
#define GL_RESET_NOTIFICATION_STRATEGY_ARB 0x8256
#define GL_LOSE_CONTEXT_ON_RESET_ARB 0x8252
#define GL_NO_RESET_NOTIFICATION_ARB 0x8261
#define GL_CONTEXT_RELEASE_BEHAVIOR 0x82fb
#define GL_CONTEXT_RELEASE_BEHAVIOR_FLUSH 0x82fc
#define GL_CONTEXT_FLAG_NO_ERROR_BIT_KHR 0x00000008

typedef int GLint;
typedef unsigned int GLuint;
typedef unsigned int GLenum;
typedef unsigned int GLbitfield;
typedef unsigned char GLubyte;

typedef void (APIENTRY * PFNGLCLEARPROC)(GLbitfield);
typedef const GLubyte* (APIENTRY * PFNGLGETSTRINGPROC)(GLenum);
typedef void (APIENTRY * PFNGLGETINTEGERVPROC)(GLenum,GLint*);
typedef const GLubyte* (APIENTRY * PFNGLGETSTRINGIPROC)(GLenum,GLuint);

#define EGL_SUCCESS 0x3000
#define EGL_NOT_INITIALIZED 0x3001
#define EGL_BAD_ACCESS 0x3002
#define EGL_BAD_ALLOC 0x3003
#define EGL_BAD_ATTRIBUTE 0x3004
#define EGL_BAD_CONFIG 0x3005
#define EGL_BAD_CONTEXT 0x3006
#define EGL_BAD_CURRENT_SURFACE 0x3007
#define EGL_BAD_DISPLAY 0x3008
#define EGL_BAD_MATCH 0x3009
#define EGL_BAD_NATIVE_PIXMAP 0x300a
#define EGL_BAD_NATIVE_WINDOW 0x300b
#define EGL_BAD_PARAMETER 0x300c
#define EGL_BAD_SURFACE 0x300d
#define EGL_CONTEXT_LOST 0x300e
#define EGL_COLOR_BUFFER_TYPE 0x303f
#define EGL_RGB_BUFFER 0x308e
#define EGL_SURFACE_TYPE 0x3033
#define EGL_WINDOW_BIT 0x0004
#define EGL_RENDERABLE_TYPE 0x3040
#define EGL_OPENGL_ES_BIT 0x0001
#define EGL_OPENGL_ES2_BIT 0x0004
#define EGL_OPENGL_BIT 0x0008
#define EGL_ALPHA_SIZE 0x3021
#define EGL_BLUE_SIZE 0x3022
#define EGL_GREEN_SIZE 0x3023
#define EGL_RED_SIZE 0x3024
#define EGL_DEPTH_SIZE 0x3025
#define EGL_STENCIL_SIZE 0x3026
#define EGL_SAMPLES 0x3031
#define EGL_OPENGL_ES_API 0x30a0
#define EGL_OPENGL_API 0x30a2
#define EGL_NONE 0x3038
#define EGL_RENDER_BUFFER 0x3086
#define EGL_SINGLE_BUFFER 0x3085
#define EGL_EXTENSIONS 0x3055
#define EGL_CONTEXT_CLIENT_VERSION 0x3098
#define EGL_NATIVE_VISUAL_ID 0x302e
#define EGL_NO_SURFACE ((EGLSurface) 0)
#define EGL_NO_DISPLAY ((EGLDisplay) 0)
#define EGL_NO_CONTEXT ((EGLContext) 0)
#define EGL_DEFAULT_DISPLAY ((EGLNativeDisplayType) 0)

#define EGL_CONTEXT_OPENGL_FORWARD_COMPATIBLE_BIT_KHR 0x00000002
#define EGL_CONTEXT_OPENGL_CORE_PROFILE_BIT_KHR 0x00000001
#define EGL_CONTEXT_OPENGL_COMPATIBILITY_PROFILE_BIT_KHR 0x00000002
#define EGL_CONTEXT_OPENGL_DEBUG_BIT_KHR 0x00000001
#define EGL_CONTEXT_OPENGL_RESET_NOTIFICATION_STRATEGY_KHR 0x31bd
#define EGL_NO_RESET_NOTIFICATION_KHR 0x31be
#define EGL_LOSE_CONTEXT_ON_RESET_KHR 0x31bf
#define EGL_CONTEXT_OPENGL_ROBUST_ACCESS_BIT_KHR 0x00000004
#define EGL_CONTEXT_MAJOR_VERSION_KHR 0x3098
#define EGL_CONTEXT_MINOR_VERSION_KHR 0x30fb
#define EGL_CONTEXT_OPENGL_PROFILE_MASK_KHR 0x30fd
#define EGL_CONTEXT_FLAGS_KHR 0x30fc
#define EGL_CONTEXT_OPENGL_NO_ERROR_KHR 0x31b3
#define EGL_GL_COLORSPACE_KHR 0x309d
#define EGL_GL_COLORSPACE_SRGB_KHR 0x3089
#define EGL_CONTEXT_RELEASE_BEHAVIOR_KHR 0x2097
#define EGL_CONTEXT_RELEASE_BEHAVIOR_NONE_KHR 0
#define EGL_CONTEXT_RELEASE_BEHAVIOR_FLUSH_KHR 0x2098
#define EGL_PLATFORM_X11_EXT 0x31d5
#define EGL_PLATFORM_WAYLAND_EXT 0x31d8
#define EGL_PRESENT_OPAQUE_EXT 0x31df
#define EGL_PLATFORM_ANGLE_ANGLE 0x3202
#define EGL_PLATFORM_ANGLE_TYPE_ANGLE 0x3203
#define EGL_PLATFORM_ANGLE_TYPE_OPENGL_ANGLE 0x320d
#define EGL_PLATFORM_ANGLE_TYPE_OPENGLES_ANGLE 0x320e
#define EGL_PLATFORM_ANGLE_TYPE_D3D9_ANGLE 0x3207
#define EGL_PLATFORM_ANGLE_TYPE_D3D11_ANGLE 0x3208
#define EGL_PLATFORM_ANGLE_TYPE_VULKAN_ANGLE 0x3450
#define EGL_PLATFORM_ANGLE_TYPE_METAL_ANGLE 0x3489
#define EGL_PLATFORM_ANGLE_NATIVE_PLATFORM_TYPE_ANGLE 0x348f

typedef int EGLint;
typedef unsigned int EGLBoolean;
typedef unsigned int EGLenum;
typedef void* EGLConfig;
typedef void* EGLContext;
typedef void* EGLDisplay;
typedef void* EGLSurface;

typedef void* EGLNativeDisplayType;
typedef void* EGLNativeWindowType;

// EGL function pointer typedefs
typedef EGLBoolean (APIENTRY * PFN_eglGetConfigAttrib)(EGLDisplay,EGLConfig,EGLint,EGLint*);
typedef EGLBoolean (APIENTRY * PFN_eglGetConfigs)(EGLDisplay,EGLConfig*,EGLint,EGLint*);
typedef EGLDisplay (APIENTRY * PFN_eglGetDisplay)(EGLNativeDisplayType);
typedef EGLint (APIENTRY * PFN_eglGetError)(void);
typedef EGLBoolean (APIENTRY * PFN_eglInitialize)(EGLDisplay,EGLint*,EGLint*);
typedef EGLBoolean (APIENTRY * PFN_eglTerminate)(EGLDisplay);
typedef EGLBoolean (APIENTRY * PFN_eglBindAPI)(EGLenum);
typedef EGLContext (APIENTRY * PFN_eglCreateContext)(EGLDisplay,EGLConfig,EGLContext,const EGLint*);
typedef EGLBoolean (APIENTRY * PFN_eglDestroySurface)(EGLDisplay,EGLSurface);
typedef EGLBoolean (APIENTRY * PFN_eglDestroyContext)(EGLDisplay,EGLContext);
typedef EGLSurface (APIENTRY * PFN_eglCreateWindowSurface)(EGLDisplay,EGLConfig,EGLNativeWindowType,const EGLint*);
typedef EGLBoolean (APIENTRY * PFN_eglMakeCurrent)(EGLDisplay,EGLSurface,EGLSurface,EGLContext);
typedef EGLBoolean (APIENTRY * PFN_eglSwapBuffers)(EGLDisplay,EGLSurface);
typedef EGLBoolean (APIENTRY * PFN_eglSwapInterval)(EGLDisplay,EGLint);
typedef const char* (APIENTRY * PFN_eglQueryString)(EGLDisplay,EGLint);
typedef GLFWglproc (APIENTRY * PFN_eglGetProcAddress)(const char*);
#define eglGetConfigAttrib _glfw.egl.GetConfigAttrib
#define eglGetConfigs _glfw.egl.GetConfigs
#define eglGetDisplay _glfw.egl.GetDisplay
#define eglGetError _glfw.egl.GetError
#define eglInitialize _glfw.egl.Initialize
#define eglTerminate _glfw.egl.Terminate
#define eglBindAPI _glfw.egl.BindAPI
#define eglCreateContext _glfw.egl.CreateContext
#define eglDestroySurface _glfw.egl.DestroySurface
#define eglDestroyContext _glfw.egl.DestroyContext
#define eglCreateWindowSurface _glfw.egl.CreateWindowSurface
#define eglMakeCurrent _glfw.egl.MakeCurrent
#define eglSwapBuffers _glfw.egl.SwapBuffers
#define eglSwapInterval _glfw.egl.SwapInterval
#define eglQueryString _glfw.egl.QueryString
#define eglGetProcAddress _glfw.egl.GetProcAddress

typedef EGLDisplay (APIENTRY * PFNEGLGETPLATFORMDISPLAYEXTPROC)(EGLenum,void*,const EGLint*);
typedef EGLSurface (APIENTRY * PFNEGLCREATEPLATFORMWINDOWSURFACEEXTPROC)(EGLDisplay,EGLConfig,void*,const EGLint*);
#define eglGetPlatformDisplayEXT _glfw.egl.GetPlatformDisplayEXT
#define eglCreatePlatformWindowSurfaceEXT _glfw.egl.CreatePlatformWindowSurfaceEXT

#define OSMESA_RGBA 0x1908
#define OSMESA_FORMAT 0x22
#define OSMESA_DEPTH_BITS 0x30
#define OSMESA_STENCIL_BITS 0x31
#define OSMESA_ACCUM_BITS 0x32
#define OSMESA_PROFILE 0x33
#define OSMESA_CORE_PROFILE 0x34
#define OSMESA_COMPAT_PROFILE 0x35
#define OSMESA_CONTEXT_MAJOR_VERSION 0x36
#define OSMESA_CONTEXT_MINOR_VERSION 0x37

typedef void* OSMesaContext;
typedef void (*OSMESAproc)(void);

typedef OSMesaContext (GLAPIENTRY * PFN_OSMesaCreateContextExt)(GLenum,GLint,GLint,GLint,OSMesaContext);
typedef OSMesaContext (GLAPIENTRY * PFN_OSMesaCreateContextAttribs)(const int*,OSMesaContext);
typedef void (GLAPIENTRY * PFN_OSMesaDestroyContext)(OSMesaContext);
typedef int (GLAPIENTRY * PFN_OSMesaMakeCurrent)(OSMesaContext,void*,int,int,int);
typedef int (GLAPIENTRY * PFN_OSMesaGetColorBuffer)(OSMesaContext,int*,int*,int*,void**);
typedef int (GLAPIENTRY * PFN_OSMesaGetDepthBuffer)(OSMesaContext,int*,int*,int*,void**);
typedef GLFWglproc (GLAPIENTRY * PFN_OSMesaGetProcAddress)(const char*);
#define OSMesaCreateContextExt _glfw.osmesa.CreateContextExt
#define OSMesaCreateContextAttribs _glfw.osmesa.CreateContextAttribs
#define OSMesaDestroyContext _glfw.osmesa.DestroyContext
#define OSMesaMakeCurrent _glfw.osmesa.MakeCurrent
#define OSMesaGetColorBuffer _glfw.osmesa.GetColorBuffer
#define OSMesaGetDepthBuffer _glfw.osmesa.GetDepthBuffer
#define OSMesaGetProcAddress _glfw.osmesa.GetProcAddress

#define VK_NULL_HANDLE 0

typedef void* VkInstance;
typedef void* VkPhysicalDevice;
typedef uint64_t VkSurfaceKHR;
typedef uint32_t VkFlags;
typedef uint32_t VkBool32;

typedef enum VkStructureType
{
    VK_STRUCTURE_TYPE_XLIB_SURFACE_CREATE_INFO_KHR = 1000004000,
    VK_STRUCTURE_TYPE_XCB_SURFACE_CREATE_INFO_KHR = 1000005000,
    VK_STRUCTURE_TYPE_WAYLAND_SURFACE_CREATE_INFO_KHR = 1000006000,
    VK_STRUCTURE_TYPE_WIN32_SURFACE_CREATE_INFO_KHR = 1000009000,
    VK_STRUCTURE_TYPE_MACOS_SURFACE_CREATE_INFO_MVK = 1000123000,
    VK_STRUCTURE_TYPE_METAL_SURFACE_CREATE_INFO_EXT = 1000217000,
    VK_STRUCTURE_TYPE_MAX_ENUM = 0x7FFFFFFF
} VkStructureType;

typedef enum VkResult
{
    VK_SUCCESS = 0,
    VK_NOT_READY = 1,
    VK_TIMEOUT = 2,
    VK_EVENT_SET = 3,
    VK_EVENT_RESET = 4,
    VK_INCOMPLETE = 5,
    VK_ERROR_OUT_OF_HOST_MEMORY = -1,
    VK_ERROR_OUT_OF_DEVICE_MEMORY = -2,
    VK_ERROR_INITIALIZATION_FAILED = -3,
    VK_ERROR_DEVICE_LOST = -4,
    VK_ERROR_MEMORY_MAP_FAILED = -5,
    VK_ERROR_LAYER_NOT_PRESENT = -6,
    VK_ERROR_EXTENSION_NOT_PRESENT = -7,
    VK_ERROR_FEATURE_NOT_PRESENT = -8,
    VK_ERROR_INCOMPATIBLE_DRIVER = -9,
    VK_ERROR_TOO_MANY_OBJECTS = -10,
    VK_ERROR_FORMAT_NOT_SUPPORTED = -11,
    VK_ERROR_SURFACE_LOST_KHR = -1000000000,
    VK_SUBOPTIMAL_KHR = 1000001003,
    VK_ERROR_OUT_OF_DATE_KHR = -1000001004,
    VK_ERROR_INCOMPATIBLE_DISPLAY_KHR = -1000003001,
    VK_ERROR_NATIVE_WINDOW_IN_USE_KHR = -1000000001,
    VK_ERROR_VALIDATION_FAILED_EXT = -1000011001,
    VK_RESULT_MAX_ENUM = 0x7FFFFFFF
} VkResult;

typedef struct VkAllocationCallbacks VkAllocationCallbacks;

typedef struct VkExtensionProperties
{
    char            extensionName[256];
    uint32_t        specVersion;
} VkExtensionProperties;

typedef void (APIENTRY * PFN_vkVoidFunction)(void);

typedef PFN_vkVoidFunction (APIENTRY * PFN_vkGetInstanceProcAddr)(VkInstance,const char*);
typedef VkResult (APIENTRY * PFN_vkEnumerateInstanceExtensionProperties)(const char*,uint32_t*,VkExtensionProperties*);
#define vkGetInstanceProcAddr _glfw.vk.GetInstanceProcAddr

/* #include "external/glfw/src/platform.h" */
//========================================================================
// GLFW 3.4 - www.glfw.org
//------------------------------------------------------------------------
// Copyright (c) 2002-2006 Marcus Geelnard
// Copyright (c) 2006-2018 Camilla Löwy <elmindreda@glfw.org>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any damages
// arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it
// freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you must not
//    claim that you wrote the original software. If you use this software
//    in a product, an acknowledgment in the product documentation would
//    be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such, and must not
//    be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source
//    distribution.
//
//========================================================================

#if defined(GLFW_BUILD_WIN32_TIMER) || \
    defined(GLFW_BUILD_WIN32_MODULE) || \
    defined(GLFW_BUILD_WIN32_THREAD) || \
    defined(GLFW_BUILD_COCOA_TIMER) || \
    defined(GLFW_BUILD_POSIX_TIMER) || \
    defined(GLFW_BUILD_POSIX_MODULE) || \
    defined(GLFW_BUILD_POSIX_THREAD) || \
    defined(GLFW_BUILD_POSIX_POLL)
 #error "You must not define these; define zero or more _GLFW_<platform> macros instead"
#endif

/* #include "external/glfw/src/null_platform.h" */
//========================================================================
// GLFW 3.4 - www.glfw.org
//------------------------------------------------------------------------
// Copyright (c) 2016 Google Inc.
// Copyright (c) 2016-2017 Camilla Löwy <elmindreda@glfw.org>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any damages
// arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it
// freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you must not
//    claim that you wrote the original software. If you use this software
//    in a product, an acknowledgment in the product documentation would
//    be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such, and must not
//    be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source
//    distribution.
//
//========================================================================

#define GLFW_NULL_WINDOW_STATE          _GLFWwindowNull null;
#define GLFW_NULL_LIBRARY_WINDOW_STATE  _GLFWlibraryNull null;
#define GLFW_NULL_MONITOR_STATE         _GLFWmonitorNull null;

#define GLFW_NULL_CONTEXT_STATE
#define GLFW_NULL_CURSOR_STATE
#define GLFW_NULL_LIBRARY_CONTEXT_STATE

#define GLFW_NULL_SC_FIRST          GLFW_NULL_SC_SPACE
#define GLFW_NULL_SC_SPACE          1
#define GLFW_NULL_SC_APOSTROPHE     2
#define GLFW_NULL_SC_COMMA          3
#define GLFW_NULL_SC_MINUS          4
#define GLFW_NULL_SC_PERIOD         5
#define GLFW_NULL_SC_SLASH          6
#define GLFW_NULL_SC_0              7
#define GLFW_NULL_SC_1              8
#define GLFW_NULL_SC_2              9
#define GLFW_NULL_SC_3              10
#define GLFW_NULL_SC_4              11
#define GLFW_NULL_SC_5              12
#define GLFW_NULL_SC_6              13
#define GLFW_NULL_SC_7              14
#define GLFW_NULL_SC_8              15
#define GLFW_NULL_SC_9              16
#define GLFW_NULL_SC_SEMICOLON      17
#define GLFW_NULL_SC_EQUAL          18
#define GLFW_NULL_SC_LEFT_BRACKET   19
#define GLFW_NULL_SC_BACKSLASH      20
#define GLFW_NULL_SC_RIGHT_BRACKET  21
#define GLFW_NULL_SC_GRAVE_ACCENT   22
#define GLFW_NULL_SC_WORLD_1        23
#define GLFW_NULL_SC_WORLD_2        24
#define GLFW_NULL_SC_ESCAPE         25
#define GLFW_NULL_SC_ENTER          26
#define GLFW_NULL_SC_TAB            27
#define GLFW_NULL_SC_BACKSPACE      28
#define GLFW_NULL_SC_INSERT         29
#define GLFW_NULL_SC_DELETE         30
#define GLFW_NULL_SC_RIGHT          31
#define GLFW_NULL_SC_LEFT           32
#define GLFW_NULL_SC_DOWN           33
#define GLFW_NULL_SC_UP             34
#define GLFW_NULL_SC_PAGE_UP        35
#define GLFW_NULL_SC_PAGE_DOWN      36
#define GLFW_NULL_SC_HOME           37
#define GLFW_NULL_SC_END            38
#define GLFW_NULL_SC_CAPS_LOCK      39
#define GLFW_NULL_SC_SCROLL_LOCK    40
#define GLFW_NULL_SC_NUM_LOCK       41
#define GLFW_NULL_SC_PRINT_SCREEN   42
#define GLFW_NULL_SC_PAUSE          43
#define GLFW_NULL_SC_A              44
#define GLFW_NULL_SC_B              45
#define GLFW_NULL_SC_C              46
#define GLFW_NULL_SC_D              47
#define GLFW_NULL_SC_E              48
#define GLFW_NULL_SC_F              49
#define GLFW_NULL_SC_G              50
#define GLFW_NULL_SC_H              51
#define GLFW_NULL_SC_I              52
#define GLFW_NULL_SC_J              53
#define GLFW_NULL_SC_K              54
#define GLFW_NULL_SC_L              55
#define GLFW_NULL_SC_M              56
#define GLFW_NULL_SC_N              57
#define GLFW_NULL_SC_O              58
#define GLFW_NULL_SC_P              59
#define GLFW_NULL_SC_Q              60
#define GLFW_NULL_SC_R              61
#define GLFW_NULL_SC_S              62
#define GLFW_NULL_SC_T              63
#define GLFW_NULL_SC_U              64
#define GLFW_NULL_SC_V              65
#define GLFW_NULL_SC_W              66
#define GLFW_NULL_SC_X              67
#define GLFW_NULL_SC_Y              68
#define GLFW_NULL_SC_Z              69
#define GLFW_NULL_SC_F1             70
#define GLFW_NULL_SC_F2             71
#define GLFW_NULL_SC_F3             72
#define GLFW_NULL_SC_F4             73
#define GLFW_NULL_SC_F5             74
#define GLFW_NULL_SC_F6             75
#define GLFW_NULL_SC_F7             76
#define GLFW_NULL_SC_F8             77
#define GLFW_NULL_SC_F9             78
#define GLFW_NULL_SC_F10            79
#define GLFW_NULL_SC_F11            80
#define GLFW_NULL_SC_F12            81
#define GLFW_NULL_SC_F13            82
#define GLFW_NULL_SC_F14            83
#define GLFW_NULL_SC_F15            84
#define GLFW_NULL_SC_F16            85
#define GLFW_NULL_SC_F17            86
#define GLFW_NULL_SC_F18            87
#define GLFW_NULL_SC_F19            88
#define GLFW_NULL_SC_F20            89
#define GLFW_NULL_SC_F21            90
#define GLFW_NULL_SC_F22            91
#define GLFW_NULL_SC_F23            92
#define GLFW_NULL_SC_F24            93
#define GLFW_NULL_SC_F25            94
#define GLFW_NULL_SC_KP_0           95
#define GLFW_NULL_SC_KP_1           96
#define GLFW_NULL_SC_KP_2           97
#define GLFW_NULL_SC_KP_3           98
#define GLFW_NULL_SC_KP_4           99
#define GLFW_NULL_SC_KP_5           100
#define GLFW_NULL_SC_KP_6           101
#define GLFW_NULL_SC_KP_7           102
#define GLFW_NULL_SC_KP_8           103
#define GLFW_NULL_SC_KP_9           104
#define GLFW_NULL_SC_KP_DECIMAL     105
#define GLFW_NULL_SC_KP_DIVIDE      106
#define GLFW_NULL_SC_KP_MULTIPLY    107
#define GLFW_NULL_SC_KP_SUBTRACT    108
#define GLFW_NULL_SC_KP_ADD         109
#define GLFW_NULL_SC_KP_ENTER       110
#define GLFW_NULL_SC_KP_EQUAL       111
#define GLFW_NULL_SC_LEFT_SHIFT     112
#define GLFW_NULL_SC_LEFT_CONTROL   113
#define GLFW_NULL_SC_LEFT_ALT       114
#define GLFW_NULL_SC_LEFT_SUPER     115
#define GLFW_NULL_SC_RIGHT_SHIFT    116
#define GLFW_NULL_SC_RIGHT_CONTROL  117
#define GLFW_NULL_SC_RIGHT_ALT      118
#define GLFW_NULL_SC_RIGHT_SUPER    119
#define GLFW_NULL_SC_MENU           120
#define GLFW_NULL_SC_LAST           GLFW_NULL_SC_MENU

// Null-specific per-window data
//
typedef struct _GLFWwindowNull
{
    int             xpos;
    int             ypos;
    int             width;
    int             height;
    GLFWbool        visible;
    GLFWbool        iconified;
    GLFWbool        maximized;
    GLFWbool        resizable;
    GLFWbool        decorated;
    GLFWbool        floating;
    GLFWbool        transparent;
    float           opacity;
} _GLFWwindowNull;

// Null-specific per-monitor data
//
typedef struct _GLFWmonitorNull
{
    GLFWgammaramp   ramp;
} _GLFWmonitorNull;

// Null-specific global data
//
typedef struct _GLFWlibraryNull
{
    int             xcursor;
    int             ycursor;
    char*           clipboardString;
    _GLFWwindow*    focusedWindow;
    uint16_t        keycodes[GLFW_NULL_SC_LAST + 1];
    uint8_t         scancodes[GLFW_KEY_LAST + 1];
} _GLFWlibraryNull;

void _glfwPollMonitorsNull(void);

GLFWbool _glfwConnectNull(int platformID, _GLFWplatform* platform);
int _glfwInitNull(void);
void _glfwTerminateNull(void);

void _glfwFreeMonitorNull(_GLFWmonitor* monitor);
void _glfwGetMonitorPosNull(_GLFWmonitor* monitor, int* xpos, int* ypos);
void _glfwGetMonitorContentScaleNull(_GLFWmonitor* monitor, float* xscale, float* yscale);
void _glfwGetMonitorWorkareaNull(_GLFWmonitor* monitor, int* xpos, int* ypos, int* width, int* height);
GLFWvidmode* _glfwGetVideoModesNull(_GLFWmonitor* monitor, int* found);
GLFWbool _glfwGetVideoModeNull(_GLFWmonitor* monitor, GLFWvidmode* mode);
GLFWbool _glfwGetGammaRampNull(_GLFWmonitor* monitor, GLFWgammaramp* ramp);
void _glfwSetGammaRampNull(_GLFWmonitor* monitor, const GLFWgammaramp* ramp);

GLFWbool _glfwCreateWindowNull(_GLFWwindow* window, const _GLFWwndconfig* wndconfig, const _GLFWctxconfig* ctxconfig, const _GLFWfbconfig* fbconfig);
void _glfwDestroyWindowNull(_GLFWwindow* window);
void _glfwSetWindowTitleNull(_GLFWwindow* window, const char* title);
void _glfwSetWindowIconNull(_GLFWwindow* window, int count, const GLFWimage* images);
void _glfwSetWindowMonitorNull(_GLFWwindow* window, _GLFWmonitor* monitor, int xpos, int ypos, int width, int height, int refreshRate);
void _glfwGetWindowPosNull(_GLFWwindow* window, int* xpos, int* ypos);
void _glfwSetWindowPosNull(_GLFWwindow* window, int xpos, int ypos);
void _glfwGetWindowSizeNull(_GLFWwindow* window, int* width, int* height);
void _glfwSetWindowSizeNull(_GLFWwindow* window, int width, int height);
void _glfwSetWindowSizeLimitsNull(_GLFWwindow* window, int minwidth, int minheight, int maxwidth, int maxheight);
void _glfwSetWindowAspectRatioNull(_GLFWwindow* window, int n, int d);
void _glfwGetFramebufferSizeNull(_GLFWwindow* window, int* width, int* height);
void _glfwGetWindowFrameSizeNull(_GLFWwindow* window, int* left, int* top, int* right, int* bottom);
void _glfwGetWindowContentScaleNull(_GLFWwindow* window, float* xscale, float* yscale);
void _glfwIconifyWindowNull(_GLFWwindow* window);
void _glfwRestoreWindowNull(_GLFWwindow* window);
void _glfwMaximizeWindowNull(_GLFWwindow* window);
GLFWbool _glfwWindowMaximizedNull(_GLFWwindow* window);
GLFWbool _glfwWindowHoveredNull(_GLFWwindow* window);
GLFWbool _glfwFramebufferTransparentNull(_GLFWwindow* window);
void _glfwSetWindowResizableNull(_GLFWwindow* window, GLFWbool enabled);
void _glfwSetWindowDecoratedNull(_GLFWwindow* window, GLFWbool enabled);
void _glfwSetWindowFloatingNull(_GLFWwindow* window, GLFWbool enabled);
void _glfwSetWindowMousePassthroughNull(_GLFWwindow* window, GLFWbool enabled);
float _glfwGetWindowOpacityNull(_GLFWwindow* window);
void _glfwSetWindowOpacityNull(_GLFWwindow* window, float opacity);
void _glfwSetRawMouseMotionNull(_GLFWwindow *window, GLFWbool enabled);
GLFWbool _glfwRawMouseMotionSupportedNull(void);
void _glfwShowWindowNull(_GLFWwindow* window);
void _glfwRequestWindowAttentionNull(_GLFWwindow* window);
void _glfwHideWindowNull(_GLFWwindow* window);
void _glfwFocusWindowNull(_GLFWwindow* window);
GLFWbool _glfwWindowFocusedNull(_GLFWwindow* window);
GLFWbool _glfwWindowIconifiedNull(_GLFWwindow* window);
GLFWbool _glfwWindowVisibleNull(_GLFWwindow* window);
void _glfwPollEventsNull(void);
void _glfwWaitEventsNull(void);
void _glfwWaitEventsTimeoutNull(double timeout);
void _glfwPostEmptyEventNull(void);
void _glfwGetCursorPosNull(_GLFWwindow* window, double* xpos, double* ypos);
void _glfwSetCursorPosNull(_GLFWwindow* window, double x, double y);
void _glfwSetCursorModeNull(_GLFWwindow* window, int mode);
GLFWbool _glfwCreateCursorNull(_GLFWcursor* cursor, const GLFWimage* image, int xhot, int yhot);
GLFWbool _glfwCreateStandardCursorNull(_GLFWcursor* cursor, int shape);
void _glfwDestroyCursorNull(_GLFWcursor* cursor);
void _glfwSetCursorNull(_GLFWwindow* window, _GLFWcursor* cursor);
void _glfwSetClipboardStringNull(const char* string);
const char* _glfwGetClipboardStringNull(void);
const char* _glfwGetScancodeNameNull(int scancode);
int _glfwGetKeyScancodeNull(int key);

EGLenum _glfwGetEGLPlatformNull(EGLint** attribs);
EGLNativeDisplayType _glfwGetEGLNativeDisplayNull(void);
EGLNativeWindowType _glfwGetEGLNativeWindowNull(_GLFWwindow* window);

void _glfwGetRequiredInstanceExtensionsNull(char** extensions);
GLFWbool _glfwGetPhysicalDevicePresentationSupportNull(VkInstance instance, VkPhysicalDevice device, uint32_t queuefamily);
VkResult _glfwCreateWindowSurfaceNull(VkInstance instance, _GLFWwindow* window, const VkAllocationCallbacks* allocator, VkSurfaceKHR* surface);

void _glfwPollMonitorsNull(void);


#define GLFW_EXPOSE_NATIVE_EGL
#define GLFW_EXPOSE_NATIVE_OSMESA

#if defined(_GLFW_WIN32)
 /* #include "external/glfw/src/win32_platform.h" */
//========================================================================
// GLFW 3.4 Win32 - www.glfw.org
//------------------------------------------------------------------------
// Copyright (c) 2002-2006 Marcus Geelnard
// Copyright (c) 2006-2019 Camilla Löwy <elmindreda@glfw.org>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any damages
// arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it
// freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you must not
//    claim that you wrote the original software. If you use this software
//    in a product, an acknowledgment in the product documentation would
//    be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such, and must not
//    be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source
//    distribution.
//
//========================================================================

// We don't need all the fancy stuff
#ifndef NOMINMAX
 #define NOMINMAX
#endif

#ifndef VC_EXTRALEAN
 #define VC_EXTRALEAN
#endif

#ifndef WIN32_LEAN_AND_MEAN
 #define WIN32_LEAN_AND_MEAN
#endif

// This is a workaround for the fact that glfw3.h needs to export APIENTRY (for
// example to allow applications to correctly declare a GL_KHR_debug callback)
// but windows.h assumes no one will define APIENTRY before it does
#undef APIENTRY

// GLFW requires Windows XP or later
#if WINVER < 0x0501
 #undef WINVER
 #define WINVER 0x0501
#endif
#if _WIN32_WINNT < 0x0501
 #undef _WIN32_WINNT
 #define _WIN32_WINNT 0x0501
#endif

// GLFW uses DirectInput8 interfaces
#define DIRECTINPUT_VERSION 0x0800

// GLFW uses OEM cursor resources
#define OEMRESOURCE

#include <wctype.h>
#include <windows.h>
#include <dinput.h>
#include <xinput.h>
#include <dbt.h>

// HACK: Define macros that some windows.h variants don't
#ifndef WM_MOUSEHWHEEL
 #define WM_MOUSEHWHEEL 0x020E
#endif
#ifndef WM_DWMCOMPOSITIONCHANGED
 #define WM_DWMCOMPOSITIONCHANGED 0x031E
#endif
#ifndef WM_DWMCOLORIZATIONCOLORCHANGED
 #define WM_DWMCOLORIZATIONCOLORCHANGED 0x0320
#endif
#ifndef WM_COPYGLOBALDATA
 #define WM_COPYGLOBALDATA 0x0049
#endif
#ifndef WM_UNICHAR
 #define WM_UNICHAR 0x0109
#endif
#ifndef UNICODE_NOCHAR
 #define UNICODE_NOCHAR 0xFFFF
#endif
#ifndef WM_DPICHANGED
 #define WM_DPICHANGED 0x02E0
#endif
#ifndef GET_XBUTTON_WPARAM
 #define GET_XBUTTON_WPARAM(w) (HIWORD(w))
#endif
#ifndef EDS_ROTATEDMODE
 #define EDS_ROTATEDMODE 0x00000004
#endif
#ifndef DISPLAY_DEVICE_ACTIVE
 #define DISPLAY_DEVICE_ACTIVE 0x00000001
#endif
#ifndef _WIN32_WINNT_WINBLUE
 #define _WIN32_WINNT_WINBLUE 0x0603
#endif
#ifndef _WIN32_WINNT_WIN8
 #define _WIN32_WINNT_WIN8 0x0602
#endif
#ifndef WM_GETDPISCALEDSIZE
 #define WM_GETDPISCALEDSIZE 0x02e4
#endif
#ifndef USER_DEFAULT_SCREEN_DPI
 #define USER_DEFAULT_SCREEN_DPI 96
#endif
#ifndef OCR_HAND
 #define OCR_HAND 32649
#endif

#if WINVER < 0x0601
typedef struct
{
    DWORD cbSize;
    DWORD ExtStatus;
} CHANGEFILTERSTRUCT;
#ifndef MSGFLT_ALLOW
 #define MSGFLT_ALLOW 1
#endif
#endif /*Windows 7*/

#if WINVER < 0x0600
#define DWM_BB_ENABLE 0x00000001
#define DWM_BB_BLURREGION 0x00000002
typedef struct
{
    DWORD dwFlags;
    BOOL fEnable;
    HRGN hRgnBlur;
    BOOL fTransitionOnMaximized;
} DWM_BLURBEHIND;
#else
 #include <dwmapi.h>
#endif /*Windows Vista*/

#ifndef DPI_ENUMS_DECLARED
typedef enum
{
    PROCESS_DPI_UNAWARE = 0,
    PROCESS_SYSTEM_DPI_AWARE = 1,
    PROCESS_PER_MONITOR_DPI_AWARE = 2
} PROCESS_DPI_AWARENESS;
typedef enum
{
    MDT_EFFECTIVE_DPI = 0,
    MDT_ANGULAR_DPI = 1,
    MDT_RAW_DPI = 2,
    MDT_DEFAULT = MDT_EFFECTIVE_DPI
} MONITOR_DPI_TYPE;
#endif /*DPI_ENUMS_DECLARED*/

#ifndef DPI_AWARENESS_CONTEXT_PER_MONITOR_AWARE_V2
#define DPI_AWARENESS_CONTEXT_PER_MONITOR_AWARE_V2 ((HANDLE) -4)
#endif /*DPI_AWARENESS_CONTEXT_PER_MONITOR_AWARE_V2*/

// Replacement for versionhelpers.h macros, as we cannot rely on the
// application having a correct embedded manifest
//
#define IsWindowsVistaOrGreater()                                     \
    _glfwIsWindowsVersionOrGreaterWin32(HIBYTE(_WIN32_WINNT_VISTA),   \
                                        LOBYTE(_WIN32_WINNT_VISTA), 0)
#define IsWindows7OrGreater()                                         \
    _glfwIsWindowsVersionOrGreaterWin32(HIBYTE(_WIN32_WINNT_WIN7),    \
                                        LOBYTE(_WIN32_WINNT_WIN7), 0)
#define IsWindows8OrGreater()                                         \
    _glfwIsWindowsVersionOrGreaterWin32(HIBYTE(_WIN32_WINNT_WIN8),    \
                                        LOBYTE(_WIN32_WINNT_WIN8), 0)
#define IsWindows8Point1OrGreater()                                   \
    _glfwIsWindowsVersionOrGreaterWin32(HIBYTE(_WIN32_WINNT_WINBLUE), \
                                        LOBYTE(_WIN32_WINNT_WINBLUE), 0)

// Windows 10 Anniversary Update
#define _glfwIsWindows10Version1607OrGreaterWin32() \
    _glfwIsWindows10BuildOrGreaterWin32(14393)
// Windows 10 Creators Update
#define _glfwIsWindows10Version1703OrGreaterWin32() \
    _glfwIsWindows10BuildOrGreaterWin32(15063)

// HACK: Define macros that some xinput.h variants don't
#ifndef XINPUT_CAPS_WIRELESS
 #define XINPUT_CAPS_WIRELESS 0x0002
#endif
#ifndef XINPUT_DEVSUBTYPE_WHEEL
 #define XINPUT_DEVSUBTYPE_WHEEL 0x02
#endif
#ifndef XINPUT_DEVSUBTYPE_ARCADE_STICK
 #define XINPUT_DEVSUBTYPE_ARCADE_STICK 0x03
#endif
#ifndef XINPUT_DEVSUBTYPE_FLIGHT_STICK
 #define XINPUT_DEVSUBTYPE_FLIGHT_STICK 0x04
#endif
#ifndef XINPUT_DEVSUBTYPE_DANCE_PAD
 #define XINPUT_DEVSUBTYPE_DANCE_PAD 0x05
#endif
#ifndef XINPUT_DEVSUBTYPE_GUITAR
 #define XINPUT_DEVSUBTYPE_GUITAR 0x06
#endif
#ifndef XINPUT_DEVSUBTYPE_DRUM_KIT
 #define XINPUT_DEVSUBTYPE_DRUM_KIT 0x08
#endif
#ifndef XINPUT_DEVSUBTYPE_ARCADE_PAD
 #define XINPUT_DEVSUBTYPE_ARCADE_PAD 0x13
#endif
#ifndef XUSER_MAX_COUNT
 #define XUSER_MAX_COUNT 4
#endif

// HACK: Define macros that some dinput.h variants don't
#ifndef DIDFT_OPTIONAL
 #define DIDFT_OPTIONAL 0x80000000
#endif

#define WGL_NUMBER_PIXEL_FORMATS_ARB 0x2000
#define WGL_SUPPORT_OPENGL_ARB 0x2010
#define WGL_DRAW_TO_WINDOW_ARB 0x2001
#define WGL_PIXEL_TYPE_ARB 0x2013
#define WGL_TYPE_RGBA_ARB 0x202b
#define WGL_ACCELERATION_ARB 0x2003
#define WGL_NO_ACCELERATION_ARB 0x2025
#define WGL_RED_BITS_ARB 0x2015
#define WGL_RED_SHIFT_ARB 0x2016
#define WGL_GREEN_BITS_ARB 0x2017
#define WGL_GREEN_SHIFT_ARB 0x2018
#define WGL_BLUE_BITS_ARB 0x2019
#define WGL_BLUE_SHIFT_ARB 0x201a
#define WGL_ALPHA_BITS_ARB 0x201b
#define WGL_ALPHA_SHIFT_ARB 0x201c
#define WGL_ACCUM_BITS_ARB 0x201d
#define WGL_ACCUM_RED_BITS_ARB 0x201e
#define WGL_ACCUM_GREEN_BITS_ARB 0x201f
#define WGL_ACCUM_BLUE_BITS_ARB 0x2020
#define WGL_ACCUM_ALPHA_BITS_ARB 0x2021
#define WGL_DEPTH_BITS_ARB 0x2022
#define WGL_STENCIL_BITS_ARB 0x2023
#define WGL_AUX_BUFFERS_ARB 0x2024
#define WGL_STEREO_ARB 0x2012
#define WGL_DOUBLE_BUFFER_ARB 0x2011
#define WGL_SAMPLES_ARB 0x2042
#define WGL_FRAMEBUFFER_SRGB_CAPABLE_ARB 0x20a9
#define WGL_CONTEXT_DEBUG_BIT_ARB 0x00000001
#define WGL_CONTEXT_FORWARD_COMPATIBLE_BIT_ARB 0x00000002
#define WGL_CONTEXT_PROFILE_MASK_ARB 0x9126
#define WGL_CONTEXT_CORE_PROFILE_BIT_ARB 0x00000001
#define WGL_CONTEXT_COMPATIBILITY_PROFILE_BIT_ARB 0x00000002
#define WGL_CONTEXT_MAJOR_VERSION_ARB 0x2091
#define WGL_CONTEXT_MINOR_VERSION_ARB 0x2092
#define WGL_CONTEXT_FLAGS_ARB 0x2094
#define WGL_CONTEXT_ES2_PROFILE_BIT_EXT 0x00000004
#define WGL_CONTEXT_ROBUST_ACCESS_BIT_ARB 0x00000004
#define WGL_LOSE_CONTEXT_ON_RESET_ARB 0x8252
#define WGL_CONTEXT_RESET_NOTIFICATION_STRATEGY_ARB 0x8256
#define WGL_NO_RESET_NOTIFICATION_ARB 0x8261
#define WGL_CONTEXT_RELEASE_BEHAVIOR_ARB 0x2097
#define WGL_CONTEXT_RELEASE_BEHAVIOR_NONE_ARB 0
#define WGL_CONTEXT_RELEASE_BEHAVIOR_FLUSH_ARB 0x2098
#define WGL_CONTEXT_OPENGL_NO_ERROR_ARB 0x31b3
#define WGL_COLORSPACE_EXT 0x309d
#define WGL_COLORSPACE_SRGB_EXT 0x3089

#define ERROR_INVALID_VERSION_ARB 0x2095
#define ERROR_INVALID_PROFILE_ARB 0x2096
#define ERROR_INCOMPATIBLE_DEVICE_CONTEXTS_ARB 0x2054

// xinput.dll function pointer typedefs
typedef DWORD (WINAPI * PFN_XInputGetCapabilities)(DWORD,DWORD,XINPUT_CAPABILITIES*);
typedef DWORD (WINAPI * PFN_XInputGetState)(DWORD,XINPUT_STATE*);
#define XInputGetCapabilities _glfw.win32.xinput.GetCapabilities
#define XInputGetState _glfw.win32.xinput.GetState

// dinput8.dll function pointer typedefs
typedef HRESULT (WINAPI * PFN_DirectInput8Create)(HINSTANCE,DWORD,REFIID,LPVOID*,LPUNKNOWN);
#define DirectInput8Create _glfw.win32.dinput8.Create

// user32.dll function pointer typedefs
typedef BOOL (WINAPI * PFN_SetProcessDPIAware)(void);
typedef BOOL (WINAPI * PFN_ChangeWindowMessageFilterEx)(HWND,UINT,DWORD,CHANGEFILTERSTRUCT*);
typedef BOOL (WINAPI * PFN_EnableNonClientDpiScaling)(HWND);
typedef BOOL (WINAPI * PFN_SetProcessDpiAwarenessContext)(HANDLE);
typedef UINT (WINAPI * PFN_GetDpiForWindow)(HWND);
typedef BOOL (WINAPI * PFN_AdjustWindowRectExForDpi)(LPRECT,DWORD,BOOL,DWORD,UINT);
typedef int (WINAPI * PFN_GetSystemMetricsForDpi)(int,UINT);
#define SetProcessDPIAware _glfw.win32.user32.SetProcessDPIAware_
#define ChangeWindowMessageFilterEx _glfw.win32.user32.ChangeWindowMessageFilterEx_
#define EnableNonClientDpiScaling _glfw.win32.user32.EnableNonClientDpiScaling_
#define SetProcessDpiAwarenessContext _glfw.win32.user32.SetProcessDpiAwarenessContext_
#define GetDpiForWindow _glfw.win32.user32.GetDpiForWindow_
#define AdjustWindowRectExForDpi _glfw.win32.user32.AdjustWindowRectExForDpi_
#define GetSystemMetricsForDpi _glfw.win32.user32.GetSystemMetricsForDpi_

// dwmapi.dll function pointer typedefs
typedef HRESULT (WINAPI * PFN_DwmIsCompositionEnabled)(BOOL*);
typedef HRESULT (WINAPI * PFN_DwmFlush)(VOID);
typedef HRESULT(WINAPI * PFN_DwmEnableBlurBehindWindow)(HWND,const DWM_BLURBEHIND*);
typedef HRESULT (WINAPI * PFN_DwmGetColorizationColor)(DWORD*,BOOL*);
#define DwmIsCompositionEnabled _glfw.win32.dwmapi.IsCompositionEnabled
#define DwmFlush _glfw.win32.dwmapi.Flush
#define DwmEnableBlurBehindWindow _glfw.win32.dwmapi.EnableBlurBehindWindow
#define DwmGetColorizationColor _glfw.win32.dwmapi.GetColorizationColor

// shcore.dll function pointer typedefs
typedef HRESULT (WINAPI * PFN_SetProcessDpiAwareness)(PROCESS_DPI_AWARENESS);
typedef HRESULT (WINAPI * PFN_GetDpiForMonitor)(HMONITOR,MONITOR_DPI_TYPE,UINT*,UINT*);
#define SetProcessDpiAwareness _glfw.win32.shcore.SetProcessDpiAwareness_
#define GetDpiForMonitor _glfw.win32.shcore.GetDpiForMonitor_

// ntdll.dll function pointer typedefs
typedef LONG (WINAPI * PFN_RtlVerifyVersionInfo)(OSVERSIONINFOEXW*,ULONG,ULONGLONG);
#define RtlVerifyVersionInfo _glfw.win32.ntdll.RtlVerifyVersionInfo_

// WGL extension pointer typedefs
typedef BOOL (WINAPI * PFNWGLSWAPINTERVALEXTPROC)(int);
typedef BOOL (WINAPI * PFNWGLGETPIXELFORMATATTRIBIVARBPROC)(HDC,int,int,UINT,const int*,int*);
typedef const char* (WINAPI * PFNWGLGETEXTENSIONSSTRINGEXTPROC)(void);
typedef const char* (WINAPI * PFNWGLGETEXTENSIONSSTRINGARBPROC)(HDC);
typedef HGLRC (WINAPI * PFNWGLCREATECONTEXTATTRIBSARBPROC)(HDC,HGLRC,const int*);
#define wglSwapIntervalEXT _glfw.wgl.SwapIntervalEXT
#define wglGetPixelFormatAttribivARB _glfw.wgl.GetPixelFormatAttribivARB
#define wglGetExtensionsStringEXT _glfw.wgl.GetExtensionsStringEXT
#define wglGetExtensionsStringARB _glfw.wgl.GetExtensionsStringARB
#define wglCreateContextAttribsARB _glfw.wgl.CreateContextAttribsARB

// opengl32.dll function pointer typedefs
typedef HGLRC (WINAPI * PFN_wglCreateContext)(HDC);
typedef BOOL (WINAPI * PFN_wglDeleteContext)(HGLRC);
typedef PROC (WINAPI * PFN_wglGetProcAddress)(LPCSTR);
typedef HDC (WINAPI * PFN_wglGetCurrentDC)(void);
typedef HGLRC (WINAPI * PFN_wglGetCurrentContext)(void);
typedef BOOL (WINAPI * PFN_wglMakeCurrent)(HDC,HGLRC);
typedef BOOL (WINAPI * PFN_wglShareLists)(HGLRC,HGLRC);
#define wglCreateContext _glfw.wgl.CreateContext
#define wglDeleteContext _glfw.wgl.DeleteContext
#define wglGetProcAddress _glfw.wgl.GetProcAddress
#define wglGetCurrentDC _glfw.wgl.GetCurrentDC
#define wglGetCurrentContext _glfw.wgl.GetCurrentContext
#define wglMakeCurrent _glfw.wgl.MakeCurrent
#define wglShareLists _glfw.wgl.ShareLists

#define GLFW_WIN32_WINDOW_STATE         _GLFWwindowWin32  win32;
#define GLFW_WIN32_LIBRARY_WINDOW_STATE _GLFWlibraryWin32 win32;
#define GLFW_WIN32_MONITOR_STATE        _GLFWmonitorWin32 win32;
#define GLFW_WIN32_CURSOR_STATE         _GLFWcursorWin32  win32;

#define GLFW_WGL_CONTEXT_STATE          _GLFWcontextWGL wgl;
#define GLFW_WGL_LIBRARY_CONTEXT_STATE  _GLFWlibraryWGL wgl;


// WGL-specific per-context data
//
typedef struct _GLFWcontextWGL
{
    HDC       dc;
    HGLRC     handle;
    int       interval;
} _GLFWcontextWGL;

// WGL-specific global data
//
typedef struct _GLFWlibraryWGL
{
    HINSTANCE                           instance;
    PFN_wglCreateContext                CreateContext;
    PFN_wglDeleteContext                DeleteContext;
    PFN_wglGetProcAddress               GetProcAddress;
    PFN_wglGetCurrentDC                 GetCurrentDC;
    PFN_wglGetCurrentContext            GetCurrentContext;
    PFN_wglMakeCurrent                  MakeCurrent;
    PFN_wglShareLists                   ShareLists;

    PFNWGLSWAPINTERVALEXTPROC           SwapIntervalEXT;
    PFNWGLGETPIXELFORMATATTRIBIVARBPROC GetPixelFormatAttribivARB;
    PFNWGLGETEXTENSIONSSTRINGEXTPROC    GetExtensionsStringEXT;
    PFNWGLGETEXTENSIONSSTRINGARBPROC    GetExtensionsStringARB;
    PFNWGLCREATECONTEXTATTRIBSARBPROC   CreateContextAttribsARB;
    GLFWbool                            EXT_swap_control;
    GLFWbool                            EXT_colorspace;
    GLFWbool                            ARB_multisample;
    GLFWbool                            ARB_framebuffer_sRGB;
    GLFWbool                            EXT_framebuffer_sRGB;
    GLFWbool                            ARB_pixel_format;
    GLFWbool                            ARB_create_context;
    GLFWbool                            ARB_create_context_profile;
    GLFWbool                            EXT_create_context_es2_profile;
    GLFWbool                            ARB_create_context_robustness;
    GLFWbool                            ARB_create_context_no_error;
    GLFWbool                            ARB_context_flush_control;
} _GLFWlibraryWGL;

// Win32-specific per-window data
//
typedef struct _GLFWwindowWin32
{
    HWND                handle;
    HICON               bigIcon;
    HICON               smallIcon;

    GLFWbool            cursorTracked;
    GLFWbool            frameAction;
    GLFWbool            iconified;
    GLFWbool            maximized;
    // Whether to enable framebuffer transparency on DWM
    GLFWbool            transparent;
    GLFWbool            scaleToMonitor;
    GLFWbool            keymenu;
    GLFWbool            showDefault;

    // Cached size used to filter out duplicate events
    int                 width, height;

    // The last received cursor position, regardless of source
    int                 lastCursorPosX, lastCursorPosY;
    // The last received high surrogate when decoding pairs of UTF-16 messages
    WCHAR               highSurrogate;
} _GLFWwindowWin32;

// Win32-specific global data
//
typedef struct _GLFWlibraryWin32
{
    HINSTANCE           instance;
    HWND                helperWindowHandle;
    ATOM                helperWindowClass;
    ATOM                mainWindowClass;
    HDEVNOTIFY          deviceNotificationHandle;
    int                 acquiredMonitorCount;
    char*               clipboardString;
    short int           keycodes[512];
    short int           scancodes[GLFW_KEY_LAST + 1];
    char                keynames[GLFW_KEY_LAST + 1][5];
    // Where to place the cursor when re-enabled
    double              restoreCursorPosX, restoreCursorPosY;
    // The window whose disabled cursor mode is active
    _GLFWwindow*        disabledCursorWindow;
    // The window the cursor is captured in
    _GLFWwindow*        capturedCursorWindow;
    RAWINPUT*           rawInput;
    int                 rawInputSize;
    UINT                mouseTrailSize;
    // The cursor handle to use to hide the cursor (NULL or a transparent cursor)
    HCURSOR             blankCursor;

    struct {
        HINSTANCE                       instance;
        PFN_DirectInput8Create          Create;
        IDirectInput8W*                 api;
    } dinput8;

    struct {
        HINSTANCE                       instance;
        PFN_XInputGetCapabilities       GetCapabilities;
        PFN_XInputGetState              GetState;
    } xinput;

    struct {
        HINSTANCE                       instance;
        PFN_SetProcessDPIAware          SetProcessDPIAware_;
        PFN_ChangeWindowMessageFilterEx ChangeWindowMessageFilterEx_;
        PFN_EnableNonClientDpiScaling   EnableNonClientDpiScaling_;
        PFN_SetProcessDpiAwarenessContext SetProcessDpiAwarenessContext_;
        PFN_GetDpiForWindow             GetDpiForWindow_;
        PFN_AdjustWindowRectExForDpi    AdjustWindowRectExForDpi_;
        PFN_GetSystemMetricsForDpi      GetSystemMetricsForDpi_;
    } user32;

    struct {
        HINSTANCE                       instance;
        PFN_DwmIsCompositionEnabled     IsCompositionEnabled;
        PFN_DwmFlush                    Flush;
        PFN_DwmEnableBlurBehindWindow   EnableBlurBehindWindow;
        PFN_DwmGetColorizationColor     GetColorizationColor;
    } dwmapi;

    struct {
        HINSTANCE                       instance;
        PFN_SetProcessDpiAwareness      SetProcessDpiAwareness_;
        PFN_GetDpiForMonitor            GetDpiForMonitor_;
    } shcore;

    struct {
        HINSTANCE                       instance;
        PFN_RtlVerifyVersionInfo        RtlVerifyVersionInfo_;
    } ntdll;
} _GLFWlibraryWin32;

// Win32-specific per-monitor data
//
typedef struct _GLFWmonitorWin32
{
    HMONITOR            handle;
    // This size matches the static size of DISPLAY_DEVICE.DeviceName
    char                adapterName[32];
    char                displayName[32];
    GLFWbool            modesPruned;
    GLFWbool            modeChanged;
} _GLFWmonitorWin32;

// Win32-specific per-cursor data
//
typedef struct _GLFWcursorWin32
{
    HCURSOR             handle;
} _GLFWcursorWin32;


GLFWbool _glfwConnectWin32(int platformID, _GLFWplatform* platform);
int _glfwInitWin32(void);
void _glfwTerminateWin32(void);

BOOL _glfwIsWindowsVersionOrGreaterWin32(WORD major, WORD minor, WORD sp);
BOOL _glfwIsWindows10BuildOrGreaterWin32(WORD build);
void _glfwInputErrorWin32(int error, const char* description);

void _glfwPollMonitorsWin32(void);
void _glfwSetVideoModeWin32(_GLFWmonitor* monitor, const GLFWvidmode* desired);
void _glfwRestoreVideoModeWin32(_GLFWmonitor* monitor);
void _glfwGetHMONITORContentScaleWin32(HMONITOR handle, float* xscale, float* yscale);

GLFWbool _glfwCreateWindowWin32(_GLFWwindow* window, const _GLFWwndconfig* wndconfig, const _GLFWctxconfig* ctxconfig, const _GLFWfbconfig* fbconfig);
void _glfwDestroyWindowWin32(_GLFWwindow* window);
void _glfwSetWindowTitleWin32(_GLFWwindow* window, const char* title);
void _glfwSetWindowIconWin32(_GLFWwindow* window, int count, const GLFWimage* images);
void _glfwGetWindowPosWin32(_GLFWwindow* window, int* xpos, int* ypos);
void _glfwSetWindowPosWin32(_GLFWwindow* window, int xpos, int ypos);
void _glfwGetWindowSizeWin32(_GLFWwindow* window, int* width, int* height);
void _glfwSetWindowSizeWin32(_GLFWwindow* window, int width, int height);
void _glfwSetWindowSizeLimitsWin32(_GLFWwindow* window, int minwidth, int minheight, int maxwidth, int maxheight);
void _glfwSetWindowAspectRatioWin32(_GLFWwindow* window, int numer, int denom);
void _glfwGetFramebufferSizeWin32(_GLFWwindow* window, int* width, int* height);
void _glfwGetWindowFrameSizeWin32(_GLFWwindow* window, int* left, int* top, int* right, int* bottom);
void _glfwGetWindowContentScaleWin32(_GLFWwindow* window, float* xscale, float* yscale);
void _glfwIconifyWindowWin32(_GLFWwindow* window);
void _glfwRestoreWindowWin32(_GLFWwindow* window);
void _glfwMaximizeWindowWin32(_GLFWwindow* window);
void _glfwShowWindowWin32(_GLFWwindow* window);
void _glfwHideWindowWin32(_GLFWwindow* window);
void _glfwRequestWindowAttentionWin32(_GLFWwindow* window);
void _glfwFocusWindowWin32(_GLFWwindow* window);
void _glfwSetWindowMonitorWin32(_GLFWwindow* window, _GLFWmonitor* monitor, int xpos, int ypos, int width, int height, int refreshRate);
GLFWbool _glfwWindowFocusedWin32(_GLFWwindow* window);
GLFWbool _glfwWindowIconifiedWin32(_GLFWwindow* window);
GLFWbool _glfwWindowVisibleWin32(_GLFWwindow* window);
GLFWbool _glfwWindowMaximizedWin32(_GLFWwindow* window);
GLFWbool _glfwWindowHoveredWin32(_GLFWwindow* window);
GLFWbool _glfwFramebufferTransparentWin32(_GLFWwindow* window);
void _glfwSetWindowResizableWin32(_GLFWwindow* window, GLFWbool enabled);
void _glfwSetWindowDecoratedWin32(_GLFWwindow* window, GLFWbool enabled);
void _glfwSetWindowFloatingWin32(_GLFWwindow* window, GLFWbool enabled);
void _glfwSetWindowMousePassthroughWin32(_GLFWwindow* window, GLFWbool enabled);
float _glfwGetWindowOpacityWin32(_GLFWwindow* window);
void _glfwSetWindowOpacityWin32(_GLFWwindow* window, float opacity);

void _glfwSetRawMouseMotionWin32(_GLFWwindow *window, GLFWbool enabled);
GLFWbool _glfwRawMouseMotionSupportedWin32(void);

void _glfwPollEventsWin32(void);
void _glfwWaitEventsWin32(void);
void _glfwWaitEventsTimeoutWin32(double timeout);
void _glfwPostEmptyEventWin32(void);

void _glfwGetCursorPosWin32(_GLFWwindow* window, double* xpos, double* ypos);
void _glfwSetCursorPosWin32(_GLFWwindow* window, double xpos, double ypos);
void _glfwSetCursorModeWin32(_GLFWwindow* window, int mode);
const char* _glfwGetScancodeNameWin32(int scancode);
int _glfwGetKeyScancodeWin32(int key);
GLFWbool _glfwCreateCursorWin32(_GLFWcursor* cursor, const GLFWimage* image, int xhot, int yhot);
GLFWbool _glfwCreateStandardCursorWin32(_GLFWcursor* cursor, int shape);
void _glfwDestroyCursorWin32(_GLFWcursor* cursor);
void _glfwSetCursorWin32(_GLFWwindow* window, _GLFWcursor* cursor);
void _glfwSetClipboardStringWin32(const char* string);
const char* _glfwGetClipboardStringWin32(void);

EGLenum _glfwGetEGLPlatformWin32(EGLint** attribs);
EGLNativeDisplayType _glfwGetEGLNativeDisplayWin32(void);
EGLNativeWindowType _glfwGetEGLNativeWindowWin32(_GLFWwindow* window);

void _glfwGetRequiredInstanceExtensionsWin32(char** extensions);

void _glfwFreeMonitorWin32(_GLFWmonitor* monitor);
void _glfwGetMonitorPosWin32(_GLFWmonitor* monitor, int* xpos, int* ypos);
void _glfwGetMonitorContentScaleWin32(_GLFWmonitor* monitor, float* xscale, float* yscale);
void _glfwGetMonitorWorkareaWin32(_GLFWmonitor* monitor, int* xpos, int* ypos, int* width, int* height);
GLFWvidmode* _glfwGetVideoModesWin32(_GLFWmonitor* monitor, int* count);
GLFWbool _glfwGetVideoModeWin32(_GLFWmonitor* monitor, GLFWvidmode* mode);
GLFWbool _glfwGetGammaRampWin32(_GLFWmonitor* monitor, GLFWgammaramp* ramp);
void _glfwSetGammaRampWin32(_GLFWmonitor* monitor, const GLFWgammaramp* ramp);

const char* _glfwGetMappingNameWin32(void);

GLFWbool _glfwInitWGL(void);
void _glfwTerminateWGL(void);
GLFWbool _glfwCreateContextWGL(_GLFWwindow* window,
                               const _GLFWctxconfig* ctxconfig,
                               const _GLFWfbconfig* fbconfig);


 #define GLFW_EXPOSE_NATIVE_WIN32
 #define GLFW_EXPOSE_NATIVE_WGL
#else
 #define GLFW_WIN32_WINDOW_STATE
 #define GLFW_WIN32_MONITOR_STATE
 #define GLFW_WIN32_CURSOR_STATE
 #define GLFW_WIN32_LIBRARY_WINDOW_STATE
 #define GLFW_WGL_CONTEXT_STATE
 #define GLFW_WGL_LIBRARY_CONTEXT_STATE
#endif

#if defined(_GLFW_COCOA)
 /* #include "external/glfw/src/cocoa_platform.h" */
//========================================================================
// GLFW 3.4 macOS - www.glfw.org
//------------------------------------------------------------------------
// Copyright (c) 2009-2019 Camilla Löwy <elmindreda@glfw.org>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any damages
// arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it
// freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you must not
//    claim that you wrote the original software. If you use this software
//    in a product, an acknowledgment in the product documentation would
//    be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such, and must not
//    be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source
//    distribution.
//
//========================================================================

#include <stdint.h>

#include <Carbon/Carbon.h>
#include <IOKit/hid/IOHIDLib.h>

// NOTE: All of NSGL was deprecated in the 10.14 SDK
//       This disables the pointless warnings for every symbol we use
#ifndef GL_SILENCE_DEPRECATION
#define GL_SILENCE_DEPRECATION
#endif

#if defined(__OBJC__)
#import <Cocoa/Cocoa.h>
#else
typedef void* id;
#endif

// NOTE: Many Cocoa enum values have been renamed and we need to build across
//       SDK versions where one is unavailable or deprecated.
//       We use the newer names in code and replace them with the older names if
//       the base SDK does not provide the newer names.

#if MAC_OS_X_VERSION_MAX_ALLOWED < 101400
 #define NSOpenGLContextParameterSwapInterval NSOpenGLCPSwapInterval
 #define NSOpenGLContextParameterSurfaceOpacity NSOpenGLCPSurfaceOpacity
#endif

#if MAC_OS_X_VERSION_MAX_ALLOWED < 101200
 #define NSBitmapFormatAlphaNonpremultiplied NSAlphaNonpremultipliedBitmapFormat
 #define NSEventMaskAny NSAnyEventMask
 #define NSEventMaskKeyUp NSKeyUpMask
 #define NSEventModifierFlagCapsLock NSAlphaShiftKeyMask
 #define NSEventModifierFlagCommand NSCommandKeyMask
 #define NSEventModifierFlagControl NSControlKeyMask
 #define NSEventModifierFlagDeviceIndependentFlagsMask NSDeviceIndependentModifierFlagsMask
 #define NSEventModifierFlagOption NSAlternateKeyMask
 #define NSEventModifierFlagShift NSShiftKeyMask
 #define NSEventTypeApplicationDefined NSApplicationDefined
 #define NSWindowStyleMaskBorderless NSBorderlessWindowMask
 #define NSWindowStyleMaskClosable NSClosableWindowMask
 #define NSWindowStyleMaskMiniaturizable NSMiniaturizableWindowMask
 #define NSWindowStyleMaskResizable NSResizableWindowMask
 #define NSWindowStyleMaskTitled NSTitledWindowMask
#endif

// NOTE: Many Cocoa dynamically linked constants have been renamed and we need
//       to build across SDK versions where one is unavailable or deprecated.
//       We use the newer names in code and replace them with the older names if
//       the deployment target is older than the newer names.

#if MAC_OS_X_VERSION_MIN_REQUIRED < 101300
 #define NSPasteboardTypeURL NSURLPboardType
#endif

typedef VkFlags VkMacOSSurfaceCreateFlagsMVK;
typedef VkFlags VkMetalSurfaceCreateFlagsEXT;

typedef struct VkMacOSSurfaceCreateInfoMVK
{
    VkStructureType                 sType;
    const void*                     pNext;
    VkMacOSSurfaceCreateFlagsMVK    flags;
    const void*                     pView;
} VkMacOSSurfaceCreateInfoMVK;

typedef struct VkMetalSurfaceCreateInfoEXT
{
    VkStructureType                 sType;
    const void*                     pNext;
    VkMetalSurfaceCreateFlagsEXT    flags;
    const void*                     pLayer;
} VkMetalSurfaceCreateInfoEXT;

typedef VkResult (APIENTRY *PFN_vkCreateMacOSSurfaceMVK)(VkInstance,const VkMacOSSurfaceCreateInfoMVK*,const VkAllocationCallbacks*,VkSurfaceKHR*);
typedef VkResult (APIENTRY *PFN_vkCreateMetalSurfaceEXT)(VkInstance,const VkMetalSurfaceCreateInfoEXT*,const VkAllocationCallbacks*,VkSurfaceKHR*);

#define GLFW_COCOA_WINDOW_STATE         _GLFWwindowNS  ns;
#define GLFW_COCOA_LIBRARY_WINDOW_STATE _GLFWlibraryNS ns;
#define GLFW_COCOA_MONITOR_STATE        _GLFWmonitorNS ns;
#define GLFW_COCOA_CURSOR_STATE         _GLFWcursorNS  ns;

#define GLFW_NSGL_CONTEXT_STATE         _GLFWcontextNSGL nsgl;
#define GLFW_NSGL_LIBRARY_CONTEXT_STATE _GLFWlibraryNSGL nsgl;

// HIToolbox.framework pointer typedefs
#define kTISPropertyUnicodeKeyLayoutData _glfw.ns.tis.kPropertyUnicodeKeyLayoutData
typedef TISInputSourceRef (*PFN_TISCopyCurrentKeyboardLayoutInputSource)(void);
#define TISCopyCurrentKeyboardLayoutInputSource _glfw.ns.tis.CopyCurrentKeyboardLayoutInputSource
typedef void* (*PFN_TISGetInputSourceProperty)(TISInputSourceRef,CFStringRef);
#define TISGetInputSourceProperty _glfw.ns.tis.GetInputSourceProperty
typedef UInt8 (*PFN_LMGetKbdType)(void);
#define LMGetKbdType _glfw.ns.tis.GetKbdType


// NSGL-specific per-context data
//
typedef struct _GLFWcontextNSGL
{
    id                pixelFormat;
    id                object;
} _GLFWcontextNSGL;

// NSGL-specific global data
//
typedef struct _GLFWlibraryNSGL
{
    // dlopen handle for OpenGL.framework (for glfwGetProcAddress)
    CFBundleRef     framework;
} _GLFWlibraryNSGL;

// Cocoa-specific per-window data
//
typedef struct _GLFWwindowNS
{
    id              object;
    id              delegate;
    id              view;
    id              layer;

    GLFWbool        maximized;
    GLFWbool        occluded;
    GLFWbool        scaleFramebuffer;

    // Cached window properties to filter out duplicate events
    int             width, height;
    int             fbWidth, fbHeight;
    float           xscale, yscale;

    // The total sum of the distances the cursor has been warped
    // since the last cursor motion event was processed
    // This is kept to counteract Cocoa doing the same internally
    double          cursorWarpDeltaX, cursorWarpDeltaY;
} _GLFWwindowNS;

// Cocoa-specific global data
//
typedef struct _GLFWlibraryNS
{
    CGEventSourceRef    eventSource;
    id                  delegate;
    GLFWbool            cursorHidden;
    TISInputSourceRef   inputSource;
    IOHIDManagerRef     hidManager;
    id                  unicodeData;
    id                  helper;
    id                  keyUpMonitor;
    id                  nibObjects;

    char                keynames[GLFW_KEY_LAST + 1][17];
    short int           keycodes[256];
    short int           scancodes[GLFW_KEY_LAST + 1];
    char*               clipboardString;
    CGPoint             cascadePoint;
    // Where to place the cursor when re-enabled
    double              restoreCursorPosX, restoreCursorPosY;
    // The window whose disabled cursor mode is active
    _GLFWwindow*        disabledCursorWindow;

    struct {
        CFBundleRef     bundle;
        PFN_TISCopyCurrentKeyboardLayoutInputSource CopyCurrentKeyboardLayoutInputSource;
        PFN_TISGetInputSourceProperty GetInputSourceProperty;
        PFN_LMGetKbdType GetKbdType;
        CFStringRef     kPropertyUnicodeKeyLayoutData;
    } tis;
} _GLFWlibraryNS;

// Cocoa-specific per-monitor data
//
typedef struct _GLFWmonitorNS
{
    CGDirectDisplayID   displayID;
    CGDisplayModeRef    previousMode;
    uint32_t            unitNumber;
    id                  screen;
    double              fallbackRefreshRate;
} _GLFWmonitorNS;

// Cocoa-specific per-cursor data
//
typedef struct _GLFWcursorNS
{
    id              object;
} _GLFWcursorNS;


GLFWbool _glfwConnectCocoa(int platformID, _GLFWplatform* platform);
int _glfwInitCocoa(void);
void _glfwTerminateCocoa(void);

GLFWbool _glfwCreateWindowCocoa(_GLFWwindow* window, const _GLFWwndconfig* wndconfig, const _GLFWctxconfig* ctxconfig, const _GLFWfbconfig* fbconfig);
void _glfwDestroyWindowCocoa(_GLFWwindow* window);
void _glfwSetWindowTitleCocoa(_GLFWwindow* window, const char* title);
void _glfwSetWindowIconCocoa(_GLFWwindow* window, int count, const GLFWimage* images);
void _glfwGetWindowPosCocoa(_GLFWwindow* window, int* xpos, int* ypos);
void _glfwSetWindowPosCocoa(_GLFWwindow* window, int xpos, int ypos);
void _glfwGetWindowSizeCocoa(_GLFWwindow* window, int* width, int* height);
void _glfwSetWindowSizeCocoa(_GLFWwindow* window, int width, int height);
void _glfwSetWindowSizeLimitsCocoa(_GLFWwindow* window, int minwidth, int minheight, int maxwidth, int maxheight);
void _glfwSetWindowAspectRatioCocoa(_GLFWwindow* window, int numer, int denom);
void _glfwGetFramebufferSizeCocoa(_GLFWwindow* window, int* width, int* height);
void _glfwGetWindowFrameSizeCocoa(_GLFWwindow* window, int* left, int* top, int* right, int* bottom);
void _glfwGetWindowContentScaleCocoa(_GLFWwindow* window, float* xscale, float* yscale);
void _glfwIconifyWindowCocoa(_GLFWwindow* window);
void _glfwRestoreWindowCocoa(_GLFWwindow* window);
void _glfwMaximizeWindowCocoa(_GLFWwindow* window);
void _glfwShowWindowCocoa(_GLFWwindow* window);
void _glfwHideWindowCocoa(_GLFWwindow* window);
void _glfwRequestWindowAttentionCocoa(_GLFWwindow* window);
void _glfwFocusWindowCocoa(_GLFWwindow* window);
void _glfwSetWindowMonitorCocoa(_GLFWwindow* window, _GLFWmonitor* monitor, int xpos, int ypos, int width, int height, int refreshRate);
GLFWbool _glfwWindowFocusedCocoa(_GLFWwindow* window);
GLFWbool _glfwWindowIconifiedCocoa(_GLFWwindow* window);
GLFWbool _glfwWindowVisibleCocoa(_GLFWwindow* window);
GLFWbool _glfwWindowMaximizedCocoa(_GLFWwindow* window);
GLFWbool _glfwWindowHoveredCocoa(_GLFWwindow* window);
GLFWbool _glfwFramebufferTransparentCocoa(_GLFWwindow* window);
void _glfwSetWindowResizableCocoa(_GLFWwindow* window, GLFWbool enabled);
void _glfwSetWindowDecoratedCocoa(_GLFWwindow* window, GLFWbool enabled);
void _glfwSetWindowFloatingCocoa(_GLFWwindow* window, GLFWbool enabled);
float _glfwGetWindowOpacityCocoa(_GLFWwindow* window);
void _glfwSetWindowOpacityCocoa(_GLFWwindow* window, float opacity);
void _glfwSetWindowMousePassthroughCocoa(_GLFWwindow* window, GLFWbool enabled);

void _glfwSetRawMouseMotionCocoa(_GLFWwindow *window, GLFWbool enabled);
GLFWbool _glfwRawMouseMotionSupportedCocoa(void);

void _glfwPollEventsCocoa(void);
void _glfwWaitEventsCocoa(void);
void _glfwWaitEventsTimeoutCocoa(double timeout);
void _glfwPostEmptyEventCocoa(void);

void _glfwGetCursorPosCocoa(_GLFWwindow* window, double* xpos, double* ypos);
void _glfwSetCursorPosCocoa(_GLFWwindow* window, double xpos, double ypos);
void _glfwSetCursorModeCocoa(_GLFWwindow* window, int mode);
const char* _glfwGetScancodeNameCocoa(int scancode);
int _glfwGetKeyScancodeCocoa(int key);
GLFWbool _glfwCreateCursorCocoa(_GLFWcursor* cursor, const GLFWimage* image, int xhot, int yhot);
GLFWbool _glfwCreateStandardCursorCocoa(_GLFWcursor* cursor, int shape);
void _glfwDestroyCursorCocoa(_GLFWcursor* cursor);
void _glfwSetCursorCocoa(_GLFWwindow* window, _GLFWcursor* cursor);
void _glfwSetClipboardStringCocoa(const char* string);
const char* _glfwGetClipboardStringCocoa(void);

EGLenum _glfwGetEGLPlatformCocoa(EGLint** attribs);
EGLNativeDisplayType _glfwGetEGLNativeDisplayCocoa(void);
EGLNativeWindowType _glfwGetEGLNativeWindowCocoa(_GLFWwindow* window);

void _glfwGetRequiredInstanceExtensionsCocoa(char** extensions);
GLFWbool _glfwGetPhysicalDevicePresentationSupportCocoa(VkInstance instance, VkPhysicalDevice device, uint32_t queuefamily);

void _glfwFreeMonitorCocoa(_GLFWmonitor* monitor);
void _glfwGetMonitorPosCocoa(_GLFWmonitor* monitor, int* xpos, int* ypos);
void _glfwGetMonitorContentScaleCocoa(_GLFWmonitor* monitor, float* xscale, float* yscale);
void _glfwGetMonitorWorkareaCocoa(_GLFWmonitor* monitor, int* xpos, int* ypos, int* width, int* height);
GLFWvidmode* _glfwGetVideoModesCocoa(_GLFWmonitor* monitor, int* count);
GLFWbool _glfwGetVideoModeCocoa(_GLFWmonitor* monitor, GLFWvidmode* mode);
GLFWbool _glfwGetGammaRampCocoa(_GLFWmonitor* monitor, GLFWgammaramp* ramp);
void _glfwSetGammaRampCocoa(_GLFWmonitor* monitor, const GLFWgammaramp* ramp);

void _glfwPollMonitorsCocoa(void);
void _glfwSetVideoModeCocoa(_GLFWmonitor* monitor, const GLFWvidmode* desired);
void _glfwRestoreVideoModeCocoa(_GLFWmonitor* monitor);

float _glfwTransformYCocoa(float y);

void* _glfwLoadLocalVulkanLoaderCocoa(void);

GLFWbool _glfwInitNSGL(void);
void _glfwTerminateNSGL(void);
GLFWbool _glfwCreateContextNSGL(_GLFWwindow* window,
                                const _GLFWctxconfig* ctxconfig,
                                const _GLFWfbconfig* fbconfig);
void _glfwDestroyContextNSGL(_GLFWwindow* window);


 #define GLFW_EXPOSE_NATIVE_COCOA
 #define GLFW_EXPOSE_NATIVE_NSGL
#else
 #define GLFW_COCOA_WINDOW_STATE
 #define GLFW_COCOA_MONITOR_STATE
 #define GLFW_COCOA_CURSOR_STATE
 #define GLFW_COCOA_LIBRARY_WINDOW_STATE
 #define GLFW_NSGL_CONTEXT_STATE
 #define GLFW_NSGL_LIBRARY_CONTEXT_STATE
#endif

#if defined(_GLFW_WAYLAND)
 /* #include "external/glfw/src/wl_platform.h" */
//========================================================================
// GLFW 3.4 Wayland - www.glfw.org
//------------------------------------------------------------------------
// Copyright (c) 2014 Jonas Ådahl <jadahl@gmail.com>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any damages
// arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it
// freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you must not
//    claim that you wrote the original software. If you use this software
//    in a product, an acknowledgment in the product documentation would
//    be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such, and must not
//    be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source
//    distribution.
//
//========================================================================

/* #include "external/wayland/wayland-client-core.h" */
/*
 * Copyright © 2008 Kristian Høgsberg
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#ifndef WAYLAND_CLIENT_CORE_H
#define WAYLAND_CLIENT_CORE_H

#include <stdint.h>
/* #include "external/wayland/wayland-util.h" */
/*
 * Copyright © 2008 Kristian Høgsberg
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

/** \file wayland-util.h
 *
 * \brief Utility classes, functions, and macros.
 */

#ifndef WAYLAND_UTIL_H
#define WAYLAND_UTIL_H

#include <math.h>
#include <stddef.h>
#include <inttypes.h>
#include <stdarg.h>

#ifdef  __cplusplus
extern "C" {
#endif

/** Visibility attribute */
#if defined(__GNUC__) && __GNUC__ >= 4
#define WL_EXPORT __attribute__ ((visibility("default")))
#else
#define WL_EXPORT
#endif

/** Deprecated attribute */
#if __STDC_VERSION__ >= 202311L
#define WL_DEPRECATED [[deprecated]]
#elif defined(__GNUC__) && __GNUC__ >= 4
#define WL_DEPRECATED __attribute__ ((deprecated))
#else
#define WL_DEPRECATED
#endif

/**
 * Printf-style argument attribute
 *
 * \param x Ordinality of the format string argument
 * \param y Ordinality of the argument to check against the format string
 *
 * \sa https://gcc.gnu.org/onlinedocs/gcc-3.2.1/gcc/Function-Attributes.html
 */
#if defined(__GNUC__) && __GNUC__ >= 4
#define WL_PRINTF(x, y) __attribute__((__format__(__printf__, x, y)))
#else
#define WL_PRINTF(x, y)
#endif

#if __STDC_VERSION__ >= 202311L
#define WL_TYPEOF(expr) typeof(expr)
#else
#define WL_TYPEOF(expr) __typeof__(expr)
#endif

/** \class wl_object
 *
 * \brief A protocol object.
 *
 * A `wl_object` is an opaque struct identifying the protocol object
 * underlying a `wl_proxy` or `wl_resource`.
 *
 * \note Functions accessing a `wl_object` are not normally used by client code.
 * Clients should normally use the higher level interface generated by the
 * scanner to interact with compositor objects.
 *
 */
struct wl_object;

/**
 * Protocol message signature
 *
 * A wl_message describes the signature of an actual protocol message, such as a
 * request or event, that adheres to the Wayland protocol wire format. The
 * protocol implementation uses a wl_message within its demarshal machinery for
 * decoding messages between a compositor and its clients. In a sense, a
 * wl_message is to a protocol message like a class is to an object.
 *
 * The `name` of a wl_message is the name of the corresponding protocol message.
 *
 * The `signature` is an ordered list of symbols representing the data types
 * of message arguments and, optionally, a protocol version and indicators for
 * nullability. A leading integer in the `signature` indicates the _since_
 * version of the protocol message. A `?` preceding a data type symbol indicates
 * that the following argument type is nullable. While it is a protocol violation
 * to send messages with non-nullable arguments set to `NULL`, event handlers in
 * clients might still get called with non-nullable object arguments set to
 * `NULL`. This can happen when the client destroyed the object being used as
 * argument on its side and an event referencing that object was sent before the
 * server knew about its destruction. As this race cannot be prevented, clients
 * should - as a general rule - program their event handlers such that they can
 * handle object arguments declared non-nullable being `NULL` gracefully.
 *
 * When no arguments accompany a message, `signature` is an empty string.
 *
 * Symbols:
 *
 * * `i`: int
 * * `u`: uint
 * * `f`: fixed
 * * `s`: string
 * * `o`: object
 * * `n`: new_id
 * * `a`: array
 * * `h`: fd
 * * `?`: following argument (`o` or `s`) is nullable
 *
 * While demarshaling primitive arguments is straightforward, when demarshaling
 * messages containing `object` or `new_id` arguments, the protocol
 * implementation often must determine the type of the object. The `types` of a
 * wl_message is an array of wl_interface references that correspond to `o` and
 * `n` arguments in `signature`, with `NULL` placeholders for arguments with
 * non-object types.
 *
 * Consider the protocol event wl_display `delete_id` that has a single `uint`
 * argument. The wl_message is:
 *
 * \code
 * { "delete_id", "u", [NULL] }
 * \endcode
 *
 * Here, the message `name` is `"delete_id"`, the `signature` is `"u"`, and the
 * argument `types` is `[NULL]`, indicating that the `uint` argument has no
 * corresponding wl_interface since it is a primitive argument.
 *
 * In contrast, consider a `wl_foo` interface supporting protocol request `bar`
 * that has existed since version 2, and has two arguments: a `uint` and an
 * object of type `wl_baz_interface` that may be `NULL`. Such a `wl_message`
 * might be:
 *
 * \code
 * { "bar", "2u?o", [NULL, &wl_baz_interface] }
 * \endcode
 *
 * Here, the message `name` is `"bar"`, and the `signature` is `"2u?o"`. Notice
 * how the `2` indicates the protocol version, the `u` indicates the first
 * argument type is `uint`, and the `?o` indicates that the second argument
 * is an object that may be `NULL`. Lastly, the argument `types` array indicates
 * that no wl_interface corresponds to the first argument, while the type
 * `wl_baz_interface` corresponds to the second argument.
 *
 * \sa wl_argument
 * \sa wl_interface
 * \sa <a href="https://wayland.freedesktop.org/docs/html/ch04.html#sect-Protocol-Wire-Format">Wire Format</a>
 */
struct wl_message {
	/** Message name */
	const char *name;
	/** Message signature */
	const char *signature;
	/** Object argument interfaces */
	const struct wl_interface **types;
};

/**
 * Protocol object interface
 *
 * A wl_interface describes the API of a protocol object defined in the Wayland
 * protocol specification. The protocol implementation uses a wl_interface
 * within its marshalling machinery for encoding client requests.
 *
 * The `name` of a wl_interface is the name of the corresponding protocol
 * interface, and `version` represents the version of the interface. The members
 * `method_count` and `event_count` represent the number of `methods` (requests)
 * and `events` in the respective wl_message members.
 *
 * For example, consider a protocol interface `foo`, marked as version `1`, with
 * two requests and one event.
 *
 * \code{.xml}
 * <interface name="foo" version="1">
 *   <request name="a"></request>
 *   <request name="b"></request>
 *   <event name="c"></event>
 * </interface>
 * \endcode
 *
 * Given two wl_message arrays `foo_requests` and `foo_events`, a wl_interface
 * for `foo` might be:
 *
 * \code
 * struct wl_interface foo_interface = {
 *         "foo", 1,
 *         2, foo_requests,
 *         1, foo_events
 * };
 * \endcode
 *
 * \note The server side of the protocol may define interface <em>implementation
 *       types</em> that incorporate the term `interface` in their name. Take
 *       care to not confuse these server-side `struct`s with a wl_interface
 *       variable whose name also ends in `interface`. For example, while the
 *       server may define a type `struct wl_foo_interface`, the client may
 *       define a `struct wl_interface wl_foo_interface`.
 *
 * \sa wl_message
 * \sa wl_proxy
 * \sa <a href="https://wayland.freedesktop.org/docs/html/ch04.html#sect-Protocol-Interfaces">Interfaces</a>
 * \sa <a href="https://wayland.freedesktop.org/docs/html/ch04.html#sect-Protocol-Versioning">Versioning</a>
 */
struct wl_interface {
	/** Interface name */
	const char *name;
	/** Interface version */
	int version;
	/** Number of methods (requests) */
	int method_count;
	/** Method (request) signatures */
	const struct wl_message *methods;
	/** Number of events */
	int event_count;
	/** Event signatures */
	const struct wl_message *events;
};

/** \class wl_list
 *
 * \brief Doubly-linked list
 *
 * On its own, an instance of `struct wl_list` represents the sentinel head of
 * a doubly-linked list, and must be initialized using wl_list_init().
 * When empty, the list head's `next` and `prev` members point to the list head
 * itself, otherwise `next` references the first element in the list, and `prev`
 * refers to the last element in the list.
 *
 * Use the `struct wl_list` type to represent both the list head and the links
 * between elements within the list. Use wl_list_empty() to determine if the
 * list is empty in O(1).
 *
 * All elements in the list must be of the same type. The element type must have
 * a `struct wl_list` member, often named `link` by convention. Prior to
 * insertion, there is no need to initialize an element's `link` - invoking
 * wl_list_init() on an individual list element's `struct wl_list` member is
 * unnecessary if the very next operation is wl_list_insert(). However, a
 * common idiom is to initialize an element's `link` prior to removal - ensure
 * safety by invoking wl_list_init() before wl_list_remove().
 *
 * Consider a list reference `struct wl_list foo_list`, an element type as
 * `struct element`, and an element's link member as `struct wl_list link`.
 *
 * The following code initializes a list and adds three elements to it.
 *
 * \code
 * struct wl_list foo_list;
 *
 * struct element {
 *         int foo;
 *         struct wl_list link;
 * };
 * struct element e1, e2, e3;
 *
 * wl_list_init(&foo_list);
 * wl_list_insert(&foo_list, &e1.link);   // e1 is the first element
 * wl_list_insert(&foo_list, &e2.link);   // e2 is now the first element
 * wl_list_insert(&e2.link, &e3.link); // insert e3 after e2
 * \endcode
 *
 * The list now looks like <em>[e2, e3, e1]</em>.
 *
 * The `wl_list` API provides some iterator macros. For example, to iterate
 * a list in ascending order:
 *
 * \code
 * struct element *e;
 * wl_list_for_each(e, foo_list, link) {
 *         do_something_with_element(e);
 * }
 * \endcode
 *
 * See the documentation of each iterator for details.
 * \sa http://git.kernel.org/cgit/linux/kernel/git/torvalds/linux.git/tree/include/linux/list.h
 */
struct wl_list {
	/** Previous list element */
	struct wl_list *prev;
	/** Next list element */
	struct wl_list *next;
};

/**
 * Initializes the list.
 *
 * \param list List to initialize
 *
 * \memberof wl_list
 */
void
wl_list_init(struct wl_list *list);

/**
 * Inserts an element into the list, after the element represented by \p list.
 * When \p list is a reference to the list itself (the head), set the containing
 * struct of \p elm as the first element in the list.
 *
 * \note If \p elm is already part of a list, inserting it again will lead to
 *       list corruption.
 *
 * \param list List element after which the new element is inserted
 * \param elm Link of the containing struct to insert into the list
 *
 * \memberof wl_list
 */
void
wl_list_insert(struct wl_list *list, struct wl_list *elm);

/**
 * Removes an element from the list.
 *
 * \note This operation leaves \p elm in an invalid state.
 *
 * \param elm Link of the containing struct to remove from the list
 *
 * \memberof wl_list
 */
void
wl_list_remove(struct wl_list *elm);

/**
 * Determines the length of the list.
 *
 * \note This is an O(n) operation.
 *
 * \param list List whose length is to be determined
 *
 * \return Number of elements in the list
 *
 * \memberof wl_list
 */
int
wl_list_length(const struct wl_list *list);

/**
 * Determines if the list is empty.
 *
 * \param list List whose emptiness is to be determined
 *
 * \return 1 if empty, or 0 if not empty
 *
 * \memberof wl_list
 */
int
wl_list_empty(const struct wl_list *list);

/**
 * Inserts all of the elements of one list into another, after the element
 * represented by \p list.
 *
 * \note This leaves \p other in an invalid state.
 *
 * \param list List element after which the other list elements will be inserted
 * \param other List of elements to insert
 *
 * \memberof wl_list
 */
void
wl_list_insert_list(struct wl_list *list, struct wl_list *other);

/**
 * Retrieves a pointer to a containing struct, given a member name.
 *
 * This macro allows "conversion" from a pointer to a member to its containing
 * struct. This is useful if you have a contained item like a wl_list,
 * wl_listener, or wl_signal, provided via a callback or other means, and would
 * like to retrieve the struct that contains it.
 *
 * To demonstrate, the following example retrieves a pointer to
 * `example_container` given only its `destroy_listener` member:
 *
 * \code
 * struct example_container {
 *         struct wl_listener destroy_listener;
 *         // other members...
 * };
 *
 * void example_container_destroy(struct wl_listener *listener, void *data)
 * {
 *         struct example_container *ctr;
 *
 *         ctr = wl_container_of(listener, ctr, destroy_listener);
 *         // destroy ctr...
 * }
 * \endcode
 *
 * \note `sample` need not be a valid pointer. A null or uninitialised pointer
 *       is sufficient.
 *
 * \param ptr Valid pointer to the contained member
 * \param sample Pointer to a struct whose type contains \p ptr
 * \param member Named location of \p ptr within the \p sample type
 *
 * \return The container for the specified pointer
 */
#define wl_container_of(ptr, sample, member)				\
	(WL_TYPEOF(sample))((char *)(ptr) -				\
			     offsetof(WL_TYPEOF(*sample), member))

/**
 * Iterates over a list.
 *
 * This macro expresses a for-each iterator for wl_list. Given a list and
 * wl_list link member name (often named `link` by convention), this macro
 * assigns each element in the list to \p pos, which can then be referenced in
 * a trailing code block. For example, given a wl_list of `struct message`
 * elements:
 *
 * \code
 * struct message {
 *         char *contents;
 *         wl_list link;
 * };
 *
 * struct wl_list *message_list;
 * // Assume message_list now "contains" many messages
 *
 * struct message *m;
 * wl_list_for_each(m, message_list, link) {
 *         do_something_with_message(m);
 * }
 * \endcode
 *
 * \param pos Cursor that each list element will be assigned to
 * \param head Head of the list to iterate over
 * \param member Name of the link member within the element struct
 *
 * \relates wl_list
 */
#define wl_list_for_each(pos, head, member)				\
	for (pos = wl_container_of((head)->next, pos, member);	\
	     &pos->member != (head);					\
	     pos = wl_container_of(pos->member.next, pos, member))

/**
 * Iterates over a list, safe against removal of the list element.
 *
 * \note Only removal of the current element, \p pos, is safe. Removing
 *       any other element during traversal may lead to a loop malfunction.
 *
 * \sa wl_list_for_each()
 *
 * \param pos Cursor that each list element will be assigned to
 * \param tmp Temporary pointer of the same type as \p pos
 * \param head Head of the list to iterate over
 * \param member Name of the link member within the element struct
 *
 * \relates wl_list
 */
#define wl_list_for_each_safe(pos, tmp, head, member)			\
	for (pos = wl_container_of((head)->next, pos, member),		\
	     tmp = wl_container_of((pos)->member.next, tmp, member);	\
	     &pos->member != (head);					\
	     pos = tmp,							\
	     tmp = wl_container_of(pos->member.next, tmp, member))

/**
 * Iterates backwards over a list.
 *
 * \sa wl_list_for_each()
 *
 * \param pos Cursor that each list element will be assigned to
 * \param head Head of the list to iterate over
 * \param member Name of the link member within the element struct
 *
 * \relates wl_list
 */
#define wl_list_for_each_reverse(pos, head, member)			\
	for (pos = wl_container_of((head)->prev, pos, member);	\
	     &pos->member != (head);					\
	     pos = wl_container_of(pos->member.prev, pos, member))

/**
 * Iterates backwards over a list, safe against removal of the list element.
 *
 * \note Only removal of the current element, \p pos, is safe. Removing
 *       any other element during traversal may lead to a loop malfunction.
 *
 * \sa wl_list_for_each()
 *
 * \param pos Cursor that each list element will be assigned to
 * \param tmp Temporary pointer of the same type as \p pos
 * \param head Head of the list to iterate over
 * \param member Name of the link member within the element struct
 *
 * \relates wl_list
 */
#define wl_list_for_each_reverse_safe(pos, tmp, head, member)		\
	for (pos = wl_container_of((head)->prev, pos, member),	\
	     tmp = wl_container_of((pos)->member.prev, tmp, member);	\
	     &pos->member != (head);					\
	     pos = tmp,							\
	     tmp = wl_container_of(pos->member.prev, tmp, member))

/**
 * \class wl_array
 *
 * Dynamic array
 *
 * A wl_array is a dynamic array that can only grow until released. It is
 * intended for relatively small allocations whose size is variable or not known
 * in advance. While construction of a wl_array does not require all elements to
 * be of the same size, wl_array_for_each() does require all elements to have
 * the same type and size.
 *
 */
struct wl_array {
	/** Array size */
	size_t size;
	/** Allocated space */
	size_t alloc;
	/** Array data */
	void *data;
};

/**
 * Initializes the array.
 *
 * \param array Array to initialize
 *
 * \memberof wl_array
 */
void
wl_array_init(struct wl_array *array);

/**
 * Releases the array data.
 *
 * \note Leaves the array in an invalid state.
 *
 * \param array Array whose data is to be released
 *
 * \memberof wl_array
 */
void
wl_array_release(struct wl_array *array);

/**
 * Increases the size of the array by \p size bytes.
 *
 * \param array Array whose size is to be increased
 * \param size Number of bytes to increase the size of the array by
 *
 * \return A pointer to the beginning of the newly appended space, or NULL when
 *         resizing fails.
 *
 * \memberof wl_array
 */
void *
wl_array_add(struct wl_array *array, size_t size);

/**
 * Copies the contents of \p source to \p array.
 *
 * \param array Destination array to copy to
 * \param source Source array to copy from
 *
 * \return 0 on success, or -1 on failure
 *
 * \memberof wl_array
 */
int
wl_array_copy(struct wl_array *array, struct wl_array *source);

/**
 * Iterates over an array.
 *
 * This macro expresses a for-each iterator for wl_array. It assigns each
 * element in the array to \p pos, which can then be referenced in a trailing
 * code block. \p pos must be a pointer to the array element type, and all
 * array elements must be of the same type and size.
 *
 * \param pos Cursor that each array element will be assigned to
 * \param array Array to iterate over
 *
 * \relates wl_array
 * \sa wl_list_for_each()
 */
#define wl_array_for_each(pos, array)					\
	for (pos = (array)->data;					\
	     (array)->size != 0 &&					\
	     (const char *) pos < ((const char *) (array)->data + (array)->size); \
	     (pos)++)

/**
 * Fixed-point number
 *
 * A `wl_fixed_t` is a 24.8 signed fixed-point number with a sign bit, 23 bits
 * of integer precision and 8 bits of decimal precision. Consider `wl_fixed_t`
 * as an opaque struct with methods that facilitate conversion to and from
 * `double` and `int` types.
 */
typedef int32_t wl_fixed_t;

/**
 * Converts a fixed-point number to a floating-point number.
 *
 * \param f Fixed-point number to convert
 *
 * \return Floating-point representation of the fixed-point argument
 */
static inline double
wl_fixed_to_double(wl_fixed_t f)
{
	return f / 256.0;
}

/**
 * Converts a floating-point number to a fixed-point number.
 *
 * \param d Floating-point number to convert
 *
 * \return Fixed-point representation of the floating-point argument
 */
static inline wl_fixed_t
wl_fixed_from_double(double d)
{
	return (wl_fixed_t) (d * 256.0);
}

/**
 * Converts a fixed-point number to an integer.
 *
 * \param f Fixed-point number to convert
 *
 * \return Integer component of the fixed-point argument
 */
static inline int
wl_fixed_to_int(wl_fixed_t f)
{
	return f / 256;
}

/**
 * Converts an integer to a fixed-point number.
 *
 * \param i Integer to convert
 *
 * \return Fixed-point representation of the integer argument
 */
static inline wl_fixed_t
wl_fixed_from_int(int i)
{
	return i * 256;
}

/**
 * Protocol message argument data types
 *
 * This union represents all of the argument types in the Wayland protocol wire
 * format. The protocol implementation uses wl_argument within its marshalling
 * machinery for dispatching messages between a client and a compositor.
 *
 * \sa wl_message
 * \sa wl_interface
 * \sa <a href="https://wayland.freedesktop.org/docs/html/ch04.html#sect-Protocol-wire-Format">Wire Format</a>
 */
union wl_argument {
	int32_t i;           /**< `int`    */
	uint32_t u;          /**< `uint`   */
	wl_fixed_t f;        /**< `fixed`  */
	const char *s;       /**< `string` */
	struct wl_object *o; /**< `object` */
	uint32_t n;          /**< `new_id` */
	struct wl_array *a;  /**< `array`  */
	int32_t h;           /**< `fd`     */
};

/**
 * Dispatcher function type alias
 *
 * A dispatcher is a function that handles the emitting of callbacks in client
 * code. For programs directly using the C library, this is done by using
 * libffi to call function pointers. When binding to languages other than C,
 * dispatchers provide a way to abstract the function calling process to be
 * friendlier to other function calling systems.
 *
 * A dispatcher takes five arguments: The first is the dispatcher-specific
 * implementation associated with the target object. The second is the object
 * upon which the callback is being invoked (either wl_proxy or wl_resource).
 * The third and fourth arguments are the opcode and the wl_message
 * corresponding to the callback. The final argument is an array of arguments
 * received from the other process via the wire protocol.
 *
 * \param user_data Dispatcher-specific implementation data
 * \param target Callback invocation target (wl_proxy or `wl_resource`)
 * \param opcode Callback opcode
 * \param msg Callback message signature
 * \param args Array of received arguments
 *
 * \return 0 on success, or -1 on failure
 */
typedef int (*wl_dispatcher_func_t)(const void *user_data, void *target,
				    uint32_t opcode, const struct wl_message *msg,
				    union wl_argument *args);

/**
 * Log function type alias
 *
 * The C implementation of the Wayland protocol abstracts the details of
 * logging. Users may customize the logging behavior, with a function conforming
 * to the `wl_log_func_t` type, via `wl_log_set_handler_client` and
 * `wl_log_set_handler_server`.
 *
 * A `wl_log_func_t` must conform to the expectations of `vprintf`, and
 * expects two arguments: a string to write and a corresponding variable
 * argument list. While the string to write may contain format specifiers and
 * use values in the variable argument list, the behavior of any `wl_log_func_t`
 * depends on the implementation.
 *
 * \note Take care to not confuse this with `wl_protocol_logger_func_t`, which
 *       is a specific server-side logger for requests and events.
 *
 * \param fmt String to write to the log, containing optional format
 *            specifiers
 * \param args Variable argument list
 *
 * \sa wl_log_set_handler_client
 * \sa wl_log_set_handler_server
 */
typedef void (*wl_log_func_t)(const char *fmt, va_list args) WL_PRINTF(1, 0);

/**
 * Return value of an iterator function
 *
 * \sa wl_client_for_each_resource_iterator_func_t
 * \sa wl_client_for_each_resource
 */
enum wl_iterator_result {
	/** Stop the iteration */
	WL_ITERATOR_STOP,
	/** Continue the iteration */
	WL_ITERATOR_CONTINUE
};

#ifdef  __cplusplus
}
#endif

#endif

/* #include "external/wayland/wayland-version.h" */
/*
 * Copyright © 2012 Intel Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#ifndef WAYLAND_VERSION_H
#define WAYLAND_VERSION_H

#define WAYLAND_VERSION_MAJOR 1
#define WAYLAND_VERSION_MINOR 23
#define WAYLAND_VERSION_MICRO 1
#define WAYLAND_VERSION "1.23.1"

#endif


#ifdef  __cplusplus
extern "C" {
#endif

/** \class wl_proxy
 *
 * \brief Represents a protocol object on the client side.
 *
 * A wl_proxy acts as a client side proxy to an object existing in the
 * compositor. The proxy is responsible for converting requests made by the
 * clients with \ref wl_proxy_marshal() into Wayland's wire format. Events
 * coming from the compositor are also handled by the proxy, which will in
 * turn call the handler set with \ref wl_proxy_add_listener().
 *
 * \note With the exception of function \ref wl_proxy_set_queue(), functions
 * accessing a wl_proxy are not normally used by client code. Clients
 * should normally use the higher level interface generated by the scanner to
 * interact with compositor objects.
 *
 */
struct wl_proxy;

/** \class wl_display
 *
 * \brief Represents a connection to the compositor and acts as a proxy to
 * the wl_display singleton object.
 *
 * A wl_display object represents a client connection to a Wayland
 * compositor. It is created with either \ref wl_display_connect() or
 * \ref wl_display_connect_to_fd(). A connection is terminated using
 * \ref wl_display_disconnect().
 *
 * A wl_display is also used as the \ref wl_proxy for the wl_display
 * singleton object on the compositor side.
 *
 * A wl_display object handles all the data sent from and to the
 * compositor. When a \ref wl_proxy marshals a request, it will write its wire
 * representation to the display's write buffer. The data is sent to the
 * compositor when the client calls \ref wl_display_flush().
 *
 * Incoming data is handled in two steps: queueing and dispatching. In the
 * queue step, the data coming from the display fd is interpreted and
 * added to a queue. On the dispatch step, the handler for the incoming
 * event set by the client on the corresponding \ref wl_proxy is called.
 *
 * A wl_display has at least one event queue, called the <em>default
 * queue</em>. Clients can create additional event queues with \ref
 * wl_display_create_queue() and assign \ref wl_proxy's to it. Events
 * occurring in a particular proxy are always queued in its assigned queue.
 * A client can ensure that a certain assumption, such as holding a lock
 * or running from a given thread, is true when a proxy event handler is
 * called by assigning that proxy to an event queue and making sure that
 * this queue is only dispatched when the assumption holds.
 *
 * The default queue is dispatched by calling \ref wl_display_dispatch().
 * This will dispatch any events queued on the default queue and attempt
 * to read from the display fd if it's empty. Events read are then queued
 * on the appropriate queues according to the proxy assignment.
 *
 * A user created queue is dispatched with \ref wl_display_dispatch_queue().
 * This function behaves exactly the same as wl_display_dispatch()
 * but it dispatches given queue instead of the default queue.
 *
 * A real world example of event queue usage is Mesa's implementation of
 * eglSwapBuffers() for the Wayland platform. This function might need
 * to block until a frame callback is received, but dispatching the default
 * queue could cause an event handler on the client to start drawing
 * again. This problem is solved using another event queue, so that only
 * the events handled by the EGL code are dispatched during the block.
 *
 * This creates a problem where a thread dispatches a non-default
 * queue, reading all the data from the display fd. If the application
 * would call \em poll(2) after that it would block, even though there
 * might be events queued on the default queue. Those events should be
 * dispatched with \ref wl_display_dispatch_pending() or \ref
 * wl_display_dispatch_queue_pending() before flushing and blocking.
 */
struct wl_display;

/** \class wl_event_queue
 *
 * \brief A queue for \ref wl_proxy object events.
 *
 * Event queues allows the events on a display to be handled in a thread-safe
 * manner. See \ref wl_display for details.
 *
 */
struct wl_event_queue;

/** Destroy proxy after marshalling
 * \relates wl_proxy
 */
#define WL_MARSHAL_FLAG_DESTROY (1 << 0)

void
wl_event_queue_destroy(struct wl_event_queue *queue);

struct wl_proxy *
wl_proxy_marshal_flags(struct wl_proxy *proxy, uint32_t opcode,
		       const struct wl_interface *interface,
		       uint32_t version,
		       uint32_t flags, ...);

struct wl_proxy *
wl_proxy_marshal_array_flags(struct wl_proxy *proxy, uint32_t opcode,
			     const struct wl_interface *interface,
			     uint32_t version,
			     uint32_t flags,
			     union wl_argument *args);

void
wl_proxy_marshal(struct wl_proxy *p, uint32_t opcode, ...);

void
wl_proxy_marshal_array(struct wl_proxy *p, uint32_t opcode,
		       union wl_argument *args);

struct wl_proxy *
wl_proxy_create(struct wl_proxy *factory,
		const struct wl_interface *interface);

void *
wl_proxy_create_wrapper(void *proxy);

void
wl_proxy_wrapper_destroy(void *proxy_wrapper);

struct wl_proxy *
wl_proxy_marshal_constructor(struct wl_proxy *proxy,
			     uint32_t opcode,
			     const struct wl_interface *interface,
			     ...);

struct wl_proxy *
wl_proxy_marshal_constructor_versioned(struct wl_proxy *proxy,
				       uint32_t opcode,
				       const struct wl_interface *interface,
				       uint32_t version,
				       ...);

struct wl_proxy *
wl_proxy_marshal_array_constructor(struct wl_proxy *proxy,
				   uint32_t opcode, union wl_argument *args,
				   const struct wl_interface *interface);

struct wl_proxy *
wl_proxy_marshal_array_constructor_versioned(struct wl_proxy *proxy,
					     uint32_t opcode,
					     union wl_argument *args,
					     const struct wl_interface *interface,
					     uint32_t version);

void
wl_proxy_destroy(struct wl_proxy *proxy);

int
wl_proxy_add_listener(struct wl_proxy *proxy,
		      void (**implementation)(void), void *data);

const void *
wl_proxy_get_listener(struct wl_proxy *proxy);

int
wl_proxy_add_dispatcher(struct wl_proxy *proxy,
			wl_dispatcher_func_t dispatcher_func,
			const void * dispatcher_data, void *data);

void
wl_proxy_set_user_data(struct wl_proxy *proxy, void *user_data);

void *
wl_proxy_get_user_data(struct wl_proxy *proxy);

uint32_t
wl_proxy_get_version(struct wl_proxy *proxy);

uint32_t
wl_proxy_get_id(struct wl_proxy *proxy);

void
wl_proxy_set_tag(struct wl_proxy *proxy,
		 const char * const *tag);

const char * const *
wl_proxy_get_tag(struct wl_proxy *proxy);

const char *
wl_proxy_get_class(struct wl_proxy *proxy);

struct wl_display *
wl_proxy_get_display(struct wl_proxy *proxy);

void
wl_proxy_set_queue(struct wl_proxy *proxy, struct wl_event_queue *queue);

struct wl_event_queue *
wl_proxy_get_queue(const struct wl_proxy *proxy);

const char *
wl_event_queue_get_name(const struct wl_event_queue *queue);

struct wl_display *
wl_display_connect(const char *name);

struct wl_display *
wl_display_connect_to_fd(int fd);

void
wl_display_disconnect(struct wl_display *display);

int
wl_display_get_fd(struct wl_display *display);

int
wl_display_dispatch(struct wl_display *display);

int
wl_display_dispatch_queue(struct wl_display *display,
			  struct wl_event_queue *queue);

int
wl_display_dispatch_queue_pending(struct wl_display *display,
				  struct wl_event_queue *queue);

int
wl_display_dispatch_pending(struct wl_display *display);

int
wl_display_get_error(struct wl_display *display);

uint32_t
wl_display_get_protocol_error(struct wl_display *display,
			      const struct wl_interface **interface,
			      uint32_t *id);

int
wl_display_flush(struct wl_display *display);

int
wl_display_roundtrip_queue(struct wl_display *display,
			   struct wl_event_queue *queue);

int
wl_display_roundtrip(struct wl_display *display);

struct wl_event_queue *
wl_display_create_queue(struct wl_display *display);

struct wl_event_queue *
wl_display_create_queue_with_name(struct wl_display *display,
				  const char *name);

int
wl_display_prepare_read_queue(struct wl_display *display,
			      struct wl_event_queue *queue);

int
wl_display_prepare_read(struct wl_display *display);

void
wl_display_cancel_read(struct wl_display *display);

int
wl_display_read_events(struct wl_display *display);

void
wl_log_set_handler_client(wl_log_func_t handler);

void
wl_display_set_max_buffer_size(struct wl_display *display,
                               size_t max_buffer_size);

#ifdef  __cplusplus
}
#endif

#endif

/* #include "external/xkbcommon/xkbcommon.h" */

/* #include "external/xkbcommon/xkbcommon-compose.h" */
/*
 * Copyright © 2013 Ran Benita
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#ifndef _XKBCOMMON_COMPOSE_H
#define _XKBCOMMON_COMPOSE_H

/* #include "external/xkbcommon/xkbcommon.h" */


#ifdef __cplusplus
extern "C" {
#endif

/**
 * @file
 * libxkbcommon Compose API - support for Compose and dead-keys.
 */

/**
 * @defgroup compose Compose and dead-keys support
 * Support for Compose and dead-keys.
 * @since 0.5.0
 *
 * @{
 */

/**
 * @page compose-overview Overview
 * @parblock
 *
 * Compose and dead-keys are a common feature of many keyboard input
 * systems.  They extend the range of the keysysm that can be produced
 * directly from a keyboard by using a sequence of key strokes, instead
 * of just one.
 *
 * Here are some example sequences, in the libX11 Compose file format:
 *
 *     <dead_acute> <a>         : "á"   aacute # LATIN SMALL LETTER A WITH ACUTE
 *     <Multi_key> <A> <T>      : "@"   at     # COMMERCIAL AT
 *
 * When the user presses a key which produces the `<dead_acute>` keysym,
 * nothing initially happens (thus the key is dubbed a "dead-key").  But
 * when the user enters `<a>`, "á" is "composed", in place of "a".  If
 * instead the user had entered a keysym which does not follow
 * `<dead_acute>` in any compose sequence, the sequence is said to be
 * "cancelled".
 *
 * Compose files define many such sequences.  For a description of the
 * common file format for Compose files, see the Compose(5) man page.
 *
 * A successfuly-composed sequence has two results: a keysym and a UTF-8
 * string.  At least one of the two is defined for each sequence.  If only
 * a keysym is given, the keysym's string representation is used for the
 * result string (using xkb_keysym_to_utf8()).
 *
 * This library provides low-level support for Compose file parsing and
 * processing.  Higher-level APIs (such as libX11's `Xutf8LookupString`(3))
 * may be built upon it, or it can be used directly.
 *
 * @endparblock
 */

/**
 * @page compose-conflicting Conflicting Sequences
 * @parblock
 *
 * To avoid ambiguity, a sequence is not allowed to be a prefix of another.
 * In such a case, the conflict is resolved thus:
 *
 * 1. A longer sequence overrides a shorter one.
 * 2. An equal sequence overrides an existing one.
 * 3. A shorter sequence does not override a longer one.
 *
 * Sequences of length 1 are allowed.
 *
 * @endparblock
 */

/**
 * @page compose-cancellation Cancellation Behavior
 * @parblock
 *
 * What should happen when a sequence is cancelled?  For example, consider
 * there are only the above sequences, and the input keysyms are
 * `<dead_acute> <b>`.  There are a few approaches:
 *
 * 1. Swallow the cancelling keysym; that is, no keysym is produced.
 *    This is the approach taken by libX11.
 * 2. Let the cancelling keysym through; that is, `<b>` is produced.
 * 3. Replay the entire sequence; that is, `<dead_acute> <b>` is produced.
 *    This is the approach taken by Microsoft Windows (approximately;
 *    instead of `<dead_acute>`, the underlying key is used.  This is
 *    difficult to simulate with XKB keymaps).
 *
 * You can program whichever approach best fits users' expectations.
 *
 * @endparblock
 */

/**
 * @struct xkb_compose_table
 * Opaque Compose table object.
 *
 * The compose table holds the definitions of the Compose sequences, as
 * gathered from Compose files.  It is immutable.
 */
struct xkb_compose_table;

/**
 * @struct xkb_compose_state
 * Opaque Compose state object.
 *
 * The compose state maintains state for compose sequence matching, such
 * as which possible sequences are being matched, and the position within
 * these sequences.  It acts as a simple state machine wherein keysyms are
 * the input, and composed keysyms and strings are the output.
 *
 * The compose state is usually associated with a keyboard device.
 */
struct xkb_compose_state;

/** Flags affecting Compose file compilation. */
enum xkb_compose_compile_flags {
    /** Do not apply any flags. */
    XKB_COMPOSE_COMPILE_NO_FLAGS = 0
};

/** The recognized Compose file formats. */
enum xkb_compose_format {
    /** The classic libX11 Compose text format, described in Compose(5). */
    XKB_COMPOSE_FORMAT_TEXT_V1 = 1
};

/**
 * @page compose-locale Compose Locale
 * @parblock
 *
 * Compose files are locale dependent:
 * - Compose files are written for a locale, and the locale is used when
 *   searching for the appropriate file to use.
 * - Compose files may reference the locale internally, with directives
 *   such as \%L.
 *
 * As such, functions like xkb_compose_table_new_from_locale() require
 * a `locale` parameter.  This will usually be the current locale (see
 * locale(7) for more details).  You may also want to allow the user to
 * explicitly configure it, so he can use the Compose file of a given
 * locale, but not use that locale for other things.
 *
 * You may query the current locale as follows:
 * @code
 *     const char *locale;
 *     locale = setlocale(LC_CTYPE, NULL);
 * @endcode
 *
 * This will only give useful results if the program had previously set
 * the current locale using setlocale(3), with `LC_CTYPE` or `LC_ALL`
 * and a non-NULL argument.
 *
 * If you prefer not to use the locale system of the C runtime library,
 * you may nevertheless obtain the user's locale directly using
 * environment variables, as described in locale(7).  For example,
 * @code
 *     const char *locale;
 *     locale = getenv("LC_ALL");
 *     if (!locale || !*locale)
 *         locale = getenv("LC_CTYPE");
 *     if (!locale || !*locale)
 *         locale = getenv("LANG");
 *     if (!locale || !*locale)
 *         locale = "C";
 * @endcode
 *
 * Note that some locales supported by the C standard library may not
 * have a Compose file assigned.
 *
 * @endparblock
 */

/**
 * Create a compose table for a given locale.
 *
 * The locale is used for searching the file-system for an appropriate
 * Compose file.  The search order is described in Compose(5).  It is
 * affected by the following environment variables:
 *
 * 1. `XCOMPOSEFILE` - see Compose(5).
 * 2. `XDG_CONFIG_HOME` - before `$HOME/.XCompose` is checked,
 *    `$XDG_CONFIG_HOME/XCompose` is checked (with a fall back to
 *    `$HOME/.config/XCompose` if `XDG_CONFIG_HOME` is not defined).
 *    This is a libxkbcommon extension to the search procedure in
 *    Compose(5) (since libxkbcommon 1.0.0). Note that other
 *    implementations, such as libX11, might not find a Compose file in
 *    this path.
 * 3. `HOME` - see Compose(5).
 * 4. `XLOCALEDIR` - if set, used as the base directory for the system's
 *    X locale files, e.g. `/usr/share/X11/locale`, instead of the
 *    preconfigured directory.
 *
 * @param context
 *     The library context in which to create the compose table.
 * @param locale
 *     The current locale.  See @ref compose-locale.
 *     \n
 *     The value is copied, so it is safe to pass the result of getenv(3)
 *     (or similar) without fear of it being invalidated by a subsequent
 *     setenv(3) (or similar).
 * @param flags
 *     Optional flags for the compose table, or 0.
 *
 * @returns A compose table for the given locale, or NULL if the
 * compilation failed or a Compose file was not found.
 *
 * @memberof xkb_compose_table
 */
struct xkb_compose_table *
xkb_compose_table_new_from_locale(struct xkb_context *context,
                                  const char *locale,
                                  enum xkb_compose_compile_flags flags);

/**
 * Create a new compose table from a Compose file.
 *
 * @param context
 *     The library context in which to create the compose table.
 * @param file
 *     The Compose file to compile.
 * @param locale
 *     The current locale.  See @ref compose-locale.
 * @param format
 *     The text format of the Compose file to compile.
 * @param flags
 *     Optional flags for the compose table, or 0.
 *
 * @returns A compose table compiled from the given file, or NULL if
 * the compilation failed.
 *
 * @memberof xkb_compose_table
 */
struct xkb_compose_table *
xkb_compose_table_new_from_file(struct xkb_context *context,
                                FILE *file,
                                const char *locale,
                                enum xkb_compose_format format,
                                enum xkb_compose_compile_flags flags);

/**
 * Create a new compose table from a memory buffer.
 *
 * This is just like xkb_compose_table_new_from_file(), but instead of
 * a file, gets the table as one enormous string.
 *
 * @see xkb_compose_table_new_from_file()
 * @memberof xkb_compose_table
 */
struct xkb_compose_table *
xkb_compose_table_new_from_buffer(struct xkb_context *context,
                                  const char *buffer, size_t length,
                                  const char *locale,
                                  enum xkb_compose_format format,
                                  enum xkb_compose_compile_flags flags);

/**
 * Take a new reference on a compose table.
 *
 * @returns The passed in object.
 *
 * @memberof xkb_compose_table
 */
struct xkb_compose_table *
xkb_compose_table_ref(struct xkb_compose_table *table);

/**
 * Release a reference on a compose table, and possibly free it.
 *
 * @param table The object.  If it is NULL, this function does nothing.
 *
 * @memberof xkb_compose_table
 */
void
xkb_compose_table_unref(struct xkb_compose_table *table);

/**
 * @struct xkb_compose_table_entry
 * Opaque Compose table entry object.
 *
 * Represents a single entry in a Compose file in the iteration API.
 * It is immutable.
 *
 * @sa xkb_compose_table_iterator_new
 * @since 1.6.0
 */
struct xkb_compose_table_entry;

/**
 * Get the left-hand keysym sequence of a Compose table entry.
 *
 * For example, given the following entry:
 *
 * ```
 * <dead_tilde> <space> : "~" asciitilde # TILDE
 * ```
 *
 * it will return `{XKB_KEY_dead_tilde, XKB_KEY_space}`.
 *
 * @param[in]  entry The compose table entry object to process.
 *
 * @param[out] sequence_length Number of keysyms in the sequence.
 *
 * @returns The array of left-hand side keysyms.  The number of keysyms
 * is returned in the @p sequence_length out-parameter.
 *
 * @memberof xkb_compose_table_entry
 * @since 1.6.0
 */
const xkb_keysym_t *
xkb_compose_table_entry_sequence(struct xkb_compose_table_entry *entry,
                                 size_t *sequence_length);

/**
 * Get the right-hand result keysym of a Compose table entry.
 *
 * For example, given the following entry:
 *
 * ```
 * <dead_tilde> <space> : "~" asciitilde # TILDE
 * ```
 *
 * it will return `XKB_KEY_asciitilde`.
 *
 * The keysym is optional; if the entry does not specify a keysym,
 * returns `XKB_KEY_NoSymbol`.
 *
 * @memberof xkb_compose_table_entry
 * @since 1.6.0
 */
xkb_keysym_t
xkb_compose_table_entry_keysym(struct xkb_compose_table_entry *entry);

/**
 * Get the right-hand result string of a Compose table entry.
 *
 * The string is UTF-8 encoded and NULL-terminated.
 *
 * For example, given the following entry:
 *
 * ```
 * <dead_tilde> <space> : "~" asciitilde # TILDE
 * ```
 *
 * it will return `"~"`.
 *
 * The string is optional; if the entry does not specify a string,
 * returns the empty string.
 *
 * @memberof xkb_compose_table_entry
 * @since 1.6.0
 */
const char *
xkb_compose_table_entry_utf8(struct xkb_compose_table_entry *entry);

/**
 * @struct xkb_compose_table_iterator
 * Iterator over a compose table’s entries.
 *
 * @sa xkb_compose_table_iterator_new()
 * @since 1.6.0
 */
struct xkb_compose_table_iterator;

/**
 * Create a new iterator for a compose table.
 *
 * Intended use:
 *
 * ```c
 * struct xkb_compose_table_iterator *iter = xkb_compose_table_iterator_new(compose_table);
 * struct xkb_compose_table_entry *entry;
 * while ((entry = xkb_compose_table_iterator_next(iter))) {
 *     // ...
 * }
 * xkb_compose_table_iterator_free(iter);
 * ```
 *
 * @returns A new compose table iterator, or `NULL` on failure.
 *
 * @memberof xkb_compose_table_iterator
 * @sa xkb_compose_table_iterator_free()
 * @since 1.6.0
 */
struct xkb_compose_table_iterator *
xkb_compose_table_iterator_new(struct xkb_compose_table *table);

/**
 * Free a compose iterator.
 *
 * @memberof xkb_compose_table_iterator
 * @since 1.6.0
 */
void
xkb_compose_table_iterator_free(struct xkb_compose_table_iterator *iter);

/**
 * Get the next compose entry from a compose table iterator.
 *
 * The entries are returned in lexicographic order of the left-hand
 * side of entries. This does not correspond to the order in which
 * the entries appear in the Compose file.
 *
 * @attention The return value is valid until the next call to this function.
 *
 * Returns `NULL` in case there is no more entries.
 *
 * @memberof xkb_compose_table_iterator
 * @since 1.6.0
 */
struct xkb_compose_table_entry *
xkb_compose_table_iterator_next(struct xkb_compose_table_iterator *iter);

/** Flags for compose state creation. */
enum xkb_compose_state_flags {
    /** Do not apply any flags. */
    XKB_COMPOSE_STATE_NO_FLAGS = 0
};

/**
 * Create a new compose state object.
 *
 * @param table
 *     The compose table the state will use.
 * @param flags
 *     Optional flags for the compose state, or 0.
 *
 * @returns A new compose state, or NULL on failure.
 *
 * @memberof xkb_compose_state
 */
struct xkb_compose_state *
xkb_compose_state_new(struct xkb_compose_table *table,
                      enum xkb_compose_state_flags flags);

/**
 * Take a new reference on a compose state object.
 *
 * @returns The passed in object.
 *
 * @memberof xkb_compose_state
 */
struct xkb_compose_state *
xkb_compose_state_ref(struct xkb_compose_state *state);

/**
 * Release a reference on a compose state object, and possibly free it.
 *
 * @param state The object.  If NULL, do nothing.
 *
 * @memberof xkb_compose_state
 */
void
xkb_compose_state_unref(struct xkb_compose_state *state);

/**
 * Get the compose table which a compose state object is using.
 *
 * @returns The compose table which was passed to xkb_compose_state_new()
 * when creating this state object.
 *
 * This function does not take a new reference on the compose table; you
 * must explicitly reference it yourself if you plan to use it beyond the
 * lifetime of the state.
 *
 * @memberof xkb_compose_state
 */
struct xkb_compose_table *
xkb_compose_state_get_compose_table(struct xkb_compose_state *state);

/** Status of the Compose sequence state machine. */
enum xkb_compose_status {
    /** The initial state; no sequence has started yet. */
    XKB_COMPOSE_NOTHING,
    /** In the middle of a sequence. */
    XKB_COMPOSE_COMPOSING,
    /** A complete sequence has been matched. */
    XKB_COMPOSE_COMPOSED,
    /** The last sequence was cancelled due to an unmatched keysym. */
    XKB_COMPOSE_CANCELLED
};

/** The effect of a keysym fed to xkb_compose_state_feed(). */
enum xkb_compose_feed_result {
    /** The keysym had no effect - it did not affect the status. */
    XKB_COMPOSE_FEED_IGNORED,
    /** The keysym started, advanced or cancelled a sequence. */
    XKB_COMPOSE_FEED_ACCEPTED
};

/**
 * Feed one keysym to the Compose sequence state machine.
 *
 * This function can advance into a compose sequence, cancel a sequence,
 * start a new sequence, or do nothing in particular.  The resulting
 * status may be observed with xkb_compose_state_get_status().
 *
 * Some keysyms, such as keysyms for modifier keys, are ignored - they
 * have no effect on the status or otherwise.
 *
 * The following is a description of the possible status transitions, in
 * the format CURRENT STATUS => NEXT STATUS, given a non-ignored input
 * keysym `keysym`:
 *
   @verbatim
   NOTHING or CANCELLED or COMPOSED =>
      NOTHING   if keysym does not start a sequence.
      COMPOSING if keysym starts a sequence.
      COMPOSED  if keysym starts and terminates a single-keysym sequence.

   COMPOSING =>
      COMPOSING if keysym advances any of the currently possible
                sequences but does not terminate any of them.
      COMPOSED  if keysym terminates one of the currently possible
                sequences.
      CANCELLED if keysym does not advance any of the currently
                possible sequences.
   @endverbatim
 *
 * The current Compose formats do not support multiple-keysyms.
 * Therefore, if you are using a function such as xkb_state_key_get_syms()
 * and it returns more than one keysym, consider feeding XKB_KEY_NoSymbol
 * instead.
 *
 * @param state
 *     The compose state object.
 * @param keysym
 *     A keysym, usually obtained after a key-press event, with a
 *     function such as xkb_state_key_get_one_sym().
 *
 * @returns Whether the keysym was ignored.  This is useful, for example,
 * if you want to keep a record of the sequence matched thus far.
 *
 * @memberof xkb_compose_state
 */
enum xkb_compose_feed_result
xkb_compose_state_feed(struct xkb_compose_state *state,
                       xkb_keysym_t keysym);

/**
 * Reset the Compose sequence state machine.
 *
 * The status is set to XKB_COMPOSE_NOTHING, and the current sequence
 * is discarded.
 *
 * @memberof xkb_compose_state
 */
void
xkb_compose_state_reset(struct xkb_compose_state *state);

/**
 * Get the current status of the compose state machine.
 *
 * @see xkb_compose_status
 * @memberof xkb_compose_state
 **/
enum xkb_compose_status
xkb_compose_state_get_status(struct xkb_compose_state *state);

/**
 * Get the result Unicode/UTF-8 string for a composed sequence.
 *
 * See @ref compose-overview for more details.  This function is only
 * useful when the status is XKB_COMPOSE_COMPOSED.
 *
 * @param[in] state
 *     The compose state.
 * @param[out] buffer
 *     A buffer to write the string into.
 * @param[in] size
 *     Size of the buffer.
 *
 * @warning If the buffer passed is too small, the string is truncated
 * (though still NUL-terminated).
 *
 * @returns
 *   The number of bytes required for the string, excluding the NUL byte.
 *   If the sequence is not complete, or does not have a viable result
 *   string, returns 0, and sets `buffer` to the empty string (if possible).
 * @returns
 *   You may check if truncation has occurred by comparing the return value
 *   with the size of `buffer`, similarly to the `snprintf`(3) function.
 *   You may safely pass NULL and 0 to `buffer` and `size` to find the
 *   required size (without the NUL-byte).
 *
 * @memberof xkb_compose_state
 **/
int
xkb_compose_state_get_utf8(struct xkb_compose_state *state,
                           char *buffer, size_t size);

/**
 * Get the result keysym for a composed sequence.
 *
 * See @ref compose-overview for more details.  This function is only
 * useful when the status is XKB_COMPOSE_COMPOSED.
 *
 * @returns The result keysym.  If the sequence is not complete, or does
 * not specify a result keysym, returns XKB_KEY_NoSymbol.
 *
 * @memberof xkb_compose_state
 **/
xkb_keysym_t
xkb_compose_state_get_one_sym(struct xkb_compose_state *state);

/** @} */

#ifdef __cplusplus
} /* extern "C" */
#endif

#endif /* _XKBCOMMON_COMPOSE_H */

#include <stdbool.h>

typedef VkFlags VkWaylandSurfaceCreateFlagsKHR;

typedef struct VkWaylandSurfaceCreateInfoKHR
{
    VkStructureType                 sType;
    const void*                     pNext;
    VkWaylandSurfaceCreateFlagsKHR  flags;
    struct wl_display*              display;
    struct wl_surface*              surface;
} VkWaylandSurfaceCreateInfoKHR;

typedef VkResult (APIENTRY *PFN_vkCreateWaylandSurfaceKHR)(VkInstance,const VkWaylandSurfaceCreateInfoKHR*,const VkAllocationCallbacks*,VkSurfaceKHR*);
typedef VkBool32 (APIENTRY *PFN_vkGetPhysicalDeviceWaylandPresentationSupportKHR)(VkPhysicalDevice,uint32_t,struct wl_display*);

/* #include "external/glfw/src/xkb_unicode.h" */

/* #include "external/glfw/src/posix_poll.h" */


typedef int (* PFN_wl_display_flush)(struct wl_display* display);
typedef void (* PFN_wl_display_cancel_read)(struct wl_display* display);
typedef int (* PFN_wl_display_dispatch_pending)(struct wl_display* display);
typedef int (* PFN_wl_display_read_events)(struct wl_display* display);
typedef struct wl_display* (* PFN_wl_display_connect)(const char*);
typedef void (* PFN_wl_display_disconnect)(struct wl_display*);
typedef int (* PFN_wl_display_roundtrip)(struct wl_display*);
typedef int (* PFN_wl_display_get_fd)(struct wl_display*);
typedef int (* PFN_wl_display_prepare_read)(struct wl_display*);
typedef void (* PFN_wl_proxy_marshal)(struct wl_proxy*,uint32_t,...);
typedef int (* PFN_wl_proxy_add_listener)(struct wl_proxy*,void(**)(void),void*);
typedef void (* PFN_wl_proxy_destroy)(struct wl_proxy*);
typedef struct wl_proxy* (* PFN_wl_proxy_marshal_constructor)(struct wl_proxy*,uint32_t,const struct wl_interface*,...);
typedef struct wl_proxy* (* PFN_wl_proxy_marshal_constructor_versioned)(struct wl_proxy*,uint32_t,const struct wl_interface*,uint32_t,...);
typedef void* (* PFN_wl_proxy_get_user_data)(struct wl_proxy*);
typedef void (* PFN_wl_proxy_set_user_data)(struct wl_proxy*,void*);
typedef void (* PFN_wl_proxy_set_tag)(struct wl_proxy*,const char*const*);
typedef const char* const* (* PFN_wl_proxy_get_tag)(struct wl_proxy*);
typedef uint32_t (* PFN_wl_proxy_get_version)(struct wl_proxy*);
typedef struct wl_proxy* (* PFN_wl_proxy_marshal_flags)(struct wl_proxy*,uint32_t,const struct wl_interface*,uint32_t,uint32_t,...);
#define wl_display_flush _glfw.wl.client.display_flush
#define wl_display_cancel_read _glfw.wl.client.display_cancel_read
#define wl_display_dispatch_pending _glfw.wl.client.display_dispatch_pending
#define wl_display_read_events _glfw.wl.client.display_read_events
#define wl_display_disconnect _glfw.wl.client.display_disconnect
#define wl_display_roundtrip _glfw.wl.client.display_roundtrip
#define wl_display_get_fd _glfw.wl.client.display_get_fd
#define wl_display_prepare_read _glfw.wl.client.display_prepare_read
#define wl_proxy_marshal _glfw.wl.client.proxy_marshal
#define wl_proxy_add_listener _glfw.wl.client.proxy_add_listener
#define wl_proxy_destroy _glfw.wl.client.proxy_destroy
#define wl_proxy_marshal_constructor _glfw.wl.client.proxy_marshal_constructor
#define wl_proxy_marshal_constructor_versioned _glfw.wl.client.proxy_marshal_constructor_versioned
#define wl_proxy_get_user_data _glfw.wl.client.proxy_get_user_data
#define wl_proxy_set_user_data _glfw.wl.client.proxy_set_user_data
#define wl_proxy_get_tag _glfw.wl.client.proxy_get_tag
#define wl_proxy_set_tag _glfw.wl.client.proxy_set_tag
#define wl_proxy_get_version _glfw.wl.client.proxy_get_version
#define wl_proxy_marshal_flags _glfw.wl.client.proxy_marshal_flags

struct wl_shm;
struct wl_output;

#define wl_display_interface _glfw_wl_display_interface
#define wl_subcompositor_interface _glfw_wl_subcompositor_interface
#define wl_compositor_interface _glfw_wl_compositor_interface
#define wl_shm_interface _glfw_wl_shm_interface
#define wl_data_device_manager_interface _glfw_wl_data_device_manager_interface
#define wl_shell_interface _glfw_wl_shell_interface
#define wl_buffer_interface _glfw_wl_buffer_interface
#define wl_callback_interface _glfw_wl_callback_interface
#define wl_data_device_interface _glfw_wl_data_device_interface
#define wl_data_offer_interface _glfw_wl_data_offer_interface
#define wl_data_source_interface _glfw_wl_data_source_interface
#define wl_keyboard_interface _glfw_wl_keyboard_interface
#define wl_output_interface _glfw_wl_output_interface
#define wl_pointer_interface _glfw_wl_pointer_interface
#define wl_region_interface _glfw_wl_region_interface
#define wl_registry_interface _glfw_wl_registry_interface
#define wl_seat_interface _glfw_wl_seat_interface
#define wl_shell_surface_interface _glfw_wl_shell_surface_interface
#define wl_shm_pool_interface _glfw_wl_shm_pool_interface
#define wl_subsurface_interface _glfw_wl_subsurface_interface
#define wl_surface_interface _glfw_wl_surface_interface
#define wl_touch_interface _glfw_wl_touch_interface
#define zwp_idle_inhibitor_v1_interface _glfw_zwp_idle_inhibitor_v1_interface
#define zwp_idle_inhibit_manager_v1_interface _glfw_zwp_idle_inhibit_manager_v1_interface
#define zwp_confined_pointer_v1_interface _glfw_zwp_confined_pointer_v1_interface
#define zwp_locked_pointer_v1_interface _glfw_zwp_locked_pointer_v1_interface
#define zwp_pointer_constraints_v1_interface _glfw_zwp_pointer_constraints_v1_interface
#define zwp_relative_pointer_v1_interface _glfw_zwp_relative_pointer_v1_interface
#define zwp_relative_pointer_manager_v1_interface _glfw_zwp_relative_pointer_manager_v1_interface
#define wp_viewport_interface _glfw_wp_viewport_interface
#define wp_viewporter_interface _glfw_wp_viewporter_interface
#define xdg_toplevel_interface _glfw_xdg_toplevel_interface
#define zxdg_toplevel_decoration_v1_interface _glfw_zxdg_toplevel_decoration_v1_interface
#define zxdg_decoration_manager_v1_interface _glfw_zxdg_decoration_manager_v1_interface
#define xdg_popup_interface _glfw_xdg_popup_interface
#define xdg_positioner_interface _glfw_xdg_positioner_interface
#define xdg_surface_interface _glfw_xdg_surface_interface
#define xdg_toplevel_interface _glfw_xdg_toplevel_interface
#define xdg_wm_base_interface _glfw_xdg_wm_base_interface
#define xdg_activation_v1_interface _glfw_xdg_activation_v1_interface
#define xdg_activation_token_v1_interface _glfw_xdg_activation_token_v1_interface
#define wl_surface_interface _glfw_wl_surface_interface
#define wp_fractional_scale_v1_interface _glfw_wp_fractional_scale_v1_interface

#define GLFW_WAYLAND_WINDOW_STATE         _GLFWwindowWayland  wl;
#define GLFW_WAYLAND_LIBRARY_WINDOW_STATE _GLFWlibraryWayland wl;
#define GLFW_WAYLAND_MONITOR_STATE        _GLFWmonitorWayland wl;
#define GLFW_WAYLAND_CURSOR_STATE         _GLFWcursorWayland  wl;

struct wl_cursor_image {
    uint32_t width;
    uint32_t height;
    uint32_t hotspot_x;
    uint32_t hotspot_y;
    uint32_t delay;
};
struct wl_cursor {
    unsigned int image_count;
    struct wl_cursor_image** images;
    char* name;
};
typedef struct wl_cursor_theme* (* PFN_wl_cursor_theme_load)(const char*, int, struct wl_shm*);
typedef void (* PFN_wl_cursor_theme_destroy)(struct wl_cursor_theme*);
typedef struct wl_cursor* (* PFN_wl_cursor_theme_get_cursor)(struct wl_cursor_theme*, const char*);
typedef struct wl_buffer* (* PFN_wl_cursor_image_get_buffer)(struct wl_cursor_image*);
#define wl_cursor_theme_load _glfw.wl.cursor.theme_load
#define wl_cursor_theme_destroy _glfw.wl.cursor.theme_destroy
#define wl_cursor_theme_get_cursor _glfw.wl.cursor.theme_get_cursor
#define wl_cursor_image_get_buffer _glfw.wl.cursor.image_get_buffer

typedef struct wl_egl_window* (* PFN_wl_egl_window_create)(struct wl_surface*, int, int);
typedef void (* PFN_wl_egl_window_destroy)(struct wl_egl_window*);
typedef void (* PFN_wl_egl_window_resize)(struct wl_egl_window*, int, int, int, int);
#define wl_egl_window_create _glfw.wl.egl.window_create
#define wl_egl_window_destroy _glfw.wl.egl.window_destroy
#define wl_egl_window_resize _glfw.wl.egl.window_resize

typedef struct xkb_context* (* PFN_xkb_context_new)(enum xkb_context_flags);
typedef void (* PFN_xkb_context_unref)(struct xkb_context*);
typedef struct xkb_keymap* (* PFN_xkb_keymap_new_from_string)(struct xkb_context*, const char*, enum xkb_keymap_format, enum xkb_keymap_compile_flags);
typedef void (* PFN_xkb_keymap_unref)(struct xkb_keymap*);
typedef xkb_mod_index_t (* PFN_xkb_keymap_mod_get_index)(struct xkb_keymap*, const char*);
typedef int (* PFN_xkb_keymap_key_repeats)(struct xkb_keymap*, xkb_keycode_t);
typedef int (* PFN_xkb_keymap_key_get_syms_by_level)(struct xkb_keymap*,xkb_keycode_t,xkb_layout_index_t,xkb_level_index_t,const xkb_keysym_t**);
typedef struct xkb_state* (* PFN_xkb_state_new)(struct xkb_keymap*);
typedef void (* PFN_xkb_state_unref)(struct xkb_state*);
typedef int (* PFN_xkb_state_key_get_syms)(struct xkb_state*, xkb_keycode_t, const xkb_keysym_t**);
typedef enum xkb_state_component (* PFN_xkb_state_update_mask)(struct xkb_state*, xkb_mod_mask_t, xkb_mod_mask_t, xkb_mod_mask_t, xkb_layout_index_t, xkb_layout_index_t, xkb_layout_index_t);
typedef xkb_layout_index_t (* PFN_xkb_state_key_get_layout)(struct xkb_state*,xkb_keycode_t);
typedef int (* PFN_xkb_state_mod_index_is_active)(struct xkb_state*,xkb_mod_index_t,enum xkb_state_component);
#define xkb_context_new _glfw.wl.xkb.context_new
#define xkb_context_unref _glfw.wl.xkb.context_unref
#define xkb_keymap_new_from_string _glfw.wl.xkb.keymap_new_from_string
#define xkb_keymap_unref _glfw.wl.xkb.keymap_unref
#define xkb_keymap_mod_get_index _glfw.wl.xkb.keymap_mod_get_index
#define xkb_keymap_key_repeats _glfw.wl.xkb.keymap_key_repeats
#define xkb_keymap_key_get_syms_by_level _glfw.wl.xkb.keymap_key_get_syms_by_level
#define xkb_state_new _glfw.wl.xkb.state_new
#define xkb_state_unref _glfw.wl.xkb.state_unref
#define xkb_state_key_get_syms _glfw.wl.xkb.state_key_get_syms
#define xkb_state_update_mask _glfw.wl.xkb.state_update_mask
#define xkb_state_key_get_layout _glfw.wl.xkb.state_key_get_layout
#define xkb_state_mod_index_is_active _glfw.wl.xkb.state_mod_index_is_active

typedef struct xkb_compose_table* (* PFN_xkb_compose_table_new_from_locale)(struct xkb_context*, const char*, enum xkb_compose_compile_flags);
typedef void (* PFN_xkb_compose_table_unref)(struct xkb_compose_table*);
typedef struct xkb_compose_state* (* PFN_xkb_compose_state_new)(struct xkb_compose_table*, enum xkb_compose_state_flags);
typedef void (* PFN_xkb_compose_state_unref)(struct xkb_compose_state*);
typedef enum xkb_compose_feed_result (* PFN_xkb_compose_state_feed)(struct xkb_compose_state*, xkb_keysym_t);
typedef enum xkb_compose_status (* PFN_xkb_compose_state_get_status)(struct xkb_compose_state*);
typedef xkb_keysym_t (* PFN_xkb_compose_state_get_one_sym)(struct xkb_compose_state*);
#define xkb_compose_table_new_from_locale _glfw.wl.xkb.compose_table_new_from_locale
#define xkb_compose_table_unref _glfw.wl.xkb.compose_table_unref
#define xkb_compose_state_new _glfw.wl.xkb.compose_state_new
#define xkb_compose_state_unref _glfw.wl.xkb.compose_state_unref
#define xkb_compose_state_feed _glfw.wl.xkb.compose_state_feed
#define xkb_compose_state_get_status _glfw.wl.xkb.compose_state_get_status
#define xkb_compose_state_get_one_sym _glfw.wl.xkb.compose_state_get_one_sym

struct libdecor;
struct libdecor_frame;
struct libdecor_state;
struct libdecor_configuration;

enum libdecor_error
{
	LIBDECOR_ERROR_COMPOSITOR_INCOMPATIBLE,
	LIBDECOR_ERROR_INVALID_FRAME_CONFIGURATION,
};

enum libdecor_window_state
{
	LIBDECOR_WINDOW_STATE_NONE = 0,
	LIBDECOR_WINDOW_STATE_ACTIVE = 1,
	LIBDECOR_WINDOW_STATE_MAXIMIZED = 2,
	LIBDECOR_WINDOW_STATE_FULLSCREEN = 4,
	LIBDECOR_WINDOW_STATE_TILED_LEFT = 8,
	LIBDECOR_WINDOW_STATE_TILED_RIGHT = 16,
	LIBDECOR_WINDOW_STATE_TILED_TOP = 32,
	LIBDECOR_WINDOW_STATE_TILED_BOTTOM = 64
};

enum libdecor_capabilities
{
	LIBDECOR_ACTION_MOVE = 1,
	LIBDECOR_ACTION_RESIZE = 2,
	LIBDECOR_ACTION_MINIMIZE = 4,
	LIBDECOR_ACTION_FULLSCREEN = 8,
	LIBDECOR_ACTION_CLOSE = 16
};

struct libdecor_interface
{
	void (* error)(struct libdecor*,enum libdecor_error,const char*);
	void (* reserved0)(void);
	void (* reserved1)(void);
	void (* reserved2)(void);
	void (* reserved3)(void);
	void (* reserved4)(void);
	void (* reserved5)(void);
	void (* reserved6)(void);
	void (* reserved7)(void);
	void (* reserved8)(void);
	void (* reserved9)(void);
};

struct libdecor_frame_interface
{
	void (* configure)(struct libdecor_frame*,struct libdecor_configuration*,void*);
	void (* close)(struct libdecor_frame*,void*);
	void (* commit)(struct libdecor_frame*,void*);
	void (* dismiss_popup)(struct libdecor_frame*,const char*,void*);
	void (* reserved0)(void);
	void (* reserved1)(void);
	void (* reserved2)(void);
	void (* reserved3)(void);
	void (* reserved4)(void);
	void (* reserved5)(void);
	void (* reserved6)(void);
	void (* reserved7)(void);
	void (* reserved8)(void);
	void (* reserved9)(void);
};

typedef struct libdecor* (* PFN_libdecor_new)(struct wl_display*,const struct libdecor_interface*);
typedef void (* PFN_libdecor_unref)(struct libdecor*);
typedef int (* PFN_libdecor_get_fd)(struct libdecor*);
typedef int (* PFN_libdecor_dispatch)(struct libdecor*,int);
typedef struct libdecor_frame* (* PFN_libdecor_decorate)(struct libdecor*,struct wl_surface*,const struct libdecor_frame_interface*,void*);
typedef void (* PFN_libdecor_frame_unref)(struct libdecor_frame*);
typedef void (* PFN_libdecor_frame_set_app_id)(struct libdecor_frame*,const char*);
typedef void (* PFN_libdecor_frame_set_title)(struct libdecor_frame*,const char*);
typedef void (* PFN_libdecor_frame_set_minimized)(struct libdecor_frame*);
typedef void (* PFN_libdecor_frame_set_fullscreen)(struct libdecor_frame*,struct wl_output*);
typedef void (* PFN_libdecor_frame_unset_fullscreen)(struct libdecor_frame*);
typedef void (* PFN_libdecor_frame_map)(struct libdecor_frame*);
typedef void (* PFN_libdecor_frame_commit)(struct libdecor_frame*,struct libdecor_state*,struct libdecor_configuration*);
typedef void (* PFN_libdecor_frame_set_min_content_size)(struct libdecor_frame*,int,int);
typedef void (* PFN_libdecor_frame_set_max_content_size)(struct libdecor_frame*,int,int);
typedef void (* PFN_libdecor_frame_set_maximized)(struct libdecor_frame*);
typedef void (* PFN_libdecor_frame_unset_maximized)(struct libdecor_frame*);
typedef void (* PFN_libdecor_frame_set_capabilities)(struct libdecor_frame*,enum libdecor_capabilities);
typedef void (* PFN_libdecor_frame_unset_capabilities)(struct libdecor_frame*,enum libdecor_capabilities);
typedef void (* PFN_libdecor_frame_set_visibility)(struct libdecor_frame*,bool visible);
typedef struct xdg_toplevel* (* PFN_libdecor_frame_get_xdg_toplevel)(struct libdecor_frame*);
typedef bool (* PFN_libdecor_configuration_get_content_size)(struct libdecor_configuration*,struct libdecor_frame*,int*,int*);
typedef bool (* PFN_libdecor_configuration_get_window_state)(struct libdecor_configuration*,enum libdecor_window_state*);
typedef struct libdecor_state* (* PFN_libdecor_state_new)(int,int);
typedef void (* PFN_libdecor_state_free)(struct libdecor_state*);
#define libdecor_new _glfw.wl.libdecor.libdecor_new_
#define libdecor_unref _glfw.wl.libdecor.libdecor_unref_
#define libdecor_get_fd _glfw.wl.libdecor.libdecor_get_fd_
#define libdecor_dispatch _glfw.wl.libdecor.libdecor_dispatch_
#define libdecor_decorate _glfw.wl.libdecor.libdecor_decorate_
#define libdecor_frame_unref _glfw.wl.libdecor.libdecor_frame_unref_
#define libdecor_frame_set_app_id _glfw.wl.libdecor.libdecor_frame_set_app_id_
#define libdecor_frame_set_title _glfw.wl.libdecor.libdecor_frame_set_title_
#define libdecor_frame_set_minimized _glfw.wl.libdecor.libdecor_frame_set_minimized_
#define libdecor_frame_set_fullscreen _glfw.wl.libdecor.libdecor_frame_set_fullscreen_
#define libdecor_frame_unset_fullscreen _glfw.wl.libdecor.libdecor_frame_unset_fullscreen_
#define libdecor_frame_map _glfw.wl.libdecor.libdecor_frame_map_
#define libdecor_frame_commit _glfw.wl.libdecor.libdecor_frame_commit_
#define libdecor_frame_set_min_content_size _glfw.wl.libdecor.libdecor_frame_set_min_content_size_
#define libdecor_frame_set_max_content_size _glfw.wl.libdecor.libdecor_frame_set_max_content_size_
#define libdecor_frame_set_maximized _glfw.wl.libdecor.libdecor_frame_set_maximized_
#define libdecor_frame_unset_maximized _glfw.wl.libdecor.libdecor_frame_unset_maximized_
#define libdecor_frame_set_capabilities _glfw.wl.libdecor.libdecor_frame_set_capabilities_
#define libdecor_frame_unset_capabilities _glfw.wl.libdecor.libdecor_frame_unset_capabilities_
#define libdecor_frame_set_visibility _glfw.wl.libdecor.libdecor_frame_set_visibility_
#define libdecor_frame_get_xdg_toplevel _glfw.wl.libdecor.libdecor_frame_get_xdg_toplevel_
#define libdecor_configuration_get_content_size _glfw.wl.libdecor.libdecor_configuration_get_content_size_
#define libdecor_configuration_get_window_state _glfw.wl.libdecor.libdecor_configuration_get_window_state_
#define libdecor_state_new _glfw.wl.libdecor.libdecor_state_new_
#define libdecor_state_free _glfw.wl.libdecor.libdecor_state_free_

typedef struct _GLFWfallbackEdgeWayland
{
    struct wl_surface*          surface;
    struct wl_subsurface*       subsurface;
    struct wp_viewport*         viewport;
} _GLFWfallbackEdgeWayland;

typedef struct _GLFWofferWayland
{
    struct wl_data_offer*       offer;
    GLFWbool                    text_plain_utf8;
    GLFWbool                    text_uri_list;
} _GLFWofferWayland;

typedef struct _GLFWscaleWayland
{
    struct wl_output*           output;
    int32_t                     factor;
} _GLFWscaleWayland;

// Wayland-specific per-window data
//
typedef struct _GLFWwindowWayland
{
    int                         width, height;
    int                         fbWidth, fbHeight;
    GLFWbool                    visible;
    GLFWbool                    maximized;
    GLFWbool                    activated;
    GLFWbool                    fullscreen;
    GLFWbool                    hovered;
    GLFWbool                    transparent;
    GLFWbool                    scaleFramebuffer;
    struct wl_surface*          surface;
    struct wl_callback*         callback;

    struct {
        struct wl_egl_window*   window;
    } egl;

    struct {
        int                     width, height;
        GLFWbool                maximized;
        GLFWbool                iconified;
        GLFWbool                activated;
        GLFWbool                fullscreen;
    } pending;

    struct {
        struct xdg_surface*     surface;
        struct xdg_toplevel*    toplevel;
        struct zxdg_toplevel_decoration_v1* decoration;
        uint32_t                decorationMode;
    } xdg;

    struct {
        struct libdecor_frame*  frame;
    } libdecor;

    _GLFWcursor*                currentCursor;
    double                      cursorPosX, cursorPosY;

    char*                       appId;

    // We need to track the monitors the window spans on to calculate the
    // optimal scaling factor.
    int32_t                     bufferScale;
    _GLFWscaleWayland*          outputScales;
    size_t                      outputScaleCount;
    size_t                      outputScaleSize;

    struct wp_viewport*             scalingViewport;
    uint32_t                        scalingNumerator;
    struct wp_fractional_scale_v1*  fractionalScale;

    struct zwp_relative_pointer_v1* relativePointer;
    struct zwp_locked_pointer_v1*   lockedPointer;
    struct zwp_confined_pointer_v1* confinedPointer;

    struct zwp_idle_inhibitor_v1*   idleInhibitor;
    struct xdg_activation_token_v1* activationToken;

    struct {
        GLFWbool                    decorations;
        struct wl_buffer*           buffer;
        _GLFWfallbackEdgeWayland    top, left, right, bottom;
        struct wl_surface*          focus;
    } fallback;
} _GLFWwindowWayland;

// Wayland-specific global data
//
typedef struct _GLFWlibraryWayland
{
    struct wl_display*          display;
    struct wl_registry*         registry;
    struct wl_compositor*       compositor;
    struct wl_subcompositor*    subcompositor;
    struct wl_shm*              shm;
    struct wl_seat*             seat;
    struct wl_pointer*          pointer;
    struct wl_keyboard*         keyboard;
    struct wl_data_device_manager*          dataDeviceManager;
    struct wl_data_device*      dataDevice;
    struct xdg_wm_base*         wmBase;
    struct zxdg_decoration_manager_v1*      decorationManager;
    struct wp_viewporter*       viewporter;
    struct zwp_relative_pointer_manager_v1* relativePointerManager;
    struct zwp_pointer_constraints_v1*      pointerConstraints;
    struct zwp_idle_inhibit_manager_v1*     idleInhibitManager;
    struct xdg_activation_v1*               activationManager;
    struct wp_fractional_scale_manager_v1*  fractionalScaleManager;

    _GLFWofferWayland*          offers;
    unsigned int                offerCount;

    struct wl_data_offer*       selectionOffer;
    struct wl_data_source*      selectionSource;

    struct wl_data_offer*       dragOffer;
    _GLFWwindow*                dragFocus;
    uint32_t                    dragSerial;

    const char*                 tag;

    struct wl_cursor_theme*     cursorTheme;
    struct wl_cursor_theme*     cursorThemeHiDPI;
    struct wl_surface*          cursorSurface;
    const char*                 cursorPreviousName;
    int                         cursorTimerfd;
    uint32_t                    serial;
    uint32_t                    pointerEnterSerial;

    int                         keyRepeatTimerfd;
    int32_t                     keyRepeatRate;
    int32_t                     keyRepeatDelay;
    int                         keyRepeatScancode;

    char*                       clipboardString;
    short int                   keycodes[256];
    short int                   scancodes[GLFW_KEY_LAST + 1];
    char                        keynames[GLFW_KEY_LAST + 1][5];

    struct {
        void*                   handle;
        struct xkb_context*     context;
        struct xkb_keymap*      keymap;
        struct xkb_state*       state;

        struct xkb_compose_state* composeState;

        xkb_mod_index_t         controlIndex;
        xkb_mod_index_t         altIndex;
        xkb_mod_index_t         shiftIndex;
        xkb_mod_index_t         superIndex;
        xkb_mod_index_t         capsLockIndex;
        xkb_mod_index_t         numLockIndex;
        unsigned int            modifiers;

        PFN_xkb_context_new context_new;
        PFN_xkb_context_unref context_unref;
        PFN_xkb_keymap_new_from_string keymap_new_from_string;
        PFN_xkb_keymap_unref keymap_unref;
        PFN_xkb_keymap_mod_get_index keymap_mod_get_index;
        PFN_xkb_keymap_key_repeats keymap_key_repeats;
        PFN_xkb_keymap_key_get_syms_by_level keymap_key_get_syms_by_level;
        PFN_xkb_state_new state_new;
        PFN_xkb_state_unref state_unref;
        PFN_xkb_state_key_get_syms state_key_get_syms;
        PFN_xkb_state_update_mask state_update_mask;
        PFN_xkb_state_key_get_layout state_key_get_layout;
        PFN_xkb_state_mod_index_is_active state_mod_index_is_active;

        PFN_xkb_compose_table_new_from_locale compose_table_new_from_locale;
        PFN_xkb_compose_table_unref compose_table_unref;
        PFN_xkb_compose_state_new compose_state_new;
        PFN_xkb_compose_state_unref compose_state_unref;
        PFN_xkb_compose_state_feed compose_state_feed;
        PFN_xkb_compose_state_get_status compose_state_get_status;
        PFN_xkb_compose_state_get_one_sym compose_state_get_one_sym;
    } xkb;

    _GLFWwindow*                pointerFocus;
    _GLFWwindow*                keyboardFocus;

    struct {
        void*                                       handle;
        PFN_wl_display_flush                        display_flush;
        PFN_wl_display_cancel_read                  display_cancel_read;
        PFN_wl_display_dispatch_pending             display_dispatch_pending;
        PFN_wl_display_read_events                  display_read_events;
        PFN_wl_display_disconnect                   display_disconnect;
        PFN_wl_display_roundtrip                    display_roundtrip;
        PFN_wl_display_get_fd                       display_get_fd;
        PFN_wl_display_prepare_read                 display_prepare_read;
        PFN_wl_proxy_marshal                        proxy_marshal;
        PFN_wl_proxy_add_listener                   proxy_add_listener;
        PFN_wl_proxy_destroy                        proxy_destroy;
        PFN_wl_proxy_marshal_constructor            proxy_marshal_constructor;
        PFN_wl_proxy_marshal_constructor_versioned  proxy_marshal_constructor_versioned;
        PFN_wl_proxy_get_user_data                  proxy_get_user_data;
        PFN_wl_proxy_set_user_data                  proxy_set_user_data;
        PFN_wl_proxy_get_tag                        proxy_get_tag;
        PFN_wl_proxy_set_tag                        proxy_set_tag;
        PFN_wl_proxy_get_version                    proxy_get_version;
        PFN_wl_proxy_marshal_flags                  proxy_marshal_flags;
    } client;

    struct {
        void*                   handle;

        PFN_wl_cursor_theme_load theme_load;
        PFN_wl_cursor_theme_destroy theme_destroy;
        PFN_wl_cursor_theme_get_cursor theme_get_cursor;
        PFN_wl_cursor_image_get_buffer image_get_buffer;
    } cursor;

    struct {
        void*                   handle;

        PFN_wl_egl_window_create window_create;
        PFN_wl_egl_window_destroy window_destroy;
        PFN_wl_egl_window_resize window_resize;
    } egl;

    struct {
        void*                   handle;
        struct libdecor*        context;
        struct wl_callback*     callback;
        GLFWbool                ready;
        PFN_libdecor_new        libdecor_new_;
        PFN_libdecor_unref      libdecor_unref_;
        PFN_libdecor_get_fd     libdecor_get_fd_;
        PFN_libdecor_dispatch   libdecor_dispatch_;
        PFN_libdecor_decorate   libdecor_decorate_;
        PFN_libdecor_frame_unref libdecor_frame_unref_;
        PFN_libdecor_frame_set_app_id libdecor_frame_set_app_id_;
        PFN_libdecor_frame_set_title libdecor_frame_set_title_;
        PFN_libdecor_frame_set_minimized libdecor_frame_set_minimized_;
        PFN_libdecor_frame_set_fullscreen libdecor_frame_set_fullscreen_;
        PFN_libdecor_frame_unset_fullscreen libdecor_frame_unset_fullscreen_;
        PFN_libdecor_frame_map libdecor_frame_map_;
        PFN_libdecor_frame_commit libdecor_frame_commit_;
        PFN_libdecor_frame_set_min_content_size libdecor_frame_set_min_content_size_;
        PFN_libdecor_frame_set_max_content_size libdecor_frame_set_max_content_size_;
        PFN_libdecor_frame_set_maximized libdecor_frame_set_maximized_;
        PFN_libdecor_frame_unset_maximized libdecor_frame_unset_maximized_;
        PFN_libdecor_frame_set_capabilities libdecor_frame_set_capabilities_;
        PFN_libdecor_frame_unset_capabilities libdecor_frame_unset_capabilities_;
        PFN_libdecor_frame_set_visibility libdecor_frame_set_visibility_;
        PFN_libdecor_frame_get_xdg_toplevel libdecor_frame_get_xdg_toplevel_;
        PFN_libdecor_configuration_get_content_size libdecor_configuration_get_content_size_;
        PFN_libdecor_configuration_get_window_state libdecor_configuration_get_window_state_;
        PFN_libdecor_state_new libdecor_state_new_;
        PFN_libdecor_state_free libdecor_state_free_;
    } libdecor;
} _GLFWlibraryWayland;

// Wayland-specific per-monitor data
//
typedef struct _GLFWmonitorWayland
{
    struct wl_output*           output;
    uint32_t                    name;
    int                         currentMode;

    int                         x;
    int                         y;
    int32_t                     scale;
} _GLFWmonitorWayland;

// Wayland-specific per-cursor data
//
typedef struct _GLFWcursorWayland
{
    struct wl_cursor*           cursor;
    struct wl_cursor*           cursorHiDPI;
    struct wl_buffer*           buffer;
    int                         width, height;
    int                         xhot, yhot;
    int                         currentImage;
} _GLFWcursorWayland;

GLFWbool _glfwConnectWayland(int platformID, _GLFWplatform* platform);
int _glfwInitWayland(void);
void _glfwTerminateWayland(void);

GLFWbool _glfwCreateWindowWayland(_GLFWwindow* window, const _GLFWwndconfig* wndconfig, const _GLFWctxconfig* ctxconfig, const _GLFWfbconfig* fbconfig);
void _glfwDestroyWindowWayland(_GLFWwindow* window);
void _glfwSetWindowTitleWayland(_GLFWwindow* window, const char* title);
void _glfwSetWindowIconWayland(_GLFWwindow* window, int count, const GLFWimage* images);
void _glfwGetWindowPosWayland(_GLFWwindow* window, int* xpos, int* ypos);
void _glfwSetWindowPosWayland(_GLFWwindow* window, int xpos, int ypos);
void _glfwGetWindowSizeWayland(_GLFWwindow* window, int* width, int* height);
void _glfwSetWindowSizeWayland(_GLFWwindow* window, int width, int height);
void _glfwSetWindowSizeLimitsWayland(_GLFWwindow* window, int minwidth, int minheight, int maxwidth, int maxheight);
void _glfwSetWindowAspectRatioWayland(_GLFWwindow* window, int numer, int denom);
void _glfwGetFramebufferSizeWayland(_GLFWwindow* window, int* width, int* height);
void _glfwGetWindowFrameSizeWayland(_GLFWwindow* window, int* left, int* top, int* right, int* bottom);
void _glfwGetWindowContentScaleWayland(_GLFWwindow* window, float* xscale, float* yscale);
void _glfwIconifyWindowWayland(_GLFWwindow* window);
void _glfwRestoreWindowWayland(_GLFWwindow* window);
void _glfwMaximizeWindowWayland(_GLFWwindow* window);
void _glfwShowWindowWayland(_GLFWwindow* window);
void _glfwHideWindowWayland(_GLFWwindow* window);
void _glfwRequestWindowAttentionWayland(_GLFWwindow* window);
void _glfwFocusWindowWayland(_GLFWwindow* window);
void _glfwSetWindowMonitorWayland(_GLFWwindow* window, _GLFWmonitor* monitor, int xpos, int ypos, int width, int height, int refreshRate);
GLFWbool _glfwWindowFocusedWayland(_GLFWwindow* window);
GLFWbool _glfwWindowIconifiedWayland(_GLFWwindow* window);
GLFWbool _glfwWindowVisibleWayland(_GLFWwindow* window);
GLFWbool _glfwWindowMaximizedWayland(_GLFWwindow* window);
GLFWbool _glfwWindowHoveredWayland(_GLFWwindow* window);
GLFWbool _glfwFramebufferTransparentWayland(_GLFWwindow* window);
void _glfwSetWindowResizableWayland(_GLFWwindow* window, GLFWbool enabled);
void _glfwSetWindowDecoratedWayland(_GLFWwindow* window, GLFWbool enabled);
void _glfwSetWindowFloatingWayland(_GLFWwindow* window, GLFWbool enabled);
float _glfwGetWindowOpacityWayland(_GLFWwindow* window);
void _glfwSetWindowOpacityWayland(_GLFWwindow* window, float opacity);
void _glfwSetWindowMousePassthroughWayland(_GLFWwindow* window, GLFWbool enabled);

void _glfwSetRawMouseMotionWayland(_GLFWwindow* window, GLFWbool enabled);
GLFWbool _glfwRawMouseMotionSupportedWayland(void);

void _glfwPollEventsWayland(void);
void _glfwWaitEventsWayland(void);
void _glfwWaitEventsTimeoutWayland(double timeout);
void _glfwPostEmptyEventWayland(void);

void _glfwGetCursorPosWayland(_GLFWwindow* window, double* xpos, double* ypos);
void _glfwSetCursorPosWayland(_GLFWwindow* window, double xpos, double ypos);
void _glfwSetCursorModeWayland(_GLFWwindow* window, int mode);
const char* _glfwGetScancodeNameWayland(int scancode);
int _glfwGetKeyScancodeWayland(int key);
GLFWbool _glfwCreateCursorWayland(_GLFWcursor* cursor, const GLFWimage* image, int xhot, int yhot);
GLFWbool _glfwCreateStandardCursorWayland(_GLFWcursor* cursor, int shape);
void _glfwDestroyCursorWayland(_GLFWcursor* cursor);
void _glfwSetCursorWayland(_GLFWwindow* window, _GLFWcursor* cursor);
void _glfwSetClipboardStringWayland(const char* string);
const char* _glfwGetClipboardStringWayland(void);

EGLenum _glfwGetEGLPlatformWayland(EGLint** attribs);
EGLNativeDisplayType _glfwGetEGLNativeDisplayWayland(void);
EGLNativeWindowType _glfwGetEGLNativeWindowWayland(_GLFWwindow* window);

void _glfwGetRequiredInstanceExtensionsWayland(char** extensions);
GLFWbool _glfwGetPhysicalDevicePresentationSupportWayland(VkInstance instance, VkPhysicalDevice device, uint32_t queuefamily);

void _glfwFreeMonitorWayland(_GLFWmonitor* monitor);
void _glfwGetMonitorPosWayland(_GLFWmonitor* monitor, int* xpos, int* ypos);
void _glfwGetMonitorContentScaleWayland(_GLFWmonitor* monitor, float* xscale, float* yscale);
void _glfwGetMonitorWorkareaWayland(_GLFWmonitor* monitor, int* xpos, int* ypos, int* width, int* height);
GLFWvidmode* _glfwGetVideoModesWayland(_GLFWmonitor* monitor, int* count);
GLFWbool _glfwGetVideoModeWayland(_GLFWmonitor* monitor, GLFWvidmode* mode);
GLFWbool _glfwGetGammaRampWayland(_GLFWmonitor* monitor, GLFWgammaramp* ramp);
void _glfwSetGammaRampWayland(_GLFWmonitor* monitor, const GLFWgammaramp* ramp);

void _glfwAddOutputWayland(uint32_t name, uint32_t version);
void _glfwUpdateBufferScaleFromOutputsWayland(_GLFWwindow* window);

void _glfwAddSeatListenerWayland(struct wl_seat* seat);
void _glfwAddDataDeviceListenerWayland(struct wl_data_device* device);


 #define GLFW_EXPOSE_NATIVE_WAYLAND
#else
 #define GLFW_WAYLAND_WINDOW_STATE
 #define GLFW_WAYLAND_MONITOR_STATE
 #define GLFW_WAYLAND_CURSOR_STATE
 #define GLFW_WAYLAND_LIBRARY_WINDOW_STATE
#endif

#if defined(_GLFW_X11)
 /* #include "external/glfw/src/x11_platform.h" */
//========================================================================
// GLFW 3.4 X11 - www.glfw.org
//------------------------------------------------------------------------
// Copyright (c) 2002-2006 Marcus Geelnard
// Copyright (c) 2006-2019 Camilla Löwy <elmindreda@glfw.org>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any damages
// arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it
// freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you must not
//    claim that you wrote the original software. If you use this software
//    in a product, an acknowledgment in the product documentation would
//    be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such, and must not
//    be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source
//    distribution.
//
//========================================================================

#include <unistd.h>
#include <signal.h>
#include <stdint.h>

/* #include "external/X11/Xlib.h" */
/*

Copyright 1985, 1986, 1987, 1991, 1998  The Open Group

Permission to use, copy, modify, distribute, and sell this software and its
documentation for any purpose is hereby granted without fee, provided that
the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation.

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
OPEN GROUP BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of The Open Group shall not be
used in advertising or otherwise to promote the sale, use or other dealings
in this Software without prior written authorization from The Open Group.

*/


/*
 *	Xlib.h - Header definition and support file for the C subroutine
 *	interface library (Xlib) to the X Window System Protocol (V11).
 *	Structures and symbols starting with "_" are private to the library.
 */
#ifndef _X11_XLIB_H_
#define _X11_XLIB_H_

#define XlibSpecificationRelease 6

#include <sys/types.h>

#if defined(__SCO__) || defined(__UNIXWARE__)
#include <stdint.h>
#endif

/* #include "external/X11/X.h" */
/* Definitions for the X window system likely to be used by applications */

#ifndef X_H
#define X_H

/***********************************************************

Copyright 1987, 1998  The Open Group

Permission to use, copy, modify, distribute, and sell this software and its
documentation for any purpose is hereby granted without fee, provided that
the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation.

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
OPEN GROUP BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of The Open Group shall not be
used in advertising or otherwise to promote the sale, use or other dealings
in this Software without prior written authorization from The Open Group.


Copyright 1987 by Digital Equipment Corporation, Maynard, Massachusetts.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of Digital not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

DIGITAL DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
DIGITAL BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

******************************************************************/

#define X_PROTOCOL	11		/* current protocol version */
#define X_PROTOCOL_REVISION 0		/* current minor version */

/* Resources */

/*
 * _XSERVER64 must ONLY be defined when compiling X server sources on
 * systems where unsigned long is not 32 bits, must NOT be used in
 * client or library code.
 */
#ifndef _XSERVER64
#  ifndef _XTYPEDEF_XID
#    define _XTYPEDEF_XID
typedef unsigned long XID;
#  endif
#  ifndef _XTYPEDEF_MASK
#    define _XTYPEDEF_MASK
typedef unsigned long Mask;
#  endif
#  ifndef _XTYPEDEF_ATOM
#    define _XTYPEDEF_ATOM
typedef unsigned long Atom;		/* Also in Xdefs.h */
#  endif
typedef unsigned long VisualID;
typedef unsigned long Time;
#else
/* #  include "external/X11/Xmd.h" */

#  ifndef _XTYPEDEF_XID
#    define _XTYPEDEF_XID
typedef CARD32 XID;
#  endif
#  ifndef _XTYPEDEF_MASK
#    define _XTYPEDEF_MASK
typedef CARD32 Mask;
#  endif
#  ifndef _XTYPEDEF_ATOM
#    define _XTYPEDEF_ATOM
typedef CARD32 Atom;
#  endif
typedef CARD32 VisualID;
typedef CARD32 Time;
#endif

typedef XID Window;
typedef XID Drawable;
#ifndef _XTYPEDEF_FONT
#  define _XTYPEDEF_FONT
typedef XID Font;
#endif
typedef XID Pixmap;
typedef XID Cursor;
typedef XID Colormap;
typedef XID GContext;
typedef XID KeySym;

typedef unsigned char KeyCode;

/*****************************************************************
 * RESERVED RESOURCE AND CONSTANT DEFINITIONS
 *****************************************************************/

#ifndef None
#define None                 0L	/* universal null resource or null atom */
#endif

#define ParentRelative       1L	/* background pixmap in CreateWindow
				    and ChangeWindowAttributes */

#define CopyFromParent       0L	/* border pixmap in CreateWindow
				       and ChangeWindowAttributes
				   special VisualID and special window
				       class passed to CreateWindow */

#define PointerWindow        0L	/* destination window in SendEvent */
#define InputFocus           1L	/* destination window in SendEvent */

#define PointerRoot          1L	/* focus window in SetInputFocus */

#define AnyPropertyType      0L	/* special Atom, passed to GetProperty */

#define AnyKey		     0L	/* special Key Code, passed to GrabKey */

#define AnyButton            0L	/* special Button Code, passed to GrabButton */

#define AllTemporary         0L	/* special Resource ID passed to KillClient */

#define CurrentTime          0L	/* special Time */

#define NoSymbol	     0L	/* special KeySym */

/*****************************************************************
 * EVENT DEFINITIONS
 *****************************************************************/

/* Input Event Masks. Used as event-mask window attribute and as arguments
   to Grab requests.  Not to be confused with event names.  */

#define NoEventMask			0L
#define KeyPressMask			(1L<<0)
#define KeyReleaseMask			(1L<<1)
#define ButtonPressMask			(1L<<2)
#define ButtonReleaseMask		(1L<<3)
#define EnterWindowMask			(1L<<4)
#define LeaveWindowMask			(1L<<5)
#define PointerMotionMask		(1L<<6)
#define PointerMotionHintMask		(1L<<7)
#define Button1MotionMask		(1L<<8)
#define Button2MotionMask		(1L<<9)
#define Button3MotionMask		(1L<<10)
#define Button4MotionMask		(1L<<11)
#define Button5MotionMask		(1L<<12)
#define ButtonMotionMask		(1L<<13)
#define KeymapStateMask			(1L<<14)
#define ExposureMask			(1L<<15)
#define VisibilityChangeMask		(1L<<16)
#define StructureNotifyMask		(1L<<17)
#define ResizeRedirectMask		(1L<<18)
#define SubstructureNotifyMask		(1L<<19)
#define SubstructureRedirectMask	(1L<<20)
#define FocusChangeMask			(1L<<21)
#define PropertyChangeMask		(1L<<22)
#define ColormapChangeMask		(1L<<23)
#define OwnerGrabButtonMask		(1L<<24)

/* Event names.  Used in "type" field in XEvent structures.  Not to be
confused with event masks above.  They start from 2 because 0 and 1
are reserved in the protocol for errors and replies. */

#define KeyPress		2
#define KeyRelease		3
#define ButtonPress		4
#define ButtonRelease		5
#define MotionNotify		6
#define EnterNotify		7
#define LeaveNotify		8
#define FocusIn			9
#define FocusOut		10
#define KeymapNotify		11
#define Expose			12
#define GraphicsExpose		13
#define NoExpose		14
#define VisibilityNotify	15
#define CreateNotify		16
#define DestroyNotify		17
#define UnmapNotify		18
#define MapNotify		19
#define MapRequest		20
#define ReparentNotify		21
#define ConfigureNotify		22
#define ConfigureRequest	23
#define GravityNotify		24
#define ResizeRequest		25
#define CirculateNotify		26
#define CirculateRequest	27
#define PropertyNotify		28
#define SelectionClear		29
#define SelectionRequest	30
#define SelectionNotify		31
#define ColormapNotify		32
#define ClientMessage		33
#define MappingNotify		34
#define GenericEvent		35
#define LASTEvent		36	/* must be bigger than any event # */


/* Key masks. Used as modifiers to GrabButton and GrabKey, results of QueryPointer,
   state in various key-, mouse-, and button-related events. */

#define ShiftMask		(1<<0)
#define LockMask		(1<<1)
#define ControlMask		(1<<2)
#define Mod1Mask		(1<<3)
#define Mod2Mask		(1<<4)
#define Mod3Mask		(1<<5)
#define Mod4Mask		(1<<6)
#define Mod5Mask		(1<<7)

/* modifier names.  Used to build a SetModifierMapping request or
   to read a GetModifierMapping request.  These correspond to the
   masks defined above. */
#define ShiftMapIndex		0
#define LockMapIndex		1
#define ControlMapIndex		2
#define Mod1MapIndex		3
#define Mod2MapIndex		4
#define Mod3MapIndex		5
#define Mod4MapIndex		6
#define Mod5MapIndex		7


/* button masks.  Used in same manner as Key masks above. Not to be confused
   with button names below. */

#define Button1Mask		(1<<8)
#define Button2Mask		(1<<9)
#define Button3Mask		(1<<10)
#define Button4Mask		(1<<11)
#define Button5Mask		(1<<12)

#define AnyModifier		(1<<15)  /* used in GrabButton, GrabKey */


/* button names. Used as arguments to GrabButton and as detail in ButtonPress
   and ButtonRelease events.  Not to be confused with button masks above.
   Note that 0 is already defined above as "AnyButton".  */

#define Button1			1
#define Button2			2
#define Button3			3
#define Button4			4
#define Button5			5

/* Notify modes */

#define NotifyNormal		0
#define NotifyGrab		1
#define NotifyUngrab		2
#define NotifyWhileGrabbed	3

#define NotifyHint		1	/* for MotionNotify events */

/* Notify detail */

#define NotifyAncestor		0
#define NotifyVirtual		1
#define NotifyInferior		2
#define NotifyNonlinear		3
#define NotifyNonlinearVirtual	4
#define NotifyPointer		5
#define NotifyPointerRoot	6
#define NotifyDetailNone	7

/* Visibility notify */

#define VisibilityUnobscured		0
#define VisibilityPartiallyObscured	1
#define VisibilityFullyObscured		2

/* Circulation request */

#define PlaceOnTop		0
#define PlaceOnBottom		1

/* protocol families */

#define FamilyInternet		0	/* IPv4 */
#define FamilyDECnet		1
#define FamilyChaos		2
#define FamilyInternet6		6	/* IPv6 */

/* authentication families not tied to a specific protocol */
#define FamilyServerInterpreted 5

/* Property notification */

#define PropertyNewValue	0
#define PropertyDelete		1

/* Color Map notification */

#define ColormapUninstalled	0
#define ColormapInstalled	1

/* GrabPointer, GrabButton, GrabKeyboard, GrabKey Modes */

#define GrabModeSync		0
#define GrabModeAsync		1

/* GrabPointer, GrabKeyboard reply status */

#define GrabSuccess		0
#define AlreadyGrabbed		1
#define GrabInvalidTime		2
#define GrabNotViewable		3
#define GrabFrozen		4

/* AllowEvents modes */

#define AsyncPointer		0
#define SyncPointer		1
#define ReplayPointer		2
#define AsyncKeyboard		3
#define SyncKeyboard		4
#define ReplayKeyboard		5
#define AsyncBoth		6
#define SyncBoth		7

/* Used in SetInputFocus, GetInputFocus */

#define RevertToNone		(int)None
#define RevertToPointerRoot	(int)PointerRoot
#define RevertToParent		2

/*****************************************************************
 * ERROR CODES
 *****************************************************************/

#define Success		   0	/* everything's okay */
#define BadRequest	   1	/* bad request code */
#define BadValue	   2	/* int parameter out of range */
#define BadWindow	   3	/* parameter not a Window */
#define BadPixmap	   4	/* parameter not a Pixmap */
#define BadAtom		   5	/* parameter not an Atom */
#define BadCursor	   6	/* parameter not a Cursor */
#define BadFont		   7	/* parameter not a Font */
#define BadMatch	   8	/* parameter mismatch */
#define BadDrawable	   9	/* parameter not a Pixmap or Window */
#define BadAccess	  10	/* depending on context:
				 - key/button already grabbed
				 - attempt to free an illegal
				   cmap entry
				- attempt to store into a read-only
				   color map entry.
 				- attempt to modify the access control
				   list from other than the local host.
				*/
#define BadAlloc	  11	/* insufficient resources */
#define BadColor	  12	/* no such colormap */
#define BadGC		  13	/* parameter not a GC */
#define BadIDChoice	  14	/* choice not in range or already used */
#define BadName		  15	/* font or color name doesn't exist */
#define BadLength	  16	/* Request length incorrect */
#define BadImplementation 17	/* server is defective */

#define FirstExtensionError	128
#define LastExtensionError	255

/*****************************************************************
 * WINDOW DEFINITIONS
 *****************************************************************/

/* Window classes used by CreateWindow */
/* Note that CopyFromParent is already defined as 0 above */

#define InputOutput		1
#define InputOnly		2

/* Window attributes for CreateWindow and ChangeWindowAttributes */

#define CWBackPixmap		(1L<<0)
#define CWBackPixel		(1L<<1)
#define CWBorderPixmap		(1L<<2)
#define CWBorderPixel           (1L<<3)
#define CWBitGravity		(1L<<4)
#define CWWinGravity		(1L<<5)
#define CWBackingStore          (1L<<6)
#define CWBackingPlanes	        (1L<<7)
#define CWBackingPixel	        (1L<<8)
#define CWOverrideRedirect	(1L<<9)
#define CWSaveUnder		(1L<<10)
#define CWEventMask		(1L<<11)
#define CWDontPropagate	        (1L<<12)
#define CWColormap		(1L<<13)
#define CWCursor	        (1L<<14)

/* ConfigureWindow structure */

#define CWX			(1<<0)
#define CWY			(1<<1)
#define CWWidth			(1<<2)
#define CWHeight		(1<<3)
#define CWBorderWidth		(1<<4)
#define CWSibling		(1<<5)
#define CWStackMode		(1<<6)


/* Bit Gravity */

#define ForgetGravity		0
#define NorthWestGravity	1
#define NorthGravity		2
#define NorthEastGravity	3
#define WestGravity		4
#define CenterGravity		5
#define EastGravity		6
#define SouthWestGravity	7
#define SouthGravity		8
#define SouthEastGravity	9
#define StaticGravity		10

/* Window gravity + bit gravity above */

#define UnmapGravity		0

/* Used in CreateWindow for backing-store hint */

#define NotUseful               0
#define WhenMapped              1
#define Always                  2

/* Used in GetWindowAttributes reply */

#define IsUnmapped		0
#define IsUnviewable		1
#define IsViewable		2

/* Used in ChangeSaveSet */

#define SetModeInsert           0
#define SetModeDelete           1

/* Used in ChangeCloseDownMode */

#define DestroyAll              0
#define RetainPermanent         1
#define RetainTemporary         2

/* Window stacking method (in configureWindow) */

#define Above                   0
#define Below                   1
#define TopIf                   2
#define BottomIf                3
#define Opposite                4

/* Circulation direction */

#define RaiseLowest             0
#define LowerHighest            1

/* Property modes */

#define PropModeReplace         0
#define PropModePrepend         1
#define PropModeAppend          2

/*****************************************************************
 * GRAPHICS DEFINITIONS
 *****************************************************************/

/* graphics functions, as in GC.alu */

#define	GXclear			0x0		/* 0 */
#define GXand			0x1		/* src AND dst */
#define GXandReverse		0x2		/* src AND NOT dst */
#define GXcopy			0x3		/* src */
#define GXandInverted		0x4		/* NOT src AND dst */
#define	GXnoop			0x5		/* dst */
#define GXxor			0x6		/* src XOR dst */
#define GXor			0x7		/* src OR dst */
#define GXnor			0x8		/* NOT src AND NOT dst */
#define GXequiv			0x9		/* NOT src XOR dst */
#define GXinvert		0xa		/* NOT dst */
#define GXorReverse		0xb		/* src OR NOT dst */
#define GXcopyInverted		0xc		/* NOT src */
#define GXorInverted		0xd		/* NOT src OR dst */
#define GXnand			0xe		/* NOT src OR NOT dst */
#define GXset			0xf		/* 1 */

/* LineStyle */

#define LineSolid		0
#define LineOnOffDash		1
#define LineDoubleDash		2

/* capStyle */

#define CapNotLast		0
#define CapButt			1
#define CapRound		2
#define CapProjecting		3

/* joinStyle */

#define JoinMiter		0
#define JoinRound		1
#define JoinBevel		2

/* fillStyle */

#define FillSolid		0
#define FillTiled		1
#define FillStippled		2
#define FillOpaqueStippled	3

/* fillRule */

#define EvenOddRule		0
#define WindingRule		1

/* subwindow mode */

#define ClipByChildren		0
#define IncludeInferiors	1

/* SetClipRectangles ordering */

#define Unsorted		0
#define YSorted			1
#define YXSorted		2
#define YXBanded		3

/* CoordinateMode for drawing routines */

#define CoordModeOrigin		0	/* relative to the origin */
#define CoordModePrevious       1	/* relative to previous point */

/* Polygon shapes */

#define Complex			0	/* paths may intersect */
#define Nonconvex		1	/* no paths intersect, but not convex */
#define Convex			2	/* wholly convex */

/* Arc modes for PolyFillArc */

#define ArcChord		0	/* join endpoints of arc */
#define ArcPieSlice		1	/* join endpoints to center of arc */

/* GC components: masks used in CreateGC, CopyGC, ChangeGC, OR'ed into
   GC.stateChanges */

#define GCFunction              (1L<<0)
#define GCPlaneMask             (1L<<1)
#define GCForeground            (1L<<2)
#define GCBackground            (1L<<3)
#define GCLineWidth             (1L<<4)
#define GCLineStyle             (1L<<5)
#define GCCapStyle              (1L<<6)
#define GCJoinStyle		(1L<<7)
#define GCFillStyle		(1L<<8)
#define GCFillRule		(1L<<9)
#define GCTile			(1L<<10)
#define GCStipple		(1L<<11)
#define GCTileStipXOrigin	(1L<<12)
#define GCTileStipYOrigin	(1L<<13)
#define GCFont 			(1L<<14)
#define GCSubwindowMode		(1L<<15)
#define GCGraphicsExposures     (1L<<16)
#define GCClipXOrigin		(1L<<17)
#define GCClipYOrigin		(1L<<18)
#define GCClipMask		(1L<<19)
#define GCDashOffset		(1L<<20)
#define GCDashList		(1L<<21)
#define GCArcMode		(1L<<22)

#define GCLastBit		22
/*****************************************************************
 * FONTS
 *****************************************************************/

/* used in QueryFont -- draw direction */

#define FontLeftToRight		0
#define FontRightToLeft		1

#define FontChange		255

/*****************************************************************
 *  IMAGING
 *****************************************************************/

/* ImageFormat -- PutImage, GetImage */

#define XYBitmap		0	/* depth 1, XYFormat */
#define XYPixmap		1	/* depth == drawable depth */
#define ZPixmap			2	/* depth == drawable depth */

/*****************************************************************
 *  COLOR MAP STUFF
 *****************************************************************/

/* For CreateColormap */

#define AllocNone		0	/* create map with no entries */
#define AllocAll		1	/* allocate entire map writeable */


/* Flags used in StoreNamedColor, StoreColors */

#define DoRed			(1<<0)
#define DoGreen			(1<<1)
#define DoBlue			(1<<2)

/*****************************************************************
 * CURSOR STUFF
 *****************************************************************/

/* QueryBestSize Class */

#define CursorShape		0	/* largest size that can be displayed */
#define TileShape		1	/* size tiled fastest */
#define StippleShape		2	/* size stippled fastest */

/*****************************************************************
 * KEYBOARD/POINTER STUFF
 *****************************************************************/

#define AutoRepeatModeOff	0
#define AutoRepeatModeOn	1
#define AutoRepeatModeDefault	2

#define LedModeOff		0
#define LedModeOn		1

/* masks for ChangeKeyboardControl */

#define KBKeyClickPercent	(1L<<0)
#define KBBellPercent		(1L<<1)
#define KBBellPitch		(1L<<2)
#define KBBellDuration		(1L<<3)
#define KBLed			(1L<<4)
#define KBLedMode		(1L<<5)
#define KBKey			(1L<<6)
#define KBAutoRepeatMode	(1L<<7)

#define MappingSuccess     	0
#define MappingBusy        	1
#define MappingFailed		2

#define MappingModifier		0
#define MappingKeyboard		1
#define MappingPointer		2

/*****************************************************************
 * SCREEN SAVER STUFF
 *****************************************************************/

#define DontPreferBlanking	0
#define PreferBlanking		1
#define DefaultBlanking		2

#define DisableScreenSaver	0
#define DisableScreenInterval	0

#define DontAllowExposures	0
#define AllowExposures		1
#define DefaultExposures	2

/* for ForceScreenSaver */

#define ScreenSaverReset 0
#define ScreenSaverActive 1

/*****************************************************************
 * HOSTS AND CONNECTIONS
 *****************************************************************/

/* for ChangeHosts */

#define HostInsert		0
#define HostDelete		1

/* for ChangeAccessControl */

#define EnableAccess		1
#define DisableAccess		0

/* Display classes  used in opening the connection
 * Note that the statically allocated ones are even numbered and the
 * dynamically changeable ones are odd numbered */

#define StaticGray		0
#define GrayScale		1
#define StaticColor		2
#define PseudoColor		3
#define TrueColor		4
#define DirectColor		5


/* Byte order  used in imageByteOrder and bitmapBitOrder */

#define LSBFirst		0
#define MSBFirst		1

#endif /* X_H */

/* applications should not depend on these two headers being included! */
/* #include "external/X11/Xfuncproto.h" */
/*
 *
Copyright 1989, 1991, 1998  The Open Group

Permission to use, copy, modify, distribute, and sell this software and its
documentation for any purpose is hereby granted without fee, provided that
the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation.

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
OPEN GROUP BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of The Open Group shall not be
used in advertising or otherwise to promote the sale, use or other dealings
in this Software without prior written authorization from The Open Group.
 *
 */

/* Definitions to make function prototypes manageable */

#ifndef _XFUNCPROTO_H_
#define _XFUNCPROTO_H_

#ifndef NeedFunctionPrototypes
#define NeedFunctionPrototypes 1
#endif /* NeedFunctionPrototypes */

#ifndef NeedVarargsPrototypes
#define NeedVarargsPrototypes 1
#endif /* NeedVarargsPrototypes */

#if NeedFunctionPrototypes

#ifndef NeedNestedPrototypes
#define NeedNestedPrototypes 1
#endif /* NeedNestedPrototypes */

#ifndef _Xconst
#define _Xconst const
#endif /* _Xconst */

/* Function prototype configuration (see configure for more info) */
#if !defined(NARROWPROTO) && \
    (defined(__linux__) || defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__))
#define NARROWPROTO
#endif
#ifndef FUNCPROTO
#define FUNCPROTO 15
#endif

#ifndef NeedWidePrototypes
#ifdef NARROWPROTO
#define NeedWidePrototypes 0
#else
#define NeedWidePrototypes 1		/* default to make interropt. easier */
#endif
#endif /* NeedWidePrototypes */

#endif /* NeedFunctionPrototypes */

#ifndef _XFUNCPROTOBEGIN
#if defined(__cplusplus) || defined(c_plusplus) /* for C++ V2.0 */
#define _XFUNCPROTOBEGIN extern "C" {	/* do not leave open across includes */
#define _XFUNCPROTOEND }
#else
#define _XFUNCPROTOBEGIN
#define _XFUNCPROTOEND
#endif
#endif /* _XFUNCPROTOBEGIN */

/* http://clang.llvm.org/docs/LanguageExtensions.html#has-attribute */
#ifndef __has_attribute
# define __has_attribute(x) 0  /* Compatibility with non-clang compilers. */
#endif
#ifndef __has_feature
# define __has_feature(x) 0    /* Compatibility with non-clang compilers. */
#endif
#ifndef __has_extension
# define __has_extension(x) 0  /* Compatibility with non-clang compilers. */
#endif

/* Added in X11R6.9, so available in any version of modular xproto */
#if __has_attribute(__sentinel__) || (defined(__GNUC__) && (__GNUC__ >= 4))
# define _X_SENTINEL(x) __attribute__ ((__sentinel__(x)))
#else
# define _X_SENTINEL(x)
#endif /* GNUC >= 4 */

/* Added in X11R6.9, so available in any version of modular xproto */
#if (__has_attribute(visibility) || (defined(__GNUC__) && (__GNUC__ >= 4))) \
    && !defined(__CYGWIN__) && !defined(__MINGW32__)
# define _X_EXPORT      __attribute__((visibility("default")))
# define _X_HIDDEN      __attribute__((visibility("hidden")))
# define _X_INTERNAL    __attribute__((visibility("internal")))
#elif defined(__SUNPRO_C) && (__SUNPRO_C >= 0x550)
# define _X_EXPORT      __global
# define _X_HIDDEN      __hidden
# define _X_INTERNAL    __hidden
#else /* not gcc >= 4 and not Sun Studio >= 8 */
# define _X_EXPORT
# define _X_HIDDEN
# define _X_INTERNAL
#endif /* GNUC >= 4 */

/* Branch prediction hints for individual conditionals */
/* requires xproto >= 7.0.9 */
#if defined(__GNUC__) && ((__GNUC__ * 100 + __GNUC_MINOR__) >= 303)
# define _X_LIKELY(x)   __builtin_expect(!!(x), 1)
# define _X_UNLIKELY(x) __builtin_expect(!!(x), 0)
#else /* not gcc >= 3.3 */
# define _X_LIKELY(x)   (x)
# define _X_UNLIKELY(x) (x)
#endif

/* Bulk branch prediction hints via marking error path functions as "cold" */
/* requires xproto >= 7.0.25 */
#if __has_attribute(__cold__) || \
    (defined(__GNUC__) && ((__GNUC__ * 100 + __GNUC_MINOR__) >= 403)) /* 4.3+ */
# define _X_COLD __attribute__((__cold__))
#else
# define _X_COLD /* nothing */
#endif

/* Added in X11R6.9, so available in any version of modular xproto */
#if __has_attribute(deprecated) \
    || (defined(__GNUC__) && ((__GNUC__ * 100 + __GNUC_MINOR__) >= 301)) \
    || (defined(__SUNPRO_C) && (__SUNPRO_C >= 0x5130))
# define _X_DEPRECATED  __attribute__((deprecated))
#else /* not gcc >= 3.1 */
# define _X_DEPRECATED
#endif

/* requires xproto >= 7.0.30 */
#if __has_extension(attribute_deprecated_with_message) || \
                (defined(__GNUC__) && ((__GNUC__ >= 5) || ((__GNUC__ == 4) && (__GNUC_MINOR__ >= 5))))
# define _X_DEPRECATED_MSG(_msg) __attribute__((deprecated(_msg)))
#else
# define _X_DEPRECATED_MSG(_msg) _X_DEPRECATED
#endif

/* requires xproto >= 7.0.17 */
#if __has_attribute(noreturn) \
    || (defined(__GNUC__) && ((__GNUC__ * 100 + __GNUC_MINOR__) >= 205)) \
    || (defined(__SUNPRO_C) && (__SUNPRO_C >= 0x590))
# define _X_NORETURN __attribute((noreturn))
#else
# define _X_NORETURN
#endif /* GNUC  */

/* Added in X11R6.9, so available in any version of modular xproto */
#if __has_attribute(__format__) \
    || defined(__GNUC__) && ((__GNUC__ * 100 + __GNUC_MINOR__) >= 203)
# define _X_ATTRIBUTE_PRINTF(x,y) __attribute__((__format__(__printf__,x,y)))
#else /* not gcc >= 2.3 */
# define _X_ATTRIBUTE_PRINTF(x,y)
#endif

/* requires xproto >= 7.0.22 */
#if __has_attribute(__unused__) \
    || defined(__GNUC__) &&  ((__GNUC__ * 100 + __GNUC_MINOR__) >= 205)
#define _X_UNUSED  __attribute__((__unused__))
#else
#define _X_UNUSED  /* */
#endif

/* C99 keyword "inline" or equivalent extensions in pre-C99 compilers */
/* requires xproto >= 7.0.9
   (introduced in 7.0.8 but didn't support all compilers until 7.0.9) */
#if defined(inline) /* assume autoconf set it correctly */ || \
   (defined(__STDC_VERSION__) && (__STDC_VERSION__ - 0 >= 199901L)) /* C99 */ || \
   (defined(__SUNPRO_C) && (__SUNPRO_C >= 0x550))
# define _X_INLINE inline
#elif defined(__GNUC__) && !defined(__STRICT_ANSI__) /* gcc w/C89+extensions */
# define _X_INLINE __inline__
#else
# define _X_INLINE
#endif

/* C99 keyword "restrict" or equivalent extensions in pre-C99 compilers */
/* requires xproto >= 7.0.21 */
#ifndef _X_RESTRICT_KYWD
# if defined(restrict) /* assume autoconf set it correctly */ || \
    (defined(__STDC_VERSION__) && (__STDC_VERSION__ - 0 >= 199901L) /* C99 */ \
     && !defined(__cplusplus)) /* Workaround g++ issue on Solaris */
#  define _X_RESTRICT_KYWD  restrict
# elif defined(__GNUC__) && !defined(__STRICT_ANSI__) /* gcc w/C89+extensions */
#  define _X_RESTRICT_KYWD __restrict__
# else
#  define _X_RESTRICT_KYWD
# endif
#endif

/* requires xproto >= 7.0.30 */
#if __has_attribute(no_sanitize_thread)
# define _X_NOTSAN __attribute__((no_sanitize_thread))
#else
# define _X_NOTSAN
#endif

/* Mark a char array/pointer as not containing a NUL-terminated string */
/* requires xproto >= 7.0.33 */
#if __has_attribute(nonstring)
# define _X_NONSTRING __attribute__((nonstring))
#else
# define _X_NONSTRING
#endif

#endif /* _XFUNCPROTO_H_ */
/* #include "external/X11/Xosdefs.h" */
/*
 * O/S-dependent (mis)feature macro definitions
 *
Copyright 1991, 1998  The Open Group

Permission to use, copy, modify, distribute, and sell this software and its
documentation for any purpose is hereby granted without fee, provided that
the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation.

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
OPEN GROUP BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of The Open Group shall not be
used in advertising or otherwise to promote the sale, use or other dealings
in this Software without prior written authorization from The Open Group.
 */

#ifndef _XOSDEFS_H_
# define _XOSDEFS_H_

/*
 * X_NOT_POSIX means does not have POSIX header files.  Lack of this
 * symbol does NOT mean that the POSIX environment is the default.
 * You may still have to define _POSIX_SOURCE to get it.
 */


# ifdef _SCO_DS
#  ifndef __SCO__
#   define __SCO__
#  endif
# endif

# ifdef __i386__
#  ifdef SYSV
#   if !defined(__SCO__) && \
	!defined(__UNIXWARE__) && !defined(__sun)
#    if !defined(_POSIX_SOURCE)
#     define X_NOT_POSIX
#    endif
#   endif
#  endif
# endif

# ifdef __sun
/* Imake configs define SVR4 on Solaris, but cc & gcc only define __SVR4
 * This check allows non-Imake configured programs to build correctly.
 */
#  if defined(__SVR4) && !defined(SVR4)
#   define SVR4 1
#  endif
#  ifdef SVR4
/* define this to whatever it needs to be */
#   define X_POSIX_C_SOURCE 199300L
#  endif
# endif

# ifdef WIN32
#  ifndef _POSIX_
#   define X_NOT_POSIX
#  endif
# endif


# ifdef __APPLE__
#  define NULL_NOT_ZERO

/* Defining any of these will sanitize the namespace to JUST want is defined by
 * that particular standard.  If that happens, we don't get some expected
 * prototypes, typedefs, etc (like fd_mask).  We can define _DARWIN_C_SOURCE to
 * loosen our belts a tad.
 */
#  if defined(_XOPEN_SOURCE) || defined(_POSIX_SOURCE) || defined(_POSIX_C_SOURCE)
#   ifndef _DARWIN_C_SOURCE
#    define _DARWIN_C_SOURCE
#   endif
#  endif

# endif

# ifdef __GNU__
#  ifndef PATH_MAX
#   define PATH_MAX 4096
#  endif
#  ifndef MAXPATHLEN
#   define MAXPATHLEN 4096
#  endif
# endif

# if defined(__SCO__) || defined(__UNIXWARE__)
#  ifndef PATH_MAX
#   define PATH_MAX	1024
#  endif
#  ifndef MAXPATHLEN
#   define MAXPATHLEN	1024
#  endif
# endif

# if defined(__OpenBSD__) || defined(__NetBSD__) || defined(__FreeBSD__) \
	|| defined(__APPLE__) || defined(__DragonFly__)
#  ifndef CSRG_BASED
#   define CSRG_BASED
#  endif
# endif

#endif /* _XOSDEFS_H_ */



#ifndef X_WCHAR
#include <stddef.h>
#else
/* replace this with #include or typedef appropriate for your system */
typedef unsigned long wchar_t;
#endif


extern int
_Xmblen(
    char *str,
    int len
    );

/* API mentioning "UTF8" or "utf8" is an XFree86 extension, introduced in
   November 2000. Its presence is indicated through the following macro. */
#define X_HAVE_UTF8_STRING 1

/* The Xlib structs are full of implicit padding to properly align members.
   We can't clean that up without breaking ABI, so tell clang not to bother
   complaining about it. */
#ifdef __clang__
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wpadded"
#endif

typedef char *XPointer;

#define Bool int
#define Status int
#define True 1
#define False 0

#define QueuedAlready 0
#define QueuedAfterReading 1
#define QueuedAfterFlush 2

#define ConnectionNumber(dpy) 	(((_XPrivDisplay)(dpy))->fd)
#define RootWindow(dpy, scr) 	(ScreenOfDisplay(dpy,scr)->root)
#define DefaultScreen(dpy) 	(((_XPrivDisplay)(dpy))->default_screen)
#define DefaultRootWindow(dpy) 	(ScreenOfDisplay(dpy,DefaultScreen(dpy))->root)
#define DefaultVisual(dpy, scr) (ScreenOfDisplay(dpy,scr)->root_visual)
#define DefaultGC(dpy, scr) 	(ScreenOfDisplay(dpy,scr)->default_gc)
#define BlackPixel(dpy, scr) 	(ScreenOfDisplay(dpy,scr)->black_pixel)
#define WhitePixel(dpy, scr) 	(ScreenOfDisplay(dpy,scr)->white_pixel)
#define AllPlanes 		((unsigned long)~0L)
#define QLength(dpy) 		(((_XPrivDisplay)(dpy))->qlen)
#define DisplayWidth(dpy, scr) 	(ScreenOfDisplay(dpy,scr)->width)
#define DisplayHeight(dpy, scr) (ScreenOfDisplay(dpy,scr)->height)
#define DisplayWidthMM(dpy, scr)(ScreenOfDisplay(dpy,scr)->mwidth)
#define DisplayHeightMM(dpy, scr)(ScreenOfDisplay(dpy,scr)->mheight)
#define DisplayPlanes(dpy, scr) (ScreenOfDisplay(dpy,scr)->root_depth)
#define DisplayCells(dpy, scr) 	(DefaultVisual(dpy,scr)->map_entries)
#define ScreenCount(dpy) 	(((_XPrivDisplay)(dpy))->nscreens)
#define ServerVendor(dpy) 	(((_XPrivDisplay)(dpy))->vendor)
#define ProtocolVersion(dpy) 	(((_XPrivDisplay)(dpy))->proto_major_version)
#define ProtocolRevision(dpy) 	(((_XPrivDisplay)(dpy))->proto_minor_version)
#define VendorRelease(dpy) 	(((_XPrivDisplay)(dpy))->release)
#define DisplayString(dpy) 	(((_XPrivDisplay)(dpy))->display_name)
#define DefaultDepth(dpy, scr) 	(ScreenOfDisplay(dpy,scr)->root_depth)
#define DefaultColormap(dpy, scr)(ScreenOfDisplay(dpy,scr)->cmap)
#define BitmapUnit(dpy) 	(((_XPrivDisplay)(dpy))->bitmap_unit)
#define BitmapBitOrder(dpy) 	(((_XPrivDisplay)(dpy))->bitmap_bit_order)
#define BitmapPad(dpy) 		(((_XPrivDisplay)(dpy))->bitmap_pad)
#define ImageByteOrder(dpy) 	(((_XPrivDisplay)(dpy))->byte_order)
#define NextRequest(dpy)	(((_XPrivDisplay)(dpy))->request + 1)
#define LastKnownRequestProcessed(dpy)	(((_XPrivDisplay)(dpy))->last_request_read)

/* macros for screen oriented applications (toolkit) */
#define ScreenOfDisplay(dpy, scr)(&((_XPrivDisplay)(dpy))->screens[scr])
#define DefaultScreenOfDisplay(dpy) ScreenOfDisplay(dpy,DefaultScreen(dpy))
#define DisplayOfScreen(s)	((s)->display)
#define RootWindowOfScreen(s)	((s)->root)
#define BlackPixelOfScreen(s)	((s)->black_pixel)
#define WhitePixelOfScreen(s)	((s)->white_pixel)
#define DefaultColormapOfScreen(s)((s)->cmap)
#define DefaultDepthOfScreen(s)	((s)->root_depth)
#define DefaultGCOfScreen(s)	((s)->default_gc)
#define DefaultVisualOfScreen(s)((s)->root_visual)
#define WidthOfScreen(s)	((s)->width)
#define HeightOfScreen(s)	((s)->height)
#define WidthMMOfScreen(s)	((s)->mwidth)
#define HeightMMOfScreen(s)	((s)->mheight)
#define PlanesOfScreen(s)	((s)->root_depth)
#define CellsOfScreen(s)	(DefaultVisualOfScreen((s))->map_entries)
#define MinCmapsOfScreen(s)	((s)->min_maps)
#define MaxCmapsOfScreen(s)	((s)->max_maps)
#define DoesSaveUnders(s)	((s)->save_unders)
#define DoesBackingStore(s)	((s)->backing_store)
#define EventMaskOfScreen(s)	((s)->root_input_mask)

/*
 * Extensions need a way to hang private data on some structures.
 */
typedef struct _XExtData {
	int number;		/* number returned by XRegisterExtension */
	struct _XExtData *next;	/* next item on list of data for structure */
	int (*free_private)(	/* called to free private storage */
	struct _XExtData *extension
	);
	XPointer private_data;	/* data private to this extension. */
} XExtData;

/*
 * This file contains structures used by the extension mechanism.
 */
typedef struct {		/* public to extension, cannot be changed */
	int extension;		/* extension number */
	int major_opcode;	/* major op-code assigned by server */
	int first_event;	/* first event number for the extension */
	int first_error;	/* first error number for the extension */
} XExtCodes;

/*
 * Data structure for retrieving info about pixmap formats.
 */

typedef struct {
    int depth;
    int bits_per_pixel;
    int scanline_pad;
} XPixmapFormatValues;


/*
 * Data structure for setting graphics context.
 */
typedef struct {
	int function;		/* logical operation */
	unsigned long plane_mask;/* plane mask */
	unsigned long foreground;/* foreground pixel */
	unsigned long background;/* background pixel */
	int line_width;		/* line width */
	int line_style;	 	/* LineSolid, LineOnOffDash, LineDoubleDash */
	int cap_style;	  	/* CapNotLast, CapButt,
				   CapRound, CapProjecting */
	int join_style;	 	/* JoinMiter, JoinRound, JoinBevel */
	int fill_style;	 	/* FillSolid, FillTiled,
				   FillStippled, FillOpaqueStippled */
	int fill_rule;	  	/* EvenOddRule, WindingRule */
	int arc_mode;		/* ArcChord, ArcPieSlice */
	Pixmap tile;		/* tile pixmap for tiling operations */
	Pixmap stipple;		/* stipple 1 plane pixmap for stippling */
	int ts_x_origin;	/* offset for tile or stipple operations */
	int ts_y_origin;
        Font font;	        /* default text font for text operations */
	int subwindow_mode;     /* ClipByChildren, IncludeInferiors */
	Bool graphics_exposures;/* boolean, should exposures be generated */
	int clip_x_origin;	/* origin for clipping */
	int clip_y_origin;
	Pixmap clip_mask;	/* bitmap clipping; other calls for rects */
	int dash_offset;	/* patterned/dashed line information */
	char dashes;
} XGCValues;

/*
 * Graphics context.  The contents of this structure are implementation
 * dependent.  A GC should be treated as opaque by application code.
 */

typedef struct _XGC
#ifdef XLIB_ILLEGAL_ACCESS
{
    XExtData *ext_data;	/* hook for extension to hang data */
    GContext gid;	/* protocol ID for graphics context */
    /* there is more to this structure, but it is private to Xlib */
}
#endif
*GC;

/*
 * Visual structure; contains information about colormapping possible.
 */
typedef struct {
	XExtData *ext_data;	/* hook for extension to hang data */
	VisualID visualid;	/* visual id of this visual */
#if defined(__cplusplus) || defined(c_plusplus)
	int c_class;		/* C++ class of screen (monochrome, etc.) */
#else
	int class;		/* class of screen (monochrome, etc.) */
#endif
	unsigned long red_mask, green_mask, blue_mask;	/* mask values */
	int bits_per_rgb;	/* log base 2 of distinct color values */
	int map_entries;	/* color map entries */
} Visual;

/*
 * Depth structure; contains information for each possible depth.
 */
typedef struct {
	int depth;		/* this depth (Z) of the depth */
	int nvisuals;		/* number of Visual types at this depth */
	Visual *visuals;	/* list of visuals possible at this depth */
} Depth;

/*
 * Information about the screen.  The contents of this structure are
 * implementation dependent.  A Screen should be treated as opaque
 * by application code.
 */

struct _XDisplay;		/* Forward declare before use for C++ */

typedef struct {
	XExtData *ext_data;	/* hook for extension to hang data */
	struct _XDisplay *display;/* back pointer to display structure */
	Window root;		/* Root window id. */
	int width, height;	/* width and height of screen */
	int mwidth, mheight;	/* width and height of  in millimeters */
	int ndepths;		/* number of depths possible */
	Depth *depths;		/* list of allowable depths on the screen */
	int root_depth;		/* bits per pixel */
	Visual *root_visual;	/* root visual */
	GC default_gc;		/* GC for the root root visual */
	Colormap cmap;		/* default color map */
	unsigned long white_pixel;
	unsigned long black_pixel;	/* White and Black pixel values */
	int max_maps, min_maps;	/* max and min color maps */
	int backing_store;	/* Never, WhenMapped, Always */
	Bool save_unders;
	long root_input_mask;	/* initial root input mask */
} Screen;

/*
 * Format structure; describes ZFormat data the screen will understand.
 */
typedef struct {
	XExtData *ext_data;	/* hook for extension to hang data */
	int depth;		/* depth of this image format */
	int bits_per_pixel;	/* bits/pixel at this depth */
	int scanline_pad;	/* scanline must padded to this multiple */
} ScreenFormat;

/*
 * Data structure for setting window attributes.
 */
typedef struct {
    Pixmap background_pixmap;	/* background or None or ParentRelative */
    unsigned long background_pixel;	/* background pixel */
    Pixmap border_pixmap;	/* border of the window */
    unsigned long border_pixel;	/* border pixel value */
    int bit_gravity;		/* one of bit gravity values */
    int win_gravity;		/* one of the window gravity values */
    int backing_store;		/* NotUseful, WhenMapped, Always */
    unsigned long backing_planes;/* planes to be preserved if possible */
    unsigned long backing_pixel;/* value to use in restoring planes */
    Bool save_under;		/* should bits under be saved? (popups) */
    long event_mask;		/* set of events that should be saved */
    long do_not_propagate_mask;	/* set of events that should not propagate */
    Bool override_redirect;	/* boolean value for override-redirect */
    Colormap colormap;		/* color map to be associated with window */
    Cursor cursor;		/* cursor to be displayed (or None) */
} XSetWindowAttributes;

typedef struct {
    int x, y;			/* location of window */
    int width, height;		/* width and height of window */
    int border_width;		/* border width of window */
    int depth;          	/* depth of window */
    Visual *visual;		/* the associated visual structure */
    Window root;        	/* root of screen containing window */
#if defined(__cplusplus) || defined(c_plusplus)
    int c_class;		/* C++ InputOutput, InputOnly*/
#else
    int class;			/* InputOutput, InputOnly*/
#endif
    int bit_gravity;		/* one of bit gravity values */
    int win_gravity;		/* one of the window gravity values */
    int backing_store;		/* NotUseful, WhenMapped, Always */
    unsigned long backing_planes;/* planes to be preserved if possible */
    unsigned long backing_pixel;/* value to be used when restoring planes */
    Bool save_under;		/* boolean, should bits under be saved? */
    Colormap colormap;		/* color map to be associated with window */
    Bool map_installed;		/* boolean, is color map currently installed*/
    int map_state;		/* IsUnmapped, IsUnviewable, IsViewable */
    long all_event_masks;	/* set of events all people have interest in*/
    long your_event_mask;	/* my event mask */
    long do_not_propagate_mask; /* set of events that should not propagate */
    Bool override_redirect;	/* boolean value for override-redirect */
    Screen *screen;		/* back pointer to correct screen */
} XWindowAttributes;

/*
 * Data structure for host setting; getting routines.
 *
 */

typedef struct {
	int family;		/* for example FamilyInternet */
	int length;		/* length of address, in bytes */
	char *address;		/* pointer to where to find the bytes */
} XHostAddress;

/*
 * Data structure for ServerFamilyInterpreted addresses in host routines
 */
typedef struct {
	int typelength;		/* length of type string, in bytes */
	int valuelength;	/* length of value string, in bytes */
	char *type;		/* pointer to where to find the type string */
	char *value;		/* pointer to where to find the address */
} XServerInterpretedAddress;

/*
 * Data structure for "image" data, used by image manipulation routines.
 */
typedef struct _XImage {
    int width, height;		/* size of image */
    int xoffset;		/* number of pixels offset in X direction */
    int format;			/* XYBitmap, XYPixmap, ZPixmap */
    char *data;			/* pointer to image data */
    int byte_order;		/* data byte order, LSBFirst, MSBFirst */
    int bitmap_unit;		/* quant. of scanline 8, 16, 32 */
    int bitmap_bit_order;	/* LSBFirst, MSBFirst */
    int bitmap_pad;		/* 8, 16, 32 either XY or ZPixmap */
    int depth;			/* depth of image */
    int bytes_per_line;		/* accelerator to next line */
    int bits_per_pixel;		/* bits per pixel (ZPixmap) */
    unsigned long red_mask;	/* bits in z arrangement */
    unsigned long green_mask;
    unsigned long blue_mask;
    XPointer obdata;		/* hook for the object routines to hang on */
    struct funcs {		/* image manipulation routines */
	struct _XImage *(*create_image)(
		struct _XDisplay* /* display */,
		Visual*		/* visual */,
		unsigned int	/* depth */,
		int		/* format */,
		int		/* offset */,
		char*		/* data */,
		unsigned int	/* width */,
		unsigned int	/* height */,
		int		/* bitmap_pad */,
		int		/* bytes_per_line */);
	int (*destroy_image)        (struct _XImage *);
	unsigned long (*get_pixel)  (struct _XImage *, int, int);
	int (*put_pixel)            (struct _XImage *, int, int, unsigned long);
	struct _XImage *(*sub_image)(struct _XImage *, int, int, unsigned int, unsigned int);
	int (*add_pixel)            (struct _XImage *, long);
	} f;
} XImage;

/*
 * Data structure for XReconfigureWindow
 */
typedef struct {
    int x, y;
    int width, height;
    int border_width;
    Window sibling;
    int stack_mode;
} XWindowChanges;

/*
 * Data structure used by color operations
 */
typedef struct {
	unsigned long pixel;
	unsigned short red, green, blue;
	char flags;  /* do_red, do_green, do_blue */
	char pad;
} XColor;

/*
 * Data structures for graphics operations.  On most machines, these are
 * congruent with the wire protocol structures, so reformatting the data
 * can be avoided on these architectures.
 */
typedef struct {
    short x1, y1, x2, y2;
} XSegment;

typedef struct {
    short x, y;
} XPoint;

typedef struct {
    short x, y;
    unsigned short width, height;
} XRectangle;

typedef struct {
    short x, y;
    unsigned short width, height;
    short angle1, angle2;
} XArc;


/* Data structure for XChangeKeyboardControl */

typedef struct {
        int key_click_percent;
        int bell_percent;
        int bell_pitch;
        int bell_duration;
        int led;
        int led_mode;
        int key;
        int auto_repeat_mode;   /* On, Off, Default */
} XKeyboardControl;

/* Data structure for XGetKeyboardControl */

typedef struct {
        int key_click_percent;
	int bell_percent;
	unsigned int bell_pitch, bell_duration;
	unsigned long led_mask;
	int global_auto_repeat;
	char auto_repeats[32];
} XKeyboardState;

/* Data structure for XGetMotionEvents.  */

typedef struct {
        Time time;
	short x, y;
} XTimeCoord;

/* Data structure for X{Set,Get}ModifierMapping */

typedef struct {
 	int max_keypermod;	/* The server's max # of keys per modifier */
 	KeyCode *modifiermap;	/* An 8 by max_keypermod array of modifiers */
} XModifierKeymap;


/*
 * Display datatype maintaining display specific data.
 * The contents of this structure are implementation dependent.
 * A Display should be treated as opaque by application code.
 */
#ifndef XLIB_ILLEGAL_ACCESS
typedef struct _XDisplay Display;
#endif

struct _XPrivate;		/* Forward declare before use for C++ */
struct _XrmHashBucketRec;

typedef struct
#ifdef XLIB_ILLEGAL_ACCESS
_XDisplay
#endif
{
	XExtData *ext_data;	/* hook for extension to hang data */
	struct _XPrivate *private1;
	int fd;			/* Network socket. */
	int private2;
	int proto_major_version;/* major version of server's X protocol */
	int proto_minor_version;/* minor version of servers X protocol */
	char *vendor;		/* vendor of the server hardware */
        XID private3;
	XID private4;
	XID private5;
	int private6;
	XID (*resource_alloc)(	/* allocator function */
		struct _XDisplay*
	);
	int byte_order;		/* screen byte order, LSBFirst, MSBFirst */
	int bitmap_unit;	/* padding and data requirements */
	int bitmap_pad;		/* padding requirements on bitmaps */
	int bitmap_bit_order;	/* LeastSignificant or MostSignificant */
	int nformats;		/* number of pixmap formats in list */
	ScreenFormat *pixmap_format;	/* pixmap format list */
	int private8;
	int release;		/* release of the server */
	struct _XPrivate *private9, *private10;
	int qlen;		/* Length of input event queue */
	unsigned long last_request_read; /* seq number of last event read */
	unsigned long request;	/* sequence number of last request. */
	XPointer private11;
	XPointer private12;
	XPointer private13;
	XPointer private14;
	unsigned max_request_size; /* maximum number 32 bit words in request*/
	struct _XrmHashBucketRec *db;
	int (*private15)(
		struct _XDisplay*
		);
	char *display_name;	/* "host:display" string used on this connect*/
	int default_screen;	/* default screen for operations */
	int nscreens;		/* number of screens on this server*/
	Screen *screens;	/* pointer to list of screens */
	unsigned long motion_buffer;	/* size of motion buffer */
	unsigned long private16;
	int min_keycode;	/* minimum defined keycode */
	int max_keycode;	/* maximum defined keycode */
	XPointer private17;
	XPointer private18;
	int private19;
	char *xdefaults;	/* contents of defaults from server */
	/* there is more to this structure, but it is private to Xlib */
}
#ifdef XLIB_ILLEGAL_ACCESS
Display,
#endif
*_XPrivDisplay;

#undef _XEVENT_
#ifndef _XEVENT_
/*
 * Definitions of specific events.
 */
typedef struct {
	int type;		/* of event */
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window window;	        /* "event" window it is reported relative to */
	Window root;	        /* root window that the event occurred on */
	Window subwindow;	/* child window */
	Time time;		/* milliseconds */
	int x, y;		/* pointer x, y coordinates in event window */
	int x_root, y_root;	/* coordinates relative to root */
	unsigned int state;	/* key or button mask */
	unsigned int keycode;	/* detail */
	Bool same_screen;	/* same screen flag */
} XKeyEvent;
typedef XKeyEvent XKeyPressedEvent;
typedef XKeyEvent XKeyReleasedEvent;

typedef struct {
	int type;		/* of event */
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window window;	        /* "event" window it is reported relative to */
	Window root;	        /* root window that the event occurred on */
	Window subwindow;	/* child window */
	Time time;		/* milliseconds */
	int x, y;		/* pointer x, y coordinates in event window */
	int x_root, y_root;	/* coordinates relative to root */
	unsigned int state;	/* key or button mask */
	unsigned int button;	/* detail */
	Bool same_screen;	/* same screen flag */
} XButtonEvent;
typedef XButtonEvent XButtonPressedEvent;
typedef XButtonEvent XButtonReleasedEvent;

typedef struct {
	int type;		/* of event */
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window window;	        /* "event" window reported relative to */
	Window root;	        /* root window that the event occurred on */
	Window subwindow;	/* child window */
	Time time;		/* milliseconds */
	int x, y;		/* pointer x, y coordinates in event window */
	int x_root, y_root;	/* coordinates relative to root */
	unsigned int state;	/* key or button mask */
	char is_hint;		/* detail */
	Bool same_screen;	/* same screen flag */
} XMotionEvent;
typedef XMotionEvent XPointerMovedEvent;

typedef struct {
	int type;		/* of event */
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window window;	        /* "event" window reported relative to */
	Window root;	        /* root window that the event occurred on */
	Window subwindow;	/* child window */
	Time time;		/* milliseconds */
	int x, y;		/* pointer x, y coordinates in event window */
	int x_root, y_root;	/* coordinates relative to root */
	int mode;		/* NotifyNormal, NotifyGrab, NotifyUngrab */
	int detail;
	/*
	 * NotifyAncestor, NotifyVirtual, NotifyInferior,
	 * NotifyNonlinear,NotifyNonlinearVirtual
	 */
	Bool same_screen;	/* same screen flag */
	Bool focus;		/* boolean focus */
	unsigned int state;	/* key or button mask */
} XCrossingEvent;
typedef XCrossingEvent XEnterWindowEvent;
typedef XCrossingEvent XLeaveWindowEvent;

typedef struct {
	int type;		/* FocusIn or FocusOut */
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window window;		/* window of event */
	int mode;		/* NotifyNormal, NotifyWhileGrabbed,
				   NotifyGrab, NotifyUngrab */
	int detail;
	/*
	 * NotifyAncestor, NotifyVirtual, NotifyInferior,
	 * NotifyNonlinear,NotifyNonlinearVirtual, NotifyPointer,
	 * NotifyPointerRoot, NotifyDetailNone
	 */
} XFocusChangeEvent;
typedef XFocusChangeEvent XFocusInEvent;
typedef XFocusChangeEvent XFocusOutEvent;

/* generated on EnterWindow and FocusIn  when KeyMapState selected */
typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window window;
	char key_vector[32];
} XKeymapEvent;

typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window window;
	int x, y;
	int width, height;
	int count;		/* if non-zero, at least this many more */
} XExposeEvent;

typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Drawable drawable;
	int x, y;
	int width, height;
	int count;		/* if non-zero, at least this many more */
	int major_code;		/* core is CopyArea or CopyPlane */
	int minor_code;		/* not defined in the core */
} XGraphicsExposeEvent;

typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Drawable drawable;
	int major_code;		/* core is CopyArea or CopyPlane */
	int minor_code;		/* not defined in the core */
} XNoExposeEvent;

typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window window;
	int state;		/* Visibility state */
} XVisibilityEvent;

typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window parent;		/* parent of the window */
	Window window;		/* window id of window created */
	int x, y;		/* window location */
	int width, height;	/* size of window */
	int border_width;	/* border width */
	Bool override_redirect;	/* creation should be overridden */
} XCreateWindowEvent;

typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window event;
	Window window;
} XDestroyWindowEvent;

typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window event;
	Window window;
	Bool from_configure;
} XUnmapEvent;

typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window event;
	Window window;
	Bool override_redirect;	/* boolean, is override set... */
} XMapEvent;

typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window parent;
	Window window;
} XMapRequestEvent;

typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window event;
	Window window;
	Window parent;
	int x, y;
	Bool override_redirect;
} XReparentEvent;

typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window event;
	Window window;
	int x, y;
	int width, height;
	int border_width;
	Window above;
	Bool override_redirect;
} XConfigureEvent;

typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window event;
	Window window;
	int x, y;
} XGravityEvent;

typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window window;
	int width, height;
} XResizeRequestEvent;

typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window parent;
	Window window;
	int x, y;
	int width, height;
	int border_width;
	Window above;
	int detail;		/* Above, Below, TopIf, BottomIf, Opposite */
	unsigned long value_mask;
} XConfigureRequestEvent;

typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window event;
	Window window;
	int place;		/* PlaceOnTop, PlaceOnBottom */
} XCirculateEvent;

typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window parent;
	Window window;
	int place;		/* PlaceOnTop, PlaceOnBottom */
} XCirculateRequestEvent;

typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window window;
	Atom atom;
	Time time;
	int state;		/* NewValue, Deleted */
} XPropertyEvent;

typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window window;
	Atom selection;
	Time time;
} XSelectionClearEvent;

typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window owner;
	Window requestor;
	Atom selection;
	Atom target;
	Atom property;
	Time time;
} XSelectionRequestEvent;

typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window requestor;
	Atom selection;
	Atom target;
	Atom property;		/* ATOM or None */
	Time time;
} XSelectionEvent;

typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window window;
	Colormap colormap;	/* COLORMAP or None */
#if defined(__cplusplus) || defined(c_plusplus)
	Bool c_new;		/* C++ */
#else
	Bool new;
#endif
	int state;		/* ColormapInstalled, ColormapUninstalled */
} XColormapEvent;

typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window window;
	Atom message_type;
	int format;
	union {
		char b[20];
		short s[10];
		long l[5];
		} data;
} XClientMessageEvent;

typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;	/* Display the event was read from */
	Window window;		/* unused */
	int request;		/* one of MappingModifier, MappingKeyboard,
				   MappingPointer */
	int first_keycode;	/* first keycode */
	int count;		/* defines range of change w. first_keycode*/
} XMappingEvent;

typedef struct {
	int type;
	Display *display;	/* Display the event was read from */
	XID resourceid;		/* resource id */
	unsigned long serial;	/* serial number of failed request */
	unsigned char error_code;	/* error code of failed request */
	unsigned char request_code;	/* Major op-code of failed request */
	unsigned char minor_code;	/* Minor op-code of failed request */
} XErrorEvent;

typedef struct {
	int type;
	unsigned long serial;	/* # of last request processed by server */
	Bool send_event;	/* true if this came from a SendEvent request */
	Display *display;/* Display the event was read from */
	Window window;	/* window on which event was requested in event mask */
} XAnyEvent;


/***************************************************************
 *
 * GenericEvent.  This event is the standard event for all newer extensions.
 */

typedef struct
    {
    int            type;         /* of event. Always GenericEvent */
    unsigned long  serial;       /* # of last request processed */
    Bool           send_event;   /* true if from SendEvent request */
    Display        *display;     /* Display the event was read from */
    int            extension;    /* major opcode of extension that caused the event */
    int            evtype;       /* actual event type. */
    } XGenericEvent;

typedef struct {
    int            type;         /* of event. Always GenericEvent */
    unsigned long  serial;       /* # of last request processed */
    Bool           send_event;   /* true if from SendEvent request */
    Display        *display;     /* Display the event was read from */
    int            extension;    /* major opcode of extension that caused the event */
    int            evtype;       /* actual event type. */
    unsigned int   cookie;
    void           *data;
} XGenericEventCookie;

/*
 * this union is defined so Xlib can always use the same sized
 * event structure internally, to avoid memory fragmentation.
 */
typedef union _XEvent {
        int type;		/* must not be changed; first element */
	XAnyEvent xany;
	XKeyEvent xkey;
	XButtonEvent xbutton;
	XMotionEvent xmotion;
	XCrossingEvent xcrossing;
	XFocusChangeEvent xfocus;
	XExposeEvent xexpose;
	XGraphicsExposeEvent xgraphicsexpose;
	XNoExposeEvent xnoexpose;
	XVisibilityEvent xvisibility;
	XCreateWindowEvent xcreatewindow;
	XDestroyWindowEvent xdestroywindow;
	XUnmapEvent xunmap;
	XMapEvent xmap;
	XMapRequestEvent xmaprequest;
	XReparentEvent xreparent;
	XConfigureEvent xconfigure;
	XGravityEvent xgravity;
	XResizeRequestEvent xresizerequest;
	XConfigureRequestEvent xconfigurerequest;
	XCirculateEvent xcirculate;
	XCirculateRequestEvent xcirculaterequest;
	XPropertyEvent xproperty;
	XSelectionClearEvent xselectionclear;
	XSelectionRequestEvent xselectionrequest;
	XSelectionEvent xselection;
	XColormapEvent xcolormap;
	XClientMessageEvent xclient;
	XMappingEvent xmapping;
	XErrorEvent xerror;
	XKeymapEvent xkeymap;
	XGenericEvent xgeneric;
	XGenericEventCookie xcookie;
	long pad[24];
} XEvent;
#endif

#define XAllocID(dpy) ((*((_XPrivDisplay)(dpy))->resource_alloc)((dpy)))

/*
 * per character font metric information.
 */
typedef struct {
    short	lbearing;	/* origin to left edge of raster */
    short	rbearing;	/* origin to right edge of raster */
    short	width;		/* advance to next char's origin */
    short	ascent;		/* baseline to top edge of raster */
    short	descent;	/* baseline to bottom edge of raster */
    unsigned short attributes;	/* per char flags (not predefined) */
} XCharStruct;

/*
 * To allow arbitrary information with fonts, there are additional properties
 * returned.
 */
typedef struct {
    Atom name;
    unsigned long card32;
} XFontProp;

typedef struct {
    XExtData	*ext_data;	/* hook for extension to hang data */
    Font        fid;            /* Font id for this font */
    unsigned	direction;	/* hint about direction the font is painted */
    unsigned	min_char_or_byte2;/* first character */
    unsigned	max_char_or_byte2;/* last character */
    unsigned	min_byte1;	/* first row that exists */
    unsigned	max_byte1;	/* last row that exists */
    Bool	all_chars_exist;/* flag if all characters have non-zero size*/
    unsigned	default_char;	/* char to print for undefined character */
    int         n_properties;   /* how many properties there are */
    XFontProp	*properties;	/* pointer to array of additional properties*/
    XCharStruct	min_bounds;	/* minimum bounds over all existing char*/
    XCharStruct	max_bounds;	/* maximum bounds over all existing char*/
    XCharStruct	*per_char;	/* first_char to last_char information */
    int		ascent;		/* log. extent above baseline for spacing */
    int		descent;	/* log. descent below baseline for spacing */
} XFontStruct;

/*
 * PolyText routines take these as arguments.
 */
typedef struct {
    char *chars;		/* pointer to string */
    int nchars;			/* number of characters */
    int delta;			/* delta between strings */
    Font font;			/* font to print it in, None don't change */
} XTextItem;

typedef struct {		/* normal 16 bit characters are two bytes */
    unsigned char byte1;
    unsigned char byte2;
} XChar2b;

typedef struct {
    XChar2b *chars;		/* two byte characters */
    int nchars;			/* number of characters */
    int delta;			/* delta between strings */
    Font font;			/* font to print it in, None don't change */
} XTextItem16;


typedef union { Display *display;
		GC gc;
		Visual *visual;
		Screen *screen;
		ScreenFormat *pixmap_format;
		XFontStruct *font; } XEDataObject;

typedef struct {
    XRectangle      max_ink_extent;
    XRectangle      max_logical_extent;
} XFontSetExtents;

/* unused:
typedef void (*XOMProc)();
 */

typedef struct _XOM *XOM;
typedef struct _XOC *XOC, *XFontSet;

typedef struct {
    char           *chars;
    int             nchars;
    int             delta;
    XFontSet        font_set;
} XmbTextItem;

typedef struct {
    wchar_t        *chars;
    int             nchars;
    int             delta;
    XFontSet        font_set;
} XwcTextItem;

#define XNRequiredCharSet "requiredCharSet"
#define XNQueryOrientation "queryOrientation"
#define XNBaseFontName "baseFontName"
#define XNOMAutomatic "omAutomatic"
#define XNMissingCharSet "missingCharSet"
#define XNDefaultString "defaultString"
#define XNOrientation "orientation"
#define XNDirectionalDependentDrawing "directionalDependentDrawing"
#define XNContextualDrawing "contextualDrawing"
#define XNFontInfo "fontInfo"

typedef struct {
    int charset_count;
    char **charset_list;
} XOMCharSetList;

typedef enum {
    XOMOrientation_LTR_TTB,
    XOMOrientation_RTL_TTB,
    XOMOrientation_TTB_LTR,
    XOMOrientation_TTB_RTL,
    XOMOrientation_Context
} XOrientation;

typedef struct {
    int num_orientation;
    XOrientation *orientation;	/* Input Text description */
} XOMOrientation;

typedef struct {
    int num_font;
    XFontStruct **font_struct_list;
    char **font_name_list;
} XOMFontInfo;

typedef struct _XIM *XIM;
typedef struct _XIC *XIC;

typedef void (*XIMProc)(
    XIM,
    XPointer,
    XPointer
);

typedef Bool (*XICProc)(
    XIC,
    XPointer,
    XPointer
);

typedef void (*XIDProc)(
    Display*,
    XPointer,
    XPointer
);

typedef unsigned long XIMStyle;

typedef struct {
    unsigned short count_styles;
    XIMStyle *supported_styles;
} XIMStyles;

#define XIMPreeditArea		0x0001L
#define XIMPreeditCallbacks	0x0002L
#define XIMPreeditPosition	0x0004L
#define XIMPreeditNothing	0x0008L
#define XIMPreeditNone		0x0010L
#define XIMStatusArea		0x0100L
#define XIMStatusCallbacks	0x0200L
#define XIMStatusNothing	0x0400L
#define XIMStatusNone		0x0800L

#define XNVaNestedList "XNVaNestedList"
#define XNQueryInputStyle "queryInputStyle"
#define XNClientWindow "clientWindow"
#define XNInputStyle "inputStyle"
#define XNFocusWindow "focusWindow"
#define XNResourceName "resourceName"
#define XNResourceClass "resourceClass"
#define XNGeometryCallback "geometryCallback"
#define XNDestroyCallback "destroyCallback"
#define XNFilterEvents "filterEvents"
#define XNPreeditStartCallback "preeditStartCallback"
#define XNPreeditDoneCallback "preeditDoneCallback"
#define XNPreeditDrawCallback "preeditDrawCallback"
#define XNPreeditCaretCallback "preeditCaretCallback"
#define XNPreeditStateNotifyCallback "preeditStateNotifyCallback"
#define XNPreeditAttributes "preeditAttributes"
#define XNStatusStartCallback "statusStartCallback"
#define XNStatusDoneCallback "statusDoneCallback"
#define XNStatusDrawCallback "statusDrawCallback"
#define XNStatusAttributes "statusAttributes"
#define XNArea "area"
#define XNAreaNeeded "areaNeeded"
#define XNSpotLocation "spotLocation"
#define XNColormap "colorMap"
#define XNStdColormap "stdColorMap"
#define XNForeground "foreground"
#define XNBackground "background"
#define XNBackgroundPixmap "backgroundPixmap"
#define XNFontSet "fontSet"
#define XNLineSpace "lineSpace"
#define XNCursor "cursor"

#define XNQueryIMValuesList "queryIMValuesList"
#define XNQueryICValuesList "queryICValuesList"
#define XNVisiblePosition "visiblePosition"
#define XNR6PreeditCallback "r6PreeditCallback"
#define XNStringConversionCallback "stringConversionCallback"
#define XNStringConversion "stringConversion"
#define XNResetState "resetState"
#define XNHotKey "hotKey"
#define XNHotKeyState "hotKeyState"
#define XNPreeditState "preeditState"
#define XNSeparatorofNestedList "separatorofNestedList"

#define XBufferOverflow		-1
#define XLookupNone		1
#define XLookupChars		2
#define XLookupKeySym		3
#define XLookupBoth		4

typedef void *XVaNestedList;

typedef struct {
    XPointer client_data;
    XIMProc callback;
} XIMCallback;

typedef struct {
    XPointer client_data;
    XICProc callback;
} XICCallback;

typedef unsigned long XIMFeedback;

#define XIMReverse		1L
#define XIMUnderline		(1L<<1)
#define XIMHighlight		(1L<<2)
#define XIMPrimary	 	(1L<<5)
#define XIMSecondary		(1L<<6)
#define XIMTertiary	 	(1L<<7)
#define XIMVisibleToForward 	(1L<<8)
#define XIMVisibleToBackword 	(1L<<9)
#define XIMVisibleToCenter 	(1L<<10)

typedef struct _XIMText {
    unsigned short length;
    XIMFeedback *feedback;
    Bool encoding_is_wchar;
    union {
	char *multi_byte;
	wchar_t *wide_char;
    } string;
} XIMText;

typedef	unsigned long	 XIMPreeditState;

#define	XIMPreeditUnKnown	0L
#define	XIMPreeditEnable	1L
#define	XIMPreeditDisable	(1L<<1)

typedef	struct	_XIMPreeditStateNotifyCallbackStruct {
    XIMPreeditState state;
} XIMPreeditStateNotifyCallbackStruct;

typedef	unsigned long	 XIMResetState;

#define	XIMInitialState		1L
#define	XIMPreserveState	(1L<<1)

typedef unsigned long XIMStringConversionFeedback;

#define	XIMStringConversionLeftEdge	(0x00000001)
#define	XIMStringConversionRightEdge	(0x00000002)
#define	XIMStringConversionTopEdge	(0x00000004)
#define	XIMStringConversionBottomEdge	(0x00000008)
#define	XIMStringConversionConcealed	(0x00000010)
#define	XIMStringConversionWrapped	(0x00000020)

typedef struct _XIMStringConversionText {
    unsigned short length;
    XIMStringConversionFeedback *feedback;
    Bool encoding_is_wchar;
    union {
	char *mbs;
	wchar_t *wcs;
    } string;
} XIMStringConversionText;

typedef	unsigned short	XIMStringConversionPosition;

typedef	unsigned short	XIMStringConversionType;

#define	XIMStringConversionBuffer	(0x0001)
#define	XIMStringConversionLine		(0x0002)
#define	XIMStringConversionWord		(0x0003)
#define	XIMStringConversionChar		(0x0004)

typedef	unsigned short	XIMStringConversionOperation;

#define	XIMStringConversionSubstitution	(0x0001)
#define	XIMStringConversionRetrieval	(0x0002)

typedef enum {
    XIMForwardChar, XIMBackwardChar,
    XIMForwardWord, XIMBackwardWord,
    XIMCaretUp, XIMCaretDown,
    XIMNextLine, XIMPreviousLine,
    XIMLineStart, XIMLineEnd,
    XIMAbsolutePosition,
    XIMDontChange
} XIMCaretDirection;

typedef struct _XIMStringConversionCallbackStruct {
    XIMStringConversionPosition position;
    XIMCaretDirection direction;
    XIMStringConversionOperation operation;
    unsigned short factor;
    XIMStringConversionText *text;
} XIMStringConversionCallbackStruct;

typedef struct _XIMPreeditDrawCallbackStruct {
    int caret;		/* Cursor offset within pre-edit string */
    int chg_first;	/* Starting change position */
    int chg_length;	/* Length of the change in character count */
    XIMText *text;
} XIMPreeditDrawCallbackStruct;

typedef enum {
    XIMIsInvisible,	/* Disable caret feedback */
    XIMIsPrimary,	/* UI defined caret feedback */
    XIMIsSecondary	/* UI defined caret feedback */
} XIMCaretStyle;

typedef struct _XIMPreeditCaretCallbackStruct {
    int position;		 /* Caret offset within pre-edit string */
    XIMCaretDirection direction; /* Caret moves direction */
    XIMCaretStyle style;	 /* Feedback of the caret */
} XIMPreeditCaretCallbackStruct;

typedef enum {
    XIMTextType,
    XIMBitmapType
} XIMStatusDataType;

typedef struct _XIMStatusDrawCallbackStruct {
    XIMStatusDataType type;
    union {
	XIMText *text;
	Pixmap  bitmap;
    } data;
} XIMStatusDrawCallbackStruct;

typedef struct _XIMHotKeyTrigger {
    KeySym	 keysym;
    int		 modifier;
    int		 modifier_mask;
} XIMHotKeyTrigger;

typedef struct _XIMHotKeyTriggers {
    int			 num_hot_key;
    XIMHotKeyTrigger	*key;
} XIMHotKeyTriggers;

typedef	unsigned long	 XIMHotKeyState;

#define	XIMHotKeyStateON	(0x0001L)
#define	XIMHotKeyStateOFF	(0x0002L)

typedef struct {
    unsigned short count_values;
    char **supported_values;
} XIMValuesList;

_XFUNCPROTOBEGIN

#if defined(WIN32) && !defined(_XLIBINT_)
#define _Xdebug (*_Xdebug_p)
#endif

extern int _Xdebug;

extern XFontStruct *XLoadQueryFont(
    Display*		/* display */,
    _Xconst char*	/* name */
);

extern XFontStruct *XQueryFont(
    Display*		/* display */,
    XID			/* font_ID */
);


extern XTimeCoord *XGetMotionEvents(
    Display*		/* display */,
    Window		/* w */,
    Time		/* start */,
    Time		/* stop */,
    int*		/* nevents_return */
);

extern XModifierKeymap *XDeleteModifiermapEntry(
    XModifierKeymap*	/* modmap */,
#if NeedWidePrototypes
    unsigned int	/* keycode_entry */,
#else
    KeyCode		/* keycode_entry */,
#endif
    int			/* modifier */
);

extern XModifierKeymap	*XGetModifierMapping(
    Display*		/* display */
);

extern XModifierKeymap	*XInsertModifiermapEntry(
    XModifierKeymap*	/* modmap */,
#if NeedWidePrototypes
    unsigned int	/* keycode_entry */,
#else
    KeyCode		/* keycode_entry */,
#endif
    int			/* modifier */
);

extern XModifierKeymap *XNewModifiermap(
    int			/* max_keys_per_mod */
);

extern XImage *XCreateImage(
    Display*		/* display */,
    Visual*		/* visual */,
    unsigned int	/* depth */,
    int			/* format */,
    int			/* offset */,
    char*		/* data */,
    unsigned int	/* width */,
    unsigned int	/* height */,
    int			/* bitmap_pad */,
    int			/* bytes_per_line */
);
extern Status XInitImage(
    XImage*		/* image */
);
extern XImage *XGetImage(
    Display*		/* display */,
    Drawable		/* d */,
    int			/* x */,
    int			/* y */,
    unsigned int	/* width */,
    unsigned int	/* height */,
    unsigned long	/* plane_mask */,
    int			/* format */
);
extern XImage *XGetSubImage(
    Display*		/* display */,
    Drawable		/* d */,
    int			/* x */,
    int			/* y */,
    unsigned int	/* width */,
    unsigned int	/* height */,
    unsigned long	/* plane_mask */,
    int			/* format */,
    XImage*		/* dest_image */,
    int			/* dest_x */,
    int			/* dest_y */
);

/*
 * X function declarations.
 */
extern Display *XOpenDisplay(
    _Xconst char*	/* display_name */
);

extern void XrmInitialize(
    void
);

extern char *XFetchBytes(
    Display*		/* display */,
    int*		/* nbytes_return */
);
extern char *XFetchBuffer(
    Display*		/* display */,
    int*		/* nbytes_return */,
    int			/* buffer */
);
extern char *XGetAtomName(
    Display*		/* display */,
    Atom		/* atom */
);
extern Status XGetAtomNames(
    Display*		/* dpy */,
    Atom*		/* atoms */,
    int			/* count */,
    char**		/* names_return */
);
extern char *XGetDefault(
    Display*		/* display */,
    _Xconst char*	/* program */,
    _Xconst char*	/* option */
);
extern char *XDisplayName(
    _Xconst char*	/* string */
);
extern char *XKeysymToString(
    KeySym		/* keysym */
);

extern int (*XSynchronize(
    Display*		/* display */,
    Bool		/* onoff */
))(
    Display*		/* display */
);
extern int (*XSetAfterFunction(
    Display*		/* display */,
    int (*) (
	     Display*	/* display */
            )		/* procedure */
))(
    Display*		/* display */
);
extern Atom XInternAtom(
    Display*		/* display */,
    _Xconst char*	/* atom_name */,
    Bool		/* only_if_exists */
);
extern Status XInternAtoms(
    Display*		/* dpy */,
    char**		/* names */,
    int			/* count */,
    Bool		/* onlyIfExists */,
    Atom*		/* atoms_return */
);
extern Colormap XCopyColormapAndFree(
    Display*		/* display */,
    Colormap		/* colormap */
);
extern Colormap XCreateColormap(
    Display*		/* display */,
    Window		/* w */,
    Visual*		/* visual */,
    int			/* alloc */
);
extern Cursor XCreatePixmapCursor(
    Display*		/* display */,
    Pixmap		/* source */,
    Pixmap		/* mask */,
    XColor*		/* foreground_color */,
    XColor*		/* background_color */,
    unsigned int	/* x */,
    unsigned int	/* y */
);
extern Cursor XCreateGlyphCursor(
    Display*		/* display */,
    Font		/* source_font */,
    Font		/* mask_font */,
    unsigned int	/* source_char */,
    unsigned int	/* mask_char */,
    XColor _Xconst *	/* foreground_color */,
    XColor _Xconst *	/* background_color */
);
extern Cursor XCreateFontCursor(
    Display*		/* display */,
    unsigned int	/* shape */
);
extern Font XLoadFont(
    Display*		/* display */,
    _Xconst char*	/* name */
);
extern GC XCreateGC(
    Display*		/* display */,
    Drawable		/* d */,
    unsigned long	/* valuemask */,
    XGCValues*		/* values */
);
extern GContext XGContextFromGC(
    GC			/* gc */
);
extern void XFlushGC(
    Display*		/* display */,
    GC			/* gc */
);
extern Pixmap XCreatePixmap(
    Display*		/* display */,
    Drawable		/* d */,
    unsigned int	/* width */,
    unsigned int	/* height */,
    unsigned int	/* depth */
);
extern Pixmap XCreateBitmapFromData(
    Display*		/* display */,
    Drawable		/* d */,
    _Xconst char*	/* data */,
    unsigned int	/* width */,
    unsigned int	/* height */
);
extern Pixmap XCreatePixmapFromBitmapData(
    Display*		/* display */,
    Drawable		/* d */,
    char*		/* data */,
    unsigned int	/* width */,
    unsigned int	/* height */,
    unsigned long	/* fg */,
    unsigned long	/* bg */,
    unsigned int	/* depth */
);
extern Window XCreateSimpleWindow(
    Display*		/* display */,
    Window		/* parent */,
    int			/* x */,
    int			/* y */,
    unsigned int	/* width */,
    unsigned int	/* height */,
    unsigned int	/* border_width */,
    unsigned long	/* border */,
    unsigned long	/* background */
);
extern Window XGetSelectionOwner(
    Display*		/* display */,
    Atom		/* selection */
);
extern Window XCreateWindow(
    Display*		/* display */,
    Window		/* parent */,
    int			/* x */,
    int			/* y */,
    unsigned int	/* width */,
    unsigned int	/* height */,
    unsigned int	/* border_width */,
    int			/* depth */,
    unsigned int	/* class */,
    Visual*		/* visual */,
    unsigned long	/* valuemask */,
    XSetWindowAttributes*	/* attributes */
);
extern Colormap *XListInstalledColormaps(
    Display*		/* display */,
    Window		/* w */,
    int*		/* num_return */
);
extern char **XListFonts(
    Display*		/* display */,
    _Xconst char*	/* pattern */,
    int			/* maxnames */,
    int*		/* actual_count_return */
);
extern char **XListFontsWithInfo(
    Display*		/* display */,
    _Xconst char*	/* pattern */,
    int			/* maxnames */,
    int*		/* count_return */,
    XFontStruct**	/* info_return */
);
extern char **XGetFontPath(
    Display*		/* display */,
    int*		/* npaths_return */
);
extern char **XListExtensions(
    Display*		/* display */,
    int*		/* nextensions_return */
);
extern Atom *XListProperties(
    Display*		/* display */,
    Window		/* w */,
    int*		/* num_prop_return */
);
extern XHostAddress *XListHosts(
    Display*		/* display */,
    int*		/* nhosts_return */,
    Bool*		/* state_return */
);
_X_DEPRECATED
extern KeySym XKeycodeToKeysym(
    Display*		/* display */,
#if NeedWidePrototypes
    unsigned int	/* keycode */,
#else
    KeyCode		/* keycode */,
#endif
    int			/* index */
);
extern KeySym XLookupKeysym(
    XKeyEvent*		/* key_event */,
    int			/* index */
);
extern KeySym *XGetKeyboardMapping(
    Display*		/* display */,
#if NeedWidePrototypes
    unsigned int	/* first_keycode */,
#else
    KeyCode		/* first_keycode */,
#endif
    int			/* keycode_count */,
    int*		/* keysyms_per_keycode_return */
);
extern KeySym XStringToKeysym(
    _Xconst char*	/* string */
);
extern long XMaxRequestSize(
    Display*		/* display */
);
extern long XExtendedMaxRequestSize(
    Display*		/* display */
);
extern char *XResourceManagerString(
    Display*		/* display */
);
extern char *XScreenResourceString(
	Screen*		/* screen */
);
extern unsigned long XDisplayMotionBufferSize(
    Display*		/* display */
);
extern VisualID XVisualIDFromVisual(
    Visual*		/* visual */
);

/* multithread routines */

extern Status XInitThreads(
    void
);

extern Status XFreeThreads(
    void
);

extern void XLockDisplay(
    Display*		/* display */
);

extern void XUnlockDisplay(
    Display*		/* display */
);

/* routines for dealing with extensions */

extern XExtCodes *XInitExtension(
    Display*		/* display */,
    _Xconst char*	/* name */
);

extern XExtCodes *XAddExtension(
    Display*		/* display */
);
extern XExtData *XFindOnExtensionList(
    XExtData**		/* structure */,
    int			/* number */
);
extern XExtData **XEHeadOfExtensionList(
    XEDataObject	/* object */
);

/* these are routines for which there are also macros */
extern Window XRootWindow(
    Display*		/* display */,
    int			/* screen_number */
);
extern Window XDefaultRootWindow(
    Display*		/* display */
);
extern Window XRootWindowOfScreen(
    Screen*		/* screen */
);
extern Visual *XDefaultVisual(
    Display*		/* display */,
    int			/* screen_number */
);
extern Visual *XDefaultVisualOfScreen(
    Screen*		/* screen */
);
extern GC XDefaultGC(
    Display*		/* display */,
    int			/* screen_number */
);
extern GC XDefaultGCOfScreen(
    Screen*		/* screen */
);
extern unsigned long XBlackPixel(
    Display*		/* display */,
    int			/* screen_number */
);
extern unsigned long XWhitePixel(
    Display*		/* display */,
    int			/* screen_number */
);
extern unsigned long XAllPlanes(
    void
);
extern unsigned long XBlackPixelOfScreen(
    Screen*		/* screen */
);
extern unsigned long XWhitePixelOfScreen(
    Screen*		/* screen */
);
extern unsigned long XNextRequest(
    Display*		/* display */
);
extern unsigned long XLastKnownRequestProcessed(
    Display*		/* display */
);
extern char *XServerVendor(
    Display*		/* display */
);
extern char *XDisplayString(
    Display*		/* display */
);
extern Colormap XDefaultColormap(
    Display*		/* display */,
    int			/* screen_number */
);
extern Colormap XDefaultColormapOfScreen(
    Screen*		/* screen */
);
extern Display *XDisplayOfScreen(
    Screen*		/* screen */
);
extern Screen *XScreenOfDisplay(
    Display*		/* display */,
    int			/* screen_number */
);
extern Screen *XDefaultScreenOfDisplay(
    Display*		/* display */
);
extern long XEventMaskOfScreen(
    Screen*		/* screen */
);

extern int XScreenNumberOfScreen(
    Screen*		/* screen */
);

typedef int (*XErrorHandler) (	    /* WARNING, this type not in Xlib spec */
    Display*		/* display */,
    XErrorEvent*	/* error_event */
);

extern XErrorHandler XSetErrorHandler (
    XErrorHandler	/* handler */
);


typedef int (*XIOErrorHandler) (    /* WARNING, this type not in Xlib spec */
    Display*		/* display */
);

extern XIOErrorHandler XSetIOErrorHandler (
    XIOErrorHandler	/* handler */
);

typedef void (*XIOErrorExitHandler) ( /* WARNING, this type not in Xlib spec */
    Display*,		/* display */
    void*		/* user_data */
);

extern void XSetIOErrorExitHandler (
    Display*,			/* display */
    XIOErrorExitHandler,	/* handler */
    void*			/* user_data */
);

extern XPixmapFormatValues *XListPixmapFormats(
    Display*		/* display */,
    int*		/* count_return */
);
extern int *XListDepths(
    Display*		/* display */,
    int			/* screen_number */,
    int*		/* count_return */
);

/* ICCCM routines for things that don't require special include files; */
/* other declarations are given in Xutil.h                             */
extern Status XReconfigureWMWindow(
    Display*		/* display */,
    Window		/* w */,
    int			/* screen_number */,
    unsigned int	/* mask */,
    XWindowChanges*	/* changes */
);

extern Status XGetWMProtocols(
    Display*		/* display */,
    Window		/* w */,
    Atom**		/* protocols_return */,
    int*		/* count_return */
);
extern Status XSetWMProtocols(
    Display*		/* display */,
    Window		/* w */,
    Atom*		/* protocols */,
    int			/* count */
);
extern Status XIconifyWindow(
    Display*		/* display */,
    Window		/* w */,
    int			/* screen_number */
);
extern Status XWithdrawWindow(
    Display*		/* display */,
    Window		/* w */,
    int			/* screen_number */
);
extern Status XGetCommand(
    Display*		/* display */,
    Window		/* w */,
    char***		/* argv_return */,
    int*		/* argc_return */
);
extern Status XGetWMColormapWindows(
    Display*		/* display */,
    Window		/* w */,
    Window**		/* windows_return */,
    int*		/* count_return */
);
extern Status XSetWMColormapWindows(
    Display*		/* display */,
    Window		/* w */,
    Window*		/* colormap_windows */,
    int			/* count */
);
extern void XFreeStringList(
    char**		/* list */
);
extern int XSetTransientForHint(
    Display*		/* display */,
    Window		/* w */,
    Window		/* prop_window */
);

/* The following are given in alphabetical order */

extern int XActivateScreenSaver(
    Display*		/* display */
);

extern int XAddHost(
    Display*		/* display */,
    XHostAddress*	/* host */
);

extern int XAddHosts(
    Display*		/* display */,
    XHostAddress*	/* hosts */,
    int			/* num_hosts */
);

extern int XAddToExtensionList(
    struct _XExtData**	/* structure */,
    XExtData*		/* ext_data */
);

extern int XAddToSaveSet(
    Display*		/* display */,
    Window		/* w */
);

extern Status XAllocColor(
    Display*		/* display */,
    Colormap		/* colormap */,
    XColor*		/* screen_in_out */
);

extern Status XAllocColorCells(
    Display*		/* display */,
    Colormap		/* colormap */,
    Bool	        /* contig */,
    unsigned long*	/* plane_masks_return */,
    unsigned int	/* nplanes */,
    unsigned long*	/* pixels_return */,
    unsigned int 	/* npixels */
);

extern Status XAllocColorPlanes(
    Display*		/* display */,
    Colormap		/* colormap */,
    Bool		/* contig */,
    unsigned long*	/* pixels_return */,
    int			/* ncolors */,
    int			/* nreds */,
    int			/* ngreens */,
    int			/* nblues */,
    unsigned long*	/* rmask_return */,
    unsigned long*	/* gmask_return */,
    unsigned long*	/* bmask_return */
);

extern Status XAllocNamedColor(
    Display*		/* display */,
    Colormap		/* colormap */,
    _Xconst char*	/* color_name */,
    XColor*		/* screen_def_return */,
    XColor*		/* exact_def_return */
);

extern int XAllowEvents(
    Display*		/* display */,
    int			/* event_mode */,
    Time		/* time */
);

extern int XAutoRepeatOff(
    Display*		/* display */
);

extern int XAutoRepeatOn(
    Display*		/* display */
);

extern int XBell(
    Display*		/* display */,
    int			/* percent */
);

extern int XBitmapBitOrder(
    Display*		/* display */
);

extern int XBitmapPad(
    Display*		/* display */
);

extern int XBitmapUnit(
    Display*		/* display */
);

extern int XCellsOfScreen(
    Screen*		/* screen */
);

extern int XChangeActivePointerGrab(
    Display*		/* display */,
    unsigned int	/* event_mask */,
    Cursor		/* cursor */,
    Time		/* time */
);

extern int XChangeGC(
    Display*		/* display */,
    GC			/* gc */,
    unsigned long	/* valuemask */,
    XGCValues*		/* values */
);

extern int XChangeKeyboardControl(
    Display*		/* display */,
    unsigned long	/* value_mask */,
    XKeyboardControl*	/* values */
);

extern int XChangeKeyboardMapping(
    Display*		/* display */,
    int			/* first_keycode */,
    int			/* keysyms_per_keycode */,
    KeySym*		/* keysyms */,
    int			/* num_codes */
);

extern int XChangePointerControl(
    Display*		/* display */,
    Bool		/* do_accel */,
    Bool		/* do_threshold */,
    int			/* accel_numerator */,
    int			/* accel_denominator */,
    int			/* threshold */
);

extern int XChangeProperty(
    Display*		/* display */,
    Window		/* w */,
    Atom		/* property */,
    Atom		/* type */,
    int			/* format */,
    int			/* mode */,
    _Xconst unsigned char*	/* data */,
    int			/* nelements */
);

extern int XChangeSaveSet(
    Display*		/* display */,
    Window		/* w */,
    int			/* change_mode */
);

extern int XChangeWindowAttributes(
    Display*		/* display */,
    Window		/* w */,
    unsigned long	/* valuemask */,
    XSetWindowAttributes* /* attributes */
);

extern Bool XCheckIfEvent(
    Display*		/* display */,
    XEvent*		/* event_return */,
    Bool (*) (
	       Display*			/* display */,
               XEvent*			/* event */,
               XPointer			/* arg */
             )		/* predicate */,
    XPointer		/* arg */
);

extern Bool XCheckMaskEvent(
    Display*		/* display */,
    long		/* event_mask */,
    XEvent*		/* event_return */
);

extern Bool XCheckTypedEvent(
    Display*		/* display */,
    int			/* event_type */,
    XEvent*		/* event_return */
);

extern Bool XCheckTypedWindowEvent(
    Display*		/* display */,
    Window		/* w */,
    int			/* event_type */,
    XEvent*		/* event_return */
);

extern Bool XCheckWindowEvent(
    Display*		/* display */,
    Window		/* w */,
    long		/* event_mask */,
    XEvent*		/* event_return */
);

extern int XCirculateSubwindows(
    Display*		/* display */,
    Window		/* w */,
    int			/* direction */
);

extern int XCirculateSubwindowsDown(
    Display*		/* display */,
    Window		/* w */
);

extern int XCirculateSubwindowsUp(
    Display*		/* display */,
    Window		/* w */
);

extern int XClearArea(
    Display*		/* display */,
    Window		/* w */,
    int			/* x */,
    int			/* y */,
    unsigned int	/* width */,
    unsigned int	/* height */,
    Bool		/* exposures */
);

extern int XClearWindow(
    Display*		/* display */,
    Window		/* w */
);

extern int XCloseDisplay(
    Display*		/* display */
);

extern int XConfigureWindow(
    Display*		/* display */,
    Window		/* w */,
    unsigned int	/* value_mask */,
    XWindowChanges*	/* values */
);

extern int XConnectionNumber(
    Display*		/* display */
);

extern int XConvertSelection(
    Display*		/* display */,
    Atom		/* selection */,
    Atom 		/* target */,
    Atom		/* property */,
    Window		/* requestor */,
    Time		/* time */
);

extern int XCopyArea(
    Display*		/* display */,
    Drawable		/* src */,
    Drawable		/* dest */,
    GC			/* gc */,
    int			/* src_x */,
    int			/* src_y */,
    unsigned int	/* width */,
    unsigned int	/* height */,
    int			/* dest_x */,
    int			/* dest_y */
);

extern int XCopyGC(
    Display*		/* display */,
    GC			/* src */,
    unsigned long	/* valuemask */,
    GC			/* dest */
);

extern int XCopyPlane(
    Display*		/* display */,
    Drawable		/* src */,
    Drawable		/* dest */,
    GC			/* gc */,
    int			/* src_x */,
    int			/* src_y */,
    unsigned int	/* width */,
    unsigned int	/* height */,
    int			/* dest_x */,
    int			/* dest_y */,
    unsigned long	/* plane */
);

extern int XDefaultDepth(
    Display*		/* display */,
    int			/* screen_number */
);

extern int XDefaultDepthOfScreen(
    Screen*		/* screen */
);

extern int XDefaultScreen(
    Display*		/* display */
);

extern int XDefineCursor(
    Display*		/* display */,
    Window		/* w */,
    Cursor		/* cursor */
);

extern int XDeleteProperty(
    Display*		/* display */,
    Window		/* w */,
    Atom		/* property */
);

extern int XDestroyWindow(
    Display*		/* display */,
    Window		/* w */
);

extern int XDestroySubwindows(
    Display*		/* display */,
    Window		/* w */
);

extern int XDoesBackingStore(
    Screen*		/* screen */
);

extern Bool XDoesSaveUnders(
    Screen*		/* screen */
);

extern int XDisableAccessControl(
    Display*		/* display */
);


extern int XDisplayCells(
    Display*		/* display */,
    int			/* screen_number */
);

extern int XDisplayHeight(
    Display*		/* display */,
    int			/* screen_number */
);

extern int XDisplayHeightMM(
    Display*		/* display */,
    int			/* screen_number */
);

extern int XDisplayKeycodes(
    Display*		/* display */,
    int*		/* min_keycodes_return */,
    int*		/* max_keycodes_return */
);

extern int XDisplayPlanes(
    Display*		/* display */,
    int			/* screen_number */
);

extern int XDisplayWidth(
    Display*		/* display */,
    int			/* screen_number */
);

extern int XDisplayWidthMM(
    Display*		/* display */,
    int			/* screen_number */
);

extern int XDrawArc(
    Display*		/* display */,
    Drawable		/* d */,
    GC			/* gc */,
    int			/* x */,
    int			/* y */,
    unsigned int	/* width */,
    unsigned int	/* height */,
    int			/* angle1 */,
    int			/* angle2 */
);

extern int XDrawArcs(
    Display*		/* display */,
    Drawable		/* d */,
    GC			/* gc */,
    XArc*		/* arcs */,
    int			/* narcs */
);

extern int XDrawImageString(
    Display*		/* display */,
    Drawable		/* d */,
    GC			/* gc */,
    int			/* x */,
    int			/* y */,
    _Xconst char*	/* string */,
    int			/* length */
);

extern int XDrawImageString16(
    Display*		/* display */,
    Drawable		/* d */,
    GC			/* gc */,
    int			/* x */,
    int			/* y */,
    _Xconst XChar2b*	/* string */,
    int			/* length */
);

extern int XDrawLine(
    Display*		/* display */,
    Drawable		/* d */,
    GC			/* gc */,
    int			/* x1 */,
    int			/* y1 */,
    int			/* x2 */,
    int			/* y2 */
);

extern int XDrawLines(
    Display*		/* display */,
    Drawable		/* d */,
    GC			/* gc */,
    XPoint*		/* points */,
    int			/* npoints */,
    int			/* mode */
);

extern int XDrawPoint(
    Display*		/* display */,
    Drawable		/* d */,
    GC			/* gc */,
    int			/* x */,
    int			/* y */
);

extern int XDrawPoints(
    Display*		/* display */,
    Drawable		/* d */,
    GC			/* gc */,
    XPoint*		/* points */,
    int			/* npoints */,
    int			/* mode */
);

extern int XDrawRectangle(
    Display*		/* display */,
    Drawable		/* d */,
    GC			/* gc */,
    int			/* x */,
    int			/* y */,
    unsigned int	/* width */,
    unsigned int	/* height */
);

extern int XDrawRectangles(
    Display*		/* display */,
    Drawable		/* d */,
    GC			/* gc */,
    XRectangle*		/* rectangles */,
    int			/* nrectangles */
);

extern int XDrawSegments(
    Display*		/* display */,
    Drawable		/* d */,
    GC			/* gc */,
    XSegment*		/* segments */,
    int			/* nsegments */
);

extern int XDrawString(
    Display*		/* display */,
    Drawable		/* d */,
    GC			/* gc */,
    int			/* x */,
    int			/* y */,
    _Xconst char*	/* string */,
    int			/* length */
);

extern int XDrawString16(
    Display*		/* display */,
    Drawable		/* d */,
    GC			/* gc */,
    int			/* x */,
    int			/* y */,
    _Xconst XChar2b*	/* string */,
    int			/* length */
);

extern int XDrawText(
    Display*		/* display */,
    Drawable		/* d */,
    GC			/* gc */,
    int			/* x */,
    int			/* y */,
    XTextItem*		/* items */,
    int			/* nitems */
);

extern int XDrawText16(
    Display*		/* display */,
    Drawable		/* d */,
    GC			/* gc */,
    int			/* x */,
    int			/* y */,
    XTextItem16*	/* items */,
    int			/* nitems */
);

extern int XEnableAccessControl(
    Display*		/* display */
);

extern int XEventsQueued(
    Display*		/* display */,
    int			/* mode */
);

extern Status XFetchName(
    Display*		/* display */,
    Window		/* w */,
    char**		/* window_name_return */
);

extern int XFillArc(
    Display*		/* display */,
    Drawable		/* d */,
    GC			/* gc */,
    int			/* x */,
    int			/* y */,
    unsigned int	/* width */,
    unsigned int	/* height */,
    int			/* angle1 */,
    int			/* angle2 */
);

extern int XFillArcs(
    Display*		/* display */,
    Drawable		/* d */,
    GC			/* gc */,
    XArc*		/* arcs */,
    int			/* narcs */
);

extern int XFillPolygon(
    Display*		/* display */,
    Drawable		/* d */,
    GC			/* gc */,
    XPoint*		/* points */,
    int			/* npoints */,
    int			/* shape */,
    int			/* mode */
);

extern int XFillRectangle(
    Display*		/* display */,
    Drawable		/* d */,
    GC			/* gc */,
    int			/* x */,
    int			/* y */,
    unsigned int	/* width */,
    unsigned int	/* height */
);

extern int XFillRectangles(
    Display*		/* display */,
    Drawable		/* d */,
    GC			/* gc */,
    XRectangle*		/* rectangles */,
    int			/* nrectangles */
);

extern int XFlush(
    Display*		/* display */
);

extern int XForceScreenSaver(
    Display*		/* display */,
    int			/* mode */
);

extern int XFree(
    void*		/* data */
);

extern int XFreeColormap(
    Display*		/* display */,
    Colormap		/* colormap */
);

extern int XFreeColors(
    Display*		/* display */,
    Colormap		/* colormap */,
    unsigned long*	/* pixels */,
    int			/* npixels */,
    unsigned long	/* planes */
);

extern int XFreeCursor(
    Display*		/* display */,
    Cursor		/* cursor */
);

extern int XFreeExtensionList(
    char**		/* list */
);

extern int XFreeFont(
    Display*		/* display */,
    XFontStruct*	/* font_struct */
);

extern int XFreeFontInfo(
    char**		/* names */,
    XFontStruct*	/* free_info */,
    int			/* actual_count */
);

extern int XFreeFontNames(
    char**		/* list */
);

extern int XFreeFontPath(
    char**		/* list */
);

extern int XFreeGC(
    Display*		/* display */,
    GC			/* gc */
);

extern int XFreeModifiermap(
    XModifierKeymap*	/* modmap */
);

extern int XFreePixmap(
    Display*		/* display */,
    Pixmap		/* pixmap */
);

extern int XGeometry(
    Display*		/* display */,
    int			/* screen */,
    _Xconst char*	/* position */,
    _Xconst char*	/* default_position */,
    unsigned int	/* bwidth */,
    unsigned int	/* fwidth */,
    unsigned int	/* fheight */,
    int			/* xadder */,
    int			/* yadder */,
    int*		/* x_return */,
    int*		/* y_return */,
    int*		/* width_return */,
    int*		/* height_return */
);

extern int XGetErrorDatabaseText(
    Display*		/* display */,
    _Xconst char*	/* name */,
    _Xconst char*	/* message */,
    _Xconst char*	/* default_string */,
    char*		/* buffer_return */,
    int			/* length */
);

extern int XGetErrorText(
    Display*		/* display */,
    int			/* code */,
    char*		/* buffer_return */,
    int			/* length */
);

extern Bool XGetFontProperty(
    XFontStruct*	/* font_struct */,
    Atom		/* atom */,
    unsigned long*	/* value_return */
);

extern Status XGetGCValues(
    Display*		/* display */,
    GC			/* gc */,
    unsigned long	/* valuemask */,
    XGCValues*		/* values_return */
);

extern Status XGetGeometry(
    Display*		/* display */,
    Drawable		/* d */,
    Window*		/* root_return */,
    int*		/* x_return */,
    int*		/* y_return */,
    unsigned int*	/* width_return */,
    unsigned int*	/* height_return */,
    unsigned int*	/* border_width_return */,
    unsigned int*	/* depth_return */
);

extern Status XGetIconName(
    Display*		/* display */,
    Window		/* w */,
    char**		/* icon_name_return */
);

extern int XGetInputFocus(
    Display*		/* display */,
    Window*		/* focus_return */,
    int*		/* revert_to_return */
);

extern int XGetKeyboardControl(
    Display*		/* display */,
    XKeyboardState*	/* values_return */
);

extern int XGetPointerControl(
    Display*		/* display */,
    int*		/* accel_numerator_return */,
    int*		/* accel_denominator_return */,
    int*		/* threshold_return */
);

extern int XGetPointerMapping(
    Display*		/* display */,
    unsigned char*	/* map_return */,
    int			/* nmap */
);

extern int XGetScreenSaver(
    Display*		/* display */,
    int*		/* timeout_return */,
    int*		/* interval_return */,
    int*		/* prefer_blanking_return */,
    int*		/* allow_exposures_return */
);

extern Status XGetTransientForHint(
    Display*		/* display */,
    Window		/* w */,
    Window*		/* prop_window_return */
);

extern int XGetWindowProperty(
    Display*		/* display */,
    Window		/* w */,
    Atom		/* property */,
    long		/* long_offset */,
    long		/* long_length */,
    Bool		/* delete */,
    Atom		/* req_type */,
    Atom*		/* actual_type_return */,
    int*		/* actual_format_return */,
    unsigned long*	/* nitems_return */,
    unsigned long*	/* bytes_after_return */,
    unsigned char**	/* prop_return */
);

extern Status XGetWindowAttributes(
    Display*		/* display */,
    Window		/* w */,
    XWindowAttributes*	/* window_attributes_return */
);

extern int XGrabButton(
    Display*		/* display */,
    unsigned int	/* button */,
    unsigned int	/* modifiers */,
    Window		/* grab_window */,
    Bool		/* owner_events */,
    unsigned int	/* event_mask */,
    int			/* pointer_mode */,
    int			/* keyboard_mode */,
    Window		/* confine_to */,
    Cursor		/* cursor */
);

extern int XGrabKey(
    Display*		/* display */,
    int			/* keycode */,
    unsigned int	/* modifiers */,
    Window		/* grab_window */,
    Bool		/* owner_events */,
    int			/* pointer_mode */,
    int			/* keyboard_mode */
);

extern int XGrabKeyboard(
    Display*		/* display */,
    Window		/* grab_window */,
    Bool		/* owner_events */,
    int			/* pointer_mode */,
    int			/* keyboard_mode */,
    Time		/* time */
);

extern int XGrabPointer(
    Display*		/* display */,
    Window		/* grab_window */,
    Bool		/* owner_events */,
    unsigned int	/* event_mask */,
    int			/* pointer_mode */,
    int			/* keyboard_mode */,
    Window		/* confine_to */,
    Cursor		/* cursor */,
    Time		/* time */
);

extern int XGrabServer(
    Display*		/* display */
);

extern int XHeightMMOfScreen(
    Screen*		/* screen */
);

extern int XHeightOfScreen(
    Screen*		/* screen */
);

extern int XIfEvent(
    Display*		/* display */,
    XEvent*		/* event_return */,
    Bool (*) (
	       Display*			/* display */,
               XEvent*			/* event */,
               XPointer			/* arg */
             )		/* predicate */,
    XPointer		/* arg */
);

extern int XImageByteOrder(
    Display*		/* display */
);

extern int XInstallColormap(
    Display*		/* display */,
    Colormap		/* colormap */
);

extern KeyCode XKeysymToKeycode(
    Display*		/* display */,
    KeySym		/* keysym */
);

extern int XKillClient(
    Display*		/* display */,
    XID			/* resource */
);

extern Status XLookupColor(
    Display*		/* display */,
    Colormap		/* colormap */,
    _Xconst char*	/* color_name */,
    XColor*		/* exact_def_return */,
    XColor*		/* screen_def_return */
);

extern int XLowerWindow(
    Display*		/* display */,
    Window		/* w */
);

extern int XMapRaised(
    Display*		/* display */,
    Window		/* w */
);

extern int XMapSubwindows(
    Display*		/* display */,
    Window		/* w */
);

extern int XMapWindow(
    Display*		/* display */,
    Window		/* w */
);

extern int XMaskEvent(
    Display*		/* display */,
    long		/* event_mask */,
    XEvent*		/* event_return */
);

extern int XMaxCmapsOfScreen(
    Screen*		/* screen */
);

extern int XMinCmapsOfScreen(
    Screen*		/* screen */
);

extern int XMoveResizeWindow(
    Display*		/* display */,
    Window		/* w */,
    int			/* x */,
    int			/* y */,
    unsigned int	/* width */,
    unsigned int	/* height */
);

extern int XMoveWindow(
    Display*		/* display */,
    Window		/* w */,
    int			/* x */,
    int			/* y */
);

extern int XNextEvent(
    Display*		/* display */,
    XEvent*		/* event_return */
);

extern int XNoOp(
    Display*		/* display */
);

extern Status XParseColor(
    Display*		/* display */,
    Colormap		/* colormap */,
    _Xconst char*	/* spec */,
    XColor*		/* exact_def_return */
);

extern int XParseGeometry(
    _Xconst char*	/* parsestring */,
    int*		/* x_return */,
    int*		/* y_return */,
    unsigned int*	/* width_return */,
    unsigned int*	/* height_return */
);

extern int XPeekEvent(
    Display*		/* display */,
    XEvent*		/* event_return */
);

extern int XPeekIfEvent(
    Display*		/* display */,
    XEvent*		/* event_return */,
    Bool (*) (
	       Display*		/* display */,
               XEvent*		/* event */,
               XPointer		/* arg */
             )		/* predicate */,
    XPointer		/* arg */
);

extern int XPending(
    Display*		/* display */
);

extern int XPlanesOfScreen(
    Screen*		/* screen */
);

extern int XProtocolRevision(
    Display*		/* display */
);

extern int XProtocolVersion(
    Display*		/* display */
);


extern int XPutBackEvent(
    Display*		/* display */,
    XEvent*		/* event */
);

extern int XPutImage(
    Display*		/* display */,
    Drawable		/* d */,
    GC			/* gc */,
    XImage*		/* image */,
    int			/* src_x */,
    int			/* src_y */,
    int			/* dest_x */,
    int			/* dest_y */,
    unsigned int	/* width */,
    unsigned int	/* height */
);

extern int XQLength(
    Display*		/* display */
);

extern Status XQueryBestCursor(
    Display*		/* display */,
    Drawable		/* d */,
    unsigned int        /* width */,
    unsigned int	/* height */,
    unsigned int*	/* width_return */,
    unsigned int*	/* height_return */
);

extern Status XQueryBestSize(
    Display*		/* display */,
    int			/* class */,
    Drawable		/* which_screen */,
    unsigned int	/* width */,
    unsigned int	/* height */,
    unsigned int*	/* width_return */,
    unsigned int*	/* height_return */
);

extern Status XQueryBestStipple(
    Display*		/* display */,
    Drawable		/* which_screen */,
    unsigned int	/* width */,
    unsigned int	/* height */,
    unsigned int*	/* width_return */,
    unsigned int*	/* height_return */
);

extern Status XQueryBestTile(
    Display*		/* display */,
    Drawable		/* which_screen */,
    unsigned int	/* width */,
    unsigned int	/* height */,
    unsigned int*	/* width_return */,
    unsigned int*	/* height_return */
);

extern int XQueryColor(
    Display*		/* display */,
    Colormap		/* colormap */,
    XColor*		/* def_in_out */
);

extern int XQueryColors(
    Display*		/* display */,
    Colormap		/* colormap */,
    XColor*		/* defs_in_out */,
    int			/* ncolors */
);

extern Bool XQueryExtension(
    Display*		/* display */,
    _Xconst char*	/* name */,
    int*		/* major_opcode_return */,
    int*		/* first_event_return */,
    int*		/* first_error_return */
);

extern int XQueryKeymap(
    Display*		/* display */,
    char [32]		/* keys_return */
);

extern Bool XQueryPointer(
    Display*		/* display */,
    Window		/* w */,
    Window*		/* root_return */,
    Window*		/* child_return */,
    int*		/* root_x_return */,
    int*		/* root_y_return */,
    int*		/* win_x_return */,
    int*		/* win_y_return */,
    unsigned int*       /* mask_return */
);

extern int XQueryTextExtents(
    Display*		/* display */,
    XID			/* font_ID */,
    _Xconst char*	/* string */,
    int			/* nchars */,
    int*		/* direction_return */,
    int*		/* font_ascent_return */,
    int*		/* font_descent_return */,
    XCharStruct*	/* overall_return */
);

extern int XQueryTextExtents16(
    Display*		/* display */,
    XID			/* font_ID */,
    _Xconst XChar2b*	/* string */,
    int			/* nchars */,
    int*		/* direction_return */,
    int*		/* font_ascent_return */,
    int*		/* font_descent_return */,
    XCharStruct*	/* overall_return */
);

extern Status XQueryTree(
    Display*		/* display */,
    Window		/* w */,
    Window*		/* root_return */,
    Window*		/* parent_return */,
    Window**		/* children_return */,
    unsigned int*	/* nchildren_return */
);

extern int XRaiseWindow(
    Display*		/* display */,
    Window		/* w */
);

extern int XReadBitmapFile(
    Display*		/* display */,
    Drawable 		/* d */,
    _Xconst char*	/* filename */,
    unsigned int*	/* width_return */,
    unsigned int*	/* height_return */,
    Pixmap*		/* bitmap_return */,
    int*		/* x_hot_return */,
    int*		/* y_hot_return */
);

extern int XReadBitmapFileData(
    _Xconst char*	/* filename */,
    unsigned int*	/* width_return */,
    unsigned int*	/* height_return */,
    unsigned char**	/* data_return */,
    int*		/* x_hot_return */,
    int*		/* y_hot_return */
);

extern int XRebindKeysym(
    Display*		/* display */,
    KeySym		/* keysym */,
    KeySym*		/* list */,
    int			/* mod_count */,
    _Xconst unsigned char*	/* string */,
    int			/* bytes_string */
);

extern int XRecolorCursor(
    Display*		/* display */,
    Cursor		/* cursor */,
    XColor*		/* foreground_color */,
    XColor*		/* background_color */
);

extern int XRefreshKeyboardMapping(
    XMappingEvent*	/* event_map */
);

extern int XRemoveFromSaveSet(
    Display*		/* display */,
    Window		/* w */
);

extern int XRemoveHost(
    Display*		/* display */,
    XHostAddress*	/* host */
);

extern int XRemoveHosts(
    Display*		/* display */,
    XHostAddress*	/* hosts */,
    int			/* num_hosts */
);

extern int XReparentWindow(
    Display*		/* display */,
    Window		/* w */,
    Window		/* parent */,
    int			/* x */,
    int			/* y */
);

extern int XResetScreenSaver(
    Display*		/* display */
);

extern int XResizeWindow(
    Display*		/* display */,
    Window		/* w */,
    unsigned int	/* width */,
    unsigned int	/* height */
);

extern int XRestackWindows(
    Display*		/* display */,
    Window*		/* windows */,
    int			/* nwindows */
);

extern int XRotateBuffers(
    Display*		/* display */,
    int			/* rotate */
);

extern int XRotateWindowProperties(
    Display*		/* display */,
    Window		/* w */,
    Atom*		/* properties */,
    int			/* num_prop */,
    int			/* npositions */
);

extern int XScreenCount(
    Display*		/* display */
);

extern int XSelectInput(
    Display*		/* display */,
    Window		/* w */,
    long		/* event_mask */
);

extern Status XSendEvent(
    Display*		/* display */,
    Window		/* w */,
    Bool		/* propagate */,
    long		/* event_mask */,
    XEvent*		/* event_send */
);

extern int XSetAccessControl(
    Display*		/* display */,
    int			/* mode */
);

extern int XSetArcMode(
    Display*		/* display */,
    GC			/* gc */,
    int			/* arc_mode */
);

extern int XSetBackground(
    Display*		/* display */,
    GC			/* gc */,
    unsigned long	/* background */
);

extern int XSetClipMask(
    Display*		/* display */,
    GC			/* gc */,
    Pixmap		/* pixmap */
);

extern int XSetClipOrigin(
    Display*		/* display */,
    GC			/* gc */,
    int			/* clip_x_origin */,
    int			/* clip_y_origin */
);

extern int XSetClipRectangles(
    Display*		/* display */,
    GC			/* gc */,
    int			/* clip_x_origin */,
    int			/* clip_y_origin */,
    XRectangle*		/* rectangles */,
    int			/* n */,
    int			/* ordering */
);

extern int XSetCloseDownMode(
    Display*		/* display */,
    int			/* close_mode */
);

extern int XSetCommand(
    Display*		/* display */,
    Window		/* w */,
    char**		/* argv */,
    int			/* argc */
);

extern int XSetDashes(
    Display*		/* display */,
    GC			/* gc */,
    int			/* dash_offset */,
    _Xconst char*	/* dash_list */,
    int			/* n */
);

extern int XSetFillRule(
    Display*		/* display */,
    GC			/* gc */,
    int			/* fill_rule */
);

extern int XSetFillStyle(
    Display*		/* display */,
    GC			/* gc */,
    int			/* fill_style */
);

extern int XSetFont(
    Display*		/* display */,
    GC			/* gc */,
    Font		/* font */
);

extern int XSetFontPath(
    Display*		/* display */,
    char**		/* directories */,
    int			/* ndirs */
);

extern int XSetForeground(
    Display*		/* display */,
    GC			/* gc */,
    unsigned long	/* foreground */
);

extern int XSetFunction(
    Display*		/* display */,
    GC			/* gc */,
    int			/* function */
);

extern int XSetGraphicsExposures(
    Display*		/* display */,
    GC			/* gc */,
    Bool		/* graphics_exposures */
);

extern int XSetIconName(
    Display*		/* display */,
    Window		/* w */,
    _Xconst char*	/* icon_name */
);

extern int XSetInputFocus(
    Display*		/* display */,
    Window		/* focus */,
    int			/* revert_to */,
    Time		/* time */
);

extern int XSetLineAttributes(
    Display*		/* display */,
    GC			/* gc */,
    unsigned int	/* line_width */,
    int			/* line_style */,
    int			/* cap_style */,
    int			/* join_style */
);

extern int XSetModifierMapping(
    Display*		/* display */,
    XModifierKeymap*	/* modmap */
);

extern int XSetPlaneMask(
    Display*		/* display */,
    GC			/* gc */,
    unsigned long	/* plane_mask */
);

extern int XSetPointerMapping(
    Display*		/* display */,
    _Xconst unsigned char*	/* map */,
    int			/* nmap */
);

extern int XSetScreenSaver(
    Display*		/* display */,
    int			/* timeout */,
    int			/* interval */,
    int			/* prefer_blanking */,
    int			/* allow_exposures */
);

extern int XSetSelectionOwner(
    Display*		/* display */,
    Atom	        /* selection */,
    Window		/* owner */,
    Time		/* time */
);

extern int XSetState(
    Display*		/* display */,
    GC			/* gc */,
    unsigned long 	/* foreground */,
    unsigned long	/* background */,
    int			/* function */,
    unsigned long	/* plane_mask */
);

extern int XSetStipple(
    Display*		/* display */,
    GC			/* gc */,
    Pixmap		/* stipple */
);

extern int XSetSubwindowMode(
    Display*		/* display */,
    GC			/* gc */,
    int			/* subwindow_mode */
);

extern int XSetTSOrigin(
    Display*		/* display */,
    GC			/* gc */,
    int			/* ts_x_origin */,
    int			/* ts_y_origin */
);

extern int XSetTile(
    Display*		/* display */,
    GC			/* gc */,
    Pixmap		/* tile */
);

extern int XSetWindowBackground(
    Display*		/* display */,
    Window		/* w */,
    unsigned long	/* background_pixel */
);

extern int XSetWindowBackgroundPixmap(
    Display*		/* display */,
    Window		/* w */,
    Pixmap		/* background_pixmap */
);

extern int XSetWindowBorder(
    Display*		/* display */,
    Window		/* w */,
    unsigned long	/* border_pixel */
);

extern int XSetWindowBorderPixmap(
    Display*		/* display */,
    Window		/* w */,
    Pixmap		/* border_pixmap */
);

extern int XSetWindowBorderWidth(
    Display*		/* display */,
    Window		/* w */,
    unsigned int	/* width */
);

extern int XSetWindowColormap(
    Display*		/* display */,
    Window		/* w */,
    Colormap		/* colormap */
);

extern int XStoreBuffer(
    Display*		/* display */,
    _Xconst char*	/* bytes */,
    int			/* nbytes */,
    int			/* buffer */
);

extern int XStoreBytes(
    Display*		/* display */,
    _Xconst char*	/* bytes */,
    int			/* nbytes */
);

extern int XStoreColor(
    Display*		/* display */,
    Colormap		/* colormap */,
    XColor*		/* color */
);

extern int XStoreColors(
    Display*		/* display */,
    Colormap		/* colormap */,
    XColor*		/* color */,
    int			/* ncolors */
);

extern int XStoreName(
    Display*		/* display */,
    Window		/* w */,
    _Xconst char*	/* window_name */
);

extern int XStoreNamedColor(
    Display*		/* display */,
    Colormap		/* colormap */,
    _Xconst char*	/* color */,
    unsigned long	/* pixel */,
    int			/* flags */
);

extern int XSync(
    Display*		/* display */,
    Bool		/* discard */
);

extern int XTextExtents(
    XFontStruct*	/* font_struct */,
    _Xconst char*	/* string */,
    int			/* nchars */,
    int*		/* direction_return */,
    int*		/* font_ascent_return */,
    int*		/* font_descent_return */,
    XCharStruct*	/* overall_return */
);

extern int XTextExtents16(
    XFontStruct*	/* font_struct */,
    _Xconst XChar2b*	/* string */,
    int			/* nchars */,
    int*		/* direction_return */,
    int*		/* font_ascent_return */,
    int*		/* font_descent_return */,
    XCharStruct*	/* overall_return */
);

extern int XTextWidth(
    XFontStruct*	/* font_struct */,
    _Xconst char*	/* string */,
    int			/* count */
);

extern int XTextWidth16(
    XFontStruct*	/* font_struct */,
    _Xconst XChar2b*	/* string */,
    int			/* count */
);

extern Bool XTranslateCoordinates(
    Display*		/* display */,
    Window		/* src_w */,
    Window		/* dest_w */,
    int			/* src_x */,
    int			/* src_y */,
    int*		/* dest_x_return */,
    int*		/* dest_y_return */,
    Window*		/* child_return */
);

extern int XUndefineCursor(
    Display*		/* display */,
    Window		/* w */
);

extern int XUngrabButton(
    Display*		/* display */,
    unsigned int	/* button */,
    unsigned int	/* modifiers */,
    Window		/* grab_window */
);

extern int XUngrabKey(
    Display*		/* display */,
    int			/* keycode */,
    unsigned int	/* modifiers */,
    Window		/* grab_window */
);

extern int XUngrabKeyboard(
    Display*		/* display */,
    Time		/* time */
);

extern int XUngrabPointer(
    Display*		/* display */,
    Time		/* time */
);

extern int XUngrabServer(
    Display*		/* display */
);

extern int XUninstallColormap(
    Display*		/* display */,
    Colormap		/* colormap */
);

extern int XUnloadFont(
    Display*		/* display */,
    Font		/* font */
);

extern int XUnmapSubwindows(
    Display*		/* display */,
    Window		/* w */
);

extern int XUnmapWindow(
    Display*		/* display */,
    Window		/* w */
);

extern int XVendorRelease(
    Display*		/* display */
);

extern int XWarpPointer(
    Display*		/* display */,
    Window		/* src_w */,
    Window		/* dest_w */,
    int			/* src_x */,
    int			/* src_y */,
    unsigned int	/* src_width */,
    unsigned int	/* src_height */,
    int			/* dest_x */,
    int			/* dest_y */
);

extern int XWidthMMOfScreen(
    Screen*		/* screen */
);

extern int XWidthOfScreen(
    Screen*		/* screen */
);

extern int XWindowEvent(
    Display*		/* display */,
    Window		/* w */,
    long		/* event_mask */,
    XEvent*		/* event_return */
);

extern int XWriteBitmapFile(
    Display*		/* display */,
    _Xconst char*	/* filename */,
    Pixmap		/* bitmap */,
    unsigned int	/* width */,
    unsigned int	/* height */,
    int			/* x_hot */,
    int			/* y_hot */
);

extern Bool XSupportsLocale (void);

extern char *XSetLocaleModifiers(
    const char*		/* modifier_list */
);

extern XOM XOpenOM(
    Display*			/* display */,
    struct _XrmHashBucketRec*	/* rdb */,
    _Xconst char*		/* res_name */,
    _Xconst char*		/* res_class */
);

extern Status XCloseOM(
    XOM			/* om */
);

extern char *XSetOMValues(
    XOM			/* om */,
    ...
) _X_SENTINEL(0);

extern char *XGetOMValues(
    XOM			/* om */,
    ...
) _X_SENTINEL(0);

extern Display *XDisplayOfOM(
    XOM			/* om */
);

extern char *XLocaleOfOM(
    XOM			/* om */
);

extern XOC XCreateOC(
    XOM			/* om */,
    ...
) _X_SENTINEL(0);

extern void XDestroyOC(
    XOC			/* oc */
);

extern XOM XOMOfOC(
    XOC			/* oc */
);

extern char *XSetOCValues(
    XOC			/* oc */,
    ...
) _X_SENTINEL(0);

extern char *XGetOCValues(
    XOC			/* oc */,
    ...
) _X_SENTINEL(0);

extern XFontSet XCreateFontSet(
    Display*		/* display */,
    _Xconst char*	/* base_font_name_list */,
    char***		/* missing_charset_list */,
    int*		/* missing_charset_count */,
    char**		/* def_string */
);

extern void XFreeFontSet(
    Display*		/* display */,
    XFontSet		/* font_set */
);

extern int XFontsOfFontSet(
    XFontSet		/* font_set */,
    XFontStruct***	/* font_struct_list */,
    char***		/* font_name_list */
);

extern char *XBaseFontNameListOfFontSet(
    XFontSet		/* font_set */
);

extern char *XLocaleOfFontSet(
    XFontSet		/* font_set */
);

extern Bool XContextDependentDrawing(
    XFontSet		/* font_set */
);

extern Bool XDirectionalDependentDrawing(
    XFontSet		/* font_set */
);

extern Bool XContextualDrawing(
    XFontSet		/* font_set */
);

extern XFontSetExtents *XExtentsOfFontSet(
    XFontSet		/* font_set */
);

extern int XmbTextEscapement(
    XFontSet		/* font_set */,
    _Xconst char*	/* text */,
    int			/* bytes_text */
);

extern int XwcTextEscapement(
    XFontSet		/* font_set */,
    _Xconst wchar_t*	/* text */,
    int			/* num_wchars */
);

extern int Xutf8TextEscapement(
    XFontSet		/* font_set */,
    _Xconst char*	/* text */,
    int			/* bytes_text */
);

extern int XmbTextExtents(
    XFontSet		/* font_set */,
    _Xconst char*	/* text */,
    int			/* bytes_text */,
    XRectangle*		/* overall_ink_return */,
    XRectangle*		/* overall_logical_return */
);

extern int XwcTextExtents(
    XFontSet		/* font_set */,
    _Xconst wchar_t*	/* text */,
    int			/* num_wchars */,
    XRectangle*		/* overall_ink_return */,
    XRectangle*		/* overall_logical_return */
);

extern int Xutf8TextExtents(
    XFontSet		/* font_set */,
    _Xconst char*	/* text */,
    int			/* bytes_text */,
    XRectangle*		/* overall_ink_return */,
    XRectangle*		/* overall_logical_return */
);

extern Status XmbTextPerCharExtents(
    XFontSet		/* font_set */,
    _Xconst char*	/* text */,
    int			/* bytes_text */,
    XRectangle*		/* ink_extents_buffer */,
    XRectangle*		/* logical_extents_buffer */,
    int			/* buffer_size */,
    int*		/* num_chars */,
    XRectangle*		/* overall_ink_return */,
    XRectangle*		/* overall_logical_return */
);

extern Status XwcTextPerCharExtents(
    XFontSet		/* font_set */,
    _Xconst wchar_t*	/* text */,
    int			/* num_wchars */,
    XRectangle*		/* ink_extents_buffer */,
    XRectangle*		/* logical_extents_buffer */,
    int			/* buffer_size */,
    int*		/* num_chars */,
    XRectangle*		/* overall_ink_return */,
    XRectangle*		/* overall_logical_return */
);

extern Status Xutf8TextPerCharExtents(
    XFontSet		/* font_set */,
    _Xconst char*	/* text */,
    int			/* bytes_text */,
    XRectangle*		/* ink_extents_buffer */,
    XRectangle*		/* logical_extents_buffer */,
    int			/* buffer_size */,
    int*		/* num_chars */,
    XRectangle*		/* overall_ink_return */,
    XRectangle*		/* overall_logical_return */
);

extern void XmbDrawText(
    Display*		/* display */,
    Drawable		/* d */,
    GC			/* gc */,
    int			/* x */,
    int			/* y */,
    XmbTextItem*	/* text_items */,
    int			/* nitems */
);

extern void XwcDrawText(
    Display*		/* display */,
    Drawable		/* d */,
    GC			/* gc */,
    int			/* x */,
    int			/* y */,
    XwcTextItem*	/* text_items */,
    int			/* nitems */
);

extern void Xutf8DrawText(
    Display*		/* display */,
    Drawable		/* d */,
    GC			/* gc */,
    int			/* x */,
    int			/* y */,
    XmbTextItem*	/* text_items */,
    int			/* nitems */
);

extern void XmbDrawString(
    Display*		/* display */,
    Drawable		/* d */,
    XFontSet		/* font_set */,
    GC			/* gc */,
    int			/* x */,
    int			/* y */,
    _Xconst char*	/* text */,
    int			/* bytes_text */
);

extern void XwcDrawString(
    Display*		/* display */,
    Drawable		/* d */,
    XFontSet		/* font_set */,
    GC			/* gc */,
    int			/* x */,
    int			/* y */,
    _Xconst wchar_t*	/* text */,
    int			/* num_wchars */
);

extern void Xutf8DrawString(
    Display*		/* display */,
    Drawable		/* d */,
    XFontSet		/* font_set */,
    GC			/* gc */,
    int			/* x */,
    int			/* y */,
    _Xconst char*	/* text */,
    int			/* bytes_text */
);

extern void XmbDrawImageString(
    Display*		/* display */,
    Drawable		/* d */,
    XFontSet		/* font_set */,
    GC			/* gc */,
    int			/* x */,
    int			/* y */,
    _Xconst char*	/* text */,
    int			/* bytes_text */
);

extern void XwcDrawImageString(
    Display*		/* display */,
    Drawable		/* d */,
    XFontSet		/* font_set */,
    GC			/* gc */,
    int			/* x */,
    int			/* y */,
    _Xconst wchar_t*	/* text */,
    int			/* num_wchars */
);

extern void Xutf8DrawImageString(
    Display*		/* display */,
    Drawable		/* d */,
    XFontSet		/* font_set */,
    GC			/* gc */,
    int			/* x */,
    int			/* y */,
    _Xconst char*	/* text */,
    int			/* bytes_text */
);

extern XIM XOpenIM(
    Display*			/* dpy */,
    struct _XrmHashBucketRec*	/* rdb */,
    char*			/* res_name */,
    char*			/* res_class */
);

extern Status XCloseIM(
    XIM /* im */
);

extern char *XGetIMValues(
    XIM /* im */, ...
) _X_SENTINEL(0);

extern char *XSetIMValues(
    XIM /* im */, ...
) _X_SENTINEL(0);

extern Display *XDisplayOfIM(
    XIM /* im */
);

extern char *XLocaleOfIM(
    XIM /* im*/
);

extern XIC XCreateIC(
    XIM /* im */, ...
) _X_SENTINEL(0);

extern void XDestroyIC(
    XIC /* ic */
);

extern void XSetICFocus(
    XIC /* ic */
);

extern void XUnsetICFocus(
    XIC /* ic */
);

extern wchar_t *XwcResetIC(
    XIC /* ic */
);

extern char *XmbResetIC(
    XIC /* ic */
);

extern char *Xutf8ResetIC(
    XIC /* ic */
);

extern char *XSetICValues(
    XIC /* ic */, ...
) _X_SENTINEL(0);

extern char *XGetICValues(
    XIC /* ic */, ...
) _X_SENTINEL(0);

extern XIM XIMOfIC(
    XIC /* ic */
);

extern Bool XFilterEvent(
    XEvent*	/* event */,
    Window	/* window */
);

extern int XmbLookupString(
    XIC			/* ic */,
    XKeyPressedEvent*	/* event */,
    char*		/* buffer_return */,
    int			/* bytes_buffer */,
    KeySym*		/* keysym_return */,
    Status*		/* status_return */
);

extern int XwcLookupString(
    XIC			/* ic */,
    XKeyPressedEvent*	/* event */,
    wchar_t*		/* buffer_return */,
    int			/* wchars_buffer */,
    KeySym*		/* keysym_return */,
    Status*		/* status_return */
);

extern int Xutf8LookupString(
    XIC			/* ic */,
    XKeyPressedEvent*	/* event */,
    char*		/* buffer_return */,
    int			/* bytes_buffer */,
    KeySym*		/* keysym_return */,
    Status*		/* status_return */
);

extern XVaNestedList XVaCreateNestedList(
    int /*unused*/, ...
) _X_SENTINEL(0);

/* internal connections for IMs */

extern Bool XRegisterIMInstantiateCallback(
    Display*			/* dpy */,
    struct _XrmHashBucketRec*	/* rdb */,
    char*			/* res_name */,
    char*			/* res_class */,
    XIDProc			/* callback */,
    XPointer			/* client_data */
);

extern Bool XUnregisterIMInstantiateCallback(
    Display*			/* dpy */,
    struct _XrmHashBucketRec*	/* rdb */,
    char*			/* res_name */,
    char*			/* res_class */,
    XIDProc			/* callback */,
    XPointer			/* client_data */
);

typedef void (*XConnectionWatchProc)(
    Display*			/* dpy */,
    XPointer			/* client_data */,
    int				/* fd */,
    Bool			/* opening */,	 /* open or close flag */
    XPointer*			/* watch_data */ /* open sets, close uses */
);


extern Status XInternalConnectionNumbers(
    Display*			/* dpy */,
    int**			/* fd_return */,
    int*			/* count_return */
);

extern void XProcessInternalConnection(
    Display*			/* dpy */,
    int				/* fd */
);

extern Status XAddConnectionWatch(
    Display*			/* dpy */,
    XConnectionWatchProc	/* callback */,
    XPointer			/* client_data */
);

extern void XRemoveConnectionWatch(
    Display*			/* dpy */,
    XConnectionWatchProc	/* callback */,
    XPointer			/* client_data */
);

extern void XSetAuthorization(
    char *			/* name */,
    int				/* namelen */,
    char *			/* data */,
    int				/* datalen */
);

extern int _Xmbtowc(
    wchar_t *			/* wstr */,
    char *			/* str */,
    int				/* len */
);

extern int _Xwctomb(
    char *			/* str */,
    wchar_t			/* wc */
);

extern Bool XGetEventData(
    Display*			/* dpy */,
    XGenericEventCookie*	/* cookie*/
);

extern void XFreeEventData(
    Display*			/* dpy */,
    XGenericEventCookie*	/* cookie*/
);

#ifdef __clang__
#pragma clang diagnostic pop
#endif

_XFUNCPROTOEND

#endif /* _X11_XLIB_H_ */
/* #include "external/X11/keysym.h" */
/***********************************************************

Copyright 1987, 1998  The Open Group

Permission to use, copy, modify, distribute, and sell this software and its
documentation for any purpose is hereby granted without fee, provided that
the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation.

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
OPEN GROUP BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of The Open Group shall not be
used in advertising or otherwise to promote the sale, use or other dealings
in this Software without prior written authorization from The Open Group.


Copyright 1987 by Digital Equipment Corporation, Maynard, Massachusetts.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of Digital not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

DIGITAL DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
DIGITAL BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

******************************************************************/

/* default keysyms */
#define XK_MISCELLANY
#define XK_XKB_KEYS
#define XK_LATIN1
#define XK_LATIN2
#define XK_LATIN3
#define XK_LATIN4
#define XK_LATIN8
#define XK_LATIN9
#define XK_CAUCASUS
#define XK_GREEK
#define XK_KATAKANA
#define XK_ARABIC
#define XK_CYRILLIC
#define XK_HEBREW
#define XK_THAI
#define XK_KOREAN
#define XK_ARMENIAN
#define XK_GEORGIAN
#define XK_VIETNAMESE
#define XK_CURRENCY
#define XK_MATHEMATICAL
#define XK_BRAILLE
#define XK_SINHALA

/* #include "external/X11/keysymdef.h" */
/***********************************************************
Copyright 1987, 1994, 1998  The Open Group

Permission to use, copy, modify, distribute, and sell this software and its
documentation for any purpose is hereby granted without fee, provided that
the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation.

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE OPEN GROUP BE LIABLE FOR ANY CLAIM, DAMAGES OR
OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of The Open Group shall
not be used in advertising or otherwise to promote the sale, use or
other dealings in this Software without prior written authorization
from The Open Group.


Copyright 1987 by Digital Equipment Corporation, Maynard, Massachusetts

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of Digital not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

DIGITAL DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
DIGITAL BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

******************************************************************/

/*
 * The "X11 Window System Protocol" standard defines in Appendix A the
 * keysym codes. These 29-bit integer values identify characters or
 * functions associated with each key (e.g., via the visible
 * engraving) of a keyboard layout. This file assigns mnemonic macro
 * names for these keysyms.
 *
 * This file is also compiled (by src/util/makekeys.c in libX11) into
 * hash tables that can be accessed with X11 library functions such as
 * XStringToKeysym() and XKeysymToString().
 *
 * Where a keysym corresponds one-to-one to an ISO 10646 / Unicode
 * character, this is noted in a comment that provides both the U+xxxx
 * Unicode position, as well as the official Unicode name of the
 * character.
 *
 * Some keysyms map to a character already mapped by another keysym,
 * with compatible but more precise semantics, such as the keypad-
 * related keysyms. In this case, none of the keysym are deprecated.
 * The most generic keysym is annotated as previously and more specific
 * keysyms have the same annotation between angle brackets:
 *
 *     #define XK_space                 0x0020  // U+0020 SPACE
 *     #define XK_KP_Space              0xff80  //<U+0020 SPACE>
 *
 * Where the correspondence is either not one-to-one or semantically
 * unclear, the Unicode position and name are enclosed in
 * parentheses. Such legacy keysyms should be considered deprecated
 * and are not recommended for use in future keyboard mappings.
 *
 * For any future extension of the keysyms with characters already
 * found in ISO 10646 / Unicode, the following algorithm shall be
 * used. The new keysym code position will simply be the character's
 * Unicode number plus 0x01000000. The keysym values in the range
 * 0x01000100 to 0x0110ffff are reserved to represent Unicode
 * characters in the range U+0100 to U+10FFFF.
 *
 * While most newer Unicode-based X11 clients do already accept
 * Unicode-mapped keysyms in the range 0x01000100 to 0x0110ffff, it
 * will remain necessary for clients -- in the interest of
 * compatibility with existing servers -- to also understand the
 * existing legacy keysym values in the range 0x0100 to 0x20ff.
 *
 * Where several mnemonic names are defined for the same keysym in this
 * file, the first one listed is considered the "canonical" name. This
 * is the name that should be used when retrieving a keysym name from
 * its code. The next names are considered "aliases" to the canonical
 * name.
 *
 * Aliases are made explicit by writing in their comment "alias for",
 * followed by the corresponding canonical name. Example:
 *
 *     #define XK_dead_tilde            0xfe53
 *     #define XK_dead_perispomeni      0xfe53 // alias for dead_tilde
 *
 * The rules to consider a keysym mnemonic name deprecated are:
 *
 *   1. A legacy keysym with its Unicode mapping in parentheses is
 *      deprecated (see above).
 *
 *   2. A keysym name is *explicitly* deprecated by starting its comment
 *      with "deprecated". Examples:
 *
 *        #define XK_L1           0xffc8  // deprecated alias for F11
 *        #define XK_quoteleft    0x0060  // deprecated
 *
 *   3. A keysym name is *explicitly* *not* deprecated by starting its
 *      comment with "non-deprecated alias". Examples:
 *
 *       #define XK_dead_tilde       0xfe53
 *       #define XK_dead_perispomeni 0xfe53 // non-deprecated alias for dead_tilde
 *
 *   4. If none of the previous rules apply, an alias is *implicitly*
 *      deprecated if there is at least one previous name for the
 *      corresponding keysym that is *not* explicitly deprecated.
 * 
 *      Examples:
 * 
 *        // SingleCandidate is the canonical name
 *        #define XK_SingleCandidate        0xff3c
 *        // Hangul_SingleCandidate is deprecated because it is an alias
 *        // and it does not start with "non-deprecated alias"
 *        #define XK_Hangul_SingleCandidate 0xff3c // Single candidate
 * 
 *        // guillemotleft is the canonical name, but it is deprecated
 *        #define XK_guillemotleft  0x00ab // deprecated alias for guillemetleft (misspelling) 
 *        // guillemetleft is not deprecated, because the keysym has no endorsed name before it.
 *        #define XK_guillemetleft  0x00ab // U+00AB LEFT-POINTING DOUBLE ANGLE QUOTATION MARK 
 *        // The following hypothetical name is deprecated because guillemetleft come before.
 *        #define XK_guillemetleft2 0x00ab
 *
 * Mnemonic names for keysyms are defined in this file with lines
 * that match one of these Perl regular expressions:
 *
 *    /^\#define XK_([a-zA-Z_0-9]+)\s+0x([0-9a-f]+)\s*\/\* U\+([0-9A-F]{4,6}) (.*) \*\/\s*$/
 *    /^\#define XK_([a-zA-Z_0-9]+)\s+0x([0-9a-f]+)\s*\/\*<U\+([0-9A-F]{4,6}) (.*)>\*\/\s*$/
 *    /^\#define XK_([a-zA-Z_0-9]+)\s+0x([0-9a-f]+)\s*\/\*\(U\+([0-9A-F]{4,6}) (.*)\)\*\/\s*$/
 *    /^\#define XK_([a-zA-Z_0-9]+)\s+0x([0-9a-f]+)\s*(\/\*\s*(.*)\s*\*\/)?\s*$/
 *
 * Before adding new keysyms, please do consider the following: In
 * addition to the keysym names defined in this file, the
 * XStringToKeysym() and XKeysymToString() functions will also handle
 * any keysym string of the form "U0020" to "U007E" and "U00A0" to
 * "U10FFFF" for all possible Unicode characters. In other words,
 * every possible Unicode character has already a keysym string
 * defined algorithmically, even if it is not listed here. Therefore,
 * defining an additional keysym macro is only necessary where a
 * non-hexadecimal mnemonic name is needed, or where the new keysym
 * does not represent any existing Unicode character.
 *
 * When adding new keysyms to this file, do not forget to also update the
 * following as needed:
 *
 *   - the mappings in src/KeyBind.c in the libX11 repo
 *     https://gitlab.freedesktop.org/xorg/lib/libx11
 *
 *   - the protocol specification in specs/keysyms.xml in this repo
 *     https://gitlab.freedesktop.org/xorg/proto/xorgproto
 *
 * Before removing or changing the order of the keysyms, please consider
 * the following: it is very difficult to know what keysyms are used and
 * how.
 *
 *   - A sandboxed application may have incompatibilities with the host
 *     system. For example, if new keysym name is introduced and is made
 *     the canonical name, then an application with an older keysym parser
 *     will not be able to parse the new name.
 *   - Customization of keyboard layout and Compose files are two popular
 *     use cases. Checking the standard keyboard layout database xkeyboard-config
 *     https://gitlab.freedesktop.org/xkeyboard-config/xkeyboard-config
 *     and the standard Compose files in libx11
 *     https://gitlab.freedesktop.org/xorg/lib/libx11 is a mandatory
 *     step, but may *not* be enough for a proper impact assessment for
 *     e.g. keysyms removals.
 *
 * Therefore, it is advised to proceed to no removal and to make a new
 * name canonical only 10 years after its introduction. This means that
 * some keysyms may have their first listed name deprecated during the
 * period of transition. Once this period is over, the deprecated name
 * should be moved after the new canonical name.
 */

#define XK_VoidSymbol                  0xffffff  /* Void symbol */

#ifdef XK_MISCELLANY
/*
 * TTY function keys, cleverly chosen to map to ASCII, for convenience of
 * programming, but could have been arbitrary (at the cost of lookup
 * tables in client code).
 */

#define XK_BackSpace                     0xff08  /* U+0008 BACKSPACE */
#define XK_Tab                           0xff09  /* U+0009 CHARACTER TABULATION */
#define XK_Linefeed                      0xff0a  /* U+000A LINE FEED */
#define XK_Clear                         0xff0b  /* U+000B LINE TABULATION */
#define XK_Return                        0xff0d  /* U+000D CARRIAGE RETURN */
#define XK_Pause                         0xff13  /* Pause, hold */
#define XK_Scroll_Lock                   0xff14
#define XK_Sys_Req                       0xff15
#define XK_Escape                        0xff1b  /* U+001B ESCAPE */
#define XK_Delete                        0xffff  /* U+007F DELETE */



/* International & multi-key character composition */

#define XK_Multi_key                     0xff20  /* Multi-key character compose */
#define XK_Codeinput                     0xff37
#define XK_SingleCandidate               0xff3c
#define XK_MultipleCandidate             0xff3d
#define XK_PreviousCandidate             0xff3e

/* Japanese keyboard support */

#define XK_Kanji                         0xff21  /* Kanji, Kanji convert */
#define XK_Muhenkan                      0xff22  /* Cancel Conversion */
#define XK_Henkan_Mode                   0xff23  /* Start/Stop Conversion */
#define XK_Henkan                        0xff23  /* non-deprecated alias for Henkan_Mode */
#define XK_Romaji                        0xff24  /* to Romaji */
#define XK_Hiragana                      0xff25  /* to Hiragana */
#define XK_Katakana                      0xff26  /* to Katakana */
#define XK_Hiragana_Katakana             0xff27  /* Hiragana/Katakana toggle */
#define XK_Zenkaku                       0xff28  /* to Zenkaku */
#define XK_Hankaku                       0xff29  /* to Hankaku */
#define XK_Zenkaku_Hankaku               0xff2a  /* Zenkaku/Hankaku toggle */
#define XK_Touroku                       0xff2b  /* Add to Dictionary */
#define XK_Massyo                        0xff2c  /* Delete from Dictionary */
#define XK_Kana_Lock                     0xff2d  /* Kana Lock */
#define XK_Kana_Shift                    0xff2e  /* Kana Shift */
#define XK_Eisu_Shift                    0xff2f  /* Alphanumeric Shift */
#define XK_Eisu_toggle                   0xff30  /* Alphanumeric toggle */
#define XK_Kanji_Bangou                  0xff37  /* Codeinput */
#define XK_Zen_Koho                      0xff3d  /* Multiple/All Candidate(s) */
#define XK_Mae_Koho                      0xff3e  /* Previous Candidate */

/* 0xff31 thru 0xff3f are under XK_KOREAN */

/* Cursor control & motion */

#define XK_Home                          0xff50
#define XK_Left                          0xff51  /* Move left, left arrow */
#define XK_Up                            0xff52  /* Move up, up arrow */
#define XK_Right                         0xff53  /* Move right, right arrow */
#define XK_Down                          0xff54  /* Move down, down arrow */
#define XK_Prior                         0xff55  /* Prior, previous */
#define XK_Page_Up                       0xff55  /* deprecated alias for Prior */
#define XK_Next                          0xff56  /* Next */
#define XK_Page_Down                     0xff56  /* deprecated alias for Next */
#define XK_End                           0xff57  /* EOL */
#define XK_Begin                         0xff58  /* BOL */


/* Misc functions */

#define XK_Select                        0xff60  /* Select, mark */
#define XK_Print                         0xff61
#define XK_Execute                       0xff62  /* Execute, run, do */
#define XK_Insert                        0xff63  /* Insert, insert here */
#define XK_Undo                          0xff65
#define XK_Redo                          0xff66  /* Redo, again */
#define XK_Menu                          0xff67
#define XK_Find                          0xff68  /* Find, search */
#define XK_Cancel                        0xff69  /* Cancel, stop, abort, exit */
#define XK_Help                          0xff6a  /* Help */
#define XK_Break                         0xff6b
#define XK_Mode_switch                   0xff7e  /* Character set switch */
#define XK_script_switch                 0xff7e  /* non-deprecated alias for Mode_switch */
#define XK_Num_Lock                      0xff7f

/* Keypad functions, keypad numbers cleverly chosen to map to ASCII */

#define XK_KP_Space                      0xff80  /*<U+0020 SPACE>*/
#define XK_KP_Tab                        0xff89  /*<U+0009 CHARACTER TABULATION>*/
#define XK_KP_Enter                      0xff8d  /*<U+000D CARRIAGE RETURN>*/
#define XK_KP_F1                         0xff91  /* PF1, KP_A, ... */
#define XK_KP_F2                         0xff92
#define XK_KP_F3                         0xff93
#define XK_KP_F4                         0xff94
#define XK_KP_Home                       0xff95
#define XK_KP_Left                       0xff96
#define XK_KP_Up                         0xff97
#define XK_KP_Right                      0xff98
#define XK_KP_Down                       0xff99
#define XK_KP_Prior                      0xff9a
#define XK_KP_Page_Up                    0xff9a  /* deprecated alias for KP_Prior */
#define XK_KP_Next                       0xff9b
#define XK_KP_Page_Down                  0xff9b  /* deprecated alias for KP_Next */
#define XK_KP_End                        0xff9c
#define XK_KP_Begin                      0xff9d
#define XK_KP_Insert                     0xff9e
#define XK_KP_Delete                     0xff9f
#define XK_KP_Equal                      0xffbd  /*<U+003D EQUALS SIGN>*/
#define XK_KP_Multiply                   0xffaa  /*<U+002A ASTERISK>*/
#define XK_KP_Add                        0xffab  /*<U+002B PLUS SIGN>*/
#define XK_KP_Separator                  0xffac  /*<U+002C COMMA>*/
#define XK_KP_Subtract                   0xffad  /*<U+002D HYPHEN-MINUS>*/
#define XK_KP_Decimal                    0xffae  /*<U+002E FULL STOP>*/
#define XK_KP_Divide                     0xffaf  /*<U+002F SOLIDUS>*/

#define XK_KP_0                          0xffb0  /*<U+0030 DIGIT ZERO>*/
#define XK_KP_1                          0xffb1  /*<U+0031 DIGIT ONE>*/
#define XK_KP_2                          0xffb2  /*<U+0032 DIGIT TWO>*/
#define XK_KP_3                          0xffb3  /*<U+0033 DIGIT THREE>*/
#define XK_KP_4                          0xffb4  /*<U+0034 DIGIT FOUR>*/
#define XK_KP_5                          0xffb5  /*<U+0035 DIGIT FIVE>*/
#define XK_KP_6                          0xffb6  /*<U+0036 DIGIT SIX>*/
#define XK_KP_7                          0xffb7  /*<U+0037 DIGIT SEVEN>*/
#define XK_KP_8                          0xffb8  /*<U+0038 DIGIT EIGHT>*/
#define XK_KP_9                          0xffb9  /*<U+0039 DIGIT NINE>*/



/*
 * Auxiliary functions; note the duplicate definitions for left and right
 * function keys;  Sun keyboards and a few other manufacturers have such
 * function key groups on the left and/or right sides of the keyboard.
 * We've not found a keyboard with more than 35 function keys total.
 */

#define XK_F1                            0xffbe
#define XK_F2                            0xffbf
#define XK_F3                            0xffc0
#define XK_F4                            0xffc1
#define XK_F5                            0xffc2
#define XK_F6                            0xffc3
#define XK_F7                            0xffc4
#define XK_F8                            0xffc5
#define XK_F9                            0xffc6
#define XK_F10                           0xffc7
#define XK_F11                           0xffc8
#define XK_L1                            0xffc8  /* deprecated alias for F11 */
#define XK_F12                           0xffc9
#define XK_L2                            0xffc9  /* deprecated alias for F12 */
#define XK_F13                           0xffca
#define XK_L3                            0xffca  /* deprecated alias for F13 */
#define XK_F14                           0xffcb
#define XK_L4                            0xffcb  /* deprecated alias for F14 */
#define XK_F15                           0xffcc
#define XK_L5                            0xffcc  /* deprecated alias for F15 */
#define XK_F16                           0xffcd
#define XK_L6                            0xffcd  /* deprecated alias for F16 */
#define XK_F17                           0xffce
#define XK_L7                            0xffce  /* deprecated alias for F17 */
#define XK_F18                           0xffcf
#define XK_L8                            0xffcf  /* deprecated alias for F18 */
#define XK_F19                           0xffd0
#define XK_L9                            0xffd0  /* deprecated alias for F19 */
#define XK_F20                           0xffd1
#define XK_L10                           0xffd1  /* deprecated alias for F20 */
#define XK_F21                           0xffd2
#define XK_R1                            0xffd2  /* deprecated alias for F21 */
#define XK_F22                           0xffd3
#define XK_R2                            0xffd3  /* deprecated alias for F22 */
#define XK_F23                           0xffd4
#define XK_R3                            0xffd4  /* deprecated alias for F23 */
#define XK_F24                           0xffd5
#define XK_R4                            0xffd5  /* deprecated alias for F24 */
#define XK_F25                           0xffd6
#define XK_R5                            0xffd6  /* deprecated alias for F25 */
#define XK_F26                           0xffd7
#define XK_R6                            0xffd7  /* deprecated alias for F26 */
#define XK_F27                           0xffd8
#define XK_R7                            0xffd8  /* deprecated alias for F27 */
#define XK_F28                           0xffd9
#define XK_R8                            0xffd9  /* deprecated alias for F28 */
#define XK_F29                           0xffda
#define XK_R9                            0xffda  /* deprecated alias for F29 */
#define XK_F30                           0xffdb
#define XK_R10                           0xffdb  /* deprecated alias for F30 */
#define XK_F31                           0xffdc
#define XK_R11                           0xffdc  /* deprecated alias for F31 */
#define XK_F32                           0xffdd
#define XK_R12                           0xffdd  /* deprecated alias for F32 */
#define XK_F33                           0xffde
#define XK_R13                           0xffde  /* deprecated alias for F33 */
#define XK_F34                           0xffdf
#define XK_R14                           0xffdf  /* deprecated alias for F34 */
#define XK_F35                           0xffe0
#define XK_R15                           0xffe0  /* deprecated alias for F35 */

/* Modifiers */

#define XK_Shift_L                       0xffe1  /* Left shift */
#define XK_Shift_R                       0xffe2  /* Right shift */
#define XK_Control_L                     0xffe3  /* Left control */
#define XK_Control_R                     0xffe4  /* Right control */
#define XK_Caps_Lock                     0xffe5  /* Caps lock */
#define XK_Shift_Lock                    0xffe6  /* Shift lock */

#define XK_Meta_L                        0xffe7  /* Left meta */
#define XK_Meta_R                        0xffe8  /* Right meta */
#define XK_Alt_L                         0xffe9  /* Left alt */
#define XK_Alt_R                         0xffea  /* Right alt */
#define XK_Super_L                       0xffeb  /* Left super */
#define XK_Super_R                       0xffec  /* Right super */
#define XK_Hyper_L                       0xffed  /* Left hyper */
#define XK_Hyper_R                       0xffee  /* Right hyper */
#endif /* XK_MISCELLANY */

/*
 * Keyboard (XKB) Extension function and modifier keys
 * (from Appendix C of "The X Keyboard Extension: Protocol Specification")
 * Byte 3 = 0xfe
 */

#ifdef XK_XKB_KEYS
#define XK_ISO_Lock                      0xfe01
#define XK_ISO_Level2_Latch              0xfe02
#define XK_ISO_Level3_Shift              0xfe03
#define XK_ISO_Level3_Latch              0xfe04
#define XK_ISO_Level3_Lock               0xfe05
#define XK_ISO_Level5_Shift              0xfe11
#define XK_ISO_Level5_Latch              0xfe12
#define XK_ISO_Level5_Lock               0xfe13
#define XK_ISO_Group_Shift               0xff7e  /* non-deprecated alias for Mode_switch */
#define XK_ISO_Group_Latch               0xfe06
#define XK_ISO_Group_Lock                0xfe07
#define XK_ISO_Next_Group                0xfe08
#define XK_ISO_Next_Group_Lock           0xfe09
#define XK_ISO_Prev_Group                0xfe0a
#define XK_ISO_Prev_Group_Lock           0xfe0b
#define XK_ISO_First_Group               0xfe0c
#define XK_ISO_First_Group_Lock          0xfe0d
#define XK_ISO_Last_Group                0xfe0e
#define XK_ISO_Last_Group_Lock           0xfe0f

#define XK_ISO_Left_Tab                  0xfe20
#define XK_ISO_Move_Line_Up              0xfe21
#define XK_ISO_Move_Line_Down            0xfe22
#define XK_ISO_Partial_Line_Up           0xfe23
#define XK_ISO_Partial_Line_Down         0xfe24
#define XK_ISO_Partial_Space_Left        0xfe25
#define XK_ISO_Partial_Space_Right       0xfe26
#define XK_ISO_Set_Margin_Left           0xfe27
#define XK_ISO_Set_Margin_Right          0xfe28
#define XK_ISO_Release_Margin_Left       0xfe29
#define XK_ISO_Release_Margin_Right      0xfe2a
#define XK_ISO_Release_Both_Margins      0xfe2b
#define XK_ISO_Fast_Cursor_Left          0xfe2c
#define XK_ISO_Fast_Cursor_Right         0xfe2d
#define XK_ISO_Fast_Cursor_Up            0xfe2e
#define XK_ISO_Fast_Cursor_Down          0xfe2f
#define XK_ISO_Continuous_Underline      0xfe30
#define XK_ISO_Discontinuous_Underline   0xfe31
#define XK_ISO_Emphasize                 0xfe32
#define XK_ISO_Center_Object             0xfe33
#define XK_ISO_Enter                     0xfe34

#define XK_dead_grave                    0xfe50
#define XK_dead_acute                    0xfe51
#define XK_dead_circumflex               0xfe52
#define XK_dead_tilde                    0xfe53
#define XK_dead_perispomeni              0xfe53  /* non-deprecated alias for dead_tilde */
#define XK_dead_macron                   0xfe54
#define XK_dead_breve                    0xfe55
#define XK_dead_abovedot                 0xfe56
#define XK_dead_diaeresis                0xfe57
#define XK_dead_abovering                0xfe58
#define XK_dead_doubleacute              0xfe59
#define XK_dead_caron                    0xfe5a
#define XK_dead_cedilla                  0xfe5b
#define XK_dead_ogonek                   0xfe5c
#define XK_dead_iota                     0xfe5d
#define XK_dead_voiced_sound             0xfe5e
#define XK_dead_semivoiced_sound         0xfe5f
#define XK_dead_belowdot                 0xfe60
#define XK_dead_hook                     0xfe61
#define XK_dead_horn                     0xfe62
#define XK_dead_stroke                   0xfe63
#define XK_dead_abovecomma               0xfe64
#define XK_dead_psili                    0xfe64  /* non-deprecated alias for dead_abovecomma */
#define XK_dead_abovereversedcomma       0xfe65
#define XK_dead_dasia                    0xfe65  /* non-deprecated alias for dead_abovereversedcomma */
#define XK_dead_doublegrave              0xfe66
#define XK_dead_belowring                0xfe67
#define XK_dead_belowmacron              0xfe68
#define XK_dead_belowcircumflex          0xfe69
#define XK_dead_belowtilde               0xfe6a
#define XK_dead_belowbreve               0xfe6b
#define XK_dead_belowdiaeresis           0xfe6c
#define XK_dead_invertedbreve            0xfe6d
#define XK_dead_belowcomma               0xfe6e
#define XK_dead_currency                 0xfe6f

/* extra dead elements for German T3 layout */
#define XK_dead_lowline                  0xfe90
#define XK_dead_aboveverticalline        0xfe91
#define XK_dead_belowverticalline        0xfe92
#define XK_dead_longsolidusoverlay       0xfe93

/* dead vowels for universal syllable entry */
#define XK_dead_a                        0xfe80
#define XK_dead_A                        0xfe81
#define XK_dead_e                        0xfe82
#define XK_dead_E                        0xfe83
#define XK_dead_i                        0xfe84
#define XK_dead_I                        0xfe85
#define XK_dead_o                        0xfe86
#define XK_dead_O                        0xfe87
#define XK_dead_u                        0xfe88
#define XK_dead_U                        0xfe89
#define XK_dead_small_schwa              0xfe8a  /* deprecated alias for dead_schwa */
#define XK_dead_schwa                    0xfe8a
#define XK_dead_capital_schwa            0xfe8b  /* deprecated alias for dead_SCHWA */
#define XK_dead_SCHWA                    0xfe8b

#define XK_dead_greek                    0xfe8c
#define XK_dead_hamza                    0xfe8d

#define XK_First_Virtual_Screen          0xfed0
#define XK_Prev_Virtual_Screen           0xfed1
#define XK_Next_Virtual_Screen           0xfed2
#define XK_Last_Virtual_Screen           0xfed4
#define XK_Terminate_Server              0xfed5

#define XK_AccessX_Enable                0xfe70
#define XK_AccessX_Feedback_Enable       0xfe71
#define XK_RepeatKeys_Enable             0xfe72
#define XK_SlowKeys_Enable               0xfe73
#define XK_BounceKeys_Enable             0xfe74
#define XK_StickyKeys_Enable             0xfe75
#define XK_MouseKeys_Enable              0xfe76
#define XK_MouseKeys_Accel_Enable        0xfe77
#define XK_Overlay1_Enable               0xfe78
#define XK_Overlay2_Enable               0xfe79
#define XK_AudibleBell_Enable            0xfe7a

#define XK_Pointer_Left                  0xfee0
#define XK_Pointer_Right                 0xfee1
#define XK_Pointer_Up                    0xfee2
#define XK_Pointer_Down                  0xfee3
#define XK_Pointer_UpLeft                0xfee4
#define XK_Pointer_UpRight               0xfee5
#define XK_Pointer_DownLeft              0xfee6
#define XK_Pointer_DownRight             0xfee7
#define XK_Pointer_Button_Dflt           0xfee8
#define XK_Pointer_Button1               0xfee9
#define XK_Pointer_Button2               0xfeea
#define XK_Pointer_Button3               0xfeeb
#define XK_Pointer_Button4               0xfeec
#define XK_Pointer_Button5               0xfeed
#define XK_Pointer_DblClick_Dflt         0xfeee
#define XK_Pointer_DblClick1             0xfeef
#define XK_Pointer_DblClick2             0xfef0
#define XK_Pointer_DblClick3             0xfef1
#define XK_Pointer_DblClick4             0xfef2
#define XK_Pointer_DblClick5             0xfef3
#define XK_Pointer_Drag_Dflt             0xfef4
#define XK_Pointer_Drag1                 0xfef5
#define XK_Pointer_Drag2                 0xfef6
#define XK_Pointer_Drag3                 0xfef7
#define XK_Pointer_Drag4                 0xfef8
#define XK_Pointer_Drag5                 0xfefd

#define XK_Pointer_EnableKeys            0xfef9
#define XK_Pointer_Accelerate            0xfefa
#define XK_Pointer_DfltBtnNext           0xfefb
#define XK_Pointer_DfltBtnPrev           0xfefc

/* Single-Stroke Multiple-Character N-Graph Keysyms For The X Input Method */

#define XK_ch                            0xfea0
#define XK_Ch                            0xfea1
#define XK_CH                            0xfea2
#define XK_c_h                           0xfea3
#define XK_C_h                           0xfea4
#define XK_C_H                           0xfea5

#endif /* XK_XKB_KEYS */

/*
 * 3270 Terminal Keys
 * Byte 3 = 0xfd
 */

#ifdef XK_3270
#define XK_3270_Duplicate                0xfd01
#define XK_3270_FieldMark                0xfd02
#define XK_3270_Right2                   0xfd03
#define XK_3270_Left2                    0xfd04
#define XK_3270_BackTab                  0xfd05
#define XK_3270_EraseEOF                 0xfd06
#define XK_3270_EraseInput               0xfd07
#define XK_3270_Reset                    0xfd08
#define XK_3270_Quit                     0xfd09
#define XK_3270_PA1                      0xfd0a
#define XK_3270_PA2                      0xfd0b
#define XK_3270_PA3                      0xfd0c
#define XK_3270_Test                     0xfd0d
#define XK_3270_Attn                     0xfd0e
#define XK_3270_CursorBlink              0xfd0f
#define XK_3270_AltCursor                0xfd10
#define XK_3270_KeyClick                 0xfd11
#define XK_3270_Jump                     0xfd12
#define XK_3270_Ident                    0xfd13
#define XK_3270_Rule                     0xfd14
#define XK_3270_Copy                     0xfd15
#define XK_3270_Play                     0xfd16
#define XK_3270_Setup                    0xfd17
#define XK_3270_Record                   0xfd18
#define XK_3270_ChangeScreen             0xfd19
#define XK_3270_DeleteWord               0xfd1a
#define XK_3270_ExSelect                 0xfd1b
#define XK_3270_CursorSelect             0xfd1c
#define XK_3270_PrintScreen              0xfd1d
#define XK_3270_Enter                    0xfd1e
#endif /* XK_3270 */

/*
 * Latin 1
 * (ISO/IEC 8859-1 = Unicode U+0020..U+00FF)
 * Byte 3 = 0
 */
#ifdef XK_LATIN1
#define XK_space                         0x0020  /* U+0020 SPACE */
#define XK_exclam                        0x0021  /* U+0021 EXCLAMATION MARK */
#define XK_quotedbl                      0x0022  /* U+0022 QUOTATION MARK */
#define XK_numbersign                    0x0023  /* U+0023 NUMBER SIGN */
#define XK_dollar                        0x0024  /* U+0024 DOLLAR SIGN */
#define XK_percent                       0x0025  /* U+0025 PERCENT SIGN */
#define XK_ampersand                     0x0026  /* U+0026 AMPERSAND */
#define XK_apostrophe                    0x0027  /* U+0027 APOSTROPHE */
#define XK_quoteright                    0x0027  /* deprecated */
#define XK_parenleft                     0x0028  /* U+0028 LEFT PARENTHESIS */
#define XK_parenright                    0x0029  /* U+0029 RIGHT PARENTHESIS */
#define XK_asterisk                      0x002a  /* U+002A ASTERISK */
#define XK_plus                          0x002b  /* U+002B PLUS SIGN */
#define XK_comma                         0x002c  /* U+002C COMMA */
#define XK_minus                         0x002d  /* U+002D HYPHEN-MINUS */
#define XK_period                        0x002e  /* U+002E FULL STOP */
#define XK_slash                         0x002f  /* U+002F SOLIDUS */
#define XK_0                             0x0030  /* U+0030 DIGIT ZERO */
#define XK_1                             0x0031  /* U+0031 DIGIT ONE */
#define XK_2                             0x0032  /* U+0032 DIGIT TWO */
#define XK_3                             0x0033  /* U+0033 DIGIT THREE */
#define XK_4                             0x0034  /* U+0034 DIGIT FOUR */
#define XK_5                             0x0035  /* U+0035 DIGIT FIVE */
#define XK_6                             0x0036  /* U+0036 DIGIT SIX */
#define XK_7                             0x0037  /* U+0037 DIGIT SEVEN */
#define XK_8                             0x0038  /* U+0038 DIGIT EIGHT */
#define XK_9                             0x0039  /* U+0039 DIGIT NINE */
#define XK_colon                         0x003a  /* U+003A COLON */
#define XK_semicolon                     0x003b  /* U+003B SEMICOLON */
#define XK_less                          0x003c  /* U+003C LESS-THAN SIGN */
#define XK_equal                         0x003d  /* U+003D EQUALS SIGN */
#define XK_greater                       0x003e  /* U+003E GREATER-THAN SIGN */
#define XK_question                      0x003f  /* U+003F QUESTION MARK */
#define XK_at                            0x0040  /* U+0040 COMMERCIAL AT */
#define XK_A                             0x0041  /* U+0041 LATIN CAPITAL LETTER A */
#define XK_B                             0x0042  /* U+0042 LATIN CAPITAL LETTER B */
#define XK_C                             0x0043  /* U+0043 LATIN CAPITAL LETTER C */
#define XK_D                             0x0044  /* U+0044 LATIN CAPITAL LETTER D */
#define XK_E                             0x0045  /* U+0045 LATIN CAPITAL LETTER E */
#define XK_F                             0x0046  /* U+0046 LATIN CAPITAL LETTER F */
#define XK_G                             0x0047  /* U+0047 LATIN CAPITAL LETTER G */
#define XK_H                             0x0048  /* U+0048 LATIN CAPITAL LETTER H */
#define XK_I                             0x0049  /* U+0049 LATIN CAPITAL LETTER I */
#define XK_J                             0x004a  /* U+004A LATIN CAPITAL LETTER J */
#define XK_K                             0x004b  /* U+004B LATIN CAPITAL LETTER K */
#define XK_L                             0x004c  /* U+004C LATIN CAPITAL LETTER L */
#define XK_M                             0x004d  /* U+004D LATIN CAPITAL LETTER M */
#define XK_N                             0x004e  /* U+004E LATIN CAPITAL LETTER N */
#define XK_O                             0x004f  /* U+004F LATIN CAPITAL LETTER O */
#define XK_P                             0x0050  /* U+0050 LATIN CAPITAL LETTER P */
#define XK_Q                             0x0051  /* U+0051 LATIN CAPITAL LETTER Q */
#define XK_R                             0x0052  /* U+0052 LATIN CAPITAL LETTER R */
#define XK_S                             0x0053  /* U+0053 LATIN CAPITAL LETTER S */
#define XK_T                             0x0054  /* U+0054 LATIN CAPITAL LETTER T */
#define XK_U                             0x0055  /* U+0055 LATIN CAPITAL LETTER U */
#define XK_V                             0x0056  /* U+0056 LATIN CAPITAL LETTER V */
#define XK_W                             0x0057  /* U+0057 LATIN CAPITAL LETTER W */
#define XK_X                             0x0058  /* U+0058 LATIN CAPITAL LETTER X */
#define XK_Y                             0x0059  /* U+0059 LATIN CAPITAL LETTER Y */
#define XK_Z                             0x005a  /* U+005A LATIN CAPITAL LETTER Z */
#define XK_bracketleft                   0x005b  /* U+005B LEFT SQUARE BRACKET */
#define XK_backslash                     0x005c  /* U+005C REVERSE SOLIDUS */
#define XK_bracketright                  0x005d  /* U+005D RIGHT SQUARE BRACKET */
#define XK_asciicircum                   0x005e  /* U+005E CIRCUMFLEX ACCENT */
#define XK_underscore                    0x005f  /* U+005F LOW LINE */
#define XK_grave                         0x0060  /* U+0060 GRAVE ACCENT */
#define XK_quoteleft                     0x0060  /* deprecated */
#define XK_a                             0x0061  /* U+0061 LATIN SMALL LETTER A */
#define XK_b                             0x0062  /* U+0062 LATIN SMALL LETTER B */
#define XK_c                             0x0063  /* U+0063 LATIN SMALL LETTER C */
#define XK_d                             0x0064  /* U+0064 LATIN SMALL LETTER D */
#define XK_e                             0x0065  /* U+0065 LATIN SMALL LETTER E */
#define XK_f                             0x0066  /* U+0066 LATIN SMALL LETTER F */
#define XK_g                             0x0067  /* U+0067 LATIN SMALL LETTER G */
#define XK_h                             0x0068  /* U+0068 LATIN SMALL LETTER H */
#define XK_i                             0x0069  /* U+0069 LATIN SMALL LETTER I */
#define XK_j                             0x006a  /* U+006A LATIN SMALL LETTER J */
#define XK_k                             0x006b  /* U+006B LATIN SMALL LETTER K */
#define XK_l                             0x006c  /* U+006C LATIN SMALL LETTER L */
#define XK_m                             0x006d  /* U+006D LATIN SMALL LETTER M */
#define XK_n                             0x006e  /* U+006E LATIN SMALL LETTER N */
#define XK_o                             0x006f  /* U+006F LATIN SMALL LETTER O */
#define XK_p                             0x0070  /* U+0070 LATIN SMALL LETTER P */
#define XK_q                             0x0071  /* U+0071 LATIN SMALL LETTER Q */
#define XK_r                             0x0072  /* U+0072 LATIN SMALL LETTER R */
#define XK_s                             0x0073  /* U+0073 LATIN SMALL LETTER S */
#define XK_t                             0x0074  /* U+0074 LATIN SMALL LETTER T */
#define XK_u                             0x0075  /* U+0075 LATIN SMALL LETTER U */
#define XK_v                             0x0076  /* U+0076 LATIN SMALL LETTER V */
#define XK_w                             0x0077  /* U+0077 LATIN SMALL LETTER W */
#define XK_x                             0x0078  /* U+0078 LATIN SMALL LETTER X */
#define XK_y                             0x0079  /* U+0079 LATIN SMALL LETTER Y */
#define XK_z                             0x007a  /* U+007A LATIN SMALL LETTER Z */
#define XK_braceleft                     0x007b  /* U+007B LEFT CURLY BRACKET */
#define XK_bar                           0x007c  /* U+007C VERTICAL LINE */
#define XK_braceright                    0x007d  /* U+007D RIGHT CURLY BRACKET */
#define XK_asciitilde                    0x007e  /* U+007E TILDE */

#define XK_nobreakspace                  0x00a0  /* U+00A0 NO-BREAK SPACE */
#define XK_exclamdown                    0x00a1  /* U+00A1 INVERTED EXCLAMATION MARK */
#define XK_cent                          0x00a2  /* U+00A2 CENT SIGN */
#define XK_sterling                      0x00a3  /* U+00A3 POUND SIGN */
#define XK_currency                      0x00a4  /* U+00A4 CURRENCY SIGN */
#define XK_yen                           0x00a5  /* U+00A5 YEN SIGN */
#define XK_brokenbar                     0x00a6  /* U+00A6 BROKEN BAR */
#define XK_section                       0x00a7  /* U+00A7 SECTION SIGN */
#define XK_diaeresis                     0x00a8  /* U+00A8 DIAERESIS */
#define XK_copyright                     0x00a9  /* U+00A9 COPYRIGHT SIGN */
#define XK_ordfeminine                   0x00aa  /* U+00AA FEMININE ORDINAL INDICATOR */
#define XK_guillemotleft                 0x00ab  /* deprecated alias for guillemetleft (misspelling) */
#define XK_guillemetleft                 0x00ab  /* U+00AB LEFT-POINTING DOUBLE ANGLE QUOTATION MARK */
#define XK_notsign                       0x00ac  /* U+00AC NOT SIGN */
#define XK_hyphen                        0x00ad  /* U+00AD SOFT HYPHEN */
#define XK_registered                    0x00ae  /* U+00AE REGISTERED SIGN */
#define XK_macron                        0x00af  /* U+00AF MACRON */
#define XK_degree                        0x00b0  /* U+00B0 DEGREE SIGN */
#define XK_plusminus                     0x00b1  /* U+00B1 PLUS-MINUS SIGN */
#define XK_twosuperior                   0x00b2  /* U+00B2 SUPERSCRIPT TWO */
#define XK_threesuperior                 0x00b3  /* U+00B3 SUPERSCRIPT THREE */
#define XK_acute                         0x00b4  /* U+00B4 ACUTE ACCENT */
#define XK_mu                            0x00b5  /* U+00B5 MICRO SIGN */
#define XK_paragraph                     0x00b6  /* U+00B6 PILCROW SIGN */
#define XK_periodcentered                0x00b7  /* U+00B7 MIDDLE DOT */
#define XK_cedilla                       0x00b8  /* U+00B8 CEDILLA */
#define XK_onesuperior                   0x00b9  /* U+00B9 SUPERSCRIPT ONE */
#define XK_masculine                     0x00ba  /* deprecated alias for ordmasculine (inconsistent name) */
#define XK_ordmasculine                  0x00ba  /* U+00BA MASCULINE ORDINAL INDICATOR */
#define XK_guillemotright                0x00bb  /* deprecated alias for guillemetright (misspelling) */
#define XK_guillemetright                0x00bb  /* U+00BB RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK */
#define XK_onequarter                    0x00bc  /* U+00BC VULGAR FRACTION ONE QUARTER */
#define XK_onehalf                       0x00bd  /* U+00BD VULGAR FRACTION ONE HALF */
#define XK_threequarters                 0x00be  /* U+00BE VULGAR FRACTION THREE QUARTERS */
#define XK_questiondown                  0x00bf  /* U+00BF INVERTED QUESTION MARK */
#define XK_Agrave                        0x00c0  /* U+00C0 LATIN CAPITAL LETTER A WITH GRAVE */
#define XK_Aacute                        0x00c1  /* U+00C1 LATIN CAPITAL LETTER A WITH ACUTE */
#define XK_Acircumflex                   0x00c2  /* U+00C2 LATIN CAPITAL LETTER A WITH CIRCUMFLEX */
#define XK_Atilde                        0x00c3  /* U+00C3 LATIN CAPITAL LETTER A WITH TILDE */
#define XK_Adiaeresis                    0x00c4  /* U+00C4 LATIN CAPITAL LETTER A WITH DIAERESIS */
#define XK_Aring                         0x00c5  /* U+00C5 LATIN CAPITAL LETTER A WITH RING ABOVE */
#define XK_AE                            0x00c6  /* U+00C6 LATIN CAPITAL LETTER AE */
#define XK_Ccedilla                      0x00c7  /* U+00C7 LATIN CAPITAL LETTER C WITH CEDILLA */
#define XK_Egrave                        0x00c8  /* U+00C8 LATIN CAPITAL LETTER E WITH GRAVE */
#define XK_Eacute                        0x00c9  /* U+00C9 LATIN CAPITAL LETTER E WITH ACUTE */
#define XK_Ecircumflex                   0x00ca  /* U+00CA LATIN CAPITAL LETTER E WITH CIRCUMFLEX */
#define XK_Ediaeresis                    0x00cb  /* U+00CB LATIN CAPITAL LETTER E WITH DIAERESIS */
#define XK_Igrave                        0x00cc  /* U+00CC LATIN CAPITAL LETTER I WITH GRAVE */
#define XK_Iacute                        0x00cd  /* U+00CD LATIN CAPITAL LETTER I WITH ACUTE */
#define XK_Icircumflex                   0x00ce  /* U+00CE LATIN CAPITAL LETTER I WITH CIRCUMFLEX */
#define XK_Idiaeresis                    0x00cf  /* U+00CF LATIN CAPITAL LETTER I WITH DIAERESIS */
#define XK_ETH                           0x00d0  /* U+00D0 LATIN CAPITAL LETTER ETH */
#define XK_Eth                           0x00d0  /* deprecated */
#define XK_Ntilde                        0x00d1  /* U+00D1 LATIN CAPITAL LETTER N WITH TILDE */
#define XK_Ograve                        0x00d2  /* U+00D2 LATIN CAPITAL LETTER O WITH GRAVE */
#define XK_Oacute                        0x00d3  /* U+00D3 LATIN CAPITAL LETTER O WITH ACUTE */
#define XK_Ocircumflex                   0x00d4  /* U+00D4 LATIN CAPITAL LETTER O WITH CIRCUMFLEX */
#define XK_Otilde                        0x00d5  /* U+00D5 LATIN CAPITAL LETTER O WITH TILDE */
#define XK_Odiaeresis                    0x00d6  /* U+00D6 LATIN CAPITAL LETTER O WITH DIAERESIS */
#define XK_multiply                      0x00d7  /* U+00D7 MULTIPLICATION SIGN */
#define XK_Oslash                        0x00d8  /* U+00D8 LATIN CAPITAL LETTER O WITH STROKE */
#define XK_Ooblique                      0x00d8  /* deprecated alias for Oslash */
#define XK_Ugrave                        0x00d9  /* U+00D9 LATIN CAPITAL LETTER U WITH GRAVE */
#define XK_Uacute                        0x00da  /* U+00DA LATIN CAPITAL LETTER U WITH ACUTE */
#define XK_Ucircumflex                   0x00db  /* U+00DB LATIN CAPITAL LETTER U WITH CIRCUMFLEX */
#define XK_Udiaeresis                    0x00dc  /* U+00DC LATIN CAPITAL LETTER U WITH DIAERESIS */
#define XK_Yacute                        0x00dd  /* U+00DD LATIN CAPITAL LETTER Y WITH ACUTE */
#define XK_THORN                         0x00de  /* U+00DE LATIN CAPITAL LETTER THORN */
#define XK_Thorn                         0x00de  /* deprecated */
#define XK_ssharp                        0x00df  /* U+00DF LATIN SMALL LETTER SHARP S */
#define XK_agrave                        0x00e0  /* U+00E0 LATIN SMALL LETTER A WITH GRAVE */
#define XK_aacute                        0x00e1  /* U+00E1 LATIN SMALL LETTER A WITH ACUTE */
#define XK_acircumflex                   0x00e2  /* U+00E2 LATIN SMALL LETTER A WITH CIRCUMFLEX */
#define XK_atilde                        0x00e3  /* U+00E3 LATIN SMALL LETTER A WITH TILDE */
#define XK_adiaeresis                    0x00e4  /* U+00E4 LATIN SMALL LETTER A WITH DIAERESIS */
#define XK_aring                         0x00e5  /* U+00E5 LATIN SMALL LETTER A WITH RING ABOVE */
#define XK_ae                            0x00e6  /* U+00E6 LATIN SMALL LETTER AE */
#define XK_ccedilla                      0x00e7  /* U+00E7 LATIN SMALL LETTER C WITH CEDILLA */
#define XK_egrave                        0x00e8  /* U+00E8 LATIN SMALL LETTER E WITH GRAVE */
#define XK_eacute                        0x00e9  /* U+00E9 LATIN SMALL LETTER E WITH ACUTE */
#define XK_ecircumflex                   0x00ea  /* U+00EA LATIN SMALL LETTER E WITH CIRCUMFLEX */
#define XK_ediaeresis                    0x00eb  /* U+00EB LATIN SMALL LETTER E WITH DIAERESIS */
#define XK_igrave                        0x00ec  /* U+00EC LATIN SMALL LETTER I WITH GRAVE */
#define XK_iacute                        0x00ed  /* U+00ED LATIN SMALL LETTER I WITH ACUTE */
#define XK_icircumflex                   0x00ee  /* U+00EE LATIN SMALL LETTER I WITH CIRCUMFLEX */
#define XK_idiaeresis                    0x00ef  /* U+00EF LATIN SMALL LETTER I WITH DIAERESIS */
#define XK_eth                           0x00f0  /* U+00F0 LATIN SMALL LETTER ETH */
#define XK_ntilde                        0x00f1  /* U+00F1 LATIN SMALL LETTER N WITH TILDE */
#define XK_ograve                        0x00f2  /* U+00F2 LATIN SMALL LETTER O WITH GRAVE */
#define XK_oacute                        0x00f3  /* U+00F3 LATIN SMALL LETTER O WITH ACUTE */
#define XK_ocircumflex                   0x00f4  /* U+00F4 LATIN SMALL LETTER O WITH CIRCUMFLEX */
#define XK_otilde                        0x00f5  /* U+00F5 LATIN SMALL LETTER O WITH TILDE */
#define XK_odiaeresis                    0x00f6  /* U+00F6 LATIN SMALL LETTER O WITH DIAERESIS */
#define XK_division                      0x00f7  /* U+00F7 DIVISION SIGN */
#define XK_oslash                        0x00f8  /* U+00F8 LATIN SMALL LETTER O WITH STROKE */
#define XK_ooblique                      0x00f8  /* deprecated alias for oslash */
#define XK_ugrave                        0x00f9  /* U+00F9 LATIN SMALL LETTER U WITH GRAVE */
#define XK_uacute                        0x00fa  /* U+00FA LATIN SMALL LETTER U WITH ACUTE */
#define XK_ucircumflex                   0x00fb  /* U+00FB LATIN SMALL LETTER U WITH CIRCUMFLEX */
#define XK_udiaeresis                    0x00fc  /* U+00FC LATIN SMALL LETTER U WITH DIAERESIS */
#define XK_yacute                        0x00fd  /* U+00FD LATIN SMALL LETTER Y WITH ACUTE */
#define XK_thorn                         0x00fe  /* U+00FE LATIN SMALL LETTER THORN */
#define XK_ydiaeresis                    0x00ff  /* U+00FF LATIN SMALL LETTER Y WITH DIAERESIS */
#endif /* XK_LATIN1 */

/*
 * Latin 2
 * Byte 3 = 1
 */

#ifdef XK_LATIN2
#define XK_Aogonek                       0x01a1  /* U+0104 LATIN CAPITAL LETTER A WITH OGONEK */
#define XK_breve                         0x01a2  /* U+02D8 BREVE */
#define XK_Lstroke                       0x01a3  /* U+0141 LATIN CAPITAL LETTER L WITH STROKE */
#define XK_Lcaron                        0x01a5  /* U+013D LATIN CAPITAL LETTER L WITH CARON */
#define XK_Sacute                        0x01a6  /* U+015A LATIN CAPITAL LETTER S WITH ACUTE */
#define XK_Scaron                        0x01a9  /* U+0160 LATIN CAPITAL LETTER S WITH CARON */
#define XK_Scedilla                      0x01aa  /* U+015E LATIN CAPITAL LETTER S WITH CEDILLA */
#define XK_Tcaron                        0x01ab  /* U+0164 LATIN CAPITAL LETTER T WITH CARON */
#define XK_Zacute                        0x01ac  /* U+0179 LATIN CAPITAL LETTER Z WITH ACUTE */
#define XK_Zcaron                        0x01ae  /* U+017D LATIN CAPITAL LETTER Z WITH CARON */
#define XK_Zabovedot                     0x01af  /* U+017B LATIN CAPITAL LETTER Z WITH DOT ABOVE */
#define XK_aogonek                       0x01b1  /* U+0105 LATIN SMALL LETTER A WITH OGONEK */
#define XK_ogonek                        0x01b2  /* U+02DB OGONEK */
#define XK_lstroke                       0x01b3  /* U+0142 LATIN SMALL LETTER L WITH STROKE */
#define XK_lcaron                        0x01b5  /* U+013E LATIN SMALL LETTER L WITH CARON */
#define XK_sacute                        0x01b6  /* U+015B LATIN SMALL LETTER S WITH ACUTE */
#define XK_caron                         0x01b7  /* U+02C7 CARON */
#define XK_scaron                        0x01b9  /* U+0161 LATIN SMALL LETTER S WITH CARON */
#define XK_scedilla                      0x01ba  /* U+015F LATIN SMALL LETTER S WITH CEDILLA */
#define XK_tcaron                        0x01bb  /* U+0165 LATIN SMALL LETTER T WITH CARON */
#define XK_zacute                        0x01bc  /* U+017A LATIN SMALL LETTER Z WITH ACUTE */
#define XK_doubleacute                   0x01bd  /* U+02DD DOUBLE ACUTE ACCENT */
#define XK_zcaron                        0x01be  /* U+017E LATIN SMALL LETTER Z WITH CARON */
#define XK_zabovedot                     0x01bf  /* U+017C LATIN SMALL LETTER Z WITH DOT ABOVE */
#define XK_Racute                        0x01c0  /* U+0154 LATIN CAPITAL LETTER R WITH ACUTE */
#define XK_Abreve                        0x01c3  /* U+0102 LATIN CAPITAL LETTER A WITH BREVE */
#define XK_Lacute                        0x01c5  /* U+0139 LATIN CAPITAL LETTER L WITH ACUTE */
#define XK_Cacute                        0x01c6  /* U+0106 LATIN CAPITAL LETTER C WITH ACUTE */
#define XK_Ccaron                        0x01c8  /* U+010C LATIN CAPITAL LETTER C WITH CARON */
#define XK_Eogonek                       0x01ca  /* U+0118 LATIN CAPITAL LETTER E WITH OGONEK */
#define XK_Ecaron                        0x01cc  /* U+011A LATIN CAPITAL LETTER E WITH CARON */
#define XK_Dcaron                        0x01cf  /* U+010E LATIN CAPITAL LETTER D WITH CARON */
#define XK_Dstroke                       0x01d0  /* U+0110 LATIN CAPITAL LETTER D WITH STROKE */
#define XK_Nacute                        0x01d1  /* U+0143 LATIN CAPITAL LETTER N WITH ACUTE */
#define XK_Ncaron                        0x01d2  /* U+0147 LATIN CAPITAL LETTER N WITH CARON */
#define XK_Odoubleacute                  0x01d5  /* U+0150 LATIN CAPITAL LETTER O WITH DOUBLE ACUTE */
#define XK_Rcaron                        0x01d8  /* U+0158 LATIN CAPITAL LETTER R WITH CARON */
#define XK_Uring                         0x01d9  /* U+016E LATIN CAPITAL LETTER U WITH RING ABOVE */
#define XK_Udoubleacute                  0x01db  /* U+0170 LATIN CAPITAL LETTER U WITH DOUBLE ACUTE */
#define XK_Tcedilla                      0x01de  /* U+0162 LATIN CAPITAL LETTER T WITH CEDILLA */
#define XK_racute                        0x01e0  /* U+0155 LATIN SMALL LETTER R WITH ACUTE */
#define XK_abreve                        0x01e3  /* U+0103 LATIN SMALL LETTER A WITH BREVE */
#define XK_lacute                        0x01e5  /* U+013A LATIN SMALL LETTER L WITH ACUTE */
#define XK_cacute                        0x01e6  /* U+0107 LATIN SMALL LETTER C WITH ACUTE */
#define XK_ccaron                        0x01e8  /* U+010D LATIN SMALL LETTER C WITH CARON */
#define XK_eogonek                       0x01ea  /* U+0119 LATIN SMALL LETTER E WITH OGONEK */
#define XK_ecaron                        0x01ec  /* U+011B LATIN SMALL LETTER E WITH CARON */
#define XK_dcaron                        0x01ef  /* U+010F LATIN SMALL LETTER D WITH CARON */
#define XK_dstroke                       0x01f0  /* U+0111 LATIN SMALL LETTER D WITH STROKE */
#define XK_nacute                        0x01f1  /* U+0144 LATIN SMALL LETTER N WITH ACUTE */
#define XK_ncaron                        0x01f2  /* U+0148 LATIN SMALL LETTER N WITH CARON */
#define XK_odoubleacute                  0x01f5  /* U+0151 LATIN SMALL LETTER O WITH DOUBLE ACUTE */
#define XK_rcaron                        0x01f8  /* U+0159 LATIN SMALL LETTER R WITH CARON */
#define XK_uring                         0x01f9  /* U+016F LATIN SMALL LETTER U WITH RING ABOVE */
#define XK_udoubleacute                  0x01fb  /* U+0171 LATIN SMALL LETTER U WITH DOUBLE ACUTE */
#define XK_tcedilla                      0x01fe  /* U+0163 LATIN SMALL LETTER T WITH CEDILLA */
#define XK_abovedot                      0x01ff  /* U+02D9 DOT ABOVE */
#endif /* XK_LATIN2 */

/*
 * Latin 3
 * Byte 3 = 2
 */

#ifdef XK_LATIN3
#define XK_Hstroke                       0x02a1  /* U+0126 LATIN CAPITAL LETTER H WITH STROKE */
#define XK_Hcircumflex                   0x02a6  /* U+0124 LATIN CAPITAL LETTER H WITH CIRCUMFLEX */
#define XK_Iabovedot                     0x02a9  /* U+0130 LATIN CAPITAL LETTER I WITH DOT ABOVE */
#define XK_Gbreve                        0x02ab  /* U+011E LATIN CAPITAL LETTER G WITH BREVE */
#define XK_Jcircumflex                   0x02ac  /* U+0134 LATIN CAPITAL LETTER J WITH CIRCUMFLEX */
#define XK_hstroke                       0x02b1  /* U+0127 LATIN SMALL LETTER H WITH STROKE */
#define XK_hcircumflex                   0x02b6  /* U+0125 LATIN SMALL LETTER H WITH CIRCUMFLEX */
#define XK_idotless                      0x02b9  /* U+0131 LATIN SMALL LETTER DOTLESS I */
#define XK_gbreve                        0x02bb  /* U+011F LATIN SMALL LETTER G WITH BREVE */
#define XK_jcircumflex                   0x02bc  /* U+0135 LATIN SMALL LETTER J WITH CIRCUMFLEX */
#define XK_Cabovedot                     0x02c5  /* U+010A LATIN CAPITAL LETTER C WITH DOT ABOVE */
#define XK_Ccircumflex                   0x02c6  /* U+0108 LATIN CAPITAL LETTER C WITH CIRCUMFLEX */
#define XK_Gabovedot                     0x02d5  /* U+0120 LATIN CAPITAL LETTER G WITH DOT ABOVE */
#define XK_Gcircumflex                   0x02d8  /* U+011C LATIN CAPITAL LETTER G WITH CIRCUMFLEX */
#define XK_Ubreve                        0x02dd  /* U+016C LATIN CAPITAL LETTER U WITH BREVE */
#define XK_Scircumflex                   0x02de  /* U+015C LATIN CAPITAL LETTER S WITH CIRCUMFLEX */
#define XK_cabovedot                     0x02e5  /* U+010B LATIN SMALL LETTER C WITH DOT ABOVE */
#define XK_ccircumflex                   0x02e6  /* U+0109 LATIN SMALL LETTER C WITH CIRCUMFLEX */
#define XK_gabovedot                     0x02f5  /* U+0121 LATIN SMALL LETTER G WITH DOT ABOVE */
#define XK_gcircumflex                   0x02f8  /* U+011D LATIN SMALL LETTER G WITH CIRCUMFLEX */
#define XK_ubreve                        0x02fd  /* U+016D LATIN SMALL LETTER U WITH BREVE */
#define XK_scircumflex                   0x02fe  /* U+015D LATIN SMALL LETTER S WITH CIRCUMFLEX */
#endif /* XK_LATIN3 */


/*
 * Latin 4
 * Byte 3 = 3
 */

#ifdef XK_LATIN4
#define XK_kra                           0x03a2  /* U+0138 LATIN SMALL LETTER KRA */
#define XK_kappa                         0x03a2  /* deprecated */
#define XK_Rcedilla                      0x03a3  /* U+0156 LATIN CAPITAL LETTER R WITH CEDILLA */
#define XK_Itilde                        0x03a5  /* U+0128 LATIN CAPITAL LETTER I WITH TILDE */
#define XK_Lcedilla                      0x03a6  /* U+013B LATIN CAPITAL LETTER L WITH CEDILLA */
#define XK_Emacron                       0x03aa  /* U+0112 LATIN CAPITAL LETTER E WITH MACRON */
#define XK_Gcedilla                      0x03ab  /* U+0122 LATIN CAPITAL LETTER G WITH CEDILLA */
#define XK_Tslash                        0x03ac  /* U+0166 LATIN CAPITAL LETTER T WITH STROKE */
#define XK_rcedilla                      0x03b3  /* U+0157 LATIN SMALL LETTER R WITH CEDILLA */
#define XK_itilde                        0x03b5  /* U+0129 LATIN SMALL LETTER I WITH TILDE */
#define XK_lcedilla                      0x03b6  /* U+013C LATIN SMALL LETTER L WITH CEDILLA */
#define XK_emacron                       0x03ba  /* U+0113 LATIN SMALL LETTER E WITH MACRON */
#define XK_gcedilla                      0x03bb  /* U+0123 LATIN SMALL LETTER G WITH CEDILLA */
#define XK_tslash                        0x03bc  /* U+0167 LATIN SMALL LETTER T WITH STROKE */
#define XK_ENG                           0x03bd  /* U+014A LATIN CAPITAL LETTER ENG */
#define XK_eng                           0x03bf  /* U+014B LATIN SMALL LETTER ENG */
#define XK_Amacron                       0x03c0  /* U+0100 LATIN CAPITAL LETTER A WITH MACRON */
#define XK_Iogonek                       0x03c7  /* U+012E LATIN CAPITAL LETTER I WITH OGONEK */
#define XK_Eabovedot                     0x03cc  /* U+0116 LATIN CAPITAL LETTER E WITH DOT ABOVE */
#define XK_Imacron                       0x03cf  /* U+012A LATIN CAPITAL LETTER I WITH MACRON */
#define XK_Ncedilla                      0x03d1  /* U+0145 LATIN CAPITAL LETTER N WITH CEDILLA */
#define XK_Omacron                       0x03d2  /* U+014C LATIN CAPITAL LETTER O WITH MACRON */
#define XK_Kcedilla                      0x03d3  /* U+0136 LATIN CAPITAL LETTER K WITH CEDILLA */
#define XK_Uogonek                       0x03d9  /* U+0172 LATIN CAPITAL LETTER U WITH OGONEK */
#define XK_Utilde                        0x03dd  /* U+0168 LATIN CAPITAL LETTER U WITH TILDE */
#define XK_Umacron                       0x03de  /* U+016A LATIN CAPITAL LETTER U WITH MACRON */
#define XK_amacron                       0x03e0  /* U+0101 LATIN SMALL LETTER A WITH MACRON */
#define XK_iogonek                       0x03e7  /* U+012F LATIN SMALL LETTER I WITH OGONEK */
#define XK_eabovedot                     0x03ec  /* U+0117 LATIN SMALL LETTER E WITH DOT ABOVE */
#define XK_imacron                       0x03ef  /* U+012B LATIN SMALL LETTER I WITH MACRON */
#define XK_ncedilla                      0x03f1  /* U+0146 LATIN SMALL LETTER N WITH CEDILLA */
#define XK_omacron                       0x03f2  /* U+014D LATIN SMALL LETTER O WITH MACRON */
#define XK_kcedilla                      0x03f3  /* U+0137 LATIN SMALL LETTER K WITH CEDILLA */
#define XK_uogonek                       0x03f9  /* U+0173 LATIN SMALL LETTER U WITH OGONEK */
#define XK_utilde                        0x03fd  /* U+0169 LATIN SMALL LETTER U WITH TILDE */
#define XK_umacron                       0x03fe  /* U+016B LATIN SMALL LETTER U WITH MACRON */
#endif /* XK_LATIN4 */

/*
 * Latin 8
 */
#ifdef XK_LATIN8
#define XK_Wcircumflex                0x1000174  /* U+0174 LATIN CAPITAL LETTER W WITH CIRCUMFLEX */
#define XK_wcircumflex                0x1000175  /* U+0175 LATIN SMALL LETTER W WITH CIRCUMFLEX */
#define XK_Ycircumflex                0x1000176  /* U+0176 LATIN CAPITAL LETTER Y WITH CIRCUMFLEX */
#define XK_ycircumflex                0x1000177  /* U+0177 LATIN SMALL LETTER Y WITH CIRCUMFLEX */
#define XK_Babovedot                  0x1001e02  /* U+1E02 LATIN CAPITAL LETTER B WITH DOT ABOVE */
#define XK_babovedot                  0x1001e03  /* U+1E03 LATIN SMALL LETTER B WITH DOT ABOVE */
#define XK_Dabovedot                  0x1001e0a  /* U+1E0A LATIN CAPITAL LETTER D WITH DOT ABOVE */
#define XK_dabovedot                  0x1001e0b  /* U+1E0B LATIN SMALL LETTER D WITH DOT ABOVE */
#define XK_Fabovedot                  0x1001e1e  /* U+1E1E LATIN CAPITAL LETTER F WITH DOT ABOVE */
#define XK_fabovedot                  0x1001e1f  /* U+1E1F LATIN SMALL LETTER F WITH DOT ABOVE */
#define XK_Mabovedot                  0x1001e40  /* U+1E40 LATIN CAPITAL LETTER M WITH DOT ABOVE */
#define XK_mabovedot                  0x1001e41  /* U+1E41 LATIN SMALL LETTER M WITH DOT ABOVE */
#define XK_Pabovedot                  0x1001e56  /* U+1E56 LATIN CAPITAL LETTER P WITH DOT ABOVE */
#define XK_pabovedot                  0x1001e57  /* U+1E57 LATIN SMALL LETTER P WITH DOT ABOVE */
#define XK_Sabovedot                  0x1001e60  /* U+1E60 LATIN CAPITAL LETTER S WITH DOT ABOVE */
#define XK_sabovedot                  0x1001e61  /* U+1E61 LATIN SMALL LETTER S WITH DOT ABOVE */
#define XK_Tabovedot                  0x1001e6a  /* U+1E6A LATIN CAPITAL LETTER T WITH DOT ABOVE */
#define XK_tabovedot                  0x1001e6b  /* U+1E6B LATIN SMALL LETTER T WITH DOT ABOVE */
#define XK_Wgrave                     0x1001e80  /* U+1E80 LATIN CAPITAL LETTER W WITH GRAVE */
#define XK_wgrave                     0x1001e81  /* U+1E81 LATIN SMALL LETTER W WITH GRAVE */
#define XK_Wacute                     0x1001e82  /* U+1E82 LATIN CAPITAL LETTER W WITH ACUTE */
#define XK_wacute                     0x1001e83  /* U+1E83 LATIN SMALL LETTER W WITH ACUTE */
#define XK_Wdiaeresis                 0x1001e84  /* U+1E84 LATIN CAPITAL LETTER W WITH DIAERESIS */
#define XK_wdiaeresis                 0x1001e85  /* U+1E85 LATIN SMALL LETTER W WITH DIAERESIS */
#define XK_Ygrave                     0x1001ef2  /* U+1EF2 LATIN CAPITAL LETTER Y WITH GRAVE */
#define XK_ygrave                     0x1001ef3  /* U+1EF3 LATIN SMALL LETTER Y WITH GRAVE */
#endif /* XK_LATIN8 */

/*
 * Latin 9
 * Byte 3 = 0x13
 */

#ifdef XK_LATIN9
#define XK_OE                            0x13bc  /* U+0152 LATIN CAPITAL LIGATURE OE */
#define XK_oe                            0x13bd  /* U+0153 LATIN SMALL LIGATURE OE */
#define XK_Ydiaeresis                    0x13be  /* U+0178 LATIN CAPITAL LETTER Y WITH DIAERESIS */
#endif /* XK_LATIN9 */

/*
 * Katakana
 * Byte 3 = 4
 */

#ifdef XK_KATAKANA
#define XK_overline                      0x047e  /* U+203E OVERLINE */
#define XK_kana_fullstop                 0x04a1  /* U+3002 IDEOGRAPHIC FULL STOP */
#define XK_kana_openingbracket           0x04a2  /* U+300C LEFT CORNER BRACKET */
#define XK_kana_closingbracket           0x04a3  /* U+300D RIGHT CORNER BRACKET */
#define XK_kana_comma                    0x04a4  /* U+3001 IDEOGRAPHIC COMMA */
#define XK_kana_conjunctive              0x04a5  /* U+30FB KATAKANA MIDDLE DOT */
#define XK_kana_middledot                0x04a5  /* deprecated */
#define XK_kana_WO                       0x04a6  /* U+30F2 KATAKANA LETTER WO */
#define XK_kana_a                        0x04a7  /* U+30A1 KATAKANA LETTER SMALL A */
#define XK_kana_i                        0x04a8  /* U+30A3 KATAKANA LETTER SMALL I */
#define XK_kana_u                        0x04a9  /* U+30A5 KATAKANA LETTER SMALL U */
#define XK_kana_e                        0x04aa  /* U+30A7 KATAKANA LETTER SMALL E */
#define XK_kana_o                        0x04ab  /* U+30A9 KATAKANA LETTER SMALL O */
#define XK_kana_ya                       0x04ac  /* U+30E3 KATAKANA LETTER SMALL YA */
#define XK_kana_yu                       0x04ad  /* U+30E5 KATAKANA LETTER SMALL YU */
#define XK_kana_yo                       0x04ae  /* U+30E7 KATAKANA LETTER SMALL YO */
#define XK_kana_tsu                      0x04af  /* U+30C3 KATAKANA LETTER SMALL TU */
#define XK_kana_tu                       0x04af  /* deprecated */
#define XK_prolongedsound                0x04b0  /* U+30FC KATAKANA-HIRAGANA PROLONGED SOUND MARK */
#define XK_kana_A                        0x04b1  /* U+30A2 KATAKANA LETTER A */
#define XK_kana_I                        0x04b2  /* U+30A4 KATAKANA LETTER I */
#define XK_kana_U                        0x04b3  /* U+30A6 KATAKANA LETTER U */
#define XK_kana_E                        0x04b4  /* U+30A8 KATAKANA LETTER E */
#define XK_kana_O                        0x04b5  /* U+30AA KATAKANA LETTER O */
#define XK_kana_KA                       0x04b6  /* U+30AB KATAKANA LETTER KA */
#define XK_kana_KI                       0x04b7  /* U+30AD KATAKANA LETTER KI */
#define XK_kana_KU                       0x04b8  /* U+30AF KATAKANA LETTER KU */
#define XK_kana_KE                       0x04b9  /* U+30B1 KATAKANA LETTER KE */
#define XK_kana_KO                       0x04ba  /* U+30B3 KATAKANA LETTER KO */
#define XK_kana_SA                       0x04bb  /* U+30B5 KATAKANA LETTER SA */
#define XK_kana_SHI                      0x04bc  /* U+30B7 KATAKANA LETTER SI */
#define XK_kana_SU                       0x04bd  /* U+30B9 KATAKANA LETTER SU */
#define XK_kana_SE                       0x04be  /* U+30BB KATAKANA LETTER SE */
#define XK_kana_SO                       0x04bf  /* U+30BD KATAKANA LETTER SO */
#define XK_kana_TA                       0x04c0  /* U+30BF KATAKANA LETTER TA */
#define XK_kana_CHI                      0x04c1  /* U+30C1 KATAKANA LETTER TI */
#define XK_kana_TI                       0x04c1  /* deprecated */
#define XK_kana_TSU                      0x04c2  /* U+30C4 KATAKANA LETTER TU */
#define XK_kana_TU                       0x04c2  /* deprecated */
#define XK_kana_TE                       0x04c3  /* U+30C6 KATAKANA LETTER TE */
#define XK_kana_TO                       0x04c4  /* U+30C8 KATAKANA LETTER TO */
#define XK_kana_NA                       0x04c5  /* U+30CA KATAKANA LETTER NA */
#define XK_kana_NI                       0x04c6  /* U+30CB KATAKANA LETTER NI */
#define XK_kana_NU                       0x04c7  /* U+30CC KATAKANA LETTER NU */
#define XK_kana_NE                       0x04c8  /* U+30CD KATAKANA LETTER NE */
#define XK_kana_NO                       