# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['eltetrado']

package_data = \
{'': ['*']}

install_requires = \
['biopython>=1.78,<2.0', 'numpy>=1.20.3,<2.0.0']

entry_points = \
{'console_scripts': ['eltetrado = eltetrado.eltetrado:main',
                     'has_tetrad = eltetrado.has_tetrad:main']}

setup_kwargs = {
    'name': 'eltetrado',
    'version': '1.2.0.dev1',
    'description': 'A Python application to find and classify tetrads and quadruplexes in DNA/RNA 3D structure',
    'long_description': '![](logo.svg)\n\n# Project description\n\nThis is an application to analyze base pairing patterns of DNA/RNA 3D\nstructures to find and classify tetrads and quadruplexes. ElTetrado\nassigns tetrads to one of the ONZ classes (O, N, Z) alongside with the\ndirectionality of the tetrad (+/-) determined by the bonds between bases\nand their non-canonical interactions. The interactions follow\nLeontis/Westhof classification (Leontis and Westhof, 2001). Watson-Crick\n(W) edge of first base in the tetrad structure exposed to the Hoogsteen\n(H) edge of the next nucleobase from the same tetrad sets the tetrad\ndirectionality, clockwise (+) or anticlockwise (-). For more details,\nplease refer to Zok *et al.* (2020) and Popenda *et al.* (2020)\n\n# Dependencies\n\nThe project is written in Python 3.6+ and requires\n[Biopython](https://biopython.org/) and [NumPy](https://numpy.org/) in\nto run. These can be installed with the following command:\n\n    pip install -r requirements.txt\n\nIf you have both Python 2 and Python 3 installed, you need to explicitly\ncall `pip3`:\n\n    pip3 install -r requirements.txt\n\nElTetrado depends on DSSR (Lu *et al.*, 2015) in terms of detection of\nbase pairing and stacking. The binary `x3dna-dssr` can be\n[downloaded](http://forum.x3dna.org/site-announcements/download-instructions/)\nand put in the same directory as `eltetrado` which will execute it\nduring analysis. Alternatively, one can pre-process the 3D data with\n`x3dna-dssr --json` and provide the JSON result as an input to ElTetrado\n(see Usage section below).\n\nVisualization is created by `R` 3.6+ script which uses\n[R4RNA](https://www.e-rna.org/r-chie/) (Lai *et al.*, 2012) library. The\ndependency will be automatically installed if not present.\n\n# Usage\n\nElTetrado is a command line application, which requires to be provided\nwith:\n\n-   either `--dssr-json` and the path to JSON generated with\n    `x3dna-dssr --json` (fast, but quadruplex parameters like `rise` or\n    `twist` are not calculated)\n-   or `--pdb` and the path to PDB or PDBx/mmCIF file (slow, because the\n    execution time is a sum of ElTetrado and DSSR times)\n-   or both `--pdb` and `--dssr-json` (recommended, all analyses are\n    made and only ElTetrado is executed while DSSR results are read from\n    the file)\n\nBy default, ElTetrado outputs textual results on the standard output. A\nJSON version of the output can be obtained with `--output` switch\nfollowed by a path where the file is supposed to be created.\n\nElTetrado prepares visualization of the whole structure and of each\nN4-helices, quadruplexes and tetrads. This can be supplemented with\ncanonical base pairs visualization when `--complete-2d` is set. All\ncolor settings are located in the first several lines of the `quadraw.R`\nfile, you can easily change them without knowledge of R language. If you\nwant ElTetrado to not visualize anything, pass `--no-image` switch to\nit.\n\n    usage: eltetrado [-h] [--pdb PDB] [--dssr-json DSSR_JSON] [--output OUTPUT]\n                     [--stacking-mismatch STACKING_MISMATCH] [--strict]\n                     [--no-reorder] [--complete-2d] [--no-image] [--version]\n\n    optional arguments:\n      -h, --help            show this help message and exit\n      --pdb PDB             path to input PDB or PDBx/mmCIF file\n      --dssr-json DSSR_JSON\n                            path to input JSON file generated with `x3dna-dssr\n                            --json`\n      --output OUTPUT       (optional) path for output JSON file\n      --stacking-mismatch STACKING_MISMATCH\n                            a perfect tetrad stacking covers 4 nucleotides; this\n                            option can be used with value 1 or 2 to allow this\n                            number of nucleotides to be non-stacked with otherwise\n                            well aligned tetrad [default=2]\n      --strict              nucleotides in tetrad are found when linked only by\n                            cWH pairing\n      --no-reorder          chains of bi- and tetramolecular quadruplexes are\n                            reordered to be able to have them classified; when\n                            this is set, chains will be processed in original\n                            order and bi-/tetramolecular quadruplexes will not be\n                            classified\n      --complete-2d         when set, the visualization will also show canonical\n                            base pairs to provide context for the quadruplex\n      --no-image            when set, the visualization will not be created at all\n      --version             show program\'s version number and exit\n\n# Chains reorder\n\nElTetrado keeps a global and unique 5’-3’ index for every nucleotide\nwhich is independent from residue numbers. For example, if a structure\nhas chain M with 60 nucleotides and chain N with 15 nucleotides, then\nElTetrado will keep index between 0 and 74 which uniquely identifies\nevery nucleotide. Initially, ElTetrado assigns this indices according to\nthe order of chains in the input file. Therefore, if M preceded N then\nnucleotides in M will be indexed from 0 to 59 and in N from 60 to 74.\nOtherwise, nucleotides in N will be indexed from 0 to 14 and in M from\n15 to 74.\n\nWhen `--no-reorder` is present, this initial assignment is used.\nOtherwise, ElTetrado exhaustively checks all permutations of chains’\norders. Every permutation check induces recalculation of the global and\nunique 5’-3’ index and in effect it changes ONZ classification of\ntetrads.\n\nElTetrado keeps a table of tetrad classification scores according to\nthese rules:\n\n-   Type preference: `O` > `N` > `Z`\n-   Direction preference: `+` > `-`\n\nThe table keeps low values for preferred classes i.e.\xa0`O+` is 0, `O-` is\n1 and so on up to `Z-` with score 5. For every permutation of chain\norders, ElTetrado computes sum of scores for tetrads classification\ninduced by 5’-3’ indexing. We select permutation with the minimum value.\n\n# Examples\n\n## 1MY9: Solution structure of a K+ cation stabilized dimeric RNA quadruplex containing two G:G(:A):G:G(:A) hexads, G:G:G:G tetrads and UUUU loops\n\n![](1my9.png)\n\n    $ curl ftp://ftp.wwpdb.org/pub/pdb/data/structures/divided/mmCIF/my/1my9.cif.gz | gzip -d > 1my9.cif\n    $ ./eltetrado --pdb 1my9.cif --output 1my9.json\n\n    Chain order: A, B\n    n4-helix with 4 tetrads\n      Op+ VIII 1a quadruplex with 2 tetrads\n        B.G16 B.G19 B.G25 B.G28 cWH-cWH-cWH-cWH O+ VIIIa planarity=0.31 ions_channel= ions_outside={}\n          direction=parallel rise=4.21 twist=33.45\n        B.G15 B.G18 B.G24 B.G27 cWH-cWH-cWH-cWH O+ VIIIa planarity=0.26 ions_channel= ions_outside={}\n\n        Tracts:\n          B.G25, B.G24\n          B.G28, B.G27\n          B.G16, B.G15\n          B.G19, B.G18\n\n        Loops:\n          propeller- B.A17\n          propeller- B.U20, B.U21, B.U22, B.U23\n          propeller- B.A26\n\n      Op+ VIII 1a quadruplex with 2 tetrads\n        A.G1 A.G4 A.G10 A.G13 cWH-cWH-cWH-cWH O+ VIIIa planarity=0.63 ions_channel= ions_outside={}\n          direction=parallel rise=4.12 twist=28.66\n        A.G2 A.G5 A.G11 A.G14 cWH-cWH-cWH-cWH O+ VIIIa planarity=0.5 ions_channel= ions_outside={}\n\n        Tracts:\n          A.G10, A.G11\n          A.G13, A.G14\n          A.G1, A.G2\n          A.G4, A.G5\n\n        Loops:\n          propeller- A.A3\n          propeller- A.U6, A.U7, A.U8, A.U9\n          propeller- A.A12\n\n\n    GGAGGUUUUGGAGG-GGAGGUUUUGGAGG\n    ([.)]....([.)]-([.)]....([.)]\n    ([.([....)].)]-([.([....)].)]\n\n<details>\n<summary>\nClick to see the output JSON\n</summary>\n\n``` json\n{\n  "metals": "",\n  "nucleotides": {\n    "A.G1": {\n      "index": 1,\n      "model": 1,\n      "chain": "A",\n      "number": 1,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "A.G1",\n      "short_name": "G",\n      "chi": -114.063,\n      "glycosidic_bond": "anti"\n    },\n    "A.G2": {\n      "index": 2,\n      "model": 1,\n      "chain": "A",\n      "number": 2,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "A.G2",\n      "short_name": "G",\n      "chi": -136.905,\n      "glycosidic_bond": "anti"\n    },\n    "A.A3": {\n      "index": 3,\n      "model": 1,\n      "chain": "A",\n      "number": 3,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "A.A3",\n      "short_name": "A",\n      "chi": -53.884,\n      "glycosidic_bond": "syn"\n    },\n    "A.G4": {\n      "index": 4,\n      "model": 1,\n      "chain": "A",\n      "number": 4,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "A.G4",\n      "short_name": "G",\n      "chi": 167.763,\n      "glycosidic_bond": "anti"\n    },\n    "A.G5": {\n      "index": 5,\n      "model": 1,\n      "chain": "A",\n      "number": 5,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "A.G5",\n      "short_name": "G",\n      "chi": -98.198,\n      "glycosidic_bond": "anti"\n    },\n    "A.U6": {\n      "index": 6,\n      "model": 1,\n      "chain": "A",\n      "number": 6,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "A.U6",\n      "short_name": "U",\n      "chi": -150.069,\n      "glycosidic_bond": "anti"\n    },\n    "A.U7": {\n      "index": 7,\n      "model": 1,\n      "chain": "A",\n      "number": 7,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "A.U7",\n      "short_name": "U",\n      "chi": -130.523,\n      "glycosidic_bond": "anti"\n    },\n    "A.U8": {\n      "index": 8,\n      "model": 1,\n      "chain": "A",\n      "number": 8,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "A.U8",\n      "short_name": "U",\n      "chi": -158.504,\n      "glycosidic_bond": "anti"\n    },\n    "A.U9": {\n      "index": 9,\n      "model": 1,\n      "chain": "A",\n      "number": 9,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "A.U9",\n      "short_name": "U",\n      "chi": -149.743,\n      "glycosidic_bond": "anti"\n    },\n    "A.G10": {\n      "index": 10,\n      "model": 1,\n      "chain": "A",\n      "number": 10,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "A.G10",\n      "short_name": "G",\n      "chi": -113.245,\n      "glycosidic_bond": "anti"\n    },\n    "A.G11": {\n      "index": 11,\n      "model": 1,\n      "chain": "A",\n      "number": 11,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "A.G11",\n      "short_name": "G",\n      "chi": -138.466,\n      "glycosidic_bond": "anti"\n    },\n    "A.A12": {\n      "index": 12,\n      "model": 1,\n      "chain": "A",\n      "number": 12,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "A.A12",\n      "short_name": "A",\n      "chi": -70.627,\n      "glycosidic_bond": "syn"\n    },\n    "A.G13": {\n      "index": 13,\n      "model": 1,\n      "chain": "A",\n      "number": 13,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "A.G13",\n      "short_name": "G",\n      "chi": 150.585,\n      "glycosidic_bond": "anti"\n    },\n    "A.G14": {\n      "index": 14,\n      "model": 1,\n      "chain": "A",\n      "number": 14,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "A.G14",\n      "short_name": "G",\n      "chi": -158.594,\n      "glycosidic_bond": "anti"\n    },\n    "B.G15": {\n      "index": 15,\n      "model": 1,\n      "chain": "B",\n      "number": 15,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "B.G15",\n      "short_name": "G",\n      "chi": -114.303,\n      "glycosidic_bond": "anti"\n    },\n    "B.G16": {\n      "index": 16,\n      "model": 1,\n      "chain": "B",\n      "number": 16,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "B.G16",\n      "short_name": "G",\n      "chi": -134.388,\n      "glycosidic_bond": "anti"\n    },\n    "B.A17": {\n      "index": 17,\n      "model": 1,\n      "chain": "B",\n      "number": 17,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "B.A17",\n      "short_name": "A",\n      "chi": -52.702,\n      "glycosidic_bond": "syn"\n    },\n    "B.G18": {\n      "index": 18,\n      "model": 1,\n      "chain": "B",\n      "number": 18,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "B.G18",\n      "short_name": "G",\n      "chi": 156.482,\n      "glycosidic_bond": "anti"\n    },\n    "B.G19": {\n      "index": 19,\n      "model": 1,\n      "chain": "B",\n      "number": 19,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "B.G19",\n      "short_name": "G",\n      "chi": -100.234,\n      "glycosidic_bond": "anti"\n    },\n    "B.U20": {\n      "index": 20,\n      "model": 1,\n      "chain": "B",\n      "number": 20,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "B.U20",\n      "short_name": "U",\n      "chi": -146.822,\n      "glycosidic_bond": "anti"\n    },\n    "B.U21": {\n      "index": 21,\n      "model": 1,\n      "chain": "B",\n      "number": 21,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "B.U21",\n      "short_name": "U",\n      "chi": -144.09,\n      "glycosidic_bond": "anti"\n    },\n    "B.U22": {\n      "index": 22,\n      "model": 1,\n      "chain": "B",\n      "number": 22,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "B.U22",\n      "short_name": "U",\n      "chi": -160.945,\n      "glycosidic_bond": "anti"\n    },\n    "B.U23": {\n      "index": 23,\n      "model": 1,\n      "chain": "B",\n      "number": 23,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "B.U23",\n      "short_name": "U",\n      "chi": -144.171,\n      "glycosidic_bond": "anti"\n    },\n    "B.G24": {\n      "index": 24,\n      "model": 1,\n      "chain": "B",\n      "number": 24,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "B.G24",\n      "short_name": "G",\n      "chi": -121.507,\n      "glycosidic_bond": "anti"\n    },\n    "B.G25": {\n      "index": 25,\n      "model": 1,\n      "chain": "B",\n      "number": 25,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "B.G25",\n      "short_name": "G",\n      "chi": -132.739,\n      "glycosidic_bond": "anti"\n    },\n    "B.A26": {\n      "index": 26,\n      "model": 1,\n      "chain": "B",\n      "number": 26,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "B.A26",\n      "short_name": "A",\n      "chi": -72.432,\n      "glycosidic_bond": "syn"\n    },\n    "B.G27": {\n      "index": 27,\n      "model": 1,\n      "chain": "B",\n      "number": 27,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "B.G27",\n      "short_name": "G",\n      "chi": -178.717,\n      "glycosidic_bond": "anti"\n    },\n    "B.G28": {\n      "index": 28,\n      "model": 1,\n      "chain": "B",\n      "number": 28,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "B.G28",\n      "short_name": "G",\n      "chi": -130.902,\n      "glycosidic_bond": "anti"\n    }\n  },\n  "base_pairs": [\n    {\n      "nt1": "A.G1",\n      "nt2": "A.A3",\n      "stericity": "trans",\n      "edge5": "Sugar",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "A.A3",\n      "nt2": "A.G1",\n      "stericity": "trans",\n      "edge5": "Hoogsteen",\n      "edge3": "Sugar"\n    },\n    {\n      "nt1": "A.G1",\n      "nt2": "A.G4",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "A.G4",\n      "nt2": "A.G1",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "A.G1",\n      "nt2": "A.G13",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "A.G13",\n      "nt2": "A.G1",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "A.G2",\n      "nt2": "A.G5",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "A.G5",\n      "nt2": "A.G2",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "A.G2",\n      "nt2": "A.G11",\n      "stericity": "trans",\n      "edge5": "Watson-Crick",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "A.G11",\n      "nt2": "A.G2",\n      "stericity": "trans",\n      "edge5": "Watson-Crick",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "A.G2",\n      "nt2": "A.G14",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "A.G14",\n      "nt2": "A.G2",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "A.G4",\n      "nt2": "A.G10",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "A.G10",\n      "nt2": "A.G4",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "A.G4",\n      "nt2": "B.A17",\n      "stericity": "trans",\n      "edge5": "Hoogsteen",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "B.A17",\n      "nt2": "A.G4",\n      "stericity": "trans",\n      "edge5": "Hoogsteen",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "A.G5",\n      "nt2": "A.G11",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "A.G11",\n      "nt2": "A.G5",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "A.G10",\n      "nt2": "A.A12",\n      "stericity": "trans",\n      "edge5": "Sugar",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "A.A12",\n      "nt2": "A.G10",\n      "stericity": "trans",\n      "edge5": "Hoogsteen",\n      "edge3": "Sugar"\n    },\n    {\n      "nt1": "A.G10",\n      "nt2": "A.G13",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "A.G13",\n      "nt2": "A.G10",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "A.G11",\n      "nt2": "A.G14",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "A.G14",\n      "nt2": "A.G11",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "A.G13",\n      "nt2": "B.G27",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "B.G27",\n      "nt2": "A.G13",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "B.G15",\n      "nt2": "B.A17",\n      "stericity": "trans",\n      "edge5": "Sugar",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "B.A17",\n      "nt2": "B.G15",\n      "stericity": "trans",\n      "edge5": "Hoogsteen",\n      "edge3": "Sugar"\n    },\n    {\n      "nt1": "B.G15",\n      "nt2": "B.G18",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "B.G18",\n      "nt2": "B.G15",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "B.G15",\n      "nt2": "B.G27",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "B.G27",\n      "nt2": "B.G15",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "B.G16",\n      "nt2": "B.G19",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "B.G19",\n      "nt2": "B.G16",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "B.G16",\n      "nt2": "B.G28",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "B.G28",\n      "nt2": "B.G16",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "B.G18",\n      "nt2": "B.G24",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "B.G24",\n      "nt2": "B.G18",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "B.G19",\n      "nt2": "B.G25",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "B.G25",\n      "nt2": "B.G19",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "B.G24",\n      "nt2": "B.A26",\n      "stericity": "trans",\n      "edge5": "Sugar",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "B.A26",\n      "nt2": "B.G24",\n      "stericity": "trans",\n      "edge5": "Hoogsteen",\n      "edge3": "Sugar"\n    },\n    {\n      "nt1": "B.G24",\n      "nt2": "B.G27",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "B.G27",\n      "nt2": "B.G24",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "B.G25",\n      "nt2": "B.G28",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "B.G28",\n      "nt2": "B.G25",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    }\n  ],\n  "helices": [\n    {\n      "quadruplexes": [\n        {\n          "tetrads": {\n            "B.G16-B.G19-B.G25-B.G28": {\n              "nt1": "B.G16",\n              "nt2": "B.G19",\n              "nt3": "B.G25",\n              "nt4": "B.G28",\n              "onz": "O+",\n              "gba_classification": "VIIIa",\n              "planarity_deviation": 0.31002684978384765,\n              "ions_channel": [],\n              "ions_outside": {}\n            },\n            "B.G15-B.G18-B.G24-B.G27": {\n              "nt1": "B.G15",\n              "nt2": "B.G18",\n              "nt3": "B.G24",\n              "nt4": "B.G27",\n              "onz": "O+",\n              "gba_classification": "VIIIa",\n              "planarity_deviation": 0.2616979115189099,\n              "ions_channel": [],\n              "ions_outside": {}\n            }\n          },\n          "onzm": "Op+",\n          "loop_classification": "1a",\n          "gba_classification": "VIII",\n          "tracts": [\n            [\n              "B.G25",\n              "B.G24"\n            ],\n            [\n              "B.G28",\n              "B.G27"\n            ],\n            [\n              "B.G16",\n              "B.G15"\n            ],\n            [\n              "B.G19",\n              "B.G18"\n            ]\n          ],\n          "loops": [\n            {\n              "loop_type": "propeller-",\n              "nucleotides": [\n                "B.A17"\n              ]\n            },\n            {\n              "loop_type": "propeller-",\n              "nucleotides": [\n                "B.U20",\n                "B.U21",\n                "B.U22",\n                "B.U23"\n              ]\n            },\n            {\n              "loop_type": "propeller-",\n              "nucleotides": [\n                "B.A26"\n              ]\n            }\n          ]\n        },\n        {\n          "tetrads": {\n            "A.G1-A.G4-A.G10-A.G13": {\n              "nt1": "A.G1",\n              "nt2": "A.G4",\n              "nt3": "A.G10",\n              "nt4": "A.G13",\n              "onz": "O+",\n              "gba_classification": "VIIIa",\n              "planarity_deviation": 0.6331163330540677,\n              "ions_channel": [],\n              "ions_outside": {}\n            },\n            "A.G2-A.G5-A.G11-A.G14": {\n              "nt1": "A.G2",\n              "nt2": "A.G5",\n              "nt3": "A.G11",\n              "nt4": "A.G14",\n              "onz": "O+",\n              "gba_classification": "VIIIa",\n              "planarity_deviation": 0.49745503003636027,\n              "ions_channel": [],\n              "ions_outside": {}\n            }\n          },\n          "onzm": "Op+",\n          "loop_classification": "1a",\n          "gba_classification": "VIII",\n          "tracts": [\n            [\n              "A.G10",\n              "A.G11"\n            ],\n            [\n              "A.G13",\n              "A.G14"\n            ],\n            [\n              "A.G1",\n              "A.G2"\n            ],\n            [\n              "A.G4",\n              "A.G5"\n            ]\n          ],\n          "loops": [\n            {\n              "loop_type": "propeller-",\n              "nucleotides": [\n                "A.A3"\n              ]\n            },\n            {\n              "loop_type": "propeller-",\n              "nucleotides": [\n                "A.U6",\n                "A.U7",\n                "A.U8",\n                "A.U9"\n              ]\n            },\n            {\n              "loop_type": "propeller-",\n              "nucleotides": [\n                "A.A12"\n              ]\n            }\n          ]\n        }\n      ],\n      "tetrad_pairs": [\n        {\n          "tetrad1": "B.G16-B.G19-B.G25-B.G28",\n          "tetrad2": "B.G15-B.G18-B.G24-B.G27",\n          "direction": "parallel",\n          "rise": 4.211160689840694,\n          "twist": 33.44705121023191\n        },\n        {\n          "tetrad1": "B.G15-B.G18-B.G24-B.G27",\n          "tetrad2": "A.G1-A.G4-A.G10-A.G13",\n          "direction": "parallel",\n          "rise": 3.184510351147094,\n          "twist": 2.4776498925690027\n        },\n        {\n          "tetrad1": "A.G1-A.G4-A.G10-A.G13",\n          "tetrad2": "A.G2-A.G5-A.G11-A.G14",\n          "direction": "parallel",\n          "rise": 4.121721871766505,\n          "twist": 28.658873381786407\n        }\n      ]\n    }\n  ]\n}\n```\n\n</details>\n\n## 4RJ1: Structural variations and solvent structure of UGGGGU quadruplexes stabilized by Sr2+ ions\n\n![](4rj1.png)\n\n    $ curl https://www.ebi.ac.uk/pdbe/static/entry/download/4rj1-assembly-1.cif.gz | gzip -d > 4rj1-1.cif\n    $ ./eltetrado --pdb 4rj1-1.cif --output 4rj1-1.json\n\n    Chain order: A, AB, AA, AC, B, BC, BA, BB\n    n4-helix with 10 tetrads\n      Op* VIII n/a quadruplex with 5 tetrads\n        B.U2006 BB.U2006 BA.U2006 BC.U2006 cWH-cWH-cWH-cWH O- VIIIa planarity=1.58 ions_channel=NA,NA ions_outside={}\n          direction=parallel rise=7.14 twist=43.41\n        B.G2005 BC.G2005 BA.G2005 BB.G2005 cWH-cWH-cWH-cWH O+ VIIIa planarity=0.78 ions_channel= ions_outside={}\n          direction=parallel rise=3.27 twist=25.15\n        B.G2004 BC.G2004 BA.G2004 BB.G2004 cWH-cWH-cWH-cWH O+ VIIIa planarity=0.23 ions_channel=SR ions_outside={}\n          direction=parallel rise=3.32 twist=35.04\n        B.G2003 BC.G2003 BA.G2003 BB.G2003 cWH-cWH-cWH-cWH O+ VIIIa planarity=0.58 ions_channel=SR ions_outside={}\n          direction=parallel rise=3.37 twist=27.41\n        B.G2002 BC.G2002 BA.G2002 BB.G2002 cWH-cWH-cWH-cWH O+ VIIIa planarity=0.67 ions_channel= ions_outside={}\n\n        Tracts:\n          BA.U2006, BA.G2005, BA.G2004, BA.G2003, BA.G2002\n          BC.U2006, BC.G2005, BC.G2004, BC.G2003, BC.G2002\n          B.U2006, B.G2005, B.G2004, B.G2003, B.G2002\n          BB.U2006, BB.G2005, BB.G2004, BB.G2003, BB.G2002\n\n      Op* VIII n/a quadruplex with 5 tetrads\n        A.G1002 AB.G1002 AA.G1002 AC.G1002 cWH-cWH-cWH-cWH O+ VIIIa planarity=0.54 ions_channel= ions_outside={}\n          direction=parallel rise=3.29 twist=27.12\n        A.G1003 AB.G1003 AA.G1003 AC.G1003 cWH-cWH-cWH-cWH O+ VIIIa planarity=0.56 ions_channel=SR ions_outside={}\n          direction=parallel rise=3.34 twist=35.81\n        A.G1004 AB.G1004 AA.G1004 AC.G1004 cWH-cWH-cWH-cWH O+ VIIIa planarity=0.41 ions_channel= ions_outside={}\n          direction=parallel rise=3.31 twist=25.9\n        A.G1005 AB.G1005 AA.G1005 AC.G1005 cWH-cWH-cWH-cWH O+ VIIIa planarity=0.8 ions_channel=SR ions_outside={}\n          direction=parallel rise=3.37 twist=39.96\n        A.U1006 AC.U1006 AA.U1006 AB.U1006 cWH-cWH-cWH-cWH O- VIIIa planarity=1.06 ions_channel= ions_outside={A.U1006: \'SR\', AA.U1006: \'SR\', AB.U1006: \'SR\', AC.U1006: \'SR\'}\n\n        Tracts:\n          AA.G1002, AA.G1003, AA.G1004, AA.G1005, AA.U1006\n          AC.G1002, AC.G1003, AC.G1004, AC.G1005, AC.U1006\n          A.G1002, A.G1003, A.G1004, A.G1005, A.U1006\n          AB.G1002, AB.G1003, AB.G1004, AB.G1005, AB.U1006\n\n\n    UGGGGU-UGGGGU-UGGGGU-UGGGGU-UGGGGU-UGGGGU-UGGGGU-UGGGGU\n    .([{<A-.)]}>A-.([{<a-.)]}>a-.([{<A-.)]}>A-.([{<a-.)]}>a\n    .([{<A-.([{<a-.)]}>A-.)]}>a-.([{<A-.([{<a-.)]}>A-.)]}>a\n\n<details>\n<summary>\nClick to see the output JSON\n</summary>\n\n``` json\n{\n  "metals": "Ca=12,Na=2,Sr=8",\n  "nucleotides": {\n    "A.U1001": {\n      "index": 1,\n      "model": 1,\n      "chain": "A",\n      "number": 1001,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "A.U1001",\n      "short_name": "U",\n      "chi": -141.927,\n      "glycosidic_bond": "anti"\n    },\n    "A.G1002": {\n      "index": 2,\n      "model": 1,\n      "chain": "A",\n      "number": 1002,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "A.G1002",\n      "short_name": "G",\n      "chi": -165.93,\n      "glycosidic_bond": "anti"\n    },\n    "A.G1003": {\n      "index": 3,\n      "model": 1,\n      "chain": "A",\n      "number": 1003,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "A.G1003",\n      "short_name": "G",\n      "chi": -121.565,\n      "glycosidic_bond": "anti"\n    },\n    "A.G1004": {\n      "index": 4,\n      "model": 1,\n      "chain": "A",\n      "number": 1004,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "A.G1004",\n      "short_name": "G",\n      "chi": -156.01,\n      "glycosidic_bond": "anti"\n    },\n    "A.G1005": {\n      "index": 5,\n      "model": 1,\n      "chain": "A",\n      "number": 1005,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "A.G1005",\n      "short_name": "G",\n      "chi": -148.101,\n      "glycosidic_bond": "anti"\n    },\n    "A.U1006": {\n      "index": 6,\n      "model": 1,\n      "chain": "A",\n      "number": 1006,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "A.U1006",\n      "short_name": "U",\n      "chi": -137.28,\n      "glycosidic_bond": "anti"\n    },\n    "AA.U1001": {\n      "index": 13,\n      "model": 1,\n      "chain": "AA",\n      "number": 1001,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "AA.U1001",\n      "short_name": "U",\n      "chi": -141.927,\n      "glycosidic_bond": "anti"\n    },\n    "AA.G1002": {\n      "index": 14,\n      "model": 1,\n      "chain": "AA",\n      "number": 1002,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "AA.G1002",\n      "short_name": "G",\n      "chi": -165.93,\n      "glycosidic_bond": "anti"\n    },\n    "AA.G1003": {\n      "index": 15,\n      "model": 1,\n      "chain": "AA",\n      "number": 1003,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "AA.G1003",\n      "short_name": "G",\n      "chi": -121.565,\n      "glycosidic_bond": "anti"\n    },\n    "AA.G1004": {\n      "index": 16,\n      "model": 1,\n      "chain": "AA",\n      "number": 1004,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "AA.G1004",\n      "short_name": "G",\n      "chi": -156.01,\n      "glycosidic_bond": "anti"\n    },\n    "AA.G1005": {\n      "index": 17,\n      "model": 1,\n      "chain": "AA",\n      "number": 1005,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "AA.G1005",\n      "short_name": "G",\n      "chi": -148.101,\n      "glycosidic_bond": "anti"\n    },\n    "AA.U1006": {\n      "index": 18,\n      "model": 1,\n      "chain": "AA",\n      "number": 1006,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "AA.U1006",\n      "short_name": "U",\n      "chi": -137.28,\n      "glycosidic_bond": "anti"\n    },\n    "AB.U1001": {\n      "index": 7,\n      "model": 1,\n      "chain": "AB",\n      "number": 1001,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "AB.U1001",\n      "short_name": "U",\n      "chi": -141.927,\n      "glycosidic_bond": "anti"\n    },\n    "AB.G1002": {\n      "index": 8,\n      "model": 1,\n      "chain": "AB",\n      "number": 1002,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "AB.G1002",\n      "short_name": "G",\n      "chi": -165.93,\n      "glycosidic_bond": "anti"\n    },\n    "AB.G1003": {\n      "index": 9,\n      "model": 1,\n      "chain": "AB",\n      "number": 1003,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "AB.G1003",\n      "short_name": "G",\n      "chi": -121.565,\n      "glycosidic_bond": "anti"\n    },\n    "AB.G1004": {\n      "index": 10,\n      "model": 1,\n      "chain": "AB",\n      "number": 1004,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "AB.G1004",\n      "short_name": "G",\n      "chi": -156.01,\n      "glycosidic_bond": "anti"\n    },\n    "AB.G1005": {\n      "index": 11,\n      "model": 1,\n      "chain": "AB",\n      "number": 1005,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "AB.G1005",\n      "short_name": "G",\n      "chi": -148.101,\n      "glycosidic_bond": "anti"\n    },\n    "AB.U1006": {\n      "index": 12,\n      "model": 1,\n      "chain": "AB",\n      "number": 1006,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "AB.U1006",\n      "short_name": "U",\n      "chi": -137.28,\n      "glycosidic_bond": "anti"\n    },\n    "AC.U1001": {\n      "index": 19,\n      "model": 1,\n      "chain": "AC",\n      "number": 1001,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "AC.U1001",\n      "short_name": "U",\n      "chi": -141.927,\n      "glycosidic_bond": "anti"\n    },\n    "AC.G1002": {\n      "index": 20,\n      "model": 1,\n      "chain": "AC",\n      "number": 1002,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "AC.G1002",\n      "short_name": "G",\n      "chi": -165.93,\n      "glycosidic_bond": "anti"\n    },\n    "AC.G1003": {\n      "index": 21,\n      "model": 1,\n      "chain": "AC",\n      "number": 1003,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "AC.G1003",\n      "short_name": "G",\n      "chi": -121.565,\n      "glycosidic_bond": "anti"\n    },\n    "AC.G1004": {\n      "index": 22,\n      "model": 1,\n      "chain": "AC",\n      "number": 1004,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "AC.G1004",\n      "short_name": "G",\n      "chi": -156.01,\n      "glycosidic_bond": "anti"\n    },\n    "AC.G1005": {\n      "index": 23,\n      "model": 1,\n      "chain": "AC",\n      "number": 1005,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "AC.G1005",\n      "short_name": "G",\n      "chi": -148.101,\n      "glycosidic_bond": "anti"\n    },\n    "AC.U1006": {\n      "index": 24,\n      "model": 1,\n      "chain": "AC",\n      "number": 1006,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "AC.U1006",\n      "short_name": "U",\n      "chi": -137.28,\n      "glycosidic_bond": "anti"\n    },\n    "B.U2001": {\n      "index": 25,\n      "model": 1,\n      "chain": "B",\n      "number": 2001,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "B.U2001",\n      "short_name": "U",\n      "chi": -146.462,\n      "glycosidic_bond": "anti"\n    },\n    "B.G2002": {\n      "index": 26,\n      "model": 1,\n      "chain": "B",\n      "number": 2002,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "B.G2002",\n      "short_name": "G",\n      "chi": -170.797,\n      "glycosidic_bond": "anti"\n    },\n    "B.G2003": {\n      "index": 27,\n      "model": 1,\n      "chain": "B",\n      "number": 2003,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "B.G2003",\n      "short_name": "G",\n      "chi": -117.687,\n      "glycosidic_bond": "anti"\n    },\n    "B.G2004": {\n      "index": 28,\n      "model": 1,\n      "chain": "B",\n      "number": 2004,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "B.G2004",\n      "short_name": "G",\n      "chi": -153.886,\n      "glycosidic_bond": "anti"\n    },\n    "B.G2005": {\n      "index": 29,\n      "model": 1,\n      "chain": "B",\n      "number": 2005,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "B.G2005",\n      "short_name": "G",\n      "chi": -148.852,\n      "glycosidic_bond": "anti"\n    },\n    "B.U2006": {\n      "index": 30,\n      "model": 1,\n      "chain": "B",\n      "number": 2006,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "B.U2006",\n      "short_name": "U",\n      "chi": -159.437,\n      "glycosidic_bond": "anti"\n    },\n    "BA.U2001": {\n      "index": 37,\n      "model": 1,\n      "chain": "BA",\n      "number": 2001,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "BA.U2001",\n      "short_name": "U",\n      "chi": -146.462,\n      "glycosidic_bond": "anti"\n    },\n    "BA.G2002": {\n      "index": 38,\n      "model": 1,\n      "chain": "BA",\n      "number": 2002,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "BA.G2002",\n      "short_name": "G",\n      "chi": -170.797,\n      "glycosidic_bond": "anti"\n    },\n    "BA.G2003": {\n      "index": 39,\n      "model": 1,\n      "chain": "BA",\n      "number": 2003,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "BA.G2003",\n      "short_name": "G",\n      "chi": -117.687,\n      "glycosidic_bond": "anti"\n    },\n    "BA.G2004": {\n      "index": 40,\n      "model": 1,\n      "chain": "BA",\n      "number": 2004,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "BA.G2004",\n      "short_name": "G",\n      "chi": -153.886,\n      "glycosidic_bond": "anti"\n    },\n    "BA.G2005": {\n      "index": 41,\n      "model": 1,\n      "chain": "BA",\n      "number": 2005,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "BA.G2005",\n      "short_name": "G",\n      "chi": -148.852,\n      "glycosidic_bond": "anti"\n    },\n    "BA.U2006": {\n      "index": 42,\n      "model": 1,\n      "chain": "BA",\n      "number": 2006,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "BA.U2006",\n      "short_name": "U",\n      "chi": -159.437,\n      "glycosidic_bond": "anti"\n    },\n    "BB.U2001": {\n      "index": 43,\n      "model": 1,\n      "chain": "BB",\n      "number": 2001,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "BB.U2001",\n      "short_name": "U",\n      "chi": -146.462,\n      "glycosidic_bond": "anti"\n    },\n    "BB.G2002": {\n      "index": 44,\n      "model": 1,\n      "chain": "BB",\n      "number": 2002,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "BB.G2002",\n      "short_name": "G",\n      "chi": -170.797,\n      "glycosidic_bond": "anti"\n    },\n    "BB.G2003": {\n      "index": 45,\n      "model": 1,\n      "chain": "BB",\n      "number": 2003,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "BB.G2003",\n      "short_name": "G",\n      "chi": -117.687,\n      "glycosidic_bond": "anti"\n    },\n    "BB.G2004": {\n      "index": 46,\n      "model": 1,\n      "chain": "BB",\n      "number": 2004,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "BB.G2004",\n      "short_name": "G",\n      "chi": -153.886,\n      "glycosidic_bond": "anti"\n    },\n    "BB.G2005": {\n      "index": 47,\n      "model": 1,\n      "chain": "BB",\n      "number": 2005,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "BB.G2005",\n      "short_name": "G",\n      "chi": -148.852,\n      "glycosidic_bond": "anti"\n    },\n    "BB.U2006": {\n      "index": 48,\n      "model": 1,\n      "chain": "BB",\n      "number": 2006,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "BB.U2006",\n      "short_name": "U",\n      "chi": -159.437,\n      "glycosidic_bond": "anti"\n    },\n    "BC.U2001": {\n      "index": 31,\n      "model": 1,\n      "chain": "BC",\n      "number": 2001,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "BC.U2001",\n      "short_name": "U",\n      "chi": -146.462,\n      "glycosidic_bond": "anti"\n    },\n    "BC.G2002": {\n      "index": 32,\n      "model": 1,\n      "chain": "BC",\n      "number": 2002,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "BC.G2002",\n      "short_name": "G",\n      "chi": -170.797,\n      "glycosidic_bond": "anti"\n    },\n    "BC.G2003": {\n      "index": 33,\n      "model": 1,\n      "chain": "BC",\n      "number": 2003,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "BC.G2003",\n      "short_name": "G",\n      "chi": -117.687,\n      "glycosidic_bond": "anti"\n    },\n    "BC.G2004": {\n      "index": 34,\n      "model": 1,\n      "chain": "BC",\n      "number": 2004,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "BC.G2004",\n      "short_name": "G",\n      "chi": -153.886,\n      "glycosidic_bond": "anti"\n    },\n    "BC.G2005": {\n      "index": 35,\n      "model": 1,\n      "chain": "BC",\n      "number": 2005,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "BC.G2005",\n      "short_name": "G",\n      "chi": -148.852,\n      "glycosidic_bond": "anti"\n    },\n    "BC.U2006": {\n      "index": 36,\n      "model": 1,\n      "chain": "BC",\n      "number": 2006,\n      "icode": " ",\n      "molecule": "RNA",\n      "full_name": "BC.U2006",\n      "short_name": "U",\n      "chi": -159.437,\n      "glycosidic_bond": "anti"\n    }\n  },\n  "base_pairs": [\n    {\n      "nt1": "A.U1001",\n      "nt2": "B.G2003",\n      "stericity": "cis",\n      "edge5": "Sugar",\n      "edge3": "Sugar"\n    },\n    {\n      "nt1": "B.G2003",\n      "nt2": "A.U1001",\n      "stericity": "cis",\n      "edge5": "Sugar",\n      "edge3": "Sugar"\n    },\n    {\n      "nt1": "A.G1002",\n      "nt2": "AB.G1002",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "AB.G1002",\n      "nt2": "A.G1002",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "A.G1002",\n      "nt2": "AC.G1002",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "AC.G1002",\n      "nt2": "A.G1002",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "A.G1003",\n      "nt2": "AB.G1003",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "AB.G1003",\n      "nt2": "A.G1003",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "A.G1003",\n      "nt2": "AC.G1003",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "AC.G1003",\n      "nt2": "A.G1003",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "A.G1003",\n      "nt2": "B.U2001",\n      "stericity": "cis",\n      "edge5": "Sugar",\n      "edge3": "Sugar"\n    },\n    {\n      "nt1": "B.U2001",\n      "nt2": "A.G1003",\n      "stericity": "cis",\n      "edge5": "Sugar",\n      "edge3": "Sugar"\n    },\n    {\n      "nt1": "A.G1004",\n      "nt2": "AB.G1004",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "AB.G1004",\n      "nt2": "A.G1004",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "A.G1004",\n      "nt2": "AC.G1004",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "AC.G1004",\n      "nt2": "A.G1004",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "A.G1005",\n      "nt2": "AB.G1005",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "AB.G1005",\n      "nt2": "A.G1005",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "A.G1005",\n      "nt2": "AC.G1005",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "AC.G1005",\n      "nt2": "A.G1005",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "A.U1006",\n      "nt2": "AB.U1006",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "AB.U1006",\n      "nt2": "A.U1006",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "A.U1006",\n      "nt2": "AC.U1006",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "AC.U1006",\n      "nt2": "A.U1006",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "AA.U1001",\n      "nt2": "BA.G2003",\n      "stericity": "cis",\n      "edge5": "Sugar",\n      "edge3": "Sugar"\n    },\n    {\n      "nt1": "BA.G2003",\n      "nt2": "AA.U1001",\n      "stericity": "cis",\n      "edge5": "Sugar",\n      "edge3": "Sugar"\n    },\n    {\n      "nt1": "AA.G1002",\n      "nt2": "AB.G1002",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "AB.G1002",\n      "nt2": "AA.G1002",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "AA.G1002",\n      "nt2": "AC.G1002",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "AC.G1002",\n      "nt2": "AA.G1002",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "AA.G1003",\n      "nt2": "AB.G1003",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "AB.G1003",\n      "nt2": "AA.G1003",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "AA.G1003",\n      "nt2": "AC.G1003",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "AC.G1003",\n      "nt2": "AA.G1003",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "AA.G1003",\n      "nt2": "BA.U2001",\n      "stericity": "cis",\n      "edge5": "Sugar",\n      "edge3": "Sugar"\n    },\n    {\n      "nt1": "BA.U2001",\n      "nt2": "AA.G1003",\n      "stericity": "cis",\n      "edge5": "Sugar",\n      "edge3": "Sugar"\n    },\n    {\n      "nt1": "AA.G1004",\n      "nt2": "AB.G1004",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "AB.G1004",\n      "nt2": "AA.G1004",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "AA.G1004",\n      "nt2": "AC.G1004",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "AC.G1004",\n      "nt2": "AA.G1004",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "AA.G1005",\n      "nt2": "AB.G1005",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "AB.G1005",\n      "nt2": "AA.G1005",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "AA.G1005",\n      "nt2": "AC.G1005",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "AC.G1005",\n      "nt2": "AA.G1005",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "AA.U1006",\n      "nt2": "AB.U1006",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "AB.U1006",\n      "nt2": "AA.U1006",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "AA.U1006",\n      "nt2": "AC.U1006",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "AC.U1006",\n      "nt2": "AA.U1006",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "AB.U1001",\n      "nt2": "BB.G2003",\n      "stericity": "cis",\n      "edge5": "Sugar",\n      "edge3": "Sugar"\n    },\n    {\n      "nt1": "BB.G2003",\n      "nt2": "AB.U1001",\n      "stericity": "cis",\n      "edge5": "Sugar",\n      "edge3": "Sugar"\n    },\n    {\n      "nt1": "AB.G1003",\n      "nt2": "BB.U2001",\n      "stericity": "cis",\n      "edge5": "Sugar",\n      "edge3": "Sugar"\n    },\n    {\n      "nt1": "BB.U2001",\n      "nt2": "AB.G1003",\n      "stericity": "cis",\n      "edge5": "Sugar",\n      "edge3": "Sugar"\n    },\n    {\n      "nt1": "AC.U1001",\n      "nt2": "BC.G2003",\n      "stericity": "cis",\n      "edge5": "Sugar",\n      "edge3": "Sugar"\n    },\n    {\n      "nt1": "BC.G2003",\n      "nt2": "AC.U1001",\n      "stericity": "cis",\n      "edge5": "Sugar",\n      "edge3": "Sugar"\n    },\n    {\n      "nt1": "AC.G1003",\n      "nt2": "BC.U2001",\n      "stericity": "cis",\n      "edge5": "Sugar",\n      "edge3": "Sugar"\n    },\n    {\n      "nt1": "BC.U2001",\n      "nt2": "AC.G1003",\n      "stericity": "cis",\n      "edge5": "Sugar",\n      "edge3": "Sugar"\n    },\n    {\n      "nt1": "B.G2002",\n      "nt2": "BB.G2002",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "BB.G2002",\n      "nt2": "B.G2002",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "B.G2002",\n      "nt2": "BC.G2002",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "BC.G2002",\n      "nt2": "B.G2002",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "B.G2003",\n      "nt2": "BB.G2003",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "BB.G2003",\n      "nt2": "B.G2003",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "B.G2003",\n      "nt2": "BC.G2003",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "BC.G2003",\n      "nt2": "B.G2003",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "B.G2004",\n      "nt2": "BB.G2004",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "BB.G2004",\n      "nt2": "B.G2004",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "B.G2004",\n      "nt2": "BC.G2004",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "BC.G2004",\n      "nt2": "B.G2004",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "B.G2005",\n      "nt2": "BB.G2005",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "BB.G2005",\n      "nt2": "B.G2005",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "B.G2005",\n      "nt2": "BC.G2005",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "BC.G2005",\n      "nt2": "B.G2005",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "B.U2006",\n      "nt2": "BB.U2006",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "BB.U2006",\n      "nt2": "B.U2006",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "B.U2006",\n      "nt2": "BC.U2006",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "BC.U2006",\n      "nt2": "B.U2006",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "BA.G2002",\n      "nt2": "BB.G2002",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "BB.G2002",\n      "nt2": "BA.G2002",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "BA.G2002",\n      "nt2": "BC.G2002",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "BC.G2002",\n      "nt2": "BA.G2002",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "BA.G2003",\n      "nt2": "BB.G2003",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "BB.G2003",\n      "nt2": "BA.G2003",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "BA.G2003",\n      "nt2": "BC.G2003",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "BC.G2003",\n      "nt2": "BA.G2003",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "BA.G2004",\n      "nt2": "BB.G2004",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "BB.G2004",\n      "nt2": "BA.G2004",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "BA.G2004",\n      "nt2": "BC.G2004",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "BC.G2004",\n      "nt2": "BA.G2004",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "BA.G2005",\n      "nt2": "BB.G2005",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "BB.G2005",\n      "nt2": "BA.G2005",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "BA.G2005",\n      "nt2": "BC.G2005",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "BC.G2005",\n      "nt2": "BA.G2005",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "BA.U2006",\n      "nt2": "BB.U2006",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    },\n    {\n      "nt1": "BB.U2006",\n      "nt2": "BA.U2006",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "BA.U2006",\n      "nt2": "BC.U2006",\n      "stericity": "cis",\n      "edge5": "Watson-Crick",\n      "edge3": "Hoogsteen"\n    },\n    {\n      "nt1": "BC.U2006",\n      "nt2": "BA.U2006",\n      "stericity": "cis",\n      "edge5": "Hoogsteen",\n      "edge3": "Watson-Crick"\n    }\n  ],\n  "helices": [\n    {\n      "quadruplexes": [\n        {\n          "tetrads": {\n            "B.U2006-BB.U2006-BA.U2006-BC.U2006": {\n              "nt1": "B.U2006",\n              "nt2": "BB.U2006",\n              "nt3": "BA.U2006",\n              "nt4": "BC.U2006",\n              "onz": "O-",\n              "gba_classification": "VIIIa",\n              "planarity_deviation": 1.5839996337890805,\n              "ions_channel": [\n                "Na",\n                "Na"\n              ],\n              "ions_outside": {}\n            },\n            "B.G2005-BC.G2005-BA.G2005-BB.G2005": {\n              "nt1": "B.G2005",\n              "nt2": "BC.G2005",\n              "nt3": "BA.G2005",\n              "nt4": "BB.G2005",\n              "onz": "O+",\n              "gba_classification": "VIIIa",\n              "planarity_deviation": 0.7810001373291016,\n              "ions_channel": [],\n              "ions_outside": {}\n            },\n            "B.G2004-BC.G2004-BA.G2004-BB.G2004": {\n              "nt1": "B.G2004",\n              "nt2": "BC.G2004",\n              "nt3": "BA.G2004",\n              "nt4": "BB.G2004",\n              "onz": "O+",\n              "gba_classification": "VIIIa",\n              "planarity_deviation": 0.2290000915528585,\n              "ions_channel": [\n                "Sr"\n              ],\n              "ions_outside": {}\n            },\n            "B.G2003-BC.G2003-BA.G2003-BB.G2003": {\n              "nt1": "B.G2003",\n              "nt2": "BC.G2003",\n              "nt3": "BA.G2003",\n              "nt4": "BB.G2003",\n              "onz": "O+",\n              "gba_classification": "VIIIa",\n              "planarity_deviation": 0.5770015716552734,\n              "ions_channel": [\n                "Sr"\n              ],\n              "ions_outside": {}\n            },\n            "B.G2002-BC.G2002-BA.G2002-BB.G2002": {\n              "nt1": "B.G2002",\n              "nt2": "BC.G2002",\n              "nt3": "BA.G2002",\n              "nt4": "BB.G2002",\n              "onz": "O+",\n              "gba_classification": "VIIIa",\n              "planarity_deviation": 0.6730003356935284,\n              "ions_channel": [],\n              "ions_outside": {}\n            }\n          },\n          "onzm": "Op*",\n          "loop_classification": "n/a",\n          "gba_classification": "VIII",\n          "tracts": [\n            [\n              "BA.U2006",\n              "BA.G2005",\n              "BA.G2004",\n              "BA.G2003",\n              "BA.G2002"\n            ],\n            [\n              "BC.U2006",\n              "BC.G2005",\n              "BC.G2004",\n              "BC.G2003",\n              "BC.G2002"\n            ],\n            [\n              "B.U2006",\n              "B.G2005",\n              "B.G2004",\n              "B.G2003",\n              "B.G2002"\n            ],\n            [\n              "BB.U2006",\n              "BB.G2005",\n              "BB.G2004",\n              "BB.G2003",\n              "BB.G2002"\n            ]\n          ],\n          "loops": []\n        },\n        {\n          "tetrads": {\n            "A.G1002-AB.G1002-AA.G1002-AC.G1002": {\n              "nt1": "A.G1002",\n              "nt2": "AB.G1002",\n              "nt3": "AA.G1002",\n              "nt4": "AC.G1002",\n              "onz": "O+",\n              "gba_classification": "VIIIa",\n              "planarity_deviation": 0.5419998168945837,\n              "ions_channel": [],\n              "ions_outside": {}\n            },\n            "A.G1003-AB.G1003-AA.G1003-AC.G1003": {\n              "nt1": "A.G1003",\n              "nt2": "AB.G1003",\n              "nt3": "AA.G1003",\n              "nt4": "AC.G1003",\n              "onz": "O+",\n              "gba_classification": "VIIIa",\n              "planarity_deviation": 0.5550003051758324,\n              "ions_channel": [\n                "Sr"\n              ],\n              "ions_outside": {}\n            },\n            "A.G1004-AB.G1004-AA.G1004-AC.G1004": {\n              "nt1": "A.G1004",\n              "nt2": "AB.G1004",\n              "nt3": "AA.G1004",\n              "nt4": "AC.G1004",\n              "onz": "O+",\n              "gba_classification": "VIIIa",\n              "planarity_deviation": 0.4060020446780144,\n              "ions_channel": [],\n              "ions_outside": {}\n            },\n            "A.G1005-AB.G1005-AA.G1005-AC.G1005": {\n              "nt1": "A.G1005",\n              "nt2": "AB.G1005",\n              "nt3": "AA.G1005",\n              "nt4": "AC.G1005",\n              "onz": "O+",\n              "gba_classification": "VIIIa",\n              "planarity_deviation": 0.7999992370605469,\n              "ions_channel": [\n                "Sr"\n              ],\n              "ions_outside": {}\n            },\n            "A.U1006-AC.U1006-AA.U1006-AB.U1006": {\n              "nt1": "A.U1006",\n              "nt2": "AC.U1006",\n              "nt3": "AA.U1006",\n              "nt4": "AB.U1006",\n              "onz": "O-",\n              "gba_classification": "VIIIa",\n              "planarity_deviation": 1.0610008239746362,\n              "ions_channel": [],\n              "ions_outside": {\n                "A.U1006": [\n                  "Sr"\n                ],\n                "AA.U1006": [\n                  "Sr"\n                ],\n                "AB.U1006": [\n                  "Sr"\n                ],\n                "AC.U1006": [\n                  "Sr"\n                ]\n              }\n            }\n          },\n          "onzm": "Op*",\n          "loop_classification": "n/a",\n          "gba_classification": "VIII",\n          "tracts": [\n            [\n              "AA.G1002",\n              "AA.G1003",\n              "AA.G1004",\n              "AA.G1005",\n              "AA.U1006"\n            ],\n            [\n              "AC.G1002",\n              "AC.G1003",\n              "AC.G1004",\n              "AC.G1005",\n              "AC.U1006"\n            ],\n            [\n              "A.G1002",\n              "A.G1003",\n              "A.G1004",\n              "A.G1005",\n              "A.U1006"\n            ],\n            [\n              "AB.G1002",\n              "AB.G1003",\n              "AB.G1004",\n              "AB.G1005",\n              "AB.U1006"\n            ]\n          ],\n          "loops": []\n        }\n      ],\n      "tetrad_pairs": [\n        {\n          "tetrad1": "B.U2006-BB.U2006-BA.U2006-BC.U2006",\n          "tetrad2": "B.G2005-BC.G2005-BA.G2005-BB.G2005",\n          "direction": "parallel",\n          "rise": 7.140500068664552,\n          "twist": 43.40609929753353\n        },\n        {\n          "tetrad1": "B.G2005-BC.G2005-BA.G2005-BB.G2005",\n          "tetrad2": "B.G2004-BC.G2004-BA.G2004-BB.G2004",\n          "direction": "parallel",\n          "rise": 3.269000053405764,\n          "twist": 25.149991205206657\n        },\n        {\n          "tetrad1": "B.G2004-BC.G2004-BA.G2004-BB.G2004",\n          "tetrad2": "B.G2003-BC.G2003-BA.G2003-BB.G2003",\n          "direction": "parallel",\n          "rise": 3.317999839782717,\n          "twist": 35.04072151642772\n        },\n        {\n          "tetrad1": "B.G2003-BC.G2003-BA.G2003-BB.G2003",\n          "tetrad2": "B.G2002-BC.G2002-BA.G2002-BB.G2002",\n          "direction": "parallel",\n          "rise": 3.370999336242684,\n          "twist": 27.410094298550373\n        },\n        {\n          "tetrad1": "B.G2002-BC.G2002-BA.G2002-BB.G2002",\n          "tetrad2": "A.G1002-AB.G1002-AA.G1002-AC.G1002",\n          "direction": "parallel",\n          "rise": 3.3695011138916207,\n          "twist": 28.9931716392613\n        },\n        {\n          "tetrad1": "A.G1002-AB.G1002-AA.G1002-AC.G1002",\n          "tetrad2": "A.G1003-AB.G1003-AA.G1003-AC.G1003",\n          "direction": "parallel",\n          "rise": 3.2864990234375084,\n          "twist": 27.115165766478476\n        },\n        {\n          "tetrad1": "A.G1003-AB.G1003-AA.G1003-AC.G1003",\n          "tetrad2": "A.G1004-AB.G1004-AA.G1004-AC.G1004",\n          "direction": "parallel",\n          "rise": 3.339500427246096,\n          "twist": 35.811154582473115\n        },\n        {\n          "tetrad1": "A.G1004-AB.G1004-AA.G1004-AC.G1004",\n          "tetrad2": "A.G1005-AB.G1005-AA.G1005-AC.G1005",\n          "direction": "parallel",\n          "rise": 3.307998657226571,\n          "twist": 25.896146524668296\n        },\n        {\n          "tetrad1": "A.G1005-AB.G1005-AA.G1005-AC.G1005",\n          "tetrad2": "A.U1006-AC.U1006-AA.U1006-AB.U1006",\n          "direction": "parallel",\n          "rise": 3.3665008544921897,\n          "twist": 39.962534923679286\n        }\n      ]\n    }\n  ]\n}\n```\n\n</details>\n\n# Funding\n\nThis research was supported by the National Science Centre, Poland\n\\[2016/23/B/ST6/03931, 2019/35/B/ST6/03074\\] and Mloda Kadra project\n\\[09/91/SBAD/0684\\] from Poznan University of Technology, and carried\nout in the European Centre for Bioinformatics and Genomics (Poland). The\nauthors also acknowledge partial support by the statutory funds of\nPoznan University of Technology, Polish Ministry of Science and Higher\nEducation, and the Institute of Bioorganic Chemistry, PAS within\nintramural financing program.\n\n# Bibliography\n\n<div id="refs" class="references csl-bib-body">\n\n1.  Topology-based classification of tetrads and quadruplex\n    structures. M. Popenda, J. Miskiewicz, J. Sarzynska, T. Zok, M.\n    Szachniuk. *Bioinformatics*. 2020. 36(4):1129–1134.\n    doi:[10.1093/bioinformatics/btz738](https://doi.org/10.1093/bioinformatics/btz738)\n\n2.  ElTetrado: A tool for identification and classification of tetrads\n    and quadruplexes. T. Zok, M. Popenda, M. Szachniuk. *BMC\n    Bioinformatics*. 2020. 21(1):40.\n    doi:[10.1186/s12859-020-3385-1](https://doi.org/10.1186/s12859-020-3385-1)\n\n3.  DSSR: An integrated software tool for dissecting the spatial\n    structure of RNA. X.-J. Lu, H.J. Bussemaker, W.K. Olson. *Nucleic\n    Acids Research*. 2015. 43(21):e142.\n    doi:[10.1093/nar/gkv716](https://doi.org/10.1093/nar/gkv716)\n\n4.  R-chie : A web server and R package for visualizing RNA secondary\n    structures. D. Lai, J.R. Proctor, J.Y.A. Zhu, I.M. Meyer. *Nucleic\n    Acids Research*. 2012. 40(12):e95.\n    doi:[10.1093/nar/gks241](https://doi.org/10.1093/nar/gks241)\n\n5.  Geometric nomenclature and classification of RNA base pairs. N.B.\n    Leontis, E. Westhof. *RNA*. 2001. 7(4):499–512.\n    doi:[10.1017/S1355838201002515](https://doi.org/10.1017/S1355838201002515)\n\n</div>\n',
    'author': 'Tomasz Żok',
    'author_email': 'Tomasz.Zok@cs.put.poznan.pl',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/tzok/eltetrado',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
