"use strict";
var _a, _b, _c, _d, _e, _f, _g;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetImageCode = exports.EcrImageCode = exports.CfnParametersCode = exports.AssetCode = exports.InlineCode = exports.S3Code = exports.Code = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ecr_assets = require("@aws-cdk/aws-ecr-assets");
const iam = require("@aws-cdk/aws-iam");
const s3_assets = require("@aws-cdk/aws-s3-assets");
const cdk = require("@aws-cdk/core");
/**
 * Represents the Lambda Handler Code.
 *
 * @stability stable
 */
class Code {
    /**
     * Lambda handler code as an S3 object.
     *
     * @param bucket The S3 bucket.
     * @param key The object key.
     * @param objectVersion Optional S3 object version.
     * @stability stable
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3Code(bucket, key, objectVersion);
    }
    /**
     * (deprecated) DEPRECATED.
     *
     * @deprecated use `fromBucket`
     */
    static bucket(bucket, key, objectVersion) {
        return this.fromBucket(bucket, key, objectVersion);
    }
    /**
     * Inline code for Lambda handler.
     *
     * @param code The actual handler code (limited to 4KiB).
     * @returns `LambdaInlineCode` with inline code.
     * @stability stable
     */
    static fromInline(code) {
        return new InlineCode(code);
    }
    /**
     * (deprecated) DEPRECATED.
     *
     * @deprecated use `fromInline`
     */
    static inline(code) {
        return this.fromInline(code);
    }
    /**
     * Loads the function code from a local disk path.
     *
     * @param path Either a directory with the Lambda code bundle or a .zip file.
     * @stability stable
     */
    static fromAsset(path, options) {
        return new AssetCode(path, options);
    }
    /**
     * Loads the function code from an asset created by a Docker build.
     *
     * By defaut, the asset is expected to be located at `/asset` in the
     * image.
     *
     * @param path The path to the directory containing the Docker file.
     * @param options Docker build options.
     * @stability stable
     */
    static fromDockerBuild(path, options = {}) {
        var _h;
        const assetPath = cdk.DockerImage
            .fromBuild(path, options)
            .cp((_h = options.imagePath) !== null && _h !== void 0 ? _h : '/asset', options.outputPath);
        return new AssetCode(assetPath);
    }
    /**
     * (deprecated) DEPRECATED.
     *
     * @deprecated use `fromAsset`
     */
    static asset(path) {
        return this.fromAsset(path);
    }
    /**
     * Creates a new Lambda source defined using CloudFormation parameters.
     *
     * @param props optional construction properties of {@link CfnParametersCode}.
     * @returns a new instance of `CfnParametersCode`
     * @stability stable
     */
    static fromCfnParameters(props) {
        return new CfnParametersCode(props);
    }
    /**
     * (deprecated) DEPRECATED.
     *
     * @deprecated use `fromCfnParameters`
     */
    static cfnParameters(props) {
        return this.fromCfnParameters(props);
    }
    /**
     * Use an existing ECR image as the Lambda code.
     *
     * @param repository the ECR repository that the image is in.
     * @param props properties to further configure the selected image.
     * @stability stable
     */
    static fromEcrImage(repository, props) {
        return new EcrImageCode(repository, props);
    }
    /**
     * Create an ECR image from the specified asset and bind it as the Lambda code.
     *
     * @param directory the directory from which the asset must be created.
     * @param props properties to further configure the selected image.
     * @stability stable
     */
    static fromAssetImage(directory, props = {}) {
        return new AssetImageCode(directory, props);
    }
    /**
     * Called after the CFN function resource has been created to allow the code class to bind to it.
     *
     * Specifically it's required to allow assets to add
     * metadata for tooling like SAM CLI to be able to find their origins.
     *
     * @stability stable
     */
    bindToResource(_resource, _options) {
        return;
    }
}
exports.Code = Code;
_a = JSII_RTTI_SYMBOL_1;
Code[_a] = { fqn: "@aws-cdk/aws-lambda.Code", version: "1.93.0" };
/**
 * Lambda code from an S3 archive.
 *
 * @stability stable
 */
class S3Code extends Code {
    /**
     * @stability stable
     */
    constructor(bucket, key, objectVersion) {
        super();
        this.key = key;
        this.objectVersion = objectVersion;
        /**
         * Determines whether this Code is inline code or not.
         *
         * @stability stable
         */
        this.isInline = false;
        if (!bucket.bucketName) {
            throw new Error('bucketName is undefined for the provided bucket');
        }
        this.bucketName = bucket.bucketName;
    }
    /**
     * Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @stability stable
     */
    bind(_scope) {
        return {
            s3Location: {
                bucketName: this.bucketName,
                objectKey: this.key,
                objectVersion: this.objectVersion,
            },
        };
    }
}
exports.S3Code = S3Code;
_b = JSII_RTTI_SYMBOL_1;
S3Code[_b] = { fqn: "@aws-cdk/aws-lambda.S3Code", version: "1.93.0" };
/**
 * Lambda code from an inline string (limited to 4KiB).
 *
 * @stability stable
 */
class InlineCode extends Code {
    /**
     * @stability stable
     */
    constructor(code) {
        super();
        this.code = code;
        /**
         * Determines whether this Code is inline code or not.
         *
         * @stability stable
         */
        this.isInline = true;
        if (code.length === 0) {
            throw new Error('Lambda inline code cannot be empty');
        }
        if (code.length > 4096) {
            throw new Error('Lambda source is too large, must be <= 4096 but is ' + code.length);
        }
    }
    /**
     * Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @stability stable
     */
    bind(_scope) {
        return {
            inlineCode: this.code,
        };
    }
}
exports.InlineCode = InlineCode;
_c = JSII_RTTI_SYMBOL_1;
InlineCode[_c] = { fqn: "@aws-cdk/aws-lambda.InlineCode", version: "1.93.0" };
/**
 * Lambda code from a local directory.
 *
 * @stability stable
 */
class AssetCode extends Code {
    /**
     * @param path The path to the asset file or directory.
     * @stability stable
     */
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
        /**
         * Determines whether this Code is inline code or not.
         *
         * @stability stable
         */
        this.isInline = false;
    }
    /**
     * Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @stability stable
     */
    bind(scope) {
        // If the same AssetCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new s3_assets.Asset(scope, 'Code', {
                path: this.path,
                ...this.options,
            });
        }
        else if (cdk.Stack.of(this.asset) !== cdk.Stack.of(scope)) {
            throw new Error(`Asset is already associated with another stack '${cdk.Stack.of(this.asset).stackName}'. ` +
                'Create a new Code instance for every stack.');
        }
        if (!this.asset.isZipArchive) {
            throw new Error(`Asset must be a .zip file or a directory (${this.path})`);
        }
        return {
            s3Location: {
                bucketName: this.asset.s3BucketName,
                objectKey: this.asset.s3ObjectKey,
            },
        };
    }
    /**
     * Called after the CFN function resource has been created to allow the code class to bind to it.
     *
     * Specifically it's required to allow assets to add
     * metadata for tooling like SAM CLI to be able to find their origins.
     *
     * @stability stable
     */
    bindToResource(resource, options = {}) {
        if (!this.asset) {
            throw new Error('bindToResource() must be called after bind()');
        }
        const resourceProperty = options.resourceProperty || 'Code';
        // https://github.com/aws/aws-cdk/issues/1432
        this.asset.addResourceMetadata(resource, resourceProperty);
    }
}
exports.AssetCode = AssetCode;
_d = JSII_RTTI_SYMBOL_1;
AssetCode[_d] = { fqn: "@aws-cdk/aws-lambda.AssetCode", version: "1.93.0" };
/**
 * Lambda code defined using 2 CloudFormation parameters.
 *
 * Useful when you don't have access to the code of your Lambda from your CDK code, so you can't use Assets,
 * and you want to deploy the Lambda in a CodePipeline, using CloudFormation Actions -
 * you can fill the parameters using the {@link #assign} method.
 *
 * @stability stable
 */
class CfnParametersCode extends Code {
    /**
     * @stability stable
     */
    constructor(props = {}) {
        super();
        /**
         * Determines whether this Code is inline code or not.
         *
         * @stability stable
         */
        this.isInline = false;
        this._bucketNameParam = props.bucketNameParam;
        this._objectKeyParam = props.objectKeyParam;
    }
    /**
     * Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @stability stable
     */
    bind(scope) {
        if (!this._bucketNameParam) {
            this._bucketNameParam = new cdk.CfnParameter(scope, 'LambdaSourceBucketNameParameter', {
                type: 'String',
            });
        }
        if (!this._objectKeyParam) {
            this._objectKeyParam = new cdk.CfnParameter(scope, 'LambdaSourceObjectKeyParameter', {
                type: 'String',
            });
        }
        return {
            s3Location: {
                bucketName: this._bucketNameParam.valueAsString,
                objectKey: this._objectKeyParam.valueAsString,
            },
        };
    }
    /**
     * Create a parameters map from this instance's CloudFormation parameters.
     *
     * It returns a map with 2 keys that correspond to the names of the parameters defined in this Lambda code,
     * and as values it contains the appropriate expressions pointing at the provided S3 location
     * (most likely, obtained from a CodePipeline Artifact by calling the `artifact.s3Location` method).
     * The result should be provided to the CloudFormation Action
     * that is deploying the Stack that the Lambda with this code is part of,
     * in the `parameterOverrides` property.
     *
     * @param location the location of the object in S3 that represents the Lambda code.
     * @stability stable
     */
    assign(location) {
        const ret = {};
        ret[this.bucketNameParam] = location.bucketName;
        ret[this.objectKeyParam] = location.objectKey;
        return ret;
    }
    /**
     * @stability stable
     */
    get bucketNameParam() {
        if (this._bucketNameParam) {
            return this._bucketNameParam.logicalId;
        }
        else {
            throw new Error('Pass CfnParametersCode to a Lambda Function before accessing the bucketNameParam property');
        }
    }
    /**
     * @stability stable
     */
    get objectKeyParam() {
        if (this._objectKeyParam) {
            return this._objectKeyParam.logicalId;
        }
        else {
            throw new Error('Pass CfnParametersCode to a Lambda Function before accessing the objectKeyParam property');
        }
    }
}
exports.CfnParametersCode = CfnParametersCode;
_e = JSII_RTTI_SYMBOL_1;
CfnParametersCode[_e] = { fqn: "@aws-cdk/aws-lambda.CfnParametersCode", version: "1.93.0" };
/**
 * Represents a Docker image in ECR that can be bound as Lambda Code.
 *
 * @stability stable
 */
class EcrImageCode extends Code {
    /**
     * @stability stable
     */
    constructor(repository, props = {}) {
        super();
        this.repository = repository;
        this.props = props;
        /**
         * Determines whether this Code is inline code or not.
         *
         * @stability stable
         */
        this.isInline = false;
    }
    /**
     * Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @stability stable
     */
    bind(_) {
        var _h, _j;
        this.repository.grantPull(new iam.ServicePrincipal('lambda.amazonaws.com'));
        return {
            image: {
                imageUri: this.repository.repositoryUriForTag((_j = (_h = this.props) === null || _h === void 0 ? void 0 : _h.tag) !== null && _j !== void 0 ? _j : 'latest'),
                cmd: this.props.cmd,
                entrypoint: this.props.entrypoint,
            },
        };
    }
}
exports.EcrImageCode = EcrImageCode;
_f = JSII_RTTI_SYMBOL_1;
EcrImageCode[_f] = { fqn: "@aws-cdk/aws-lambda.EcrImageCode", version: "1.93.0" };
/**
 * Represents an ECR image that will be constructed from the specified asset and can be bound as Lambda code.
 *
 * @stability stable
 */
class AssetImageCode extends Code {
    /**
     * @stability stable
     */
    constructor(directory, props) {
        super();
        this.directory = directory;
        this.props = props;
        /**
         * Determines whether this Code is inline code or not.
         *
         * @stability stable
         */
        this.isInline = false;
    }
    /**
     * Called when the lambda or layer is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @stability stable
     */
    bind(scope) {
        const asset = new ecr_assets.DockerImageAsset(scope, 'AssetImage', {
            directory: this.directory,
            ...this.props,
        });
        asset.repository.grantPull(new iam.ServicePrincipal('lambda.amazonaws.com'));
        return {
            image: {
                imageUri: asset.imageUri,
                entrypoint: this.props.entrypoint,
                cmd: this.props.cmd,
            },
        };
    }
}
exports.AssetImageCode = AssetImageCode;
_g = JSII_RTTI_SYMBOL_1;
AssetImageCode[_g] = { fqn: "@aws-cdk/aws-lambda.AssetImageCode", version: "1.93.0" };
//# sourceMappingURL=data:application/json;base64,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