from math import pi
import numpy as np
from simudo.physics import Material


class SiliconMaterial(Material):
    """Silicon material data based on Palankovski.

       V. Palankovski and R. Quay, "Analysis and Simulation of Heterostructure Devices",
       Springer-Verlag (2004).
    """

    name = "Silicon"

    def get_dict(self):
        d = super().get_dict()
        U = self.unit_registry

        d.update(
            {
                # Static dielectric constant, Palankovski Table 3.3
                "poisson/permittivity": U("11.9 vacuum_permittivity"),
                # Table 3.9
                "CB/EgX_0K": U("1.1695 eV"),
                "CB/ChiX_0K": U("4.0045 eV") - U("4 eV"),
                "CB/varshni_alphaX": U("4.73e-4 eV/K"),
                "CB/varshni_betaX": U("636 K"),
                # Table 3.21
                "CB/MCX": U("6"),
                # Table 3.18
                "CB/mX": U("0.328"),
                "CB/m1n": U("0.009"),
                # Table 3.19
                "VB/m0p": U("0.55"),
                "VB/m1p": U("0.6"),
                "VB/m2p": U("-0.1"),
                # TODO - update these mobilities from Palankovski
                "CB/mobility": U("1400 cm^2/V/s"),
                "VB/mobility": U(" 450 cm^2/V/s"),
                # SRH recombination lifetimes
                "SRH/CB/tau": U("1e-9 s"),
                "SRH/VB/tau": U("1e-9 s"),
                # Properties for band-to-band tunneling
                "CB/eff_mass_tunnel": U("0.05"),
                "VB/eff_mass_tunnel": U("0.1"),
                "CB/tunnel_offset_energy": U("0 eV"),
                "VB/tunnel_offset_energy": U("0 eV"),
            }
        )

        T = self.temperature
        Eg = d["CB/EgX_0K"] - d["CB/varshni_alphaX"] * T ** 2 / (
            d["CB/varshni_betaX"] + T
        )
        Chi = d["CB/ChiX_0K"] + d["CB/varshni_alphaX"] * T ** 2 / (
            d["CB/varshni_betaX"] + T
        )

        mn = d["CB/mX"] + d["CB/m1n"] * (T / U("300 K"))
        mp = (
            d["VB/m0p"]
            + d["VB/m1p"] * (T / U("300 K"))
            + d["VB/m2p"] * (T / U("300 K")) ** 2
        )

        m_e = U.electron_mass
        k_B = U.boltzmann_constant
        h = U.planck_constant
        NC = 2 * d["CB/MCX"] * (2 * pi * mn * m_e * k_B * T / h ** 2) ** (3 / 2)
        NV = 2 * (2 * pi * mp * m_e * k_B * T / h ** 2) ** (3 / 2)

        # Effective masses and thermal velocity from Green (1990)
        # mtc actually has a small temperature dependence acording to Green
        mtc = U("0.28")
        vth_c = (
            8 * U.boltzmann_constant * T / (pi * mtc * U.electron_mass)
        ) ** 0.5

        mtv = (
            0.3676
            + U("1.98738e-5 /K^2") * T ** 2
            - U("2.588144e-7 /K^3") * T ** 3
            + U("1.415372e-9 /K^4") * T ** 4
            - U("3.919169e-12 /K^5") * T ** 5
            + U("5.410849e-15 /K^6") * T ** 6
            - U("2.959797e-18 /K^7") * T ** 7
        )
        vth_v = (
            8 * U.boltzmann_constant * T / (pi * mtv * U.electron_mass)
        ) ** 0.5

        d.update(
            {
                "CB/mDOS": mn,
                "VB/mDOS": mp,
                "CB/energy_level": -Chi,
                "VB/energy_level": -Chi - Eg,
                "CB/effective_density_of_states": NC,
                "VB/effective_density_of_states": NV,
                "CB/vth": vth_c,
                # "CB/vth": 7e5*U("cm/s"),
                "VB/vth": vth_v,
                "SRH/energy_level": -Chi - Eg / 2,
            }
        )

        return d

    def optical_properties_table(self):
        """
        Optical properties from M.A. Green,
        "Self-consistent optical parameters of intrinsic silicon at 300 K
        including temperature coefficients", Sol. Ener. Mat. and Sol. Cells.,
        https://doi.org/10.1016/j.solmat.2008.06.009
        """

        # TODO: The reference includes data on temperature dependence.
        # Incorporate it here.

        # wavelength (nm), alpha (/cm), n, k
        np.array(
            [
                [250, 1.84e06, 1.694, 3.66056369111359],
                [260, 1.97e06, 1.8, 4.07595809258344],
                [270, 2.18e06, 2.129, 4.68392997519448],
                [280, 2.36e06, 3.052, 5.25847931975622],
                [290, 2.24e06, 4.426, 5.16935255162476],
                [300, 1.73e06, 5.055, 4.13007077323468],
                [310, 1.44e06, 5.074, 3.5523383298111],
                [320, 1.28e06, 5.102, 3.25949323452202],
                [330, 1.17e06, 5.179, 3.07248617638904],
                [340, 1.09e06, 5.293, 2.94914109549282],
                [350, 1.04e06, 5.483, 2.89661996427249],
                [360, 1.02e06, 6.014, 2.9220847551672],
                [370, 6.97e05, 6.863, 2.05222341369844],
                [380, 2.93e05, 6.548, 0.886015568192581],
                [390, 1.50e05, 5.976, 0.465528208543794],
                [400, 9.52e04, 5.587, 0.303031011646969],
                [410, 6.74e04, 5.305, 0.219904384870072],
                [420, 5.00e04, 5.091, 0.16711269024649],
                [430, 3.92e04, 4.925, 0.134135786037849],
                [440, 3.11e04, 4.793, 0.108893812063475],
                [450, 2.55e04, 4.676, 0.091315148598975],
                [460, 2.10e04, 4.577, 0.076871837513386],
                [470, 1.72e04, 4.491, 0.064330427997744],
                [480, 1.48e04, 4.416, 0.056531835786241],
                [490, 1.27e04, 4.348, 0.049521060543043],
                [500, 1.11e04, 4.293, 0.044165496708001],
                [510, 9.70e03, 4.239, 0.03936697517378],
                [520, 8.80e03, 4.192, 0.036414650979426],
                [530, 7.85e03, 4.15, 0.033108207036692],
                [540, 7.05e03, 4.11, 0.030295143417542],
                [550, 6.39e03, 4.077, 0.027967502374823],
                [560, 5.78e03, 4.044, 0.025757635989992],
                [570, 5.32e03, 4.015, 0.024131072471593],
                [580, 4.88e03, 3.986, 0.022523607546365],
                [590, 4.49e03, 3.962, 0.021080867987237],
                [600, 4.14e03, 3.939, 0.019767043932013],
                [610, 3.81e03, 3.916, 0.018494600161994],
                [620, 3.52e03, 3.895, 0.017366987390188],
                [630, 3.27e03, 3.879, 0.016393754913181],
                [640, 3.04e03, 3.861, 0.01548259286398],
                [650, 2.81e03, 3.844, 0.014534825177867],
                [660, 2.58e03, 3.83, 0.013550451854844],
                [670, 2.38e03, 3.815, 0.012689423612717],
                [680, 2.21e03, 3.8, 0.011958902423925],
                [690, 2.05e03, 3.787, 0.011256233350174],
                [700, 1.90e03, 3.774, 0.010583803715611],
                [710, 1.77e03, 3.762, 0.010000500849179],
                [720, 1.66e03, 3.751, 0.009511099399172],
                [730, 1.54e03, 3.741, 0.008946099351195],
                [740, 1.42e03, 3.732, 0.008362000710048],
                [750, 1.30e03, 3.723, 0.00775880347573],
                [760, 1.19e03, 3.714, 0.007196986526616],
                [770, 1.10e03, 3.705, 0.006740211839942],
                [780, 1.01e03, 3.696, 0.00626911320839],
                [790, 9.28e02, 3.688, 0.005833983593977],
                [800, 8.50e02, 3.681, 0.005411268065124],
                [810, 7.75e02, 3.674, 0.004995475776297],
                [820, 7.07e02, 3.668, 0.004613424335405],
                [830, 6.47e02, 3.662, 0.004273389799489],
                [840, 5.91e02, 3.656, 0.003950543997427],
                [850, 5.35e02, 3.65, 0.003618785518552],
                [860, 4.80e02, 3.644, 0.003284958025417],
                [870, 4.32e02, 3.638, 0.002990839690583],
                [880, 3.83e02, 3.632, 0.002682079100985],
                [890, 3.43e02, 3.626, 0.002429261473883],
                [900, 3.06e02, 3.62, 0.002191563566375],
                [910, 2.72e02, 3.614, 0.001969701575705],
                [920, 2.40e02, 3.608, 0.001757070571735],
                [930, 2.10e02, 3.602, 0.001554148019292],
                [940, 1.83e02, 3.597, 0.001368891665533],
                [950, 1.57e02, 3.592, 0.001186897988108],
                [960, 1.34e02, 3.587, 0.001023684593967],
                [970, 1.14e02, 3.582, 0.000879967680355],
                [980, 9.59e01, 3.578, 0.000747884993083],
                [990, 7.92e01, 3.574, 0.000623951038897],
                [1000, 6.40e01, 3.57, 0.000509295817894],
                [1010, 5.11e01, 3.566, 0.000410707288396],
                [1020, 3.99e01, 3.563, 0.000323864393698],
                [1030, 3.02e01, 3.56, 0.000247533682991],
                [1040, 2.26e01, 3.557, 0.000187038889122],
                [1050, 1.63e01, 3.554, 0.000136196842551],
                [1060, 1.11e01, 3.551, 9.3630853020962e-05],
                [1070, 8.00e00, 3.548, 6.81183156433312e-05],
                [1080, 6.20e00, 3.546, 5.32850749471666e-05],
                [1090, 4.70e00, 3.544, 4.0767538672989e-05],
                [1100, 3.50e00, 3.541, 3.06373265451899e-05],
                [1110, 2.70e00, 3.539, 2.38493682223205e-05],
                [1120, 2.00e00, 3.537, 1.78253536262923e-05],
                [1130, 1.50e00, 3.534, 1.34883814270381e-05],
                [1140, 1.00e00, 3.532, 9.07183175623803e-06],
                [1150, 6.80e-01, 3.53, 6.22295827489311e-06],
                [1160, 4.20e-01, 3.528, 3.87701441371857e-06],
                [1170, 2.20e-01, 3.526, 2.04832411759269e-06],
                [1180, 6.50e-02, 3.524, 6.10359206757419e-07],
                [1190, 3.60e-02, 3.522, 3.4090988810284e-07],
                [1200, 2.20e-02, 3.52, 2.10084524881302e-07],
                [1210, 1.30e-02, 3.528, 1.25175362741776e-07],
                [1220, 8.20e-03, 3.516, 7.9609302534566e-08],
                [1230, 4.70e-03, 3.515, 4.60037363007123e-08],
                [1240, 2.40e-03, 3.513, 2.3682255532074e-08],
                [1250, 1.00e-03, 3.511, 9.94718394324346e-09],
                [1260, 3.60e-04, 3.51, 3.60963410932419e-09],
                [1270, 2.00e-04, 3.508, 2.02126777726707e-09],
                [1280, 1.20e-04, 3.507, 1.22230996294576e-09],
                [1290, 7.10e-05, 3.506, 7.28850061889335e-10],
                [1300, 4.50e-05, 3.504, 4.65528208543794e-10],
                [1310, 2.70e-05, 3.503, 2.81465516858017e-10],
                [1320, 1.60e-05, 3.501, 1.68067619905041e-10],
                [1330, 8.00e-06, 3.5, 8.46704297248883e-11],
                [1340, 3.50e-06, 3.498, 3.73218341550495e-11],
                [1350, 1.70e-06, 3.497, 1.8263029719795e-11],
                [1360, 1.00e-06, 3.496, 1.08225361302489e-11],
                [1370, 6.70e-07, 3.495, 7.30441611320254e-12],
                [1380, 4.50e-07, 3.493, 4.94176098300335e-12],
                [1390, 2.50e-07, 3.492, 2.76531713622168e-12],
                [1400, 2.00e-07, 3.491, 2.22816920328653e-12],
                [1410, 1.50e-07, 3.49, 1.68306352319679e-12],
                [1420, 8.50e-08, 3.489, 9.60500081559588e-13],
                [1430, 7.70e-08, 3.488, 8.7622753919243e-13],
                [1440, 4.20e-08, 3.487, 4.81284547909892e-13],
                [1450, 3.20e-08, 3.486, 3.69239467973197e-13],
            ]
        )
