# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/rolling.ipynb (unless otherwise specified).

__all__ = ['rolling_mean', 'rolling_std', 'rolling_max', 'rolling_min', 'seasonal_rolling_mean', 'seasonal_rolling_std',
           'seasonal_rolling_max', 'seasonal_rolling_min']

# Cell
from math import sqrt
from typing import Callable, Optional

import numpy as np
from numba import njit

from .utils import _gt, _lt, _rolling_std, _validate_rolling_sizes

# Cell
@njit
def rolling_mean(x: np.ndarray,
                 window_size: int,
                 min_samples: Optional[int] = None) -> np.ndarray:
    n_samples = x.size
    window_size, min_samples = _validate_rolling_sizes(n_samples, window_size, min_samples)
    out = np.full(n_samples, np.nan, dtype=np.float32)
    accum = 0.
    for i in range(window_size):
        accum += x[i]
        if i + 1 >= min_samples:
            out[i] = accum / (i+1)
    for i in range(window_size, n_samples):
        accum += x[i] - x[i-window_size]
        out[i] = accum / window_size
    return out

# Cell
@njit
def rolling_std(x: np.ndarray,
                window_size: int,
                min_samples: Optional[int] = None) -> np.ndarray:
    out, _, _ = _rolling_std(x, window_size, min_samples)
    return out

# Internal Cell
@njit
def _rolling_comp(comp: Callable,
                  x: np.ndarray,
                  window_size: int,
                  min_samples: Optional[int] = None):
    n_samples = x.size
    window_size, min_samples = _validate_rolling_sizes(n_samples, window_size, min_samples)
    out = np.full(n_samples, np.nan, dtype=np.float32)
    for i in range(min_samples - 1, n_samples):
        pivot = x[i]
        ws = min(i+1, window_size)
        for j in range(1, ws):
            if comp(x[i - j], pivot) > 0:
                pivot = x[i - j]
        out[i] = pivot
    return out

# Cell
@njit
def rolling_max(x: np.ndarray,
                window_size: int,
                min_samples: Optional[int] = None):
    return _rolling_comp(_gt, x, window_size, min_samples)

# Cell
@njit
def rolling_min(x: np.ndarray,
                window_size: int,
                min_samples: Optional[int] = None):
    return _rolling_comp(_lt, x, window_size, min_samples)

# Internal Cell
def _seasonal_rolling_op(rolling_op: Callable,
                         x: np.ndarray,
                         season_length: int,
                         window_size: int,
                         min_samples: Optional[int] = None) -> np.ndarray:
    n_samples = x.size
    out = np.full(n_samples, np.nan, dtype=np.float32)
    for season in range(season_length):
        out[season::season_length] = rolling_op(x[season::season_length], window_size, min_samples)
    return out

# Cell
def seasonal_rolling_mean(x: np.ndarray,
                          season_length: int,
                          window_size: int,
                          min_samples: Optional[int] = None) -> np.ndarray:
    return _seasonal_rolling_op(rolling_mean, x, season_length, window_size=window_size, min_samples=min_samples)

# Cell
def seasonal_rolling_std(x: np.ndarray,
                         season_length: int,
                         window_size: int,
                         min_samples: Optional[int] = None
                         ) -> np.ndarray:
    return _seasonal_rolling_op(rolling_std, x, season_length, window_size, min_samples)

# Cell
def seasonal_rolling_max(x: np.ndarray,
                         season_length: int,
                         window_size: int,
                         min_samples: Optional[int] = None
                         ) -> np.ndarray:
    return _seasonal_rolling_op(rolling_max, x, season_length, window_size, min_samples)

# Cell
def seasonal_rolling_min(x: np.ndarray,
                         season_length: int,
                         window_size: int,
                         min_samples: Optional[int] = None
                         ) -> np.ndarray:
    return _seasonal_rolling_op(rolling_min, x, season_length, window_size, min_samples)