# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['emo_platform']

package_data = \
{'': ['*'], 'emo_platform': ['tokens/.gitkeep']}

install_requires = \
['aiohttp>=3.7.4,<4.0.0',
 'fire>=0.4.0,<0.5.0',
 'pydantic>=1.9.0,<2.0.0',
 'requests>=2.26.0,<3.0.0']

entry_points = \
{'console_scripts': ['emo_platform_cli = emo_platform.cli:main']}

setup_kwargs = {
    'name': 'emo-platform-api-sdk',
    'version': '1.0.0',
    'description': 'emo platform api python sdk',
    'long_description': '# BOCCO emo platform api python sdk\n\n## How to install\n### Using poetry (if you want to use in python virtual environment)\nIf poetry has not been installed, please see [this page](https://python-poetry.org/docs/) to install.\n\n```bash\n# Python 3.7+ required\npoetry install --no-dev\n```\n\nWhen you execute python code,\n\n```bash\npoetry run python your_python_code.py\n```\n\n### Using PyPl\n\n```\n# Python 3.7+ required\n$ pip3 install emo-platform-api-sdk\n```\n\n## Setting api tokens\n\nYou can see access token & refresh token from dashboard in [this page](https://platform-api.bocco.me/dashboard/login) after login.\n\nThen, set those tokens as environment variables in terminal.\n\n```bash\nexport EMO_PLATFORM_API_ACCESS_TOKEN="***"\nexport EMO_PLATFORM_API_REFRESH_TOKEN="***"\n```\n\nOr, you can give as argument when initializing client in python code.\n\n```python\nfrom emo_platform import Client, Tokens\n\nclient = Client(Tokens(access_token="***", refresh_token="***"))\n```\n\n### Note\n- Once you set tokens as arguments or environment variables, the tokens are saved in the sdk and you don\'t need to specify any arguments or set any environment variables next time.\n- If you want to overwrite the tokens with the other tokens, for example if you want to change your account, set the new tokens again with arguments or environment variables.\n\n\n## Usage Example\n\nYou can also see other examples in "examples" directory.\n\n### Note\n- When you initialize emo_platform.Client, without the argument `use_cached_credentials` given as `True` , two json files (emo-platform-api.json & emo-platform-api_previous.json) are created in the path where emo_platform module was installed.\n\t- These files are used to store the tokens information.\n\t- See the documentation for details.\n- You can change the path where these json files are created, as shown below.\n\n```python\nimport os\nfrom emo_platform import Client\n\nCURRENT_DIR = os.path.abspath(os.path.dirname(__file__))\n\nclient = Client(token_file_path=CURRENT_DIR)\n```\n\n### Example1 : Using client\n```python\nfrom emo_platform import Client, Head\n\nclient = Client()\n\nprint(client.get_account_info())\nprint(client.get_stamps_list())\n\nroom_id_list = client.get_rooms_id()\nroom_client = client.create_room_client(room_id_list[0])\n\nprint(room_client.get_msgs())\nroom_client.move_to(Head(10,10))\n```\n\n#### For business user\nWhen you use business version, you need to give api_key as argument for some functions.\n\nYou can find the API key in [this page](https://platform-api.bocco.me/dashboard/login) after login with business account.\n```python\nfrom emo_platform import BizBasicClient, BizAdvancedClient\n\n# business basic version\nclient = BizBasicClient()\n\n# business advanced version\nclient = BizAdvancedClient()\n\nprint(client.get_account_info())\nprint(client.get_stamps_list(api_key))\n\nroom_id_list = client.get_rooms_id(api_key)\n# give api_key to room client\nroom_client = client.create_room_client(api_key, room_id_list[0])\n\n# need not to give api_key to room client method\n# if you want to change api_key, please create another room_client for each api_key\nprint(room_client.get_msgs())\n```\n\n\n### Example2 : Receive webhook\n\nIn another terminal, execute ngrok and copy URL forwarded to http://localhost:8000.\n```bash\nngrok http 8000\n```\n\n#### Case1 : Using function `start_webhook_event()` (**Recommended**)\nYou can use the decorator `event()` to register functions as callback functions.\n\nAnd, you can get the corresponding callback function and the parsed body by giving the webhook request body as the argument of the function `get_cb_func()`.\n\nPlease check if the `X-Platform-Api-Secret` in the header of the webhook request is the same as the return value of `start_webhook_event()` to avoid unexpected requests from third parties.\n```python\nimport json, http.server\nfrom emo_platform import Client, WebHook, EmoPlatformError\n\nclient = Client()\n# Please replace "YOUR WEBHOOK URL" with the URL forwarded to http://localhost:8000\nclient.create_webhook_setting(WebHook("YOUR WEBHOOK URL"))\n\n@client.event(\'message.received\')\ndef message_callback(data):\n\tprint("message received")\n\tprint(data)\n\n@client.event(\'illuminance.changed\')\ndef illuminance_callback(data):\n\tprint("illuminance changed")\n\tprint(data)\n\nsecret_key = client.start_webhook_event()\n\n\n# localserver\nclass Handler(http.server.BaseHTTPRequestHandler):\n\tdef _send_status(self, status):\n\t\tself.send_response(status)\n\t\tself.send_header(\'Content-type\', \'text/plain; charset=utf-8\')\n\t\tself.end_headers()\n\n\tdef do_POST(self):\n\t\t# check secret_key\n\t\tif not secret_key == self.headers["X-Platform-Api-Secret"]:\n\t\t\tself._send_status(401)\n\t\t\treturn\n\n\t\tcontent_len = int(self.headers[\'content-length\'])\n\t\trequest_body = json.loads(self.rfile.read(content_len).decode(\'utf-8\'))\n\n\t\ttry:\n\t\t\tcb_func, emo_webhook_body = client.get_cb_func(request_body)\n\t\texcept EmoPlatformError:\n\t\t\tself._send_status(501)\n\t\t\treturn\n\n\t\tcb_func(emo_webhook_body)\n\n\t\tself._send_status(200)\n\nwith http.server.HTTPServer((\'\', 8000), Handler) as httpd:\n\thttpd.serve_forever()\n\n```\n\n#### Case2 : Using function `register_webhook_event()` (**Not recommended**)\nYou can\'t use the decorator `event()` to register functions as callback functions.\n\nSo, you need to call the callback functions yourself after webhook request body is parsed using `parse_webhook_body()`.\n\nPlease check if the `X-Platform-Api-Secret` in the header of the webhook request is correct using the return value of `register_webhook_event()` to avoid unexpected requests from third parties.\n\n```python\nimport json, http.server\nfrom emo_platform import Client, WebHook, parse_webhook_body\n\nclient = Client()\n# Please replace "YOUR WEBHOOK URL" with the URL forwarded to http://localhost:8000\nclient.create_webhook_setting(WebHook("YOUR WEBHOOK URL"))\n\ndef message_callback(data):\n\tprint("message received")\n\tprint(data)\n\ndef illuminance_callback(data):\n\tprint("illuminance changed")\n\tprint(data)\n\nresponse = client.register_webhook_event(\n\t[\'message.received\',\'illuminance.changed\' ]\n)\nsecret_key = response.secret\n\n# localserver\nclass Handler(http.server.BaseHTTPRequestHandler):\n\tdef _send_status(self, status):\n\t\tself.send_response(status)\n\t\tself.send_header(\'Content-type\', \'text/plain; charset=utf-8\')\n\t\tself.end_headers()\n\n\tdef do_POST(self):\n\t\t# check secret_key\n\t\tif not secret_key == self.headers["X-Platform-Api-Secret"]:\n\t\t\tself._send_status(401)\n\t\t\treturn\n\n\t\tcontent_len = int(self.headers[\'content-length\'])\n\t\trequest_body = json.loads(self.rfile.read(content_len).decode(\'utf-8\'))\n\n\t\temo_webhook_body = parse_webhook_body(request_body)\n\t\tif emo_webhook_body.event == "message.received":\n\t\t\tmessage_callback(emo_webhook_body)\n\t\telif emo_webhook_body.event == "illuminance.changed":\n\t\t\tilluminance_callback(emo_webhook_body)\n\t\telse:\n\t\t\tpass\n\n\t\tself._send_status(200)\n\nwith http.server.HTTPServer((\'\', 8000), Handler) as httpd:\n\thttpd.serve_forever()\n\n```\n\n## Cli Tool\nYou can use the command `emo_platform_cli` after installing this package.\n\n### Example1 : Use client\nInitially, you need to specify your account refresh token.\n```bash\n$ emo_platform_cli personal --refresh_token *** get_account_info\n```\nOnce you set refresh_token, you don\'t need to set again.\n```bash\n$ emo_platform_cli personal get_account_info\n```\n\n### Example2 : Use room client\nPlease replace `ROOM_ID` with room id which you want to use.\n```\n$ emo_platform_cli personal create_room_client ROOM_ID change_led_color 10 10 200\n```\nYou can get room id as shown below.\n```\n$ emo_platform_cli personal get_rooms_id\n```\n\nOr, you can use "room" command which does not require the room id to be specified.\nThis is because it calls `get_rooms_id()` internally and specifies the first room id.\n```\n$ emo_platform_cli personal room change_led_color 10 10 200\n```\n\n### Help\nIf you run a command without giving any arguments, a list of available commands(functions) or required arguments will be displayed with explanations.\n\n#### Example1 : Show available clients\n```bash\n$ emo_platform_cli\n\nNAME\n    emo_platform_cli\n\nSYNOPSIS\n    emo_platform_cli COMMAND\n\nCOMMANDS\n    COMMAND is one of the following:\n\n     biz_advanced\n       各種apiを呼び出す同期版のclient(Business Advanced版)\n\n     biz_basic\n       各種apiを呼び出す同期版のclient(Business Basic版)\n\n     personal\n       各種apiを呼び出す同期版のclient(Personal版)\n\n```\n\n#### Example2 : Show available functions in personal version\n```bash\n$ emo_platform_cli personal\nNAME\n    emo_platform_cli personal\n\nSYNOPSIS\n    emo_platform_cli personal - COMMAND\n\nCOMMANDS\n    COMMAND is one of the following:\n\n     change_webhook_setting\n       Webhookの設定の変更\n\n     create_room_client\n       部屋固有の各種apiを呼び出すclientの作成\n\n     create_webhook_setting\n       Webhookの設定の作成\n\n     delete_account_info\n       アカウントの削除\n\n: ...\n```\n\n#### Example3 : Show available functions unique to room in personal version\n```bash\n$ emo_platform_cli personal room\nNAME\n    emo_platform_cli personal room\n\nSYNOPSIS\n    emo_platform_cli personal room COMMAND | VALUE\n\nCOMMANDS\n    COMMAND is one of the following:\n\n     change_led_color\n       ほっぺたの色の変更\n\n     get_emo_settings\n       現在のBOCCO emoの設定値を取得\n\n     get_msgs\n       部屋に投稿されたメッセージの取得\n\n     get_sensor_values\n       部屋センサの送信値を取得\n\n: ...\n```\n\n#### Example4 : Show the required argument of `change_webhook_setting()`\n```bash\n$ emo_platform_cli personal change_webhook_setting\nERROR: The function received no value for the required argument: url\nUsage: emo_platform_cli personal change_webhook_setting URL <flags>\n  optional flags:        --description\n\nFor detailed information on this command, run:\n  emo_platform_cli personal change_webhook_setting --help\n```\n',
    'author': 'Keita Ito',
    'author_email': 'kito@ux-xu.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://yukai.github.io/emo-platform-api-python/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
