# -*- coding: utf-8 -*-
# vim: tabstop=4 shiftwidth=4 softtabstop=4
#
# Copyright (C) 2014-2023 GEM Foundation
#
# OpenQuake is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenQuake is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with OpenQuake. If not, see <http://www.gnu.org/licenses/>.

"""
Module exports :class:`WongEtAl2015`.
               :class:`WongEtAl2022Shallow`.
               :class:`WongEtAl2022Deep.
"""
import numpy as np

from openquake.hazardlib.gsim.base import GMPE, CoeffsTable
from openquake.hazardlib import const
from openquake.hazardlib.imt import PGA, PGV, SA


def _compute_distance(ctx, C):
    """
    Compute the second term of the equation 7 described in Results:

    """

    return (C['C6'] + C['C7'] * ctx.mag) * np.log(ctx.rjb + np.exp(C['C4']))


def _compute_magnitude(ctx, C):
    """
    Compute the third term of the equation 1:

    """
    return C['C2'] * ctx.mag + C['C10'] * (ctx.mag - 6)**2


class WongEtAl2022Shallow(GMPE):
    """
    Implements GMPE developed by Ivan Wong, Robert Darragh, Sarah Smith, Qimin
    Wu, Walter Silva, Tadahiro Kishida; "Ground motion models for shallow
    crustal and deep earthquakes in Hawaii and analyses of the 2018 M 6.9
    Kalapana sequence." Earthquake Spectra 2022;; 38 (1): 579–614.  doi:
        https://doi.org/10.1177/87552930211044521

    SA are given up to 10 s.  The regressions are developed considering the
    average horizontal component of the as-recorded horizontal components
    Ground motion refers to Vs30 of 150, 185, 260, 365, 428, 530, 760, 1080,
    1500, m/s Model is for deep events (depth >20km) and crustal events.  this
    class is only for shallow ebents
    """
    #: Supported tectonic region type is 'active shallow crust' because the
    #: equations have been derived from data from Italian database ITACA, as
    #: explained in the 'Introduction'.
    DEFINED_FOR_TECTONIC_REGION_TYPE = const.TRT.VOLCANIC

    #: Set of :mod:`intensity measure types <openquake.hazardlib.imt>`
    #: this GSIM can calculate. A set should contain classes from module
    #: :mod:`openquake.hazardlib.imt`.
    DEFINED_FOR_INTENSITY_MEASURE_TYPES = {PGA, PGV, SA}

    #: Supported intensity measure component is the geometric mean of two
    #: horizontal components
    DEFINED_FOR_INTENSITY_MEASURE_COMPONENT = const.IMC.HORIZONTAL

    #: Supported standard deviation types are inter-event, intra-event
    #: and total, page 1904
    DEFINED_FOR_STANDARD_DEVIATION_TYPES = {
        const.StdDev.TOTAL}

    #: Required rupture parameters are magnitude and rake (eq. 1).
    REQUIRES_RUPTURE_PARAMETERS = {'mag'}

    #: Required distance measure is RRup (eq. 1).
    REQUIRES_DISTANCES = {'rjb'}

    #: Required site parameter is only Vs30
    REQUIRES_SITES_PARAMETERS = {'vs30'}

    def compute(self, ctx: np.recarray, imts, mean, sig, tau, phi):

        for m, imt in enumerate(imts):

            for vs in ctx.vs30:
                if (vs >= 1E-10) & (vs < 185.0):
                    C = self.COEFFS_shallow_Vs30_150[imt]
                elif (vs == 185.0):
                    C = self.COEFFS_shallow_Vs30_185[imt]
                elif (vs > 185.0) & (vs < 365.0):
                    C = self.COEFFS_shallow_Vs30_260[imt]
                elif vs == 365.0:
                    C = self.COEFFS_shallow_Vs30_365[imt]
                elif vs == 428.0:
                    C = self.COEFFS_shallow_Vs30_428[imt]
                elif (vs > 365.0) & (vs < 760.0):
                    C = self.COEFFS_shallow_Vs30_530[imt]
                elif vs == 760.0:
                    C = self.COEFFS_shallow_Vs30_760[imt]
                elif (vs > 760.0) & (vs < 1500.0):
                    C = self.COEFFS_shallow_Vs30_1080[imt]
                elif (vs >= 1500.0):
                    C = self.COEFFS_shallow_Vs30_1500[imt]

            mean[m] = (C['C1'] +
                       _compute_magnitude(ctx, C) +
                       _compute_distance(ctx, C))

            if imt.string.startswith(('PGV')):
                sig[m] = 0
            else:
                sig[m] = C['SigmaTot']

    #: Coefficients from SA from Tables from Appendic C in Wong et al. 2022
    COEFFS_shallow_Vs30_150 = CoeffsTable(sa_damping=5, table="""
imt	Freq 	C1 	C2 	C4 	C5 	C6 	C7 	C8 	C10 	param_sigma	model_sigma	SigmaTot
10	0.1	-12.69233	1.64091	3.2	0	-1.89422	0.1493	0	-0.27992	0.3866	1.2756	1.332897
5	0.2	-8.77422	1.45379	3.5	0	-2.0482	0.13771	0	-0.33978	0.4094	1.1358	1.207332
3.0211	0.331	-4.66273	1.21469	3.8	0	-2.41811	0.13997	0	-0.34608	0.4781	0.9702	1.081604
1.996	0.501	-0.84421	0.99123	4	0	-2.84022	0.15017	0	-0.33357	0.5543	0.8636	1.026184
1.5848	0.631	2.81197	0.71563	4.2	0	-3.3263	0.18443	0	-0.3301	0.6183	0.768	0.9859609
1	1	13.51442	-0.32873	4.6	0	-4.82239	0.33458	0	-0.31854	0.699	0.6627	0.9632094
0.7413	1.349	21.65911	-1.31756	4.8	0	-5.9689	0.47585	0	-0.28891	0.7309	0.6566	0.9825164
0.5013	1.995	26.47325	-1.71937	4.9	0	-6.66762	0.52223	0	-0.24974	0.7253	0.5902	0.9350915
0.3981	2.512	29.74357	-2.08263	4.9	0	-7.23778	0.58194	0	-0.23657	0.7454	0.5655	0.9356343
0.302	3.311	37.07298	-2.95007	5	0	-8.36127	0.71381	0	-0.21387	0.7752	0.5581	0.955202
0.2399	4.169	46.5483	-3.9291	5.2	0	-9.75429	0.85504	0	-0.18908	0.7851	0.533	0.9489315
0.1995	5.012	44.60371	-3.85907	5.1	0	-9.52258	0.84894	0	-0.17647	0.7565	0.5209	0.9184929
0.1585	6.31	43.27945	-3.87522	5	0	-9.41691	0.86085	0	-0.16574	0.7485	0.5103	0.905902
0.1514	6.607	43.64014	-3.92958	5	0	-9.48336	0.8702	0	-0.16314	0.7502	0.5113	0.90787
0.1202	8.318	41.95651	-3.87512	4.9	0	-9.28916	0.86845	0	-0.15395	0.7574	0.5184	0.9178199
0.1	10	42.98182	-4.04797	4.9	0	-9.47691	0.89816	0	-0.14469	0.7638	0.4997	0.912738
0.0794	12.589	39.98228	-3.80803	4.8	0	-9.05806	0.86566	0	-0.13695	0.7455	0.4866	0.8902527
0.0692	14.454	39.77138	-3.79197	4.8	0	-9.04302	0.8651	0	-0.13572	0.7381	0.4887	0.8852228
0.0603	16.596	36.23857	-3.44597	4.7	0	-8.51344	0.8136	0	-0.13721	0.7353	0.4918	0.8846092
0.055	18.197	35.86794	-3.3996	4.7	0	-8.4612	0.80702	0	-0.13873	0.7284	0.4852	0.875206
0.0501	19.953	32.84651	-3.10617	4.6	0	-7.99857	0.76237	0	-0.14048	0.7249	0.489	0.8736687
0.0398	25.119	32.00853	-2.99776	4.6	0	-7.87069	0.74591	0	-0.1452	0.7171	0.4846	0.8654881
0.0324	30.903	31.11209	-2.87756	4.6	0	-7.7263	0.72676	0	-0.15116	0.7096	0.4793	0.8563064
0.0251	39.811	30.25219	-2.76067	4.6	0	-7.5846	0.70777	0	-0.15726	0.7018	0.4744	0.8471002
0.02	50.119	29.6859	-2.68299	4.6	0	-7.48962	0.69495	0	-0.16144	0.6974	0.4768	0.8448106
0.01	100	29.14579	-2.60861	4.6	0	-7.3986	0.68263	0	-0.16565	0.6932	0.4774	0.8416871
pga	-1 	29.05542	-2.59586	4.6	0	-7.37863	0.67997	0	-0.16529	0.6922	0.4774	0.8409
pgv	-2 	16.07683	-0.83229	3.9	0	-4.79022	0.45682	0	-0.20432	0.5283
    """)

    COEFFS_shallow_Vs30_185 = CoeffsTable(sa_damping=5, table="""
imt	Freq 	C1 	C2 	C4 	C5 	C6 	C7 	C8 	C10 	param_sigma	model_sigma	SigmaTot
10	0.1	-13.19364	1.68037	3.1	0	-1.83593	0.14544	0	-0.28148	0.3724	1.2756	1.328848
5	0.2	-8.79408	1.43896	3.5	0	-2.06632	0.14222	0	-0.34248	0.381	1.1358	1.197999
3.0211	0.331	-4.28488	1.11086	3.8	0	-2.51048	0.15977	0	-0.35034	0.4421	0.9702	1.06618
1.996	0.501	-0.47932	0.87423	4	0	-2.93176	0.17056	0	-0.33456	0.5276	0.8636	1.012011
1.5848	0.631	1.57935	0.78998	4.1	0	-3.1502	0.16996	0	-0.32264	0.586	0.768	0.9660332
1	1	8.3541	0.34371	4.4	0	-4.04345	0.21998	0	-0.29742	0.6509	0.6627	0.928893
0.7413	1.349	17.28152	-0.52518	4.7	0	-5.34888	0.35015	0	-0.28162	0.6864	0.6566	0.9498782
0.5013	1.995	26.75557	-1.6281	4.9	0	-6.7129	0.50911	0	-0.24583	0.7187	0.5902	0.9299817
0.3981	2.512	31.18577	-1.99195	5	0	-7.40066	0.55927	0	-0.22335	0.7165	0.5655	0.9127774
0.302	3.311	31.47319	-2.10584	4.9	0	-7.50354	0.57767	0	-0.20194	0.7658	0.5581	0.9475892
0.2399	4.169	37.42274	-2.73145	5	0	-8.43049	0.67351	0	-0.18545	0.7621	0.533	0.9299921
0.1995	5.012	39.52031	-3.02774	5	0	-8.78509	0.72108	0	-0.17329	0.7624	0.5209	0.9233584
0.1585	6.31	41.67412	-3.3466	5	0	-9.15102	0.77233	0	-0.16017	0.7725	0.5103	0.9258307
0.1514	6.607	42.04087	-3.39681	5	0	-9.22402	0.78164	0	-0.15942	0.7686	0.5113	0.9231326
0.1202	8.318	40.79807	-3.42792	4.9	0	-9.11464	0.79508	0	-0.15423	0.7561	0.5184	0.9167474
0.1	10	38.68828	-3.33067	4.8	0	-8.83502	0.78481	0	-0.14727	0.7377	0.4997	0.8910115
0.0794	12.589	36.4883	-3.22079	4.7	0	-8.54079	0.77318	0	-0.14298	0.7387	0.4866	0.8845662
0.0692	14.454	36.6643	-3.27179	4.7	0	-8.5856	0.78298	0	-0.13869	0.7368	0.4887	0.8841392
0.0603	16.596	36.26675	-3.23525	4.7	0	-8.53532	0.77836	0	-0.13732	0.7283	0.4918	0.8787993
0.055	18.197	33.34174	-2.97142	4.6	0	-8.09078	0.73865	0	-0.13824	0.728	0.4852	0.8748732
0.0501	19.953	33.0138	-2.9344	4.6	0	-8.045	0.73351	0	-0.13942	0.7179	0.489	0.8686204
0.0398	25.119	29.65054	-2.60564	4.5	0	-7.52549	0.68302	0	-0.14336	0.7076	0.4846	0.8576334
0.0324	30.903	28.8738	-2.50642	4.5	0	-7.40161	0.66727	0	-0.14758	0.701	0.4793	0.8491934
0.0251	39.811	27.84722	-2.37095	4.5	0	-7.2314	0.64501	0	-0.15345	0.6911	0.4744	0.838257
0.02	50.119	27.23831	-2.28934	4.5	0	-7.12908	0.63145	0	-0.15728	0.6856	0.4768	0.8350962
0.01	100	26.58137	-2.20049	4.5	0	-7.01758	0.61659	0	-0.16184	0.6798	0.4774	0.8306858
pga	-1 	26.45904	-2.18418	4.5	0	-6.99144	0.61318	0	-0.16145	0.6782	0.4774	0.8294
pgv	-2 	15.35176	-0.78585	3.8	0	-4.71966	0.45311	0	-0.19689	0.5031

    """)

    COEFFS_shallow_Vs30_260 = CoeffsTable(sa_damping=5, table="""
imt	Freq 	C1 	C2 	C4 	C5 	C6 	C7 	C8 	C10 	param_sigma	model_sigma	SigmaTot
10	0.1	-14.35371	1.79484	2.9	0	-1.66745	0.13007	0	-0.28825	0.3861	1.2756	1.332752
5	0.2	-9.63654	1.51074	3.4	0	-1.94809	0.13306	0	-0.34759	0.4125	1.1358	1.208387
3.0211	0.331	-4.52783	1.11009	3.8	0	-2.49561	0.16327	0	-0.35676	0.4723	0.9702	1.079053
1.996	0.501	-0.30511	0.77396	4	0	-2.99194	0.19132	0	-0.34351	0.5085	0.8636	0.1002186
1.5848	0.631	2.01423	0.62282	4.1	0	-3.25746	0.20131	0	-0.33096	0.5454	0.768	0.9419583
1	1	7.07371	0.32481	4.3	0	-3.87826	0.22334	0	-0.29361	0.6465	0.6627	0.9258151
0.7413	1.349	10.51269	0.10274	4.4	0	-4.34111	0.24548	0	-0.26396	0.6921	0.6566	0.9540053
0.5013	1.995	13.49041	-0.07927	4.4	0	-4.74109	0.26056	0	-0.22238	0.7132	0.5902	0.9257377
0.3981	2.512	19.49423	-0.57854	4.6	0	-5.64339	0.33567	0	-0.20571	0.7284	0.5655	0.922148
0.302	3.311	24.3216	-1.07113	4.7	0	-6.39802	0.40954	0	-0.18472	0.7542	0.5581	0.9382394
0.2399	4.169	29.2079	-1.52145	4.8	0	-7.18386	0.48002	0	-0.16958	0.795	0.533	0.9571385
0.1995	5.012	31.05221	-1.75437	4.8	0	-7.49281	0.51675	0	-0.15865	0.8041	0.5209	0.9580781
0.1585	6.31	30.56005	-1.84421	4.7	0	-7.47969	0.53572	0	-0.15006	0.8111	0.5103	0.9582741
0.1514	6.607	31.01753	-1.90066	4.7	0	-7.56496	0.54571	0	-0.14872	0.8054	0.5113	0.95399
0.1202	8.318	33.39284	-2.24085	4.7	0	-7.98795	0.60413	0	-0.143	0.7864	0.5184	0.9418936
0.1	10	34.64468	-2.46235	4.7	0	-8.21594	0.64198	0	-0.13782	0.7909	0.4997	0.9355335
0.0794	12.589	32.41158	-2.35857	4.6	0	-7.92393	0.63118	0	-0.1351	0.7903	0.4866	0.9280915
0.0692	14.454	30.22104	-2.21542	4.5	0	-7.60779	0.61167	0	-0.13481	0.7777	0.4887	0.9185015
0.0603	16.596	28.14254	-2.08659	4.4	0	-7.30054	0.5939	0	-0.13579	0.7706	0.4918	0.9141618
0.055	18.197	28.3528	-2.13895	4.4	0	-7.3468	0.60392	0	-0.13581	0.7586	0.4852	0.900496
0.0501	19.953	28.30711	-2.15428	4.4	0	-7.34798	0.60734	0	-0.13525	0.7583	0.489	0.9022971
0.0398	25.119	27.49727	-2.09579	4.4	0	-7.22715	0.59882	0	-0.13457	0.7431	0.4846	0.8871499
0.0324	30.903	24.50783	-1.82495	4.3	0	-6.75313	0.55607	0	-0.1386	0.7317	0.4793	0.8747076
0.0251	39.811	23.67945	-1.7301	4.3	0	-6.61713	0.54052	0	-0.14177	0.7221	0.4744	0.8639929
0.02	50.119	23.09489	-1.65994	4.3	0	-6.51861	0.52873	0	-0.14419	0.7175	0.4768	0.8614781
0.01	100	22.39352	-1.5731	4.3	0	-6.39882	0.51398	0	-0.14734	0.7119	0.4774	0.8571537
pga	-1 	22.30187	-1.5588	4.3	0	-6.37736	0.51072	0	-0.14672	0.71144	0.4774	0.8578
pgv	-2 	14.66968	-0.75914	3.7	0	-4.70358	0.45882	0	-0.18164	0.5243
    """)

    COEFFS_shallow_Vs30_365 = CoeffsTable(sa_damping=5, table="""
imt	Freq 	C1 	C2 	C4 	C5 	C6 	C7 	C8 	C10 	param_sigma	model_sigma	SigmaTot
10	0.1	-14.90354	1.85488	2.9	0	-1.58566	0.1203	0	-0.29728	0.3831	1.2756	1.331886
5	0.2	-10.47331	1.59148	3.3	0	-1.81477	0.12087	0	-0.35424	0.4161	1.1358	1.20962
3.0211	0.331	-5.00108	1.16029	3.8	0	-2.42774	0.15631	0	-0.36186	0.48	0.9702	1.082445
1.996	0.501	-0.73342	0.81024	4	0	-2.93365	0.18683	0	-0.34845	0.5113	0.8636	1.00361
1.5848	0.631	1.68809	0.63636	4.1	0	-3.21847	0.20079	0	-0.33559	0.5477	0.768	0.9432918
1	1	7.13072	0.24902	4.3	0	-3.91056	0.23837	0	-0.29855	0.6116	0.6627	0.9017904
0.7413	1.349	10.79472	-0.03598	4.4	0	-4.42784	0.27156	0	-0.26869	0.6393	0.6566	0.9164214
0.5013	1.995	13.92087	-0.30043	4.4	0	-4.87604	0.3013	0	-0.22558	0.7179	0.5902	0.9293635
0.3981	2.512	17.15407	-0.49325	4.5	0	-5.35135	0.32506	0	-0.20107	0.7612	0.5655	0.9482699
0.302	3.311	21.41014	-0.82771	4.6	0	-6.00587	0.37333	0	-0.1793	0.7875	0.5581	0.9652109
0.2399	4.169	23.37461	-1.02097	4.6	0	-6.34039	0.40334	0	-0.16322	0.8046	0.533	0.965127
0.1995	5.012	24.938	-1.17987	4.6	0	-6.60933	0.42943	0	-0.15331	0.8314	0.5209	0.9811029
0.1585	6.31	27.34521	-1.48185	4.6	0	-7.00808	0.47891	0	-0.1448	0.8387	0.5103	0.9817453
0.1514	6.607	27.89548	-1.55521	4.6	0	-7.10133	0.49126	0	-0.14366	0.8258	0.5113	0.9712741
0.1202	8.318	29.53385	-1.78086	4.6	0	-7.41676	0.53109	0	-0.13617	0.7994	0.5184	0.9527743
0.1	10	28.2026	-1.74082	4.5	0	-7.25641	0.52914	0	-0.13329	0.8002	0.4997	0.9434088
0.0794	12.589	27.30592	-1.78899	4.4	0	-7.16301	0.54227	0	-0.13319	0.8081	0.4866	0.9432949
0.0692	14.454	27.96499	-1.9181	4.4	0	-7.2942	0.56582	0	-0.13173	0.7895	0.4887	0.9285139
0.0603	16.596	25.90233	-1.80073	4.3	0	-6.99013	0.54985	0	-0.13201	0.7867	0.4918	0.9277738
0.055	18.197	25.9192	-1.83006	4.3	0	-7.00435	0.55594	0	-0.13239	0.7907	0.4852	0.9276991
0.0501	19.953	23.88298	-1.68952	4.2	0	-6.68792	0.53463	0	-0.13309	0.7824	0.489	0.9226434
0.0398	25.119	23.52147	-1.70415	4.2	0	-6.65023	0.53957	0	-0.13434	0.756	0.4846	0.8979828
0.0324	30.903	22.75054	-1.64369	4.2	0	-6.53022	0.53053	0	-0.13679	0.749	0.4793	0.8892297
0.0251	39.811	21.94363	-1.566	4.2	0	-6.3975	0.51788	0	-0.13971	0.7383	0.4744	0.8775776
0.02	50.119	21.37331	-1.50684	4.2	0	-6.30212	0.50803	0	-0.14149	0.7311	0.4768	0.8728377
0.01	100	20.64147	-1.42407	4.2	0	-6.17683	0.49391	0	-0.1441	0.7238	0.4774	0.8670624
pga	-1 	20.58466	-1.41581	4.2	0	-6.16192	0.49169	0	-0.14324	0.7231	0.4774	0.8665
pgv	-2 	13.12894	-0.62021	3.6	0	-4.50438	0.44333	0	-0.18596	0.5198
    """)

    COEFFS_shallow_Vs30_428 = CoeffsTable(sa_damping=5, table="""
imt	Freq 	C1 	C2 	C4 	C5 	C6 	C7 	C8 	C10 	param_sigma	model_sigma	SigmaTot
10	0.1	-15.19784	1.90028	2.9	0	-1.54536	0.11537	0	-0.3012	0.3726	1.2756	1.328904
5	0.2	-10.71071	1.61659	3.3	0	-1.78403	0.11728	0	-0.35703	0.391	1.1358	1.201217
3.0211	0.331	-5.24628	1.1842	3.8	0	-2.39631	0.15279	0	-0.36419	0.4532	0.9702	1.070831
1.996	0.501	-0.9783	0.83131	4	0	-2.90421	0.18404	0	-0.35075	0.4919	0.8636	0.9938666
1.5848	0.631	1.48968	0.64607	4.1	0	-3.20088	0.20003	0	-0.33732	0.5343	0.768	0.935575
1	1	7.05101	0.22829	4.3	0	-3.92056	0.24294	0	-0.2992	0.6426	0.6627	0.9230959
0.7413	1.349	10.66644	-0.04191	4.4	0	-4.4185	0.27296	0	-0.26955	0.6368	0.6566	0.9146791
0.5013	1.995	15.40632	-0.43346	4.4	0	-5.13011	0.32339	0	-0.22784	0.6466	0.5902	0.8754586
0.3981	2.512	17.27335	-0.59059	4.5	0	-5.41213	0.34345	0	-0.20377	0.7065	0.5655	0.9049489
0.302	3.311	19.41612	-0.75243	4.6	0	-5.73872	0.36436	0	-0.17824	0.7426	0.5581	0.9289405
0.2399	4.169	21.22878	-0.88364	4.6	0	-6.01767	0.38218	0	-0.16204	0.7578	0.533	0.9264718
0.1995	5.012	24.83756	-1.19754	4.6	0	-6.60747	0.43276	0	-0.1527	0.788	0.5209	0.9446063
0.1585	6.31	24.65807	-1.2831	4.6	0	-6.62166	0.44902	0	-0.14261	0.8033	0.5103	0.9516812
0.1514	6.607	25.00269	-1.32047	4.6	0	-6.68548	0.45564	0	-0.14106	0.8034	0.5113	0.9523021
0.1202	8.318	26.72766	-1.5579	4.6	0	-7.00238	0.49733	0	-0.13575	0.8355	0.5184	0.9832593
0.1	10	25.42663	-1.52143	4.5	0	-6.86383	0.49534	0	-0.13222	0.8147	0.4997	0.9557386
0.0794	12.589	26.97275	-1.74579	4.4	0	-7.13266	0.53623	0	-0.13127	0.8225	0.4866	0.9556599
0.0692	14.454	25.42663	-1.70772	4.4	0	-6.91502	0.53347	0	-0.13121	0.8092	0.4887	0.9453213
0.0603	16.596	25.3101	-1.74063	4.3	0	-6.92114	0.54121	0	-0.13069	0.779	0.4918	0.9212536
0.055	18.197	23.21994	-1.58948	4.3	0	-6.59887	0.51833	0	-0.13128	0.7643	0.4852	0.905303
0.0501	19.953	23.09248	-1.59088	4.2	0	-6.58967	0.5199	0	-0.13283	0.7577	0.489	0.9017929
0.0398	25.119	21.23085	-1.49661	4.2	0	-6.30799	0.50735	0	-0.13454	0.7445	0.4846	0.8883228
0.0324	30.903	20.66127	-1.46626	4.2	0	-6.22043	0.50331	0	-0.13658	0.7405	0.4793	0.8820821
0.0251	39.811	20.04865	-1.41386	4.2	0	-6.12221	0.49519	0	-0.1396	0.7309	0.4744	0.8713611
0.02	50.119	19.50112	-1.36578	4.2	0	-6.03003	0.4872	0	-0.14119	0.7221	0.4768	0.8653131
0.01	100	18.73341	-1.2838	4.2	0	-5.89812	0.4732	0	-0.144	0.7128	0.4774	0.8579013
pga	-1 	18.67909	-1.2768	4.2	0	-5.88336	0.47114	0	-0.14312	0.7121	0.4774	0.8573
pgv	-2 	12.77377	-0.59006	3.6	0	-4.47669	0.44189	0	-0.18818	0.5039
    """)

    COEFFS_shallow_Vs30_530 = CoeffsTable(sa_damping=5, table="""
imt	Freq 	C1 	C2 	C4 	C5 	C6 	C7 	C8 	C10 	param_sigma	model_sigma	SigmaTot
10	0.1	-15.7025	1.95736	2.8	0	-1.45569	0.10622	0	-0.30744	0.3528	1.2756	1.323489
5	0.2	-10.93376	1.65285	3.3	0	-1.74195	0.11191	0	-0.36216	0.3634	1.1358	1.192519
3.0211	0.331	-6.04163	1.26021	3.7	0	-2.25721	0.14099	0	-0.36847	0.4005	0.9702	1.049613
1.996	0.501	-1.2924	0.87728	4	0	-2.84878	0.17718	0	-0.35395	0.456	0.8636	0.9765967
1.5848	0.631	1.25369	0.67569	4.1	0	-3.16012	0.19617	0	-0.34026	0.517	0.768	0.9258041
1	1	6.85477	0.23616	4.3	0	-3.8988	0.2431	0	-0.302	0.5824	0.6627	0.8822478
0.7413	1.349	10.56733	-0.07379	4.4	0	-4.42871	0.28097	0	-0.27186	0.6668	0.6566	0.935813
0.5013	1.995	13.97374	-0.41849	4.4	0	-4.93825	0.32546	0	-0.23079	0.6973	0.5902	0.9135445
0.3981	2.512	15.88206	-0.61695	4.4	0	-5.24764	0.3538	0	-0.20725	0.6952	0.5655	0.8961548
0.302	3.311	18.03911	-0.80805	4.4	0	-5.59024	0.3802	0	-0.18076	0.7333	0.5581	0.9215229
0.2399	4.169	19.70215	-0.92336	4.4	0	-5.8474	0.3959	0	-0.16301	0.7848	0.533	0.9486834
0.1995	5.012	21.02148	-1.03819	4.4	0	-6.07017	0.41379	0	-0.15168	0.7844	0.5209	0.9416051
0.1585	6.31	24.64531	-1.30643	4.5	0	-6.65529	0.45646	0	-0.14173	0.8184	0.5103	0.9644608
0.1514	6.607	25.00777	-0.134016	4.5	0	-6.7164	0.46209	0	-0.14037	0.8251	0.5113	0.970679
0.1202	8.318	24.72683	-1.40483	4.4	0	-6.71801	0.47609	0	-0.1342	0.7873	0.5184	0.9426452
0.1	10	26.01249	-1.56856	4.4	0	-6.96623	0.50638	0	-0.132	0.765	0.4997	0.9137424
0.0794	12.589	27.27882	-1.77106	4.4	0	-7.21109	0.54332	0	-0.13012	0.7782	0.4866	0.9178098
0.0692	14.454	25.50016	-1.68058	4.3	0	-6.95687	0.53189	0	-0.12961	0.7805	0.4887	0.9208735
0.0603	16.596	23.72271	-1.5875	4.2	0	-6.69752	0.5198	0	-0.13007	0.7831	0.4918	0.9247232
0.055	18.197	23.79793	-1.62919	4.2	0	-6.72667	0.52856	0	-0.13097	0.7783	0.4852	0.9171532
0.0501	19.953	22.02051	-1.5193	4.1	0	-6.45252	0.51236	0	-0.13142	0.7736	0.489	0.9151928
0.0398	25.119	21.84634	-1.56141	4.1	0	-6.44792	0.52224	0	-0.1335	0.7488	0.4846	0.8919298
0.0324	30.903	19.61937	-1.40448	4	0	-6.09483	0.49812	0	-0.1359	0.7313	0.4793	0.8743731
0.0251	39.811	19.06739	-1.37853	4	0	-6.00962	0.49485	0	-0.13833	0.7209	0.4744	0.8629903
0.02	50.119	18.56849	-1.33799	4	0	-5.92705	0.48832	0	-0.14006	0.7112	0.4768	0.8562381
0.01	100	19.27022	-1.38064	4.1	0	-6.03413	0.49464	0	-0.1428	0.7009	0.4774	0.8480399
pga	-1 	19.25381	-1.37968	4.1	0	-6.02726	-6.02726	0	-0.14172	0.6998	0.4774	0.8471
pgv	-2 	12.31387	-0.53167	3.6	0	-4.42465	-4.42465	0	-0.19744	0.4784
    """)

    COEFFS_shallow_Vs30_760 = CoeffsTable(sa_damping=5, table="""
imt	Freq 	C1 	C2 	C4 	C5 	C6 	C7 	C8 	C10 	param_sigma	model_sigma	SigmaTot
10	0.1	-16.07739	1.99692	2.8	0	-1.41449	0.10118	0	-0.31138	0.3618	1.2756	1.325916
5	0.2	-11.21796	1.67885	3.3	0	-1.71272	0.10846	0	-0.36544	0.3777	1.1358	1.196954
3.0211	0.331	-6.31976	1.27385	3.7	0	-2.23711	0.13967	0	-0.37034	0.4343	0.9702	1.06297
1.996	0.501	-1.59169	0.89174	4	0	-2.82243	0.17611	0	-0.35661	0.4767	0.8636	0.9864319
1.5848	0.631	0.95326	0.68439	4.1	0	-3.13975	0.19606	0	-0.3425	0.5154	0.768	0.9249114
1	1	6.70174	0.1992	4.3	0	-3.93011	0.25129	0	-0.30248	0.5781	0.6627	0.8794152
0.7413	1.349	10.48506	-0.11811	4.4	0	-4.47259	0.29008	0	-0.27125	0.6092	0.6566	0.8956831
0.5013	1.995	13.7448	-0.42077	4.4	0	-4.94146	0.32682	0	-0.22914	0.6643	0.5902	0.8886116
0.3981	2.512	15.57489	-0.57308	4.4	0	-5.20939	0.34553	0	-0.20499	0.7041	0.5655	0.9030765
0.302	3.311	19.42413	-0.86915	4.5	0	-5.81266	0.38841	0	-0.18	0.7329	0.5581	0.9212047
0.2399	4.169	21.19987	-1.01613	4.5	0	-6.10069	0.40997	0	-0.16275	0.7529	0.533	0.9224681
0.1995	5.012	22.62015	-1.14773	4.5	0	-6.34543	0.43119	0	-0.15172	0.7597	0.5209	0.9211303
0.1585	6.31	22.35435	-1.19335	4.4	0	-6.34551	0.44127	0	-0.14128	0.7667	0.5103	0.9209968
0.1514	6.607	22.66839	-1.22722	4.4	0	-6.40412	0.44731	0	-0.13957	0.7656	0.5113	0.9206363
0.1202	8.318	24.12291	-1.39341	4.4	0	-6.67951	0.47756	0	-0.13311	0.7725	0.5184	0.9303197
0.1	10	23.18347	-1.37453	4.3	0	-6.56811	0.47798	0	-0.12976	0.7558	0.4997	0.906054
0.0794	12.589	22.31324	-1.36426	4.2	0	-6.47038	0.48047	0	-0.12769	0.7454	0.4866	0.890169
0.0692	14.454	22.69928	-1.4341	4.2	0	-6.55795	0.49446	0	-0.12767	0.7462	0.4887	0.8919878
0.0603	16.596	21.11329	-1.35791	4.1	0	-6.32642	0.48488	0	-0.12833	0.7426	0.4918	0.8906863
0.055	18.197	21.11927	-1.38114	4.1	0	-6.34191	0.49021	0	-0.12885	0.738	0.4852	0.8832118
0.0501	19.953	19.49197	-1.26593	4	0	-6.08766	0.47253	0	-0.12931	0.7392	0.489	0.8863057
0.0398	25.119	18.06259	-1.18715	3.9	0	-5.87261	0.46191	0	-0.13111	0.7555	0.4846	0.897562
0.0324	30.903	17.92045	-1.20062	3.9	0	-5.86142	0.4657	0	-0.13299	0.7549	0.4793	0.8942049
0.0251	39.811	17.40417	-1.19515	3.9	0	-5.78455	0.46629	0	-0.13621	0.7253	0.4744	0.8666692
0.02	50.119	16.82685	-1.16893	3.9	0	-5.68937	0.46249	0	-0.13895	0.7144	0.4768	0.8588979
0.01	100	17.24628	-1.21208	4	0	-5.7507	0.46938	0	-0.14292	0.6878	0.4774	0.8372452
pga	-1 	17.13204	-1.20352	4	0	-5.7267	0.46719	0	-0.14231	0.6844	0.4774	0.8345
pgv	-2 	10.89224	-0.40124	3.5	0	-4.23673	0.41962	0	-0.1976	0.4824
    """)

    COEFFS_shallow_Vs30_1080 = CoeffsTable(sa_damping=5, table="""
imt	Freq 	C1 	C2 	C4 	C5 	C6 	C7 	C8 	C10 	param_sigma	model_sigma	SigmaTot
10	0.1	-16.2567	2.02096	2.8	0	-1.38545	0.09737	0	-0.31511	0.3571	1.2756	1.324642
5	0.2	-11.36847	1.69993	3.3	0	-1.68801	0.10522	0	-0.36887	0.3719	1.1358	1.195137
3.0211	0.331	-6.48545	1.29604	3.7	0	-2.21	0.1363	0	-0.37332	0.4221	0.9702	1.058044
1.996	0.501	-1.77648	0.92028	4	0	-2.78726	0.17173	0	-0.35924	0.4707	0.8636	0.9835464
1.5848	0.631	0.76984	0.71363	4.1	0	-3.10407	0.19161	0	-0.34477	0.5035	0.768	0.9183335
1	1	6.50828	0.21606	4.3	0	-3.90645	0.24934	0	-0.30449	0.5635	0.6627	0.8698871
0.7413	1.349	10.29307	-0.10381	4.4	0	-4.44633	0.28846	0	-0.27277	0.6084	0.6566	0.8951392
0.5013	1.995	13.61576	-0.42921	4.4	0	-4.93645	0.3297	0	-0.23108	0.6505	0.5902	0.878343
0.3981	2.512	15.50359	-0.60408	4.4	0	-5.22168	0.35276	0	-0.20699	0.6761	0.5655	0.8814202
0.302	3.311	17.72611	-0.82521	4.4	0	-5.58438	0.38536	0	-0.18188	0.6932	0.5581	0.8899449
0.2399	4.169	19.47596	-1.00146	4.4	0	-5.88134	0.41307	0	-0.16456	0.7144	0.533	0.8913228
0.1995	5.012	20.81498	-1.13064	4.4	0	-6.11272	0.4341	0	-0.15315	0.7331	0.5209	0.8993178
0.1585	6.31	22.40769	-1.28594	4.4	0	-6.39676	0.46065	0	-0.14218	0.7481	0.5103	0.9055715
0.1514	6.607	22.71156	-1.31593	4.4	0	-6.45211	0.46592	0	-0.14039	0.7516	0.5113	0.9090271
0.1202	8.318	22.21752	-1.31371	4.3	0	-6.40947	0.46841	0	-0.13333	0.7662	0.5184	0.9250952
0.1	10	23.22242	-1.40958	4.3	0	-6.59942	0.48626	0	-0.12959	0.7733	0.4997	0.9207025
0.0794	12.589	22.42362	-1.37659	4.2	0	-6.50795	0.4843	0	-0.12683	0.782	0.4866	0.921034
0.0692	14.454	22.95399	-1.43367	4.2	0	-6.61553	0.49558	0	-0.12598	0.7823	0.4887	0.9223997
0.0603	16.596	21.58647	-1.35098	4.1	0	-6.41588	0.48428	0	-0.12568	0.779	0.4918	0.9212536
0.055	18.197	21.79868	-1.38267	4.1	0	-6.46452	0.49084	0	-0.12575	0.7758	0.4852	0.9150326
0.0501	19.953	20.28014	-1.28304	4	0	-6.2299	0.47591	0	-0.12603	0.7715	0.489	0.9134184
0.0398	25.119	18.87089	-1.2177	3.9	0	-6.02753	0.46829	0	-0.12745	0.759	0.4846	0.90051
0.0324	30.903	17.36066	-1.1335	3.8	0	-5.79682	0.45656	0	-0.12923	0.7492	0.4793	0.8893982
0.0251	39.811	17.15664	-1.15033	3.8	0	-5.77816	0.46133	0	-0.13161	0.7446	0.4744	0.8828843
0.02	50.119	15.36392	-1.03981	3.7	0	-5.48466	0.44406	0	-0.1343	0.7246	0.4768	0.8674004
0.01	100	16.49999	-1.18349	3.9	0	-5.6669	0.46861	0	-0.1402	0.6921	0.4774	0.8407814
pga	-1 	16.31762	-1.1733	3.9	0	-5.63243	0.46634	0	-0.13988	0.6881	0.4774	0.8346
pgv	-2 	10.55	-0.35673	3.5	0	-4.19532	0.41473	0	-0.2055	0.4691
    """)

    COEFFS_shallow_Vs30_1500 = CoeffsTable(sa_damping=5, table="""
imt	Freq 	C1 	C2 	C4 	C5 	C6 	C7 	C8 	C10 	param_sigma	model_sigma	SigmaTot
10	0.1	-16.44065	2.03994	2.8	0	-1.37085	0.09545	0	-0.31641	0.3583	1.2756	1.324966
5	0.2	-11.48095	1.70249	3.3	0	-1.68898	0.1055	0	-0.36877	0.3717	1.1358	1.195074
3.0211	0.331	-6.60644	1.2842	3.7	0	-2.2251	0.13914	0	-0.37189	0.4119	0.9702	1.054016
1.996	0.501	-1.86329	0.90457	4	0	-2.80543	0.1749	0	-0.35821	0.4643	0.8636	0.9804997
1.5848	0.631	0.72537	0.68975	4.1	0	-3.13014	0.19575	0	-0.34335	0.4991	0.768	0.9159284
1	1	6.42075	0.21818	4.3	0	-3.90468	0.24863	0	-0.30345	0.5664	0.6627	0.8717685
0.7413	1.349	10.17112	-0.09873	4.4	0	-4.44264	0.28767	0	-0.27274	0.6054	0.6566	0.8931029
0.5013	1.995	13.47447	-0.40616	4.4	0	-4.91804	0.32501	0	-0.23021	0.6524	0.5902	0.8797511
0.3981	2.512	15.44042	-0.61296	4.4	0	-5.23324	0.3545	0	-0.20656	0.6731	0.5655	0.8791211
0.302	3.311	17.6789	-0.83551	4.4	0	-5.59778	0.38728	0	-0.1811	0.6988	0.5581	0.8943138
0.2399	4.169	19.41548	-0.99937	4.4	0	-5.88699	0.41248	0	-0.16369	0.7202	0.533	0.8959783
0.1995	5.012	20.77758	1.13487	4.4	0	-6.12493	0.43478	0	-0.15227	0.7345	0.5209	0.9004594
0.1585	6.31	22.38266	1.29975	4.4	0	-6.41524	0.46328	0	-0.14141	0.749	0.5103	0.9063151
0.1514	6.607	22.68901	1.32638	4.4	0	-6.4694	0.46785	0	-0.13958	0.7527	0.5113	0.9099369
0.1202	8.318	22.19347	1.32914	4.3	0	-6.42884	0.47137	0	-0.13256	0.7644	0.5184	0.9236049
0.1	10	23.19162	1.42897	4.3	0	-6.61927	0.49003	0	-0.12886	0.7717	0.4997	0.9193591
0.0794	12.589	22.38657	1.39567	4.2	0	-6.52703	0.48803	0	-0.12604	0.7785	0.4866	0.9180643
0.0692	14.454	21.1486	1.31533	4.1	0	-6.34677	0.47686	0	-0.12508	0.7817	0.4887	0.9218909
0.0603	16.596	21.6033	1.36136	4.1	0	-6.43987	0.48613	0	-0.12454	0.7841	0.4918	0.9255702
0.055	18.197	21.87417	1.38906	4.1	0	-6.496	0.49177	0	-0.12436	0.7859	0.4852	0.9236113
0.0501	19.953	20.42827	1.28535	4	0	-6.27101	0.47585	0	-0.12432	0.7875	0.489	0.9269721
0.0398	25.119	19.309	1.21542	3.9	0	-6.11174	0.46667	0	-0.12465	0.7922	0.4846	0.9286647
0.0324	30.903	18.06549	1.13695	3.8	0	-5.92459	0.45551	0	-0.12545	0.7966	0.4793	0.9296774
0.0251	39.811	18.05173	1.16741	3.8	0	-5.94216	0.46273	0	-0.12707	0.7977	0.4744	0.928106
0.02	50.119	16.32087	1.07317	3.7	0	-5.6651	0.44866	0	-0.12936	0.782	0.4768	0.9158943
0.01	100	15.66781	1.11454	3.8	0	-5.5543	0.45726	0	-0.13663	0.713	0.4774	0.8580675
pga	-1 	15.39305	1.10031	3.8	0	-5.50442	0.4544	0	-0.13677	0.7055	0.4774	0.8518
pgv	-2 	10.5337	-0.37189	3.5	0	-4.23064	0.42027	0	-0.20041	0.4729
    """)

    COEFFS_deep_Vs30_150 = CoeffsTable(sa_damping=5, table="""
imt	Freq 	C1 	C2 	C4 	C5 	C6 	C7 	C8 	C10 	param_sigma	model_sigma	SigmaTot
10	0.1	4.37553	0.77975	5.8	0	-4.59986	0.30767	0	-0.14658	0.2813	1.2756	1.30625
5	0.2	6.55163	1.33405	5.9	0	-4.46379	0.19195	0	-0.223	0.3285	1.1358	1.18235
3.0211	0.331	11.38311	1.68224	6	0	-4.85417	0.10946	0	-0.26454	0.3719	0.9702	1.03904
1.996	0.501	30.98126	0.54258	6.3	0	-7.3971	0.25264	0	-0.30809	0.4095	0.8636	0.95577
1.5848	0.631	49.92847	-1.04082	6.5	0	-9.86362	0.46506	0	-0.33988	0.45	0.768	0.89013
1	1	84.68634	-4.87454	6.7	0	-14.29627	0.9698	0	-0.37189	0.5312	0.6627	0.84932
0.7413	1.349	119.7338	-8.69895	6.9	0	-18.67737	1.45506	0	-0.35283	0.5612	0.6566	0.86375
0.5013	1.995	135.0739	-10.27271	6.9	0	-20.71831	1.65558	0	-0.32696	0.6008	0.5902	0.8422
0.3981	2.512	153.2049	-12.41574	6.9	0	-23.23709	1.95012	0	-0.32053	0.5926	0.5655	0.81912
0.302	3.311	233.3452	-20.53758	7.2	0	-33.22908	2.97031	0	-0.28776	0.6288	0.5581	0.84075
0.2399	4.169	301.7976	-27.35859	7.4	0	-41.47969	3.79423	0	-0.24674	0.6299	0.533	0.82514
0.1995	5.012	284.1134	-26.06946	7.3	0	-39.59374	3.6627	0	-0.2195	0.6197	0.5209	0.80955
0.1585	6.31	221.0541	-20.66703	7	0	-32.2576	3.04183	0	-0.19271	0.6396	0.5103	0.81823
0.1514	6.607	223.8744	-21.03829	7	0	-32.65327	3.09331	0	-0.18668	0.6389	0.5113	0.8183
0.1202	8.318	213.0411	-20.43183	6.9	0	-31.52074	3.04296	0	-0.1592	0.6456	0.5184	0.82797
0.1	10	220.6796	-21.50187	6.9	0	-32.60587	3.19304	0	-0.13747	0.6484	0.4997	0.81861
0.0794	12.589	184.5597	-18.16765	6.7	0	-28.19168	2.78915	0	-0.11745	0.6407	0.4866	0.80453
0.0692	14.454	169.095	-16.73068	6.6	0	-26.2848	2.61423	0	-0.11189	0.6268	0.4887	0.7948
0.0603	16.596	153.6647	-15.23057	6.5	0	-24.33881	2.42658	0	-0.11226	0.6204	0.4918	0.79168
0.055	18.197	138.9711	-13.76625	6.4	0	-22.43781	2.23752	0	-0.11217	0.6218	0.4852	0.7887
0.0501	19.953	138.4665	-13.70573	6.4	0	-22.37661	2.23038	0	-0.1138	0.6213	0.489	0.79065
0.0398	25.119	112.5185	-11.07698	6.2	0	-18.94926	1.88358	0	-0.12076	0.6174	0.4846	0.78487
0.0324	30.903	109.6691	-10.69246	6.2	0	-18.5375	1.82857	0	-0.13173	0.6055	0.4793	0.77224
0.0251	39.811	106.448	-10.25181	6.2	0	-18.0633	1.76436	0	-0.14413	0.5971	0.4744	0.76262
0.02	50.119	104.0392	-9.92014	6.2	0	-17.70504	1.71552	0	-0.153	0.5876	0.4768	0.75671
0.01	100	101.6177	-9.58564	6.2	0	-17.34384	1.66611	0	-0.16184	0.5783	0.4774	0.74989
pga	-1 	101.4428	-9.5506	6.2	0	-17.31475	1.66075	0	-0.16232	0.5788	0.4774	0.75028
pgv	-2 	52.29073	-3.78249	5.7	0	-10.07493	0.90801	0	-0.1844	0.445
    """)


class WongEtAl2022Deep(WongEtAl2022Shallow):
    """
    For deep events (depth >20km)
    """

    def compute(self, ctx: np.recarray, imts, mean, sig, tau, phi):

        for m, imt in enumerate(imts):

            for vs in ctx.vs30:
                if (vs >= 1E-10) & (vs < 185.0):
                    C = self.COEFFS_deep_Vs30_150[imt]
                elif (vs == 185.0):
                    C = self.COEFFS_deep_Vs30_185[imt]
                elif (vs > 185.0) & (vs < 365.0):
                    C = self.COEFFS_deep_Vs30_260[imt]
                elif vs == 365.0:
                    C = self.COEFFS_deep_Vs30_365[imt]
                elif vs == 428.0:
                    C = self.COEFFS_deep_Vs30_428[imt]
                elif (vs > 365.0) & (vs < 760.0):
                    C = self.COEFFS_deep_Vs30_530[imt]
                elif vs == 760.0:
                    C = self.COEFFS_deep_Vs30_760[imt]
                elif (vs > 760.0) & (vs < 1500.0):
                    C = self.COEFFS_deep_Vs30_1080[imt]
                elif (vs >= 1500.0):
                    C = self.COEFFS_deep_Vs30_1500[imt]

            mean[m] = (C['C1'] +
                       _compute_magnitude(ctx, C) +
                       _compute_distance(ctx, C))

            if imt.string.startswith(('PGV')):
                sig[m] = 0
            else:
                sig[m] = C['SigmaTot']

    COEFFS_deep_Vs30_185 = CoeffsTable(sa_damping=5, table="""
imt	Freq 	C1 	C2 	C4 	C5 	C6 	C7 	C8 	C10 	param_sigma	model_sigma	SigmaTot
10	0.1	2.23924	0.88904	5.7	0	-4.3267	0.29496	0	-0.14503	0.2903	1.2756	1.30822
5	0.2	4.9761	1.26923	5.8	0	-4.2701	0.20407	0	-0.22666	0.3373	1.1358	1.18483
3.0211	0.331	10.39555	1.40754	5.9	0	-4.75552	0.15144	0	-0.26862	0.3936	0.9702	1.047
1.996	0.501	17.88831	1.30809	6	0	-5.63213	0.14066	0	-0.28729	0.4219	0.8636	0.96115
1.5848	0.631	29.04721	0.80828	6.2	0	-7.06492	0.20046	0	-0.30768	0.4435	0.768	0.88686
1	1	70.86868	-2.72757	6.6	0	-12.54106	0.67623	0	-0.35302	0.5067	0.6627	0.83422
0.7413	1.349	106.1116	-6.43004	6.8	0	-17.05878	1.16334	0	-0.3575	0.5359	0.6566	0.84753
0.5013	1.995	167.8015	-12.39433	7.1	0	-24.73777	1.91219	0	-0.32696	0.5593	0.5902	0.81311
0.3981	2.512	162.4307	-12.23821	7	0	-24.23815	1.9031	0	-0.30798	0.582	0.5655	0.81149
0.302	3.311	182.1445	-14.54356	7	0	-26.93524	2.21379	0	-0.28892	0.6045	0.5581	0.82274
0.2399	4.169	218.5392	-18.39985	7.1	0	-31.56796	2.70736	0	-0.26152	0.6315	0.533	0.82637
0.1995	5.012	229.6358	-19.84729	7.1	0	-33.07245	2.90063	0	-0.23693	0.6548	0.5209	0.83672
0.1585	6.31	242.3088	-21.47872	7.1	0	-34.80092	3.11927	0	-0.20269	0.6413	0.5103	0.81956
0.1514	6.607	221.0366	-19.64746	7	0	-32.25693	2.90152	0	-0.19915	0.6388	0.5113	0.81823
0.1202	8.318	191.9292	-17.5185	6.8	0	-28.89044	2.66676	0	-0.17815	0.6303	0.5184	0.8161
0.1	10	199.455	-18.55872	6.8	0	-29.97464	2.81452	0	-0.15835	0.617	0.4997	0.79397
0.0794	12.589	168.4743	-15.93822	6.6	0	-26.18095	2.4994	0	-0.13857	0.6442	0.4866	0.80732
0.0692	14.454	170.5263	-16.25515	6.6	0	-26.49816	2.54691	0	-0.12739	0.6432	0.4887	0.8078
0.0603	16.596	154.9056	-14.79413	6.5	0	-24.5222	2.36322	0	-0.12262	0.6327	0.4918	0.80136
0.055	18.197	140.912	-13.47447	6.4	0	-22.72883	2.19519	0	-0.12298	0.6263	0.4852	0.79226
0.0501	19.953	127.56	-12.18776	6.3	0	-20.98351	2.02746	0	-0.12309	0.6253	0.489	0.7938
0.0398	25.119	113.869	-10.80928	6.2	0	-19.16232	1.8444	0	-0.12886	0.6243	0.4846	0.79031
0.0324	30.903	101.2428	-9.51463	6.1	0	-17.44776	1.66844	0	-0.13768	0.6159	0.4793	0.78042
0.0251	39.811	97.71396	-9.03698	6.1	0	-16.92421	1.59824	0	-0.15042	0.6056	0.4744	0.76929
0.02	50.119	104.9656	-9.60834	6.2	0	-17.85769	1.66995	0	-0.15902	0.5959	0.4768	0.76317
0.01	100	102.0858	-9.21284	6.2	0	-17.42784	1.61145	0	-0.16875	0.5835	0.4774	0.75391
pga	-1 	101.7488	-9.15899	6.2	0	-17.37388	1.60314	0	-0.16908	0.583	0.4774	0.75352
pgv	-2 	49.26893	-3.53262	5.6	0	-9.70275	0.87669	0	-0.18271	0.4627
    """)

    COEFFS_deep_Vs30_260 = CoeffsTable(sa_damping=5, table="""
imt	Freq 	C1 	C2 	C4 	C5 	C6 	C7 	C8 	C10 	param_sigma	model_sigma	SigmaTot
10	0.1	-4.06428	1.34282	5.4	0	-3.4532	0.23289	0	-0.1427	0.3058	1.2756	1.31174
5	0.2	2.52438	1.35767	5.7	0	-3.94546	0.19376	0	-0.22802	0.3468	1.1358	1.18757
3.0211	0.331	11.21896	1.12792	5.9	0	-4.89811	0.19535	0	-0.27638	0.3887	0.9702	1.04517
1.996	0.501	16.13302	1.01694	5.9	0	-5.42645	0.18666	0	-0.29415	0.4178	0.8636	0.95935
1.5848	0.631	21.10197	1.01206	6	0	-6.00444	0.17189	0	-0.29984	0.4435	0.768	0.88686
1	1	30.96416	0.74999	6.1	0	-7.18627	0.18324	0	-0.30587	0.4828	0.6627	0.81992
0.7413	1.349	43.71734	-0.26484	6.2	0	-8.88793	0.3174	0	-0.30885	0.5089	0.6566	0.83072
0.5013	1.995	80.53577	-3.3402	6.5	0	-13.82011	0.73594	0	-0.31268	0.5498	0.5902	0.80661
0.3981	2.512	112.4397	-6.21106	6.7	0	-17.97597	1.11532	0	-0.2958	0.5827	0.5655	0.81199
0.302	3.311	139.2257	-8.85118	6.8	0	-21.49511	1.46421	0	-0.2793	0.6222	0.5581	0.83583
0.2399	4.169	168.3329	-11.62806	6.9	0	-25.29347	1.82776	0	-0.2555	0.6441	0.533	0.83603
0.1995	5.012	197.6861	-14.40394	7	0	-29.04471	2.18513	0	-0.2396	0.6366	0.5209	0.82255
0.1585	6.31	172.9564	-13.05183	6.8	0	-26.20392	2.0398	0	-0.21531	0.6385	0.5103	0.81737
0.1514	6.607	175.796	-13.4259	6.8	0	-26.60735	2.09247	0	-0.21223	0.6442	0.5113	0.82245
0.1202	8.318	187.3089	-14.99079	6.8	0	-28.23981	2.31085	0	-0.18484	0.645	0.5184	0.8275
0.1	10	177.0336	-14.59615	6.7	0	-27.09436	2.27932	0	-0.16581	0.6292	0.4997	0.80349
0.0794	12.589	136.0109	-11.43608	6.4	0	-21.95613	1.89006	0	-0.15274	0.6616	0.4866	0.82128
0.0692	14.454	125.7472	-10.7008	6.3	0	-20.67593	1.80192	0	-0.14316	0.665	0.4887	0.82526
0.0603	16.596	115.3937	-9.89023	6.2	0	-19.34768	1.70006	0	-0.13524	0.6679	0.4918	0.82943
0.055	18.197	115.9236	-9.9845	6.2	0	-19.44174	1.71563	0	-0.13121	0.6574	0.4852	0.81706
0.0501	19.953	105.567	-9.1025	6.1	0	-18.07121	1.59993	0	-0.12999	0.6462	0.489	0.81037
0.0398	25.119	94.88692	-8.14908	6	0	-16.63719	1.47287	0	-0.13059	0.6467	0.4846	0.80812
0.0324	30.903	91.89919	-7.77002	6	0	-16.20113	1.41794	0	-0.13896	0.6383	0.4793	0.79822
0.0251	39.811	88.90305	-7.382	6	0	-15.75632	1.36069	0	-0.14686	0.6268	0.4744	0.78609
0.02	50.119	86.55507	-7.07048	6	0	-15.40218	1.31405	0	-0.15396	0.6173	0.4768	0.78
0.01	100	83.82279	-6.70478	6	0	-14.98769	1.25899	0	-0.16222	0.6055	0.4774	0.77106
pga	-1 	83.41772	-6.63195	6	0	-14.92092	1.24728	0	-0.16212	0.606	0.4774	0.77146
pgv	-2 	43.611	-3.09358	5.4	0	-9.00818	0.82445	0	-0.16302	0.4921
    """)

    COEFFS_deep_Vs30_365 = CoeffsTable(sa_damping=5, table="""
imt	Freq 	C1 	C2 	C4 	C5 	C6 	C7 	C8 	C10 	param_sigma	model_sigma	SigmaTot
10	0.1	-5.89428	1.54704	5.4	0	-3.18565	0.20388	0	-0.14813	0.2898	1.2756	1.30811
5	0.2	0.82909	1.53292	5.7	0	-3.69799	0.16915	0	-0.23479	0.3377	1.1358	1.18494
3.0211	0.331	10.22099	1.15948	5.9	0	-4.76276	0.19303	0	-0.28367	0.3661	0.9702	1.03698
1.996	0.501	19.13934	0.72139	6	0	-5.85525	0.23212	0	-0.3042	0.3761	0.8636	0.94194
1.5848	0.631	25.40152	0.53718	6.1	0	-6.61504	0.24338	0	-0.31135	0.3848	0.768	0.85901
1	1	32.53355	0.21408	6.1	0	-7.43641	0.26362	0	-0.3121	0.4529	0.6627	0.80268
0.7413	1.349	43.69998	-0.48336	6.2	0	-8.91529	0.34927	0	-0.30116	0.5169	0.6566	0.83565
0.5013	1.995	47.13694	-0.88354	6.1	0	-9.3807	0.39279	0	-0.27345	0.5631	0.5902	0.81573
0.3981	2.512	66.01481	-2.09548	6.3	0	-11.92174	0.5546	0	-0.26212	0.5865	0.5655	0.81472
0.302	3.311	84.00746	-3.74371	6.4	0	-14.36503	0.77954	0	-0.2489	0.6116	0.5581	0.82797
0.2399	4.169	92.99872	-4.78684	6.4	0	-15.65712	0.92627	0	-0.23303	0.6256	0.533	0.82187
0.1995	5.012	112.0083	-6.51004	6.5	0	-18.24856	1.16354	0	-0.22159	0.6181	0.5209	0.80832
0.1585	6.31	136	-8.89998	6.6	0	-21.47893	1.48895	0	-0.20588	0.6414	0.5103	0.81963
0.1514	6.607	138.3169	-9.24763	6.6	0	-21.80543	1.53773	0	-0.20243	0.6584	0.5113	0.83362
0.1202	8.318	133.3029	-9.43377	6.5	0	-21.32144	1.57964	0	-0.18015	0.6909	0.5184	0.86376
0.1	10	115.2858	-8.43449	6.3	0	-19.10105	1.46438	0	-0.16934	0.6836	0.4997	0.84676
0.0794	12.589	122.4671	-9.49702	6.3	0	-20.19708	1.62386	0	-0.15751	0.6951	0.4866	0.8485
0.0692	14.454	114.5081	-9.10871	6.2	0	-19.22034	1.58278	0	-0.14825	0.6715	0.4887	0.83051
0.0603	16.596	105.397	-8.51048	6.1	0	-18.05312	1.51	0	-0.14108	0.677	0.4918	0.83678
0.055	18.197	96.25155	-7.80813	6	0	-16.83717	1.41798	0	-0.13678	0.6859	0.4852	0.84017
0.0501	19.953	96.17582	-7.83323	6	0	-16.84412	1.42371	0	-0.13404	0.6781	0.489	0.83602
0.0398	25.119	86.83934	-7.08907	5.9	0	-15.59138	1.32578	0	-0.13191	0.6578	0.4846	0.81703
0.0324	30.903	76.95651	-6.19404	5.8	0	-14.21214	1.20096	0	-0.13501	0.6498	0.4793	0.80745
0.0251	39.811	74.29043	-5.87185	5.8	0	-13.81202	1.1529	0	-0.14086	0.6378	0.4744	0.79489
0.02	50.119	72.02564	-5.58468	5.8	0	-13.46659	1.10935	0	-0.14657	0.6224	0.4768	0.78404
0.01	100	69.57445	-5.26972	5.8	0	-13.08968	1.06122	0	-0.1531	0.6103	0.4774	0.77484
pga	-1 	69.33356	-5.22513	5.8	0	-13.04691	1.05354	0	-0.15287	0.6102	0.4774	0.77476
pgv	-2 	44.09953	-3.2484	5.4	0	-9.16765	0.85807	0	-0.15093	0.4599
    """)

    COEFFS_deep_Vs30_428 = CoeffsTable(sa_damping=5, table="""
imt	Freq 	C1 	C2 	C4 	C5 	C6 	C7 	C8 	C10 	param_sigma	model_sigma	SigmaTot
10	0.1	-6.9155	1.66675	5.4	0	-3.03522	0.18615	0	-0.15158	0.2945	1.2756	1.3092
5	0.2	-0.08881	1.63345	5.7	0	-3.56561	0.15429	0	-0.23807	0.3392	1.1358	1.1854
3.0211	0.331	9.41267	1.23611	5.9	0	-4.64883	0.18193	0	-0.28698	0.3861	0.9702	1.0442
1.996	0.501	21.84053	0.6191	6.1	0	-6.22484	0.24654	0	-0.30737	0.3986	0.8636	0.9512
1.5848	0.631	25.1181	0.51244	6.1	0	-6.58828	0.24816	0	-0.31414	0.41	0.768	0.8706
1	1	32.93593	0.08357	6.1	0	-7.51731	0.28394	0	-0.31387	0.4398	0.6627	0.7954
0.7413	1.349	43.47321	-0.50426	6.2	0	-8.89132	0.35245	0	-0.30295	0.4883	0.6566	0.8183
0.5013	1.995	46.91267	-0.93125	6.1	0	-9.36813	0.40094	0	-0.27491	0.5676	0.5902	0.8188
0.3981	2.512	58.53881	-1.71001	6.2	0	-10.95674	0.50386	0	-0.25817	0.6228	0.5655	0.8412
0.302	3.311	73.99161	-2.90241	6.3	0	-13.06945	0.66484	0	-0.2386	0.64	0.5581	0.8492
0.2399	4.169	91.64108	-4.50724	6.4	0	-15.47178	0.88528	0	-0.22629	0.6315	0.533	0.8264
0.1995	5.012	98.22293	-5.32783	6.4	0	-16.41981	1.00088	0	-0.20822	0.6565	0.5209	0.8381
0.1585	6.31	106.9383	-6.37317	6.4	0	-17.69223	1.15058	0	-0.19072	0.6497	0.5103	0.8261
0.1514	6.607	108.8136	-6.61834	6.4	0	-17.96824	1.18618	0	-0.1876	0.6526	0.5113	0.829
0.1202	8.318	106.7208	-7.05038	6.3	0	-17.84235	1.26035	0	-0.17244	0.6654	0.5184	0.8435
0.1	10	103.0432	-7.20814	6.2	0	-17.48802	1.29736	0	-0.16376	0.6569	0.4997	0.8254
0.0794	12.589	109.7436	-8.23258	6.2	0	-18.51599	1.45171	0	-0.14918	0.6755	0.4866	0.8325
0.0692	14.454	102.118	-7.86625	6.1	0	-17.56159	1.41182	0	-0.14031	0.6823	0.4887	0.8393
0.0603	16.596	93.82941	-7.33934	6	0	-16.49267	1.34782	0	-0.13461	0.6789	0.4918	0.8383
0.055	18.197	85.58938	-6.72014	5.9	0	-15.38958	1.26654	0	-0.13275	0.6781	0.4852	0.8338
0.0501	19.953	85.62312	-6.76578	5.9	0	-15.41352	1.27568	0	-0.13241	0.674	0.489	0.8327
0.0398	25.119	78.01486	-6.22686	5.8	0	-14.39446	1.20623	0	-0.12877	0.6714	0.4846	0.828
0.0324	30.903	69.46166	-5.48027	5.7	0	-13.1943	1.10188	0	-0.13149	0.6646	0.4793	0.8194
0.0251	39.811	67.40193	-5.24299	5.7	0	-12.88585	1.06672	0	-0.13721	0.652	0.4744	0.8063
0.02	50.119	65.29077	-4.98649	5.7	0	-12.56134	1.02752	0	-0.14202	0.6393	0.4768	0.7975
0.01	100	68.77858	-5.13883	5.8	0	-13.00533	1.04366	0	-0.14918	0.6189	0.4774	0.7816
pga	-1 	68.52187	-5.09631	5.8	0	-12.9601	1.03629	0	-0.14898	0.6172	0.4774	0.7803
pgv	-2 	39.70192	-2.85255	5.3	0	-8.55941	0.80414	0	-0.15077	0.4487
    """)

    COEFFS_deep_Vs30_530 = CoeffsTable(sa_damping=5, table="""
imt	Freq 	C1 	C2 	C4 	C5 	C6 	C7 	C8 	C10 	param_sigma	model_sigma	SigmaTot
10	0.1	-8.03767	1.81	5.4	0	-2.85976	0.16457	0	-0.15643	0.2767	1.2756	1.30527
5	0.2	-1.25751	1.78877	5.7	0	-3.38239	0.13112	0	-0.24196	0.3092	1.1358	1.17714
3.0211	0.331	8.15232	1.40183	5.9	0	-4.45366	0.15768	0	-0.28962	0.3223	0.9702	1.02233
1.996	0.501	20.94299	0.69302	6.1	0	-6.09092	0.23685	0	-0.31168	0.3451	0.8636	0.93
1.5848	0.631	25.01453	0.42709	6.1	0	-6.57384	0.26243	0	-0.31979	0.3806	0.768	0.85714
1	1	38.35402	-0.4174	6.2	0	-8.27339	0.35831	0	-0.31925	0.4037	0.6627	0.77598
0.7413	1.349	44.88622	-0.88836	6.2	0	-9.13036	0.41238	0	-0.30586	0.4852	0.6566	0.81642
0.5013	1.995	48.98657	-1.46751	6.1	0	-9.70979	0.48418	0	-0.27903	0.58	0.5902	0.82749
0.3981	2.512	54.65042	-2.00377	6.1	0	-10.51665	0.55588	0	-0.25843	0.6022	0.5655	0.8261
0.302	3.311	61.09278	-2.53809	6.1	0	-11.42838	0.62602	0	-0.23021	0.6315	0.5581	0.84277
0.2399	4.169	74.6057	-3.59587	6.2	0	-13.28874	0.77046	0	-0.21288	0.6833	0.533	0.8666
0.1995	5.012	80.00266	-4.15288	6.2	0	-14.07377	0.84958	0	-0.19862	0.7007	0.5209	0.87311
0.1585	6.31	87.95488	-5.11364	6.2	0	-15.23049	0.98796	0	-0.18458	0.7117	0.5103	0.87574
0.1514	6.607	98.8055	-5.92063	6.3	0	-16.69061	1.09728	0	-0.18137	0.7167	0.5113	0.88039
0.1202	8.318	97.18981	-6.29408	6.2	0	-16.61059	1.16151	0	-0.1664	0.693	0.5184	0.86544
0.1	10	103.4824	-7.16712	6.2	0	-17.571	1.29274	0	-0.15638	0.6944	0.4997	0.85551
0.0794	12.589	100.129	-7.40399	6.1	0	-17.25625	1.34181	0	-0.14196	0.6943	0.4866	0.84784
0.0692	14.454	93.65316	-7.10737	6	0	-16.45844	1.31177	0	-0.13559	0.6817	0.4887	0.83878
0.0603	16.596	86.99494	-6.75528	5.9	0	-15.61034	1.27237	0	-0.13166	0.6812	0.4918	0.84018
0.055	18.197	87.73768	-6.92513	5.9	0	-15.74427	1.3006	0	-0.13039	0.6952	0.4852	0.84777
0.0501	19.953	80.4328	-6.40826	5.8	0	-14.76011	1.23307	0	-0.1284	0.7043	0.489	0.85741
0.0398	25.119	73.42247	-5.94296	5.7	0	-13.82274	1.17454	0	-0.12507	0.6939	0.4846	0.84637
0.0324	30.903	71.87675	-5.81419	5.7	0	-13.60719	1.15756	0	-0.12684	0.6778	0.4793	0.83015
0.0251	39.811	63.55502	-5.0654	5.6	0	-12.41712	1.05065	0	-0.13091	0.6674	0.4744	0.81883
0.02	50.119	61.90769	-4.87518	5.6	0	-12.16475	1.02174	0	-0.13495	0.6566	0.4768	0.81146
0.01	100	64.88462	-5.0074	5.7	0	-12.54193	1.03547	0	-0.14146	0.6367	0.4774	0.7958
pga	-1 	64.73113	-4.9809	5.7	0	-12.513	1.03056	0	-0.1412	0.637	0.4774	0.79604
pgv	-2 	39.64739	-2.89068	5.3	0	-8.5942	0.81631	0	-0.14725	0.4549
    """)

    COEFFS_deep_Vs30_760 = CoeffsTable(sa_damping=5, table="""
imt	Freq 	C1 	C2 	C4 	C5 	C6 	C7 	C8 	C10 	param_sigma	model_sigma	SigmaTot
10	0.1	-9.22402	1.94021	5.4	0	-2.69597	0.14582	0	-0.15966	0.2835	1.2756	1.3067
5	0.2	-2.19341	1.87567	5.7	0	-3.25694	0.11873	0	-0.24647	0.3272	1.1358	1.182
3.0211	0.331	10.209	1.28757	6	0	-4.75429	0.17465	0	-0.29404	0.3752	0.9702	1.0402
1.996	0.501	20.33703	0.67544	6.1	0	-6.02319	0.24061	0	-0.31542	0.3731	0.8636	0.94075
1.5848	0.631	28.17415	0.28135	6.2	0	-7.01483	0.28296	0	-0.32196	0.3683	0.768	0.85174
1	1	37.99058	-0.39601	6.2	0	-8.27053	0.3564	0	-0.31746	0.4298	0.6627	0.78987
0.7413	1.349	44.5577	-0.86527	6.2	0	-9.13144	0.40982	0	-0.30304	0.4817	0.6566	0.81435
0.5013	1.995	47.53404	-1.20048	6.1	0	-9.52729	0.44397	0	-0.27306	0.5276	0.5902	0.79164
0.3981	2.512	52.11226	-1.49083	6.1	0	-10.14325	0.4767	0	-0.25038	0.5565	0.5655	0.7934
0.302	3.311	57.95681	-1.92831	6.1	0	-10.95883	0.53184	0	-0.22169	0.5979	0.5581	0.8179
0.2399	4.169	63.14175	-2.35722	6.1	0	-11.6992	0.5891	0	-0.19799	0.6183	0.533	0.81632
0.1995	5.012	67.73399	-2.80825	6.1	0	-12.37446	0.65295	0	-0.18027	0.64	0.5209	0.82519
0.1585	6.31	66.85735	-3.08451	6	0	-12.34889	0.69779	0	-0.15953	0.6706	0.5103	0.84268
0.1514	6.607	67.98064	-3.20709	6	0	-12.52122	0.71605	0	-0.15554	0.6724	0.5113	0.84472
0.1202	8.318	73.97302	-3.92204	6	0	-13.44834	0.82423	0	-0.13877	0.6812	0.5184	0.85602
0.1	10	71.43771	-4.06123	5.9	0	-13.19303	0.85325	0	-0.12811	0.6812	0.4997	0.84483
0.0794	12.589	76.61803	-4.77255	5.9	0	-14.02002	0.9647	0	-0.11795	0.689	0.4866	0.84351
0.0692	14.454	71.88163	-4.64213	5.8	0	-13.42685	0.95465	0	-0.11275	0.7025	0.4887	0.85577
0.0603	16.596	66.97212	-4.46293	5.7	0	-12.79742	0.93716	0	-0.10889	0.7031	0.4918	0.85803
0.055	18.197	67.62684	-4.59324	5.7	0	-12.92078	0.95962	0	-0.10653	0.7067	0.4852	0.85723
0.0501	19.953	62.08216	-4.23285	5.6	0	-12.16202	0.91183	0	-0.10505	0.718	0.489	0.8687
0.0398	25.119	57.73029	-4.06066	5.5	0	-11.58973	0.8945	0	-0.10552	0.7204	0.4846	0.86822
0.0324	30.903	57.97868	-4.17019	5.5	0	-11.65156	0.91463	0	-0.10812	0.72	0.4793	0.86494
0.0251	39.811	51.7562	-3.71567	5.4	0	-10.76019	0.85118	0	-0.11264	0.6975	0.4744	0.84354
0.02	50.119	50.09983	-3.58159	5.4	0	-10.50493	0.83123	0	-0.11897	0.6786	0.4768	0.82936
0.01	100	51.52443	-3.61327	5.5	0	-10.66384	0.83166	0	-0.12816	0.6373	0.4774	0.79628
pga	-1 	51.14649	-3.57103	5.5	0	-10.59901	0.82422	0	-0.12837	0.6336	0.4774	0.79332
pgv	-2 	38.93484	-2.83525	5.3	0	-8.54137	0.81287	0	-0.14409	0.4554
    """)

    COEFFS_deep_Vs30_1080 = CoeffsTable(sa_damping=5, table="""
imt	Freq 	C1 	C2 	C4 	C5 	C6 	C7 	C8 	C10 	param_sigma	model_sigma	SigmaTot
10	0.1	-9.94467	2.03274	5.4	0	-2.58611	0.13184	0	-0.16348	0.2739	1.2756	1.30468
5	0.2	-2.85757	1.96038	5.7	0	-3.15687	0.10616	0	-0.24948	0.311	1.1358	1.17761
3.0211	0.331	9.50447	1.37435	6	0	-4.65042	0.16201	0	-0.29654	0.335	0.9702	1.02641
1.996	0.501	19.42515	0.78887	6.1	0	-5.88408	0.22405	0	-0.31785	0.3511	0.8636	0.932243
1.5848	0.631	27.22984	0.39421	6.2	0	-6.87085	0.26668	0	-0.32407	0.3734	0.768	0.853962
1	1	37.72668	-0.41228	6.2	0	-8.24274	0.36012	0	-0.3185	0.4143	0.6627	0.781547
0.7413	1.349	44.39633	-0.89301	6.2	0	-9.1152	0.41483	0	-0.30422	0.4455	0.6566	0.79347
0.5013	1.995	48.02106	-1.37182	6.1	0	-9.61985	0.47167	0	-0.27523	0.4975	0.5902	0.771908
0.3981	2.512	53.04999	-1.76242	6.1	0	-10.3109	0.51995	0	-0.2532	0.5282	0.5655	0.773812
0.302	3.311	59.63075	-2.36002	6.1	0	-11.25182	0.60028	0	-0.22461	0.5586	0.5581	0.789626
0.2399	4.169	58.88488	-2.54016	6	0	-11.20858	0.62655	0	-0.20009	0.593	0.533	0.797332
0.1995	5.012	63.04949	-2.93622	6	0	-11.8256	0.68261	0	-0.18061	0.6078	0.5209	0.800473
0.1585	6.31	68.3811	-3.46609	6	0	-12.629	0.7597	0	-0.15764	0.6285	0.5103	0.809579
0.1514	6.607	69.46798	-3.57603	6	0	-12.79463	0.77593	0	-0.1533	0.6274	0.5113	0.809357
0.1202	8.318	67.81388	-3.68465	5.9	0	-12.65502	0.79768	0	-0.13347	0.6533	0.5184	0.83399
0.1	10	71.67037	-4.09398	5.9	0	-13.25869	0.86044	0	-0.12008	0.6724	0.4997	0.837748
0.0794	12.589	69.3763	-4.14677	5.8	0	-13.03577	0.87696	0	-0.10694	0.6944	0.4866	0.847922
0.0692	14.454	65.34	-3.99283	5.7	0	-12.52696	0.86127	0	-0.10106	0.7044	0.4887	0.857326
0.0603	16.596	67.42964	-4.27086	5.7	0	-12.87548	0.90671	0	-0.09675	0.7128	0.4918	0.865997
0.055	18.197	62.56392	-4.01021	5.6	0	-12.22293	0.87403	0	-0.09471	0.7176	0.4852	0.866238
0.0501	19.953	63.57597	-4.1706	5.6	0	-12.3988	0.90097	0	-0.0933	0.7216	0.489	0.871681
0.0398	25.119	59.57322	-4.06961	5.5	0	-11.89416	0.89577	0	-0.09197	0.7281	0.4846	0.874624
0.0324	30.903	54.89888	-3.83474	5.4	0	-11.25994	0.86786	0	-0.09258	0.7281	0.4793	0.871698
0.0251	39.811	50.08947	-3.54479	5.3	0	-10.58589	0.83003	0	-0.09554	0.7265	0.4744	0.867674
0.02	50.119	49.10114	-3.52464	5.3	0	-10.44304	0.82905	0	-0.1011	0.7273	0.4768	0.869657
0.01	100	49.50914	-3.58029	5.4	0	-10.45222	0.8349	0	-0.11428	0.6471	0.4774	0.804145
pga	-1 	49.01306	-3.53963	5.4	0	-10.36996	0.8279	0	-0.1149	0.6388	0.4774	0.797481
pgv	-2 	35.32381	-2.5049	5.2	0	-8.04105	0.76843	0	-0.14424	0.4401
    """)

    COEFFS_deep_Vs30_1500 = CoeffsTable(sa_damping=5, table="""
imt	Freq 	C1 	C2 	C4 	C5 	C6 	C7 	C8 	C10 	param_sigma	model_sigma	SigmaTot
10	0.1	-10.49826	2.09582	5.4	0	-2.51426	0.12288	0	-0.16307	0.2731	1.2756	1.30451
5	0.2	-3.07494	1.97375	5.7	0	-3.14119	0.10461	0	-0.24902	0.3084	1.1358	1.17692
3.0211	0.331	9.83182	1.28979	6	0	-4.73407	0.17553	0	-0.29387	0.3216	0.9702	1.02211
1.996	0.501	19.75484	0.70839	6.1	0	-5.9646	0.23664	0	-0.31617	0.3406	0.8636	0.92834
1.5848	0.631	27.73978	0.29543	6.2	0	-6.97664	0.28134	0	-0.32269	0.3595	0.768	0.84798
1	1	37.4756	-0.37858	6.2	0	-8.21498	0.35451	0	-0.31819	0.4115	0.6627	0.78007
0.7413	1.349	44.11293	-0.87037	6.2	0	-9.08773	0.41153	0	-0.30435	0.4509	0.6566	0.79651
0.5013	1.995	47.47013	-1.27662	6.1	0	-9.53965	0.45639	0	-0.27493	0.5044	0.5902	0.77637
0.3981	2.512	53.04236	-1.78513	6.1	0	-10.32908	0.52354	0	-0.25306	0.5327	0.5655	0.77689
0.302	3.311	59.67915	-2.38283	6.1	0	-11.27705	0.60363	0	-0.22401	0.5644	0.5581	0.79374
0.2399	4.169	58.74831	-2.51538	6	0	-11.19996	0.6222	0	-0.19957	0.5889	0.533	0.79429
0.1995	5.012	63.04345	-2.93681	6	0	-11.83945	0.68231	0	-0.18005	0.6063	0.5209	0.79934
0.1585	6.31	68.54849	-3.51136	6	0	-12.67392	0.76662	0	-0.15731	0.6261	0.5103	0.80772
0.1514	6.607	69.59005	-3.6096	6	0	-12.8307	0.78093	0	-0.15297	0.6305	0.5113	0.81176
0.1202	8.318	68.04929	-3.74767	5.9	0	-12.7117	0.80748	0	-0.13335	0.649	0.5184	0.83063
0.1	10	72.01244	-4.18625	5.9	0	-13.33419	0.87497	0	-0.12009	0.6626	0.4997	0.8299
0.0794	12.589	69.74603	-4.24538	5.8	0	-13.11702	0.89264	0	-0.10685	0.6775	0.4866	0.83414
0.0692	14.454	65.68249	-4.07579	5.7	0	-12.60331	0.87448	0	-0.10067	0.6853	0.4887	0.8417
0.0603	16.596	67.79343	-4.3293	5.7	0	-12.95181	0.9157	0	-0.09573	0.6919	0.4918	0.84888
0.055	18.197	62.94144	-4.04373	5.6	0	-12.29884	0.8788	0	-0.09307	0.6957	0.4852	0.84819
0.0501	19.953	64.05569	-4.18264	5.6	0	-12.48718	0.9019	0	-0.09087	0.6991	0.489	0.85315
0.0398	25.119	60.57238	-4.05466	5.5	0	-12.05671	0.89126	0	-0.08726	0.7054	0.4846	0.85582
0.0324	30.903	51.6826	-3.48432	5.3	0	-10.81647	0.81431	0	-0.08616	0.7077	0.4793	0.85473
0.0251	39.811	47.7734	-3.29534	5.2	0	-10.28098	0.7922	0	-0.08751	0.7071	0.4744	0.8515
0.02	50.119	47.17186	-3.34137	5.2	0	-10.20652	0.80257	0	-0.09145	0.7024	0.4768	0.84894
0.01	100	51.14855	-3.7548	5.4	0	-10.73373	0.86222	0	-0.10771	0.6371	0.4774	0.79612
pga	-1 	50.38409	-3.69514	5.4	0	-10.60943	0.85235	0	-0.10911	0.6287	0.4774	0.78941
pgv	-2 	35.6397	-2.57346	5.2	0	-8.1267	0.78163	0	-0.13717	0.4352
    """)


class WongEtAl2015(GMPE):
    """
    Implements GMPE developed by Wong IG, Silva WJ, Darragh R, Gregor N and
    Dober M (2015) "Ground motion prediction modelfor deep earthquakes beneath
    the Island of Hawaii" Earthquake Spectra31: 1763–1788. SA are given up to
    10 s. Ground motion refers to Vs30 of 428m/s Model is for deep events depth
    >20km
    """
    def compute(self, ctx: np.recarray, imts, mean, sig, tau, phi):
        for m, imt in enumerate(imts):

            C = self.COEFFS_2015[imt]

            mean[m] = (C['C1'] +
                    _compute_magnitude(ctx, C) +
                    _compute_distance(ctx, C))

            if imt.string.startswith(('PGV')):
                sig[m]=0
            else:
                sig[m] = C['SigmaTot']


    #: Coefficients from SA from Table 1 in Wong et al. 2015
    #: Coefficients from PGA e PGV from Table 1
    #: c3, c4, c5 and c6 in the original table coresponds to c4, c6, c7 and c10 here as in eq of wong 2022
    #  imt	freq	c1	  c2	c3	c4	c5	c6	param_sigma	model_sigma	SigmaTot Wong 2015
    #  imt	freq	c1	  c2	c4	c6	c7	c10	param_sigma	model_sigma	SigmaTot Wong 2022
    COEFFS_2015 = CoeffsTable(sa_damping=5, table="""
imt	freq	C1	  C2	C4	C6	C7	C10	param_sigma	model_sigma	SigmaTot
pga	-1	68.52187	-5.09631	5.8	-12.9601	1.0363	-0.1490	0.6172	0.4774	0.7803
10.000	0.100	-6.91550	1.66675	5.4	-3.0352	0.1862	-0.1516	0.2945	1.2756	1.3092
5.0000	0.200	-0.08881	1.63345	5.7	-3.5656	0.1543	-0.2381	0.3392	1.1358	1.1854
3.0211	0.331	9.41267	        1.23611	5.9	-4.6488	0.1819	-0.2870	0.3861	0.9702	1.0442
1.9960	0.501	21.84053	0.61910	6.1	-6.2248	0.2465	-0.3074	0.3986	0.8636	0.9512
1.5848	0.631	25.11810	0.51244	6.1	-6.5883	0.2482	-0.3141	0.4100	0.7680	0.8706
1.0000	1.000	32.93593	0.08357	6.1	-7.5173	0.2839	-0.3139	0.4398	0.6627	0.7954
0.7413	1.349	43.47321	-0.50426	6.2	-8.8913	0.3525	-0.3030	0.4883	0.6566	0.8183
0.5013	1.995	46.91267	-0.93125	6.1	-9.3681	0.4009	0.2749	0.5676	0.5902	0.8188
0.3981	2.512	58.53881	-1.71001	6.2	-10.9567	0.5039	-0.2582	0.6228	0.5655	0.8412
0.3020	3.311	73.99161	-2.90241	6.3	-13.0695	0.6648	-0.2386	0.6400	0.5581	0.8492
0.2399	4.169	91.64108	-4.50724	6.4	-15.4718	0.8853	-0.2263	0.6315	0.5330	0.8264
0.1995	5.012	98.22293	-5.32783	6.4	-16.4198	1.0009	-0.2082	0.6565	0.5209	0.8381
0.1585	6.310	106.93830	-6.37317	6.4	-17.6922	1.1506	-0.1907	0.6497	0.5103	0.8261
0.1514	6.607	108.81360	-6.61834	6.4	-17.9682	1.1862	-0.1876	0.6526	0.5113	0.8290
0.1202	8.318	106.72080	-7.05038	6.3	-17.8424	1.2604	-0.1724	0.6654	0.5184	0.8435
0.1000	10.000	103.04320	-7.20814	6.2	-17.4880	1.2974	-0.1638	0.6569	0.4997	0.8254
0.0794	12.589	109.74360	-8.23258	6.2	-18.5160	1.4517	-0.1492	0.6755	0.4866	0.8325
0.0692	14.454	102.11800	-7.86625	6.1	-17.5616	1.4118	-0.1403	0.6823	0.4887	0.8393
0.0603	16.596	93.82941	-7.33934	6.0	-16.4927	1.3478	-0.1346	0.6789	0.4918	0.8383
0.0550	18.197	85.58938	-6.72014	5.9	-15.3896	1.2665	-0.1328	0.6781	0.4852	0.8338
0.0501	19.953	85.62312	-6.76578	5.9	-15.4135	1.2757	-0.1324	0.6740	0.4890	0.8327
0.0398	25.119	78.01486	-6.22686	5.8	-14.3945	1.2062	-0.1288	0.6714	0.4846	0.8280
0.0324	30.903	69.46166	-5.48027	5.7	-13.1943	1.1019	-0.1315	0.6646	0.4793	0.8194
0.0251	39.811	67.40193	-5.24299	5.7	-12.8859	1.0667	-0.1372	0.6520	0.4744	0.8063
0.0200	50.119	65.29077	-4.98649	5.7	-12.5613	1.0275	-0.1420	0.6393	0.4768	0.7975
0.0100	100.00	68.77858	-5.13883	5.8	-13.0053	1.0437	-0.1492	0.6189	0.4774	0.7816
pgv	-2	39.70192	-2.85255	5.3	-8.5594	0.8041	-0.1508	0.4487
    """)
