# -*- coding: utf-8 -*-
# vim: tabstop=4 shiftwidth=4 softtabstop=4
#
# Copyright (C) 2014-2023 GEM Foundation
#
# OpenQuake is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenQuake is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with OpenQuake. If not, see <http://www.gnu.org/licenses/>.

"""
Module exports :class:`RietbrockEdwards2019Mean`,
               :class:`RietbrockEdwards2019Low`,
               :class:`RietbrockEdwards2019Up`
"""

import numpy as np

from openquake.hazardlib.gsim.base import CoeffsTable, GMPE
from openquake.hazardlib import const
from openquake.hazardlib.imt import PGA, PGV, SA

CONSTS = {"r0": 10.0, "r1": 50.0, "r2": 100.0}


def _get_distance_segment_coefficients(rval):
    """
    Returns the coefficients describing the distance attenuation shape
    for three different distance bins, equations 12a - 12c
    """
    # Get distance segment ends
    # Equation 12a
    f_0 = np.log10(CONSTS["r0"] / rval)
    f_0[rval > CONSTS["r0"]] = 0.0

    # Equation 12b
    f_1 = np.log10(rval)
    f_1[rval > CONSTS["r1"]] = np.log10(CONSTS["r1"])
    # Equation 12c
    f_2 = np.log10(rval / CONSTS["r2"])
    f_2[rval <= CONSTS["r2"]] = 0.0
    return f_0, f_1, f_2


def _get_distance_term(C, rjb, mag):
    """
    Returns the distance scaling component of the model
    Equation 10, Page 63
    """
    # Depth adjusted distance, equation 11 (Page 63)
    rval = np.sqrt(rjb ** 2.0 + C["c11"] ** 2.0)
    f_0, f_1, f_2 = _get_distance_segment_coefficients(rval)
    return ((C["c4"] + C["c5"] * mag) * f_0 +
            (C["c6"] + C["c7"] * mag) * f_1 +
            (C["c8"] + C["c9"] * mag) * f_2 +
            (C["c10"] * rval))


def _get_magnitude_term(C, mag):
    """
    Returns the magnitude scaling component of the model
    Equation 10, Page 63
    """
    return (C["c2"] * mag) + (C["c3"] * (mag ** 2.0))


class RietbrockEdwards2019Mean(GMPE):
    """
    Implements the ground motion prediction equation of Rietbrock et al
    (2019):

    Rietbrock, A., Edwards, B. (2019). Update of the UK stochastic ground
    motion model using a decade of broadband data. In Proceedings of the
    2019 SECED conference.
    """
    #: Supported tectonic region type is stabe continental crust,
    DEFINED_FOR_TECTONIC_REGION_TYPE = const.TRT.STABLE_CONTINENTAL

    #: Supported intensity measure types are spectral acceleration, peak
    #: ground acceleration and peak ground velocity.
    DEFINED_FOR_INTENSITY_MEASURE_TYPES = {PGA, PGV, SA}

    #: Supported intensity measure component is the geometric mean of two
    #: horizontal components
    DEFINED_FOR_INTENSITY_MEASURE_COMPONENT = const.IMC.GEOMETRIC_MEAN

    #: Supported standard deviation types are inter-event, intra-event and
    #: total
    DEFINED_FOR_STANDARD_DEVIATION_TYPES = {
        const.StdDev.INTER_EVENT, const.StdDev.INTRA_EVENT, const.StdDev.TOTAL}

    #: No site parameter is required
    REQUIRES_SITES_PARAMETERS = set()

    #: Required rupture parameters are magnitude
    REQUIRES_RUPTURE_PARAMETERS = {'mag'}

    #: Required distance measure is Rjb
    REQUIRES_DISTANCES = {'rjb'}

    def compute(self, ctx: np.recarray, imts, mean, sig, tau, phi):
        """
        See :meth:`superclass method
        <.base.GroundShakingIntensityModel.compute>`
        for spec of input and result values.
        """
        for m, imt in enumerate(imts):
            C = self.COEFFS[imt]
            imean = C["c1"] + (_get_magnitude_term(C, ctx.mag) +
                               _get_distance_term(C, ctx.rjb, ctx.mag))
            # Converting from log10 to log
            if imt.string.startswith(("SA", "PGA")):
                mean[m] = np.log(10.0**(imean) / 980.665)
            else:
                mean[m] = np.log(10 ** imean)
            sig[m] = np.log(10.0 ** C["total"])
            phi[m] = np.log(10.0 ** C["phi"])
            tau[m] = np.log(10.0 ** C["tau"])

    COEFFS = CoeffsTable(sa_damping=5, table="""\
  IMT      c1      c2      c3      c4      c5      c6      c7      c8      c9       c10     c11   total     tau     phi
  pgv -4.4578  1.6540 -0.1044 -1.6308  0.2082 -1.6465  0.1568 -2.3547  0.0676 -0.000991  2.8899  0.2704  0.2199  0.1573
  pga -2.1780  1.6355 -0.1241 -1.8303  0.2165 -1.8318  0.1622 -1.9899  0.0678 -0.002073  2.3460  0.3323  0.2858  0.1695
 0.03 -1.3946  1.5566 -0.1210 -2.1447  0.2324 -2.0955  0.1777 -1.7240  0.0891 -0.003079  1.5613  0.3642  0.2951  0.2133
 0.04 -1.5298  1.5684 -0.1181 -1.8704  0.1906 -1.8175  0.1420 -1.5458  0.0827 -0.003522  1.4637  0.3573  0.2977  0.1975
 0.05 -1.6747  1.5876 -0.1178 -1.7218  0.1719 -1.6779  0.1258 -1.5486  0.0695 -0.003350  1.5137  0.3506  0.2976  0.1853
 0.06 -1.8507  1.6171 -0.1187 -1.5996  0.1587 -1.5637  0.1136 -1.6102  0.0560 -0.002996  1.6318  0.3422  0.2959  0.1718
 0.08 -2.0520  1.6568 -0.1209 -1.5126  0.1517 -1.4804  0.1057 -1.7039  0.0458 -0.002586  1.8079  0.3336  0.2928  0.1598
  0.1 -2.3048  1.7127 -0.1246 -1.4463  0.1480 -1.4141  0.1000 -1.8096  0.0382 -0.002172  2.0171  0.3250  0.2880  0.1506
 0.12 -2.5981  1.7821 -0.1295 -1.3958  0.1454 -1.3622  0.0956 -1.9028  0.0331 -0.001822  2.1704  0.3174  0.2821  0.1454
 0.16 -2.9711  1.8735 -0.1360 -1.3507  0.1428 -1.3141  0.0915 -1.9872  0.0290 -0.001508  2.2948  0.3097  0.2745  0.1433
  0.2 -3.4163  1.9839 -0.1439 -1.3080  0.1393 -1.2687  0.0872 -2.0567  0.0256 -0.001243  2.3514  0.3023  0.2657  0.1441
 0.25 -3.9043  2.1039 -0.1522 -1.2667  0.1347 -1.2260  0.0829 -2.1103  0.0228 -0.001030  2.3495  0.2954  0.2563  0.1470
 0.31 -4.4356  2.2302 -0.1606 -1.2230  0.1286 -1.1838  0.0783 -2.1517  0.0202 -0.000850  2.2892  0.2889  0.2462  0.1512
  0.4 -4.9975  2.3558 -0.1683 -1.1762  0.1213 -1.1419  0.0736 -2.1841  0.0178 -0.000694  2.1896  0.2827  0.2355  0.1563
  0.5 -5.5468  2.4664 -0.1741 -1.1287  0.1135 -1.1038  0.0693 -2.2080  0.0155 -0.000560  2.0648  0.2770  0.2248  0.1618
 0.63 -6.0465  2.5499 -0.1771 -1.0857  0.1067 -1.0742  0.0663 -2.2253  0.0133 -0.000442  1.9685  0.2719  0.2144  0.1673
 0.79 -6.4705  2.5968 -0.1765 -1.0512  0.1016 -1.0584  0.0655 -2.2384  0.0115 -0.000334  1.8968  0.2676  0.2044  0.1726
    1 -6.7771  2.5981 -0.1719 -1.0343  0.1002 -1.0642  0.0681 -2.2491  0.0102 -0.000239  1.8893  0.2640  0.1954  0.1775
 1.25 -6.9495  2.5538 -0.1637 -1.0411  0.1032 -1.0957  0.0748 -2.2593  0.0101 -0.000157  1.9356  0.2610  0.1872  0.1819
 1.59 -6.9977  2.4597 -0.1513 -1.0802  0.1121 -1.1604  0.0866 -2.2724  0.0117 -0.000086  2.0388  0.2580  0.1787  0.1862
    2 -6.9298  2.3348 -0.1370 -1.1486  0.1257 -1.2477  0.1018 -2.2897  0.0158 -0.000043  2.1604  0.2550  0.1703  0.1898
  2.5 -6.7909  2.1932 -0.1220 -1.2383  0.1424 -1.3473  0.1184 -2.3139  0.0230 -0.000029  2.2903  0.2517  0.1616  0.1930
 3.13 -6.6206  2.0449 -0.1069 -1.3397  0.1604 -1.4492  0.1350 -2.3503  0.0340 -0.000047  2.4199  0.2480  0.1523  0.1957
    4 -6.4512  1.8930 -0.0918 -1.4444  0.1782 -1.5462  0.1501 -2.4088  0.0503 -0.000098  2.5430  0.2440  0.1423  0.1982
    5 -6.3504  1.7758 -0.0801 -1.5184  0.1896 -1.6103  0.1596 -2.4808  0.0678 -0.000161  2.6018  0.2407  0.1339  0.2001
    """)


class RietbrockEdwards2019Low(RietbrockEdwards2019Mean):
    COEFFS = CoeffsTable(sa_damping=5, table="""\
IMT c1 c2 c3 c4 c5 c6 c7 c8 c9 c10 c11 total tau phi
pgv -4.05078870 1.46427880 -0.08998341 -1.60167410 0.20356859 -1.63160860 0.15821518 -2.35049000 0.07392323 -0.00102311 2.58615440 0.2704 0.2199 0.1573
pga -1.60259970 1.37285850 -0.10353892 -1.81282250 0.21307317 -1.82589140 0.16432264 -2.00090990 0.07491388 -0.00207667 2.11450340 0.3323 0.2858 0.1695
0.03 -0.79003607 1.28300850 -0.09942094 -2.14492450 0.23583211 -2.09833560 0.18115663 -1.74644450 0.09577206 -0.00304929 1.58617360 0.3642 0.2951 0.2133
0.04 -0.90611552 1.28711160 -0.09581079 -1.86923660 0.19267400 -1.82419210 0.14569685 -1.56107040 0.08927442 -0.00350578 1.44234570 0.3573 0.2977 0.1975
0.05 -1.04711370 1.30468530 -0.09530169 -1.71853200 0.17328611 -1.68445320 0.12936888 -1.55828000 0.07651830 -0.00335303 1.47727750 0.3506 0.2976 0.1853
0.06 -1.22529050 1.33456630 -0.09619364 -1.59144300 0.15873918 -1.56825330 0.11683660 -1.61530290 0.06338805 -0.00301709 1.55257230 0.3422 0.2959 0.1718
0.08 -1.43321740 1.37676870 -0.09855742 -1.49850160 0.14999751 -1.48220220 0.10845706 -1.70529800 0.05342553 -0.00262136 1.67266580 0.3336 0.2928 0.1598
0.10 -1.69652830 1.43701970 -0.10263528 -1.42558870 0.14447179 -1.41274050 0.10225828 -1.80738030 0.04604166 -0.00221917 1.81493860 0.3250 0.2880 0.1506
0.12 -2.00184330 1.51208770 -0.10808787 -1.37084800 0.14098987 -1.35835070 0.09757216 -1.89632580 0.04094712 -0.00188155 1.94759430 0.3174 0.2821 0.1454
0.16 -2.38953910 1.61139970 -0.11549289 -1.32360220 0.13843898 -1.30823810 0.09318406 -1.97527920 0.03679033 -0.00158092 2.08350470 0.3097 0.2745 0.1433
0.20 -2.85337300 1.73236370 -0.12453166 -1.28020540 0.13551058 -1.26116250 0.08880509 -2.03827740 0.03326299 -0.00133074 2.17943210 0.3023 0.2657 0.1441
0.25 -3.36449850 1.86576200 -0.13436886 -1.24057390 0.13208753 -1.21750150 0.08442438 -2.08503360 0.03018515 -0.00113085 2.22844100 0.2954 0.2563 0.1470
0.31 -3.92691690 2.00971750 -0.14469228 -1.19983970 0.12736874 -1.17462020 0.07984518 -2.11938810 0.02723030 -0.00096440 2.21646870 0.2889 0.2462 0.1512
0.40 -4.53146080 2.15840620 -0.15486393 -1.15608380 0.12116727 -1.13208020 0.07509116 -2.14463720 0.02428715 -0.00082028 2.14497910 0.2827 0.2355 0.1563
0.50 -5.13557600 2.29730030 -0.16364663 -1.11209000 0.11445474 -1.09326350 0.07073403 -2.16161500 0.02135015 -0.00069608 2.05140710 0.2770 0.2248 0.1618
0.63 -5.70334260 2.41394850 -0.17000065 -1.07006960 0.10775827 -1.06176940 0.06740720 -2.17291110 0.01848641 -0.00058557 1.94452320 0.2719 0.2144 0.1673
0.79 -6.20728400 2.49807490 -0.17307640 -1.03571610 0.10261704 -1.04302850 0.06607659 -2.18072640 0.01579112 -0.00048295 1.86409070 0.2676 0.2044 0.1726
1.00 -6.60027110 2.53752730 -0.17211859 -1.01708030 0.10057252 -1.04434490 0.06790519 -2.18706150 0.01359976 -0.00038858 1.82909740 0.2640 0.1954 0.1775
1.25 -6.85660140 2.52859420 -0.16715486 -1.02108720 0.10277088 -1.07045080 0.07361143 -2.19391070 0.01239902 -0.00030534 1.84418640 0.2610 0.1872 0.1819
1.59 -6.98749290 2.46775380 -0.15766195 -1.05638850 0.11072683 -1.12910930 0.08453909 -2.20447290 0.01286382 -0.00022919 1.91129290 0.2580 0.1787 0.1862
2.00 -6.98306750 2.36691560 -0.14535246 -1.12123440 0.12363125 -1.21181040 0.09903945 -2.22023740 0.01593622 -0.00017758 2.01133540 0.2550 0.1703 0.1898
2.50 -6.88527590 2.23966130 -0.13133178 -1.20922230 0.14002460 -1.30921710 0.11554340 -2.24444960 0.02230296 -0.00015407 2.11738920 0.2517 0.1616 0.1930
3.13 -6.73371000 2.09612620 -0.11630054 -1.31027640 0.15779586 -1.41112190 0.13237332 -2.28236380 0.03279477 -0.00016054 2.20057140 0.2480 0.1523 0.1957
4.00 -6.55726550 1.93825510 -0.10018965 -1.41727510 0.17610179 -1.51131020 0.14847939 -2.34423460 0.04903702 -0.00019987 2.29561760 0.2440 0.1423 0.1982
5.00 -6.42782830 1.80694050 -0.08686407 -1.49651890 0.18916280 -1.58086690 0.15925255 -2.42120890 0.06725696 -0.00025529 2.37054260 0.2407 0.1339 0.2001
""")


class RietbrockEdwards2019Up(RietbrockEdwards2019Mean):
    COEFFS = CoeffsTable(sa_damping=5, table="""\
 IMT            c1           c2            c3            c4           c5            c6           c7            c8           c9           c10          c11  total    tau    phi
pgv  -4.9816838000 1.8789784000 -0.1225694500 -1.6433560000 0.2147766200 -1.6443706000 0.1547758200 -2.2806838000 0.0606829370 -0.0010935213 3.2639678000 0.2704 0.2199 0.1573
pga  -2.8890438000 1.9430938000 -0.1495854600 -1.8472696000 0.2209417900 -1.8290992000 0.1597624600 -1.9370594000 0.0632183350 -0.0021546010 2.5369700000 0.3323 0.2858 0.1695
0.03 -2.1156934000 1.8705434000 -0.1472062100 -2.1624550000 0.2354056400 -2.0928843000 0.1757926000 -1.6724533000 0.0860580350 -0.0031813478 1.6568565000 0.3642 0.2951 0.2133
0.04 -2.2877062000 1.8938687000 -0.1453637700 -1.8698183000 0.1908067100 -1.8053171000 0.1388823400 -1.5007491000 0.0795500100 -0.0036060903 1.5483894000 0.3573 0.2977 0.1975
0.05 -2.4460150000 1.9176132000 -0.1454325400 -1.7166526000 0.1715511600 -1.6637661000 0.1223268900 -1.5076678000 0.0661223340 -0.0034181412 1.6117140000 0.3506 0.2976 0.1853
0.06 -2.6298308000 1.9492395000 -0.1464915600 -1.5915475000 0.1582005100 -1.5486034000 0.1099611100 -1.5718625000 0.0522047140 -0.0030503915 1.7469826000 0.3422 0.2959 0.1718
0.08 -2.8307059000 1.9881561000 -0.1485462900 -1.5058464000 0.1522958900 -1.4657355000 0.1020838300 -1.6653204000 0.0415340580 -0.0026332462 1.9869103000 0.3336 0.2928 0.1598
0.10 -3.0774070000 2.0403979000 -0.1518020500 -1.4397947000 0.1489292800 -1.3996737000 0.0963509340 -1.7699356000 0.0335532470 -0.0022139177 2.2111670000 0.325  0.288  0.1506
0.12 -3.3588080000 2.1032578000 -0.1559664400 -1.3895383000 0.1466020600 -1.3479606000 0.0920333910 -1.8606712000 0.0279983910 -0.0018633807 2.3836040000 0.3174 0.2821 0.1454
0.16 -3.7125483000 2.1841739000 -0.1613883900 -1.3415308000 0.1432496400 -1.2992204000 0.0878464610 -1.9416882000 0.0234928810 -0.0015507221 2.4829925000 0.3097 0.2745 0.1433
0.20 -4.1266541000 2.2790044000 -0.1676410800 -1.2986449000 0.1399474800 -1.2540464000 0.0837600260 -2.0068362000 0.0197410390 -0.0012898922 2.5586652000 0.3023 0.2657 0.1441
0.25 -4.5741285000 2.3788992000 -0.1739195400 -1.2558993000 0.1351325600 -1.2113657000 0.0796017520 -2.0558883000 0.0166173510 -0.0010812314 2.5561511000 0.2954 0.2563 0.147
0.31 -5.0532534000 2.4799325000 -0.1797357500 -1.2106965000 0.1287642800 -1.1693226000 0.0752791130 -2.0928919000 0.0138181050 -0.0009074376 2.4881098000 0.2889 0.2462 0.1512
0.40 -5.5507974000 2.5747522000 -0.1843312700 -1.1609460000 0.1206814400 -1.1275806000 0.0708689960 -2.1210247000 0.0112592820 -0.0007579100 2.3573241000 0.2827 0.2355 0.1563
0.50 -6.0249280000 2.6502658000 -0.1866701700 -1.1120213000 0.1124871600 -1.0903864000 0.0670389270 -2.1412864000 0.0089653868 -0.0006301900 2.2141131000 0.277  0.2248 0.1618
0.63 -6.4425123000 2.6961429000 -0.1859792100 -1.0674637000 0.1050725100 -1.0620167000 0.0645216780 -2.1560319000 0.0069921178 -0.0005184826 2.0774605000 0.2719 0.2144 0.1673
0.79 -6.7785112000 2.7039614000 -0.1816479400 -1.0337643000 0.1002312000 -1.0486458000 0.0643952410 -2.1672652000 0.0054542399 -0.0004177409 1.9994682000 0.2676 0.2044 0.1726
1.00 -6.9991518000 2.6680998000 -0.1735975900 -1.0188091000 0.0996645640 -1.0576641000 0.0677898930 -2.1764288000 0.0046443344 -0.0003293746 2.0208491000 0.264  0.1954 0.1775
1.25 -7.0947910000 2.5918299000 -0.1625220300 -1.0279357000 0.1041191100 -1.0929838000 0.0752154880 -2.1853865000 0.0049835925 -0.0002562350 2.1420859000 0.261  0.1872 0.1819
1.59 -7.0733139000 2.4701997000 -0.1477864400 -1.0685423000 0.1151509200 -1.1620181000 0.0879039360 -2.1969973000 0.0072376443 -0.0001962806 2.3910408000 0.258  0.1787 0.1862
2.00 -6.9533573000 2.3271338000 -0.1321765800 -1.1413338000 0.1323921200 -1.2552796000 0.1038796800 -2.2128362000 0.0119140890 -0.0001626182 2.7515016000 0.255  0.1703 0.1898
2.50 -6.7808107000 2.1765794000 -0.1167466000 -1.2397545000 0.1534875000 -1.3615027000 0.1212399900 -2.2365279000 0.0195113130 -0.0001561904 3.1259577000 0.2517 0.1616 0.193
3.13 -6.5978140000 2.0284144000 -0.1021115000 -1.3523298000 0.1754121900 -1.4684528000 0.1380543200 -2.2726833000 0.0306087950 -0.0001780563 3.4367318000 0.248  0.1523 0.1957
4.00 -6.4312318000 1.8852996000 -0.0883183780 -1.4847659000 0.2004538200 -1.5725205000 0.1534078500 -2.3314115000 0.0463264380 -0.0002269082 3.7836231000 0.244  0.1423 0.1982
5.00 -6.3575321000 1.7829213000 -0.0784036930 -1.5682796000 0.2134715300 -1.6358638000 0.1622203500 -2.4008112000 0.0627602400 -0.0002901302 3.8793342000 0.2407 0.1339 0.2001
""")

