import torch.utils.data as data
from nerf.model import NeRF


class PixelRayDataset(data.Dataset):
    """Generate a dataset containing rays from a pinhole camera with the
    specified focal length with an imaging plane of size equal to the
    height and width of the provided images tensor

    Arguments:

    images: torch.Tensor
        a torch tensor of values on [0, 1] representing images to be
        generated by a neural radiance field: [batch, height, width, 3]
    poses: torch.Tensor
        a torch tensor containing poses of a camera represented by a
        transformation in homogeneous coordinates: [batch, 4, 4]
    focal_length: float
        a float representing the focal length of a pinhole camera and
        used to scale the physical size of the imaging plane

    """

    def __init__(self, images, poses, focal_length):
        """Generate a dataset containing rays from a pinhole camera with the
        specified focal length with an imaging plane of size equal to the
        height and width of the provided images tensor

        Arguments:

        images: torch.Tensor
            a torch tensor of values on [0, 1] representing images to be
            generated by a neural radiance field: [batch, height, width, 3]
        poses: torch.Tensor
            a torch tensor containing poses of a camera represented by a
            transformation in homogeneous coordinates: [batch, 4, 4]
        focal_length: float
            a float representing the focal length of a pinhole camera and
            used to scale the physical size of the imaging plane

        """

        self.images = images
        self.poses = poses

        # generate all possible rays to cast through the scene
        kwargs = dict(dtype=images.dtype, device=images.device)
        self.rays = NeRF.generate_rays(images.shape[1], images.shape[2],
                                       focal_length, **kwargs)

    def __len__(self):
        """Return the size of the a dataset containing pixels and rays,
        where each example in the dataset represents a single pixel
        and ray cast through the scene in world coordinates

        Returns:

        dataset_size: int
            the integer size of the dataset, which represents the total
            number of pixels in all images in the dataset

        """
        return (self.images.shape[0] *
                self.images.shape[1] * self.images.shape[2])

    def __getitem__(self, idx):
        """Retrieve a single example from the dataset by selecting the
        corresponding pose, pixel, and ray, transforming each ray to world
        coordinates, and returning pairs of rays and pixels

        Arguments:

        idx: int
            the example id in the dataset to load, which encodes the image
            id, and height width location of a pixel

        Returns:

        pixels: torch.Tensor
            a tensor that represents a pixel to be rendered for the given
            rays by alpha compositing predicted colors in space
        rays_o: torch.Tensor
            a single ray location in world coordinates, represented as a
            3-vector and paired with a single ray direction
        rays_d: torch.Tensor
            a single ray direction in world coordinates, represented as a
            3-vector and paired with a single ray location

        """

        # select the column of the image to select pixels from
        image_wi = idx % self.images.shape[2]
        idx = idx // self.images.shape[2]

        # select the row of the image to select pixels from
        image_hi = idx % self.images.shape[1]
        idx = idx // self.images.shape[1]

        # select the image id to select pixels from
        image_bi = idx % self.images.shape[0]

        # select a pixel from the image and the pose of the camera
        pixel = self.images[image_bi, image_hi, image_wi]
        pose = self.poses[image_bi]

        # transform the ray to world coordinates using the pose
        return (pixel, *NeRF.rays_to_world_coordinates(
            self.rays[image_hi, image_wi], pose[:3, 3], pose[:3, :3]))
