# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['misty2py', 'misty2py.basic_skills', 'misty2py.utils']

package_data = \
{'': ['*']}

install_requires = \
['python-dotenv>=0.17.1,<0.18.0',
 'requests>=2.25.1,<3.0.0',
 'websocket-client>=0.58.0,<0.59.0']

setup_kwargs = {
    'name': 'misty2py',
    'version': '5.0.0',
    'description': 'A Python package for Misty II development',
    'long_description': '# Misty2py\n\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black) [![GitHub license](https://img.shields.io/github/license/Naereen/StrapDown.js.svg)](https://github.com/ChrisScarred/misty2py/blob/main/LICENSE)\n\nMisty2py is a Python 3 package for Misty II development using [Misty\'s REST API](https://docs.mistyrobotics.com/misty-ii/rest-api/api-reference/ "Misty Robotics REST API").\n\nRead the full documentation [here](https://chrisscarred.github.io/misty2py)!\n\n## Installation\n\n### Poetry\n\nTo install misty2py, run `pip install misty2py`.\n\n### From source\n\n- If this is your first time using `misty2py` from source, do following:\n\n  - Get Poetry (`python -m pip install poetry`) if you do not have it yet.\n  - Copy `.env.example` to `.env`.\n  - Replace the placeholder values in the new `.env` file.\n  - Run `poetry install` to obtain all dependencies.\n\n- Run the desired script via `poetry run python -m [name]` where `[name]` is the placeholder for the module location (in Python notation).\n- If the scripts run but your Misty does not seem to respond, you have most likely provided an incorrect IP address for `MISTY_IP_ADDRESS` in `.env`.\n- Pytests can be run via `poetry run pytest .`.\n- The coverage report can be obtained via `poetry run pytest --cov-report html --cov=misty2py tests` for HTML output or via `poetry run pytest --cov=misty2py tests` for terminal output.\n\n## Features\n\nMisty2py can be used to develop complex skills (behaviours) for the Misty II robot utilising:\n\n- **actions** via sending a `POST` or `DELETE` requests to Misty\'s API;\n- **informations** via sending a `GET` request to Misty\'s API;\n- **continuous streams of data** via subscribing to event types on Misty\'s websockets.\n\nMisty2py uses following concepts for easy of usage:\n\n- **action keywords** - customisable python-styled keywords for endpoints of Misty\'s API that correspond to performing actions;\n- **information keywords** - customisable python-styled keywords for endpoints of Misty\'s API that correspond to retrieving information;\n- **data shortcuts** - customisable python-styled keywords for commonly used data that are supplied to Misty\'s API as the body of a `POST` request.\n\n## Usage\n\n### Getting started\n\nThe main object of this package is `Misty`, which is an abstract representation of Misty the robot. To initialise this object, it is required to know the IP address of the Misty robot that should be used.\n\nThe most direct way to initialise a `Misty` object is to use the IP address directly, which allows the user to get the object in one step via:\n\n``` python\nfrom misty2py.robot import Misty\n\nmy_misty = Misty("192.168.0.1")  #example IP address\n```\n\nThis may be impractical and potentially even unsafe, so it is recommended to create a .env file in the project\'s directory, specify the IP address there via `MISTY_IP_ADDRESS="[ip_address_here]"` and use Misty2py\'s `EnvLoader` to load the IP address via:\n\n\n``` python\nfrom misty2py.robot import Misty\nfrom misty2py.utils.env_loader import EnvLoader\n\nenv_loader = EnvLoader()\nmy_misty = Misty(env_loader.get_ip())\n```\n\nAssuming a `Misty` object called `my_misty` was obtained, all required actions can be performed via the following three methods:\n\n``` python\n# Performing an action (a POST or DELETE request):\nmy_misty.perform_action("<action_keyword>")\n\n# Obtaining information (a GET request):\nmy_misty.get_info("<information_keyword>")\n\n# Event related methods \n# (subscribing to an event, getting event data\n# or event log and unsubscribing from an event):\nmy_misty.event("<parameter>")\n```\n\n### Responses\n\nAny action performed via Misty2py which contains communication with Misty\'s APIs returns the `Misty2pyResponse` object. `Misty2pyResponse` is a uniform representation of two sub-responses that are present in any HTTP or WebSocket communication with Misty\'s APIs using Misty2py. The first sub-response is always from Misty2py and is represented by the attributes `Misty2pyResponse.misty2py_status` (`True` if no Misty2py-related errors were encountered) and potentially empty `Misty2pyResponse.error_msg` and `Misty2pyResponse.error_type` that contain error information if a Misty2py-related error was encountered. The other sub-response is either from Misty\'s REST API or Misty\'s WebSocket API. In the first case, it is represented by the attribute `Misty2pyResponse.rest_response` (Dict), and in the second case, it is represented by the attribute `Misty2pyResponse.ws_response`. One of these is always empty, because no action in Misty2py includes simultaneous communication with both APIs. For convenience, a `Misty2pyResponse` object can be easily parsed to a dictionary via the method `Misty2pyResponse.parse_to_dict`.\n\n### Obtaining information\n\nObtaining digital information is handled by `misty2py.robot::get_info` method which has two arguments. The argument `info_name` is required and it specifies the string information keyword corresponding to an endpoint in Misty\'s REST API. The argument `params` is optional and it supplies a dictionary of parameter name and parameter value pairs. This argument defaults to `{}` (an empty dictionary).\n\n### Performing actions\n\nPerforming physical and digital actions including removal of non-system files is handled by `misty2py.robot::perform_action()` method which takes two arguments. The argument `action_name` is required and it specifies the string action keyword corresponding to an endpoint in Misty’s REST API. The second argument, `data`, is optional and it specifies the data to pass to the request as a dictionary or a data shortcut (string). The `data` argument defaults to `{}` (an empty dictionary).\n\n### Event types\n\nMisty\'s WebSocket API follows PUB-SUB architecture, which means that in order to obtain event data in Misty\'s framework, it is required to **subscribe** to an event type on Misty\'s WebSocket API. The WebSocket server then streams data to the WebSocket client, which receives it a separate thread. To **access the data,** `misty2py.robot::event` method must be called with `"get_data"` parameter from the main thread. When the data are no longer required to be streamed to the client, an event type can be **unsubscribed** which both kills the event thread and stops the API from sending more data.\n\n**Subscribing** to an event is done via `misty2py.robot::event` with the parameter `"subscribe"` and following keyword arguments:\n\n- `type` - *required;* event type string as documented in [Event Types Docs](https://docs.mistyrobotics.com/misty-ii/robot/sensor-data/ "Misty Robotics Event Types").\n- `name` - *optional;* a custom event name string; must be unique.\n- `return_property` - *optional;* the property to return from Misty\'s websockets; all properties are returned if return_property is not supplied.\n- `debounce` - *optional;* the interval in ms at which new information is sent; defaults to `250`.\n- `len_data_entries` - *optional;* the maximum number of data entries to keep (discards in fifo style); defaults to `10`.\n- `event_emitter` - *optional;* an event emitter function which emits an event upon message recieval. Supplies the message content as an argument.\n\n**Accessing the data** of an event or its log is done via `misty2py.robot::event` with the parameter `"get_data"` or `"get_log"` and a keyword argument `name` (the name of the event).\n\n**Unsubscribing** from an event is done via `misty2py.robot::event` with the parameter `"unsubscribe"` and a keyword argument `name` (the name of the event).\n\nA bare-bones implementation of event subscription can be seen below.\n\n```python\nimport time\n\nfrom misty2py.robot import Misty\nfrom misty2py.utils.env_loader import EnvLoader\n\nenv_loader = EnvLoader\n\nm = Misty(env_loader.get_ip())\n\nd = m.event("subscribe", type = "BatteryCharge")\ne_name = d.get("event_name")\n\ntime.sleep(1)\n\nd = m.event("get_data", name = e_name)\n\nd = m.event("unsubscribe", name = e_name)\n```\n\nThe following example shows a more realistic scenario which includes an event emitter and an event listener.\n\n```python\nimport time\nfrom pymitter import EventEmitter\n\nfrom misty2py.robot import Misty\nfrom misty2py.utils.env_loader import EnvLoader\n\nenv_loader = EnvLoader\n\nm = Misty(env_loader.get_ip())\nee = EventEmitter()\nevent_name = "myevent_001"\n\n@ee.on(event_name)\ndef listener(data):\n    print(data)\n\nd = m.event("subscribe", type = "BatteryCharge", \n            name = event_name, event_emitter = ee)\n\ntime.sleep(2)\n\nd = m.event("unsubscribe", name = event_name)\n```\n\n### Adding custom keywords and shortcuts\n\nCustom keywords and shortcuts can be passed to a Misty object while declaring a new instance by using the optional arguments `custom_info`, `custom_actions` and `custom_data`. \n\nThe argument `custom_info` can be used to pass custom information keywords as a dictionary with keys being the information keywords and values being the endpoints. An information keyword can only be used for a `GET` method supporting endpoint.\n\nThe argument `custom_actions` can be used to pass custom action keywords as a dictionary with keys being the action keywords and values being a dictionary of an `"endpoint"` key (str) and a `"method"` key (str). The `"method"` values must be one of `post`, `delete`, `put`, `head`, `options` and `patch`. However, it should be noted that Misty\'s REST API currently only has `GET`, `POST` and `DELETE` methods. The rest of the methods was implement in Misty2py for forwards-compatibility.\n\nThe argument `custom_data` can be used to pass custom data shortcuts as a dictionary with keys being the data shortcuts and values being the dictionary of data values.\n\nFor futher illustration, an example of passing custom keywords and shortcuts can be seen below.\n\n```python\ncustom_allowed_infos = {\n    "hazards_settings": "api/hazards/settings"\n}\n\ncustom_allowed_data = {\n    "amazement": {\n        "FileName": "s_Amazement.wav"\n    },\n    "red": {\n        "red": "255",\n        "green": "0",\n        "blue": "0"\n    }\n}\n\ncustom_allowed_actions = {\n    "audio_play" : {\n        "endpoint" : "api/audio/play",\n        "method" : "post"\n    },\n    "delete_audio" : {\n        "endpoint" : "api/audio",\n        "method" : "delete"\n    }\n}\n\nmisty_robot = Misty("0.0.0.0", \n    custom_info=custom_allowed_infos, \n    custom_actions=custom_allowed_actions, \n    custom_data=custom_allowed_data)\n```',
    'author': "Chris 'Scar(R)ed' Vajdík",
    'author_email': 'scarred.chris@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://chrisscarred.github.io/misty2py',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
