#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# Copyright (c) 2007 - 2022 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
eric Editor.

This is the main Python script that performs the necessary initialization
of the MiniEditor module and starts the Qt event loop. This is a standalone
version of the integrated MiniEditor module.
"""

import sys
import os

sys.path.insert(1, os.path.dirname(__file__))

for arg in sys.argv[:]:
    if arg.startswith("--config="):
        import Globals

        configDir = arg.replace("--config=", "")
        Globals.setConfigDir(configDir)
        sys.argv.remove(arg)
    elif arg.startswith("--settings="):
        from PyQt6.QtCore import QSettings

        settingsDir = os.path.expanduser(arg.replace("--settings=", ""))
        if not os.path.isdir(settingsDir):
            os.makedirs(settingsDir)
        QSettings.setPath(
            QSettings.Format.IniFormat, QSettings.Scope.UserScope, settingsDir
        )
        sys.argv.remove(arg)

from Globals import AppInfo

from Toolbox import Startup


def createMainWidget(argv):
    """
    Function to create the main widget.

    @param argv list of commandline parameters (list of strings)
    @return reference to the main widget (QWidget)
    """
    from QScintilla.MiniEditor import MiniEditor

    if len(argv) > 1:
        return MiniEditor(argv[1])
    else:
        return MiniEditor()


def main():
    """
    Main entry point into the application.
    """
    from PyQt6.QtGui import QGuiApplication

    QGuiApplication.setDesktopFileName("eric7_editor.desktop")

    options = [
        (
            "--config=configDir",
            "use the given directory as the one containing the config files",
        ),
        (
            "--settings=settingsDir",
            "use the given directory to store the settings files",
        ),
        ("", "name of file to edit"),
    ]
    appinfo = AppInfo.makeAppInfo(
        sys.argv, "eric Editor", "", "Simplified version of the eric editor", options
    )
    res = Startup.simpleAppStartup(sys.argv, appinfo, createMainWidget)
    sys.exit(res)


if __name__ == "__main__":
    main()
