"use strict";
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
function mockVpcContextProviderWith(test, response, paramValidator) {
    const previous = core_1.ContextProvider.getValue;
    core_1.ContextProvider.getValue = (_scope, options) => {
        // do some basic sanity checks
        test.equal(options.provider, cxapi.VPC_PROVIDER, `Expected provider to be: '${cxapi.VPC_PROVIDER}', got: '${options.provider}'`);
        test.equal((options.props || {}).returnAsymmetricSubnets, true, `Expected options.props.returnAsymmetricSubnets to be true, got: '${(options.props || {}).returnAsymmetricSubnets}'`);
        if (paramValidator) {
            paramValidator(options.props);
        }
        return {
            value: {
                availabilityZones: [],
                isolatedSubnetIds: undefined,
                isolatedSubnetNames: undefined,
                isolatedSubnetRouteTableIds: undefined,
                privateSubnetIds: undefined,
                privateSubnetNames: undefined,
                privateSubnetRouteTableIds: undefined,
                publicSubnetIds: undefined,
                publicSubnetNames: undefined,
                publicSubnetRouteTableIds: undefined,
                ...response,
            },
        };
    };
    return previous;
}
function restoreContextProvider(previous) {
    core_1.ContextProvider.getValue = previous;
}
module.exports = {
    'Vpc.fromLookup()': {
        'requires concrete values'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            test.throws(() => {
                lib_1.Vpc.fromLookup(stack, 'Vpc', {
                    vpcId: core_1.Lazy.stringValue({ produce: () => 'some-id' }),
                });
            }, 'All arguments to Vpc.fromLookup() must be concrete');
            test.done();
        },
        'selecting subnets by name from a looked-up VPC does not throw'(test) {
            // GIVEN
            const stack = new core_1.Stack(undefined, undefined, { env: { region: 'us-east-1', account: '123456789012' } });
            const vpc = lib_1.Vpc.fromLookup(stack, 'VPC', {
                vpcId: 'vpc-1234',
            });
            // WHEN
            vpc.selectSubnets({ subnetName: 'Bleep' });
            // THEN: no exception
            test.done();
        },
        'accepts asymmetric subnets'(test) {
            const previous = mockVpcContextProviderWith(test, {
                vpcId: 'vpc-1234',
                subnetGroups: [
                    {
                        name: 'Public',
                        type: cxapi.VpcSubnetGroupType.PUBLIC,
                        subnets: [
                            {
                                subnetId: 'pub-sub-in-us-east-1a',
                                availabilityZone: 'us-east-1a',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pub-sub-in-us-east-1b',
                                availabilityZone: 'us-east-1b',
                                routeTableId: 'rt-123',
                            },
                        ],
                    },
                    {
                        name: 'Private',
                        type: cxapi.VpcSubnetGroupType.PRIVATE,
                        subnets: [
                            {
                                subnetId: 'pri-sub-1-in-us-east-1c',
                                availabilityZone: 'us-east-1c',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pri-sub-2-in-us-east-1c',
                                availabilityZone: 'us-east-1c',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pri-sub-1-in-us-east-1d',
                                availabilityZone: 'us-east-1d',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pri-sub-2-in-us-east-1d',
                                availabilityZone: 'us-east-1d',
                                routeTableId: 'rt-123',
                            },
                        ],
                    },
                ],
            }, options => {
                test.deepEqual(options.filter, {
                    isDefault: 'true',
                });
                test.equal(options.subnetGroupNameTag, undefined);
            });
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromLookup(stack, 'Vpc', {
                isDefault: true,
            });
            test.deepEqual(vpc.availabilityZones, ['us-east-1a', 'us-east-1b', 'us-east-1c', 'us-east-1d']);
            test.equal(vpc.publicSubnets.length, 2);
            test.equal(vpc.privateSubnets.length, 4);
            test.equal(vpc.isolatedSubnets.length, 0);
            restoreContextProvider(previous);
            test.done();
        },
        'selectSubnets onePerAz works on imported VPC'(test) {
            const previous = mockVpcContextProviderWith(test, {
                vpcId: 'vpc-1234',
                subnetGroups: [
                    {
                        name: 'Public',
                        type: cxapi.VpcSubnetGroupType.PUBLIC,
                        subnets: [
                            {
                                subnetId: 'pub-sub-in-us-east-1a',
                                availabilityZone: 'us-east-1a',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pub-sub-in-us-east-1b',
                                availabilityZone: 'us-east-1b',
                                routeTableId: 'rt-123',
                            },
                        ],
                    },
                    {
                        name: 'Private',
                        type: cxapi.VpcSubnetGroupType.PRIVATE,
                        subnets: [
                            {
                                subnetId: 'pri-sub-1-in-us-east-1c',
                                availabilityZone: 'us-east-1c',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pri-sub-2-in-us-east-1c',
                                availabilityZone: 'us-east-1c',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pri-sub-1-in-us-east-1d',
                                availabilityZone: 'us-east-1d',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pri-sub-2-in-us-east-1d',
                                availabilityZone: 'us-east-1d',
                                routeTableId: 'rt-123',
                            },
                        ],
                    },
                ],
            }, options => {
                test.deepEqual(options.filter, {
                    isDefault: 'true',
                });
                test.equal(options.subnetGroupNameTag, undefined);
            });
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromLookup(stack, 'Vpc', {
                isDefault: true,
            });
            // WHEN
            const subnets = vpc.selectSubnets({ subnetType: lib_1.SubnetType.PRIVATE, onePerAz: true });
            // THEN: we got 2 subnets and not 4
            test.deepEqual(subnets.subnets.map(s => s.availabilityZone), ['us-east-1c', 'us-east-1d']);
            restoreContextProvider(previous);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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