"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'security group can allows all outbound traffic by default'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.SecurityGroup(stack, 'SG1', { vpc, allowAllOutbound: true });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupEgress: [
                {
                    CidrIp: '0.0.0.0/0',
                    Description: 'Allow all outbound traffic by default',
                    IpProtocol: '-1',
                },
            ],
        }));
        test.done();
    },
    'no new outbound rule is added if we are allowing all traffic anyway'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        // WHEN
        const sg = new lib_1.SecurityGroup(stack, 'SG1', { vpc, allowAllOutbound: true });
        sg.addEgressRule(lib_1.Peer.anyIpv4(), lib_1.Port.tcp(86), 'This does not show up');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupEgress: [
                {
                    CidrIp: '0.0.0.0/0',
                    Description: 'Allow all outbound traffic by default',
                    IpProtocol: '-1',
                },
            ],
        }));
        test.done();
    },
    'security group disallow outbound traffic by default'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        // WHEN
        new lib_1.SecurityGroup(stack, 'SG1', { vpc, allowAllOutbound: false });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupEgress: [
                {
                    CidrIp: '255.255.255.255/32',
                    Description: 'Disallow all traffic',
                    FromPort: 252,
                    IpProtocol: 'icmp',
                    ToPort: 86,
                },
            ],
        }));
        test.done();
    },
    'bogus outbound rule disappears if another rule is added'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        // WHEN
        const sg = new lib_1.SecurityGroup(stack, 'SG1', { vpc, allowAllOutbound: false });
        sg.addEgressRule(lib_1.Peer.anyIpv4(), lib_1.Port.tcp(86), 'This replaces the other one');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupEgress: [
                {
                    CidrIp: '0.0.0.0/0',
                    Description: 'This replaces the other one',
                    FromPort: 86,
                    IpProtocol: 'tcp',
                    ToPort: 86,
                },
            ],
        }));
        test.done();
    },
    'all outbound rule cannot be added after creation'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VPC');
        // WHEN
        const sg = new lib_1.SecurityGroup(stack, 'SG1', { vpc, allowAllOutbound: false });
        test.throws(() => {
            sg.addEgressRule(lib_1.Peer.anyIpv4(), lib_1.Port.allTraffic(), 'All traffic');
        }, /Cannot add/);
        test.done();
    },
    'immutable imports do not add rules'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const sg = lib_1.SecurityGroup.fromSecurityGroupId(stack, 'SG1', 'test-id', { mutable: false });
        sg.addEgressRule(lib_1.Peer.anyIpv4(), lib_1.Port.tcp(86), 'This rule was not added');
        sg.addIngressRule(lib_1.Peer.anyIpv4(), lib_1.Port.tcp(86), 'This rule was not added');
        assert_1.expect(stack).to(assert_1.not(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupEgress: [
                {
                    CidrIp: '0.0.0.0/0',
                    Description: 'This rule was not added',
                    FromPort: 86,
                    IpProtocol: 'tcp',
                    ToPort: 86,
                },
            ],
        })));
        assert_1.expect(stack).to(assert_1.not(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupIngress: [
                {
                    CidrIp: '0.0.0.0/0',
                    Description: 'This rule was not added',
                    FromPort: 86,
                    IpProtocol: 'tcp',
                    ToPort: 86,
                },
            ],
        })));
        test.done();
    },
    'peer between all types of peers and port range types'(test) {
        // GIVEN
        const stack = new core_1.Stack(undefined, 'TestStack', { env: { account: '12345678', region: 'dummy' } });
        const vpc = new lib_1.Vpc(stack, 'VPC');
        const sg = new lib_1.SecurityGroup(stack, 'SG', { vpc });
        const peers = [
            new lib_1.SecurityGroup(stack, 'PeerGroup', { vpc }),
            lib_1.Peer.anyIpv4(),
            lib_1.Peer.anyIpv6(),
            lib_1.Peer.prefixList('pl-012345'),
        ];
        const ports = [
            lib_1.Port.tcp(1234),
            lib_1.Port.tcp(core_1.Lazy.numberValue({ produce: () => 5000 })),
            lib_1.Port.allTcp(),
            lib_1.Port.tcpRange(80, 90),
            lib_1.Port.udp(2345),
            lib_1.Port.udp(core_1.Lazy.numberValue({ produce: () => 7777 })),
            lib_1.Port.allUdp(),
            lib_1.Port.udpRange(85, 95),
            lib_1.Port.icmpTypeAndCode(5, 1),
            lib_1.Port.icmpType(8),
            lib_1.Port.allIcmp(),
            lib_1.Port.icmpPing(),
            lib_1.Port.allTraffic(),
        ];
        // WHEN
        for (const peer of peers) {
            for (const port of ports) {
                sg.connections.allowTo(peer, port);
                sg.connections.allowFrom(peer, port);
            }
        }
        // THEN -- no crash
        test.done();
    },
    'if tokens are used in ports, `canInlineRule` should be false to avoid cycles'(test) {
        // GIVEN
        const p1 = core_1.Lazy.numberValue({ produce: () => 80 });
        const p2 = core_1.Lazy.numberValue({ produce: () => 5000 });
        // WHEN
        const ports = [
            lib_1.Port.tcp(p1),
            lib_1.Port.tcp(p2),
            lib_1.Port.tcpRange(p1, 90),
            lib_1.Port.tcpRange(80, p2),
            lib_1.Port.tcpRange(p1, p2),
            lib_1.Port.udp(p1),
            lib_1.Port.udpRange(p1, 95),
            lib_1.Port.udpRange(85, p2),
            lib_1.Port.udpRange(p1, p2),
            lib_1.Port.icmpTypeAndCode(p1, 1),
            lib_1.Port.icmpTypeAndCode(5, p1),
            lib_1.Port.icmpTypeAndCode(p1, p2),
            lib_1.Port.icmpType(p1),
        ];
        // THEN
        for (const range of ports) {
            test.equal(range.canInlineRule, false, range.toString());
        }
        test.done();
    },
    'Peer IP CIDR validation': {
        'passes with valid IPv4 CIDR block'(test) {
            // GIVEN
            const cidrIps = ['0.0.0.0/0', '192.168.255.255/24'];
            // THEN
            for (const cidrIp of cidrIps) {
                test.equal(lib_1.Peer.ipv4(cidrIp).uniqueId, cidrIp);
            }
            test.done();
        },
        'passes with unresolved IP CIDR token'(test) {
            // GIVEN
            core_1.Token.asString(new core_1.Intrinsic('ip'));
            // THEN: don't throw
            test.done();
        },
        'throws if invalid IPv4 CIDR block'(test) {
            // THEN
            test.throws(() => {
                lib_1.Peer.ipv4('invalid');
            }, /Invalid IPv4 CIDR/);
            test.done();
        },
        'throws if missing mask in IPv4 CIDR block'(test) {
            test.throws(() => {
                lib_1.Peer.ipv4('0.0.0.0');
            }, /CIDR mask is missing in IPv4/);
            test.done();
        },
        'passes with valid IPv6 CIDR block'(test) {
            // GIVEN
            const cidrIps = [
                '::/0',
                '2001:db8::/32',
                '2001:0db8:0000:0000:0000:8a2e:0370:7334/32',
                '2001:db8::8a2e:370:7334/32',
            ];
            // THEN
            for (const cidrIp of cidrIps) {
                test.equal(lib_1.Peer.ipv6(cidrIp).uniqueId, cidrIp);
            }
            test.done();
        },
        'throws if invalid IPv6 CIDR block'(test) {
            // THEN
            test.throws(() => {
                lib_1.Peer.ipv6('invalid');
            }, /Invalid IPv6 CIDR/);
            test.done();
        },
        'throws if missing mask in IPv6 CIDR block'(test) {
            test.throws(() => {
                lib_1.Peer.ipv6('::');
            }, /IDR mask is missing in IPv6/);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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