"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const _1 = require(".");
const instance_1 = require("./instance");
const machine_image_1 = require("./machine-image");
const port_1 = require("./port");
/**
 * This creates a linux bastion host you can use to connect to other instances or services in your VPC.
 * The recommended way to connect to the bastion host is by using AWS Systems Manager Session Manager.
 *
 * The operating system is Amazon Linux 2 with the latest SSM agent installed
 *
 * You can also configure this bastion host to allow connections via SSH
 *
 * @experimental
 */
class BastionHostLinux extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id);
        this.stack = core_1.Stack.of(scope);
        this.instance = new instance_1.Instance(this, 'Resource', {
            vpc: props.vpc,
            availabilityZone: props.availabilityZone,
            securityGroup: props.securityGroup,
            instanceName: (_a = props.instanceName) !== null && _a !== void 0 ? _a : 'BastionHost',
            instanceType: (_b = props.instanceType) !== null && _b !== void 0 ? _b : _1.InstanceType.of(_1.InstanceClass.T3, _1.InstanceSize.NANO),
            machineImage: (_c = props.machineImage) !== null && _c !== void 0 ? _c : machine_image_1.MachineImage.latestAmazonLinux({ generation: _1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            vpcSubnets: (_d = props.subnetSelection) !== null && _d !== void 0 ? _d : {},
        });
        this.instance.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ssmmessages:*',
                'ssm:UpdateInstanceInformation',
                'ec2messages:*',
            ],
            resources: ['*'],
        }));
        this.instance.addUserData('yum install -y https://s3.amazonaws.com/ec2-downloads-windows/SSMAgent/latest/linux_amd64/amazon-ssm-agent.rpm');
        this.connections = this.instance.connections;
        this.role = this.instance.role;
        this.grantPrincipal = this.instance.role;
        this.instanceId = this.instance.instanceId;
        this.instancePrivateIp = this.instance.instancePrivateIp;
        this.instanceAvailabilityZone = this.instance.instanceAvailabilityZone;
        this.instancePrivateDnsName = this.instance.instancePrivateDnsName;
        this.instancePublicIp = this.instance.instancePublicIp;
        this.instancePublicDnsName = this.instance.instancePublicDnsName;
        new core_1.CfnOutput(this, 'BastionHostId', {
            description: 'Instance ID of the bastion host. Use this to connect via SSM Session Manager',
            value: this.instanceId,
        });
    }
    /**
     * Allow SSH access from the given peer or peers
     *
     * Necessary if you want to connect to the instance using ssh. If not
     * called, you should use SSM Session Manager to connect to the instance.
     */
    allowSshAccessFrom(...peer) {
        peer.forEach(p => {
            this.connections.allowFrom(p, port_1.Port.tcp(22), 'SSH access');
        });
    }
}
exports.BastionHostLinux = BastionHostLinux;
//# sourceMappingURL=data:application/json;base64,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