from typing import List, Union, Dict, Any
from csle_common.dao.emulation_config.transport_protocol import TransportProtocol
from csle_common.dao.emulation_config.node_vulnerability_config import NodeVulnerabilityConfig
from csle_common.dao.emulation_config.credential import Credential
from csle_common.dao.emulation_config.network_service import NetworkService
from csle_common.dao.emulation_config.vulnerability_type import VulnType


class EmulationVulnerabilityObservationState:
    """
    A DTO representing an observed vulnerability in the emulation
    """

    def __init__(self, name: str, port: Union[int, None],
                 protocol: TransportProtocol, cvss: float, osvdbid: int = None,
                 description: str = None, credentials: List[Credential] = None, service: str = ""):
        """
        Initializes the DTO

        :param name: the name of the vulnerability
        :param port: the port of the vulnerability
        :param protocol: the protocol of the vulnerability
        :param cvss: the CVSS of the vulnerability
        :param osvdbid: the OSVDBID of the vulnerability
        :param description: the description of the vulnerability
        :param credentials: the credentials of the vulnerability
        :param service: the service of the vulnerability
        """
        self.name = name
        self.port = port
        self.protocol = protocol
        self.cvss = cvss
        self.osvdb_id = osvdbid
        self.description = description
        self.credentials = credentials
        self.service = service
        if service is None or service == "":
            for cr in self.credentials:
                if cr.service is not None and cr.service != "":
                    self.service = cr.service

    def to_dict(self) -> Dict[str, Any]:
        """
        :return: a dict representation of the object
        """
        d = {}
        d["name"] = self.name
        d["protocol"] = self.protocol
        d["cvss"] = self.cvss
        d["osvdb_id"] = self.osvdb_id
        d["description"] = self.description
        d["credentials"] = list(map(lambda x: x.to_dict(), self.credentials))
        d["service"] = self.service
        d["port"] = self.port
        return d

    @staticmethod
    def from_dict(d: Dict[str, Any]) -> "EmulationVulnerabilityObservationState":
        """
        Converts a dict representation of the object into an instance

        :param d: the dict to convert
        :return: the created instance
        """
        obj = EmulationVulnerabilityObservationState(
            name=d["name"],
            protocol=d["protocol"],
            cvss=d["cvss"],
            osvdbid=d["osvdb_id"],
            description=d["description"],
            credentials=list(map(lambda x: Credential.from_dict(x), d["credentials"])),
            service=d["service"],
            port=d["port"]
        )
        return obj

    def __str__(self):
        """
        :return: a string representation of the observed vulnerability
        """
        return "name:{}, port:{}, protocol:{}, cvss:{}, osvdb_id:{}, desc:{}, service:{}, credentials:{}".format(
            self.name, self.port, self.protocol, self.cvss, self.osvdb_id, self.description, self.service,
            list(map(lambda x: str(x), self.credentials)))

    def to_vulnerability(self) -> NodeVulnerabilityConfig:
        """
        Converts the object into a vulnerability representation

        :return: A vulnerability representation of the object
        """
        vuln = NodeVulnerabilityConfig(name=self.name, port=self.port, protocol=self.protocol, cvss=self.cvss,
                                       cve="", credentials=self.credentials, service=self.service, ip="",
                                       vuln_type=VulnType.WEAK_PW)
        return vuln

    def to_network_services(self) -> List[NetworkService]:
        """
        Converts the object into a network service representation

        :return: the network service representation
        """
        services = [NetworkService(protocol=self.protocol, port=self.port, name=self.service,
                                   credentials=self.credentials)]
        for cr in self.credentials:
            new_service = True
            for s in services:
                if s.name == cr.service:
                    new_service = False
            if new_service:
                services.append(NetworkService.from_credential(cr))
        return services

    def to_json_str(self) -> str:
        """
        Converts the DTO into a json string

        :return: the json string representation of the DTO
        """
        import json
        json_str = json.dumps(self.to_dict(), indent=4, sort_keys=True)
        return json_str

    def to_json_file(self, json_file_path: str) -> None:
        """
        Saves the DTO to a json file

        :param json_file_path: the json file path to save  the DTO to
        :return: None
        """
        import io
        json_str = self.to_json_str()
        with io.open(json_file_path, 'w', encoding='utf-8') as f:
            f.write(json_str)

    @staticmethod
    def from_json_file(json_file_path: str) -> "EmulationVulnerabilityObservationState":
        """
        Reads a json file and converts it to a DTO

        :param json_file_path: the json file path
        :return: the converted DTO
        """
        import io
        import json
        with io.open(json_file_path, 'r') as f:
            json_str = f.read()
        return EmulationVulnerabilityObservationState.from_dict(json.loads(json_str))

    def num_attributes(self) -> int:
        """
        :return: The number of attribute of the DTO
        """
        if len(self.credentials) > 0:
            return 7 + len(self.credentials) * self.credentials[0].num_attributes()

    @staticmethod
    def schema() -> "EmulationVulnerabilityObservationState":
        """
        :return: get the schema of the DTO
        """
        return EmulationVulnerabilityObservationState(name="", port=-1, protocol=TransportProtocol.TCP, cvss=0.5,
                                                      osvdbid=-1, description="", credentials=[Credential.schema()],
                                                      service="")
