"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Origin = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const http_origin_1 = require("./http-origin");
/**
 * An Origin that is backed by an S3 bucket.
 *
 * If the bucket is configured for website hosting, this origin will be configured to use the bucket as an
 * HTTP server origin and will use the bucket's configured website redirects and error handling. Otherwise,
 * the origin is created as a bucket origin and will use CloudFront's redirect and error handling.
 *
 * @stability stable
 */
class S3Origin {
    /**
     * @stability stable
     */
    constructor(bucket, props = {}) {
        this.origin = bucket.isWebsite ?
            new http_origin_1.HttpOrigin(bucket.bucketWebsiteDomainName, {
                protocolPolicy: cloudfront.OriginProtocolPolicy.HTTP_ONLY,
                ...props,
            }) :
            new S3BucketOrigin(bucket, props);
    }
    /**
     * The method called when a given Origin is added (for the first time) to a Distribution.
     *
     * @stability stable
     */
    bind(scope, options) {
        return this.origin.bind(scope, options);
    }
}
exports.S3Origin = S3Origin;
_a = JSII_RTTI_SYMBOL_1;
S3Origin[_a] = { fqn: "@aws-cdk/aws-cloudfront-origins.S3Origin", version: "1.124.0" };
/**
 * An Origin specific to a S3 bucket (not configured for website hosting).
 *
 * Contains additional logic around bucket permissions and origin access identities.
 */
class S3BucketOrigin extends cloudfront.OriginBase {
    constructor(bucket, { originAccessIdentity, ...props }) {
        super(bucket.bucketRegionalDomainName, props);
        this.bucket = bucket;
        if (originAccessIdentity) {
            this.originAccessIdentity = originAccessIdentity;
        }
    }
    bind(scope, options) {
        if (!this.originAccessIdentity) {
            // Using a bucket from another stack creates a cyclic reference with
            // the bucket taking a dependency on the generated S3CanonicalUserId for the grant principal,
            // and the distribution having a dependency on the bucket's domain name.
            // Fix this by parenting the OAI in the bucket's stack when cross-stack usage is detected.
            const bucketStack = cdk.Stack.of(this.bucket);
            const bucketInDifferentStack = bucketStack !== cdk.Stack.of(scope);
            const oaiScope = bucketInDifferentStack ? bucketStack : scope;
            const oaiId = bucketInDifferentStack ? `${cdk.Names.uniqueId(scope)}S3Origin` : 'S3Origin';
            this.originAccessIdentity = new cloudfront.OriginAccessIdentity(oaiScope, oaiId, {
                comment: `Identity for ${options.originId}`,
            });
            // Used rather than `grantRead` because `grantRead` will grant overly-permissive policies.
            // Only GetObject is needed to retrieve objects for the distribution.
            // This also excludes KMS permissions; currently, OAI only supports SSE-S3 for buckets.
            // Source: https://aws.amazon.com/blogs/networking-and-content-delivery/serving-sse-kms-encrypted-content-from-s3-using-cloudfront/
            this.bucket.addToResourcePolicy(new iam.PolicyStatement({
                resources: [this.bucket.arnForObjects('*')],
                actions: ['s3:GetObject'],
                principals: [this.originAccessIdentity.grantPrincipal],
            }));
        }
        return super.bind(scope, options);
    }
    renderS3OriginConfig() {
        return { originAccessIdentity: `origin-access-identity/cloudfront/${this.originAccessIdentity.originAccessIdentityName}` };
    }
}
//# sourceMappingURL=data:application/json;base64,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