# -*- coding: utf-8 -*-

"""
pywebcopy.elements
~~~~~~~~~~~~~~~~~~

Asset elements of a web page.

"""

import os
import re
import sys
import threading
from threading import Thread

from six.moves.urllib.request import pathname2url

from . import LOGGER, utils
from .configs import config
from .urls import URLTransformer
from .core import get, new_file


_main = threading.current_thread
_enumerate = threading.enumerate

PY3 = (sys.version_info[0] == 3)

CSS_URLS = re.compile(b'''url\\(['"]?(.*?)["']?\\)''')
"""Matches any url() declaration in a css file."""

CSS_IMPORTS = re.compile(b'''@import\\s*['"](.*?)['"]\\s*''')
"""Matches any @import declaration in a css file."""


class FileMixin(URLTransformer, Thread):
    """Wrapper for every Asset type which is used by a web page. e.g. css, js, img, link.

     It inherits the URLTransformer() object to provide file path manipulations.

     :type url: str
     :type base_url: str
     :param str url: a url type string for working
     :param optional str base_url: base url of the website i.e. domain name
     :param optional str base_path: base path where the files will be stored after download
     """
    rel_path = None     # Initialiser for a dummy use case

    def __init__(self, url, base_url=None, base_path=None):
        # this file needs to go in the project_folder to be able to correctly used by
        # the html file
        URLTransformer.__init__(self, url, base_url, base_path)
        Thread.__init__(self)

    def run(self):
        """Saves the file from url and returns the file location. """

        # The file path provided can already be existing
        if os.path.exists(self.file_path) and not config['over_write']:
            LOGGER.info("File already exists at location: {!r}".format(self.file_path))
            return

        new_file(content_url=self.url, location=self.file_path)

    def save_file(self):
        """Saves the file in background thread."""
        self.start()


class AnchorTag(FileMixin):
    """Customises the FileMixin() object for Anchor Tags"""

    def __init__(self, *args, **kwargs):
        super(AnchorTag, self).__init__(*args, **kwargs)

        self.default_filename = 'index.html'
        self.default_fileext = 'html'
        self.check_fileext = True

    def run(self):
        # Override the saving action the file object
        return


class LinkTag(FileMixin):
    """Customises the FileMixin() object for css file type.

    :param bytes contents: contents of the file to be written to disk
    """
    contents = b''  # binary file data
    files = []       # sub-files counter

    def repl(self, match_obj):
        """Processes an url and returns a suited replaceable string.

        :type match_obj: re.MatchObject
        :param match_obj: regex match object to be processed
        :rtype: str
        :return: processed url
        """
        url = match_obj.group(1)

        # url can be base64 encoded content which is not required to be stored
        if url[:4] == b'data':
            return url

        # a path is generated by the cssAsset object and tried to store the file
        # but file could be corrupted to open or write
        # NOTE: self.base_path property needs to be set in order to work properly
        if self.base_path:
            base_path = self.base_path
        else:
            base_path = config['project_folder']

        # decode the url
        str_url = url.decode()

        # If the url is also a css file then it that file also
        # needs to be scanned for urls.
        if str_url.endswith('.css'):
            new_element = LinkTag(str_url, self.url, base_path)

        else:
            new_element = FileMixin(str_url, self.url, base_path)

        # Start the download of the file
        new_element.start()
        self.files.append(new_element)

        # generate a relative path for this downloaded file
        url = pathname2url(utils.relate(new_element.file_path, self.file_path))

        return "url({})".format(url).encode()

    def extract_css_urls(self):
        """Extracts url() links and @imports in css.

        All the linked files will be saved and file path
        would be replaced accordingly
        """

        # the regex matches all those with double mix-match quotes and normal ones
        self.contents = CSS_URLS.sub(self.repl, self.contents)
        self.contents = CSS_IMPORTS.sub(self.repl, self.contents)

        # log amount of links found
        LOGGER.info('%d CSS linked files are found in file %s' % (len(self.files), self.file_path))

        # wait for the still downloading files
        for t in self.files:
            if t is _main():
                continue
            t.join()

        # otherwise return the rewritten bytes self.contents
        return self.contents

    def run(self):
        """Css files are saved differently because they could have url() files in them which also links
        for different files. Thus css file content needs to be searched for urls and then it will proceed
        as usual."""

        # LinkTags can also be specified for elements like favicon etc. Thus a check is necessary
        # to validate it is a proper css file or not.
        if not self.file_name.endswith('.css'):
            super(LinkTag, self).run()

        # Custom request object creation
        req = get(self.url, stream=True)

        # if some error occurs
        if not req or not req.ok:
            LOGGER.error("URL returned an unknown response %s" % self.url)
            return

        # Send the contents for urls
        self.contents = req.content

        # Find and save any urls  or imports in css file
        self.extract_css_urls()

        # Save the content with original encoding
        new_file(location=self.file_path, content=self.contents)


class ScriptTag(FileMixin):
    """Customises the Asset() object for js file type."""


class ImgTag(FileMixin):
    """Customises the Asset() object for images file type."""
