# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pyplexity', 'pyplexity.dataset_processor']

package_data = \
{'': ['*']}

install_requires = \
['Flask>=2.0.2,<3.0.0',
 'cached-path>=1.0.2,<2.0.0',
 'html5lib>=1.1,<2.0',
 'lxml>=4.7.1,<5.0.0',
 'memory-tempfile>=2.2.3,<3.0.0',
 'nltk>=3.6.7,<4.0.0',
 'pandas>=1.1.5,<2.0.0',
 'storable>=1.2.4,<2.0.0',
 'typer[all]>=0.4.0,<0.5.0',
 'warcio>=1.7.4,<2.0.0']

entry_points = \
{'console_scripts': ['pyplexity = pyplexity.__main__:app']}

setup_kwargs = {
    'name': 'pyplexity',
    'version': '0.2.9',
    'description': 'Perplexity filter for documents and bulk HTML and WARC boilerplate removal.',
    'long_description': '# PyPlexity\n\nThis package provides a simple interface to apply perplexity filters to any document. A possible use case for this technology could be the removal of boilerplate (sentences with a high perplexity score). \nFurthermore, it provides a rough HTML tag cleaner and a WARC and HTML bulk processor, with distributed capabilities.\n\n![](imgs/perpl.PNG)\n\n## Models\nMemory intensive but does not scale on CPU. \n| Model | RAM usage | Download size |\n| --- | --- | --- |\n| bigrams-cord19 | 2GB | 230MB |\n| bigrams-bnc | 5GB | 660MB |\n| trigrams-cord19 | 6,6GB | 1GB |\n| trigrams-bnc | 14GB | 2,2GB |\n\nTwo different datasets were selected to build the background language model (LM): CORD-19 dataset [1] and the British National Corpus (BNC) [2]. \n\n[1] Wang, L. L., Lo, K., Chandrasekhar, Y., Reas, R., Yang, J., Eide, D., ... & Kohlmeier, S. (2020). Cord-19: The covid-19 open research dataset. ArXiv.\n\n[2] BNC Consortium. (2007). British national corpus. Oxford Text Archive Core Collection.\n\n## Installation process\n\nThis package can be directly found in [Pypi](https://pypi.org/project/pyplexity/) repository or installed in two ways: \n\n```\npython3 -m pip install pyplexity\n```\nor\n\n```\npip install -r requirements.txt\npython setup.py install\n```\n\n## Examples of usage options\n\n### Compute perplexity from console\nCommand "perplexity". This very first command computes the perplexity score or the probability for a given sentence according to a given distribution, in this case, the background LM. By default, bigrams-bnc. Argument "--model bigrams-bnc" changes the model.  \n*Documentation*:\n```\ncitius@pc:~$ pyplexity perplexity --help\nUsage: pyplexity perplexity [OPTIONS] TEXT\n\nArguments:\n  TEXT  [required]\n\nOptions:\n  --model TEXT  [default: bigrams-bnc]\n  --help        Show this message and exit.\n```\nBy default, models are stored in ~/.cache/cached_path/, as per cached-path package documentation. *Example*:\n```\ncitius@pc:~$ pyplexity perplexity "this is normal text"\ndownloading: 100%|##########| 660M/660M [00:11<00:00, 59.0MiB/s]\nLoading model... Done.\n1844.85540669094\ncitius@pc:~$ pyplexity perplexity "this is normal HTML PAGE BOI%& 678346 NOR  text"\nLoading model... Done.\n44787.99199563819\n```\nAs can be seen, malformed sentences obtain a higher value. \n\n### Bulk perplexity computation and cleaning of a directory\n\nThe previous command was a toy example, as normally in real applications, we will want to score complete datasets to clean them up. This scenario is where the bulk-perplexity functionality that supports WARC or HTML directories comes in.\n\n*Documentation*:\n```\ncitius@pc:~$ pyplexity bulk-perplexity --help\nUsage: pyplexity bulk-perplexity [OPTIONS] INPUT_DIR\n\nArguments:\n  INPUT_DIR  [required]\n\nOptions:\n  --output-dir TEXT                [default: out_dir]\n  --model TEXT                     [default: bigrams-bnc]\n  --perpl-limit FLOAT              [default: 8000.0]\n  --warc-input / --no-warc-input   [default: no-warc-input]\nDistributed computing options:\n  --distributed / --no-distributed [default: no-distributed]\n  --n-workers INTEGER              [default: 1]\n  --node INTEGER                   [default: 1]\n  --port INTEGER                   [default: 8866]\n  --help                           Show this message and exit.\n```\nWe will explain the distributed computing capabilities later. Input directory is allowed to have recursive subdirectories with files. WARC containers and HTML files should have been previously tag-cleaned with the command below. *Example*:\n```\ncitius@pc:~$ pyplexity bulk-perplexity ./out_dir/ --output-dir cleaned_files --model bigrams-cord19\ndownloading: 100%|##########| 233M/233M [00:03<00:00, 63.3MiB/s] \nLoading model... Done.\nComputed 1124 files in 0:00:01.905390.\n```\n\n### Perform HTML tag cleaning of a directory\n\nOur method does not remove HTML tags by default. This fact could impoverish its global performance. That\'s why we recommend removing HTML tags first, and we offer this option inside our package.\n\n*Documentation*:\n```\ncitius@pc:~$ pyplexity tag-remover --help\nUsage: pyplexity tag-remover [OPTIONS] BASE_DIR\n\nArguments:\n  BASE_DIR  [required]\n\nOptions:\n  --output-dir TEXT                [default: out_dir]\n  --warc-input / --no-warc-input   [default: no-warc-input]\nDistributed computing options:\n  --distributed / --no-distributed [default: no-distributed]\n  --n-workers INTEGER              [default: 1]\n  --node INTEGER                   [default: 1]\n  --port INTEGER                   [default: 8866]\n  --help                           Show this message and exit.\n\n```\nWe will explain the distributed computing capabilities later. Input directory is allowed to have recursive subdirectories with files. It can process HTML files or WARC files. In this case, it will recompress the WARC efficiently, after stripping out all the tags. *Example*:\n```\ncitius@pc:~$ pyplexity tag-remover ./html_source --output-dir ./output\nComputed 1124 files in 0:00:00.543175.\n```\n## Parallel mode (cluster)\nPrevious documentation shows that our commands have integrated distributed computing capabilities. When using the cluster mode, all the nodes must be interconnected in a local network, having the access to the same files mounted via SSHFS or other filesystem. A master node will recursively load the folder of files to be computed, with the command:\n```\npyplexity fileserver /mnt/input_dir --port 8866\n```\nNow, clients from the nodes will connect to the master node asking for file names to be processed. This mechanism allows for load distribution, as clients are able to ask for files in queue for processing from the master. For example, from a node:\n```\npyplexity bulk-perplexity /mnt/input_dir --output-dir /mnt/output_dir --warc-input --distributed --n-workers 10 --node 2 --url master.local --port 8866\n```\nThat command should be executed in every machine of the cluster. The node argument identifies the machine for logging purposes, and has no functional relevance. The n-workers argument controls the number of thread workers per machine that will be querying the master node for files concurrently. When the master server has served all the files, worker procceses will shutdown accordingly. In our experiments, we use this feature to run the HTML tag removal and perplexity computation in 20 threads * 15 machines.\n\n## Interfacing from Python\n\nWe also offer the possibility of utilising *pyplexity* from Python code. As an example, we provide an API that serves a web app to make some small tests on how to directly clean texts or raw files.\n\nExample: computing the perplexity score for a sentence:\n```\nfrom pyplexity import PerplexityComputer\n\nmodel = PerplexityModel.from_str("bigrams-cord19")\nperpl = model.compute_sentence("this is normal text")\n```\nExample 2: Cleaning sentences from a text:\n```\nfrom pyplexity import PerplexityModel, PerplexityProcessor\n\nmodel = PerplexityModel.from_str("bigrams-cord19")\ntext_processor = PerplexityProcessor(perpl_model=model, perpl_limit=8000.0)\nclean_text = text_processor.process("This is a normal sentence. Meanwhile, hjldfuia HTML BODY this one will be deleted LINK URL COUISUDOANLHJWQKEJK")\n```\nExample 3: Removing HTML tags from a website:\n```\nimport requests\nfrom pyplexity.tag_remover import HTMLTagRemover\n\nhtml = requests.get("https://example.com").text\ntext = HTMLTagRemover().process(html)\n```\n## Web Demo\nWe also provide a [web demo](https://tec.citius.usc.es/pyplexity/) as a simple example of the power of our tool. Screenshot:\n<p align="center">\n  <img src="https://user-images.githubusercontent.com/6536835/158210142-c0b04512-f482-49fc-9261-adb15628984f.png" alt="screenshot" width="600"/>\n</p>\n\n\n## Building the package\n\nIf you are interested, you can also build the same package version we have currently deployed in the Pypi repository.\n\n```\ngit clone https://github.com/citiususc/pyplexity && cd pyplexity\ncurl -sSL https://raw.githubusercontent.com/python-poetry/poetry/master/get-poetry.py | python3 -\nsource $HOME/.poetry/env\npoetry build\npip3 install dist/pyplexity-X.X.X-py3-none-any.whl\n```\n\n## General Advice\n\nAs you may have noticed, this is an unsupervised method that requires setting the optimal model and threshold. From our experimentation (**PONER REF**), we have concluded that the bigrams-bnc model and removing sentences with a value higher than 8k is a robust strategy both for an IR search task and a text classification task.\n\n## Cite\n\nBibtex\n\n',
    'author': 'Manuel de Prada Corral',
    'author_email': 'manuel.deprada.corral@usc.es',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/citiususc/pyplexity',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.6.1,<4.0.0',
}


setup(**setup_kwargs)
