import copy
from src.Config import Config
from urllib.parse import urljoin
import requests



class ApiRequest:
    """Request Object for RestApi requests

    ApiRequest class represent a request object
    ApiRequest instances are generated by RequestFactory classes
    There is no reason to create ApiRequestInstance out of a RequestFactory instance

    """


    _headers: dict = None
    _auth: dict = None
    _base_url: str = None
    _end_url: str = None
    _params: dict = None
    _api_name: str = None


    def set_config(self, config: Config):
        """Set request configuration with a Config instance to prepare the request

        Arguments:

            config(Required):

                Config instance
                --> see Config documentation
        """

        self._headers = config.headers
        self._auth = config.auth
        self._base_url = config.base_url


    def set_request(self, end_url: str = None, params: dict = None):
        """Set end_url and params request arguments to prepare the request

        Arguments:

            end_url(Optional):
                String.
                Complete base_url request argument (from set_config)
                ex

                    base_url = "https://api/"
                    end_url = "end_of_url"

                    --> url = "https://api/end_of_url"


            params(Optional):
                Dict.
                Parameters of the request url
                ex:
                    params = {"plan":"Tier_one"}

                    --> url = "https://api/end_of_url?plan=Tier_one"

                Multiple parameters can be provided by key/value pairs in this dict
        """

        self._end_url = end_url
        self._params = params


    def _get_url(self):
        """Join base_url and end_url and return the complete request url"""
        if self._base_url is not None:
            if self._end_url is not None:
                if self._base_url[-1] != "/":
                    self._base_url = self._base_url + "/"
                return urljoin(self._base_url, self._end_url)
            else:
                return self._base_url
        else:
            raise ValueError("Cannot operate request while 'base_url' argument isn't set")



    def _get_header(self):
        """return header(auth + headers arguments)"""
        ret_head = dict()
        if self._headers is not None:
            ret_head.update(self._headers)
        if self._auth is not None:
            ret_head.update(self._auth)
        return ret_head


    def get_response(self):
        """Execute request and return a Request instance(requests library)"""
        return requests.get(self._get_url(), params=self._params, headers=self._get_header())


    def __repr__(self):
        return "ApiRequest(%r, %r, %r, %r %r)" % (self._base_url, self._end_url, self._auth, self._headers, self._params)


    def __str__(self):
        return "ApiRequest: %r / params: %r" % (self._get_url(), self._params)


    def get_request_params(self):
        """return request parameters and end_url reprensented what was passed to the set_request() method"""
        return self._end_url, self._params


    def __copy__(self):
        copy_obj = ApiRequest()
        copy_obj._base_url = self._base_url
        copy_obj._end_url = self._end_url
        copy_obj._headers = copy.copy(self._headers)
        copy_obj._auth = copy.copy(self._auth)
        copy_obj._params = copy.copy(self._params)
        return copy_obj

    def copy(self):
        return self.__copy__()