"use strict";
// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
Object.defineProperty(exports, "__esModule", { value: true });
const coreutils_1 = require("@jupyterlab/coreutils");
const serverconnection_1 = require("../serverconnection");
/**
 * The url for the lab settings service.
 */
const SERVICE_SETTINGS_URL = 'api/settings';
/**
 * The settings API service manager.
 */
class SettingManager extends coreutils_1.DataConnector {
    /**
     * Create a new setting manager.
     */
    constructor(options = {}) {
        super();
        this.serverSettings =
            options.serverSettings || serverconnection_1.ServerConnection.makeSettings();
    }
    /**
     * Fetch a plugin's settings.
     *
     * @param id - The plugin's ID.
     *
     * @returns A promise that resolves if successful.
     */
    async fetch(id) {
        const { serverSettings } = this;
        const { baseUrl, appUrl } = serverSettings;
        const { makeRequest, ResponseError } = serverconnection_1.ServerConnection;
        const base = baseUrl + appUrl;
        const url = Private.url(base, id);
        const response = await makeRequest(url, {}, serverSettings);
        if (!id) {
            throw new Error('Plugin `id` parameter is required for settings fetch.');
        }
        if (response.status !== 200) {
            throw new ResponseError(response);
        }
        // Assert what type the server response is returning.
        return response.json();
    }
    /**
     * Fetch the list of all plugin setting bundles.
     *
     * @returns A promise that resolves if successful.
     */
    async list() {
        const { serverSettings } = this;
        const { baseUrl, appUrl } = serverSettings;
        const { makeRequest, ResponseError } = serverconnection_1.ServerConnection;
        const base = baseUrl + appUrl;
        const url = Private.url(base, '');
        const response = await makeRequest(url, {}, serverSettings);
        if (response.status !== 200) {
            throw new ResponseError(response);
        }
        const json = await response.json();
        const values = ((json || {})['settings'] || []).map((plugin) => {
            plugin.data = { composite: {}, user: {} };
            return plugin;
        });
        const ids = values.map(plugin => plugin.id);
        return { ids, values };
    }
    /**
     * Save a plugin's settings.
     *
     * @param id - The plugin's ID.
     *
     * @param raw - The user setting values as a raw string of JSON with comments.
     *
     * @returns A promise that resolves if successful.
     */
    async save(id, raw) {
        const { serverSettings } = this;
        const { baseUrl, appUrl } = serverSettings;
        const { makeRequest, ResponseError } = serverconnection_1.ServerConnection;
        const base = baseUrl + appUrl;
        const url = Private.url(base, id);
        const init = { body: raw, method: 'PUT' };
        const response = await makeRequest(url, init, serverSettings);
        if (response.status !== 204) {
            throw new ResponseError(response);
        }
    }
}
exports.SettingManager = SettingManager;
/**
 * A namespace for private data.
 */
var Private;
(function (Private) {
    /**
     * Get the url for a plugin's settings.
     */
    function url(base, id) {
        return coreutils_1.URLExt.join(base, SERVICE_SETTINGS_URL, id);
    }
    Private.url = url;
})(Private || (Private = {}));
//# sourceMappingURL=index.js.map