import { ISignal } from '@phosphor/signaling';
import { Kernel, KernelMessage } from '../kernel';
import { ServerConnection } from '..';
import { Session } from './session';
/**
 * Session object for accessing the session REST api. The session
 * should be used to start kernels and then shut them down -- for
 * all other operations, the kernel object should be used.
 */
export declare class DefaultSession implements Session.ISession {
    /**
     * Construct a new session.
     */
    constructor(options: Session.IOptions, id: string, model: Kernel.IModel);
    /**
     * A signal emitted when the session is shut down.
     */
    readonly terminated: ISignal<this, void>;
    /**
     * A signal emitted when the kernel changes.
     */
    readonly kernelChanged: ISignal<this, Session.IKernelChangedArgs>;
    /**
     * A signal emitted when the kernel status changes.
     */
    readonly statusChanged: ISignal<this, Kernel.Status>;
    /**
     * A signal emitted for a kernel messages.
     */
    readonly iopubMessage: ISignal<this, KernelMessage.IIOPubMessage>;
    /**
     * A signal emitted for an unhandled kernel message.
     */
    readonly unhandledMessage: ISignal<this, KernelMessage.IMessage>;
    /**
     * A signal emitted for any kernel message.
     *
     * Note: The behavior is undefined if the message is modified
     * during message handling. As such, it should be treated as read-only.
     */
    readonly anyMessage: ISignal<this, Kernel.IAnyMessageArgs>;
    /**
     * A signal emitted when a session property changes.
     */
    readonly propertyChanged: ISignal<this, 'path' | 'name' | 'type'>;
    /**
     * Get the session id.
     */
    readonly id: string;
    /**
     * Get the session kernel object.
     *
     * #### Notes
     * This is a read-only property, and can be altered by [changeKernel].
     */
    readonly kernel: Kernel.IKernelConnection;
    /**
     * Get the session path.
     */
    readonly path: string;
    /**
     * Get the session type.
     */
    readonly type: string;
    /**
     * Get the session name.
     */
    readonly name: string;
    /**
     * Get the model associated with the session.
     */
    readonly model: Session.IModel;
    /**
     * The current status of the session.
     *
     * #### Notes
     * This is a delegate to the kernel status.
     */
    readonly status: Kernel.Status;
    /**
     * The server settings of the session.
     */
    readonly serverSettings: ServerConnection.ISettings;
    /**
     * Test whether the session has been disposed.
     */
    readonly isDisposed: boolean;
    /**
     * Clone the current session with a new clientId.
     */
    clone(): Session.ISession;
    /**
     * Update the session based on a session model from the server.
     */
    update(model: Session.IModel): void;
    /**
     * Dispose of the resources held by the session.
     */
    dispose(): void;
    /**
     * Change the session path.
     *
     * @param path - The new session path.
     *
     * @returns A promise that resolves when the session has renamed.
     *
     * #### Notes
     * This uses the Jupyter REST API, and the response is validated.
     * The promise is fulfilled on a valid response and rejected otherwise.
     */
    setPath(path: string): Promise<void>;
    /**
     * Change the session name.
     */
    setName(name: string): Promise<void>;
    /**
     * Change the session type.
     */
    setType(type: string): Promise<void>;
    /**
     * Change the kernel.
     *
     * @params options - The name or id of the new kernel.
     *
     * #### Notes
     * This shuts down the existing kernel and creates a new kernel,
     * keeping the existing session ID and session path.
     */
    changeKernel(options: Partial<Kernel.IModel>): Promise<Kernel.IKernelConnection>;
    /**
     * Kill the kernel and shutdown the session.
     *
     * @returns - The promise fulfilled on a valid response from the server.
     *
     * #### Notes
     * Uses the [Jupyter Notebook API](http://petstore.swagger.io/?url=https://raw.githubusercontent.com/jupyter/notebook/master/notebook/services/api/api.yaml#!/sessions), and validates the response.
     * Disposes of the session and emits a [sessionDied] signal on success.
     */
    shutdown(): Promise<void>;
    /**
     * Create a new kernel connection and hook up to its events.
     *
     * #### Notes
     * This method is not meant to be subclassed.
     */
    protected setupKernel(model: Kernel.IModel): void;
    /**
     * Handle to changes in the Kernel status.
     */
    protected onKernelStatus(sender: Kernel.IKernel, state: Kernel.Status): void;
    /**
     * Handle iopub kernel messages.
     */
    protected onIOPubMessage(sender: Kernel.IKernel, msg: KernelMessage.IIOPubMessage): void;
    /**
     * Handle unhandled kernel messages.
     */
    protected onUnhandledMessage(sender: Kernel.IKernel, msg: KernelMessage.IMessage): void;
    /**
     * Handle any kernel messages.
     */
    protected onAnyMessage(sender: Kernel.IKernel, args: Kernel.IAnyMessageArgs): void;
    /**
     * Send a PATCH to the server, updating the session path or the kernel.
     */
    private _patch;
    /**
     * Handle a change to the model.
     */
    private _handleModelChange;
    private _id;
    private _path;
    private _name;
    private _type;
    private _kernel;
    private _isDisposed;
    private _updating;
    private _kernelChanged;
    private _statusChanged;
    private _iopubMessage;
    private _unhandledMessage;
    private _anyMessage;
    private _propertyChanged;
    private _terminated;
}
/**
 * The namespace for `DefaultSession` statics.
 */
export declare namespace DefaultSession {
    /**
     * List the running sessions.
     */
    function listRunning(settings?: ServerConnection.ISettings): Promise<Session.IModel[]>;
    /**
     * Start a new session.
     */
    function startNew(options: Session.IOptions): Promise<Session.ISession>;
    /**
     * Find a session by id.
     */
    function findById(id: string, settings?: ServerConnection.ISettings): Promise<Session.IModel>;
    /**
     * Find a session by path.
     */
    function findByPath(path: string, settings?: ServerConnection.ISettings): Promise<Session.IModel>;
    /**
     * Connect to a running session.
     */
    function connectTo(model: Session.IModel, settings?: ServerConnection.ISettings): Session.ISession;
    /**
     * Shut down a session by id.
     */
    function shutdown(id: string, settings?: ServerConnection.ISettings): Promise<void>;
    /**
     * Shut down all sessions.
     *
     * @param settings - The server settings to use.
     *
     * @returns A promise that resolves when all the sessions are shut down.
     */
    function shutdownAll(settings?: ServerConnection.ISettings): Promise<void>;
}
