"use strict";
// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
Object.defineProperty(exports, "__esModule", { value: true });
// Notebook format interfaces
// https://nbformat.readthedocs.io/en/latest/format_description.html
// https://github.com/jupyter/nbformat/blob/master/nbformat/v4/nbformat.v4.schema.json
const coreutils_1 = require("@phosphor/coreutils");
/**
 * A namespace for nbformat interfaces.
 */
var nbformat;
(function (nbformat) {
    /**
     * The major version of the notebook format.
     */
    nbformat.MAJOR_VERSION = 4;
    /**
     * The minor version of the notebook format.
     */
    nbformat.MINOR_VERSION = 4;
    /**
     * Validate a mime type/value pair.
     *
     * @param type - The mimetype name.
     *
     * @param value - The value associated with the type.
     *
     * @returns Whether the type/value pair are valid.
     */
    function validateMimeValue(type, value) {
        // Check if "application/json" or "application/foo+json"
        const jsonTest = /^application\/(.*?)+\+json$/;
        const isJSONType = type === 'application/json' || jsonTest.test(type);
        let isString = (x) => {
            return Object.prototype.toString.call(x) === '[object String]';
        };
        // If it is an array, make sure if is not a JSON type and it is an
        // array of strings.
        if (Array.isArray(value)) {
            if (isJSONType) {
                return false;
            }
            let valid = true;
            value.forEach(v => {
                if (!isString(v)) {
                    valid = false;
                }
            });
            return valid;
        }
        // If it is a string, make sure we are not a JSON type.
        if (isString(value)) {
            return !isJSONType;
        }
        // It is not a string, make sure it is a JSON type.
        if (!isJSONType) {
            return false;
        }
        // It is a JSON type, make sure it is a valid JSON object.
        return coreutils_1.JSONExt.isObject(value);
    }
    nbformat.validateMimeValue = validateMimeValue;
    /**
     * Test whether a cell is a raw cell.
     */
    function isRaw(cell) {
        return cell.cell_type === 'raw';
    }
    nbformat.isRaw = isRaw;
    /**
     * Test whether a cell is a markdown cell.
     */
    function isMarkdown(cell) {
        return cell.cell_type === 'markdown';
    }
    nbformat.isMarkdown = isMarkdown;
    /**
     * Test whether a cell is a code cell.
     */
    function isCode(cell) {
        return cell.cell_type === 'code';
    }
    nbformat.isCode = isCode;
    /**
     * Test whether an output is an execute result.
     */
    function isExecuteResult(output) {
        return output.output_type === 'execute_result';
    }
    nbformat.isExecuteResult = isExecuteResult;
    /**
     * Test whether an output is from display data.
     */
    function isDisplayData(output) {
        return output.output_type === 'display_data';
    }
    nbformat.isDisplayData = isDisplayData;
    /**
     * Test whether an output is from updated display data.
     */
    function isDisplayUpdate(output) {
        return output.output_type === 'update_display_data';
    }
    nbformat.isDisplayUpdate = isDisplayUpdate;
    /**
     * Test whether an output is from a stream.
     */
    function isStream(output) {
        return output.output_type === 'stream';
    }
    nbformat.isStream = isStream;
    /**
     * Test whether an output is an error.
     */
    function isError(output) {
        return output.output_type === 'error';
    }
    nbformat.isError = isError;
})(nbformat = exports.nbformat || (exports.nbformat = {}));
//# sourceMappingURL=nbformat.js.map