from __future__ import annotations

import os
import sys
import importlib
import importlib.util
import inspect
import json
import logging
import zipfile
import copy
from pathlib import Path
from typing import Any, Dict, List, Optional, Set, Union, get_type_hints
import numpy as np
import pandas as pd
from anytree import AnyNode, PreOrderIter
import ast
import weakref
from jsonschema import validate
import hashlib
import types

from flexiconc.resources import ResourceRegistry
from flexiconc.utils.conditions import *

# Configure logging
logging.basicConfig(level=logging.WARNING)
logger = logging.getLogger(__name__)

class ConcordanceSubset:
    """
    A class that holds the data corresponding to a subset of concordance lines.
    """

    def __init__(
        self,
        metadata: pd.DataFrame,
        tokens: pd.DataFrame,
        matches: pd.DataFrame,
        root: AnalysisTreeNode = None,
        original_size: Optional[int] = None,
        active_node: AnalysisTreeNode = None
    ):
        self.metadata = metadata
        self.tokens = tokens
        self.matches = matches
        self.root = root
        self.resources = self.root.concordance().resources
        self.original_size = original_size
        self.active_node = active_node


class AnalysisTreeNode(AnyNode):
    """
    Represents a node within the analysis tree, derived from AnyNode.

    Attributes:
        id (int, optional): The unique identifier of the node, if not supplied one chosen by t_last_idhe root node.
        node_type (str): Either "subset" or "arrangement". The root node is treated as "subset".
        bookmarked (bool): Indicates whether the node is bookmarked. Default is False.
        label (str): A label for the node. Default is an empty string.
        selected_lines (list): For subset nodes, contains the line IDs of the selected concordance lines.
        sort_keys (dict): For arrangement nodes, contains surrogate sort keys generated by the operation.
        rank_keys (dict): For arrangement nodes, contains natural ranking keys from the operation's algorithms.
        differentiation_info (dict): For arrangement nodes, indicates how many pairs of adjacent lines are differentiated by each ordering algorithm.
        partitions (dict): For arrangement nodes, contains partition information, where keys are partition IDs, and values include:
            - label (str): Label of the partition.
            - line_ids (list): Sorted line IDs of the lines in the partition.
            - best_example (int, optional): The line ID representing the most prototypical line of the partition.
    """

    def __init__(self, node_type, parent, id=None, concordance=None, **kwargs):
        super().__init__(parent=parent)
        if self.parent == None:
            self._last_id = 0
        self.id = id if id is not None else self._next_id()
        self.node_type = node_type
        self.concordance = weakref.ref(concordance) if concordance else parent.concordance
        self.bookmarked = kwargs.get("bookmarked", False)
        self.label = kwargs.get("label", "")

        # Extract additional attributes
        for key, value in kwargs.items():
            setattr(self, key, value)

    def _next_id(self):
        """
        Counter to return the next unique id for the tree
        """
        if self.parent is not None:
            # Subnodes should ask the root for the next_id
            return self.root._next_id()
        if not hasattr(self, "_last_id"):
            self._last_id = 0
        else:
            self._last_id += 1
        return self._last_id

    def _execute_algorithm(self, algorithm_name, args):
        """
        Executes the specified algorithm on the subset of this node.

        Args:
            algorithm_name (str): The name of the algorithm to execute.
            args (dict): Arguments to pass to the algorithm.

        Returns:
            dict: The result of the algorithm execution as a dictionary.

        Raises:
            RuntimeError: If the Concordance reference is invalid or the algorithm fails.
            ValueError: If the algorithm is not found in available algorithms.
        """
        # Dereference the weak reference to get the Concordance object
        concordance = self.concordance()
        if not concordance:
            raise RuntimeError("Concordance reference is no longer valid.")

        # Find the algorithm in available_algorithms
        algorithm_metadata = copy.deepcopy(concordance.available_algorithms.get(algorithm_name))
        if not algorithm_metadata:
            raise ValueError(f"Algorithm '{algorithm_name}' not found in available algorithms.")
        algorithm_metadata["args_schema"] = self.schema_for(algorithm_name)

        # Retrieve the subset at this node
        subset = concordance.subset_at_node(self)

        # Dynamically load and execute the algorithm
        module_name = algorithm_metadata["module"]
        function_name = algorithm_metadata["function"]

        try:
            module = __import__(module_name, fromlist=[function_name])
            algorithm_function = getattr(module, function_name)

            # Add default values from metadata if missing in args
            schema_properties = algorithm_metadata.get("args_schema", {}).get("properties", {})
            for key, details in schema_properties.items():
                if key not in args and "default" in details:
                    args[key] = details["default"]

            # Remove keys with value None from args
            args = {k: v for k, v in args.items() if v is not None}

            # Validate args against the enriched schema in context.
            enriched_schema = self.schema_for(algorithm_name)
            validate(instance=args, schema=enriched_schema)

            # Execute the algorithm with validated args
            return algorithm_function(subset, **args)

        except Exception as e:
            raise RuntimeError(f"Failed to execute algorithm '{algorithm_name}': {e}")

    @staticmethod
    def _compute_algorithms_hash(target_algos: dict) -> str:
        """
        Computes a hash of the target algorithms dictionary by serializing it to JSON with sorted keys
        and then taking its MD5 hash.
        """
        return hashlib.md5(json.dumps(str(target_algos), sort_keys=True).encode('utf-8')).hexdigest()

    def add_subset_node(self, algo_tuple, keep_arrangement=True):
        """
        Adds a subset node to the analysis tree by applying the specified algorithm.
        The algorithm is provided as a tuple (algorithm_name, args). If a sibling subset node (sharing the same parent)
        already exists with the same algorithm name and arguments, that node is returned rather than creating a new one.
        Optionally, if keep_arrangement is True, the parent's ordering_result and grouping_result are copied and restricted
        to the new subset's selected lines. For grouping_result, if partitions are used, each partition's line_ids (and
        prototypes) are intersected with the new subset; if clusters are used, the clustering tree is recursively restricted,
        and empty clusters are discarded.

        Args:
            algo_tuple (tuple): A tuple of (algorithm_name, args), where algorithm_name is a string and args is a dict.
            keep_arrangement (bool): If True, copy and restrict parent's ordering and grouping information.

        Returns:
            AnalysisTreeNode: The subset node corresponding to the given algorithm.
        """
        algorithm_name, args = algo_tuple
        algo_meta = self.concordance().available_algorithms.get(algorithm_name, {})
        target_algo = {
            "subset": {
                "algorithm_name": algorithm_name,
                "args": args,
                "algorithm_type": algo_meta.get("algorithm_type", "Unknown")
            }
        }
        target_hash = self._compute_algorithms_hash(target_algo)

        # Determine parent (for subset nodes, parent is self).
        parent = self

        # Check for existing sibling subset node with identical algorithm configuration.
        if parent and hasattr(parent, "children"):
            for sibling in parent.children:
                if sibling.node_type == "subset" and getattr(sibling, "algorithms_hash", None) == target_hash:
                    return sibling

        # Execute the algorithm and get the result.
        algorithm_result = self._execute_algorithm(algorithm_name, args)
        new_node_attrs = {key: value for key, value in algorithm_result.items()}

        new_node = AnalysisTreeNode(
            node_type="subset",
            parent=parent,
            algorithms=target_algo,
            **new_node_attrs
        )
        new_node.algorithms_hash = target_hash
        new_node.line_count = len(new_node.selected_lines)

        if keep_arrangement:
            # Determine the new node's selected lines.
            if hasattr(new_node, "selected_lines") and new_node.selected_lines is not None:
                selected_lines = set(new_node.selected_lines)
            elif hasattr(parent, "selected_lines") and parent.selected_lines is not None:
                selected_lines = set(parent.selected_lines)
            else:
                selected_lines = set(range(len(self.concordance().metadata)))

            # Restrict ordering_result if present.
            if hasattr(parent, "ordering_result"):
                parent_ordering = parent.ordering_result
                # Build restricted sort keys for selected lines.
                restricted_keys = [(line_id, rank) for line_id, rank in parent_ordering.get("sort_keys", {}).items() if
                                   line_id in selected_lines]
                restricted_keys = sorted(restricted_keys, key=lambda x: (x[1], x[0]))
                new_sort_keys = {}
                current_rank = 0
                tie_count = 0
                for i, (line_id, rank) in enumerate(restricted_keys):
                    if i > 0 and rank == restricted_keys[i - 1][1]:
                        new_sort_keys[line_id] = current_rank
                        tie_count += 1
                    else:
                        current_rank += tie_count + 1
                        new_sort_keys[line_id] = current_rank
                        tie_count = 0
                new_ordering_result = parent_ordering.copy()
                new_ordering_result["sort_keys"] = new_sort_keys
                if "rank_keys" in parent_ordering:
                    new_rank_keys = {}
                    for algo, ranks in parent_ordering["rank_keys"].items():
                        filtered = {line_id: r for line_id, r in ranks.items() if line_id in selected_lines}
                        if filtered:
                            new_rank_keys[algo] = filtered
                    new_ordering_result["rank_keys"] = new_rank_keys
                new_node.ordering_result = new_ordering_result

            # Restrict grouping_result if present.
            if hasattr(parent, "grouping_result"):
                parent_grouping = parent.grouping_result
                # Handle flat partitions.
                if isinstance(parent_grouping, dict) and "partitions" in parent_grouping:
                    new_partitions = []
                    for partition in parent_grouping["partitions"]:
                        new_lines = [line_id for line_id in partition.get("line_ids", []) if line_id in selected_lines]
                        if not new_lines:
                            continue
                        new_partition = partition.copy()
                        new_partition["line_ids"] = new_lines
                        if "prototypes" in new_partition:
                            new_partition["prototypes"] = [p for p in new_partition["prototypes"] if p in new_lines]
                        new_partitions.append(new_partition)
                    new_node.grouping_result = {"partitions": new_partitions}
                # Handle clusters.
                elif isinstance(parent_grouping, dict) and "clusters" in parent_grouping:
                    def restrict_cluster(cluster, selected):
                        # Restrict the current cluster's line_ids.
                        new_line_ids = [lid for lid in cluster.get("line_ids", []) if lid in selected]
                        new_cluster = cluster.copy()
                        new_cluster["line_ids"] = new_line_ids
                        # Restrict prototypes if available.
                        if "prototypes" in new_cluster:
                            new_cluster["prototypes"] = [p for p in new_cluster["prototypes"] if p in new_line_ids]
                        # Process children recursively.
                        if "children" in cluster:
                            new_children = []
                            for child in cluster["children"]:
                                restricted_child = restrict_cluster(child, selected)
                                if restricted_child is not None:
                                    new_children.append(restricted_child)
                            if new_children:
                                new_cluster["children"] = new_children
                            else:
                                new_cluster.pop("children", None)
                        # Discard the cluster if empty.
                        if not new_cluster.get("line_ids") and "children" not in new_cluster:
                            return None
                        return new_cluster

                    new_clusters = []
                    for cluster in parent_grouping["clusters"]:
                        restricted = restrict_cluster(cluster, selected_lines)
                        if restricted is not None:
                            new_clusters.append(restricted)
                    new_node.grouping_result = {"clusters": new_clusters}

        return new_node

    def add_arrangement_node(self, ordering=[], grouping=None):
        """
        Adds an arrangement node to the analysis tree by applying ordering and grouping algorithms.
        In addition to storing ordering_result and grouping_result, this version combines any token_spans
        returned by ordering and grouping algorithms and computes a hash of the provided algorithm parameters.
        If a sibling arrangement node (sharing the same parent) already exists with exactly the same algorithm
        configuration (as determined by the hash), that node is returned rather than creating a new one.

        Args:
            ordering (list of dict): A list of tuples of ordering algorithms.
            grouping (tuple): A single grouping algorithm as a tuple.
            Each algorithm tuple is of a form (algorithm_name, args), where algorithm_name is a string and args is a dict.

        Returns:
            AnalysisTreeNode: The arrangement node corresponding to the given algorithms.
        """
        # Define target algorithm configuration.
        conc_obj = self.concordance()
        target_algos = {
            "ordering": [
                {
                    "algorithm_name": name,
                    "args": a,
                    "algorithm_type": conc_obj.available_algorithms.get(name, {}).get("algorithm_type", "Unknown")
                }
                for name, a in ordering
            ],
            "grouping": (
                {
                    "algorithm_name": grouping[0],
                    "args": grouping[1],
                    "algorithm_type": conc_obj.available_algorithms.get(grouping[0], {}).get("algorithm_type",
                                                                                             "Unknown")
                } if grouping is not None else None
            )
        }
        target_hash = self._compute_algorithms_hash(target_algos)

        # Determine the parent: if self is a subset, then parent is self; otherwise, parent's parent.
        parent = self if self.node_type == "subset" else self.parent

        # Check if a sibling arrangement node with the same algorithm hash already exists.
        if parent and hasattr(parent, "children"):
            for sibling in parent.children:
                if (sibling.node_type == "arrangement" and
                        getattr(sibling, "algorithms_hash", None) == target_hash):
                    return sibling

        # Proceed to compute ordering_result and grouping_result.
        new_node_attrs = {}
        ordering_result = None
        grouping_result = None

        if ordering:
            ordering_result = self._combine_ordering_algorithms([{"algorithm_name": name, "args": a} for name, a in ordering])
            new_node_attrs["ordering_result"] = ordering_result

        if grouping:
            grouping_name, grouping_args = grouping[0], grouping[1]
            grouping_result = self._execute_algorithm(grouping_name, grouping_args)
            if isinstance(grouping_result, dict) and "token_spans" in grouping_result:
                ts_grouping = grouping_result["token_spans"].copy()
                ts_grouping["algo"] = "grouping"
                grouping_result["token_spans"] = ts_grouping

            # TODO: update to include clusters when implemented
            if "partitions" in grouping_result:
                sort_keys = new_node_attrs.get("ordering_result", {}).get("sort_keys", {})
                for partition in grouping_result["partitions"]:
                    partition["line_ids"] = sorted(
                        partition["line_ids"],
                        key=lambda lid: sort_keys.get(lid, float('inf'))
                    )
            new_node_attrs["grouping_result"] = grouping_result

        combined_token_spans = pd.DataFrame()
        if ordering_result and "token_spans" in ordering_result:
            ts_ordering = ordering_result["token_spans"].copy()
            if "algo" not in ts_ordering.columns:
                ts_ordering["algo"] = "ordering_algo_unknown"
            combined_token_spans = ts_ordering
        if grouping_result and "token_spans" in grouping_result:
            ts_grouping = grouping_result["token_spans"].copy()
            ts_grouping["algo"] = "grouping"
            if combined_token_spans.empty:
                combined_token_spans = ts_grouping
            else:
                combined_token_spans = pd.concat([combined_token_spans, ts_grouping], ignore_index=True)

        if not combined_token_spans.empty:
            def algo_sort_key(algo_val):
                if algo_val == "grouping":
                    return 0
                elif algo_val.startswith("ordering_algo_"):
                    try:
                        num = int(algo_val.split("_")[-1])
                        return num + 1
                    except Exception:
                        return 1000
                else:
                    return 1000

            combined_token_spans["sort_key"] = combined_token_spans["algo"].apply(algo_sort_key)
            combined_token_spans.sort_values(by="sort_key", inplace=True)
            combined_token_spans.drop(columns=["sort_key"], inplace=True)
            unique_pairs = combined_token_spans[["algo", "category"]].drop_duplicates()

            def algo_pair_sort_key(row):
                algo_val = row["algo"]
                orig_cat = row["category"]
                if algo_val == "grouping":
                    return (0, orig_cat)
                elif algo_val.startswith("ordering_algo_"):
                    try:
                        num = int(algo_val.split("_")[-1])
                    except Exception:
                        num = 1000
                    return (num + 1, orig_cat)
                else:
                    return (1000, orig_cat)

            unique_pairs["sort_order"] = unique_pairs.apply(algo_pair_sort_key, axis=1)
            unique_pairs.sort_values(by="sort_order", inplace=True)
            mapping = {}
            letters = [chr(65 + i) for i in range(len(unique_pairs))]
            for new_label, (_, row) in zip(letters, unique_pairs.iterrows()):
                mapping[(row["algo"], row["category"])] = new_label
            combined_token_spans["category"] = combined_token_spans.apply(
                lambda row: mapping.get((row["algo"], row["category"]), row["category"]), axis=1
            )
            combined_token_spans.drop(columns=["algo"], inplace=True)
            new_node_attrs["token_spans"] = combined_token_spans

        # Create new node and assign the computed algorithms hash.
        new_node = AnalysisTreeNode(
            node_type="arrangement",
            parent=parent,
            algorithms=target_algos,
            line_count=parent.line_count,
            **new_node_attrs
        )
        new_node.algorithms_hash = target_hash

        return new_node


    def _combine_ordering_algorithms(self, ordering):
        """
        Combines and executes multiple ordering algorithms into a single sorted order.
        Also provides information on how many adjacent line pairs are differentiated by each algorithm,
        and handles tie ranks correctly. If "rank_keys" are present, includes them.
        Additionally, collects token_spans from each ordering algorithm and fills an "algo" column
        with values like "ordering_algo_0", "ordering_algo_1", etc.

        Args:
            ordering (list of dict): A list of ordering algorithms with their arguments.

        Returns:
            dict: The combined result containing:
                - "sort_keys": A mapping from line IDs to their final sorted ranks.
                - "differentiation_info": How many pairs were differentiated by each algorithm.
                - "rank_keys": Original rank keys, if any.
                - "token_spans": A unified DataFrame of token spans from all ordering algorithms,
                                 each with an "algo" column indicating the source algorithm.
        """

        sort_df = pd.DataFrame()
        rank_keys_dict = {}
        token_spans_list = []

        # Process each ordering algorithm
        for algo_idx, algo in enumerate(ordering):
            order_name, order_args = algo["algorithm_name"], algo["args"]
            algo_result = self._execute_algorithm(order_name, order_args)

            # If token_spans are returned, ensure the "algo" column is set accordingly.
            if "token_spans" in algo_result:
                spans_df = algo_result["token_spans"].copy()
                spans_df["algo"] = f"ordering_algo_{algo_idx}"
                token_spans_list.append(spans_df)

            # Build sort keys using sort_keys if available, or rank_keys otherwise.
            if "sort_keys" in algo_result:
                sort_df[algo_idx] = pd.Series(algo_result["sort_keys"])
            elif "rank_keys" in algo_result:
                sort_df[algo_idx] = -pd.Series(algo_result["rank_keys"])
                rank_keys_dict[algo_idx] = algo_result["rank_keys"]
            else:
                raise RuntimeError(f"Ordering algorithm '{order_name}' did not return valid keys.")

        sort_df.fillna(float('inf'), inplace=True)
        sort_df = sort_df.sort_values(by=list(sort_df.columns))

        # Calculate final sort keys and differentiation info.
        final_sort_keys = {}
        current_rank = 0
        differentiation_info = [0] * len(ordering)
        sorted_line_ids = sort_df.index.tolist()

        i = 0
        while i < len(sorted_line_ids):
            start = i
            while i < len(sorted_line_ids) - 1 and sort_df.iloc[i].equals(sort_df.iloc[i + 1]):
                i += 1
            for j in range(start, i + 1):
                final_sort_keys[sorted_line_ids[j]] = current_rank + 1
            current_rank += (i - start + 1)
            if start != 0:
                for algo_idx in sort_df.columns:
                    if sort_df.iloc[start][algo_idx] != sort_df.iloc[start - 1][algo_idx]:
                        differentiation_info[algo_idx] += 1
                        break
            i += 1

        # Combine token spans from all ordering algorithms.
        if token_spans_list:
            combined_token_spans = pd.concat(token_spans_list, ignore_index=True)
        else:
            combined_token_spans = pd.DataFrame(
                columns=["line_id", "start_id_in_line", "end_id_in_line", "category", "weight", "algo"]
            )

        return {
            "sort_keys": final_sort_keys,
            "differentiation_info": {f"algo_{i}": differentiation_info[i] for i in range(len(differentiation_info))},
            "rank_keys": rank_keys_dict,
            "token_spans": combined_token_spans
        }

    def remove(self):
        """
        Removes the node and all its descendants from the tree.

        The node is detached from its parent, and all references to it and its children are deleted.
        """
        if self.parent:
            self.parent.children = tuple(child for child in self.parent.children if child is not self)
        self.parent = None  # Detach from parent

        # Recursively remove all children
        def delete_descendants(node):
            for child in node.children:
                delete_descendants(child)
            node.children = ()  # Clear children

        delete_descendants(self)

    def view(self, inherit_spans: bool = True) -> dict:
        """
        Returns a concordance view for this node, structured according to the following specification:

        1. "selected_lines": A list of line numbers visible in the view. These refer to the original concordance
           lines (their indices in the concordance metadata). If the node has a 'selected_lines' attribute, that list
           is used; otherwise, the view inherits the selected_lines from the nearest ancestor that has them.
           In the worst-case scenario (if no such ancestor exists), the default is the entire set of line numbers from
           the concordance metadata.

        2. "ordering": A complete ordering of the visible lines, represented as a list of line numbers.
           This ordering is determined by the node's ordering_result. If the node does not have its own ordering_result,
           the ordering is copied from the nearest ancestor that has one. In either case, only the line numbers that are
           present in the current node's selected_lines are retained. If no ordering_result is found, the ordering defaults to
           the original ordering by line numbers (using the line number as the sort key).

        3. "grouping": (Optional) If a grouping algorithm has been applied, this key contains an ordered list of groups.
           The format depends on the grouping strategy:
             a. For partitions, grouping is represented as a flat list of dictionaries, where each partition has:
                 - "id": A unique integer identifier.
                 - "label": A display label.
                 - "line_ids": An ordered list of line numbers (sorted according to the global ordering) belonging to the partition.
                 - Optionally, "prototypes": A list of prototypical line numbers.
                 - Optionally, "info": Additional information (as a dictionary).
             b. For clusters, grouping is represented as a nested (hierarchical) tree. Each cluster dictionary contains the same keys
                as a partition, plus a "children" key with a list of sub-clusters.

        4. "global_info": (Optional) A dictionary containing overall information about the view. For example, this can include
           differentiation information from the ordering_result and any additional info stored in the node (e.g., self.info).

        5. "line_info": (Optional) A dictionary mapping each visible line number to per-line information.
           This information is derived solely from the "rank_keys" provided by ranking algorithms (of type "ranking")
           in the ordering_result, if present. Only entries corresponding to line numbers in selected_lines are included.

        6. "token_spans": (Optional) A list of token span objects (each a dict) to be marked in the KWIC display.
           Each token span dictionary has:
             - "line_id": The line number in which the span occurs.
             - "start_id_in_line": The starting token id (inclusive) relative to the line.
             - "end_id_in_line": The ending token id (inclusive) relative to the line.
             - "category": A string indicating the mark (e.g., "A").
             - "weight": A numerical weight in the range [0,1].

        Additionally, the view always includes a "node_type" key indicating the type of the node (e.g., "subset", "arrangement").

        Returns:
            dict: The concordance view, suitable for JSON serialization.
        """
        view_dict = {}

        # 1. Selected lines.
        if hasattr(self, "selected_lines") and self.selected_lines is not None:
            selected = self.selected_lines
        else:
            # Inherit selected_lines from the nearest ancestor that has it.
            inherited = None
            ancestor = self.parent
            while ancestor:
                if hasattr(ancestor, "selected_lines") and ancestor.selected_lines is not None:
                    inherited = ancestor.selected_lines
                    break
                ancestor = ancestor.parent
            if inherited is not None:
                selected = inherited
            else:
                n = len(self.concordance().metadata)
                selected = list(range(n))
        view_dict["selected_lines"] = selected

        # 2. Ordering.
        if hasattr(self, "ordering_result") and "sort_keys" in self.ordering_result:
            sort_keys = self.ordering_result["sort_keys"]
        else:
            ancestor = self.parent
            sort_keys = None
            while ancestor:
                if hasattr(ancestor, "ordering_result") and "sort_keys" in ancestor.ordering_result:
                    sort_keys = ancestor.ordering_result["sort_keys"]
                    break
                ancestor = ancestor.parent
            if sort_keys is None:
                sort_keys = {line_id: line_id for line_id in selected}
        ordering = [line_id for line_id, _ in sorted(sort_keys.items(), key=lambda item: (item[1], item[0]))]
        # Filter ordering to only include lines that are in the current node's selected_lines.
        ordering = [line for line in ordering if line in selected]
        view_dict["ordering"] = ordering

        # 3. Grouping.
        if hasattr(self, "grouping_result"):
            view_dict["grouping"] = self.grouping_result

        # 4. Global info.
        global_info = {}
        if hasattr(self, "ordering_result") and "differentiation_info" in self.ordering_result:
            global_info["differentiation_info"] = self.ordering_result["differentiation_info"]
        if hasattr(self, "info"):
            global_info.update(self.info)
        if global_info:
            view_dict["global_info"] = global_info

        # 5. Line info (currently taking care of ranking only)
        column_info = []
        data = {}
        if hasattr(self, "ordering_result") and "rank_keys" in self.ordering_result:
            ordering_algos = self.algorithms.get("ordering", [])
            for algo_key, rank_map in self.ordering_result["rank_keys"].items():
                # map algo_key → algorithm name from self.algorithms["ordering"]
                if isinstance(algo_key, int) and algo_key < len(ordering_algos):
                    algo_name = ordering_algos[algo_key].get("algorithm_name", f"Algorithm_{algo_key}")
                else:
                    algo_name = f"Algorithm_{algo_key}"
                col_key = f"Ranking: {algo_name}"

                column_info.append({
                    "key": col_key,
                    "algorithm": algo_name,
                    "algorithm_index_within_ordering": algo_key,
                    "type": "rank",
                    "description": f"Ranking score of the line from {algo_name} (ordering algorithm #{algo_key})"
                })

                for line_id, rank_val in rank_map.items():
                    if line_id in selected:
                        data.setdefault(line_id, {})[col_key] = rank_val

        if column_info:
            view_dict["line_info"] = {
                "column_info": column_info,
                "data": data
            }

        # 6. Token-spans (inherit from ancestors if requested)
        selected_set = set(view_dict["selected_lines"])

        # gather span sources:  (depth_from_self, spans)
        if inherit_spans:
            spans_srcs = []
            depth, node = 0, self
            while node is not None:
                spans_srcs.append((depth, getattr(node, "token_spans", None)))
                node, depth = node.parent, depth + 1
        else:
            spans_srcs = [(0, getattr(self, "token_spans", None))]

        # helper → always return a DataFrame (or None)
        def _as_df(sp):
            if sp is None:
                return None
            if isinstance(sp, pd.DataFrame):
                return sp.copy()
            return pd.DataFrame(sp)

        used_labels: list[str] = []      # overall label order A,B,C…
        dfs: list[pd.DataFrame] = []

        for depth, spans in spans_srcs:          # child first
            df = _as_df(spans)
            if df is None or df.empty:
                continue

            # keep only spans belonging to selected lines
            df = df[df["line_id"].isin(selected_set)]
            if df.empty:
                continue

            # label-renaming to avoid clashes
            cat_map = {}
            for cat in df["category"]:
                if cat in cat_map:
                    continue

                keep_label = (depth == 0) and (cat not in used_labels)
                if keep_label:
                    new_lab = cat
                else:
                    # find next unused letter(s): A..Z, AA, BB, …
                    idx = len(used_labels)
                    while True:
                        letter = chr(65 + idx % 26)
                        wraps = idx // 26
                        new_lab = letter if wraps == 0 else letter * (wraps + 1)
                        if new_lab not in used_labels:
                            break
                        idx += 1

                cat_map[cat] = new_lab
                used_labels.append(new_lab)

            df["category"] = df["category"].map(cat_map)
            dfs.append(df)

        if dfs:
            view_dict["token_spans"] = (
                pd.concat(dfs, ignore_index=True)
                  .sort_values(["line_id", "start_id_in_line"])
                  .to_dict(orient="records")
            )

        # Node type.
        view_dict["node_type"] = self.node_type

        return view_dict

    def available_algorithms(self):
        """
        Returns all algorithms available at the current node with their metadata,
        sorted by algorithm type (in the order: selection, sorting, ranking, partitioning, clustering)
        then alphabetically by module name and algorithm name.
        """
        from collections import OrderedDict

        # Define desired order for algorithm types
        type_order = {
            "selection": 1,
            "sorting": 2,
            "ranking": 3,
            "partitioning": 4,
            "clustering": 5
        }

        # Get the Concordance object from the weak reference
        concordance = self.concordance()
        if not concordance:
            raise RuntimeError("Concordance reference is no longer valid.")

        # Filter available algorithms: skip annotations and check conditions.
        available_algos = {}
        for algo_name, algo_metadata in concordance.available_algorithms.items():
            if algo_metadata.get("algorithm_type") == "annotation":
                continue

            conditions = algo_metadata.get("conditions", {})
            matches_conditions = True

            # Check conditions on the concordance object.
            if "concordance" in conditions:
                conc_conditions = conditions["concordance"]
                conc = self.concordance()
                for attr, expected in conc_conditions.items():
                    if getattr(conc, attr, None) != expected:
                        matches_conditions = False
                        break

            # Check conditions on the node.
            if matches_conditions and "node" in conditions:
                node_conditions = conditions["node"]
                for attr, expected in node_conditions.items():
                    if getattr(self, attr, None) != expected:
                        matches_conditions = False
                        break

            if "x-eval" in conditions:
                eval_conditions = conditions["x-eval"]
                try:
                    res = eval(eval_conditions, {**globals(), "conc": self.concordance(), "node": self})
                    if res == False:
                        matches_conditions = False
                except Exception as e:
                    matches_conditions = False

            if matches_conditions:
                available_algos[algo_name] = copy.deepcopy(algo_metadata)
                available_algos[algo_name]["args_schema"] = self.schema_for(algo_name)

        # Sort first by type (using the ordering above), then by module name, then by algorithm name.
        sorted_items = sorted(
            available_algos.items(),
            key=lambda item: (
                type_order.get(item[1].get("algorithm_type", "").lower(), 99),
                item[1].get("module", "").lower(),
                item[1].get("name", item[0]).lower()
            )
        )

        return OrderedDict(sorted_items)

    def schema_for(self, algorithm_name: str) -> dict:
        """
        Returns the enriched args_schema for the specified algorithm in the context of this node.
        The enrichment process evaluates any Python expressions under the custom keyword 'x-eval' in the schema.

        Parameters:
            algorithm_name (str): The name of the algorithm (e.g., "sort_by_token_attribute").

        Returns:
            dict: The enriched args_schema, with dynamic values (like enum, min, max, etc.) added.
        """
        # Retrieve the algorithm metadata from the Concordance available_algorithms.
        conc = self.concordance()
        algo_meta = conc.available_algorithms.get(algorithm_name)
        if not algo_meta:
            raise ValueError(f"Algorithm '{algorithm_name}' not found.")

        base_schema = algo_meta.get("args_schema", {})
        enriched_schema = self._enrich_schema(base_schema)
        return enriched_schema

    def _enrich_schema(self, schema: dict) -> dict:
        """
        Enrich the base *args* schema by **recursively** evaluating every
        ``x-eval`` expression, no matter where it appears in the JSON-Schema
        structure.

        The expression is executed in a restricted namespace where

        * ``conc`` – the current :class:`Concordance` object
        * ``node`` – this :class:`AnalysisTreeNode`

        are available.
        The result is handled as follows
        • *dict* → merged into the current schema node
        • *list* → stored under ``"enum"``
        • anything else → stored under ``"default"``

        If evaluation fails, the exception text is placed under
        ``"x-eval-error"`` so the UI can fall back gracefully.

        Parameters
        ----------
        schema : dict
            The original (un-enriched) JSON-Schema.

        Returns
        -------
        dict
            The schema with all ``x-eval`` expressions resolved.
        """
        def _process(node):
            if isinstance(node, dict):
                node = node.copy()
                if "x-eval" in node:
                    expr = node.pop("x-eval")
                    try:
                        res = eval(expr, {"conc": self.concordance(), "node": self})
                        if isinstance(res, dict):
                            node.update(res)
                        elif isinstance(res, list):
                            node.setdefault("enum", res)
                        else:
                            node.setdefault("default", res)
                    except Exception as e:
                        node["x-eval-error"] = str(e)
                # recurse into schema containers
                for k, v in list(node.items()):
                    if k in ("properties",):
                        node[k] = {p: _process(s) for p, s in v.items()}
                    elif k in ("items", "additionalProperties"):
                        node[k] = _process(v)
                    elif isinstance(v, (dict, list)):
                        node[k] = _process(v)
                return node
            if isinstance(node, list):
                return [_process(x) for x in node]
            return node

        return _process(schema)

    def summary(self, format="json"):
        """
        Returns a summary of this analysis tree node in the specified format.

        Parameters:
            format (str): The output format. One of:
                - "dict": returns a Python dictionary,
                - "json": returns a JSON-formatted string,
                - "text": returns a plain text summary,
                - "html": returns an HTML-formatted summary.
                Default is "json".

        Returns:
            If format == "dict": a dictionary with keys:
                - "node_type": the node type,
                - "line_count": the number of lines,
                - "label": the node's label,
                - "algorithms": a list where each element is a dictionary with:
                    - "type": the algorithm's type,
                    - "name": the algorithm's name,
                    - "args": the raw arguments (a dictionary).
            If format == "json": a JSON string representing the above dictionary.
            If format == "text": a plain text summary.
            If format == "html": an HTML-formatted summary.
        """
        import json

        # Build the structured summary dictionary.
        summary_dict = {
            "node_type": self.node_type,
            "line_count": self.line_count,
            "label": self.label,
            "algorithms": []
        }
        if hasattr(self, "algorithms") and self.algorithms:
            for algo_key, algo_val in self.algorithms.items():
                if algo_val is None:
                    continue
                if isinstance(algo_val, list):
                    for a in algo_val:
                        summary_dict["algorithms"].append({
                            "type": a.get("algorithm_type", "Unknown"),
                            "name": a.get("algorithm_name", "Unknown"),
                            "args": a.get("args", {})
                        })
                elif isinstance(algo_val, dict):
                    summary_dict["algorithms"].append({
                        "type": algo_val.get("algorithm_type", "Unknown"),
                        "name": algo_val.get("algorithm_name", "Unknown"),
                        "args": algo_val.get("args", {})
                    })

        if format == "dict":
            return summary_dict
        elif format == "json":
            return json.dumps(summary_dict)
        elif format == "text":
            parts = [
                f"Node type: {self.node_type}",
                f"Line count: {self.line_count}",
                f"Label: {self.label}"
            ]
            if summary_dict["algorithms"]:
                parts.append("Algorithms:")
                for algo in summary_dict["algorithms"]:
                    args_str = json.dumps(algo["args"], indent=2)
                    parts.append(f"  - {algo['name']} ({algo['type']}): args: {args_str}")
            return "\n".join(parts)
        elif format == "html":
            html_parts = [
                f"<div><strong>Node type:</strong> {self.node_type}</div>",
                f"<div><strong>Line count:</strong> {self.line_count}</div>",
                f"<div><strong>Label:</strong> {self.label}</div>"
            ]
            if summary_dict["algorithms"]:
                html_parts.append("<div><strong>Algorithms:</strong><ul>")
                for algo in summary_dict["algorithms"]:
                    args_str = json.dumps(algo["args"], indent=2)
                    html_parts.append(f"<li>{algo['name']} ({algo['type']}): args: {args_str}</li>")
                html_parts.append("</ul></div>")
            return "\n".join(html_parts)
        else:
            raise ValueError("Unsupported format. Use 'dict', 'json', 'text', or 'html'.")

class Concordance:
    """
    The Concordance class manages and manipulates concordance data, including metadata,
    tokens, matches, and an analysis tree.
    """

    def __init__(
        self,
        metadata: Union[str, pd.DataFrame] = None,
        tokens: Union[str, pd.DataFrame] = None,
        matches: Union[str, pd.DataFrame] = None,
        root: AnalysisTreeNode = None,
        info: Union[str, Dict[str, Any]] = {},
        resources: ResourceRegistry = None,
        algorithms_folder: str = "algorithms"
    ) -> Concordance:

        self.metadata: Optional[pd.DataFrame] = None
        self.tokens: Optional[pd.DataFrame] = None
        self.matches: Optional[pd.DataFrame] = None
        self.info: Dict[str, Any] = {}
        if resources is None:
            self.resources = ResourceRegistry()
        else:
            self.resources = resources
        self.subsets: Dict[int, ConcordanceSubset] = {}
        self.algorithms: Dict[str, Dict[str, Any]] = {}
        self.annotations: list[dict] = []

        if metadata is not None or tokens is not None or matches is not None or info:
            self.load(metadata, tokens, matches, info=info)

        # Ensure offset column if it can be derived
        self._ensure_offset_column()

        # Ensure root is set; if None, create an empty root node
        self.root = root if root else AnalysisTreeNode(node_type="subset", parent=None, concordance=self, label="Root")
        if not hasattr(self.root, "line_count"):
            if self.metadata is not None:
                self.root.line_count = len(self.metadata)
            else:
                self.root.line_count = 0

        self.node_counter = len(self.root.descendants) + 1
        self._load_methods(paths=["utils/retrieve.py"])
        self._load_methods(paths=["utils/retrieve_resources.py"])
        self.available_algorithms = self._scan_algorithms(algorithms_folder)

        # Automatically attach algorithm helper functions
        # self._initialize_algorithm_functions()

    def _ensure_offset_column(self):
        """
        If every line uses the same slot and *tokens* lack an “offset”
        column, create it.  Offsets are computed against the token’s
        **global** row index:

            i < match_start   →  i - match_start   (negative)
            i ∈ match_span    →  0
            i > match_end     →  i - match_end     (positive)
        """
        if (
                self.tokens is None or self.matches is None
                or 'offset' in self.tokens.columns
                or 'slot' not in self.matches.columns
                or self.matches['slot'].nunique(dropna=True) != 1
        ):
            return  # nothing to do

        # map line_id → (match_start, match_end) in *global* token index
        bounds = (
            self.matches
            .set_index('line_id')[['match_start', 'match_end']]
            .dropna()
            .astype(int)
            .to_dict(orient='index')
        )

        def _offset(row):
            b = bounds.get(row['line_id'])
            if not b:
                return 0
            ms, me = b['match_start'], b['match_end']
            idx = row.name  # global token row index
            if idx < ms:
                return idx - ms
            if idx > me:
                return idx - me
            return 0

        self.tokens['offset'] = self.tokens.apply(_offset, axis=1)


    def _scan_algorithms(self, folder_path):
        """
        Scans the algorithms folder for Python files and stores available algorithms with metadata.

        Args:
            folder_path (str): Path to the folder containing algorithm files.

        Returns:
            dict: A dictionary where keys are full algorithm names, and values include:
                  - module: The module containing the algorithm
                  - function: The function name
                  - conditions: Attributes required for the algorithm to be applicable to a node
                  - short_names: List of short names for the algorithm
        """
        module_dir = Path(__file__).parent
        algorithms = {}

        for filename in os.listdir(module_dir / folder_path):
            if filename.endswith(".py") and filename != "__init__.py":
                # Build the full module name with dots for later loading
                relative_path = Path(folder_path) / filename
                module_name = relative_path.as_posix().replace('/', '.')[:-3]  # Strip '.py'

                module_path = module_dir / folder_path / filename

                try:
                    # Parse the file with `ast` to avoid heavy imports
                    with open(module_path, "r", encoding="utf-8") as f:
                        source_code = f.read()

                    tree = ast.parse(source_code)

                    for node in tree.body:
                        if isinstance(node, ast.FunctionDef):  # Look for function definitions
                            function_name = node.name
                            function_metadata = None

                            # Search for _algorithm_metadata inside the function body
                            for inner_node in node.body:
                                if (
                                        isinstance(inner_node, ast.Assign)
                                        and len(inner_node.targets) == 1
                                        and isinstance(inner_node.targets[0], ast.Attribute)
                                        and inner_node.targets[0].attr == "_algorithm_metadata"
                                        and isinstance(inner_node.targets[0].value, ast.Name)
                                        and inner_node.targets[0].value.id == function_name
                                ):
                                    # Extract metadata from the assignment
                                    function_metadata = ast.literal_eval(inner_node.value)
                                    break

                            # If metadata is found, store it
                            if function_metadata:
                                full_name = function_metadata["name"]
                                function_metadata["short_names"] = function_metadata.get("short_names", [])
                                algorithms[full_name] = {
                                    "full_name": full_name,
                                    "module": module_name,
                                    "function": function_name,
                                    "algorithm_type": function_metadata.get("algorithm_type", ""),
                                    "requires": function_metadata.get("requires", []),
                                    "scope": function_metadata.get("scope", None),
                                    "conditions": function_metadata.get("conditions", {}),
                                    "short_names": function_metadata["short_names"],
                                    "args_schema": function_metadata["args_schema"]
                                }

                except Exception as e:
                    print(f"Failed to process {filename}: {e}")

        return algorithms

    def _load_methods(self, paths: List[str]) -> None:
        """
        Dynamically load methods from the specified Python files or folders
        and attach them to the class instance.

        Args:
            paths (List[str]): A list of module paths or folder paths containing Python files to import methods from.
        """
        current_dir = os.path.dirname(os.path.abspath(__file__))
        parent_dir = current_dir

        if parent_dir not in sys.path:
            sys.path.append(parent_dir)

        for path in paths:
            full_path = os.path.join(parent_dir, path)

            if os.path.isdir(full_path):
                for filename in os.listdir(full_path):
                    if filename.endswith('.py') and filename != '__init__.py':
                        module_name = f"{path.replace('/', '.')}.{filename[:-3]}"
                        self._import_and_bind_methods(module_name)
            elif os.path.isfile(full_path) and full_path.endswith('.py'):
                module_name = path.replace('/', '.').rstrip('.py')
                self._import_and_bind_methods(module_name)
            else:
                raise FileNotFoundError(f"Invalid path or file: '{full_path}'")

    def _import_and_bind_methods(self, module_name: str) -> None:
        """
        Import a module by name and bind all callable methods with 'self' as the first parameter
        to the class instance.

        Args:
            module_name (str): The module name to import and bind methods from.
        """
        try:
            module = importlib.import_module(module_name)

            for func_name in dir(module):
                func = getattr(module, func_name)
                if callable(func):
                    sig = inspect.signature(func)
                    params = list(sig.parameters.values())

                    if params and params[0].name == 'self':
                        setattr(self, func_name, func.__get__(self, self.__class__))
        except Exception as e:
            logger.error(f"Error importing {module_name}: {e}")
            raise

    def _initialize_algorithm_functions(self):
        """
        Attach algorithm helper functions (with tab-completion and signatures) to this Concordance object.
        Each function returns (algorithm_name, args_dict).
        """
        algorithms = self.root.available_algorithms()

        for function_name, schema in algorithms.items():
            func_name = "_" + schema['function']

            props = schema.get("args_schema", {}).get("properties", {})

            parameters = []
            docstring_lines = [f"{schema.get('full_name', function_name)}: {schema.get('description', '').strip()}\n"]
            docstring_lines.append("Args:")

            for arg_name, arg_info in props.items():
                # arg_type = self._map_json_type_to_python(arg_info.get("type", "string"))
                default = inspect.Parameter.empty

                parameters.append(inspect.Parameter(
                    name=arg_name,
                    kind=inspect.Parameter.KEYWORD_ONLY,
                    # annotation=arg_type,
                    default=default
                ))

                #docstring_lines.append(f"    {arg_name} ({arg_type.__name__}): {arg_info.get('description', '')}")
                docstring_lines.append(f"    {arg_name}: {arg_info.get('description', '')}")

            docstring_lines.append("\nReturns:\n    Tuple[str, dict]: (algorithm_name, args_dict)")

            sig = inspect.Signature(parameters)

            def make_algo_func(algorithm_name):
                def algo_func(self, **kwargs):
                    return (algorithm_name, kwargs)
                return algo_func

            func = make_algo_func(function_name)
            func.__signature__ = sig
            func.__doc__ = "\n".join(docstring_lines)

            setattr(self, func_name, types.MethodType(func, self))


    def _map_json_type_to_python(self, json_type):
        """Simple mapping from JSON schema types to Python types."""
        return {
            "integer": int,
            "number": float,
            "string": str,
            "boolean": bool,
            "array": list,
            "object": dict,
        }.get(json_type, str)

    def add_annotation(self, algo_tuple, column_names=None):
        """
        Executes an annotation algorithm and attaches the resulting column(s) to the appropriate dataframe.
        The algorithm is provided as a tuple (algorithm_name, args). Multiple annotations can be requested at the same time.
        If column_names is not provided, the default naming is taken from the returned Series/DataFrame's column names.

        Parameters:
            algo_tuple (tuple): A tuple (algorithm_name, args) where:
                - algorithm_name (str): The name of the annotation algorithm to execute.
                - args (dict): The arguments to pass to the algorithm.
            column_names (list of str, optional): A list of column names to add to the dataframe.
                For single-column annotations, provide a one-element list.
                If omitted, default names are taken from the algorithm result.

        Returns:
            None

        Raises:
            ValueError: If the algorithm is not of type "annotation", if the scope is invalid,
                        if the returned Series does not have an expected name (for single-column results),
                        or if the number of columns returned does not match the number of provided column names.
            RuntimeError: If the algorithm fails or returns invalid results.
        """
        # Extract algorithm_name and args from the tuple.
        algorithm_name, args = algo_tuple

        # Retrieve the algorithm metadata.
        algorithm_metadata = self.available_algorithms.get(algorithm_name)
        if not algorithm_metadata:
            raise ValueError(f"Algorithm '{algorithm_name}' not found in available algorithms.")

        # Check if the algorithm type is "annotation".
        if algorithm_metadata.get("algorithm_type") != "annotation":
            raise ValueError(f"Algorithm '{algorithm_name}' is not of type 'annotation'.")

        # Determine the scope: "token" or "line".
        scope = algorithm_metadata.get("scope")
        if scope not in ["token", "type", "line"]:
            raise ValueError(
                f"Invalid scope '{scope}' for algorithm '{algorithm_name}'. Must be 'token', 'type', or 'line'.")

        # Ensure column_names is a list of strings if provided.
        if column_names is not None:
            if isinstance(column_names, str):
                column_names = [column_names]
            elif not (isinstance(column_names, list) and all(isinstance(col, str) for col in column_names)):
                raise ValueError("column_names must be a list of strings.")

        # Ensure the column names do not already exist in the target dataframe.
        if scope == "line":
            if column_names is not None:
                for col in column_names:
                    if col in self.metadata.columns:
                        raise ValueError(f"Column '{col}' already exists in metadata.")
        elif scope == "token":
            if column_names is not None:
                for col in column_names:
                    if col in self.tokens.columns:
                        raise ValueError(f"Column '{col}' already exists in tokens.")

        # Dynamically load and execute the algorithm.
        module_name = algorithm_metadata["module"]
        function_name = algorithm_metadata["function"]

        try:
            module = __import__(module_name, fromlist=[function_name])
            algorithm_function = getattr(module, function_name)

            # Add default values from metadata if missing in args.
            schema_properties = algorithm_metadata.get("args_schema", {}).get("properties", {})
            for key, details in schema_properties.items():
                if key not in args and "default" in details:
                    args[key] = details["default"]

            # Validate args against the enriched schema.
            enriched_schema = self.root.schema_for(algorithm_name)
            from jsonschema import validate
            validate(instance=args, schema=enriched_schema)

            # Execute the algorithm and get the result.
            annotation_result = algorithm_function(self, **args)

            # Process the result.
            success = False
            if scope=="type":
                if annotation_result.get("type") == "frequency_list":
                    if isinstance(column_names, str):
                        column_names = [column_names]
                    self.resources.register_frequency_list(name=column_names[0],
                                                        df=annotation_result.get("df"),
                                                        sample_size=annotation_result.get("sample_size"),
                                                        info=annotation_result.get("info", {}))
                    success = True

                elif annotation_result.get("type") == "scores":
                    if isinstance(column_names, str):
                        column_names = [column_names]
                    self.resources.register_scores(name=column_names[0],
                                                df=annotation_result.get("df"),
                                                info=annotation_result.get("info", {}))
                    success = True


            if isinstance(annotation_result, pd.Series):
                # Use the result's name as default if column_names not provided.
                if column_names is None:
                    column_names = [annotation_result.name if annotation_result.name is not None else algorithm_name]
                if len(column_names) != 1:
                    raise ValueError("Annotation returned a single column, but multiple column names were provided.")
                if scope == "line":
                    self.metadata[column_names[0]] = annotation_result
                elif scope == "token":
                    self.tokens[column_names[0]] = annotation_result
                success = True
            elif isinstance(annotation_result, pd.DataFrame):
                if column_names is None:
                    column_names = list(annotation_result.columns)
                if len(annotation_result.columns) != len(column_names):
                    raise ValueError(
                        "The number of columns returned does not match the number of provided column names.")
                # Rename columns of the returned DataFrame to the provided column names.
                annotation_result = annotation_result.rename(columns=dict(zip(annotation_result.columns, column_names)))
                if scope == "line":
                    self.metadata = pd.concat([self.metadata, annotation_result], axis=1)
                elif scope == "token":
                    self.tokens = pd.concat([self.tokens, annotation_result], axis=1)
                success = True
            else:
                raise RuntimeError(f"Algorithm '{algorithm_name}' did not return a pandas Series or DataFrame.")

            self.annotations.append({
                "algorithm": algorithm_name,
                "args": args,
                "column_names": column_names,
                "scope": scope
            })

        except Exception as e:
            raise RuntimeError(f"Failed to execute annotation algorithm '{algorithm_name}': {e}")

    def _all_keys_convertible_to_int(self, d: Any) -> bool:
        """
        Check if all keys in a dictionary are convertible to integers.

        Parameters:
            d (Any): The object to check.

        Returns:
            bool: True if all keys are convertible to integers, False otherwise.
        """
        if isinstance(d, dict):
            try:
                return all(isinstance(int(k), int) for k in d.keys())
            except (ValueError, TypeError):
                return False
        return False

    def _convert_keys_to_int(self, d: Any) -> Any:
        """
        Convert string keys to integers in a nested dictionary if possible.

        Parameters:
            d (Any): The dictionary or list to convert.

        Returns:
            Any: The dictionary or list with keys converted to integers where possible.
        """
        if isinstance(d, dict):
            if self._all_keys_convertible_to_int(d):
                return {int(k): self._convert_keys_to_int(v) for k, v in d.items()}
            else:
                return {k: self._convert_keys_to_int(v) for k, v in d.items()}
        elif isinstance(d, list):
            return [self._convert_keys_to_int(item) for item in d]
        else:
            return d

    def load(
            self,
            metadata: Union[str, pd.DataFrame],
            tokens: Union[str, pd.DataFrame],
            matches: Union[str, pd.DataFrame],
            root: Optional[Union[str, AnalysisTreeNode]] = None,
            info: Optional[Union[str, Dict[str, Any]]] = None
    ) -> None:
        """
        Populates the Concordance object with provided metadata, tokens, matches, an optional root node, and additional info.
        They can be provided as pandas DataFrames or file paths to TSV files, which follow the specifications of MTSV format (Anthony & Evert 2019).

        Parameters:
            metadata (str or pd.DataFrame): The path to the TSV file containing the structural attributes of concordance lines,
                or a Pandas DataFrame with the same data.
            tokens (str or pd.DataFrame): The path to the TSV file containing the tokens of concordance lines,
                or a Pandas DataFrame with the same data.
            matches (str or pd.DataFrame): The path to the TSV file containing the query matches of concordance lines,
                or a Pandas DataFrame with the same data.
            root (str, AnalysisTreeNode, optional): The path to the JSON file containing the analysis root node,
                or an `AnalysisTreeNode` instance. If None, a new root node is created.
            info (str, dict, optional): The path to the JSON file containing additional info, or a dictionary with the same data.
                If None, no additional info is added.

        Raises:
            ValueError: If the input types are incorrect.
            FileNotFoundError: If specified files are not found.
        """
        # Load metadata
        if isinstance(metadata, str):
            self.metadata = pd.read_csv(metadata, sep='\t')
        elif isinstance(metadata, pd.DataFrame):
            self.metadata = metadata
        else:
            raise ValueError("Metadata should be either a path to a TSV file or a Pandas DataFrame.")

        # Ensure line_id column is present
        if "line_id" not in metadata.columns:
            metadata["line_id"] = metadata.index

        # Load tokens
        if isinstance(tokens, str):
            self.tokens = pd.read_csv(tokens, sep='\t')
        elif isinstance(tokens, pd.DataFrame):
            self.tokens = tokens
        else:
            raise ValueError("Tokens should be either a path to a TSV file or a Pandas DataFrame.")

        # Load matches
        if isinstance(matches, str):
            self.matches = pd.read_csv(matches, sep='\t')
        elif isinstance(matches, pd.DataFrame):
            self.matches = matches
        else:
            raise ValueError("Matches should be either a path to a TSV file or a Pandas DataFrame.")

        # Load or create the root node
        if root is None:
            self.root = AnalysisTreeNode(node_type="subset", parent=None, concordance=self, label="Root")
        elif isinstance(root, str):
            with open(root, 'r') as f:
                root_data = json.load(f)
                self.root = self._build_tree_from_data(root_data)
        elif isinstance(root, AnalysisTreeNode):
            self.root = root
        else:
            raise ValueError("Root should be either a path to a JSON file or an instance of AnalysisTreeNode.")
        self.root.line_count = len(self.metadata)

        # Load or assign additional info
        if info is None:
            self.info = {}
        elif isinstance(info, str):
            with open(info, 'r') as f:
                self.info = json.load(f)
                self.info = self._convert_keys_to_int(self.info)
        elif isinstance(info, dict):
            self.info = self._convert_keys_to_int(info)
        else:
            raise ValueError("Info should be either a path to a JSON file or a dictionary.")

    def _build_tree_from_data(self, root_data: Dict[str, Any]) -> AnalysisTreeNode:
        """
        Helper method to build the analysis tree from a JSON structure.

        Parameters:
            root_data (dict): Dictionary representing the root node and its children.

        Returns:
            AnalysisTreeNode: The root of the reconstructed analysis tree.
        """
        nodes: Dict[int, AnalysisTreeNode] = {}

        # Recursive function to construct tree
        def construct_node(node_data, parent=None):
            node_id = node_data["id"]

            # Create the node and store it in the dictionary
            node = AnalysisTreeNode(
                id=node_id,
                node_type=node_data["node_type"],
                parent=parent,
                concordance=self,
                label=node_data.get("label", ""),
                **{k: v for k, v in node_data.items() if k not in ["id", "node_type", "parent_id", "children"]}
            )
            nodes[node_id] = node

            # Recursively process children if they exist
            for child_data in node_data.get("children", []):
                construct_node(child_data, parent=node)

            return node

        # Start tree construction
        return construct_node(root_data)


    def _locate_slot_to_use(self, node: AnalysisTreeNode) -> int:
        """
        Traverses the operations tree upwards from the specified node to find the closest node with the 'slot_to_use' attribute.
        Returns the value of 'slot_to_use' if found, otherwise returns 0.
        """
        current_node = node
        while current_node is not None:
            if hasattr(current_node, 'slot_to_use'):
                return current_node.slot_to_use
            current_node = current_node.parent
        return 0

    def subset_at_node(self, node: AnalysisTreeNode) -> ConcordanceSubset:
        """
        Get the subset of the data at a given node in the analysis tree.

        Parameters:
        - node (AnalysisTreeNode): The node to get the subset from.

        Returns:
        - ConcordanceSubset: An object representing the subset of data at the specified node.
        """

        # Find slot_to_use
        slot_to_use = self._locate_slot_to_use(node)

        # Initialize selected_lines as None
        selected_lines = None

        # Check if the node has selected_lines; if not, check ancestors
        if hasattr(node, 'selected_lines') and node.selected_lines:
            selected_lines = node.selected_lines
        else:
            current_node = node
            while current_node is not None:
                if hasattr(current_node, 'selected_lines') and current_node.selected_lines:
                    selected_lines = current_node.selected_lines
                    break
                current_node = current_node.parent

        # Use the selected_lines (if found), otherwise use the full dataset
        if selected_lines is not None:
            metadata_subset = self.metadata.loc[selected_lines]
            tokens_subset = self.tokens[self.tokens['line_id'].isin(selected_lines)]
            matches_subset = self.matches[self.matches['line_id'].isin(selected_lines)]
        else:
            metadata_subset = self.metadata
            tokens_subset = self.tokens
            matches_subset = self.matches

        # Compute "offset" column in tokens_subset
        matches_subset_slot = matches_subset[matches_subset['slot'] == slot_to_use]

        # For each line_id, take the first match
        matches_subset_slot = matches_subset_slot.groupby('line_id').first().reset_index()

        # Merge tokens_subset with matches_subset_slot on 'line_id'
        tokens_with_match = tokens_subset.merge(
            matches_subset[['line_id', 'match_start', 'match_end']],
            on='line_id',
            how='left'
        ).set_index(tokens_subset.index)

        tokens_with_match['offset'] = tokens_with_match.apply(
            lambda row: 0 if pd.isnull(row['match_start']) or pd.isnull(row['match_end'])
            else (row.name - row['match_start'] if row.name < row['match_start']
                  else row.name - row['match_end'] if row.name > row['match_end']
            else 0),
            axis=1
        )

        # Remove temporary columns
        tokens_with_match.drop(columns=['match_start', 'match_end'], inplace=True)

        # Return a ConcordanceSubset object containing filtered data
        return ConcordanceSubset(
            metadata=metadata_subset,
            tokens=tokens_with_match,
            matches=matches_subset,
            root=self.root,
            original_size=len(self.metadata),
            active_node=node
        )

    def _get_ordered_line_ids(self, order_keys: Dict[int, Union[int, float]], reverse: bool = False) -> List[int]:
        """
        Returns the list of line IDs ordered by their associated order keys (surrogate keys or natural keys).

        Parameters:
        - order_keys (dict): A dictionary where keys are line IDs and values are order keys (surrogate keys or natural keys).
        - reverse (bool): If True, return the line IDs in descending order. Default is False (ascending order).

        Returns:
        - list: A list of line IDs ordered by their associated surrogate keys.
        """
        # Sort the line_ids based on the values (random integers or floats), optionally in reverse order
        ordered_line_ids = [line_id for line_id, _ in
                            sorted(order_keys.items(), key=lambda item: (item[1], item[0]), reverse=reverse)]

        return ordered_line_ids


    def export(self, export_path: str, as_zip: Optional[bool] = False) -> None:
        """
        Exports the metadata, tokens, matches as TSV files, and the tree and info as JSON files.
        The files will be stored either in a folder or as a zip archive.

        Parameters:
        - export_path (str): The path to the export folder or zip archive.
        - as_zip (bool, optional): If True, compress the files into a zip archive. Defaults to False.
        """

        # Prepare file content as dictionary with filenames and content
        export_data = {
            'metadata.tsv': self.metadata.to_csv(sep='\t', index=False),
            'tokens.tsv': self.tokens.to_csv(sep='\t', index=False),
            'matches.tsv': self.matches.to_csv(sep='\t', index=False),
            'root.json': json.dumps(self._node_to_dict(self.root), indent=4),
            'info.json': json.dumps(self.info, indent=4)
        }

        export_path = Path(export_path)

        if as_zip:
            # If exporting as zip, create zip archive directly
            zip_file = export_path.with_suffix('.zip') if not export_path.suffix else export_path
            with zipfile.ZipFile(zip_file, 'w', zipfile.ZIP_DEFLATED) as zipf:
                for filename, content in export_data.items():
                    zipf.writestr(filename, content)
        else:
            # Create the export directory if it doesn't exist
            export_path.mkdir(parents=True, exist_ok=True)

            # Write each file to the export directory
            for filename, content in export_data.items():
                with open(export_path / filename, 'w') as f:
                    f.write(content)

    def _node_to_dict(self, node):
        """
        Converts a tree node to a dictionary for serialization to JSON.

        Parameters:
        - node: The node to convert.

        Returns:
        - dict: A dictionary representation of the node.
        """
        node_dict = {key: getattr(node, key) for key in node.__dict__ if not key.startswith('_')}
        if node.children:
            node_dict['children'] = [self._node_to_dict(child) for child in node.children]
        return node_dict

    def view_to_csv(self, node_id: int, output_file: str, token_attr: str = 'word') -> None:
        """
        Outputs a CSV file with columns from metadata, Left context, KWIC, Right context.
        If the node has 'rank_keys', it also includes ranking columns corresponding to each algorithm.
        If the node is of type 'partition', it also includes a 'Partition label' column.
        Lines are ordered according to the node's ordering or partitions.
        The order of columns is:
        - line_id
        - Partition label (if present)
        - Ranking columns (if present)
        - Left context
        - KWIC
        - Right context
        - Metadata columns

        Parameters:
        - node_id (int): The ID of the node to start from.
        - output_file (str): The path to the output CSV file.
        - token_attr (str): The token attribute to use for joining tokens. Default is 'word'.
        """

        node = self.find_node_by_id(node_id)

        # Get the subset at this node
        subset = self.subset_at_node(node)

        # Get tokens and metadata
        tokens = subset.tokens.copy()
        metadata = subset.metadata.copy()

        # Ensure line_id is a column (and not just index)
        if 'line_id' not in metadata.columns:
            metadata = metadata.reset_index()

        # Add 'context_type' column based on 'offset'
        tokens['context_type'] = np.where(
            tokens['offset'] < 0, 'Left context',
            np.where(tokens['offset'] == 0, 'KWIC', 'Right context')
        )

        # If no tokens, create an empty contexts DataFrame with the same line_ids as metadata
        if len(tokens) == 0:
            contexts = pd.DataFrame({'line_id': metadata['line_id']})
            # Add empty columns for contexts
            for ccol in ['Left context', 'KWIC', 'Right context']:
                contexts[ccol] = ''
        else:
            # Group tokens by 'line_id' and 'context_type', and join tokens
            contexts = tokens.groupby(['line_id', 'context_type'])[token_attr].apply(' '.join).unstack(
                fill_value='').reset_index()

        # Merge contexts with metadata on 'line_id'
        merged_data = contexts.merge(metadata, on='line_id', how='left')

        #print(merged_data)

        # If the node has 'rank_keys', add ranking columns
        if hasattr(node, 'rank_keys'):
            for algo_key, rank_dict in node.rank_keys.items():
                # Create a Series from rank_dict with name as algo_key
                rank_series = pd.Series(rank_dict, name=algo_key)
                # Merge rank_series into merged_data
                merged_data = merged_data.merge(rank_series, on='line_id', how='left')

        # If the node is partitioned, reorder lines by partitions
        if node.type == "partition" and hasattr(node, 'partitions'):
            partitions = node.partitions
            data = []

            for partition_index in sorted(partitions.keys()):
                partition = partitions[partition_index]
                partition_label = partition.get('label', '')
                line_ids = partition.get('line_ids', [])
                # Select merged_data for the current partition's line_ids
                partition_data = merged_data[merged_data['line_id'].isin(line_ids)].copy()
                # Add 'Partition label' column
                partition_data['Partition label'] = partition_label
                # Preserve the order of line_ids within the partition
                line_id_order = {lid: idx for idx, lid in enumerate(line_ids)}
                partition_data['line_order'] = partition_data['line_id'].map(line_id_order)
                partition_data = partition_data.sort_values('line_order')
                data.append(partition_data)
            # Concatenate all partitions
            if data:
                df = pd.concat(data, ignore_index=True)
            else:
                df = merged_data.copy()
            # Drop 'line_order' column if exists
            if 'line_order' in df.columns:
                df = df.drop(columns=['line_order'])
        else:
            # Determine order of line_ids if node is of type 'order' and has 'sort_keys'
            if node.type == "order" and hasattr(node, 'sort_keys'):
                ordered_line_ids = self._get_ordered_line_ids(node.sort_keys)
                order_df = pd.DataFrame({'line_id': ordered_line_ids})
                df = order_df.merge(merged_data, on='line_id', how='left')
            else:
                df = merged_data

        # Reorder columns
        ranking_columns = list(node.rank_keys.keys()) if hasattr(node, 'rank_keys') else []
        metadata_columns = [col for col in metadata.columns if col != 'line_id']  # exclude line_id if duplicated

        columns_order = ['line_id']
        if 'Partition label' in df.columns:
            columns_order.append('Partition label')
        columns_order.extend(ranking_columns)
        # Add context columns if present
        for ccol in ['Left context', 'KWIC', 'Right context']:
            if ccol in df.columns:
                columns_order.append(ccol)
        columns_order.extend(metadata_columns)

        # Ensure only existing columns are included
        columns_order = [col for col in columns_order if col in df.columns]

        df = df[columns_order]

        # Output to CSV
        df.to_csv(output_file, index=False)


    def find_node_by_id(self, node_id):
        """
        Finds a node in the analysis tree by its unique ID.

        Args:
            node_id (int): The unique identifier of the node to find.

        Returns:
            AnalysisTreeNode: The node with the matching ID, or None if not found.
        """
        def traverse(node):
            if node.id == node_id:
                return node
            for child in node.children:
                result = traverse(child)
                if result:
                    return result
            return None

        return traverse(self.root)

    def _template_from_tree(self):
        """Return a serialisable list describing every node in build order."""
        out = []
        for n in PreOrderIter(self.root):
            if n is self.root:
                continue
            # keep *only* the information needed to rebuild the node
            out.append({
                "id": n.id,
                "parent_id": n.parent.id if n.parent else None,
                "node_type": n.node_type,
                "algorithms": n.algorithms
            })
        return out

    def save_template(self, file):
        """Store annotations+tree-recipe.  *file* = path or open file object."""
        data = {
            "root_id": self.root.id,
            "annotations": self.annotations,
            "nodes": self._template_from_tree()
        }
        if isinstance(file, (str, os.PathLike)):
            with open(file, "w", encoding="utf8") as fh:
                json.dump(data, fh, indent=2)
        else:
            json.dump(data, file, indent=2)

    @staticmethod
    def load_template(file) -> dict:
        """Read the JSON produced by *save_template()*."""
        if isinstance(file, (str, os.PathLike)):
            with open(file, encoding="utf8") as fh:
                return json.load(fh)
        return json.load(file)

    def apply_template(self, template: dict, *, skip_annotations=False):
        """
        Re-apply *template* (loaded via *load_template*) to *this* concordance.
        Returns a dict {old_node_id → new_node}.
        """

        root_id_orig = template.get("root_id")
        id_map = {}
        if root_id_orig is not None:  # map original-root → new root
            id_map[root_id_orig] = self.root
        else:
            id_map[None] = self.root

        # 1  annotations --------------------------------------------------
        if not skip_annotations:
            for ann in template["annotations"]:
                self.add_annotation(
                    (ann["algorithm"], ann["args"]),
                    column_names=ann["column_names"]
                )

        # 2  rebuild the analysis tree -----------------------------------
        for spec in template["nodes"]:
            parent_new = id_map[spec["parent_id"]]

            if spec["node_type"] == "subset":
                algo = spec["algorithms"]["subset"]
                new_node = parent_new.add_subset_node(
                    (algo["algorithm_name"], algo["args"])
                )

            elif spec["node_type"] == "arrangement":
                ordering = [
                    (a["algorithm_name"], a["args"])
                    for a in spec["algorithms"]["ordering"]
                ]
                group = spec["algorithms"]["grouping"]
                grouping = None if group is None else (group["algorithm_name"],
                                                       group["args"])
                new_node = parent_new.add_arrangement_node(
                    ordering=ordering, grouping=grouping
                )
            else:
                raise RuntimeError(f"Unknown node_type {spec['node_type']}")

            id_map[spec["id"]] = new_node

        return id_map


