# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['paprika']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'paprika',
    'version': '1.0.0',
    'description': 'Paprika is a python library that reduces boilerplate. Heavily inspired by Project Lombok.',
    'long_description': 'Paprika is a python library that reduces boilerplate. Heavily inspired by\nProject Lombok.\n\n## Installation\n\npaprika is available on PyPi.\n\n```bash\n$ pip install paprika\n```\n\n## Usage\n\n`paprika` is a decorator-only library and all decorators are exposed at the\ntop-level of the module. If you want to use shorthand notation (i.e. `@data`),\nyou can import all decorators as follows:\n\n```python3\nfrom paprika import *\n```\n\nAlternatively, you can opt to use the longhand notation (i.e. `@paprika.data`)\nby importing `paprika` as follows:\n\n```python3\nimport paprika\n```\n\n## Features & Examples\n\n### @to_string\n\nThe `@to_string` decorator automatically overrides `__str__`\n\n#### Python\n\n```python3\nclass Person:\n    def __init__(self, name: str, age: int):\n        self.name = name\n        self.age = age\n\n    def __str__(self):\n        return f"{self.__name__}@[name={self.name}, age={self.age}]"\n```\n\n#### Python with paprika\n\n```python3\n@to_string\nclass Person:\n    def __init__(self, name: str, age: int):\n        self.name = name\n        self.age = age\n```\n\n----\n\n### @equals_and_hashcode\n\nThe `@equals_and_hashcode` decorator automatically overrides `__eq__`\nand `__hash__`\n\n#### Python\n\n```python3\nclass Person:\n    def __init__(self, name: str, age: int):\n        self.name = name\n        self.age = age\n\n    def __eq__(self, other):\n        return (self.__class__ == other.__class__\n                and\n                self.__dict__ == other.__dict__)\n\n    def __hash__(self):\n        return hash((self.name, self.age))\n```\n\n#### Python with paprika\n\n```python3\n@equals_and_hashcode\nclass Person:\n    def __init__(self, name: str, age: int):\n        self.name = name\n        self.age = age\n```\n\n---\n\n### @data\n\nThe `@data` decorator creates a dataclass by combining `@to_string`\nand `@equals_and_hashcode` and automatically creating a constructor!\n\n#### Python\n\n```python3\nclass Person:\n    def __init__(self, name: str, age: int):\n        self.name = name\n        self.age = age\n\n    def __str__(self):\n        return f"{self.__name__}@[name={self.name}, age={self.age}]"\n\n    def __eq__(self, other):\n        return (self.__class__ == other.__class__\n                and\n                self.__dict__ == other.__dict__)\n\n    def __hash__(self):\n        return hash((self.name, self.age))\n```\n\n#### Python with paprika\n\n```python3\n@data\nclass Person:\n    name: str\n    age: int\n```\n\n#### Footnote on @data and NonNull\n\n`paprika` exposes a `NonNull` generic type that can be used in conjunction with\nthe `@data` decorator to enforce that certain arguments passed to the\nconstructor are not null. The following snippet will raise a `ValueError`:\n\n```python3\n@data\nclass Person:\n    name: NonNull[str]\n    age: int\n\n\np = Person(name=None, age=42)  # ValueError ❌\n```\n\n----\n\n### @singleton\n\nThe `@singleton` decorator can be used to enforce that a class only gets\ninstantiated once within the lifetime of a program. Any subsequent instantiation\nwill return the same original instance.\n\n```python3\n@singleton\nclass Person:\n    def __init__(self, name: str, age: int):\n        self.name = name\n        self.age = age\n\n\np1 = Person(name="Rayan", age=19)\np2 = Person()\nprint(p1 == p2 and p1 is p2)  # True ✅\n```\n\n`@singleton` can be seamlessly combined with `@data`!\n\n```python3\n@singleton\n@data\nclass Person:\n    name: str\n    age: int\n\n\np1 = Person(name="Rayan", age=19)\np2 = Person()\nprint(p1 == p2 and p1 is p2)  # True ✅\n```\n\n---\n\n### @threaded\n\nThe `@threaded` decorator will run the decorated function in a thread by\nsubmitting it to a `ThreadPoolExecutor`. When the decorated function is called,\nit will immediately return a `Future` object. The result can be extracted by\ncalling `.result()` on that `Future`\n\n```python3\n@threaded\ndef waste_time(sleep_time):\n    thread_name = threading.current_thread().name\n    time.sleep(sleep_time)\n    print(f"{thread_name} woke up after {sleep_time}s!")\n    return 42\n\nt1 = waste_time(5)\nt2 = waste_time(2)\nprint(t1)           # <Future at 0x104130a90 state=running>\nprint(t1.result())  # 42\n```\n\n```\nThreadPoolExecutor-0_1 woke up after 2s!\nThreadPoolExecutor-0_0 woke up after 5s!\n```\n\n---\n\n### @repeat\n\nThe `@repeat` decorator will run the decorated function consecutively, as many\ntimes as specified.\n\n```python3\n@repeat(n=5)\ndef hello_world():\n    print("Hello world!")\n\nhello_world()\n```\n\n```\nHello world!\nHello world!\nHello world!\nHello world!\nHello world!\n```\n\n\n## Authors\n\n* **Rayan Hatout** - [GitHub](https://github.com/rayanht)\n  | [Twitter](https://twitter.com/rayanhtt)\n  | [LinkedIn](https://www.linkedin.com/in/rayan-hatout/)\n\nSee also the list of [contributors](https://github.com/rayanht/paprika) who\nparticipated in this project.\n\n## License\n\nThis project is licensed under the MIT License - see\nthe [LICENSE.md](LICENSE.md) file for details',
    'author': 'Rayan Hatout',
    'author_email': 'rayan.hatout@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/rayanht/paprika',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
