"""
Module: System
    This module shall be used to implement subclasses of system. It wraps all information needed and generated by a simulation.
"""

import numpy as np
import pandas as pd

pd.options.mode.use_inf_as_na = True

from ensembler.util.ensemblerTypes import samplerCls, conditionCls, Number, Iterable, Union

from ensembler.util import dataStructure as data
from ensembler.potentials import OneD as pot
from ensembler.samplers.stochastic import metropolisMonteCarloIntegrator

from ensembler.system.basic_system import system


class edsSystem(system):
    """
        The EDS-System is collecting and providing information essential to EDS.
        The Trajectory contains s and Eoff values for each step.
        Functions like set_s (or simply access s) or set_eoff (or simply access Eoff) give direct acces to the EDS potential.

    """
    name = "eds system"
    # EDS Dependend Settings
    state = data.envelopedPStstate

    current_state: data.envelopedPStstate
    potential: pot.envelopedPotential

    # current lambda
    _currentEdsS: Number = np.nan
    _currentEdsEoffs: Iterable[Number] = np.nan

    """
    Attributes
    """

    @property
    def s(self) -> Number:
        """
            s
                smoothing parameter for the EDS-potential
        """
        return self._currentEdsS

    @s.setter
    def s(self, s: Number):
        self._currentEdsS = s
        self.potential.set_s(self._currentEdsS)
        self.update_system_properties()

    def set_s(self, s: Number):
        """
            set_s
                setting a new s-value to the system.

        Parameters
        ----------
        s: Number
            the new smoothing parameter s

        """
        self.s = s

    @property
    def eoff(self):
        """
            Eoff
                Energy Offsets for the EDS-potential

        """
        return self._currentEdsEoffs

    @eoff.setter
    def eoff(self, eoff: Iterable[Number]):
        self._currentEdsEoffs = eoff
        self.potential.Eoff_i = self._currentEdsEoffs
        self.update_system_properties()

    def set_eoff(self, eoff: Iterable[Number]):
        """
            set_Eoff
                setting new Energy offsets.

        Parameters
        ----------
        eoff: Iterable[Number]
            vector of new energy offsets

        """
        self.eoff = eoff

    """
    Magics
    """

    def __init__(self, potential: pot.envelopedPotential = pot.envelopedPotential(
        V_is=[pot.harmonicOscillatorPotential(x_shift=2), pot.harmonicOscillatorPotential(x_shift=-2)], eoff=[0, 0]),
                 sampler: samplerCls = metropolisMonteCarloIntegrator(),
                 conditions: Iterable[conditionCls] = [],
                 temperature: float = 298.0, start_position: Union[Number, Iterable[Number]] = None,
                 eds_s: float = 1, eds_Eoff: Iterable[Number] = [0, 0]):
        """
            __init__
                construct a eds-System that can be used to manage a simulation.

        Parameters
        ----------
        potential:  pot.envelopedPotential, optional
            potential function class to be explored by sampling
        sampler: sampler, optional
            sampling method, that allows exploring the potential function
        conditions: Iterable[condition], optional
            conditions that shall be applied to the system.
        temperature: float, optional
            The temperature of the system (default: 298K)
        start_position:
            starting position for the simulation and setup of the system.
        eds_s: float, optional
            is the S-value of the EDS-Potential
        eds_Eoff: Iterable[Number], optional
            giving the energy offsets for the

        """
        ################################
        # Declare Attributes
        #################################

        self._currentEdsS = eds_s
        self._currentEdsEoffs = eds_Eoff
        self.state = data.envelopedPStstate

        super().__init__(potential=potential, sampler=sampler, conditions=conditions, temperature=temperature,
                         start_position=start_position)

        # Output
        self.set_s(self._currentEdsS)
        self.set_eoff(self._currentEdsEoffs)

    """
    Overwrite Functions to adapt to EDS
    """

    def set_current_state(self, current_position: Union[Number, Iterable[Number]],
                          current_velocities: Union[Number, Iterable[Number]] = 0,
                          current_force: Union[Number, Iterable[Number]] = 0,
                          current_temperature: Union[Number, Iterable[Number]] = 298,
                          current_s: Union[Number, Iterable[Number]] = 1.0,
                          current_eoff: Iterable[Number] = None):
        """
            set_current_state
                set s the current state to the given variables.

        Parameters
        ----------
        current_position: Union[Number, Iterable[Number]]
            The new Position
        current_velocities: Union[Number, Iterable[Number]],
            The new Velocities
        current_force: Union[Number, Iterable[Number]],
            The new Forces of the system
        current_temperature: Union[Number, Iterable[Number]],
            The new S_value
        current_s: Union[Number, Iterable[Number]],
            the new s values.
        current_eoff: Iterable[Number],
            the new eoff values.
        """
        self._currentPosition = current_position
        self._currentForce = current_force
        self._currentVelocities = current_velocities
        self._currentTemperature = current_temperature

        self._currentEdsS = current_s
        if (current_eoff is None):
            self._currentEdsEoffs = [0 for x in range(self.nStates)]
        else:
            self._currentEdsEoffs = current_eoff

        self.update_system_properties()
        self.update_current_state()

    def update_current_state(self):
        """
            updateCurrentState
                This function updates the current state from the _current Variables.
        """
        self._currentState = self.state(position=self._currentPosition, temperature=self._currentTemperature,
                                        total_system_energy=self._currentTotE,
                                        total_potential_energy=self._currentTotPot,
                                        total_kinetic_energy=self._currentTotKin,
                                        dhdpos=self._currentForce, velocity=self._currentVelocities,
                                        s=self._currentEdsS, eoff=self._currentEdsEoffs)

    def append_state(self, new_position: Union[Number, Iterable[Number]], new_velocity: Union[Number, Iterable[Number]],
                     new_forces: Union[Number, Iterable[Number]], new_s: Union[Number, Iterable[Number]],
                     new_eoff: Iterable[Number]):
        """
            append_state
                Append a new state to the trajectory.

        Parameters
        ----------
        new_position: Union[Number, Iterable[Number]]
            new position for the system
        new_velocity: Union[Number, Iterable[Number]]
            new velocity for the system
        new_forces: Union[Number, Iterable[Number]]
            new forces for the system
        new_s: Union[Number, Iterable[Number]]
            new s-value for the system
        new_eoff: Iterable[Number]
            new eoffs for the system
        """

        self._currentPosition = new_position
        self._currentVelocities = new_velocity
        self._currentForce = new_forces
        self._currentEdsS = new_s
        self._currentEdsEoffs = new_eoff

        self.update_system_properties()
        self.update_current_state()

        self._trajectory.append(self.current_state)
