"""
Test to analyze token usage of tool schema expansion in AnswerFormat model.

This test examines:
1. The actual schema generated by model_json_schema()
2. Token count of the schema in prompts
3. Redundant or unnecessary characters
4. Optimization opportunities

Run this test:
    python tests/test_tool_schema_token_usage.py
"""

import sys
import os
import json
from typing import Dict, List, Any

# Add src directory to path
sys.path.insert(0, os.path.abspath(os.path.join(os.path.dirname(__file__), '..', 'src')))

from masai.Tools.Tool import tool
from masai.pydanticModels.AnswerModel import answermodel

# Import tiktoken for accurate token counting (OpenAI's tokenizer)
try:
    import tiktoken
    TIKTOKEN_AVAILABLE = True
except ImportError:
    TIKTOKEN_AVAILABLE = False
    print("⚠️  Warning: tiktoken not installed. Using character count approximation.")
    print("   Install with: pip install tiktoken")


def count_tokens(text: str, model: str = "gpt-4") -> int:
    """Count tokens in text using tiktoken (if available) or approximation."""
    if TIKTOKEN_AVAILABLE:
        encoding = tiktoken.encoding_for_model(model)
        return len(encoding.encode(text))
    else:
        # Rough approximation: 1 token ≈ 4 characters
        return len(text) // 4


def analyze_schema_output(tools: List[Any]) -> Dict[str, Any]:
    """Analyze the schema output for a list of tools."""
    
    results = {
        'tools': [],
        'total_analysis': {}
    }
    
    for tool_obj in tools:
        tool_analysis = {
            'name': tool_obj.name,
            'description': tool_obj.args_schema.model_json_schema().get('description', 'N/A'),
            'properties': tool_obj.args_schema.model_json_schema().get('properties', {}),
            'full_schema': tool_obj.args_schema.model_json_schema(),
        }
        
        # Analyze what gets included in the prompt
        # Line 12: [(tool.name, tool.args_schema.model_json_schema()['description']) for tool in tools]
        tool_description_tuple = (tool_obj.name, tool_obj.args_schema.model_json_schema().get('description', ''))
        
        # Line 19: [(tool.name, tool.args_schema.model_json_schema()['properties']) for tool in tools]
        tool_properties_tuple = (tool_obj.name, tool_obj.args_schema.model_json_schema().get('properties', {}))
        
        tool_analysis['description_tuple'] = str(tool_description_tuple)
        tool_analysis['properties_tuple'] = str(tool_properties_tuple)
        
        # Count tokens
        tool_analysis['description_tuple_tokens'] = count_tokens(str(tool_description_tuple))
        tool_analysis['properties_tuple_tokens'] = count_tokens(str(tool_properties_tuple))
        tool_analysis['description_tuple_chars'] = len(str(tool_description_tuple))
        tool_analysis['properties_tuple_chars'] = len(str(tool_properties_tuple))
        
        results['tools'].append(tool_analysis)
    
    return results


def create_sample_tools():
    """Create sample tools for testing."""

    @tool("search_web")
    def search_web(query: str, max_results: int = 5) -> str:
        """Search the web for information about a topic."""
        return f"Search results for: {query}"

    @tool("calculate")
    def calculate(expression: str) -> str:
        """Perform mathematical calculations."""
        return f"Result: {expression}"

    @tool("send_email")
    def send_email(to: str, subject: str, body: str, cc: str = None, bcc: str = None) -> str:
        """Send an email to a recipient with a subject and body."""
        return f"Email sent to {to}"

    @tool("fetch_data")
    def fetch_data(table: str, filters: Dict = None, limit: int = 100, offset: int = 0) -> str:
        """Fetch data from a database with optional filters and pagination."""
        return f"Data from {table}"

    return [search_web, calculate, send_email, fetch_data]


def test_schema_expansion():
    """Test the schema expansion in AnswerFormat model."""
    
    print("\n" + "="*80)
    print("TEST: Tool Schema Expansion Analysis")
    print("="*80)
    
    # Create sample tools
    tools = create_sample_tools()
    
    print(f"\n📊 Testing with {len(tools)} tools:")
    for tool_obj in tools:
        print(f"   • {tool_obj.name}")
    
    # Analyze each tool's schema
    results = analyze_schema_output(tools)
    
    print("\n" + "="*80)
    print("DETAILED ANALYSIS PER TOOL")
    print("="*80)
    
    total_description_tokens = 0
    total_properties_tokens = 0
    total_description_chars = 0
    total_properties_chars = 0
    
    for i, tool_analysis in enumerate(results['tools'], 1):
        print(f"\n{'─'*80}")
        print(f"Tool {i}: {tool_analysis['name']}")
        print(f"{'─'*80}")
        
        print(f"\n📝 Description Schema (Line 12):")
        print(f"   Output: {tool_analysis['description_tuple']}")
        print(f"   Tokens: {tool_analysis['description_tuple_tokens']}")
        print(f"   Characters: {tool_analysis['description_tuple_chars']}")
        
        print(f"\n📝 Properties Schema (Line 19):")
        print(f"   Output: {tool_analysis['properties_tuple']}")
        print(f"   Tokens: {tool_analysis['properties_tuple_tokens']}")
        print(f"   Characters: {tool_analysis['properties_tuple_chars']}")
        
        print(f"\n📋 Full JSON Schema:")
        print(f"   {json.dumps(tool_analysis['full_schema'], indent=2)}")
        
        total_description_tokens += tool_analysis['description_tuple_tokens']
        total_properties_tokens += tool_analysis['properties_tuple_tokens']
        total_description_chars += tool_analysis['description_tuple_chars']
        total_properties_chars += tool_analysis['properties_tuple_chars']
    
    # Analyze the complete AnswerFormat description strings
    print("\n" + "="*80)
    print("COMPLETE ANSWERFORMAT FIELD DESCRIPTIONS")
    print("="*80)
    
    # Simulate what gets put in the prompt
    tool_description_list = [(tool.name, tool.args_schema.model_json_schema().get('description', '')) for tool in tools]
    tool_properties_list = [(tool.name, tool.args_schema.model_json_schema().get('properties', {})) for tool in tools]
    
    # Line 12: tool field description
    tool_field_description = f"""Select tools among: {tool_description_list}.
                            To use tool, set satisfied=False and specify the tool name, tool_input.
                            To return answer field, set satisfied=True and tool,tool_input=None. 
                            On returning answer, you can not work on current problem anymore."""
    
    # Line 19: tool_input field description
    tool_input_field_description = f"""Always provide tool input as valid JSON. TOOL INPUT SCHEMA: {tool_properties_list}.
                            \n\nSet to None if no tool is needed.
                            If optional parameters are not needed in tool input, for a task, refrain from using them."""
    
    print(f"\n📝 Tool Field Description (Line 12-15):")
    print(f"{'─'*80}")
    print(tool_field_description)
    print(f"{'─'*80}")
    print(f"   Tokens: {count_tokens(tool_field_description)}")
    print(f"   Characters: {len(tool_field_description)}")
    
    print(f"\n📝 Tool Input Field Description (Line 19-21):")
    print(f"{'─'*80}")
    print(tool_input_field_description)
    print(f"{'─'*80}")
    print(f"   Tokens: {count_tokens(tool_input_field_description)}")
    print(f"   Characters: {len(tool_input_field_description)}")
    
    # Total analysis
    print("\n" + "="*80)
    print("TOTAL TOKEN USAGE SUMMARY")
    print("="*80)
    
    total_tokens = count_tokens(tool_field_description) + count_tokens(tool_input_field_description)
    total_chars = len(tool_field_description) + len(tool_input_field_description)
    
    print(f"\n📊 Total Usage:")
    print(f"   Total Tokens: {total_tokens}")
    print(f"   Total Characters: {total_chars}")
    print(f"   Description Tokens: {count_tokens(tool_field_description)}")
    print(f"   Properties Tokens: {count_tokens(tool_input_field_description)}")
    
    # Identify redundant elements
    print("\n" + "="*80)
    print("REDUNDANCY ANALYSIS")
    print("="*80)
    
    redundancies = []
    
    # Check for redundant whitespace
    import re
    whitespace_count = len(re.findall(r'\s{2,}', tool_field_description + tool_input_field_description))
    if whitespace_count > 0:
        redundancies.append(f"• Multiple consecutive whitespaces: {whitespace_count} occurrences")
    
    # Check for redundant newlines
    newline_count = (tool_field_description + tool_input_field_description).count('\n')
    if newline_count > 2:
        redundancies.append(f"• Newline characters: {newline_count} (could be reduced)")
    
    # Check for redundant punctuation
    if tool_field_description.count('.') > 3:
        redundancies.append(f"• Multiple periods: {tool_field_description.count('.')} (could use semicolons)")
    
    # Check for verbose phrasing
    verbose_phrases = [
        ("To use tool, set satisfied=False and specify the tool name, tool_input.", 
         "Use tool: set satisfied=False, specify tool name & input."),
        ("To return answer field, set satisfied=True and tool,tool_input=None.",
         "Return answer: set satisfied=True, tool=None, tool_input=None."),
        ("On returning answer, you can not work on current problem anymore.",
         "Returning answer ends current task."),
        ("Always provide tool input as valid JSON.",
         "Provide tool input as JSON."),
        ("Set to None if no tool is needed.",
         "Set None if no tool needed."),
        ("If optional parameters are not needed in tool input, for a task, refrain from using them.",
         "Omit optional parameters if not needed."),
    ]
    
    print(f"\n🔍 Identified Redundancies:")
    if redundancies:
        for redundancy in redundancies:
            print(f"   {redundancy}")
    else:
        print("   ✅ No obvious redundancies in whitespace/newlines")
    
    print(f"\n💡 Verbose Phrasing Opportunities:")
    total_saved_tokens = 0
    for original, optimized in verbose_phrases:
        original_tokens = count_tokens(original)
        optimized_tokens = count_tokens(optimized)
        saved_tokens = original_tokens - optimized_tokens
        total_saved_tokens += saved_tokens
        print(f"\n   Original ({original_tokens} tokens):")
        print(f"      \"{original}\"")
        print(f"   Optimized ({optimized_tokens} tokens, saves {saved_tokens} tokens):")
        print(f"      \"{optimized}\"")
    
    print(f"\n📊 Potential Token Savings:")
    print(f"   Current Total: {total_tokens} tokens")
    print(f"   Potential Savings: ~{total_saved_tokens} tokens")
    print(f"   Optimized Total: ~{total_tokens - total_saved_tokens} tokens")
    print(f"   Reduction: ~{(total_saved_tokens / total_tokens * 100):.1f}%")
    
    return results, total_tokens, total_saved_tokens


def test_json_schema_verbosity():
    """Test the verbosity of JSON schema output."""
    
    print("\n" + "="*80)
    print("TEST: JSON Schema Verbosity Analysis")
    print("="*80)
    
    tools = create_sample_tools()
    
    print(f"\n📊 Analyzing JSON schema structure...")
    
    for tool_obj in tools:
        schema = tool_obj.args_schema.model_json_schema()
        schema_str = json.dumps(schema, indent=2)
        schema_compact = json.dumps(schema)
        
        print(f"\n{'─'*80}")
        print(f"Tool: {tool_obj.name}")
        print(f"{'─'*80}")
        print(f"   Pretty JSON tokens: {count_tokens(schema_str)}")
        print(f"   Compact JSON tokens: {count_tokens(schema_compact)}")
        print(f"   Savings: {count_tokens(schema_str) - count_tokens(schema_compact)} tokens")
        
        # Check for unnecessary fields
        unnecessary_fields = []
        if 'title' in schema:
            unnecessary_fields.append(f"'title': {schema['title']}")
        if 'type' in schema and schema['type'] == 'object':
            unnecessary_fields.append("'type': 'object' (implicit)")
        
        if 'properties' in schema:
            for prop_name, prop_schema in schema['properties'].items():
                if 'title' in prop_schema:
                    unnecessary_fields.append(f"properties.{prop_name}.title: {prop_schema['title']}")
        
        if unnecessary_fields:
            print(f"\n   ⚠️  Potentially unnecessary fields:")
            for field in unnecessary_fields:
                print(f"      • {field}")
    
    print("\n" + "="*80)
    print("💡 RECOMMENDATIONS")
    print("="*80)
    print("""
   1. Use compact JSON (no indentation) in prompts
   2. Remove 'title' fields from schema (auto-generated from field names)
   3. Remove 'type': 'object' (implicit for Pydantic models)
   4. Consider custom schema serialization to exclude unnecessary fields
    """)


def main():
    """Run all tests."""
    
    print("\n" + "="*80)
    print("TOOL SCHEMA TOKEN USAGE ANALYSIS")
    print("="*80)
    print("\nThis test analyzes token usage of tool schema expansion in AnswerFormat.")
    print("It examines:")
    print("  1. Actual schema generated by model_json_schema()")
    print("  2. Token count in prompts")
    print("  3. Redundant or unnecessary characters")
    print("  4. Optimization opportunities")
    print("\n" + "="*80)
    
    # Run tests
    results, total_tokens, potential_savings = test_schema_expansion()
    test_json_schema_verbosity()
    
    # Final summary
    print("\n" + "="*80)
    print("FINAL SUMMARY")
    print("="*80)
    print(f"\n📊 Current State:")
    print(f"   • Total tokens in tool descriptions: {total_tokens}")
    print(f"   • Number of tools tested: 4")
    print(f"   • Average tokens per tool: {total_tokens // 4}")
    
    print(f"\n💡 Optimization Potential:")
    print(f"   • Potential token savings: ~{potential_savings} tokens")
    print(f"   • Reduction percentage: ~{(potential_savings / total_tokens * 100):.1f}%")
    
    print(f"\n✅ Key Findings:")
    print(f"   1. Verbose phrasing in descriptions can be shortened")
    print(f"   2. JSON schema includes unnecessary 'title' fields")
    print(f"   3. Multiple newlines and whitespace can be reduced")
    print(f"   4. Compact JSON format should be used instead of pretty-printed")
    
    print(f"\n🎯 Recommended Actions:")
    print(f"   1. Shorten field descriptions in AnswerModel.py")
    print(f"   2. Use compact JSON serialization")
    print(f"   3. Create custom schema serializer to exclude 'title' fields")
    print(f"   4. Remove redundant whitespace and newlines")
    
    print("\n" + "="*80)
    
    return 0


if __name__ == "__main__":
    exit_code = main()
    sys.exit(exit_code)

