# AUTOGENERATED! DO NOT EDIT! File to edit: dataset_torch.ipynb (unless otherwise specified).

__all__ = ['SequentialDataset', 'load_dataloaders']

# Cell
import torch
import recsys_slates_dataset.datahelper as datahelper
from torch.utils.data import Dataset, DataLoader
import torch
import json
import numpy as np
import logging
logging.basicConfig(format='%(asctime)s %(message)s', level='INFO')

class SequentialDataset(Dataset):
    '''
    '''
    def __init__(self, data, sample_uniform_slate=False):

        self.data = data
        self.num_items = self.data['slate'].max()+1
        self.sample_uniform_slate = sample_uniform_slate
        self.mask2ind = {'train' : 1, 'valid' : 2, 'test' : 3}

        logging.info(
            "Loading dataset with slate size={} and uniform candidate sampling={}"
            .format(self.data['slate'].size(), self.sample_uniform_slate))

    def __getitem__(self, idx):
        batch = {key: val[idx] for key, val in self.data.items()}

        if self.sample_uniform_slate:
            # Sample actions uniformly:
            action = torch.randint_like(batch['slate'], low=3, high=self.num_items)

            # Add noclick action at pos0
            # and the actual click action at pos 1 (unless noclick):
            action[:,0] = 1
            clicked = batch['click']!=1
            action[:,1][clicked] = batch['click'][clicked]
            batch['slate'] = action
            # Set click idx to 0 if noclick, and 1 otherwise:
            batch['click_idx'] = clicked.long()

        return batch

    def __len__(self):
        return len(self.data['click'])

# Cell
def load_dataloaders(data_dir= "dat",
                     batch_size=1024,
                     num_workers= 0,
                     sample_uniform_slate=False,
                     valid_pct= 0.05,
                     test_pct= 0.05,
                     t_testsplit= 5):
    """
    Loads pytorch dataloaders to be used in training. If used with standard settings, the train/val/test split is equivalent to Eide et. al. 2021
    """

    logging.info("Download data if not in data folder..")
    datahelper.download_data_files(data_dir=data_dir)

    logging.info('Load data..')
    with np.load("{}/data.npz".format(data_dir)) as data_np:
        data = {key: torch.tensor(val) for key, val in data_np.items()}
    dataset = SequentialDataset(data, sample_uniform_slate)

    with open('{}/ind2val.json'.format(data_dir), 'rb') as handle:
        # Use string2int object_hook found here: https://stackoverflow.com/a/54112705
        ind2val = json.load(
            handle,
            object_hook=lambda d: {
                int(k) if k.lstrip('-').isdigit() else k: v
                for k, v in d.items()
                }
            )

    num_users = len(data['click'])
    num_validusers = int(num_users * valid_pct)
    num_testusers = int(num_users * test_pct)
    torch.manual_seed(0)
    perm_user = torch.randperm(num_users)
    valid_user_idx = perm_user[:num_validusers]
    test_user_idx  = perm_user[num_validusers:(num_validusers+num_testusers)]
    train_user_idx = perm_user[(num_validusers+num_testusers):]

    # Split dictionary into train/valid/test with a phase mask that shows which interactions are in different sets
    # (as some users have both train and valid data)
    data_train = data
    data_train['phase_mask'] = torch.ones_like(data['click']).bool()
    data_train['phase_mask'][test_user_idx,t_testsplit:]=False
    data_train['phase_mask'][valid_user_idx,t_testsplit:]=False

    data_valid = {key: val[valid_user_idx] for key, val in data.items()}
    data_valid['phase_mask'] = torch.zeros_like(data_valid['click']).bool()
    data_valid['phase_mask'][:,t_testsplit:] = True

    data_test = {key: val[test_user_idx] for key, val in data.items()}
    data_test['phase_mask'] = torch.zeros_like(data_test['click']).bool()
    data_test['phase_mask'][:,t_testsplit:] = True

    data_dicts = {
        "train" : data_train,
        "valid" : data_valid,
        "test" : data_test}

    datasets = {
        phase : SequentialDataset(data, sample_uniform_slate)
        for phase, data in data_dicts.items()
        }


    # Build dataloaders for each data subset:
    dataloaders = {
        phase: DataLoader(ds, batch_size=batch_size, shuffle=(phase=="train"), num_workers=num_workers)
        for phase, ds in datasets.items()
    }
    for key, dl in dataloaders.items():
        logging.info(
            "In {}: num_users: {}, num_batches: {}".format(key, len(dl.dataset), len(dl))
        )

    # Load item attributes:
    with np.load('{}/itemattr.npz'.format(data_dir), mmap_mode=None) as itemattr_file:
        itemattr = {key : val for key, val in itemattr_file.items()}

    return ind2val, itemattr, dataloaders