"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.clusterArnComponents = exports.ClusterResource = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const consts_1 = require("./cluster-resource-handler/consts");
const cluster_resource_provider_1 = require("./cluster-resource-provider");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * A low-level CFN resource Amazon EKS cluster implemented through a custom
 * resource.
 *
 * Implements EKS create/update/delete through a CloudFormation custom resource
 * in order to allow us to control the IAM role which creates the cluster. This
 * is required in order to be able to allow CloudFormation to interact with the
 * cluster via `kubectl` to enable Kubernetes management capabilities like apply
 * manifest and IAM role/user RBAC mapping.
 */
class ClusterResource extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.roleArn) {
            throw new Error('"roleArn" is required');
        }
        this.adminRole = this.createAdminRole(props);
        const provider = cluster_resource_provider_1.ClusterResourceProvider.getOrCreate(this, {
            adminRole: this.adminRole,
            subnets: props.subnets,
            vpc: props.vpc,
            environment: props.environment,
            onEventLayer: props.onEventLayer,
        });
        const resource = new core_1.CustomResource(this, 'Resource', {
            resourceType: consts_1.CLUSTER_RESOURCE_TYPE,
            serviceToken: provider.serviceToken,
            properties: {
                // the structure of config needs to be that of 'aws.EKS.CreateClusterRequest' since its passed as is
                // to the eks.createCluster sdk invocation.
                Config: {
                    name: props.name,
                    version: props.version,
                    roleArn: props.roleArn,
                    encryptionConfig: props.encryptionConfig,
                    resourcesVpcConfig: {
                        subnetIds: props.resourcesVpcConfig.subnetIds,
                        securityGroupIds: props.resourcesVpcConfig.securityGroupIds,
                        endpointPublicAccess: props.endpointPublicAccess,
                        endpointPrivateAccess: props.endpointPrivateAccess,
                        publicAccessCidrs: props.publicAccessCidrs,
                    },
                },
                AssumeRoleArn: this.adminRole.roleArn,
                // IMPORTANT: increment this number when you add new attributes to the
                // resource. Otherwise, CloudFormation will error with "Vendor response
                // doesn't contain XXX key in object" (see #8276) by incrementing this
                // number, you will effectively cause a "no-op update" to the cluster
                // which will return the new set of attribute.
                AttributesRevision: 2,
            },
        });
        resource.node.addDependency(this.adminRole);
        this.ref = resource.ref;
        this.attrEndpoint = core_1.Token.asString(resource.getAtt('Endpoint'));
        this.attrArn = core_1.Token.asString(resource.getAtt('Arn'));
        this.attrCertificateAuthorityData = core_1.Token.asString(resource.getAtt('CertificateAuthorityData'));
        this.attrClusterSecurityGroupId = core_1.Token.asString(resource.getAtt('ClusterSecurityGroupId'));
        this.attrEncryptionConfigKeyArn = core_1.Token.asString(resource.getAtt('EncryptionConfigKeyArn'));
        this.attrOpenIdConnectIssuerUrl = core_1.Token.asString(resource.getAtt('OpenIdConnectIssuerUrl'));
        this.attrOpenIdConnectIssuer = core_1.Token.asString(resource.getAtt('OpenIdConnectIssuer'));
    }
    createAdminRole(props) {
        const stack = core_1.Stack.of(this);
        // the role used to create the cluster. this becomes the administrator role
        // of the cluster.
        const creationRole = new iam.Role(this, 'CreationRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        // the CreateCluster API will allow the cluster to assume this role, so we
        // need to allow the lambda execution role to pass it.
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [props.roleArn],
        }));
        // if we know the cluster name, restrict the policy to only allow
        // interacting with this specific cluster otherwise, we will have to grant
        // this role to manage all clusters in the account. this must be lazy since
        // `props.name` may contain a lazy value that conditionally resolves to a
        // physical name.
        const resourceArns = core_1.Lazy.list({
            produce: () => {
                const arn = stack.formatArn(clusterArnComponents(stack.resolve(props.name)));
                return stack.resolve(props.name)
                    ? [arn, `${arn}/*`] // see https://github.com/aws/aws-cdk/issues/6060
                    : ['*'];
            },
        });
        const fargateProfileResourceArn = core_1.Lazy.string({
            produce: () => stack.resolve(props.name)
                ? stack.formatArn({ service: 'eks', resource: 'fargateprofile', resourceName: stack.resolve(props.name) + '/*' })
                : '*',
        });
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'eks:CreateCluster',
                'eks:DescribeCluster',
                'eks:DescribeUpdate',
                'eks:DeleteCluster',
                'eks:UpdateClusterVersion',
                'eks:UpdateClusterConfig',
                'eks:CreateFargateProfile',
                'eks:TagResource',
                'eks:UntagResource',
            ],
            resources: resourceArns,
        }));
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['eks:DescribeFargateProfile', 'eks:DeleteFargateProfile'],
            resources: [fargateProfileResourceArn],
        }));
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:GetRole', 'iam:listAttachedRolePolicies'],
            resources: ['*'],
        }));
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:CreateServiceLinkedRole'],
            resources: ['*'],
        }));
        // see https://github.com/aws/aws-cdk/issues/9027
        // these actions are the combined 'ec2:Describe*' actions taken from the EKS SLR policies.
        // (AWSServiceRoleForAmazonEKS, AWSServiceRoleForAmazonEKSForFargate, AWSServiceRoleForAmazonEKSNodegroup)
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'ec2:DescribeInstances',
                'ec2:DescribeNetworkInterfaces',
                'ec2:DescribeSecurityGroups',
                'ec2:DescribeSubnets',
                'ec2:DescribeRouteTables',
                'ec2:DescribeDhcpOptions',
                'ec2:DescribeVpcs',
            ],
            resources: ['*'],
        }));
        // grant cluster creation role sufficient permission to access the specified key
        // see https://docs.aws.amazon.com/eks/latest/userguide/create-cluster.html
        if (props.secretsEncryptionKey) {
            creationRole.addToPolicy(new iam.PolicyStatement({
                actions: [
                    'kms:Encrypt',
                    'kms:Decrypt',
                    'kms:DescribeKey',
                    'kms:CreateGrant',
                ],
                resources: [props.secretsEncryptionKey.keyArn],
            }));
        }
        return creationRole;
    }
}
exports.ClusterResource = ClusterResource;
function clusterArnComponents(clusterName) {
    return {
        service: 'eks',
        resource: 'cluster',
        resourceName: clusterName,
    };
}
exports.clusterArnComponents = clusterArnComponents;
//# sourceMappingURL=data:application/json;base64,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