"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClusterResourceProvider = void 0;
const path = require("path");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const lambda_layer_node_proxy_agent_1 = require("@aws-cdk/lambda-layer-node-proxy-agent");
const HANDLER_DIR = path.join(__dirname, 'cluster-resource-handler');
const HANDLER_RUNTIME = lambda.Runtime.NODEJS_12_X;
/**
 * A custom resource provider that handles cluster operations. It serves
 * multiple custom resources such as the cluster resource and the fargate
 * resource.
 *
 * @internal
 */
class ClusterResourceProvider extends core_1.NestedStack {
    constructor(scope, id, props) {
        super(scope, id);
        const onEvent = new lambda.Function(this, 'OnEventHandler', {
            code: lambda.Code.fromAsset(HANDLER_DIR),
            description: 'onEvent handler for EKS cluster resource provider',
            runtime: HANDLER_RUNTIME,
            environment: props.environment,
            handler: 'index.onEvent',
            timeout: core_1.Duration.minutes(1),
            vpc: props.subnets ? props.vpc : undefined,
            vpcSubnets: props.subnets ? { subnets: props.subnets } : undefined,
        });
        // Allow user to customize the layer
        if (!props.onEventLayer) {
            // `NodeProxyAgentLayer` provides `proxy-agent` which is needed to configure `aws-sdk-js` with a user provided proxy.
            onEvent.addLayers(new lambda_layer_node_proxy_agent_1.NodeProxyAgentLayer(this, 'NodeProxyAgentLayer'));
        }
        else {
            onEvent.addLayers(props.onEventLayer);
        }
        const isComplete = new lambda.Function(this, 'IsCompleteHandler', {
            code: lambda.Code.fromAsset(HANDLER_DIR),
            description: 'isComplete handler for EKS cluster resource provider',
            runtime: HANDLER_RUNTIME,
            handler: 'index.isComplete',
            timeout: core_1.Duration.minutes(1),
            vpc: props.subnets ? props.vpc : undefined,
            vpcSubnets: props.subnets ? { subnets: props.subnets } : undefined,
        });
        this.provider = new cr.Provider(this, 'Provider', {
            onEventHandler: onEvent,
            isCompleteHandler: isComplete,
            totalTimeout: core_1.Duration.hours(1),
            queryInterval: core_1.Duration.minutes(1),
            vpc: props.subnets ? props.vpc : undefined,
            vpcSubnets: props.subnets ? { subnets: props.subnets } : undefined,
        });
        props.adminRole.grant(onEvent.role, 'sts:AssumeRole');
        props.adminRole.grant(isComplete.role, 'sts:AssumeRole');
    }
    static getOrCreate(scope, props) {
        var _a;
        const stack = core_1.Stack.of(scope);
        const uid = '@aws-cdk/aws-eks.ClusterResourceProvider';
        return (_a = stack.node.tryFindChild(uid)) !== null && _a !== void 0 ? _a : new ClusterResourceProvider(stack, uid, props);
    }
    /**
     * The custom resource service token for this provider.
     */
    get serviceToken() { return this.provider.serviceToken; }
}
exports.ClusterResourceProvider = ClusterResourceProvider;
//# sourceMappingURL=data:application/json;base64,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