"use strict";
const network_util_1 = require("../lib/network-util");
module.exports = {
    IP: {
        'should convert a valid IP Address to an integer'(test) {
            test.strictEqual(network_util_1.NetworkUtils.ipToNum('174.66.173.168'), 2923605416);
            test.done();
        },
        'should throw on invalid IP Address'(test) {
            test.throws(() => {
                network_util_1.NetworkUtils.ipToNum('174.266.173.168');
            }, Error, 'is not valid');
            test.done();
        },
        'should convert a valid IP integer to a staring'(test) {
            test.strictEqual(network_util_1.NetworkUtils.numToIp(2923605416), '174.66.173.168');
            test.done();
        },
        'should throw an error for invalid IP'(test) {
            test.throws(() => {
                network_util_1.NetworkUtils.numToIp(2923605416 * 5);
            }, /is not a valid/);
            test.throws(() => {
                network_util_1.NetworkUtils.numToIp(-1);
            }, /is not a valid/);
            test.done();
        },
        'validIp returns true if octect is in 0-255'(test) {
            const invalidIps = ['255.255.0.0', '0.0.0.0', '1.2.3.4', '10.0.0.0', '255.01.01.255'];
            for (const ip of invalidIps) {
                test.strictEqual(true, network_util_1.NetworkUtils.validIp(ip));
            }
            test.done();
        },
        'validIp returns false if octect is not in 0-255'(test) {
            const invalidIps = ['1.2.3.4.689', '-1.55.22.22', '', ' ', '255.264.1.01'];
            for (const ip of invalidIps) {
                test.strictEqual(false, network_util_1.NetworkUtils.validIp(ip));
            }
            test.done();
        },
    },
    CidrBlock: {
        'should return the next valid subnet from offset IP'(test) {
            const num = network_util_1.NetworkUtils.ipToNum('10.0.1.255');
            const newBlock = new network_util_1.CidrBlock(num, 24);
            test.strictEqual(newBlock.cidr, '10.0.2.0/24');
            test.done();
        },
        'nextBlock() returns the next higher CIDR space'(test) {
            const testValues = [
                ['192.168.0.0/24', '192.168.1.0/24'],
                ['10.85.7.0/28', '10.85.7.16/28'],
            ];
            for (const value of testValues) {
                const block = new network_util_1.CidrBlock(value[0]);
                test.strictEqual(block.nextBlock().cidr, value[1]);
            }
            test.done();
        },
        'maxIp() should return the last usable IP from the CidrBlock'(test) {
            const testValues = [
                ['10.0.3.0/28', '10.0.3.15'],
                ['10.0.3.1/28', '10.0.3.31'],
                ['10.0.2.254/28', '10.0.3.15'],
            ];
            for (const value of testValues) {
                const block = new network_util_1.CidrBlock(value[0]);
                test.strictEqual(block.maxIp(), value[1]);
            }
            test.done();
        },
        'minIp() should return the first usable IP from the CidrBlock'(test) {
            const testValues = [
                ['192.168.0.0/18', '192.168.0.0'],
                ['10.0.3.0/24', '10.0.3.0'],
            ];
            for (const answer of testValues) {
                const block = new network_util_1.CidrBlock(answer[0]);
                test.strictEqual(block.minIp(), answer[1]);
            }
            test.done();
        },
        'containsCidr returns true if fully contained'(test) {
            const block = new network_util_1.CidrBlock('10.0.3.0/24');
            const contained = new network_util_1.CidrBlock('10.0.3.0/26');
            test.strictEqual(block.containsCidr(contained), true);
            test.done();
        },
        'containsCidr returns false if not fully contained'(test) {
            const block = new network_util_1.CidrBlock('10.0.3.0/26');
            const notContained = new network_util_1.CidrBlock('10.0.3.0/25');
            test.strictEqual(block.containsCidr(notContained), false);
            test.done();
        },
        'calculateNetmask returns the ip string mask'(test) {
            const netmask = network_util_1.CidrBlock.calculateNetmask(27);
            test.strictEqual(netmask, '255.255.255.224');
            test.done();
        },
    },
    NetworkBuilder: {
        'allows you to carve subnets our of CIDR network'(test) {
            const answers = [
                [
                    '192.168.0.0/28',
                    '192.168.0.16/28',
                    '192.168.0.32/28',
                    '192.168.0.128/25',
                    '192.168.1.0/25',
                    '192.168.4.0/22',
                ],
                [
                    '192.168.0.0/24',
                    '192.168.1.0/24',
                    '192.168.2.0/24',
                    '192.168.3.0/25',
                    '192.168.3.128/25',
                    '192.168.4.0/25',
                    '192.168.4.128/28',
                    '192.168.4.144/28',
                    '192.168.4.160/28',
                ],
            ];
            const wasteful = new network_util_1.NetworkBuilder('192.168.0.0/18');
            const efficient = new network_util_1.NetworkBuilder('192.168.0.0/18');
            wasteful.addSubnets(28, 3);
            wasteful.addSubnets(25, 2);
            wasteful.addSubnets(22, 1);
            efficient.addSubnets(24, 3);
            efficient.addSubnets(25, 3);
            efficient.addSubnets(28, 3);
            const expected = [
                wasteful.cidrStrings.sort(),
                efficient.cidrStrings.sort(),
            ];
            for (let i = 0; i < answers.length; i++) {
                test.deepEqual(answers[i].sort(), expected[i]);
            }
            test.done();
        },
        'throws on subnets < 16 or > 28'(test) {
            const builder = new network_util_1.NetworkBuilder('192.168.0.0/18');
            test.throws(() => {
                builder.addSubnet(15);
            }, network_util_1.InvalidCidrRangeError);
            test.throws(() => {
                builder.addSubnet(29);
            }, network_util_1.InvalidCidrRangeError);
            test.done();
        },
        'throws if you add a subnet outside of the cidr'(test) {
            const builder = new network_util_1.NetworkBuilder('192.168.0.0/18');
            const builder2 = new network_util_1.NetworkBuilder('10.0.0.0/21');
            builder.addSubnets(19, 1);
            builder2.addSubnets(24, 8);
            test.throws(() => {
                builder.addSubnet(19);
                builder.addSubnet(28);
            }, /exceeds remaining space/);
            test.throws(() => {
                builder2.addSubnet(28);
            }, /exceeds remaining space/);
            test.done();
        },
        'maskForRemainingSubnets calcs mask for even split of remaining'(test) {
            const builder = new network_util_1.NetworkBuilder('10.0.0.0/24');
            builder.addSubnet(25);
            test.strictEqual(27, builder.maskForRemainingSubnets(3));
            const builder2 = new network_util_1.NetworkBuilder('192.168.176.0/20');
            builder2.addSubnets(22, 2);
            test.strictEqual(22, builder2.maskForRemainingSubnets(2));
            const builder3 = new network_util_1.NetworkBuilder('192.168.0.0/16');
            test.strictEqual(17, builder3.maskForRemainingSubnets(2));
            const builder4 = new network_util_1.NetworkBuilder('10.0.0.0/16');
            test.strictEqual(18, builder4.maskForRemainingSubnets(4));
            const builder5 = new network_util_1.NetworkBuilder('10.0.0.0/16');
            builder5.addSubnets(26, 3);
            builder5.addSubnets(27, 3);
            test.strictEqual(18, builder5.maskForRemainingSubnets(3));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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