"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const tmp = require("tmp");
const fs = require("fs");
class Seeder extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.setup || !Array.isArray(props.setup))
            throw new Error('setup value must be an array of JSON objects');
        this.props = props;
        const destinationBucket = new aws_s3_1.Bucket(this, 'acds-bucket', {
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        });
        tmp.setGracefulCleanup();
        tmp.dir((err, dir) => {
            if (err)
                throw err;
            this.writeTempFile(dir, 'setup.json', props.setup);
            if (props.teardown) {
                this.writeTempFile(dir, 'teardown.json', props.teardown);
            }
            new aws_s3_deployment_1.BucketDeployment(this, id, {
                sources: [aws_s3_deployment_1.Source.asset(dir)],
                destinationBucket,
                retainOnDelete: false,
            });
        });
        const fn = new aws_lambda_1.Function(this, 'handler', {
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            timeout: core_1.Duration.seconds(60),
            code: aws_lambda_1.Code.fromInline(`
console.log('function loaded');

const AWS = require('aws-sdk');
const s3 = new AWS.S3();

const writeTypeFromAction = (action) => {
  if (action === "put")
    return "Item";
  if (action === "delete")
    return "Key";
}

const run = async (filename, action) => {
  console.log('reading from s3');
  const data = await s3.getObject({
    Bucket: "${destinationBucket.bucketName}", 
    Key: filename
  }).promise();
  console.log('finished reading from s3');
  
  console.log('transforming seed data');
  const seed = JSON.parse(data.Body.toString());
  console.log('finished transforming seed data');
  
  const documentClient = new AWS.DynamoDB.DocumentClient({
    convertEmptyValues: true
  });
  console.log('sending data to dynamodb');
  for(let i = 0; i < seed.length;i++) {
    await documentClient[action]({
      TableName: '${props.table.tableName}',
      [writeTypeFromAction(action)]: seed[i]
    }).promise();
  };
  console.log('finished sending data to dynamodb');
}

exports.handler = async (event) => {
  if (event.mode === "delete" || event.mode === "update")
    await run("teardown.json", "delete");
  if (event.mode === "create" || event.mode === "update")
    await run("setup.json", "put");
}`),
        });
        destinationBucket.grantRead(fn);
        props.table.grantWriteData(fn);
        const onEvent = new custom_resources_1.AwsCustomResource(this, 'on-event', {
            onCreate: {
                ...this.callLambdaOptions(),
                parameters: {
                    FunctionName: fn.functionArn,
                    InvokeArgs: JSON.stringify({
                        mode: 'create',
                    }),
                },
            },
            onDelete: props.teardown
                ? {
                    ...this.callLambdaOptions(),
                    parameters: {
                        FunctionName: fn.functionArn,
                        InvokeArgs: JSON.stringify({
                            mode: 'delete',
                        }),
                    },
                }
                : undefined,
            onUpdate: props.refreshOnUpdate
                ? {
                    ...this.callLambdaOptions(),
                    parameters: {
                        FunctionName: fn.functionArn,
                        InvokeArgs: JSON.stringify({
                            mode: 'update',
                        }),
                    },
                }
                : undefined,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({ resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE }),
        });
        fn.grantInvoke(onEvent);
    }
    callLambdaOptions() {
        return {
            service: 'Lambda',
            action: 'invokeAsync',
            apiVersion: '2015-03-31',
            physicalResourceId: {
                id: `${this.props.table.tableArn}-seeder`,
            },
        };
    }
    writeTempFile(dir, filename, data) {
        const buffer = Buffer.from(JSON.stringify(data));
        const filepath = dir + '/' + filename;
        fs.writeFileSync(filepath, buffer);
    }
}
exports.Seeder = Seeder;
//# sourceMappingURL=data:application/json;base64,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