# AUTOGENERATED! DO NOT EDIT! File to edit: ../notebooks/survey_scenario/modules/inflation_and_calibration_margins.ipynb.

# %% auto 0
__all__ = [
    "config",
    "total_annuel_salaires",
    "inflation_idc",
    "inflation_insee",
    "revalorisation_retraite",
    "reval_chomage",
    "nb_foyers_par_annee",
    "aggm",
    "projection_salaires",
    "inflator_salarial",
    "inflator_economique",
    "inflator_retraite",
    "inflator_chomage",
    "inflation_coefs",
    "inflation_foyers_fiscaux",
    "get_total_population",
    "get_margins",
]

# %% ../notebooks/survey_scenario/modules/inflation_and_calibration_margins.ipynb 4
import numpy as np
from leximpact_aggregates.aggregate import AggregateManager
from leximpact_socio_fisca_simu_etat.config import Configuration
from sklearn.linear_model import LinearRegression

from .helpers_survey_scenario import get_total_extractions

config = Configuration(project_folder="leximpact-prepare-data")

# %% ../notebooks/survey_scenario/modules/inflation_and_calibration_margins.ipynb 11
total_annuel_salaires = {
    "2003": 505_413_219_971,
    "2004": 521_074_208_677,
    "2005": 535_224_289_868,
    "2006": 555_718_533_592,
    "2007": 576_696_329_635,
    "2008": 598_350_414_779,
    "2009": 603_748_075_702,
    "2010": 621_113_771_549,
    "2011": 638_494_009_748,
    "2012": 651_235_711_284,
    "2013": 664_720_821_400,
    "2014": 674_188_954_200,
    "2015": 683_665_874_000,
    "2016": 695_665_787_000,
    "2017": 713_523_524_000,
    "2018": 738_077_177_000,
    "2019": 743_500_200_000,  # Sur le nouveau site
    "2020": 758_741_700_000,  # Sur le nouveau site
    "2021": 758_741_700_000 * 1.023,
    "2022": 758_741_700_000 * 1.023 * 1.031,
    "2023": 758_741_700_000 * 1.023 * 1.031 * 1.033,
}

# %% ../notebooks/survey_scenario/modules/inflation_and_calibration_margins.ipynb 15
# Regression linéaire pour estimer les salaires des années futures
def projection_salaires(yearf):
    yearf = int(yearf)
    salaires_annuels = total_annuel_salaires.copy()

    last_known = int(list(total_annuel_salaires.keys())[-1])
    print(
        "Attention, les années au-dessus de ",
        last_known,
        " sont estimées par régression linéaire",
    )

    # Linear Regression
    # X = np.array([int(i) for i in range(len(inflation_sal.values()))]).reshape(-1, 1)
    X = np.array([int(i) for i in total_annuel_salaires.keys()]).reshape(-1, 1)
    Y = np.array([float(i) for i in total_annuel_salaires.values()]).reshape(-1, 1)

    model = LinearRegression().fit(X, Y)
    for year in range(last_known + 1, yearf + 1):
        to_predict = np.array(year).reshape(-1, 1)

        prediction = model.predict(to_predict)[0][0]
        salaires_annuels[str(year)] = prediction

    return salaires_annuels


# %% ../notebooks/survey_scenario/modules/inflation_and_calibration_margins.ipynb 19
# Création de la liste des taux d'inflation (en %)
# Données issues du PLF - 'Chiffres clés' (=inflation hors tabac = prix à la consommation)
inflation_idc = {
    "2017": 1.0,
    "2018": 1.6,
    "2019": 0.9,
    "2020": 0.2,  # PLF 2022
    "2021": 1.6,  # PLF 2023
    "2022": 5.4,  # PLF 2023
    "2023": 4.3,  # PLF 2023
}

# %% ../notebooks/survey_scenario/modules/inflation_and_calibration_margins.ipynb 20
# Pour info, pas utilisé dans ce code:
# Création de la liste des taux d'inflation (en %)
# Source: https://www.insee.fr/fr/statistiques/2122401#tableau-figure1
inflation_insee = {
    "2009": 0.1,
    "2010": 1.5,
    "2011": 2.0,
    "2012": 2.0,
    "2013": 0.9,
    "2014": 0.5,
    "2015": 0.0,
    "2016": 0.2,
    "2017": 1.0,
    "2018": 1.8,
    "2019": 1.1,
    "2020": 0.5,
    "2021": 1.6,
    "2022": 5.9,  # https://www.insee.fr/fr/statistiques?debut=0&theme=30&conjoncture=43
}

# %% ../notebooks/survey_scenario/modules/inflation_and_calibration_margins.ipynb 23
# Création de la liste des taux d'inflation (en %)
# Source : https://www.ipp.eu/baremes-ipp/regimes-de-retraites/0/0/revalorisation_pension/
# On ne traite que le cas du régime général privé
revalorisation_retraite = {
    "2015": 0.1,
    "2016": 0.0,
    "2017": 0.8,
    "2018": 0.0,
    "2019": 0.3,
    "2020": 0.3,
    "2021": 0.4,
    "2022": 1.1 + 4,  # 1.1% en janvier + 4% en juin
}

# %% ../notebooks/survey_scenario/modules/inflation_and_calibration_margins.ipynb 26
# Création de la liste des taux d'inflation (en %)
# Source : https://www.ipp.eu/baremes-ipp/chomage/allocations_assurance_chomage/sr_alloc/
reval_chomage = {
    "2017": 0.65,
    "2018": 0.7,
    "2019": 0.7,
    "2020": 0.4,
    "2021": 0.6,  # https://www.service-public.fr/particuliers/actualites/A15021
    "2022": 2.9,  # https://www.service-public.fr/particuliers/actualites/A15787
}

# %% ../notebooks/survey_scenario/modules/inflation_and_calibration_margins.ipynb 30
# INFLATION SALARIALE
def inflator_salarial(startp, endp):
    startp = int(startp)
    endp = int(endp)
    adjrate_sal = 1
    total_annuel_salaires_proj = projection_salaires(endp)
    # Calcul du taux d'inflation
    annees_connues = list(total_annuel_salaires_proj.keys())
    annees_connues.remove(annees_connues[0])  # On supprime l'année 1
    inflation_sal_proj = {}
    for year in annees_connues:
        inflation_sal_proj[year] = (
            total_annuel_salaires_proj[year]
            / total_annuel_salaires_proj[str(int(year) - 1)]
        )
    # Calcul du taux d'ajustement
    for year in range(startp + 1, endp + 1):
        adjrate_sal = adjrate_sal * inflation_sal_proj[str(year)]

    return adjrate_sal


# INFLATION ECONOMIQUE
def inflator_economique(startp, endp):
    startp = int(startp)
    endp = int(endp)
    # Calcul du taux d'inflation economique sur plusieurs années
    adjrate_eco = 1
    for year in range(startp, endp):
        rateinfla = inflation_idc[str(year)]
        adjrate_eco = adjrate_eco * (1 + rateinfla / 100)  # Car on a les données en %
    # print("Taux d'ajustement economique : ", adjrate_eco)

    return adjrate_eco


# REVALORISATION DES RETRAITES
def inflator_retraite(startp, endp):
    startp = int(startp)
    endp = int(endp)
    # Calcul du taux d'inflation des retraites sur plusieurs années
    adjrate_ret = 1
    for year in range(startp, endp):
        print(year)
        rateinfla = revalorisation_retraite[str(year)]
        adjrate_ret = adjrate_ret * (1 + rateinfla / 100)  # Car on a les données en %
    # print("Taux d'ajustement de revalorisation des retraites: ", adjrate_ret)

    return adjrate_ret


# REVALORISATION DU CHOMAGE
def inflator_chomage(startp, endp):
    startp = int(startp)
    endp = int(endp)
    # Calcul du taux d'inflation du chomage sur plusieurs années
    adjrate_chom = 1
    for year in range(startp, endp):
        rateinfla = reval_chomage[str(year)]
        adjrate_chom = adjrate_chom * (1 + rateinfla / 100)  # Car on a les données en %
    # print("Taux d'ajustement de revalorisation du chômage: ", adjrate_chom)

    return adjrate_chom


# %% ../notebooks/survey_scenario/modules/inflation_and_calibration_margins.ipynb 32
def inflation_coefs(variables, startp, endp):

    inflator_by_variable = {}

    # Variables indexées sur l'inflation salariale
    cols_type_salarial = [
        "pensions_alimentaires_percues",
        "rag",
        "ric",
        "rnc",
        "salaire_de_base",
        "taxe_habitation",
        "rfr",
        "salaire_imposable",
        "crds_salaire",
    ]
    adjrate_sal = inflator_salarial(startp, endp)

    # Variables indexées sur l'indice des prix à la consommation
    cols_type_idc = [
        "f4ba",
        "loyer",
        "revenus_capitaux_prelevement_bareme",
        "revenus_capitaux_prelevement_liberatoire",
        "revenus_capitaux_prelevement_forfaitaire_unique_ir",
    ]
    adjrate_eco = inflator_economique(startp, endp)

    # Variables indexées sur la revalorisation des retraites
    cols_retraite = ["retraite_brute", "pre_retraite"]
    adjrate_ret = inflator_retraite(startp, endp)

    # Variables indexées sur la revalorisation des indemnisations de chômage
    cols_chomage = ["chomage_brut"]
    adjrate_chom = inflator_chomage(startp, endp)

    # ON INFLATE
    for col in variables:
        # Type 'salarial'
        if col in cols_type_salarial:
            inflator_by_variable.update({col: adjrate_sal})
        # Type IDC
        elif col in cols_type_idc:
            inflator_by_variable.update({col: adjrate_eco})
        # Type retraite
        elif col in cols_retraite:
            inflator_by_variable.update({col: adjrate_ret})
        # Type chomage
        elif col in cols_chomage:
            inflator_by_variable.update({col: adjrate_chom})

    return inflator_by_variable


# %% ../notebooks/survey_scenario/modules/inflation_and_calibration_margins.ipynb 35
nb_foyers_par_annee = {
    "2011": 36_389_256,
    "2012": 36_720_036,
    "2013": 37_119_219,
    "2014": 37_429_459,
    "2015": 37_683_595,
    "2016": 37_889_181,
    "2017": 38_332_977,
    "2018": 38_549_926,
    "2019": 39_331_689,  # 39_167_000 selon un autre document sur le meme site...
    "2020": 39_887_586,  # Source : national.xls # 39_714_000 selon un autre document sur le meme site...
}

# %% ../notebooks/survey_scenario/modules/inflation_and_calibration_margins.ipynb 38
def inflation_foyers_fiscaux(year_end):

    # FOYERS FISCAUX, DATA ET PROJECTION
    nb_foyers = nb_foyers_par_annee.copy()
    last_known = int(list(nb_foyers.keys())[-1])

    # Regression linéaire pour estimer le nombre de foyers des années futures
    if last_known < year_end:
        print(
            "Attention, les années au-dessus de ",
            last_known,
            " sont estimées par régression linéaire",
        )

        # Linear Regression
        X = np.array([int(i) for i in nb_foyers_par_annee.keys()]).reshape(-1, 1)
        Y = np.array([float(i) for i in nb_foyers_par_annee.values()]).reshape(-1, 1)

        model = LinearRegression().fit(X, Y)
        for year in range(last_known + 1, year_end + 1):
            to_predict = np.array(year).reshape(-1, 1)
            prediction = round(model.predict(to_predict)[0][0], 0)
            nb_foyers[str(year)] = prediction

    print("Objectif de ", nb_foyers[str(year_end)], "foyers fiscaux en ", year_end)

    return nb_foyers[str(year_end)]


# %% ../notebooks/survey_scenario/modules/inflation_and_calibration_margins.ipynb 39
def get_total_population(year):

    print("On calibre sur le total de foyers")

    return inflation_foyers_fiscaux(year)


# %% ../notebooks/survey_scenario/modules/inflation_and_calibration_margins.ipynb 41
aggm = AggregateManager(aggregates_path="/mnt/data-out/leximpact-aggregates")


def get_margins(scenario, year, var_list=None):
    """
    Par défaut, si on ne donne pas une liste de variables attendues,
    on renvoie les variables principales
    """

    if var_list is None:
        var_list = scenario._leximpact_variables

    target_margins_by_var = {}
    # On va chercher tous les agrégats de l'année
    for var in var_list:
        # Salaire de base
        if var == "salaire_de_base":
            target_margins_by_var["salaire_de_base"] = total_annuel_salaires[str(year)]
            print("On va pouvoir calibrer le salaire_de_base ")

        else:
            marge = get_total_extractions(var, year)
            if marge != 0:
                target_margins_by_var[var] = marge
                print("On va pouvoir calibrer ", var, "sur un total de ", marge)

    return target_margins_by_var
