"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Document = void 0;
const cfn = require("@aws-cdk/aws-cloudformation");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const statement = require("iam-floyd");
const yaml = require("js-yaml");
const path = require("path");
const resourceType = 'Custom::SSM-Document';
const ID = `CFN::Resource::${resourceType}`;
const cleanID = ID.replace(/:+/g, '-');
const lambdaTimeout = 3; // minutes
/**
 * An SSM document
 */
class Document extends cdk.Construct {
    /**
     * Defines a new SSM document
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Name of the document
         */
        this.name = '';
        this.tags = new cdk.TagManager(cdk.TagType.MAP, 'Custom::SSM-Document');
        this.tags.setTag('CreatedBy', ID);
        const stack = cdk.Stack.of(this).stackName;
        const fn = this.ensureLambda();
        const name = this.fixDocumentName(props.name);
        if (name.length < 3 || name.length > 128) {
            this.node.addError(`SSM Document name ${name} is invalid. The name must be between 3 and 128 characters.`);
            return;
        }
        let content = props.content;
        if (typeof content === 'string') {
            content = yaml.safeLoad(content);
        }
        const document = new cfn.CustomResource(this, `SSM-Document-${name}`, {
            provider: cfn.CustomResourceProvider.fromLambda(fn),
            resourceType: resourceType,
            properties: {
                updateDefaultVersion: props.updateDefaultVersion || true,
                name: name,
                content: content,
                documentType: props.documentType || 'Command',
                targetType: props.targetType || '/',
                StackName: stack,
                tags: cdk.Lazy.anyValue({
                    produce: () => this.tags.renderTags(),
                }),
            },
        });
        this.name = document.getAttString('Name');
    }
    ensureLambda() {
        const stack = cdk.Stack.of(this);
        const constructName = 'SSM-Document-Manager-Lambda';
        const existing = stack.node.tryFindChild(constructName);
        if (existing) {
            return existing;
        }
        const policy = new iam.ManagedPolicy(stack, 'SSM-Document-Manager-Policy', {
            managedPolicyName: `${stack.stackName}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing SSM documents`,
            statements: [
                new statement.Ssm().allow().listDocuments().listTagsForResource(),
                new statement.Ssm()
                    .allow()
                    .createDocument()
                    .addTagsToResource()
                    .ifRequestTag('CreatedBy', ID),
                new statement.Ssm()
                    .allow()
                    .deleteDocument()
                    .describeDocument()
                    .getDocument()
                    .listDocumentVersions()
                    .modifyDocumentPermission()
                    .updateDocument()
                    .updateDocumentDefaultVersion()
                    .addTagsToResource()
                    .removeTagsFromResource()
                    .ifResourceTag('CreatedBy', ID),
            ],
        });
        const role = new iam.Role(stack, 'SSM-Document-Manager-Role', {
            roleName: `${stack.stackName}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing SSM documents`,
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                policy,
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        const fn = new lambda.Function(stack, constructName, {
            functionName: `${stack.stackName}-${cleanID}`,
            role: role,
            description: 'Custom CFN resource: Manage SSM Documents',
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/code.zip')),
            timeout: cdk.Duration.minutes(lambdaTimeout),
        });
        return fn;
    }
    fixDocumentName(name) {
        return name.replace(/[^a-zA-Z0-9_.-]+/g, '-');
    }
}
exports.Document = Document;
//# sourceMappingURL=data:application/json;base64,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