import React from "react";
import {Candidate, Explanations, MetaInformation, Structure} from "../model";
import {Components, JupyterContext} from "../util";
import {TwoColumnLayout} from "../util/layout";
import {LocalSurrogateComponent} from "./details/local_surrogate";
import {FeatureImportanceComponent} from "./details/feature_importance";
import {RawDataset} from "../util/raw_dataset";
import {ComparisonType, DetailsModel} from "./details/model";
import {GlobalSurrogateComponent} from "./details/global_surrogate";
import {CollapseComp} from "../util/collapse";
import {PerformanceDetailsComponent} from "./details/performance_details";
import {HPImportanceComp} from "./details/hp_importance";
import {ConfigurationComponent} from "./details/configuration";

interface CandidateInspectionsProps {
    candidate: Candidate
    structure: Structure
    componentId: string
    componentLabel: string
    meta: MetaInformation

    structures: Structure[]
    explanations: Explanations

    onComparisonRequest: (type: ComparisonType, selectedRow: number) => void
}

interface CandidateInspectionsState {
    selectedSample: number
}

export class CandidateInspections extends React.Component<CandidateInspectionsProps, CandidateInspectionsState> {

    static contextType = JupyterContext;
    context: React.ContextType<typeof JupyterContext>;

    constructor(props: CandidateInspectionsProps) {
        super(props);
        this.state = {selectedSample: undefined}

        this.handleSampleSelection = this.handleSampleSelection.bind(this)
        this.onComparisonRequest = this.onComparisonRequest.bind(this)
    }

    private handleSampleSelection(idx: number) {
        this.setState({selectedSample: idx})
    }

    private onComparisonRequest(type: ComparisonType) {
        this.props.onComparisonRequest(type, this.state.selectedSample)
    }

    render() {
        const {candidate, structure, meta, componentId, componentLabel, structures, explanations} = this.props
        const {selectedSample} = this.state

        const model = new DetailsModel(structure, candidate, componentId, componentLabel, selectedSample)

        return (
            <>
                <h3>Insights for <i>
                    {Components.isPipEnd(model.component) ? `${model.component === Components.SOURCE ? 'Beginning' : 'End'} of the Pipeline` : `${model.algorithm} (${model.component})`}
                </i>
                </h3>
                <p style={{marginBottom: '15px', marginTop: '-15px'}}>
                    Select any step in the pipeline above to calculate the analysis in the following views for the
                    output generated by the selected pipeline step.
                </p>

                <CollapseComp name={'performance'} showInitial={false} help={PerformanceDetailsComponent.HELP}
                              onComparisonRequest={() => this.onComparisonRequest('performance')}>
                    <h3>Performance Details</h3>
                    <PerformanceDetailsComponent model={model} meta={meta}/>
                </CollapseComp>

                <CollapseComp name={'raw-dataset'} showInitial={false} help={RawDataset.HELP}>
                    <h3>Data Set Preview</h3>
                    <TwoColumnLayout widthRight={'25%'}>
                        <RawDataset model={model} onSampleClick={this.handleSampleSelection}/>
                        <LocalSurrogateComponent model={model} orientation={'vertical'}
                                                 onComparisonRequest={this.onComparisonRequest}/>
                    </TwoColumnLayout>
                </CollapseComp>

                <CollapseComp name={'feature-importance'} showInitial={false} help={FeatureImportanceComponent.HELP}
                              onComparisonRequest={() => this.onComparisonRequest('feature_importance')}>
                    <h3>Feature Importance</h3>
                    <FeatureImportanceComponent model={model}/>
                </CollapseComp>

                <CollapseComp name={'global-surrogate'} showInitial={false} help={GlobalSurrogateComponent.HELP}
                              onComparisonRequest={() => this.onComparisonRequest('global_surrogate')}>
                    <h3>Global Surrogate</h3>
                    <GlobalSurrogateComponent model={model}/>
                </CollapseComp>

                <hr/>
                <CollapseComp name={'config-origin'} showInitial={false} help={ConfigurationComponent.HELP}
                              onComparisonRequest={() => this.onComparisonRequest('configuration')}>
                    <h3>Configuration</h3>
                    <ConfigurationComponent model={model} structures={structures} explanations={explanations}/>
                </CollapseComp>

                <CollapseComp name={'hp-importance'} showInitial={false} help={HPImportanceComp.HELP}
                              onComparisonRequest={() => this.onComparisonRequest('hp_importance')}>
                    <h3>Hyperparameter Importance</h3>
                    <HPImportanceComp model={model} metric={meta.metric}/>
                </CollapseComp>
            </>
        )
    }
}
