# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['dollar_lambda']

package_data = \
{'': ['*']}

install_requires = \
['pytypeclass>=0.1.1,<0.2.0']

setup_kwargs = {
    'name': 'dollar-lambda',
    'version': '0.3.4',
    'description': 'An argument parser for Python built from functional first principles',
    'long_description': '<p align="center">\n  <img height="300" src="https://ethanabrooks.github.io/dollar-lambda/logo.png">\n</p>\n\n[$λ](https://ethanabrooks.github.io/dollar-lambda/) provides an alternative to [`argparse`](https://docs.python.org/3/library/argparse.html)\nbased on parser combinators and functional first principles. Arguably, `$λ` is way more expressive than any reasonable\nperson would ever need... but even if it\'s not the parser that we need, it\'s the parser we deserve.\n\n# Installation\n```\npip install dollar-lambda\n```\n\n# [Documentation](https://ethanabrooks.github.io/dollar-lambda/dollar_lambda/)\n\n# Highlights\n`$λ` comes with syntactic sugar that came make building parsers completely boilerplate-free.\nHowever, with more concise syntax comes less flexibility. For more complex parsing situations,\nthere are modular building blocks that lie behind the syntactic sugar which enable parsers to\nhandle any reasonable amount of logical complexity.\n\n## The [`@command`](https://ethanabrooks.github.io/dollar-lambda/dollar_lambda/#dollar_lambda.command) decorator\nThis syntax is best for simple parsers that take a set of unordered arguments:\n\n\n```python\nfrom dollar_lambda import command\n\n\n@command()\ndef main(x: int, dev: bool = False, prod: bool = False):\n    return dict(x=x, dev=dev, prod=prod)\n```\n\nHere is the help text generated by this parser:\n\n\n```python\nmain("-h")\n```\n\n    usage: -x X --dev --prod\n\n\n\n```python\nmain("-x", "1", "--dev")\n```\n\n\n\n\n    {\'x\': 1, \'dev\': True, \'prod\': False}\n\n\n\nUse the `parsers` argument do add custom logic to this parser:\n\n\n```python\nfrom dollar_lambda import flag\n\n\n@command(parsers=dict(kwargs=(flag("dev") | flag("prod"))))\ndef main(x: int, **kwargs):\n    return dict(x=x, **kwargs)\n\n\nmain("-h")\n```\n\n    usage: -x X [--dev | --prod]\n\n\nThis parser requires either a `--dev` or `--prod` flag and maps them to the `kwargs` argument:\n\n\n```python\nmain("-x", "1", "--dev")\n```\n\n\n\n\n    {\'x\': 1, \'dev\': True}\n\n\n\n\n```python\nmain("-x", "1", "--prod")\n```\n\n\n\n\n    {\'x\': 1, \'prod\': True}\n\n\n\n\n```python\nmain("-x", "1")\n```\n\n    usage: -x X [--dev | --prod]\n    The following arguments are required: --dev\n\n\n## `CommandTree` for dynamic dispatch\nFor many programs, a user will want to use one entrypoint for one set of\narguments, and another for another set of arguments. Returning to our example,\nlet\'s say we wanted to execute `prod_function` when the user provides the\n`--prod` flag, and `dev_function` when the user provides the `--dev` flag:\n\n\n```python\nfrom dollar_lambda import CommandTree\n\ntree = CommandTree()\n\n\n@tree.command()\ndef base_function(x: int):\n    print("Ran base_function with arguments:", dict(x=x))\n\n\n@base_function.command()\ndef prod_function(x: int, prod: bool):\n    print("Ran prod_function with arguments:", dict(x=x, prod=prod))\n\n\n@base_function.command()\ndef dev_function(x: int, dev: bool):\n    print("Ran dev_function with arguments:", dict(x=x, dev=dev))\n```\n\nLet\'s see how this parser handles different inputs.\nIf we provide the `--prod` flag, `$λ` automatically invokes\n `prod_function` with the parsed arguments:\n\n\n```python\ntree("-x", "1", "--prod")\n```\n\n    Ran prod_function with arguments: {\'x\': 1, \'prod\': True}\n\n\nIf we provide the `--dev` flag, `$λ` invokes `dev_function`:\n\n\n```python\ntree("-x", "1", "--dev")\n```\n\n    Ran dev_function with arguments: {\'x\': 1, \'dev\': True}\n\n\nWith this configuration, the parser will run `base_function` if neither\n`--prod` nor `--dev` are given:\n\n\n```python\ntree("-x", "1")\n```\n\n    Ran base_function with arguments: {\'x\': 1}\n\n\nAs with `main` in the previous example, you would ordinarily provide `tree` no arguments and it would get them\nfrom the command line.\n\nThere are many other ways to use `CommandTree`,\nincluding some that make use of the `base_function`.\nTo learn more, we recommend the [`CommandTree` tutorial](https://ethanabrooks.github.io/dollar-lambda/dollar_lambda/#commandtree-tutorial).\n\n## Lower-level syntax\n[`@command`](https://ethanabrooks.github.io/dollar-lambda/dollar_lambda/#dollar_lambda.command) and `CommandTree` cover many use cases,\nbut they are both syntactic sugar for a lower-level interface that is far\nmore expressive.\n\nSuppose you want to implement a parser that first tries to parse an option\n(a flag that takes an argument),\n`-x X` and if that fails, tries to parse the input as a variadic sequence of\nfloats:\n\n\n```python\nfrom dollar_lambda import argument, option\n\np = option("x", type=int) | argument("y", type=float).many()\n```\n\nWe go over this syntax in greater detail in the [tutorial](https://ethanabrooks.github.io/dollar-lambda/dollar_lambda/#tutorial).\nFor now, suffice to say that `argument` defines a positional argument,\n[`many`](https://ethanabrooks.github.io/dollar-lambda/dollar_lambda/#dollar_lambda.Parser.many) allows parsers to be applied\nzero or more times, and [`|`](https://ethanabrooks.github.io/dollar-lambda/dollar_lambda/#dollar_lambda.Parser.__or__) expresses alternatives.\n\nHere is the help text:\n\n\n```python\np.parse_args("-h")\n```\n\n    usage: [-x X | [Y ...]]\n\n\nAs promised, this succeeds:\n\n\n```python\np.parse_args("-x", "1")\n```\n\n\n\n\n    {\'x\': 1}\n\n\n\nAnd this succeeds:\n\n\n```python\np.parse_args("1", "2", "3", return_dict=False)\n```\n\n\n\n\n    [(\'y\', 1.0), (\'y\', 2.0), (\'y\', 3.0)]\n\n\n\nAgain, you would ordinarily provide `parse_args` no arguments and it would get the.\nfrom the command line.\n\n### Thanks\nSpecial thanks to ["Functional Pearls"](https://www.cs.nott.ac.uk/~pszgmh/pearl.pdf) by Graham Hutton and Erik Meijer for bringing these topics to life.\n',
    'author': 'Ethan Brooks',
    'author_email': 'ethanabrooks@gmail.com',
    'maintainer': 'Ethan Brooks',
    'maintainer_email': 'ethanabrooks@gmail.com',
    'url': 'https://ethanabrooks.github.io/dollar-lambda/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
